# documentation -- lintian check script -*- perl -*-

# Copyright © 1998 Christian Schwarz and Richard Braakman
# Copyright © 2020 Felix Lechner
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, you can find it on the World Wide
# Web at http://www.gnu.org/copyleft/gpl.html, or write to the Free
# Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
# MA 02110-1301, USA.

package Lintian::Check::Documentation;

use v5.20;
use warnings;
use utf8;

use Const::Fast;
use List::SomeUtils qw(any);
use Unicode::UTF8 qw(encode_utf8);

use Moo;
use namespace::clean;

with 'Lintian::Check';

const my $VERTICAL_BAR => q{|};

# 276 is 255 bytes (maximal length for a filename) plus gzip overhead
const my $MAXIMUM_EMPTY_GZIP_SIZE => 276;

# a list of regex for detecting documentation file checked against basename (xi)
my @DOCUMENTATION_FILE_REGEXES = qw{
  \.docx?$
  \.html?$
  \.info$
  \.latex$
  \.markdown$
  \.md$
  \.odt$
  \.pdf$
  \.readme$
  \.rmd$
  \.rst$
  \.rtf$
  \.tex$
  \.txt$
  ^code[-_]of[-_]conduct$
  ^contribut(?:e|ing)$
  ^copyright$
  ^licen[sc]es?$
  ^howto$
  ^patents?$
  ^readme(?:\.?first|\.1st|\.debian|\.source)?$
  ^todos?$
};

has COMPRESS_FILE_EXTENSIONS => (
    is => 'rw',
    lazy => 1,
    default => sub {
        my ($self) = @_;

        return $self->profile->load_data('files/compressed-file-extensions',
            qr/\s++/,sub { return qr/\Q$_[0]\E/ });
    });

# an OR (|) regex of all compressed extension
has COMPRESS_FILE_EXTENSIONS_OR_ALL => (
    is => 'rw',
    lazy => 1,
    default => sub {
        my ($self) = @_;

        my $text = join($VERTICAL_BAR,
            map {$self->COMPRESS_FILE_EXTENSIONS->value($_) }
              $self->COMPRESS_FILE_EXTENSIONS->all);

        return qr/$text/;
    });

sub visit_installed_files {
    my ($self, $file) = @_;

    my $ppkg = quotemeta($self->processable->name);

    if ($self->processable->type eq 'udeb') {
        if ($file->name =~ m{^usr/share/(?:doc|info)/\S}) {

            $self->hint('udeb-contains-documentation-file', $file->name);
            return;
        }
    }

    if ($file->name =~ m{^usr/share/info/dir(?:\.old)?(?:\.gz)?$}) {
        $self->hint('package-contains-info-dir-file', $file->name);
    }

    # doxygen md5sum
    if ($file->name =~ m{^usr/share/doc/$ppkg/[^/]+/.+\.md5$}s) {

        $self->hint('useless-autogenerated-doxygen-file', $file->name)
          if $file->parent_dir->child('doxygen.png');
    }

    my $regex = $self->COMPRESS_FILE_EXTENSIONS_OR_ALL;

    # doxygen compressed map
    if (
        $file->name =~ m{^usr/share/doc/(?:.+/)?(?:doxygen|html)/
                         .*\.map\.$regex}xs
    ) {
        $self->hint('compressed-documentation', $file->name);
    }

    if ($file->is_file
        && any { $file->basename =~ m{$_}xi } @DOCUMENTATION_FILE_REGEXES) {

        $self->hint('package-contains-documentation-outside-usr-share-doc',
            $file->name)
          unless $file->name =~ m{^etc/}
          || $file->name =~ m{^usr/share/(?:doc|help)/}
          # see Bug#981268
          # usr/lib/python3/dist-packages/*.dist-info/entry_points.txt
          || $file->name =~ m{^ usr/lib/python3/dist-packages/
                              .+ [.] dist-info/entry_points.txt $}sx
          # No need for dh-r packages to automatically
          # create overrides if we just allow them all to
          # begin with.
          || $file->dirname =~ 'usr/lib/R/site-library/'
          # SNMP MIB files, see Bug#971427
          || $file->dirname eq 'usr/share/snmp/mibs/'
          # see Bug#904852
          || $file->dirname =~ m{templates?(?:\.d)?/}
          || ( $file->basename =~ m{\.txt$}
            && $file->dirname =~ m{^usr/lib/python3/.*\.egg-info/}s)
          || ( $file->basename =~ m{^README}xi
            && $file->bytes =~ m{this directory}xi);
    }

    if ($file->name =~ m{^usr/share/doc/\S}) {

        # file not owned by root?
        unless ($file->identity eq 'root/root' || $file->identity eq '0/0') {
            $self->hint('bad-owner-for-doc-file', $file->name, $file->identity,
                '!= root/root (or 0/0)');
        }

        # executable in /usr/share/doc ?
        if (   $file->is_file
            && $file->name !~ m{^usr/share/doc/(?:[^/]+/)?examples/}
            && $file->is_executable) {

            if ($file->is_script) {
                $self->hint('script-in-usr-share-doc', $file->name);
            } else {
                $self->hint('executable-in-usr-share-doc', $file->name,
                    (sprintf '%04o', $file->operm));
            }
        }

        # zero byte file in /usr/share/doc/
        if ($file->is_regular_file and $file->size == 0) {
            # Exceptions: examples may contain empty files for various
            # reasons, Doxygen generates empty *.map files, and Python
            # uses __init__.py to mark module directories.
            unless ($file->name =~ m{^usr/share/doc/(?:[^/]+/)?examples/}
                || $file->name
                =~ m{^usr/share/doc/(?:.+/)?(?:doxygen|html)/.*\.map$}s
                || $file->name=~ m{^usr/share/doc/(?:.+/)?__init__\.py$}s){

                $self->hint('zero-byte-file-in-doc-directory', $file->name);
            }
        }

        if (   $file->name =~ / [.]gz $/msx
            && $file->is_regular_file
            && $file->size <= $MAXIMUM_EMPTY_GZIP_SIZE
            && $file->file_info =~ / gzip \s compressed /msx) {

            open(my $fd, '<:gzip', $file->unpacked_path)
              or die encode_utf8('Cannot open ' . $file->unpacked_path);

            my $f = <$fd>;
            close($fd);

            unless (defined $f and length $f) {
                $self->hint('zero-byte-file-in-doc-directory', $file->name);
            }
        }
    }

    # file directly in /usr/share/doc ?
    if (   $file->is_file
        && $file->name =~ m{^usr/share/doc/[^/]+$}) {
        $self->hint('file-directly-in-usr-share-doc', $file->name);
    }

    # contains an INSTALL file?
    if ($file->name =~ m{^usr/share/doc/$ppkg/INSTALL(?:\..+)*$}s){
        $self->hint('package-contains-upstream-installation-documentation',
            $file->name);
    }

    # contains a README for another distribution/platform?
    if (
        $file->name =~ m{^usr/share/doc/$ppkg/readme\.
                             (?:apple|aix|atari|be|beos|bsd|bsdi
                               |cygwin|darwin|irix|gentoo|freebsd|mac|macos
                               |macosx|netbsd|openbsd|osf|redhat|sco|sgi
                               |solaris|suse|sun|vms|win32|win9x|windows
                             )(?:\.txt)?(?:\.gz)?$}xi
    ) {
        $self->hint('package-contains-readme-for-other-platform-or-distro',
            $file->name);
    }

    # contains a compressed version of objects.inv in
    # sphinx-generated documentation?
    if (   $file->name=~ m{^usr/share/doc/$ppkg/(?:[^/]+/)+objects\.inv\.gz$}
        && $file->file_info =~ m/gzip compressed/) {
        $self->hint('compressed-documentation', $file->name);
    }

    return;
}

1;

# Local Variables:
# indent-tabs-mode: nil
# cperl-indent-level: 4
# End:
# vim: syntax=perl sw=4 sts=4 sr et
