/*
 * Copyright 2017-2018 The OpenSSL Project Authors. All Rights Reserved.
 *
 * Licensed under the OpenSSL license (the "License").  You may not use
 * this file except in compliance with the License.  You can obtain a copy
 * in the file LICENSE in the source distribution or at
 * https://www.openssl.org/source/license.html
 */

/*
 * DRBG test vectors from:
 * https://csrc.nist.gov/projects/cryptographic-algorithm-validation-program/
 */

#include <openssl/obj_mac.h>
#include "internal/nelem.h"
#include "drbg_cavs_data.h"

static const unsigned char kat1_entropyin[] = {
    0x89, 0x0e, 0xb0, 0x67, 0xac, 0xf7, 0x38, 0x2e, 0xff, 0x80, 0xb0, 0xc7,
    0x3b, 0xc8, 0x72, 0xc6,
};
static const unsigned char kat1_nonce[] = {
    0xaa, 0xd4, 0x71, 0xef, 0x3e, 0xf1, 0xd2, 0x03,
};
static const unsigned char kat1_persstr[] = {0};
static const unsigned char kat1_addin0[] = {0};
static const unsigned char kat1_addin1[] = {0};
static const unsigned char kat1_retbits[] = {
    0xa5, 0x51, 0x4e, 0xd7, 0x09, 0x5f, 0x64, 0xf3, 0xd0, 0xd3, 0xa5, 0x76,
    0x03, 0x94, 0xab, 0x42, 0x06, 0x2f, 0x37, 0x3a, 0x25, 0x07, 0x2a, 0x6e,
    0xa6, 0xbc, 0xfd, 0x84, 0x89, 0xe9, 0x4a, 0xf6, 0xcf, 0x18, 0x65, 0x9f,
    0xea, 0x22, 0xed, 0x1c, 0xa0, 0xa9, 0xe3, 0x3f, 0x71, 0x8b, 0x11, 0x5e,
    0xe5, 0x36, 0xb1, 0x28, 0x09, 0xc3, 0x1b, 0x72, 0xb0, 0x8d, 0xdd, 0x8b,
    0xe1, 0x91, 0x0f, 0xa3,
};
static const struct drbg_kat_no_reseed kat1_t = {
    0, kat1_entropyin, kat1_nonce, kat1_persstr,
    kat1_addin0, kat1_addin1, kat1_retbits
};
static const struct drbg_kat kat1 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat1_t
};

static const unsigned char kat2_entropyin[] = {
    0xc4, 0x7b, 0xe8, 0xe8, 0x21, 0x9a, 0x5a, 0x87, 0xc9, 0x40, 0x64, 0xa5,
    0x12, 0x08, 0x9f, 0x2b,
};
static const unsigned char kat2_nonce[] = {
    0xf2, 0xa2, 0x3e, 0x63, 0x6a, 0xee, 0x75, 0xc6,
};
static const unsigned char kat2_persstr[] = {0};
static const unsigned char kat2_addin0[] = {0};
static const unsigned char kat2_addin1[] = {0};
static const unsigned char kat2_retbits[] = {
    0x5a, 0x16, 0x50, 0xbb, 0x6d, 0x6a, 0x16, 0xf6, 0x04, 0x05, 0x91, 0xd5,
    0x6a, 0xbc, 0xd5, 0xdd, 0x3d, 0xb8, 0x77, 0x2a, 0x9c, 0x75, 0xc4, 0x4d,
    0x9f, 0xc6, 0x4d, 0x51, 0xb7, 0x33, 0xd4, 0xa6, 0x75, 0x9b, 0xd5, 0xa6,
    0x4e, 0xc4, 0x23, 0x1a, 0x24, 0xe6, 0x62, 0xfd, 0xd4, 0x7c, 0x82, 0xdb,
    0x63, 0xb2, 0x00, 0xda, 0xf8, 0xd0, 0x98, 0x56, 0x0e, 0xb5, 0xba, 0x7b,
    0xf3, 0xf9, 0xab, 0xf7,
};
static const struct drbg_kat_no_reseed kat2_t = {
    1, kat2_entropyin, kat2_nonce, kat2_persstr,
    kat2_addin0, kat2_addin1, kat2_retbits
};
static const struct drbg_kat kat2 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat2_t
};

static const unsigned char kat3_entropyin[] = {
    0x13, 0x0b, 0x8c, 0x3d, 0x2d, 0x7b, 0x6e, 0x02, 0xc4, 0x10, 0xb4, 0x16,
    0x8e, 0x12, 0x2c, 0x38,
};
static const unsigned char kat3_nonce[] = {
    0x79, 0xa6, 0x74, 0xc5, 0xb2, 0xc5, 0x1a, 0xa9,
};
static const unsigned char kat3_persstr[] = {0};
static const unsigned char kat3_addin0[] = {0};
static const unsigned char kat3_addin1[] = {0};
static const unsigned char kat3_retbits[] = {
    0x57, 0xe8, 0xa1, 0xe5, 0x78, 0xed, 0xe1, 0xc6, 0x68, 0x79, 0xc4, 0x30,
    0xdf, 0x72, 0x64, 0x35, 0xd5, 0x1a, 0x36, 0x9a, 0x0f, 0xe5, 0x9a, 0x03,
    0x58, 0xd1, 0xde, 0x35, 0x2d, 0x42, 0x80, 0xfd, 0x7b, 0x22, 0x5f, 0x5f,
    0x38, 0x6a, 0x4f, 0xcf, 0x12, 0xf7, 0x27, 0x94, 0xad, 0x0f, 0x37, 0x57,
    0xfb, 0x25, 0xde, 0xba, 0x3c, 0x75, 0x12, 0xce, 0x4d, 0x37, 0x33, 0xc7,
    0xee, 0x06, 0x70, 0x43,
};
static const struct drbg_kat_no_reseed kat3_t = {
    2, kat3_entropyin, kat3_nonce, kat3_persstr,
    kat3_addin0, kat3_addin1, kat3_retbits
};
static const struct drbg_kat kat3 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat3_t
};

static const unsigned char kat4_entropyin[] = {
    0x04, 0xd2, 0x41, 0x45, 0x28, 0x76, 0x64, 0xf6, 0x76, 0x2b, 0x5d, 0x2a,
    0x10, 0x2a, 0xc6, 0x03,
};
static const unsigned char kat4_nonce[] = {
    0xec, 0xac, 0x63, 0xe1, 0x21, 0x7e, 0xe3, 0x35,
};
static const unsigned char kat4_persstr[] = {0};
static const unsigned char kat4_addin0[] = {0};
static const unsigned char kat4_addin1[] = {0};
static const unsigned char kat4_retbits[] = {
    0xfa, 0x5e, 0x74, 0x6d, 0xec, 0xd6, 0x80, 0x1e, 0xb7, 0x08, 0x3b, 0x6f,
    0x0e, 0x72, 0x43, 0x2e, 0x1f, 0xd4, 0x24, 0x31, 0x04, 0xf7, 0x48, 0xd0,
    0xf1, 0x90, 0x83, 0x92, 0x3b, 0x95, 0x55, 0x68, 0x8f, 0x43, 0x14, 0x6d,
    0x5a, 0xce, 0xa9, 0x62, 0xda, 0x01, 0x23, 0x1d, 0x9e, 0x5f, 0xaf, 0xf0,
    0xe8, 0x1f, 0x3d, 0x39, 0x4a, 0xce, 0x3a, 0x34, 0x54, 0x53, 0x6d, 0x72,
    0x65, 0x75, 0x04, 0x1f,
};
static const struct drbg_kat_no_reseed kat4_t = {
    3, kat4_entropyin, kat4_nonce, kat4_persstr,
    kat4_addin0, kat4_addin1, kat4_retbits
};
static const struct drbg_kat kat4 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat4_t
};

static const unsigned char kat5_entropyin[] = {
    0xae, 0xde, 0x4e, 0x61, 0x35, 0x56, 0xb1, 0xd5, 0xa3, 0x0f, 0xce, 0x26,
    0x1f, 0xbb, 0x82, 0x0c,
};
static const unsigned char kat5_nonce[] = {
    0x39, 0xac, 0xba, 0x03, 0xc5, 0xf1, 0x0a, 0xf4,
};
static const unsigned char kat5_persstr[] = {0};
static const unsigned char kat5_addin0[] = {0};
static const unsigned char kat5_addin1[] = {0};
static const unsigned char kat5_retbits[] = {
    0x23, 0x2c, 0x44, 0xb8, 0x19, 0xb8, 0x8f, 0x1a, 0xeb, 0x83, 0xf2, 0x03,
    0x4f, 0x84, 0x2d, 0x5a, 0x00, 0xf0, 0x30, 0x15, 0x05, 0xd2, 0xaa, 0x69,
    0xaa, 0xec, 0xb3, 0xcb, 0x14, 0xbc, 0xb1, 0x58, 0x75, 0xe0, 0xfd, 0x60,
    0x07, 0x1a, 0x80, 0xf6, 0x26, 0x2d, 0xce, 0xbc, 0xf4, 0x1a, 0x0e, 0x14,
    0x76, 0xd9, 0x6f, 0x40, 0x97, 0x12, 0xd8, 0x28, 0xae, 0x31, 0x3a, 0x9d,
    0x28, 0xec, 0x2d, 0xee,
};
static const struct drbg_kat_no_reseed kat5_t = {
    4, kat5_entropyin, kat5_nonce, kat5_persstr,
    kat5_addin0, kat5_addin1, kat5_retbits
};
static const struct drbg_kat kat5 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat5_t
};

static const unsigned char kat6_entropyin[] = {
    0x9f, 0x90, 0x54, 0x1c, 0x10, 0xd4, 0xb7, 0xc0, 0x89, 0xfe, 0x68, 0x8e,
    0xa3, 0xef, 0x4f, 0xc6,
};
static const unsigned char kat6_nonce[] = {
    0x1e, 0xac, 0x1c, 0x22, 0x03, 0x6e, 0x2b, 0x22,
};
static const unsigned char kat6_persstr[] = {0};
static const unsigned char kat6_addin0[] = {0};
static const unsigned char kat6_addin1[] = {0};
static const unsigned char kat6_retbits[] = {
    0x71, 0xaf, 0x3f, 0xdf, 0x67, 0x34, 0x04, 0x16, 0x3b, 0x06, 0x73, 0x7e,
    0x0f, 0x39, 0x91, 0x5f, 0xae, 0xc2, 0x18, 0x21, 0x81, 0x6c, 0x31, 0x42,
    0xe8, 0x0a, 0x50, 0x3c, 0x70, 0xcb, 0x2e, 0xdd, 0x46, 0x8c, 0x3f, 0x03,
    0xcb, 0x1f, 0x8a, 0x2f, 0x92, 0x48, 0x63, 0x5b, 0x63, 0xd7, 0xb8, 0xf1,
    0x9e, 0x9e, 0x11, 0xca, 0xf0, 0xab, 0x0c, 0x3d, 0x2f, 0xf9, 0xc7, 0x13,
    0x21, 0xca, 0xc1, 0x54,
};
static const struct drbg_kat_no_reseed kat6_t = {
    5, kat6_entropyin, kat6_nonce, kat6_persstr,
    kat6_addin0, kat6_addin1, kat6_retbits
};
static const struct drbg_kat kat6 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat6_t
};

static const unsigned char kat7_entropyin[] = {
    0x3b, 0x8a, 0x18, 0x1c, 0xea, 0x83, 0x90, 0xd5, 0xd7, 0xa4, 0xe6, 0x51,
    0x5c, 0xf9, 0x2e, 0x3e,
};
static const unsigned char kat7_nonce[] = {
    0x37, 0xf4, 0xd1, 0x74, 0x87, 0x14, 0x34, 0x5b,
};
static const unsigned char kat7_persstr[] = {0};
static const unsigned char kat7_addin0[] = {0};
static const unsigned char kat7_addin1[] = {0};
static const unsigned char kat7_retbits[] = {
    0xd6, 0xfd, 0x0f, 0xfb, 0x10, 0x85, 0x0e, 0xb6, 0xeb, 0x70, 0x50, 0xe6,
    0x1e, 0xac, 0x00, 0xd4, 0x72, 0xf6, 0x5c, 0xd3, 0xd9, 0x35, 0x08, 0x14,
    0x77, 0xfe, 0x44, 0xaa, 0x85, 0x69, 0x4e, 0x12, 0x0a, 0xb5, 0xa1, 0xca,
    0x1f, 0xa1, 0x98, 0xaf, 0x76, 0xdf, 0xa8, 0xd0, 0xab, 0xdf, 0x53, 0xe8,
    0x5a, 0xa8, 0xc8, 0x7f, 0xed, 0x0a, 0x8c, 0x24, 0x16, 0x39, 0x43, 0xb9,
    0x6d, 0x80, 0xaa, 0xfb,
};
static const struct drbg_kat_no_reseed kat7_t = {
    6, kat7_entropyin, kat7_nonce, kat7_persstr,
    kat7_addin0, kat7_addin1, kat7_retbits
};
static const struct drbg_kat kat7 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat7_t
};

static const unsigned char kat8_entropyin[] = {
    0x85, 0x4c, 0xaa, 0x2a, 0x74, 0xf3, 0xf4, 0x3b, 0x6a, 0xbd, 0x80, 0x6d,
    0x67, 0x48, 0xed, 0x80,
};
static const unsigned char kat8_nonce[] = {
    0x0c, 0xbd, 0x13, 0x72, 0xbe, 0xb6, 0x27, 0x36,
};
static const unsigned char kat8_persstr[] = {0};
static const unsigned char kat8_addin0[] = {0};
static const unsigned char kat8_addin1[] = {0};
static const unsigned char kat8_retbits[] = {
    0x46, 0x53, 0x40, 0x28, 0x16, 0x5f, 0xfe, 0x28, 0x45, 0xa9, 0x72, 0x62,
    0x7e, 0x67, 0xe1, 0x53, 0x51, 0x0c, 0xa1, 0x88, 0x0e, 0x6a, 0x3a, 0xd3,
    0x1f, 0xde, 0xe7, 0x1f, 0xf2, 0x40, 0xd3, 0x27, 0x86, 0x24, 0xb3, 0x1f,
    0x2d, 0x38, 0x6d, 0x7b, 0x22, 0x28, 0xce, 0xd6, 0x24, 0xa4, 0x2e, 0x7a,
    0x3b, 0x07, 0x48, 0x0b, 0x23, 0x23, 0x16, 0x6c, 0x18, 0xd1, 0xac, 0x0f,
    0x60, 0x00, 0x2e, 0xc4,
};
static const struct drbg_kat_no_reseed kat8_t = {
    7, kat8_entropyin, kat8_nonce, kat8_persstr,
    kat8_addin0, kat8_addin1, kat8_retbits
};
static const struct drbg_kat kat8 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat8_t
};

static const unsigned char kat9_entropyin[] = {
    0x87, 0xe1, 0xc5, 0x32, 0x99, 0x7f, 0x57, 0xa3, 0x5c, 0x28, 0x6d, 0xe8,
    0x64, 0xbf, 0xf2, 0x64,
};
static const unsigned char kat9_nonce[] = {
    0xa3, 0x9e, 0x98, 0xdb, 0x6c, 0x10, 0x78, 0x7f,
};
static const unsigned char kat9_persstr[] = {0};
static const unsigned char kat9_addin0[] = {0};
static const unsigned char kat9_addin1[] = {0};
static const unsigned char kat9_retbits[] = {
    0x2c, 0x14, 0x7e, 0x24, 0x11, 0x9a, 0xd8, 0xd4, 0xb2, 0xed, 0x61, 0xc1,
    0x53, 0xd0, 0x50, 0xc9, 0x24, 0xff, 0x59, 0x75, 0x15, 0xf1, 0x17, 0x3a,
    0x3d, 0xf4, 0x4b, 0x2c, 0x84, 0x28, 0xef, 0x89, 0x0e, 0xb9, 0xde, 0xf3,
    0xe4, 0x78, 0x04, 0xb2, 0xfd, 0x9b, 0x35, 0x7f, 0xe1, 0x3f, 0x8a, 0x3e,
    0x10, 0xc8, 0x67, 0x0a, 0xf9, 0xdf, 0x2d, 0x6c, 0x96, 0xfb, 0xb2, 0xb8,
    0xcb, 0x2d, 0xd6, 0xb0,
};
static const struct drbg_kat_no_reseed kat9_t = {
    8, kat9_entropyin, kat9_nonce, kat9_persstr,
    kat9_addin0, kat9_addin1, kat9_retbits
};
static const struct drbg_kat kat9 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat9_t
};

static const unsigned char kat10_entropyin[] = {
    0x2f, 0xc6, 0x23, 0x42, 0x90, 0x29, 0xc9, 0x6e, 0xde, 0xf6, 0x16, 0x62,
    0x47, 0xb0, 0x8c, 0xb0,
};
static const unsigned char kat10_nonce[] = {
    0x99, 0x82, 0x66, 0x33, 0x55, 0x58, 0x27, 0x88,
};
static const unsigned char kat10_persstr[] = {0};
static const unsigned char kat10_addin0[] = {0};
static const unsigned char kat10_addin1[] = {0};
static const unsigned char kat10_retbits[] = {
    0x55, 0x96, 0xcb, 0x16, 0xf3, 0xbe, 0x85, 0x52, 0xc1, 0xe5, 0xc1, 0x64,
    0xd5, 0x40, 0xcb, 0x1f, 0xaf, 0x4b, 0xea, 0x87, 0x33, 0xb6, 0x0a, 0x8a,
    0xd0, 0xc4, 0x06, 0x26, 0x25, 0x65, 0x48, 0xc7, 0xaa, 0x96, 0xd1, 0xd2,
    0x72, 0x9d, 0x26, 0xf0, 0x08, 0x73, 0x1f, 0xc3, 0x93, 0x07, 0xbe, 0x5b,
    0xcd, 0x20, 0x81, 0xc6, 0x9e, 0x31, 0x4e, 0x0c, 0x73, 0xe3, 0xd0, 0xfd,
    0x1d, 0x90, 0x58, 0x28,
};
static const struct drbg_kat_no_reseed kat10_t = {
    9, kat10_entropyin, kat10_nonce, kat10_persstr,
    kat10_addin0, kat10_addin1, kat10_retbits
};
static const struct drbg_kat kat10 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat10_t
};

static const unsigned char kat11_entropyin[] = {
    0x98, 0xab, 0x8b, 0x4e, 0xaf, 0xab, 0x6e, 0x53, 0x6f, 0x78, 0x45, 0xab,
    0xec, 0x13, 0x78, 0x08,
};
static const unsigned char kat11_nonce[] = {
    0xdb, 0xa9, 0x44, 0xc9, 0x8b, 0x31, 0x1d, 0x8e,
};
static const unsigned char kat11_persstr[] = {0};
static const unsigned char kat11_addin0[] = {0};
static const unsigned char kat11_addin1[] = {0};
static const unsigned char kat11_retbits[] = {
    0x86, 0xee, 0xd3, 0xa9, 0xfa, 0x53, 0x45, 0x2f, 0xb1, 0x1d, 0xba, 0x9c,
    0xac, 0x8e, 0x44, 0x02, 0x52, 0x29, 0x28, 0xf2, 0x70, 0x5a, 0x5e, 0x58,
    0x2f, 0x4d, 0x00, 0xeb, 0x8f, 0xed, 0x81, 0x8e, 0x62, 0x9c, 0x72, 0xa6,
    0xa7, 0x79, 0xbe, 0xb4, 0xed, 0x9a, 0x23, 0x93, 0x68, 0x23, 0x3c, 0xbf,
    0xcf, 0x55, 0x68, 0x5d, 0xbf, 0x2d, 0xe3, 0x4a, 0xb5, 0x89, 0x20, 0xcf,
    0xac, 0xa4, 0xaa, 0xfe,
};
static const struct drbg_kat_no_reseed kat11_t = {
    10, kat11_entropyin, kat11_nonce, kat11_persstr,
    kat11_addin0, kat11_addin1, kat11_retbits
};
static const struct drbg_kat kat11 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat11_t
};

static const unsigned char kat12_entropyin[] = {
    0x7d, 0xcf, 0x4f, 0xa7, 0x31, 0x13, 0x9c, 0x5b, 0xb6, 0x44, 0x2f, 0xed,
    0x91, 0xe8, 0x9f, 0x68,
};
static const unsigned char kat12_nonce[] = {
    0xff, 0xd9, 0x4f, 0xe2, 0x1a, 0x80, 0x8b, 0x15,
};
static const unsigned char kat12_persstr[] = {0};
static const unsigned char kat12_addin0[] = {0};
static const unsigned char kat12_addin1[] = {0};
static const unsigned char kat12_retbits[] = {
    0x8e, 0xca, 0x20, 0xe3, 0x1c, 0x98, 0x39, 0xb7, 0x41, 0xaa, 0xa9, 0xbf,
    0x6c, 0xee, 0xe2, 0x24, 0xd3, 0x21, 0x26, 0xb3, 0x19, 0x6e, 0xef, 0x3e,
    0xcd, 0x34, 0x3d, 0x41, 0x4d, 0x32, 0x33, 0xb9, 0xfd, 0x0e, 0xa0, 0xed,
    0x1b, 0xc7, 0x70, 0x0c, 0x88, 0xcd, 0x7c, 0x88, 0xd3, 0xc0, 0x76, 0x13,
    0xc4, 0x2c, 0xd1, 0xf9, 0x09, 0xfe, 0xd8, 0xc6, 0xa7, 0x08, 0xd0, 0x5d,
    0x6b, 0x68, 0xfb, 0x2e,
};
static const struct drbg_kat_no_reseed kat12_t = {
    11, kat12_entropyin, kat12_nonce, kat12_persstr,
    kat12_addin0, kat12_addin1, kat12_retbits
};
static const struct drbg_kat kat12 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat12_t
};

static const unsigned char kat13_entropyin[] = {
    0x51, 0x9c, 0x6e, 0xfe, 0xde, 0xd2, 0xa1, 0x10, 0xac, 0x41, 0x83, 0x9a,
    0x8b, 0x8a, 0xbf, 0xce,
};
static const unsigned char kat13_nonce[] = {
    0x81, 0xd9, 0x5e, 0xdc, 0x06, 0xdd, 0xe6, 0xb3,
};
static const unsigned char kat13_persstr[] = {0};
static const unsigned char kat13_addin0[] = {0};
static const unsigned char kat13_addin1[] = {0};
static const unsigned char kat13_retbits[] = {
    0x39, 0xcd, 0xd1, 0x0e, 0x49, 0xe0, 0x35, 0x81, 0xe3, 0x81, 0x1d, 0xdd,
    0x07, 0xd9, 0xd0, 0xac, 0xc3, 0x40, 0x85, 0xa1, 0x2c, 0x1c, 0x6b, 0x87,
    0xa7, 0x63, 0x86, 0xe9, 0x70, 0x32, 0xdc, 0x01, 0xf5, 0x23, 0xf6, 0x32,
    0xec, 0x95, 0x0d, 0x04, 0x34, 0xc2, 0x59, 0x37, 0xe4, 0x32, 0xd9, 0x85,
    0x54, 0x59, 0x9a, 0x85, 0x5d, 0xb9, 0xad, 0xf5, 0x8b, 0x9e, 0x04, 0x59,
    0x7a, 0x21, 0xd0, 0x0d,
};
static const struct drbg_kat_no_reseed kat13_t = {
    12, kat13_entropyin, kat13_nonce, kat13_persstr,
    kat13_addin0, kat13_addin1, kat13_retbits
};
static const struct drbg_kat kat13 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat13_t
};

static const unsigned char kat14_entropyin[] = {
    0x80, 0xc1, 0x9e, 0xea, 0xad, 0x1a, 0x58, 0x37, 0x94, 0x66, 0xaf, 0xae,
    0x1d, 0x80, 0x87, 0x6d,
};
static const unsigned char kat14_nonce[] = {
    0xd9, 0xa2, 0x09, 0x3f, 0x11, 0x89, 0x2b, 0x82,
};
static const unsigned char kat14_persstr[] = {0};
static const unsigned char kat14_addin0[] = {0};
static const unsigned char kat14_addin1[] = {0};
static const unsigned char kat14_retbits[] = {
    0xce, 0xc5, 0x1b, 0x98, 0x5e, 0xc9, 0x7b, 0x18, 0xee, 0x8e, 0xad, 0x36,
    0x15, 0x7b, 0xea, 0xf9, 0x6f, 0x12, 0x9f, 0x28, 0x28, 0x08, 0x89, 0xec,
    0x87, 0x3c, 0x27, 0xb5, 0x62, 0x71, 0x98, 0xc5, 0x85, 0xa6, 0xad, 0x21,
    0xae, 0x23, 0xa9, 0x59, 0xc9, 0xfa, 0x49, 0xd9, 0x85, 0xaf, 0x0d, 0xf4,
    0x02, 0x8f, 0xdf, 0x1f, 0x51, 0xd8, 0x2e, 0x8f, 0x2b, 0x3f, 0x02, 0x88,
    0x53, 0xf1, 0x4e, 0x8f,
};
static const struct drbg_kat_no_reseed kat14_t = {
    13, kat14_entropyin, kat14_nonce, kat14_persstr,
    kat14_addin0, kat14_addin1, kat14_retbits
};
static const struct drbg_kat kat14 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat14_t
};

static const unsigned char kat15_entropyin[] = {
    0xb7, 0x7b, 0xd2, 0x2e, 0xfe, 0xb7, 0x71, 0x58, 0x6d, 0x51, 0x6f, 0x58,
    0x21, 0x92, 0xa3, 0x11,
};
static const unsigned char kat15_nonce[] = {
    0xa6, 0x99, 0xf4, 0x2a, 0x49, 0x81, 0xfe, 0xfc,
};
static const unsigned char kat15_persstr[] = {0};
static const unsigned char kat15_addin0[] = {0};
static const unsigned char kat15_addin1[] = {0};
static const unsigned char kat15_retbits[] = {
    0x7f, 0xf0, 0xab, 0xea, 0xff, 0xc6, 0xec, 0x92, 0x3c, 0xeb, 0xd9, 0x10,
    0xf5, 0x93, 0x7b, 0xf1, 0x4f, 0xc5, 0x2d, 0x2a, 0x74, 0x25, 0x83, 0x88,
    0xc7, 0x6c, 0x1b, 0xc9, 0xe6, 0x35, 0xad, 0xf1, 0x75, 0x85, 0x8f, 0x0a,
    0x55, 0x87, 0x7e, 0x7e, 0x9f, 0x5f, 0x86, 0x7d, 0x00, 0xb9, 0xb1, 0x36,
    0x3d, 0xde, 0x46, 0x28, 0x8a, 0x6f, 0xff, 0x53, 0x21, 0xf4, 0x69, 0x44,
    0x5a, 0xad, 0x41, 0x98,
};
static const struct drbg_kat_no_reseed kat15_t = {
    14, kat15_entropyin, kat15_nonce, kat15_persstr,
    kat15_addin0, kat15_addin1, kat15_retbits
};
static const struct drbg_kat kat15 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat15_t
};

static const unsigned char kat16_entropyin[] = {
    0xb4, 0x08, 0xce, 0xfb, 0x5b, 0xc7, 0x15, 0x7d, 0x3f, 0x26, 0xcb, 0x95,
    0xa8, 0xb1, 0xd7, 0xac,
};
static const unsigned char kat16_nonce[] = {
    0x02, 0x6c, 0x76, 0x8f, 0xd5, 0x77, 0xb9, 0x2a,
};
static const unsigned char kat16_persstr[] = {0};
static const unsigned char kat16_addin0[] = {
    0x57, 0x37, 0xef, 0x81, 0xde, 0xe3, 0x65, 0xb6, 0xda, 0xdb, 0x3f, 0xee,
    0xbf, 0x5d, 0x10, 0x84,
};
static const unsigned char kat16_addin1[] = {
    0x33, 0x68, 0xa5, 0x16, 0xb3, 0x43, 0x1a, 0x3d, 0xaa, 0xa6, 0x0d, 0xc8,
    0x74, 0x3c, 0x82, 0x97,
};
static const unsigned char kat16_retbits[] = {
    0x4e, 0x90, 0x9e, 0xbb, 0x24, 0x14, 0x7a, 0x00, 0x04, 0x06, 0x3a, 0x5e,
    0x47, 0xee, 0x04, 0x4f, 0xea, 0xd6, 0x10, 0xd6, 0x23, 0x24, 0xbd, 0x0f,
    0x96, 0x3f, 0x75, 0x6f, 0xb9, 0x13, 0x61, 0xe8, 0xb8, 0x7e, 0x3a, 0x76,
    0xa3, 0x98, 0x14, 0x3f, 0xe8, 0x81, 0x30, 0xfe, 0x1b, 0x54, 0x7b, 0x66,
    0x1a, 0x64, 0x80, 0xc7, 0x11, 0xb7, 0x39, 0xf1, 0x8a, 0x9d, 0xf3, 0xae,
    0x51, 0xd4, 0x1b, 0xc9,
};
static const struct drbg_kat_no_reseed kat16_t = {
    0, kat16_entropyin, kat16_nonce, kat16_persstr,
    kat16_addin0, kat16_addin1, kat16_retbits
};
static const struct drbg_kat kat16 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat16_t
};

static const unsigned char kat17_entropyin[] = {
    0x71, 0xbd, 0xce, 0x35, 0x42, 0x7d, 0x20, 0xbf, 0x58, 0xcf, 0x17, 0x74,
    0xce, 0x72, 0xd8, 0x33,
};
static const unsigned char kat17_nonce[] = {
    0x34, 0x50, 0x2d, 0x8f, 0x5b, 0x14, 0xc4, 0xdd,
};
static const unsigned char kat17_persstr[] = {0};
static const unsigned char kat17_addin0[] = {
    0x66, 0xef, 0x42, 0xd6, 0x9a, 0x8c, 0x3d, 0x6d, 0x4a, 0x9e, 0x95, 0xa6,
    0x91, 0x4d, 0x81, 0x56,
};
static const unsigned char kat17_addin1[] = {
    0xe3, 0x18, 0x83, 0xd9, 0x4b, 0x5e, 0xc4, 0xcc, 0xaa, 0x61, 0x2f, 0xbb,
    0x4a, 0x55, 0xd1, 0xc6,
};
static const unsigned char kat17_retbits[] = {
    0x97, 0x33, 0xe8, 0x20, 0x12, 0xe2, 0x7b, 0xa1, 0x46, 0x8f, 0xf2, 0x34,
    0xb3, 0xc9, 0xb6, 0x6b, 0x20, 0xb2, 0x4f, 0xee, 0x27, 0xd8, 0x0b, 0x21,
    0x8c, 0xff, 0x63, 0x73, 0x69, 0x29, 0xfb, 0xf3, 0x85, 0xcd, 0x88, 0x8e,
    0x43, 0x2c, 0x71, 0x8b, 0xa2, 0x55, 0xd2, 0x0f, 0x1d, 0x7f, 0xe3, 0xe1,
    0x2a, 0xa3, 0xe9, 0x2c, 0x25, 0x89, 0xc7, 0x14, 0x52, 0x99, 0x56, 0xcc,
    0xc3, 0xdf, 0xb3, 0x81,
};
static const struct drbg_kat_no_reseed kat17_t = {
    1, kat17_entropyin, kat17_nonce, kat17_persstr,
    kat17_addin0, kat17_addin1, kat17_retbits
};
static const struct drbg_kat kat17 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat17_t
};

static const unsigned char kat18_entropyin[] = {
    0x9c, 0x0e, 0x4a, 0xea, 0xfc, 0x35, 0x7f, 0xf8, 0xe3, 0xf5, 0x40, 0xa4,
    0x55, 0x67, 0x8d, 0x7e,
};
static const unsigned char kat18_nonce[] = {
    0xef, 0xd8, 0x3d, 0xc5, 0xc5, 0x31, 0xd1, 0x67,
};
static const unsigned char kat18_persstr[] = {0};
static const unsigned char kat18_addin0[] = {
    0x58, 0xa1, 0x18, 0x14, 0x08, 0x1c, 0x1c, 0x35, 0xaf, 0x89, 0x88, 0xf1,
    0x31, 0x40, 0xf6, 0xcc,
};
static const unsigned char kat18_addin1[] = {
    0x19, 0x19, 0x22, 0x70, 0xd6, 0x1a, 0x43, 0x26, 0x4a, 0x30, 0xba, 0x9e,
    0xbc, 0x72, 0x8a, 0x5b,
};
static const unsigned char kat18_retbits[] = {
    0x1e, 0xff, 0x1c, 0xf6, 0xb5, 0x75, 0x31, 0x55, 0x90, 0x70, 0x38, 0x82,
    0x35, 0x9b, 0x6a, 0xdb, 0x3c, 0x21, 0xeb, 0xcf, 0xda, 0x5d, 0x3b, 0x5d,
    0x6e, 0x88, 0xac, 0xab, 0x6e, 0x87, 0x9e, 0x0a, 0x6f, 0x75, 0x37, 0x5c,
    0x02, 0xd1, 0xd3, 0xfb, 0xae, 0x94, 0x3c, 0x34, 0x43, 0xe3, 0x9c, 0x29,
    0x9b, 0x80, 0xe2, 0x41, 0xcd, 0x36, 0x5d, 0x4a, 0xf8, 0xcd, 0x1a, 0xd5,
    0xd9, 0xd5, 0x4c, 0x58,
};
static const struct drbg_kat_no_reseed kat18_t = {
    2, kat18_entropyin, kat18_nonce, kat18_persstr,
    kat18_addin0, kat18_addin1, kat18_retbits
};
static const struct drbg_kat kat18 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat18_t
};

static const unsigned char kat19_entropyin[] = {
    0x95, 0x07, 0x01, 0x9b, 0x1f, 0xe8, 0x79, 0x84, 0xbe, 0xc1, 0xd0, 0x9c,
    0xec, 0xcf, 0x36, 0x35,
};
static const unsigned char kat19_nonce[] = {
    0x47, 0x58, 0x62, 0x74, 0xfa, 0x75, 0x3e, 0x0d,
};
static const unsigned char kat19_persstr[] = {0};
static const unsigned char kat19_addin0[] = {
    0xf1, 0x7d, 0xe4, 0xce, 0xef, 0x9f, 0x2f, 0x23, 0x42, 0x20, 0xcb, 0x71,
    0xd8, 0xce, 0xb7, 0x0d,
};
static const unsigned char kat19_addin1[] = {
    0x99, 0xd4, 0x8d, 0x9c, 0xe2, 0xfe, 0x76, 0xdc, 0xb8, 0xd0, 0x5f, 0xa4,
    0xec, 0x07, 0x1a, 0xec,
};
static const unsigned char kat19_retbits[] = {
    0x25, 0xa0, 0xd1, 0x67, 0x8e, 0x25, 0xb1, 0x94, 0x39, 0xd8, 0x33, 0x4c,
    0xa9, 0x02, 0xdb, 0x63, 0x47, 0xa9, 0xa4, 0x2b, 0x86, 0xf3, 0x54, 0xb9,
    0xe0, 0xb3, 0xc4, 0xf4, 0xe8, 0xba, 0x9d, 0x19, 0x2a, 0x5d, 0x76, 0x5a,
    0xae, 0x5f, 0x4c, 0x5a, 0x74, 0xa0, 0x17, 0x7d, 0x3e, 0xb6, 0x13, 0x70,
    0x62, 0xe6, 0xd9, 0x8b, 0x65, 0x18, 0x22, 0x26, 0x5b, 0xfe, 0xfc, 0x92,
    0x9a, 0x39, 0x69, 0xdd,
};
static const struct drbg_kat_no_reseed kat19_t = {
    3, kat19_entropyin, kat19_nonce, kat19_persstr,
    kat19_addin0, kat19_addin1, kat19_retbits
};
static const struct drbg_kat kat19 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat19_t
};

static const unsigned char kat20_entropyin[] = {
    0x12, 0xe8, 0xb3, 0xda, 0x3a, 0x3b, 0x3e, 0x77, 0x3a, 0xfc, 0x23, 0x05,
    0xbd, 0xa8, 0x6d, 0xb6,
};
static const unsigned char kat20_nonce[] = {
    0xc4, 0x9f, 0x26, 0x1a, 0x9a, 0xb9, 0xd9, 0xf8,
};
static const unsigned char kat20_persstr[] = {0};
static const unsigned char kat20_addin0[] = {
    0x86, 0xb6, 0x38, 0x52, 0x41, 0x96, 0x15, 0x9d, 0xea, 0x01, 0x76, 0xe5,
    0x46, 0x2a, 0xe6, 0x5c,
};
static const unsigned char kat20_addin1[] = {
    0x5e, 0xad, 0xb7, 0x67, 0x72, 0xbe, 0x3c, 0xe8, 0x47, 0xaf, 0x6b, 0x77,
    0x94, 0xe0, 0x02, 0xe3,
};
static const unsigned char kat20_retbits[] = {
    0x1b, 0xbf, 0x85, 0xe0, 0x2c, 0x1f, 0xc0, 0x4b, 0xea, 0x5f, 0xc4, 0x7f,
    0x23, 0x6a, 0x07, 0x1c, 0x13, 0xf8, 0xf5, 0x2e, 0xb1, 0x13, 0xcb, 0x83,
    0xc4, 0x15, 0x2a, 0xc8, 0x32, 0x62, 0x7a, 0xbb, 0xa6, 0xec, 0xde, 0xc1,
    0x7a, 0x9f, 0xf0, 0x30, 0x5b, 0x7a, 0x49, 0x81, 0x79, 0x83, 0xfa, 0x5b,
    0x66, 0x51, 0x45, 0xe7, 0xba, 0x9a, 0x5d, 0xf5, 0x72, 0x26, 0xec, 0xcf,
    0xc1, 0x64, 0xe3, 0xbb,
};
static const struct drbg_kat_no_reseed kat20_t = {
    4, kat20_entropyin, kat20_nonce, kat20_persstr,
    kat20_addin0, kat20_addin1, kat20_retbits
};
static const struct drbg_kat kat20 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat20_t
};

static const unsigned char kat21_entropyin[] = {
    0xb6, 0x08, 0xb1, 0x49, 0xcf, 0x44, 0xfe, 0x06, 0x06, 0x13, 0xb1, 0x66,
    0x1d, 0xa0, 0xdf, 0x63,
};
static const unsigned char kat21_nonce[] = {
    0x95, 0x17, 0xc9, 0x93, 0xf1, 0x50, 0x34, 0x77,
};
static const unsigned char kat21_persstr[] = {0};
static const unsigned char kat21_addin0[] = {
    0x55, 0x9c, 0xb8, 0x12, 0xc3, 0x75, 0x8b, 0x2b, 0x31, 0x9d, 0xd9, 0xf5,
    0x76, 0x8e, 0x8d, 0x3d,
};
static const unsigned char kat21_addin1[] = {
    0xc9, 0xaf, 0x98, 0x7e, 0xfc, 0xe4, 0x70, 0x0f, 0xa9, 0xc8, 0xeb, 0xb9,
    0xfe, 0xcf, 0x00, 0xda,
};
static const unsigned char kat21_retbits[] = {
    0x5b, 0x48, 0x13, 0x4d, 0xc7, 0xe6, 0x58, 0x9f, 0x1b, 0x40, 0x2c, 0xf5,
    0xc4, 0xf6, 0xe3, 0xa3, 0xeb, 0xfd, 0xa8, 0x26, 0x72, 0x50, 0xe8, 0xe8,
    0x11, 0xec, 0x05, 0x98, 0x6f, 0xf2, 0x1c, 0x78, 0xa1, 0x69, 0x30, 0xa8,
    0xe0, 0x81, 0x03, 0x07, 0x76, 0x08, 0xc0, 0x6f, 0xa5, 0xe2, 0xa0, 0x48,
    0xe8, 0x5d, 0xd6, 0xf3, 0x95, 0xce, 0xe2, 0x01, 0xd3, 0xcd, 0xe6, 0x35,
    0xe1, 0x7e, 0x20, 0x3e,
};
static const struct drbg_kat_no_reseed kat21_t = {
    5, kat21_entropyin, kat21_nonce, kat21_persstr,
    kat21_addin0, kat21_addin1, kat21_retbits
};
static const struct drbg_kat kat21 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat21_t
};

static const unsigned char kat22_entropyin[] = {
    0xe9, 0x2f, 0xd0, 0x7a, 0x09, 0xf2, 0xbc, 0xcb, 0x43, 0x42, 0x4c, 0x57,
    0x57, 0x73, 0xb0, 0xf0,
};
static const unsigned char kat22_nonce[] = {
    0xb4, 0xe3, 0xca, 0x71, 0xae, 0x15, 0xa0, 0x36,
};
static const unsigned char kat22_persstr[] = {0};
static const unsigned char kat22_addin0[] = {
    0xe1, 0xb4, 0xb0, 0x40, 0x13, 0xdc, 0x12, 0x18, 0x8c, 0x62, 0x3d, 0x10,
    0x11, 0xd4, 0x08, 0xf1,
};
static const unsigned char kat22_addin1[] = {
    0x79, 0x56, 0xbf, 0x5b, 0x16, 0x58, 0x0b, 0x3e, 0x72, 0x0a, 0x81, 0x33,
    0x3e, 0xae, 0x24, 0x83,
};
static const unsigned char kat22_retbits[] = {
    0x21, 0xa4, 0x67, 0x22, 0x42, 0x46, 0x56, 0x37, 0x0c, 0x47, 0x5b, 0x69,
    0xa1, 0xfb, 0xc7, 0x0c, 0x8c, 0xbf, 0x44, 0x2d, 0xc2, 0xbc, 0x28, 0xe0,
    0xab, 0x34, 0xbb, 0x6f, 0xfe, 0xd0, 0x6d, 0x10, 0xf4, 0x90, 0xdf, 0x7e,
    0x78, 0x7c, 0x12, 0xb6, 0x52, 0xd2, 0x19, 0xa5, 0x77, 0x30, 0x6f, 0x04,
    0xa9, 0xc8, 0x78, 0xb6, 0xcf, 0xdf, 0xf4, 0x3e, 0x13, 0x55, 0xc4, 0xd1,
    0x10, 0xfb, 0x0f, 0xb2,
};
static const struct drbg_kat_no_reseed kat22_t = {
    6, kat22_entropyin, kat22_nonce, kat22_persstr,
    kat22_addin0, kat22_addin1, kat22_retbits
};
static const struct drbg_kat kat22 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat22_t
};

static const unsigned char kat23_entropyin[] = {
    0xde, 0x0b, 0x4e, 0xa1, 0xa4, 0x40, 0xce, 0x68, 0x0e, 0xd0, 0x04, 0xcf,
    0x64, 0xaf, 0xbe, 0x0f,
};
static const unsigned char kat23_nonce[] = {
    0xab, 0xb3, 0x1a, 0x3a, 0xb1, 0x5e, 0xd4, 0x49,
};
static const unsigned char kat23_persstr[] = {0};
static const unsigned char kat23_addin0[] = {
    0x45, 0xe7, 0x10, 0x10, 0x0c, 0xad, 0x00, 0xf3, 0xd2, 0x93, 0xdb, 0x75,
    0x5e, 0xe1, 0x99, 0x0d,
};
static const unsigned char kat23_addin1[] = {
    0xb9, 0x11, 0xc9, 0xa4, 0x38, 0x9e, 0x71, 0x67, 0x98, 0x2a, 0x51, 0x24,
    0x8e, 0xe2, 0x93, 0x7e,
};
static const unsigned char kat23_retbits[] = {
    0x01, 0xc3, 0xaf, 0x83, 0x55, 0x7c, 0x63, 0x2d, 0xd8, 0x22, 0x83, 0x3f,
    0x18, 0xe5, 0x48, 0xfb, 0x60, 0x36, 0xe9, 0xcc, 0xac, 0xc3, 0xb4, 0x33,
    0xc5, 0xa7, 0x00, 0x79, 0x39, 0x93, 0x30, 0x89, 0x16, 0x22, 0x0e, 0x5a,
    0x74, 0x16, 0x39, 0x86, 0x1c, 0x2f, 0x17, 0x71, 0x9b, 0x32, 0x51, 0x22,
    0x06, 0xe3, 0x6e, 0x94, 0x7b, 0x28, 0x60, 0x07, 0x3b, 0x11, 0x3c, 0xcd,
    0xa3, 0xfe, 0x31, 0x68,
};
static const struct drbg_kat_no_reseed kat23_t = {
    7, kat23_entropyin, kat23_nonce, kat23_persstr,
    kat23_addin0, kat23_addin1, kat23_retbits
};
static const struct drbg_kat kat23 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat23_t
};

static const unsigned char kat24_entropyin[] = {
    0x72, 0x27, 0xe1, 0x1c, 0xfc, 0x1a, 0x6b, 0x84, 0xb4, 0x0e, 0xca, 0x24,
    0xe7, 0x7f, 0x8c, 0x64,
};
static const unsigned char kat24_nonce[] = {
    0xab, 0x7e, 0x9f, 0xc4, 0x5e, 0x58, 0x63, 0xfb,
};
static const unsigned char kat24_persstr[] = {0};
static const unsigned char kat24_addin0[] = {
    0xca, 0xeb, 0xb3, 0xcd, 0x32, 0xc9, 0xd3, 0xe1, 0xf5, 0xd3, 0x8f, 0x8b,
    0xdd, 0x89, 0x58, 0x7d,
};
static const unsigned char kat24_addin1[] = {
    0x31, 0x38, 0x09, 0x26, 0xb8, 0x60, 0x80, 0x06, 0xb7, 0xc7, 0x93, 0x61,
    0xf0, 0xf9, 0x1d, 0x5e,
};
static const unsigned char kat24_retbits[] = {
    0x3a, 0xd7, 0xb9, 0x32, 0x59, 0xc6, 0xd0, 0xfd, 0xb5, 0x14, 0x27, 0x7a,
    0xe9, 0x95, 0x01, 0x22, 0xfb, 0x97, 0x11, 0x33, 0x74, 0x2d, 0xe8, 0xe5,
    0x17, 0xdd, 0x22, 0x69, 0xdd, 0x5a, 0x3c, 0x25, 0x7e, 0x3d, 0x10, 0x6b,
    0x55, 0x56, 0x60, 0xcf, 0xdd, 0x6a, 0x92, 0xc2, 0xcd, 0xdf, 0x1e, 0x53,
    0xc4, 0x94, 0x5e, 0xd7, 0x72, 0xee, 0x30, 0x38, 0x04, 0xd0, 0x56, 0xf2,
    0x42, 0xbf, 0x53, 0x99,
};
static const struct drbg_kat_no_reseed kat24_t = {
    8, kat24_entropyin, kat24_nonce, kat24_persstr,
    kat24_addin0, kat24_addin1, kat24_retbits
};
static const struct drbg_kat kat24 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat24_t
};

static const unsigned char kat25_entropyin[] = {
    0x3b, 0x9e, 0xe2, 0x0c, 0x82, 0x42, 0xb3, 0xf3, 0x58, 0xdb, 0xe8, 0xea,
    0x58, 0x9e, 0x6e, 0x7c,
};
static const unsigned char kat25_nonce[] = {
    0xb1, 0xdc, 0x3a, 0x47, 0x03, 0x68, 0x52, 0x15,
};
static const unsigned char kat25_persstr[] = {0};
static const unsigned char kat25_addin0[] = {
    0x47, 0x33, 0x34, 0x47, 0x86, 0x73, 0xe7, 0xb4, 0xa4, 0x3e, 0x9a, 0xe6,
    0xa8, 0xac, 0x98, 0xbf,
};
static const unsigned char kat25_addin1[] = {
    0x06, 0x1f, 0xe4, 0x45, 0x5a, 0x17, 0x08, 0x74, 0x98, 0xfe, 0x20, 0x9c,
    0x2c, 0xba, 0x7a, 0x91,
};
static const unsigned char kat25_retbits[] = {
    0x8e, 0x50, 0x62, 0xb1, 0xfb, 0x29, 0x05, 0xc1, 0x57, 0xc5, 0x26, 0xfa,
    0x5f, 0x67, 0xcc, 0x99, 0xc6, 0xd7, 0x08, 0x0a, 0x3d, 0x31, 0x42, 0xfa,
    0x37, 0xc2, 0x09, 0xcf, 0x53, 0x4a, 0xf1, 0x6a, 0x9a, 0x2f, 0x28, 0x61,
    0xb8, 0x94, 0xea, 0x84, 0xa3, 0x73, 0xd4, 0xa8, 0x04, 0xc3, 0xdc, 0xd5,
    0x1f, 0xcc, 0x09, 0xdd, 0x0f, 0x5b, 0x39, 0x45, 0x69, 0x7a, 0xac, 0xd0,
    0xb8, 0xf3, 0xea, 0xfd,
};
static const struct drbg_kat_no_reseed kat25_t = {
    9, kat25_entropyin, kat25_nonce, kat25_persstr,
    kat25_addin0, kat25_addin1, kat25_retbits
};
static const struct drbg_kat kat25 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat25_t
};

static const unsigned char kat26_entropyin[] = {
    0x8f, 0xf7, 0x57, 0xb2, 0x11, 0x69, 0xb3, 0xfd, 0x6e, 0x88, 0x00, 0xaa,
    0xc0, 0xe3, 0xc4, 0x1f,
};
static const unsigned char kat26_nonce[] = {
    0x49, 0x4d, 0x42, 0x51, 0xe8, 0x88, 0x15, 0xb4,
};
static const unsigned char kat26_persstr[] = {0};
static const unsigned char kat26_addin0[] = {
    0xe0, 0x19, 0x9f, 0xb9, 0xf0, 0xf0, 0xa1, 0xa8, 0xdd, 0x9d, 0x19, 0xd1,
    0x43, 0x68, 0xfa, 0x1e,
};
static const unsigned char kat26_addin1[] = {
    0x21, 0x91, 0xab, 0xe7, 0x24, 0x41, 0x28, 0x56, 0x36, 0xc1, 0x1b, 0xde,
    0xe9, 0x83, 0xed, 0x8c,
};
static const unsigned char kat26_retbits[] = {
    0x06, 0x75, 0xb2, 0x25, 0x4b, 0x71, 0x31, 0x8e, 0xb3, 0xa3, 0x60, 0x4d,
    0x7b, 0x6d, 0x77, 0x80, 0x6c, 0xd7, 0xdf, 0x32, 0x5c, 0x7c, 0xe8, 0x9e,
    0x16, 0xd3, 0xbd, 0x56, 0x16, 0xb0, 0xfa, 0xec, 0x33, 0x0c, 0xfe, 0x6b,
    0x84, 0xb6, 0x87, 0x61, 0xad, 0xf7, 0x0c, 0xc1, 0x4e, 0x46, 0x25, 0x3e,
    0x11, 0x55, 0x72, 0xc1, 0xa4, 0xac, 0xbb, 0xae, 0xee, 0x1a, 0xd6, 0x81,
    0xb5, 0x38, 0xd6, 0xc9,
};
static const struct drbg_kat_no_reseed kat26_t = {
    10, kat26_entropyin, kat26_nonce, kat26_persstr,
    kat26_addin0, kat26_addin1, kat26_retbits
};
static const struct drbg_kat kat26 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat26_t
};

static const unsigned char kat27_entropyin[] = {
    0x46, 0x0c, 0x0a, 0x1e, 0x92, 0x9b, 0x75, 0x18, 0xdb, 0xcf, 0x44, 0xe4,
    0xe7, 0xee, 0x4d, 0x83,
};
static const unsigned char kat27_nonce[] = {
    0x88, 0x65, 0xd5, 0x01, 0x53, 0x48, 0x25, 0x9e,
};
static const unsigned char kat27_persstr[] = {0};
static const unsigned char kat27_addin0[] = {
    0xf9, 0x32, 0x78, 0x91, 0x40, 0xa0, 0xa7, 0x6f, 0x07, 0xb2, 0x5b, 0x5f,
    0x8e, 0x1d, 0x00, 0xdc,
};
static const unsigned char kat27_addin1[] = {
    0xe6, 0x44, 0x6a, 0xe2, 0xc6, 0x62, 0x66, 0x73, 0xe1, 0x8e, 0x37, 0x3d,
    0xeb, 0x12, 0xbd, 0x26,
};
static const unsigned char kat27_retbits[] = {
    0xc5, 0xde, 0x2e, 0x1a, 0x2a, 0xa7, 0x0a, 0x28, 0xac, 0xe6, 0x8e, 0x62,
    0x0c, 0x94, 0xb8, 0x88, 0xa0, 0x7b, 0x07, 0x80, 0xb3, 0x14, 0x38, 0xc5,
    0xe1, 0x1c, 0x5d, 0x76, 0xdf, 0xd6, 0x15, 0xf4, 0x1d, 0x4b, 0xa3, 0x24,
    0xd2, 0x2c, 0x12, 0xfe, 0xe3, 0x8f, 0x58, 0x93, 0x5d, 0xad, 0x35, 0xe0,
    0x0f, 0xe6, 0xa6, 0xe6, 0x11, 0x86, 0xc7, 0x88, 0x1c, 0x30, 0x6e, 0x96,
    0x31, 0xcc, 0x15, 0xec,
};
static const struct drbg_kat_no_reseed kat27_t = {
    11, kat27_entropyin, kat27_nonce, kat27_persstr,
    kat27_addin0, kat27_addin1, kat27_retbits
};
static const struct drbg_kat kat27 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat27_t
};

static const unsigned char kat28_entropyin[] = {
    0xb2, 0xda, 0x44, 0xbc, 0x3b, 0xe2, 0xd1, 0x7a, 0x9a, 0xc2, 0x08, 0x80,
    0x85, 0xe9, 0x26, 0xf8,
};
static const unsigned char kat28_nonce[] = {
    0x04, 0xbe, 0x54, 0x49, 0x35, 0x26, 0xc5, 0x78,
};
static const unsigned char kat28_persstr[] = {0};
static const unsigned char kat28_addin0[] = {
    0x67, 0xbe, 0x89, 0x19, 0x59, 0x65, 0xcd, 0x7d, 0xc9, 0x84, 0x27, 0x1b,
    0x49, 0x7d, 0x61, 0x90,
};
static const unsigned char kat28_addin1[] = {
    0xc9, 0x95, 0x6b, 0x00, 0x4b, 0x0c, 0x26, 0xb3, 0xf6, 0x57, 0xc6, 0xc1,
    0x55, 0xd3, 0x60, 0xcc,
};
static const unsigned char kat28_retbits[] = {
    0xc7, 0x74, 0x56, 0x62, 0x19, 0x48, 0xf6, 0x08, 0x38, 0xe3, 0x77, 0x9a,
    0x14, 0x60, 0x63, 0x22, 0x52, 0x96, 0xda, 0x38, 0x18, 0x0e, 0x2c, 0x4c,
    0x58, 0x35, 0xea, 0x27, 0xb2, 0x84, 0x24, 0xa8, 0x83, 0x1c, 0xee, 0x8c,
    0x8d, 0xf7, 0x39, 0xa2, 0xb1, 0x73, 0xb3, 0x7b, 0xe2, 0xc0, 0x2b, 0xff,
    0x38, 0xe0, 0xb3, 0x67, 0xd4, 0x82, 0xfb, 0x76, 0x54, 0x5f, 0xb3, 0xc9,
    0x8d, 0x82, 0xb6, 0xe7,
};
static const struct drbg_kat_no_reseed kat28_t = {
    12, kat28_entropyin, kat28_nonce, kat28_persstr,
    kat28_addin0, kat28_addin1, kat28_retbits
};
static const struct drbg_kat kat28 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat28_t
};

static const unsigned char kat29_entropyin[] = {
    0xf1, 0x77, 0xbc, 0x8a, 0xf7, 0x5c, 0x49, 0xe1, 0xd3, 0x79, 0xb5, 0x72,
    0x26, 0x92, 0xbb, 0x3f,
};
static const unsigned char kat29_nonce[] = {
    0xd8, 0xb5, 0xfb, 0xdc, 0x5d, 0xa7, 0xea, 0x7b,
};
static const unsigned char kat29_persstr[] = {0};
static const unsigned char kat29_addin0[] = {
    0x37, 0x41, 0x94, 0x38, 0x35, 0x54, 0xa4, 0x21, 0x41, 0x0a, 0xf5, 0xde,
    0x2f, 0xc2, 0xec, 0x82,
};
static const unsigned char kat29_addin1[] = {
    0x00, 0x0b, 0xe9, 0xac, 0x8e, 0x4b, 0x0c, 0xd1, 0x24, 0x5d, 0x7b, 0x7d,
    0xbf, 0x8d, 0x87, 0x4f,
};
static const unsigned char kat29_retbits[] = {
    0xb4, 0xa0, 0x0c, 0xd6, 0xe5, 0xf4, 0x7e, 0xa8, 0x7b, 0x12, 0x07, 0xc3,
    0xa2, 0x18, 0xd8, 0x98, 0x8d, 0x4c, 0xb7, 0xed, 0x52, 0xb1, 0x4b, 0x79,
    0xbb, 0xa5, 0x6a, 0xa7, 0xe5, 0x7e, 0x63, 0x64, 0x05, 0x70, 0x25, 0xab,
    0x8f, 0x60, 0xd9, 0x7b, 0x84, 0x71, 0x7b, 0xbf, 0x25, 0x58, 0xdd, 0x43,
    0x1f, 0x4b, 0xc3, 0x8c, 0xf1, 0xf6, 0x1e, 0xc0, 0xe7, 0xc2, 0xb4, 0xae,
    0xb5, 0x4b, 0x2f, 0xaa,
};
static const struct drbg_kat_no_reseed kat29_t = {
    13, kat29_entropyin, kat29_nonce, kat29_persstr,
    kat29_addin0, kat29_addin1, kat29_retbits
};
static const struct drbg_kat kat29 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat29_t
};

static const unsigned char kat30_entropyin[] = {
    0x52, 0x6b, 0xc8, 0xb7, 0x97, 0xad, 0x67, 0xeb, 0x9b, 0x11, 0xbb, 0x5d,
    0x17, 0x32, 0x00, 0x87,
};
static const unsigned char kat30_nonce[] = {
    0x64, 0x7b, 0xd2, 0x32, 0xad, 0x4d, 0x13, 0x00,
};
static const unsigned char kat30_persstr[] = {0};
static const unsigned char kat30_addin0[] = {
    0x58, 0xc8, 0xbf, 0x95, 0x0f, 0xab, 0x02, 0x71, 0x19, 0xee, 0x4a, 0x59,
    0x27, 0x27, 0xaa, 0x02,
};
static const unsigned char kat30_addin1[] = {
    0x25, 0x39, 0xe1, 0x23, 0x92, 0x6b, 0x74, 0xab, 0x23, 0x41, 0x4d, 0x63,
    0x50, 0xb8, 0x92, 0x87,
};
static const unsigned char kat30_retbits[] = {
    0x35, 0x9c, 0x56, 0x08, 0x34, 0xbb, 0xeb, 0xdb, 0x41, 0x34, 0xcc, 0x3e,
    0x66, 0xbc, 0x06, 0x95, 0xce, 0xfd, 0xb4, 0xfe, 0x56, 0x63, 0x75, 0x2b,
    0x6e, 0xba, 0x97, 0x4c, 0x4b, 0x85, 0x92, 0x4a, 0x01, 0xdc, 0xd4, 0xea,
    0x3e, 0xd7, 0x7f, 0x2a, 0xf3, 0x95, 0x7b, 0x87, 0x51, 0xf7, 0x37, 0x15,
    0x74, 0xbb, 0x8b, 0x09, 0x0b, 0xc0, 0xd5, 0x12, 0xc1, 0x57, 0xc6, 0x55,
    0x9c, 0x16, 0xd0, 0xfd,
};
static const struct drbg_kat_no_reseed kat30_t = {
    14, kat30_entropyin, kat30_nonce, kat30_persstr,
    kat30_addin0, kat30_addin1, kat30_retbits
};
static const struct drbg_kat kat30 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat30_t
};

static const unsigned char kat31_entropyin[] = {
    0xe1, 0x0b, 0xc2, 0x8a, 0x0b, 0xfd, 0xdf, 0xe9, 0x3e, 0x7f, 0x51, 0x86,
    0xe0, 0xca, 0x0b, 0x3b,
};
static const unsigned char kat31_nonce[] = {
    0x9f, 0xf4, 0x77, 0xc1, 0x86, 0x73, 0x84, 0x0d,
};
static const unsigned char kat31_persstr[] = {
    0xc9, 0x80, 0xde, 0xdf, 0x98, 0x82, 0xed, 0x44, 0x64, 0xa6, 0x74, 0x96,
    0x78, 0x68, 0xf1, 0x43,
};
static const unsigned char kat31_addin0[] = {0};
static const unsigned char kat31_addin1[] = {0};
static const unsigned char kat31_retbits[] = {
    0x35, 0xb0, 0x0d, 0xf6, 0x26, 0x9b, 0x66, 0x41, 0xfd, 0x4c, 0xcb, 0x35,
    0x4d, 0x56, 0xd8, 0x51, 0xde, 0x7a, 0x77, 0x52, 0x7e, 0x03, 0x4d, 0x60,
    0xc9, 0xe1, 0xa9, 0xe1, 0x52, 0x5a, 0x30, 0xed, 0x36, 0x1f, 0xde, 0xd8,
    0x9d, 0x3d, 0xcc, 0xb9, 0x78, 0xd4, 0xe7, 0xa9, 0xe1, 0x00, 0xeb, 0xf6,
    0x30, 0x62, 0x73, 0x5b, 0x52, 0x83, 0x1c, 0x6f, 0x0a, 0x1d, 0x3e, 0x1b,
    0xdc, 0x5e, 0xbc, 0x72,
};
static const struct drbg_kat_no_reseed kat31_t = {
    0, kat31_entropyin, kat31_nonce, kat31_persstr,
    kat31_addin0, kat31_addin1, kat31_retbits
};
static const struct drbg_kat kat31 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat31_t
};

static const unsigned char kat32_entropyin[] = {
    0xca, 0x4b, 0x1e, 0xfa, 0x75, 0xbd, 0x69, 0x36, 0x38, 0x73, 0xb8, 0xf9,
    0xdb, 0x4d, 0x35, 0x0e,
};
static const unsigned char kat32_nonce[] = {
    0x47, 0xbf, 0x6c, 0x37, 0x72, 0xfd, 0xf7, 0xa9,
};
static const unsigned char kat32_persstr[] = {
    0xeb, 0xaa, 0x60, 0x2c, 0x4d, 0xbe, 0x33, 0xff, 0x1b, 0xef, 0xbf, 0x0a,
    0x0b, 0xc6, 0x97, 0x54,
};
static const unsigned char kat32_addin0[] = {0};
static const unsigned char kat32_addin1[] = {0};
static const unsigned char kat32_retbits[] = {
    0x59, 0xc3, 0x19, 0x79, 0x1b, 0xb1, 0xf3, 0x0e, 0xe9, 0x34, 0xae, 0x6e,
    0x8b, 0x1f, 0xad, 0x1f, 0x74, 0xca, 0x25, 0x45, 0x68, 0xb8, 0x7f, 0x75,
    0x12, 0xf8, 0xf2, 0xab, 0x4c, 0x23, 0x01, 0x03, 0x05, 0xe1, 0x70, 0xee,
    0x75, 0xd8, 0xcb, 0xeb, 0x23, 0x4c, 0x7a, 0x23, 0x6e, 0x12, 0x27, 0xdb,
    0x6f, 0x7a, 0xac, 0x3c, 0x44, 0xb7, 0x87, 0x4b, 0x65, 0x56, 0x74, 0x45,
    0x34, 0x30, 0x0c, 0x3d,
};
static const struct drbg_kat_no_reseed kat32_t = {
    1, kat32_entropyin, kat32_nonce, kat32_persstr,
    kat32_addin0, kat32_addin1, kat32_retbits
};
static const struct drbg_kat kat32 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat32_t
};

static const unsigned char kat33_entropyin[] = {
    0x35, 0x19, 0x49, 0x15, 0x74, 0x71, 0x9d, 0xdb, 0x1f, 0x51, 0xb5, 0xf5,
    0x9e, 0x21, 0xad, 0x3d,
};
static const unsigned char kat33_nonce[] = {
    0xef, 0x1d, 0xbe, 0xae, 0x79, 0xf0, 0x89, 0x8b,
};
static const unsigned char kat33_persstr[] = {
    0x7f, 0xdc, 0x21, 0xe3, 0x53, 0x24, 0x9e, 0x93, 0xdf, 0x98, 0xf2, 0x91,
    0x02, 0xb9, 0xac, 0xec,
};
static const unsigned char kat33_addin0[] = {0};
static const unsigned char kat33_addin1[] = {0};
static const unsigned char kat33_retbits[] = {
    0x19, 0x2c, 0x71, 0x85, 0x74, 0x47, 0xaf, 0x82, 0x01, 0x37, 0x06, 0xb8,
    0xeb, 0x00, 0x8f, 0x8e, 0x2c, 0x2e, 0xb3, 0x99, 0x4a, 0x23, 0x8c, 0xf7,
    0x39, 0xe1, 0x74, 0x60, 0xfe, 0x84, 0xef, 0x71, 0xca, 0x43, 0x7e, 0x8a,
    0xcb, 0xb8, 0xd9, 0xd2, 0x88, 0x07, 0xc4, 0x7a, 0x5f, 0x03, 0x44, 0x42,
    0xaa, 0xf3, 0xc2, 0xf3, 0xe7, 0xde, 0xbe, 0x53, 0x1b, 0xc3, 0x59, 0x60,
    0x56, 0x68, 0x58, 0x48,
};
static const struct drbg_kat_no_reseed kat33_t = {
    2, kat33_entropyin, kat33_nonce, kat33_persstr,
    kat33_addin0, kat33_addin1, kat33_retbits
};
static const struct drbg_kat kat33 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat33_t
};

static const unsigned char kat34_entropyin[] = {
    0x68, 0xed, 0xc1, 0xc7, 0xf0, 0x4f, 0xa2, 0x2f, 0x75, 0x1b, 0x91, 0x8b,
    0x01, 0x6b, 0xbf, 0xb5,
};
static const unsigned char kat34_nonce[] = {
    0xa9, 0xb6, 0x7a, 0x39, 0x58, 0xf2, 0x02, 0xf0,
};
static const unsigned char kat34_persstr[] = {
    0x9d, 0x6f, 0xbb, 0x02, 0x0e, 0x3d, 0x99, 0xd1, 0x95, 0xff, 0x46, 0x9f,
    0x0a, 0xc0, 0xb8, 0xca,
};
static const unsigned char kat34_addin0[] = {0};
static const unsigned char kat34_addin1[] = {0};
static const unsigned char kat34_retbits[] = {
    0xd9, 0x02, 0x89, 0xb1, 0xb6, 0x76, 0x3a, 0x76, 0x90, 0xfc, 0xb5, 0xd6,
    0x7c, 0x81, 0xc2, 0x03, 0xa1, 0xcf, 0xb0, 0x88, 0x71, 0xb9, 0x4a, 0xf6,
    0x51, 0xc4, 0xcd, 0x7e, 0xc2, 0xfa, 0x4b, 0x9d, 0x47, 0xc8, 0xe5, 0xfc,
    0xc9, 0x9a, 0xff, 0xc2, 0x21, 0x43, 0xb6, 0x12, 0x70, 0x18, 0xe9, 0xe2,
    0xc0, 0x91, 0x4f, 0x9d, 0x47, 0xa5, 0xa3, 0x0b, 0x26, 0xe7, 0x4d, 0xdc,
    0x31, 0xa4, 0x48, 0x42,
};
static const struct drbg_kat_no_reseed kat34_t = {
    3, kat34_entropyin, kat34_nonce, kat34_persstr,
    kat34_addin0, kat34_addin1, kat34_retbits
};
static const struct drbg_kat kat34 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat34_t
};

static const unsigned char kat35_entropyin[] = {
    0x8e, 0x6b, 0xaf, 0x81, 0xa3, 0xf9, 0xe7, 0x32, 0xef, 0xb9, 0x65, 0xaf,
    0xa5, 0x9e, 0x71, 0xee,
};
static const unsigned char kat35_nonce[] = {
    0x9a, 0x40, 0x36, 0xfe, 0x7f, 0xf2, 0x25, 0xb8,
};
static const unsigned char kat35_persstr[] = {
    0xc5, 0x7e, 0x42, 0xad, 0x7a, 0xef, 0x53, 0x77, 0xfc, 0x4c, 0xd6, 0x20,
    0xd6, 0x31, 0xb0, 0x61,
};
static const unsigned char kat35_addin0[] = {0};
static const unsigned char kat35_addin1[] = {0};
static const unsigned char kat35_retbits[] = {
    0x71, 0x39, 0xa2, 0xd6, 0x74, 0xee, 0xfe, 0x54, 0x08, 0x7a, 0x2d, 0xd5,
    0xf4, 0x97, 0xf5, 0xa1, 0xcb, 0x44, 0x44, 0x72, 0xe3, 0x64, 0x04, 0x5c,
    0xb9, 0x20, 0x2b, 0xce, 0x24, 0x58, 0x1b, 0x4b, 0x1b, 0x80, 0x3f, 0xa5,
    0xf4, 0xa9, 0xac, 0x6d, 0x66, 0x37, 0xbd, 0xd9, 0x06, 0xaf, 0x3d, 0x8f,
    0x49, 0xb5, 0xb8, 0x0f, 0x44, 0xcd, 0x16, 0xee, 0x5b, 0x1a, 0xe3, 0xbf,
    0x51, 0xb5, 0x98, 0xba,
};
static const struct drbg_kat_no_reseed kat35_t = {
    4, kat35_entropyin, kat35_nonce, kat35_persstr,
    kat35_addin0, kat35_addin1, kat35_retbits
};
static const struct drbg_kat kat35 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat35_t
};

static const unsigned char kat36_entropyin[] = {
    0xf8, 0x99, 0x2f, 0x34, 0x0f, 0x90, 0xc8, 0xd8, 0x16, 0xb4, 0xbd, 0x8b,
    0x43, 0x5d, 0xa0, 0x35,
};
static const unsigned char kat36_nonce[] = {
    0xfc, 0xf9, 0x44, 0x00, 0x6f, 0x29, 0x27, 0x98,
};
static const unsigned char kat36_persstr[] = {
    0x56, 0x32, 0x23, 0x5e, 0x85, 0x2c, 0x07, 0x40, 0x09, 0xb4, 0x97, 0xed,
    0xa4, 0x38, 0xa1, 0x6f,
};
static const unsigned char kat36_addin0[] = {0};
static const unsigned char kat36_addin1[] = {0};
static const unsigned char kat36_retbits[] = {
    0x9e, 0xab, 0x55, 0xfa, 0x4b, 0x2c, 0x1c, 0xfa, 0xc6, 0x3d, 0x4d, 0xe4,
    0x7a, 0xa8, 0xf5, 0x9f, 0x78, 0x5e, 0xe4, 0x90, 0xda, 0x44, 0xce, 0xe3,
    0x0f, 0x91, 0x61, 0xe9, 0x3a, 0xfc, 0x46, 0x1d, 0xad, 0x0e, 0x7f, 0x4a,
    0x06, 0x4e, 0x92, 0xef, 0xd6, 0x00, 0x89, 0x16, 0xc2, 0x2f, 0x83, 0x64,
    0xc1, 0x90, 0x46, 0x7a, 0xde, 0x12, 0xab, 0x4e, 0x18, 0xb1, 0xf3, 0x9c,
    0xe1, 0x8b, 0x35, 0x8a,
};
static const struct drbg_kat_no_reseed kat36_t = {
    5, kat36_entropyin, kat36_nonce, kat36_persstr,
    kat36_addin0, kat36_addin1, kat36_retbits
};
static const struct drbg_kat kat36 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat36_t
};

static const unsigned char kat37_entropyin[] = {
    0x07, 0x43, 0x8f, 0x83, 0x53, 0xb7, 0xb0, 0x67, 0x42, 0x18, 0x31, 0x9e,
    0x29, 0x54, 0x0b, 0xca,
};
static const unsigned char kat37_nonce[] = {
    0x41, 0x14, 0x77, 0x63, 0x6a, 0x50, 0x0c, 0x62,
};
static const unsigned char kat37_persstr[] = {
    0xcf, 0x36, 0xf2, 0xaa, 0x72, 0xf2, 0xf3, 0x5e, 0x33, 0x5e, 0x65, 0xcf,
    0xd1, 0x7e, 0x2d, 0x3d,
};
static const unsigned char kat37_addin0[] = {0};
static const unsigned char kat37_addin1[] = {0};
static const unsigned char kat37_retbits[] = {
    0x6c, 0xfa, 0xcf, 0xe4, 0x7a, 0xcf, 0x8f, 0x61, 0xd8, 0xaf, 0x3e, 0xd4,
    0xfb, 0xac, 0xef, 0x5c, 0xd7, 0x41, 0xac, 0x2f, 0x16, 0x5e, 0x15, 0xbd,
    0xaa, 0xb1, 0xd0, 0x30, 0xc6, 0x85, 0x67, 0x83, 0x7e, 0xf3, 0xda, 0x1e,
    0xc8, 0x3d, 0xa0, 0x42, 0x3c, 0xa3, 0x42, 0x4c, 0xf2, 0x93, 0xf6, 0x1f,
    0x71, 0x75, 0xfb, 0xa8, 0xe9, 0x1e, 0x5f, 0xdc, 0x8b, 0x39, 0x55, 0x77,
    0x73, 0x61, 0xd6, 0x90,
};
static const struct drbg_kat_no_reseed kat37_t = {
    6, kat37_entropyin, kat37_nonce, kat37_persstr,
    kat37_addin0, kat37_addin1, kat37_retbits
};
static const struct drbg_kat kat37 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat37_t
};

static const unsigned char kat38_entropyin[] = {
    0x8e, 0x93, 0x73, 0x5c, 0x34, 0xae, 0x0c, 0x5e, 0xa5, 0x21, 0x3d, 0x83,
    0xa8, 0xda, 0x80, 0x45,
};
static const unsigned char kat38_nonce[] = {
    0x2a, 0x0b, 0x27, 0x9a, 0x9d, 0x67, 0x78, 0x38,
};
static const unsigned char kat38_persstr[] = {
    0x14, 0xf4, 0x52, 0x66, 0x52, 0x4a, 0x81, 0x52, 0x85, 0x7a, 0x83, 0x31,
    0x6a, 0x17, 0x75, 0x85,
};
static const unsigned char kat38_addin0[] = {0};
static const unsigned char kat38_addin1[] = {0};
static const unsigned char kat38_retbits[] = {
    0xe0, 0x0c, 0xb8, 0x63, 0x6f, 0x96, 0x02, 0xc3, 0x30, 0x21, 0xa8, 0x2a,
    0x7f, 0x9d, 0xc9, 0xc7, 0x4e, 0xd3, 0x43, 0x5b, 0x34, 0x89, 0x7d, 0x6e,
    0xfd, 0x8c, 0x05, 0xf1, 0x77, 0xe8, 0x55, 0x6a, 0x44, 0x3b, 0xd1, 0x8c,
    0x20, 0xa5, 0x57, 0xb9, 0xbd, 0xc4, 0x17, 0xe8, 0x99, 0x85, 0x06, 0x48,
    0x50, 0x91, 0xd6, 0xbc, 0x91, 0x8e, 0x9c, 0xbf, 0xfb, 0xb8, 0x5c, 0x62,
    0x0f, 0x74, 0xc6, 0xec,
};
static const struct drbg_kat_no_reseed kat38_t = {
    7, kat38_entropyin, kat38_nonce, kat38_persstr,
    kat38_addin0, kat38_addin1, kat38_retbits
};
static const struct drbg_kat kat38 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat38_t
};

static const unsigned char kat39_entropyin[] = {
    0xec, 0x6b, 0x87, 0xe9, 0xe0, 0x3f, 0x44, 0xf9, 0x19, 0xe4, 0x0e, 0x1d,
    0xac, 0x02, 0xf8, 0x73,
};
static const unsigned char kat39_nonce[] = {
    0x5e, 0x9a, 0xc5, 0x09, 0x0a, 0x5e, 0x28, 0x97,
};
static const unsigned char kat39_persstr[] = {
    0x0d, 0x9b, 0xa4, 0x62, 0xbf, 0x16, 0x6f, 0x3b, 0x36, 0x6a, 0x2a, 0xdf,
    0x55, 0xd2, 0x76, 0x45,
};
static const unsigned char kat39_addin0[] = {0};
static const unsigned char kat39_addin1[] = {0};
static const unsigned char kat39_retbits[] = {
    0xec, 0x37, 0xac, 0xbd, 0xdf, 0xb0, 0x41, 0xe4, 0x24, 0x9c, 0x1c, 0xb9,
    0x83, 0x3a, 0x31, 0xfc, 0x6f, 0xbd, 0xaa, 0x66, 0x58, 0xe4, 0xb7, 0x48,
    0xc0, 0xeb, 0xf6, 0xab, 0x54, 0xa3, 0xb9, 0xc0, 0xf6, 0x2d, 0x7c, 0x89,
    0xdb, 0xb2, 0x1d, 0x1d, 0xe1, 0x33, 0x15, 0xcb, 0xae, 0x2f, 0xf4, 0xf5,
    0x46, 0x24, 0x91, 0x32, 0x1f, 0xbe, 0x04, 0xb4, 0x14, 0xfb, 0xa2, 0x80,
    0x7e, 0xd9, 0x14, 0xee,
};
static const struct drbg_kat_no_reseed kat39_t = {
    8, kat39_entropyin, kat39_nonce, kat39_persstr,
    kat39_addin0, kat39_addin1, kat39_retbits
};
static const struct drbg_kat kat39 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat39_t
};

static const unsigned char kat40_entropyin[] = {
    0x9d, 0x28, 0x2f, 0x5a, 0x0c, 0x15, 0xa6, 0x53, 0x04, 0x73, 0x10, 0xb5,
    0x62, 0x7d, 0x8f, 0xa2,
};
static const unsigned char kat40_nonce[] = {
    0x55, 0x22, 0xc4, 0x2f, 0x4b, 0xb0, 0x76, 0xdc,
};
static const unsigned char kat40_persstr[] = {
    0x84, 0x6b, 0x2e, 0x17, 0xb0, 0x62, 0xc4, 0xc7, 0xc5, 0x61, 0x94, 0x3b,
    0x5d, 0x97, 0x56, 0x5d,
};
static const unsigned char kat40_addin0[] = {0};
static const unsigned char kat40_addin1[] = {0};
static const unsigned char kat40_retbits[] = {
    0x45, 0xf6, 0x0e, 0x1b, 0xa9, 0x81, 0xaf, 0x7f, 0x52, 0xef, 0x49, 0x39,
    0xc0, 0xbb, 0x0b, 0xdb, 0x6f, 0xe4, 0x6f, 0x37, 0x2c, 0xc6, 0x48, 0x06,
    0x4b, 0xbd, 0x3a, 0x0a, 0x2b, 0x85, 0xc0, 0x23, 0x91, 0xb2, 0x97, 0x92,
    0xcc, 0x88, 0x05, 0xbb, 0x5d, 0x45, 0x3e, 0xe2, 0x90, 0xcd, 0x1b, 0x9c,
    0x9f, 0x8f, 0x20, 0xb0, 0x91, 0x16, 0xaf, 0x1f, 0xd5, 0xec, 0xea, 0xe8,
    0x0d, 0x03, 0x58, 0xd8,
};
static const struct drbg_kat_no_reseed kat40_t = {
    9, kat40_entropyin, kat40_nonce, kat40_persstr,
    kat40_addin0, kat40_addin1, kat40_retbits
};
static const struct drbg_kat kat40 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat40_t
};

static const unsigned char kat41_entropyin[] = {
    0x83, 0x82, 0x7c, 0x08, 0x52, 0x4b, 0x54, 0xe8, 0x0f, 0x60, 0xe1, 0x86,
    0xcf, 0xce, 0x36, 0xdb,
};
static const unsigned char kat41_nonce[] = {
    0x17, 0x7e, 0x5c, 0x00, 0xf7, 0x85, 0xa6, 0xf9,
};
static const unsigned char kat41_persstr[] = {
    0x71, 0x75, 0xa2, 0xc2, 0x2e, 0xab, 0x4d, 0x0b, 0x1a, 0x65, 0xfd, 0xad,
    0xed, 0x35, 0x6a, 0xeb,
};
static const unsigned char kat41_addin0[] = {0};
static const unsigned char kat41_addin1[] = {0};
static const unsigned char kat41_retbits[] = {
    0x84, 0x24, 0xe7, 0x6c, 0x73, 0x6c, 0x03, 0x09, 0x42, 0x3c, 0xf4, 0x8d,
    0xc1, 0x05, 0xfe, 0xef, 0x22, 0x8d, 0xd4, 0x7b, 0xfd, 0x8d, 0x33, 0x80,
    0x19, 0x50, 0xd2, 0x10, 0x2d, 0x5d, 0xba, 0xfb, 0x51, 0xe1, 0xa8, 0x5b,
    0x4a, 0x97, 0x1e, 0x4f, 0x0b, 0x9d, 0x52, 0x65, 0x69, 0x73, 0xdb, 0x45,
    0x5b, 0x28, 0x6f, 0x58, 0x8c, 0xa6, 0x1d, 0x15, 0x99, 0xc4, 0xec, 0x60,
    0xdd, 0x80, 0xbe, 0x98,
};
static const struct drbg_kat_no_reseed kat41_t = {
    10, kat41_entropyin, kat41_nonce, kat41_persstr,
    kat41_addin0, kat41_addin1, kat41_retbits
};
static const struct drbg_kat kat41 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat41_t
};

static const unsigned char kat42_entropyin[] = {
    0x9f, 0x32, 0x13, 0xa2, 0xd6, 0xfe, 0x5f, 0x3d, 0xfa, 0xf0, 0xe4, 0xed,
    0x29, 0x2e, 0x0f, 0x2a,
};
static const unsigned char kat42_nonce[] = {
    0xf9, 0xc0, 0x55, 0x3b, 0x18, 0xb2, 0x1c, 0xc3,
};
static const unsigned char kat42_persstr[] = {
    0x21, 0x73, 0xd3, 0x9b, 0x37, 0x55, 0x04, 0x19, 0x11, 0x98, 0x9f, 0x89,
    0xf1, 0xe9, 0x5d, 0x3e,
};
static const unsigned char kat42_addin0[] = {0};
static const unsigned char kat42_addin1[] = {0};
static const unsigned char kat42_retbits[] = {
    0x10, 0x45, 0xf7, 0xb5, 0x36, 0xda, 0xb4, 0xde, 0x01, 0x54, 0x48, 0xac,
    0x47, 0x66, 0x18, 0x8a, 0x36, 0x7b, 0xb6, 0x93, 0x97, 0x91, 0x3d, 0xfd,
    0x3d, 0x5b, 0xbf, 0x83, 0xf1, 0xbb, 0xac, 0x73, 0x23, 0x2d, 0x47, 0x64,
    0x4e, 0x5a, 0x9c, 0x16, 0x99, 0x60, 0x58, 0x1e, 0x4a, 0xd5, 0xed, 0x26,
    0x95, 0x36, 0x0b, 0x94, 0xf4, 0x20, 0x9c, 0x77, 0xd7, 0x69, 0x40, 0x16,
    0xb4, 0xb3, 0x97, 0xc2,
};
static const struct drbg_kat_no_reseed kat42_t = {
    11, kat42_entropyin, kat42_nonce, kat42_persstr,
    kat42_addin0, kat42_addin1, kat42_retbits
};
static const struct drbg_kat kat42 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat42_t
};

static const unsigned char kat43_entropyin[] = {
    0xfb, 0xc8, 0x19, 0x90, 0x74, 0xa7, 0x6f, 0x23, 0x68, 0x86, 0x36, 0xae,
    0x31, 0x14, 0x28, 0x1e,
};
static const unsigned char kat43_nonce[] = {
    0xeb, 0xc2, 0x2e, 0x95, 0xdd, 0xae, 0xb6, 0xfc,
};
static const unsigned char kat43_persstr[] = {
    0x71, 0xfb, 0xd5, 0x9f, 0x88, 0xfc, 0x99, 0xa9, 0xb6, 0x07, 0x00, 0x27,
    0xe7, 0x35, 0x41, 0xc2,
};
static const unsigned char kat43_addin0[] = {0};
static const unsigned char kat43_addin1[] = {0};
static const unsigned char kat43_retbits[] = {
    0x87, 0x25, 0x66, 0xa7, 0x00, 0x47, 0x98, 0xe0, 0x06, 0x97, 0xbd, 0xb3,
    0x3a, 0xf1, 0xee, 0x7a, 0x07, 0x6e, 0x9f, 0xd2, 0x89, 0xaf, 0x83, 0xaf,
    0xcb, 0x10, 0xa0, 0x25, 0x2b, 0xb5, 0xc1, 0x06, 0xc8, 0xdd, 0x20, 0xd7,
    0x7c, 0x85, 0x9f, 0x14, 0x01, 0x11, 0x8c, 0x24, 0xf5, 0xa8, 0x96, 0x8f,
    0x11, 0xfa, 0x9f, 0x77, 0x78, 0xc2, 0x80, 0x31, 0x89, 0xc8, 0xda, 0x87,
    0xe6, 0x49, 0x45, 0xc8,
};
static const struct drbg_kat_no_reseed kat43_t = {
    12, kat43_entropyin, kat43_nonce, kat43_persstr,
    kat43_addin0, kat43_addin1, kat43_retbits
};
static const struct drbg_kat kat43 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat43_t
};

static const unsigned char kat44_entropyin[] = {
    0x43, 0xf4, 0xd0, 0x89, 0x06, 0x0c, 0x90, 0x7f, 0x38, 0x76, 0x05, 0x1f,
    0xd5, 0xe3, 0x6f, 0x74,
};
static const unsigned char kat44_nonce[] = {
    0x45, 0x71, 0xe6, 0x61, 0x5b, 0xa3, 0xbb, 0xce,
};
static const unsigned char kat44_persstr[] = {
    0xfc, 0x90, 0x57, 0x74, 0x9b, 0xa4, 0x37, 0xd8, 0xfb, 0xe9, 0xf8, 0x1d,
    0x29, 0xdf, 0x5e, 0xf1,
};
static const unsigned char kat44_addin0[] = {0};
static const unsigned char kat44_addin1[] = {0};
static const unsigned char kat44_retbits[] = {
    0x90, 0x78, 0xf1, 0xbc, 0x92, 0x91, 0x0e, 0xbc, 0xf8, 0xe3, 0x17, 0x57,
    0xb6, 0x24, 0xbd, 0x23, 0xdb, 0xcd, 0x74, 0xf9, 0xca, 0x70, 0xff, 0x1c,
    0x6f, 0x2b, 0x21, 0xd8, 0x59, 0xa4, 0xe8, 0xa4, 0xe9, 0x96, 0x3e, 0xf1,
    0x32, 0xa0, 0x25, 0xb3, 0xae, 0x28, 0x5b, 0x43, 0xa6, 0x96, 0x98, 0x90,
    0x0d, 0x4f, 0x8a, 0x30, 0xbb, 0x5e, 0x99, 0x29, 0x0e, 0x45, 0x04, 0x16,
    0x19, 0x97, 0xa8, 0x37,
};
static const struct drbg_kat_no_reseed kat44_t = {
    13, kat44_entropyin, kat44_nonce, kat44_persstr,
    kat44_addin0, kat44_addin1, kat44_retbits
};
static const struct drbg_kat kat44 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat44_t
};

static const unsigned char kat45_entropyin[] = {
    0x0e, 0xe3, 0x8e, 0xa2, 0x9d, 0x71, 0x95, 0x00, 0x3e, 0x2b, 0x94, 0x2a,
    0xbf, 0x13, 0x48, 0xfc,
};
static const unsigned char kat45_nonce[] = {
    0x80, 0xa7, 0x5c, 0x77, 0xce, 0xff, 0x54, 0xe4,
};
static const unsigned char kat45_persstr[] = {
    0x83, 0x8e, 0xae, 0xbc, 0x98, 0x2c, 0xaa, 0x15, 0x62, 0xc3, 0x06, 0x4c,
    0xce, 0x42, 0xe5, 0x1e,
};
static const unsigned char kat45_addin0[] = {0};
static const unsigned char kat45_addin1[] = {0};
static const unsigned char kat45_retbits[] = {
    0x66, 0x29, 0x1b, 0x86, 0x85, 0xe9, 0x7c, 0x76, 0xe2, 0x21, 0x6d, 0x70,
    0x8b, 0x40, 0x70, 0x22, 0x68, 0x4b, 0x28, 0x29, 0x1f, 0x3d, 0xc7, 0x1b,
    0x5d, 0x60, 0xde, 0x14, 0x3f, 0x5f, 0xdf, 0x0b, 0xb0, 0x7a, 0xe9, 0xd7,
    0x52, 0x44, 0x56, 0xc4, 0xdf, 0xd0, 0x89, 0xea, 0x88, 0xde, 0x86, 0xdd,
    0xfa, 0x1d, 0x48, 0xca, 0x54, 0x2f, 0x00, 0xfa, 0x58, 0x6d, 0xa7, 0xa6,
    0x02, 0x7d, 0x83, 0x7b,
};
static const struct drbg_kat_no_reseed kat45_t = {
    14, kat45_entropyin, kat45_nonce, kat45_persstr,
    kat45_addin0, kat45_addin1, kat45_retbits
};
static const struct drbg_kat kat45 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat45_t
};

static const unsigned char kat46_entropyin[] = {
    0xca, 0xe4, 0x8d, 0xd8, 0x0d, 0x29, 0x81, 0x03, 0xef, 0x1e, 0xc0, 0xbf,
    0x1b, 0xb9, 0x62, 0x70,
};
static const unsigned char kat46_nonce[] = {
    0xd8, 0x27, 0xf9, 0x16, 0x13, 0xe0, 0xb4, 0x7f,
};
static const unsigned char kat46_persstr[] = {
    0xcc, 0x92, 0x8f, 0x3d, 0x2d, 0xf3, 0x1a, 0x29, 0xf4, 0xe4, 0x44, 0xf3,
    0xdf, 0x08, 0xbe, 0x21,
};
static const unsigned char kat46_addin0[] = {
    0x7e, 0xaa, 0x1b, 0xbe, 0xc7, 0x93, 0x93, 0xa7, 0xf4, 0xa8, 0x22, 0x7b,
    0x69, 0x1e, 0xcb, 0x68,
};
static const unsigned char kat46_addin1[] = {
    0x68, 0x69, 0xc6, 0xc7, 0xb9, 0xe6, 0x65, 0x3b, 0x39, 0x77, 0xf0, 0x78,
    0x9e, 0x94, 0x47, 0x8a,
};
static const unsigned char kat46_retbits[] = {
    0x92, 0x01, 0x32, 0xcd, 0x28, 0x46, 0x95, 0xb8, 0x68, 0xb5, 0xbc, 0x4b,
    0x70, 0x3a, 0xfe, 0xa4, 0xd9, 0x96, 0x62, 0x4a, 0x8f, 0x57, 0xe9, 0xfb,
    0xf5, 0xe7, 0x93, 0xb5, 0x09, 0xcb, 0x15, 0xb4, 0xbe, 0xaf, 0x70, 0x2d,
    0xac, 0x28, 0x71, 0x2d, 0x24, 0x9a, 0xe7, 0x50, 0x90, 0xa9, 0x1f, 0xd3,
    0x57, 0x75, 0x29, 0x4b, 0xf2, 0x4d, 0xde, 0xbf, 0xd2, 0x4e, 0x45, 0xd1,
    0x3f, 0x4a, 0x17, 0x48,
};
static const struct drbg_kat_no_reseed kat46_t = {
    0, kat46_entropyin, kat46_nonce, kat46_persstr,
    kat46_addin0, kat46_addin1, kat46_retbits
};
static const struct drbg_kat kat46 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat46_t
};

static const unsigned char kat47_entropyin[] = {
    0xc0, 0x70, 0x1f, 0x92, 0x50, 0x75, 0x8f, 0xcd, 0xf2, 0xbe, 0x73, 0x98,
    0x80, 0xdb, 0x66, 0xeb,
};
static const unsigned char kat47_nonce[] = {
    0x14, 0x68, 0xb4, 0xa5, 0x87, 0x9c, 0x2d, 0xa6,
};
static const unsigned char kat47_persstr[] = {
    0x80, 0x08, 0xae, 0xe8, 0xe9, 0x69, 0x40, 0xc5, 0x08, 0x73, 0xc7, 0x9f,
    0x8e, 0xcf, 0xe0, 0x02,
};
static const unsigned char kat47_addin0[] = {
    0xf9, 0x01, 0xf8, 0x16, 0x7a, 0x1d, 0xff, 0xde, 0x8e, 0x3c, 0x83, 0xe2,
    0x44, 0x85, 0xe7, 0xfe,
};
static const unsigned char kat47_addin1[] = {
    0x17, 0x1c, 0x09, 0x38, 0xc2, 0x38, 0x9f, 0x97, 0x87, 0x60, 0x55, 0xb4,
    0x82, 0x16, 0x62, 0x7f,
};
static const unsigned char kat47_retbits[] = {
    0x97, 0xc0, 0xc0, 0xe5, 0xa0, 0xcc, 0xf2, 0x4f, 0x33, 0x63, 0x48, 0x8a,
    0xdb, 0x13, 0x0a, 0x35, 0x89, 0xbf, 0x80, 0x65, 0x62, 0xee, 0x13, 0x95,
    0x7c, 0x33, 0xd3, 0x7d, 0xf4, 0x07, 0x77, 0x7a, 0x2b, 0x65, 0x0b, 0x5f,
    0x45, 0x5c, 0x13, 0xf1, 0x90, 0x77, 0x7f, 0xc5, 0x04, 0x3f, 0xcc, 0x1a,
    0x38, 0xf8, 0xcd, 0x1b, 0xbb, 0xd5, 0x57, 0xd1, 0x4a, 0x4c, 0x2e, 0x8a,
    0x2b, 0x49, 0x1e, 0x5c,
};
static const struct drbg_kat_no_reseed kat47_t = {
    1, kat47_entropyin, kat47_nonce, kat47_persstr,
    kat47_addin0, kat47_addin1, kat47_retbits
};
static const struct drbg_kat kat47 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat47_t
};

static const unsigned char kat48_entropyin[] = {
    0x6b, 0x98, 0x53, 0x2d, 0x67, 0x8f, 0xd2, 0x79, 0xcf, 0x73, 0x7c, 0x58,
    0x15, 0x0c, 0xe5, 0x89,
};
static const unsigned char kat48_nonce[] = {
    0xca, 0x55, 0x45, 0x07, 0x3d, 0x54, 0x9d, 0x06,
};
static const unsigned char kat48_persstr[] = {
    0x80, 0x90, 0x8a, 0xc1, 0x34, 0xe0, 0xcd, 0x23, 0xef, 0x7e, 0x31, 0x85,
    0x48, 0xf9, 0x21, 0x91,
};
static const unsigned char kat48_addin0[] = {
    0x53, 0x79, 0x0d, 0xa9, 0x09, 0xbf, 0xde, 0xed, 0xaa, 0xfd, 0x99, 0x61,
    0x9f, 0x3f, 0x46, 0x1b,
};
static const unsigned char kat48_addin1[] = {
    0xc7, 0xa5, 0x87, 0x77, 0xd1, 0xfb, 0xd1, 0x55, 0x3e, 0x95, 0x46, 0xc8,
    0x3e, 0x40, 0x9f, 0xc2,
};
static const unsigned char kat48_retbits[] = {
    0xd9, 0x84, 0xa2, 0x40, 0x58, 0x87, 0x39, 0x02, 0x27, 0x94, 0x09, 0x59,
    0xe4, 0x9f, 0x94, 0x4e, 0x49, 0x04, 0xc4, 0xf0, 0x46, 0x7f, 0xf2, 0x4d,
    0x87, 0x64, 0xdb, 0x91, 0x53, 0x0a, 0x1e, 0x59, 0x65, 0x68, 0x84, 0x72,
    0xfa, 0xc1, 0x31, 0xbe, 0xa0, 0x55, 0xc7, 0x15, 0x12, 0x27, 0xe2, 0xfc,
    0xb0, 0xda, 0xd9, 0x65, 0x5e, 0xc3, 0xac, 0x22, 0xcd, 0x7d, 0xff, 0x72,
    0xf6, 0xc7, 0x4a, 0xa0,
};
static const struct drbg_kat_no_reseed kat48_t = {
    2, kat48_entropyin, kat48_nonce, kat48_persstr,
    kat48_addin0, kat48_addin1, kat48_retbits
};
static const struct drbg_kat kat48 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat48_t
};

static const unsigned char kat49_entropyin[] = {
    0x97, 0x51, 0x6e, 0x48, 0xd9, 0x10, 0x37, 0x2a, 0x3f, 0x1f, 0xb4, 0xaa,
    0x7a, 0xe4, 0x93, 0x0f,
};
static const unsigned char kat49_nonce[] = {
    0xbb, 0xa3, 0x0b, 0x3d, 0xeb, 0x19, 0x33, 0x89,
};
static const unsigned char kat49_persstr[] = {
    0x8c, 0x0e, 0x3e, 0x71, 0xc8, 0x20, 0x1a, 0x77, 0x10, 0x55, 0x19, 0xdf,
    0xfa, 0x89, 0xc1, 0xaa,
};
static const unsigned char kat49_addin0[] = {
    0x0a, 0x18, 0x78, 0x7e, 0x46, 0x50, 0xfa, 0x02, 0x49, 0xf7, 0x79, 0x62,
    0xac, 0x15, 0x8c, 0x63,
};
static const unsigned char kat49_addin1[] = {
    0x2f, 0xb1, 0x92, 0x31, 0x4c, 0x3e, 0x07, 0xf9, 0x63, 0xfb, 0xa3, 0x4d,
    0x8d, 0xfc, 0x8c, 0x9c,
};
static const unsigned char kat49_retbits[] = {
    0xd7, 0xb7, 0x3e, 0x44, 0x9a, 0xed, 0x8f, 0x0a, 0x06, 0x93, 0x84, 0x83,
    0xe8, 0x78, 0x11, 0x04, 0x9b, 0x4a, 0xcf, 0xe5, 0x1c, 0xa5, 0x55, 0xf1,
    0xee, 0x97, 0x86, 0xe3, 0x3f, 0xce, 0x90, 0x69, 0x7d, 0xb6, 0x8d, 0x2d,
    0xc6, 0x36, 0x1c, 0x94, 0x2c, 0xf6, 0x7d, 0x89, 0x26, 0x19, 0x49, 0x7e,
    0x96, 0x19, 0x43, 0x32, 0xae, 0x32, 0xa6, 0x35, 0xcf, 0x35, 0x5b, 0xbc,
    0x3c, 0x66, 0x79, 0x55,
};
static const struct drbg_kat_no_reseed kat49_t = {
    3, kat49_entropyin, kat49_nonce, kat49_persstr,
    kat49_addin0, kat49_addin1, kat49_retbits
};
static const struct drbg_kat kat49 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat49_t
};

static const unsigned char kat50_entropyin[] = {
    0x62, 0xce, 0xf3, 0x5a, 0x51, 0x8e, 0xf6, 0x52, 0x0b, 0x6e, 0xdb, 0x1a,
    0xa9, 0xa2, 0x14, 0x96,
};
static const unsigned char kat50_nonce[] = {
    0x2b, 0x98, 0xe2, 0x69, 0xf5, 0x1f, 0xb3, 0xba,
};
static const unsigned char kat50_persstr[] = {
    0x89, 0xa6, 0xc9, 0x11, 0x27, 0x13, 0xd0, 0xd1, 0x96, 0xc8, 0xe0, 0x26,
    0x60, 0xa2, 0x51, 0x31,
};
static const unsigned char kat50_addin0[] = {
    0x35, 0xed, 0xb8, 0x77, 0x01, 0xcb, 0x50, 0x01, 0x43, 0x4d, 0xb1, 0x85,
    0x7b, 0xf9, 0x8e, 0xfb,
};
static const unsigned char kat50_addin1[] = {
    0xa9, 0xfa, 0x53, 0xb5, 0xd3, 0x03, 0x2e, 0x4b, 0xe6, 0x0d, 0xcd, 0xb1,
    0xe9, 0x4b, 0x4f, 0x50,
};
static const unsigned char kat50_retbits[] = {
    0xde, 0x8b, 0x77, 0x0b, 0x3e, 0xd4, 0x33, 0x15, 0xab, 0x7e, 0x83, 0x31,
    0xe2, 0xdf, 0x12, 0xcb, 0x9a, 0xbe, 0xb3, 0xf6, 0xb1, 0x4c, 0x45, 0x7a,
    0x30, 0x4b, 0xcd, 0x1d, 0xf7, 0x30, 0xdb, 0x52, 0xd4, 0xb4, 0x02, 0xf7,
    0x8b, 0x80, 0x54, 0x59, 0x65, 0xfc, 0x5a, 0x83, 0xdd, 0xf9, 0x99, 0x63,
    0xaa, 0xb5, 0xe1, 0xb3, 0x79, 0x24, 0x57, 0xdf, 0x4b, 0xf9, 0x27, 0xb6,
    0x6a, 0x4d, 0x67, 0x45,
};
static const struct drbg_kat_no_reseed kat50_t = {
    4, kat50_entropyin, kat50_nonce, kat50_persstr,
    kat50_addin0, kat50_addin1, kat50_retbits
};
static const struct drbg_kat kat50 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat50_t
};

static const unsigned char kat51_entropyin[] = {
    0xc0, 0x2a, 0xf7, 0x05, 0xf6, 0x99, 0xc3, 0xa7, 0xf5, 0xe2, 0xbb, 0x9b,
    0x14, 0xc7, 0xce, 0xa7,
};
static const unsigned char kat51_nonce[] = {
    0xeb, 0x24, 0x6b, 0x26, 0xa3, 0xa0, 0xa1, 0x0b,
};
static const unsigned char kat51_persstr[] = {
    0x07, 0xdf, 0x3d, 0x3d, 0x48, 0x6b, 0xbe, 0x68, 0x19, 0x04, 0x40, 0x94,
    0xc9, 0x15, 0x89, 0x99,
};
static const unsigned char kat51_addin0[] = {
    0xf4, 0x9d, 0x27, 0x9f, 0x04, 0x81, 0x5d, 0x98, 0x30, 0x92, 0xb6, 0xb5,
    0xfe, 0x4f, 0x10, 0xf7,
};
static const unsigned char kat51_addin1[] = {
    0xd7, 0x66, 0x07, 0x00, 0x64, 0xdf, 0xe2, 0x03, 0xb2, 0xa3, 0x9d, 0x1b,
    0x24, 0x75, 0x84, 0x10,
};
static const unsigned char kat51_retbits[] = {
    0xdd, 0xbb, 0x48, 0x8c, 0x6b, 0x6f, 0x0f, 0x9c, 0xb7, 0x05, 0xa3, 0xc5,
    0xab, 0x13, 0xc3, 0x17, 0x6f, 0x86, 0x7b, 0x12, 0x0c, 0x29, 0xdb, 0x37,
    0xf7, 0x9c, 0x00, 0x23, 0x8a, 0x77, 0xa0, 0x41, 0x45, 0x04, 0xd3, 0x28,
    0x93, 0xa0, 0x8c, 0x7e, 0x99, 0xc2, 0xb1, 0xa8, 0x8d, 0x50, 0x92, 0x2e,
    0x52, 0xb0, 0xdf, 0x56, 0x0b, 0x75, 0x3a, 0xdd, 0x48, 0x60, 0x4c, 0xaa,
    0xbb, 0xb9, 0x75, 0x2f,
};
static const struct drbg_kat_no_reseed kat51_t = {
    5, kat51_entropyin, kat51_nonce, kat51_persstr,
    kat51_addin0, kat51_addin1, kat51_retbits
};
static const struct drbg_kat kat51 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat51_t
};

static const unsigned char kat52_entropyin[] = {
    0x56, 0x6f, 0xc3, 0x38, 0x98, 0x5a, 0x8f, 0xc8, 0x67, 0xda, 0xbe, 0x24,
    0xf7, 0xf9, 0x09, 0x7d,
};
static const unsigned char kat52_nonce[] = {
    0xdb, 0xf7, 0x1d, 0x79, 0x17, 0xe0, 0xb1, 0x51,
};
static const unsigned char kat52_persstr[] = {
    0x66, 0x56, 0x94, 0x18, 0x17, 0xa6, 0x15, 0xe2, 0x9f, 0x62, 0xbd, 0x4f,
    0x1d, 0x93, 0xcb, 0x11,
};
static const unsigned char kat52_addin0[] = {
    0xa2, 0x74, 0xbe, 0x52, 0x97, 0x46, 0xbd, 0x0b, 0x6a, 0x58, 0x73, 0xdc,
    0x3c, 0x1d, 0xae, 0xbc,
};
static const unsigned char kat52_addin1[] = {
    0x4e, 0x23, 0xfa, 0x29, 0xd2, 0x6b, 0x1f, 0xe9, 0x2b, 0xb4, 0x2b, 0x3e,
    0x9f, 0x2b, 0x4c, 0x08,
};
static const unsigned char kat52_retbits[] = {
    0xb5, 0xda, 0x60, 0xb8, 0x70, 0x22, 0x6a, 0x48, 0xb2, 0xb9, 0xb5, 0xd9,
    0xa9, 0x46, 0x49, 0x54, 0xa8, 0xdb, 0x0d, 0x3e, 0x5f, 0xb3, 0x06, 0x3c,
    0x80, 0xe6, 0x75, 0x19, 0x7d, 0xc5, 0x84, 0x07, 0xb4, 0xc0, 0x40, 0x9c,
    0xd0, 0x69, 0x75, 0xed, 0xa8, 0xaf, 0xe6, 0x66, 0x7a, 0xa0, 0xd7, 0x9e,
    0xcf, 0x55, 0x5e, 0x23, 0x8f, 0x83, 0x42, 0x4f, 0x13, 0x08, 0x19, 0x20,
    0x0c, 0x24, 0x4a, 0x77,
};
static const struct drbg_kat_no_reseed kat52_t = {
    6, kat52_entropyin, kat52_nonce, kat52_persstr,
    kat52_addin0, kat52_addin1, kat52_retbits
};
static const struct drbg_kat kat52 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat52_t
};

static const unsigned char kat53_entropyin[] = {
    0x67, 0x43, 0xbc, 0x12, 0x2a, 0xaa, 0x13, 0x4f, 0x82, 0xac, 0x8c, 0x78,
    0xcc, 0x0b, 0x64, 0xe7,
};
static const unsigned char kat53_nonce[] = {
    0x2d, 0xc3, 0x2e, 0x4f, 0x3b, 0xba, 0x5a, 0x20,
};
static const unsigned char kat53_persstr[] = {
    0xb8, 0x07, 0x7f, 0x23, 0xa9, 0x08, 0xc4, 0x25, 0x92, 0xc1, 0x3b, 0x63,
    0x89, 0x09, 0x98, 0xd0,
};
static const unsigned char kat53_addin0[] = {
    0xd6, 0x83, 0x74, 0xf5, 0x8c, 0xe2, 0x5b, 0xd8, 0xe9, 0x5f, 0xe0, 0x6d,
    0xe8, 0xa9, 0x50, 0xa1,
};
static const unsigned char kat53_addin1[] = {
    0x07, 0xb7, 0xab, 0x9c, 0x32, 0xa9, 0x78, 0x30, 0xa0, 0x0e, 0xd9, 0x89,
    0x5e, 0x3b, 0xbe, 0x19,
};
static const unsigned char kat53_retbits[] = {
    0xfe, 0x82, 0x7f, 0x53, 0x07, 0x6d, 0xf3, 0x08, 0x47, 0xd0, 0xfb, 0x0f,
    0x55, 0xa2, 0xcd, 0x63, 0x81, 0xca, 0x0e, 0x79, 0xe6, 0x81, 0x62, 0x1f,
    0x49, 0xf3, 0x1e, 0x48, 0x11, 0xe9, 0xa6, 0x4c, 0x57, 0xc0, 0x99, 0x4f,
    0x18, 0x0c, 0x31, 0x1a, 0x40, 0x44, 0x40, 0xf2, 0x77, 0xfa, 0x62, 0xe0,
    0xf9, 0x9c, 0x6f, 0x5c, 0x7b, 0x88, 0xbe, 0xe2, 0x07, 0xfc, 0xa9, 0x92,
    0x6b, 0xfc, 0x9e, 0x46,
};
static const struct drbg_kat_no_reseed kat53_t = {
    7, kat53_entropyin, kat53_nonce, kat53_persstr,
    kat53_addin0, kat53_addin1, kat53_retbits
};
static const struct drbg_kat kat53 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat53_t
};

static const unsigned char kat54_entropyin[] = {
    0xf2, 0xad, 0xdf, 0x01, 0x67, 0xa1, 0x6d, 0xa3, 0x8d, 0x2f, 0xd7, 0x92,
    0xa7, 0x18, 0x9c, 0x06,
};
static const unsigned char kat54_nonce[] = {
    0x2b, 0x50, 0xd8, 0xcc, 0x7f, 0x71, 0x40, 0x1c,
};
static const unsigned char kat54_persstr[] = {
    0xb9, 0x17, 0xa5, 0x4c, 0x13, 0xd5, 0x8c, 0xb6, 0xad, 0x47, 0xc8, 0xfa,
    0xf6, 0xf9, 0xa7, 0xf5,
};
static const unsigned char kat54_addin0[] = {
    0xcf, 0x2e, 0x78, 0x5a, 0x06, 0xbc, 0x5e, 0xb7, 0x1e, 0x3b, 0x52, 0x4f,
    0x11, 0x72, 0xc7, 0x55,
};
static const unsigned char kat54_addin1[] = {
    0xc9, 0xa7, 0x53, 0xa9, 0xce, 0xdd, 0xd7, 0x5d, 0x1a, 0x62, 0x6d, 0xa7,
    0x27, 0x95, 0xa3, 0xc6,
};
static const unsigned char kat54_retbits[] = {
    0xe2, 0xaa, 0xf1, 0x1c, 0x20, 0xde, 0x2b, 0xaf, 0x89, 0x23, 0x04, 0x1d,
    0x83, 0x8f, 0x4a, 0xcb, 0x38, 0xe2, 0x22, 0xd3, 0x06, 0xb9, 0x7c, 0x97,
    0xbf, 0xe5, 0x46, 0x61, 0x7b, 0x3d, 0x8d, 0x92, 0xe4, 0x0c, 0xaa, 0x2e,
    0xdc, 0xac, 0x7a, 0x2f, 0xb0, 0x64, 0x3e, 0xd5, 0x7d, 0x47, 0xa9, 0x0a,
    0x11, 0x64, 0x10, 0x44, 0xb6, 0xc6, 0xa7, 0xdf, 0x42, 0x4d, 0xbd, 0xa2,
    0xb4, 0x5f, 0x0d, 0xf9,
};
static const struct drbg_kat_no_reseed kat54_t = {
    8, kat54_entropyin, kat54_nonce, kat54_persstr,
    kat54_addin0, kat54_addin1, kat54_retbits
};
static const struct drbg_kat kat54 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat54_t
};

static const unsigned char kat55_entropyin[] = {
    0xca, 0x23, 0x34, 0x58, 0xf6, 0x6b, 0x26, 0xf2, 0xc1, 0x35, 0x7a, 0xef,
    0xce, 0xb6, 0x79, 0xef,
};
static const unsigned char kat55_nonce[] = {
    0xfe, 0x13, 0x64, 0x04, 0xad, 0x7a, 0x49, 0xe7,
};
static const unsigned char kat55_persstr[] = {
    0x26, 0x83, 0xdc, 0x31, 0x3b, 0xc0, 0xbb, 0x93, 0xeb, 0xb4, 0x6b, 0xbb,
    0x55, 0x3f, 0x58, 0x75,
};
static const unsigned char kat55_addin0[] = {
    0xf7, 0x8a, 0xb1, 0xf4, 0x52, 0x6f, 0x0f, 0xa5, 0xcb, 0x75, 0x43, 0xd1,
    0x46, 0x43, 0xfe, 0xb5,
};
static const unsigned char kat55_addin1[] = {
    0x40, 0xb9, 0x26, 0x32, 0x4c, 0xc2, 0x94, 0x3a, 0xca, 0x83, 0x7f, 0x69,
    0x0d, 0x0b, 0xcd, 0xa4,
};
static const unsigned char kat55_retbits[] = {
    0x37, 0x24, 0xd0, 0xe5, 0x80, 0xe9, 0x0d, 0x4e, 0xf3, 0x2e, 0x5f, 0x8a,
    0x5b, 0xb5, 0x62, 0x25, 0xa9, 0x13, 0xf2, 0x10, 0xb7, 0x6d, 0x5a, 0x78,
    0xfd, 0x8e, 0xf2, 0x5d, 0x6e, 0xec, 0x9f, 0xf1, 0x47, 0x11, 0xd6, 0xdb,
    0x80, 0xe1, 0xe8, 0x11, 0x42, 0xa5, 0x60, 0x75, 0x34, 0xa7, 0x25, 0xd2,
    0x78, 0x5c, 0x8e, 0x3e, 0x4a, 0x48, 0xa5, 0x94, 0xb0, 0xd9, 0x1b, 0xc9,
    0x32, 0x82, 0x7b, 0xd7,
};
static const struct drbg_kat_no_reseed kat55_t = {
    9, kat55_entropyin, kat55_nonce, kat55_persstr,
    kat55_addin0, kat55_addin1, kat55_retbits
};
static const struct drbg_kat kat55 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat55_t
};

static const unsigned char kat56_entropyin[] = {
    0x2b, 0x52, 0x1d, 0xc4, 0x03, 0x12, 0x9f, 0xd9, 0xed, 0x0b, 0xe9, 0xf5,
    0xfd, 0xe0, 0x66, 0x17,
};
static const unsigned char kat56_nonce[] = {
    0xf0, 0xdf, 0xa0, 0x1e, 0x09, 0x9b, 0x95, 0xa1,
};
static const unsigned char kat56_persstr[] = {
    0xfe, 0x85, 0x8d, 0x59, 0xd7, 0x34, 0xf6, 0x0f, 0x71, 0xa3, 0x87, 0x55,
    0x3c, 0xca, 0x46, 0xff,
};
static const unsigned char kat56_addin0[] = {
    0x6f, 0xba, 0x8d, 0x06, 0xff, 0x43, 0x30, 0x07, 0xbd, 0x7d, 0x15, 0x56,
    0x9a, 0xf3, 0xd0, 0x69,
};
static const unsigned char kat56_addin1[] = {
    0x80, 0xb0, 0xc1, 0x1f, 0x18, 0xb5, 0x92, 0x3a, 0x89, 0xf0, 0xfe, 0x23,
    0x77, 0x6d, 0x39, 0x05,
};
static const unsigned char kat56_retbits[] = {
    0x41, 0x73, 0x88, 0xde, 0xe9, 0x08, 0x4f, 0x58, 0x8d, 0x0a, 0xdb, 0x5e,
    0x5d, 0x1b, 0x54, 0x6b, 0x12, 0x55, 0xcd, 0xf5, 0x9d, 0x69, 0x95, 0x7a,
    0xe6, 0x87, 0xc7, 0x82, 0x2b, 0x93, 0x5c, 0xe0, 0x29, 0x30, 0x6c, 0x02,
    0x50, 0xd4, 0x69, 0x83, 0x9c, 0x7d, 0xf9, 0xd3, 0xd0, 0x8e, 0xc8, 0xc5,
    0x0a, 0x66, 0x7a, 0xf1, 0x0d, 0x8d, 0x5b, 0x00, 0x73, 0x02, 0xa6, 0xe8,
    0x7b, 0x4e, 0x7f, 0x2c,
};
static const struct drbg_kat_no_reseed kat56_t = {
    10, kat56_entropyin, kat56_nonce, kat56_persstr,
    kat56_addin0, kat56_addin1, kat56_retbits
};
static const struct drbg_kat kat56 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat56_t
};

static const unsigned char kat57_entropyin[] = {
    0xd0, 0x50, 0x2b, 0x78, 0xf7, 0xaa, 0x6f, 0xa1, 0xd8, 0x88, 0x42, 0x1d,
    0x70, 0x2c, 0xad, 0xd1,
};
static const unsigned char kat57_nonce[] = {
    0x4b, 0x37, 0xb8, 0x96, 0x59, 0x69, 0xea, 0x8d,
};
static const unsigned char kat57_persstr[] = {
    0x56, 0xbd, 0x43, 0x1f, 0x04, 0x88, 0xfe, 0x6d, 0x7b, 0xa3, 0x83, 0x6e,
    0x5c, 0x8c, 0x7a, 0xa9,
};
static const unsigned char kat57_addin0[] = {
    0x26, 0x92, 0x5d, 0xbc, 0xbe, 0xd7, 0x33, 0xbe, 0x2a, 0x5e, 0x76, 0x49,
    0xc8, 0x76, 0xa0, 0xf8,
};
static const unsigned char kat57_addin1[] = {
    0x9c, 0x70, 0x35, 0x2d, 0x4f, 0x9c, 0x88, 0x0e, 0x76, 0xd4, 0x68, 0x08,
    0x38, 0xdd, 0x0d, 0xea,
};
static const unsigned char kat57_retbits[] = {
    0x65, 0x00, 0x24, 0x2c, 0x35, 0x88, 0xb4, 0x52, 0x30, 0x7b, 0x2a, 0x06,
    0x34, 0x90, 0x0e, 0x75, 0x6a, 0x3d, 0x26, 0x19, 0xfc, 0x2a, 0xdf, 0x10,
    0x77, 0x69, 0x22, 0xb0, 0x06, 0x61, 0x7e, 0xe8, 0x81, 0x82, 0x83, 0x66,
    0xac, 0x52, 0x7f, 0xad, 0xa0, 0x01, 0x86, 0xbc, 0x8b, 0xfd, 0x6b, 0xee,
    0x66, 0xd6, 0x38, 0x03, 0x5a, 0xaf, 0xff, 0xbc, 0x83, 0x32, 0xa5, 0x4c,
    0xa1, 0x7d, 0xf1, 0x6f,
};
static const struct drbg_kat_no_reseed kat57_t = {
    11, kat57_entropyin, kat57_nonce, kat57_persstr,
    kat57_addin0, kat57_addin1, kat57_retbits
};
static const struct drbg_kat kat57 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat57_t
};

static const unsigned char kat58_entropyin[] = {
    0xa3, 0xb9, 0x5f, 0x32, 0x01, 0xfa, 0x01, 0x01, 0x07, 0x57, 0x30, 0x74,
    0xb1, 0x19, 0x9b, 0xc0,
};
static const unsigned char kat58_nonce[] = {
    0x60, 0x30, 0xe5, 0xb6, 0x13, 0x47, 0xb0, 0xe2,
};
static const unsigned char kat58_persstr[] = {
    0x9e, 0x77, 0x85, 0x46, 0xd4, 0x5f, 0x17, 0x93, 0xa1, 0x1d, 0x80, 0xa0,
    0x2e, 0x4c, 0xa3, 0xc3,
};
static const unsigned char kat58_addin0[] = {
    0xe5, 0x07, 0x33, 0x61, 0x98, 0x25, 0x31, 0x37, 0xa5, 0x55, 0xe8, 0xc1,
    0x18, 0xa1, 0x2d, 0x2f,
};
static const unsigned char kat58_addin1[] = {
    0xe5, 0xce, 0xea, 0xcb, 0xb5, 0x22, 0xf5, 0xde, 0x81, 0x78, 0x9c, 0x32,
    0xcb, 0xcd, 0xef, 0x30,
};
static const unsigned char kat58_retbits[] = {
    0x4a, 0xd1, 0x8e, 0xa0, 0x83, 0x7f, 0x06, 0x3e, 0x9f, 0xbf, 0x6c, 0x72,
    0xd1, 0xaf, 0x6f, 0x77, 0x96, 0x41, 0xf4, 0xe1, 0x1c, 0x40, 0x0d, 0x95,
    0xcb, 0x2b, 0xa4, 0x22, 0x99, 0xa9, 0x66, 0x48, 0xe0, 0x59, 0xe1, 0xa6,
    0x3f, 0x7c, 0x03, 0x79, 0x25, 0x44, 0x89, 0xc7, 0xcb, 0xff, 0xfb, 0x22,
    0x11, 0x84, 0x57, 0x18, 0xc5, 0x26, 0x00, 0xc9, 0xe8, 0x58, 0x49, 0xd6,
    0xe0, 0x95, 0x1f, 0x7b,
};
static const struct drbg_kat_no_reseed kat58_t = {
    12, kat58_entropyin, kat58_nonce, kat58_persstr,
    kat58_addin0, kat58_addin1, kat58_retbits
};
static const struct drbg_kat kat58 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat58_t
};

static const unsigned char kat59_entropyin[] = {
    0x88, 0x18, 0xbe, 0x7e, 0x8e, 0x81, 0x5b, 0xca, 0x0c, 0xb5, 0xa0, 0x14,
    0x4d, 0x44, 0xb6, 0x72,
};
static const unsigned char kat59_nonce[] = {
    0x12, 0x2b, 0x14, 0x8c, 0xa1, 0xaa, 0xba, 0x60,
};
static const unsigned char kat59_persstr[] = {
    0x90, 0x6d, 0xc3, 0x0b, 0x39, 0x54, 0x27, 0x38, 0x56, 0x73, 0x26, 0x13,
    0x66, 0xf5, 0xe9, 0xa8,
};
static const unsigned char kat59_addin0[] = {
    0x56, 0xee, 0x67, 0xff, 0x14, 0x66, 0xce, 0xed, 0xd7, 0x28, 0x1d, 0x73,
    0xf3, 0x9c, 0x6c, 0x85,
};
static const unsigned char kat59_addin1[] = {
    0x82, 0xe4, 0xa6, 0xed, 0xb0, 0x78, 0x00, 0x79, 0xfa, 0xee, 0x25, 0x8e,
    0xd1, 0x13, 0x7e, 0xcf,
};
static const unsigned char kat59_retbits[] = {
    0x88, 0x55, 0xad, 0x03, 0xe3, 0x87, 0x27, 0xd0, 0x49, 0xe8, 0x55, 0x8e,
    0xcc, 0x66, 0x3b, 0x8c, 0x89, 0xcb, 0x5b, 0xd1, 0x38, 0xac, 0x34, 0x9c,
    0x6e, 0x5f, 0x78, 0x41, 0x70, 0xc5, 0x07, 0x69, 0xc8, 0xc9, 0x7b, 0x09,
    0x4d, 0xfa, 0x07, 0x52, 0xd0, 0x65, 0x53, 0x1d, 0xa4, 0x00, 0x85, 0x81,
    0x7c, 0xcb, 0x08, 0xf2, 0xb9, 0x4c, 0xbb, 0xba, 0xe7, 0x59, 0x8c, 0x96,
    0x3e, 0x3a, 0xd6, 0xd1,
};
static const struct drbg_kat_no_reseed kat59_t = {
    13, kat59_entropyin, kat59_nonce, kat59_persstr,
    kat59_addin0, kat59_addin1, kat59_retbits
};
static const struct drbg_kat kat59 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat59_t
};

static const unsigned char kat60_entropyin[] = {
    0x00, 0x90, 0xdc, 0xb9, 0x1f, 0x02, 0x70, 0xf6, 0x58, 0x20, 0x7d, 0x49,
    0xa7, 0x47, 0x38, 0xe2,
};
static const unsigned char kat60_nonce[] = {
    0x9d, 0x18, 0xae, 0xa2, 0x68, 0x4e, 0xe2, 0xb7,
};
static const unsigned char kat60_persstr[] = {
    0x64, 0xa8, 0x18, 0x57, 0xce, 0x29, 0xf0, 0xd2, 0xa8, 0x51, 0x03, 0x00,
    0xb0, 0xa3, 0xfa, 0xc0,
};
static const unsigned char kat60_addin0[] = {
    0xcd, 0x7f, 0x89, 0xf8, 0xd5, 0x46, 0x8e, 0xae, 0x8b, 0xca, 0x31, 0x4f,
    0xd9, 0xf2, 0x5b, 0x23,
};
static const unsigned char kat60_addin1[] = {
    0x94, 0x9b, 0x94, 0x41, 0x7f, 0xb4, 0x2b, 0xbc, 0x7c, 0xdb, 0x3f, 0x53,
    0xa0, 0x55, 0x80, 0x19,
};
static const unsigned char kat60_retbits[] = {
    0x63, 0xd3, 0xe9, 0x19, 0x88, 0x1f, 0x87, 0xd6, 0x9c, 0xba, 0xdc, 0xb6,
    0x78, 0x9e, 0x0b, 0x8f, 0x2e, 0x49, 0xf4, 0xc9, 0xd0, 0xe3, 0xba, 0x1e,
    0x4a, 0x59, 0x16, 0x2a, 0xf7, 0x18, 0xbc, 0x40, 0xdd, 0x67, 0x3e, 0xae,
    0x8f, 0x48, 0xfd, 0x27, 0x82, 0xe0, 0x6c, 0xde, 0xc5, 0x63, 0x2a, 0x4b,
    0x2e, 0x60, 0x65, 0x5c, 0xba, 0x75, 0xaf, 0xaf, 0x4c, 0x12, 0xa6, 0x74,
    0xee, 0xe4, 0xd3, 0xfc,
};
static const struct drbg_kat_no_reseed kat60_t = {
    14, kat60_entropyin, kat60_nonce, kat60_persstr,
    kat60_addin0, kat60_addin1, kat60_retbits
};
static const struct drbg_kat kat60 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat60_t
};

static const unsigned char kat61_entropyin[] = {
    0x2d, 0x2a, 0xb5, 0x64, 0x20, 0x29, 0x18, 0xc4, 0xef, 0x5b, 0x10, 0x2d,
    0xda, 0x38, 0x5a, 0x18,
};
static const unsigned char kat61_nonce[] = {
    0x25, 0x91, 0x95, 0x26, 0x9e, 0xc1, 0x1a, 0xf6,
};
static const unsigned char kat61_persstr[] = {0};
static const unsigned char kat61_addin0[] = {0};
static const unsigned char kat61_addin1[] = {0};
static const unsigned char kat61_retbits[] = {
    0x2c, 0x5c, 0xd7, 0x9e, 0xd8, 0x76, 0x22, 0xa9, 0x1b, 0x86, 0x54, 0xc8,
    0x90, 0x3d, 0x85, 0x22, 0x42, 0xcd, 0x49, 0xcb, 0x5d, 0xf2, 0xd4, 0xb4,
    0x15, 0x05, 0x84, 0x30, 0x1c, 0x59, 0xf0, 0x1f, 0xd9, 0x5a, 0x70, 0x2a,
    0xc1, 0x57, 0xc8, 0x4c, 0xc1, 0x5f, 0x42, 0xc8, 0x21, 0x13, 0x35, 0x67,
    0x2d, 0x8c, 0xe1, 0x29, 0x1e, 0xf9, 0xb1, 0xde, 0xf7, 0x81, 0x49, 0xa0,
    0x4f, 0xa2, 0x69, 0x7c,
};
static const struct drbg_kat_no_reseed kat61_t = {
    0, kat61_entropyin, kat61_nonce, kat61_persstr,
    kat61_addin0, kat61_addin1, kat61_retbits
};
static const struct drbg_kat kat61 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat61_t
};

static const unsigned char kat62_entropyin[] = {
    0xa0, 0x16, 0x46, 0x3d, 0xbb, 0x49, 0x99, 0x90, 0xcb, 0xcd, 0xa4, 0x50,
    0x46, 0xd8, 0xf3, 0x37,
};
static const unsigned char kat62_nonce[] = {
    0x24, 0x9d, 0x02, 0xde, 0x2d, 0xcf, 0x3e, 0x57,
};
static const unsigned char kat62_persstr[] = {0};
static const unsigned char kat62_addin0[] = {0};
static const unsigned char kat62_addin1[] = {0};
static const unsigned char kat62_retbits[] = {
    0x35, 0xb8, 0x1f, 0xb9, 0x42, 0x41, 0xf4, 0xc9, 0x31, 0x9a, 0x7f, 0x16,
    0xb4, 0x42, 0x49, 0x52, 0x52, 0xdb, 0x4c, 0x98, 0x48, 0x60, 0xd0, 0x35,
    0xf6, 0xc4, 0x54, 0x03, 0xf9, 0x74, 0xf5, 0x34, 0xfa, 0x93, 0xb2, 0x1b,
    0x6b, 0x89, 0x44, 0x1b, 0xe0, 0x7c, 0x5c, 0x29, 0x31, 0x7f, 0x86, 0x8d,
    0xc9, 0xab, 0x5c, 0x18, 0x37, 0x74, 0x37, 0xfa, 0xdb, 0x4d, 0x85, 0x7e,
    0xe0, 0x92, 0xf9, 0x23,
};
static const struct drbg_kat_no_reseed kat62_t = {
    1, kat62_entropyin, kat62_nonce, kat62_persstr,
    kat62_addin0, kat62_addin1, kat62_retbits
};
static const struct drbg_kat kat62 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat62_t
};

static const unsigned char kat63_entropyin[] = {
    0x1e, 0x38, 0x20, 0xc4, 0x6e, 0x50, 0xb7, 0xcb, 0xc5, 0x8b, 0x05, 0xaa,
    0x72, 0x24, 0x27, 0xa8,
};
static const unsigned char kat63_nonce[] = {
    0xfe, 0x0f, 0x40, 0x01, 0x4e, 0xb3, 0x27, 0x9f,
};
static const unsigned char kat63_persstr[] = {0};
static const unsigned char kat63_addin0[] = {0};
static const unsigned char kat63_addin1[] = {0};
static const unsigned char kat63_retbits[] = {
    0x18, 0x53, 0x0b, 0x9d, 0xe8, 0x89, 0xd8, 0xad, 0x3f, 0x41, 0xd5, 0x39,
    0x79, 0x6c, 0x2a, 0x95, 0xd6, 0x82, 0xfb, 0x02, 0x6d, 0xf7, 0xca, 0x09,
    0x4a, 0xf4, 0xab, 0x23, 0x95, 0x29, 0x2e, 0x6f, 0xcd, 0xb1, 0x75, 0xba,
    0x25, 0x7d, 0x61, 0x27, 0x9d, 0xa4, 0xd0, 0xfc, 0x78, 0xfd, 0xf8, 0xa4,
    0xeb, 0x46, 0xa0, 0xe2, 0xd7, 0x54, 0xf5, 0xf8, 0xf0, 0x9e, 0xcb, 0x15,
    0x7c, 0x8a, 0x58, 0xa1,
};
static const struct drbg_kat_no_reseed kat63_t = {
    2, kat63_entropyin, kat63_nonce, kat63_persstr,
    kat63_addin0, kat63_addin1, kat63_retbits
};
static const struct drbg_kat kat63 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat63_t
};

static const unsigned char kat64_entropyin[] = {
    0x3b, 0x02, 0xc5, 0xa4, 0x74, 0x67, 0x96, 0x48, 0xe2, 0xfe, 0x09, 0x0a,
    0x13, 0xbe, 0x04, 0x1c,
};
static const unsigned char kat64_nonce[] = {
    0x71, 0xed, 0x5a, 0xa0, 0x78, 0xb8, 0x3a, 0x7b,
};
static const unsigned char kat64_persstr[] = {0};
static const unsigned char kat64_addin0[] = {0};
static const unsigned char kat64_addin1[] = {0};
static const unsigned char kat64_retbits[] = {
    0xe8, 0xe2, 0xe5, 0xd8, 0xbd, 0x99, 0x29, 0xe4, 0xe1, 0x67, 0x5c, 0x84,
    0x61, 0x45, 0x7c, 0xab, 0x06, 0x33, 0xd8, 0x3f, 0x9b, 0xd2, 0x43, 0xbc,
    0x58, 0x29, 0x37, 0xc2, 0xb9, 0x61, 0xb7, 0x1c, 0x11, 0x85, 0x6f, 0x83,
    0x6c, 0x37, 0x32, 0x9d, 0x3f, 0xa8, 0x63, 0x76, 0xee, 0xc9, 0x71, 0x39,
    0x10, 0x4d, 0xe5, 0x32, 0x60, 0xe1, 0x22, 0xcb, 0x2b, 0x6d, 0x82, 0x4e,
    0x21, 0xe7, 0x5c, 0xe7,
};
static const struct drbg_kat_no_reseed kat64_t = {
    3, kat64_entropyin, kat64_nonce, kat64_persstr,
    kat64_addin0, kat64_addin1, kat64_retbits
};
static const struct drbg_kat kat64 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat64_t
};

static const unsigned char kat65_entropyin[] = {
    0x05, 0x0c, 0x48, 0xab, 0x8a, 0x05, 0xff, 0x69, 0x0b, 0x62, 0xe3, 0x7d,
    0x9b, 0x21, 0xf2, 0xfa,
};
static const unsigned char kat65_nonce[] = {
    0xf7, 0x96, 0x48, 0x75, 0xbe, 0x28, 0x83, 0x53,
};
static const unsigned char kat65_persstr[] = {0};
static const unsigned char kat65_addin0[] = {0};
static const unsigned char kat65_addin1[] = {0};
static const unsigned char kat65_retbits[] = {
    0x24, 0xcb, 0x0e, 0xdc, 0xb8, 0x9e, 0xec, 0x61, 0x76, 0x13, 0xa2, 0x7f,
    0xe3, 0xe4, 0x47, 0x5a, 0xff, 0xab, 0x49, 0x8d, 0x57, 0xbc, 0x3c, 0x4e,
    0xf8, 0x44, 0xfe, 0x19, 0xb3, 0x84, 0x13, 0x47, 0x7c, 0x1d, 0xb0, 0x40,
    0xf1, 0x67, 0x57, 0xf1, 0x85, 0x1b, 0x79, 0x9c, 0x1f, 0x26, 0x69, 0xaa,
    0x9f, 0x9c, 0x50, 0x53, 0x2e, 0xaf, 0xbe, 0x1e, 0x29, 0x48, 0xe3, 0x38,
    0xd1, 0xca, 0x4b, 0xd7,
};
static const struct drbg_kat_no_reseed kat65_t = {
    4, kat65_entropyin, kat65_nonce, kat65_persstr,
    kat65_addin0, kat65_addin1, kat65_retbits
};
static const struct drbg_kat kat65 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat65_t
};

static const unsigned char kat66_entropyin[] = {
    0xec, 0xd9, 0x9c, 0x0b, 0x49, 0x1c, 0xf6, 0xa3, 0x23, 0xbc, 0x33, 0x36,
    0x46, 0xe5, 0xc5, 0x6a,
};
static const unsigned char kat66_nonce[] = {
    0xb1, 0x0a, 0x17, 0x84, 0x99, 0x5c, 0xb3, 0x4c,
};
static const unsigned char kat66_persstr[] = {0};
static const unsigned char kat66_addin0[] = {0};
static const unsigned char kat66_addin1[] = {0};
static const unsigned char kat66_retbits[] = {
    0x65, 0x35, 0xc4, 0x8d, 0x99, 0x9a, 0x30, 0x6a, 0xcb, 0x83, 0xf7, 0xb7,
    0x20, 0xb3, 0xc3, 0x97, 0x72, 0xa4, 0x96, 0xe7, 0x5a, 0x89, 0x0f, 0x6e,
    0x14, 0xd7, 0x0a, 0x82, 0x41, 0x42, 0xa1, 0x2d, 0x01, 0xde, 0xe5, 0x52,
    0x71, 0xb8, 0x2c, 0x7b, 0x02, 0x7b, 0x94, 0x68, 0xc3, 0x09, 0xfa, 0x2d,
    0x08, 0xe9, 0x1e, 0x5e, 0x37, 0x9e, 0xb9, 0x0c, 0xf8, 0x91, 0x5c, 0xf4,
    0xcc, 0x24, 0x1b, 0x50,
};
static const struct drbg_kat_no_reseed kat66_t = {
    5, kat66_entropyin, kat66_nonce, kat66_persstr,
    kat66_addin0, kat66_addin1, kat66_retbits
};
static const struct drbg_kat kat66 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat66_t
};

static const unsigned char kat67_entropyin[] = {
    0x34, 0xae, 0xe3, 0xef, 0x35, 0xb2, 0xa1, 0xe3, 0x1c, 0x42, 0x97, 0x25,
    0xfe, 0xaa, 0x6b, 0xfd,
};
static const unsigned char kat67_nonce[] = {
    0xe4, 0x36, 0x9b, 0x6a, 0xda, 0x90, 0x91, 0xed,
};
static const unsigned char kat67_persstr[] = {0};
static const unsigned char kat67_addin0[] = {0};
static const unsigned char kat67_addin1[] = {0};
static const unsigned char kat67_retbits[] = {
    0xaf, 0x9d, 0xb4, 0xdf, 0x46, 0xac, 0x80, 0xfe, 0xca, 0xd6, 0xc3, 0xd1,
    0xd9, 0x75, 0xd2, 0xd0, 0xa6, 0x65, 0xf3, 0x1b, 0x4e, 0x3b, 0x07, 0xd9,
    0x62, 0x63, 0x22, 0x96, 0xb7, 0x60, 0x39, 0xdd, 0xca, 0xb8, 0xed, 0xde,
    0x17, 0xba, 0xd6, 0x23, 0x01, 0x31, 0x16, 0x2e, 0x59, 0xd0, 0x7d, 0xbb,
    0xd0, 0xf0, 0xb5, 0x2e, 0x5c, 0xa9, 0x3c, 0xf3, 0x7f, 0x12, 0xc0, 0x30,
    0x42, 0xbf, 0x0c, 0x82,
};
static const struct drbg_kat_no_reseed kat67_t = {
    6, kat67_entropyin, kat67_nonce, kat67_persstr,
    kat67_addin0, kat67_addin1, kat67_retbits
};
static const struct drbg_kat kat67 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat67_t
};

static const unsigned char kat68_entropyin[] = {
    0x91, 0x15, 0x64, 0x69, 0x54, 0x0c, 0x78, 0x5d, 0xf5, 0x6a, 0x10, 0xe1,
    0x6b, 0x66, 0x10, 0x7c,
};
static const unsigned char kat68_nonce[] = {
    0x7e, 0x2b, 0x2a, 0x7f, 0x58, 0xf3, 0xef, 0x20,
};
static const unsigned char kat68_persstr[] = {0};
static const unsigned char kat68_addin0[] = {0};
static const unsigned char kat68_addin1[] = {0};
static const unsigned char kat68_retbits[] = {
    0x9a, 0x76, 0x3a, 0x4c, 0x05, 0x27, 0x25, 0xe7, 0xae, 0x9d, 0x21, 0x22,
    0xa4, 0x5c, 0xd1, 0x8e, 0xa2, 0x8c, 0x58, 0x17, 0x2b, 0x87, 0xc5, 0x7e,
    0xdb, 0x9d, 0xaa, 0x32, 0xf8, 0xa2, 0x6e, 0xb9, 0x8d, 0x6b, 0x88, 0x12,
    0x09, 0x5a, 0x54, 0xcf, 0x62, 0x82, 0xc5, 0xe0, 0xfc, 0x18, 0xcd, 0x30,
    0x12, 0x44, 0x15, 0xd2, 0x67, 0xc9, 0xfc, 0x2c, 0xda, 0xe9, 0xbe, 0xed,
    0xbb, 0x86, 0x58, 0x4e,
};
static const struct drbg_kat_no_reseed kat68_t = {
    7, kat68_entropyin, kat68_nonce, kat68_persstr,
    kat68_addin0, kat68_addin1, kat68_retbits
};
static const struct drbg_kat kat68 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat68_t
};

static const unsigned char kat69_entropyin[] = {
    0xcd, 0x5d, 0xbc, 0x35, 0xd1, 0xb4, 0x43, 0x5c, 0x4c, 0x94, 0x5b, 0xaf,
    0xe4, 0xb8, 0xfc, 0x7a,
};
static const unsigned char kat69_nonce[] = {
    0xaf, 0x4a, 0xcd, 0xfc, 0x81, 0x19, 0x32, 0x2c,
};
static const unsigned char kat69_persstr[] = {0};
static const unsigned char kat69_addin0[] = {0};
static const unsigned char kat69_addin1[] = {0};
static const unsigned char kat69_retbits[] = {
    0xfe, 0x67, 0x82, 0x99, 0xf1, 0xad, 0xe1, 0x96, 0x21, 0x88, 0x02, 0x35,
    0xfa, 0x6f, 0xe2, 0xb5, 0x01, 0x84, 0xf8, 0x12, 0x43, 0x83, 0x3e, 0xbe,
    0x68, 0xc6, 0x16, 0xa9, 0x03, 0x4e, 0x23, 0xe0, 0xcc, 0x87, 0x3e, 0x8a,
    0xc4, 0xd0, 0xe3, 0x65, 0xb5, 0x24, 0xa3, 0x78, 0x32, 0xb5, 0x2d, 0x97,
    0xa8, 0x38, 0x62, 0x93, 0x24, 0x91, 0xd2, 0x64, 0x3a, 0x73, 0xe4, 0x6a,
    0x80, 0xb5, 0x14, 0x9c,
};
static const struct drbg_kat_no_reseed kat69_t = {
    8, kat69_entropyin, kat69_nonce, kat69_persstr,
    kat69_addin0, kat69_addin1, kat69_retbits
};
static const struct drbg_kat kat69 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat69_t
};

static const unsigned char kat70_entropyin[] = {
    0xd5, 0x2d, 0x11, 0x59, 0x78, 0x7b, 0xaa, 0xb2, 0x13, 0xe8, 0x79, 0x21,
    0xfd, 0x95, 0x91, 0x10,
};
static const unsigned char kat70_nonce[] = {
    0x7a, 0xb3, 0x75, 0x38, 0xee, 0x02, 0x45, 0xc8,
};
static const unsigned char kat70_persstr[] = {0};
static const unsigned char kat70_addin0[] = {0};
static const unsigned char kat70_addin1[] = {0};
static const unsigned char kat70_retbits[] = {
    0xf4, 0xd9, 0xea, 0xd3, 0x59, 0x46, 0x06, 0xac, 0x51, 0x94, 0x22, 0x15,
    0xaa, 0x0b, 0xd2, 0x9b, 0xaa, 0x7c, 0x28, 0x1e, 0x62, 0xd2, 0xbc, 0xc1,
    0xa7, 0x7f, 0x57, 0xe3, 0x59, 0x66, 0x83, 0x19, 0x9d, 0x70, 0xc0, 0x09,
    0xc0, 0x84, 0xeb, 0x36, 0x54, 0x22, 0x4e, 0xd7, 0xb0, 0x7a, 0x09, 0xb4,
    0x94, 0x48, 0x0a, 0x11, 0x3b, 0x33, 0xe0, 0x36, 0x6c, 0xb2, 0x25, 0x3e,
    0xdb, 0xb0, 0xcb, 0x5e,
};
static const struct drbg_kat_no_reseed kat70_t = {
    9, kat70_entropyin, kat70_nonce, kat70_persstr,
    kat70_addin0, kat70_addin1, kat70_retbits
};
static const struct drbg_kat kat70 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat70_t
};

static const unsigned char kat71_entropyin[] = {
    0x51, 0x10, 0xdb, 0x9d, 0x5f, 0x4f, 0xbd, 0xc0, 0x37, 0x55, 0x0e, 0x83,
    0xc2, 0xbd, 0x96, 0xfe,
};
static const unsigned char kat71_nonce[] = {
    0xde, 0xe6, 0x41, 0xf9, 0x17, 0x7c, 0x9d, 0xa4,
};
static const unsigned char kat71_persstr[] = {0};
static const unsigned char kat71_addin0[] = {0};
static const unsigned char kat71_addin1[] = {0};
static const unsigned char kat71_retbits[] = {
    0x05, 0xf8, 0xca, 0xd2, 0xd0, 0x31, 0xce, 0x6b, 0x52, 0x7e, 0xe0, 0x43,
    0xb5, 0x05, 0x6d, 0xb2, 0x0e, 0xe5, 0xca, 0x66, 0x0a, 0x95, 0xc2, 0x9b,
    0x8f, 0x8b, 0x2e, 0x43, 0xef, 0xa5, 0xc7, 0xaa, 0x62, 0x35, 0x12, 0x33,
    0xf4, 0xf1, 0x60, 0x36, 0x8c, 0x3d, 0xad, 0x4a, 0x74, 0xf4, 0x71, 0xbb,
    0x02, 0x15, 0x3f, 0x86, 0xc8, 0x7a, 0x14, 0x0a, 0x36, 0xe0, 0xf1, 0x22,
    0xc6, 0x8b, 0xa7, 0x7d,
};
static const struct drbg_kat_no_reseed kat71_t = {
    10, kat71_entropyin, kat71_nonce, kat71_persstr,
    kat71_addin0, kat71_addin1, kat71_retbits
};
static const struct drbg_kat kat71 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat71_t
};

static const unsigned char kat72_entropyin[] = {
    0x97, 0x90, 0x65, 0xbc, 0x1a, 0x1c, 0x8c, 0x82, 0x90, 0xdd, 0xe2, 0x19,
    0xf0, 0x12, 0x1a, 0xbe,
};
static const unsigned char kat72_nonce[] = {
    0x59, 0x3a, 0x0e, 0x1f, 0xa9, 0x85, 0x64, 0xa3,
};
static const unsigned char kat72_persstr[] = {0};
static const unsigned char kat72_addin0[] = {0};
static const unsigned char kat72_addin1[] = {0};
static const unsigned char kat72_retbits[] = {
    0x19, 0x9e, 0x68, 0xe5, 0xb6, 0x37, 0xa2, 0x7b, 0xe7, 0x94, 0x53, 0xd9,
    0x95, 0x3a, 0x0e, 0x52, 0x95, 0x36, 0x37, 0x4e, 0x7f, 0x63, 0xe8, 0x59,
    0x5a, 0xfe, 0x4b, 0x5d, 0x4e, 0x3b, 0xb3, 0x79, 0xc6, 0xd4, 0x62, 0xf6,
    0x43, 0x12, 0x44, 0xaf, 0x43, 0x36, 0xb1, 0x7f, 0x9d, 0xfd, 0xfe, 0xaf,
    0xa6, 0xf4, 0xff, 0xf5, 0x00, 0x4a, 0xa6, 0x5f, 0x7d, 0x6b, 0xb3, 0xb8,
    0x8a, 0xf8, 0xe9, 0xe2,
};
static const struct drbg_kat_no_reseed kat72_t = {
    11, kat72_entropyin, kat72_nonce, kat72_persstr,
    kat72_addin0, kat72_addin1, kat72_retbits
};
static const struct drbg_kat kat72 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat72_t
};

static const unsigned char kat73_entropyin[] = {
    0x89, 0xa7, 0x87, 0xbe, 0x64, 0xba, 0x33, 0x58, 0x95, 0x2f, 0x9d, 0x60,
    0x2b, 0xd8, 0xfa, 0xb4,
};
static const unsigned char kat73_nonce[] = {
    0xab, 0x96, 0x1b, 0x68, 0xc6, 0x58, 0x92, 0xd3,
};
static const unsigned char kat73_persstr[] = {0};
static const unsigned char kat73_addin0[] = {0};
static const unsigned char kat73_addin1[] = {0};
static const unsigned char kat73_retbits[] = {
    0x6e, 0x5d, 0xd2, 0xa6, 0x51, 0xf2, 0x61, 0x4f, 0xca, 0x88, 0x1f, 0x5e,
    0xdb, 0x05, 0xd5, 0xee, 0xc6, 0x12, 0x75, 0xdd, 0x05, 0xb3, 0x1f, 0xd8,
    0xa3, 0x06, 0x2f, 0xac, 0x51, 0x59, 0x60, 0xde, 0xf7, 0xb0, 0xd4, 0xdb,
    0x30, 0x4f, 0x64, 0x2b, 0x91, 0xe6, 0x40, 0x1f, 0x2a, 0x00, 0xb0, 0xfb,
    0x69, 0x3e, 0xdd, 0x73, 0x13, 0x10, 0x2f, 0x58, 0xd7, 0x33, 0x02, 0xea,
    0x8b, 0x5a, 0x9d, 0x7a,
};
static const struct drbg_kat_no_reseed kat73_t = {
    12, kat73_entropyin, kat73_nonce, kat73_persstr,
    kat73_addin0, kat73_addin1, kat73_retbits
};
static const struct drbg_kat kat73 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat73_t
};

static const unsigned char kat74_entropyin[] = {
    0x21, 0xf0, 0x2f, 0xc7, 0x5a, 0x28, 0x2d, 0x7f, 0x87, 0xca, 0xb0, 0x57,
    0x67, 0x55, 0x8e, 0x08,
};
static const unsigned char kat74_nonce[] = {
    0x3b, 0x43, 0xd7, 0x7b, 0x0c, 0x64, 0x71, 0xe9,
};
static const unsigned char kat74_persstr[] = {0};
static const unsigned char kat74_addin0[] = {0};
static const unsigned char kat74_addin1[] = {0};
static const unsigned char kat74_retbits[] = {
    0x3b, 0x87, 0x09, 0xce, 0xfe, 0x27, 0x06, 0x27, 0xeb, 0x49, 0x47, 0xf3,
    0xc7, 0x69, 0x5e, 0xed, 0x2c, 0x85, 0xcc, 0x15, 0x25, 0xbe, 0x25, 0x3f,
    0x20, 0x0a, 0x44, 0xad, 0x55, 0xf7, 0x58, 0x3e, 0x10, 0x91, 0xb8, 0x80,
    0xee, 0x16, 0xa8, 0x08, 0xe9, 0x0d, 0xc1, 0xdd, 0x5c, 0x0e, 0xd3, 0xb8,
    0xd1, 0x96, 0x40, 0x76, 0x61, 0xde, 0x80, 0x61, 0xf6, 0xa2, 0xd7, 0x8c,
    0xdc, 0x93, 0x42, 0x9d,
};
static const struct drbg_kat_no_reseed kat74_t = {
    13, kat74_entropyin, kat74_nonce, kat74_persstr,
    kat74_addin0, kat74_addin1, kat74_retbits
};
static const struct drbg_kat kat74 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat74_t
};

static const unsigned char kat75_entropyin[] = {
    0xdf, 0x5b, 0x1c, 0xd5, 0x10, 0x45, 0xc8, 0x38, 0x01, 0x53, 0x24, 0x49,
    0xd6, 0xa8, 0x7a, 0xed,
};
static const unsigned char kat75_nonce[] = {
    0x8d, 0xd1, 0x72, 0xb8, 0x12, 0x86, 0xa2, 0x38,
};
static const unsigned char kat75_persstr[] = {0};
static const unsigned char kat75_addin0[] = {0};
static const unsigned char kat75_addin1[] = {0};
static const unsigned char kat75_retbits[] = {
    0xf0, 0x17, 0x2b, 0x3a, 0x5b, 0x42, 0xea, 0xd6, 0xe6, 0x54, 0xd3, 0xab,
    0xc7, 0x8d, 0xda, 0x08, 0x78, 0x0e, 0xa4, 0x32, 0x23, 0x62, 0x57, 0x07,
    0xf8, 0xab, 0x02, 0x1b, 0xef, 0x54, 0x12, 0xdb, 0x89, 0xb0, 0x89, 0xcf,
    0x75, 0x11, 0x78, 0xeb, 0x1b, 0x97, 0xbe, 0x13, 0xf5, 0xe8, 0x3b, 0x9f,
    0x10, 0x76, 0xd0, 0xfc, 0x88, 0xe6, 0xbb, 0xd6, 0xee, 0xf1, 0xef, 0x63,
    0x55, 0xe3, 0x67, 0x2b,
};
static const struct drbg_kat_no_reseed kat75_t = {
    14, kat75_entropyin, kat75_nonce, kat75_persstr,
    kat75_addin0, kat75_addin1, kat75_retbits
};
static const struct drbg_kat kat75 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat75_t
};

static const unsigned char kat76_entropyin[] = {
    0xad, 0xf5, 0x71, 0x1f, 0x93, 0xd8, 0xc8, 0x99, 0x73, 0x49, 0x42, 0x9c,
    0xca, 0xed, 0xae, 0x0a,
};
static const unsigned char kat76_nonce[] = {
    0xb2, 0x57, 0x16, 0x93, 0x1b, 0x6e, 0x3c, 0xc1,
};
static const unsigned char kat76_persstr[] = {0};
static const unsigned char kat76_addin0[] = {
    0xab, 0xf8, 0xcd, 0x66, 0xdd, 0x39, 0x75, 0x8b, 0x01, 0xd7, 0xdb, 0xb9,
    0x9a, 0xb1, 0x7d, 0xc3,
};
static const unsigned char kat76_addin1[] = {
    0x4b, 0xe0, 0xf6, 0xb2, 0x75, 0x53, 0x77, 0xc6, 0xe8, 0x81, 0xfb, 0xb2,
    0x61, 0xb5, 0x6b, 0xeb,
};
static const unsigned char kat76_retbits[] = {
    0xd4, 0x20, 0x60, 0x4d, 0xee, 0x64, 0x67, 0x49, 0x2d, 0xb5, 0x95, 0x7c,
    0x86, 0x20, 0x7a, 0x70, 0x8f, 0xd2, 0x42, 0xed, 0x67, 0x94, 0x2a, 0xed,
    0x29, 0x94, 0x25, 0x33, 0x5c, 0x83, 0xb4, 0x14, 0x37, 0x41, 0x85, 0x82,
    0xf4, 0x1b, 0xc7, 0xfc, 0x0e, 0xf0, 0xd6, 0x92, 0x7f, 0x34, 0xd8, 0x3a,
    0xcd, 0x67, 0xc7, 0x01, 0x33, 0x64, 0x4f, 0xd7, 0x11, 0xdd, 0x5a, 0x65,
    0x73, 0x1f, 0x9f, 0x02,
};
static const struct drbg_kat_no_reseed kat76_t = {
    0, kat76_entropyin, kat76_nonce, kat76_persstr,
    kat76_addin0, kat76_addin1, kat76_retbits
};
static const struct drbg_kat kat76 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat76_t
};

static const unsigned char kat77_entropyin[] = {
    0xe9, 0x21, 0x00, 0x02, 0x07, 0x34, 0x90, 0x9d, 0x61, 0x09, 0xd2, 0x9c,
    0xf2, 0xc8, 0x98, 0xb0,
};
static const unsigned char kat77_nonce[] = {
    0x91, 0xfb, 0xd9, 0xa3, 0xa8, 0xed, 0xc3, 0xa4,
};
static const unsigned char kat77_persstr[] = {0};
static const unsigned char kat77_addin0[] = {
    0x09, 0x80, 0xc7, 0xdb, 0xcf, 0xd0, 0xe7, 0xf7, 0x7a, 0xf8, 0x35, 0xac,
    0xc7, 0x4a, 0x4b, 0xa3,
};
static const unsigned char kat77_addin1[] = {
    0x3c, 0xbf, 0x47, 0xd4, 0x66, 0x55, 0xa0, 0xce, 0x75, 0xa5, 0xd5, 0x65,
    0x28, 0x12, 0x7b, 0xc0,
};
static const unsigned char kat77_retbits[] = {
    0xe7, 0x7c, 0xcc, 0x52, 0x63, 0x62, 0x54, 0x29, 0xce, 0x7e, 0x53, 0x56,
    0x22, 0x48, 0x1d, 0x0b, 0xef, 0x47, 0x61, 0x41, 0xd1, 0xa0, 0xab, 0xfa,
    0x7f, 0x69, 0x67, 0xc1, 0x1f, 0x47, 0xf7, 0xa7, 0x0f, 0xa9, 0xbc, 0x70,
    0x59, 0x8e, 0xce, 0x3f, 0xf2, 0x83, 0xe5, 0xae, 0x04, 0xf1, 0x05, 0x35,
    0xd3, 0x49, 0xc2, 0x31, 0x34, 0x9a, 0xf8, 0x07, 0x1a, 0x22, 0x32, 0x3a,
    0x6e, 0x95, 0x10, 0x39,
};
static const struct drbg_kat_no_reseed kat77_t = {
    1, kat77_entropyin, kat77_nonce, kat77_persstr,
    kat77_addin0, kat77_addin1, kat77_retbits
};
static const struct drbg_kat kat77 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat77_t
};

static const unsigned char kat78_entropyin[] = {
    0x09, 0x2f, 0xfe, 0x74, 0x57, 0xd6, 0x17, 0x3e, 0x67, 0x13, 0xb0, 0xaf,
    0x76, 0x52, 0x1d, 0x82,
};
static const unsigned char kat78_nonce[] = {
    0xd0, 0xb8, 0xca, 0xe6, 0xab, 0x9e, 0x29, 0xac,
};
static const unsigned char kat78_persstr[] = {0};
static const unsigned char kat78_addin0[] = {
    0xe3, 0x1a, 0xf5, 0xe1, 0x81, 0xc4, 0x34, 0x6e, 0x42, 0x3e, 0xbc, 0x1f,
    0xf9, 0xf4, 0x6d, 0xdd,
};
static const unsigned char kat78_addin1[] = {
    0x02, 0xb9, 0x49, 0xd1, 0x34, 0xfc, 0x89, 0x68, 0xba, 0x51, 0x8f, 0xda,
    0x3b, 0x42, 0xc9, 0x7f,
};
static const unsigned char kat78_retbits[] = {
    0x0b, 0x64, 0xd9, 0xc6, 0x1d, 0x1f, 0x23, 0x44, 0x0c, 0xe5, 0x7d, 0xbf,
    0x46, 0x5f, 0x8c, 0x62, 0x9a, 0x9f, 0xfe, 0xdd, 0x16, 0x43, 0x99, 0xab,
    0x84, 0xfc, 0x11, 0x84, 0x66, 0xa5, 0x67, 0x9b, 0xb2, 0xd9, 0xc8, 0x05,
    0xea, 0x9d, 0x29, 0x49, 0x72, 0x61, 0xea, 0x58, 0xbd, 0x8f, 0xa4, 0x5c,
    0xf2, 0x4e, 0x21, 0x2c, 0x07, 0x7b, 0xab, 0xa4, 0xd4, 0xe5, 0x73, 0x8b,
    0xb8, 0x83, 0xc8, 0x83,
};
static const struct drbg_kat_no_reseed kat78_t = {
    2, kat78_entropyin, kat78_nonce, kat78_persstr,
    kat78_addin0, kat78_addin1, kat78_retbits
};
static const struct drbg_kat kat78 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat78_t
};

static const unsigned char kat79_entropyin[] = {
    0xa7, 0xa4, 0x78, 0x41, 0xcf, 0x1a, 0x75, 0xee, 0x96, 0x93, 0x5f, 0x52,
    0xa3, 0x32, 0x2d, 0x9d,
};
static const unsigned char kat79_nonce[] = {
    0x5b, 0x48, 0xc3, 0xc3, 0x47, 0x39, 0x46, 0x56,
};
static const unsigned char kat79_persstr[] = {0};
static const unsigned char kat79_addin0[] = {
    0xaa, 0xc2, 0x03, 0x1f, 0x6a, 0xa4, 0x0f, 0x08, 0x75, 0x49, 0x33, 0x70,
    0x92, 0x7c, 0x7c, 0xa8,
};
static const unsigned char kat79_addin1[] = {
    0x1a, 0x67, 0x44, 0xd0, 0x3f, 0xc2, 0x15, 0x12, 0x94, 0x70, 0x7c, 0x3b,
    0xce, 0xa8, 0x84, 0x02,
};
static const unsigned char kat79_retbits[] = {
    0xb4, 0x67, 0x48, 0x67, 0xcb, 0x6c, 0xd4, 0xfa, 0xde, 0x78, 0x46, 0xf4,
    0x92, 0xbd, 0x9e, 0xca, 0x69, 0xc1, 0x33, 0xf1, 0x17, 0x1c, 0xc6, 0x0b,
    0x6c, 0xd7, 0x5f, 0x22, 0xe2, 0xba, 0x7e, 0x70, 0x20, 0x7c, 0xa3, 0x8b,
    0xdc, 0xef, 0x17, 0xac, 0x15, 0xe4, 0x88, 0x49, 0xba, 0x13, 0x0d, 0x5f,
    0xb8, 0x10, 0x84, 0x4c, 0x25, 0x39, 0xfe, 0xc5, 0x19, 0xc4, 0xfa, 0x97,
    0x7f, 0xfa, 0x07, 0x38,
};
static const struct drbg_kat_no_reseed kat79_t = {
    3, kat79_entropyin, kat79_nonce, kat79_persstr,
    kat79_addin0, kat79_addin1, kat79_retbits
};
static const struct drbg_kat kat79 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat79_t
};

static const unsigned char kat80_entropyin[] = {
    0x52, 0x89, 0xc3, 0x33, 0xce, 0x1c, 0xee, 0xf5, 0x6a, 0x19, 0xa1, 0x11,
    0xe9, 0x20, 0x1a, 0x38,
};
static const unsigned char kat80_nonce[] = {
    0xdb, 0xd6, 0xb7, 0x6b, 0x1f, 0xdf, 0x5d, 0x1b,
};
static const unsigned char kat80_persstr[] = {0};
static const unsigned char kat80_addin0[] = {
    0xf1, 0xfa, 0x6e, 0xc8, 0xca, 0xf9, 0x4f, 0x65, 0xf1, 0x2e, 0x9c, 0x0d,
    0x8f, 0x57, 0x04, 0xf6,
};
static const unsigned char kat80_addin1[] = {
    0x6e, 0xb6, 0xa5, 0x57, 0xf1, 0x70, 0x18, 0x83, 0x0b, 0x5d, 0x72, 0xb6,
    0x7d, 0x32, 0x22, 0x63,
};
static const unsigned char kat80_retbits[] = {
    0x39, 0xec, 0x96, 0xbc, 0x71, 0x6e, 0x1d, 0x21, 0x48, 0x80, 0xd1, 0xf4,
    0xb8, 0xa9, 0x84, 0xbb, 0x83, 0x39, 0x94, 0x53, 0xec, 0x1e, 0xb9, 0xf3,
    0x71, 0x28, 0x2d, 0x65, 0x8e, 0x32, 0xd7, 0x39, 0xb4, 0x89, 0xf4, 0xc5,
    0xee, 0xae, 0x62, 0xa8, 0xfe, 0xe8, 0xe0, 0x84, 0xee, 0xae, 0x6d, 0x01,
    0xcc, 0xa0, 0xad, 0x80, 0xda, 0x1c, 0x53, 0x23, 0x0f, 0x8a, 0x1f, 0xaf,
    0x7c, 0x33, 0x5e, 0xdf,
};
static const struct drbg_kat_no_reseed kat80_t = {
    4, kat80_entropyin, kat80_nonce, kat80_persstr,
    kat80_addin0, kat80_addin1, kat80_retbits
};
static const struct drbg_kat kat80 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat80_t
};

static const unsigned char kat81_entropyin[] = {
    0x4f, 0x05, 0xef, 0x26, 0xc6, 0x85, 0x44, 0x4d, 0x6e, 0x65, 0xd8, 0xf2,
    0x6e, 0x0f, 0x2e, 0xf2,
};
static const unsigned char kat81_nonce[] = {
    0xd0, 0xab, 0x06, 0x6c, 0x7a, 0x49, 0x03, 0xb7,
};
static const unsigned char kat81_persstr[] = {0};
static const unsigned char kat81_addin0[] = {
    0xd2, 0x75, 0x42, 0x60, 0x43, 0xa8, 0x2d, 0xe2, 0x0f, 0x89, 0xf1, 0xcb,
    0x2c, 0xff, 0x5c, 0x9b,
};
static const unsigned char kat81_addin1[] = {
    0x02, 0xa3, 0x27, 0x63, 0xed, 0x9a, 0xab, 0x16, 0x72, 0x82, 0xd0, 0xa0,
    0x97, 0xf3, 0x10, 0x3e,
};
static const unsigned char kat81_retbits[] = {
    0xaf, 0xa2, 0xd3, 0x7c, 0xc1, 0xc7, 0x9d, 0x20, 0xa6, 0x10, 0x4d, 0x67,
    0x29, 0xfc, 0x75, 0x44, 0x36, 0x1b, 0x95, 0xd2, 0x36, 0x33, 0x48, 0x7e,
    0x8b, 0xb2, 0x82, 0xb2, 0x77, 0xe2, 0x5f, 0x8a, 0xd5, 0x31, 0x36, 0x10,
    0x9b, 0x21, 0xc3, 0x80, 0xaa, 0x05, 0xe4, 0x1d, 0xd9, 0xef, 0xd8, 0xad,
    0xf4, 0x6b, 0x7d, 0x69, 0x5b, 0xba, 0xf5, 0xb0, 0x30, 0x72, 0x16, 0xa1,
    0x94, 0xfd, 0x91, 0x28,
};
static const struct drbg_kat_no_reseed kat81_t = {
    5, kat81_entropyin, kat81_nonce, kat81_persstr,
    kat81_addin0, kat81_addin1, kat81_retbits
};
static const struct drbg_kat kat81 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat81_t
};

static const unsigned char kat82_entropyin[] = {
    0x0c, 0x2c, 0xcf, 0xd6, 0x5c, 0x23, 0xbc, 0x06, 0x93, 0xc9, 0xd4, 0x66,
    0x7a, 0x94, 0x2b, 0x42,
};
static const unsigned char kat82_nonce[] = {
    0xc0, 0x8d, 0x55, 0x22, 0x8f, 0xe8, 0x4b, 0xd7,
};
static const unsigned char kat82_persstr[] = {0};
static const unsigned char kat82_addin0[] = {
    0xcc, 0x01, 0x7b, 0xb7, 0x96, 0x2d, 0xa6, 0xdb, 0x0f, 0x51, 0x7b, 0xe9,
    0x18, 0x3f, 0xa6, 0xf4,
};
static const unsigned char kat82_addin1[] = {
    0xc4, 0x24, 0xaa, 0x0c, 0xe8, 0x4a, 0x80, 0x7a, 0x78, 0xb7, 0xb7, 0x32,
    0x11, 0x3c, 0x22, 0xa1,
};
static const unsigned char kat82_retbits[] = {
    0xcb, 0x55, 0x6f, 0x6d, 0x22, 0x3a, 0x51, 0xa2, 0x00, 0xd7, 0x35, 0xf2,
    0xf4, 0xba, 0x21, 0x7a, 0x67, 0x2e, 0xf6, 0xbc, 0x14, 0x51, 0x97, 0x6d,
    0x1d, 0x05, 0xb4, 0x4a, 0x2c, 0x0d, 0x54, 0x27, 0x8a, 0xf7, 0x47, 0x83,
    0x79, 0x22, 0xcd, 0x0b, 0x28, 0xd7, 0xaa, 0xa2, 0xa8, 0x7e, 0x23, 0x05,
    0x5e, 0x18, 0xd6, 0x7d, 0x76, 0xfa, 0x90, 0x72, 0x82, 0x94, 0xab, 0xa4,
    0xac, 0x36, 0x96, 0x5e,
};
static const struct drbg_kat_no_reseed kat82_t = {
    6, kat82_entropyin, kat82_nonce, kat82_persstr,
    kat82_addin0, kat82_addin1, kat82_retbits
};
static const struct drbg_kat kat82 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat82_t
};

static const unsigned char kat83_entropyin[] = {
    0x3b, 0x74, 0x81, 0x3e, 0xeb, 0xf5, 0xf5, 0xf2, 0xa6, 0xa5, 0xa4, 0xf3,
    0xf9, 0x07, 0xdc, 0xb8,
};
static const unsigned char kat83_nonce[] = {
    0x35, 0xda, 0xa2, 0xef, 0xda, 0xd8, 0x8f, 0xb8,
};
static const unsigned char kat83_persstr[] = {0};
static const unsigned char kat83_addin0[] = {
    0x5e, 0x0a, 0xe3, 0x69, 0xb6, 0xe8, 0x50, 0xed, 0x30, 0xf9, 0xae, 0x64,
    0xfe, 0x10, 0x5a, 0xf3,
};
static const unsigned char kat83_addin1[] = {
    0x3e, 0x69, 0x7b, 0xb2, 0xbf, 0x25, 0x85, 0x26, 0x45, 0xf2, 0xd1, 0x6f,
    0x67, 0x6f, 0x90, 0x78,
};
static const unsigned char kat83_retbits[] = {
    0x79, 0x0b, 0x1f, 0x6c, 0xcd, 0xf3, 0x52, 0xde, 0xc0, 0xe5, 0x7f, 0xa0,
    0xc7, 0x72, 0xc6, 0x4d, 0x77, 0x77, 0xe4, 0xbd, 0xb9, 0xff, 0x8a, 0x91,
    0xda, 0x01, 0x89, 0x02, 0x20, 0x3d, 0xb5, 0x82, 0xf9, 0x01, 0xc2, 0x0d,
    0xd2, 0x53, 0x61, 0xdf, 0x22, 0xe7, 0x17, 0x12, 0xa4, 0x97, 0x57, 0x8d,
    0x22, 0xee, 0xa8, 0x52, 0x46, 0xce, 0xfd, 0x42, 0x5b, 0x08, 0xf1, 0x50,
    0x74, 0x34, 0x1f, 0x39,
};
static const struct drbg_kat_no_reseed kat83_t = {
    7, kat83_entropyin, kat83_nonce, kat83_persstr,
    kat83_addin0, kat83_addin1, kat83_retbits
};
static const struct drbg_kat kat83 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat83_t
};

static const unsigned char kat84_entropyin[] = {
    0x46, 0x26, 0x07, 0xb2, 0xb7, 0x7b, 0x90, 0x3f, 0xdb, 0xd4, 0x2b, 0xf1,
    0xd7, 0x7b, 0xc8, 0x1a,
};
static const unsigned char kat84_nonce[] = {
    0xb5, 0x4d, 0x43, 0x00, 0x6a, 0xde, 0x2a, 0x7c,
};
static const unsigned char kat84_persstr[] = {0};
static const unsigned char kat84_addin0[] = {
    0x61, 0x05, 0xc5, 0xdd, 0xed, 0xcc, 0x74, 0x46, 0x65, 0x91, 0x5e, 0xbe,
    0x48, 0xad, 0x3b, 0x54,
};
static const unsigned char kat84_addin1[] = {
    0xfd, 0xde, 0xe2, 0x35, 0xc3, 0x7e, 0xd4, 0xbc, 0x21, 0xea, 0xe0, 0x8e,
    0x61, 0x53, 0x33, 0x1f,
};
static const unsigned char kat84_retbits[] = {
    0x16, 0x33, 0x9e, 0x91, 0x0e, 0xaf, 0x9b, 0xad, 0x90, 0x4a, 0x7b, 0x3f,
    0x9b, 0x2e, 0x02, 0x41, 0xd2, 0xb4, 0x66, 0x5d, 0x1f, 0x9c, 0x76, 0x55,
    0xf9, 0x21, 0x7e, 0x34, 0x25, 0x3a, 0xcc, 0x33, 0x0d, 0xfc, 0x98, 0x81,
    0x7d, 0x97, 0x81, 0x86, 0xf8, 0xad, 0x77, 0xb1, 0x0f, 0x38, 0xdd, 0x72,
    0xf4, 0x71, 0xb4, 0x61, 0x5b, 0x28, 0xf3, 0xa3, 0x9c, 0x40, 0x66, 0x23,
    0xea, 0x5d, 0x63, 0x77,
};
static const struct drbg_kat_no_reseed kat84_t = {
    8, kat84_entropyin, kat84_nonce, kat84_persstr,
    kat84_addin0, kat84_addin1, kat84_retbits
};
static const struct drbg_kat kat84 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat84_t
};

static const unsigned char kat85_entropyin[] = {
    0xba, 0x2e, 0xa9, 0x9b, 0x79, 0x2b, 0xde, 0xe8, 0xbd, 0xba, 0xf9, 0xc0,
    0x5a, 0xb7, 0xb6, 0x33,
};
static const unsigned char kat85_nonce[] = {
    0x44, 0x53, 0x34, 0x46, 0x44, 0xd4, 0x34, 0xa5,
};
static const unsigned char kat85_persstr[] = {0};
static const unsigned char kat85_addin0[] = {
    0x81, 0x72, 0x7d, 0x30, 0xd6, 0x4a, 0xe0, 0x7f, 0xf6, 0x15, 0x69, 0xfc,
    0x7a, 0x35, 0x0c, 0xcc,
};
static const unsigned char kat85_addin1[] = {
    0x65, 0x56, 0xf6, 0x2f, 0x75, 0xd5, 0x26, 0x84, 0x93, 0xcc, 0xa5, 0x12,
    0x01, 0x14, 0x93, 0x4c,
};
static const unsigned char kat85_retbits[] = {
    0x2a, 0x3b, 0x39, 0x87, 0x02, 0xbc, 0x5f, 0xff, 0xd9, 0x14, 0xf9, 0x16,
    0x72, 0xaa, 0x05, 0xbe, 0xb6, 0x87, 0x82, 0xe0, 0xa6, 0x22, 0xf6, 0x7b,
    0x21, 0x82, 0x6c, 0x23, 0x32, 0x6c, 0x50, 0x78, 0xcc, 0xda, 0x6a, 0xa3,
    0xdb, 0x01, 0x4c, 0xec, 0x56, 0xa0, 0x65, 0x63, 0x6b, 0x80, 0x57, 0xe8,
    0x12, 0xa6, 0xf4, 0xb7, 0xc6, 0xc4, 0xde, 0x6e, 0x9f, 0x06, 0xaa, 0x53,
    0x77, 0xb2, 0x8b, 0xe6,
};
static const struct drbg_kat_no_reseed kat85_t = {
    9, kat85_entropyin, kat85_nonce, kat85_persstr,
    kat85_addin0, kat85_addin1, kat85_retbits
};
static const struct drbg_kat kat85 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat85_t
};

static const unsigned char kat86_entropyin[] = {
    0x61, 0xcf, 0x4e, 0x09, 0xd9, 0xa4, 0x6a, 0xf3, 0x18, 0xc7, 0x06, 0x42,
    0x5c, 0xd3, 0x50, 0xfb,
};
static const unsigned char kat86_nonce[] = {
    0x28, 0xec, 0x9c, 0x4a, 0xe9, 0x8f, 0x05, 0x74,
};
static const unsigned char kat86_persstr[] = {0};
static const unsigned char kat86_addin0[] = {
    0x7f, 0xc3, 0xbf, 0x8a, 0xd3, 0x3e, 0x59, 0x61, 0x74, 0x5b, 0xb2, 0xae,
    0x7c, 0x39, 0x9b, 0xa0,
};
static const unsigned char kat86_addin1[] = {
    0x31, 0x35, 0x71, 0xd3, 0x0d, 0xc5, 0x90, 0x9a, 0x03, 0x38, 0x8c, 0x69,
    0x3f, 0x4e, 0xf9, 0x4c,
};
static const unsigned char kat86_retbits[] = {
    0xd2, 0x25, 0xf0, 0xc6, 0x5e, 0x9a, 0xe8, 0xc3, 0x64, 0x93, 0xb0, 0x09,
    0x47, 0x6f, 0x5a, 0x56, 0x36, 0xd4, 0x52, 0x54, 0x4c, 0xe1, 0x6b, 0x99,
    0x1c, 0x73, 0xd6, 0xea, 0xe5, 0xef, 0x5e, 0xff, 0x3b, 0x74, 0xe0, 0x03,
    0x28, 0xf8, 0x7f, 0x20, 0x30, 0xf3, 0xdf, 0x53, 0xb2, 0xfc, 0x26, 0x47,
    0xd6, 0x3a, 0x28, 0xe5, 0xc7, 0xb5, 0x83, 0x44, 0xed, 0x99, 0xc5, 0xd0,
    0xbc, 0x62, 0x93, 0xb4,
};
static const struct drbg_kat_no_reseed kat86_t = {
    10, kat86_entropyin, kat86_nonce, kat86_persstr,
    kat86_addin0, kat86_addin1, kat86_retbits
};
static const struct drbg_kat kat86 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat86_t
};

static const unsigned char kat87_entropyin[] = {
    0xdd, 0x6c, 0x7d, 0xba, 0xc4, 0xfc, 0x4c, 0x8d, 0x9c, 0x69, 0xce, 0x4d,
    0xcf, 0x43, 0xb6, 0x67,
};
static const unsigned char kat87_nonce[] = {
    0x83, 0x26, 0x16, 0x1c, 0x2a, 0xac, 0x48, 0x06,
};
static const unsigned char kat87_persstr[] = {0};
static const unsigned char kat87_addin0[] = {
    0xcf, 0x95, 0x9f, 0x9e, 0x93, 0x89, 0x09, 0xd9, 0x5c, 0x47, 0x25, 0xdc,
    0x39, 0xd7, 0xef, 0xd7,
};
static const unsigned char kat87_addin1[] = {
    0xe4, 0x37, 0xd2, 0x26, 0xc4, 0x02, 0x68, 0xcb, 0x9f, 0x63, 0x94, 0x5c,
    0xbc, 0xe5, 0x4b, 0x2d,
};
static const unsigned char kat87_retbits[] = {
    0x90, 0xdd, 0xc5, 0x3c, 0x48, 0x04, 0x1f, 0x0c, 0x45, 0x01, 0xff, 0xee,
    0x61, 0x7b, 0x3a, 0x35, 0x83, 0xcb, 0xf5, 0x1a, 0x4c, 0x45, 0xad, 0x7b,
    0xa5, 0xb3, 0xdd, 0x34, 0x65, 0xff, 0x13, 0x92, 0xbd, 0x2e, 0xb4, 0x65,
    0x1a, 0xc2, 0x2a, 0x34, 0xa0, 0xf5, 0x37, 0x72, 0x01, 0x27, 0x12, 0xf2,
    0x39, 0x31, 0xba, 0x8f, 0x0a, 0x3b, 0x74, 0xfa, 0xe0, 0xa1, 0x6b, 0xd4,
    0xa1, 0x88, 0x35, 0x13,
};
static const struct drbg_kat_no_reseed kat87_t = {
    11, kat87_entropyin, kat87_nonce, kat87_persstr,
    kat87_addin0, kat87_addin1, kat87_retbits
};
static const struct drbg_kat kat87 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat87_t
};

static const unsigned char kat88_entropyin[] = {
    0xdf, 0x4f, 0xef, 0xc1, 0xc3, 0xd4, 0x03, 0xfb, 0x17, 0x9b, 0x9b, 0x95,
    0x77, 0x53, 0x81, 0xe3,
};
static const unsigned char kat88_nonce[] = {
    0xd6, 0x84, 0x83, 0xae, 0xe4, 0x2c, 0xfc, 0x00,
};
static const unsigned char kat88_persstr[] = {0};
static const unsigned char kat88_addin0[] = {
    0xe8, 0x77, 0xeb, 0x37, 0x51, 0xe2, 0xba, 0xfe, 0xa0, 0x33, 0xac, 0x8c,
    0xc5, 0xba, 0x66, 0xf6,
};
static const unsigned char kat88_addin1[] = {
    0x79, 0x56, 0x87, 0x40, 0x22, 0xb5, 0x0a, 0xa0, 0x32, 0xba, 0xfa, 0x29,
    0xe7, 0x21, 0xf3, 0x56,
};
static const unsigned char kat88_retbits[] = {
    0x6a, 0xda, 0x06, 0x20, 0xd4, 0x29, 0xcd, 0x38, 0x75, 0x4e, 0x55, 0x9a,
    0x2e, 0xb8, 0x8c, 0x08, 0x26, 0x6c, 0xd9, 0xcb, 0xac, 0x24, 0x48, 0xb2,
    0xb8, 0xa0, 0x5d, 0x92, 0x7d, 0xa1, 0x72, 0x59, 0x19, 0x9d, 0xbc, 0xc5,
    0xeb, 0x0b, 0x7a, 0xde, 0xe6, 0x1c, 0x60, 0x74, 0xae, 0x2e, 0x7a, 0xf6,
    0xb2, 0xc9, 0xb1, 0xfd, 0x52, 0x89, 0xfb, 0xc3, 0xd9, 0xbe, 0x30, 0x85,
    0x84, 0xd0, 0xae, 0x4a,
};
static const struct drbg_kat_no_reseed kat88_t = {
    12, kat88_entropyin, kat88_nonce, kat88_persstr,
    kat88_addin0, kat88_addin1, kat88_retbits
};
static const struct drbg_kat kat88 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat88_t
};

static const unsigned char kat89_entropyin[] = {
    0x62, 0x0e, 0xa3, 0x17, 0x34, 0x4d, 0x86, 0x85, 0x8d, 0x64, 0xf6, 0xbb,
    0x05, 0xd4, 0x47, 0x97,
};
static const unsigned char kat89_nonce[] = {
    0xf6, 0xdd, 0xc5, 0x84, 0x4f, 0xbd, 0x73, 0x9e,
};
static const unsigned char kat89_persstr[] = {0};
static const unsigned char kat89_addin0[] = {
    0x3c, 0xcf, 0xb0, 0x69, 0xef, 0x27, 0xf8, 0x64, 0xe3, 0xb7, 0x31, 0xd9,
    0x4c, 0x3f, 0x2c, 0x6c,
};
static const unsigned char kat89_addin1[] = {
    0x80, 0x85, 0x7d, 0x3f, 0x59, 0x8b, 0x1b, 0x16, 0x01, 0x4e, 0x6a, 0x35,
    0x49, 0xa2, 0xd2, 0x2c,
};
static const unsigned char kat89_retbits[] = {
    0xd7, 0xa7, 0x87, 0xbe, 0xeb, 0x7e, 0xfc, 0x9c, 0x58, 0x4f, 0x93, 0x34,
    0x55, 0x33, 0xa5, 0x99, 0xed, 0x61, 0x5e, 0xb2, 0x13, 0x6b, 0x8f, 0xb8,
    0x74, 0xa7, 0xf5, 0xa7, 0x1a, 0xea, 0x98, 0xf8, 0x0c, 0x56, 0x92, 0x30,
    0x98, 0x56, 0x7a, 0xa3, 0xb4, 0xc3, 0x16, 0x96, 0x82, 0x6f, 0x08, 0x16,
    0x76, 0xa3, 0xce, 0xed, 0x1d, 0x91, 0x4c, 0x94, 0x9d, 0x54, 0x23, 0x90,
    0xb6, 0x99, 0xd3, 0xfe,
};
static const struct drbg_kat_no_reseed kat89_t = {
    13, kat89_entropyin, kat89_nonce, kat89_persstr,
    kat89_addin0, kat89_addin1, kat89_retbits
};
static const struct drbg_kat kat89 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat89_t
};

static const unsigned char kat90_entropyin[] = {
    0xd8, 0xba, 0x9d, 0x16, 0xa0, 0x07, 0x0f, 0xbd, 0xf5, 0x10, 0x9d, 0x3c,
    0x1e, 0xd8, 0x59, 0x16,
};
static const unsigned char kat90_nonce[] = {
    0x29, 0xce, 0x07, 0xdd, 0xa6, 0x1b, 0x50, 0x14,
};
static const unsigned char kat90_persstr[] = {0};
static const unsigned char kat90_addin0[] = {
    0x16, 0x0a, 0xd7, 0x76, 0x75, 0xc6, 0x4c, 0x74, 0x87, 0xee, 0xdb, 0x34,
    0x34, 0x37, 0x00, 0xc0,
};
static const unsigned char kat90_addin1[] = {
    0x16, 0x32, 0x54, 0x60, 0x58, 0xf1, 0x9e, 0xe1, 0xb8, 0x49, 0xe5, 0xd2,
    0x7e, 0x2d, 0x75, 0x5a,
};
static const unsigned char kat90_retbits[] = {
    0xe0, 0x5c, 0x62, 0xcb, 0xef, 0xaf, 0x03, 0x80, 0x43, 0xd4, 0x9c, 0x97,
    0x6b, 0xa8, 0x55, 0x21, 0xc2, 0x06, 0x71, 0xd9, 0xf8, 0x1c, 0xeb, 0xf5,
    0xa8, 0x75, 0x81, 0x65, 0x9e, 0x3d, 0x18, 0x19, 0x90, 0x8e, 0x84, 0xec,
    0x01, 0x93, 0xa5, 0x30, 0xf3, 0x10, 0xcc, 0x35, 0x44, 0xba, 0x90, 0xad,
    0x92, 0xa6, 0xc9, 0x9d, 0x3a, 0x1a, 0xf5, 0x5b, 0x5a, 0xed, 0x72, 0x87,
    0x1d, 0xb8, 0x90, 0x02,
};
static const struct drbg_kat_no_reseed kat90_t = {
    14, kat90_entropyin, kat90_nonce, kat90_persstr,
    kat90_addin0, kat90_addin1, kat90_retbits
};
static const struct drbg_kat kat90 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat90_t
};

static const unsigned char kat91_entropyin[] = {
    0x0a, 0xf1, 0x3f, 0x64, 0x59, 0x02, 0xaf, 0x49, 0xe9, 0xa7, 0xce, 0xd6,
    0xe3, 0x6a, 0x21, 0x0d,
};
static const unsigned char kat91_nonce[] = {
    0xc3, 0xbf, 0xf2, 0x91, 0xa1, 0x1a, 0xc4, 0x97,
};
static const unsigned char kat91_persstr[] = {
    0xe8, 0xf1, 0xd1, 0xb4, 0x73, 0x1c, 0x4d, 0x57, 0xd7, 0xea, 0xd9, 0xc2,
    0xf6, 0x00, 0xfd, 0xc6,
};
static const unsigned char kat91_addin0[] = {0};
static const unsigned char kat91_addin1[] = {0};
static const unsigned char kat91_retbits[] = {
    0xac, 0x6f, 0x94, 0x5a, 0x4b, 0x9f, 0xd3, 0xb4, 0x7c, 0x74, 0x37, 0x9e,
    0xb1, 0xf2, 0xa7, 0xbb, 0xed, 0xf8, 0xee, 0xc6, 0x8e, 0xfd, 0x3c, 0x7a,
    0x6c, 0xf6, 0x8c, 0x54, 0xae, 0x7a, 0x3f, 0x7b, 0xe7, 0x28, 0x0f, 0x45,
    0x9c, 0x2e, 0x0b, 0x72, 0xaf, 0xa4, 0x5c, 0xeb, 0xbe, 0xbb, 0xa1, 0x7c,
    0x86, 0x7e, 0x96, 0x11, 0xc8, 0x96, 0xa5, 0x7d, 0x51, 0x5b, 0xeb, 0x06,
    0xa7, 0xb9, 0x1f, 0x4c,
};
static const struct drbg_kat_no_reseed kat91_t = {
    0, kat91_entropyin, kat91_nonce, kat91_persstr,
    kat91_addin0, kat91_addin1, kat91_retbits
};
static const struct drbg_kat kat91 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat91_t
};

static const unsigned char kat92_entropyin[] = {
    0x0e, 0x11, 0x3f, 0x47, 0xf2, 0xfc, 0x76, 0xe8, 0x3e, 0x2d, 0x13, 0xd5,
    0x72, 0x24, 0x56, 0x08,
};
static const unsigned char kat92_nonce[] = {
    0x5f, 0xf8, 0x5c, 0xc6, 0xa5, 0x34, 0xf1, 0x5a,
};
static const unsigned char kat92_persstr[] = {
    0x50, 0x25, 0x06, 0x68, 0xe5, 0x9d, 0xe3, 0x5f, 0xde, 0x91, 0xe0, 0x8f,
    0xe1, 0x84, 0x84, 0xab,
};
static const unsigned char kat92_addin0[] = {0};
static const unsigned char kat92_addin1[] = {0};
static const unsigned char kat92_retbits[] = {
    0xdc, 0xc6, 0x4a, 0x96, 0x6a, 0x52, 0xd6, 0x00, 0x8d, 0xbe, 0x07, 0xa2,
    0x48, 0x4b, 0xca, 0xad, 0x67, 0xb2, 0x54, 0xd6, 0xf2, 0x46, 0xe4, 0x50,
    0x1d, 0x98, 0x64, 0xb6, 0x4a, 0xd8, 0xb7, 0xed, 0xf1, 0x0f, 0xdb, 0xc6,
    0xdd, 0xc4, 0x14, 0xa9, 0xb4, 0x31, 0xb0, 0x58, 0xa7, 0xee, 0x5c, 0xed,
    0x23, 0xf7, 0xa6, 0xac, 0x7e, 0xea, 0x0f, 0xe6, 0x13, 0x1c, 0x9e, 0xb7,
    0x41, 0x2e, 0x68, 0xdf,
};
static const struct drbg_kat_no_reseed kat92_t = {
    1, kat92_entropyin, kat92_nonce, kat92_persstr,
    kat92_addin0, kat92_addin1, kat92_retbits
};
static const struct drbg_kat kat92 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat92_t
};

static const unsigned char kat93_entropyin[] = {
    0x16, 0xd2, 0xa2, 0x33, 0xf0, 0x49, 0x7c, 0x75, 0x1d, 0xdd, 0x85, 0xef,
    0x7c, 0xd8, 0x62, 0xe6,
};
static const unsigned char kat93_nonce[] = {
    0x7e, 0x7e, 0x1a, 0xab, 0x1b, 0x82, 0x67, 0x5d,
};
static const unsigned char kat93_persstr[] = {
    0x15, 0xf8, 0x1e, 0x40, 0xa2, 0x49, 0x3b, 0x85, 0x9c, 0xce, 0xd3, 0x3c,
    0xdf, 0x7b, 0x75, 0x90,
};
static const unsigned char kat93_addin0[] = {0};
static const unsigned char kat93_addin1[] = {0};
static const unsigned char kat93_retbits[] = {
    0xc5, 0x20, 0x97, 0xd2, 0xd0, 0x09, 0xdc, 0xe7, 0xcb, 0xd4, 0x6d, 0x74,
    0x0d, 0xe5, 0x4d, 0x70, 0xc0, 0x73, 0x2f, 0x96, 0xfa, 0xc0, 0xbd, 0x16,
    0x9f, 0x38, 0x56, 0xe5, 0x0f, 0x9e, 0x46, 0x01, 0x67, 0x25, 0x38, 0xb2,
    0x3f, 0x37, 0x1f, 0x78, 0x0e, 0x61, 0xe8, 0x2f, 0x2d, 0xb9, 0x9e, 0xf3,
    0x83, 0x40, 0x08, 0xaa, 0xdc, 0x26, 0xc2, 0x78, 0x55, 0xa6, 0x86, 0x96,
    0xee, 0x81, 0x2c, 0x9c,
};
static const struct drbg_kat_no_reseed kat93_t = {
    2, kat93_entropyin, kat93_nonce, kat93_persstr,
    kat93_addin0, kat93_addin1, kat93_retbits
};
static const struct drbg_kat kat93 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat93_t
};

static const unsigned char kat94_entropyin[] = {
    0xea, 0x36, 0x89, 0x1e, 0x0d, 0x64, 0x26, 0x94, 0x5a, 0x6e, 0x1f, 0x33,
    0x8c, 0x86, 0xca, 0x4b,
};
static const unsigned char kat94_nonce[] = {
    0x89, 0x69, 0x66, 0x60, 0x0e, 0x85, 0xd3, 0xda,
};
static const unsigned char kat94_persstr[] = {
    0xb8, 0xc7, 0x01, 0xa1, 0x1e, 0x7c, 0x00, 0x8e, 0x37, 0xc2, 0x32, 0x62,
    0x7c, 0x24, 0xb6, 0xd1,
};
static const unsigned char kat94_addin0[] = {0};
static const unsigned char kat94_addin1[] = {0};
static const unsigned char kat94_retbits[] = {
    0x44, 0x63, 0x60, 0xf1, 0x6b, 0x1e, 0x59, 0xa8, 0x25, 0x96, 0xe6, 0x1d,
    0xf2, 0x3a, 0xf1, 0x8f, 0x5a, 0x95, 0xb7, 0xd8, 0xb7, 0xdd, 0x61, 0x72,
    0xf3, 0x57, 0x5d, 0x45, 0x25, 0x13, 0x86, 0xb0, 0x4b, 0x69, 0x78, 0x5b,
    0xa4, 0x4a, 0x75, 0xeb, 0x2a, 0x54, 0x15, 0x71, 0x3a, 0x84, 0x36, 0x5c,
    0xac, 0x81, 0xb3, 0x0d, 0xf0, 0xb4, 0x8a, 0x61, 0x09, 0x77, 0xfe, 0x69,
    0x61, 0xe7, 0xee, 0x8a,
};
static const struct drbg_kat_no_reseed kat94_t = {
    3, kat94_entropyin, kat94_nonce, kat94_persstr,
    kat94_addin0, kat94_addin1, kat94_retbits
};
static const struct drbg_kat kat94 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat94_t
};

static const unsigned char kat95_entropyin[] = {
    0x3f, 0x41, 0x38, 0xd8, 0x4d, 0x7a, 0xbd, 0x13, 0xbb, 0xe1, 0x5e, 0x18,
    0xb2, 0x92, 0xf1, 0x5f,
};
static const unsigned char kat95_nonce[] = {
    0x34, 0x62, 0xe5, 0x2e, 0x82, 0xf2, 0x86, 0x53,
};
static const unsigned char kat95_persstr[] = {
    0x24, 0xc5, 0x17, 0x70, 0x84, 0x95, 0x1c, 0xf5, 0x32, 0xf2, 0x43, 0x2a,
    0x18, 0x82, 0x59, 0x6f,
};
static const unsigned char kat95_addin0[] = {0};
static const unsigned char kat95_addin1[] = {0};
static const unsigned char kat95_retbits[] = {
    0xba, 0x2c, 0xa1, 0xe4, 0x43, 0xc7, 0x5d, 0x55, 0xc4, 0x20, 0x65, 0xba,
    0x91, 0xf7, 0x7d, 0xb5, 0xb8, 0xde, 0x6c, 0x34, 0x2b, 0x65, 0xa2, 0xc1,
    0x49, 0x44, 0x59, 0x79, 0xff, 0x26, 0x3e, 0x9d, 0x01, 0x8f, 0x56, 0x44,
    0x30, 0xfe, 0x1e, 0xdb, 0x48, 0xb4, 0x03, 0xce, 0xf2, 0x3f, 0x86, 0x0e,
    0xa2, 0x71, 0x46, 0xd8, 0x51, 0x18, 0x43, 0xe0, 0xa3, 0x9f, 0xf3, 0x37,
    0xba, 0x74, 0xd2, 0x21,
};
static const struct drbg_kat_no_reseed kat95_t = {
    4, kat95_entropyin, kat95_nonce, kat95_persstr,
    kat95_addin0, kat95_addin1, kat95_retbits
};
static const struct drbg_kat kat95 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat95_t
};

static const unsigned char kat96_entropyin[] = {
    0xa6, 0xae, 0xf8, 0x0f, 0xab, 0x61, 0xe2, 0x7e, 0xf6, 0x11, 0xb8, 0x6f,
    0x92, 0x4e, 0x97, 0x6c,
};
static const unsigned char kat96_nonce[] = {
    0x74, 0x27, 0x8f, 0x50, 0x23, 0xc2, 0x55, 0x2c,
};
static const unsigned char kat96_persstr[] = {
    0xa3, 0x85, 0x16, 0x79, 0xe1, 0x12, 0x94, 0x82, 0x01, 0x39, 0x6d, 0x98,
    0xd6, 0xbe, 0x5c, 0x13,
};
static const unsigned char kat96_addin0[] = {0};
static const unsigned char kat96_addin1[] = {0};
static const unsigned char kat96_retbits[] = {
    0x68, 0x57, 0x26, 0x82, 0x7f, 0x3d, 0x38, 0x42, 0x7d, 0x07, 0xdf, 0xcf,
    0xb6, 0x4a, 0x7f, 0x95, 0xb6, 0xf0, 0x6e, 0xf0, 0x1c, 0xed, 0x28, 0x1d,
    0xce, 0x7f, 0x41, 0x30, 0x34, 0x12, 0x36, 0x11, 0x24, 0x36, 0x2b, 0xa5,
    0x1c, 0xe8, 0xd2, 0x55, 0x2a, 0xf0, 0x54, 0xa2, 0x0f, 0xda, 0xc9, 0x4e,
    0x51, 0x69, 0x16, 0x5b, 0x71, 0x52, 0x67, 0x63, 0x8c, 0x1e, 0x0b, 0x9b,
    0x34, 0x52, 0xb5, 0xb7,
};
static const struct drbg_kat_no_reseed kat96_t = {
    5, kat96_entropyin, kat96_nonce, kat96_persstr,
    kat96_addin0, kat96_addin1, kat96_retbits
};
static const struct drbg_kat kat96 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat96_t
};

static const unsigned char kat97_entropyin[] = {
    0x01, 0x64, 0xae, 0x53, 0xe5, 0x33, 0xf2, 0xf1, 0x22, 0x2d, 0x4a, 0xa2,
    0x72, 0x76, 0x81, 0x4c,
};
static const unsigned char kat97_nonce[] = {
    0x26, 0x05, 0xf5, 0x89, 0xcb, 0x1b, 0xea, 0x9c,
};
static const unsigned char kat97_persstr[] = {
    0x9d, 0xe7, 0x03, 0xa5, 0xe2, 0xbb, 0xf2, 0xe5, 0x3d, 0x0f, 0xe3, 0xb5,
    0x73, 0xf5, 0x2a, 0xcf,
};
static const unsigned char kat97_addin0[] = {0};
static const unsigned char kat97_addin1[] = {0};
static const unsigned char kat97_retbits[] = {
    0xfd, 0x7f, 0x64, 0x62, 0x2e, 0x87, 0xde, 0x3b, 0x66, 0xe1, 0x99, 0x30,
    0xcf, 0x4c, 0xcd, 0xc3, 0xe6, 0x72, 0xf5, 0xa4, 0xde, 0xf0, 0xda, 0xdd,
    0xb3, 0x71, 0x07, 0x43, 0x0e, 0xc7, 0xf6, 0x91, 0xc6, 0x32, 0x15, 0x45,
    0x10, 0x3c, 0x2a, 0x61, 0x4d, 0x7f, 0x0f, 0x33, 0x17, 0x3b, 0xde, 0xc4,
    0x20, 0x58, 0x58, 0x22, 0xcd, 0xa8, 0xc2, 0x72, 0x20, 0x19, 0x22, 0xb6,
    0x02, 0xf5, 0xd8, 0xa7,
};
static const struct drbg_kat_no_reseed kat97_t = {
    6, kat97_entropyin, kat97_nonce, kat97_persstr,
    kat97_addin0, kat97_addin1, kat97_retbits
};
static const struct drbg_kat kat97 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat97_t
};

static const unsigned char kat98_entropyin[] = {
    0x29, 0xdc, 0x8b, 0xad, 0x50, 0x0c, 0xce, 0x29, 0x5b, 0x9e, 0x86, 0xb2,
    0x15, 0x9c, 0x6d, 0xcd,
};
static const unsigned char kat98_nonce[] = {
    0xb6, 0x9a, 0x5a, 0x77, 0x69, 0x4f, 0xc4, 0xa0,
};
static const unsigned char kat98_persstr[] = {
    0xc5, 0x3b, 0x13, 0xaf, 0xbf, 0x21, 0xf1, 0x7e, 0xe9, 0xf4, 0xdd, 0x7c,
    0x09, 0x93, 0xe0, 0xf9,
};
static const unsigned char kat98_addin0[] = {0};
static const unsigned char kat98_addin1[] = {0};
static const unsigned char kat98_retbits[] = {
    0x08, 0xc9, 0x7d, 0x63, 0x8b, 0x5d, 0x82, 0x59, 0x93, 0x10, 0x32, 0x5d,
    0x3b, 0x1e, 0x63, 0x27, 0x74, 0x1c, 0xd9, 0x18, 0x87, 0xd6, 0x58, 0xb5,
    0xd8, 0x15, 0xee, 0xed, 0x7a, 0x72, 0xab, 0x86, 0xe2, 0xfa, 0x17, 0xe9,
    0xd0, 0x13, 0xe1, 0x7a, 0x92, 0x14, 0xd6, 0x39, 0x6a, 0xc1, 0x48, 0x1f,
    0x76, 0x88, 0x30, 0xa4, 0xd8, 0xc4, 0xfc, 0x39, 0x20, 0x12, 0xe9, 0x07,
    0x64, 0x34, 0x93, 0x9f,
};
static const struct drbg_kat_no_reseed kat98_t = {
    7, kat98_entropyin, kat98_nonce, kat98_persstr,
    kat98_addin0, kat98_addin1, kat98_retbits
};
static const struct drbg_kat kat98 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat98_t
};

static const unsigned char kat99_entropyin[] = {
    0xb3, 0xae, 0xe2, 0xcf, 0x80, 0xe3, 0xd0, 0x5a, 0xed, 0xa0, 0xe0, 0x1c,
    0xf6, 0x5e, 0xa9, 0x65,
};
static const unsigned char kat99_nonce[] = {
    0x24, 0xb5, 0x8b, 0x7c, 0x6a, 0x99, 0xeb, 0x2a,
};
static const unsigned char kat99_persstr[] = {
    0x73, 0x44, 0x04, 0x3c, 0x84, 0x91, 0x5b, 0xcd, 0xd6, 0xd8, 0x1f, 0x3f,
    0xe2, 0x3b, 0xa2, 0x73,
};
static const unsigned char kat99_addin0[] = {0};
static const unsigned char kat99_addin1[] = {0};
static const unsigned char kat99_retbits[] = {
    0xbe, 0x35, 0x3c, 0xf8, 0x32, 0xe8, 0x46, 0x44, 0x4a, 0xf0, 0x01, 0x5e,
    0xd0, 0x5d, 0x54, 0xb7, 0x2c, 0x9a, 0x1a, 0x02, 0x31, 0xf2, 0xa4, 0xc4,
    0x75, 0x61, 0x17, 0x46, 0xef, 0x86, 0x10, 0x29, 0xe1, 0x88, 0x42, 0xc1,
    0xb7, 0x61, 0x48, 0xd2, 0xa3, 0xeb, 0x01, 0x74, 0xa3, 0x0c, 0xad, 0x81,
    0xb5, 0x58, 0xd0, 0xd1, 0xe0, 0x1a, 0x8c, 0x7d, 0xd5, 0x10, 0xf0, 0xb5,
    0x6f, 0x1f, 0xdd, 0xc3,
};
static const struct drbg_kat_no_reseed kat99_t = {
    8, kat99_entropyin, kat99_nonce, kat99_persstr,
    kat99_addin0, kat99_addin1, kat99_retbits
};
static const struct drbg_kat kat99 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat99_t
};

static const unsigned char kat100_entropyin[] = {
    0xa5, 0x25, 0xf6, 0x91, 0x65, 0xc4, 0x4f, 0x3a, 0x8a, 0xf8, 0x08, 0x57,
    0xa5, 0xe8, 0x74, 0xb0,
};
static const unsigned char kat100_nonce[] = {
    0x0c, 0x80, 0x6a, 0x40, 0xc1, 0xdc, 0xb7, 0xcc,
};
static const unsigned char kat100_persstr[] = {
    0x43, 0x12, 0xc2, 0x30, 0x9a, 0x1d, 0xfe, 0x2d, 0x9a, 0x66, 0xf5, 0xc1,
    0x8f, 0xc4, 0x0b, 0x87,
};
static const unsigned char kat100_addin0[] = {0};
static const unsigned char kat100_addin1[] = {0};
static const unsigned char kat100_retbits[] = {
    0xd0, 0xb4, 0x6f, 0x7e, 0x18, 0x77, 0xdb, 0x24, 0x4f, 0xf4, 0xe0, 0xae,
    0xb6, 0x4c, 0x4e, 0xfa, 0x3e, 0x8c, 0x2a, 0x86, 0x62, 0x41, 0x5c, 0xf5,
    0xba, 0x67, 0xf2, 0xdc, 0x91, 0xe8, 0x23, 0x10, 0xa6, 0xa9, 0x33, 0xc4,
    0xdf, 0x6a, 0x40, 0x2f, 0x8f, 0xae, 0x27, 0x0b, 0x22, 0x9b, 0xf2, 0x54,
    0xcd, 0x35, 0x94, 0x3a, 0x13, 0x34, 0x09, 0x79, 0x37, 0x66, 0x69, 0x11,
    0x9e, 0x0e, 0x5c, 0xf8,
};
static const struct drbg_kat_no_reseed kat100_t = {
    9, kat100_entropyin, kat100_nonce, kat100_persstr,
    kat100_addin0, kat100_addin1, kat100_retbits
};
static const struct drbg_kat kat100 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat100_t
};

static const unsigned char kat101_entropyin[] = {
    0x06, 0x08, 0xdc, 0x6c, 0x8f, 0x9f, 0x19, 0x56, 0xfb, 0xe7, 0xd5, 0xa7,
    0xc8, 0x2f, 0xd3, 0xb3,
};
static const unsigned char kat101_nonce[] = {
    0x61, 0x5b, 0x62, 0xa0, 0x0d, 0x6d, 0x8d, 0x85,
};
static const unsigned char kat101_persstr[] = {
    0x70, 0xdd, 0x43, 0xc1, 0x76, 0x43, 0x18, 0x17, 0x4a, 0x28, 0x57, 0xc4,
    0x20, 0xdd, 0xf4, 0x07,
};
static const unsigned char kat101_addin0[] = {0};
static const unsigned char kat101_addin1[] = {0};
static const unsigned char kat101_retbits[] = {
    0x71, 0xf4, 0xb5, 0xdf, 0x5b, 0xfa, 0x5f, 0xd5, 0x2c, 0xdc, 0x98, 0x51,
    0xa6, 0x33, 0xf7, 0x7e, 0xd9, 0x04, 0x58, 0xb5, 0xa2, 0x90, 0xb0, 0x4d,
    0x2f, 0x35, 0x13, 0x0d, 0x67, 0xa4, 0x2c, 0x16, 0x98, 0xac, 0x5f, 0x9b,
    0x13, 0x9a, 0x9e, 0xcf, 0x35, 0x90, 0x75, 0x5a, 0x20, 0x41, 0x60, 0xa3,
    0xa8, 0xf1, 0x7b, 0x77, 0x72, 0x66, 0x52, 0xc6, 0xdc, 0x6e, 0x9f, 0x00,
    0x96, 0x64, 0x54, 0xfd,
};
static const struct drbg_kat_no_reseed kat101_t = {
    10, kat101_entropyin, kat101_nonce, kat101_persstr,
    kat101_addin0, kat101_addin1, kat101_retbits
};
static const struct drbg_kat kat101 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat101_t
};

static const unsigned char kat102_entropyin[] = {
    0x7b, 0x39, 0x1f, 0x35, 0xdb, 0x0e, 0x53, 0xbf, 0x48, 0xee, 0x55, 0x76,
    0xb0, 0x23, 0xff, 0x90,
};
static const unsigned char kat102_nonce[] = {
    0x76, 0x21, 0xaa, 0xd5, 0x32, 0x98, 0x2b, 0x23,
};
static const unsigned char kat102_persstr[] = {
    0xa6, 0xe7, 0x57, 0x78, 0xb7, 0x77, 0xd0, 0x1f, 0x00, 0x62, 0x27, 0xc4,
    0x00, 0xd8, 0xff, 0x86,
};
static const unsigned char kat102_addin0[] = {0};
static const unsigned char kat102_addin1[] = {0};
static const unsigned char kat102_retbits[] = {
    0xd6, 0x17, 0x80, 0xb5, 0xc7, 0x81, 0x41, 0x2b, 0xf3, 0x08, 0x5e, 0xec,
    0xce, 0xe4, 0x9b, 0x99, 0x35, 0x8a, 0x18, 0x32, 0x23, 0x82, 0x8c, 0x1d,
    0x80, 0x13, 0xac, 0xe6, 0x13, 0xd8, 0x9a, 0x45, 0x04, 0xd7, 0x5b, 0xa3,
    0x09, 0xe5, 0x10, 0x58, 0x9a, 0x53, 0xb4, 0x72, 0xbd, 0x5f, 0xa1, 0xee,
    0x2a, 0x22, 0x39, 0x2b, 0x82, 0x65, 0x70, 0x7c, 0x15, 0xe3, 0x29, 0x35,
    0xbc, 0x8e, 0xfb, 0x4e,
};
static const struct drbg_kat_no_reseed kat102_t = {
    11, kat102_entropyin, kat102_nonce, kat102_persstr,
    kat102_addin0, kat102_addin1, kat102_retbits
};
static const struct drbg_kat kat102 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat102_t
};

static const unsigned char kat103_entropyin[] = {
    0x09, 0x9f, 0x1f, 0x59, 0x17, 0x07, 0xd1, 0xf9, 0xab, 0xae, 0x7e, 0x2b,
    0x65, 0xad, 0xab, 0xc3,
};
static const unsigned char kat103_nonce[] = {
    0xf9, 0x0d, 0x35, 0x7a, 0x90, 0x1f, 0x11, 0x35,
};
static const unsigned char kat103_persstr[] = {
    0x74, 0xb8, 0x26, 0x0c, 0xed, 0x4b, 0xbb, 0x80, 0x41, 0x77, 0x92, 0xf1,
    0x93, 0xc6, 0xf9, 0xac,
};
static const unsigned char kat103_addin0[] = {0};
static const unsigned char kat103_addin1[] = {0};
static const unsigned char kat103_retbits[] = {
    0x86, 0xf5, 0x06, 0x25, 0x6d, 0x29, 0x45, 0x0f, 0x16, 0x07, 0x31, 0x27,
    0x39, 0xe5, 0xa0, 0x89, 0xb6, 0xb7, 0x86, 0xfd, 0xf5, 0x21, 0x45, 0x76,
    0x9c, 0xb7, 0x0e, 0xf4, 0xee, 0xd3, 0x21, 0x09, 0xcf, 0x62, 0x36, 0x9d,
    0x7a, 0x76, 0x59, 0x55, 0x78, 0x1e, 0xae, 0x63, 0x52, 0x02, 0x75, 0x16,
    0x93, 0x68, 0x25, 0x7c, 0x6d, 0x34, 0x8e, 0xcc, 0x90, 0x0a, 0x38, 0x98,
    0x77, 0x8d, 0x9a, 0x1a,
};
static const struct drbg_kat_no_reseed kat103_t = {
    12, kat103_entropyin, kat103_nonce, kat103_persstr,
    kat103_addin0, kat103_addin1, kat103_retbits
};
static const struct drbg_kat kat103 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat103_t
};

static const unsigned char kat104_entropyin[] = {
    0xc9, 0x05, 0x9d, 0x7c, 0x01, 0x5c, 0x64, 0xf6, 0xe4, 0x1a, 0xa4, 0x92,
    0x09, 0x9f, 0x4e, 0xf0,
};
static const unsigned char kat104_nonce[] = {
    0x3f, 0x6e, 0x07, 0x69, 0xce, 0xe1, 0x77, 0xc6,
};
static const unsigned char kat104_persstr[] = {
    0x76, 0xc6, 0xc6, 0x61, 0x32, 0x98, 0xd9, 0x25, 0xa6, 0xc5, 0x07, 0xae,
    0x65, 0x4d, 0x6d, 0x9a,
};
static const unsigned char kat104_addin0[] = {0};
static const unsigned char kat104_addin1[] = {0};
static const unsigned char kat104_retbits[] = {
    0x23, 0x81, 0x11, 0xc0, 0x06, 0xe5, 0x14, 0xc0, 0x5f, 0x2a, 0xe9, 0x35,
    0xa3, 0x2d, 0xba, 0xa0, 0xb1, 0x28, 0xda, 0xa0, 0x73, 0x6e, 0x83, 0xd9,
    0x34, 0x03, 0x77, 0x6c, 0x91, 0xe4, 0x77, 0xd0, 0xd0, 0x2c, 0x4a, 0x24,
    0x87, 0x6a, 0x23, 0x29, 0xa0, 0xf1, 0x06, 0x03, 0x8d, 0x70, 0x1f, 0xed,
    0xdf, 0x02, 0x47, 0xcc, 0x75, 0x84, 0x6d, 0xd3, 0x01, 0x08, 0x29, 0x9d,
    0x84, 0x0e, 0xc4, 0xa8,
};
static const struct drbg_kat_no_reseed kat104_t = {
    13, kat104_entropyin, kat104_nonce, kat104_persstr,
    kat104_addin0, kat104_addin1, kat104_retbits
};
static const struct drbg_kat kat104 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat104_t
};

static const unsigned char kat105_entropyin[] = {
    0x98, 0xd1, 0x3c, 0x30, 0xa6, 0xda, 0xe2, 0xca, 0x76, 0xd7, 0xd6, 0x67,
    0x9f, 0x1f, 0xfb, 0xc4,
};
static const unsigned char kat105_nonce[] = {
    0x49, 0x0b, 0x93, 0x9e, 0xb5, 0x56, 0xff, 0x1b,
};
static const unsigned char kat105_persstr[] = {
    0x6c, 0x10, 0xa7, 0xc3, 0x0f, 0xb5, 0x34, 0xb5, 0x93, 0x4b, 0x1c, 0x29,
    0x80, 0x84, 0x94, 0xb9,
};
static const unsigned char kat105_addin0[] = {0};
static const unsigned char kat105_addin1[] = {0};
static const unsigned char kat105_retbits[] = {
    0xcb, 0x7d, 0xee, 0xea, 0x97, 0xbd, 0xb0, 0x3f, 0x72, 0xa0, 0x3b, 0x2d,
    0x9b, 0x5f, 0x1b, 0x33, 0x03, 0x44, 0x48, 0x6e, 0x53, 0xe0, 0x49, 0x27,
    0xc5, 0x8f, 0xb6, 0x3c, 0x97, 0x1a, 0xc9, 0x87, 0x33, 0xb0, 0x24, 0x9a,
    0x1e, 0xfa, 0x85, 0xf8, 0x8b, 0xec, 0x48, 0x18, 0xe1, 0x31, 0x10, 0x22,
    0x07, 0x76, 0xe4, 0xfd, 0xe4, 0x29, 0xe6, 0x39, 0x0a, 0xf4, 0x66, 0x71,
    0x29, 0x01, 0x18, 0x9b,
};
static const struct drbg_kat_no_reseed kat105_t = {
    14, kat105_entropyin, kat105_nonce, kat105_persstr,
    kat105_addin0, kat105_addin1, kat105_retbits
};
static const struct drbg_kat kat105 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat105_t
};

static const unsigned char kat106_entropyin[] = {
    0x25, 0xad, 0x6e, 0x73, 0x05, 0x5e, 0xcb, 0xc9, 0x49, 0xf2, 0x91, 0xf0,
    0xf7, 0x97, 0xf1, 0x7a,
};
static const unsigned char kat106_nonce[] = {
    0x1f, 0x85, 0xa9, 0x2a, 0x46, 0x0a, 0x2e, 0xea,
};
static const unsigned char kat106_persstr[] = {
    0x83, 0xd0, 0x4b, 0x4b, 0x15, 0x91, 0xc7, 0xba, 0xc9, 0xbc, 0xa9, 0x07,
    0xbb, 0xe9, 0xdd, 0xd4,
};
static const unsigned char kat106_addin0[] = {
    0x52, 0x3c, 0x6c, 0xb0, 0xbc, 0x27, 0xd1, 0x9f, 0x8d, 0x2e, 0xbe, 0xf5,
    0x79, 0x26, 0xdd, 0xe4,
};
static const unsigned char kat106_addin1[] = {
    0xd3, 0x3c, 0xd1, 0x4b, 0x5b, 0x79, 0x37, 0x38, 0x8e, 0x89, 0xec, 0xc0,
    0x80, 0x63, 0x03, 0xf0,
};
static const unsigned char kat106_retbits[] = {
    0x23, 0xd9, 0x19, 0x5e, 0x40, 0x26, 0xed, 0xf0, 0x70, 0x88, 0xb8, 0x37,
    0x62, 0x75, 0x47, 0xa0, 0xea, 0x91, 0x37, 0xed, 0x03, 0x36, 0xd7, 0x69,
    0x60, 0x15, 0xdd, 0x6f, 0x29, 0x84, 0xde, 0xdb, 0x2a, 0x20, 0x7f, 0x2e,
    0xb8, 0xa2, 0x56, 0x60, 0xa5, 0xee, 0x78, 0x11, 0x78, 0x57, 0x9a, 0x0f,
    0x23, 0x3f, 0xb6, 0xf7, 0x26, 0x03, 0x58, 0xdb, 0xfd, 0x53, 0x25, 0xc3,
    0xf8, 0xc8, 0xfe, 0x33,
};
static const struct drbg_kat_no_reseed kat106_t = {
    0, kat106_entropyin, kat106_nonce, kat106_persstr,
    kat106_addin0, kat106_addin1, kat106_retbits
};
static const struct drbg_kat kat106 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat106_t
};

static const unsigned char kat107_entropyin[] = {
    0xae, 0x33, 0x2f, 0x73, 0x39, 0x0e, 0x27, 0x75, 0x6b, 0xc9, 0x3d, 0x2b,
    0x95, 0x1b, 0x8b, 0x44,
};
static const unsigned char kat107_nonce[] = {
    0xe6, 0x9f, 0xc8, 0xa1, 0x64, 0x50, 0xfb, 0x6d,
};
static const unsigned char kat107_persstr[] = {
    0x5f, 0x76, 0x86, 0x3d, 0x3a, 0x20, 0x87, 0xe9, 0xcb, 0x90, 0xe1, 0x12,
    0xc1, 0x68, 0x06, 0xe2,
};
static const unsigned char kat107_addin0[] = {
    0xbf, 0x6a, 0x8e, 0x05, 0x6e, 0x6c, 0x3c, 0xc1, 0xa6, 0xcb, 0xdb, 0x6b,
    0x59, 0xeb, 0xea, 0xe2,
};
static const unsigned char kat107_addin1[] = {
    0xd8, 0x7a, 0xb9, 0x22, 0x4a, 0xa9, 0xcd, 0x9b, 0x5b, 0x84, 0x78, 0x35,
    0xcb, 0x0d, 0xaa, 0xc4,
};
static const unsigned char kat107_retbits[] = {
    0xe0, 0xa2, 0xdd, 0x56, 0x06, 0xa0, 0xa2, 0x61, 0x57, 0xf9, 0x21, 0x05,
    0x11, 0xbb, 0xde, 0x50, 0xc9, 0x2f, 0x34, 0xad, 0x92, 0x36, 0x3c, 0xb9,
    0x2a, 0x05, 0x20, 0x8b, 0x60, 0xb3, 0x3d, 0x21, 0xc3, 0xd6, 0xc0, 0xc6,
    0xef, 0x05, 0x4e, 0xc4, 0x09, 0xb4, 0x63, 0x0d, 0xea, 0xc4, 0xc6, 0xd1,
    0xad, 0x1b, 0x6b, 0xe7, 0x5f, 0xff, 0x27, 0xaa, 0x74, 0x94, 0x13, 0xc8,
    0xb6, 0x4c, 0xf3, 0xe1,
};
static const struct drbg_kat_no_reseed kat107_t = {
    1, kat107_entropyin, kat107_nonce, kat107_persstr,
    kat107_addin0, kat107_addin1, kat107_retbits
};
static const struct drbg_kat kat107 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat107_t
};

static const unsigned char kat108_entropyin[] = {
    0xe1, 0xa5, 0x89, 0xaf, 0xab, 0x83, 0x91, 0x69, 0x2d, 0xba, 0x52, 0x31,
    0x8a, 0xa8, 0xdc, 0x0d,
};
static const unsigned char kat108_nonce[] = {
    0x22, 0xeb, 0xcf, 0x2e, 0x24, 0xa9, 0x55, 0x28,
};
static const unsigned char kat108_persstr[] = {
    0x82, 0xe8, 0xc3, 0x55, 0xdb, 0xc9, 0xcd, 0x4c, 0x8d, 0x52, 0xfa, 0x45,
    0x21, 0x18, 0xf9, 0xf8,
};
static const unsigned char kat108_addin0[] = {
    0xf7, 0x74, 0x57, 0xec, 0x32, 0xde, 0x66, 0x6f, 0x37, 0x0d, 0xa9, 0xe6,
    0xc2, 0x20, 0xc5, 0x80,
};
static const unsigned char kat108_addin1[] = {
    0xba, 0x25, 0xaa, 0x3a, 0xb3, 0xd5, 0x0a, 0x72, 0xb6, 0xc1, 0xb4, 0x3f,
    0x30, 0xdb, 0x44, 0x55,
};
static const unsigned char kat108_retbits[] = {
    0x88, 0x59, 0x55, 0xd1, 0x27, 0xfa, 0xf4, 0xc9, 0x21, 0xc4, 0xed, 0x3f,
    0x38, 0xbc, 0x15, 0x42, 0xb9, 0x5a, 0x87, 0x83, 0x4d, 0x4d, 0x73, 0xd1,
    0x56, 0x24, 0x00, 0xa6, 0xd3, 0xaa, 0x8e, 0x6c, 0x69, 0x7b, 0x71, 0x1d,
    0x91, 0x0e, 0xec, 0xeb, 0xc9, 0xca, 0x4e, 0xc2, 0xdc, 0x17, 0xe7, 0x98,
    0x93, 0xe7, 0xf7, 0x34, 0xc4, 0x56, 0x75, 0xde, 0x47, 0x7f, 0x55, 0x45,
    0xbd, 0x94, 0xcd, 0xee,
};
static const struct drbg_kat_no_reseed kat108_t = {
    2, kat108_entropyin, kat108_nonce, kat108_persstr,
    kat108_addin0, kat108_addin1, kat108_retbits
};
static const struct drbg_kat kat108 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat108_t
};

static const unsigned char kat109_entropyin[] = {
    0x6f, 0xb7, 0x6f, 0xcd, 0x3d, 0x76, 0x2f, 0xa4, 0xc5, 0x0d, 0xe1, 0xf2,
    0x2e, 0x92, 0x76, 0x83,
};
static const unsigned char kat109_nonce[] = {
    0xa5, 0xee, 0xf1, 0x2f, 0x3b, 0x3c, 0x25, 0xa3,
};
static const unsigned char kat109_persstr[] = {
    0x54, 0x86, 0x7d, 0x7d, 0xc8, 0xce, 0xf5, 0x23, 0x42, 0xbb, 0xc0, 0x09,
    0x95, 0x4e, 0xfb, 0xf8,
};
static const unsigned char kat109_addin0[] = {
    0xfa, 0xc5, 0x52, 0xaa, 0x87, 0x96, 0xe8, 0x85, 0x8d, 0x55, 0xae, 0x37,
    0x1a, 0x0f, 0x54, 0x8f,
};
static const unsigned char kat109_addin1[] = {
    0xfe, 0xfb, 0xe9, 0xc8, 0x3c, 0x15, 0x9f, 0xcf, 0xa9, 0x2e, 0xe8, 0x18,
    0x76, 0x1d, 0x49, 0xdc,
};
static const unsigned char kat109_retbits[] = {
    0x09, 0x4f, 0x86, 0x3b, 0x88, 0xe2, 0xb5, 0x5e, 0xf4, 0x15, 0xd2, 0x68,
    0x82, 0x94, 0xc7, 0xd8, 0xd2, 0xd7, 0x76, 0xc1, 0x69, 0x16, 0x73, 0x91,
    0xdf, 0x36, 0xf8, 0x0a, 0xd8, 0xe9, 0x83, 0x3e, 0x97, 0x61, 0x6e, 0xf6,
    0xe0, 0xa4, 0x5f, 0x94, 0x0d, 0x66, 0x38, 0x30, 0x18, 0xae, 0xe8, 0x6a,
    0x48, 0x0a, 0x26, 0x51, 0xd6, 0xe2, 0xef, 0xad, 0x61, 0x5a, 0x05, 0x1f,
    0x0d, 0xca, 0x64, 0xae,
};
static const struct drbg_kat_no_reseed kat109_t = {
    3, kat109_entropyin, kat109_nonce, kat109_persstr,
    kat109_addin0, kat109_addin1, kat109_retbits
};
static const struct drbg_kat kat109 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat109_t
};

static const unsigned char kat110_entropyin[] = {
    0x5b, 0x32, 0xa3, 0x92, 0xdc, 0x66, 0x5a, 0xb5, 0x8f, 0x25, 0x98, 0xea,
    0xb0, 0x7d, 0x8e, 0x53,
};
static const unsigned char kat110_nonce[] = {
    0xc6, 0xf6, 0x13, 0x10, 0x73, 0x88, 0x60, 0xf5,
};
static const unsigned char kat110_persstr[] = {
    0x9f, 0x30, 0x9a, 0x68, 0x53, 0x29, 0x82, 0x16, 0x92, 0xa9, 0x73, 0x00,
    0x62, 0x0c, 0x01, 0x56,
};
static const unsigned char kat110_addin0[] = {
    0x87, 0x5a, 0x4a, 0x43, 0xbd, 0xdf, 0x1a, 0x73, 0x05, 0x52, 0x6a, 0x48,
    0x8a, 0x1e, 0x0e, 0xb4,
};
static const unsigned char kat110_addin1[] = {
    0x21, 0x14, 0xa7, 0xcb, 0x15, 0x49, 0x0c, 0x06, 0x3b, 0xb4, 0x66, 0x8f,
    0x36, 0x9a, 0x3d, 0x8e,
};
static const unsigned char kat110_retbits[] = {
    0x91, 0xc1, 0xf1, 0x99, 0xf6, 0x71, 0xff, 0x56, 0x8e, 0x0e, 0xce, 0x46,
    0x08, 0x87, 0x65, 0x16, 0x23, 0x22, 0xa8, 0x21, 0x60, 0x66, 0xf9, 0xa4,
    0x15, 0xd7, 0xcc, 0xd9, 0xdf, 0xe7, 0x23, 0xc5, 0x21, 0xf5, 0x7d, 0xbd,
    0x84, 0x69, 0xe3, 0x9b, 0xac, 0xda, 0xc3, 0x1e, 0x74, 0x2a, 0x2d, 0x2f,
    0xf2, 0xa3, 0x1d, 0x5a, 0x15, 0xa0, 0x1b, 0xfd, 0x32, 0xbe, 0x50, 0x28,
    0x6c, 0x73, 0x0e, 0xad,
};
static const struct drbg_kat_no_reseed kat110_t = {
    4, kat110_entropyin, kat110_nonce, kat110_persstr,
    kat110_addin0, kat110_addin1, kat110_retbits
};
static const struct drbg_kat kat110 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat110_t
};

static const unsigned char kat111_entropyin[] = {
    0x79, 0x95, 0x7c, 0x33, 0xe4, 0x5b, 0xba, 0x1c, 0x8a, 0xe6, 0x90, 0x29,
    0x07, 0x7a, 0xe3, 0x6b,
};
static const unsigned char kat111_nonce[] = {
    0x5b, 0x49, 0xd8, 0xe7, 0xc1, 0x80, 0xbc, 0xb6,
};
static const unsigned char kat111_persstr[] = {
    0x2a, 0xa8, 0xe7, 0x7b, 0x94, 0xfe, 0x0d, 0xdf, 0x0e, 0x57, 0x30, 0x2c,
    0x8e, 0xbd, 0xc2, 0x2a,
};
static const unsigned char kat111_addin0[] = {
    0x14, 0x84, 0x4d, 0x85, 0xe5, 0x0c, 0xb1, 0xfc, 0xb4, 0xe6, 0x88, 0x7b,
    0x3f, 0xc6, 0x7f, 0x54,
};
static const unsigned char kat111_addin1[] = {
    0x4c, 0x7c, 0xb0, 0x69, 0x29, 0x7c, 0x18, 0x65, 0x33, 0x94, 0xb8, 0x2b,
    0xe6, 0xa9, 0x30, 0x99,
};
static const unsigned char kat111_retbits[] = {
    0x06, 0x61, 0xe2, 0x41, 0x6e, 0x71, 0xfe, 0x49, 0x11, 0x9f, 0x96, 0x76,
    0x42, 0xfd, 0x37, 0xdd, 0xb5, 0x82, 0xa1, 0x89, 0xa8, 0x88, 0xe7, 0xd4,
    0xa2, 0xaa, 0x56, 0x27, 0x51, 0x94, 0xc8, 0xca, 0x69, 0x8f, 0xb2, 0xd6,
    0x96, 0x83, 0x4b, 0x88, 0xb5, 0x72, 0xf0, 0xfd, 0xf5, 0xba, 0xdb, 0x6b,
    0xf5, 0x8d, 0x16, 0x12, 0xd2, 0x5f, 0x5e, 0xe3, 0xf6, 0x2d, 0x33, 0xdb,
    0x9a, 0x86, 0xb0, 0x87,
};
static const struct drbg_kat_no_reseed kat111_t = {
    5, kat111_entropyin, kat111_nonce, kat111_persstr,
    kat111_addin0, kat111_addin1, kat111_retbits
};
static const struct drbg_kat kat111 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat111_t
};

static const unsigned char kat112_entropyin[] = {
    0x70, 0x50, 0x89, 0x80, 0x36, 0x2e, 0x96, 0x2a, 0xe7, 0x78, 0x45, 0x80,
    0x81, 0xcf, 0xbb, 0xe6,
};
static const unsigned char kat112_nonce[] = {
    0x29, 0xcc, 0xf6, 0xed, 0xba, 0x46, 0x2d, 0xc2,
};
static const unsigned char kat112_persstr[] = {
    0x48, 0xa3, 0xd8, 0xdb, 0xdc, 0x85, 0x18, 0x8a, 0xbb, 0x35, 0xb5, 0x6c,
    0x19, 0x4d, 0x60, 0x20,
};
static const unsigned char kat112_addin0[] = {
    0xa8, 0x14, 0x72, 0x42, 0xe2, 0x47, 0x37, 0xfc, 0xf3, 0xb0, 0x07, 0xd2,
    0x84, 0xd6, 0x61, 0xc4,
};
static const unsigned char kat112_addin1[] = {
    0x96, 0x4a, 0x9d, 0x06, 0x86, 0xd9, 0x61, 0x93, 0xfb, 0x2e, 0x4a, 0x35,
    0x14, 0x5e, 0xdc, 0xc8,
};
static const unsigned char kat112_retbits[] = {
    0xd8, 0x1b, 0xd5, 0xda, 0x60, 0xd9, 0x9d, 0x6d, 0xaa, 0xec, 0xeb, 0xc0,
    0x60, 0x65, 0x26, 0x33, 0xec, 0x14, 0xdc, 0x3b, 0x84, 0x69, 0x3b, 0xdf,
    0xf7, 0x67, 0x77, 0x1f, 0x03, 0x2b, 0xfa, 0x2e, 0x4d, 0xfa, 0xd5, 0x3e,
    0xf6, 0xd1, 0x99, 0x10, 0xbd, 0x92, 0x5d, 0xf1, 0x20, 0x9f, 0x33, 0xea,
    0x11, 0xe3, 0xf1, 0xe8, 0x37, 0xb0, 0x3e, 0x7a, 0x70, 0x03, 0x25, 0x7f,
    0xf4, 0x20, 0xa6, 0x92,
};
static const struct drbg_kat_no_reseed kat112_t = {
    6, kat112_entropyin, kat112_nonce, kat112_persstr,
    kat112_addin0, kat112_addin1, kat112_retbits
};
static const struct drbg_kat kat112 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat112_t
};

static const unsigned char kat113_entropyin[] = {
    0xb7, 0x83, 0x71, 0xba, 0xf5, 0x86, 0xc0, 0xdd, 0x05, 0x4b, 0xfd, 0x29,
    0x3f, 0xc7, 0xe8, 0xdc,
};
static const unsigned char kat113_nonce[] = {
    0x11, 0x9b, 0x23, 0x2c, 0xe4, 0xca, 0xda, 0x97,
};
static const unsigned char kat113_persstr[] = {
    0x3a, 0x58, 0x75, 0xb4, 0x43, 0x07, 0x3c, 0x9b, 0xf6, 0x4a, 0x0a, 0xaa,
    0xf2, 0x6b, 0xa4, 0x4f,
};
static const unsigned char kat113_addin0[] = {
    0x4d, 0x2d, 0x4f, 0x16, 0xb7, 0xb7, 0xd2, 0xd2, 0xbb, 0x4d, 0x83, 0xa4,
    0x92, 0x99, 0x96, 0xb2,
};
static const unsigned char kat113_addin1[] = {
    0x7d, 0x31, 0x73, 0xe0, 0x8a, 0xea, 0x4c, 0x8b, 0x6e, 0xf9, 0xe3, 0xcd,
    0x69, 0x8f, 0x9b, 0x09,
};
static const unsigned char kat113_retbits[] = {
    0xbb, 0x04, 0x86, 0x41, 0xf0, 0x36, 0xdc, 0x11, 0xb8, 0xdf, 0x8a, 0xd3,
    0xe4, 0xd5, 0xbd, 0xfc, 0x30, 0x9b, 0x2c, 0x3f, 0x22, 0xb3, 0xd2, 0xb8,
    0xac, 0xf1, 0x96, 0xd5, 0x0c, 0xd5, 0x42, 0xde, 0x13, 0xcd, 0x99, 0x97,
    0x8a, 0x92, 0x0a, 0xcb, 0xa6, 0x1d, 0xc0, 0x7c, 0x64, 0x6b, 0x48, 0x09,
    0x44, 0x79, 0xbb, 0x44, 0x17, 0xd8, 0x40, 0x9a, 0xba, 0xe3, 0xd6, 0x9e,
    0xd5, 0x2a, 0x2a, 0x3e,
};
static const struct drbg_kat_no_reseed kat113_t = {
    7, kat113_entropyin, kat113_nonce, kat113_persstr,
    kat113_addin0, kat113_addin1, kat113_retbits
};
static const struct drbg_kat kat113 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat113_t
};

static const unsigned char kat114_entropyin[] = {
    0x96, 0xbf, 0xc3, 0x47, 0xb3, 0x06, 0x80, 0x5a, 0xf8, 0x8d, 0xa0, 0x07,
    0x67, 0x5d, 0xb4, 0x7c,
};
static const unsigned char kat114_nonce[] = {
    0x12, 0x95, 0x37, 0x06, 0xa0, 0x6e, 0xb5, 0x30,
};
static const unsigned char kat114_persstr[] = {
    0x46, 0x73, 0x3d, 0xdd, 0xaf, 0x70, 0xe3, 0xf2, 0x07, 0x5c, 0xb8, 0x70,
    0xf4, 0x7e, 0x72, 0xdf,
};
static const unsigned char kat114_addin0[] = {
    0x64, 0xf7, 0x9a, 0xd1, 0xdf, 0xa3, 0x93, 0xd3, 0x6b, 0xf5, 0xbe, 0x83,
    0x32, 0x90, 0xb6, 0x94,
};
static const unsigned char kat114_addin1[] = {
    0x5d, 0x21, 0x0f, 0x12, 0x85, 0xf0, 0xfb, 0x03, 0x8a, 0x75, 0x34, 0x76,
    0xef, 0x24, 0x91, 0x79,
};
static const unsigned char kat114_retbits[] = {
    0x67, 0xb6, 0x9b, 0xa4, 0x34, 0x90, 0x3e, 0x2b, 0xbf, 0x46, 0xd3, 0xf5,
    0xe3, 0x00, 0xb9, 0xe5, 0xcf, 0x33, 0x01, 0xbe, 0xaa, 0xb7, 0xd3, 0x0b,
    0x46, 0x52, 0xed, 0x3b, 0xf6, 0x62, 0x65, 0x79, 0x50, 0x3a, 0x54, 0x4e,
    0x58, 0x23, 0x7c, 0x5e, 0x55, 0x1a, 0xd5, 0xee, 0x17, 0xb7, 0x36, 0x12,
    0x0c, 0x37, 0xa3, 0x02, 0x9b, 0xd7, 0xa1, 0x03, 0x4d, 0x4c, 0x49, 0xf6,
    0x1e, 0x04, 0x0a, 0xca,
};
static const struct drbg_kat_no_reseed kat114_t = {
    8, kat114_entropyin, kat114_nonce, kat114_persstr,
    kat114_addin0, kat114_addin1, kat114_retbits
};
static const struct drbg_kat kat114 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat114_t
};

static const unsigned char kat115_entropyin[] = {
    0x9a, 0x4c, 0x8b, 0x46, 0xa9, 0x74, 0xfb, 0xc8, 0x5e, 0x25, 0x6d, 0x1f,
    0x07, 0xcc, 0x59, 0xaa,
};
static const unsigned char kat115_nonce[] = {
    0xeb, 0x8d, 0x66, 0x21, 0x02, 0x09, 0x01, 0x14,
};
static const unsigned char kat115_persstr[] = {
    0xb9, 0x4b, 0x8d, 0x2a, 0x64, 0x92, 0x66, 0x64, 0x5a, 0x71, 0x5a, 0xfc,
    0xa0, 0x99, 0x5d, 0x9d,
};
static const unsigned char kat115_addin0[] = {
    0xf3, 0x36, 0xe1, 0xd6, 0x2a, 0xc5, 0x68, 0xed, 0x13, 0x2e, 0xa0, 0x18,
    0x15, 0xda, 0xd5, 0x08,
};
static const unsigned char kat115_addin1[] = {
    0x66, 0x76, 0x26, 0xa2, 0x3d, 0x23, 0xb7, 0x16, 0x5a, 0x6a, 0x88, 0x5d,
    0x41, 0xfa, 0x90, 0x1c,
};
static const unsigned char kat115_retbits[] = {
    0x2d, 0xf3, 0x3f, 0x7a, 0xd2, 0xc2, 0xd8, 0x6e, 0x61, 0x63, 0x3c, 0x25,
    0xa8, 0xac, 0x57, 0xb5, 0xde, 0xad, 0x88, 0x02, 0x1c, 0x93, 0xad, 0x69,
    0xba, 0x6c, 0x52, 0x27, 0x99, 0xc4, 0xbe, 0xbf, 0xda, 0x78, 0x27, 0x72,
    0xc8, 0x1e, 0x7e, 0x02, 0xd8, 0xb8, 0x82, 0x54, 0x6b, 0x24, 0x2b, 0xb2,
    0x2f, 0x31, 0x05, 0x65, 0xba, 0xc3, 0x16, 0x26, 0x9f, 0xfc, 0x80, 0x04,
    0xd5, 0x44, 0x06, 0x31,
};
static const struct drbg_kat_no_reseed kat115_t = {
    9, kat115_entropyin, kat115_nonce, kat115_persstr,
    kat115_addin0, kat115_addin1, kat115_retbits
};
static const struct drbg_kat kat115 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat115_t
};

static const unsigned char kat116_entropyin[] = {
    0xc7, 0xd6, 0x55, 0x75, 0x65, 0xa1, 0xfd, 0xe4, 0x80, 0x84, 0x1f, 0x1f,
    0x1a, 0x10, 0xf6, 0x7d,
};
static const unsigned char kat116_nonce[] = {
    0x02, 0xf8, 0x47, 0x98, 0x66, 0x26, 0xbb, 0x4b,
};
static const unsigned char kat116_persstr[] = {
    0x61, 0x8c, 0xac, 0x99, 0x53, 0x17, 0x91, 0x67, 0x36, 0x03, 0xa1, 0x6b,
    0x99, 0xab, 0x0e, 0x80,
};
static const unsigned char kat116_addin0[] = {
    0x81, 0x0c, 0x8b, 0xe4, 0xb1, 0x80, 0x1f, 0xec, 0x75, 0x40, 0x63, 0xc2,
    0x61, 0x15, 0x66, 0x60,
};
static const unsigned char kat116_addin1[] = {
    0xe2, 0x59, 0x4d, 0x34, 0xce, 0x1a, 0xd6, 0x53, 0x51, 0x28, 0x60, 0x69,
    0xeb, 0x1a, 0x63, 0x08,
};
static const unsigned char kat116_retbits[] = {
    0xa3, 0xc4, 0xd0, 0x2d, 0xa0, 0x16, 0x6f, 0xef, 0x32, 0xce, 0x69, 0x97,
    0x5d, 0xb9, 0x4d, 0x19, 0x60, 0x44, 0xe8, 0x1f, 0x59, 0x04, 0x4d, 0x22,
    0xdf, 0x84, 0x05, 0xb4, 0x68, 0xeb, 0xd5, 0xa9, 0x29, 0x97, 0xfe, 0xeb,
    0xa8, 0x2d, 0x7a, 0xdf, 0x97, 0xcc, 0xbb, 0x1a, 0x59, 0x35, 0x56, 0x26,
    0xae, 0xfa, 0xe0, 0xd4, 0x35, 0xb4, 0x03, 0xb7, 0xde, 0xb9, 0x00, 0x5b,
    0x8f, 0x5e, 0xa9, 0xa5,
};
static const struct drbg_kat_no_reseed kat116_t = {
    10, kat116_entropyin, kat116_nonce, kat116_persstr,
    kat116_addin0, kat116_addin1, kat116_retbits
};
static const struct drbg_kat kat116 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat116_t
};

static const unsigned char kat117_entropyin[] = {
    0x14, 0x0d, 0xbf, 0xf6, 0x3e, 0x70, 0x83, 0xc3, 0x08, 0x41, 0x9f, 0x86,
    0x4c, 0x09, 0x3e, 0x99,
};
static const unsigned char kat117_nonce[] = {
    0xeb, 0x1c, 0x66, 0x92, 0xdb, 0x94, 0x9d, 0x5a,
};
static const unsigned char kat117_persstr[] = {
    0x23, 0xd3, 0x27, 0x21, 0xf1, 0xc0, 0x3a, 0x90, 0xf3, 0x9e, 0xf0, 0x21,
    0xac, 0x88, 0x84, 0xc7,
};
static const unsigned char kat117_addin0[] = {
    0xc6, 0xe3, 0x80, 0x8d, 0xb9, 0xf9, 0xda, 0x14, 0x23, 0x9a, 0xd7, 0xd7,
    0x18, 0x28, 0x6a, 0x96,
};
static const unsigned char kat117_addin1[] = {
    0xa8, 0x4a, 0xaa, 0xdb, 0x7d, 0x57, 0x2a, 0x9a, 0xfc, 0xa3, 0xf7, 0x6c,
    0xed, 0xf0, 0xa4, 0xb9,
};
static const unsigned char kat117_retbits[] = {
    0x74, 0x3d, 0xf9, 0x7c, 0x2e, 0x99, 0xd9, 0x74, 0x07, 0x77, 0x72, 0xa6,
    0xfe, 0xa4, 0x03, 0x12, 0x8f, 0x41, 0xff, 0x32, 0xa9, 0x5f, 0xb4, 0xf8,
    0xb0, 0xec, 0xca, 0xa0, 0x57, 0x1c, 0xd2, 0x57, 0x8e, 0x48, 0x2c, 0xe7,
    0x09, 0x84, 0x99, 0xc4, 0xd3, 0x01, 0x5f, 0x91, 0x92, 0x35, 0x52, 0x38,
    0xc7, 0x0f, 0x8b, 0x76, 0xa5, 0x04, 0x27, 0x2a, 0x87, 0x09, 0xcd, 0x0e,
    0x2d, 0xc8, 0x2c, 0x9c,
};
static const struct drbg_kat_no_reseed kat117_t = {
    11, kat117_entropyin, kat117_nonce, kat117_persstr,
    kat117_addin0, kat117_addin1, kat117_retbits
};
static const struct drbg_kat kat117 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat117_t
};

static const unsigned char kat118_entropyin[] = {
    0x6a, 0xf3, 0x96, 0x2f, 0xd6, 0xae, 0x84, 0xde, 0xc3, 0xf8, 0x94, 0xbf,
    0x39, 0xc6, 0x46, 0x42,
};
static const unsigned char kat118_nonce[] = {
    0xc0, 0x20, 0x49, 0x64, 0x61, 0xa3, 0x91, 0x61,
};
static const unsigned char kat118_persstr[] = {
    0x75, 0xca, 0xef, 0xe7, 0x75, 0x20, 0x45, 0xfc, 0xb1, 0xe5, 0xfc, 0x22,
    0x7f, 0xfb, 0x94, 0x0d,
};
static const unsigned char kat118_addin0[] = {
    0x16, 0x64, 0xf7, 0xcc, 0x88, 0xd0, 0x1d, 0x7f, 0x27, 0x86, 0x15, 0xc5,
    0x63, 0x09, 0x2f, 0xd6,
};
static const unsigned char kat118_addin1[] = {
    0x60, 0xb7, 0xa8, 0x6b, 0xc0, 0xc8, 0xdd, 0x2b, 0xd0, 0x7a, 0xda, 0xa4,
    0xd5, 0xc2, 0x4d, 0x68,
};
static const unsigned char kat118_retbits[] = {
    0xb4, 0xed, 0xc9, 0x65, 0x0d, 0x81, 0x47, 0x17, 0x0c, 0x81, 0x52, 0x9f,
    0x2f, 0xd7, 0x33, 0x11, 0x53, 0xa6, 0x1d, 0x0d, 0x58, 0xb7, 0x6b, 0x50,
    0xbb, 0x52, 0xda, 0x0a, 0x79, 0x51, 0x64, 0xbe, 0xeb, 0xc6, 0x09, 0x32,
    0x05, 0x20, 0x32, 0x8f, 0x3d, 0x4e, 0x83, 0xcd, 0x80, 0xd0, 0xb1, 0xcb,
    0xbb, 0xfd, 0x0a, 0x84, 0xa9, 0x65, 0x10, 0x88, 0x11, 0x01, 0xfb, 0xc7,
    0x40, 0xb2, 0x81, 0x7f,
};
static const struct drbg_kat_no_reseed kat118_t = {
    12, kat118_entropyin, kat118_nonce, kat118_persstr,
    kat118_addin0, kat118_addin1, kat118_retbits
};
static const struct drbg_kat kat118 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat118_t
};

static const unsigned char kat119_entropyin[] = {
    0x36, 0xe2, 0x7a, 0xa6, 0xb3, 0x66, 0x99, 0xe9, 0x46, 0x45, 0x95, 0xa9,
    0x32, 0xcc, 0x92, 0x8e,
};
static const unsigned char kat119_nonce[] = {
    0x65, 0x16, 0xbb, 0x91, 0x85, 0x66, 0xbf, 0x89,
};
static const unsigned char kat119_persstr[] = {
    0x9b, 0xc6, 0x83, 0xa1, 0xf4, 0x65, 0xba, 0x0f, 0x2c, 0xe1, 0x8c, 0x8b,
    0x9c, 0x61, 0x42, 0x44,
};
static const unsigned char kat119_addin0[] = {
    0x0d, 0x17, 0x5b, 0x34, 0xd8, 0xb9, 0xe2, 0x70, 0x74, 0x5a, 0x71, 0xf3,
    0x5d, 0x41, 0x6f, 0x10,
};
static const unsigned char kat119_addin1[] = {
    0xa7, 0x90, 0xc8, 0x6f, 0xdf, 0x0c, 0xa8, 0x0e, 0xb3, 0xd6, 0xf7, 0x02,
    0x91, 0x6e, 0xa9, 0x9f,
};
static const unsigned char kat119_retbits[] = {
    0x34, 0xea, 0x96, 0x29, 0x62, 0x7f, 0x66, 0x84, 0xb9, 0xab, 0x85, 0xf8,
    0x16, 0x46, 0x86, 0x4d, 0xd8, 0xef, 0x3a, 0x15, 0xed, 0xc8, 0x01, 0xdc,
    0x39, 0xa7, 0x31, 0xdc, 0x9e, 0x80, 0xe6, 0x26, 0xfc, 0x8c, 0xe2, 0x1e,
    0x9b, 0x62, 0xe9, 0x48, 0x88, 0x64, 0x89, 0xd5, 0xe4, 0x43, 0x6e, 0x65,
    0x9f, 0x26, 0xbf, 0xb3, 0x43, 0x51, 0x04, 0xad, 0x9b, 0xfa, 0x0a, 0xaf,
    0xc5, 0x50, 0x69, 0x17,
};
static const struct drbg_kat_no_reseed kat119_t = {
    13, kat119_entropyin, kat119_nonce, kat119_persstr,
    kat119_addin0, kat119_addin1, kat119_retbits
};
static const struct drbg_kat kat119 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat119_t
};

static const unsigned char kat120_entropyin[] = {
    0xac, 0xa3, 0x59, 0xf1, 0x07, 0xc3, 0x24, 0x64, 0x8a, 0xd4, 0x45, 0x56,
    0x4b, 0xfa, 0xe5, 0x6d,
};
static const unsigned char kat120_nonce[] = {
    0xf9, 0x5d, 0x1c, 0x8f, 0xd2, 0x66, 0xdf, 0x64,
};
static const unsigned char kat120_persstr[] = {
    0xa5, 0xcf, 0xdc, 0x76, 0xc4, 0xe8, 0x49, 0xe4, 0x5d, 0x8b, 0xcc, 0x68,
    0xee, 0xb0, 0x12, 0x84,
};
static const unsigned char kat120_addin0[] = {
    0x57, 0x80, 0x2c, 0xc0, 0x6e, 0xf0, 0x34, 0x4b, 0xd1, 0x4a, 0x48, 0x57,
    0xc6, 0x9f, 0x68, 0xd4,
};
static const unsigned char kat120_addin1[] = {
    0x53, 0x14, 0xfc, 0xcb, 0xe4, 0x52, 0x01, 0x8f, 0xc1, 0x6e, 0x5f, 0xfb,
    0x5d, 0x90, 0xe8, 0x88,
};
static const unsigned char kat120_retbits[] = {
    0x21, 0x1d, 0x90, 0xec, 0xd7, 0xd3, 0xdd, 0xd0, 0x3f, 0x91, 0xe8, 0xa6,
    0x74, 0x40, 0x0f, 0xb1, 0x8d, 0x3f, 0xb3, 0x98, 0x86, 0xa2, 0x26, 0x2f,
    0x0e, 0x09, 0xff, 0x07, 0x5f, 0x99, 0x8d, 0x21, 0x74, 0x19, 0xf2, 0xe9,
    0x35, 0x3d, 0x45, 0xb9, 0xdf, 0x10, 0x72, 0x3e, 0xba, 0xeb, 0xcf, 0xf1,
    0xaa, 0x80, 0x24, 0xd9, 0xb0, 0x96, 0x06, 0x4d, 0x6f, 0x31, 0x83, 0x5b,
    0x75, 0xf8, 0xea, 0xa6,
};
static const struct drbg_kat_no_reseed kat120_t = {
    14, kat120_entropyin, kat120_nonce, kat120_persstr,
    kat120_addin0, kat120_addin1, kat120_retbits
};
static const struct drbg_kat kat120 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat120_t
};

static const unsigned char kat121_entropyin[] = {
    0x2e, 0x17, 0x24, 0xdb, 0x48, 0x22, 0x32, 0xa3, 0xe6, 0x1f, 0x92, 0xc1,
    0xc2, 0x66, 0xfa, 0xf8,
};
static const unsigned char kat121_nonce[] = {
    0x38, 0xaa, 0x55, 0x90, 0xf6, 0xbf, 0xaa, 0x4b,
};
static const unsigned char kat121_persstr[] = {0};
static const unsigned char kat121_addin0[] = {0};
static const unsigned char kat121_addin1[] = {0};
static const unsigned char kat121_retbits[] = {
    0x44, 0x38, 0xb4, 0x8a, 0x45, 0xfb, 0x01, 0x41, 0xe3, 0x1f, 0x0a, 0x96,
    0x24, 0xdf, 0xe6, 0xfc, 0xc2, 0xf9, 0xed, 0xc0, 0x75, 0xc0, 0xa5, 0x2b,
    0xc5, 0xfc, 0x46, 0xd8, 0x5a, 0x96, 0x6c, 0x85, 0x3f, 0xee, 0xe6, 0xaf,
    0x91, 0x32, 0x34, 0xb3, 0xf9, 0xa6, 0x79, 0xf6, 0x67, 0x89, 0x8d, 0xc1,
    0x5a, 0x24, 0xaa, 0xed, 0x89, 0xf0, 0x35, 0xbf, 0xa5, 0xda, 0x51, 0x6e,
    0x43, 0x5b, 0xba, 0xd1,
};
static const struct drbg_kat_no_reseed kat121_t = {
    0, kat121_entropyin, kat121_nonce, kat121_persstr,
    kat121_addin0, kat121_addin1, kat121_retbits
};
static const struct drbg_kat kat121 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat121_t
};

static const unsigned char kat122_entropyin[] = {
    0x22, 0x56, 0x4f, 0x77, 0xc4, 0x5b, 0x05, 0x3c, 0xdf, 0x61, 0x43, 0x3e,
    0xb9, 0x6b, 0x1d, 0x7c,
};
static const unsigned char kat122_nonce[] = {
    0xcf, 0x73, 0xe6, 0x20, 0xf8, 0x51, 0x52, 0x03,
};
static const unsigned char kat122_persstr[] = {0};
static const unsigned char kat122_addin0[] = {0};
static const unsigned char kat122_addin1[] = {0};
static const unsigned char kat122_retbits[] = {
    0xc7, 0x90, 0x8e, 0x71, 0x2c, 0x71, 0x6d, 0x1f, 0x5e, 0xd5, 0x33, 0xe1,
    0x42, 0xe7, 0x21, 0x87, 0xea, 0x77, 0xfb, 0x4f, 0x51, 0x6d, 0xc3, 0x1a,
    0xa1, 0x0a, 0x1e, 0x54, 0x9d, 0x85, 0xea, 0xdb, 0x7a, 0x46, 0x46, 0x17,
    0x04, 0x64, 0xc1, 0xf7, 0xa7, 0x52, 0xc0, 0x1a, 0x94, 0x06, 0xbe, 0x66,
    0x43, 0xee, 0x96, 0x7d, 0x04, 0x64, 0xb8, 0x4b, 0x6a, 0x08, 0xb2, 0xed,
    0x0a, 0x7a, 0xcb, 0x07,
};
static const struct drbg_kat_no_reseed kat122_t = {
    1, kat122_entropyin, kat122_nonce, kat122_persstr,
    kat122_addin0, kat122_addin1, kat122_retbits
};
static const struct drbg_kat kat122 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat122_t
};

static const unsigned char kat123_entropyin[] = {
    0x2a, 0x5b, 0x01, 0x2b, 0x09, 0x79, 0x26, 0xe8, 0xf8, 0x57, 0x0f, 0xf8,
    0x69, 0x2c, 0xc5, 0xd1,
};
static const unsigned char kat123_nonce[] = {
    0xde, 0x8e, 0x07, 0x2d, 0x15, 0x81, 0xaf, 0xe6,
};
static const unsigned char kat123_persstr[] = {0};
static const unsigned char kat123_addin0[] = {0};
static const unsigned char kat123_addin1[] = {0};
static const unsigned char kat123_retbits[] = {
    0x81, 0xf2, 0xe2, 0xc8, 0x58, 0x5e, 0x8d, 0xc4, 0x65, 0xd7, 0x8e, 0x7b,
    0x75, 0xb9, 0xf6, 0xc2, 0xbb, 0xdc, 0xde, 0x94, 0x75, 0xd4, 0x25, 0x0f,
    0xb4, 0x9e, 0x04, 0xc5, 0x6e, 0x30, 0x48, 0x9e, 0x24, 0xdf, 0x48, 0x58,
    0xf7, 0x4e, 0xd0, 0x85, 0xcb, 0xa9, 0xf9, 0x92, 0xeb, 0x7d, 0x13, 0xe4,
    0xe0, 0x64, 0xa7, 0x45, 0xf4, 0x51, 0xcb, 0x6e, 0xdf, 0xc3, 0x7c, 0x57,
    0xf3, 0x5e, 0x8d, 0x57,
};
static const struct drbg_kat_no_reseed kat123_t = {
    2, kat123_entropyin, kat123_nonce, kat123_persstr,
    kat123_addin0, kat123_addin1, kat123_retbits
};
static const struct drbg_kat kat123 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat123_t
};

static const unsigned char kat124_entropyin[] = {
    0x2c, 0x6b, 0x05, 0xd1, 0xc8, 0x6a, 0xae, 0x86, 0xa8, 0x9e, 0x81, 0x64,
    0x82, 0x98, 0x32, 0x36,
};
static const unsigned char kat124_nonce[] = {
    0x33, 0x85, 0xd9, 0x29, 0xad, 0xe9, 0x96, 0xe8,
};
static const unsigned char kat124_persstr[] = {0};
static const unsigned char kat124_addin0[] = {0};
static const unsigned char kat124_addin1[] = {0};
static const unsigned char kat124_retbits[] = {
    0x52, 0x56, 0xa6, 0x44, 0x29, 0xa5, 0x56, 0xd1, 0xa9, 0x1d, 0x58, 0x99,
    0x9c, 0x75, 0xb3, 0x6d, 0xe7, 0xcc, 0x01, 0xf7, 0x46, 0x3c, 0x4e, 0x24,
    0xaf, 0xd1, 0x5d, 0xe0, 0xa3, 0x5d, 0xcb, 0x5a, 0xda, 0x26, 0x79, 0x13,
    0x4f, 0x15, 0xf4, 0xc5, 0x1d, 0xc0, 0x6b, 0x34, 0x45, 0x4d, 0x6d, 0xca,
    0xa1, 0xd2, 0x51, 0x1c, 0x1d, 0x22, 0x6f, 0x23, 0x2f, 0x44, 0x52, 0x76,
    0x25, 0x57, 0x51, 0xe6,
};
static const struct drbg_kat_no_reseed kat124_t = {
    3, kat124_entropyin, kat124_nonce, kat124_persstr,
    kat124_addin0, kat124_addin1, kat124_retbits
};
static const struct drbg_kat kat124 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat124_t
};

static const unsigned char kat125_entropyin[] = {
    0x48, 0xba, 0xb2, 0xf8, 0x2d, 0x80, 0x59, 0x7f, 0x93, 0xad, 0xdb, 0x7e,
    0x1f, 0x0f, 0x2e, 0x72,
};
static const unsigned char kat125_nonce[] = {
    0xf5, 0xf4, 0x90, 0x18, 0xe8, 0x55, 0x9b, 0x0a,
};
static const unsigned char kat125_persstr[] = {0};
static const unsigned char kat125_addin0[] = {0};
static const unsigned char kat125_addin1[] = {0};
static const unsigned char kat125_retbits[] = {
    0xfa, 0x32, 0x3d, 0x2d, 0xae, 0x96, 0x74, 0xd8, 0x0b, 0xd2, 0xf1, 0x42,
    0x7c, 0x1c, 0x79, 0x53, 0x2b, 0x23, 0x74, 0xed, 0x1f, 0xb3, 0xa1, 0x3c,
    0x62, 0x06, 0x05, 0x04, 0x8a, 0xc5, 0x78, 0xb3, 0x07, 0x0c, 0x67, 0x48,
    0x31, 0x4e, 0x5e, 0xd1, 0xdb, 0xd3, 0xea, 0xaa, 0x64, 0x1e, 0x50, 0x5c,
    0x3d, 0x3f, 0x59, 0xfa, 0xc2, 0x5d, 0x89, 0x7b, 0xf3, 0x94, 0xdc, 0xad,
    0xb6, 0x3b, 0x7f, 0xf9,
};
static const struct drbg_kat_no_reseed kat125_t = {
    4, kat125_entropyin, kat125_nonce, kat125_persstr,
    kat125_addin0, kat125_addin1, kat125_retbits
};
static const struct drbg_kat kat125 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat125_t
};

static const unsigned char kat126_entropyin[] = {
    0xad, 0x7a, 0xff, 0x42, 0x24, 0xe9, 0x3f, 0x32, 0x35, 0x45, 0x41, 0x6a,
    0x1e, 0x56, 0x97, 0xce,
};
static const unsigned char kat126_nonce[] = {
    0x9c, 0xc1, 0x2c, 0xe2, 0xed, 0x5e, 0x8d, 0x1c,
};
static const unsigned char kat126_persstr[] = {0};
static const unsigned char kat126_addin0[] = {0};
static const unsigned char kat126_addin1[] = {0};
static const unsigned char kat126_retbits[] = {
    0x36, 0xc7, 0x22, 0x23, 0xb9, 0x24, 0x4c, 0xdb, 0x2c, 0x2c, 0x0d, 0xd5,
    0xa5, 0x97, 0x06, 0x55, 0x8e, 0x2e, 0x5a, 0x11, 0x84, 0x5c, 0xab, 0xf9,
    0x54, 0x5c, 0xd4, 0xad, 0x08, 0x15, 0x4a, 0x46, 0x70, 0x3a, 0xe7, 0x50,
    0xb7, 0xf0, 0xc4, 0xf5, 0xbb, 0x33, 0xac, 0xd3, 0xc3, 0x81, 0xe5, 0xee,
    0x4c, 0xe0, 0x99, 0x16, 0x43, 0x11, 0x44, 0xe8, 0x51, 0x5f, 0xed, 0x91,
    0x4d, 0x5c, 0x5b, 0x5e,
};
static const struct drbg_kat_no_reseed kat126_t = {
    5, kat126_entropyin, kat126_nonce, kat126_persstr,
    kat126_addin0, kat126_addin1, kat126_retbits
};
static const struct drbg_kat kat126 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat126_t
};

static const unsigned char kat127_entropyin[] = {
    0x29, 0x9c, 0x01, 0xd3, 0xa2, 0xf1, 0x32, 0x3d, 0xf7, 0x53, 0xcf, 0x14,
    0x84, 0x5e, 0x0d, 0xe5,
};
static const unsigned char kat127_nonce[] = {
    0x51, 0x1e, 0x36, 0x23, 0x2a, 0x11, 0x29, 0x1d,
};
static const unsigned char kat127_persstr[] = {0};
static const unsigned char kat127_addin0[] = {0};
static const unsigned char kat127_addin1[] = {0};
static const unsigned char kat127_retbits[] = {
    0x77, 0x09, 0xfd, 0xc6, 0x27, 0x8d, 0xb4, 0x4b, 0x21, 0xd3, 0x9a, 0x19,
    0x4b, 0x80, 0x6e, 0x48, 0xe7, 0xfe, 0x3e, 0x9a, 0xe1, 0x16, 0xe2, 0x38,
    0xc2, 0x05, 0xc2, 0xc3, 0x45, 0x98, 0x1c, 0xe8, 0x1f, 0x25, 0x57, 0x13,
    0x59, 0x7c, 0xec, 0x2b, 0x3a, 0xd3, 0x91, 0x0f, 0x2b, 0x67, 0x42, 0xae,
    0xa6, 0x64, 0x04, 0x71, 0x2d, 0xf8, 0x32, 0x8d, 0x2f, 0x2d, 0x19, 0x48,
    0x41, 0x30, 0x97, 0xdb,
};
static const struct drbg_kat_no_reseed kat127_t = {
    6, kat127_entropyin, kat127_nonce, kat127_persstr,
    kat127_addin0, kat127_addin1, kat127_retbits
};
static const struct drbg_kat kat127 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat127_t
};

static const unsigned char kat128_entropyin[] = {
    0xb6, 0xee, 0x77, 0x79, 0x99, 0x4b, 0xa8, 0xcd, 0x49, 0x0a, 0x03, 0xcf,
    0x68, 0x99, 0xb1, 0x35,
};
static const unsigned char kat128_nonce[] = {
    0x66, 0xed, 0xa9, 0xb5, 0xa5, 0x4d, 0x7d, 0xed,
};
static const unsigned char kat128_persstr[] = {0};
static const unsigned char kat128_addin0[] = {0};
static const unsigned char kat128_addin1[] = {0};
static const unsigned char kat128_retbits[] = {
    0x4e, 0x21, 0xb4, 0x8f, 0xde, 0x08, 0x22, 0x63, 0xd7, 0x6a, 0x10, 0x34,
    0xe8, 0x7a, 0x56, 0x6e, 0x1a, 0x1c, 0x9d, 0x2e, 0x1b, 0xd5, 0xc7, 0x48,
    0xe3, 0x0e, 0x1d, 0x87, 0x50, 0xf2, 0xff, 0x03, 0x93, 0x1c, 0x4b, 0xfe,
    0x19, 0x4d, 0x2d, 0xa4, 0xed, 0x1c, 0xf1, 0x53, 0x03, 0x01, 0xe5, 0xb1,
    0xab, 0xc4, 0xbd, 0x2b, 0xda, 0x7b, 0xe8, 0x92, 0x84, 0xf8, 0xc2, 0x19,
    0x36, 0x88, 0xc9, 0x82,
};
static const struct drbg_kat_no_reseed kat128_t = {
    7, kat128_entropyin, kat128_nonce, kat128_persstr,
    kat128_addin0, kat128_addin1, kat128_retbits
};
static const struct drbg_kat kat128 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat128_t
};

static const unsigned char kat129_entropyin[] = {
    0xe3, 0xcc, 0xb9, 0x91, 0xc3, 0xe1, 0xb3, 0xed, 0xa0, 0xb4, 0x0b, 0x51,
    0x42, 0xac, 0x84, 0xd3,
};
static const unsigned char kat129_nonce[] = {
    0x99, 0x87, 0x57, 0xe0, 0x0d, 0xa2, 0xb9, 0xef,
};
static const unsigned char kat129_persstr[] = {0};
static const unsigned char kat129_addin0[] = {0};
static const unsigned char kat129_addin1[] = {0};
static const unsigned char kat129_retbits[] = {
    0xd3, 0x2b, 0xc1, 0x90, 0x99, 0x8f, 0x18, 0xe9, 0xd5, 0x50, 0x9f, 0x46,
    0x02, 0x29, 0x05, 0xd0, 0x11, 0xbb, 0xec, 0x77, 0x4f, 0x05, 0x83, 0x69,
    0x1d, 0x48, 0x12, 0x43, 0x79, 0xe8, 0x1d, 0x99, 0xf0, 0xcd, 0xd4, 0x61,
    0x38, 0xbc, 0xcc, 0x47, 0xeb, 0x77, 0x3f, 0x25, 0x7a, 0x66, 0x2b, 0x79,
    0x8f, 0xab, 0x27, 0x58, 0x86, 0x94, 0x89, 0x36, 0xbc, 0xe7, 0x2d, 0xbd,
    0x2c, 0x61, 0x88, 0xac,
};
static const struct drbg_kat_no_reseed kat129_t = {
    8, kat129_entropyin, kat129_nonce, kat129_persstr,
    kat129_addin0, kat129_addin1, kat129_retbits
};
static const struct drbg_kat kat129 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat129_t
};

static const unsigned char kat130_entropyin[] = {
    0xc3, 0x4e, 0x39, 0x80, 0x41, 0xcc, 0xee, 0x23, 0x89, 0x7f, 0x7c, 0xa3,
    0x43, 0xf3, 0x56, 0x40,
};
static const unsigned char kat130_nonce[] = {
    0xa2, 0x4f, 0x8e, 0xa0, 0x88, 0x6b, 0xf6, 0xbf,
};
static const unsigned char kat130_persstr[] = {0};
static const unsigned char kat130_addin0[] = {0};
static const unsigned char kat130_addin1[] = {0};
static const unsigned char kat130_retbits[] = {
    0x5f, 0xf0, 0xda, 0xef, 0x30, 0x04, 0xc5, 0x03, 0xb8, 0x09, 0x8e, 0x3b,
    0x96, 0x8a, 0x8e, 0x32, 0x33, 0x19, 0xbe, 0x78, 0x6c, 0x7b, 0x74, 0x2a,
    0xac, 0xee, 0x35, 0x5f, 0x1a, 0x3c, 0x9d, 0xe7, 0x50, 0x61, 0x10, 0x8e,
    0x79, 0x18, 0x13, 0x61, 0xf2, 0xe3, 0x30, 0x6a, 0xf0, 0x7b, 0xcd, 0xed,
    0x10, 0xe3, 0x2d, 0xef, 0x1b, 0x7b, 0xb3, 0xe4, 0xeb, 0xc1, 0x70, 0x96,
    0xc6, 0x93, 0x30, 0x58,
};
static const struct drbg_kat_no_reseed kat130_t = {
    9, kat130_entropyin, kat130_nonce, kat130_persstr,
    kat130_addin0, kat130_addin1, kat130_retbits
};
static const struct drbg_kat kat130 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat130_t
};

static const unsigned char kat131_entropyin[] = {
    0xfc, 0x1f, 0xd2, 0x5c, 0xed, 0x5b, 0xd3, 0x0a, 0x62, 0x1c, 0xd4, 0xae,
    0x77, 0x96, 0x80, 0xad,
};
static const unsigned char kat131_nonce[] = {
    0xaf, 0x17, 0xa9, 0xc9, 0x74, 0xb3, 0x6e, 0x6c,
};
static const unsigned char kat131_persstr[] = {0};
static const unsigned char kat131_addin0[] = {0};
static const unsigned char kat131_addin1[] = {0};
static const unsigned char kat131_retbits[] = {
    0x59, 0xf9, 0x9d, 0x08, 0x57, 0x49, 0x36, 0x74, 0x78, 0x68, 0x4a, 0x5d,
    0xdc, 0x8f, 0xe1, 0x81, 0xb9, 0x7a, 0x4e, 0x67, 0xfd, 0xe5, 0xc1, 0x51,
    0xc4, 0x69, 0x6d, 0x52, 0x3d, 0x7c, 0x14, 0xb7, 0x26, 0x89, 0xa9, 0x5a,
    0x5b, 0x60, 0x92, 0xe9, 0x49, 0xdd, 0x16, 0x3b, 0xd8, 0xf9, 0xe4, 0x57,
    0x27, 0xd2, 0xb8, 0xa3, 0x1e, 0xd2, 0x88, 0xc8, 0xc6, 0x22, 0x9e, 0x8b,
    0xe6, 0x80, 0x8e, 0xc8,
};
static const struct drbg_kat_no_reseed kat131_t = {
    10, kat131_entropyin, kat131_nonce, kat131_persstr,
    kat131_addin0, kat131_addin1, kat131_retbits
};
static const struct drbg_kat kat131 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat131_t
};

static const unsigned char kat132_entropyin[] = {
    0x72, 0xc0, 0xe2, 0x3d, 0x92, 0x07, 0x0a, 0x8b, 0xab, 0x70, 0x7f, 0x65,
    0xd5, 0x95, 0x18, 0x6d,
};
static const unsigned char kat132_nonce[] = {
    0x1a, 0x63, 0xdc, 0xfc, 0x52, 0xe5, 0x5c, 0x58,
};
static const unsigned char kat132_persstr[] = {0};
static const unsigned char kat132_addin0[] = {0};
static const unsigned char kat132_addin1[] = {0};
static const unsigned char kat132_retbits[] = {
    0x49, 0xfe, 0xc2, 0x67, 0x58, 0x85, 0xd5, 0x4a, 0x4c, 0x6b, 0x10, 0x1f,
    0x29, 0x19, 0x45, 0xc7, 0x35, 0xad, 0x9c, 0x2d, 0xb5, 0x1a, 0x63, 0xb9,
    0x41, 0xbc, 0x18, 0x2e, 0xe5, 0x1f, 0xd8, 0xfd, 0x84, 0xb8, 0xc6, 0x33,
    0x7b, 0x0f, 0x77, 0xf3, 0x10, 0xca, 0x50, 0x69, 0x3b, 0x91, 0xb5, 0x90,
    0xe3, 0xef, 0x65, 0x5b, 0xe7, 0xad, 0x76, 0x21, 0xed, 0x21, 0xff, 0x39,
    0xd3, 0x29, 0x2e, 0xfd,
};
static const struct drbg_kat_no_reseed kat132_t = {
    11, kat132_entropyin, kat132_nonce, kat132_persstr,
    kat132_addin0, kat132_addin1, kat132_retbits
};
static const struct drbg_kat kat132 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat132_t
};

static const unsigned char kat133_entropyin[] = {
    0xa0, 0x90, 0x91, 0xca, 0x32, 0x80, 0xf7, 0xf5, 0x83, 0x76, 0xbf, 0x68,
    0x0d, 0xe1, 0x11, 0x92,
};
static const unsigned char kat133_nonce[] = {
    0x1d, 0x08, 0xdc, 0x0d, 0x06, 0x00, 0x95, 0xb3,
};
static const unsigned char kat133_persstr[] = {0};
static const unsigned char kat133_addin0[] = {0};
static const unsigned char kat133_addin1[] = {0};
static const unsigned char kat133_retbits[] = {
    0x0d, 0x59, 0xac, 0xa0, 0x23, 0x64, 0x7f, 0x0b, 0xf2, 0x68, 0x81, 0xd9,
    0x12, 0x6b, 0xab, 0x8c, 0x7f, 0xea, 0x92, 0x2d, 0x2b, 0x4c, 0x24, 0xf1,
    0xdc, 0xc0, 0xbf, 0xf8, 0x7a, 0x3d, 0x0d, 0x1b, 0x1d, 0xa0, 0xe8, 0x75,
    0x62, 0x6a, 0x56, 0x42, 0x61, 0x86, 0xef, 0xd0, 0x07, 0x1f, 0x5a, 0x78,
    0x9f, 0xbf, 0x35, 0xfa, 0x8b, 0xfc, 0x85, 0xaf, 0xaf, 0xd3, 0xaf, 0x6c,
    0x9c, 0x7c, 0xd0, 0x7a,
};
static const struct drbg_kat_no_reseed kat133_t = {
    12, kat133_entropyin, kat133_nonce, kat133_persstr,
    kat133_addin0, kat133_addin1, kat133_retbits
};
static const struct drbg_kat kat133 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat133_t
};

static const unsigned char kat134_entropyin[] = {
    0x8b, 0x41, 0x75, 0xd0, 0xa1, 0x95, 0x39, 0xef, 0x3d, 0x5d, 0x40, 0x84,
    0x6c, 0xb4, 0x0a, 0xb3,
};
static const unsigned char kat134_nonce[] = {
    0x8c, 0x31, 0x0d, 0x43, 0x1c, 0xf0, 0x0b, 0x3e,
};
static const unsigned char kat134_persstr[] = {0};
static const unsigned char kat134_addin0[] = {0};
static const unsigned char kat134_addin1[] = {0};
static const unsigned char kat134_retbits[] = {
    0x92, 0xb3, 0xb0, 0xe5, 0x7a, 0xe4, 0xce, 0xfc, 0x40, 0xd6, 0xe5, 0xfa,
    0x0d, 0x9f, 0xa8, 0x5c, 0x97, 0x0c, 0x2d, 0xd0, 0xcd, 0x4e, 0x04, 0xd7,
    0x27, 0x27, 0x56, 0x63, 0x4b, 0x84, 0xcc, 0x20, 0xc5, 0x36, 0x8f, 0x3a,
    0x7b, 0x3e, 0x12, 0x11, 0xc5, 0xfa, 0x2e, 0x63, 0x35, 0x43, 0x6b, 0x88,
    0x58, 0x2d, 0x04, 0x8b, 0xc7, 0x6a, 0x7c, 0x19, 0xbb, 0xfe, 0xc1, 0x35,
    0xa1, 0x05, 0x5c, 0xbd,
};
static const struct drbg_kat_no_reseed kat134_t = {
    13, kat134_entropyin, kat134_nonce, kat134_persstr,
    kat134_addin0, kat134_addin1, kat134_retbits
};
static const struct drbg_kat kat134 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat134_t
};

static const unsigned char kat135_entropyin[] = {
    0x68, 0xc8, 0x53, 0xb9, 0x12, 0x9c, 0xb2, 0x0a, 0x3d, 0xdf, 0x11, 0xe7,
    0x8a, 0x58, 0x75, 0xba,
};
static const unsigned char kat135_nonce[] = {
    0xf5, 0xab, 0x54, 0x86, 0x05, 0xa5, 0x11, 0x03,
};
static const unsigned char kat135_persstr[] = {0};
static const unsigned char kat135_addin0[] = {0};
static const unsigned char kat135_addin1[] = {0};
static const unsigned char kat135_retbits[] = {
    0x43, 0xcf, 0xb0, 0x3a, 0x51, 0xd7, 0xda, 0x40, 0xb3, 0x94, 0x68, 0x36,
    0x1c, 0x2e, 0xe0, 0x78, 0x81, 0x90, 0x75, 0x87, 0x2f, 0x4f, 0x7c, 0x5d,
    0x2b, 0x09, 0xef, 0x39, 0x91, 0x46, 0x02, 0xa7, 0x2a, 0x62, 0xc6, 0x3e,
    0x29, 0x38, 0x3f, 0xbb, 0x9e, 0x45, 0x0f, 0xb2, 0xae, 0xf3, 0x2e, 0xb9,
    0xf3, 0x70, 0xcb, 0xbc, 0x1a, 0xb4, 0x70, 0x8a, 0x5d, 0x28, 0x98, 0xdf,
    0x8a, 0xe4, 0xf6, 0x26,
};
static const struct drbg_kat_no_reseed kat135_t = {
    14, kat135_entropyin, kat135_nonce, kat135_persstr,
    kat135_addin0, kat135_addin1, kat135_retbits
};
static const struct drbg_kat kat135 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat135_t
};

static const unsigned char kat136_entropyin[] = {
    0x9b, 0xfa, 0xef, 0xb6, 0x98, 0xb1, 0xb5, 0xfc, 0xc6, 0x2d, 0xb2, 0xc1,
    0x64, 0x98, 0xc3, 0x3a,
};
static const unsigned char kat136_nonce[] = {
    0x11, 0x1d, 0x86, 0x12, 0xa0, 0xf0, 0x4e, 0x2a,
};
static const unsigned char kat136_persstr[] = {0};
static const unsigned char kat136_addin0[] = {
    0xae, 0xdb, 0xe0, 0x28, 0x47, 0xb1, 0xb0, 0x8b, 0x6a, 0x67, 0x3b, 0xdf,
    0x25, 0xb0, 0x22, 0x4c,
};
static const unsigned char kat136_addin1[] = {
    0x99, 0x01, 0xea, 0xd6, 0x2c, 0xe5, 0x65, 0x73, 0xb0, 0xf7, 0x1c, 0xd0,
    0x20, 0xfe, 0x34, 0x69,
};
static const unsigned char kat136_retbits[] = {
    0xdf, 0xf8, 0xbf, 0x2a, 0xec, 0x53, 0x1f, 0x85, 0x32, 0x60, 0x7e, 0x73,
    0x8b, 0xd7, 0x9f, 0x91, 0xd6, 0x08, 0x5c, 0xb1, 0x95, 0x68, 0xb7, 0xb0,
    0x24, 0x0c, 0xe6, 0xa6, 0xb3, 0x71, 0xa2, 0x82, 0xba, 0xfc, 0xdb, 0xa0,
    0x21, 0x37, 0xdf, 0x99, 0x05, 0x35, 0xd9, 0xeb, 0xf0, 0xba, 0x77, 0x11,
    0x77, 0x51, 0x62, 0x6b, 0x26, 0x78, 0xac, 0xa7, 0xbe, 0x4d, 0xec, 0xfd,
    0x6b, 0x9d, 0x4b, 0x38,
};
static const struct drbg_kat_no_reseed kat136_t = {
    0, kat136_entropyin, kat136_nonce, kat136_persstr,
    kat136_addin0, kat136_addin1, kat136_retbits
};
static const struct drbg_kat kat136 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat136_t
};

static const unsigned char kat137_entropyin[] = {
    0xdf, 0x99, 0x3f, 0xed, 0xd5, 0x96, 0x74, 0xa8, 0x7a, 0x15, 0x14, 0x7b,
    0x80, 0xbe, 0x37, 0xae,
};
static const unsigned char kat137_nonce[] = {
    0x22, 0xb3, 0x31, 0x5a, 0xcc, 0xf1, 0xad, 0x13,
};
static const unsigned char kat137_persstr[] = {0};
static const unsigned char kat137_addin0[] = {
    0x9f, 0x9b, 0x94, 0x09, 0x04, 0x8a, 0x71, 0x17, 0x45, 0xc7, 0xef, 0x7a,
    0x6d, 0xdd, 0xc1, 0x7d,
};
static const unsigned char kat137_addin1[] = {
    0xc5, 0x60, 0x68, 0x5b, 0xd4, 0x9c, 0x05, 0x9f, 0x04, 0x38, 0xe9, 0xdf,
    0xf6, 0x2d, 0x82, 0xd7,
};
static const unsigned char kat137_retbits[] = {
    0x04, 0xd7, 0x4f, 0xa1, 0xb6, 0x9d, 0xe6, 0x89, 0x3a, 0x47, 0xbf, 0xb0,
    0xb6, 0xae, 0x58, 0xa7, 0x98, 0x4b, 0xbb, 0x08, 0x8f, 0xce, 0x62, 0x0b,
    0x9d, 0x8e, 0xbc, 0x0b, 0x54, 0xcd, 0xdb, 0xca, 0x00, 0x45, 0xd7, 0x5d,
    0x5b, 0x04, 0x6f, 0xcd, 0x88, 0x95, 0xc1, 0x6b, 0x05, 0x13, 0xaa, 0x52,
    0x1b, 0x8d, 0x4a, 0xf2, 0x76, 0x78, 0x3d, 0x9d, 0x25, 0x77, 0xac, 0xb3,
    0x2c, 0xea, 0xdb, 0x89,
};
static const struct drbg_kat_no_reseed kat137_t = {
    1, kat137_entropyin, kat137_nonce, kat137_persstr,
    kat137_addin0, kat137_addin1, kat137_retbits
};
static const struct drbg_kat kat137 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat137_t
};

static const unsigned char kat138_entropyin[] = {
    0xdb, 0x4f, 0xae, 0x0c, 0x43, 0x02, 0xeb, 0xe8, 0x4f, 0x2a, 0x28, 0xad,
    0x98, 0x4d, 0xae, 0xc1,
};
static const unsigned char kat138_nonce[] = {
    0x07, 0x9e, 0xbf, 0x0f, 0x00, 0x93, 0xcd, 0xb2,
};
static const unsigned char kat138_persstr[] = {0};
static const unsigned char kat138_addin0[] = {
    0x2f, 0xfb, 0x64, 0x85, 0xac, 0xe2, 0xad, 0x77, 0xd5, 0xf8, 0xad, 0xbd,
    0x09, 0xb3, 0xf3, 0x72,
};
static const unsigned char kat138_addin1[] = {
    0xf4, 0xb8, 0x44, 0x74, 0x3d, 0x00, 0x65, 0xa0, 0x38, 0x98, 0x80, 0x49,
    0xf6, 0xaa, 0x53, 0xb5,
};
static const unsigned char kat138_retbits[] = {
    0xcb, 0x8e, 0x71, 0x44, 0x8f, 0xf7, 0x91, 0x1e, 0x5c, 0xae, 0xa7, 0xd5,
    0x4a, 0x12, 0x94, 0x5e, 0xd3, 0x4b, 0xea, 0x42, 0xdb, 0xb6, 0x57, 0x3b,
    0xf8, 0x42, 0x0d, 0xbb, 0x6b, 0xae, 0x6a, 0x11, 0xe0, 0x24, 0x82, 0x92,
    0xc9, 0xd9, 0x3d, 0xa3, 0x09, 0x68, 0x56, 0xe0, 0xf2, 0x94, 0x18, 0xf1,
    0xc8, 0x0d, 0x13, 0x8b, 0x34, 0x15, 0x66, 0x7b, 0xfb, 0x45, 0x6b, 0x08,
    0x9f, 0x26, 0x62, 0x1a,
};
static const struct drbg_kat_no_reseed kat138_t = {
    2, kat138_entropyin, kat138_nonce, kat138_persstr,
    kat138_addin0, kat138_addin1, kat138_retbits
};
static const struct drbg_kat kat138 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat138_t
};

static const unsigned char kat139_entropyin[] = {
    0xa4, 0x23, 0x48, 0xf1, 0xbe, 0xc0, 0x6f, 0x58, 0x36, 0xfc, 0xa0, 0x60,
    0x66, 0x14, 0x34, 0xc0,
};
static const unsigned char kat139_nonce[] = {
    0x2f, 0x70, 0xf6, 0xa2, 0xe7, 0xd0, 0xb4, 0x36,
};
static const unsigned char kat139_persstr[] = {0};
static const unsigned char kat139_addin0[] = {
    0x54, 0x70, 0x91, 0xef, 0x85, 0xb0, 0x13, 0xf3, 0xf5, 0xdc, 0x82, 0x2a,
    0x5b, 0x3f, 0x27, 0xf9,
};
static const unsigned char kat139_addin1[] = {
    0x2c, 0xa9, 0x9e, 0xe7, 0x97, 0xf4, 0xa9, 0xb1, 0x67, 0x88, 0xd2, 0x98,
    0xbb, 0xa2, 0xd1, 0x83,
};
static const unsigned char kat139_retbits[] = {
    0x4b, 0x41, 0xed, 0x62, 0x81, 0xa8, 0x1d, 0x44, 0x2a, 0x8c, 0xb0, 0x3a,
    0x81, 0x45, 0xfe, 0x78, 0x86, 0x3d, 0x25, 0xd7, 0xee, 0x70, 0xe7, 0x2d,
    0x4d, 0x3a, 0xfd, 0x51, 0x68, 0x16, 0x48, 0x76, 0xa6, 0x6b, 0x5a, 0xca,
    0x31, 0x8c, 0x9c, 0x91, 0x17, 0x23, 0x05, 0xe3, 0x88, 0xe1, 0xda, 0xc3,
    0x86, 0xcb, 0xab, 0x6d, 0x15, 0x0a, 0x39, 0x12, 0x36, 0x9c, 0x93, 0x90,
    0x08, 0x6b, 0xd7, 0x44,
};
static const struct drbg_kat_no_reseed kat139_t = {
    3, kat139_entropyin, kat139_nonce, kat139_persstr,
    kat139_addin0, kat139_addin1, kat139_retbits
};
static const struct drbg_kat kat139 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat139_t
};

static const unsigned char kat140_entropyin[] = {
    0x14, 0x83, 0xb9, 0x8e, 0x08, 0xc0, 0x40, 0x12, 0xb0, 0x3f, 0x1f, 0x61,
    0x02, 0xa2, 0x83, 0x06,
};
static const unsigned char kat140_nonce[] = {
    0x89, 0xe6, 0x7c, 0xa1, 0xda, 0x05, 0xdd, 0x4f,
};
static const unsigned char kat140_persstr[] = {0};
static const unsigned char kat140_addin0[] = {
    0x30, 0x85, 0x9e, 0xcc, 0x30, 0x25, 0x04, 0x79, 0x30, 0xb3, 0x85, 0x3a,
    0xe6, 0x8a, 0xbd, 0xa2,
};
static const unsigned char kat140_addin1[] = {
    0xf5, 0x05, 0x57, 0xf0, 0x40, 0x29, 0x28, 0x6c, 0xaf, 0xb2, 0xfc, 0xbf,
    0x14, 0x64, 0xe4, 0xe5,
};
static const unsigned char kat140_retbits[] = {
    0x45, 0xac, 0xf5, 0x2f, 0xf5, 0x69, 0xcc, 0x48, 0x15, 0x95, 0x88, 0x01,
    0x62, 0x89, 0x3b, 0x57, 0x9b, 0xe4, 0x05, 0x43, 0xc5, 0x6a, 0x94, 0xb7,
    0x18, 0x4b, 0xc9, 0x92, 0xd1, 0xdf, 0x37, 0x85, 0x18, 0xf9, 0x3f, 0xb4,
    0xd9, 0x75, 0x5d, 0x86, 0x58, 0xe9, 0x72, 0x2b, 0x34, 0x79, 0x53, 0x6b,
    0xa4, 0x83, 0x9a, 0x8e, 0xee, 0x7a, 0x65, 0x5f, 0xed, 0x88, 0x4a, 0xaf,
    0x5a, 0x55, 0x6e, 0xb7,
};
static const struct drbg_kat_no_reseed kat140_t = {
    4, kat140_entropyin, kat140_nonce, kat140_persstr,
    kat140_addin0, kat140_addin1, kat140_retbits
};
static const struct drbg_kat kat140 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat140_t
};

static const unsigned char kat141_entropyin[] = {
    0xbe, 0x9f, 0xcd, 0xd7, 0x11, 0xce, 0x1b, 0x0a, 0x3c, 0x97, 0xb1, 0xac,
    0x19, 0x79, 0x20, 0x7d,
};
static const unsigned char kat141_nonce[] = {
    0xa8, 0x02, 0x94, 0xb5, 0xd6, 0x59, 0xff, 0xb4,
};
static const unsigned char kat141_persstr[] = {0};
static const unsigned char kat141_addin0[] = {
    0xb2, 0x0c, 0x04, 0x54, 0x6a, 0x0c, 0x6a, 0x5e, 0x6b, 0x95, 0x63, 0x7c,
    0x96, 0x0b, 0xca, 0x63,
};
static const unsigned char kat141_addin1[] = {
    0x67, 0x07, 0xcc, 0x21, 0x7f, 0xb1, 0x98, 0xaf, 0x85, 0x2e, 0x06, 0xdb,
    0x14, 0x61, 0x58, 0x05,
};
static const unsigned char kat141_retbits[] = {
    0x6b, 0x62, 0x0c, 0x76, 0xf0, 0xb1, 0xa4, 0xa3, 0xd7, 0xf0, 0xf6, 0x0a,
    0x76, 0x45, 0x81, 0x1b, 0xb7, 0x9b, 0xf1, 0x84, 0x77, 0xf8, 0x5b, 0x94,
    0xf0, 0xee, 0x09, 0xe0, 0xc6, 0x8b, 0x0f, 0xdb, 0x3e, 0x11, 0xd8, 0xe5,
    0x8a, 0x34, 0xd2, 0x4f, 0xe3, 0x6e, 0x8b, 0x7e, 0xa7, 0x2a, 0x26, 0xb6,
    0xcb, 0x92, 0xd9, 0x87, 0xc5, 0x81, 0xab, 0x48, 0xe5, 0xfb, 0x8f, 0xb7,
    0x0f, 0x76, 0x5f, 0x5f,
};
static const struct drbg_kat_no_reseed kat141_t = {
    5, kat141_entropyin, kat141_nonce, kat141_persstr,
    kat141_addin0, kat141_addin1, kat141_retbits
};
static const struct drbg_kat kat141 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat141_t
};

static const unsigned char kat142_entropyin[] = {
    0xa5, 0x8e, 0xb5, 0x4d, 0x10, 0x9b, 0x51, 0x4d, 0xb3, 0x38, 0xfc, 0xce,
    0xa2, 0x8e, 0xf2, 0xea,
};
static const unsigned char kat142_nonce[] = {
    0x91, 0x2d, 0x24, 0xa1, 0xd8, 0x1d, 0xe7, 0xdc,
};
static const unsigned char kat142_persstr[] = {0};
static const unsigned char kat142_addin0[] = {
    0x59, 0xd4, 0x59, 0xb0, 0x73, 0x52, 0x60, 0x21, 0xec, 0x45, 0x51, 0x12,
    0x56, 0xcf, 0xb3, 0x58,
};
static const unsigned char kat142_addin1[] = {
    0xf7, 0x9f, 0x7e, 0x5a, 0x2a, 0x79, 0x1a, 0x39, 0xbb, 0x32, 0xb6, 0xb1,
    0x5f, 0xe0, 0x14, 0x61,
};
static const unsigned char kat142_retbits[] = {
    0xc1, 0x26, 0xd4, 0x09, 0xc2, 0x0d, 0x5d, 0x3d, 0x4c, 0x79, 0x46, 0x12,
    0x56, 0xf5, 0x40, 0x3d, 0x1c, 0xc5, 0x9b, 0xcf, 0x11, 0xa4, 0xa6, 0x16,
    0xab, 0xf0, 0x65, 0x30, 0x32, 0xe0, 0x45, 0x0a, 0x11, 0xf3, 0x2f, 0x38,
    0x16, 0xc3, 0x51, 0x91, 0x2f, 0xe8, 0xe3, 0x00, 0x84, 0xed, 0x34, 0xdb,
    0xcd, 0x0a, 0x9c, 0x95, 0xe1, 0xc0, 0x13, 0x62, 0xef, 0x61, 0x6d, 0xd2,
    0x2a, 0x13, 0x7f, 0x72,
};
static const struct drbg_kat_no_reseed kat142_t = {
    6, kat142_entropyin, kat142_nonce, kat142_persstr,
    kat142_addin0, kat142_addin1, kat142_retbits
};
static const struct drbg_kat kat142 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat142_t
};

static const unsigned char kat143_entropyin[] = {
    0xe0, 0xa5, 0xd9, 0x4d, 0x0c, 0xa9, 0x5a, 0x7b, 0x38, 0x0b, 0x99, 0x9c,
    0x69, 0xd9, 0x01, 0x1c,
};
static const unsigned char kat143_nonce[] = {
    0x47, 0xcd, 0xc5, 0x5a, 0x19, 0x56, 0xaf, 0x8d,
};
static const unsigned char kat143_persstr[] = {0};
static const unsigned char kat143_addin0[] = {
    0xa9, 0x16, 0x84, 0xd0, 0x12, 0xdc, 0x48, 0x83, 0x0b, 0xe4, 0xcf, 0xae,
    0xe2, 0xfc, 0x88, 0x54,
};
static const unsigned char kat143_addin1[] = {
    0xa0, 0x61, 0xc2, 0xb1, 0x3b, 0xf1, 0x00, 0x6d, 0x3b, 0xa2, 0xf2, 0x29,
    0x7f, 0x95, 0x4f, 0xb5,
};
static const unsigned char kat143_retbits[] = {
    0xe1, 0x36, 0xea, 0x57, 0x3d, 0x1d, 0x81, 0xec, 0xb7, 0x84, 0x2e, 0xc4,
    0x3a, 0xf0, 0xb4, 0xb9, 0x78, 0x3f, 0x3f, 0xe4, 0xb1, 0xab, 0x90, 0x24,
    0xc6, 0x2e, 0xae, 0xf0, 0x86, 0x0e, 0x81, 0x3d, 0x5a, 0x24, 0xf1, 0xa5,
    0xfa, 0xb7, 0x4b, 0x8f, 0x1f, 0x66, 0x1b, 0x50, 0x39, 0x29, 0x02, 0x56,
    0xd0, 0xe5, 0xaa, 0xa9, 0xe0, 0xfd, 0xab, 0x3a, 0x18, 0x9b, 0x2d, 0x66,
    0x95, 0x89, 0x39, 0x99,
};
static const struct drbg_kat_no_reseed kat143_t = {
    7, kat143_entropyin, kat143_nonce, kat143_persstr,
    kat143_addin0, kat143_addin1, kat143_retbits
};
static const struct drbg_kat kat143 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat143_t
};

static const unsigned char kat144_entropyin[] = {
    0x91, 0x8b, 0xda, 0x9e, 0x89, 0x72, 0x7b, 0xbd, 0x97, 0x04, 0x25, 0xc7,
    0x48, 0x3e, 0xf1, 0xaf,
};
static const unsigned char kat144_nonce[] = {
    0xfc, 0x3b, 0xee, 0xec, 0x38, 0x0b, 0x02, 0xc2,
};
static const unsigned char kat144_persstr[] = {0};
static const unsigned char kat144_addin0[] = {
    0x6f, 0x48, 0x36, 0x73, 0xaf, 0x64, 0xbc, 0xd7, 0x4c, 0xfe, 0xff, 0x3f,
    0x98, 0xb6, 0xcd, 0x5a,
};
static const unsigned char kat144_addin1[] = {
    0x15, 0xbd, 0x5e, 0xab, 0xba, 0x23, 0x2d, 0xf5, 0xa1, 0xb6, 0x03, 0xa2,
    0xfc, 0x16, 0x39, 0xc0,
};
static const unsigned char kat144_retbits[] = {
    0x43, 0x85, 0x55, 0x1a, 0x3a, 0x7d, 0xd9, 0x0d, 0x77, 0xa7, 0x65, 0xf7,
    0xcb, 0x85, 0x9d, 0x80, 0xab, 0x6a, 0xc4, 0x84, 0x8e, 0x3b, 0xd9, 0x11,
    0x01, 0xb4, 0x52, 0x32, 0x34, 0xca, 0x0f, 0x16, 0xc0, 0x6a, 0x45, 0x79,
    0xa9, 0x7e, 0xa5, 0x1d, 0xb0, 0x37, 0xfd, 0x8b, 0x8c, 0x40, 0x83, 0xcd,
    0xb6, 0xf4, 0xed, 0x97, 0x56, 0xf5, 0xa4, 0x48, 0x8b, 0xb5, 0xdf, 0xcf,
    0x7a, 0x8f, 0x8a, 0x9a,
};
static const struct drbg_kat_no_reseed kat144_t = {
    8, kat144_entropyin, kat144_nonce, kat144_persstr,
    kat144_addin0, kat144_addin1, kat144_retbits
};
static const struct drbg_kat kat144 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat144_t
};

static const unsigned char kat145_entropyin[] = {
    0xd8, 0xad, 0xd3, 0x82, 0xd4, 0xdc, 0x7a, 0x1d, 0x10, 0xd4, 0x63, 0x03,
    0x17, 0x39, 0x18, 0x01,
};
static const unsigned char kat145_nonce[] = {
    0x6f, 0xc0, 0xb1, 0xa3, 0xaa, 0x2b, 0x3a, 0xdb,
};
static const unsigned char kat145_persstr[] = {0};
static const unsigned char kat145_addin0[] = {
    0x10, 0x1f, 0xcf, 0xd6, 0x60, 0x80, 0x75, 0xdc, 0xbb, 0xf2, 0xeb, 0xe8,
    0x32, 0xd9, 0x6b, 0x13,
};
static const unsigned char kat145_addin1[] = {
    0x63, 0x2f, 0x7a, 0x11, 0xfb, 0xa9, 0x79, 0xd2, 0x97, 0x9e, 0x41, 0x7e,
    0x2d, 0xed, 0x1d, 0x30,
};
static const unsigned char kat145_retbits[] = {
    0xa6, 0xa1, 0x55, 0x95, 0xbb, 0xd8, 0x57, 0xbb, 0x62, 0x15, 0x04, 0xc8,
    0x5f, 0x03, 0x03, 0x3c, 0xe4, 0x7b, 0x5d, 0xf8, 0x62, 0x96, 0xb8, 0x0d,
    0x40, 0xd0, 0x09, 0xfe, 0x6c, 0x5e, 0x1f, 0xff, 0xf0, 0x84, 0x04, 0x56,
    0x99, 0x0a, 0x14, 0xf3, 0xa1, 0xc4, 0x9c, 0x36, 0x73, 0x7b, 0x70, 0xb6,
    0x2f, 0x40, 0x6f, 0xa3, 0xc5, 0x32, 0x95, 0x2f, 0xb2, 0x2e, 0xfc, 0x76,
    0x00, 0x9a, 0x1b, 0x1a,
};
static const struct drbg_kat_no_reseed kat145_t = {
    9, kat145_entropyin, kat145_nonce, kat145_persstr,
    kat145_addin0, kat145_addin1, kat145_retbits
};
static const struct drbg_kat kat145 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat145_t
};

static const unsigned char kat146_entropyin[] = {
    0xd1, 0x9f, 0xa3, 0x38, 0xc7, 0xbd, 0x52, 0x40, 0x74, 0x7b, 0xac, 0xab,
    0x70, 0x32, 0x8f, 0xb3,
};
static const unsigned char kat146_nonce[] = {
    0x5b, 0x42, 0x82, 0x35, 0x41, 0x42, 0x83, 0xb4,
};
static const unsigned char kat146_persstr[] = {0};
static const unsigned char kat146_addin0[] = {
    0xf3, 0x7c, 0x9f, 0xeb, 0x90, 0xc9, 0x6a, 0x26, 0x3f, 0x41, 0xb1, 0xc4,
    0x98, 0xd5, 0xe0, 0x75,
};
static const unsigned char kat146_addin1[] = {
    0xa3, 0xd8, 0x86, 0x96, 0xfc, 0x1a, 0x44, 0x70, 0x28, 0x42, 0x31, 0x71,
    0xba, 0xd4, 0x65, 0x24,
};
static const unsigned char kat146_retbits[] = {
    0x11, 0xee, 0x72, 0xb4, 0x81, 0xd5, 0x54, 0xf0, 0xfe, 0x49, 0xdc, 0x27,
    0x37, 0x46, 0x6d, 0x5f, 0x5a, 0x64, 0x76, 0xa2, 0xb5, 0xb2, 0xf9, 0x3c,
    0xd6, 0x0e, 0xe8, 0xab, 0x1b, 0xf7, 0x56, 0x3d, 0x3e, 0xbc, 0x60, 0x5e,
    0x44, 0xc3, 0x65, 0xe7, 0x86, 0x5b, 0xff, 0x31, 0xc0, 0x77, 0xd1, 0x76,
    0xd3, 0x61, 0xa4, 0x24, 0x06, 0x27, 0xde, 0xb2, 0x8a, 0xd5, 0x68, 0x50,
    0x46, 0x91, 0xf9, 0x47,
};
static const struct drbg_kat_no_reseed kat146_t = {
    10, kat146_entropyin, kat146_nonce, kat146_persstr,
    kat146_addin0, kat146_addin1, kat146_retbits
};
static const struct drbg_kat kat146 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat146_t
};

static const unsigned char kat147_entropyin[] = {
    0xc3, 0xd8, 0x2c, 0x63, 0x01, 0x41, 0x77, 0xe6, 0x2b, 0xda, 0x82, 0xdd,
    0xe9, 0x11, 0x54, 0x50,
};
static const unsigned char kat147_nonce[] = {
    0x5e, 0x6f, 0x3b, 0x1c, 0x75, 0x70, 0x6f, 0x5f,
};
static const unsigned char kat147_persstr[] = {0};
static const unsigned char kat147_addin0[] = {
    0x72, 0xa7, 0x17, 0xc6, 0x6f, 0xb2, 0x69, 0x06, 0xd6, 0x99, 0x5d, 0x3e,
    0xec, 0xc5, 0x79, 0xd9,
};
static const unsigned char kat147_addin1[] = {
    0xb1, 0xb7, 0xa9, 0xad, 0x3c, 0x64, 0x35, 0x5d, 0x5d, 0x70, 0xf0, 0x3c,
    0x7b, 0x83, 0x29, 0xb0,
};
static const unsigned char kat147_retbits[] = {
    0x34, 0xc3, 0x09, 0xf2, 0x35, 0x27, 0x09, 0xa9, 0x1f, 0x1f, 0x1a, 0x6a,
    0xfb, 0x60, 0x14, 0x66, 0xf8, 0x6d, 0xc8, 0x02, 0x2b, 0xcb, 0xfd, 0xee,
    0x09, 0x51, 0x90, 0xa8, 0x5f, 0xe1, 0xf0, 0x33, 0x2b, 0x8b, 0xab, 0x32,
    0xf4, 0x42, 0x49, 0x43, 0x00, 0x41, 0xcb, 0x0c, 0xf2, 0xc4, 0x0d, 0x9c,
    0xdd, 0x0c, 0x0d, 0xf4, 0x23, 0xe3, 0x4f, 0x2d, 0xa8, 0xd7, 0xf7, 0xb8,
    0x32, 0x94, 0x56, 0x19,
};
static const struct drbg_kat_no_reseed kat147_t = {
    11, kat147_entropyin, kat147_nonce, kat147_persstr,
    kat147_addin0, kat147_addin1, kat147_retbits
};
static const struct drbg_kat kat147 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat147_t
};

static const unsigned char kat148_entropyin[] = {
    0x1c, 0x34, 0x80, 0x46, 0x79, 0x40, 0x8c, 0x0d, 0xd1, 0xc0, 0xb0, 0xb2,
    0xcb, 0x4c, 0x0c, 0x8d,
};
static const unsigned char kat148_nonce[] = {
    0x23, 0xe6, 0xd5, 0xa1, 0x31, 0x74, 0x0e, 0xc9,
};
static const unsigned char kat148_persstr[] = {0};
static const unsigned char kat148_addin0[] = {
    0x8b, 0x99, 0xe4, 0x48, 0x55, 0x10, 0xe7, 0xc4, 0xef, 0x31, 0x86, 0xc7,
    0x5d, 0x0f, 0x42, 0x1d,
};
static const unsigned char kat148_addin1[] = {
    0xcc, 0x1e, 0xcf, 0x02, 0x3c, 0x9b, 0xea, 0xfb, 0x63, 0x48, 0x69, 0x57,
    0x32, 0x7c, 0x2b, 0xde,
};
static const unsigned char kat148_retbits[] = {
    0x7f, 0xbf, 0x33, 0xf0, 0x51, 0x08, 0x78, 0x6f, 0xe0, 0x19, 0x97, 0x14,
    0x6a, 0x27, 0xe5, 0x94, 0x68, 0x76, 0x49, 0x9a, 0x8b, 0xa5, 0x2a, 0x71,
    0x47, 0x16, 0xd9, 0x82, 0xce, 0xa2, 0x3f, 0x39, 0x2d, 0x40, 0x23, 0x4d,
    0x3e, 0x00, 0x33, 0x8d, 0x1a, 0xc8, 0x80, 0x9d, 0x43, 0xc7, 0x79, 0x42,
    0x40, 0x30, 0x19, 0x3b, 0xc6, 0x12, 0x3b, 0x70, 0x67, 0xb6, 0xa8, 0xc2,
    0xed, 0x17, 0x9a, 0x25,
};
static const struct drbg_kat_no_reseed kat148_t = {
    12, kat148_entropyin, kat148_nonce, kat148_persstr,
    kat148_addin0, kat148_addin1, kat148_retbits
};
static const struct drbg_kat kat148 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat148_t
};

static const unsigned char kat149_entropyin[] = {
    0xaa, 0x70, 0xaa, 0x9d, 0x3c, 0x0a, 0x77, 0xf8, 0x68, 0x38, 0x80, 0x5e,
    0xef, 0x34, 0x82, 0xba,
};
static const unsigned char kat149_nonce[] = {
    0xcf, 0x7a, 0x0f, 0x57, 0xc7, 0xde, 0xdf, 0x50,
};
static const unsigned char kat149_persstr[] = {0};
static const unsigned char kat149_addin0[] = {
    0xf6, 0xd4, 0x76, 0xea, 0xe4, 0x2f, 0x02, 0xc9, 0xec, 0x21, 0x98, 0x12,
    0x29, 0xd0, 0xf9, 0x76,
};
static const unsigned char kat149_addin1[] = {
    0x1a, 0xa4, 0x1e, 0xfd, 0xf1, 0x06, 0xe5, 0xa3, 0xe5, 0xd9, 0x76, 0x77,
    0x8b, 0x8f, 0x0c, 0x32,
};
static const unsigned char kat149_retbits[] = {
    0x84, 0xc1, 0xc0, 0x61, 0x9f, 0x69, 0x79, 0xe2, 0xf4, 0xd2, 0xab, 0xc6,
    0x33, 0x35, 0x42, 0xbf, 0x42, 0xfd, 0x32, 0x79, 0x14, 0x2f, 0x07, 0x1d,
    0xad, 0xb2, 0x64, 0x45, 0xfb, 0x2d, 0xe5, 0x1c, 0x4f, 0xad, 0x68, 0x1d,
    0xce, 0x89, 0x18, 0xaf, 0x3a, 0xe1, 0x64, 0x19, 0xfa, 0xfa, 0x3c, 0x55,
    0x26, 0xd8, 0xc4, 0x78, 0x59, 0x9e, 0x85, 0xee, 0x61, 0xa2, 0x73, 0x18,
    0x36, 0x05, 0x45, 0x0e,
};
static const struct drbg_kat_no_reseed kat149_t = {
    13, kat149_entropyin, kat149_nonce, kat149_persstr,
    kat149_addin0, kat149_addin1, kat149_retbits
};
static const struct drbg_kat kat149 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat149_t
};

static const unsigned char kat150_entropyin[] = {
    0x94, 0xc6, 0x97, 0x57, 0xc9, 0x75, 0xe5, 0x3c, 0xa9, 0x44, 0xcb, 0xe8,
    0xf7, 0x68, 0x6f, 0x85,
};
static const unsigned char kat150_nonce[] = {
    0x6e, 0x14, 0x5c, 0xc7, 0xd7, 0x2d, 0xec, 0x57,
};
static const unsigned char kat150_persstr[] = {0};
static const unsigned char kat150_addin0[] = {
    0xce, 0x08, 0x94, 0x6b, 0x04, 0xfd, 0x83, 0x13, 0xdc, 0xda, 0x8c, 0xde,
    0x3b, 0x0f, 0xf9, 0xdf,
};
static const unsigned char kat150_addin1[] = {
    0x9f, 0xfc, 0x93, 0x9d, 0x30, 0x5a, 0xe5, 0x86, 0xd8, 0x6a, 0x1a, 0x14,
    0x70, 0x32, 0x68, 0x0b,
};
static const unsigned char kat150_retbits[] = {
    0xb1, 0xd7, 0xb3, 0x87, 0xa9, 0xf6, 0x7b, 0x91, 0x50, 0x55, 0xe6, 0x82,
    0x98, 0xe0, 0x37, 0x73, 0xa0, 0x19, 0x75, 0x56, 0xf8, 0xd4, 0xb7, 0xe0,
    0x29, 0x52, 0x03, 0x35, 0xef, 0xdc, 0xe2, 0xac, 0xb0, 0xd4, 0xd4, 0xcd,
    0xae, 0x8a, 0xf8, 0xc0, 0x9f, 0xb5, 0x7a, 0x21, 0x5f, 0x20, 0x87, 0xab,
    0xb8, 0x4e, 0xc8, 0x5b, 0x12, 0xdc, 0x3a, 0x85, 0x3d, 0x73, 0xa4, 0xdd,
    0x11, 0x43, 0x00, 0xc5,
};
static const struct drbg_kat_no_reseed kat150_t = {
    14, kat150_entropyin, kat150_nonce, kat150_persstr,
    kat150_addin0, kat150_addin1, kat150_retbits
};
static const struct drbg_kat kat150 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat150_t
};

static const unsigned char kat151_entropyin[] = {
    0xab, 0xa3, 0x6f, 0xf7, 0xa5, 0x35, 0x37, 0x45, 0x4b, 0x5c, 0xb2, 0x68,
    0x39, 0x17, 0x15, 0x40,
};
static const unsigned char kat151_nonce[] = {
    0x68, 0x71, 0xc0, 0xf5, 0x26, 0xfb, 0xcd, 0xc7,
};
static const unsigned char kat151_persstr[] = {
    0xdb, 0xc4, 0x4a, 0xf4, 0x98, 0x16, 0x1f, 0x1f, 0x2a, 0xf6, 0xfc, 0xe6,
    0x6c, 0xcc, 0x30, 0xa8,
};
static const unsigned char kat151_addin0[] = {0};
static const unsigned char kat151_addin1[] = {0};
static const unsigned char kat151_retbits[] = {
    0xaf, 0x68, 0x6e, 0x9a, 0xaf, 0x10, 0xaa, 0xbc, 0xbb, 0x44, 0xb3, 0x74,
    0x89, 0x53, 0xad, 0x18, 0x5d, 0xbf, 0x12, 0x89, 0x8e, 0x52, 0x4d, 0x04,
    0x08, 0x61, 0x02, 0xe4, 0x5f, 0x38, 0x41, 0xc6, 0x50, 0xf6, 0x23, 0xf4,
    0x8f, 0x54, 0x2c, 0xaa, 0x14, 0x79, 0x3e, 0x4f, 0xcb, 0xbc, 0xf2, 0xe4,
    0x61, 0xbe, 0x1c, 0x01, 0xed, 0x8f, 0x1f, 0x48, 0xb9, 0x70, 0x4d, 0x79,
    0xa8, 0xeb, 0xf7, 0x9d,
};
static const struct drbg_kat_no_reseed kat151_t = {
    0, kat151_entropyin, kat151_nonce, kat151_persstr,
    kat151_addin0, kat151_addin1, kat151_retbits
};
static const struct drbg_kat kat151 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat151_t
};

static const unsigned char kat152_entropyin[] = {
    0x47, 0xaf, 0xb8, 0x3e, 0x3a, 0xda, 0x22, 0x0f, 0x7d, 0x7e, 0x63, 0x82,
    0xa8, 0xb3, 0x8c, 0xbb,
};
static const unsigned char kat152_nonce[] = {
    0x7a, 0xb8, 0xae, 0x9b, 0xac, 0x8b, 0x15, 0xa5,
};
static const unsigned char kat152_persstr[] = {
    0x8b, 0xec, 0x1c, 0xb2, 0xd1, 0x80, 0xb3, 0x67, 0x7c, 0xd1, 0xa8, 0x60,
    0x4b, 0x61, 0x4d, 0xbe,
};
static const unsigned char kat152_addin0[] = {0};
static const unsigned char kat152_addin1[] = {0};
static const unsigned char kat152_retbits[] = {
    0x90, 0xcf, 0x1d, 0x9e, 0x65, 0xd9, 0x76, 0xca, 0xce, 0x2f, 0x20, 0xe7,
    0x81, 0x47, 0xd5, 0x04, 0x0d, 0x02, 0x23, 0x7e, 0x04, 0x17, 0x3f, 0x1f,
    0x37, 0x10, 0xe5, 0x22, 0x7d, 0xcb, 0x85, 0x64, 0x68, 0x4f, 0x2e, 0xba,
    0x38, 0xe1, 0xde, 0xf7, 0x2b, 0x93, 0xbe, 0xdb, 0x44, 0x85, 0xf2, 0xb8,
    0x17, 0xee, 0x66, 0xc1, 0x89, 0x02, 0x4b, 0x2a, 0x12, 0x73, 0x65, 0xbc,
    0x83, 0x50, 0x08, 0x71,
};
static const struct drbg_kat_no_reseed kat152_t = {
    1, kat152_entropyin, kat152_nonce, kat152_persstr,
    kat152_addin0, kat152_addin1, kat152_retbits
};
static const struct drbg_kat kat152 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat152_t
};

static const unsigned char kat153_entropyin[] = {
    0xbb, 0x69, 0x93, 0xaf, 0xf4, 0x80, 0x4a, 0x87, 0xdd, 0x42, 0x59, 0x73,
    0xfd, 0xe3, 0x53, 0x11,
};
static const unsigned char kat153_nonce[] = {
    0xfb, 0x14, 0x70, 0x0b, 0x33, 0x5a, 0x01, 0x46,
};
static const unsigned char kat153_persstr[] = {
    0x1e, 0xb0, 0x65, 0x44, 0xca, 0xa8, 0x6e, 0x2a, 0xc4, 0x58, 0x8a, 0xa8,
    0x51, 0xe8, 0xfc, 0x0e,
};
static const unsigned char kat153_addin0[] = {0};
static const unsigned char kat153_addin1[] = {0};
static const unsigned char kat153_retbits[] = {
    0x4e, 0xe7, 0x27, 0x07, 0x3a, 0xba, 0xa3, 0xd7, 0x41, 0x8d, 0x6b, 0x3e,
    0x3b, 0xd4, 0x67, 0xc9, 0x28, 0x48, 0x54, 0x42, 0x3b, 0xa6, 0xd4, 0xef,
    0x02, 0xb2, 0xda, 0xbf, 0x14, 0xb9, 0xb8, 0x24, 0xb2, 0x7a, 0xda, 0x2b,
    0x4a, 0x42, 0xf7, 0xdd, 0x1c, 0xd3, 0x9d, 0xc4, 0x42, 0x0e, 0xe6, 0xe8,
    0x43, 0xfa, 0x7f, 0x2e, 0xee, 0x06, 0xbb, 0x05, 0xc6, 0x47, 0xdc, 0xc0,
    0xd6, 0x97, 0xc0, 0x09,
};
static const struct drbg_kat_no_reseed kat153_t = {
    2, kat153_entropyin, kat153_nonce, kat153_persstr,
    kat153_addin0, kat153_addin1, kat153_retbits
};
static const struct drbg_kat kat153 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat153_t
};

static const unsigned char kat154_entropyin[] = {
    0xa0, 0x5d, 0x9d, 0x84, 0xa5, 0x0d, 0xe7, 0x97, 0x01, 0xc7, 0x59, 0x59,
    0xaf, 0x26, 0xa8, 0xfa,
};
static const unsigned char kat154_nonce[] = {
    0x7e, 0xde, 0x7f, 0x16, 0x30, 0xdc, 0x01, 0xf7,
};
static const unsigned char kat154_persstr[] = {
    0x54, 0xf5, 0xd6, 0xe5, 0xd9, 0xb1, 0xfe, 0xb0, 0xa1, 0xc3, 0xd9, 0xa7,
    0xec, 0x81, 0xdd, 0x98,
};
static const unsigned char kat154_addin0[] = {0};
static const unsigned char kat154_addin1[] = {0};
static const unsigned char kat154_retbits[] = {
    0xf4, 0x7a, 0xab, 0x57, 0xdf, 0xff, 0x93, 0x1d, 0x4b, 0xfc, 0x75, 0xa0,
    0x13, 0x11, 0x86, 0xf0, 0xf5, 0xc1, 0x50, 0x5e, 0x9c, 0x6c, 0x7e, 0xb9,
    0x35, 0xe3, 0x1b, 0x49, 0xf1, 0x34, 0xef, 0xc0, 0x0e, 0x45, 0xfc, 0x96,
    0x73, 0x58, 0xee, 0xc4, 0xa9, 0x21, 0xfd, 0xa0, 0xd0, 0x53, 0x7d, 0x9e,
    0x4f, 0xb3, 0x3b, 0x26, 0x3d, 0xb8, 0xe0, 0x8e, 0x73, 0xf2, 0x1f, 0xe1,
    0x75, 0x05, 0x19, 0x6f,
};
static const struct drbg_kat_no_reseed kat154_t = {
    3, kat154_entropyin, kat154_nonce, kat154_persstr,
    kat154_addin0, kat154_addin1, kat154_retbits
};
static const struct drbg_kat kat154 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat154_t
};

static const unsigned char kat155_entropyin[] = {
    0x80, 0x7d, 0x1b, 0x10, 0x96, 0xe3, 0xd7, 0xf7, 0x89, 0x03, 0x6d, 0x64,
    0x4d, 0xfb, 0x9e, 0x8b,
};
static const unsigned char kat155_nonce[] = {
    0x8c, 0xd7, 0x7f, 0x71, 0xcc, 0xd8, 0xa3, 0x22,
};
static const unsigned char kat155_persstr[] = {
    0x4f, 0x72, 0x74, 0x5e, 0x2f, 0x2a, 0xa5, 0x43, 0x61, 0x89, 0xdb, 0x92,
    0x27, 0x82, 0x0e, 0x46,
};
static const unsigned char kat155_addin0[] = {0};
static const unsigned char kat155_addin1[] = {0};
static const unsigned char kat155_retbits[] = {
    0xff, 0x4a, 0xe0, 0x06, 0x4c, 0x25, 0xfc, 0xef, 0x07, 0x4a, 0xb3, 0x65,
    0x0e, 0xb6, 0xd3, 0x04, 0x4f, 0x86, 0x68, 0x7e, 0x6d, 0xb2, 0x26, 0x29,
    0x99, 0x2b, 0x08, 0xed, 0x07, 0x8c, 0x65, 0xd0, 0x3d, 0xaf, 0x62, 0x41,
    0xa3, 0x10, 0xe5, 0x76, 0x3c, 0x29, 0x86, 0x63, 0x84, 0x8e, 0x32, 0xd0,
    0x61, 0x4e, 0x98, 0x98, 0x9a, 0x16, 0xd3, 0x7d, 0xc1, 0x72, 0x91, 0x35,
    0xfc, 0xa5, 0xe6, 0x2e,
};
static const struct drbg_kat_no_reseed kat155_t = {
    4, kat155_entropyin, kat155_nonce, kat155_persstr,
    kat155_addin0, kat155_addin1, kat155_retbits
};
static const struct drbg_kat kat155 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat155_t
};

static const unsigned char kat156_entropyin[] = {
    0x1a, 0xbc, 0xe2, 0x1f, 0x12, 0x33, 0x68, 0xb2, 0x6c, 0x66, 0x56, 0xb0,
    0xce, 0xbc, 0x03, 0xe8,
};
static const unsigned char kat156_nonce[] = {
    0x1f, 0x7a, 0x6d, 0x5b, 0xc2, 0xcb, 0x97, 0x59,
};
static const unsigned char kat156_persstr[] = {
    0x00, 0xcd, 0x25, 0x9b, 0xb8, 0x7e, 0x4e, 0x6e, 0x21, 0x3e, 0xd4, 0x4e,
    0xec, 0x19, 0xd9, 0x9d,
};
static const unsigned char kat156_addin0[] = {0};
static const unsigned char kat156_addin1[] = {0};
static const unsigned char kat156_retbits[] = {
    0x86, 0xc0, 0x0d, 0x31, 0x81, 0xc1, 0xb6, 0x06, 0xc5, 0x1f, 0x90, 0x98,
    0x93, 0x83, 0xb4, 0xb8, 0x92, 0x2e, 0x4a, 0x19, 0x0b, 0x94, 0x01, 0x65,
    0x8f, 0x8d, 0x45, 0x13, 0x69, 0x7c, 0xa7, 0xd5, 0x84, 0xf5, 0xfc, 0xce,
    0xb3, 0x32, 0x4f, 0x62, 0x4c, 0x47, 0x81, 0xdf, 0xe5, 0x5c, 0x0d, 0xba,
    0x2d, 0x66, 0xbd, 0x85, 0x8f, 0xb6, 0x43, 0xf2, 0x3c, 0xe5, 0xd6, 0x7e,
    0x57, 0x25, 0x00, 0x07,
};
static const struct drbg_kat_no_reseed kat156_t = {
    5, kat156_entropyin, kat156_nonce, kat156_persstr,
    kat156_addin0, kat156_addin1, kat156_retbits
};
static const struct drbg_kat kat156 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat156_t
};

static const unsigned char kat157_entropyin[] = {
    0x06, 0xd5, 0xfb, 0x81, 0x4c, 0xa4, 0xb2, 0xba, 0xc9, 0xe1, 0xcf, 0xba,
    0x0f, 0x13, 0x69, 0x9d,
};
static const unsigned char kat157_nonce[] = {
    0x0e, 0xa8, 0x7b, 0x9f, 0x3b, 0xb1, 0xe6, 0x29,
};
static const unsigned char kat157_persstr[] = {
    0xf6, 0x36, 0x79, 0xdf, 0xa3, 0x53, 0x70, 0x3f, 0x12, 0xe7, 0x23, 0x61,
    0x73, 0xc7, 0xd3, 0x20,
};
static const unsigned char kat157_addin0[] = {0};
static const unsigned char kat157_addin1[] = {0};
static const unsigned char kat157_retbits[] = {
    0xe0, 0xc8, 0xff, 0x1f, 0x1d, 0x2c, 0x69, 0xaf, 0xca, 0xe0, 0xb7, 0x3b,
    0xe8, 0xb3, 0xc4, 0xc7, 0x41, 0x3f, 0x7f, 0xbc, 0xbe, 0xfc, 0x3b, 0xcf,
    0x1e, 0x68, 0x8d, 0x2a, 0x7d, 0x08, 0x49, 0xfd, 0xee, 0x60, 0xbd, 0xe9,
    0x1a, 0x0f, 0xb1, 0xa5, 0xef, 0x4b, 0xf3, 0xdf, 0xb3, 0x36, 0xb7, 0x3e,
    0xd0, 0x47, 0xdf, 0xa7, 0x74, 0x77, 0xa5, 0x1c, 0x6d, 0xee, 0x81, 0x72,
    0x63, 0xa2, 0x0c, 0x37,
};
static const struct drbg_kat_no_reseed kat157_t = {
    6, kat157_entropyin, kat157_nonce, kat157_persstr,
    kat157_addin0, kat157_addin1, kat157_retbits
};
static const struct drbg_kat kat157 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat157_t
};

static const unsigned char kat158_entropyin[] = {
    0xdd, 0x3b, 0x7c, 0xfc, 0x51, 0x8c, 0x18, 0x0c, 0xf2, 0x89, 0xf1, 0x45,
    0x25, 0x15, 0x0c, 0xa5,
};
static const unsigned char kat158_nonce[] = {
    0x36, 0x94, 0x04, 0x3c, 0x71, 0xc2, 0xdd, 0xc5,
};
static const unsigned char kat158_persstr[] = {
    0x3b, 0x1c, 0x08, 0xfa, 0xb6, 0x36, 0x18, 0x51, 0xd7, 0xa5, 0x18, 0xae,
    0x35, 0x5b, 0x8c, 0x9b,
};
static const unsigned char kat158_addin0[] = {0};
static const unsigned char kat158_addin1[] = {0};
static const unsigned char kat158_retbits[] = {
    0x59, 0x16, 0x4b, 0xac, 0x7a, 0x71, 0x4f, 0xcd, 0x5a, 0x45, 0x80, 0xfb,
    0x54, 0xed, 0x4d, 0xdf, 0x99, 0xc3, 0x9c, 0xf5, 0x9c, 0x23, 0xf8, 0x5f,
    0x6c, 0x52, 0x16, 0xf4, 0xe8, 0x9c, 0xf2, 0x8d, 0xa1, 0x59, 0x9f, 0x82,
    0x57, 0xa6, 0xaf, 0xc3, 0x02, 0xed, 0x3a, 0x1d, 0xec, 0x00, 0x3f, 0xf4,
    0x50, 0x91, 0x2c, 0x2b, 0xcd, 0x68, 0x2c, 0xd3, 0x40, 0x79, 0xcf, 0xb3,
    0xcc, 0xf2, 0x59, 0x3a,
};
static const struct drbg_kat_no_reseed kat158_t = {
    7, kat158_entropyin, kat158_nonce, kat158_persstr,
    kat158_addin0, kat158_addin1, kat158_retbits
};
static const struct drbg_kat kat158 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat158_t
};

static const unsigned char kat159_entropyin[] = {
    0xb7, 0x31, 0x4b, 0xac, 0xd0, 0xb3, 0xe9, 0xe2, 0x21, 0x4e, 0x11, 0xa4,
    0x9c, 0x4f, 0xab, 0x54,
};
static const unsigned char kat159_nonce[] = {
    0x8c, 0xd5, 0x9a, 0x9c, 0x39, 0x50, 0x1c, 0x0b,
};
static const unsigned char kat159_persstr[] = {
    0xb6, 0xe3, 0xa4, 0x64, 0x4f, 0xb9, 0x13, 0xa5, 0x4c, 0x89, 0x21, 0xcb,
    0xc1, 0x73, 0x72, 0x38,
};
static const unsigned char kat159_addin0[] = {0};
static const unsigned char kat159_addin1[] = {0};
static const unsigned char kat159_retbits[] = {
    0xe9, 0x2c, 0x97, 0xcc, 0xbb, 0xd6, 0x01, 0x31, 0x78, 0xee, 0x06, 0xd0,
    0x1a, 0xd2, 0xc9, 0xeb, 0x54, 0x64, 0xa7, 0xe3, 0x04, 0x32, 0xb9, 0x43,
    0xe0, 0xb3, 0x71, 0xf1, 0x36, 0xa9, 0x94, 0xb9, 0xf5, 0x44, 0xf3, 0x7b,
    0x60, 0x56, 0x1e, 0x10, 0x25, 0xb1, 0x2b, 0x5a, 0x15, 0xcb, 0x66, 0x1b,
    0x30, 0x1b, 0x5d, 0xd4, 0x38, 0x4b, 0x8b, 0xc0, 0x0d, 0x1d, 0x72, 0xb3,
    0xc6, 0x18, 0xf8, 0x75,
};
static const struct drbg_kat_no_reseed kat159_t = {
    8, kat159_entropyin, kat159_nonce, kat159_persstr,
    kat159_addin0, kat159_addin1, kat159_retbits
};
static const struct drbg_kat kat159 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat159_t
};

static const unsigned char kat160_entropyin[] = {
    0xd6, 0x2b, 0x1f, 0xc2, 0xb0, 0x45, 0x71, 0x8a, 0x3f, 0xe5, 0x9d, 0xfa,
    0xc4, 0xa1, 0xc0, 0x3b,
};
static const unsigned char kat160_nonce[] = {
    0x41, 0x99, 0x17, 0xd5, 0x8a, 0x67, 0x56, 0xa5,
};
static const unsigned char kat160_persstr[] = {
    0xf4, 0x79, 0x4b, 0xdc, 0x10, 0x9f, 0x13, 0x00, 0x4b, 0x89, 0x3b, 0x32,
    0x83, 0xc0, 0x97, 0x7e,
};
static const unsigned char kat160_addin0[] = {0};
static const unsigned char kat160_addin1[] = {0};
static const unsigned char kat160_retbits[] = {
    0x05, 0x42, 0xdf, 0xda, 0xb8, 0x8e, 0x34, 0x91, 0x63, 0xed, 0x9b, 0xa6,
    0x34, 0xee, 0x76, 0x25, 0x7d, 0xc9, 0x27, 0x66, 0x61, 0xcd, 0x5d, 0xd2,
    0xfa, 0xa9, 0x31, 0xbc, 0x3a, 0x2e, 0x9c, 0x2d, 0x17, 0xe5, 0x70, 0xff,
    0xa1, 0xa5, 0xf1, 0x44, 0x96, 0xf0, 0xea, 0xc3, 0x33, 0x9e, 0xfd, 0xe4,
    0x6a, 0xa4, 0x0e, 0x87, 0xf2, 0x1a, 0x98, 0x54, 0x95, 0xfd, 0xa3, 0x94,
    0xf2, 0x06, 0x6e, 0xbb,
};
static const struct drbg_kat_no_reseed kat160_t = {
    9, kat160_entropyin, kat160_nonce, kat160_persstr,
    kat160_addin0, kat160_addin1, kat160_retbits
};
static const struct drbg_kat kat160 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat160_t
};

static const unsigned char kat161_entropyin[] = {
    0x91, 0xb4, 0xaa, 0x7d, 0x56, 0x58, 0x32, 0xe9, 0x6a, 0x21, 0xd6, 0x80,
    0xa0, 0xec, 0xdb, 0x4c,
};
static const unsigned char kat161_nonce[] = {
    0x06, 0x09, 0x09, 0xe2, 0xce, 0x8b, 0x2d, 0xc8,
};
static const unsigned char kat161_persstr[] = {
    0x5c, 0xba, 0xce, 0xb8, 0x9c, 0x15, 0x0d, 0x22, 0x9b, 0x51, 0x6c, 0x34,
    0x93, 0x60, 0xf2, 0x7f,
};
static const unsigned char kat161_addin0[] = {0};
static const unsigned char kat161_addin1[] = {0};
static const unsigned char kat161_retbits[] = {
    0xc0, 0x79, 0xc2, 0x91, 0x7e, 0x8f, 0x6b, 0x84, 0xc5, 0x8e, 0x02, 0x26,
    0xad, 0x0b, 0x8a, 0x60, 0xa8, 0x7b, 0x88, 0x22, 0xa9, 0x90, 0x45, 0x9d,
    0xe2, 0x4c, 0xc6, 0x55, 0x4c, 0x7f, 0x24, 0x1a, 0xff, 0x30, 0xcd, 0xea,
    0x61, 0xa7, 0x48, 0x47, 0x0a, 0x58, 0xc9, 0x4a, 0x15, 0x0e, 0xbd, 0xdc,
    0x35, 0x5c, 0x64, 0x4d, 0xd4, 0x78, 0x6e, 0x36, 0xf1, 0x7e, 0xcf, 0xce,
    0xa7, 0x33, 0x9d, 0x42,
};
static const struct drbg_kat_no_reseed kat161_t = {
    10, kat161_entropyin, kat161_nonce, kat161_persstr,
    kat161_addin0, kat161_addin1, kat161_retbits
};
static const struct drbg_kat kat161 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat161_t
};

static const unsigned char kat162_entropyin[] = {
    0x32, 0xe1, 0x66, 0xc0, 0x93, 0x8d, 0x6d, 0xa7, 0x21, 0xcf, 0xce, 0x23,
    0x5a, 0x66, 0xa1, 0x80,
};
static const unsigned char kat162_nonce[] = {
    0xeb, 0x68, 0x79, 0x96, 0xa8, 0xff, 0x63, 0x94,
};
static const unsigned char kat162_persstr[] = {
    0x05, 0xb9, 0x9f, 0x8d, 0x9f, 0x10, 0x2c, 0x5f, 0xf6, 0xd0, 0xd4, 0x5c,
    0x4d, 0x68, 0x5f, 0xf8,
};
static const unsigned char kat162_addin0[] = {0};
static const unsigned char kat162_addin1[] = {0};
static const unsigned char kat162_retbits[] = {
    0xec, 0x5a, 0xe3, 0x97, 0x25, 0x72, 0x20, 0x45, 0x2e, 0x61, 0x05, 0xa5,
    0xa2, 0x9f, 0x56, 0xd9, 0x33, 0x21, 0x32, 0xf5, 0x33, 0x8d, 0x19, 0xe8,
    0x1a, 0xf6, 0xd3, 0x10, 0x22, 0xbb, 0x91, 0xf8, 0x7f, 0x04, 0xe4, 0x52,
    0x0d, 0x50, 0x76, 0x61, 0x43, 0xd3, 0x53, 0x3e, 0x7b, 0x9d, 0x57, 0x0a,
    0x80, 0x68, 0x84, 0xa3, 0xdf, 0xe2, 0x81, 0xe0, 0x00, 0xf7, 0x7e, 0x7b,
    0x85, 0x59, 0x44, 0x34,
};
static const struct drbg_kat_no_reseed kat162_t = {
    11, kat162_entropyin, kat162_nonce, kat162_persstr,
    kat162_addin0, kat162_addin1, kat162_retbits
};
static const struct drbg_kat kat162 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat162_t
};

static const unsigned char kat163_entropyin[] = {
    0x3a, 0x3d, 0x6a, 0xa9, 0xdb, 0x0f, 0x54, 0x0a, 0x37, 0xeb, 0x81, 0x50,
    0x9d, 0xdb, 0x65, 0x5d,
};
static const unsigned char kat163_nonce[] = {
    0x1f, 0xe8, 0xc2, 0x5b, 0x27, 0xee, 0x3a, 0xbe,
};
static const unsigned char kat163_persstr[] = {
    0x8e, 0x15, 0xa5, 0xad, 0x7a, 0x5f, 0x29, 0x38, 0xeb, 0xdb, 0x5c, 0x08,
    0x8b, 0xdc, 0xf3, 0x07,
};
static const unsigned char kat163_addin0[] = {0};
static const unsigned char kat163_addin1[] = {0};
static const unsigned char kat163_retbits[] = {
    0x8d, 0x7e, 0xae, 0xe5, 0x0e, 0x87, 0xf0, 0xbc, 0x34, 0xa4, 0x9b, 0xe9,
    0x4e, 0x7a, 0x4a, 0x64, 0xfd, 0x1e, 0xc9, 0x86, 0x1e, 0xf3, 0x34, 0x22,
    0x2f, 0xfb, 0x15, 0xaf, 0x89, 0x29, 0xf9, 0xed, 0x42, 0xfb, 0x3f, 0xa6,
    0x29, 0xd6, 0xba, 0x39, 0x70, 0x6b, 0x91, 0x93, 0xe1, 0x2b, 0xa2, 0x40,
    0x44, 0x76, 0x17, 0x7c, 0x4f, 0x69, 0xbf, 0xd1, 0x8c, 0xfe, 0x59, 0x55,
    0x5d, 0xca, 0x7c, 0x04,
};
static const struct drbg_kat_no_reseed kat163_t = {
    12, kat163_entropyin, kat163_nonce, kat163_persstr,
    kat163_addin0, kat163_addin1, kat163_retbits
};
static const struct drbg_kat kat163 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat163_t
};

static const unsigned char kat164_entropyin[] = {
    0xc0, 0x84, 0x69, 0xf8, 0x36, 0xfc, 0x12, 0x48, 0xde, 0xd4, 0xed, 0xd8,
    0xaf, 0x4d, 0x9d, 0xe8,
};
static const unsigned char kat164_nonce[] = {
    0x74, 0x6e, 0xa2, 0xd7, 0x4c, 0xcc, 0xc4, 0xcd,
};
static const unsigned char kat164_persstr[] = {
    0xa5, 0x18, 0x18, 0xae, 0x54, 0x37, 0x56, 0x25, 0x52, 0x65, 0x11, 0x80,
    0xf3, 0xde, 0x5d, 0xae,
};
static const unsigned char kat164_addin0[] = {0};
static const unsigned char kat164_addin1[] = {0};
static const unsigned char kat164_retbits[] = {
    0xa0, 0x8a, 0xee, 0xde, 0x46, 0x3f, 0xc9, 0xac, 0x48, 0x90, 0x7a, 0xb4,
    0xb0, 0xb3, 0x9d, 0x6f, 0x86, 0xa8, 0x79, 0xf4, 0x21, 0x81, 0x61, 0x41,
    0xa0, 0x52, 0xeb, 0x48, 0xd8, 0x7f, 0xf5, 0xd9, 0xb5, 0xb6, 0xcb, 0xfe,
    0xef, 0xdd, 0x8c, 0xfc, 0x17, 0x72, 0xeb, 0x62, 0x67, 0x12, 0x45, 0x3f,
    0x88, 0xec, 0x74, 0x7f, 0x6a, 0x05, 0xaf, 0x91, 0x7c, 0x9f, 0xa1, 0x61,
    0xed, 0xe1, 0x1d, 0x7f,
};
static const struct drbg_kat_no_reseed kat164_t = {
    13, kat164_entropyin, kat164_nonce, kat164_persstr,
    kat164_addin0, kat164_addin1, kat164_retbits
};
static const struct drbg_kat kat164 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat164_t
};

static const unsigned char kat165_entropyin[] = {
    0x0c, 0xe4, 0xc1, 0x82, 0x8f, 0x9c, 0x9f, 0x0a, 0xea, 0xb4, 0x95, 0x36,
    0x72, 0x23, 0x59, 0x23,
};
static const unsigned char kat165_nonce[] = {
    0x46, 0x4b, 0x42, 0xc9, 0xfc, 0x5b, 0x38, 0x31,
};
static const unsigned char kat165_persstr[] = {
    0x48, 0xde, 0x34, 0xd8, 0x48, 0x29, 0x8d, 0xcf, 0x3f, 0x58, 0xc5, 0x2d,
    0x96, 0xc7, 0x7b, 0xf5,
};
static const unsigned char kat165_addin0[] = {0};
static const unsigned char kat165_addin1[] = {0};
static const unsigned char kat165_retbits[] = {
    0xa7, 0xd3, 0x79, 0x02, 0x26, 0xf5, 0xab, 0x5b, 0x38, 0x33, 0xdc, 0xc7,
    0x63, 0xc2, 0xe9, 0xe7, 0xc5, 0xb7, 0x7d, 0x57, 0xc2, 0xfa, 0x26, 0x15,
    0x47, 0xae, 0x0e, 0x39, 0xe9, 0x78, 0x4d, 0xf2, 0x69, 0xd0, 0x8b, 0xbd,
    0x40, 0x36, 0x46, 0x2f, 0x3a, 0xcc, 0xc7, 0x1b, 0x37, 0x8b, 0x09, 0x41,
    0xe9, 0x9c, 0x32, 0x7c, 0x4a, 0x50, 0x34, 0x39, 0xa9, 0x3b, 0x4e, 0xc7,
    0xa0, 0x39, 0xa2, 0xa8,
};
static const struct drbg_kat_no_reseed kat165_t = {
    14, kat165_entropyin, kat165_nonce, kat165_persstr,
    kat165_addin0, kat165_addin1, kat165_retbits
};
static const struct drbg_kat kat165 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat165_t
};

static const unsigned char kat166_entropyin[] = {
    0x70, 0x7a, 0x43, 0x07, 0x2e, 0xac, 0xb8, 0xce, 0xa9, 0x5f, 0xb8, 0x52,
    0x1a, 0xe5, 0x16, 0x0b,
};
static const unsigned char kat166_nonce[] = {
    0xfd, 0x62, 0x12, 0x13, 0x26, 0x16, 0x30, 0xad,
};
static const unsigned char kat166_persstr[] = {
    0x6b, 0xf9, 0xa7, 0x2e, 0xaf, 0xe3, 0x5d, 0x58, 0x3c, 0x91, 0x5c, 0x95,
    0x25, 0x74, 0x7b, 0xa0,
};
static const unsigned char kat166_addin0[] = {
    0xe3, 0xdf, 0xd6, 0x92, 0x7c, 0x4a, 0xe1, 0x03, 0x43, 0x2e, 0xb6, 0x19,
    0x63, 0x67, 0xec, 0xee,
};
static const unsigned char kat166_addin1[] = {
    0xe1, 0x8c, 0xd5, 0x04, 0xe9, 0x40, 0x27, 0x53, 0x3c, 0xf3, 0x30, 0x71,
    0xca, 0x93, 0x1b, 0x60,
};
static const unsigned char kat166_retbits[] = {
    0xee, 0x46, 0x3c, 0xdc, 0x78, 0xdd, 0x25, 0x3c, 0x44, 0x66, 0xdd, 0xc2,
    0xe3, 0x5c, 0xc4, 0xa9, 0x1a, 0xf2, 0x0a, 0xd3, 0xb3, 0x39, 0x66, 0x69,
    0xce, 0xf5, 0x22, 0x1b, 0x0a, 0x9c, 0xcc, 0x5d, 0xfc, 0x72, 0x3b, 0x2f,
    0xe1, 0xe5, 0x7f, 0xa2, 0x6d, 0xd0, 0x30, 0xad, 0x0b, 0x6e, 0xa1, 0x4e,
    0xcb, 0x89, 0xf9, 0x1c, 0x4b, 0xc6, 0x93, 0x82, 0xa0, 0x1d, 0x8d, 0xfa,
    0x6f, 0x1d, 0xff, 0x8a,
};
static const struct drbg_kat_no_reseed kat166_t = {
    0, kat166_entropyin, kat166_nonce, kat166_persstr,
    kat166_addin0, kat166_addin1, kat166_retbits
};
static const struct drbg_kat kat166 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat166_t
};

static const unsigned char kat167_entropyin[] = {
    0x35, 0x67, 0xac, 0x54, 0xb9, 0x03, 0xfd, 0x5d, 0xdc, 0x57, 0x23, 0x4c,
    0x31, 0x9b, 0x14, 0x15,
};
static const unsigned char kat167_nonce[] = {
    0x8f, 0x75, 0xc5, 0x6a, 0x85, 0x77, 0x21, 0x92,
};
static const unsigned char kat167_persstr[] = {
    0xd1, 0xe3, 0xff, 0xed, 0x85, 0x21, 0x1c, 0x40, 0x8e, 0x31, 0xdb, 0x12,
    0xe8, 0x5b, 0x75, 0xdb,
};
static const unsigned char kat167_addin0[] = {
    0xcc, 0x38, 0x84, 0x33, 0x5e, 0x19, 0x42, 0x7c, 0xaf, 0x09, 0xf0, 0x1c,
    0x63, 0x4b, 0x30, 0x8a,
};
static const unsigned char kat167_addin1[] = {
    0xf7, 0x73, 0x3d, 0x02, 0x51, 0xff, 0x02, 0xaf, 0xc0, 0xe9, 0x61, 0xc1,
    0x12, 0x28, 0x84, 0xfc,
};
static const unsigned char kat167_retbits[] = {
    0x28, 0xc6, 0x9b, 0xb4, 0x98, 0x76, 0x42, 0xe5, 0xa4, 0x67, 0x93, 0x8f,
    0x52, 0xa7, 0x37, 0x14, 0xd3, 0x9a, 0xd0, 0x89, 0xee, 0x09, 0xda, 0xbe,
    0x70, 0xd6, 0x90, 0x36, 0x17, 0x8f, 0x59, 0x98, 0x18, 0x6c, 0xc6, 0x45,
    0x10, 0x0b, 0x56, 0xf9, 0x8e, 0xf4, 0x5c, 0xe3, 0x5b, 0x06, 0xf0, 0x9c,
    0x26, 0x20, 0xba, 0x68, 0x0a, 0x5a, 0x03, 0xd3, 0xd9, 0x59, 0x43, 0xed,
    0x17, 0x16, 0x0f, 0xe4,
};
static const struct drbg_kat_no_reseed kat167_t = {
    1, kat167_entropyin, kat167_nonce, kat167_persstr,
    kat167_addin0, kat167_addin1, kat167_retbits
};
static const struct drbg_kat kat167 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat167_t
};

static const unsigned char kat168_entropyin[] = {
    0x41, 0x76, 0x72, 0x58, 0xe0, 0x6e, 0x50, 0x11, 0x24, 0xf3, 0xdb, 0x8b,
    0xe4, 0x58, 0xd3, 0xbf,
};
static const unsigned char kat168_nonce[] = {
    0xde, 0x8f, 0xa0, 0xa7, 0x9f, 0x1a, 0x7f, 0x0c,
};
static const unsigned char kat168_persstr[] = {
    0x9d, 0xab, 0x2f, 0x46, 0xd2, 0x2d, 0x37, 0x86, 0x08, 0xe2, 0x1f, 0xc2,
    0x37, 0xe9, 0x98, 0x71,
};
static const unsigned char kat168_addin0[] = {
    0xea, 0x41, 0xd1, 0x33, 0xb4, 0xf8, 0x6a, 0x35, 0x22, 0x7b, 0x76, 0x16,
    0x7f, 0x51, 0xbb, 0x5f,
};
static const unsigned char kat168_addin1[] = {
    0x71, 0x61, 0xe1, 0x2c, 0x6f, 0x65, 0xc1, 0x42, 0xba, 0xf9, 0x4f, 0x1a,
    0x09, 0xe9, 0x4e, 0x22,
};
static const unsigned char kat168_retbits[] = {
    0xb3, 0x90, 0xc9, 0x52, 0x0e, 0xad, 0x28, 0x82, 0x69, 0x2a, 0x29, 0x59,
    0x3d, 0x16, 0x5c, 0x58, 0xbc, 0x66, 0x14, 0x62, 0x11, 0x2a, 0xf6, 0xd2,
    0x5a, 0x57, 0x14, 0x73, 0xe6, 0x1f, 0x1c, 0xbc, 0xd0, 0x75, 0xe3, 0xad,
    0x1c, 0xae, 0xeb, 0x51, 0xcb, 0x99, 0x28, 0x40, 0x51, 0xce, 0xa4, 0x5a,
    0x17, 0x06, 0xe9, 0x84, 0xe2, 0x7c, 0x32, 0x84, 0xfc, 0x1e, 0x87, 0xab,
    0x68, 0xa6, 0xf1, 0x46,
};
static const struct drbg_kat_no_reseed kat168_t = {
    2, kat168_entropyin, kat168_nonce, kat168_persstr,
    kat168_addin0, kat168_addin1, kat168_retbits
};
static const struct drbg_kat kat168 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat168_t
};

static const unsigned char kat169_entropyin[] = {
    0x19, 0xdb, 0x2d, 0xcd, 0xc5, 0x7d, 0xf2, 0x1e, 0x55, 0xb6, 0x56, 0x4e,
    0xae, 0x47, 0x41, 0x0b,
};
static const unsigned char kat169_nonce[] = {
    0xeb, 0xec, 0x8e, 0xce, 0x62, 0xd3, 0x29, 0xb6,
};
static const unsigned char kat169_persstr[] = {
    0x80, 0xba, 0xec, 0xa8, 0x2b, 0x3b, 0xa2, 0x07, 0xb9, 0xf3, 0x75, 0x08,
    0x38, 0xbd, 0xe4, 0x98,
};
static const unsigned char kat169_addin0[] = {
    0x5d, 0x74, 0xcd, 0xd7, 0xbc, 0xbe, 0x7b, 0xc9, 0x48, 0xb8, 0xf1, 0x3e,
    0xfa, 0x83, 0x5a, 0xd4,
};
static const unsigned char kat169_addin1[] = {
    0x17, 0xbb, 0x7c, 0xb9, 0x1b, 0x48, 0x11, 0xa5, 0xe7, 0xdb, 0x26, 0x78,
    0x4c, 0xbb, 0xc2, 0xe5,
};
static const unsigned char kat169_retbits[] = {
    0x26, 0x54, 0xd9, 0x86, 0x67, 0xf7, 0x56, 0x15, 0xfb, 0x6c, 0x74, 0xd5,
    0xb9, 0x0b, 0x62, 0x3c, 0xcc, 0xb9, 0x62, 0x49, 0xe1, 0x4e, 0x08, 0xe8,
    0x8d, 0x1c, 0xb5, 0xe0, 0x90, 0x93, 0x26, 0xb6, 0x77, 0x67, 0xa5, 0x73,
    0xe0, 0xa4, 0xd1, 0xdf, 0x05, 0x7f, 0x2d, 0xe7, 0xa4, 0x95, 0xde, 0xae,
    0xcb, 0xeb, 0x7e, 0x25, 0xe3, 0xaa, 0x82, 0xde, 0x55, 0x59, 0x08, 0xbf,
    0xe3, 0xcc, 0x6a, 0x85,
};
static const struct drbg_kat_no_reseed kat169_t = {
    3, kat169_entropyin, kat169_nonce, kat169_persstr,
    kat169_addin0, kat169_addin1, kat169_retbits
};
static const struct drbg_kat kat169 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat169_t
};

static const unsigned char kat170_entropyin[] = {
    0x64, 0xd7, 0xb4, 0x2c, 0x52, 0x64, 0x33, 0x76, 0xd8, 0xa1, 0x1b, 0x0d,
    0xa5, 0xcc, 0xa5, 0xe6,
};
static const unsigned char kat170_nonce[] = {
    0x6f, 0xeb, 0x03, 0x5f, 0x39, 0x14, 0x25, 0x14,
};
static const unsigned char kat170_persstr[] = {
    0x3f, 0x12, 0xa6, 0xf2, 0x40, 0x01, 0xb9, 0x8c, 0x1f, 0x24, 0xab, 0xcf,
    0xab, 0x68, 0x95, 0x08,
};
static const unsigned char kat170_addin0[] = {
    0xfe, 0xe4, 0x82, 0xb1, 0xfe, 0xa0, 0x53, 0x65, 0xab, 0xa0, 0x7b, 0xc7,
    0xc3, 0x35, 0x1c, 0x6d,
};
static const unsigned char kat170_addin1[] = {
    0xaf, 0x13, 0x59, 0x83, 0xab, 0x81, 0x13, 0x91, 0x45, 0x35, 0x13, 0x49,
    0x18, 0x78, 0xb5, 0xb5,
};
static const unsigned char kat170_retbits[] = {
    0x0c, 0x21, 0xf6, 0x97, 0xf1, 0xb0, 0x0d, 0x22, 0x83, 0xae, 0xe4, 0xe7,
    0xb7, 0xe1, 0x01, 0x33, 0x38, 0x6a, 0xb0, 0xab, 0xfa, 0xdb, 0x5e, 0x14,
    0xbf, 0xe5, 0xb4, 0x52, 0xa6, 0xd9, 0x17, 0x46, 0x2a, 0x7b, 0xd0, 0xed,
    0xa0, 0x6f, 0xa1, 0x41, 0xb1, 0xbc, 0x3e, 0xab, 0x4f, 0x47, 0x73, 0xa2,
    0x40, 0xbe, 0x73, 0x51, 0xe3, 0x2a, 0x1d, 0xab, 0xdd, 0xc5, 0x7b, 0xae,
    0x02, 0xd2, 0x6c, 0xd2,
};
static const struct drbg_kat_no_reseed kat170_t = {
    4, kat170_entropyin, kat170_nonce, kat170_persstr,
    kat170_addin0, kat170_addin1, kat170_retbits
};
static const struct drbg_kat kat170 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat170_t
};

static const unsigned char kat171_entropyin[] = {
    0xa5, 0x5a, 0x5a, 0x07, 0x29, 0xc3, 0x9e, 0xfb, 0xc8, 0x6c, 0xe6, 0x4d,
    0x91, 0x51, 0x03, 0x4e,
};
static const unsigned char kat171_nonce[] = {
    0x10, 0xf1, 0x15, 0x57, 0x73, 0xfb, 0xcd, 0x3b,
};
static const unsigned char kat171_persstr[] = {
    0x6c, 0x6f, 0x3f, 0x77, 0x80, 0x15, 0xb2, 0xd3, 0x34, 0x92, 0x2e, 0x7a,
    0x13, 0xea, 0xd6, 0xe6,
};
static const unsigned char kat171_addin0[] = {
    0x84, 0x10, 0x15, 0x5a, 0x9f, 0x4d, 0x40, 0xa3, 0x29, 0x8d, 0x5b, 0x0f,
    0x42, 0xc0, 0xef, 0x46,
};
static const unsigned char kat171_addin1[] = {
    0xfd, 0x91, 0x1c, 0x8b, 0xb6, 0xff, 0x14, 0xe7, 0xd7, 0x6a, 0x61, 0x1a,
    0xdd, 0xda, 0x72, 0x11,
};
static const unsigned char kat171_retbits[] = {
    0x16, 0x5f, 0x07, 0xd1, 0x4b, 0xb1, 0xb3, 0x7b, 0xd9, 0xe0, 0x94, 0x9e,
    0x7b, 0x0c, 0x0d, 0xfa, 0xdd, 0x06, 0x7a, 0xdf, 0x15, 0xb6, 0x6c, 0xb3,
    0x96, 0xe6, 0x44, 0x4b, 0xcd, 0xf1, 0x1b, 0xc2, 0x34, 0xb0, 0xd5, 0x9b,
    0x23, 0x17, 0x21, 0x4a, 0x9c, 0xde, 0x72, 0xe5, 0x32, 0xf3, 0x00, 0xc9,
    0xd0, 0x90, 0x7f, 0xf5, 0x9e, 0x62, 0x12, 0x82, 0x6a, 0x92, 0x0f, 0xd9,
    0x2b, 0x53, 0xad, 0xf7,
};
static const struct drbg_kat_no_reseed kat171_t = {
    5, kat171_entropyin, kat171_nonce, kat171_persstr,
    kat171_addin0, kat171_addin1, kat171_retbits
};
static const struct drbg_kat kat171 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat171_t
};

static const unsigned char kat172_entropyin[] = {
    0x53, 0xea, 0x1b, 0xef, 0x71, 0x12, 0xfd, 0xcb, 0x15, 0x51, 0x36, 0x16,
    0xae, 0x7e, 0x07, 0x03,
};
static const unsigned char kat172_nonce[] = {
    0xca, 0xac, 0x5b, 0x2f, 0xbf, 0x5b, 0xd1, 0xb8,
};
static const unsigned char kat172_persstr[] = {
    0x79, 0x04, 0x26, 0x8d, 0x84, 0x6f, 0x81, 0x41, 0xcd, 0x78, 0x3e, 0xcb,
    0xa6, 0xef, 0xc9, 0x01,
};
static const unsigned char kat172_addin0[] = {
    0x03, 0x64, 0xae, 0xea, 0xf7, 0x42, 0x4c, 0x92, 0xc2, 0x61, 0x1f, 0x33,
    0x86, 0xfb, 0xe3, 0x8a,
};
static const unsigned char kat172_addin1[] = {
    0x9c, 0x77, 0xd6, 0x80, 0xc5, 0xf2, 0x70, 0xae, 0xfa, 0x8a, 0x45, 0xb3,
    0xe9, 0x35, 0x53, 0x30,
};
static const unsigned char kat172_retbits[] = {
    0xe5, 0x6f, 0xa9, 0x81, 0x29, 0x23, 0x89, 0x0b, 0x94, 0x81, 0xee, 0x07,
    0x80, 0xea, 0x23, 0xb2, 0x05, 0x94, 0x7e, 0x80, 0xf3, 0xe3, 0x8f, 0x55,
    0xc2, 0x2d, 0xef, 0xce, 0x6c, 0xae, 0xa8, 0x4f, 0xd9, 0x85, 0x0f, 0x2c,
    0x50, 0x9c, 0x51, 0x48, 0xe0, 0x1c, 0xd9, 0x77, 0x34, 0x8c, 0xc8, 0xb3,
    0xcc, 0xab, 0xd3, 0xd7, 0x51, 0x5c, 0xc2, 0xca, 0x95, 0xd0, 0xbc, 0x13,
    0x48, 0xba, 0xc5, 0xc2,
};
static const struct drbg_kat_no_reseed kat172_t = {
    6, kat172_entropyin, kat172_nonce, kat172_persstr,
    kat172_addin0, kat172_addin1, kat172_retbits
};
static const struct drbg_kat kat172 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat172_t
};

static const unsigned char kat173_entropyin[] = {
    0x1f, 0xae, 0xf5, 0x93, 0x5c, 0xb8, 0x2d, 0x27, 0x08, 0x13, 0x25, 0x2c,
    0x77, 0xef, 0x3d, 0x6c,
};
static const unsigned char kat173_nonce[] = {
    0x75, 0xf6, 0x8a, 0x1b, 0x65, 0x8c, 0x72, 0x77,
};
static const unsigned char kat173_persstr[] = {
    0x08, 0x15, 0xfa, 0x38, 0x8d, 0xb0, 0x3a, 0x89, 0x2d, 0xf4, 0x47, 0x4f,
    0x19, 0xde, 0x31, 0xbc,
};
static const unsigned char kat173_addin0[] = {
    0x9f, 0x92, 0x23, 0xae, 0x37, 0x86, 0xd3, 0x8c, 0x43, 0xe0, 0xe9, 0x2a,
    0xa1, 0xbf, 0x5e, 0x3f,
};
static const unsigned char kat173_addin1[] = {
    0x4d, 0x9c, 0x28, 0xcd, 0x69, 0x4e, 0x01, 0x6f, 0xaf, 0xbd, 0xe3, 0xcb,
    0x30, 0x85, 0xb8, 0xd9,
};
static const unsigned char kat173_retbits[] = {
    0xcd, 0x69, 0x3a, 0xab, 0x02, 0xec, 0x80, 0xa0, 0x4b, 0xeb, 0xcf, 0xef,
    0x66, 0x04, 0x38, 0x4f, 0x48, 0xc8, 0xeb, 0x7d, 0x44, 0xd6, 0x91, 0xac,
    0x35, 0xfb, 0x11, 0xe1, 0x81, 0x83, 0x88, 0x4b, 0x9e, 0x35, 0xd5, 0x8e,
    0x62, 0xfc, 0x56, 0x7b, 0x9a, 0x3d, 0xa4, 0x0e, 0x3a, 0xee, 0xa7, 0x83,
    0x11, 0x5e, 0x80, 0x19, 0x22, 0x09, 0x03, 0xab, 0x55, 0x9b, 0xe9, 0x83,
    0x4c, 0x37, 0x77, 0x58,
};
static const struct drbg_kat_no_reseed kat173_t = {
    7, kat173_entropyin, kat173_nonce, kat173_persstr,
    kat173_addin0, kat173_addin1, kat173_retbits
};
static const struct drbg_kat kat173 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat173_t
};

static const unsigned char kat174_entropyin[] = {
    0x0b, 0x8c, 0x52, 0x08, 0xd9, 0x11, 0x3c, 0xbe, 0x8a, 0x31, 0xcf, 0x3d,
    0xe1, 0x43, 0x38, 0x1a,
};
static const unsigned char kat174_nonce[] = {
    0x44, 0x9e, 0xbf, 0x57, 0xb8, 0x2a, 0xa5, 0xcb,
};
static const unsigned char kat174_persstr[] = {
    0x7f, 0x60, 0x01, 0xd6, 0x0c, 0x31, 0xcf, 0x8a, 0x90, 0x4c, 0xb3, 0x18,
    0xcb, 0x70, 0x04, 0x7b,
};
static const unsigned char kat174_addin0[] = {
    0x4f, 0xe0, 0x45, 0xed, 0xbb, 0x98, 0x16, 0x0a, 0x42, 0x63, 0x5b, 0x77,
    0x79, 0x6e, 0x8a, 0x97,
};
static const unsigned char kat174_addin1[] = {
    0xd8, 0x96, 0x4e, 0x5f, 0xa8, 0x16, 0xea, 0x83, 0xd1, 0xa5, 0xab, 0xbd,
    0x15, 0x68, 0xad, 0xaa,
};
static const unsigned char kat174_retbits[] = {
    0xab, 0x1f, 0xba, 0xf1, 0x54, 0xa9, 0x07, 0x2b, 0x62, 0xc3, 0xd6, 0xbb,
    0x9d, 0x01, 0x48, 0xac, 0xe4, 0xb4, 0xcd, 0xe0, 0x8d, 0x9a, 0x71, 0x2a,
    0xa2, 0x94, 0x10, 0xab, 0x51, 0x4d, 0x03, 0xe3, 0x5d, 0xb3, 0xf4, 0x91,
    0x80, 0x66, 0x73, 0x04, 0x18, 0x33, 0xa2, 0xeb, 0xeb, 0xe8, 0x70, 0x55,
    0xf1, 0xf5, 0xfb, 0x64, 0xdd, 0x8e, 0x40, 0x8e, 0x3b, 0x99, 0x57, 0xd2,
    0x1e, 0x66, 0xca, 0xa2,
};
static const struct drbg_kat_no_reseed kat174_t = {
    8, kat174_entropyin, kat174_nonce, kat174_persstr,
    kat174_addin0, kat174_addin1, kat174_retbits
};
static const struct drbg_kat kat174 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat174_t
};

static const unsigned char kat175_entropyin[] = {
    0xae, 0x90, 0x6d, 0x89, 0x04, 0x15, 0x45, 0xc7, 0xef, 0x70, 0x71, 0xf5,
    0x3e, 0xce, 0xea, 0x52,
};
static const unsigned char kat175_nonce[] = {
    0x1f, 0x4b, 0x28, 0x4c, 0x36, 0xaa, 0xe3, 0xba,
};
static const unsigned char kat175_persstr[] = {
    0xb9, 0x19, 0x98, 0xc3, 0x21, 0x83, 0x00, 0x6f, 0x7d, 0xf1, 0xc5, 0xd7,
    0x1f, 0xba, 0xc5, 0x7b,
};
static const unsigned char kat175_addin0[] = {
    0x86, 0x47, 0x5c, 0xa2, 0xb4, 0x84, 0xb8, 0xc6, 0xf3, 0x3b, 0x11, 0x22,
    0x8b, 0xc6, 0x65, 0x05,
};
static const unsigned char kat175_addin1[] = {
    0x0c, 0x34, 0xe0, 0xea, 0xe8, 0xfa, 0x94, 0xc2, 0x6c, 0xb5, 0x11, 0xc7,
    0x9a, 0x49, 0x83, 0x3e,
};
static const unsigned char kat175_retbits[] = {
    0x89, 0xeb, 0xf1, 0x81, 0x5e, 0x2c, 0x33, 0x32, 0x59, 0xd6, 0xaa, 0xa0,
    0x07, 0xd1, 0x9c, 0xa5, 0xe0, 0x1e, 0x59, 0x87, 0x76, 0xac, 0x2a, 0xf5,
    0x5a, 0xbf, 0x78, 0xc1, 0x1b, 0x4e, 0xd7, 0x44, 0xb6, 0xde, 0x87, 0x5a,
    0x36, 0xf4, 0x52, 0xc2, 0x7a, 0xc4, 0xa0, 0xe6, 0x77, 0x93, 0x8b, 0x16,
    0x1b, 0x2c, 0x00, 0xeb, 0x06, 0x59, 0xbd, 0xf2, 0xdc, 0x42, 0x5a, 0x89,
    0xa2, 0x4a, 0xff, 0xd0,
};
static const struct drbg_kat_no_reseed kat175_t = {
    9, kat175_entropyin, kat175_nonce, kat175_persstr,
    kat175_addin0, kat175_addin1, kat175_retbits
};
static const struct drbg_kat kat175 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat175_t
};

static const unsigned char kat176_entropyin[] = {
    0x1d, 0x0e, 0x60, 0xe6, 0x80, 0x17, 0x40, 0x42, 0x1d, 0x29, 0xf6, 0x5a,
    0x60, 0x60, 0x6c, 0x1b,
};
static const unsigned char kat176_nonce[] = {
    0x37, 0x03, 0x36, 0xdb, 0x08, 0x35, 0xe9, 0x1a,
};
static const unsigned char kat176_persstr[] = {
    0x7a, 0x27, 0x3b, 0x4d, 0x99, 0x75, 0x35, 0x11, 0xfc, 0xf3, 0x4f, 0x3b,
    0x1b, 0xb2, 0x9e, 0x16,
};
static const unsigned char kat176_addin0[] = {
    0xd6, 0x44, 0xcb, 0x46, 0xe0, 0x1c, 0x07, 0x12, 0xef, 0x9d, 0xa4, 0xed,
    0xe5, 0xca, 0x40, 0x7c,
};
static const unsigned char kat176_addin1[] = {
    0x5c, 0x99, 0x05, 0x45, 0xee, 0xaf, 0x59, 0xde, 0x4d, 0x34, 0x9a, 0xe6,
    0x66, 0x1f, 0x70, 0x41,
};
static const unsigned char kat176_retbits[] = {
    0xda, 0xd5, 0x03, 0xe7, 0x20, 0x96, 0x02, 0xa8, 0x6f, 0x18, 0xd0, 0x1a,
    0x2b, 0x05, 0x15, 0xb4, 0xb3, 0xd4, 0xa4, 0xc0, 0x37, 0xaf, 0x71, 0x86,
    0xec, 0xba, 0x25, 0xdc, 0xb7, 0x81, 0xd9, 0x73, 0x11, 0x81, 0x0f, 0x98,
    0xc1, 0x7c, 0x34, 0x17, 0x30, 0x61, 0xab, 0x78, 0xec, 0xcb, 0xd4, 0x91,
    0x85, 0xcf, 0x5d, 0x4f, 0xb9, 0x1d, 0xff, 0xf4, 0xb0, 0xb6, 0x53, 0xdc,
    0x15, 0x89, 0x92, 0x29,
};
static const struct drbg_kat_no_reseed kat176_t = {
    10, kat176_entropyin, kat176_nonce, kat176_persstr,
    kat176_addin0, kat176_addin1, kat176_retbits
};
static const struct drbg_kat kat176 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat176_t
};

static const unsigned char kat177_entropyin[] = {
    0x67, 0x58, 0x85, 0x38, 0x3c, 0x6b, 0x3b, 0x00, 0x8e, 0x88, 0x49, 0x37,
    0xed, 0xa7, 0x7b, 0xe8,
};
static const unsigned char kat177_nonce[] = {
    0xb9, 0x1c, 0x7c, 0x06, 0x87, 0x85, 0x32, 0xf2,
};
static const unsigned char kat177_persstr[] = {
    0x12, 0xce, 0x97, 0xb4, 0x0a, 0x1b, 0xcd, 0xe5, 0x03, 0x90, 0xe3, 0x15,
    0xcd, 0x91, 0x1d, 0x53,
};
static const unsigned char kat177_addin0[] = {
    0x10, 0x58, 0x43, 0x4c, 0xf9, 0xb3, 0x8f, 0xd7, 0x2a, 0x14, 0x0c, 0xf0,
    0x94, 0x24, 0x36, 0xe3,
};
static const unsigned char kat177_addin1[] = {
    0x59, 0x8b, 0xf8, 0x01, 0x43, 0x39, 0x82, 0x09, 0x66, 0xd9, 0x4b, 0xe1,
    0xe2, 0xe9, 0xc8, 0x55,
};
static const unsigned char kat177_retbits[] = {
    0x45, 0x48, 0x8c, 0xa0, 0xe2, 0xf2, 0x83, 0xf1, 0x74, 0x1b, 0xcb, 0xf4,
    0x9a, 0x79, 0x36, 0x54, 0x59, 0x07, 0x52, 0x49, 0xe1, 0xc4, 0x17, 0x77,
    0xf3, 0xe0, 0x08, 0xd8, 0x02, 0x0e, 0xe1, 0xf5, 0x98, 0xac, 0xa0, 0xad,
    0x26, 0xed, 0xcd, 0x92, 0x0c, 0x85, 0x59, 0xd0, 0xea, 0xf7, 0xcc, 0x5d,
    0x5a, 0x71, 0x44, 0x37, 0xc0, 0x45, 0x81, 0xa6, 0x4c, 0x74, 0x99, 0xe5,
    0xf0, 0xbe, 0x08, 0x9c,
};
static const struct drbg_kat_no_reseed kat177_t = {
    11, kat177_entropyin, kat177_nonce, kat177_persstr,
    kat177_addin0, kat177_addin1, kat177_retbits
};
static const struct drbg_kat kat177 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat177_t
};

static const unsigned char kat178_entropyin[] = {
    0x71, 0x23, 0x93, 0xa9, 0xf4, 0xa1, 0xb0, 0xbb, 0xab, 0xf4, 0xf4, 0x96,
    0xe8, 0x17, 0x0a, 0xdc,
};
static const unsigned char kat178_nonce[] = {
    0x0f, 0x59, 0x0c, 0x32, 0x39, 0xc6, 0xcf, 0x47,
};
static const unsigned char kat178_persstr[] = {
    0xb7, 0x2b, 0x40, 0x97, 0xa9, 0xfb, 0x3d, 0x45, 0xbb, 0x06, 0x24, 0x15,
    0xb6, 0xf2, 0xdf, 0xb1,
};
static const unsigned char kat178_addin0[] = {
    0xe2, 0x7a, 0xde, 0x7c, 0x11, 0x86, 0x56, 0x55, 0x3f, 0x06, 0xec, 0x20,
    0x19, 0x9b, 0x53, 0x70,
};
static const unsigned char kat178_addin1[] = {
    0x4e, 0xb4, 0xfd, 0xa3, 0xff, 0xda, 0xd4, 0xf9, 0xdd, 0xaf, 0xd0, 0xd0,
    0xbf, 0xec, 0xd4, 0x44,
};
static const unsigned char kat178_retbits[] = {
    0x04, 0x49, 0xa8, 0x2a, 0x31, 0x7e, 0x22, 0xe5, 0xa7, 0xa6, 0x68, 0x4d,
    0x08, 0xb8, 0xf6, 0x3e, 0x02, 0xd8, 0xb9, 0x08, 0x5d, 0x61, 0x9a, 0xb5,
    0x6c, 0xde, 0x52, 0x2c, 0x86, 0x67, 0xd5, 0xf8, 0x8b, 0x2e, 0xa8, 0x83,
    0x03, 0x29, 0x49, 0x1a, 0x5e, 0xa8, 0x28, 0xe0, 0xa3, 0x66, 0xae, 0x7e,
    0x84, 0xf2, 0xcd, 0xee, 0x69, 0x8a, 0x92, 0x41, 0x18, 0x88, 0x83, 0xbe,
    0x00, 0xc5, 0x33, 0xe8,
};
static const struct drbg_kat_no_reseed kat178_t = {
    12, kat178_entropyin, kat178_nonce, kat178_persstr,
    kat178_addin0, kat178_addin1, kat178_retbits
};
static const struct drbg_kat kat178 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat178_t
};

static const unsigned char kat179_entropyin[] = {
    0x57, 0x66, 0x40, 0xf9, 0x76, 0xa6, 0x28, 0xb0, 0x9e, 0x98, 0xaf, 0x48,
    0x3b, 0x64, 0x5e, 0x56,
};
static const unsigned char kat179_nonce[] = {
    0x82, 0x2e, 0x8a, 0x10, 0xdf, 0x06, 0xd0, 0xfe,
};
static const unsigned char kat179_persstr[] = {
    0x6d, 0x6a, 0x6f, 0x68, 0xbb, 0x38, 0x1c, 0x29, 0x1f, 0x3f, 0xa6, 0x21,
    0xd6, 0xab, 0x64, 0xb2,
};
static const unsigned char kat179_addin0[] = {
    0xb0, 0x5f, 0x14, 0xe4, 0x1f, 0x54, 0x33, 0x8c, 0x44, 0x15, 0xa5, 0xb5,
    0xc8, 0xc5, 0x61, 0x99,
};
static const unsigned char kat179_addin1[] = {
    0xc6, 0x84, 0x7c, 0x9e, 0xd7, 0x4c, 0x86, 0x9a, 0xc4, 0xa5, 0x6f, 0xf2,
    0xd0, 0x95, 0x6d, 0x6b,
};
static const unsigned char kat179_retbits[] = {
    0x5c, 0x6a, 0x7e, 0x58, 0xea, 0xcf, 0xa5, 0x8d, 0xca, 0x93, 0x99, 0xa3,
    0xf9, 0xb6, 0xcb, 0x41, 0xd2, 0xbc, 0x24, 0xf6, 0x72, 0x58, 0x7d, 0xb2,
    0x29, 0xbf, 0x22, 0x20, 0xc2, 0x85, 0x13, 0x47, 0xd4, 0x7d, 0xa1, 0xca,
    0x0a, 0x6a, 0x13, 0x77, 0x96, 0x05, 0xdb, 0x90, 0xd9, 0x1a, 0x61, 0x48,
    0x88, 0x18, 0xd3, 0x3b, 0x2a, 0x4a, 0xd6, 0x74, 0x09, 0x00, 0x68, 0x5f,
    0x8e, 0xc6, 0x13, 0x6f,
};
static const struct drbg_kat_no_reseed kat179_t = {
    13, kat179_entropyin, kat179_nonce, kat179_persstr,
    kat179_addin0, kat179_addin1, kat179_retbits
};
static const struct drbg_kat kat179 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat179_t
};

static const unsigned char kat180_entropyin[] = {
    0x69, 0x65, 0x03, 0xc8, 0x48, 0x60, 0x51, 0x87, 0x84, 0xdd, 0x4c, 0x6c,
    0x31, 0x71, 0x0f, 0x43,
};
static const unsigned char kat180_nonce[] = {
    0xee, 0x52, 0xc0, 0xa4, 0xcd, 0x90, 0xdf, 0x1e,
};
static const unsigned char kat180_persstr[] = {
    0x9a, 0x08, 0xd6, 0xb4, 0x80, 0xef, 0x3a, 0xc2, 0x00, 0xa1, 0x30, 0x87,
    0x8d, 0x80, 0xd0, 0x79,
};
static const unsigned char kat180_addin0[] = {
    0x5c, 0x51, 0x31, 0x59, 0x7b, 0xd3, 0xab, 0x02, 0x5c, 0xa0, 0x05, 0x79,
    0xc8, 0x6f, 0xfd, 0xed,
};
static const unsigned char kat180_addin1[] = {
    0xa8, 0xcc, 0x33, 0xc8, 0x51, 0xee, 0x11, 0xe4, 0x70, 0x1a, 0x32, 0x19,
    0x7b, 0x61, 0x76, 0x23,
};
static const unsigned char kat180_retbits[] = {
    0x9a, 0xfd, 0xc4, 0x54, 0xf5, 0xc2, 0xce, 0x29, 0x95, 0xfc, 0x7c, 0xcf,
    0xb4, 0xe8, 0x67, 0x1a, 0x4b, 0x27, 0xdf, 0xcb, 0x1c, 0x02, 0xfe, 0xc6,
    0x07, 0xaa, 0xfc, 0x64, 0x22, 0x65, 0x70, 0x96, 0xce, 0x18, 0x14, 0x10,
    0x1b, 0x8a, 0x74, 0x3e, 0x2d, 0x4a, 0x2c, 0x4e, 0x2e, 0x8d, 0xa2, 0x57,
    0x57, 0x8d, 0xe7, 0x10, 0x54, 0x12, 0xf3, 0x52, 0xa4, 0xcd, 0x14, 0xaf,
    0x7d, 0xb0, 0xeb, 0x03,
};
static const struct drbg_kat_no_reseed kat180_t = {
    14, kat180_entropyin, kat180_nonce, kat180_persstr,
    kat180_addin0, kat180_addin1, kat180_retbits
};
static const struct drbg_kat kat180 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat180_t
};

static const unsigned char kat181_entropyin[] = {
    0x6b, 0xdf, 0x53, 0x32, 0xbd, 0xce, 0x46, 0x55, 0xd4, 0x5c, 0x2c, 0xfe,
    0xa8, 0x97, 0xb0, 0x00,
};
static const unsigned char kat181_nonce[] = {
    0xe7, 0x8c, 0x55, 0x71, 0xc5, 0xf9, 0x26, 0xf9,
};
static const unsigned char kat181_persstr[] = {0};
static const unsigned char kat181_addin0[] = {0};
static const unsigned char kat181_addin1[] = {0};
static const unsigned char kat181_retbits[] = {
    0xe0, 0x71, 0x56, 0x88, 0x76, 0x5a, 0x32, 0x85, 0xe7, 0xb7, 0xdb, 0x55,
    0x5f, 0x27, 0x79, 0x24, 0xe7, 0x17, 0x1f, 0x75, 0x41, 0xbf, 0x26, 0x12,
    0x2b, 0x13, 0xdb, 0xaa, 0xa3, 0x9f, 0x9e, 0x2b, 0x03, 0x45, 0xc6, 0x59,
    0x58, 0x3f, 0xf8, 0xc9, 0xcf, 0xd8, 0x88, 0xf1, 0xab, 0xd2, 0xf3, 0xb3,
    0x6a, 0x7c, 0x9d, 0x47, 0xc6, 0x87, 0xb0, 0x1c, 0x81, 0x9a, 0x9f, 0x98,
    0x88, 0x54, 0x2e, 0x0f,
};
static const struct drbg_kat_no_reseed kat181_t = {
    0, kat181_entropyin, kat181_nonce, kat181_persstr,
    kat181_addin0, kat181_addin1, kat181_retbits
};
static const struct drbg_kat kat181 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat181_t
};

static const unsigned char kat182_entropyin[] = {
    0xa4, 0x73, 0x7d, 0x48, 0xa8, 0x93, 0x25, 0x07, 0x85, 0x79, 0xe6, 0x49,
    0xe2, 0xfa, 0x65, 0xeb,
};
static const unsigned char kat182_nonce[] = {
    0x6a, 0x79, 0x9a, 0x7a, 0x2f, 0x13, 0xe8, 0x13,
};
static const unsigned char kat182_persstr[] = {0};
static const unsigned char kat182_addin0[] = {0};
static const unsigned char kat182_addin1[] = {0};
static const unsigned char kat182_retbits[] = {
    0x75, 0x2a, 0x91, 0x6d, 0x98, 0x05, 0x18, 0xd9, 0xe7, 0xe4, 0x75, 0x99,
    0x06, 0x6b, 0x45, 0x71, 0x46, 0x61, 0xf3, 0x41, 0x59, 0xf0, 0xc8, 0xce,
    0xa8, 0xda, 0xbd, 0x59, 0x6a, 0x06, 0x6a, 0xff, 0x7a, 0xe6, 0xc2, 0x1e,
    0x69, 0xa3, 0x56, 0xdd, 0x2e, 0xe0, 0xda, 0x55, 0x42, 0x9c, 0x67, 0x5a,
    0xa6, 0xfa, 0x09, 0x00, 0x17, 0x3f, 0x54, 0x77, 0xcd, 0x7f, 0xd6, 0x49,
    0xea, 0xe0, 0xc9, 0x9a,
};
static const struct drbg_kat_no_reseed kat182_t = {
    1, kat182_entropyin, kat182_nonce, kat182_persstr,
    kat182_addin0, kat182_addin1, kat182_retbits
};
static const struct drbg_kat kat182 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat182_t
};

static const unsigned char kat183_entropyin[] = {
    0x99, 0xb4, 0xe9, 0xe5, 0xba, 0xaa, 0x99, 0xbf, 0x8b, 0x8a, 0x0c, 0x2c,
    0xfd, 0x9a, 0x89, 0xce,
};
static const unsigned char kat183_nonce[] = {
    0xf4, 0x8b, 0x31, 0x29, 0x41, 0xd3, 0x55, 0x4d,
};
static const unsigned char kat183_persstr[] = {0};
static const unsigned char kat183_addin0[] = {0};
static const unsigned char kat183_addin1[] = {0};
static const unsigned char kat183_retbits[] = {
    0x85, 0xcf, 0x14, 0x8c, 0x65, 0xec, 0xe7, 0x52, 0x5d, 0xea, 0x96, 0x34,
    0x4d, 0xa6, 0x45, 0x11, 0x99, 0x49, 0x21, 0x85, 0xc8, 0xb8, 0xdf, 0xff,
    0x50, 0x0d, 0xdd, 0x68, 0xca, 0xab, 0x50, 0xbd, 0x74, 0x18, 0x66, 0x93,
    0x7a, 0x50, 0x1a, 0xf8, 0x76, 0xae, 0x84, 0x9b, 0x5b, 0x1b, 0x53, 0x30,
    0xde, 0x65, 0xeb, 0xf3, 0x8e, 0x9d, 0x55, 0x96, 0x93, 0xfa, 0xe8, 0x05,
    0xcc, 0xc9, 0xaa, 0xed,
};
static const struct drbg_kat_no_reseed kat183_t = {
    2, kat183_entropyin, kat183_nonce, kat183_persstr,
    kat183_addin0, kat183_addin1, kat183_retbits
};
static const struct drbg_kat kat183 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat183_t
};

static const unsigned char kat184_entropyin[] = {
    0x7c, 0x1e, 0xf8, 0x46, 0xe7, 0xf3, 0xa7, 0xcf, 0xe5, 0xc7, 0xbb, 0x47,
    0xe4, 0x6a, 0xa0, 0xe8,
};
static const unsigned char kat184_nonce[] = {
    0x78, 0xeb, 0x00, 0x29, 0x55, 0xd2, 0x70, 0xd3,
};
static const unsigned char kat184_persstr[] = {0};
static const unsigned char kat184_addin0[] = {0};
static const unsigned char kat184_addin1[] = {0};
static const unsigned char kat184_retbits[] = {
    0xb7, 0x39, 0xf9, 0x53, 0x39, 0xe8, 0x3f, 0xe8, 0xa3, 0x39, 0x16, 0x2f,
    0x1e, 0x96, 0xcc, 0x82, 0xb6, 0x7a, 0xf4, 0x17, 0x59, 0xf4, 0x83, 0x06,
    0x4e, 0x61, 0xa0, 0x39, 0x13, 0x7a, 0xf0, 0x7d, 0x93, 0x4a, 0x0e, 0xb7,
    0xca, 0x72, 0x84, 0xc1, 0x46, 0x86, 0xca, 0x04, 0x88, 0x95, 0x3e, 0xe4,
    0x40, 0xf6, 0xcd, 0x11, 0x45, 0xd0, 0x76, 0x6f, 0x4f, 0xf1, 0xc7, 0x7b,
    0x0d, 0x81, 0xd3, 0x61,
};
static const struct drbg_kat_no_reseed kat184_t = {
    3, kat184_entropyin, kat184_nonce, kat184_persstr,
    kat184_addin0, kat184_addin1, kat184_retbits
};
static const struct drbg_kat kat184 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat184_t
};

static const unsigned char kat185_entropyin[] = {
    0x6f, 0x75, 0x0f, 0xb7, 0x3f, 0xbc, 0xb6, 0xdb, 0x57, 0x50, 0x6c, 0xf2,
    0xde, 0xfa, 0xf6, 0xd4,
};
static const unsigned char kat185_nonce[] = {
    0xa5, 0x2c, 0x92, 0x97, 0xe0, 0x2f, 0x42, 0x55,
};
static const unsigned char kat185_persstr[] = {0};
static const unsigned char kat185_addin0[] = {0};
static const unsigned char kat185_addin1[] = {0};
static const unsigned char kat185_retbits[] = {
    0x0c, 0xa7, 0x48, 0xfc, 0xb3, 0xba, 0x23, 0xbd, 0x04, 0x3f, 0x48, 0x50,
    0x75, 0x85, 0x42, 0x52, 0x34, 0xec, 0x4a, 0x0f, 0x35, 0x0e, 0xfd, 0xcc,
    0x87, 0xf9, 0x06, 0x2c, 0x8a, 0xe0, 0xcf, 0x1e, 0x03, 0x3b, 0x7d, 0xf8,
    0x35, 0x7f, 0x5b, 0x0e, 0x4f, 0x7c, 0x21, 0xd4, 0xfb, 0xdc, 0xf8, 0x9c,
    0xe0, 0x19, 0x9c, 0x25, 0x79, 0x02, 0x70, 0xfe, 0x67, 0xff, 0xee, 0xc2,
    0x6c, 0xfc, 0x4d, 0x18,
};
static const struct drbg_kat_no_reseed kat185_t = {
    4, kat185_entropyin, kat185_nonce, kat185_persstr,
    kat185_addin0, kat185_addin1, kat185_retbits
};
static const struct drbg_kat kat185 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat185_t
};

static const unsigned char kat186_entropyin[] = {
    0x44, 0xb6, 0xc3, 0x9a, 0x3a, 0xf6, 0xa4, 0x73, 0x14, 0x8e, 0x32, 0xd9,
    0x79, 0x2b, 0x9e, 0xee,
};
static const unsigned char kat186_nonce[] = {
    0xde, 0x44, 0xdd, 0x36, 0x00, 0x3e, 0x48, 0x22,
};
static const unsigned char kat186_persstr[] = {0};
static const unsigned char kat186_addin0[] = {0};
static const unsigned char kat186_addin1[] = {0};
static const unsigned char kat186_retbits[] = {
    0x10, 0x12, 0x20, 0x08, 0x7c, 0xa7, 0x70, 0xe4, 0x57, 0x4f, 0xd0, 0x5b,
    0x2e, 0x88, 0x85, 0x1b, 0x48, 0x09, 0xcf, 0x21, 0xe6, 0x5e, 0xb0, 0xe3,
    0xd1, 0xec, 0xda, 0x29, 0xaa, 0xcf, 0x2d, 0x93, 0xe7, 0x5d, 0x79, 0x3b,
    0x99, 0x24, 0xae, 0xdd, 0x9b, 0x9a, 0x38, 0x84, 0x4d, 0xb4, 0x30, 0x47,
    0x0d, 0x01, 0x5d, 0xa6, 0x04, 0x18, 0x75, 0x3f, 0x2e, 0x3c, 0x6a, 0x15,
    0xf5, 0x58, 0xf4, 0xa9,
};
static const struct drbg_kat_no_reseed kat186_t = {
    5, kat186_entropyin, kat186_nonce, kat186_persstr,
    kat186_addin0, kat186_addin1, kat186_retbits
};
static const struct drbg_kat kat186 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat186_t
};

static const unsigned char kat187_entropyin[] = {
    0xe4, 0xd1, 0xff, 0x1c, 0x3c, 0xd2, 0x78, 0x49, 0xa2, 0x97, 0xe5, 0xa9,
    0x8e, 0xde, 0x08, 0x6a,
};
static const unsigned char kat187_nonce[] = {
    0xf8, 0x9d, 0x58, 0xb0, 0x61, 0x7e, 0x5d, 0x10,
};
static const unsigned char kat187_persstr[] = {0};
static const unsigned char kat187_addin0[] = {0};
static const unsigned char kat187_addin1[] = {0};
static const unsigned char kat187_retbits[] = {
    0x54, 0xc4, 0x67, 0x97, 0xa3, 0x8e, 0x11, 0xc5, 0x43, 0x47, 0x78, 0x09,
    0xd9, 0xd6, 0xed, 0x0a, 0xe4, 0x02, 0x02, 0x85, 0xed, 0xf9, 0x9e, 0x7e,
    0xc9, 0x45, 0xcf, 0xcf, 0x21, 0xdd, 0xaa, 0x6d, 0x9a, 0x71, 0x85, 0x6b,
    0xdb, 0xd6, 0x61, 0x3b, 0xf8, 0xe9, 0x7e, 0x0a, 0xf8, 0xd3, 0xc4, 0xce,
    0xdd, 0x0d, 0xfc, 0xfb, 0x47, 0x42, 0xa2, 0xef, 0x0a, 0x44, 0x3e, 0xf8,
    0x78, 0x96, 0x0e, 0x6f,
};
static const struct drbg_kat_no_reseed kat187_t = {
    6, kat187_entropyin, kat187_nonce, kat187_persstr,
    kat187_addin0, kat187_addin1, kat187_retbits
};
static const struct drbg_kat kat187 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat187_t
};

static const unsigned char kat188_entropyin[] = {
    0x15, 0x14, 0xbe, 0x70, 0x6e, 0x4c, 0x16, 0x72, 0x90, 0x55, 0x06, 0xb0,
    0xc5, 0x53, 0x47, 0xf2,
};
static const unsigned char kat188_nonce[] = {
    0xad, 0x8d, 0x88, 0x9f, 0x2b, 0x91, 0xa6, 0xdd,
};
static const unsigned char kat188_persstr[] = {0};
static const unsigned char kat188_addin0[] = {0};
static const unsigned char kat188_addin1[] = {0};
static const unsigned char kat188_retbits[] = {
    0xb3, 0x64, 0x4a, 0x6d, 0x16, 0x34, 0x0e, 0xf8, 0xb2, 0x8a, 0xd0, 0x6f,
    0x4c, 0xbf, 0xd5, 0xe7, 0x99, 0xf0, 0x80, 0xd4, 0xb2, 0xf8, 0x31, 0xfd,
    0x90, 0xb3, 0x13, 0xc8, 0x62, 0xd5, 0x76, 0x3d, 0x22, 0xb1, 0x21, 0x7f,
    0xe3, 0xd4, 0xf0, 0x7c, 0x00, 0x6a, 0x39, 0x89, 0x5d, 0x46, 0xd5, 0x24,
    0x70, 0xb3, 0xf8, 0xb7, 0x75, 0x21, 0xe1, 0x41, 0xf6, 0x8b, 0x06, 0xb5,
    0x4c, 0x38, 0x66, 0x6f,
};
static const struct drbg_kat_no_reseed kat188_t = {
    7, kat188_entropyin, kat188_nonce, kat188_persstr,
    kat188_addin0, kat188_addin1, kat188_retbits
};
static const struct drbg_kat kat188 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat188_t
};

static const unsigned char kat189_entropyin[] = {
    0x83, 0x35, 0xa4, 0x33, 0xa8, 0x0d, 0x5a, 0x3e, 0x8a, 0xb1, 0xf7, 0x38,
    0x55, 0xf9, 0x7a, 0x9b,
};
static const unsigned char kat189_nonce[] = {
    0x02, 0x62, 0xb4, 0x7c, 0x2a, 0xb5, 0xfc, 0xba,
};
static const unsigned char kat189_persstr[] = {0};
static const unsigned char kat189_addin0[] = {0};
static const unsigned char kat189_addin1[] = {0};
static const unsigned char kat189_retbits[] = {
    0x66, 0xa0, 0x3a, 0x0b, 0x3b, 0x2e, 0xf8, 0xd0, 0x4e, 0x01, 0xda, 0x6e,
    0x31, 0xbc, 0x90, 0x02, 0xeb, 0xac, 0xa9, 0x8d, 0x63, 0xfd, 0xde, 0xb0,
    0x7f, 0xa3, 0xe0, 0x65, 0x0a, 0x32, 0xe1, 0x42, 0xfc, 0xab, 0x99, 0x63,
    0xf0, 0x6b, 0xa8, 0x8c, 0xd5, 0x8b, 0xe7, 0x5a, 0x99, 0xdf, 0xa1, 0x3b,
    0x4f, 0x8f, 0xea, 0xe6, 0xf8, 0x4c, 0xe2, 0xde, 0x4d, 0x20, 0x19, 0x92,
    0x23, 0x1f, 0xf3, 0xb8,
};
static const struct drbg_kat_no_reseed kat189_t = {
    8, kat189_entropyin, kat189_nonce, kat189_persstr,
    kat189_addin0, kat189_addin1, kat189_retbits
};
static const struct drbg_kat kat189 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat189_t
};

static const unsigned char kat190_entropyin[] = {
    0x37, 0xfa, 0xeb, 0x0b, 0x3f, 0x85, 0x37, 0x2c, 0xcb, 0x44, 0x22, 0xe3,
    0x66, 0x90, 0xe9, 0x6c,
};
static const unsigned char kat190_nonce[] = {
    0xda, 0x16, 0x5c, 0x92, 0xd5, 0xfa, 0xaa, 0x3a,
};
static const unsigned char kat190_persstr[] = {0};
static const unsigned char kat190_addin0[] = {0};
static const unsigned char kat190_addin1[] = {0};
static const unsigned char kat190_retbits[] = {
    0x47, 0x75, 0x15, 0x90, 0x25, 0xd5, 0xb8, 0x82, 0xf3, 0x7c, 0x7e, 0xdb,
    0x8d, 0x89, 0xb3, 0x8a, 0xcb, 0x14, 0xa0, 0x28, 0x39, 0x80, 0x2c, 0x37,
    0x7a, 0xc4, 0x82, 0xf8, 0xc6, 0xb5, 0x9a, 0xd6, 0x9c, 0x7d, 0x67, 0x55,
    0x6b, 0xc4, 0x1c, 0xc3, 0x22, 0xd9, 0xcd, 0x75, 0x03, 0x2a, 0xf5, 0x27,
    0x15, 0xb3, 0x60, 0x2f, 0x24, 0xc2, 0xb0, 0x99, 0x84, 0x0e, 0xab, 0x00,
    0xb7, 0x42, 0x93, 0xad,
};
static const struct drbg_kat_no_reseed kat190_t = {
    9, kat190_entropyin, kat190_nonce, kat190_persstr,
    kat190_addin0, kat190_addin1, kat190_retbits
};
static const struct drbg_kat kat190 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat190_t
};

static const unsigned char kat191_entropyin[] = {
    0xb6, 0x83, 0x0f, 0x64, 0xe4, 0x73, 0x88, 0xa9, 0xc3, 0xc6, 0x4a, 0xa0,
    0xbe, 0x70, 0x84, 0x51,
};
static const unsigned char kat191_nonce[] = {
    0x5c, 0xd4, 0x30, 0x34, 0xc9, 0x89, 0xda, 0x56,
};
static const unsigned char kat191_persstr[] = {0};
static const unsigned char kat191_addin0[] = {0};
static const unsigned char kat191_addin1[] = {0};
static const unsigned char kat191_retbits[] = {
    0xe1, 0x2f, 0x1e, 0x9f, 0xd9, 0x0e, 0x4f, 0x16, 0x6a, 0x13, 0xa4, 0xd3,
    0x5f, 0x90, 0x91, 0xa9, 0x74, 0x44, 0x32, 0x91, 0xff, 0xae, 0x44, 0x02,
    0xcb, 0xcd, 0x9a, 0x24, 0x89, 0x3b, 0x77, 0x09, 0x6b, 0x0a, 0x00, 0xdb,
    0x20, 0xe4, 0x32, 0x88, 0x62, 0x61, 0x39, 0xf7, 0xc1, 0x76, 0x7f, 0x6e,
    0x7f, 0xfe, 0xe9, 0x02, 0x2d, 0xde, 0x41, 0xb4, 0x78, 0x48, 0x5e, 0xe9,
    0x12, 0x5f, 0x3e, 0xb8,
};
static const struct drbg_kat_no_reseed kat191_t = {
    10, kat191_entropyin, kat191_nonce, kat191_persstr,
    kat191_addin0, kat191_addin1, kat191_retbits
};
static const struct drbg_kat kat191 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat191_t
};

static const unsigned char kat192_entropyin[] = {
    0xf7, 0x1a, 0x01, 0xef, 0x00, 0xb3, 0xd6, 0x13, 0x46, 0x07, 0xdf, 0xc5,
    0x73, 0x25, 0xf6, 0x87,
};
static const unsigned char kat192_nonce[] = {
    0x49, 0x76, 0x0a, 0xeb, 0xbc, 0x89, 0x61, 0xfe,
};
static const unsigned char kat192_persstr[] = {0};
static const unsigned char kat192_addin0[] = {0};
static const unsigned char kat192_addin1[] = {0};
static const unsigned char kat192_retbits[] = {
    0x0f, 0x00, 0x8e, 0x60, 0xab, 0x48, 0x14, 0xcc, 0xd8, 0xd2, 0x89, 0x77,
    0x53, 0xf5, 0x8c, 0xa3, 0x58, 0xba, 0xcd, 0x3f, 0x2e, 0xfc, 0x22, 0xc0,
    0xec, 0x89, 0xa6, 0x5b, 0x0a, 0x91, 0x82, 0xf5, 0x55, 0xcc, 0x49, 0x7a,
    0xa5, 0x9a, 0xf9, 0x14, 0xbc, 0x9c, 0x65, 0xbe, 0x7c, 0x09, 0x21, 0x46,
    0xcb, 0x78, 0xfc, 0xe2, 0x40, 0xfc, 0x8f, 0xe1, 0x36, 0x72, 0x9e, 0xa7,
    0x77, 0x16, 0x44, 0x7f,
};
static const struct drbg_kat_no_reseed kat192_t = {
    11, kat192_entropyin, kat192_nonce, kat192_persstr,
    kat192_addin0, kat192_addin1, kat192_retbits
};
static const struct drbg_kat kat192 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat192_t
};

static const unsigned char kat193_entropyin[] = {
    0x79, 0x0e, 0xf8, 0xf0, 0xee, 0x6d, 0xf4, 0x80, 0xb2, 0xd4, 0x4d, 0xac,
    0x1c, 0x37, 0x61, 0x4c,
};
static const unsigned char kat193_nonce[] = {
    0xd5, 0x52, 0x44, 0x4d, 0x7d, 0xc3, 0x5e, 0x5c,
};
static const unsigned char kat193_persstr[] = {0};
static const unsigned char kat193_addin0[] = {0};
static const unsigned char kat193_addin1[] = {0};
static const unsigned char kat193_retbits[] = {
    0x2a, 0xf8, 0xe3, 0x09, 0x77, 0x55, 0x6f, 0xae, 0x7c, 0x18, 0xae, 0x12,
    0xba, 0x22, 0x01, 0xb4, 0x0d, 0xe8, 0xc0, 0x21, 0x66, 0xed, 0x94, 0xc1,
    0x41, 0x27, 0x20, 0x50, 0xdb, 0xea, 0x72, 0xf3, 0xb8, 0xf9, 0x91, 0x54,
    0x7f, 0xc0, 0xf5, 0x58, 0x36, 0x17, 0x12, 0x67, 0xd4, 0x2a, 0x53, 0xdc,
    0xfb, 0x37, 0xc1, 0x5a, 0x22, 0x76, 0x76, 0xbf, 0x21, 0x8a, 0x49, 0xd0,
    0xb7, 0x23, 0x68, 0x9c,
};
static const struct drbg_kat_no_reseed kat193_t = {
    12, kat193_entropyin, kat193_nonce, kat193_persstr,
    kat193_addin0, kat193_addin1, kat193_retbits
};
static const struct drbg_kat kat193 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat193_t
};

static const unsigned char kat194_entropyin[] = {
    0xbe, 0x19, 0x6c, 0x9c, 0xe3, 0x83, 0xc8, 0xb1, 0x01, 0xcd, 0x27, 0xa1,
    0x12, 0x91, 0x9f, 0xbc,
};
static const unsigned char kat194_nonce[] = {
    0x33, 0x2e, 0xbe, 0x3d, 0x47, 0x3a, 0x3a, 0x35,
};
static const unsigned char kat194_persstr[] = {0};
static const unsigned char kat194_addin0[] = {0};
static const unsigned char kat194_addin1[] = {0};
static const unsigned char kat194_retbits[] = {
    0x01, 0x6b, 0x05, 0xd5, 0x57, 0x77, 0x7b, 0x36, 0xa1, 0x92, 0x04, 0x0c,
    0x58, 0x0a, 0x5c, 0x51, 0x6d, 0xda, 0x11, 0x55, 0x93, 0x4a, 0xfb, 0xd9,
    0x6c, 0xc8, 0x5b, 0x6e, 0x8b, 0x00, 0x83, 0xbb, 0xda, 0x27, 0x3e, 0xa6,
    0xdb, 0x52, 0xa0, 0x58, 0x9d, 0x9d, 0xe2, 0xe5, 0x56, 0x8f, 0xfe, 0xf9,
    0xdb, 0x39, 0x50, 0x09, 0x3c, 0x4d, 0xb2, 0xe8, 0xce, 0x9c, 0x25, 0x13,
    0x08, 0x46, 0x93, 0xe4,
};
static const struct drbg_kat_no_reseed kat194_t = {
    13, kat194_entropyin, kat194_nonce, kat194_persstr,
    kat194_addin0, kat194_addin1, kat194_retbits
};
static const struct drbg_kat kat194 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat194_t
};

static const unsigned char kat195_entropyin[] = {
    0x76, 0x39, 0x61, 0x6e, 0xe5, 0xcb, 0x26, 0x9b, 0x5f, 0x54, 0x0d, 0xe1,
    0x50, 0xb4, 0x65, 0x0e,
};
static const unsigned char kat195_nonce[] = {
    0x3c, 0x93, 0x3a, 0xbe, 0xd3, 0x07, 0xf6, 0xe2,
};
static const unsigned char kat195_persstr[] = {0};
static const unsigned char kat195_addin0[] = {0};
static const unsigned char kat195_addin1[] = {0};
static const unsigned char kat195_retbits[] = {
    0x06, 0x33, 0xc9, 0xaf, 0x4d, 0x40, 0xf3, 0xc2, 0x1a, 0xd4, 0xe0, 0x78,
    0xa0, 0xc8, 0x4e, 0xe0, 0x3a, 0xd7, 0xad, 0x41, 0x0a, 0x0f, 0x7c, 0xf3,
    0xdf, 0xdf, 0xf5, 0x92, 0x0d, 0xcd, 0x39, 0x87, 0xfc, 0xec, 0xed, 0x11,
    0xa2, 0xb3, 0x8b, 0x15, 0x53, 0x5e, 0x44, 0xb5, 0x53, 0x77, 0xa8, 0xf2,
    0x0f, 0x4f, 0xe0, 0x51, 0x87, 0xf9, 0x76, 0xa2, 0x76, 0xe6, 0x4b, 0x81,
    0x3e, 0x55, 0xa8, 0xb9,
};
static const struct drbg_kat_no_reseed kat195_t = {
    14, kat195_entropyin, kat195_nonce, kat195_persstr,
    kat195_addin0, kat195_addin1, kat195_retbits
};
static const struct drbg_kat kat195 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat195_t
};

static const unsigned char kat196_entropyin[] = {
    0x8b, 0x80, 0x93, 0x6e, 0x69, 0xc6, 0x7e, 0xdb, 0x77, 0x1c, 0x28, 0xf9,
    0xb9, 0x45, 0x21, 0x24,
};
static const unsigned char kat196_nonce[] = {
    0x7e, 0xe2, 0x61, 0x4e, 0xad, 0x3c, 0x12, 0x8e,
};
static const unsigned char kat196_persstr[] = {0};
static const unsigned char kat196_addin0[] = {
    0xfc, 0x35, 0xcb, 0xa9, 0x7a, 0x1e, 0x21, 0x1b, 0xc4, 0x20, 0xe8, 0xaf,
    0x53, 0xf8, 0xe1, 0x3c,
};
static const unsigned char kat196_addin1[] = {
    0xfb, 0xa4, 0x38, 0xaa, 0xa7, 0x5a, 0x3c, 0xd4, 0xcd, 0x0c, 0xce, 0x39,
    0x9b, 0xfe, 0xc7, 0x4a,
};
static const unsigned char kat196_retbits[] = {
    0x67, 0x21, 0xcc, 0x1a, 0xda, 0x5e, 0xbc, 0x17, 0x13, 0xf7, 0x4c, 0x75,
    0x90, 0x00, 0x76, 0x56, 0x52, 0xee, 0xb5, 0xf3, 0xf9, 0xc2, 0x4f, 0xb9,
    0x34, 0x1b, 0x36, 0xa3, 0x69, 0xce, 0xc1, 0xd2, 0x7e, 0xa8, 0x0d, 0x6b,
    0x73, 0xb5, 0x60, 0x47, 0xaf, 0x07, 0x13, 0x8c, 0x5a, 0x43, 0xc9, 0x9a,
    0x87, 0x75, 0x31, 0x15, 0xc4, 0x71, 0xb8, 0x58, 0x7e, 0xa6, 0x5f, 0xa2,
    0x06, 0x5e, 0x3c, 0xe0,
};
static const struct drbg_kat_no_reseed kat196_t = {
    0, kat196_entropyin, kat196_nonce, kat196_persstr,
    kat196_addin0, kat196_addin1, kat196_retbits
};
static const struct drbg_kat kat196 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat196_t
};

static const unsigned char kat197_entropyin[] = {
    0x89, 0xce, 0x6c, 0x76, 0xbc, 0x41, 0xaa, 0x32, 0xa9, 0xc8, 0xe3, 0xe3,
    0x7d, 0x62, 0x02, 0xef,
};
static const unsigned char kat197_nonce[] = {
    0xa3, 0x47, 0xf6, 0xcb, 0x64, 0xed, 0x19, 0xdd,
};
static const unsigned char kat197_persstr[] = {0};
static const unsigned char kat197_addin0[] = {
    0xc5, 0x03, 0xb1, 0x22, 0x95, 0x7d, 0xce, 0xc8, 0xd9, 0xeb, 0x9c, 0xc9,
    0x94, 0xb8, 0xd1, 0x22,
};
static const unsigned char kat197_addin1[] = {
    0x3b, 0xb2, 0xf1, 0x19, 0x7a, 0x99, 0x58, 0x8d, 0xf3, 0x57, 0xc0, 0xd1,
    0x98, 0x6a, 0xc6, 0xed,
};
static const unsigned char kat197_retbits[] = {
    0xf0, 0xf4, 0x25, 0xb1, 0x9e, 0xb7, 0x5a, 0xea, 0x68, 0x99, 0xfa, 0xb7,
    0x61, 0x2c, 0x98, 0xa0, 0xb5, 0xc5, 0xc5, 0xa3, 0xc8, 0x61, 0x07, 0xa8,
    0xc2, 0x01, 0x62, 0x3f, 0x75, 0x99, 0x31, 0x90, 0x9a, 0xfd, 0x63, 0x41,
    0x9d, 0x1c, 0xce, 0xb8, 0x6b, 0xd4, 0xf1, 0x6e, 0x94, 0x8a, 0xef, 0x08,
    0x47, 0x61, 0x70, 0x75, 0x7b, 0xca, 0xf7, 0x98, 0x84, 0xf9, 0xc3, 0x6b,
    0xc7, 0x7e, 0x9f, 0xf9,
};
static const struct drbg_kat_no_reseed kat197_t = {
    1, kat197_entropyin, kat197_nonce, kat197_persstr,
    kat197_addin0, kat197_addin1, kat197_retbits
};
static const struct drbg_kat kat197 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat197_t
};

static const unsigned char kat198_entropyin[] = {
    0xbc, 0x51, 0x67, 0x8d, 0x1f, 0x31, 0x28, 0x65, 0x7b, 0x38, 0x84, 0x9a,
    0xe9, 0xc4, 0xbb, 0x1e,
};
static const unsigned char kat198_nonce[] = {
    0xdd, 0x29, 0xd9, 0xd8, 0x3f, 0xdb, 0x5e, 0x09,
};
static const unsigned char kat198_persstr[] = {0};
static const unsigned char kat198_addin0[] = {
    0xd4, 0xcd, 0x00, 0x67, 0x5c, 0xfe, 0x22, 0x7d, 0xe0, 0xcb, 0xab, 0x65,
    0x11, 0x35, 0x22, 0x89,
};
static const unsigned char kat198_addin1[] = {
    0x55, 0xbb, 0x93, 0x62, 0x80, 0xae, 0x46, 0xb2, 0x6d, 0x66, 0xb2, 0xf3,
    0xfe, 0xb2, 0x68, 0xd8,
};
static const unsigned char kat198_retbits[] = {
    0xe5, 0x81, 0x38, 0xb6, 0xb0, 0x23, 0x2e, 0x4d, 0x6d, 0x9e, 0xa8, 0xb6,
    0x5b, 0xd7, 0x69, 0x65, 0xcd, 0xbc, 0x58, 0x4c, 0xca, 0x2e, 0x83, 0xd4,
    0xea, 0xd9, 0x36, 0xbf, 0xb9, 0xc7, 0x9d, 0x77, 0xde, 0xc9, 0x42, 0x4f,
    0x2c, 0xf8, 0x95, 0x01, 0x1e, 0xf1, 0x03, 0x3c, 0x61, 0x9e, 0x28, 0xd1,
    0xdb, 0xa5, 0xdd, 0x3c, 0xed, 0x04, 0x42, 0x9c, 0x1b, 0xd3, 0xc0, 0x3e,
    0x5d, 0x13, 0xe2, 0xb8,
};
static const struct drbg_kat_no_reseed kat198_t = {
    2, kat198_entropyin, kat198_nonce, kat198_persstr,
    kat198_addin0, kat198_addin1, kat198_retbits
};
static const struct drbg_kat kat198 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat198_t
};

static const unsigned char kat199_entropyin[] = {
    0x06, 0x5a, 0x5d, 0x4f, 0xdb, 0x34, 0xb0, 0x8f, 0xd8, 0xef, 0xc8, 0x1f,
    0x99, 0x62, 0x69, 0xab,
};
static const unsigned char kat199_nonce[] = {
    0xd5, 0xa6, 0x74, 0x7c, 0x7f, 0x8e, 0xef, 0x46,
};
static const unsigned char kat199_persstr[] = {0};
static const unsigned char kat199_addin0[] = {
    0x72, 0xeb, 0xd3, 0x51, 0xed, 0xc0, 0x00, 0xd8, 0x36, 0x32, 0x58, 0xa6,
    0x9f, 0x2b, 0x98, 0xcc,
};
static const unsigned char kat199_addin1[] = {
    0xe5, 0xc8, 0x61, 0x21, 0x32, 0x96, 0xea, 0x2a, 0xd5, 0x09, 0x8f, 0x9b,
    0x4c, 0x50, 0x64, 0x7d,
};
static const unsigned char kat199_retbits[] = {
    0xa8, 0xea, 0x47, 0x0f, 0x17, 0xbe, 0xe5, 0x02, 0x1d, 0x49, 0xc8, 0x4b,
    0x72, 0x1e, 0x0b, 0x67, 0x47, 0x0d, 0x3b, 0xc4, 0xf9, 0x8f, 0xac, 0x68,
    0xc6, 0x74, 0x42, 0x97, 0x9d, 0x4f, 0xbc, 0xcb, 0xbc, 0xd0, 0x47, 0x7c,
    0x1e, 0x74, 0x93, 0x12, 0x6c, 0xbe, 0x16, 0x9b, 0x84, 0x44, 0x6f, 0x5f,
    0xcf, 0x66, 0x67, 0x86, 0x77, 0x5d, 0x19, 0xf3, 0x5c, 0x61, 0x52, 0x52,
    0x9a, 0x41, 0xde, 0x6d,
};
static const struct drbg_kat_no_reseed kat199_t = {
    3, kat199_entropyin, kat199_nonce, kat199_persstr,
    kat199_addin0, kat199_addin1, kat199_retbits
};
static const struct drbg_kat kat199 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat199_t
};

static const unsigned char kat200_entropyin[] = {
    0x12, 0x48, 0xed, 0x4b, 0xf6, 0x06, 0xcc, 0x65, 0x8b, 0x64, 0x1b, 0x79,
    0x58, 0x59, 0x7f, 0x5a,
};
static const unsigned char kat200_nonce[] = {
    0x16, 0xc8, 0x8d, 0xea, 0x4e, 0x75, 0x3d, 0x50,
};
static const unsigned char kat200_persstr[] = {0};
static const unsigned char kat200_addin0[] = {
    0xfc, 0x67, 0xe7, 0x63, 0x41, 0x32, 0x02, 0xe0, 0x4c, 0x55, 0x69, 0x7e,
    0xb7, 0x54, 0x8c, 0x56,
};
static const unsigned char kat200_addin1[] = {
    0x21, 0xe0, 0xe3, 0x86, 0x65, 0xec, 0xf6, 0x0f, 0xa9, 0x73, 0x0b, 0xdf,
    0x7e, 0x22, 0xcc, 0x7f,
};
static const unsigned char kat200_retbits[] = {
    0x63, 0xd9, 0xb5, 0x0d, 0xd6, 0x9e, 0xa9, 0xda, 0xc7, 0x5f, 0x48, 0xae,
    0x01, 0x2f, 0xce, 0xc2, 0x7c, 0x2e, 0x1d, 0xcf, 0xc2, 0xac, 0x2e, 0x59,
    0x24, 0x4a, 0xf4, 0xa6, 0xce, 0x9a, 0x73, 0xc4, 0xd8, 0xbd, 0xe6, 0x57,
    0x0d, 0x85, 0xee, 0x15, 0xb0, 0x8a, 0x48, 0xef, 0x47, 0x3d, 0x12, 0xdb,
    0x87, 0x1f, 0x6d, 0xf1, 0xd8, 0x14, 0x41, 0x61, 0x3d, 0xcc, 0x1e, 0xfb,
    0x20, 0x18, 0xf0, 0xc3,
};
static const struct drbg_kat_no_reseed kat200_t = {
    4, kat200_entropyin, kat200_nonce, kat200_persstr,
    kat200_addin0, kat200_addin1, kat200_retbits
};
static const struct drbg_kat kat200 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat200_t
};

static const unsigned char kat201_entropyin[] = {
    0x6f, 0xa6, 0x29, 0xd0, 0x3c, 0xb4, 0xeb, 0x53, 0x4e, 0xfa, 0x03, 0x33,
    0x4d, 0x3b, 0x1d, 0x02,
};
static const unsigned char kat201_nonce[] = {
    0xd9, 0x52, 0x15, 0x3e, 0x79, 0xdf, 0x53, 0x8f,
};
static const unsigned char kat201_persstr[] = {0};
static const unsigned char kat201_addin0[] = {
    0x2e, 0x5c, 0x55, 0x45, 0x78, 0xa0, 0x69, 0xf5, 0xe4, 0x95, 0x9d, 0xcb,
    0x35, 0x1a, 0x29, 0x4d,
};
static const unsigned char kat201_addin1[] = {
    0x2d, 0x26, 0xae, 0xaa, 0xd9, 0xcb, 0xf2, 0x53, 0xad, 0xd8, 0x68, 0x4d,
    0x29, 0xb1, 0xa6, 0x33,
};
static const unsigned char kat201_retbits[] = {
    0xf5, 0x82, 0x46, 0x31, 0x32, 0x84, 0x99, 0x02, 0x98, 0x86, 0x4a, 0x89,
    0xbf, 0xe7, 0xbe, 0x19, 0x70, 0xc0, 0x33, 0xdc, 0x16, 0x64, 0xad, 0x7f,
    0x5c, 0xd9, 0xb8, 0x12, 0xf5, 0xb7, 0xe9, 0x0f, 0x69, 0xa4, 0x9c, 0xdf,
    0xbe, 0xe0, 0xe2, 0x79, 0xf7, 0xdf, 0x1a, 0x8e, 0xd7, 0x9e, 0xca, 0x6e,
    0x68, 0x0d, 0x74, 0x0c, 0x0f, 0x2a, 0xe1, 0x2f, 0x87, 0x79, 0x8a, 0xcc,
    0x6d, 0x73, 0xd4, 0x2f,
};
static const struct drbg_kat_no_reseed kat201_t = {
    5, kat201_entropyin, kat201_nonce, kat201_persstr,
    kat201_addin0, kat201_addin1, kat201_retbits
};
static const struct drbg_kat kat201 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat201_t
};

static const unsigned char kat202_entropyin[] = {
    0xc5, 0x8e, 0xeb, 0x2e, 0x57, 0xf2, 0x95, 0xd7, 0xaf, 0x1a, 0x37, 0x67,
    0x93, 0x5b, 0x85, 0xeb,
};
static const unsigned char kat202_nonce[] = {
    0x98, 0x61, 0x2b, 0x6c, 0xa0, 0x2f, 0x60, 0xe4,
};
static const unsigned char kat202_persstr[] = {0};
static const unsigned char kat202_addin0[] = {
    0x10, 0xe8, 0x9c, 0x27, 0x1e, 0x2b, 0x28, 0x3d, 0x69, 0xb0, 0x4a, 0xbf,
    0x6c, 0x54, 0xb1, 0xe1,
};
static const unsigned char kat202_addin1[] = {
    0x20, 0x79, 0x50, 0xc2, 0x8b, 0x26, 0xec, 0x16, 0xc4, 0xf2, 0x81, 0x21,
    0x60, 0x7f, 0x4a, 0x5a,
};
static const unsigned char kat202_retbits[] = {
    0x28, 0xef, 0xf6, 0x03, 0xff, 0x0d, 0xc8, 0x39, 0xf8, 0x8f, 0x84, 0x47,
    0x33, 0xb2, 0x75, 0xbc, 0x6f, 0xda, 0x6d, 0x45, 0x46, 0xee, 0x26, 0x03,
    0x4f, 0x51, 0xf8, 0x4c, 0xe4, 0x30, 0xb0, 0x73, 0xc3, 0xda, 0x8a, 0xbf,
    0xe8, 0x2b, 0x53, 0x13, 0xba, 0x2e, 0x0f, 0x60, 0x04, 0x5b, 0x80, 0x96,
    0xb9, 0xcf, 0x31, 0x42, 0x48, 0x63, 0xe0, 0x6f, 0x72, 0xef, 0xf1, 0xba,
    0xa4, 0xb2, 0x52, 0x70,
};
static const struct drbg_kat_no_reseed kat202_t = {
    6, kat202_entropyin, kat202_nonce, kat202_persstr,
    kat202_addin0, kat202_addin1, kat202_retbits
};
static const struct drbg_kat kat202 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat202_t
};

static const unsigned char kat203_entropyin[] = {
    0x8a, 0x34, 0x5b, 0x5a, 0x5e, 0xb7, 0xd7, 0xaa, 0x70, 0x0b, 0xff, 0x26,
    0x61, 0x25, 0xed, 0xe1,
};
static const unsigned char kat203_nonce[] = {
    0x00, 0xb2, 0x96, 0x37, 0x27, 0x8d, 0xa4, 0x57,
};
static const unsigned char kat203_persstr[] = {0};
static const unsigned char kat203_addin0[] = {
    0x44, 0x53, 0x4b, 0x92, 0xaf, 0x03, 0x81, 0x71, 0xb6, 0x7b, 0x70, 0xf4,
    0xe1, 0xd5, 0xe7, 0xf5,
};
static const unsigned char kat203_addin1[] = {
    0x53, 0x14, 0x76, 0x60, 0xc0, 0xfd, 0xe5, 0xf9, 0x89, 0x34, 0x74, 0xf0,
    0x38, 0x84, 0xab, 0x14,
};
static const unsigned char kat203_retbits[] = {
    0xa6, 0xa7, 0xa5, 0x6f, 0xc3, 0x2c, 0x3d, 0x3c, 0x33, 0x06, 0xdd, 0x65,
    0x10, 0x9f, 0xf8, 0xd6, 0x80, 0x31, 0x19, 0x3b, 0xbf, 0x5b, 0x38, 0x38,
    0x0e, 0x38, 0x25, 0xdb, 0x7b, 0xef, 0x72, 0x94, 0x05, 0x14, 0x16, 0x26,
    0x37, 0x95, 0xf3, 0x34, 0xd1, 0xf8, 0x70, 0x54, 0xe9, 0x7d, 0xbb, 0x52,
    0xdd, 0x24, 0x4d, 0x52, 0x7a, 0x6f, 0xfc, 0xe0, 0x86, 0xd1, 0xad, 0x17,
    0x7b, 0xa8, 0xfb, 0x81,
};
static const struct drbg_kat_no_reseed kat203_t = {
    7, kat203_entropyin, kat203_nonce, kat203_persstr,
    kat203_addin0, kat203_addin1, kat203_retbits
};
static const struct drbg_kat kat203 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat203_t
};

static const unsigned char kat204_entropyin[] = {
    0xfe, 0x50, 0x4e, 0x06, 0x89, 0x5d, 0x34, 0x08, 0x90, 0x50, 0x9f, 0x39,
    0x8f, 0xe7, 0x08, 0x32,
};
static const unsigned char kat204_nonce[] = {
    0x3b, 0x3d, 0x94, 0xb6, 0x94, 0x8f, 0x74, 0x01,
};
static const unsigned char kat204_persstr[] = {0};
static const unsigned char kat204_addin0[] = {
    0x64, 0x20, 0x03, 0x18, 0x1f, 0x79, 0x1c, 0xb5, 0xba, 0xce, 0xf7, 0x23,
    0xc9, 0x6c, 0xe2, 0x62,
};
static const unsigned char kat204_addin1[] = {
    0x34, 0x5e, 0x8d, 0x29, 0xd5, 0xef, 0xdd, 0xc8, 0xab, 0x37, 0x83, 0x8b,
    0x18, 0x91, 0xbc, 0xd0,
};
static const unsigned char kat204_retbits[] = {
    0x45, 0x30, 0x40, 0x99, 0x14, 0x5b, 0x8e, 0x3d, 0xa8, 0x00, 0xa9, 0x2e,
    0x0f, 0x59, 0x15, 0xc8, 0x98, 0x1f, 0xb7, 0x70, 0x6a, 0x1d, 0xf0, 0xa6,
    0xc0, 0xb2, 0xad, 0x54, 0x35, 0xda, 0x19, 0xce, 0xd3, 0xac, 0xdf, 0xd5,
    0x83, 0x27, 0x95, 0x01, 0x3a, 0xfa, 0x5f, 0xf2, 0x1f, 0xf3, 0x32, 0x6a,
    0xda, 0x7d, 0x57, 0x78, 0x17, 0xf6, 0xb9, 0x4e, 0xb3, 0x5f, 0x33, 0x62,
    0x3d, 0xb4, 0x71, 0x76,
};
static const struct drbg_kat_no_reseed kat204_t = {
    8, kat204_entropyin, kat204_nonce, kat204_persstr,
    kat204_addin0, kat204_addin1, kat204_retbits
};
static const struct drbg_kat kat204 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat204_t
};

static const unsigned char kat205_entropyin[] = {
    0xe0, 0x22, 0x7d, 0xe1, 0x4d, 0x5b, 0xc6, 0xad, 0xe2, 0xa0, 0x14, 0xee,
    0x43, 0x01, 0x51, 0xb4,
};
static const unsigned char kat205_nonce[] = {
    0x15, 0x37, 0xf8, 0x56, 0x16, 0x75, 0x87, 0x53,
};
static const unsigned char kat205_persstr[] = {0};
static const unsigned char kat205_addin0[] = {
    0x87, 0xee, 0xfe, 0xf0, 0xe0, 0xcf, 0xd9, 0x0d, 0xd2, 0x6d, 0x48, 0x33,
    0x79, 0x78, 0x7c, 0x7f,
};
static const unsigned char kat205_addin1[] = {
    0x74, 0xd2, 0xff, 0xf6, 0xf2, 0x9f, 0x89, 0x2e, 0x66, 0xa9, 0xf8, 0x0b,
    0x2a, 0x5d, 0x9d, 0x20,
};
static const unsigned char kat205_retbits[] = {
    0xdd, 0xe0, 0x86, 0xce, 0xc0, 0xad, 0xd3, 0x1f, 0xb9, 0x55, 0x3d, 0x56,
    0xcc, 0x2c, 0x6b, 0x7c, 0x02, 0x2d, 0x8c, 0x87, 0xff, 0x54, 0x99, 0xfa,
    0x78, 0xa6, 0x8e, 0xeb, 0x05, 0x1c, 0xdb, 0xf8, 0x99, 0x67, 0x3b, 0x4e,
    0xd7, 0x6f, 0xe8, 0x67, 0xb7, 0x9e, 0xcf, 0x6f, 0xc1, 0xd7, 0xc6, 0xf8,
    0xf5, 0x97, 0x01, 0x31, 0x39, 0xd5, 0x28, 0xd3, 0x48, 0xca, 0x3b, 0xee,
    0x96, 0x05, 0xdd, 0x80,
};
static const struct drbg_kat_no_reseed kat205_t = {
    9, kat205_entropyin, kat205_nonce, kat205_persstr,
    kat205_addin0, kat205_addin1, kat205_retbits
};
static const struct drbg_kat kat205 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat205_t
};

static const unsigned char kat206_entropyin[] = {
    0x44, 0xb7, 0x28, 0x2f, 0x95, 0x1c, 0x59, 0x1a, 0x7a, 0x3d, 0x9f, 0xf6,
    0x96, 0x0f, 0x20, 0xb9,
};
static const unsigned char kat206_nonce[] = {
    0xb7, 0x7e, 0xbe, 0x26, 0x99, 0x0c, 0xbe, 0x8c,
};
static const unsigned char kat206_persstr[] = {0};
static const unsigned char kat206_addin0[] = {
    0x90, 0x00, 0x58, 0x7b, 0xb1, 0x8d, 0x20, 0x1e, 0xcf, 0xd5, 0x6f, 0x30,
    0xdb, 0xa4, 0x83, 0xd2,
};
static const unsigned char kat206_addin1[] = {
    0x24, 0x55, 0x23, 0x06, 0x0b, 0x0a, 0xf3, 0xe5, 0x75, 0xb1, 0x48, 0x0a,
    0xa6, 0xd8, 0xa3, 0x3b,
};
static const unsigned char kat206_retbits[] = {
    0x9f, 0x25, 0x67, 0x4a, 0xc0, 0x85, 0xa7, 0xa4, 0x87, 0xe7, 0xd7, 0x20,
    0x84, 0xd5, 0xd3, 0xc0, 0xfb, 0xd7, 0xd4, 0x18, 0x70, 0xf0, 0xf7, 0x66,
    0xd6, 0x13, 0x01, 0x59, 0x96, 0x05, 0x2e, 0xbf, 0xc6, 0xf6, 0x2e, 0x4c,
    0xf3, 0x89, 0xaf, 0x85, 0xcf, 0x12, 0x5d, 0x9d, 0x99, 0xc6, 0x49, 0xd4,
    0xe8, 0x75, 0x50, 0x79, 0xa6, 0x81, 0x7a, 0x9e, 0x81, 0x44, 0x5d, 0x1b,
    0x99, 0x4d, 0x29, 0x61,
};
static const struct drbg_kat_no_reseed kat206_t = {
    10, kat206_entropyin, kat206_nonce, kat206_persstr,
    kat206_addin0, kat206_addin1, kat206_retbits
};
static const struct drbg_kat kat206 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat206_t
};

static const unsigned char kat207_entropyin[] = {
    0x74, 0x5c, 0xa1, 0x6e, 0xbc, 0x9e, 0x1a, 0x58, 0xda, 0x81, 0x02, 0x78,
    0x88, 0x5d, 0xff, 0x33,
};
static const unsigned char kat207_nonce[] = {
    0x7e, 0x11, 0xf9, 0x49, 0x86, 0x2c, 0xbc, 0xc9,
};
static const unsigned char kat207_persstr[] = {0};
static const unsigned char kat207_addin0[] = {
    0x4b, 0x89, 0x8e, 0xbd, 0xd3, 0x21, 0x42, 0x62, 0x78, 0x15, 0xbe, 0x10,
    0x35, 0x43, 0x56, 0x96,
};
static const unsigned char kat207_addin1[] = {
    0x33, 0x0b, 0x0c, 0xe2, 0x87, 0x84, 0x2d, 0x3d, 0x71, 0x9a, 0x0c, 0xe4,
    0x15, 0x36, 0x3b, 0x77,
};
static const unsigned char kat207_retbits[] = {
    0x9c, 0x6d, 0xd7, 0x63, 0x88, 0x80, 0x65, 0x54, 0x1b, 0x1a, 0x38, 0x0c,
    0xf4, 0xf4, 0x59, 0x83, 0x9f, 0xb4, 0xf1, 0xef, 0x4f, 0x78, 0x17, 0x28,
    0x6b, 0xe4, 0x50, 0x33, 0x95, 0x7f, 0x79, 0xd1, 0x42, 0x9c, 0xfe, 0xfe,
    0xd1, 0xab, 0xd6, 0x22, 0x8b, 0x26, 0x74, 0x7e, 0x69, 0xec, 0x9b, 0x1a,
    0x02, 0x7b, 0x3a, 0x0b, 0x9a, 0x28, 0x48, 0x7e, 0x95, 0x4d, 0xda, 0x06,
    0xb7, 0xee, 0xb4, 0x54,
};
static const struct drbg_kat_no_reseed kat207_t = {
    11, kat207_entropyin, kat207_nonce, kat207_persstr,
    kat207_addin0, kat207_addin1, kat207_retbits
};
static const struct drbg_kat kat207 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat207_t
};

static const unsigned char kat208_entropyin[] = {
    0x1e, 0x54, 0x6f, 0xee, 0xba, 0xf0, 0xfe, 0xba, 0x3a, 0x07, 0xc6, 0x35,
    0xe6, 0x23, 0x5b, 0xfb,
};
static const unsigned char kat208_nonce[] = {
    0x17, 0x88, 0xd9, 0xee, 0xa4, 0x44, 0xad, 0x7b,
};
static const unsigned char kat208_persstr[] = {0};
static const unsigned char kat208_addin0[] = {
    0x51, 0x99, 0xd4, 0xaa, 0x59, 0x30, 0x22, 0x0c, 0x27, 0x34, 0x2e, 0x1c,
    0x0b, 0x99, 0x46, 0x4f,
};
static const unsigned char kat208_addin1[] = {
    0x16, 0xdd, 0x52, 0x0f, 0x00, 0x93, 0x29, 0xed, 0x7b, 0xe1, 0x4d, 0x25,
    0xe5, 0xc7, 0x91, 0x12,
};
static const unsigned char kat208_retbits[] = {
    0x34, 0xfa, 0x78, 0x30, 0x6c, 0x9e, 0xfb, 0x91, 0x2e, 0xe7, 0x8f, 0xb9,
    0x8a, 0x03, 0x54, 0x6d, 0x16, 0xc1, 0xcc, 0x9b, 0xc3, 0xb6, 0x7a, 0x04,
    0xf2, 0xa8, 0xe6, 0xfb, 0x26, 0x24, 0x14, 0xbc, 0xf3, 0xcc, 0x51, 0xd3,
    0xcf, 0x3a, 0x1a, 0x30, 0x48, 0x09, 0xf1, 0x86, 0xbd, 0x74, 0xf2, 0x09,
    0xb2, 0x42, 0x86, 0xf7, 0x71, 0x76, 0x67, 0xcc, 0x56, 0x60, 0xd3, 0xec,
    0x73, 0x2f, 0xb0, 0xd3,
};
static const struct drbg_kat_no_reseed kat208_t = {
    12, kat208_entropyin, kat208_nonce, kat208_persstr,
    kat208_addin0, kat208_addin1, kat208_retbits
};
static const struct drbg_kat kat208 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat208_t
};

static const unsigned char kat209_entropyin[] = {
    0xde, 0x1f, 0xca, 0xee, 0x63, 0x29, 0xe9, 0xac, 0x06, 0xa6, 0xdc, 0x18,
    0xa1, 0x8d, 0x14, 0x7d,
};
static const unsigned char kat209_nonce[] = {
    0xe1, 0x0e, 0xa5, 0x48, 0xcc, 0x37, 0x73, 0xab,
};
static const unsigned char kat209_persstr[] = {0};
static const unsigned char kat209_addin0[] = {
    0x5c, 0xdd, 0xd4, 0xfd, 0x07, 0xb6, 0xac, 0x3e, 0x96, 0xb1, 0x33, 0x39,
    0xa7, 0x5f, 0x56, 0xf9,
};
static const unsigned char kat209_addin1[] = {
    0x45, 0x47, 0x14, 0xbe, 0xbf, 0x96, 0x03, 0xe1, 0xcf, 0xef, 0x7b, 0x80,
    0x38, 0x37, 0x48, 0x99,
};
static const unsigned char kat209_retbits[] = {
    0x5e, 0x34, 0x83, 0x12, 0x58, 0x35, 0x24, 0x30, 0x48, 0xa5, 0x10, 0x18,
    0x4f, 0x08, 0x64, 0x87, 0xe9, 0xb0, 0x2f, 0xc0, 0xca, 0x60, 0xfb, 0x46,
    0x41, 0x6f, 0xb0, 0xcc, 0xc1, 0x3e, 0x45, 0x08, 0x1d, 0xa5, 0x96, 0x91,
    0xe3, 0x26, 0x78, 0xfc, 0xb6, 0xaa, 0xbe, 0xf5, 0x85, 0xfb, 0x49, 0x2b,
    0x19, 0x4e, 0x06, 0x17, 0x1a, 0x8d, 0x17, 0xaf, 0x85, 0x56, 0xa3, 0x6b,
    0xf4, 0x93, 0x1f, 0xf5,
};
static const struct drbg_kat_no_reseed kat209_t = {
    13, kat209_entropyin, kat209_nonce, kat209_persstr,
    kat209_addin0, kat209_addin1, kat209_retbits
};
static const struct drbg_kat kat209 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat209_t
};

static const unsigned char kat210_entropyin[] = {
    0x8b, 0x52, 0x95, 0x07, 0xbc, 0x2e, 0x8f, 0x71, 0x19, 0x34, 0xad, 0xc8,
    0x8c, 0x65, 0x30, 0xa3,
};
static const unsigned char kat210_nonce[] = {
    0x72, 0x15, 0x95, 0x6b, 0x7c, 0x57, 0x4e, 0x71,
};
static const unsigned char kat210_persstr[] = {0};
static const unsigned char kat210_addin0[] = {
    0x1a, 0xdb, 0x1e, 0xf6, 0x33, 0x47, 0xb2, 0x1e, 0x33, 0x51, 0x8b, 0x96,
    0x2f, 0x6d, 0xb8, 0x98,
};
static const unsigned char kat210_addin1[] = {
    0xd5, 0xc6, 0x9f, 0x42, 0xf3, 0x6e, 0xa2, 0x24, 0x98, 0x42, 0xb6, 0xfc,
    0x26, 0xac, 0x54, 0xfe,
};
static const unsigned char kat210_retbits[] = {
    0xc3, 0x74, 0xbd, 0xd6, 0x2c, 0xb1, 0xe4, 0x2e, 0x64, 0x84, 0x03, 0x84,
    0x3b, 0x8c, 0x06, 0xc0, 0xb3, 0x05, 0xf9, 0x45, 0x68, 0x5f, 0x72, 0xd1,
    0xbc, 0x2e, 0x42, 0x8c, 0x19, 0xdf, 0x45, 0x20, 0x18, 0xdd, 0xea, 0x81,
    0xeb, 0xe5, 0xb1, 0x5f, 0xad, 0x3b, 0xe4, 0xeb, 0x17, 0xfa, 0x2d, 0x2c,
    0xf5, 0x78, 0x36, 0xdd, 0x08, 0x09, 0x70, 0xf7, 0x16, 0xc0, 0xb2, 0x2f,
    0x72, 0x63, 0x41, 0x4a,
};
static const struct drbg_kat_no_reseed kat210_t = {
    14, kat210_entropyin, kat210_nonce, kat210_persstr,
    kat210_addin0, kat210_addin1, kat210_retbits
};
static const struct drbg_kat kat210 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat210_t
};

static const unsigned char kat211_entropyin[] = {
    0xf4, 0x1f, 0x46, 0x6b, 0x32, 0x19, 0xbe, 0x21, 0x59, 0x77, 0x63, 0xfa,
    0x7b, 0x76, 0xfb, 0x40,
};
static const unsigned char kat211_nonce[] = {
    0xcd, 0x93, 0xfe, 0xb9, 0x96, 0x2e, 0x81, 0xac,
};
static const unsigned char kat211_persstr[] = {
    0xb5, 0x8f, 0x86, 0x9a, 0xd0, 0xaa, 0x98, 0x08, 0xf6, 0x64, 0x61, 0x37,
    0x43, 0x1d, 0x43, 0x0c,
};
static const unsigned char kat211_addin0[] = {0};
static const unsigned char kat211_addin1[] = {0};
static const unsigned char kat211_retbits[] = {
    0x2f, 0xb6, 0xd7, 0xec, 0xa3, 0x92, 0x67, 0x4f, 0xc7, 0x22, 0xa6, 0x19,
    0x20, 0x2e, 0x81, 0x9d, 0x0d, 0xa9, 0xd1, 0x1b, 0xc6, 0x7d, 0xb1, 0x0b,
    0xe4, 0xc1, 0x3c, 0xb9, 0x64, 0xe3, 0x0a, 0xda, 0x96, 0xdc, 0xcf, 0x0c,
    0x92, 0x2b, 0x71, 0x0a, 0xc0, 0x0d, 0xed, 0x54, 0x57, 0xfa, 0x97, 0x1b,
    0xb1, 0xc6, 0x61, 0xa0, 0x9a, 0xfa, 0x72, 0x0a, 0x58, 0x64, 0x34, 0x4b,
    0xf7, 0x7a, 0x36, 0xae,
};
static const struct drbg_kat_no_reseed kat211_t = {
    0, kat211_entropyin, kat211_nonce, kat211_persstr,
    kat211_addin0, kat211_addin1, kat211_retbits
};
static const struct drbg_kat kat211 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat211_t
};

static const unsigned char kat212_entropyin[] = {
    0x9d, 0x1b, 0x88, 0x34, 0x83, 0x2f, 0xfa, 0x13, 0x83, 0x2e, 0xb0, 0x86,
    0x04, 0x7b, 0xf3, 0xb1,
};
static const unsigned char kat212_nonce[] = {
    0xd0, 0xf1, 0x5e, 0xfe, 0x86, 0x47, 0x7f, 0x75,
};
static const unsigned char kat212_persstr[] = {
    0x73, 0xc9, 0x37, 0x34, 0xf6, 0xea, 0x39, 0xae, 0x04, 0xe6, 0xa4, 0xb4,
    0x97, 0x66, 0xb8, 0x20,
};
static const unsigned char kat212_addin0[] = {0};
static const unsigned char kat212_addin1[] = {0};
static const unsigned char kat212_retbits[] = {
    0x9f, 0xb6, 0x7d, 0x35, 0x37, 0x89, 0x40, 0xa5, 0xd7, 0x6b, 0x96, 0x3a,
    0xce, 0x4f, 0x81, 0x58, 0xe9, 0x3f, 0xe0, 0xca, 0x06, 0x4f, 0x96, 0x56,
    0xd4, 0x6d, 0xf1, 0xc1, 0x0d, 0x02, 0x5f, 0x48, 0xb3, 0x35, 0x69, 0xda,
    0x07, 0xc7, 0x7e, 0xc5, 0x12, 0x23, 0x6d, 0x08, 0xd2, 0x69, 0x97, 0xd6,
    0xb9, 0xbb, 0x69, 0x15, 0xdf, 0x63, 0x9e, 0xa8, 0x9d, 0xa9, 0x57, 0xe6,
    0x6f, 0xc2, 0x90, 0x03,
};
static const struct drbg_kat_no_reseed kat212_t = {
    1, kat212_entropyin, kat212_nonce, kat212_persstr,
    kat212_addin0, kat212_addin1, kat212_retbits
};
static const struct drbg_kat kat212 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat212_t
};

static const unsigned char kat213_entropyin[] = {
    0x29, 0x89, 0x86, 0x2a, 0x79, 0xe2, 0x55, 0x19, 0x5a, 0x24, 0x82, 0x8f,
    0xd3, 0x00, 0xeb, 0x34,
};
static const unsigned char kat213_nonce[] = {
    0x30, 0xff, 0xda, 0xcb, 0x3a, 0xc7, 0xb2, 0x7b,
};
static const unsigned char kat213_persstr[] = {
    0x71, 0x9b, 0x89, 0x9c, 0x9e, 0x4a, 0x5d, 0xb9, 0xe7, 0x1d, 0xfd, 0xa4,
    0x8f, 0xa6, 0x58, 0xcd,
};
static const unsigned char kat213_addin0[] = {0};
static const unsigned char kat213_addin1[] = {0};
static const unsigned char kat213_retbits[] = {
    0xb4, 0xf2, 0x00, 0x60, 0xea, 0x30, 0x01, 0xef, 0xdb, 0xd5, 0xcc, 0x89,
    0x83, 0x8e, 0x0a, 0x08, 0xc0, 0x9f, 0x7a, 0x6f, 0xe5, 0xbc, 0x02, 0x3c,
    0x33, 0xd1, 0x15, 0xfe, 0xdd, 0x6a, 0xe1, 0x51, 0x30, 0x74, 0x22, 0xf9,
    0x97, 0xd3, 0x2b, 0x3c, 0xea, 0xb8, 0x79, 0x95, 0x86, 0x23, 0x68, 0xc4,
    0xc3, 0xaf, 0x7a, 0xc4, 0x81, 0x58, 0x74, 0xc0, 0x08, 0x4e, 0xa1, 0xdc,
    0xec, 0x50, 0x58, 0xba,
};
static const struct drbg_kat_no_reseed kat213_t = {
    2, kat213_entropyin, kat213_nonce, kat213_persstr,
    kat213_addin0, kat213_addin1, kat213_retbits
};
static const struct drbg_kat kat213 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat213_t
};

static const unsigned char kat214_entropyin[] = {
    0x74, 0x82, 0xb2, 0xb0, 0x2a, 0x74, 0x46, 0xde, 0x58, 0x9d, 0x7e, 0x60,
    0xcf, 0x01, 0x29, 0x69,
};
static const unsigned char kat214_nonce[] = {
    0x1c, 0x4b, 0xda, 0x6a, 0x57, 0xf4, 0x1f, 0xfb,
};
static const unsigned char kat214_persstr[] = {
    0x52, 0x9e, 0x4e, 0x80, 0xf5, 0x01, 0xe7, 0x3e, 0xc8, 0x6e, 0xaa, 0x25,
    0x96, 0x28, 0xa1, 0x97,
};
static const unsigned char kat214_addin0[] = {0};
static const unsigned char kat214_addin1[] = {0};
static const unsigned char kat214_retbits[] = {
    0x74, 0x98, 0xf9, 0x17, 0x2a, 0xf7, 0xf5, 0xf2, 0x6d, 0x84, 0x77, 0x97,
    0x76, 0x8e, 0x45, 0x91, 0x70, 0xdd, 0x9e, 0xc7, 0xf4, 0x2a, 0x1f, 0xe9,
    0x79, 0xa2, 0xe4, 0xfa, 0x32, 0xa5, 0xe1, 0x24, 0xc5, 0xcb, 0x1a, 0xd4,
    0xc3, 0x94, 0xa2, 0xc2, 0x09, 0x9e, 0x8f, 0x94, 0x2e, 0xfb, 0xe5, 0x9a,
    0xf0, 0x97, 0x5b, 0x56, 0xa9, 0xaf, 0xa7, 0x74, 0x33, 0x16, 0x12, 0xad,
    0x88, 0x7b, 0x3f, 0x55,
};
static const struct drbg_kat_no_reseed kat214_t = {
    3, kat214_entropyin, kat214_nonce, kat214_persstr,
    kat214_addin0, kat214_addin1, kat214_retbits
};
static const struct drbg_kat kat214 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat214_t
};

static const unsigned char kat215_entropyin[] = {
    0x0a, 0x84, 0x05, 0x99, 0x1a, 0xeb, 0x64, 0xf3, 0xa8, 0x2d, 0x8b, 0xef,
    0x2b, 0x6c, 0x94, 0x22,
};
static const unsigned char kat215_nonce[] = {
    0xa7, 0xa5, 0x8d, 0xa9, 0xb2, 0x16, 0xf7, 0xfb,
};
static const unsigned char kat215_persstr[] = {
    0x68, 0xdf, 0x62, 0xfc, 0x01, 0xd3, 0xdb, 0xb0, 0x18, 0xc1, 0x63, 0xbe,
    0x34, 0x29, 0xf2, 0xaa,
};
static const unsigned char kat215_addin0[] = {0};
static const unsigned char kat215_addin1[] = {0};
static const unsigned char kat215_retbits[] = {
    0x7a, 0x9b, 0xa1, 0xe8, 0x25, 0x13, 0x3f, 0xf4, 0xc1, 0xd6, 0x46, 0xce,
    0x55, 0x77, 0xf3, 0x5a, 0x17, 0x84, 0xee, 0xc2, 0xc1, 0x97, 0x70, 0x90,
    0xb4, 0x8e, 0x30, 0xbd, 0x3b, 0x75, 0x06, 0xf4, 0x47, 0xee, 0x62, 0xd0,
    0x21, 0xca, 0xe1, 0x2a, 0xd2, 0x87, 0xb4, 0x17, 0xed, 0xdb, 0x9e, 0xc6,
    0x46, 0x0e, 0x3e, 0x28, 0x4a, 0xfa, 0x73, 0xb7, 0x39, 0x56, 0x4e, 0x40,
    0x73, 0xd0, 0x0e, 0x3c,
};
static const struct drbg_kat_no_reseed kat215_t = {
    4, kat215_entropyin, kat215_nonce, kat215_persstr,
    kat215_addin0, kat215_addin1, kat215_retbits
};
static const struct drbg_kat kat215 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat215_t
};

static const unsigned char kat216_entropyin[] = {
    0xc7, 0xe9, 0xf6, 0x88, 0x33, 0xb1, 0x8e, 0x03, 0x6a, 0xa1, 0xda, 0x02,
    0x5a, 0x35, 0x9e, 0xd7,
};
static const unsigned char kat216_nonce[] = {
    0x99, 0xf0, 0xe4, 0x9c, 0xe8, 0x11, 0xee, 0x7e,
};
static const unsigned char kat216_persstr[] = {
    0x8d, 0xe7, 0xc8, 0x6b, 0x8e, 0xc6, 0x1d, 0x6a, 0xbb, 0x52, 0xa7, 0x91,
    0x66, 0x71, 0xad, 0xb9,
};
static const unsigned char kat216_addin0[] = {0};
static const unsigned char kat216_addin1[] = {0};
static const unsigned char kat216_retbits[] = {
    0x14, 0x07, 0xb6, 0x81, 0x51, 0xfc, 0xb0, 0xf0, 0x8e, 0xba, 0xbc, 0x21,
    0xc6, 0xc1, 0x81, 0xac, 0x1d, 0xbf, 0x9c, 0x6f, 0xb1, 0xb2, 0xc1, 0x6e,
    0xaf, 0x1f, 0x8c, 0x49, 0x0d, 0x6f, 0x7d, 0x52, 0xd0, 0xf4, 0x21, 0x11,
    0x6a, 0x59, 0x98, 0x33, 0x0d, 0x81, 0x05, 0xf5, 0x02, 0x76, 0x17, 0xdc,
    0x94, 0xb1, 0x4c, 0x08, 0x3f, 0x49, 0xd1, 0x1c, 0x34, 0xf4, 0xf2, 0x63,
    0x02, 0x31, 0x66, 0x24,
};
static const struct drbg_kat_no_reseed kat216_t = {
    5, kat216_entropyin, kat216_nonce, kat216_persstr,
    kat216_addin0, kat216_addin1, kat216_retbits
};
static const struct drbg_kat kat216 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat216_t
};

static const unsigned char kat217_entropyin[] = {
    0xd6, 0x7e, 0xeb, 0x00, 0xa4, 0x29, 0x1a, 0x18, 0x47, 0x11, 0x84, 0xb4,
    0x31, 0x59, 0xd2, 0xbd,
};
static const unsigned char kat217_nonce[] = {
    0xe4, 0xa3, 0x61, 0x49, 0x7e, 0xe1, 0x43, 0x8a,
};
static const unsigned char kat217_persstr[] = {
    0xf0, 0x54, 0xad, 0xf2, 0xad, 0x38, 0x49, 0xda, 0x72, 0x72, 0xb3, 0xb3,
    0x2a, 0xe0, 0xfc, 0xab,
};
static const unsigned char kat217_addin0[] = {0};
static const unsigned char kat217_addin1[] = {0};
static const unsigned char kat217_retbits[] = {
    0x52, 0x45, 0xc3, 0x0a, 0x65, 0x1d, 0x98, 0x61, 0xb6, 0x36, 0xc8, 0xe8,
    0xcc, 0x8b, 0x84, 0x52, 0x46, 0xda, 0x10, 0xc1, 0x04, 0xd7, 0x81, 0x34,
    0xa8, 0x4e, 0x41, 0xfe, 0xa8, 0x0e, 0x0e, 0x73, 0x69, 0x2f, 0x84, 0x81,
    0xcd, 0x4d, 0x75, 0x0c, 0x79, 0xe3, 0x87, 0x6b, 0x9a, 0xbf, 0xa9, 0xd6,
    0x14, 0xd8, 0x68, 0x24, 0x96, 0x05, 0x61, 0x9d, 0xef, 0xef, 0x29, 0x68,
    0xfd, 0x33, 0x14, 0x1e,
};
static const struct drbg_kat_no_reseed kat217_t = {
    6, kat217_entropyin, kat217_nonce, kat217_persstr,
    kat217_addin0, kat217_addin1, kat217_retbits
};
static const struct drbg_kat kat217 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat217_t
};

static const unsigned char kat218_entropyin[] = {
    0xce, 0x08, 0xf0, 0x2b, 0xcd, 0xe4, 0x7f, 0x60, 0x30, 0x82, 0x02, 0x43,
    0xe1, 0x01, 0x38, 0x87,
};
static const unsigned char kat218_nonce[] = {
    0xd1, 0xb1, 0x51, 0x30, 0xcd, 0x03, 0x8d, 0x6d,
};
static const unsigned char kat218_persstr[] = {
    0xad, 0x0b, 0xcb, 0x82, 0x01, 0x16, 0x0d, 0x82, 0xf1, 0x79, 0x66, 0xd4,
    0xc7, 0xb6, 0xa4, 0xec,
};
static const unsigned char kat218_addin0[] = {0};
static const unsigned char kat218_addin1[] = {0};
static const unsigned char kat218_retbits[] = {
    0xf4, 0x8e, 0xd0, 0x3b, 0xdb, 0xa9, 0x31, 0x0d, 0x7f, 0xe0, 0xa5, 0xdd,
    0xdf, 0x9d, 0xf4, 0x9c, 0x0d, 0xbe, 0x07, 0xb9, 0x5b, 0xde, 0x25, 0xa0,
    0xb6, 0x6e, 0xd0, 0x1a, 0x9f, 0x7a, 0x07, 0x82, 0x0f, 0x2d, 0x7e, 0xaa,
    0x98, 0x63, 0x70, 0xa0, 0xce, 0x00, 0x01, 0x3b, 0x43, 0x31, 0xe4, 0x4b,
    0xeb, 0x30, 0x10, 0x57, 0x5a, 0xf7, 0xd6, 0x25, 0xbe, 0xd5, 0x5a, 0x59,
    0x2d, 0x97, 0x38, 0x28,
};
static const struct drbg_kat_no_reseed kat218_t = {
    7, kat218_entropyin, kat218_nonce, kat218_persstr,
    kat218_addin0, kat218_addin1, kat218_retbits
};
static const struct drbg_kat kat218 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat218_t
};

static const unsigned char kat219_entropyin[] = {
    0x8d, 0xbc, 0xda, 0xa7, 0x8a, 0xdd, 0xb2, 0x85, 0xdf, 0xdb, 0x5e, 0x41,
    0xee, 0xf6, 0x7e, 0x6f,
};
static const unsigned char kat219_nonce[] = {
    0x49, 0x06, 0x2c, 0x09, 0x8b, 0x64, 0xe7, 0xf8,
};
static const unsigned char kat219_persstr[] = {
    0x7d, 0xc1, 0x4e, 0x73, 0x96, 0xf6, 0x24, 0x50, 0xbe, 0xbd, 0xf2, 0xeb,
    0xf0, 0x17, 0xaa, 0xd5,
};
static const unsigned char kat219_addin0[] = {0};
static const unsigned char kat219_addin1[] = {0};
static const unsigned char kat219_retbits[] = {
    0x49, 0xd0, 0x3f, 0xe3, 0xb7, 0x2d, 0x44, 0xa8, 0xa1, 0xe4, 0x69, 0xd2,
    0x5a, 0x14, 0x5b, 0xa6, 0x4d, 0x61, 0x69, 0xdd, 0x94, 0x7f, 0x87, 0x93,
    0xd5, 0xef, 0x43, 0xde, 0x7b, 0x23, 0x94, 0x13, 0x70, 0x83, 0xe6, 0xe7,
    0x69, 0xbb, 0xfd, 0xe9, 0x60, 0x0c, 0x36, 0xb0, 0x32, 0xb7, 0x78, 0x65,
    0x22, 0x07, 0x0b, 0x5a, 0x65, 0xc7, 0x93, 0x92, 0x68, 0x92, 0xb9, 0xfb,
    0x0d, 0x1c, 0x1d, 0x54,
};
static const struct drbg_kat_no_reseed kat219_t = {
    8, kat219_entropyin, kat219_nonce, kat219_persstr,
    kat219_addin0, kat219_addin1, kat219_retbits
};
static const struct drbg_kat kat219 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat219_t
};

static const unsigned char kat220_entropyin[] = {
    0x0d, 0x98, 0x37, 0x0c, 0xff, 0xa0, 0xa8, 0x79, 0xfe, 0x85, 0xb5, 0xde,
    0x4e, 0x69, 0xcf, 0x7d,
};
static const unsigned char kat220_nonce[] = {
    0x93, 0x50, 0xa8, 0x6b, 0x7a, 0x1b, 0xc6, 0x8a,
};
static const unsigned char kat220_persstr[] = {
    0x01, 0x61, 0xb2, 0x4d, 0x1c, 0x3c, 0xa5, 0x90, 0x11, 0x7e, 0xc2, 0x9a,
    0xcc, 0x39, 0x44, 0x6f,
};
static const unsigned char kat220_addin0[] = {0};
static const unsigned char kat220_addin1[] = {0};
static const unsigned char kat220_retbits[] = {
    0x7c, 0xd6, 0x46, 0x84, 0x28, 0x9b, 0x43, 0xb1, 0xe5, 0x93, 0xd9, 0x4a,
    0xe9, 0x79, 0xf7, 0xfa, 0x5d, 0xe8, 0x95, 0x77, 0xf3, 0x7c, 0x7a, 0xea,
    0x0b, 0x58, 0x4d, 0x18, 0x60, 0x2c, 0x25, 0x1b, 0x1e, 0xc2, 0x5f, 0xf6,
    0x12, 0xd9, 0x7f, 0xda, 0x37, 0x84, 0xe9, 0x65, 0x59, 0x73, 0xe3, 0x1f,
    0xd4, 0xb5, 0xd1, 0xea, 0xde, 0xc6, 0x68, 0x61, 0xae, 0x2e, 0x97, 0x19,
    0xad, 0x34, 0x47, 0x30,
};
static const struct drbg_kat_no_reseed kat220_t = {
    9, kat220_entropyin, kat220_nonce, kat220_persstr,
    kat220_addin0, kat220_addin1, kat220_retbits
};
static const struct drbg_kat kat220 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat220_t
};

static const unsigned char kat221_entropyin[] = {
    0xd9, 0xb6, 0xfb, 0x62, 0xda, 0x0a, 0x02, 0x32, 0x35, 0xe7, 0xcd, 0x78,
    0x16, 0x37, 0x7c, 0xf5,
};
static const unsigned char kat221_nonce[] = {
    0x1c, 0xa8, 0xc5, 0xfa, 0x5b, 0x9f, 0x8c, 0x90,
};
static const unsigned char kat221_persstr[] = {
    0x17, 0xbe, 0x48, 0x85, 0xb5, 0xf9, 0x64, 0x1b, 0xf5, 0xf6, 0x89, 0xbc,
    0x97, 0x97, 0x88, 0x58,
};
static const unsigned char kat221_addin0[] = {0};
static const unsigned char kat221_addin1[] = {0};
static const unsigned char kat221_retbits[] = {
    0xe1, 0x1e, 0x3c, 0xa6, 0xc8, 0x32, 0xf4, 0x6f, 0xf1, 0xf9, 0x71, 0xc0,
    0x7e, 0x7b, 0x66, 0xc7, 0xd5, 0xb1, 0xb2, 0xe6, 0xec, 0x8c, 0x5f, 0xfc,
    0x77, 0x10, 0x3f, 0x0a, 0xd3, 0x08, 0x80, 0x0b, 0xb9, 0x89, 0xb9, 0xab,
    0x70, 0x10, 0x68, 0x33, 0x78, 0xa3, 0xf7, 0x81, 0x9a, 0x29, 0x7a, 0x37,
    0x65, 0x25, 0x6a, 0xc4, 0x0f, 0xb0, 0xa9, 0xcb, 0x22, 0x46, 0xae, 0xb8,
    0x5d, 0x73, 0x60, 0x1b,
};
static const struct drbg_kat_no_reseed kat221_t = {
    10, kat221_entropyin, kat221_nonce, kat221_persstr,
    kat221_addin0, kat221_addin1, kat221_retbits
};
static const struct drbg_kat kat221 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat221_t
};

static const unsigned char kat222_entropyin[] = {
    0xa7, 0x11, 0x39, 0x44, 0xfc, 0x4d, 0x4b, 0x15, 0xa9, 0x49, 0x67, 0xc8,
    0xf5, 0x65, 0x2a, 0x36,
};
static const unsigned char kat222_nonce[] = {
    0xa5, 0x06, 0xb7, 0x9a, 0xfe, 0x6a, 0xf8, 0x22,
};
static const unsigned char kat222_persstr[] = {
    0x91, 0x8f, 0xe4, 0xf3, 0x65, 0x25, 0x9c, 0x18, 0xeb, 0x68, 0x50, 0xfb,
    0xee, 0x40, 0x3f, 0x5e,
};
static const unsigned char kat222_addin0[] = {0};
static const unsigned char kat222_addin1[] = {0};
static const unsigned char kat222_retbits[] = {
    0x60, 0x76, 0xb2, 0xc0, 0xf7, 0xde, 0x1d, 0xd3, 0xe4, 0x6a, 0xdb, 0x11,
    0x61, 0xb7, 0x2a, 0x7f, 0x83, 0xb8, 0x77, 0x3f, 0xab, 0x0d, 0xcb, 0x1c,
    0x1f, 0xde, 0x25, 0xdb, 0x0d, 0x09, 0x86, 0x2a, 0xcd, 0xd3, 0x8f, 0x8d,
    0x21, 0x64, 0x90, 0x3a, 0x8e, 0x88, 0x58, 0xf8, 0xa9, 0xb6, 0x1b, 0xd7,
    0xda, 0xe3, 0xf6, 0x06, 0x68, 0xe6, 0xee, 0x26, 0x4b, 0x9f, 0xfa, 0xf5,
    0x78, 0xb5, 0x56, 0x46,
};
static const struct drbg_kat_no_reseed kat222_t = {
    11, kat222_entropyin, kat222_nonce, kat222_persstr,
    kat222_addin0, kat222_addin1, kat222_retbits
};
static const struct drbg_kat kat222 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat222_t
};

static const unsigned char kat223_entropyin[] = {
    0xac, 0xf7, 0x62, 0x38, 0x39, 0xb9, 0x4b, 0xd7, 0xa8, 0x93, 0xc2, 0x87,
    0x61, 0x6c, 0xdb, 0x6a,
};
static const unsigned char kat223_nonce[] = {
    0xa1, 0xf6, 0x03, 0x9f, 0x91, 0xc1, 0x76, 0x23,
};
static const unsigned char kat223_persstr[] = {
    0x5f, 0x9c, 0x69, 0xde, 0x2c, 0x32, 0x87, 0x3f, 0x67, 0x9d, 0x17, 0x68,
    0xef, 0x3c, 0x70, 0xfb,
};
static const unsigned char kat223_addin0[] = {0};
static const unsigned char kat223_addin1[] = {0};
static const unsigned char kat223_retbits[] = {
    0xf6, 0xab, 0xbf, 0xfd, 0x79, 0x65, 0x64, 0x94, 0x36, 0xb4, 0x8b, 0x09,
    0x20, 0x86, 0xcd, 0xf0, 0x50, 0x2f, 0x52, 0xc8, 0x7c, 0x8f, 0xdc, 0x8f,
    0x6d, 0x49, 0xf0, 0x84, 0x33, 0xa2, 0x02, 0xd8, 0xbe, 0x44, 0xf6, 0x56,
    0x2a, 0xce, 0x58, 0x0e, 0x80, 0x75, 0xf0, 0xbc, 0x67, 0x0b, 0x2e, 0x9d,
    0x95, 0x1f, 0x15, 0xf8, 0x4d, 0x82, 0xaf, 0xe6, 0xb8, 0x32, 0x76, 0x5e,
    0xca, 0xb0, 0x37, 0xee,
};
static const struct drbg_kat_no_reseed kat223_t = {
    12, kat223_entropyin, kat223_nonce, kat223_persstr,
    kat223_addin0, kat223_addin1, kat223_retbits
};
static const struct drbg_kat kat223 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat223_t
};

static const unsigned char kat224_entropyin[] = {
    0xd9, 0x0d, 0x78, 0x6e, 0x9d, 0x19, 0x62, 0x84, 0x3b, 0x02, 0x7a, 0xa0,
    0xe8, 0x59, 0x8f, 0xd4,
};
static const unsigned char kat224_nonce[] = {
    0x04, 0x17, 0xf1, 0x4f, 0x96, 0xf2, 0x02, 0x9c,
};
static const unsigned char kat224_persstr[] = {
    0xec, 0x5c, 0x55, 0x14, 0x5a, 0x4f, 0x95, 0x52, 0xf2, 0x51, 0xf7, 0xdf,
    0xb0, 0x17, 0xc1, 0xd4,
};
static const unsigned char kat224_addin0[] = {0};
static const unsigned char kat224_addin1[] = {0};
static const unsigned char kat224_retbits[] = {
    0xfe, 0x2c, 0xa4, 0x44, 0x55, 0xa5, 0xe7, 0xde, 0x70, 0x8a, 0x71, 0x0a,
    0xb6, 0x46, 0xcf, 0xcf, 0xbc, 0x87, 0xd6, 0x78, 0xb3, 0xe9, 0x41, 0x06,
    0x3f, 0xa7, 0x01, 0xda, 0xc7, 0xcb, 0x4f, 0x09, 0x53, 0xaf, 0x8f, 0xce,
    0x45, 0x8c, 0xee, 0x54, 0x9f, 0x53, 0x7f, 0xa7, 0x71, 0xb8, 0x9b, 0x04,
    0x94, 0xc0, 0xbb, 0x5c, 0xfd, 0x35, 0xbf, 0x67, 0xf4, 0xb2, 0x70, 0xd6,
    0x20, 0x5f, 0x47, 0x03,
};
static const struct drbg_kat_no_reseed kat224_t = {
    13, kat224_entropyin, kat224_nonce, kat224_persstr,
    kat224_addin0, kat224_addin1, kat224_retbits
};
static const struct drbg_kat kat224 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat224_t
};

static const unsigned char kat225_entropyin[] = {
    0x1c, 0xdb, 0x74, 0x45, 0x59, 0xa8, 0x2c, 0x62, 0xfb, 0x3b, 0x15, 0xa2,
    0xa3, 0xaa, 0xd1, 0x89,
};
static const unsigned char kat225_nonce[] = {
    0x43, 0x43, 0x2c, 0xe3, 0x0a, 0xe7, 0xf9, 0x6b,
};
static const unsigned char kat225_persstr[] = {
    0xf5, 0x9a, 0xd9, 0x93, 0x9f, 0xcd, 0x6c, 0x0f, 0x47, 0x8c, 0xc5, 0x08,
    0x39, 0xf8, 0xff, 0xce,
};
static const unsigned char kat225_addin0[] = {0};
static const unsigned char kat225_addin1[] = {0};
static const unsigned char kat225_retbits[] = {
    0x91, 0xb9, 0xeb, 0xeb, 0x92, 0x9d, 0xc6, 0x99, 0x94, 0x91, 0x22, 0x97,
    0x0b, 0x21, 0x77, 0xd5, 0xd5, 0xb7, 0x05, 0x04, 0x2d, 0x3a, 0x0d, 0x60,
    0xd7, 0xd3, 0xbb, 0x21, 0x8b, 0x7a, 0x69, 0xe1, 0xcd, 0x68, 0x64, 0xb8,
    0xa2, 0x68, 0xca, 0x78, 0xc8, 0x34, 0x23, 0x2d, 0x0c, 0xb8, 0x8f, 0x93,
    0x77, 0x30, 0x08, 0x4b, 0x1e, 0xc7, 0xd2, 0x86, 0x7f, 0xbc, 0x85, 0x0e,
    0x04, 0x05, 0x03, 0x24,
};
static const struct drbg_kat_no_reseed kat225_t = {
    14, kat225_entropyin, kat225_nonce, kat225_persstr,
    kat225_addin0, kat225_addin1, kat225_retbits
};
static const struct drbg_kat kat225 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat225_t
};

static const unsigned char kat226_entropyin[] = {
    0x71, 0xba, 0x02, 0x9a, 0x7a, 0x92, 0x39, 0x1b, 0x3f, 0x29, 0x4f, 0x2f,
    0xbf, 0x57, 0x27, 0xab,
};
static const unsigned char kat226_nonce[] = {
    0xf0, 0xe9, 0x12, 0x78, 0x8f, 0x98, 0x27, 0xff,
};
static const unsigned char kat226_persstr[] = {
    0x23, 0xf3, 0x69, 0x80, 0xda, 0x40, 0x16, 0x64, 0x2c, 0x81, 0x0d, 0xa2,
    0x99, 0x0a, 0xa2, 0x5e,
};
static const unsigned char kat226_addin0[] = {
    0x59, 0xab, 0x41, 0xb2, 0x4e, 0xe8, 0xe2, 0x71, 0xe2, 0x53, 0xc6, 0xcc,
    0x40, 0x48, 0x7c, 0xb5,
};
static const unsigned char kat226_addin1[] = {
    0x91, 0x64, 0xf0, 0x28, 0x60, 0x77, 0x3e, 0x3b, 0x96, 0xd8, 0x5b, 0x37,
    0x38, 0x38, 0x50, 0x66,
};
static const unsigned char kat226_retbits[] = {
    0xde, 0x1b, 0x8a, 0x25, 0x95, 0x89, 0x23, 0x54, 0xda, 0x47, 0xb4, 0xea,
    0xaf, 0x9d, 0xdc, 0xec, 0x64, 0xa9, 0x61, 0x01, 0x17, 0xb0, 0x5e, 0x40,
    0xd0, 0x76, 0x60, 0xa8, 0x0b, 0xcf, 0x82, 0x5e, 0xef, 0xdb, 0xd2, 0x8e,
    0x07, 0xd5, 0x96, 0x81, 0xf9, 0xe0, 0x03, 0x7b, 0xdb, 0x72, 0x5f, 0xe6,
    0xce, 0x84, 0x6d, 0x82, 0x4b, 0x3b, 0x34, 0xc2, 0xc2, 0x1a, 0x48, 0xf8,
    0x89, 0x5f, 0x9f, 0x5e,
};
static const struct drbg_kat_no_reseed kat226_t = {
    0, kat226_entropyin, kat226_nonce, kat226_persstr,
    kat226_addin0, kat226_addin1, kat226_retbits
};
static const struct drbg_kat kat226 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat226_t
};

static const unsigned char kat227_entropyin[] = {
    0x67, 0x73, 0x94, 0xf0, 0x3e, 0xac, 0xb5, 0xa1, 0x37, 0x51, 0x5f, 0xcd,
    0x3e, 0xa2, 0xbb, 0xc7,
};
static const unsigned char kat227_nonce[] = {
    0x9a, 0x5c, 0x1c, 0xea, 0x26, 0xef, 0xbf, 0x76,
};
static const unsigned char kat227_persstr[] = {
    0xdd, 0x73, 0xb1, 0x8e, 0xee, 0x62, 0x94, 0x34, 0x9a, 0xa6, 0x45, 0x6d,
    0xaa, 0x77, 0xd2, 0x4f,
};
static const unsigned char kat227_addin0[] = {
    0xeb, 0x70, 0x19, 0x82, 0x6d, 0xb3, 0x27, 0x17, 0xa5, 0x73, 0x5e, 0xb4,
    0x07, 0x73, 0xea, 0x56,
};
static const unsigned char kat227_addin1[] = {
    0xb6, 0x2d, 0xe2, 0xb4, 0xaa, 0x8d, 0x97, 0xf5, 0x26, 0xdd, 0xba, 0x44,
    0x09, 0xf5, 0xaa, 0x26,
};
static const unsigned char kat227_retbits[] = {
    0x22, 0x22, 0xa6, 0xbb, 0x0f, 0xfe, 0x14, 0xd2, 0xf7, 0x89, 0xe6, 0x4d,
    0xcd, 0xf4, 0x78, 0x51, 0xa6, 0xc3, 0xa6, 0xe1, 0xd0, 0x83, 0x7d, 0x8b,
    0x51, 0x1a, 0xa2, 0xf5, 0x6a, 0x6d, 0x08, 0x53, 0x4c, 0x97, 0xf4, 0xb1,
    0x2a, 0x77, 0x44, 0x7d, 0xb2, 0x04, 0x09, 0xd3, 0x27, 0xfc, 0x08, 0x81,
    0x62, 0xc0, 0xf2, 0xc5, 0x9f, 0xe4, 0x7e, 0x8c, 0x92, 0xae, 0x5d, 0xcd,
    0xd7, 0x38, 0xc7, 0x68,
};
static const struct drbg_kat_no_reseed kat227_t = {
    1, kat227_entropyin, kat227_nonce, kat227_persstr,
    kat227_addin0, kat227_addin1, kat227_retbits
};
static const struct drbg_kat kat227 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat227_t
};

static const unsigned char kat228_entropyin[] = {
    0x3f, 0x51, 0xf6, 0xcc, 0xf2, 0x24, 0x79, 0xed, 0x89, 0x1b, 0x89, 0x1f,
    0xc4, 0xc6, 0xb5, 0x51,
};
static const unsigned char kat228_nonce[] = {
    0x19, 0xe0, 0x75, 0x4f, 0x7b, 0xfa, 0x0c, 0xef,
};
static const unsigned char kat228_persstr[] = {
    0x40, 0x2a, 0x1d, 0xab, 0x45, 0x0e, 0xa9, 0x04, 0x97, 0x38, 0x78, 0x2a,
    0x93, 0x34, 0x7a, 0xff,
};
static const unsigned char kat228_addin0[] = {
    0x52, 0xc9, 0x0f, 0x57, 0x1c, 0x8c, 0x63, 0xf0, 0xdc, 0xdb, 0xd4, 0xe6,
    0xc9, 0x65, 0x34, 0x78,
};
static const unsigned char kat228_addin1[] = {
    0x8b, 0xe6, 0x3b, 0x5e, 0xf2, 0x12, 0x69, 0x4e, 0xdb, 0xc2, 0xc8, 0xde,
    0xc9, 0xa0, 0x58, 0x7b,
};
static const unsigned char kat228_retbits[] = {
    0x1a, 0xbf, 0xe2, 0xd7, 0x21, 0x93, 0xb6, 0xe3, 0xd9, 0xbe, 0x85, 0x10,
    0x7e, 0xce, 0x8f, 0xea, 0x5d, 0x22, 0x95, 0x68, 0x56, 0x8c, 0x93, 0xeb,
    0x5e, 0xae, 0xd4, 0x63, 0xbf, 0x2b, 0x79, 0xcf, 0xdb, 0xd0, 0x47, 0x40,
    0x2f, 0xf4, 0xe4, 0x22, 0x34, 0xc4, 0xe3, 0xe1, 0x50, 0xc1, 0xf0, 0xe8,
    0x24, 0x00, 0xf6, 0xa5, 0x9e, 0xeb, 0xd0, 0x92, 0xee, 0x73, 0xad, 0xf1,
    0xff, 0xca, 0x64, 0x94,
};
static const struct drbg_kat_no_reseed kat228_t = {
    2, kat228_entropyin, kat228_nonce, kat228_persstr,
    kat228_addin0, kat228_addin1, kat228_retbits
};
static const struct drbg_kat kat228 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat228_t
};

static const unsigned char kat229_entropyin[] = {
    0x5a, 0x01, 0xb6, 0x38, 0x31, 0x1a, 0x29, 0x6b, 0xfe, 0x1f, 0x6e, 0x18,
    0x0f, 0x24, 0x3d, 0x5b,
};
static const unsigned char kat229_nonce[] = {
    0x16, 0xbb, 0x4e, 0x36, 0xfd, 0x04, 0xb0, 0x5b,
};
static const unsigned char kat229_persstr[] = {
    0xd5, 0x44, 0x0a, 0x6f, 0xa4, 0xd3, 0x45, 0xcb, 0x84, 0xd5, 0x6d, 0xec,
    0xaa, 0x3d, 0xbd, 0x18,
};
static const unsigned char kat229_addin0[] = {
    0xbd, 0x3e, 0x11, 0xdb, 0xb9, 0xf4, 0x0a, 0x13, 0x51, 0xc8, 0xbf, 0x7b,
    0xd8, 0x94, 0xad, 0x20,
};
static const unsigned char kat229_addin1[] = {
    0xe0, 0x4a, 0x83, 0xa5, 0x18, 0x75, 0x15, 0x10, 0x4a, 0xa1, 0xe7, 0xf4,
    0x34, 0xff, 0x43, 0x36,
};
static const unsigned char kat229_retbits[] = {
    0x27, 0x28, 0xfc, 0xc1, 0xfe, 0xa7, 0xfe, 0x63, 0x5d, 0xdf, 0xed, 0x8b,
    0xb3, 0xf7, 0xca, 0x29, 0xe1, 0xcb, 0xd8, 0x51, 0x6a, 0x2b, 0x5e, 0xab,
    0xb7, 0xd0, 0x91, 0xe6, 0xe7, 0xf2, 0x04, 0x6a, 0x80, 0x77, 0xef, 0x6d,
    0x1a, 0x90, 0x16, 0x5f, 0x18, 0x9d, 0x07, 0xa8, 0x97, 0xcd, 0xfc, 0x06,
    0x88, 0x2a, 0xdb, 0x78, 0x1b, 0xe5, 0x89, 0xe0, 0xe5, 0xeb, 0x2d, 0xc9,
    0x7b, 0xe6, 0x54, 0x6b,
};
static const struct drbg_kat_no_reseed kat229_t = {
    3, kat229_entropyin, kat229_nonce, kat229_persstr,
    kat229_addin0, kat229_addin1, kat229_retbits
};
static const struct drbg_kat kat229 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat229_t
};

static const unsigned char kat230_entropyin[] = {
    0x97, 0xe4, 0x15, 0xb3, 0xca, 0x11, 0x0a, 0x80, 0x9e, 0xa6, 0x00, 0xa1,
    0x3e, 0x47, 0xaa, 0x04,
};
static const unsigned char kat230_nonce[] = {
    0x12, 0x6e, 0x16, 0xdc, 0x57, 0x22, 0xa4, 0x49,
};
static const unsigned char kat230_persstr[] = {
    0xa9, 0xbe, 0x38, 0xe3, 0x01, 0x19, 0x86, 0xb9, 0x04, 0x71, 0x71, 0x93,
    0xec, 0x4a, 0x7d, 0x4c,
};
static const unsigned char kat230_addin0[] = {
    0x19, 0x68, 0x69, 0x92, 0x21, 0xeb, 0x8b, 0x5e, 0xd0, 0x23, 0xa5, 0xd2,
    0xd2, 0xd8, 0x12, 0x56,
};
static const unsigned char kat230_addin1[] = {
    0x4b, 0x5e, 0x1b, 0xde, 0x38, 0xb2, 0xa9, 0x79, 0x3b, 0x16, 0x6d, 0x86,
    0x81, 0x6e, 0xd2, 0x31,
};
static const unsigned char kat230_retbits[] = {
    0x4b, 0xe9, 0x89, 0x89, 0xdf, 0x2f, 0xe6, 0x68, 0x7e, 0x50, 0x88, 0xe5,
    0x06, 0x98, 0x50, 0x7d, 0x91, 0x06, 0x1a, 0x8d, 0x6e, 0xb9, 0x5d, 0x66,
    0x27, 0xd2, 0xb8, 0x57, 0x25, 0x3f, 0x05, 0x3c, 0x99, 0xee, 0x79, 0x84,
    0xa4, 0x98, 0xde, 0x4d, 0xf3, 0x12, 0x44, 0x42, 0x85, 0x6f, 0x18, 0xe9,
    0x8e, 0x66, 0x41, 0xf1, 0x52, 0xcd, 0x12, 0x9b, 0xd0, 0x4e, 0x99, 0x71,
    0x09, 0x30, 0xc3, 0xe1,
};
static const struct drbg_kat_no_reseed kat230_t = {
    4, kat230_entropyin, kat230_nonce, kat230_persstr,
    kat230_addin0, kat230_addin1, kat230_retbits
};
static const struct drbg_kat kat230 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat230_t
};

static const unsigned char kat231_entropyin[] = {
    0xb6, 0xc4, 0xcd, 0x96, 0xf7, 0xfd, 0xe8, 0x3e, 0xb5, 0x71, 0x22, 0x9b,
    0x53, 0x7a, 0xcf, 0x5a,
};
static const unsigned char kat231_nonce[] = {
    0x4c, 0x4e, 0x0a, 0xac, 0xb5, 0xc5, 0x28, 0xe7,
};
static const unsigned char kat231_persstr[] = {
    0x5c, 0xe1, 0x09, 0x4d, 0x0d, 0x30, 0x7b, 0x41, 0xdb, 0x9b, 0xf2, 0x6d,
    0xa4, 0x1d, 0xb1, 0x00,
};
static const unsigned char kat231_addin0[] = {
    0x98, 0x04, 0x7e, 0xe9, 0x87, 0x63, 0x7e, 0x45, 0x84, 0xfe, 0x7f, 0xe2,
    0x86, 0x9c, 0x84, 0x5d,
};
static const unsigned char kat231_addin1[] = {
    0x17, 0x7e, 0x3c, 0x29, 0xd4, 0xfa, 0xf0, 0x43, 0x07, 0x7f, 0xbe, 0xfe,
    0x0b, 0x99, 0x4b, 0x91,
};
static const unsigned char kat231_retbits[] = {
    0x5e, 0xaf, 0x01, 0x3f, 0xd3, 0x77, 0x5a, 0x2d, 0x3e, 0x97, 0x16, 0x6d,
    0x23, 0x15, 0x3a, 0x00, 0xf1, 0x86, 0x44, 0x96, 0x3e, 0x69, 0xcb, 0x96,
    0x2a, 0x7b, 0xd2, 0xc3, 0x0a, 0xf9, 0xbd, 0x6a, 0x29, 0xd4, 0xbc, 0x70,
    0x93, 0x0a, 0x92, 0x9b, 0x92, 0x52, 0xd8, 0x08, 0xdb, 0x41, 0x0c, 0x66,
    0x41, 0x23, 0xbd, 0x69, 0xc1, 0xd0, 0xc1, 0xd4, 0xba, 0x3c, 0x7d, 0x8f,
    0xf7, 0xae, 0x00, 0xf0,
};
static const struct drbg_kat_no_reseed kat231_t = {
    5, kat231_entropyin, kat231_nonce, kat231_persstr,
    kat231_addin0, kat231_addin1, kat231_retbits
};
static const struct drbg_kat kat231 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat231_t
};

static const unsigned char kat232_entropyin[] = {
    0x2d, 0x1b, 0x80, 0x61, 0xd7, 0xd5, 0x83, 0xf8, 0x3b, 0xe5, 0x53, 0xc4,
    0x83, 0x72, 0x12, 0xa0,
};
static const unsigned char kat232_nonce[] = {
    0x62, 0xa1, 0xfa, 0x74, 0x53, 0x62, 0xba, 0xce,
};
static const unsigned char kat232_persstr[] = {
    0x4d, 0x97, 0x6a, 0xdd, 0x18, 0xd5, 0xe2, 0x48, 0x4f, 0x67, 0xf7, 0xc8,
    0x4c, 0xc6, 0x8b, 0x52,
};
static const unsigned char kat232_addin0[] = {
    0x57, 0x64, 0xde, 0x5d, 0xb9, 0x7e, 0x22, 0x3c, 0x04, 0x4a, 0x83, 0x3f,
    0xa4, 0x2c, 0xf6, 0x29,
};
static const unsigned char kat232_addin1[] = {
    0xde, 0xef, 0x8d, 0x58, 0x1f, 0x6e, 0x4f, 0x80, 0x89, 0x92, 0xfd, 0x20,
    0xc0, 0x6e, 0x10, 0x2d,
};
static const unsigned char kat232_retbits[] = {
    0xac, 0xb7, 0x8f, 0x27, 0x33, 0x50, 0xd4, 0x59, 0xc7, 0x98, 0x09, 0xd8,
    0x9b, 0x40, 0x28, 0x25, 0xbb, 0x00, 0xe3, 0x8a, 0xbd, 0x5e, 0xc1, 0x2f,
    0xea, 0xdc, 0xde, 0xf6, 0x11, 0x46, 0x84, 0xb6, 0xfc, 0xb0, 0xe1, 0x76,
    0xcb, 0xe4, 0x65, 0x91, 0x0f, 0xc2, 0x84, 0x8f, 0x0a, 0x70, 0x88, 0x44,
    0x79, 0x4c, 0x5e, 0xea, 0xfe, 0xad, 0xa7, 0x56, 0x3c, 0x38, 0xc3, 0xee,
    0x63, 0xc9, 0x7e, 0xf9,
};
static const struct drbg_kat_no_reseed kat232_t = {
    6, kat232_entropyin, kat232_nonce, kat232_persstr,
    kat232_addin0, kat232_addin1, kat232_retbits
};
static const struct drbg_kat kat232 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat232_t
};

static const unsigned char kat233_entropyin[] = {
    0xef, 0x30, 0xfd, 0x99, 0x5f, 0xf0, 0xef, 0x44, 0xcc, 0xe6, 0x07, 0x7b,
    0xb2, 0x2d, 0xe6, 0x99,
};
static const unsigned char kat233_nonce[] = {
    0x85, 0xc7, 0xd6, 0x05, 0x4a, 0x57, 0x76, 0x22,
};
static const unsigned char kat233_persstr[] = {
    0x5d, 0x59, 0xb3, 0xf0, 0x25, 0x00, 0x60, 0xe2, 0x20, 0x02, 0xad, 0xf7,
    0x06, 0x48, 0x6d, 0x85,
};
static const unsigned char kat233_addin0[] = {
    0x79, 0x46, 0x81, 0xf7, 0x5d, 0xf9, 0xc9, 0x88, 0x30, 0x3f, 0x75, 0x1f,
    0xb7, 0xe7, 0xfb, 0xa9,
};
static const unsigned char kat233_addin1[] = {
    0x7b, 0x13, 0x74, 0xc8, 0x47, 0x25, 0x37, 0xfa, 0x31, 0x11, 0xeb, 0x44,
    0xe5, 0x2a, 0xfa, 0x23,
};
static const unsigned char kat233_retbits[] = {
    0xb0, 0x49, 0x50, 0xf1, 0xa3, 0x57, 0x52, 0xc0, 0x67, 0xbc, 0xe8, 0x35,
    0x76, 0x35, 0xf4, 0x70, 0xd1, 0x0b, 0x3b, 0xae, 0x47, 0x46, 0x6f, 0xd1,
    0x16, 0x6f, 0x9f, 0xfa, 0xef, 0xbd, 0x32, 0x4b, 0x26, 0x53, 0xc7, 0xf3,
    0x0e, 0xe4, 0x84, 0x86, 0x73, 0xd7, 0x04, 0x18, 0x65, 0xd9, 0x95, 0x45,
    0x12, 0x58, 0x94, 0x92, 0x98, 0x9a, 0x00, 0xea, 0x4d, 0xa1, 0xdd, 0x28,
    0x9e, 0x87, 0x17, 0xc3,
};
static const struct drbg_kat_no_reseed kat233_t = {
    7, kat233_entropyin, kat233_nonce, kat233_persstr,
    kat233_addin0, kat233_addin1, kat233_retbits
};
static const struct drbg_kat kat233 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat233_t
};

static const unsigned char kat234_entropyin[] = {
    0xee, 0x4b, 0xde, 0x4f, 0x71, 0x2d, 0xc9, 0xfc, 0x1f, 0x2c, 0x01, 0xf1,
    0xc6, 0x08, 0x6f, 0x35,
};
static const unsigned char kat234_nonce[] = {
    0xf6, 0x83, 0x7c, 0xfe, 0x53, 0xec, 0x48, 0x74,
};
static const unsigned char kat234_persstr[] = {
    0xfd, 0xb8, 0xf6, 0x4d, 0x99, 0xbd, 0x71, 0x7e, 0xee, 0x3d, 0x89, 0x35,
    0x3b, 0x73, 0xc9, 0xc1,
};
static const unsigned char kat234_addin0[] = {
    0x09, 0xe5, 0xba, 0x3b, 0x23, 0x89, 0x5c, 0x5d, 0xff, 0x89, 0x38, 0x2d,
    0x5e, 0x91, 0x70, 0x0d,
};
static const unsigned char kat234_addin1[] = {
    0xeb, 0x1a, 0x98, 0xde, 0x89, 0x62, 0xbb, 0xc4, 0xcb, 0x75, 0xcf, 0x0b,
    0xf0, 0xf8, 0xdf, 0xa4,
};
static const unsigned char kat234_retbits[] = {
    0xaa, 0xcd, 0xee, 0x0f, 0x3d, 0x1f, 0x95, 0x5f, 0x89, 0x6d, 0x5a, 0x5a,
    0x25, 0x30, 0xfd, 0x80, 0x9a, 0x20, 0x2c, 0x92, 0x1d, 0x90, 0xa9, 0xcf,
    0x59, 0x3e, 0x03, 0x93, 0x4e, 0x07, 0xc3, 0x92, 0xcc, 0x55, 0x54, 0xf5,
    0x99, 0x21, 0xb4, 0x0c, 0xc0, 0x03, 0xfc, 0x26, 0x82, 0xca, 0xef, 0x9e,
    0x6e, 0x61, 0xee, 0x4a, 0xbf, 0x45, 0x14, 0x8e, 0x5f, 0xf3, 0xff, 0x65,
    0x87, 0x1b, 0x0d, 0x81,
};
static const struct drbg_kat_no_reseed kat234_t = {
    8, kat234_entropyin, kat234_nonce, kat234_persstr,
    kat234_addin0, kat234_addin1, kat234_retbits
};
static const struct drbg_kat kat234 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat234_t
};

static const unsigned char kat235_entropyin[] = {
    0xa6, 0xaa, 0x94, 0x5e, 0xbe, 0x19, 0x29, 0x98, 0xbe, 0x8e, 0x24, 0x7d,
    0xb9, 0x9c, 0xe7, 0xd0,
};
static const unsigned char kat235_nonce[] = {
    0x7d, 0xff, 0x26, 0xe9, 0x7a, 0x0a, 0x8d, 0xba,
};
static const unsigned char kat235_persstr[] = {
    0x4b, 0x6c, 0xcf, 0xac, 0x6c, 0x47, 0xfd, 0x65, 0x64, 0xbc, 0xfd, 0x94,
    0xfe, 0x9e, 0xb2, 0x8d,
};
static const unsigned char kat235_addin0[] = {
    0x07, 0x83, 0x70, 0x53, 0xd2, 0x4c, 0xd0, 0xf4, 0xf4, 0x32, 0x7e, 0x97,
    0xd2, 0x9d, 0x67, 0x95,
};
static const unsigned char kat235_addin1[] = {
    0x3b, 0x17, 0x2d, 0xdc, 0x8d, 0x38, 0x3b, 0x98, 0xd4, 0x08, 0x39, 0x4c,
    0x0b, 0xa1, 0x52, 0xc1,
};
static const unsigned char kat235_retbits[] = {
    0xd9, 0x0a, 0xa4, 0x22, 0xa8, 0xd2, 0xb8, 0x6e, 0x1f, 0x09, 0xde, 0x29,
    0xdd, 0x16, 0xd4, 0x60, 0x50, 0x98, 0xa4, 0x5a, 0xea, 0xbe, 0x69, 0xc6,
    0xbc, 0x8a, 0xe0, 0xc8, 0xf0, 0x1b, 0x7c, 0xc7, 0x3f, 0x88, 0x50, 0xec,
    0x0c, 0x47, 0xd6, 0x16, 0xcb, 0x6c, 0xca, 0xb7, 0x7f, 0x22, 0x07, 0x99,
    0x38, 0xed, 0x11, 0xb4, 0x8f, 0xcf, 0x3c, 0xf4, 0x6a, 0x3f, 0x64, 0x2f,
    0x05, 0xae, 0xb8, 0xb7,
};
static const struct drbg_kat_no_reseed kat235_t = {
    9, kat235_entropyin, kat235_nonce, kat235_persstr,
    kat235_addin0, kat235_addin1, kat235_retbits
};
static const struct drbg_kat kat235 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat235_t
};

static const unsigned char kat236_entropyin[] = {
    0x7e, 0xb5, 0x0f, 0x53, 0x2b, 0x00, 0x76, 0x52, 0x8e, 0x2d, 0x1f, 0x26,
    0x6b, 0x38, 0x14, 0x06,
};
static const unsigned char kat236_nonce[] = {
    0x2a, 0x5e, 0x92, 0xe8, 0xfb, 0xf0, 0xe9, 0xee,
};
static const unsigned char kat236_persstr[] = {
    0x7e, 0xcb, 0xfd, 0x22, 0x98, 0x87, 0x2a, 0x79, 0x34, 0xe4, 0xed, 0x61,
    0xa9, 0xf0, 0x04, 0xcd,
};
static const unsigned char kat236_addin0[] = {
    0x17, 0x9f, 0xa5, 0xd2, 0xeb, 0x90, 0xd4, 0x15, 0x48, 0x19, 0x2c, 0xc7,
    0x44, 0x98, 0xbd, 0x42,
};
static const unsigned char kat236_addin1[] = {
    0x92, 0xdc, 0x9d, 0x60, 0xd2, 0xe3, 0x6d, 0x4b, 0xfb, 0x97, 0x90, 0x78,
    0x3d, 0x1c, 0x58, 0x19,
};
static const unsigned char kat236_retbits[] = {
    0x9a, 0x17, 0x1b, 0x50, 0x48, 0xba, 0xf1, 0x37, 0x5c, 0x14, 0x07, 0xa8,
    0xdc, 0x1c, 0x8e, 0xda, 0x53, 0x2d, 0xf5, 0x52, 0x96, 0x77, 0x06, 0x88,
    0x4e, 0x5f, 0xe3, 0xd4, 0x67, 0x86, 0x07, 0x74, 0x48, 0xf2, 0x8f, 0x89,
    0xb3, 0x8e, 0xb7, 0x64, 0xdf, 0x7e, 0x0e, 0x3b, 0x40, 0x1a, 0xed, 0x3a,
    0x23, 0x06, 0xfa, 0x2b, 0xed, 0xe3, 0x82, 0xaa, 0xe1, 0x08, 0xe1, 0x6a,
    0xb4, 0x85, 0x69, 0x1e,
};
static const struct drbg_kat_no_reseed kat236_t = {
    10, kat236_entropyin, kat236_nonce, kat236_persstr,
    kat236_addin0, kat236_addin1, kat236_retbits
};
static const struct drbg_kat kat236 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat236_t
};

static const unsigned char kat237_entropyin[] = {
    0x7d, 0x33, 0x40, 0xa8, 0xb1, 0xe1, 0x51, 0x71, 0xee, 0x34, 0x03, 0xcc,
    0x19, 0xd3, 0xfe, 0xa9,
};
static const unsigned char kat237_nonce[] = {
    0xfd, 0xe6, 0x97, 0xac, 0xc7, 0xe7, 0x2b, 0x75,
};
static const unsigned char kat237_persstr[] = {
    0xa1, 0x28, 0x79, 0x71, 0x79, 0xc5, 0xca, 0xd6, 0xa9, 0xb4, 0x76, 0xff,
    0x99, 0xf9, 0x82, 0x8f,
};
static const unsigned char kat237_addin0[] = {
    0x2d, 0xfa, 0x30, 0x85, 0xfa, 0x02, 0x3b, 0x53, 0x61, 0xdb, 0xa7, 0x9d,
    0x40, 0x77, 0x38, 0xbc,
};
static const unsigned char kat237_addin1[] = {
    0xae, 0xf5, 0x3a, 0xf3, 0xbe, 0x89, 0x41, 0xe7, 0xd2, 0x71, 0x3c, 0x71,
    0x28, 0x41, 0xdc, 0x68,
};
static const unsigned char kat237_retbits[] = {
    0xc1, 0xf7, 0x25, 0xb2, 0x90, 0xed, 0x13, 0x10, 0xdd, 0x3f, 0x39, 0xe9,
    0x9c, 0x7a, 0x65, 0xb8, 0x01, 0xb4, 0x74, 0x2f, 0x50, 0x66, 0xb3, 0xc7,
    0x1b, 0xb4, 0x6f, 0x3d, 0xe7, 0x4d, 0xd2, 0xca, 0xae, 0xa4, 0x08, 0x21,
    0x17, 0x4f, 0x67, 0x67, 0x22, 0xdb, 0x38, 0xfe, 0xda, 0x5d, 0x10, 0x96,
    0xc7, 0x9a, 0xd1, 0xea, 0xbe, 0xff, 0x78, 0xde, 0xf0, 0x84, 0x7a, 0x24,
    0x23, 0xbb, 0x46, 0x02,
};
static const struct drbg_kat_no_reseed kat237_t = {
    11, kat237_entropyin, kat237_nonce, kat237_persstr,
    kat237_addin0, kat237_addin1, kat237_retbits
};
static const struct drbg_kat kat237 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat237_t
};

static const unsigned char kat238_entropyin[] = {
    0x05, 0x71, 0xd4, 0x1f, 0x98, 0xb7, 0x1e, 0xad, 0x41, 0x5f, 0xcb, 0x97,
    0xfd, 0x05, 0xa1, 0xf1,
};
static const unsigned char kat238_nonce[] = {
    0x42, 0x16, 0x18, 0x2c, 0x09, 0x4c, 0x4b, 0x14,
};
static const unsigned char kat238_persstr[] = {
    0x6b, 0xed, 0x5c, 0xc7, 0x26, 0x60, 0xf7, 0xfd, 0x4c, 0x32, 0xab, 0xe5,
    0xcb, 0x1a, 0xef, 0xed,
};
static const unsigned char kat238_addin0[] = {
    0xb1, 0x98, 0x97, 0x3f, 0x68, 0x9c, 0x37, 0xeb, 0x63, 0x76, 0xa0, 0x88,
    0xa7, 0xc7, 0x3e, 0x97,
};
static const unsigned char kat238_addin1[] = {
    0xdf, 0xf3, 0xa3, 0xb0, 0x72, 0x43, 0x02, 0xf8, 0x3b, 0x85, 0x4f, 0x2f,
    0xa8, 0xec, 0x34, 0x96,
};
static const unsigned char kat238_retbits[] = {
    0x66, 0x7d, 0xdb, 0x87, 0x07, 0x9b, 0x39, 0x49, 0x03, 0x0f, 0x41, 0xff,
    0x91, 0xb2, 0xec, 0x0a, 0x59, 0x8a, 0xb2, 0xd5, 0x5c, 0x99, 0xd0, 0x17,
    0xe4, 0xfc, 0x79, 0xe9, 0x4f, 0x15, 0xb4, 0x10, 0xcc, 0x29, 0x70, 0xff,
    0x01, 0x74, 0x82, 0xb2, 0x9e, 0xef, 0x2c, 0x25, 0x0a, 0x09, 0x12, 0x8a,
    0x18, 0x60, 0x5e, 0x0f, 0xa1, 0x4a, 0xcc, 0xe3, 0x7b, 0x89, 0xc4, 0x38,
    0xc8, 0x21, 0x31, 0x94,
};
static const struct drbg_kat_no_reseed kat238_t = {
    12, kat238_entropyin, kat238_nonce, kat238_persstr,
    kat238_addin0, kat238_addin1, kat238_retbits
};
static const struct drbg_kat kat238 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat238_t
};

static const unsigned char kat239_entropyin[] = {
    0x8e, 0xe3, 0xed, 0xc1, 0xc6, 0x4e, 0xa7, 0xd6, 0xf0, 0xba, 0xb9, 0x73,
    0x1e, 0x5d, 0x17, 0x62,
};
static const unsigned char kat239_nonce[] = {
    0x92, 0xa1, 0x42, 0x26, 0x08, 0x7a, 0xfa, 0xfb,
};
static const unsigned char kat239_persstr[] = {
    0x2a, 0x24, 0x04, 0xbe, 0xf6, 0xf7, 0x11, 0x34, 0x17, 0xdf, 0x0b, 0x18,
    0x54, 0xd8, 0x47, 0x3d,
};
static const unsigned char kat239_addin0[] = {
    0xa2, 0x78, 0xe8, 0xa5, 0x0b, 0xcc, 0xd6, 0x12, 0x97, 0x4d, 0x02, 0x6e,
    0xde, 0xf7, 0x50, 0x1c,
};
static const unsigned char kat239_addin1[] = {
    0xe7, 0x48, 0xf5, 0xe2, 0xf9, 0x8b, 0x48, 0x8e, 0x74, 0xb0, 0xbb, 0xd5,
    0x59, 0xb5, 0xba, 0xbc,
};
static const unsigned char kat239_retbits[] = {
    0x81, 0xec, 0xf8, 0xb7, 0xb0, 0x7c, 0x64, 0xce, 0xd1, 0xa4, 0x9b, 0x21,
    0x33, 0x54, 0x99, 0x5a, 0x7b, 0xf1, 0x68, 0xd3, 0x8a, 0x83, 0x0b, 0x15,
    0x70, 0x41, 0x4f, 0x5a, 0x20, 0x15, 0x49, 0x9e, 0x09, 0xaf, 0xf3, 0xbf,
    0x58, 0x37, 0x19, 0x88, 0x6f, 0xe9, 0x9d, 0x00, 0xe6, 0xd9, 0xeb, 0x9a,
    0x4f, 0xde, 0x29, 0xd8, 0xd9, 0x78, 0x8d, 0xd6, 0xff, 0xe2, 0x09, 0x83,
    0x9a, 0x73, 0x9c, 0xd2,
};
static const struct drbg_kat_no_reseed kat239_t = {
    13, kat239_entropyin, kat239_nonce, kat239_persstr,
    kat239_addin0, kat239_addin1, kat239_retbits
};
static const struct drbg_kat kat239 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat239_t
};

static const unsigned char kat240_entropyin[] = {
    0x06, 0x18, 0xc2, 0x88, 0x49, 0x14, 0x71, 0x22, 0x26, 0xad, 0x4e, 0x95,
    0x80, 0xc0, 0x95, 0x23,
};
static const unsigned char kat240_nonce[] = {
    0x6e, 0xbc, 0x4d, 0x5d, 0xb4, 0x76, 0x7f, 0x5f,
};
static const unsigned char kat240_persstr[] = {
    0x7a, 0x82, 0x50, 0xa3, 0xbc, 0xbd, 0x81, 0x54, 0xc5, 0x05, 0x8b, 0x55,
    0x1c, 0xf2, 0x20, 0x36,
};
static const unsigned char kat240_addin0[] = {
    0x30, 0x26, 0x05, 0x32, 0xb5, 0x6c, 0x0f, 0x8e, 0xde, 0x9b, 0x72, 0x5c,
    0x32, 0x20, 0x8f, 0x31,
};
static const unsigned char kat240_addin1[] = {
    0x1e, 0x90, 0xdc, 0x4f, 0x67, 0xc1, 0xcb, 0x2d, 0xa1, 0x2d, 0x10, 0xa0,
    0xb0, 0xfb, 0xb6, 0x26,
};
static const unsigned char kat240_retbits[] = {
    0x49, 0x93, 0x57, 0x58, 0x80, 0xd0, 0x01, 0x45, 0xc1, 0xf9, 0x67, 0xdb,
    0x25, 0xcc, 0xc8, 0xf3, 0x4c, 0xf0, 0xc6, 0x2a, 0xcc, 0xcb, 0xcf, 0xa0,
    0xe7, 0x2c, 0x89, 0xea, 0xcf, 0x9f, 0xbe, 0x56, 0x14, 0x50, 0x74, 0xe4,
    0xd4, 0x7f, 0x2b, 0x76, 0x74, 0xf4, 0xb0, 0xe0, 0x7e, 0xe3, 0xb5, 0x10,
    0x4b, 0xf7, 0xcc, 0xfe, 0xd8, 0x00, 0x1d, 0x67, 0x11, 0xc6, 0x6f, 0xda,
    0x94, 0x44, 0xfb, 0x8a,
};
static const struct drbg_kat_no_reseed kat240_t = {
    14, kat240_entropyin, kat240_nonce, kat240_persstr,
    kat240_addin0, kat240_addin1, kat240_retbits
};
static const struct drbg_kat kat240 = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat240_t
};

static const unsigned char kat241_entropyin[] = {
    0xc3, 0x5c, 0x2f, 0xa2, 0xa8, 0x9d, 0x52, 0xa1, 0x1f, 0xa3, 0x2a, 0xa9,
    0x6c, 0x95, 0xb8, 0xf1, 0xc9, 0xa8, 0xf9, 0xcb, 0x24, 0x5a, 0x8b, 0x40,
};
static const unsigned char kat241_nonce[] = {
    0xf3, 0xa6, 0xe5, 0xa7, 0xfb, 0xd9, 0xd3, 0xc6, 0x8e, 0x27, 0x7b, 0xa9,
    0xac, 0x9b, 0xbb, 0x00,
};
static const unsigned char kat241_persstr[] = {0};
static const unsigned char kat241_addin0[] = {0};
static const unsigned char kat241_addin1[] = {0};
static const unsigned char kat241_retbits[] = {
    0x8c, 0x2e, 0x72, 0xab, 0xfd, 0x9b, 0xb8, 0x28, 0x4d, 0xb7, 0x9e, 0x17,
    0xa4, 0x3a, 0x31, 0x46, 0xcd, 0x76, 0x94, 0xe3, 0x52, 0x49, 0xfc, 0x33,
    0x83, 0x91, 0x4a, 0x71, 0x17, 0xf4, 0x13, 0x68, 0xe6, 0xd4, 0xf1, 0x48,
    0xff, 0x49, 0xbf, 0x29, 0x07, 0x6b, 0x50, 0x15, 0xc5, 0x9f, 0x45, 0x79,
    0x45, 0x66, 0x2e, 0x3d, 0x35, 0x03, 0x84, 0x3f, 0x4a, 0xa5, 0xa3, 0xdf,
    0x9a, 0x9d, 0xf1, 0x0d,
};
static const struct drbg_kat_no_reseed kat241_t = {
    0, kat241_entropyin, kat241_nonce, kat241_persstr,
    kat241_addin0, kat241_addin1, kat241_retbits
};
static const struct drbg_kat kat241 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat241_t
};

static const unsigned char kat242_entropyin[] = {
    0xc1, 0x5f, 0x9f, 0xc5, 0x74, 0x1f, 0x2c, 0xac, 0xe0, 0xb5, 0x8d, 0x72,
    0x49, 0xbd, 0x03, 0x77, 0xbd, 0x57, 0x08, 0xe3, 0x65, 0x88, 0x4b, 0x59,
};
static const unsigned char kat242_nonce[] = {
    0xc3, 0x34, 0x3e, 0x3a, 0x11, 0xb2, 0xdc, 0x15, 0x26, 0x1c, 0x51, 0x75,
    0x1f, 0x51, 0x3b, 0x60,
};
static const unsigned char kat242_persstr[] = {0};
static const unsigned char kat242_addin0[] = {0};
static const unsigned char kat242_addin1[] = {0};
static const unsigned char kat242_retbits[] = {
    0x5c, 0x6c, 0xed, 0x80, 0x50, 0xbc, 0x5a, 0xde, 0x3f, 0x9a, 0xca, 0xcf,
    0x23, 0x89, 0x9f, 0x98, 0xf5, 0xe0, 0x45, 0xa7, 0x68, 0xba, 0x53, 0x8f,
    0xc3, 0x71, 0x74, 0x74, 0x62, 0xeb, 0x9b, 0x84, 0x82, 0x8c, 0x9c, 0xe8,
    0x8e, 0x41, 0x99, 0x05, 0x23, 0x59, 0xb1, 0x58, 0x33, 0x66, 0x89, 0x44,
    0xd6, 0x18, 0x76, 0x7d, 0x0c, 0x6c, 0xfc, 0x24, 0x11, 0xf8, 0x2e, 0x04,
    0x12, 0x06, 0x7a, 0xf6,
};
static const struct drbg_kat_no_reseed kat242_t = {
    1, kat242_entropyin, kat242_nonce, kat242_persstr,
    kat242_addin0, kat242_addin1, kat242_retbits
};
static const struct drbg_kat kat242 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat242_t
};

static const unsigned char kat243_entropyin[] = {
    0x34, 0xdb, 0xd5, 0xcf, 0xe7, 0x9c, 0xac, 0xa4, 0x25, 0x8c, 0xd1, 0xae,
    0x24, 0xb5, 0xaf, 0xe2, 0xd3, 0x7c, 0xb4, 0x69, 0x5d, 0x65, 0x5a, 0x7b,
};
static const unsigned char kat243_nonce[] = {
    0x55, 0xc2, 0x8f, 0xb4, 0x4e, 0x58, 0x49, 0xb5, 0x7a, 0x77, 0x87, 0x35,
    0x13, 0xc6, 0x57, 0x05,
};
static const unsigned char kat243_persstr[] = {0};
static const unsigned char kat243_addin0[] = {0};
static const unsigned char kat243_addin1[] = {0};
static const unsigned char kat243_retbits[] = {
    0x9d, 0x47, 0x1d, 0x2c, 0xb8, 0xe4, 0x20, 0x02, 0xce, 0x24, 0xe0, 0xc6,
    0xbe, 0x84, 0xc8, 0x99, 0x37, 0x59, 0x5e, 0x61, 0xfa, 0xf0, 0x37, 0x1c,
    0xb7, 0xe8, 0x28, 0x5d, 0x76, 0xbe, 0x6b, 0x34, 0xef, 0x1d, 0xb5, 0xf7,
    0xd2, 0xeb, 0xad, 0x4b, 0xf8, 0xff, 0x77, 0x65, 0x13, 0x89, 0xef, 0xcf,
    0x13, 0xba, 0xe7, 0x2b, 0xd8, 0xa1, 0x3c, 0x8e, 0x9f, 0xc6, 0x18, 0x55,
    0x58, 0x1b, 0x8e, 0xa9,
};
static const struct drbg_kat_no_reseed kat243_t = {
    2, kat243_entropyin, kat243_nonce, kat243_persstr,
    kat243_addin0, kat243_addin1, kat243_retbits
};
static const struct drbg_kat kat243 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat243_t
};

static const unsigned char kat244_entropyin[] = {
    0x64, 0x3e, 0x95, 0xbf, 0xea, 0xca, 0xfc, 0x7d, 0x72, 0x1b, 0x48, 0xb9,
    0xf1, 0x5a, 0x80, 0xce, 0x8a, 0xd7, 0x07, 0x7f, 0x30, 0x1f, 0x16, 0xc5,
};
static const unsigned char kat244_nonce[] = {
    0x2d, 0x90, 0x8d, 0xd6, 0x95, 0x2f, 0x86, 0x2a, 0xda, 0x90, 0x2b, 0xc1,
    0xa1, 0x4a, 0x61, 0x29,
};
static const unsigned char kat244_persstr[] = {0};
static const unsigned char kat244_addin0[] = {0};
static const unsigned char kat244_addin1[] = {0};
static const unsigned char kat244_retbits[] = {
    0xd6, 0xee, 0x5e, 0x37, 0x96, 0x34, 0x9d, 0x47, 0xc2, 0xed, 0xe3, 0x2c,
    0x93, 0x8f, 0x2c, 0x06, 0xef, 0x8a, 0x65, 0x11, 0xb4, 0xe9, 0xbb, 0xbe,
    0x74, 0xeb, 0x71, 0xbc, 0x3c, 0x47, 0x70, 0xd9, 0x0c, 0x70, 0xcd, 0xcd,
    0xff, 0x6b, 0x3c, 0x9f, 0xbf, 0x4f, 0xa6, 0xbc, 0x38, 0x0a, 0xc2, 0xf2,
    0xe1, 0x61, 0x96, 0xb4, 0xc4, 0xc8, 0xa6, 0x44, 0x8f, 0x04, 0x0d, 0x11,
    0xd3, 0x6b, 0x42, 0x14,
};
static const struct drbg_kat_no_reseed kat244_t = {
    3, kat244_entropyin, kat244_nonce, kat244_persstr,
    kat244_addin0, kat244_addin1, kat244_retbits
};
static const struct drbg_kat kat244 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat244_t
};

static const unsigned char kat245_entropyin[] = {
    0x0f, 0x47, 0x7a, 0x59, 0x14, 0x90, 0xa0, 0xac, 0xcf, 0x82, 0x9c, 0x67,
    0x3f, 0x15, 0x06, 0xda, 0xca, 0x70, 0xdf, 0x96, 0x58, 0x2c, 0x3d, 0x3a,
};
static const unsigned char kat245_nonce[] = {
    0x1a, 0x8d, 0xd4, 0x23, 0x3b, 0xed, 0xe3, 0xf2, 0x83, 0x86, 0x34, 0xc0,
    0x95, 0x44, 0xab, 0x6e,
};
static const unsigned char kat245_persstr[] = {0};
static const unsigned char kat245_addin0[] = {0};
static const unsigned char kat245_addin1[] = {0};
static const unsigned char kat245_retbits[] = {
    0x43, 0x16, 0x75, 0x73, 0x86, 0x7d, 0x53, 0xca, 0x92, 0xe2, 0x96, 0x86,
    0xd9, 0xfd, 0x99, 0x3c, 0x69, 0x80, 0x5a, 0x63, 0x11, 0xe3, 0x3e, 0xb9,
    0x63, 0x30, 0xc5, 0x37, 0xdc, 0xbd, 0xea, 0xb1, 0x1b, 0x0b, 0xa1, 0x5e,
    0xe5, 0xb2, 0x28, 0x7f, 0x24, 0x85, 0xe2, 0xe3, 0xa6, 0xe0, 0xed, 0x62,
    0xd2, 0xeb, 0x81, 0x08, 0xc1, 0xfc, 0x73, 0x90, 0x8d, 0xfd, 0x4b, 0xc7,
    0xb7, 0x1b, 0x16, 0x6f,
};
static const struct drbg_kat_no_reseed kat245_t = {
    4, kat245_entropyin, kat245_nonce, kat245_persstr,
    kat245_addin0, kat245_addin1, kat245_retbits
};
static const struct drbg_kat kat245 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat245_t
};

static const unsigned char kat246_entropyin[] = {
    0x00, 0xa6, 0xed, 0x77, 0x83, 0x11, 0x52, 0x81, 0x09, 0xe6, 0xfb, 0x5c,
    0xbd, 0x60, 0xd1, 0xa7, 0x6c, 0x4b, 0xd5, 0xd0, 0x0d, 0xd1, 0x03, 0xa4,
};
static const unsigned char kat246_nonce[] = {
    0x40, 0x89, 0xf2, 0xf0, 0xb7, 0x4c, 0x18, 0x4a, 0xc9, 0xaf, 0xc9, 0x56,
    0x47, 0xcd, 0xe8, 0x4b,
};
static const unsigned char kat246_persstr[] = {0};
static const unsigned char kat246_addin0[] = {0};
static const unsigned char kat246_addin1[] = {0};
static const unsigned char kat246_retbits[] = {
    0xb9, 0x08, 0xd9, 0x3a, 0xb2, 0xfd, 0xeb, 0x1a, 0x54, 0x7b, 0x6d, 0x15,
    0xeb, 0x95, 0x0e, 0xdd, 0xce, 0x25, 0x95, 0x6d, 0x8f, 0x5e, 0xe9, 0xa4,
    0xb1, 0xd5, 0x08, 0xc0, 0x82, 0x77, 0xdd, 0x7a, 0x1a, 0x10, 0x60, 0xfc,
    0xec, 0xaf, 0x93, 0xeb, 0x7a, 0x47, 0x2a, 0x36, 0x3e, 0x97, 0xd7, 0xe6,
    0x3c, 0xde, 0xc7, 0xf9, 0x4a, 0x5d, 0xc2, 0x6d, 0xb7, 0xb2, 0xb7, 0xf1,
    0x2c, 0xd8, 0xf3, 0xee,
};
static const struct drbg_kat_no_reseed kat246_t = {
    5, kat246_entropyin, kat246_nonce, kat246_persstr,
    kat246_addin0, kat246_addin1, kat246_retbits
};
static const struct drbg_kat kat246 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat246_t
};

static const unsigned char kat247_entropyin[] = {
    0x5f, 0x34, 0x55, 0x2a, 0x45, 0x5a, 0x02, 0x9b, 0x4b, 0xac, 0xf1, 0xb8,
    0x0c, 0xb1, 0xcf, 0xb4, 0x12, 0x56, 0x5f, 0xb7, 0x73, 0x4f, 0x7e, 0x0a,
};
static const unsigned char kat247_nonce[] = {
    0xbd, 0x55, 0xae, 0xf0, 0x6d, 0xb6, 0xf2, 0x0d, 0xf2, 0x71, 0xcf, 0x60,
    0x59, 0x4c, 0x22, 0xac,
};
static const unsigned char kat247_persstr[] = {0};
static const unsigned char kat247_addin0[] = {0};
static const unsigned char kat247_addin1[] = {0};
static const unsigned char kat247_retbits[] = {
    0xd4, 0xb1, 0x00, 0xef, 0x27, 0xd7, 0x9a, 0x5d, 0x10, 0x03, 0x1d, 0xb0,
    0xcc, 0xc7, 0x2b, 0xa8, 0x9c, 0x68, 0x19, 0x77, 0x6c, 0x19, 0xf0, 0x2f,
    0x9a, 0xfe, 0x4a, 0x44, 0x05, 0xc5, 0x84, 0xa0, 0x66, 0x21, 0xf1, 0xbb,
    0x5b, 0xa2, 0x9a, 0xf5, 0xea, 0x69, 0x63, 0x83, 0xa6, 0x7a, 0x9e, 0xc6,
    0x52, 0xbe, 0xe7, 0x59, 0x28, 0x11, 0x3d, 0xb2, 0x8f, 0x39, 0x40, 0x4f,
    0xf0, 0xba, 0x37, 0x38,
};
static const struct drbg_kat_no_reseed kat247_t = {
    6, kat247_entropyin, kat247_nonce, kat247_persstr,
    kat247_addin0, kat247_addin1, kat247_retbits
};
static const struct drbg_kat kat247 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat247_t
};

static const unsigned char kat248_entropyin[] = {
    0x6d, 0x7c, 0xb2, 0x32, 0xa3, 0x20, 0x3c, 0xa6, 0x95, 0x5c, 0x1f, 0x1e,
    0xdd, 0xd6, 0xb8, 0xe0, 0x9d, 0x82, 0x0b, 0xf6, 0xc0, 0x5f, 0xfb, 0x6e,
};
static const unsigned char kat248_nonce[] = {
    0xc0, 0x4f, 0x16, 0x27, 0xed, 0x1d, 0x7b, 0x31, 0x0a, 0xa4, 0x7e, 0xcc,
    0x79, 0x83, 0x83, 0x6a,
};
static const unsigned char kat248_persstr[] = {0};
static const unsigned char kat248_addin0[] = {0};
static const unsigned char kat248_addin1[] = {0};
static const unsigned char kat248_retbits[] = {
    0x63, 0xd6, 0x96, 0x3e, 0xff, 0x3d, 0x50, 0xfc, 0x0c, 0x21, 0xab, 0xa6,
    0x1d, 0x63, 0x65, 0x74, 0x2c, 0x4b, 0xce, 0xa7, 0xbf, 0x76, 0xc9, 0x5d,
    0x64, 0x5e, 0x88, 0xcb, 0x13, 0x92, 0x53, 0xd5, 0x63, 0x50, 0x3e, 0xc3,
    0x99, 0x9e, 0xe7, 0x72, 0xc8, 0x97, 0xc9, 0xd3, 0x66, 0x3f, 0x64, 0x5a,
    0xb3, 0x04, 0x1d, 0xcf, 0x3d, 0x9e, 0xe8, 0xf4, 0xc9, 0x2e, 0x89, 0x5b,
    0x76, 0x44, 0x41, 0x4d,
};
static const struct drbg_kat_no_reseed kat248_t = {
    7, kat248_entropyin, kat248_nonce, kat248_persstr,
    kat248_addin0, kat248_addin1, kat248_retbits
};
static const struct drbg_kat kat248 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat248_t
};

static const unsigned char kat249_entropyin[] = {
    0x48, 0xbf, 0xce, 0xf7, 0xdb, 0xe6, 0x8d, 0x5e, 0x09, 0xdf, 0x38, 0xe9,
    0x7f, 0x89, 0x77, 0xe2, 0xa1, 0x50, 0x14, 0xbf, 0x75, 0x1f, 0x9e, 0x27,
};
static const unsigned char kat249_nonce[] = {
    0xee, 0xa0, 0x3c, 0x6d, 0x77, 0x80, 0x2a, 0xa2, 0xc5, 0xd5, 0xe1, 0x29,
    0x36, 0x31, 0xe3, 0x9b,
};
static const unsigned char kat249_persstr[] = {0};
static const unsigned char kat249_addin0[] = {0};
static const unsigned char kat249_addin1[] = {0};
static const unsigned char kat249_retbits[] = {
    0xec, 0xdd, 0x8e, 0x7c, 0x78, 0x81, 0x94, 0x69, 0x82, 0xff, 0x3a, 0x11,
    0xa4, 0x3b, 0xd0, 0x20, 0xe9, 0x70, 0xa3, 0xb8, 0x90, 0x13, 0x87, 0xa1,
    0xcd, 0x3e, 0xb6, 0x3d, 0xf3, 0xfb, 0x51, 0x06, 0x60, 0xbc, 0x82, 0xa5,
    0xe0, 0x29, 0xbe, 0xee, 0x26, 0xb4, 0xcc, 0x5d, 0x89, 0x1f, 0xaf, 0xcf,
    0x5e, 0x42, 0xb6, 0xf4, 0xdc, 0x41, 0x49, 0x16, 0xfc, 0xd8, 0xb0, 0x5e,
    0xd3, 0xf4, 0xa6, 0x85,
};
static const struct drbg_kat_no_reseed kat249_t = {
    8, kat249_entropyin, kat249_nonce, kat249_persstr,
    kat249_addin0, kat249_addin1, kat249_retbits
};
static const struct drbg_kat kat249 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat249_t
};

static const unsigned char kat250_entropyin[] = {
    0x5b, 0xe9, 0x61, 0x75, 0x4f, 0xe9, 0xe9, 0x72, 0x06, 0x35, 0x87, 0x5f,
    0xeb, 0x2f, 0x37, 0xc7, 0x4f, 0xb9, 0x62, 0x0e, 0x47, 0xb8, 0x81, 0x41,
};
static const unsigned char kat250_nonce[] = {
    0x23, 0xd3, 0x5a, 0xac, 0xee, 0xfb, 0x41, 0x2a, 0x6f, 0xbd, 0x18, 0x7f,
    0x57, 0x87, 0xbc, 0x53,
};
static const unsigned char kat250_persstr[] = {0};
static const unsigned char kat250_addin0[] = {0};
static const unsigned char kat250_addin1[] = {0};
static const unsigned char kat250_retbits[] = {
    0x8a, 0x8e, 0x0e, 0x44, 0x3b, 0x8e, 0x94, 0x4d, 0x58, 0xc8, 0x40, 0x73,
    0x92, 0x29, 0x6a, 0x2c, 0x7f, 0xc4, 0xeb, 0x04, 0xfc, 0xb7, 0x2c, 0xc7,
    0xea, 0x99, 0x51, 0x35, 0x59, 0xc0, 0x91, 0x76, 0x61, 0xdb, 0x2e, 0xda,
    0x20, 0x9f, 0xbb, 0xed, 0x40, 0x00, 0x89, 0xe3, 0xfc, 0x48, 0xff, 0x69,
    0x90, 0xd9, 0x7c, 0x04, 0x1b, 0xb5, 0x75, 0x27, 0x92, 0xd1, 0x30, 0xa1,
    0x52, 0x57, 0xca, 0x44,
};
static const struct drbg_kat_no_reseed kat250_t = {
    9, kat250_entropyin, kat250_nonce, kat250_persstr,
    kat250_addin0, kat250_addin1, kat250_retbits
};
static const struct drbg_kat kat250 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat250_t
};

static const unsigned char kat251_entropyin[] = {
    0x2a, 0x25, 0x35, 0xa5, 0xaf, 0xc9, 0xb1, 0x71, 0x75, 0x42, 0x92, 0x8a,
    0x3d, 0x35, 0xe9, 0x5a, 0x7f, 0x63, 0xb8, 0x77, 0xa8, 0x38, 0x8a, 0x09,
};
static const unsigned char kat251_nonce[] = {
    0xab, 0x21, 0x7b, 0x02, 0x9a, 0x33, 0xf0, 0x56, 0x35, 0x30, 0xd3, 0x7f,
    0x4f, 0xbf, 0xf1, 0x0b,
};
static const unsigned char kat251_persstr[] = {0};
static const unsigned char kat251_addin0[] = {0};
static const unsigned char kat251_addin1[] = {0};
static const unsigned char kat251_retbits[] = {
    0x31, 0xa2, 0x10, 0x65, 0x55, 0x29, 0xfb, 0x46, 0xeb, 0x4e, 0x79, 0x97,
    0x10, 0xc6, 0x31, 0x92, 0xe3, 0x65, 0x6c, 0xb1, 0xea, 0x24, 0x04, 0x08,
    0x52, 0x7d, 0xf7, 0xfe, 0xf8, 0x2b, 0x30, 0x5c, 0x5f, 0xa9, 0x8a, 0x1d,
    0x82, 0x4f, 0x78, 0x30, 0xd8, 0x4a, 0x53, 0x64, 0x3c, 0xfb, 0x52, 0x51,
    0x76, 0x63, 0xd5, 0x16, 0x11, 0x5d, 0x45, 0x27, 0x2f, 0x59, 0x72, 0x24,
    0x0f, 0xe8, 0x48, 0x27,
};
static const struct drbg_kat_no_reseed kat251_t = {
    10, kat251_entropyin, kat251_nonce, kat251_persstr,
    kat251_addin0, kat251_addin1, kat251_retbits
};
static const struct drbg_kat kat251 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat251_t
};

static const unsigned char kat252_entropyin[] = {
    0xff, 0xd8, 0xb1, 0x50, 0xb2, 0x87, 0xb3, 0xed, 0x9d, 0x01, 0x28, 0xdc,
    0xc8, 0xcc, 0x5d, 0xeb, 0x90, 0xbd, 0xa9, 0x8a, 0xca, 0x79, 0x14, 0x83,
};
static const unsigned char kat252_nonce[] = {
    0xbc, 0x7b, 0x6a, 0x7f, 0xb0, 0x59, 0x2f, 0xda, 0x4a, 0xbc, 0x43, 0xee,
    0xee, 0x42, 0xb1, 0xf6,
};
static const unsigned char kat252_persstr[] = {0};
static const unsigned char kat252_addin0[] = {0};
static const unsigned char kat252_addin1[] = {0};
static const unsigned char kat252_retbits[] = {
    0x3f, 0xdb, 0x4f, 0xf8, 0x3c, 0x36, 0x2b, 0x76, 0xa8, 0x3b, 0x00, 0xa8,
    0x90, 0x5f, 0x2c, 0x05, 0x8b, 0xaf, 0xe8, 0xeb, 0xdf, 0xf8, 0xeb, 0x2e,
    0x5f, 0x53, 0xcb, 0x9c, 0xa5, 0xc8, 0x2e, 0x32, 0x17, 0xc2, 0xc6, 0x03,
    0x26, 0xb0, 0xbb, 0xa6, 0xc9, 0x1c, 0x44, 0x0a, 0x48, 0x52, 0xc7, 0x2c,
    0xd7, 0x33, 0x45, 0xc5, 0xe0, 0xd4, 0xcc, 0x2e, 0x15, 0x84, 0x01, 0x04,
    0x5a, 0x77, 0xe0, 0xf9,
};
static const struct drbg_kat_no_reseed kat252_t = {
    11, kat252_entropyin, kat252_nonce, kat252_persstr,
    kat252_addin0, kat252_addin1, kat252_retbits
};
static const struct drbg_kat kat252 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat252_t
};

static const unsigned char kat253_entropyin[] = {
    0xcb, 0x33, 0xfc, 0x14, 0xf2, 0xb6, 0x94, 0x01, 0xb7, 0x8b, 0xbd, 0x78,
    0x1c, 0xbf, 0xe6, 0x58, 0x5f, 0x1e, 0x73, 0x64, 0x59, 0x00, 0x0c, 0x26,
};
static const unsigned char kat253_nonce[] = {
    0x5a, 0x91, 0x4f, 0x81, 0xd6, 0x5b, 0x19, 0x7a, 0x5d, 0x89, 0x24, 0xfa,
    0x88, 0xfa, 0x9b, 0xa3,
};
static const unsigned char kat253_persstr[] = {0};
static const unsigned char kat253_addin0[] = {0};
static const unsigned char kat253_addin1[] = {0};
static const unsigned char kat253_retbits[] = {
    0x6c, 0x63, 0x20, 0x91, 0x88, 0x6e, 0x24, 0x26, 0x5d, 0x12, 0x3b, 0x5c,
    0x7f, 0xca, 0x06, 0x3d, 0x70, 0xdf, 0xea, 0x32, 0xc7, 0x78, 0x0a, 0xfa,
    0xab, 0x89, 0x5c, 0x10, 0xb5, 0x12, 0x62, 0xbc, 0x4d, 0x36, 0x3d, 0x21,
    0x65, 0x99, 0x83, 0x67, 0xc9, 0xa8, 0xa5, 0x16, 0x47, 0x6b, 0xc7, 0x63,
    0x99, 0x02, 0x41, 0xba, 0xf9, 0x2d, 0x6b, 0x87, 0xe4, 0xde, 0x62, 0x2d,
    0x40, 0x19, 0xdb, 0x6c,
};
static const struct drbg_kat_no_reseed kat253_t = {
    12, kat253_entropyin, kat253_nonce, kat253_persstr,
    kat253_addin0, kat253_addin1, kat253_retbits
};
static const struct drbg_kat kat253 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat253_t
};

static const unsigned char kat254_entropyin[] = {
    0xc4, 0x57, 0xe7, 0xd1, 0xc6, 0x9b, 0x1a, 0x53, 0x00, 0xb4, 0x75, 0xef,
    0x80, 0x44, 0x1c, 0x40, 0x80, 0x0c, 0xd3, 0x76, 0x4e, 0x41, 0x4c, 0xe1,
};
static const unsigned char kat254_nonce[] = {
    0x39, 0x33, 0xe4, 0x98, 0xa0, 0xf2, 0x0d, 0x3f, 0xcf, 0x05, 0xa3, 0xb8,
    0x2b, 0x67, 0xac, 0x6d,
};
static const unsigned char kat254_persstr[] = {0};
static const unsigned char kat254_addin0[] = {0};
static const unsigned char kat254_addin1[] = {0};
static const unsigned char kat254_retbits[] = {
    0x26, 0x89, 0x41, 0x9f, 0xa0, 0xf9, 0xb3, 0xa4, 0xc5, 0xe9, 0x6a, 0x71,
    0xf6, 0x76, 0xef, 0x7f, 0x76, 0x76, 0x7b, 0x51, 0xb3, 0xcd, 0x5c, 0xe8,
    0x37, 0xe9, 0xd8, 0x28, 0x6f, 0xa8, 0x37, 0xa5, 0xa3, 0x45, 0x66, 0x95,
    0xe3, 0xe9, 0xca, 0x93, 0x96, 0xe8, 0xd5, 0xb9, 0x47, 0x8a, 0x1e, 0x8c,
    0xdd, 0x5c, 0x31, 0xfb, 0xaa, 0x5e, 0x07, 0x43, 0xd2, 0x77, 0xfd, 0xf4,
    0xaf, 0xc7, 0x95, 0x9a,
};
static const struct drbg_kat_no_reseed kat254_t = {
    13, kat254_entropyin, kat254_nonce, kat254_persstr,
    kat254_addin0, kat254_addin1, kat254_retbits
};
static const struct drbg_kat kat254 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat254_t
};

static const unsigned char kat255_entropyin[] = {
    0x49, 0xbc, 0x4a, 0xa4, 0xca, 0xba, 0x36, 0x42, 0x42, 0xdf, 0x51, 0x84,
    0x90, 0x9c, 0xe5, 0x88, 0xf2, 0x4a, 0x2d, 0xe3, 0x40, 0xe3, 0x30, 0x82,
};
static const unsigned char kat255_nonce[] = {
    0x51, 0x01, 0xf0, 0x0a, 0xc1, 0x89, 0x9c, 0x9b, 0xa4, 0x5e, 0x92, 0x3e,
    0x07, 0x16, 0x64, 0x07,
};
static const unsigned char kat255_persstr[] = {0};
static const unsigned char kat255_addin0[] = {0};
static const unsigned char kat255_addin1[] = {0};
static const unsigned char kat255_retbits[] = {
    0x9b, 0x71, 0xe0, 0xa0, 0x9c, 0x39, 0x3e, 0xb0, 0xf0, 0x34, 0xa1, 0x2d,
    0x3a, 0xd1, 0x32, 0x0a, 0xc1, 0xca, 0x42, 0x2a, 0x82, 0x96, 0x7c, 0xc5,
    0x4d, 0x14, 0xed, 0x0e, 0x36, 0x42, 0x9a, 0xb2, 0xa0, 0xe8, 0xc6, 0x7a,
    0x5c, 0x7f, 0x60, 0xe5, 0x37, 0xe4, 0x10, 0xeb, 0x5f, 0xf0, 0x75, 0xb7,
    0xc6, 0x7a, 0xc4, 0x80, 0x52, 0x00, 0xa5, 0xab, 0x2f, 0xb6, 0x29, 0xb4,
    0x84, 0x60, 0xcc, 0xe4,
};
static const struct drbg_kat_no_reseed kat255_t = {
    14, kat255_entropyin, kat255_nonce, kat255_persstr,
    kat255_addin0, kat255_addin1, kat255_retbits
};
static const struct drbg_kat kat255 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat255_t
};

static const unsigned char kat256_entropyin[] = {
    0x89, 0xd3, 0x2f, 0x3d, 0xe1, 0x67, 0xde, 0xbe, 0xdf, 0xc1, 0x43, 0xe4,
    0xda, 0x78, 0x9a, 0x5f, 0x83, 0x31, 0x2a, 0x9d, 0x42, 0x0c, 0x29, 0xd4,
};
static const unsigned char kat256_nonce[] = {
    0x9a, 0xe3, 0xba, 0xb2, 0xd9, 0x65, 0xdb, 0xe9, 0x96, 0xa6, 0xc4, 0xc0,
    0xde, 0x28, 0x05, 0x01,
};
static const unsigned char kat256_persstr[] = {0};
static const unsigned char kat256_addin0[] = {
    0xd5, 0x16, 0x5f, 0xbf, 0x88, 0x50, 0xe2, 0xec, 0xfb, 0x46, 0x87, 0xaf,
    0x79, 0xdc, 0x62, 0x36, 0x1e, 0x75, 0x57, 0x70, 0x70, 0x51, 0xa1, 0x36,
    0x80, 0x47, 0x1e, 0xe1, 0x29, 0xf2, 0x84, 0xf9,
};
static const unsigned char kat256_addin1[] = {
    0x85, 0x22, 0x56, 0xde, 0x34, 0x79, 0xdc, 0x86, 0xa3, 0xb4, 0xd4, 0x04,
    0xc2, 0x64, 0x7b, 0x74, 0xf5, 0xf8, 0xa1, 0xc0, 0x1b, 0x68, 0x1f, 0x1d,
    0x8e, 0xfd, 0xfe, 0xdc, 0x54, 0xc1, 0x0c, 0x07,
};
static const unsigned char kat256_retbits[] = {
    0xb0, 0x39, 0x7e, 0xda, 0x1d, 0xaf, 0x45, 0x44, 0x10, 0x4f, 0x73, 0x04,
    0x01, 0xcb, 0xd1, 0xbe, 0x29, 0x98, 0x9c, 0xd2, 0x87, 0x97, 0xde, 0x2c,
    0x13, 0xcd, 0xbf, 0x18, 0x4f, 0x86, 0xc6, 0x37, 0x8f, 0x8d, 0xfa, 0x39,
    0x4e, 0x08, 0xd9, 0xa7, 0x1d, 0xe1, 0xf1, 0x63, 0x3b, 0x5b, 0x28, 0x36,
    0x3d, 0xf2, 0x16, 0x59, 0xce, 0x58, 0xb1, 0x25, 0x92, 0xe8, 0xee, 0x4f,
    0x55, 0xcb, 0x81, 0xac,
};
static const struct drbg_kat_no_reseed kat256_t = {
    0, kat256_entropyin, kat256_nonce, kat256_persstr,
    kat256_addin0, kat256_addin1, kat256_retbits
};
static const struct drbg_kat kat256 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat256_t
};

static const unsigned char kat257_entropyin[] = {
    0xad, 0x3b, 0x6f, 0xa3, 0xd6, 0x32, 0xa4, 0x2f, 0x8c, 0x58, 0x0b, 0x59,
    0xfa, 0xf8, 0x82, 0xc0, 0x33, 0x8c, 0x5b, 0xb0, 0x44, 0xb8, 0x8f, 0x37,
};
static const unsigned char kat257_nonce[] = {
    0xf9, 0x99, 0x66, 0xdb, 0x52, 0xa2, 0xf7, 0x27, 0x11, 0xb5, 0xaf, 0x10,
    0xef, 0xf4, 0x2f, 0xef,
};
static const unsigned char kat257_persstr[] = {0};
static const unsigned char kat257_addin0[] = {
    0xbe, 0xe5, 0x97, 0x4c, 0x34, 0x30, 0x2a, 0xee, 0xfe, 0x31, 0xc7, 0x32,
    0x59, 0x87, 0xc1, 0x58, 0xdf, 0x80, 0x38, 0xa2, 0x37, 0xef, 0x9e, 0xcb,
    0xb8, 0xfe, 0x74, 0xfe, 0xfb, 0xb3, 0xe2, 0x29,
};
static const unsigned char kat257_addin1[] = {
    0x47, 0xb9, 0xbd, 0x9b, 0xab, 0xc4, 0x15, 0xaf, 0xaa, 0xed, 0xbb, 0x8a,
    0xc9, 0xeb, 0xf0, 0x2e, 0xb5, 0x81, 0xd1, 0xeb, 0x5f, 0xb0, 0xb7, 0xd4,
    0x34, 0x55, 0x84, 0x8d, 0x66, 0xce, 0xb5, 0x3b,
};
static const unsigned char kat257_retbits[] = {
    0x75, 0x92, 0x2a, 0x7b, 0x19, 0x33, 0xf4, 0xbe, 0xc2, 0x75, 0xc1, 0x50,
    0xe2, 0xf9, 0x89, 0x37, 0x14, 0xcb, 0xa1, 0x19, 0xc5, 0xdf, 0x7d, 0xc2,
    0xcf, 0xaa, 0x96, 0x18, 0xf1, 0xeb, 0xa5, 0x4d, 0x68, 0xd5, 0x58, 0xe9,
    0x76, 0xce, 0x29, 0x2e, 0x4b, 0x9f, 0x83, 0x06, 0xf1, 0xd1, 0x75, 0x59,
    0x78, 0x04, 0x12, 0x24, 0x74, 0x80, 0x06, 0xbd, 0x57, 0x12, 0x31, 0x0f,
    0xd0, 0x85, 0xaf, 0xa0,
};
static const struct drbg_kat_no_reseed kat257_t = {
    1, kat257_entropyin, kat257_nonce, kat257_persstr,
    kat257_addin0, kat257_addin1, kat257_retbits
};
static const struct drbg_kat kat257 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat257_t
};

static const unsigned char kat258_entropyin[] = {
    0x16, 0xd3, 0x99, 0xb5, 0x26, 0xa8, 0x37, 0x66, 0xb5, 0xf9, 0x35, 0xe4,
    0xc1, 0x23, 0x41, 0x98, 0x17, 0x24, 0xcd, 0x87, 0x24, 0x25, 0xd8, 0x59,
};
static const unsigned char kat258_nonce[] = {
    0xc0, 0xf3, 0x83, 0x54, 0xae, 0x69, 0x68, 0x35, 0xf9, 0xc8, 0x56, 0x71,
    0x6f, 0x18, 0x6e, 0x4d,
};
static const unsigned char kat258_persstr[] = {0};
static const unsigned char kat258_addin0[] = {
    0x56, 0x94, 0x3f, 0x5e, 0xf7, 0xee, 0xa6, 0x6a, 0xf2, 0xec, 0x60, 0x7f,
    0xcf, 0xa2, 0xbd, 0x76, 0x69, 0x89, 0x9b, 0x6e, 0xce, 0x4c, 0xab, 0xac,
    0x8d, 0x6e, 0x7a, 0x08, 0x68, 0xbe, 0x42, 0x2b,
};
static const unsigned char kat258_addin1[] = {
    0x2f, 0x7e, 0x74, 0x6e, 0x2d, 0x05, 0xcd, 0x1e, 0x52, 0x33, 0x39, 0x92,
    0x6e, 0x3e, 0x96, 0x56, 0x8f, 0x5d, 0x23, 0x1b, 0x80, 0xc2, 0x68, 0xf4,
    0x4b, 0x1d, 0xf9, 0x62, 0x22, 0x9e, 0x5b, 0x27,
};
static const unsigned char kat258_retbits[] = {
    0x0b, 0xd1, 0xc4, 0x1c, 0x30, 0x28, 0x4f, 0xad, 0xa0, 0x9e, 0xb2, 0x44,
    0x7d, 0x56, 0x38, 0xa7, 0xa0, 0xb7, 0x3d, 0x99, 0xc6, 0x51, 0xed, 0xc5,
    0xb8, 0xaa, 0x15, 0xd4, 0xe9, 0x81, 0x40, 0x48, 0xd7, 0x9c, 0x52, 0x38,
    0x2b, 0xaa, 0x20, 0x3d, 0x3f, 0x72, 0x78, 0xf4, 0x86, 0x4d, 0x18, 0x36,
    0x65, 0x66, 0x24, 0x80, 0x06, 0x58, 0x7c, 0xf3, 0xf9, 0x02, 0x71, 0xc4,
    0x8c, 0x49, 0x3d, 0x5d,
};
static const struct drbg_kat_no_reseed kat258_t = {
    2, kat258_entropyin, kat258_nonce, kat258_persstr,
    kat258_addin0, kat258_addin1, kat258_retbits
};
static const struct drbg_kat kat258 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat258_t
};

static const unsigned char kat259_entropyin[] = {
    0x8d, 0x2b, 0xcb, 0x66, 0xbd, 0x17, 0xe8, 0xb2, 0xe3, 0x6a, 0x2a, 0x5f,
    0xd1, 0xc1, 0xa0, 0x86, 0x1f, 0xe3, 0x4e, 0x1c, 0xb8, 0xf2, 0x21, 0x9b,
};
static const unsigned char kat259_nonce[] = {
    0x86, 0x8c, 0x2b, 0xab, 0xe9, 0x69, 0xd7, 0x77, 0x6f, 0x92, 0x92, 0x97,
    0x95, 0xa9, 0x1e, 0xb7,
};
static const unsigned char kat259_persstr[] = {0};
static const unsigned char kat259_addin0[] = {
    0x46, 0x18, 0x36, 0x6b, 0x0d, 0x51, 0xf9, 0x09, 0x5f, 0xd8, 0x67, 0xfa,
    0x7a, 0x40, 0xe7, 0x73, 0x62, 0x94, 0xa9, 0x2f, 0xdc, 0xc5, 0xd0, 0xec,
    0xf2, 0x93, 0x04, 0x25, 0x66, 0x68, 0x34, 0x48,
};
static const unsigned char kat259_addin1[] = {
    0xc4, 0x5c, 0x21, 0x8f, 0xc6, 0x48, 0xde, 0x1b, 0xfb, 0xaa, 0xa6, 0x3b,
    0xe0, 0x27, 0xaf, 0xa0, 0x8c, 0xde, 0x5c, 0x7d, 0x84, 0xa6, 0x29, 0xdb,
    0x9b, 0x54, 0xa2, 0x0c, 0xf6, 0x4c, 0x5a, 0x75,
};
static const unsigned char kat259_retbits[] = {
    0xb8, 0x5b, 0x85, 0xcf, 0xf7, 0x18, 0x51, 0xce, 0x21, 0x1d, 0x4e, 0xb2,
    0xdc, 0x56, 0x9a, 0xc2, 0xd4, 0x3c, 0xcf, 0x52, 0xff, 0x71, 0xcc, 0x00,
    0x90, 0xed, 0x59, 0xc8, 0x7e, 0x78, 0x50, 0x61, 0xb9, 0x9f, 0x35, 0x2e,
    0x39, 0x8f, 0x61, 0xd2, 0x3d, 0xc0, 0xa1, 0x7e, 0x44, 0x76, 0xda, 0x39,
    0xc3, 0x1f, 0x7b, 0x81, 0x73, 0x8c, 0xaa, 0x61, 0x47, 0x53, 0x17, 0x36,
    0x3c, 0x20, 0x5a, 0xba,
};
static const struct drbg_kat_no_reseed kat259_t = {
    3, kat259_entropyin, kat259_nonce, kat259_persstr,
    kat259_addin0, kat259_addin1, kat259_retbits
};
static const struct drbg_kat kat259 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat259_t
};

static const unsigned char kat260_entropyin[] = {
    0xf3, 0xca, 0x9e, 0xe3, 0x24, 0xcb, 0x01, 0x11, 0xd5, 0xec, 0x08, 0x1e,
    0xdf, 0x77, 0xd3, 0xfa, 0x0b, 0x77, 0x71, 0x8a, 0x87, 0xfb, 0xf4, 0xf5,
};
static const unsigned char kat260_nonce[] = {
    0x98, 0xd1, 0x2b, 0xbe, 0x87, 0xe4, 0x76, 0x03, 0x7a, 0x5b, 0x24, 0x47,
    0x35, 0xd2, 0x10, 0x2a,
};
static const unsigned char kat260_persstr[] = {0};
static const unsigned char kat260_addin0[] = {
    0x19, 0x06, 0x9b, 0x8f, 0xb8, 0x14, 0x8a, 0xc0, 0x0c, 0xf6, 0x07, 0xcf,
    0xce, 0x76, 0xa9, 0x54, 0xf3, 0x12, 0x9c, 0x79, 0x75, 0x19, 0x6b, 0x8b,
    0x61, 0x28, 0xa3, 0x00, 0xce, 0x0d, 0xc0, 0xa7,
};
static const unsigned char kat260_addin1[] = {
    0x4f, 0x07, 0x8d, 0xaa, 0xbd, 0x37, 0x76, 0x80, 0xea, 0xe1, 0x82, 0x30,
    0x74, 0xab, 0x0d, 0x40, 0xa6, 0x93, 0xd6, 0x81, 0x21, 0xbb, 0xa4, 0x29,
    0xde, 0xbd, 0xfe, 0x0f, 0x88, 0x37, 0x03, 0xdd,
};
static const unsigned char kat260_retbits[] = {
    0x67, 0x28, 0x47, 0x2e, 0xbc, 0x52, 0xa8, 0x75, 0xb1, 0xa1, 0x63, 0x21,
    0x1c, 0x2b, 0x77, 0x1b, 0xdc, 0xa9, 0x9e, 0x03, 0xf0, 0xde, 0x54, 0xa6,
    0xf4, 0xa1, 0x41, 0xa7, 0x69, 0x1d, 0x6f, 0x96, 0x79, 0xf9, 0x63, 0xc2,
    0xfa, 0x37, 0x7f, 0x0a, 0xe4, 0x71, 0xec, 0xc4, 0x70, 0x76, 0x88, 0x18,
    0xab, 0x4e, 0x62, 0x77, 0x11, 0xa0, 0xf9, 0x88, 0xff, 0x97, 0x3e, 0x4b,
    0xce, 0x50, 0x1b, 0x69,
};
static const struct drbg_kat_no_reseed kat260_t = {
    4, kat260_entropyin, kat260_nonce, kat260_persstr,
    kat260_addin0, kat260_addin1, kat260_retbits
};
static const struct drbg_kat kat260 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat260_t
};

static const unsigned char kat261_entropyin[] = {
    0x54, 0x47, 0x86, 0x8c, 0xd9, 0x25, 0x24, 0x23, 0xf0, 0x3c, 0xa3, 0xbd,
    0x8b, 0x0a, 0x45, 0x43, 0x35, 0xd3, 0x2e, 0xa5, 0x4f, 0xbd, 0x3e, 0xdb,
};
static const unsigned char kat261_nonce[] = {
    0x64, 0xf2, 0xde, 0x76, 0x5d, 0x20, 0x28, 0xa8, 0xba, 0x06, 0xe6, 0xd2,
    0x04, 0xac, 0x4b, 0xc4,
};
static const unsigned char kat261_persstr[] = {0};
static const unsigned char kat261_addin0[] = {
    0x19, 0x1a, 0x39, 0xd6, 0x30, 0xd6, 0x6c, 0xa9, 0x46, 0x72, 0x0a, 0xef,
    0xe8, 0x0c, 0x94, 0xbf, 0xdd, 0x3e, 0x24, 0xdd, 0xa3, 0xe3, 0x7c, 0x41,
    0x4a, 0x93, 0xeb, 0xb7, 0x84, 0xd3, 0xe7, 0xc4,
};
static const unsigned char kat261_addin1[] = {
    0x8b, 0x66, 0x4b, 0xa7, 0x4f, 0xd0, 0xbb, 0x50, 0x06, 0xed, 0xd3, 0x37,
    0xa1, 0xc4, 0xdb, 0xbf, 0xe7, 0x73, 0x90, 0x2e, 0x08, 0x4c, 0x55, 0x8b,
    0xa1, 0x45, 0xe8, 0xf8, 0xca, 0x34, 0x54, 0x87,
};
static const unsigned char kat261_retbits[] = {
    0xce, 0x5f, 0x7a, 0x69, 0x68, 0xf3, 0xcf, 0x5f, 0xb0, 0x1e, 0x62, 0x08,
    0xa1, 0x65, 0x17, 0x20, 0xfe, 0x1e, 0x90, 0x76, 0x4d, 0xea, 0x46, 0x50,
    0x4e, 0x10, 0x6f, 0x13, 0xeb, 0xff, 0xf3, 0xd9, 0x79, 0x07, 0x2b, 0x6f,
    0x54, 0x59, 0x06, 0x9a, 0x77, 0x30, 0x03, 0xe3, 0xec, 0xa4, 0xee, 0x3b,
    0x73, 0x0a, 0xeb, 0xfe, 0x95, 0x7c, 0x0e, 0x68, 0x77, 0x4d, 0xdc, 0xb9,
    0x71, 0xbf, 0x46, 0x03,
};
static const struct drbg_kat_no_reseed kat261_t = {
    5, kat261_entropyin, kat261_nonce, kat261_persstr,
    kat261_addin0, kat261_addin1, kat261_retbits
};
static const struct drbg_kat kat261 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat261_t
};

static const unsigned char kat262_entropyin[] = {
    0xe8, 0xd9, 0x3d, 0x86, 0xc3, 0xe9, 0x8d, 0xe6, 0x4f, 0x40, 0x80, 0xbb,
    0x64, 0x52, 0x05, 0x5c, 0xc7, 0xdb, 0x86, 0xd2, 0x5b, 0xb3, 0xa9, 0x23,
};
static const unsigned char kat262_nonce[] = {
    0xa5, 0x44, 0x7f, 0x91, 0x15, 0x89, 0x1d, 0x48, 0x30, 0x37, 0xad, 0x4d,
    0xd4, 0xc2, 0x2a, 0x8a,
};
static const unsigned char kat262_persstr[] = {0};
static const unsigned char kat262_addin0[] = {
    0x56, 0xc6, 0xce, 0xa0, 0x19, 0x72, 0x73, 0x74, 0x21, 0xa0, 0x7c, 0xb2,
    0x0c, 0x79, 0xb8, 0xf7, 0x6b, 0x32, 0xa9, 0xe9, 0xfb, 0xbb, 0xe4, 0x6d,
    0xd7, 0x1f, 0x02, 0xc7, 0x2d, 0x1e, 0x45, 0x0a,
};
static const unsigned char kat262_addin1[] = {
    0x96, 0x42, 0xc8, 0x21, 0x59, 0xc3, 0x6a, 0x48, 0xa2, 0x52, 0xcc, 0xfe,
    0xe6, 0x1c, 0x5c, 0x1c, 0xed, 0x6d, 0x3a, 0x04, 0x09, 0x9d, 0x98, 0xf9,
    0xdb, 0xe5, 0x34, 0x87, 0x7b, 0x80, 0xa9, 0x8f,
};
static const unsigned char kat262_retbits[] = {
    0x18, 0x7f, 0x40, 0x48, 0x9f, 0x22, 0xf3, 0x55, 0x83, 0xe7, 0x58, 0x21,
    0xd6, 0x9f, 0x36, 0x9d, 0x12, 0x59, 0x35, 0x21, 0x71, 0xc5, 0x69, 0xf2,
    0xd8, 0xe9, 0xa1, 0x50, 0x97, 0x87, 0x6c, 0xe5, 0x26, 0x1b, 0x41, 0x74,
    0x53, 0x04, 0x69, 0x82, 0x9b, 0xa2, 0xc0, 0x01, 0x45, 0xcb, 0xf4, 0xd2,
    0xa1, 0x20, 0x5e, 0x91, 0x1c, 0x2c, 0x32, 0xd3, 0x3b, 0x5d, 0x67, 0x06,
    0x40, 0x97, 0x72, 0xd2,
};
static const struct drbg_kat_no_reseed kat262_t = {
    6, kat262_entropyin, kat262_nonce, kat262_persstr,
    kat262_addin0, kat262_addin1, kat262_retbits
};
static const struct drbg_kat kat262 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat262_t
};

static const unsigned char kat263_entropyin[] = {
    0x59, 0xb9, 0xb4, 0xdf, 0x1b, 0xb4, 0xf7, 0x78, 0xea, 0x1a, 0x1b, 0xfc,
    0x41, 0x25, 0x5c, 0xc3, 0x25, 0xd6, 0xbf, 0x4f, 0xfc, 0x42, 0x8c, 0xe1,
};
static const unsigned char kat263_nonce[] = {
    0xe0, 0xfb, 0x72, 0xdd, 0x71, 0xff, 0x3c, 0x2e, 0x07, 0x2f, 0x27, 0xd6,
    0x57, 0x26, 0x0d, 0xa2,
};
static const unsigned char kat263_persstr[] = {0};
static const unsigned char kat263_addin0[] = {
    0xcc, 0xfb, 0x64, 0xf4, 0x3f, 0x84, 0xf0, 0x3c, 0xd9, 0x8e, 0x1a, 0xa3,
    0x56, 0x6c, 0xa0, 0xc9, 0xb7, 0x43, 0x82, 0x04, 0xc8, 0xe6, 0x5a, 0x9c,
    0xb8, 0x22, 0x7e, 0x20, 0x11, 0xb9, 0xa7, 0x16,
};
static const unsigned char kat263_addin1[] = {
    0x30, 0x64, 0xa3, 0x1a, 0x84, 0xc4, 0xea, 0xa6, 0x98, 0x0e, 0x6d, 0x41,
    0x2c, 0xd4, 0xf4, 0x19, 0x11, 0x4b, 0xf0, 0xe8, 0xb2, 0x27, 0x45, 0x6a,
    0x91, 0xda, 0x0f, 0xc0, 0xf7, 0xdc, 0xb7, 0x5f,
};
static const unsigned char kat263_retbits[] = {
    0x7d, 0x48, 0x83, 0x6e, 0xd6, 0x57, 0x23, 0x1f, 0x62, 0x62, 0x7b, 0xa5,
    0x08, 0x97, 0xca, 0x8f, 0x37, 0x9f, 0x68, 0xd8, 0x6f, 0xab, 0xe2, 0x6c,
    0xaa, 0x7b, 0x44, 0x41, 0xcd, 0x2b, 0x6c, 0x2a, 0xd1, 0xda, 0xf4, 0xea,
    0xbc, 0x2c, 0x34, 0xb3, 0x96, 0x34, 0x00, 0xd2, 0x36, 0x11, 0x74, 0xee,
    0x22, 0x95, 0x2d, 0x6a, 0x28, 0xe8, 0x99, 0x37, 0xb2, 0x31, 0xc9, 0xbc,
    0x22, 0x8a, 0x78, 0xc5,
};
static const struct drbg_kat_no_reseed kat263_t = {
    7, kat263_entropyin, kat263_nonce, kat263_persstr,
    kat263_addin0, kat263_addin1, kat263_retbits
};
static const struct drbg_kat kat263 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat263_t
};

static const unsigned char kat264_entropyin[] = {
    0x0d, 0x03, 0xee, 0x60, 0xbc, 0xa7, 0x23, 0x99, 0x9d, 0x49, 0x58, 0x9e,
    0xb6, 0xe4, 0x04, 0x2f, 0xf9, 0x7b, 0x22, 0x43, 0x3c, 0x81, 0x22, 0x42,
};
static const unsigned char kat264_nonce[] = {
    0x8a, 0x2f, 0x6f, 0xc8, 0xd4, 0xc5, 0x4b, 0xf6, 0x85, 0x2b, 0x90, 0x2b,
    0xcf, 0x2c, 0x41, 0xec,
};
static const unsigned char kat264_persstr[] = {0};
static const unsigned char kat264_addin0[] = {
    0xcb, 0x26, 0x94, 0x41, 0x03, 0x5e, 0x07, 0x11, 0x01, 0xac, 0x8e, 0xfb,
    0x4e, 0xb7, 0xf9, 0xf1, 0xde, 0x21, 0x84, 0x77, 0x1e, 0xc0, 0xea, 0xc6,
    0x2d, 0x06, 0x92, 0xe6, 0x7b, 0xdf, 0x1f, 0xae,
};
static const unsigned char kat264_addin1[] = {
    0x90, 0x5e, 0x3d, 0x9c, 0x18, 0x00, 0xe1, 0x36, 0x6b, 0xc7, 0x49, 0xa6,
    0x0d, 0x21, 0xce, 0x51, 0x8f, 0xde, 0xf2, 0xa3, 0x6f, 0x88, 0x0c, 0x26,
    0xc1, 0x52, 0x8f, 0x12, 0xdf, 0xfb, 0x91, 0x76,
};
static const unsigned char kat264_retbits[] = {
    0xa4, 0xcb, 0x87, 0xde, 0x76, 0xb2, 0xeb, 0x39, 0x59, 0x9f, 0x68, 0x40,
    0x22, 0x93, 0xce, 0xe9, 0xc6, 0x1d, 0xc9, 0xdc, 0x12, 0x57, 0x78, 0x99,
    0x64, 0xa2, 0xac, 0xf0, 0xc3, 0x2f, 0x61, 0xc9, 0x07, 0x29, 0x4b, 0xde,
    0xca, 0x88, 0xb0, 0x5d, 0xaa, 0xe0, 0x75, 0x4d, 0x21, 0xf0, 0xc2, 0xb7,
    0x56, 0x97, 0xd8, 0x61, 0x42, 0x37, 0x4f, 0x96, 0x76, 0xdb, 0x34, 0x75,
    0x33, 0x71, 0xe6, 0x18,
};
static const struct drbg_kat_no_reseed kat264_t = {
    8, kat264_entropyin, kat264_nonce, kat264_persstr,
    kat264_addin0, kat264_addin1, kat264_retbits
};
static const struct drbg_kat kat264 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat264_t
};

static const unsigned char kat265_entropyin[] = {
    0xb9, 0x5d, 0x8b, 0x26, 0x6a, 0xcc, 0xcd, 0x6c, 0xba, 0xc0, 0xc0, 0x61,
    0xec, 0x32, 0xa1, 0xea, 0x71, 0x1f, 0xfe, 0xc8, 0x00, 0x6b, 0xc7, 0xcc,
};
static const unsigned char kat265_nonce[] = {
    0xbf, 0xa0, 0x84, 0x02, 0xc0, 0xc6, 0xb8, 0x7d, 0xec, 0xa2, 0x24, 0x6d,
    0x81, 0xf6, 0x7f, 0x58,
};
static const unsigned char kat265_persstr[] = {0};
static const unsigned char kat265_addin0[] = {
    0x72, 0xff, 0xd4, 0xa1, 0x83, 0x99, 0x0d, 0xac, 0x6b, 0xc7, 0xa9, 0x46,
    0x47, 0x14, 0x27, 0x59, 0xbf, 0x88, 0x1c, 0xc0, 0xb3, 0x17, 0x8f, 0x60,
    0xe0, 0xe2, 0xcb, 0xc3, 0x33, 0x79, 0xf7, 0xad,
};
static const unsigned char kat265_addin1[] = {
    0x2e, 0x9f, 0xe4, 0xd1, 0xaf, 0x62, 0xb1, 0x5f, 0x85, 0x35, 0xe1, 0x98,
    0xaa, 0xd7, 0x93, 0x61, 0xe9, 0x46, 0x16, 0x41, 0x8e, 0xd2, 0xd8, 0x01,
    0xe7, 0x0f, 0xad, 0x9e, 0x0e, 0x9c, 0xa5, 0xb8,
};
static const unsigned char kat265_retbits[] = {
    0x1b, 0x0d, 0x15, 0xfb, 0x4f, 0xcd, 0xca, 0x14, 0x97, 0xdc, 0xfe, 0x60,
    0xe7, 0xd8, 0x2b, 0xd0, 0x40, 0x1b, 0x16, 0x23, 0x36, 0x6a, 0x71, 0xcf,
    0x8f, 0xa5, 0x75, 0x4f, 0x87, 0x8e, 0x20, 0xd8, 0xf2, 0x54, 0x5b, 0xc3,
    0xd4, 0x1e, 0x79, 0x73, 0xae, 0x32, 0x90, 0x50, 0x6b, 0xa2, 0xa9, 0x76,
    0xb8, 0x3b, 0x7b, 0x98, 0xc7, 0x66, 0xd8, 0xec, 0x0c, 0x26, 0x4b, 0xe2,
    0x87, 0xee, 0x63, 0xf5,
};
static const struct drbg_kat_no_reseed kat265_t = {
    9, kat265_entropyin, kat265_nonce, kat265_persstr,
    kat265_addin0, kat265_addin1, kat265_retbits
};
static const struct drbg_kat kat265 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat265_t
};

static const unsigned char kat266_entropyin[] = {
    0x6d, 0x30, 0x46, 0x58, 0x37, 0xa6, 0x1e, 0x49, 0xda, 0xc0, 0xfc, 0x84,
    0x48, 0xde, 0x3a, 0xfa, 0x2e, 0x22, 0xe0, 0x94, 0x28, 0x6a, 0x5a, 0xed,
};
static const unsigned char kat266_nonce[] = {
    0x94, 0x18, 0xc0, 0x28, 0x70, 0xef, 0x4c, 0xd1, 0x19, 0xdd, 0xb6, 0x73,
    0xee, 0x69, 0x9d, 0xde,
};
static const unsigned char kat266_persstr[] = {0};
static const unsigned char kat266_addin0[] = {
    0xac, 0x32, 0x2e, 0xcf, 0xff, 0x1f, 0x73, 0x26, 0xa4, 0xf4, 0xcb, 0x33,
    0xc1, 0x76, 0xfe, 0x83, 0x1c, 0x0d, 0xdf, 0x37, 0x37, 0x69, 0xfd, 0xe6,
    0x1a, 0x42, 0x42, 0x6f, 0xe9, 0x9e, 0xa8, 0x49,
};
static const unsigned char kat266_addin1[] = {
    0x5b, 0x70, 0x52, 0x49, 0x1c, 0x3f, 0x53, 0x6e, 0xcb, 0x91, 0xfd, 0x04,
    0x38, 0x0f, 0x9f, 0x03, 0x80, 0xa4, 0x15, 0xdc, 0xde, 0x60, 0x1a, 0x67,
    0x0a, 0x1e, 0xe6, 0xb8, 0xf2, 0x2e, 0x96, 0xda,
};
static const unsigned char kat266_retbits[] = {
    0xdc, 0xa3, 0xfd, 0xae, 0xf8, 0xbf, 0xb2, 0xd0, 0x9b, 0xb9, 0x06, 0xab,
    0xcb, 0x21, 0x81, 0x8a, 0xa6, 0x6a, 0x06, 0x4e, 0x71, 0x18, 0xff, 0x3d,
    0xac, 0x55, 0x66, 0xca, 0x99, 0x1e, 0xbb, 0x37, 0x33, 0x82, 0x41, 0xe0,
    0xc8, 0x29, 0x76, 0x8f, 0x71, 0x5e, 0x4c, 0x2f, 0x11, 0x57, 0xeb, 0x0a,
    0xc4, 0x73, 0xba, 0xd8, 0x6b, 0x05, 0x13, 0xb6, 0x48, 0x54, 0xf3, 0xed,
    0x8d, 0xa4, 0x83, 0x28,
};
static const struct drbg_kat_no_reseed kat266_t = {
    10, kat266_entropyin, kat266_nonce, kat266_persstr,
    kat266_addin0, kat266_addin1, kat266_retbits
};
static const struct drbg_kat kat266 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat266_t
};

static const unsigned char kat267_entropyin[] = {
    0xc6, 0x83, 0x21, 0x3a, 0x47, 0x37, 0x5e, 0x29, 0x75, 0xd3, 0x00, 0x4b,
    0xcd, 0x6f, 0xb4, 0xaa, 0xb6, 0x55, 0x31, 0xc7, 0x52, 0x93, 0x1d, 0x6b,
};
static const unsigned char kat267_nonce[] = {
    0x11, 0xf2, 0x78, 0xb8, 0x64, 0x4f, 0x52, 0xa0, 0xa4, 0x07, 0x41, 0x64,
    0x09, 0x7d, 0xe4, 0x58,
};
static const unsigned char kat267_persstr[] = {0};
static const unsigned char kat267_addin0[] = {
    0x37, 0xc0, 0x65, 0x5e, 0xa6, 0xc2, 0x0e, 0xc4, 0x3c, 0x58, 0x85, 0x5b,
    0xe3, 0x14, 0xaf, 0xde, 0x29, 0x32, 0x4a, 0x81, 0x86, 0xae, 0x9c, 0x08,
    0xc8, 0xad, 0x4d, 0x85, 0x70, 0x08, 0x1a, 0x72,
};
static const unsigned char kat267_addin1[] = {
    0xc3, 0x9b, 0xc5, 0x60, 0xc7, 0x1a, 0xa5, 0xcd, 0xf2, 0xa0, 0xec, 0xa0,
    0xff, 0xa4, 0xfb, 0x56, 0x2c, 0xab, 0x13, 0x79, 0xbc, 0x90, 0x43, 0x25,
    0x9e, 0xf8, 0x93, 0x44, 0x36, 0x14, 0x23, 0x23,
};
static const unsigned char kat267_retbits[] = {
    0x7a, 0xf8, 0xcd, 0xc7, 0x05, 0xcb, 0x06, 0xc4, 0x08, 0xf7, 0x89, 0xb5,
    0xa0, 0xd6, 0x77, 0xe6, 0x6f, 0xb3, 0x2b, 0xc7, 0xa5, 0x78, 0xc4, 0xcd,
    0x6d, 0xe3, 0x42, 0x02, 0xf1, 0x7a, 0x88, 0xd9, 0x1a, 0x23, 0x85, 0x74,
    0x66, 0x64, 0x49, 0xc4, 0x05, 0x9f, 0x2b, 0xdf, 0x59, 0x35, 0x57, 0xfb,
    0x78, 0xd0, 0x77, 0x95, 0x37, 0x16, 0x46, 0xf0, 0x3c, 0xf3, 0x85, 0xf3,
    0x05, 0x8b, 0x1d, 0x78,
};
static const struct drbg_kat_no_reseed kat267_t = {
    11, kat267_entropyin, kat267_nonce, kat267_persstr,
    kat267_addin0, kat267_addin1, kat267_retbits
};
static const struct drbg_kat kat267 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat267_t
};

static const unsigned char kat268_entropyin[] = {
    0x8b, 0xd5, 0xa5, 0x95, 0x22, 0x83, 0x77, 0x67, 0xfb, 0xc7, 0x83, 0xca,
    0xa9, 0xa7, 0xfe, 0xec, 0x68, 0xdf, 0x13, 0x56, 0x16, 0xe9, 0x88, 0x78,
};
static const unsigned char kat268_nonce[] = {
    0x14, 0xaa, 0xa4, 0x14, 0xad, 0xe4, 0x8e, 0x33, 0xac, 0x5b, 0x44, 0x3b,
    0x94, 0xa9, 0xf1, 0xac,
};
static const unsigned char kat268_persstr[] = {0};
static const unsigned char kat268_addin0[] = {
    0x23, 0x3b, 0x1e, 0x2c, 0x23, 0x0c, 0xea, 0xba, 0xf8, 0x8e, 0xc1, 0xba,
    0x30, 0xef, 0xb7, 0x2e, 0x35, 0xcb, 0xf9, 0x9f, 0xdc, 0x92, 0x59, 0x51,
    0x40, 0x19, 0xbc, 0x96, 0xf6, 0xee, 0xcb, 0xde,
};
static const unsigned char kat268_addin1[] = {
    0xf8, 0xb1, 0x99, 0x22, 0xb8, 0x89, 0x61, 0x75, 0xad, 0xa1, 0x15, 0x24,
    0xb9, 0x8f, 0xff, 0xc1, 0x15, 0x9f, 0x14, 0x56, 0xdd, 0x7a, 0xa6, 0x57,
    0xb7, 0x54, 0x7f, 0x0b, 0xf9, 0xe4, 0xfd, 0xe4,
};
static const unsigned char kat268_retbits[] = {
    0x49, 0x4a, 0x5d, 0x31, 0xa5, 0x8a, 0x50, 0xf5, 0x06, 0x82, 0x52, 0x79,
    0xb2, 0x48, 0x83, 0x59, 0x14, 0x08, 0x1d, 0xdc, 0xcd, 0x63, 0x80, 0x67,
    0xdf, 0x0e, 0x2d, 0x3c, 0x62, 0x00, 0x8a, 0x4f, 0x10, 0x8c, 0x81, 0x91,
    0x59, 0x00, 0x0b, 0xc0, 0xd0, 0xc7, 0x51, 0x5d, 0xbe, 0x48, 0x8c, 0x3d,
    0xc6, 0xca, 0x3f, 0x28, 0x78, 0xca, 0x58, 0x96, 0xb1, 0x33, 0x50, 0x98,
    0x5e, 0xd7, 0x60, 0x2d,
};
static const struct drbg_kat_no_reseed kat268_t = {
    12, kat268_entropyin, kat268_nonce, kat268_persstr,
    kat268_addin0, kat268_addin1, kat268_retbits
};
static const struct drbg_kat kat268 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat268_t
};

static const unsigned char kat269_entropyin[] = {
    0x97, 0xfe, 0xf1, 0xc3, 0x6e, 0x3c, 0xab, 0x72, 0xd5, 0x04, 0x3e, 0x41,
    0x6c, 0xb1, 0x57, 0x06, 0xd6, 0xd7, 0x6e, 0xc5, 0x01, 0x64, 0x51, 0xe2,
};
static const unsigned char kat269_nonce[] = {
    0xaa, 0x86, 0x57, 0x62, 0xb8, 0x01, 0x65, 0xe2, 0x39, 0xed, 0x79, 0x51,
    0xb0, 0x56, 0xc7, 0x70,
};
static const unsigned char kat269_persstr[] = {0};
static const unsigned char kat269_addin0[] = {
    0x41, 0x89, 0x07, 0xa2, 0x53, 0x32, 0xb9, 0xd2, 0x42, 0xbb, 0x18, 0x77,
    0x5f, 0xdc, 0xb0, 0xb4, 0xf2, 0xec, 0x98, 0x11, 0x75, 0x0e, 0x05, 0x1f,
    0x96, 0xe3, 0xd6, 0xd2, 0x52, 0xeb, 0x87, 0xde,
};
static const unsigned char kat269_addin1[] = {
    0x57, 0x4b, 0x64, 0x27, 0x93, 0x94, 0x23, 0x4f, 0xe5, 0x9a, 0x14, 0xcb,
    0xc9, 0x40, 0x4a, 0x0b, 0xc1, 0x46, 0x9e, 0x6d, 0x18, 0x1e, 0xeb, 0x4e,
    0x74, 0x14, 0xa4, 0xa6, 0xde, 0x0b, 0xab, 0x7d,
};
static const unsigned char kat269_retbits[] = {
    0x8c, 0x6b, 0x47, 0x5b, 0xac, 0xf9, 0x33, 0xc0, 0xf7, 0xa0, 0x7c, 0x88,
    0xa5, 0x28, 0xfe, 0xef, 0xc9, 0x52, 0xb9, 0x55, 0x31, 0x05, 0xb2, 0x0a,
    0x17, 0xe3, 0xba, 0xd3, 0x93, 0x9e, 0x94, 0x32, 0x0f, 0xa8, 0xe2, 0x80,
    0xfc, 0x54, 0xb4, 0x85, 0xd4, 0xcf, 0xe7, 0x94, 0x08, 0x1c, 0x05, 0x4e,
    0xe8, 0xe7, 0xc2, 0x4c, 0x65, 0xf6, 0xbb, 0x95, 0xd8, 0x82, 0x72, 0x87,
    0x4d, 0x2a, 0x04, 0x2f,
};
static const struct drbg_kat_no_reseed kat269_t = {
    13, kat269_entropyin, kat269_nonce, kat269_persstr,
    kat269_addin0, kat269_addin1, kat269_retbits
};
static const struct drbg_kat kat269 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat269_t
};

static const unsigned char kat270_entropyin[] = {
    0x54, 0x32, 0xa3, 0x40, 0x20, 0x56, 0x25, 0xc2, 0x0a, 0xfd, 0xd4, 0x2e,
    0xcd, 0x54, 0xe8, 0xd6, 0xef, 0x80, 0xb2, 0x9c, 0x73, 0xf6, 0x2d, 0xba,
};
static const unsigned char kat270_nonce[] = {
    0xb8, 0xfb, 0x44, 0x40, 0xd8, 0x95, 0x54, 0x99, 0xcd, 0xad, 0x97, 0xb9,
    0xb0, 0xe3, 0x3f, 0x63,
};
static const unsigned char kat270_persstr[] = {0};
static const unsigned char kat270_addin0[] = {
    0x9a, 0x53, 0x54, 0x10, 0x62, 0x6b, 0x87, 0x2d, 0xa5, 0x06, 0x43, 0x19,
    0xdb, 0x92, 0x1d, 0x9f, 0xa0, 0x15, 0xa6, 0x75, 0x48, 0xf1, 0x2e, 0x8a,
    0xf2, 0xb1, 0x55, 0x13, 0x0a, 0xb6, 0x32, 0xd0,
};
static const unsigned char kat270_addin1[] = {
    0x3a, 0x12, 0x51, 0x9f, 0x49, 0x7a, 0x8c, 0x3a, 0xf0, 0xf3, 0x5f, 0xdf,
    0xe4, 0xaa, 0x47, 0xfe, 0xf0, 0x39, 0x56, 0x3c, 0x03, 0x8c, 0x52, 0xaa,
    0x4e, 0xe4, 0xce, 0xcd, 0x05, 0x5f, 0x1b, 0xb6,
};
static const unsigned char kat270_retbits[] = {
    0x57, 0x01, 0xe2, 0xee, 0x57, 0x46, 0x84, 0xc8, 0x55, 0xd4, 0x03, 0xe5,
    0xa6, 0x1b, 0x13, 0x2c, 0x21, 0x1e, 0x64, 0x6f, 0xf5, 0xed, 0x41, 0xcf,
    0xcb, 0x81, 0xf7, 0x9b, 0xdd, 0x86, 0x7e, 0xf5, 0xcd, 0xa7, 0xf0, 0xba,
    0x57, 0x99, 0xf9, 0x3c, 0x07, 0xf4, 0xb5, 0x58, 0x8c, 0x34, 0xd4, 0x33,
    0xdf, 0x93, 0x35, 0x01, 0x0f, 0xb6, 0x01, 0x10, 0x91, 0x17, 0x7f, 0x5e,
    0x0a, 0x7c, 0xed, 0xe7,
};
static const struct drbg_kat_no_reseed kat270_t = {
    14, kat270_entropyin, kat270_nonce, kat270_persstr,
    kat270_addin0, kat270_addin1, kat270_retbits
};
static const struct drbg_kat kat270 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat270_t
};

static const unsigned char kat271_entropyin[] = {
    0xeb, 0x45, 0x53, 0xf7, 0x14, 0x1b, 0xc1, 0x08, 0x82, 0xf0, 0xa3, 0x74,
    0x17, 0x80, 0xa4, 0x2d, 0xf8, 0xbc, 0x38, 0xa7, 0x12, 0x1d, 0x3b, 0x7e,
};
static const unsigned char kat271_nonce[] = {
    0x6f, 0x34, 0x7f, 0x9c, 0x1d, 0xe8, 0x4f, 0xd5, 0x34, 0x16, 0x25, 0xae,
    0x8d, 0x6b, 0xf5, 0x0c,
};
static const unsigned char kat271_persstr[] = {
    0x5e, 0x2e, 0x73, 0xb8, 0x6c, 0xa2, 0xf3, 0x15, 0x0d, 0x53, 0xd2, 0x3d,
    0x59, 0x0a, 0xcb, 0xee, 0xda, 0xaf, 0x91, 0x63, 0x8b, 0xdc, 0x3f, 0x9d,
    0x58, 0x8e, 0x94, 0x5a, 0xf4, 0xbb, 0x6e, 0xa2,
};
static const unsigned char kat271_addin0[] = {0};
static const unsigned char kat271_addin1[] = {0};
static const unsigned char kat271_retbits[] = {
    0xa1, 0xde, 0xb9, 0xa5, 0xaa, 0xd6, 0x08, 0xa5, 0x87, 0xd6, 0x1c, 0xe5,
    0xe0, 0xd7, 0xc7, 0xdd, 0x44, 0x9b, 0x8c, 0x87, 0x89, 0x83, 0x54, 0xad,
    0x1a, 0xdd, 0x6e, 0x05, 0x85, 0x38, 0x73, 0xd2, 0x79, 0xeb, 0xe4, 0x13,
    0x2f, 0xc2, 0x36, 0xa4, 0x2d, 0x8f, 0xf0, 0xdc, 0x3a, 0xce, 0x95, 0xd2,
    0xcd, 0xf9, 0xd0, 0xb0, 0x57, 0x11, 0x7c, 0xb1, 0x19, 0xee, 0x75, 0x50,
    0xce, 0x03, 0x08, 0x5c,
};
static const struct drbg_kat_no_reseed kat271_t = {
    0, kat271_entropyin, kat271_nonce, kat271_persstr,
    kat271_addin0, kat271_addin1, kat271_retbits
};
static const struct drbg_kat kat271 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat271_t
};

static const unsigned char kat272_entropyin[] = {
    0xf9, 0xce, 0x2d, 0x06, 0x49, 0xbc, 0x99, 0x28, 0x8a, 0xf1, 0x5f, 0xdf,
    0xbc, 0x3d, 0xb8, 0x89, 0x56, 0xd9, 0x6c, 0x84, 0xc0, 0xd7, 0xe5, 0xd2,
};
static const unsigned char kat272_nonce[] = {
    0x8c, 0xf0, 0x0c, 0x63, 0x7a, 0x07, 0x9a, 0x98, 0x36, 0x2e, 0xad, 0x51,
    0x14, 0x9e, 0x55, 0x67,
};
static const unsigned char kat272_persstr[] = {
    0xb2, 0x44, 0xd6, 0x8a, 0x9b, 0x30, 0xf3, 0xac, 0x88, 0x04, 0x0d, 0x64,
    0x58, 0xa6, 0x25, 0x08, 0x00, 0x20, 0x53, 0x53, 0x41, 0x53, 0x3b, 0xe2,
    0x70, 0xe8, 0x94, 0x00, 0x2c, 0x07, 0x69, 0x7d,
};
static const unsigned char kat272_addin0[] = {0};
static const unsigned char kat272_addin1[] = {0};
static const unsigned char kat272_retbits[] = {
    0x28, 0x51, 0x19, 0x2f, 0xd3, 0xb3, 0x73, 0x51, 0xd0, 0x51, 0x10, 0x97,
    0x43, 0x88, 0xec, 0xe0, 0x11, 0xd1, 0x0e, 0x7b, 0x9d, 0x38, 0x01, 0x40,
    0x29, 0x10, 0x48, 0xce, 0x36, 0x72, 0xc1, 0x34, 0xbc, 0xb4, 0xa0, 0xcd,
    0x07, 0x4f, 0xff, 0xf3, 0x89, 0xa0, 0x2a, 0xf5, 0x9c, 0x52, 0x26, 0xbe,
    0x02, 0x53, 0xe7, 0xb7, 0x40, 0x0e, 0x63, 0x44, 0xb1, 0xa0, 0xd0, 0xd1,
    0x45, 0xff, 0x36, 0x6c,
};
static const struct drbg_kat_no_reseed kat272_t = {
    1, kat272_entropyin, kat272_nonce, kat272_persstr,
    kat272_addin0, kat272_addin1, kat272_retbits
};
static const struct drbg_kat kat272 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat272_t
};

static const unsigned char kat273_entropyin[] = {
    0x61, 0x8a, 0xb7, 0xda, 0x12, 0xa5, 0x83, 0x9b, 0x4d, 0x80, 0x8b, 0xc2,
    0x7c, 0xd5, 0xd3, 0x25, 0x00, 0x00, 0x78, 0x14, 0xa5, 0x4e, 0x5d, 0xe0,
};
static const unsigned char kat273_nonce[] = {
    0xc3, 0xef, 0xab, 0x85, 0x7f, 0x1f, 0xeb, 0x04, 0x9e, 0xe0, 0x60, 0xba,
    0x76, 0x0f, 0x17, 0xe0,
};
static const unsigned char kat273_persstr[] = {
    0x33, 0xc6, 0xaf, 0x4e, 0x26, 0x4f, 0x0d, 0x19, 0xc3, 0x61, 0xec, 0xec,
    0xf8, 0x9b, 0xd7, 0x86, 0x9f, 0xb0, 0xaf, 0x7f, 0x9b, 0x39, 0x15, 0x9c,
    0x0f, 0xab, 0xe0, 0x81, 0x14, 0x31, 0xe6, 0x2c,
};
static const unsigned char kat273_addin0[] = {0};
static const unsigned char kat273_addin1[] = {0};
static const unsigned char kat273_retbits[] = {
    0x1d, 0x74, 0xba, 0x44, 0xea, 0xdb, 0xae, 0x17, 0x6a, 0x0a, 0x87, 0x06,
    0x22, 0x17, 0x5e, 0x4b, 0x0e, 0xe4, 0xe4, 0x35, 0x2f, 0x8c, 0x2e, 0xe1,
    0x95, 0x53, 0xdc, 0xb2, 0x10, 0x0f, 0x8e, 0x2f, 0x13, 0x2d, 0xfd, 0x4f,
    0x4c, 0xad, 0x5e, 0x01, 0xe3, 0xb7, 0x02, 0x22, 0x89, 0x02, 0xdc, 0xbe,
    0xe5, 0xaf, 0xd5, 0x39, 0x09, 0x39, 0xc3, 0x61, 0x88, 0x2a, 0x0b, 0x67,
    0x9d, 0xc2, 0xcd, 0x69,
};
static const struct drbg_kat_no_reseed kat273_t = {
    2, kat273_entropyin, kat273_nonce, kat273_persstr,
    kat273_addin0, kat273_addin1, kat273_retbits
};
static const struct drbg_kat kat273 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat273_t
};

static const unsigned char kat274_entropyin[] = {
    0x7d, 0xf2, 0xa3, 0xc0, 0xbd, 0x95, 0xc6, 0xba, 0x88, 0x73, 0xd6, 0xa5,
    0xad, 0x5d, 0xbf, 0x0e, 0x06, 0x9d, 0xa3, 0xcd, 0xe1, 0xe3, 0x94, 0x3d,
};
static const unsigned char kat274_nonce[] = {
    0xfa, 0x28, 0x22, 0x4a, 0x89, 0x49, 0x13, 0x4e, 0x28, 0x50, 0xc5, 0x2a,
    0x28, 0x57, 0x6f, 0x65,
};
static const unsigned char kat274_persstr[] = {
    0xe2, 0xde, 0xa1, 0x5d, 0x8d, 0x3a, 0xee, 0xd8, 0x7f, 0xf4, 0x5e, 0x79,
    0xa4, 0xa7, 0x60, 0xa8, 0x96, 0x83, 0xdc, 0xf8, 0x2c, 0xfe, 0x35, 0x64,
    0x67, 0xaf, 0xfc, 0x44, 0x59, 0x2e, 0x2b, 0xf5,
};
static const unsigned char kat274_addin0[] = {0};
static const unsigned char kat274_addin1[] = {0};
static const unsigned char kat274_retbits[] = {
    0x3c, 0x48, 0x82, 0x3f, 0x45, 0x28, 0xb3, 0x96, 0xc8, 0x66, 0x74, 0x07,
    0xcb, 0x88, 0x46, 0xa2, 0x29, 0x52, 0x7d, 0x95, 0x89, 0xf1, 0xce, 0xb2,
    0xfe, 0xe6, 0x40, 0xa8, 0x3f, 0x93, 0x32, 0x71, 0x07, 0xc4, 0xc9, 0x91,
    0xc2, 0xc8, 0xba, 0x3e, 0xe8, 0x18, 0x53, 0x14, 0x0a, 0x34, 0x8c, 0x1c,
    0xa5, 0xce, 0x26, 0x4e, 0xf7, 0x5d, 0xb4, 0x95, 0x67, 0x94, 0x26, 0x8c,
    0x55, 0x38, 0x10, 0x1f,
};
static const struct drbg_kat_no_reseed kat274_t = {
    3, kat274_entropyin, kat274_nonce, kat274_persstr,
    kat274_addin0, kat274_addin1, kat274_retbits
};
static const struct drbg_kat kat274 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat274_t
};

static const unsigned char kat275_entropyin[] = {
    0x2d, 0x8a, 0x1a, 0x3a, 0x6a, 0x77, 0xe3, 0xcd, 0x46, 0x40, 0xa0, 0x78,
    0x0f, 0x59, 0xcb, 0x97, 0x5f, 0x81, 0xc2, 0x73, 0x3a, 0xd7, 0xf4, 0x98,
};
static const unsigned char kat275_nonce[] = {
    0x50, 0xfc, 0x6b, 0xcc, 0x7e, 0xcb, 0xdb, 0x3d, 0x57, 0xbe, 0xab, 0x7e,
    0x28, 0xa4, 0x9b, 0xc0,
};
static const unsigned char kat275_persstr[] = {
    0x95, 0xf9, 0xc3, 0x56, 0x3b, 0x53, 0x5e, 0x69, 0xa4, 0x91, 0x34, 0xc3,
    0x36, 0xcb, 0x80, 0xfa, 0x9a, 0xd9, 0x51, 0x08, 0xc7, 0x56, 0xea, 0x26,
    0x1f, 0x5b, 0x3a, 0xe9, 0xcb, 0xaf, 0xd4, 0x1b,
};
static const unsigned char kat275_addin0[] = {0};
static const unsigned char kat275_addin1[] = {0};
static const unsigned char kat275_retbits[] = {
    0x86, 0x87, 0x1f, 0x9c, 0xb6, 0xb4, 0xed, 0x25, 0x2b, 0xd1, 0xe8, 0x68,
    0xc8, 0x0a, 0x26, 0x3e, 0x02, 0x5b, 0xba, 0xe2, 0x28, 0x5c, 0xca, 0x59,
    0xc6, 0x29, 0x98, 0x27, 0x32, 0xa5, 0x06, 0x3e, 0x5c, 0xbd, 0xa2, 0x76,
    0xf2, 0x82, 0xfd, 0xaa, 0x90, 0xae, 0xf8, 0xec, 0x6d, 0xd3, 0x1d, 0x32,
    0xb7, 0x04, 0xde, 0x50, 0x28, 0xdd, 0xd3, 0x2e, 0x22, 0xde, 0x36, 0x80,
    0x08, 0x6f, 0x9a, 0x89,
};
static const struct drbg_kat_no_reseed kat275_t = {
    4, kat275_entropyin, kat275_nonce, kat275_persstr,
    kat275_addin0, kat275_addin1, kat275_retbits
};
static const struct drbg_kat kat275 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat275_t
};

static const unsigned char kat276_entropyin[] = {
    0x13, 0x35, 0x24, 0xed, 0xab, 0xd5, 0xa7, 0xf7, 0x85, 0x8c, 0x13, 0xf2,
    0x79, 0x53, 0xe9, 0x87, 0x28, 0x3c, 0xb1, 0x72, 0xa2, 0xc3, 0x7f, 0x17,
};
static const unsigned char kat276_nonce[] = {
    0x5e, 0x6c, 0xb7, 0xf7, 0xc9, 0x8e, 0xe1, 0x3f, 0x46, 0x7d, 0x6b, 0xda,
    0x28, 0x8b, 0x57, 0x7e,
};
static const unsigned char kat276_persstr[] = {
    0x67, 0xd3, 0x91, 0x60, 0xcc, 0xee, 0x30, 0x40, 0xdb, 0x78, 0x20, 0xa3,
    0x7a, 0xef, 0xb7, 0xd4, 0xa1, 0x0f, 0x7d, 0xcd, 0xf3, 0xbc, 0xa6, 0x69,
    0xfe, 0x23, 0x5d, 0xb6, 0x3a, 0xb2, 0x47, 0x60,
};
static const unsigned char kat276_addin0[] = {0};
static const unsigned char kat276_addin1[] = {0};
static const unsigned char kat276_retbits[] = {
    0x2d, 0x13, 0xee, 0x8b, 0x73, 0xd5, 0x18, 0xb0, 0xaf, 0xbf, 0x4e, 0x1e,
    0xa6, 0xf6, 0x32, 0x00, 0x6b, 0xa5, 0x6e, 0x3f, 0x4e, 0x24, 0xa2, 0x1f,
    0x68, 0x77, 0xed, 0x0f, 0x79, 0xc3, 0xec, 0x7f, 0xfd, 0x19, 0xce, 0x81,
    0xbb, 0x17, 0x0b, 0x3a, 0xa9, 0x0d, 0xf6, 0x97, 0xf5, 0xcd, 0x98, 0x72,
    0xcc, 0xb3, 0xd1, 0xe3, 0x38, 0x94, 0xdd, 0x16, 0xc6, 0xf5, 0xcf, 0x2f,
    0xb1, 0x07, 0xae, 0xdd,
};
static const struct drbg_kat_no_reseed kat276_t = {
    5, kat276_entropyin, kat276_nonce, kat276_persstr,
    kat276_addin0, kat276_addin1, kat276_retbits
};
static const struct drbg_kat kat276 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat276_t
};

static const unsigned char kat277_entropyin[] = {
    0xf2, 0x97, 0x47, 0x4b, 0x5b, 0x7e, 0x68, 0xd5, 0xae, 0xa9, 0x48, 0xf7,
    0x51, 0xbc, 0x89, 0x9b, 0x36, 0xc2, 0x12, 0x63, 0x6f, 0x28, 0x8b, 0x84,
};
static const unsigned char kat277_nonce[] = {
    0xb8, 0xa6, 0x0d, 0xd7, 0x26, 0xa0, 0x3f, 0xd0, 0xd2, 0xbf, 0x19, 0x5d,
    0x2c, 0xb0, 0x9a, 0x03,
};
static const unsigned char kat277_persstr[] = {
    0x8e, 0x32, 0xdb, 0x43, 0x66, 0x90, 0x7a, 0xbb, 0x98, 0xc0, 0xe0, 0x9c,
    0x53, 0x60, 0xc5, 0x6f, 0xdb, 0x6f, 0x48, 0x3c, 0x84, 0xe6, 0x06, 0xf0,
    0x7f, 0x5d, 0x1d, 0x45, 0xa0, 0x93, 0x46, 0xca,
};
static const unsigned char kat277_addin0[] = {0};
static const unsigned char kat277_addin1[] = {0};
static const unsigned char kat277_retbits[] = {
    0x92, 0x08, 0x90, 0x94, 0xa8, 0x9f, 0xb5, 0x32, 0xf0, 0x68, 0xa8, 0x63,
    0x0f, 0xa9, 0x47, 0xb8, 0xf8, 0x6e, 0xee, 0x22, 0xd5, 0x6f, 0x22, 0xa5,
    0x14, 0xf8, 0xa6, 0x87, 0x1a, 0xa4, 0xc8, 0x08, 0xc8, 0xc9, 0xf4, 0x7c,
    0x13, 0x54, 0xf1, 0x51, 0xc6, 0x8b, 0xc1, 0x30, 0xc8, 0xe8, 0x5f, 0xe4,
    0x7e, 0x8b, 0xea, 0xc8, 0xcb, 0x34, 0x6b, 0x8e, 0xe2, 0xa7, 0xe0, 0x01,
    0x59, 0xa0, 0xea, 0x80,
};
static const struct drbg_kat_no_reseed kat277_t = {
    6, kat277_entropyin, kat277_nonce, kat277_persstr,
    kat277_addin0, kat277_addin1, kat277_retbits
};
static const struct drbg_kat kat277 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat277_t
};

static const unsigned char kat278_entropyin[] = {
    0x70, 0x50, 0x8f, 0x60, 0x35, 0xcd, 0xc4, 0x1a, 0xf6, 0x32, 0x52, 0xaf,
    0x23, 0xbe, 0x67, 0xaf, 0x4a, 0x46, 0x8f, 0x45, 0x53, 0x31, 0x10, 0xc8,
};
static const unsigned char kat278_nonce[] = {
    0x77, 0xc2, 0xbc, 0x1a, 0x84, 0xf4, 0xf9, 0x91, 0x79, 0x6f, 0x9c, 0x07,
    0x09, 0x2a, 0xd5, 0xc7,
};
static const unsigned char kat278_persstr[] = {
    0x47, 0x73, 0x82, 0xb2, 0xf6, 0xde, 0x44, 0xe2, 0xf0, 0x9a, 0x1d, 0x13,
    0x5a, 0x35, 0xf1, 0xcc, 0xb0, 0x28, 0x51, 0xf9, 0xe1, 0xf0, 0x03, 0xf9,
    0x8e, 0x02, 0x20, 0x43, 0x45, 0x8f, 0x5b, 0x66,
};
static const unsigned char kat278_addin0[] = {0};
static const unsigned char kat278_addin1[] = {0};
static const unsigned char kat278_retbits[] = {
    0xfc, 0x63, 0x4f, 0xd4, 0xdc, 0xa7, 0xfc, 0xd6, 0x6c, 0x38, 0x92, 0x65,
    0x18, 0xa8, 0xd6, 0xfd, 0xee, 0xca, 0x07, 0xf8, 0x7e, 0x9b, 0x9f, 0xe5,
    0x40, 0x5e, 0xfd, 0x9a, 0xf9, 0xc9, 0x41, 0xcb, 0xf1, 0x56, 0xbc, 0x5b,
    0x09, 0xa5, 0x3f, 0x5c, 0xc8, 0xb5, 0xbf, 0x94, 0x37, 0xe6, 0x76, 0x90,
    0x5a, 0xfa, 0xee, 0x58, 0x02, 0x7b, 0xc2, 0x5a, 0xd4, 0x6c, 0x32, 0xab,
    0xbe, 0xa0, 0x5c, 0x85,
};
static const struct drbg_kat_no_reseed kat278_t = {
    7, kat278_entropyin, kat278_nonce, kat278_persstr,
    kat278_addin0, kat278_addin1, kat278_retbits
};
static const struct drbg_kat kat278 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat278_t
};

static const unsigned char kat279_entropyin[] = {
    0x2e, 0x5d, 0x95, 0x68, 0x7b, 0x0e, 0x9b, 0x77, 0x7f, 0x13, 0x94, 0xf1,
    0x86, 0x63, 0xe7, 0x98, 0xed, 0xbd, 0x24, 0xcf, 0x0c, 0x3b, 0x94, 0x58,
};
static const unsigned char kat279_nonce[] = {
    0x02, 0x05, 0x5f, 0xa3, 0xeb, 0x3d, 0x12, 0x04, 0xc2, 0x53, 0xeb, 0xf3,
    0x5e, 0x31, 0x43, 0xbd,
};
static const unsigned char kat279_persstr[] = {
    0x17, 0x56, 0x63, 0x5c, 0x33, 0xf8, 0x68, 0x6b, 0x45, 0x8d, 0xaa, 0xec,
    0x9b, 0x9b, 0x44, 0x60, 0xb8, 0xa1, 0xd7, 0x5a, 0xa2, 0xe3, 0x00, 0xe7,
    0x55, 0x57, 0x41, 0x12, 0x49, 0xab, 0xc0, 0x0f,
};
static const unsigned char kat279_addin0[] = {0};
static const unsigned char kat279_addin1[] = {0};
static const unsigned char kat279_retbits[] = {
    0x91, 0xc7, 0xe7, 0x18, 0x3e, 0xed, 0x6b, 0xa6, 0x64, 0x96, 0x67, 0x8b,
    0x9e, 0xe8, 0xec, 0x8b, 0x86, 0xde, 0x02, 0xfd, 0x7c, 0x6c, 0xb9, 0x77,
    0x48, 0x2f, 0x0d, 0xf4, 0x84, 0x9a, 0x72, 0x85, 0x9a, 0x80, 0x4d, 0x26,
    0x86, 0x68, 0xa8, 0xf4, 0xc9, 0x49, 0x34, 0x41, 0x3a, 0x94, 0xa2, 0xff,
    0x0d, 0x9d, 0x39, 0xb5, 0x18, 0x86, 0x07, 0xcc, 0x75, 0xa0, 0x79, 0xa7,
    0xe4, 0x84, 0x61, 0x69,
};
static const struct drbg_kat_no_reseed kat279_t = {
    8, kat279_entropyin, kat279_nonce, kat279_persstr,
    kat279_addin0, kat279_addin1, kat279_retbits
};
static const struct drbg_kat kat279 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat279_t
};

static const unsigned char kat280_entropyin[] = {
    0x2e, 0x99, 0x70, 0x3a, 0x2b, 0xf1, 0x95, 0xa1, 0x82, 0xd2, 0x21, 0xce,
    0x79, 0xbd, 0xbd, 0xfc, 0xa4, 0xdb, 0x53, 0xe1, 0x6e, 0x75, 0x7a, 0x5b,
};
static const unsigned char kat280_nonce[] = {
    0x4b, 0xe5, 0x2f, 0xba, 0x5d, 0x1f, 0xbc, 0x7e, 0xa7, 0x7b, 0x4a, 0xe3,
    0x0d, 0x16, 0xe0, 0xe9,
};
static const unsigned char kat280_persstr[] = {
    0x8a, 0xf3, 0x46, 0xe5, 0x2f, 0x8c, 0x9b, 0xe6, 0x8a, 0x58, 0xf4, 0x0d,
    0x50, 0x05, 0x70, 0x04, 0xb7, 0xd7, 0x58, 0x07, 0xaf, 0x92, 0x27, 0x32,
    0x84, 0x36, 0x96, 0xc1, 0x82, 0x90, 0xd5, 0x89,
};
static const unsigned char kat280_addin0[] = {0};
static const unsigned char kat280_addin1[] = {0};
static const unsigned char kat280_retbits[] = {
    0x23, 0x35, 0x2a, 0x39, 0x99, 0x4a, 0x03, 0xf4, 0x25, 0x06, 0xc3, 0x4d,
    0xdb, 0x8e, 0x0d, 0x19, 0x12, 0x7e, 0xbb, 0xfe, 0x82, 0x39, 0xc3, 0x4a,
    0x27, 0x11, 0xcd, 0xf1, 0xb1, 0xbe, 0xb1, 0xcf, 0x75, 0x40, 0x2a, 0x61,
    0x3c, 0x85, 0x31, 0xd1, 0xdc, 0xce, 0xac, 0xdb, 0x49, 0x00, 0x73, 0xc7,
    0xa5, 0x6f, 0x0a, 0xff, 0xb3, 0xf1, 0x1f, 0x26, 0xe8, 0xc1, 0x90, 0x1f,
    0xaf, 0x3d, 0xfb, 0x2f,
};
static const struct drbg_kat_no_reseed kat280_t = {
    9, kat280_entropyin, kat280_nonce, kat280_persstr,
    kat280_addin0, kat280_addin1, kat280_retbits
};
static const struct drbg_kat kat280 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat280_t
};

static const unsigned char kat281_entropyin[] = {
    0x23, 0x93, 0x1f, 0x7e, 0x10, 0xad, 0x6e, 0x55, 0x3a, 0x28, 0x54, 0x81,
    0xac, 0xae, 0x7c, 0xfc, 0xb3, 0x2f, 0x64, 0x4e, 0x08, 0xc5, 0xb0, 0x09,
};
static const unsigned char kat281_nonce[] = {
    0x7b, 0xc2, 0xb5, 0x7e, 0xde, 0x91, 0x35, 0xb4, 0x19, 0x2c, 0x51, 0x81,
    0xf9, 0x08, 0x08, 0xc6,
};
static const unsigned char kat281_persstr[] = {
    0x6e, 0xcd, 0x84, 0xef, 0x10, 0xb4, 0xe8, 0x62, 0xbc, 0x34, 0x47, 0x86,
    0x7b, 0x3e, 0x80, 0x88, 0xe9, 0xdb, 0x4c, 0x5b, 0xe5, 0x38, 0x1d, 0xbc,
    0xb6, 0x05, 0x14, 0x05, 0x13, 0x90, 0xbf, 0xaf,
};
static const unsigned char kat281_addin0[] = {0};
static const unsigned char kat281_addin1[] = {0};
static const unsigned char kat281_retbits[] = {
    0xa5, 0xc1, 0x15, 0xbf, 0x8f, 0x95, 0x0b, 0x5d, 0x18, 0x71, 0x8a, 0x9a,
    0x22, 0x69, 0x75, 0x9e, 0x78, 0xe4, 0xfb, 0x79, 0x89, 0x37, 0xaf, 0xd0,
    0x6d, 0xaf, 0xc9, 0xea, 0xc9, 0xf5, 0x72, 0x60, 0x27, 0xd6, 0x39, 0xde,
    0x08, 0xd0, 0xb7, 0xb0, 0x08, 0x53, 0x2f, 0x70, 0xbc, 0x48, 0xa8, 0xf8,
    0x8f, 0x63, 0x7e, 0x67, 0xca, 0x7f, 0xb3, 0xfc, 0xe5, 0x19, 0x6e, 0x99,
    0x3f, 0x6d, 0x3a, 0x25,
};
static const struct drbg_kat_no_reseed kat281_t = {
    10, kat281_entropyin, kat281_nonce, kat281_persstr,
    kat281_addin0, kat281_addin1, kat281_retbits
};
static const struct drbg_kat kat281 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat281_t
};

static const unsigned char kat282_entropyin[] = {
    0x61, 0x96, 0x42, 0xe8, 0x63, 0xa8, 0xf8, 0xd2, 0xf7, 0xdb, 0x9c, 0x2e,
    0xe5, 0x6c, 0x13, 0xa2, 0xf0, 0x39, 0x59, 0x5c, 0x29, 0xed, 0x74, 0x96,
};
static const unsigned char kat282_nonce[] = {
    0x9c, 0x50, 0xab, 0x67, 0xfc, 0x76, 0x68, 0xde, 0xc6, 0x03, 0x4e, 0x82,
    0x24, 0xae, 0x1d, 0x54,
};
static const unsigned char kat282_persstr[] = {
    0x23, 0x0b, 0x18, 0x7d, 0x67, 0xc9, 0x31, 0x2e, 0xaf, 0x19, 0x23, 0x52,
    0x6b, 0xfb, 0xcb, 0x31, 0x9d, 0x9d, 0x33, 0x9f, 0xe8, 0xc8, 0x62, 0xd1,
    0xa9, 0xfa, 0x1e, 0xa7, 0x83, 0x09, 0x29, 0xb2,
};
static const unsigned char kat282_addin0[] = {0};
static const unsigned char kat282_addin1[] = {0};
static const unsigned char kat282_retbits[] = {
    0x0e, 0xce, 0x18, 0x8d, 0x8d, 0x1a, 0x37, 0xf1, 0x58, 0xa6, 0xd0, 0x65,
    0xf9, 0x07, 0x37, 0x69, 0xd7, 0x50, 0xb6, 0xd3, 0x3b, 0xf8, 0x49, 0x3d,
    0xf9, 0x6d, 0x99, 0xfa, 0x98, 0xc9, 0x90, 0x00, 0x76, 0xf7, 0xab, 0xeb,
    0x02, 0x31, 0x2a, 0xd4, 0xe0, 0xc6, 0xed, 0xde, 0x99, 0xeb, 0xd6, 0x1c,
    0x39, 0x6a, 0x83, 0x75, 0xa1, 0xd7, 0x1f, 0x6b, 0x20, 0x86, 0xd9, 0x02,
    0x1c, 0x11, 0xa1, 0x4c,
};
static const struct drbg_kat_no_reseed kat282_t = {
    11, kat282_entropyin, kat282_nonce, kat282_persstr,
    kat282_addin0, kat282_addin1, kat282_retbits
};
static const struct drbg_kat kat282 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat282_t
};

static const unsigned char kat283_entropyin[] = {
    0xbf, 0x4e, 0x56, 0x6b, 0x60, 0xa5, 0x92, 0xf6, 0xc8, 0x89, 0xbb, 0x19,
    0x25, 0x2b, 0x54, 0x48, 0x9a, 0x58, 0x15, 0xf2, 0xbd, 0x07, 0x4b, 0x41,
};
static const unsigned char kat283_nonce[] = {
    0x32, 0xa2, 0xf8, 0xe7, 0x8e, 0x17, 0x57, 0x4d, 0x63, 0x84, 0xe1, 0xf5,
    0x39, 0x95, 0x9a, 0xda,
};
static const unsigned char kat283_persstr[] = {
    0xcf, 0x75, 0x8d, 0x77, 0x29, 0x63, 0xf5, 0x76, 0xb5, 0x1d, 0x32, 0x7d,
    0xba, 0xa3, 0x51, 0x76, 0x13, 0xf3, 0x38, 0x7c, 0xb9, 0x0c, 0x85, 0x49,
    0x7a, 0x93, 0x60, 0xe0, 0x4e, 0xbf, 0xb1, 0x79,
};
static const unsigned char kat283_addin0[] = {0};
static const unsigned char kat283_addin1[] = {0};
static const unsigned char kat283_retbits[] = {
    0xf6, 0xbf, 0x71, 0x1d, 0x50, 0x02, 0x99, 0x7a, 0x3e, 0xca, 0xa3, 0x6c,
    0xeb, 0x5e, 0xe6, 0x3c, 0xae, 0x53, 0x01, 0x72, 0x89, 0x07, 0x64, 0xae,
    0xb8, 0xaf, 0xd5, 0xd8, 0x12, 0xc3, 0x68, 0x56, 0x8e, 0x4a, 0xb0, 0xaf,
    0xd2, 0xdd, 0xf9, 0x7d, 0xc3, 0x10, 0xb1, 0xc3, 0x53, 0xe6, 0x73, 0xfd,
    0xac, 0x59, 0x2a, 0xc6, 0x08, 0xe1, 0x30, 0x4f, 0x97, 0xa5, 0xf2, 0x57,
    0x8c, 0x1e, 0xc1, 0xd4,
};
static const struct drbg_kat_no_reseed kat283_t = {
    12, kat283_entropyin, kat283_nonce, kat283_persstr,
    kat283_addin0, kat283_addin1, kat283_retbits
};
static const struct drbg_kat kat283 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat283_t
};

static const unsigned char kat284_entropyin[] = {
    0x18, 0x01, 0x0f, 0xfb, 0xae, 0xac, 0xab, 0x06, 0xa4, 0xc5, 0x5a, 0x38,
    0xe8, 0xe9, 0x36, 0xd7, 0x9a, 0xd4, 0x9a, 0xf9, 0x76, 0xb8, 0xf7, 0x48,
};
static const unsigned char kat284_nonce[] = {
    0xcf, 0x55, 0xbc, 0x78, 0xa1, 0xf1, 0xee, 0x4a, 0xbd, 0xc5, 0x54, 0xe8,
    0x7b, 0x3f, 0x11, 0x4d,
};
static const unsigned char kat284_persstr[] = {
    0x53, 0xa9, 0x74, 0x38, 0x6b, 0xa1, 0x01, 0xba, 0x34, 0xea, 0x65, 0xb7,
    0x5b, 0xc3, 0x60, 0xe9, 0xc1, 0xcb, 0x80, 0xc2, 0xa6, 0x50, 0x82, 0x90,
    0xa7, 0x8a, 0xe2, 0x3e, 0x5c, 0x70, 0x15, 0x37,
};
static const unsigned char kat284_addin0[] = {0};
static const unsigned char kat284_addin1[] = {0};
static const unsigned char kat284_retbits[] = {
    0x57, 0x26, 0xde, 0x68, 0x48, 0xee, 0x07, 0x4f, 0x35, 0x3c, 0xf1, 0x7b,
    0xf3, 0xee, 0x0b, 0x09, 0xb6, 0xb6, 0x03, 0xb1, 0x24, 0x64, 0x25, 0xdd,
    0xde, 0x1d, 0x01, 0xb5, 0xbf, 0x5a, 0xf0, 0xd8, 0x88, 0x88, 0x21, 0x7d,
    0x59, 0x01, 0x8d, 0xb3, 0x17, 0x18, 0x11, 0xda, 0x02, 0xe6, 0x67, 0xfd,
    0xdf, 0x8c, 0xb3, 0x51, 0x00, 0x36, 0x9d, 0xc9, 0xa8, 0xcf, 0x2a, 0xaf,
    0xc5, 0x45, 0x21, 0x82,
};
static const struct drbg_kat_no_reseed kat284_t = {
    13, kat284_entropyin, kat284_nonce, kat284_persstr,
    kat284_addin0, kat284_addin1, kat284_retbits
};
static const struct drbg_kat kat284 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat284_t
};

static const unsigned char kat285_entropyin[] = {
    0xf6, 0x20, 0x87, 0x73, 0xae, 0x06, 0xc6, 0x14, 0x10, 0x7a, 0x98, 0x65,
    0xc3, 0x6e, 0xeb, 0x2a, 0xc2, 0xe3, 0x48, 0xee, 0x7c, 0xd6, 0x9d, 0xde,
};
static const unsigned char kat285_nonce[] = {
    0xee, 0x7c, 0x9e, 0xc2, 0x68, 0x77, 0x03, 0xaa, 0x10, 0x30, 0x70, 0x57,
    0x4d, 0x61, 0x67, 0xd9,
};
static const unsigned char kat285_persstr[] = {
    0xa8, 0x5a, 0x25, 0xe5, 0xcd, 0x63, 0x90, 0xbe, 0xab, 0x64, 0xab, 0x8b,
    0x4f, 0x53, 0x42, 0x80, 0xf3, 0xfc, 0x7b, 0x16, 0x9e, 0xb0, 0xa7, 0x5c,
    0xb7, 0x7d, 0x9b, 0xf0, 0x62, 0x92, 0xab, 0xe8,
};
static const unsigned char kat285_addin0[] = {0};
static const unsigned char kat285_addin1[] = {0};
static const unsigned char kat285_retbits[] = {
    0xcd, 0x5f, 0x73, 0xd4, 0xfe, 0xbc, 0x7f, 0xe1, 0x73, 0x51, 0xbb, 0xb9,
    0x09, 0x47, 0x29, 0x75, 0x90, 0x16, 0xbd, 0x3e, 0xae, 0x0d, 0xd3, 0xd9,
    0x12, 0x86, 0x97, 0x81, 0x3a, 0x0b, 0x92, 0x9d, 0xcf, 0xce, 0x6b, 0xfe,
    0xfc, 0x1e, 0x08, 0xde, 0xdd, 0xf6, 0x17, 0xd4, 0xe7, 0x27, 0xaa, 0xa3,
    0xc7, 0xcb, 0x1f, 0xf2, 0x3b, 0xf0, 0x2f, 0xcd, 0xa7, 0x7d, 0x0b, 0x15,
    0x02, 0x39, 0x03, 0x49,
};
static const struct drbg_kat_no_reseed kat285_t = {
    14, kat285_entropyin, kat285_nonce, kat285_persstr,
    kat285_addin0, kat285_addin1, kat285_retbits
};
static const struct drbg_kat kat285 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat285_t
};

static const unsigned char kat286_entropyin[] = {
    0xd5, 0x97, 0x3b, 0x5c, 0x91, 0x05, 0xcb, 0xf6, 0x7e, 0x97, 0x8f, 0x41,
    0x99, 0x24, 0x79, 0x0d, 0x83, 0x02, 0x3e, 0x86, 0xa8, 0xb5, 0xdd, 0x6b,
};
static const unsigned char kat286_nonce[] = {
    0x35, 0x8a, 0xf1, 0xae, 0x9a, 0x84, 0x2c, 0x6e, 0x03, 0xf8, 0x8d, 0xfa,
    0x2a, 0x31, 0x11, 0x61,
};
static const unsigned char kat286_persstr[] = {
    0x29, 0x4d, 0x7d, 0x35, 0xf5, 0x3a, 0x5d, 0x7d, 0xde, 0xf5, 0xca, 0x41,
    0x00, 0xf3, 0x54, 0x71, 0x12, 0xc9, 0x3e, 0x41, 0x25, 0x12, 0x57, 0xdc,
    0x0a, 0x19, 0xb6, 0xdf, 0xaa, 0x4a, 0x60, 0xa4,
};
static const unsigned char kat286_addin0[] = {
    0x08, 0x05, 0xf3, 0x14, 0x46, 0xc5, 0x1d, 0x5d, 0x9d, 0x27, 0xb7, 0xcb,
    0xb1, 0x6e, 0x84, 0x0b, 0x9e, 0x8b, 0x0d, 0xfe, 0x6f, 0xb4, 0xb6, 0x97,
    0x92, 0xbc, 0x8d, 0xe9, 0xe3, 0xbd, 0x6d, 0x92,
};
static const unsigned char kat286_addin1[] = {
    0x93, 0x4d, 0x7f, 0xd5, 0xe7, 0x16, 0x37, 0x63, 0x42, 0x60, 0x71, 0x23,
    0xea, 0x11, 0x3d, 0x6b, 0x20, 0x17, 0x0c, 0xcd, 0xa5, 0x3f, 0xc8, 0x65,
    0x41, 0x40, 0x7a, 0x15, 0x6c, 0xd9, 0x49, 0x04,
};
static const unsigned char kat286_retbits[] = {
    0xcb, 0x95, 0x45, 0x9d, 0x17, 0x35, 0xcb, 0x9b, 0xce, 0x8a, 0x75, 0xbf,
    0x09, 0x7a, 0x09, 0x9c, 0x9f, 0x7c, 0x70, 0xba, 0xd4, 0x3e, 0x3e, 0x43,
    0x1f, 0x2d, 0x38, 0x29, 0xd7, 0xca, 0x9d, 0x06, 0x17, 0xb9, 0xa9, 0x93,
    0x37, 0xaf, 0x52, 0x48, 0xd4, 0x74, 0x1c, 0xb5, 0xa6, 0x0d, 0xff, 0x6f,
    0x8c, 0x52, 0x21, 0xe2, 0x3f, 0x3c, 0xb5, 0x24, 0xa9, 0x4f, 0xfd, 0xd2,
    0x19, 0x0b, 0xfb, 0x3b,
};
static const struct drbg_kat_no_reseed kat286_t = {
    0, kat286_entropyin, kat286_nonce, kat286_persstr,
    kat286_addin0, kat286_addin1, kat286_retbits
};
static const struct drbg_kat kat286 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat286_t
};

static const unsigned char kat287_entropyin[] = {
    0xa0, 0x44, 0x99, 0xcc, 0x2b, 0x36, 0x20, 0xa4, 0x7e, 0xe6, 0x6d, 0xe9,
    0xce, 0x65, 0x10, 0x03, 0x39, 0x40, 0x95, 0x85, 0x39, 0xf7, 0x54, 0xfb,
};
static const unsigned char kat287_nonce[] = {
    0xfa, 0xa4, 0x92, 0xb5, 0xee, 0xb6, 0x20, 0xe7, 0x8f, 0xb6, 0x37, 0x5c,
    0x62, 0x0f, 0x49, 0x5c,
};
static const unsigned char kat287_persstr[] = {
    0x31, 0x8a, 0x01, 0xb4, 0x75, 0x60, 0x11, 0x91, 0x43, 0x8c, 0xcf, 0x44,
    0xfe, 0xd9, 0xc0, 0xc6, 0xaf, 0x5f, 0x44, 0x15, 0xc2, 0x66, 0x8b, 0x2d,
    0x39, 0xf9, 0x59, 0xef, 0x6f, 0xa2, 0xe2, 0xee,
};
static const unsigned char kat287_addin0[] = {
    0xf1, 0xb9, 0x2d, 0x7f, 0xb1, 0x26, 0x79, 0x9f, 0x08, 0x02, 0x64, 0xbb,
    0xd2, 0xef, 0xf3, 0x9b, 0xd7, 0x55, 0xb8, 0xd1, 0xcb, 0xb8, 0x79, 0x63,
    0xa7, 0x71, 0xe7, 0xac, 0x54, 0x94, 0xc1, 0x54,
};
static const unsigned char kat287_addin1[] = {
    0xab, 0x79, 0x75, 0xe3, 0x42, 0xc2, 0x60, 0xf4, 0x01, 0x38, 0x56, 0xe5,
    0x9d, 0x76, 0x77, 0xe7, 0x07, 0x18, 0x21, 0xe4, 0x2d, 0xcb, 0x0b, 0x14,
    0x7d, 0xcc, 0x74, 0x07, 0x48, 0x32, 0x06, 0x1c,
};
static const unsigned char kat287_retbits[] = {
    0x64, 0x7a, 0x3e, 0x82, 0xbe, 0x71, 0x73, 0xe4, 0x57, 0x82, 0xd4, 0x2e,
    0xe0, 0xbc, 0x52, 0xff, 0xef, 0xc3, 0x07, 0x2e, 0xca, 0xb3, 0x06, 0x0a,
    0xe6, 0x06, 0x31, 0xb8, 0x48, 0x62, 0x89, 0x23, 0x0c, 0x00, 0xb3, 0xe9,
    0x64, 0x1f, 0x88, 0xd3, 0x42, 0x7f, 0xbf, 0xb1, 0x50, 0xfd, 0x14, 0xdc,
    0xf8, 0xf2, 0xc8, 0xdb, 0x14, 0xc7, 0x15, 0x45, 0x78, 0x89, 0x51, 0xef,
    0xa0, 0x74, 0xb2, 0x27,
};
static const struct drbg_kat_no_reseed kat287_t = {
    1, kat287_entropyin, kat287_nonce, kat287_persstr,
    kat287_addin0, kat287_addin1, kat287_retbits
};
static const struct drbg_kat kat287 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat287_t
};

static const unsigned char kat288_entropyin[] = {
    0x56, 0x82, 0x91, 0x4c, 0x36, 0x77, 0x69, 0x45, 0xe0, 0x23, 0xf9, 0x8e,
    0x14, 0x17, 0x17, 0x25, 0xa2, 0xc0, 0x20, 0x4d, 0xe7, 0x99, 0xa3, 0x6f,
};
static const unsigned char kat288_nonce[] = {
    0x6d, 0x6f, 0x71, 0xcf, 0xef, 0x95, 0x7f, 0xf4, 0x75, 0x58, 0x33, 0x15,
    0x12, 0x5e, 0xad, 0x6b,
};
static const unsigned char kat288_persstr[] = {
    0x55, 0x7f, 0xde, 0x94, 0xd4, 0x1c, 0x13, 0xb2, 0x48, 0x29, 0xb8, 0xfd,
    0x42, 0x41, 0x24, 0x82, 0x5a, 0xac, 0xb1, 0x40, 0x94, 0xe5, 0xe7, 0xd4,
    0xf8, 0xde, 0xea, 0x7d, 0xb2, 0xac, 0x29, 0x40,
};
static const unsigned char kat288_addin0[] = {
    0x4c, 0x64, 0xc5, 0xa5, 0xc1, 0xc6, 0x91, 0x4b, 0x61, 0xfa, 0xf2, 0x64,
    0x92, 0x4c, 0x10, 0xde, 0xbc, 0x2f, 0x36, 0x7a, 0xc4, 0xcb, 0x1e, 0x01,
    0x24, 0x88, 0x32, 0x2f, 0xb1, 0x9d, 0x69, 0xa8,
};
static const unsigned char kat288_addin1[] = {
    0x71, 0x93, 0xe6, 0x1b, 0xb3, 0xe7, 0x5d, 0x48, 0xe4, 0xa3, 0xf9, 0x0b,
    0x10, 0x8e, 0x07, 0x07, 0x4c, 0x35, 0x39, 0x06, 0x93, 0xcd, 0x0a, 0x4c,
    0xd3, 0xbb, 0x6c, 0xd2, 0x46, 0xbe, 0xb7, 0x40,
};
static const unsigned char kat288_retbits[] = {
    0x7a, 0xed, 0x93, 0x1e, 0xe1, 0x7c, 0x7b, 0x05, 0x53, 0x3b, 0x52, 0xf6,
    0xb3, 0xdf, 0xc9, 0x76, 0x88, 0xf6, 0xf3, 0xb3, 0x97, 0x44, 0xed, 0xed,
    0x73, 0xdd, 0xe2, 0x7e, 0x03, 0x4a, 0xfb, 0x18, 0x01, 0xb3, 0xd1, 0x16,
    0x0d, 0xba, 0x79, 0x75, 0x59, 0xfd, 0x72, 0x16, 0x10, 0xf6, 0xd6, 0x2f,
    0x6d, 0x0c, 0x48, 0xa6, 0x74, 0x03, 0x07, 0x5f, 0x6c, 0x24, 0x50, 0xe6,
    0x24, 0x6a, 0x4d, 0x66,
};
static const struct drbg_kat_no_reseed kat288_t = {
    2, kat288_entropyin, kat288_nonce, kat288_persstr,
    kat288_addin0, kat288_addin1, kat288_retbits
};
static const struct drbg_kat kat288 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat288_t
};

static const unsigned char kat289_entropyin[] = {
    0x19, 0x9c, 0xee, 0x8a, 0x88, 0x18, 0x88, 0x16, 0x6c, 0xb5, 0x2c, 0xf6,
    0x8f, 0xde, 0x78, 0x0f, 0x32, 0x55, 0xb1, 0x3d, 0x37, 0x0b, 0x68, 0x9a,
};
static const unsigned char kat289_nonce[] = {
    0x84, 0x80, 0x16, 0xaa, 0x71, 0x57, 0x1e, 0x1c, 0x63, 0xf3, 0x17, 0x4b,
    0xf9, 0x35, 0x87, 0xc9,
};
static const unsigned char kat289_persstr[] = {
    0x66, 0x21, 0x2f, 0x4c, 0x19, 0xb5, 0xcd, 0xf4, 0xa6, 0x59, 0x4f, 0xd4,
    0xc4, 0xf7, 0xb7, 0x83, 0x72, 0x30, 0xde, 0x71, 0xb9, 0x8d, 0x50, 0x6d,
    0x46, 0xde, 0xce, 0xac, 0xb5, 0xf0, 0xc6, 0x9c,
};
static const unsigned char kat289_addin0[] = {
    0x59, 0x34, 0xb3, 0x95, 0x8e, 0xbe, 0x21, 0x54, 0xc2, 0x46, 0xce, 0xa5,
    0x5d, 0xf2, 0x31, 0xe4, 0x1a, 0x86, 0x7e, 0x88, 0x44, 0x12, 0x9f, 0xd6,
    0xc7, 0xde, 0x6c, 0x9c, 0xf8, 0xd4, 0xd7, 0xb8,
};
static const unsigned char kat289_addin1[] = {
    0x0b, 0x4a, 0x32, 0xba, 0xb8, 0x42, 0xc4, 0xd2, 0x49, 0x99, 0xe9, 0xac,
    0x6b, 0xb1, 0x02, 0x4b, 0x76, 0x73, 0x08, 0x68, 0x95, 0xe8, 0x9a, 0x70,
    0x72, 0xa5, 0x9c, 0x8c, 0x75, 0xfe, 0x3e, 0x0a,
};
static const unsigned char kat289_retbits[] = {
    0x84, 0x50, 0x41, 0x0e, 0x9a, 0x7f, 0xf9, 0xa1, 0x04, 0xd6, 0xd0, 0xec,
    0xd2, 0x31, 0x5e, 0xa5, 0x40, 0xf5, 0x4c, 0xd8, 0x8d, 0x21, 0xac, 0x6d,
    0x7c, 0xae, 0x8a, 0x86, 0x18, 0x1d, 0xcb, 0x53, 0xf1, 0x88, 0x3e, 0xe1,
    0xdc, 0xe8, 0xa2, 0x86, 0x22, 0x61, 0x62, 0x08, 0x59, 0x33, 0x4f, 0xde,
    0x9c, 0xe2, 0xde, 0xaf, 0x1b, 0x58, 0xd4, 0xd4, 0xd0, 0x1d, 0xbd, 0xfe,
    0xe7, 0x73, 0x2e, 0x8e,
};
static const struct drbg_kat_no_reseed kat289_t = {
    3, kat289_entropyin, kat289_nonce, kat289_persstr,
    kat289_addin0, kat289_addin1, kat289_retbits
};
static const struct drbg_kat kat289 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat289_t
};

static const unsigned char kat290_entropyin[] = {
    0x2f, 0xf3, 0xc6, 0x29, 0x0f, 0x51, 0x40, 0xb6, 0xc4, 0x96, 0x8a, 0xfe,
    0x0c, 0x6a, 0xc1, 0x8e, 0x55, 0x74, 0xc7, 0x11, 0x73, 0x29, 0x2e, 0x54,
};
static const unsigned char kat290_nonce[] = {
    0x93, 0xdd, 0x9d, 0xfd, 0xb4, 0x5f, 0x96, 0xb8, 0x05, 0x9c, 0xb3, 0xfd,
    0xc1, 0x25, 0xa3, 0xb5,
};
static const unsigned char kat290_persstr[] = {
    0xdb, 0xf6, 0x83, 0x70, 0x58, 0xa3, 0x74, 0x1a, 0xbf, 0x3a, 0xd0, 0x62,
    0x97, 0xc1, 0x9b, 0x81, 0xf4, 0xc4, 0x6a, 0xdf, 0x7d, 0x46, 0xc5, 0x8b,
    0xbb, 0xce, 0x5b, 0xcf, 0x29, 0x64, 0x00, 0xd5,
};
static const unsigned char kat290_addin0[] = {
    0x32, 0x9d, 0x4f, 0xe2, 0x73, 0x66, 0x16, 0xb0, 0x65, 0x28, 0x7f, 0x3c,
    0xad, 0x21, 0x63, 0x42, 0xb9, 0xbe, 0x26, 0xe8, 0xf2, 0xc5, 0x94, 0x64,
    0x5e, 0x7e, 0x27, 0x88, 0x07, 0x09, 0x7d, 0x5c,
};
static const unsigned char kat290_addin1[] = {
    0x8a, 0x1e, 0x25, 0x8c, 0x3a, 0x3f, 0xb9, 0xb0, 0x31, 0x45, 0x62, 0xa3,
    0x20, 0xc0, 0xc1, 0x1e, 0x6f, 0x81, 0x37, 0xe2, 0x91, 0xa7, 0x9e, 0xd5,
    0x68, 0x48, 0x9a, 0xd0, 0x56, 0x0c, 0x7c, 0x28,
};
static const unsigned char kat290_retbits[] = {
    0x28, 0x1c, 0xd7, 0x62, 0xf1, 0x0d, 0xdd, 0x29, 0xe0, 0xab, 0x11, 0xcf,
    0x79, 0x49, 0x4a, 0x7e, 0xb5, 0x92, 0xdd, 0x84, 0xdc, 0xe6, 0x58, 0x04,
    0x28, 0x6e, 0xea, 0x86, 0x4d, 0x8b, 0x85, 0x44, 0x28, 0x2d, 0x3f, 0xe8,
    0x0d, 0x76, 0xfe, 0x55, 0xf7, 0xfc, 0x66, 0xd0, 0x97, 0x1b, 0x8d, 0xcb,
    0x84, 0xa3, 0xf2, 0x5d, 0x85, 0x97, 0x21, 0x33, 0xe0, 0x74, 0xcd, 0xf1,
    0xe6, 0x0b, 0xf0, 0x69,
};
static const struct drbg_kat_no_reseed kat290_t = {
    4, kat290_entropyin, kat290_nonce, kat290_persstr,
    kat290_addin0, kat290_addin1, kat290_retbits
};
static const struct drbg_kat kat290 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat290_t
};

static const unsigned char kat291_entropyin[] = {
    0x15, 0xb6, 0x96, 0xec, 0xc0, 0x14, 0x9b, 0x78, 0x05, 0xbd, 0xbe, 0xaa,
    0x65, 0xb9, 0xea, 0x9b, 0x76, 0xe7, 0x6d, 0xd6, 0x8b, 0xb9, 0xba, 0x5f,
};
static const unsigned char kat291_nonce[] = {
    0xee, 0x41, 0x74, 0xe0, 0x87, 0xf3, 0xe7, 0x4a, 0x3f, 0xd2, 0xfc, 0x40,
    0x35, 0x59, 0x20, 0x9c,
};
static const unsigned char kat291_persstr[] = {
    0xa4, 0x2d, 0xd8, 0x8c, 0xdb, 0xa2, 0x5d, 0x88, 0x9d, 0xbc, 0xd9, 0xc7,
    0xb7, 0xa7, 0xbc, 0xfe, 0x25, 0xad, 0x1a, 0x81, 0xa1, 0x76, 0x1e, 0x24,
    0xe0, 0x46, 0xca, 0xd4, 0x75, 0x91, 0x02, 0x5c,
};
static const unsigned char kat291_addin0[] = {
    0xa3, 0x73, 0xc6, 0x7b, 0xf7, 0xa8, 0x78, 0x98, 0x70, 0x76, 0x8a, 0xf6,
    0x01, 0x5a, 0x13, 0x39, 0x36, 0x82, 0x1d, 0x97, 0xcf, 0xe1, 0xdc, 0x49,
    0x73, 0x1b, 0x14, 0x5f, 0x66, 0x66, 0xdf, 0xe3,
};
static const unsigned char kat291_addin1[] = {
    0x7b, 0x93, 0xdd, 0x28, 0x55, 0xff, 0x93, 0xec, 0x1d, 0x19, 0x4c, 0x86,
    0x71, 0xa6, 0x25, 0x99, 0x39, 0xbc, 0x4a, 0x57, 0x29, 0x94, 0xae, 0x3d,
    0x52, 0xa8, 0x44, 0x11, 0x74, 0x90, 0x53, 0x3a,
};
static const unsigned char kat291_retbits[] = {
    0x78, 0xca, 0x84, 0xfe, 0x42, 0xb1, 0x6e, 0xab, 0x52, 0x88, 0x8d, 0x54,
    0x0c, 0x50, 0xe5, 0x21, 0x61, 0x5c, 0x5f, 0x4d, 0xff, 0x04, 0x56, 0xcb,
    0x80, 0x8d, 0x10, 0xf6, 0x66, 0xd0, 0x8d, 0xd1, 0x9c, 0x68, 0x78, 0x9b,
    0xe1, 0xd9, 0xd8, 0x55, 0xff, 0x99, 0x5d, 0xeb, 0x8e, 0x02, 0x61, 0xe8,
    0xc4, 0x4b, 0x12, 0x48, 0xb0, 0xb2, 0xde, 0xc2, 0x59, 0x9d, 0x8e, 0x77,
    0x9d, 0x24, 0xc8, 0x39,
};
static const struct drbg_kat_no_reseed kat291_t = {
    5, kat291_entropyin, kat291_nonce, kat291_persstr,
    kat291_addin0, kat291_addin1, kat291_retbits
};
static const struct drbg_kat kat291 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat291_t
};

static const unsigned char kat292_entropyin[] = {
    0xdd, 0xc6, 0xe5, 0x0f, 0xf9, 0x08, 0x3f, 0xed, 0xf4, 0xa5, 0xd5, 0x6f,
    0xd9, 0x23, 0x8a, 0x25, 0x65, 0xf5, 0x01, 0x84, 0xdf, 0x28, 0x3a, 0x78,
};
static const unsigned char kat292_nonce[] = {
    0xa3, 0x06, 0x0e, 0x17, 0x32, 0xf1, 0xb1, 0xde, 0xfc, 0x8e, 0x5f, 0x39,
    0xe0, 0x12, 0xe3, 0x03,
};
static const unsigned char kat292_persstr[] = {
    0x86, 0x98, 0x83, 0x57, 0x95, 0x80, 0x4b, 0xa3, 0x46, 0x5f, 0x3e, 0xcd,
    0xd4, 0xac, 0xe0, 0x47, 0x50, 0x6c, 0xb1, 0x5d, 0xd9, 0xc8, 0x72, 0x8f,
    0x07, 0x5f, 0xce, 0xa5, 0x03, 0x3f, 0xa0, 0xd8,
};
static const unsigned char kat292_addin0[] = {
    0xb1, 0x01, 0x1f, 0x40, 0xcf, 0x33, 0x1e, 0x33, 0xd2, 0x4f, 0xfe, 0x86,
    0x17, 0x8f, 0x70, 0xe6, 0xc7, 0xd4, 0xd2, 0xb4, 0x43, 0xb6, 0x4f, 0xbd,
    0x5c, 0x97, 0x9b, 0x94, 0x19, 0x3c, 0xac, 0x60,
};
static const unsigned char kat292_addin1[] = {
    0x3a, 0x72, 0xe1, 0xe6, 0x63, 0x64, 0x04, 0xbe, 0x69, 0xf8, 0x53, 0xbe,
    0xb0, 0x63, 0x3d, 0x5d, 0xef, 0x03, 0xb1, 0x6b, 0x4b, 0x20, 0x5a, 0x02,
    0x38, 0x4c, 0x26, 0x95, 0x1c, 0xa0, 0xf2, 0x20,
};
static const unsigned char kat292_retbits[] = {
    0x7d, 0xbe, 0xac, 0xe1, 0xa9, 0x0a, 0x28, 0x43, 0xc3, 0xde, 0x36, 0xf3,
    0xbb, 0xac, 0x6e, 0x91, 0x40, 0xa9, 0x38, 0xce, 0xd6, 0x30, 0x41, 0x6e,
    0xb1, 0xe1, 0x63, 0x87, 0x08, 0xec, 0x12, 0xe4, 0x80, 0xda, 0x65, 0xe0,
    0x0c, 0x82, 0x94, 0x15, 0x1e, 0xa4, 0xb9, 0xa1, 0xd5, 0xb2, 0x59, 0x75,
    0xd7, 0x94, 0x0d, 0x57, 0xc8, 0xde, 0x72, 0xe2, 0x03, 0x82, 0x99, 0x6f,
    0xf4, 0xdc, 0x64, 0xc2,
};
static const struct drbg_kat_no_reseed kat292_t = {
    6, kat292_entropyin, kat292_nonce, kat292_persstr,
    kat292_addin0, kat292_addin1, kat292_retbits
};
static const struct drbg_kat kat292 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat292_t
};

static const unsigned char kat293_entropyin[] = {
    0xdd, 0xe2, 0x70, 0x18, 0xcd, 0xea, 0x02, 0x7b, 0x27, 0x3a, 0x83, 0xab,
    0x28, 0xf8, 0x18, 0x22, 0x93, 0xe5, 0x37, 0x32, 0x91, 0x64, 0x12, 0x62,
};
static const unsigned char kat293_nonce[] = {
    0xe8, 0x81, 0x0e, 0x24, 0xfb, 0x96, 0xba, 0x67, 0x68, 0xb6, 0x57, 0x47,
    0xbb, 0x27, 0x93, 0xb8,
};
static const unsigned char kat293_persstr[] = {
    0xd8, 0xad, 0xa0, 0x62, 0xfb, 0x8a, 0x56, 0x7f, 0x42, 0x15, 0xeb, 0x1f,
    0xf8, 0x7a, 0x2c, 0xd6, 0xfe, 0x96, 0x4d, 0x7b, 0xeb, 0xcc, 0x9c, 0xf4,
    0xec, 0xc5, 0x31, 0xbe, 0x21, 0xad, 0x47, 0x8f,
};
static const unsigned char kat293_addin0[] = {
    0xee, 0xa0, 0xc6, 0x51, 0xaf, 0xae, 0x1c, 0xca, 0x7d, 0x6b, 0x4a, 0xef,
    0x4c, 0x1e, 0x0c, 0x22, 0x19, 0x4c, 0x70, 0xa6, 0x3f, 0xe4, 0x74, 0xa8,
    0xb5, 0xfc, 0xbd, 0x26, 0xa5, 0x69, 0x6c, 0x3c,
};
static const unsigned char kat293_addin1[] = {
    0x71, 0xee, 0xe7, 0x0d, 0x6b, 0x57, 0x85, 0x4d, 0xa6, 0x87, 0x89, 0xd9,
    0x92, 0xe4, 0x13, 0x24, 0x85, 0xa9, 0xaa, 0xcc, 0x3f, 0xb8, 0xfa, 0xab,
    0xc9, 0x08, 0xed, 0xda, 0x65, 0x3b, 0xf7, 0x40,
};
static const unsigned char kat293_retbits[] = {
    0xce, 0x2f, 0x8c, 0x2a, 0xcc, 0xc4, 0xc4, 0x5f, 0x18, 0x3b, 0xfe, 0xbe,
    0xcf, 0x1a, 0xdd, 0x4f, 0x32, 0xed, 0x36, 0x0d, 0x35, 0x6e, 0xfb, 0x90,
    0x63, 0x15, 0xc0, 0x43, 0xd1, 0x50, 0x66, 0x0a, 0x35, 0xb5, 0xd0, 0x35,
    0x0c, 0xb0, 0x53, 0xd4, 0x58, 0x16, 0x7e, 0xac, 0xf3, 0x5c, 0xc7, 0x01,
    0x2b, 0xdd, 0x1e, 0xf7, 0x08, 0x18, 0xde, 0xdd, 0xf3, 0x8b, 0x4e, 0x85,
    0x57, 0xbc, 0xc0, 0x3e,
};
static const struct drbg_kat_no_reseed kat293_t = {
    7, kat293_entropyin, kat293_nonce, kat293_persstr,
    kat293_addin0, kat293_addin1, kat293_retbits
};
static const struct drbg_kat kat293 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat293_t
};

static const unsigned char kat294_entropyin[] = {
    0x49, 0xb2, 0xae, 0x2b, 0x29, 0x14, 0xd6, 0x88, 0x53, 0x8b, 0xb2, 0x91,
    0x93, 0xae, 0x63, 0x78, 0xe0, 0xe8, 0xb9, 0x41, 0xdd, 0x3e, 0x31, 0x52,
};
static const unsigned char kat294_nonce[] = {
    0x27, 0x9f, 0x5f, 0x18, 0x32, 0xcc, 0xd1, 0x0d, 0x08, 0x51, 0x40, 0x45,
    0x27, 0xed, 0x25, 0xfc,
};
static const unsigned char kat294_persstr[] = {
    0xee, 0x3a, 0x40, 0xee, 0x43, 0x22, 0xba, 0xb1, 0x68, 0x23, 0x59, 0x23,
    0xb5, 0x2b, 0x80, 0xa3, 0xc7, 0x11, 0xcd, 0x7a, 0xd7, 0x61, 0x8e, 0x76,
    0x40, 0x0e, 0x46, 0xf1, 0x53, 0xa1, 0x07, 0x22,
};
static const unsigned char kat294_addin0[] = {
    0xca, 0x3e, 0x55, 0xa7, 0xe1, 0xa5, 0x1d, 0x7d, 0xa3, 0x37, 0x9a, 0x2c,
    0xbc, 0x31, 0x62, 0xfe, 0xe7, 0x05, 0xff, 0xc8, 0xe2, 0xbb, 0x72, 0x94,
    0xe9, 0xf2, 0xd9, 0xc2, 0x9f, 0xe8, 0xb3, 0x86,
};
static const unsigned char kat294_addin1[] = {
    0x09, 0xac, 0xba, 0x8e, 0x05, 0x59, 0x39, 0x94, 0x24, 0x8d, 0x40, 0x66,
    0xb3, 0xd7, 0x70, 0xf4, 0x1b, 0x47, 0xab, 0x82, 0x44, 0x96, 0x8b, 0xb6,
    0x26, 0xbd, 0xfb, 0x6d, 0xb1, 0x41, 0x76, 0x0f,
};
static const unsigned char kat294_retbits[] = {
    0xaf, 0x2e, 0xb7, 0x3e, 0x49, 0xb1, 0x3c, 0x1c, 0x42, 0x62, 0x95, 0xfa,
    0x26, 0x43, 0xde, 0x08, 0xb1, 0x92, 0xac, 0x24, 0x07, 0x9b, 0x71, 0xc8,
    0xc1, 0x75, 0x50, 0x96, 0x44, 0x93, 0x7f, 0x22, 0x2d, 0xfb, 0xd6, 0xc7,
    0x7d, 0xd2, 0x5e, 0xa3, 0xea, 0xfe, 0xf7, 0x1d, 0xa2, 0xaa, 0x7b, 0xdb,
    0x68, 0xab, 0x4c, 0x8e, 0x97, 0xbd, 0xeb, 0x7c, 0x70, 0x6d, 0x99, 0xb5,
    0xb6, 0x7c, 0x6f, 0x9d,
};
static const struct drbg_kat_no_reseed kat294_t = {
    8, kat294_entropyin, kat294_nonce, kat294_persstr,
    kat294_addin0, kat294_addin1, kat294_retbits
};
static const struct drbg_kat kat294 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat294_t
};

static const unsigned char kat295_entropyin[] = {
    0xe8, 0xc5, 0x38, 0xae, 0x11, 0x61, 0x98, 0x96, 0xbd, 0x34, 0x7d, 0x67,
    0x90, 0x07, 0x66, 0x1a, 0xce, 0x7e, 0x51, 0x92, 0xb8, 0x39, 0x19, 0x42,
};
static const unsigned char kat295_nonce[] = {
    0xf8, 0x26, 0x28, 0xa0, 0x5a, 0x8d, 0xfc, 0x06, 0x68, 0xf7, 0xf7, 0x54,
    0xb1, 0x71, 0xd5, 0xdb,
};
static const unsigned char kat295_persstr[] = {
    0x2c, 0x2a, 0x7e, 0xc3, 0x15, 0x64, 0xa3, 0x4e, 0x9c, 0xfb, 0x9f, 0x0f,
    0x42, 0xd8, 0xba, 0xb6, 0x81, 0x2b, 0x4a, 0xde, 0x26, 0x3a, 0xf4, 0xf0,
    0x02, 0x29, 0xe6, 0xeb, 0x99, 0x1e, 0x25, 0x92,
};
static const unsigned char kat295_addin0[] = {
    0x8d, 0x9e, 0x18, 0x24, 0xd2, 0x77, 0x37, 0xaf, 0xc4, 0x11, 0xb2, 0x4b,
    0xe6, 0x5e, 0xac, 0x50, 0x48, 0x0f, 0xce, 0x4a, 0xfb, 0xf7, 0x31, 0x7a,
    0x24, 0xb0, 0x4e, 0x24, 0x67, 0x20, 0x6f, 0x54,
};
static const unsigned char kat295_addin1[] = {
    0x3f, 0x02, 0x6a, 0x1a, 0x2b, 0xb8, 0xd0, 0x59, 0xc4, 0x95, 0xb0, 0xd4,
    0xa4, 0xf9, 0x73, 0x2d, 0x94, 0xaa, 0xf1, 0x5d, 0xe4, 0x2b, 0xf3, 0x44,
    0x12, 0xd7, 0x27, 0x0a, 0xaf, 0xca, 0x09, 0x9b,
};
static const unsigned char kat295_retbits[] = {
    0xb3, 0x96, 0xa8, 0x56, 0x93, 0x88, 0x93, 0x2a, 0x6f, 0xd2, 0x8c, 0xab,
    0x6c, 0xfb, 0x53, 0x37, 0x57, 0x1c, 0x1c, 0x91, 0xb8, 0xde, 0xf9, 0x83,
    0x5a, 0x2d, 0xcf, 0x23, 0x53, 0x66, 0x93, 0xb6, 0xa2, 0x75, 0xab, 0xb4,
    0xd4, 0xa3, 0xa2, 0xa1, 0xf4, 0x92, 0xb6, 0xfe, 0x5b, 0xce, 0x76, 0x30,
    0xde, 0x03, 0x99, 0x4e, 0xf2, 0x34, 0xd4, 0x25, 0x5e, 0xa3, 0x38, 0xa1,
    0x3a, 0xfb, 0x8e, 0x6d,
};
static const struct drbg_kat_no_reseed kat295_t = {
    9, kat295_entropyin, kat295_nonce, kat295_persstr,
    kat295_addin0, kat295_addin1, kat295_retbits
};
static const struct drbg_kat kat295 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat295_t
};

static const unsigned char kat296_entropyin[] = {
    0x95, 0x44, 0xea, 0x65, 0x20, 0x52, 0x83, 0xf3, 0x66, 0x82, 0x4c, 0x55,
    0x60, 0x21, 0x78, 0xeb, 0x82, 0x13, 0x5d, 0x01, 0x57, 0x78, 0x8f, 0x86,
};
static const unsigned char kat296_nonce[] = {
    0x5e, 0x03, 0x79, 0xad, 0xec, 0x23, 0x87, 0xfc, 0xdb, 0xc6, 0x4d, 0x0f,
    0x9e, 0x23, 0xcc, 0xb5,
};
static const unsigned char kat296_persstr[] = {
    0xbe, 0x01, 0x6b, 0x1e, 0xf2, 0x74, 0xc7, 0x2c, 0x74, 0x97, 0x0c, 0x7f,
    0x16, 0xff, 0x9f, 0x6d, 0xd5, 0x79, 0xad, 0x73, 0x6a, 0xb8, 0x02, 0x38,
    0x5b, 0xb7, 0xc5, 0xee, 0xf7, 0x08, 0x6a, 0x95,
};
static const unsigned char kat296_addin0[] = {
    0xaa, 0xbf, 0xe4, 0x85, 0x76, 0x08, 0x9e, 0x44, 0x43, 0x1a, 0xe8, 0x94,
    0xcc, 0x31, 0x6e, 0xe9, 0x6b, 0xf7, 0xa4, 0x91, 0x89, 0xc7, 0xe4, 0x2a,
    0xbc, 0xca, 0x27, 0x87, 0x8b, 0x30, 0xef, 0x24,
};
static const unsigned char kat296_addin1[] = {
    0xb5, 0x29, 0xe2, 0xd3, 0x7b, 0x08, 0x08, 0xb1, 0xc9, 0x25, 0x34, 0x40,
    0x19, 0xdd, 0xdd, 0x3e, 0xea, 0x22, 0x77, 0x68, 0x38, 0xa0, 0x6e, 0x67,
    0x7f, 0xff, 0x67, 0x12, 0x7b, 0xb6, 0x8a, 0x6d,
};
static const unsigned char kat296_retbits[] = {
    0xab, 0x1a, 0x41, 0x84, 0x13, 0x30, 0xdf, 0x7e, 0x73, 0x4a, 0x57, 0xbe,
    0x20, 0x81, 0x71, 0x95, 0x0f, 0xa8, 0xe6, 0x4f, 0x4a, 0x2e, 0x26, 0xc7,
    0x2a, 0x10, 0xb3, 0x3d, 0xf9, 0xa7, 0x7b, 0xb2, 0x42, 0x84, 0xbf, 0xa7,
    0x22, 0x0b, 0xf8, 0xef, 0x03, 0x5e, 0x4d, 0x15, 0x84, 0x2b, 0xe5, 0x51,
    0x03, 0xec, 0x47, 0x43, 0xe2, 0xa9, 0x5b, 0xad, 0xd9, 0x48, 0x4b, 0x36,
    0x0d, 0x22, 0xe2, 0xdb,
};
static const struct drbg_kat_no_reseed kat296_t = {
    10, kat296_entropyin, kat296_nonce, kat296_persstr,
    kat296_addin0, kat296_addin1, kat296_retbits
};
static const struct drbg_kat kat296 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat296_t
};

static const unsigned char kat297_entropyin[] = {
    0x25, 0xec, 0x07, 0x42, 0x27, 0x5e, 0x6f, 0xd6, 0xd3, 0x09, 0xa0, 0x26,
    0x99, 0xb7, 0xf2, 0x1a, 0xa1, 0x83, 0x62, 0xea, 0x68, 0x3b, 0xa7, 0x32,
};
static const unsigned char kat297_nonce[] = {
    0x54, 0x24, 0xe5, 0x50, 0x26, 0xf0, 0x28, 0x36, 0x9f, 0xd9, 0x6f, 0x57,
    0x50, 0x9b, 0xfe, 0x92,
};
static const unsigned char kat297_persstr[] = {
    0xef, 0xed, 0x65, 0x1e, 0x62, 0x79, 0x63, 0xc4, 0x3f, 0xbf, 0x16, 0x4b,
    0xd7, 0xb9, 0xb3, 0xbc, 0xf1, 0x8c, 0xb7, 0x8a, 0xe9, 0x7d, 0xc4, 0x8a,
    0x85, 0xd5, 0x21, 0x2a, 0xc7, 0x1a, 0x11, 0xfc,
};
static const unsigned char kat297_addin0[] = {
    0x26, 0xb0, 0xf1, 0x02, 0x5f, 0x31, 0x83, 0x14, 0xb5, 0x5f, 0xc1, 0xa4,
    0xdd, 0x32, 0x55, 0x13, 0xc4, 0x53, 0xb6, 0x6d, 0x0b, 0x21, 0x29, 0x3f,
    0x77, 0x5c, 0xcd, 0x7d, 0x3b, 0x9f, 0x43, 0x43,
};
static const unsigned char kat297_addin1[] = {
    0x2f, 0x94, 0xd7, 0xde, 0x2c, 0xe2, 0x5d, 0x12, 0x4d, 0x9d, 0xc6, 0x8b,
    0x39, 0xf9, 0x07, 0x6e, 0xcd, 0x2f, 0xf1, 0x4f, 0xe4, 0xe1, 0xd5, 0xc1,
    0x7c, 0x6e, 0x4d, 0xf2, 0x25, 0x7f, 0xc1, 0xde,
};
static const unsigned char kat297_retbits[] = {
    0xf9, 0xfd, 0x57, 0x61, 0x04, 0xd7, 0xf6, 0x8a, 0x2a, 0xf7, 0xd6, 0xa6,
    0x3f, 0x82, 0x44, 0xb0, 0x63, 0xe2, 0x87, 0x78, 0x7d, 0x8c, 0x93, 0xf3,
    0x16, 0x7e, 0xed, 0x9f, 0x8f, 0x83, 0x20, 0xc8, 0x61, 0xcd, 0x8c, 0xcb,
    0x51, 0x24, 0xd3, 0x2a, 0xc7, 0x5c, 0x57, 0x6d, 0xaa, 0xc3, 0x08, 0x2b,
    0x3f, 0x5f, 0x75, 0x50, 0x64, 0x0b, 0x77, 0x9e, 0x6f, 0xbe, 0xb7, 0xa3,
    0xf8, 0x9f, 0xcb, 0x11,
};
static const struct drbg_kat_no_reseed kat297_t = {
    11, kat297_entropyin, kat297_nonce, kat297_persstr,
    kat297_addin0, kat297_addin1, kat297_retbits
};
static const struct drbg_kat kat297 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat297_t
};

static const unsigned char kat298_entropyin[] = {
    0xb7, 0x2b, 0xc2, 0xe7, 0x5e, 0x22, 0x7a, 0x5f, 0xab, 0xc2, 0x67, 0xc7,
    0x7f, 0xc7, 0xf9, 0x9c, 0x81, 0xd2, 0xe4, 0xe4, 0x1e, 0xfc, 0x31, 0xb0,
};
static const unsigned char kat298_nonce[] = {
    0x3e, 0x28, 0x60, 0x39, 0xed, 0x89, 0x9f, 0x2b, 0xdc, 0x70, 0x25, 0xa7,
    0xd3, 0x07, 0x12, 0xdc,
};
static const unsigned char kat298_persstr[] = {
    0x35, 0x90, 0x15, 0x3e, 0xf3, 0xb5, 0x80, 0x51, 0x6a, 0x0a, 0xae, 0x3e,
    0x9c, 0xc7, 0xe3, 0x8c, 0x3e, 0xe6, 0xf1, 0x70, 0x3b, 0xf7, 0x7a, 0x3a,
    0x6d, 0x8f, 0xdb, 0x58, 0x40, 0x0c, 0xf1, 0x20,
};
static const unsigned char kat298_addin0[] = {
    0x7d, 0x62, 0xa4, 0x68, 0x7d, 0x5c, 0x11, 0xd3, 0x7c, 0x3a, 0x86, 0x47,
    0x4a, 0x53, 0x22, 0x8c, 0xa5, 0x30, 0x7e, 0xc5, 0x80, 0x1d, 0xc2, 0x49,
    0x53, 0xa0, 0x12, 0x88, 0xde, 0x86, 0x8b, 0xdb,
};
static const unsigned char kat298_addin1[] = {
    0xf4, 0xce, 0x4d, 0x4b, 0x6c, 0xc8, 0x04, 0x1c, 0xc0, 0x94, 0xf0, 0x5f,
    0xb4, 0x68, 0x36, 0xfe, 0x79, 0x4c, 0x81, 0x0a, 0x00, 0xea, 0x54, 0x54,
    0x26, 0x29, 0x74, 0x96, 0xd1, 0xb3, 0xda, 0x84,
};
static const unsigned char kat298_retbits[] = {
    0x91, 0x91, 0x88, 0x42, 0x1a, 0xc9, 0x04, 0xb1, 0x86, 0x98, 0x66, 0x2d,
    0xf0, 0x6b, 0x1b, 0x2c, 0x61, 0xd9, 0x38, 0xee, 0x32, 0x45, 0x70, 0xa4,
    0xd3, 0xfd, 0x84, 0x53, 0x47, 0xf7, 0x6a, 0xb9, 0x0e, 0xa8, 0x98, 0xe8,
    0x74, 0xd5, 0x29, 0x07, 0x75, 0x42, 0x94, 0x1d, 0xf5, 0x47, 0xdc, 0xd1,
    0xb0, 0xa3, 0x70, 0xad, 0xb8, 0xf3, 0xcf, 0x2b, 0x66, 0x50, 0x9f, 0x2a,
    0xa0, 0x0c, 0xe0, 0x07,
};
static const struct drbg_kat_no_reseed kat298_t = {
    12, kat298_entropyin, kat298_nonce, kat298_persstr,
    kat298_addin0, kat298_addin1, kat298_retbits
};
static const struct drbg_kat kat298 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat298_t
};

static const unsigned char kat299_entropyin[] = {
    0x80, 0x0e, 0xa2, 0xb7, 0xe2, 0xa3, 0x4b, 0xa4, 0xe0, 0x4a, 0x8b, 0x2e,
    0x3f, 0x40, 0x4a, 0x4e, 0xbe, 0x78, 0xd1, 0xb1, 0x19, 0x64, 0x71, 0x0b,
};
static const unsigned char kat299_nonce[] = {
    0x26, 0x14, 0xf4, 0x44, 0xb7, 0x0a, 0x50, 0xd8, 0xae, 0x85, 0x27, 0x07,
    0x86, 0x74, 0x16, 0x35,
};
static const unsigned char kat299_persstr[] = {
    0x61, 0x11, 0x2f, 0x45, 0x79, 0xe8, 0xda, 0x87, 0xf9, 0xbd, 0x33, 0xaa,
    0xd0, 0x8f, 0xbe, 0xea, 0x17, 0xe7, 0x8d, 0xe3, 0xdb, 0xf6, 0xa7, 0x55,
    0x49, 0xdc, 0x50, 0xd2, 0xa9, 0x22, 0x72, 0x16,
};
static const unsigned char kat299_addin0[] = {
    0x67, 0x1b, 0x8c, 0xc5, 0x1f, 0x1c, 0x53, 0xe4, 0x25, 0xd5, 0xf9, 0x7d,
    0x73, 0x05, 0x30, 0x21, 0xdd, 0x6a, 0xf2, 0x3f, 0x6a, 0xbd, 0xa8, 0x2e,
    0xfb, 0x00, 0x63, 0x75, 0xfd, 0xfe, 0x68, 0x18,
};
static const unsigned char kat299_addin1[] = {
    0x50, 0x2f, 0xc3, 0xd1, 0xe8, 0x9e, 0x17, 0xf0, 0xf1, 0x11, 0xdf, 0xa3,
    0x2c, 0x74, 0x8a, 0x79, 0xdd, 0x1b, 0x1e, 0x19, 0x02, 0x40, 0x27, 0xbc,
    0x0d, 0x92, 0xed, 0x30, 0x63, 0x85, 0x5b, 0x07,
};
static const unsigned char kat299_retbits[] = {
    0xad, 0xfc, 0xe1, 0x6f, 0x2f, 0x64, 0x82, 0xd0, 0x19, 0x0d, 0xdd, 0xbb,
    0x89, 0xea, 0xd4, 0x48, 0xe3, 0x32, 0x64, 0xfb, 0xe9, 0xf7, 0x6e, 0xf7,
    0x53, 0x07, 0x62, 0x25, 0xf1, 0xc4, 0x30, 0x68, 0x6f, 0x33, 0x2c, 0xba,
    0x84, 0x54, 0xab, 0x81, 0xef, 0x91, 0x86, 0xa8, 0x2a, 0x81, 0x45, 0x31,
    0xc4, 0x9a, 0x1e, 0x46, 0xa2, 0x37, 0x2f, 0xd0, 0x6f, 0xfc, 0x57, 0x68,
    0x23, 0x00, 0x1b, 0x97,
};
static const struct drbg_kat_no_reseed kat299_t = {
    13, kat299_entropyin, kat299_nonce, kat299_persstr,
    kat299_addin0, kat299_addin1, kat299_retbits
};
static const struct drbg_kat kat299 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat299_t
};

static const unsigned char kat300_entropyin[] = {
    0x46, 0xbc, 0x30, 0x11, 0x14, 0x29, 0x14, 0xe8, 0x7f, 0x3b, 0xed, 0xea,
    0x5d, 0x09, 0xed, 0xc1, 0x88, 0xac, 0xdb, 0x2b, 0xd4, 0x96, 0x2d, 0x5e,
};
static const unsigned char kat300_nonce[] = {
    0x99, 0xb0, 0xb4, 0xb4, 0x42, 0xe8, 0x5e, 0xaf, 0x63, 0x8a, 0xb4, 0x65,
    0x44, 0x37, 0xc6, 0xbf,
};
static const unsigned char kat300_persstr[] = {
    0x5b, 0xe5, 0x1d, 0x43, 0x7e, 0x75, 0x05, 0x7d, 0xd2, 0xef, 0x4c, 0xa7,
    0x44, 0x3f, 0xad, 0xe0, 0x7d, 0x52, 0x0d, 0x17, 0x07, 0x4b, 0x5b, 0xd7,
    0x3a, 0xca, 0x11, 0x17, 0x31, 0xb5, 0x08, 0xa5,
};
static const unsigned char kat300_addin0[] = {
    0x81, 0xa5, 0x40, 0xc5, 0xf3, 0x37, 0x9a, 0x46, 0xa2, 0xea, 0x35, 0xcf,
    0xa3, 0x58, 0xd6, 0x7f, 0x6b, 0x35, 0xc0, 0x23, 0xf9, 0x10, 0x30, 0x4b,
    0x87, 0x37, 0x44, 0xaa, 0xb6, 0x32, 0x55, 0xbf,
};
static const unsigned char kat300_addin1[] = {
    0x55, 0xa2, 0xe3, 0x45, 0x99, 0xb0, 0x05, 0x08, 0xea, 0x03, 0x89, 0x56,
    0x3c, 0xf5, 0xc6, 0x63, 0x77, 0x5e, 0x4c, 0x19, 0x13, 0x01, 0x3e, 0x65,
    0x99, 0x9e, 0xa1, 0x52, 0x94, 0x78, 0x84, 0xfb,
};
static const unsigned char kat300_retbits[] = {
    0x56, 0x22, 0xea, 0x84, 0x41, 0x40, 0x52, 0x21, 0x81, 0xd8, 0x3c, 0x5d,
    0xb4, 0x31, 0x8e, 0xfe, 0xa2, 0x59, 0x07, 0x80, 0x70, 0x5b, 0x94, 0x9a,
    0x94, 0x22, 0xf0, 0xee, 0x57, 0x0c, 0x3e, 0x61, 0x35, 0x61, 0x64, 0xc2,
    0xe8, 0xc3, 0x27, 0x5f, 0x5e, 0xdc, 0x8c, 0xf1, 0x8b, 0x71, 0xec, 0xd6,
    0x34, 0xea, 0xe5, 0x29, 0x01, 0xca, 0xa5, 0x23, 0x04, 0x7d, 0xc1, 0x78,
    0xf0, 0x2f, 0x2c, 0xc6,
};
static const struct drbg_kat_no_reseed kat300_t = {
    14, kat300_entropyin, kat300_nonce, kat300_persstr,
    kat300_addin0, kat300_addin1, kat300_retbits
};
static const struct drbg_kat kat300 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat300_t
};

static const unsigned char kat301_entropyin[] = {
    0xa5, 0x7c, 0xc3, 0xb9, 0x95, 0xf3, 0x55, 0x62, 0xba, 0x30, 0x61, 0x9c,
    0xe6, 0xc2, 0xb5, 0x1f, 0x22, 0x17, 0xbf, 0xf0, 0x14, 0x00, 0x6e, 0xf1,
};
static const unsigned char kat301_nonce[] = {
    0xe5, 0x0a, 0x31, 0x2b, 0x22, 0xd6, 0x8f, 0x32, 0x0d, 0x4b, 0xac, 0x24,
    0x0d, 0x41, 0x4f, 0x47,
};
static const unsigned char kat301_persstr[] = {0};
static const unsigned char kat301_addin0[] = {0};
static const unsigned char kat301_addin1[] = {0};
static const unsigned char kat301_retbits[] = {
    0xf2, 0xcf, 0xfe, 0xb0, 0xf4, 0xff, 0xbe, 0x77, 0x3d, 0xc8, 0x04, 0x72,
    0x08, 0x2b, 0x3c, 0x0a, 0x87, 0x7a, 0xaa, 0x11, 0x3d, 0xc4, 0xd8, 0x67,
    0x8b, 0x25, 0xd8, 0x42, 0x02, 0x70, 0xe3, 0x50, 0x88, 0xcd, 0x9e, 0xb2,
    0xcb, 0xaa, 0xd9, 0xbc, 0x3d, 0x5b, 0x51, 0x86, 0x54, 0x47, 0x24, 0x5a,
    0x3a, 0x78, 0xb3, 0x8c, 0x51, 0xd0, 0xa1, 0x9c, 0xa0, 0x8c, 0x61, 0x95,
    0x58, 0x7d, 0xfd, 0x7a,
};
static const struct drbg_kat_no_reseed kat301_t = {
    0, kat301_entropyin, kat301_nonce, kat301_persstr,
    kat301_addin0, kat301_addin1, kat301_retbits
};
static const struct drbg_kat kat301 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat301_t
};

static const unsigned char kat302_entropyin[] = {
    0x0b, 0x61, 0x0e, 0x89, 0x79, 0x68, 0x2f, 0x44, 0xd9, 0x37, 0xc9, 0x74,
    0xe7, 0x3a, 0x4c, 0x3d, 0xf9, 0x5a, 0x34, 0xb0, 0x92, 0x40, 0x5f, 0xe4,
};
static const unsigned char kat302_nonce[] = {
    0x08, 0x55, 0x81, 0xc0, 0x1f, 0xb9, 0x16, 0x15, 0x84, 0xb9, 0xf6, 0x52,
    0x6f, 0x54, 0x7b, 0x44,
};
static const unsigned char kat302_persstr[] = {0};
static const unsigned char kat302_addin0[] = {0};
static const unsigned char kat302_addin1[] = {0};
static const unsigned char kat302_retbits[] = {
    0x28, 0x2d, 0xfb, 0x8a, 0x9e, 0xd6, 0x72, 0x12, 0x29, 0xf7, 0x81, 0xa1,
    0x5a, 0x70, 0x33, 0xad, 0xbc, 0xd4, 0x9a, 0x21, 0x0a, 0x23, 0x1e, 0x17,
    0xeb, 0x70, 0xd6, 0x4c, 0xa8, 0x03, 0x75, 0xa8, 0xff, 0xcb, 0x4e, 0x9a,
    0xc6, 0xe2, 0x32, 0x73, 0xca, 0x96, 0x54, 0xe6, 0x71, 0xcc, 0xb1, 0xae,
    0x3b, 0xb5, 0x96, 0xbf, 0x7a, 0x8d, 0xf5, 0xdd, 0x23, 0x0b, 0xf4, 0xa2,
    0xb3, 0x9b, 0xf9, 0x6f,
};
static const struct drbg_kat_no_reseed kat302_t = {
    1, kat302_entropyin, kat302_nonce, kat302_persstr,
    kat302_addin0, kat302_addin1, kat302_retbits
};
static const struct drbg_kat kat302 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat302_t
};

static const unsigned char kat303_entropyin[] = {
    0xf9, 0x3e, 0x04, 0x17, 0x48, 0x91, 0x02, 0x81, 0xa4, 0xff, 0x37, 0xef,
    0xe8, 0x30, 0xac, 0x02, 0x64, 0xe4, 0xb1, 0x2e, 0xce, 0x06, 0x78, 0xee,
};
static const unsigned char kat303_nonce[] = {
    0x7d, 0x4d, 0x6c, 0x54, 0x5d, 0xe8, 0xc8, 0x11, 0x84, 0x55, 0x59, 0x54,
    0x34, 0x65, 0x1b, 0xbc,
};
static const unsigned char kat303_persstr[] = {0};
static const unsigned char kat303_addin0[] = {0};
static const unsigned char kat303_addin1[] = {0};
static const unsigned char kat303_retbits[] = {
    0xa7, 0x3e, 0x59, 0xa6, 0x6d, 0xe5, 0x02, 0x29, 0x8a, 0x81, 0x06, 0x93,
    0x9f, 0xfa, 0xf8, 0x56, 0xaf, 0x9f, 0x17, 0x2b, 0x85, 0xf5, 0xe9, 0x9b,
    0x05, 0x7d, 0x6b, 0x4d, 0x6e, 0xf0, 0x52, 0xf1, 0x22, 0x67, 0x00, 0xe5,
    0xe3, 0x98, 0xe6, 0x80, 0x2f, 0x2a, 0x4a, 0xfa, 0x04, 0xe7, 0xff, 0x00,
    0xc4, 0x6f, 0x57, 0xf3, 0x45, 0xeb, 0x03, 0xde, 0x8f, 0xa7, 0x79, 0x5a,
    0x06, 0x0c, 0xf3, 0x40,
};
static const struct drbg_kat_no_reseed kat303_t = {
    2, kat303_entropyin, kat303_nonce, kat303_persstr,
    kat303_addin0, kat303_addin1, kat303_retbits
};
static const struct drbg_kat kat303 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat303_t
};

static const unsigned char kat304_entropyin[] = {
    0x81, 0x48, 0x86, 0xa6, 0xb9, 0xa3, 0xcf, 0xc3, 0x18, 0x78, 0x5b, 0x41,
    0x93, 0x9f, 0x7b, 0x07, 0x5e, 0xc6, 0xb0, 0x0f, 0xc7, 0xff, 0x1c, 0x9c,
};
static const unsigned char kat304_nonce[] = {
    0xd1, 0x33, 0x13, 0x12, 0x3f, 0xbc, 0x45, 0xce, 0x8b, 0x9e, 0x8a, 0x0a,
    0x4d, 0x82, 0xb2, 0x7a,
};
static const unsigned char kat304_persstr[] = {0};
static const unsigned char kat304_addin0[] = {0};
static const unsigned char kat304_addin1[] = {0};
static const unsigned char kat304_retbits[] = {
    0xba, 0x2a, 0x7c, 0x50, 0xc6, 0xd3, 0xe7, 0xef, 0x2d, 0x84, 0x91, 0x06,
    0xc6, 0xe8, 0x33, 0x10, 0xd9, 0xad, 0x1c, 0x1d, 0x0e, 0xf5, 0x3e, 0x61,
    0xc8, 0xb9, 0x55, 0xac, 0xb4, 0xef, 0x43, 0x42, 0xe1, 0xc0, 0xf5, 0xf2,
    0x2e, 0x19, 0xbb, 0x1e, 0xea, 0x5d, 0x0f, 0x03, 0xb6, 0xd5, 0x96, 0x92,
    0x91, 0xc4, 0x01, 0xfb, 0x6c, 0x55, 0xc9, 0x9f, 0x42, 0x52, 0xb6, 0xca,
    0x7f, 0x34, 0x19, 0x35,
};
static const struct drbg_kat_no_reseed kat304_t = {
    3, kat304_entropyin, kat304_nonce, kat304_persstr,
    kat304_addin0, kat304_addin1, kat304_retbits
};
static const struct drbg_kat kat304 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat304_t
};

static const unsigned char kat305_entropyin[] = {
    0x69, 0xce, 0x9b, 0xc6, 0xa2, 0x66, 0x95, 0x02, 0x70, 0xba, 0xb4, 0x7f,
    0x50, 0x19, 0xf5, 0x76, 0xfb, 0x9d, 0xee, 0x08, 0x17, 0x9b, 0x34, 0xeb,
};
static const unsigned char kat305_nonce[] = {
    0x63, 0x02, 0x5d, 0x58, 0x8f, 0x1d, 0x08, 0xf3, 0x81, 0x16, 0x96, 0xa6,
    0x3a, 0x10, 0x02, 0xe7,
};
static const unsigned char kat305_persstr[] = {0};
static const unsigned char kat305_addin0[] = {0};
static const unsigned char kat305_addin1[] = {0};
static const unsigned char kat305_retbits[] = {
    0x5e, 0x52, 0xaf, 0xd2, 0xa8, 0x67, 0x01, 0xcb, 0x40, 0xbd, 0x77, 0xba,
    0xf5, 0x5e, 0xff, 0x3a, 0x6c, 0x9f, 0x84, 0x34, 0x56, 0xed, 0x64, 0xd4,
    0x01, 0x53, 0x32, 0x77, 0x9f, 0xe0, 0xb2, 0x6a, 0xea, 0xc6, 0x34, 0xcb,
    0x06, 0x21, 0xc4, 0x3b, 0x3a, 0xb7, 0x1e, 0xea, 0x2a, 0xdf, 0x96, 0x31,
    0x2d, 0x38, 0x5c, 0x62, 0xc3, 0x1c, 0x09, 0x51, 0x52, 0x32, 0x60, 0xc4,
    0x69, 0xb2, 0x7d, 0x88,
};
static const struct drbg_kat_no_reseed kat305_t = {
    4, kat305_entropyin, kat305_nonce, kat305_persstr,
    kat305_addin0, kat305_addin1, kat305_retbits
};
static const struct drbg_kat kat305 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat305_t
};

static const unsigned char kat306_entropyin[] = {
    0x25, 0x46, 0x9a, 0x3c, 0x9a, 0x37, 0xbd, 0x9e, 0xa9, 0x1c, 0x6f, 0x24,
    0x3e, 0x44, 0x4c, 0x1b, 0x36, 0x03, 0x01, 0x52, 0xd6, 0x38, 0x00, 0xf5,
};
static const unsigned char kat306_nonce[] = {
    0xa2, 0xb8, 0x39, 0xab, 0xcf, 0xe4, 0xa8, 0x73, 0x68, 0x65, 0x6f, 0x43,
    0xb5, 0xbb, 0x4e, 0x2e,
};
static const unsigned char kat306_persstr[] = {0};
static const unsigned char kat306_addin0[] = {0};
static const unsigned char kat306_addin1[] = {0};
static const unsigned char kat306_retbits[] = {
    0x32, 0xc1, 0xfd, 0x38, 0xda, 0x53, 0xad, 0xbc, 0x0a, 0x21, 0xa8, 0x28,
    0xdd, 0xdc, 0xb6, 0x80, 0x9f, 0xb0, 0x1d, 0x28, 0x30, 0xc6, 0xa5, 0x44,
    0xd1, 0x2f, 0x55, 0xfa, 0x2e, 0x44, 0x43, 0xf9, 0x5f, 0x00, 0x5c, 0x0f,
    0xf2, 0x4c, 0x64, 0x92, 0xb6, 0x40, 0x95, 0xe3, 0x74, 0x6f, 0x07, 0x12,
    0x2f, 0x98, 0x3d, 0x23, 0x8f, 0x23, 0xda, 0xff, 0xf7, 0x10, 0x3a, 0x7c,
    0xda, 0x57, 0xeb, 0x0e,
};
static const struct drbg_kat_no_reseed kat306_t = {
    5, kat306_entropyin, kat306_nonce, kat306_persstr,
    kat306_addin0, kat306_addin1, kat306_retbits
};
static const struct drbg_kat kat306 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat306_t
};

static const unsigned char kat307_entropyin[] = {
    0xb8, 0x8a, 0x61, 0x0e, 0x68, 0x7c, 0x1e, 0x06, 0xd1, 0x8c, 0xa9, 0x12,
    0x86, 0x6b, 0x55, 0xd2, 0x76, 0xa7, 0x89, 0xcf, 0xf9, 0x43, 0x95, 0x95,
};
static const unsigned char kat307_nonce[] = {
    0x25, 0x4f, 0x4b, 0xaa, 0x2b, 0x34, 0xe9, 0x89, 0x8d, 0x5f, 0x78, 0x59,
    0x77, 0x24, 0xdd, 0xc7,
};
static const unsigned char kat307_persstr[] = {0};
static const unsigned char kat307_addin0[] = {0};
static const unsigned char kat307_addin1[] = {0};
static const unsigned char kat307_retbits[] = {
    0x32, 0x41, 0x1f, 0x32, 0x50, 0xbf, 0x2a, 0x18, 0xce, 0xd4, 0x8f, 0x52,
    0x2b, 0x3c, 0xc3, 0x06, 0xaf, 0xc1, 0x8e, 0x7a, 0x36, 0xcd, 0x64, 0x81,
    0x66, 0x47, 0xa0, 0x0c, 0x22, 0x1b, 0x99, 0x6a, 0x82, 0xee, 0x85, 0x0b,
    0x7c, 0x0a, 0x8c, 0xa2, 0x24, 0x1b, 0x7c, 0x6e, 0x64, 0x5b, 0x45, 0x3e,
    0x01, 0xfc, 0x35, 0xfb, 0xa5, 0x4d, 0x5b, 0xae, 0xd9, 0x22, 0x8a, 0x98,
    0xa6, 0x72, 0xdb, 0x67,
};
static const struct drbg_kat_no_reseed kat307_t = {
    6, kat307_entropyin, kat307_nonce, kat307_persstr,
    kat307_addin0, kat307_addin1, kat307_retbits
};
static const struct drbg_kat kat307 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat307_t
};

static const unsigned char kat308_entropyin[] = {
    0xf3, 0x2b, 0xb1, 0x0c, 0x9e, 0x0d, 0x8f, 0x53, 0x45, 0x38, 0x94, 0x0f,
    0xba, 0xa3, 0x68, 0x44, 0xe7, 0xf6, 0xf0, 0x91, 0x97, 0x50, 0xe1, 0xeb,
};
static const unsigned char kat308_nonce[] = {
    0x33, 0x4b, 0x85, 0xe0, 0x31, 0x51, 0x96, 0x07, 0xe0, 0x01, 0xa2, 0x87,
    0xe3, 0x25, 0x83, 0x22,
};
static const unsigned char kat308_persstr[] = {0};
static const unsigned char kat308_addin0[] = {0};
static const unsigned char kat308_addin1[] = {0};
static const unsigned char kat308_retbits[] = {
    0x75, 0x58, 0xc5, 0x96, 0x84, 0xf7, 0xa0, 0x47, 0xa1, 0xe5, 0x92, 0x9c,
    0x8a, 0x37, 0xde, 0xfc, 0x2f, 0x1f, 0x55, 0x4a, 0x34, 0xc4, 0x7f, 0xcd,
    0x9d, 0xda, 0xcf, 0x88, 0xbb, 0xf7, 0xea, 0xc4, 0x07, 0x04, 0xae, 0xaa,
    0x50, 0x3c, 0xff, 0x89, 0x63, 0x29, 0x5e, 0x70, 0x24, 0x35, 0xf7, 0x3c,
    0x93, 0xff, 0x35, 0x4f, 0x52, 0x9e, 0xe2, 0xda, 0x08, 0xc3, 0x50, 0xd4,
    0xcc, 0xb6, 0xd4, 0x5f,
};
static const struct drbg_kat_no_reseed kat308_t = {
    7, kat308_entropyin, kat308_nonce, kat308_persstr,
    kat308_addin0, kat308_addin1, kat308_retbits
};
static const struct drbg_kat kat308 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat308_t
};

static const unsigned char kat309_entropyin[] = {
    0xcf, 0x14, 0x4d, 0xdf, 0xc1, 0x32, 0xa6, 0xa6, 0xe7, 0xb5, 0xa1, 0xef,
    0xd5, 0x15, 0xc7, 0x04, 0xfc, 0x22, 0x82, 0x3e, 0x21, 0x7d, 0x32, 0xdd,
};
static const unsigned char kat309_nonce[] = {
    0x5a, 0x0b, 0xc0, 0xc8, 0x7c, 0xcc, 0xd9, 0x08, 0xb4, 0x40, 0xd6, 0xa4,
    0xc7, 0x38, 0xfa, 0x74,
};
static const unsigned char kat309_persstr[] = {0};
static const unsigned char kat309_addin0[] = {0};
static const unsigned char kat309_addin1[] = {0};
static const unsigned char kat309_retbits[] = {
    0x86, 0x5d, 0xb2, 0xd0, 0x71, 0x1d, 0xd3, 0xfd, 0xd5, 0xd2, 0xa1, 0x9a,
    0x65, 0xfb, 0x93, 0x1e, 0xec, 0xd8, 0x2b, 0xa2, 0xa7, 0xf2, 0xe5, 0x3b,
    0xd6, 0x42, 0x0a, 0x8b, 0x16, 0x47, 0xa4, 0x18, 0x02, 0xc0, 0x4f, 0x9b,
    0x6b, 0xa3, 0x4b, 0x85, 0xa7, 0x2b, 0x48, 0xfd, 0xe7, 0x60, 0x52, 0x3e,
    0x28, 0xd8, 0x7e, 0xda, 0xc9, 0xb8, 0xad, 0x6c, 0xba, 0x36, 0x1f, 0x13,
    0x53, 0xfd, 0x39, 0xae,
};
static const struct drbg_kat_no_reseed kat309_t = {
    8, kat309_entropyin, kat309_nonce, kat309_persstr,
    kat309_addin0, kat309_addin1, kat309_retbits
};
static const struct drbg_kat kat309 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat309_t
};

static const unsigned char kat310_entropyin[] = {
    0xad, 0xe1, 0x90, 0x85, 0x34, 0x62, 0x4c, 0x6b, 0xc3, 0x75, 0x4d, 0xe0,
    0xdb, 0x75, 0x31, 0x38, 0x35, 0xa8, 0x28, 0x57, 0x2c, 0xd2, 0x77, 0xbd,
};
static const unsigned char kat310_nonce[] = {
    0x05, 0xa5, 0xb6, 0xd2, 0x1b, 0x98, 0x05, 0x05, 0x93, 0xab, 0x98, 0x13,
    0x26, 0x8e, 0x2d, 0xc2,
};
static const unsigned char kat310_persstr[] = {0};
static const unsigned char kat310_addin0[] = {0};
static const unsigned char kat310_addin1[] = {0};
static const unsigned char kat310_retbits[] = {
    0xef, 0xb7, 0xb7, 0x7b, 0x76, 0xd7, 0xbf, 0x57, 0x1a, 0x9b, 0xb1, 0x3e,
    0x39, 0xec, 0x46, 0xa1, 0x9e, 0x91, 0x35, 0xab, 0x08, 0x6d, 0x06, 0xd5,
    0x9b, 0x20, 0xf0, 0x0c, 0x78, 0xf2, 0xaa, 0xa8, 0x48, 0xf6, 0x29, 0x32,
    0x7a, 0x0d, 0x67, 0x07, 0x29, 0xa4, 0x53, 0x91, 0xf2, 0xf0, 0x23, 0xa8,
    0xf4, 0x5f, 0x1c, 0xb1, 0x0b, 0x40, 0xc6, 0x61, 0x83, 0x17, 0xa4, 0x6a,
    0xfe, 0x0a, 0xe1, 0x30,
};
static const struct drbg_kat_no_reseed kat310_t = {
    9, kat310_entropyin, kat310_nonce, kat310_persstr,
    kat310_addin0, kat310_addin1, kat310_retbits
};
static const struct drbg_kat kat310 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat310_t
};

static const unsigned char kat311_entropyin[] = {
    0x38, 0x80, 0xc2, 0x26, 0x90, 0x82, 0x47, 0x5c, 0x16, 0xb9, 0x78, 0xc7,
    0x2a, 0xf1, 0xde, 0x9d, 0x74, 0xec, 0x3c, 0x76, 0x94, 0xc5, 0x19, 0xca,
};
static const unsigned char kat311_nonce[] = {
    0x83, 0xea, 0xd0, 0xf2, 0xd4, 0x6d, 0x8d, 0x1a, 0x77, 0x86, 0x90, 0xae,
    0x51, 0x2d, 0x66, 0xbd,
};
static const unsigned char kat311_persstr[] = {0};
static const unsigned char kat311_addin0[] = {0};
static const unsigned char kat311_addin1[] = {0};
static const unsigned char kat311_retbits[] = {
    0xaf, 0x4d, 0x71, 0xc1, 0x14, 0x75, 0x52, 0xbf, 0x8b, 0x35, 0x8e, 0x16,
    0xc9, 0xc7, 0xfe, 0x7f, 0x03, 0x66, 0x20, 0xcb, 0xd6, 0x1f, 0x14, 0x32,
    0xc8, 0xc5, 0xf2, 0x70, 0xda, 0xcc, 0xb2, 0xb9, 0xe1, 0x5c, 0x12, 0xe9,
    0xf5, 0x92, 0x10, 0x47, 0xdc, 0x92, 0xd4, 0x0a, 0x90, 0xe4, 0x5b, 0xf1,
    0x5f, 0xef, 0x12, 0x0a, 0x3e, 0x08, 0x42, 0x13, 0x54, 0x67, 0x4c, 0x95,
    0xa6, 0x30, 0x77, 0xe3,
};
static const struct drbg_kat_no_reseed kat311_t = {
    10, kat311_entropyin, kat311_nonce, kat311_persstr,
    kat311_addin0, kat311_addin1, kat311_retbits
};
static const struct drbg_kat kat311 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat311_t
};

static const unsigned char kat312_entropyin[] = {
    0xb1, 0xd2, 0xc9, 0xab, 0x5a, 0x90, 0x45, 0x67, 0xbd, 0x03, 0x11, 0xc6,
    0x42, 0x04, 0x7d, 0xe1, 0xb3, 0x94, 0x41, 0xa5, 0xe2, 0xd9, 0x01, 0xbe,
};
static const unsigned char kat312_nonce[] = {
    0x91, 0xc8, 0xd1, 0x9b, 0xf5, 0xd4, 0xf9, 0x51, 0x1f, 0x6c, 0x66, 0x53,
    0xa6, 0x58, 0x5b, 0xa6,
};
static const unsigned char kat312_persstr[] = {0};
static const unsigned char kat312_addin0[] = {0};
static const unsigned char kat312_addin1[] = {0};
static const unsigned char kat312_retbits[] = {
    0x8b, 0xc7, 0xb7, 0x90, 0xc7, 0x7b, 0x04, 0x7f, 0x7a, 0x17, 0x86, 0x41,
    0x61, 0x5c, 0x29, 0x5b, 0x16, 0x4b, 0x09, 0x03, 0x7c, 0x06, 0xa8, 0xdf,
    0x76, 0xbd, 0xa5, 0x9b, 0x77, 0xed, 0x28, 0x5f, 0x80, 0x8c, 0x96, 0xa4,
    0xb4, 0x9b, 0x45, 0x94, 0xa1, 0x82, 0x99, 0xc2, 0xa6, 0x2b, 0x1e, 0x0d,
    0x9c, 0xaa, 0xeb, 0x0f, 0x82, 0xc1, 0xde, 0x35, 0xb4, 0x28, 0x03, 0x8d,
    0x10, 0xab, 0x00, 0xc0,
};
static const struct drbg_kat_no_reseed kat312_t = {
    11, kat312_entropyin, kat312_nonce, kat312_persstr,
    kat312_addin0, kat312_addin1, kat312_retbits
};
static const struct drbg_kat kat312 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat312_t
};

static const unsigned char kat313_entropyin[] = {
    0x19, 0x70, 0x0c, 0xe2, 0x88, 0x3e, 0xac, 0x51, 0x58, 0x75, 0xfc, 0x91,
    0x35, 0x3c, 0x9a, 0x9d, 0x59, 0x89, 0xb0, 0xd2, 0x9f, 0xa2, 0xee, 0xab,
};
static const unsigned char kat313_nonce[] = {
    0x56, 0x56, 0x44, 0x93, 0x11, 0xe2, 0x46, 0xee, 0x80, 0x25, 0xcf, 0x30,
    0x3d, 0x6b, 0x6d, 0xa9,
};
static const unsigned char kat313_persstr[] = {0};
static const unsigned char kat313_addin0[] = {0};
static const unsigned char kat313_addin1[] = {0};
static const unsigned char kat313_retbits[] = {
    0x00, 0x02, 0x17, 0x84, 0x99, 0x39, 0xed, 0x26, 0xc9, 0xd5, 0xfd, 0xd4,
    0xd0, 0x03, 0xd5, 0x7c, 0x12, 0xf5, 0x2f, 0x6e, 0x7d, 0xae, 0x36, 0x4e,
    0x73, 0x75, 0x8c, 0x7f, 0xd1, 0x22, 0x5b, 0x17, 0xae, 0x66, 0xf3, 0x83,
    0x79, 0xa9, 0x03, 0xe7, 0x84, 0x54, 0x46, 0xf0, 0x99, 0x8a, 0x87, 0x45,
    0xe9, 0xae, 0xc9, 0xb4, 0xba, 0x31, 0x73, 0xba, 0x1b, 0x29, 0x21, 0xea,
    0xac, 0x31, 0x3e, 0x58,
};
static const struct drbg_kat_no_reseed kat313_t = {
    12, kat313_entropyin, kat313_nonce, kat313_persstr,
    kat313_addin0, kat313_addin1, kat313_retbits
};
static const struct drbg_kat kat313 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat313_t
};

static const unsigned char kat314_entropyin[] = {
    0x55, 0x21, 0x10, 0x48, 0x6f, 0x04, 0xb8, 0xb1, 0x73, 0x74, 0x8b, 0xe0,
    0x9d, 0xb8, 0x53, 0x02, 0x3d, 0x7d, 0x1e, 0x6b, 0x9e, 0x63, 0x11, 0xf4,
};
static const unsigned char kat314_nonce[] = {
    0xd2, 0x8c, 0x45, 0xca, 0x09, 0xd5, 0x11, 0x04, 0xd5, 0x19, 0xd3, 0x81,
    0xb0, 0x0c, 0xd7, 0x12,
};
static const unsigned char kat314_persstr[] = {0};
static const unsigned char kat314_addin0[] = {0};
static const unsigned char kat314_addin1[] = {0};
static const unsigned char kat314_retbits[] = {
    0xb3, 0x5b, 0x8c, 0xde, 0x56, 0xd5, 0xb2, 0xa3, 0x65, 0xa7, 0x55, 0xe4,
    0xd0, 0xbd, 0xbf, 0x70, 0x02, 0xe4, 0xb0, 0x61, 0x99, 0xa3, 0x42, 0xf1,
    0xb9, 0x8b, 0x73, 0x43, 0x11, 0xfc, 0xab, 0x66, 0xaf, 0x1e, 0xea, 0x2c,
    0x7f, 0xbc, 0x26, 0x42, 0xac, 0x36, 0x4c, 0x92, 0x0a, 0xf7, 0x24, 0xe3,
    0x7d, 0x45, 0xe8, 0xc1, 0x7c, 0xd4, 0x46, 0x7b, 0x89, 0x2a, 0x45, 0x36,
    0x47, 0x94, 0xa7, 0x46,
};
static const struct drbg_kat_no_reseed kat314_t = {
    13, kat314_entropyin, kat314_nonce, kat314_persstr,
    kat314_addin0, kat314_addin1, kat314_retbits
};
static const struct drbg_kat kat314 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat314_t
};

static const unsigned char kat315_entropyin[] = {
    0xb2, 0xf6, 0x05, 0xfc, 0xf9, 0x92, 0x9b, 0x12, 0x43, 0xc8, 0x81, 0x99,
    0x14, 0x3f, 0x15, 0x42, 0x66, 0xfd, 0x85, 0x87, 0xc9, 0x0a, 0xaa, 0xb2,
};
static const unsigned char kat315_nonce[] = {
    0x78, 0x1c, 0x85, 0x7f, 0x75, 0xbb, 0x59, 0x48, 0xc2, 0xa2, 0x2d, 0xd7,
    0x57, 0xb2, 0x47, 0xbe,
};
static const unsigned char kat315_persstr[] = {0};
static const unsigned char kat315_addin0[] = {0};
static const unsigned char kat315_addin1[] = {0};
static const unsigned char kat315_retbits[] = {
    0x77, 0x53, 0x79, 0x7b, 0x7b, 0x37, 0x25, 0xc8, 0x73, 0x9f, 0xf2, 0x11,
    0xa0, 0xcc, 0x8a, 0xe8, 0xa1, 0xe0, 0xf2, 0x80, 0x86, 0xed, 0xe4, 0xd4,
    0xf8, 0x36, 0xde, 0xe5, 0x7f, 0xbd, 0x78, 0x80, 0xf1, 0x05, 0x21, 0x24,
    0x27, 0x59, 0xb9, 0x72, 0x5a, 0x08, 0xcb, 0x64, 0x96, 0x59, 0xa2, 0xd8,
    0x1f, 0x54, 0x0e, 0x93, 0xf8, 0x7e, 0x49, 0x31, 0xd8, 0x1b, 0x59, 0xd9,
    0xcd, 0xcf, 0x45, 0xc0,
};
static const struct drbg_kat_no_reseed kat315_t = {
    14, kat315_entropyin, kat315_nonce, kat315_persstr,
    kat315_addin0, kat315_addin1, kat315_retbits
};
static const struct drbg_kat kat315 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat315_t
};

static const unsigned char kat316_entropyin[] = {
    0xe6, 0x20, 0x98, 0xa1, 0x6d, 0x60, 0xd2, 0x75, 0x98, 0xea, 0x4c, 0x97,
    0xf2, 0xf0, 0x13, 0xc4, 0xff, 0xbd, 0x45, 0x6b, 0x54, 0xa8, 0xfc, 0xf7,
};
static const unsigned char kat316_nonce[] = {
    0x10, 0x39, 0x1c, 0x9e, 0x4d, 0x79, 0x69, 0xfd, 0x2b, 0x1f, 0x89, 0x86,
    0xe0, 0x48, 0x60, 0xc8,
};
static const unsigned char kat316_persstr[] = {0};
static const unsigned char kat316_addin0[] = {
    0x6a, 0x7d, 0xb6, 0xd2, 0xf6, 0xf2, 0x35, 0x72, 0xfe, 0xff, 0xd3, 0xd7,
    0x74, 0x46, 0xf0, 0x7b, 0x92, 0x2e, 0xe7, 0xa9, 0xfe, 0x58, 0x01, 0x60,
    0xed, 0x2d, 0x8d, 0xd7, 0xff, 0xb5, 0x0b, 0x00,
};
static const unsigned char kat316_addin1[] = {
    0x1c, 0x4f, 0x98, 0x2a, 0xce, 0x96, 0xe7, 0x84, 0x10, 0x3e, 0xa2, 0x54,
    0xcf, 0xd6, 0x85, 0xe9, 0x5a, 0xda, 0xfe, 0x79, 0x88, 0xd9, 0xee, 0xbd,
    0xcd, 0x82, 0xe7, 0xba, 0x02, 0x5f, 0x31, 0x53,
};
static const unsigned char kat316_retbits[] = {
    0xaf, 0x0e, 0x2c, 0x20, 0x53, 0xe5, 0xa0, 0xef, 0x2d, 0xef, 0xa7, 0xdf,
    0xa6, 0xb4, 0x16, 0x4b, 0x4d, 0x83, 0x78, 0xf0, 0x0d, 0xfd, 0x97, 0x62,
    0xdb, 0xa6, 0x3a, 0x67, 0xef, 0x5b, 0x58, 0xc8, 0xe1, 0xa8, 0x6f, 0x56,
    0x0c, 0xcd, 0x93, 0x37, 0x11, 0x62, 0x43, 0x60, 0x9b, 0xc4, 0xb4, 0x54,
    0xbf, 0x8d, 0x1a, 0x1a, 0xa3, 0xf9, 0x9a, 0xd1, 0xd2, 0x58, 0xc3, 0x18,
    0xc7, 0x2d, 0x14, 0x4c,
};
static const struct drbg_kat_no_reseed kat316_t = {
    0, kat316_entropyin, kat316_nonce, kat316_persstr,
    kat316_addin0, kat316_addin1, kat316_retbits
};
static const struct drbg_kat kat316 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat316_t
};

static const unsigned char kat317_entropyin[] = {
    0x83, 0x02, 0x64, 0x0e, 0x79, 0xce, 0xab, 0x3c, 0x2b, 0x9f, 0x2b, 0x3e,
    0xf8, 0x7e, 0x6a, 0xd9, 0x07, 0xd1, 0x2e, 0xa9, 0xad, 0x1a, 0x07, 0xc7,
};
static const unsigned char kat317_nonce[] = {
    0xbc, 0xf4, 0xb6, 0xd5, 0xf6, 0x9c, 0xa0, 0x9b, 0xa4, 0xb3, 0xbf, 0x58,
    0x17, 0x23, 0xcd, 0x6b,
};
static const unsigned char kat317_persstr[] = {0};
static const unsigned char kat317_addin0[] = {
    0xc2, 0x7f, 0x38, 0x0b, 0x39, 0xcb, 0xb7, 0x15, 0x25, 0x67, 0x28, 0xde,
    0x46, 0x85, 0xf3, 0xc4, 0x28, 0x7b, 0x6e, 0x3b, 0xaf, 0x13, 0xc0, 0xf1,
    0xbe, 0x9b, 0xb4, 0xc8, 0x4d, 0xec, 0xad, 0x11,
};
static const unsigned char kat317_addin1[] = {
    0x13, 0x52, 0x30, 0xd9, 0x56, 0x35, 0xfb, 0x88, 0xc7, 0x37, 0x07, 0x07,
    0x80, 0x18, 0x79, 0x2d, 0x40, 0x65, 0x84, 0x8a, 0xd8, 0x13, 0xab, 0x98,
    0x14, 0x95, 0x57, 0x1c, 0xf9, 0x88, 0x8e, 0x7c,
};
static const unsigned char kat317_retbits[] = {
    0x9c, 0x61, 0xdc, 0xac, 0x29, 0xe1, 0xd8, 0x50, 0x8a, 0xef, 0x25, 0x38,
    0xad, 0xe8, 0xa8, 0x38, 0xec, 0xf5, 0xc5, 0x06, 0x1a, 0xdc, 0xda, 0xfb,
    0xce, 0x37, 0x74, 0xbd, 0x24, 0x0c, 0xc6, 0x23, 0x0b, 0x68, 0x48, 0x55,
    0x92, 0xa2, 0xec, 0x2c, 0x0b, 0x9d, 0x6f, 0x4f, 0x82, 0x02, 0xb1, 0x63,
    0xb8, 0x3e, 0xbc, 0x75, 0x2c, 0xc0, 0x8c, 0xc9, 0x27, 0xe2, 0xa7, 0x9d,
    0xb1, 0xba, 0xb8, 0x1e,
};
static const struct drbg_kat_no_reseed kat317_t = {
    1, kat317_entropyin, kat317_nonce, kat317_persstr,
    kat317_addin0, kat317_addin1, kat317_retbits
};
static const struct drbg_kat kat317 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat317_t
};

static const unsigned char kat318_entropyin[] = {
    0xee, 0xad, 0x03, 0x90, 0xcd, 0x19, 0x34, 0x10, 0xc5, 0x20, 0xeb, 0x54,
    0x14, 0xae, 0x6f, 0x2e, 0x2e, 0xb0, 0x0b, 0xf5, 0xae, 0x0a, 0xf3, 0xde,
};
static const unsigned char kat318_nonce[] = {
    0xd4, 0x0c, 0xa3, 0xe5, 0x21, 0xf2, 0xa5, 0x7e, 0x35, 0xbf, 0x3a, 0xed,
    0x30, 0x02, 0xab, 0xc0,
};
static const unsigned char kat318_persstr[] = {0};
static const unsigned char kat318_addin0[] = {
    0xe4, 0x30, 0x9e, 0xc8, 0x51, 0x9e, 0xed, 0x08, 0xe1, 0x54, 0x21, 0x30,
    0x14, 0x51, 0x8f, 0x02, 0xa8, 0x38, 0x75, 0xc7, 0x7e, 0x6b, 0xcb, 0x6e,
    0xb1, 0xdc, 0x21, 0xb1, 0x48, 0xbc, 0x17, 0xc1,
};
static const unsigned char kat318_addin1[] = {
    0xa7, 0x6a, 0xae, 0xb9, 0xbd, 0xda, 0x28, 0x26, 0x38, 0xae, 0x77, 0x4f,
    0x35, 0xdf, 0x92, 0xa1, 0x0e, 0x2f, 0xdc, 0xb2, 0xfc, 0x91, 0x70, 0x3e,
    0xc8, 0xd5, 0x69, 0x37, 0x2d, 0x86, 0x89, 0x44,
};
static const unsigned char kat318_retbits[] = {
    0xb4, 0xb8, 0x0f, 0x0d, 0x94, 0xa4, 0xf9, 0x31, 0x07, 0xe4, 0xb5, 0x4c,
    0x11, 0x8f, 0xd8, 0xa8, 0xca, 0x62, 0xd6, 0x0c, 0x54, 0x25, 0x86, 0x54,
    0xfb, 0x83, 0xdc, 0x3f, 0x4d, 0x24, 0xcf, 0xeb, 0x58, 0xa3, 0x71, 0x4e,
    0xe6, 0xfa, 0x47, 0x8e, 0x86, 0xea, 0x64, 0x9a, 0x64, 0xe9, 0x4e, 0xa4,
    0x9e, 0x10, 0x88, 0x54, 0xbf, 0x04, 0x8e, 0x4b, 0xc6, 0xa0, 0xf0, 0xf9,
    0x74, 0x0a, 0xaf, 0x92,
};
static const struct drbg_kat_no_reseed kat318_t = {
    2, kat318_entropyin, kat318_nonce, kat318_persstr,
    kat318_addin0, kat318_addin1, kat318_retbits
};
static const struct drbg_kat kat318 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat318_t
};

static const unsigned char kat319_entropyin[] = {
    0xee, 0x1e, 0x3a, 0xf7, 0x32, 0xbf, 0xf0, 0x4e, 0x9e, 0x45, 0xba, 0x62,
    0xdb, 0xe6, 0x27, 0x04, 0x97, 0x08, 0xb6, 0xef, 0x4a, 0xa7, 0x93, 0xe7,
};
static const unsigned char kat319_nonce[] = {
    0x53, 0x1e, 0xdb, 0xba, 0x82, 0xb1, 0x20, 0x02, 0x68, 0x35, 0xb8, 0x56,
    0xbb, 0x5b, 0x63, 0xe0,
};
static const unsigned char kat319_persstr[] = {0};
static const unsigned char kat319_addin0[] = {
    0xc3, 0x8b, 0x7a, 0x6c, 0xc5, 0xd5, 0xd2, 0x3c, 0xe4, 0x16, 0xf9, 0x24,
    0xc6, 0x04, 0xd6, 0x4e, 0x37, 0x8d, 0xee, 0x31, 0xf3, 0xdd, 0x70, 0xb7,
    0xb1, 0x1d, 0x26, 0x5a, 0xcb, 0x45, 0xf9, 0x8f,
};
static const unsigned char kat319_addin1[] = {
    0x49, 0x3e, 0x62, 0x62, 0x78, 0xf6, 0xaf, 0x2a, 0xb9, 0x9c, 0x53, 0x35,
    0xc2, 0x9d, 0xac, 0x38, 0x25, 0xf8, 0x52, 0xaa, 0xcd, 0x3b, 0x23, 0x7a,
    0x3f, 0x01, 0x4b, 0x9a, 0xdf, 0xbe, 0x4d, 0x4a,
};
static const unsigned char kat319_retbits[] = {
    0x07, 0x6a, 0xa4, 0x2c, 0x35, 0x03, 0x55, 0x75, 0xf1, 0xf0, 0xa4, 0x92,
    0x28, 0x8d, 0xbb, 0x9c, 0x46, 0x28, 0xc0, 0x91, 0xfc, 0xb3, 0x4e, 0xb6,
    0xb8, 0x03, 0x9a, 0x88, 0x7b, 0xa1, 0xed, 0x8e, 0x00, 0x29, 0xb2, 0xd1,
    0x8c, 0x3c, 0x1c, 0x32, 0x1d, 0x7f, 0x0e, 0x20, 0xb7, 0xdb, 0xcc, 0xea,
    0x88, 0x6a, 0xd5, 0x18, 0xae, 0xff, 0x08, 0x9d, 0x44, 0x70, 0x9e, 0xaa,
    0x58, 0x3e, 0x45, 0xef,
};
static const struct drbg_kat_no_reseed kat319_t = {
    3, kat319_entropyin, kat319_nonce, kat319_persstr,
    kat319_addin0, kat319_addin1, kat319_retbits
};
static const struct drbg_kat kat319 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat319_t
};

static const unsigned char kat320_entropyin[] = {
    0x79, 0xb2, 0x03, 0x9d, 0xf2, 0x89, 0x3b, 0x45, 0xd7, 0xc4, 0xbe, 0x89,
    0xd0, 0xe0, 0xac, 0xf4, 0x1f, 0xb1, 0x1c, 0xf7, 0x0b, 0x12, 0xdb, 0xe4,
};
static const unsigned char kat320_nonce[] = {
    0x35, 0xd1, 0x9a, 0x10, 0x42, 0x0d, 0x53, 0x97, 0xd1, 0xa6, 0xfa, 0x35,
    0xa7, 0x67, 0x6e, 0xe6,
};
static const unsigned char kat320_persstr[] = {0};
static const unsigned char kat320_addin0[] = {
    0xe3, 0x38, 0x4b, 0x09, 0xc4, 0xde, 0xad, 0xf0, 0x94, 0x67, 0xfb, 0x4c,
    0x68, 0xf2, 0x89, 0x8e, 0xd6, 0x41, 0x28, 0x0c, 0x2c, 0x61, 0xd8, 0xad,
    0x7f, 0x86, 0xbe, 0xd0, 0x56, 0x69, 0x27, 0xb3,
};
static const unsigned char kat320_addin1[] = {
    0x39, 0xda, 0x7d, 0x65, 0x30, 0x08, 0xb8, 0xb5, 0x99, 0xcd, 0xda, 0x25,
    0xcc, 0xba, 0x4d, 0x2d, 0x15, 0xa9, 0x6a, 0xdc, 0x1d, 0x12, 0xac, 0x13,
    0x1c, 0xbc, 0x43, 0x64, 0x88, 0xe1, 0xf7, 0xb0,
};
static const unsigned char kat320_retbits[] = {
    0xbe, 0xc5, 0x5b, 0x20, 0x67, 0xac, 0x47, 0x0d, 0xba, 0x7b, 0x2a, 0xe7,
    0xc7, 0x04, 0x3c, 0xa0, 0x41, 0x59, 0x95, 0x7b, 0xc5, 0x47, 0x28, 0x16,
    0x57, 0x41, 0xd0, 0x57, 0xf8, 0x09, 0xbf, 0x56, 0x7b, 0x9d, 0x7b, 0x58,
    0x89, 0xa5, 0x79, 0x95, 0x8d, 0x6b, 0xf2, 0xea, 0x11, 0xd3, 0x8e, 0xba,
    0x75, 0x43, 0x42, 0x0f, 0x80, 0x94, 0x8f, 0x71, 0x20, 0xd9, 0x64, 0x07,
    0x60, 0x67, 0xbe, 0xe7,
};
static const struct drbg_kat_no_reseed kat320_t = {
    4, kat320_entropyin, kat320_nonce, kat320_persstr,
    kat320_addin0, kat320_addin1, kat320_retbits
};
static const struct drbg_kat kat320 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat320_t
};

static const unsigned char kat321_entropyin[] = {
    0x6a, 0x91, 0x0a, 0x86, 0x59, 0x1c, 0xf0, 0x55, 0x21, 0xc8, 0x9f, 0xa0,
    0x6b, 0x0f, 0xd7, 0xeb, 0xe3, 0xc8, 0xf1, 0xe0, 0xde, 0x18, 0x09, 0xe9,
};
static const unsigned char kat321_nonce[] = {
    0x44, 0x1e, 0xa2, 0x6f, 0xe9, 0x46, 0x39, 0x08, 0x5b, 0x0d, 0x8e, 0x28,
    0x22, 0x4b, 0xf1, 0x11,
};
static const unsigned char kat321_persstr[] = {0};
static const unsigned char kat321_addin0[] = {
    0x29, 0x28, 0xcc, 0xab, 0x0d, 0xde, 0xb1, 0xe9, 0x68, 0x1f, 0xd9, 0x40,
    0x3a, 0x48, 0xe7, 0x34, 0xc3, 0x47, 0xae, 0x80, 0x45, 0x0b, 0xa8, 0x4b,
    0x2c, 0x3a, 0x68, 0x6c, 0x1a, 0xf6, 0xf2, 0x03,
};
static const unsigned char kat321_addin1[] = {
    0x22, 0xcf, 0x81, 0x0f, 0xff, 0x93, 0x50, 0xb5, 0xd3, 0xb3, 0x45, 0x24,
    0x76, 0x67, 0x0a, 0xb9, 0x64, 0x88, 0xb8, 0x6d, 0x64, 0x43, 0xdd, 0x87,
    0xc4, 0xc2, 0x49, 0xa7, 0x9e, 0xf1, 0xfa, 0xce,
};
static const unsigned char kat321_retbits[] = {
    0xa7, 0xb0, 0xa1, 0x39, 0xcd, 0x26, 0x27, 0x4c, 0x0f, 0xd5, 0xb7, 0x09,
    0x0a, 0xe5, 0xe6, 0x07, 0x5c, 0xd7, 0x8d, 0x93, 0x49, 0x77, 0x75, 0x11,
    0x32, 0xc0, 0xb8, 0xa7, 0xf2, 0x7b, 0x5f, 0xf1, 0xe8, 0x68, 0xe2, 0x01,
    0x76, 0x52, 0x31, 0x04, 0xe2, 0x34, 0x60, 0x27, 0xc4, 0xef, 0x3a, 0x69,
    0x72, 0xac, 0x75, 0x63, 0x28, 0xdc, 0x5f, 0x1c, 0xe1, 0x52, 0xb0, 0x1f,
    0x48, 0x61, 0xb5, 0x07,
};
static const struct drbg_kat_no_reseed kat321_t = {
    5, kat321_entropyin, kat321_nonce, kat321_persstr,
    kat321_addin0, kat321_addin1, kat321_retbits
};
static const struct drbg_kat kat321 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat321_t
};

static const unsigned char kat322_entropyin[] = {
    0x70, 0x87, 0x4f, 0xaa, 0x74, 0xd1, 0xa1, 0xaf, 0x88, 0x2a, 0x36, 0x96,
    0xff, 0x2f, 0x40, 0x9f, 0x9c, 0x2e, 0x05, 0x96, 0xcc, 0xe1, 0xfb, 0x9c,
};
static const unsigned char kat322_nonce[] = {
    0x77, 0xc4, 0x98, 0xe0, 0xd5, 0x1a, 0xf0, 0xc6, 0x87, 0x36, 0x45, 0xfd,
    0x88, 0x89, 0xfc, 0x58,
};
static const unsigned char kat322_persstr[] = {0};
static const unsigned char kat322_addin0[] = {
    0xe6, 0xed, 0x21, 0x0e, 0x04, 0x59, 0x16, 0xcf, 0x1a, 0x69, 0x1e, 0x36,
    0x69, 0x71, 0x22, 0x89, 0x04, 0x85, 0xf2, 0x85, 0x2a, 0x27, 0x85, 0x11,
    0x25, 0xeb, 0x17, 0xdb, 0xf5, 0x79, 0x1f, 0x91,
};
static const unsigned char kat322_addin1[] = {
    0x1f, 0x0d, 0x29, 0x59, 0xf9, 0x72, 0x56, 0x04, 0xe6, 0xa1, 0x3b, 0x7e,
    0x2e, 0x0d, 0x72, 0xad, 0xc8, 0xbb, 0xdf, 0xb4, 0xa3, 0x1b, 0x9b, 0x38,
    0xbd, 0x8c, 0xae, 0x26, 0x4d, 0x4d, 0xcb, 0x01,
};
static const unsigned char kat322_retbits[] = {
    0xb6, 0xef, 0x96, 0x03, 0xce, 0x5c, 0xb3, 0xb3, 0xde, 0x76, 0x87, 0xcc,
    0x73, 0x88, 0x32, 0xa1, 0x1c, 0x39, 0x8e, 0x73, 0x30, 0xaa, 0x55, 0x2f,
    0x67, 0x8e, 0x13, 0x2c, 0x71, 0xb7, 0xc0, 0x20, 0x98, 0x0c, 0x3c, 0xab,
    0x47, 0x3d, 0x17, 0xc8, 0x43, 0x99, 0xca, 0x19, 0x2e, 0x6b, 0x29, 0x35,
    0xde, 0x21, 0x1e, 0x17, 0x9f, 0x39, 0xaa, 0xb3, 0x05, 0x52, 0xc9, 0x7e,
    0x11, 0x1f, 0xf3, 0xc2,
};
static const struct drbg_kat_no_reseed kat322_t = {
    6, kat322_entropyin, kat322_nonce, kat322_persstr,
    kat322_addin0, kat322_addin1, kat322_retbits
};
static const struct drbg_kat kat322 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat322_t
};

static const unsigned char kat323_entropyin[] = {
    0xaa, 0x69, 0x94, 0x9c, 0x60, 0x5a, 0x8c, 0x99, 0x7a, 0xa6, 0x50, 0x76,
    0xe8, 0xce, 0xb6, 0x3f, 0xc9, 0xd0, 0xaf, 0x84, 0x0e, 0xa8, 0x73, 0xbb,
};
static const unsigned char kat323_nonce[] = {
    0x6b, 0xe6, 0xf8, 0x9f, 0x54, 0x9f, 0x0e, 0xb6, 0xdf, 0xd7, 0x61, 0xf0,
    0x9b, 0xfb, 0xcc, 0xb5,
};
static const unsigned char kat323_persstr[] = {0};
static const unsigned char kat323_addin0[] = {
    0xfa, 0x18, 0x0a, 0x06, 0xf0, 0x7f, 0xcd, 0xcc, 0x95, 0xff, 0x37, 0x11,
    0xc3, 0x48, 0xc1, 0x5b, 0xae, 0x75, 0x0b, 0x8c, 0xbd, 0x98, 0xe4, 0x3a,
    0xbe, 0x61, 0xfe, 0x15, 0x4f, 0x3f, 0x5c, 0xd5,
};
static const unsigned char kat323_addin1[] = {
    0xa5, 0x59, 0xc3, 0xf7, 0xcc, 0x60, 0x44, 0xed, 0xc0, 0x29, 0xa3, 0x5b,
    0xf0, 0x2a, 0x3a, 0x96, 0xad, 0x01, 0x83, 0x86, 0xe1, 0xbd, 0x99, 0x8a,
    0xf1, 0x08, 0xc8, 0xfd, 0xba, 0x6f, 0x78, 0xa1,
};
static const unsigned char kat323_retbits[] = {
    0xa1, 0x8f, 0x0f, 0xfe, 0xc3, 0x92, 0x70, 0x6f, 0xc8, 0x37, 0xc0, 0xa5,
    0x32, 0x15, 0x58, 0x08, 0x62, 0x32, 0xf7, 0xf6, 0x94, 0x70, 0xb6, 0xd2,
    0x99, 0xd3, 0x2b, 0xba, 0x24, 0xa3, 0xb1, 0xd5, 0x44, 0x1e, 0x27, 0xf0,
    0x83, 0x57, 0xb0, 0xe7, 0x0f, 0x00, 0x0c, 0x32, 0x1b, 0x70, 0x81, 0xdc,
    0xff, 0xad, 0x82, 0x4d, 0xe1, 0x30, 0x47, 0x32, 0x2d, 0xf5, 0xf7, 0xf4,
    0xaf, 0x25, 0x84, 0x3b,
};
static const struct drbg_kat_no_reseed kat323_t = {
    7, kat323_entropyin, kat323_nonce, kat323_persstr,
    kat323_addin0, kat323_addin1, kat323_retbits
};
static const struct drbg_kat kat323 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat323_t
};

static const unsigned char kat324_entropyin[] = {
    0xd4, 0x17, 0x6b, 0x63, 0xa9, 0x19, 0x97, 0x39, 0xff, 0x7d, 0x7f, 0x46,
    0x20, 0x3a, 0x4a, 0x01, 0xbf, 0xcc, 0x34, 0xfa, 0x7b, 0xd7, 0x01, 0x9f,
};
static const unsigned char kat324_nonce[] = {
    0x05, 0x06, 0x01, 0xd5, 0x0c, 0xd5, 0x04, 0x4f, 0x97, 0x8d, 0xfe, 0x9e,
    0xe2, 0x69, 0x5a, 0x44,
};
static const unsigned char kat324_persstr[] = {0};
static const unsigned char kat324_addin0[] = {
    0xa7, 0x6a, 0x67, 0x40, 0x9e, 0x51, 0x64, 0xea, 0x0a, 0xba, 0xd6, 0x2a,
    0x41, 0x8a, 0xbd, 0x8c, 0xd2, 0xe7, 0x29, 0xb6, 0x34, 0x82, 0xf2, 0xae,
    0xaf, 0xc0, 0xa3, 0x65, 0x0e, 0xa8, 0xb9, 0x52,
};
static const unsigned char kat324_addin1[] = {
    0x83, 0xc9, 0x37, 0x10, 0x55, 0xee, 0x7f, 0xbb, 0x93, 0x54, 0x83, 0x76,
    0x2e, 0x16, 0xe2, 0xd8, 0x5d, 0xcd, 0x70, 0xe1, 0x34, 0x97, 0xe6, 0x9a,
    0xab, 0x9a, 0xd3, 0x50, 0x01, 0x62, 0x7a, 0x4b,
};
static const unsigned char kat324_retbits[] = {
    0xe7, 0xb6, 0x7e, 0x05, 0x2c, 0x3b, 0x43, 0x97, 0x42, 0xdc, 0x52, 0x28,
    0x9f, 0x38, 0x84, 0x53, 0x4a, 0x0e, 0x85, 0xb8, 0x34, 0x19, 0x30, 0x8b,
    0xe0, 0xb4, 0xe6, 0x32, 0x77, 0x9b, 0x4c, 0x72, 0x2a, 0x4d, 0x8b, 0xd3,
    0x8c, 0x8f, 0xeb, 0x3a, 0x5c, 0x21, 0x33, 0x0d, 0xb1, 0xa2, 0xf5, 0xbd,
    0x6a, 0xde, 0x1e, 0x0a, 0x77, 0x88, 0x13, 0x9f, 0x9f, 0x0e, 0x52, 0xf2,
    0xe0, 0x1e, 0xf7, 0x59,
};
static const struct drbg_kat_no_reseed kat324_t = {
    8, kat324_entropyin, kat324_nonce, kat324_persstr,
    kat324_addin0, kat324_addin1, kat324_retbits
};
static const struct drbg_kat kat324 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat324_t
};

static const unsigned char kat325_entropyin[] = {
    0xe1, 0xae, 0xa3, 0x22, 0x9c, 0x6c, 0x18, 0x1e, 0x32, 0xaf, 0x65, 0x00,
    0xae, 0x4d, 0xb0, 0x93, 0xd1, 0xd5, 0x70, 0x0d, 0x64, 0x7e, 0x27, 0x24,
};
static const unsigned char kat325_nonce[] = {
    0x65, 0x6c, 0x62, 0x42, 0xbb, 0x52, 0x0a, 0x88, 0x64, 0x09, 0xca, 0x4f,
    0x3a, 0xf4, 0x0a, 0x1c,
};
static const unsigned char kat325_persstr[] = {0};
static const unsigned char kat325_addin0[] = {
    0x57, 0x48, 0x3f, 0x32, 0x65, 0x38, 0x02, 0xb2, 0xbf, 0xc8, 0x06, 0xd0,
    0x28, 0xf2, 0xb1, 0x68, 0x27, 0x1e, 0x55, 0xd4, 0x9a, 0x43, 0x6a, 0x5c,
    0x4e, 0x89, 0xc2, 0x0c, 0xaa, 0x0c, 0xa1, 0x07,
};
static const unsigned char kat325_addin1[] = {
    0x5c, 0xc0, 0x20, 0x2e, 0x9f, 0x5f, 0x19, 0xf4, 0x01, 0x0b, 0x68, 0xde,
    0x2b, 0x2c, 0x28, 0xc1, 0x99, 0x41, 0xce, 0xc8, 0xcf, 0x31, 0xf5, 0x9c,
    0x29, 0x9c, 0x89, 0x58, 0xe0, 0x7c, 0xed, 0xd7,
};
static const unsigned char kat325_retbits[] = {
    0x5b, 0xb5, 0x65, 0x0e, 0xcb, 0x99, 0x68, 0x2a, 0xfe, 0xf4, 0xba, 0x87,
    0xc2, 0x27, 0x27, 0xee, 0x7a, 0xb0, 0x80, 0xbc, 0xab, 0x60, 0xf1, 0x57,
    0x6e, 0x46, 0x27, 0xf7, 0xe9, 0x05, 0xa2, 0x48, 0xd7, 0x00, 0x29, 0xca,
    0xf3, 0x71, 0xab, 0x58, 0x98, 0x82, 0x3a, 0x35, 0xee, 0x5b, 0x81, 0x18,
    0x2c, 0x24, 0x03, 0x1d, 0xc3, 0x96, 0x6d, 0x7c, 0x24, 0x4b, 0x28, 0x80,
    0x3b, 0x36, 0x37, 0x00,
};
static const struct drbg_kat_no_reseed kat325_t = {
    9, kat325_entropyin, kat325_nonce, kat325_persstr,
    kat325_addin0, kat325_addin1, kat325_retbits
};
static const struct drbg_kat kat325 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat325_t
};

static const unsigned char kat326_entropyin[] = {
    0x9c, 0xf9, 0x22, 0x67, 0x96, 0x9b, 0x13, 0x62, 0x4a, 0x01, 0x6a, 0xfc,
    0xbe, 0xdc, 0x99, 0x64, 0xc0, 0x71, 0xb6, 0xd5, 0x7b, 0x9b, 0xc1, 0xaa,
};
static const unsigned char kat326_nonce[] = {
    0x5f, 0xe0, 0xed, 0x8b, 0xec, 0xd3, 0xbb, 0xb4, 0x4a, 0x0d, 0xfa, 0x8d,
    0xe9, 0xcb, 0x47, 0xff,
};
static const unsigned char kat326_persstr[] = {0};
static const unsigned char kat326_addin0[] = {
    0x3a, 0x10, 0xc0, 0x87, 0xcf, 0x4d, 0x09, 0xd1, 0x3c, 0x93, 0x3b, 0x73,
    0xce, 0x0c, 0xb2, 0x3d, 0x35, 0x21, 0x2a, 0x09, 0x80, 0xca, 0x88, 0xa9,
    0x5f, 0x71, 0xc0, 0x82, 0x61, 0x28, 0x99, 0x24,
};
static const unsigned char kat326_addin1[] = {
    0x23, 0xa3, 0x91, 0x51, 0x60, 0xa4, 0xc8, 0xc2, 0x2a, 0xdb, 0x26, 0xa8,
    0x6e, 0x29, 0x6f, 0x23, 0x63, 0x60, 0x12, 0x44, 0xfb, 0xc8, 0xcf, 0xed,
    0x07, 0x57, 0xf9, 0x31, 0xe4, 0x95, 0xa6, 0xb9,
};
static const unsigned char kat326_retbits[] = {
    0x65, 0xba, 0x2a, 0x31, 0x91, 0x1a, 0x00, 0x98, 0x33, 0x0a, 0xd9, 0x14,
    0xe7, 0x9c, 0x71, 0x4b, 0xb1, 0xb9, 0xa0, 0x06, 0x15, 0x51, 0xf9, 0x81,
    0xf3, 0xde, 0x6a, 0x88, 0x08, 0x74, 0x73, 0x72, 0xbe, 0xd7, 0xb5, 0xcc,
    0xb4, 0x53, 0x81, 0xab, 0x98, 0xb1, 0xc5, 0x56, 0x59, 0xb0, 0x23, 0x42,
    0x2b, 0x75, 0x39, 0x83, 0xb2, 0xd3, 0x54, 0x55, 0x98, 0xf3, 0x11, 0xfd,
    0x2a, 0xcd, 0xdb, 0x32,
};
static const struct drbg_kat_no_reseed kat326_t = {
    10, kat326_entropyin, kat326_nonce, kat326_persstr,
    kat326_addin0, kat326_addin1, kat326_retbits
};
static const struct drbg_kat kat326 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat326_t
};

static const unsigned char kat327_entropyin[] = {
    0xf1, 0x3d, 0xa4, 0x54, 0xa1, 0xa1, 0x68, 0x51, 0x88, 0xce, 0x40, 0x2a,
    0xfc, 0x5f, 0x1a, 0xb8, 0xc2, 0x32, 0x40, 0xb2, 0x2c, 0x57, 0x0f, 0xe8,
};
static const unsigned char kat327_nonce[] = {
    0x65, 0x4d, 0x75, 0xd1, 0x03, 0xfb, 0xf3, 0x58, 0x9a, 0x66, 0x64, 0x24,
    0x7f, 0x6f, 0x32, 0xd7,
};
static const unsigned char kat327_persstr[] = {0};
static const unsigned char kat327_addin0[] = {
    0x0f, 0x72, 0x58, 0x42, 0xcc, 0x60, 0x61, 0x97, 0xd3, 0xc3, 0xa3, 0x72,
    0x60, 0xfd, 0x24, 0x23, 0x71, 0x98, 0xb7, 0xe4, 0x86, 0x80, 0x1d, 0x32,
    0xa8, 0xe1, 0x55, 0xca, 0x22, 0xf8, 0x1e, 0xac,
};
static const unsigned char kat327_addin1[] = {
    0x78, 0x8e, 0xe6, 0xcd, 0x54, 0x3e, 0x8b, 0x1f, 0x69, 0xc6, 0x98, 0xb4,
    0xf9, 0xcc, 0xc3, 0x3f, 0x76, 0xf6, 0x63, 0x99, 0xe8, 0xbd, 0xf8, 0x9e,
    0xe2, 0xb6, 0x29, 0x54, 0x0f, 0xf7, 0x58, 0x50,
};
static const unsigned char kat327_retbits[] = {
    0xcc, 0xf1, 0x24, 0x79, 0x17, 0x7b, 0xbe, 0xca, 0xf4, 0x58, 0x50, 0xf2,
    0x30, 0x76, 0x8c, 0x97, 0x01, 0x91, 0xb7, 0x1b, 0x98, 0xf4, 0x68, 0x5f,
    0x47, 0x98, 0x49, 0x6c, 0x5c, 0x87, 0x7b, 0xc7, 0xa5, 0xda, 0xe4, 0x08,
    0x86, 0x27, 0x26, 0xac, 0xa4, 0xbb, 0xaf, 0xf3, 0x5c, 0x20, 0x08, 0x96,
    0x58, 0x6a, 0x4e, 0xad, 0x49, 0x3a, 0x4b, 0x4e, 0xb8, 0xaf, 0xb0, 0x1d,
    0xed, 0xf6, 0xea, 0xcb,
};
static const struct drbg_kat_no_reseed kat327_t = {
    11, kat327_entropyin, kat327_nonce, kat327_persstr,
    kat327_addin0, kat327_addin1, kat327_retbits
};
static const struct drbg_kat kat327 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat327_t
};

static const unsigned char kat328_entropyin[] = {
    0xff, 0x69, 0x97, 0x35, 0x5e, 0x83, 0x80, 0xc6, 0xf1, 0x0f, 0xf0, 0x52,
    0xb1, 0x85, 0x34, 0xbe, 0x0d, 0x7e, 0x34, 0x4d, 0x11, 0xce, 0x6a, 0x37,
};
static const unsigned char kat328_nonce[] = {
    0xae, 0x30, 0xf9, 0x9e, 0xe7, 0xdb, 0x3f, 0xb0, 0xa6, 0xbe, 0x60, 0x5f,
    0x14, 0xa4, 0xa0, 0xf5,
};
static const unsigned char kat328_persstr[] = {0};
static const unsigned char kat328_addin0[] = {
    0xd3, 0xaa, 0x9a, 0x02, 0xe5, 0x09, 0x31, 0xcd, 0xb6, 0x57, 0xe5, 0xac,
    0x56, 0x57, 0x22, 0xea, 0xc7, 0x69, 0xb1, 0x7e, 0xca, 0x63, 0xff, 0xe9,
    0x16, 0x92, 0x8b, 0x52, 0x6c, 0x58, 0x65, 0x87,
};
static const unsigned char kat328_addin1[] = {
    0x06, 0x79, 0x40, 0xbc, 0xb6, 0x55, 0x3a, 0xf8, 0xdb, 0x6e, 0x21, 0xfa,
    0x49, 0xc9, 0xf9, 0xba, 0x22, 0x27, 0x21, 0x64, 0x22, 0xae, 0x20, 0xdd,
    0xf7, 0x3b, 0x65, 0xa6, 0x65, 0xcd, 0x81, 0xc4,
};
static const unsigned char kat328_retbits[] = {
    0x00, 0x76, 0xb4, 0x66, 0x4e, 0x83, 0xfb, 0xe6, 0x93, 0xd5, 0x1e, 0x31,
    0x2d, 0x92, 0xf4, 0x1a, 0x29, 0xa7, 0x92, 0x16, 0xc1, 0x4d, 0xfa, 0x28,
    0xf6, 0xc2, 0xde, 0xd9, 0xd0, 0xe6, 0x9c, 0x6f, 0x51, 0x29, 0x80, 0xc2,
    0x5d, 0x72, 0x38, 0xcf, 0xbe, 0x6a, 0x39, 0x23, 0xd3, 0x0e, 0x2a, 0xf3,
    0xcb, 0xab, 0x71, 0x34, 0xc6, 0x53, 0x7f, 0xe1, 0x99, 0x91, 0x10, 0x5e,
    0x3d, 0xac, 0x89, 0x4b,
};
static const struct drbg_kat_no_reseed kat328_t = {
    12, kat328_entropyin, kat328_nonce, kat328_persstr,
    kat328_addin0, kat328_addin1, kat328_retbits
};
static const struct drbg_kat kat328 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat328_t
};

static const unsigned char kat329_entropyin[] = {
    0xf7, 0x00, 0xbb, 0x6a, 0x92, 0xec, 0xc9, 0xee, 0x07, 0xd5, 0xc5, 0x96,
    0xbd, 0x66, 0x78, 0x62, 0x45, 0x56, 0x51, 0xb5, 0xe0, 0x9c, 0xc7, 0x16,
};
static const unsigned char kat329_nonce[] = {
    0x41, 0x19, 0x2c, 0x65, 0xd6, 0x6b, 0xa5, 0xf0, 0x9c, 0x03, 0x24, 0x91,
    0xa2, 0xc5, 0x3e, 0xde,
};
static const unsigned char kat329_persstr[] = {0};
static const unsigned char kat329_addin0[] = {
    0x9e, 0x74, 0x82, 0x5c, 0xa0, 0xc3, 0xf9, 0xce, 0xb2, 0x72, 0x15, 0x0a,
    0xa5, 0x1f, 0x9f, 0x92, 0xe1, 0x99, 0xf3, 0x13, 0x3d, 0x91, 0x9d, 0x4f,
    0xa2, 0x16, 0x11, 0x9a, 0x13, 0x30, 0xe8, 0x8d,
};
static const unsigned char kat329_addin1[] = {
    0xc5, 0x94, 0x03, 0x87, 0x0a, 0xa5, 0xb1, 0x48, 0xec, 0x6f, 0x99, 0xff,
    0x8b, 0x80, 0x65, 0x69, 0xe9, 0x9f, 0xa8, 0x36, 0x75, 0x01, 0x71, 0x78,
    0x7b, 0xcc, 0x8c, 0x89, 0x31, 0xaa, 0x82, 0xac,
};
static const unsigned char kat329_retbits[] = {
    0x32, 0x21, 0x7f, 0xf9, 0x37, 0x1a, 0x6c, 0xbf, 0x79, 0x6c, 0x13, 0x8f,
    0x9c, 0xe3, 0xbe, 0x65, 0x8f, 0xba, 0x03, 0xdf, 0x9c, 0x10, 0xcd, 0x3d,
    0xe0, 0x9e, 0x82, 0xd1, 0xd9, 0xce, 0xc4, 0x3a, 0x96, 0x6a, 0xb8, 0x64,
    0xaa, 0xca, 0x56, 0x02, 0x4e, 0x8b, 0x36, 0x46, 0x3a, 0x31, 0x24, 0x56,
    0xb4, 0x21, 0xb2, 0x57, 0x66, 0x3f, 0xa5, 0xd4, 0x2a, 0xdb, 0xfa, 0x4a,
    0x54, 0x10, 0x79, 0x65,
};
static const struct drbg_kat_no_reseed kat329_t = {
    13, kat329_entropyin, kat329_nonce, kat329_persstr,
    kat329_addin0, kat329_addin1, kat329_retbits
};
static const struct drbg_kat kat329 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat329_t
};

static const unsigned char kat330_entropyin[] = {
    0x15, 0x31, 0xb5, 0x6b, 0xff, 0x3f, 0x34, 0x44, 0x96, 0x74, 0xc5, 0xf9,
    0xb3, 0x10, 0x13, 0x8a, 0x51, 0x72, 0x1a, 0x40, 0xa4, 0x63, 0xe5, 0xbe,
};
static const unsigned char kat330_nonce[] = {
    0xeb, 0x78, 0x09, 0xf3, 0xf3, 0xba, 0x6e, 0xb5, 0x77, 0x75, 0xf1, 0x28,
    0x4f, 0xc2, 0x9f, 0x54,
};
static const unsigned char kat330_persstr[] = {0};
static const unsigned char kat330_addin0[] = {
    0x85, 0xe6, 0xaa, 0x85, 0x9f, 0x43, 0x1d, 0xaa, 0xaf, 0xc9, 0x29, 0x9e,
    0xa6, 0x1f, 0xcd, 0xcf, 0xde, 0x71, 0x2f, 0xc2, 0xd0, 0x1d, 0x94, 0xfe,
    0x70, 0xcd, 0x0b, 0xc6, 0xda, 0xb7, 0xf3, 0xe9,
};
static const unsigned char kat330_addin1[] = {
    0xcb, 0x7f, 0xb3, 0x63, 0x3f, 0x42, 0x65, 0x92, 0x01, 0x02, 0x6c, 0x63,
    0x48, 0x94, 0xec, 0xc7, 0x19, 0xc0, 0x00, 0x32, 0x59, 0xa6, 0x92, 0x83,
    0x76, 0x09, 0xab, 0x90, 0x78, 0x2a, 0xe2, 0x3a,
};
static const unsigned char kat330_retbits[] = {
    0x74, 0x45, 0xb8, 0x12, 0xa7, 0xf5, 0xd3, 0xb8, 0xba, 0x79, 0xf7, 0x7a,
    0x80, 0x1a, 0xfc, 0xda, 0xb8, 0x23, 0x14, 0x0e, 0x5e, 0xb4, 0xa6, 0x6e,
    0x50, 0xf8, 0x58, 0x3d, 0x90, 0xdb, 0xad, 0xde, 0xe8, 0x69, 0x7d, 0x37,
    0xc0, 0xfd, 0x62, 0x47, 0xec, 0x6a, 0x3b, 0xf7, 0x66, 0x40, 0x18, 0xa5,
    0x8c, 0x04, 0x2b, 0xc3, 0xb3, 0xbe, 0x91, 0x2e, 0x88, 0xc2, 0xbe, 0x96,
    0xb9, 0xc6, 0x9e, 0x85,
};
static const struct drbg_kat_no_reseed kat330_t = {
    14, kat330_entropyin, kat330_nonce, kat330_persstr,
    kat330_addin0, kat330_addin1, kat330_retbits
};
static const struct drbg_kat kat330 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat330_t
};

static const unsigned char kat331_entropyin[] = {
    0x50, 0x13, 0x4a, 0x63, 0x80, 0x92, 0xb0, 0x9e, 0x78, 0x70, 0x80, 0x50,
    0xdc, 0x40, 0x44, 0xe6, 0x3c, 0x7a, 0xbd, 0x22, 0x77, 0xbe, 0x3d, 0x71,
};
static const unsigned char kat331_nonce[] = {
    0xda, 0x88, 0x07, 0x00, 0x9d, 0x40, 0x04, 0x82, 0xc6, 0xda, 0x8e, 0x4e,
    0x9a, 0x4e, 0xe3, 0x3c,
};
static const unsigned char kat331_persstr[] = {
    0x8e, 0x01, 0x53, 0xaa, 0xbf, 0xd2, 0xca, 0x23, 0x74, 0xbd, 0xd1, 0xe9,
    0x7f, 0xf2, 0x3e, 0xe2, 0xe7, 0xdb, 0xff, 0x7e, 0x83, 0x6f, 0xa5, 0xad,
    0xb6, 0x57, 0x64, 0xf7, 0x07, 0x8d, 0xf1, 0xb8,
};
static const unsigned char kat331_addin0[] = {0};
static const unsigned char kat331_addin1[] = {0};
static const unsigned char kat331_retbits[] = {
    0x50, 0x27, 0x3a, 0xa2, 0x0f, 0xeb, 0xe8, 0x26, 0x85, 0xd4, 0x9a, 0x01,
    0x3e, 0x75, 0xa8, 0x4b, 0xcc, 0xc7, 0xc2, 0x01, 0x28, 0xbd, 0x09, 0x8a,
    0x22, 0x8c, 0x77, 0x1d, 0x08, 0xbb, 0x53, 0x03, 0xe7, 0x15, 0xfc, 0x30,
    0x82, 0x3d, 0xae, 0x08, 0x57, 0x80, 0xd6, 0xd2, 0x8d, 0x10, 0x71, 0xa2,
    0x65, 0x08, 0x13, 0x0f, 0x35, 0x25, 0xb3, 0xbf, 0xd5, 0x05, 0xf0, 0x75,
    0x75, 0xad, 0xd8, 0x74,
};
static const struct drbg_kat_no_reseed kat331_t = {
    0, kat331_entropyin, kat331_nonce, kat331_persstr,
    kat331_addin0, kat331_addin1, kat331_retbits
};
static const struct drbg_kat kat331 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat331_t
};

static const unsigned char kat332_entropyin[] = {
    0x9b, 0x6c, 0x7b, 0x1c, 0x72, 0x9c, 0xeb, 0xca, 0xf4, 0xc4, 0xaf, 0xb9,
    0xa3, 0x51, 0xd2, 0xf6, 0xd1, 0xb3, 0x8a, 0xff, 0xc3, 0x5f, 0xdc, 0x2b,
};
static const unsigned char kat332_nonce[] = {
    0xd0, 0x46, 0x71, 0x5f, 0xee, 0x95, 0x6f, 0xa5, 0xe1, 0xa9, 0xd1, 0xaa,
    0xca, 0x6e, 0xd6, 0x7e,
};
static const unsigned char kat332_persstr[] = {
    0x37, 0x6b, 0x88, 0x73, 0xca, 0x51, 0x20, 0x9c, 0x93, 0xfb, 0xfb, 0x15,
    0x89, 0x96, 0x41, 0x2b, 0xc6, 0xcd, 0x1f, 0x41, 0xe9, 0xa2, 0x09, 0x3d,
    0x9e, 0xd6, 0xfb, 0x91, 0xca, 0x3b, 0xa2, 0xc7,
};
static const unsigned char kat332_addin0[] = {0};
static const unsigned char kat332_addin1[] = {0};
static const unsigned char kat332_retbits[] = {
    0x05, 0x92, 0x7f, 0xeb, 0x60, 0x85, 0x51, 0x27, 0xb4, 0xd6, 0x63, 0x2f,
    0x12, 0x17, 0xef, 0x30, 0x72, 0xa0, 0x3c, 0x03, 0x74, 0x0f, 0xdc, 0x14,
    0x1e, 0x56, 0x36, 0x0e, 0xfc, 0xbb, 0xd5, 0x5c, 0x5e, 0xf5, 0x16, 0xe3,
    0x91, 0x3b, 0xb2, 0x0d, 0xc4, 0xda, 0x9e, 0x79, 0x98, 0xb8, 0xb5, 0x93,
    0xe3, 0xa0, 0x21, 0x5d, 0xc0, 0x32, 0x24, 0x12, 0x14, 0xa3, 0x5e, 0x5e,
    0x4d, 0xdd, 0xf1, 0xa0,
};
static const struct drbg_kat_no_reseed kat332_t = {
    1, kat332_entropyin, kat332_nonce, kat332_persstr,
    kat332_addin0, kat332_addin1, kat332_retbits
};
static const struct drbg_kat kat332 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat332_t
};

static const unsigned char kat333_entropyin[] = {
    0x32, 0xf7, 0x9c, 0x64, 0x90, 0x4f, 0x65, 0x16, 0x92, 0x77, 0xcf, 0x0e,
    0xaa, 0xd8, 0xce, 0xe6, 0xa6, 0x13, 0xf9, 0x2f, 0x08, 0x63, 0xb6, 0x62,
};
static const unsigned char kat333_nonce[] = {
    0xf4, 0x19, 0x07, 0xab, 0x3d, 0xd3, 0xfc, 0x0c, 0x33, 0x75, 0x81, 0xdb,
    0xf3, 0xcf, 0x6a, 0x61,
};
static const unsigned char kat333_persstr[] = {
    0xfe, 0x9b, 0xaf, 0x70, 0x43, 0xf1, 0x5c, 0x4e, 0xbe, 0x93, 0x30, 0x43,
    0x2d, 0xac, 0xda, 0xc1, 0x30, 0x6d, 0x42, 0x37, 0xe7, 0x65, 0xd5, 0xb8,
    0x26, 0x36, 0x0a, 0xad, 0x36, 0x84, 0xa2, 0x35,
};
static const unsigned char kat333_addin0[] = {0};
static const unsigned char kat333_addin1[] = {0};
static const unsigned char kat333_retbits[] = {
    0xa2, 0xce, 0x0a, 0xfe, 0x7f, 0xdb, 0xeb, 0x4a, 0xd1, 0x95, 0xc4, 0x61,
    0x0c, 0xae, 0x40, 0x68, 0x83, 0xb6, 0x9c, 0xbc, 0x85, 0x48, 0xa8, 0x2d,
    0x12, 0x2c, 0x46, 0x13, 0xa6, 0x2e, 0xb3, 0x6e, 0x98, 0x6c, 0xaf, 0xce,
    0x10, 0xfc, 0x32, 0x00, 0xae, 0xf2, 0x97, 0xa3, 0xdf, 0xf0, 0x1b, 0xe3,
    0xa5, 0xdf, 0x6c, 0x82, 0x58, 0xc0, 0xa6, 0x01, 0xd8, 0x91, 0x88, 0xd5,
    0xc0, 0x65, 0xce, 0x1e,
};
static const struct drbg_kat_no_reseed kat333_t = {
    2, kat333_entropyin, kat333_nonce, kat333_persstr,
    kat333_addin0, kat333_addin1, kat333_retbits
};
static const struct drbg_kat kat333 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat333_t
};

static const unsigned char kat334_entropyin[] = {
    0xb7, 0x36, 0x7a, 0x69, 0xf9, 0x1c, 0x68, 0xc6, 0x75, 0x64, 0x83, 0x70,
    0x28, 0x5b, 0x0e, 0x48, 0x6a, 0x3a, 0x97, 0x0d, 0x12, 0xc5, 0x81, 0xac,
};
static const unsigned char kat334_nonce[] = {
    0xbd, 0x49, 0x96, 0x7a, 0x92, 0xa0, 0xf3, 0xca, 0x17, 0x31, 0x12, 0x5d,
    0x33, 0x5f, 0x86, 0xd3,
};
static const unsigned char kat334_persstr[] = {
    0x5e, 0x7d, 0x45, 0x32, 0x6c, 0x6e, 0x53, 0xf1, 0x90, 0x2e, 0x0b, 0x5a,
    0x31, 0x4e, 0x59, 0xa0, 0x44, 0x74, 0x78, 0x1a, 0x1a, 0x4c, 0xd3, 0xdc,
    0xf1, 0x3b, 0xe1, 0x78, 0xce, 0x73, 0x70, 0x70,
};
static const unsigned char kat334_addin0[] = {0};
static const unsigned char kat334_addin1[] = {0};
static const unsigned char kat334_retbits[] = {
    0x59, 0x6f, 0x86, 0xa7, 0x8c, 0x76, 0xd6, 0x93, 0xa6, 0x6b, 0xc0, 0x7e,
    0xd1, 0xf0, 0xd9, 0xfd, 0x3b, 0xa3, 0x06, 0xfd, 0xa4, 0xfb, 0x45, 0x6e,
    0x7d, 0xc6, 0x81, 0x29, 0x96, 0xe2, 0xf7, 0xba, 0xe4, 0x57, 0x95, 0xa9,
    0x0d, 0x9a, 0x92, 0xab, 0x4e, 0x06, 0x0d, 0x5b, 0x02, 0xe1, 0x50, 0x7a,
    0xc6, 0x81, 0x49, 0xeb, 0xd6, 0xf2, 0x37, 0xd3, 0xdf, 0x2e, 0x40, 0xa9,
    0xba, 0xf8, 0x91, 0x8c,
};
static const struct drbg_kat_no_reseed kat334_t = {
    3, kat334_entropyin, kat334_nonce, kat334_persstr,
    kat334_addin0, kat334_addin1, kat334_retbits
};
static const struct drbg_kat kat334 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat334_t
};

static const unsigned char kat335_entropyin[] = {
    0xb6, 0x14, 0xaa, 0xf7, 0x99, 0x22, 0xf8, 0x3f, 0x67, 0xa4, 0xd4, 0x9d,
    0xf5, 0xc2, 0x40, 0x5f, 0xb5, 0xa9, 0x71, 0x26, 0xfd, 0x79, 0x2d, 0x80,
};
static const unsigned char kat335_nonce[] = {
    0xcc, 0xc1, 0x86, 0x76, 0x0d, 0x36, 0x33, 0x56, 0x8a, 0x5e, 0x9f, 0x38,
    0xc2, 0xdb, 0x4c, 0xa0,
};
static const unsigned char kat335_persstr[] = {
    0x65, 0x34, 0x7e, 0x04, 0xd7, 0x68, 0x8d, 0x1b, 0x0b, 0x69, 0xa3, 0xb1,
    0x81, 0x61, 0x3e, 0x6a, 0xbc, 0x78, 0x03, 0xc6, 0x48, 0x82, 0xbf, 0x62,
    0xd4, 0x10, 0x38, 0x95, 0x30, 0x06, 0x2e, 0x53,
};
static const unsigned char kat335_addin0[] = {0};
static const unsigned char kat335_addin1[] = {0};
static const unsigned char kat335_retbits[] = {
    0x9d, 0x57, 0x4a, 0xc6, 0x7c, 0xc3, 0x84, 0xa8, 0x8b, 0x5a, 0xa1, 0x5e,
    0x65, 0x6f, 0xe9, 0x4b, 0xc8, 0x0b, 0xb0, 0x0b, 0xfe, 0xe7, 0xfc, 0x79,
    0xaa, 0x2f, 0xa7, 0xd9, 0x8e, 0x6d, 0x87, 0x45, 0xc0, 0x38, 0x1e, 0xab,
    0x01, 0x06, 0x3b, 0x18, 0x90, 0xed, 0xac, 0x7e, 0xf3, 0x0e, 0x34, 0xb5,
    0xde, 0xdf, 0xe9, 0xb1, 0xc7, 0xf2, 0x1d, 0x48, 0x4b, 0x2f, 0x47, 0x0c,
    0xbe, 0x7b, 0xac, 0x9d,
};
static const struct drbg_kat_no_reseed kat335_t = {
    4, kat335_entropyin, kat335_nonce, kat335_persstr,
    kat335_addin0, kat335_addin1, kat335_retbits
};
static const struct drbg_kat kat335 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat335_t
};

static const unsigned char kat336_entropyin[] = {
    0xa4, 0xcf, 0xbf, 0xc8, 0x94, 0x17, 0xd3, 0x55, 0x98, 0x86, 0xfd, 0xaf,
    0x38, 0x8d, 0x60, 0xb7, 0x82, 0x27, 0xae, 0xad, 0x26, 0xcd, 0x47, 0x2f,
};
static const unsigned char kat336_nonce[] = {
    0xe1, 0x8c, 0xd3, 0x79, 0x48, 0x0e, 0xf2, 0x68, 0x55, 0x2f, 0x74, 0x8e,
    0x9c, 0xc0, 0xf1, 0x33,
};
static const unsigned char kat336_persstr[] = {
    0xe6, 0xc4, 0x92, 0x78, 0x42, 0x42, 0xbc, 0x71, 0xc5, 0x82, 0xe4, 0xbd,
    0x79, 0xcc, 0xad, 0x4c, 0xf6, 0xb1, 0x12, 0x49, 0x01, 0xea, 0x7a, 0x4a,
    0x60, 0x1e, 0x91, 0x00, 0x67, 0x86, 0x41, 0x1a,
};
static const unsigned char kat336_addin0[] = {0};
static const unsigned char kat336_addin1[] = {0};
static const unsigned char kat336_retbits[] = {
    0x05, 0xbd, 0x50, 0x50, 0x20, 0xc3, 0xb1, 0x6d, 0xff, 0xc5, 0x11, 0xd6,
    0x25, 0xbd, 0xb3, 0x6d, 0x46, 0xca, 0x8a, 0x6a, 0xa1, 0xfa, 0xe9, 0x01,
    0x54, 0x46, 0x9a, 0x1c, 0x1a, 0x2c, 0xac, 0x50, 0x59, 0x8a, 0xcc, 0xb9,
    0x94, 0xd0, 0x89, 0x40, 0x26, 0xe4, 0xb3, 0x83, 0xf0, 0xf3, 0x11, 0x88,
    0xa7, 0x98, 0x9f, 0x98, 0xe9, 0xa7, 0x32, 0x95, 0x3c, 0x82, 0xa2, 0xff,
    0x32, 0xbf, 0x54, 0xed,
};
static const struct drbg_kat_no_reseed kat336_t = {
    5, kat336_entropyin, kat336_nonce, kat336_persstr,
    kat336_addin0, kat336_addin1, kat336_retbits
};
static const struct drbg_kat kat336 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat336_t
};

static const unsigned char kat337_entropyin[] = {
    0xfa, 0x72, 0xab, 0x81, 0xa6, 0x3b, 0xa4, 0x90, 0x67, 0x76, 0x84, 0x8a,
    0xd1, 0xfb, 0xc6, 0x72, 0xae, 0xc0, 0x98, 0x3c, 0x9f, 0x09, 0xbe, 0x1f,
};
static const unsigned char kat337_nonce[] = {
    0x9d, 0x5f, 0x58, 0x70, 0x9f, 0xd2, 0x3f, 0xc2, 0x30, 0xdb, 0xd8, 0x33,
    0xac, 0x34, 0x2d, 0x3f,
};
static const unsigned char kat337_persstr[] = {
    0xa6, 0xce, 0x15, 0xe6, 0xb2, 0xce, 0x26, 0x0f, 0x8f, 0xb5, 0xd3, 0xd6,
    0xb9, 0x2a, 0xa2, 0x67, 0xe1, 0xa3, 0x23, 0x6e, 0x09, 0x10, 0xbe, 0x3a,
    0x9a, 0x06, 0xc7, 0xec, 0x86, 0xde, 0x5a, 0x58,
};
static const unsigned char kat337_addin0[] = {0};
static const unsigned char kat337_addin1[] = {0};
static const unsigned char kat337_retbits[] = {
    0xbd, 0x9b, 0x6f, 0x17, 0xc2, 0xa0, 0x6a, 0x7f, 0xa7, 0x42, 0x09, 0x2f,
    0x2d, 0xa8, 0x1f, 0x82, 0x49, 0x89, 0x9d, 0x13, 0xf0, 0x5b, 0x9b, 0xc6,
    0x3e, 0x37, 0x00, 0xf8, 0x16, 0x89, 0xb1, 0x11, 0x3e, 0x1d, 0x32, 0x4e,
    0x3b, 0x34, 0x12, 0xdc, 0xf2, 0xf8, 0x5b, 0x46, 0x9c, 0xbc, 0xe4, 0x4d,
    0x25, 0x9e, 0x71, 0x53, 0x33, 0x9a, 0x47, 0x57, 0xf8, 0x70, 0x31, 0xbf,
    0x59, 0x00, 0x30, 0x47,
};
static const struct drbg_kat_no_reseed kat337_t = {
    6, kat337_entropyin, kat337_nonce, kat337_persstr,
    kat337_addin0, kat337_addin1, kat337_retbits
};
static const struct drbg_kat kat337 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat337_t
};

static const unsigned char kat338_entropyin[] = {
    0x6f, 0x8b, 0xd5, 0x7f, 0x52, 0x3a, 0x9a, 0x7d, 0x1b, 0xff, 0x9e, 0x6e,
    0x21, 0xd9, 0x34, 0xb6, 0x5e, 0x1c, 0x00, 0xf2, 0xb0, 0x69, 0x17, 0xeb,
};
static const unsigned char kat338_nonce[] = {
    0x74, 0x41, 0x0e, 0x8f, 0x55, 0xfd, 0x7d, 0x8c, 0x6e, 0x58, 0xc2, 0x81,
    0xc9, 0x74, 0x8a, 0xdd,
};
static const unsigned char kat338_persstr[] = {
    0x4b, 0x16, 0x63, 0xae, 0x4b, 0x57, 0x58, 0xbd, 0x57, 0x03, 0xc9, 0xae,
    0xed, 0xd7, 0xc9, 0x74, 0x0c, 0xe2, 0x57, 0xf0, 0x06, 0xbc, 0x7b, 0x68,
    0xf9, 0x0f, 0x71, 0xf6, 0x37, 0x19, 0xf7, 0x8a,
};
static const unsigned char kat338_addin0[] = {0};
static const unsigned char kat338_addin1[] = {0};
static const unsigned char kat338_retbits[] = {
    0xdc, 0x84, 0xd5, 0xef, 0x81, 0x94, 0xfa, 0x6c, 0x89, 0x75, 0x35, 0x88,
    0x0b, 0xf4, 0x84, 0x76, 0xf1, 0xe5, 0x31, 0x24, 0xb7, 0xad, 0x72, 0x99,
    0xcc, 0x1f, 0xf5, 0xe8, 0x56, 0x7e, 0xd4, 0xd5, 0x04, 0x1b, 0xa6, 0x2a,
    0x29, 0xb6, 0x32, 0x4e, 0x4a, 0x69, 0x40, 0xab, 0x8f, 0xba, 0xf3, 0x58,
    0xe9, 0xaa, 0x2d, 0xb4, 0x5f, 0x1c, 0x26, 0x69, 0xb7, 0x57, 0xeb, 0x3e,
    0xe9, 0xb9, 0xce, 0x70,
};
static const struct drbg_kat_no_reseed kat338_t = {
    7, kat338_entropyin, kat338_nonce, kat338_persstr,
    kat338_addin0, kat338_addin1, kat338_retbits
};
static const struct drbg_kat kat338 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat338_t
};

static const unsigned char kat339_entropyin[] = {
    0x18, 0xa2, 0x35, 0x03, 0x50, 0xdc, 0x88, 0x05, 0x8d, 0x97, 0x18, 0x33,
    0x55, 0x26, 0xa3, 0x92, 0x59, 0x73, 0xd3, 0x69, 0x2c, 0xec, 0x6c, 0x2d,
};
static const unsigned char kat339_nonce[] = {
    0x8b, 0x0c, 0x04, 0x3b, 0x89, 0x71, 0x7a, 0xca, 0x31, 0x9f, 0x96, 0x72,
    0x1e, 0x45, 0xf0, 0xeb,
};
static const unsigned char kat339_persstr[] = {
    0x38, 0x5f, 0x0d, 0xdb, 0xe8, 0xa3, 0xd5, 0xba, 0x48, 0x0a, 0xbb, 0x7a,
    0xd5, 0x4d, 0x2a, 0xae, 0xa1, 0x29, 0x53, 0xdf, 0x7b, 0xe1, 0xd3, 0x48,
    0xfb, 0x38, 0x83, 0x86, 0xbe, 0x66, 0x51, 0xfa,
};
static const unsigned char kat339_addin0[] = {0};
static const unsigned char kat339_addin1[] = {0};
static const unsigned char kat339_retbits[] = {
    0xbf, 0x27, 0x45, 0xcc, 0x69, 0xc8, 0xe3, 0x76, 0xfe, 0x2d, 0x60, 0x63,
    0x5c, 0xd2, 0xb7, 0xf1, 0x32, 0x4b, 0x58, 0x29, 0xb3, 0xd2, 0x31, 0x49,
    0xed, 0xa9, 0x16, 0xd1, 0x92, 0x61, 0x61, 0xb2, 0x98, 0x8b, 0x6d, 0x01,
    0x43, 0x48, 0x89, 0x12, 0x6a, 0x5f, 0xbe, 0xca, 0xc4, 0xbb, 0xf3, 0x9a,
    0xb3, 0x52, 0x04, 0x2f, 0xa0, 0x78, 0x1a, 0x3b, 0x0b, 0x9d, 0x04, 0x6c,
    0xb9, 0xad, 0xbe, 0x9f,
};
static const struct drbg_kat_no_reseed kat339_t = {
    8, kat339_entropyin, kat339_nonce, kat339_persstr,
    kat339_addin0, kat339_addin1, kat339_retbits
};
static const struct drbg_kat kat339 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat339_t
};

static const unsigned char kat340_entropyin[] = {
    0xcf, 0x2e, 0x38, 0xdd, 0x52, 0x04, 0x1d, 0xe7, 0xb4, 0x48, 0x63, 0x2b,
    0xbc, 0x11, 0xa9, 0x94, 0x79, 0x88, 0x7d, 0x44, 0xf3, 0xcc, 0xfc, 0xaf,
};
static const unsigned char kat340_nonce[] = {
    0x19, 0xc4, 0x29, 0x64, 0x80, 0xca, 0xa9, 0xe1, 0x60, 0x2e, 0x1c, 0x5e,
    0x8f, 0x21, 0x36, 0xb2,
};
static const unsigned char kat340_persstr[] = {
    0x64, 0xd3, 0x8b, 0xe3, 0x66, 0x26, 0xe8, 0x5f, 0xda, 0x02, 0x6a, 0x2a,
    0xdd, 0x7f, 0x98, 0x1b, 0x2e, 0x81, 0x44, 0x9e, 0xba, 0xf4, 0x5b, 0xfb,
    0xb5, 0x80, 0xab, 0x07, 0x62, 0x65, 0xb1, 0x61,
};
static const unsigned char kat340_addin0[] = {0};
static const unsigned char kat340_addin1[] = {0};
static const unsigned char kat340_retbits[] = {
    0x68, 0x77, 0x58, 0xf9, 0x5c, 0x73, 0x92, 0x86, 0x77, 0x23, 0x5d, 0x46,
    0x99, 0x4e, 0x90, 0x23, 0x13, 0xd4, 0x24, 0x61, 0x01, 0x69, 0x53, 0x59,
    0xcd, 0x81, 0xbd, 0x03, 0x5f, 0xda, 0xd3, 0xe8, 0xd8, 0xdc, 0x91, 0xd9,
    0x07, 0x63, 0x0e, 0xee, 0xfc, 0xfa, 0x44, 0x44, 0x5b, 0x00, 0xd1, 0x40,
    0x71, 0x49, 0xa1, 0x60, 0x6e, 0xdd, 0x72, 0x84, 0xaf, 0xca, 0xb5, 0xc8,
    0x69, 0xd7, 0x62, 0xfb,
};
static const struct drbg_kat_no_reseed kat340_t = {
    9, kat340_entropyin, kat340_nonce, kat340_persstr,
    kat340_addin0, kat340_addin1, kat340_retbits
};
static const struct drbg_kat kat340 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat340_t
};

static const unsigned char kat341_entropyin[] = {
    0x75, 0xb3, 0xf1, 0x49, 0x96, 0xb0, 0x8a, 0x9c, 0xb2, 0x11, 0x11, 0xc1,
    0x05, 0x64, 0xf1, 0x24, 0x2f, 0xbc, 0x7e, 0xc2, 0xc1, 0x14, 0x56, 0xdd,
};
static const unsigned char kat341_nonce[] = {
    0x87, 0xac, 0x27, 0x4b, 0xad, 0x1f, 0x34, 0x7e, 0x05, 0x3f, 0x5f, 0x24,
    0x2e, 0x5b, 0x2a, 0x49,
};
static const unsigned char kat341_persstr[] = {
    0x1d, 0x4b, 0x88, 0xab, 0xff, 0xd4, 0x80, 0xbe, 0xdf, 0x4e, 0x4f, 0xcb,
    0xd8, 0x6e, 0x2c, 0xd3, 0x8c, 0x18, 0x3c, 0x93, 0xf2, 0xef, 0xe6, 0x21,
    0x8e, 0x94, 0x97, 0x63, 0xea, 0xfb, 0xa9, 0x81,
};
static const unsigned char kat341_addin0[] = {0};
static const unsigned char kat341_addin1[] = {0};
static const unsigned char kat341_retbits[] = {
    0x89, 0x58, 0x44, 0x32, 0x63, 0xf9, 0x4b, 0x01, 0x96, 0x34, 0xe3, 0x7f,
    0x7e, 0x5d, 0xec, 0x3b, 0x9f, 0xa7, 0xce, 0x24, 0xd0, 0xbf, 0xf6, 0x1c,
    0x5b, 0x8e, 0xe5, 0x4d, 0x0f, 0x29, 0x91, 0xfd, 0x2a, 0xf7, 0x2e, 0xf9,
    0x3b, 0x82, 0x2c, 0x55, 0xbd, 0xd2, 0xd2, 0x0e, 0xd3, 0xa7, 0x89, 0x05,
    0xa8, 0xc6, 0x01, 0xb4, 0xad, 0xd9, 0x8e, 0x66, 0x59, 0xb2, 0x17, 0x4c,
    0x45, 0x8c, 0x25, 0x87,
};
static const struct drbg_kat_no_reseed kat341_t = {
    10, kat341_entropyin, kat341_nonce, kat341_persstr,
    kat341_addin0, kat341_addin1, kat341_retbits
};
static const struct drbg_kat kat341 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat341_t
};

static const unsigned char kat342_entropyin[] = {
    0x82, 0x0c, 0xcc, 0x71, 0xe2, 0x47, 0x27, 0x11, 0xe6, 0x3d, 0x44, 0x58,
    0xcc, 0x0b, 0x84, 0x66, 0xba, 0x0a, 0x79, 0xc2, 0x43, 0x9a, 0x8b, 0x9b,
};
static const unsigned char kat342_nonce[] = {
    0x98, 0xed, 0x21, 0xc6, 0x7c, 0x7c, 0x8d, 0x19, 0xbc, 0x21, 0x83, 0x70,
    0x58, 0x30, 0x6b, 0xc9,
};
static const unsigned char kat342_persstr[] = {
    0xa3, 0xdf, 0xf1, 0x63, 0x0c, 0x7f, 0xee, 0x3d, 0x69, 0x6a, 0x35, 0xfc,
    0xb6, 0x75, 0x4b, 0x63, 0xbd, 0xf1, 0x6c, 0x0a, 0x84, 0x9a, 0x54, 0x05,
    0x59, 0xea, 0xf3, 0x50, 0xa8, 0xa0, 0x3a, 0x80,
};
static const unsigned char kat342_addin0[] = {0};
static const unsigned char kat342_addin1[] = {0};
static const unsigned char kat342_retbits[] = {
    0x67, 0x01, 0xd4, 0x2f, 0xd9, 0xbe, 0x4a, 0x6e, 0xf9, 0x75, 0x0a, 0x5e,
    0xd6, 0x81, 0x7f, 0xd1, 0x6c, 0x06, 0xe3, 0x79, 0x1f, 0x4e, 0x7e, 0xd6,
    0xbf, 0x7e, 0x07, 0x84, 0x23, 0x9e, 0xb5, 0x5b, 0xbd, 0x5b, 0xf8, 0xbf,
    0x75, 0x7f, 0x73, 0x9a, 0x53, 0xd9, 0xe2, 0xce, 0xcc, 0x85, 0xdb, 0x1c,
    0x35, 0x72, 0x7e, 0xf0, 0x98, 0xd1, 0x9c, 0x09, 0x00, 0x0a, 0x7f, 0xb6,
    0x3a, 0x83, 0x6c, 0x68,
};
static const struct drbg_kat_no_reseed kat342_t = {
    11, kat342_entropyin, kat342_nonce, kat342_persstr,
    kat342_addin0, kat342_addin1, kat342_retbits
};
static const struct drbg_kat kat342 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat342_t
};

static const unsigned char kat343_entropyin[] = {
    0x2a, 0xb2, 0x2f, 0x44, 0xac, 0x41, 0x93, 0xb1, 0x11, 0x15, 0x52, 0xd2,
    0x95, 0xe2, 0x41, 0xed, 0xf3, 0x7d, 0x6f, 0x4c, 0x28, 0xbb, 0xb3, 0xfc,
};
static const unsigned char kat343_nonce[] = {
    0x83, 0xb2, 0xdd, 0xa7, 0x9e, 0x88, 0xac, 0xb5, 0xb9, 0xbf, 0xf1, 0xbb,
    0xf6, 0x6f, 0x38, 0x78,
};
static const unsigned char kat343_persstr[] = {
    0xa6, 0xb0, 0xa7, 0x32, 0x04, 0xbc, 0x0d, 0xcc, 0x83, 0xfa, 0x2f, 0x48,
    0x0d, 0xb3, 0x71, 0xeb, 0x5b, 0x18, 0x3d, 0xaf, 0x59, 0x96, 0xed, 0xea,
    0xae, 0xb0, 0x9b, 0x82, 0x1a, 0x51, 0x66, 0x20,
};
static const unsigned char kat343_addin0[] = {0};
static const unsigned char kat343_addin1[] = {0};
static const unsigned char kat343_retbits[] = {
    0x69, 0x0e, 0x7f, 0x00, 0xc5, 0x57, 0xf7, 0x14, 0x29, 0xbc, 0x36, 0x42,
    0x5e, 0xd7, 0xd6, 0x21, 0x5d, 0x0f, 0xc1, 0x81, 0x29, 0xcc, 0x0f, 0x93,
    0x5f, 0xfb, 0x2b, 0xc4, 0xfb, 0x2c, 0x2b, 0x88, 0x3d, 0x6d, 0x95, 0xdd,
    0xcc, 0x07, 0x1f, 0xff, 0x5a, 0xd1, 0x99, 0x66, 0x09, 0x68, 0x0a, 0x7c,
    0xdc, 0xf5, 0x09, 0x9c, 0x88, 0x37, 0x1b, 0x6a, 0xc0, 0x65, 0x08, 0xd3,
    0x52, 0xcb, 0x91, 0x05,
};
static const struct drbg_kat_no_reseed kat343_t = {
    12, kat343_entropyin, kat343_nonce, kat343_persstr,
    kat343_addin0, kat343_addin1, kat343_retbits
};
static const struct drbg_kat kat343 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat343_t
};

static const unsigned char kat344_entropyin[] = {
    0xe5, 0x99, 0x5b, 0x25, 0xb4, 0x74, 0x12, 0xc2, 0xcb, 0x2e, 0xcd, 0x2a,
    0x32, 0xb4, 0x06, 0x12, 0x4d, 0x90, 0xba, 0x5e, 0x26, 0xe6, 0x5b, 0x87,
};
static const unsigned char kat344_nonce[] = {
    0x12, 0x64, 0x3e, 0xa9, 0xfd, 0x57, 0x40, 0xa7, 0x38, 0x6f, 0x5f, 0x6a,
    0x01, 0xe6, 0xfe, 0x44,
};
static const unsigned char kat344_persstr[] = {
    0x3c, 0xef, 0x84, 0x53, 0x15, 0xcd, 0x98, 0x32, 0x2e, 0x2a, 0x28, 0xa4,
    0x44, 0x68, 0xcc, 0x14, 0xab, 0x68, 0x05, 0x03, 0xa6, 0xfa, 0x89, 0xad,
    0x4f, 0x87, 0x6b, 0x0e, 0x2c, 0x4f, 0x3c, 0xa7,
};
static const unsigned char kat344_addin0[] = {0};
static const unsigned char kat344_addin1[] = {0};
static const unsigned char kat344_retbits[] = {
    0x61, 0x29, 0xed, 0xa9, 0xef, 0x3b, 0x9e, 0xb9, 0xf3, 0x8f, 0xd8, 0xe9,
    0xca, 0x93, 0xe7, 0x31, 0x9b, 0x6f, 0xde, 0xd3, 0x63, 0xee, 0x15, 0x5c,
    0x97, 0x99, 0x19, 0xc3, 0x4b, 0x98, 0xce, 0x39, 0x9b, 0x0b, 0x4f, 0x99,
    0xf2, 0x83, 0x6f, 0x98, 0xea, 0x84, 0xd5, 0x26, 0x03, 0xf5, 0x79, 0x0d,
    0xa5, 0x20, 0x29, 0xa4, 0xe7, 0xcf, 0x87, 0x17, 0xdb, 0x4a, 0x62, 0x22,
    0x50, 0x0d, 0x7d, 0x2a,
};
static const struct drbg_kat_no_reseed kat344_t = {
    13, kat344_entropyin, kat344_nonce, kat344_persstr,
    kat344_addin0, kat344_addin1, kat344_retbits
};
static const struct drbg_kat kat344 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat344_t
};

static const unsigned char kat345_entropyin[] = {
    0xf8, 0x1b, 0x47, 0x52, 0xd9, 0xb3, 0x4f, 0xea, 0x62, 0x82, 0x78, 0xbe,
    0xff, 0x92, 0x9d, 0x17, 0x9d, 0x04, 0x44, 0xa8, 0x1b, 0x0e, 0x1b, 0x75,
};
static const unsigned char kat345_nonce[] = {
    0xeb, 0xe6, 0xe2, 0x74, 0x3f, 0x5f, 0xd9, 0xc5, 0xb1, 0xbc, 0xeb, 0x12,
    0xf3, 0xf7, 0x9c, 0x2b,
};
static const unsigned char kat345_persstr[] = {
    0xd8, 0x6a, 0x1d, 0x62, 0x8b, 0xb8, 0x3d, 0x73, 0x86, 0x14, 0xa9, 0x3c,
    0x44, 0xc0, 0x8d, 0x37, 0x17, 0xf8, 0x03, 0xa2, 0x7b, 0xee, 0x61, 0xbe,
    0x24, 0xb8, 0xed, 0xf5, 0x2e, 0xfc, 0xd3, 0x68,
};
static const unsigned char kat345_addin0[] = {0};
static const unsigned char kat345_addin1[] = {0};
static const unsigned char kat345_retbits[] = {
    0xac, 0xd3, 0x01, 0xc2, 0x92, 0xbb, 0x6b, 0x80, 0x14, 0x01, 0xff, 0x9e,
    0x3b, 0x65, 0x56, 0xfa, 0x1c, 0xa8, 0x06, 0x1b, 0x99, 0xb8, 0xbd, 0x54,
    0x5b, 0x16, 0x33, 0x18, 0xbb, 0x88, 0x02, 0x28, 0xee, 0xb4, 0xee, 0x3a,
    0xd5, 0x6b, 0x21, 0x34, 0x33, 0x77, 0x92, 0x86, 0x2d, 0xa2, 0x51, 0xc5,
    0xac, 0x43, 0xc2, 0xa2, 0xa5, 0xd8, 0xcb, 0x38, 0xc1, 0x84, 0x36, 0x83,
    0x9f, 0xba, 0xb4, 0xdc,
};
static const struct drbg_kat_no_reseed kat345_t = {
    14, kat345_entropyin, kat345_nonce, kat345_persstr,
    kat345_addin0, kat345_addin1, kat345_retbits
};
static const struct drbg_kat kat345 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat345_t
};

static const unsigned char kat346_entropyin[] = {
    0xf3, 0x58, 0x3f, 0x3c, 0x8b, 0x14, 0x7a, 0x7e, 0xe9, 0x92, 0xd5, 0xca,
    0x61, 0x1e, 0x47, 0x85, 0xb7, 0x72, 0xcb, 0x57, 0x5a, 0x53, 0xa5, 0x73,
};
static const unsigned char kat346_nonce[] = {
    0xeb, 0x1b, 0xce, 0xf5, 0x54, 0x1a, 0xba, 0x81, 0x29, 0xcd, 0xaa, 0x8b,
    0x23, 0x3b, 0x60, 0xf0,
};
static const unsigned char kat346_persstr[] = {
    0x08, 0xc1, 0x8b, 0x3c, 0x8f, 0x79, 0xed, 0xa3, 0xba, 0x18, 0xfc, 0xd2,
    0x71, 0x1c, 0x86, 0xfd, 0x1d, 0x6a, 0x28, 0x42, 0xea, 0x95, 0xf1, 0xd8,
    0x96, 0x17, 0x24, 0x01, 0x2c, 0x0a, 0x61, 0x21,
};
static const unsigned char kat346_addin0[] = {
    0x65, 0x7d, 0x1d, 0xf7, 0x80, 0xa9, 0xb6, 0x74, 0xb9, 0x0c, 0xce, 0x0d,
    0x49, 0xe2, 0xd5, 0xc0, 0xc0, 0x1e, 0xd2, 0xd2, 0xd7, 0x99, 0xf5, 0x04,
    0x34, 0x8d, 0x86, 0x37, 0x45, 0x76, 0xc4, 0x04,
};
static const unsigned char kat346_addin1[] = {
    0x94, 0x23, 0x4f, 0x2b, 0x6c, 0x9f, 0x32, 0x7c, 0xd8, 0x23, 0xe7, 0x53,
    0xec, 0x9c, 0x4d, 0xe4, 0x11, 0x97, 0x37, 0xaf, 0x91, 0x4d, 0xea, 0x5e,
    0x57, 0xe0, 0xa5, 0x0e, 0x33, 0x76, 0x13, 0x5f,
};
static const unsigned char kat346_retbits[] = {
    0x3c, 0x2e, 0x80, 0xe7, 0x21, 0x1e, 0x6a, 0x19, 0xa2, 0x7a, 0x53, 0xb4,
    0x95, 0x71, 0x65, 0xed, 0x87, 0xd1, 0xed, 0xb6, 0x62, 0x42, 0x6f, 0xee,
    0x92, 0x73, 0xae, 0x0d, 0x85, 0x40, 0x2c, 0xe2, 0x50, 0x7c, 0xda, 0x18,
    0x12, 0x6c, 0x07, 0xfb, 0x20, 0x11, 0x71, 0xf1, 0x76, 0xb0, 0x69, 0xd2,
    0xd2, 0xa1, 0x26, 0xaf, 0x5f, 0xb3, 0x1c, 0xe1, 0xd1, 0x99, 0x97, 0x8e,
    0x3d, 0x11, 0xd7, 0x71,
};
static const struct drbg_kat_no_reseed kat346_t = {
    0, kat346_entropyin, kat346_nonce, kat346_persstr,
    kat346_addin0, kat346_addin1, kat346_retbits
};
static const struct drbg_kat kat346 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat346_t
};

static const unsigned char kat347_entropyin[] = {
    0x4e, 0x00, 0x8d, 0xc8, 0x72, 0xa7, 0x4a, 0x6a, 0x9d, 0x4c, 0x71, 0xc1,
    0x4a, 0x2d, 0x9b, 0x1a, 0xa7, 0xc0, 0x5a, 0x03, 0x40, 0x2c, 0xa1, 0x81,
};
static const unsigned char kat347_nonce[] = {
    0x33, 0xc5, 0x0f, 0xa9, 0x5f, 0x0d, 0x9d, 0x03, 0xbc, 0xc1, 0xde, 0x53,
    0x0e, 0x04, 0x44, 0x0d,
};
static const unsigned char kat347_persstr[] = {
    0x57, 0x87, 0x57, 0x74, 0x9b, 0x8e, 0xbe, 0x44, 0x2d, 0x93, 0xb6, 0x2c,
    0xb9, 0x42, 0x27, 0x8d, 0xe5, 0xaa, 0x89, 0x09, 0xf9, 0x14, 0x12, 0x0a,
    0x90, 0x58, 0xb2, 0xc3, 0x41, 0xe8, 0x86, 0xc8,
};
static const unsigned char kat347_addin0[] = {
    0xea, 0x62, 0x7c, 0xb9, 0xfe, 0x1a, 0x49, 0x9e, 0x3b, 0xc1, 0xae, 0x73,
    0x1a, 0xa3, 0x73, 0x2b, 0x53, 0x93, 0x97, 0x17, 0x88, 0x98, 0xba, 0x2d,
    0xc4, 0x0c, 0x04, 0xae, 0x68, 0x50, 0x48, 0x86,
};
static const unsigned char kat347_addin1[] = {
    0x77, 0x0a, 0xcb, 0x69, 0x0f, 0xd0, 0xba, 0x83, 0x09, 0x71, 0x93, 0x30,
    0x3d, 0x68, 0x44, 0x05, 0xdd, 0xb3, 0xdd, 0x4f, 0x67, 0x70, 0xf0, 0x14,
    0x1d, 0x58, 0x04, 0x62, 0x60, 0xe3, 0xf2, 0x73,
};
static const unsigned char kat347_retbits[] = {
    0xb0, 0xe1, 0x3a, 0xf5, 0x42, 0xc0, 0x77, 0x79, 0x61, 0xcb, 0xec, 0x4c,
    0x61, 0xc9, 0xb2, 0x25, 0xa1, 0x21, 0x01, 0x16, 0xcb, 0xc7, 0x4e, 0xed,
    0x6e, 0xa8, 0xb7, 0x3e, 0x96, 0x8c, 0x11, 0x8f, 0xe6, 0x0e, 0x36, 0x07,
    0x98, 0xa1, 0x77, 0x9f, 0x07, 0xe0, 0x0b, 0x3f, 0xd1, 0x24, 0xf2, 0x78,
    0xb2, 0x7f, 0x70, 0x04, 0x66, 0x12, 0x02, 0x1d, 0xb8, 0xb8, 0xad, 0x32,
    0x92, 0x89, 0x40, 0x64,
};
static const struct drbg_kat_no_reseed kat347_t = {
    1, kat347_entropyin, kat347_nonce, kat347_persstr,
    kat347_addin0, kat347_addin1, kat347_retbits
};
static const struct drbg_kat kat347 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat347_t
};

static const unsigned char kat348_entropyin[] = {
    0x09, 0x65, 0xa0, 0xca, 0x7c, 0x1e, 0x68, 0x46, 0x26, 0xa4, 0x09, 0xd8,
    0xcb, 0xed, 0x26, 0x73, 0x0e, 0x86, 0x18, 0xbf, 0x7c, 0x2c, 0x70, 0x31,
};
static const unsigned char kat348_nonce[] = {
    0x19, 0xc4, 0x20, 0x55, 0xf5, 0xcd, 0x94, 0xdf, 0x0f, 0xe3, 0x03, 0x7a,
    0xf9, 0x81, 0x55, 0x0b,
};
static const unsigned char kat348_persstr[] = {
    0xc9, 0x08, 0x8b, 0x4b, 0x23, 0xa1, 0xbd, 0x96, 0x96, 0x3e, 0x43, 0xe1,
    0xe7, 0x74, 0x28, 0x70, 0xb6, 0xb0, 0xad, 0x58, 0xac, 0xe3, 0x32, 0x6f,
    0x1c, 0x7e, 0x67, 0x5a, 0xff, 0x6e, 0x53, 0x97,
};
static const unsigned char kat348_addin0[] = {
    0xcf, 0x58, 0x27, 0x39, 0x45, 0xa0, 0xc5, 0x26, 0x2b, 0xb1, 0x7e, 0x32,
    0xd2, 0xdc, 0x7a, 0xfa, 0x5f, 0xf9, 0x19, 0xee, 0xaf, 0xee, 0xf7, 0x32,
    0x2c, 0xa0, 0x2c, 0xa5, 0x05, 0x53, 0xf4, 0x91,
};
static const unsigned char kat348_addin1[] = {
    0x9d, 0x64, 0xc2, 0xef, 0x27, 0xde, 0xda, 0xe0, 0x99, 0xa1, 0x01, 0x6c,
    0x10, 0x93, 0x71, 0x5b, 0xd1, 0x36, 0x87, 0xfa, 0xa1, 0x72, 0x42, 0xfe,
    0xa8, 0xbf, 0x25, 0x48, 0xa6, 0xdd, 0x7d, 0xfd,
};
static const unsigned char kat348_retbits[] = {
    0x03, 0xa7, 0x58, 0x95, 0x2a, 0xc1, 0x7a, 0x40, 0x0a, 0xe9, 0x74, 0x28,
    0xba, 0xa6, 0x41, 0xd7, 0xb9, 0xe3, 0xcd, 0xdb, 0x16, 0x3c, 0x1f, 0x38,
    0x19, 0x69, 0xb2, 0x09, 0x5b, 0x5e, 0x00, 0x76, 0xc3, 0x53, 0xc8, 0xf5,
    0x54, 0x60, 0xf1, 0x62, 0xae, 0x85, 0x23, 0xa5, 0x2b, 0x46, 0xa0, 0x1d,
    0x5d, 0x26, 0x43, 0xac, 0x81, 0x52, 0x9e, 0x7f, 0xc7, 0x73, 0xac, 0xda,
    0xf6, 0x74, 0xed, 0x37,
};
static const struct drbg_kat_no_reseed kat348_t = {
    2, kat348_entropyin, kat348_nonce, kat348_persstr,
    kat348_addin0, kat348_addin1, kat348_retbits
};
static const struct drbg_kat kat348 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat348_t
};

static const unsigned char kat349_entropyin[] = {
    0x0b, 0x74, 0xa0, 0xd1, 0x4e, 0x27, 0x94, 0xd8, 0x8b, 0x66, 0x17, 0x0c,
    0x90, 0xc4, 0xf9, 0x21, 0xaf, 0xa6, 0xe6, 0x78, 0xd4, 0x8a, 0x38, 0x6a,
};
static const unsigned char kat349_nonce[] = {
    0x2c, 0x38, 0x0c, 0xba, 0xbd, 0x98, 0x44, 0x3d, 0x64, 0x56, 0xea, 0xcf,
    0x25, 0xff, 0xdc, 0xfc,
};
static const unsigned char kat349_persstr[] = {
    0x11, 0x0b, 0xf5, 0xb1, 0x76, 0xd8, 0x7a, 0x0a, 0x4e, 0x6c, 0x48, 0x28,
    0x29, 0x72, 0x85, 0x4c, 0x3e, 0xde, 0xbd, 0xd2, 0x08, 0xc9, 0xa6, 0xa3,
    0xef, 0x52, 0x0f, 0x93, 0xa2, 0x1e, 0x1b, 0xbf,
};
static const unsigned char kat349_addin0[] = {
    0x0e, 0xee, 0xec, 0x35, 0x9c, 0x4c, 0x7b, 0x15, 0x5c, 0xd4, 0x91, 0x71,
    0x13, 0x65, 0x14, 0xde, 0xa7, 0x1a, 0xb8, 0xe4, 0x35, 0xa1, 0x22, 0x60,
    0x6c, 0xcc, 0x50, 0x40, 0xd4, 0xce, 0x82, 0x5b,
};
static const unsigned char kat349_addin1[] = {
    0x9d, 0xe0, 0xef, 0x32, 0x02, 0x88, 0x82, 0x3b, 0x06, 0xce, 0x81, 0x89,
    0x5e, 0x57, 0x23, 0xf1, 0x32, 0x39, 0xd5, 0x26, 0x9d, 0x50, 0x7c, 0xca,
    0xb6, 0xbc, 0xd9, 0x6b, 0x3b, 0x9b, 0x3f, 0xbf,
};
static const unsigned char kat349_retbits[] = {
    0x3f, 0xe1, 0x12, 0xcc, 0xd4, 0x4d, 0x50, 0xd5, 0xd5, 0x4e, 0x33, 0xbd,
    0xd6, 0xf1, 0x12, 0xf1, 0xef, 0x4c, 0xaa, 0xc0, 0x69, 0x58, 0x6d, 0xd4,
    0xd8, 0x9c, 0x64, 0xe6, 0x93, 0x14, 0xe2, 0xef, 0xc3, 0xcb, 0x77, 0xaa,
    0x5f, 0xe2, 0x08, 0x61, 0x59, 0xab, 0xfe, 0x21, 0x9e, 0xfd, 0xf5, 0x2b,
    0x86, 0x70, 0x41, 0x14, 0x75, 0x58, 0x57, 0xd7, 0x4c, 0x45, 0x9a, 0xf5,
    0x22, 0x85, 0xa4, 0x4a,
};
static const struct drbg_kat_no_reseed kat349_t = {
    3, kat349_entropyin, kat349_nonce, kat349_persstr,
    kat349_addin0, kat349_addin1, kat349_retbits
};
static const struct drbg_kat kat349 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat349_t
};

static const unsigned char kat350_entropyin[] = {
    0xfe, 0x01, 0xd0, 0x64, 0xd2, 0x98, 0xa1, 0x6a, 0xa1, 0x38, 0xbf, 0x10,
    0xfe, 0x69, 0x2a, 0xd6, 0xaa, 0xae, 0x53, 0xf5, 0x19, 0xe6, 0xad, 0x13,
};
static const unsigned char kat350_nonce[] = {
    0x14, 0x32, 0xc2, 0x4e, 0x78, 0xa2, 0x45, 0x40, 0xdc, 0xb5, 0x91, 0x4e,
    0xe8, 0x51, 0x0c, 0xfa,
};
static const unsigned char kat350_persstr[] = {
    0x76, 0x99, 0x12, 0xbd, 0xd5, 0x9b, 0xcd, 0x14, 0x5c, 0x52, 0x7e, 0x9c,
    0x13, 0x2c, 0x5e, 0xca, 0x56, 0xb2, 0x87, 0x8f, 0x12, 0x2c, 0x0e, 0x53,
    0xfa, 0x34, 0x53, 0x14, 0x0a, 0x2b, 0x4b, 0x01,
};
static const unsigned char kat350_addin0[] = {
    0x30, 0x0b, 0x91, 0x87, 0x40, 0x26, 0x19, 0xf7, 0x75, 0xca, 0x2c, 0x53,
    0x52, 0xa3, 0x3a, 0x03, 0x4a, 0xd8, 0x00, 0x68, 0xcf, 0xef, 0x0f, 0xb1,
    0xea, 0xa7, 0xb5, 0x76, 0xa0, 0xc9, 0x10, 0xbd,
};
static const unsigned char kat350_addin1[] = {
    0xb6, 0xb1, 0x76, 0xd0, 0x6c, 0xe4, 0xfe, 0x18, 0x9d, 0x87, 0x73, 0xf4,
    0x13, 0xf2, 0x59, 0x26, 0x10, 0x63, 0x79, 0x69, 0xab, 0x9a, 0x1d, 0xde,
    0x51, 0x96, 0x0f, 0xcd, 0x42, 0x75, 0xfc, 0x80,
};
static const unsigned char kat350_retbits[] = {
    0x98, 0x90, 0x5c, 0x12, 0xc0, 0xe5, 0xc2, 0xe0, 0x7b, 0x32, 0x61, 0x7f,
    0xac, 0x64, 0x34, 0x34, 0x21, 0xfe, 0x69, 0x7b, 0x18, 0x88, 0x6a, 0x33,
    0x44, 0xa4, 0xe6, 0x28, 0x67, 0x73, 0x2e, 0x90, 0xe2, 0x85, 0x75, 0x04,
    0xf7, 0x53, 0xc1, 0x11, 0x83, 0xfd, 0xb4, 0xdb, 0xe4, 0x6e, 0x78, 0xfb,
    0x74, 0xe5, 0xbe, 0x32, 0xa6, 0x72, 0xb8, 0xe5, 0xc6, 0x99, 0xc3, 0x45,
    0xa8, 0x4a, 0x6d, 0xcc,
};
static const struct drbg_kat_no_reseed kat350_t = {
    4, kat350_entropyin, kat350_nonce, kat350_persstr,
    kat350_addin0, kat350_addin1, kat350_retbits
};
static const struct drbg_kat kat350 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat350_t
};

static const unsigned char kat351_entropyin[] = {
    0x25, 0x38, 0x36, 0xb5, 0xb8, 0xe1, 0xda, 0x4a, 0x96, 0xf2, 0x01, 0xec,
    0xdc, 0xce, 0xbc, 0x2f, 0x9a, 0xed, 0xb9, 0xf2, 0x20, 0xed, 0x26, 0x61,
};
static const unsigned char kat351_nonce[] = {
    0x1e, 0x91, 0xb8, 0x74, 0x68, 0xf3, 0x28, 0x4a, 0x4e, 0x55, 0x08, 0x40,
    0x2a, 0x32, 0x74, 0x25,
};
static const unsigned char kat351_persstr[] = {
    0x25, 0xe8, 0x4e, 0x59, 0x21, 0xcd, 0x54, 0x8c, 0x38, 0x95, 0xac, 0xb0,
    0x1d, 0xcd, 0xff, 0xc9, 0xa4, 0x79, 0x22, 0x56, 0xf3, 0x1d, 0x1a, 0xdf,
    0x30, 0xe0, 0x89, 0xb6, 0xa4, 0x0c, 0xd0, 0x29,
};
static const unsigned char kat351_addin0[] = {
    0x0b, 0x8b, 0x88, 0xd3, 0xc3, 0x78, 0x73, 0x6b, 0x60, 0x69, 0x49, 0x0c,
    0xc8, 0xf8, 0x1b, 0x70, 0xb1, 0xc4, 0x2d, 0xc7, 0xb6, 0x30, 0xb5, 0xe8,
    0xb5, 0xd2, 0x57, 0xfa, 0x73, 0xda, 0x77, 0xcf,
};
static const unsigned char kat351_addin1[] = {
    0xe7, 0x92, 0x5d, 0xbd, 0x1b, 0x6e, 0xc3, 0x62, 0xf2, 0xb3, 0x94, 0x1c,
    0x34, 0xad, 0x6c, 0xc9, 0xf0, 0x9b, 0x7e, 0xe1, 0xe6, 0xc9, 0xff, 0xd5,
    0x12, 0x05, 0x4b, 0xdd, 0x40, 0x3d, 0xe0, 0xe1,
};
static const unsigned char kat351_retbits[] = {
    0xa4, 0x33, 0x61, 0xdf, 0xe3, 0x61, 0x97, 0x9a, 0xbf, 0xf5, 0x57, 0xf6,
    0x50, 0x06, 0xc5, 0x2e, 0x77, 0x64, 0xa2, 0x8f, 0x7d, 0x13, 0x69, 0x7c,
    0x94, 0x09, 0x85, 0x52, 0xaf, 0x40, 0x05, 0x97, 0x89, 0x3b, 0x9f, 0x02,
    0x71, 0x3f, 0x96, 0xd3, 0x94, 0xbb, 0x17, 0x83, 0x8f, 0x3c, 0x7b, 0xc8,
    0xdc, 0xb7, 0xaa, 0x03, 0x32, 0xb7, 0xef, 0x6b, 0x51, 0xde, 0xec, 0x7a,
    0x2f, 0x38, 0x16, 0x07,
};
static const struct drbg_kat_no_reseed kat351_t = {
    5, kat351_entropyin, kat351_nonce, kat351_persstr,
    kat351_addin0, kat351_addin1, kat351_retbits
};
static const struct drbg_kat kat351 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat351_t
};

static const unsigned char kat352_entropyin[] = {
    0x0b, 0xa8, 0xc3, 0x0d, 0x69, 0x75, 0x23, 0x95, 0xd3, 0xa6, 0xdd, 0x5c,
    0x3d, 0x2e, 0xb1, 0x7c, 0x98, 0x4b, 0x40, 0xb4, 0x03, 0x82, 0xac, 0x84,
};
static const unsigned char kat352_nonce[] = {
    0x60, 0x79, 0x40, 0xf2, 0x5e, 0x5d, 0xb8, 0x9a, 0x7b, 0x2d, 0xa5, 0xd4,
    0xee, 0x79, 0xb8, 0x46,
};
static const unsigned char kat352_persstr[] = {
    0x87, 0x74, 0xab, 0x97, 0xc6, 0xfa, 0x93, 0x8e, 0x49, 0x37, 0x3e, 0x94,
    0x83, 0xc1, 0xa1, 0x8c, 0x8b, 0xf4, 0x3d, 0x3e, 0x38, 0xf9, 0xb9, 0x5e,
    0x23, 0xab, 0x04, 0x12, 0x24, 0x5b, 0x8f, 0xb5,
};
static const unsigned char kat352_addin0[] = {
    0x94, 0x6c, 0xe7, 0xd8, 0xda, 0xf6, 0x45, 0xbb, 0x0a, 0x21, 0x87, 0x85,
    0xc0, 0x24, 0x61, 0xdb, 0xf4, 0x97, 0x3a, 0xf9, 0xf9, 0xb1, 0xb7, 0x70,
    0x94, 0x89, 0xe1, 0xb4, 0xe2, 0xf9, 0x1b, 0xad,
};
static const unsigned char kat352_addin1[] = {
    0x73, 0xde, 0xf1, 0x10, 0x43, 0x86, 0xd8, 0x3e, 0xf8, 0xbc, 0x71, 0x7e,
    0x52, 0x2b, 0x14, 0x8c, 0xca, 0xac, 0xd4, 0x3a, 0x27, 0x9b, 0x73, 0xc3,
    0x73, 0xa6, 0xd5, 0xa6, 0x62, 0xf0, 0x26, 0xa3,
};
static const unsigned char kat352_retbits[] = {
    0x54, 0xbe, 0x8d, 0xaf, 0xf4, 0xb6, 0x5b, 0x03, 0x2b, 0xfa, 0x52, 0x34,
    0x92, 0x06, 0x1c, 0x46, 0xb5, 0xce, 0x36, 0xb2, 0x64, 0x60, 0x84, 0x47,
    0x43, 0x03, 0xd2, 0x62, 0x2b, 0x00, 0x61, 0x12, 0xdd, 0x18, 0x6d, 0xce,
    0x10, 0xc3, 0x06, 0x85, 0x6c, 0x2e, 0x88, 0xc1, 0xce, 0xdc, 0x40, 0x42,
    0x69, 0xb8, 0x92, 0x4f, 0x92, 0xd8, 0x43, 0x97, 0x30, 0xd6, 0x47, 0xda,
    0x55, 0xd6, 0x0e, 0x9b,
};
static const struct drbg_kat_no_reseed kat352_t = {
    6, kat352_entropyin, kat352_nonce, kat352_persstr,
    kat352_addin0, kat352_addin1, kat352_retbits
};
static const struct drbg_kat kat352 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat352_t
};

static const unsigned char kat353_entropyin[] = {
    0x61, 0x91, 0x07, 0x37, 0x79, 0x9f, 0xa4, 0x68, 0xa3, 0xf6, 0x6d, 0xad,
    0xaf, 0x56, 0xf4, 0xe3, 0xb8, 0xf7, 0x98, 0xca, 0x65, 0xce, 0x03, 0x91,
};
static const unsigned char kat353_nonce[] = {
    0x72, 0x33, 0x99, 0xac, 0x93, 0x58, 0x69, 0xce, 0x20, 0x5d, 0x85, 0x50,
    0x9c, 0xc2, 0x78, 0x42,
};
static const unsigned char kat353_persstr[] = {
    0x29, 0xe3, 0xea, 0x51, 0x80, 0x5c, 0x78, 0xfe, 0x84, 0x91, 0x15, 0xc4,
    0xb7, 0x73, 0x61, 0xd7, 0x5b, 0x1b, 0x92, 0x46, 0xe5, 0x6c, 0x46, 0x41,
    0x66, 0xac, 0x7e, 0x8d, 0x1c, 0xa8, 0xe3, 0xab,
};
static const unsigned char kat353_addin0[] = {
    0xab, 0x8a, 0xa0, 0x35, 0xc7, 0x15, 0x01, 0xc1, 0xf4, 0x9d, 0xea, 0x7c,
    0x67, 0x8e, 0x02, 0xd6, 0x9e, 0x2f, 0x84, 0x3a, 0xf7, 0x44, 0x79, 0x36,
    0x7a, 0x38, 0x6f, 0x68, 0x4c, 0xc3, 0x60, 0xdb,
};
static const unsigned char kat353_addin1[] = {
    0x22, 0xdc, 0xd9, 0xf0, 0x31, 0x38, 0xf3, 0x72, 0xb1, 0x5f, 0x15, 0xac,
    0x6d, 0xc3, 0x1d, 0x13, 0x57, 0x10, 0x2e, 0xd4, 0xe8, 0x03, 0xb6, 0xd3,
    0x84, 0x29, 0x8e, 0x4c, 0xde, 0xb2, 0xe9, 0x01,
};
static const unsigned char kat353_retbits[] = {
    0x01, 0x86, 0x2e, 0x1d, 0x6f, 0xa1, 0x11, 0x73, 0x60, 0x9a, 0x18, 0x2d,
    0x6a, 0x5a, 0x73, 0x31, 0x0c, 0x76, 0xdf, 0xa8, 0x8d, 0x1e, 0xbf, 0xf2,
    0x5e, 0xf4, 0x5d, 0x57, 0x96, 0xbd, 0xf4, 0x7a, 0x2b, 0xb6, 0xb9, 0xcf,
    0x88, 0x17, 0xe6, 0xd7, 0x6b, 0x70, 0xbf, 0x0b, 0x9e, 0x83, 0xd8, 0xb4,
    0xa1, 0x28, 0xbe, 0xe0, 0x6b, 0x6b, 0x0e, 0xe3, 0x7b, 0x60, 0x21, 0xad,
    0x47, 0x22, 0xc4, 0xe7,
};
static const struct drbg_kat_no_reseed kat353_t = {
    7, kat353_entropyin, kat353_nonce, kat353_persstr,
    kat353_addin0, kat353_addin1, kat353_retbits
};
static const struct drbg_kat kat353 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat353_t
};

static const unsigned char kat354_entropyin[] = {
    0x6d, 0xf2, 0x09, 0x57, 0x0e, 0xd2, 0x0e, 0x3d, 0x9f, 0x13, 0xe9, 0x6c,
    0x29, 0xd2, 0xb0, 0x14, 0xbc, 0xa9, 0xb9, 0xfb, 0xeb, 0x5a, 0x32, 0xcf,
};
static const unsigned char kat354_nonce[] = {
    0x53, 0x1a, 0x31, 0xc8, 0x7e, 0xc9, 0x2b, 0xa0, 0x6e, 0xf8, 0xa6, 0xdb,
    0x62, 0x3c, 0xf3, 0xc0,
};
static const unsigned char kat354_persstr[] = {
    0xf2, 0x55, 0x70, 0x3d, 0xc8, 0xcd, 0x5c, 0xeb, 0xba, 0x02, 0x29, 0xe4,
    0x31, 0x5a, 0xee, 0xe8, 0x6b, 0xee, 0x45, 0xae, 0xb7, 0x52, 0x79, 0xc0,
    0xe1, 0x20, 0xe4, 0x4e, 0x46, 0x93, 0x5b, 0x15,
};
static const unsigned char kat354_addin0[] = {
    0x7e, 0x16, 0x03, 0x2a, 0x61, 0x55, 0xc3, 0x7c, 0x24, 0x69, 0xb7, 0x3a,
    0x2a, 0x50, 0xba, 0xa8, 0x96, 0x53, 0xe9, 0xc1, 0xe7, 0xb2, 0x5b, 0xdb,
    0xf0, 0x67, 0x5c, 0xa6, 0x84, 0xec, 0x8a, 0x05,
};
static const unsigned char kat354_addin1[] = {
    0x95, 0x60, 0x81, 0xc8, 0x08, 0xea, 0x4b, 0x9f, 0xd3, 0xc9, 0x88, 0x01,
    0x40, 0xcd, 0xbe, 0x41, 0xcb, 0x0a, 0xa0, 0x39, 0xf0, 0xff, 0xad, 0x1e,
    0xfc, 0xb9, 0x4b, 0x41, 0x31, 0x7a, 0xb3, 0x21,
};
static const unsigned char kat354_retbits[] = {
    0x5e, 0x8d, 0x8b, 0x4b, 0x79, 0xeb, 0x53, 0x42, 0x2f, 0x92, 0x68, 0x36,
    0xef, 0x78, 0x05, 0x1f, 0x53, 0x3b, 0x93, 0x18, 0xf6, 0x1a, 0x62, 0x55,
    0x29, 0xb0, 0xad, 0xcc, 0x07, 0x0d, 0x71, 0xd2, 0xef, 0xd3, 0x3f, 0x61,
    0x14, 0xdb, 0x13, 0xcf, 0x26, 0xb6, 0x62, 0x1f, 0xe8, 0x56, 0xb7, 0x08,
    0xaf, 0x48, 0xa6, 0x7f, 0x3c, 0xb2, 0xed, 0x8b, 0x9a, 0x92, 0xe0, 0xb7,
    0x3b, 0xfb, 0xce, 0x57,
};
static const struct drbg_kat_no_reseed kat354_t = {
    8, kat354_entropyin, kat354_nonce, kat354_persstr,
    kat354_addin0, kat354_addin1, kat354_retbits
};
static const struct drbg_kat kat354 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat354_t
};

static const unsigned char kat355_entropyin[] = {
    0x73, 0x2a, 0xda, 0x29, 0x7f, 0x8e, 0x2f, 0x25, 0x4e, 0x5c, 0x3c, 0xd3,
    0x27, 0x58, 0xa7, 0x00, 0xa4, 0xaa, 0x63, 0xea, 0x35, 0x31, 0x78, 0xf1,
};
static const unsigned char kat355_nonce[] = {
    0x19, 0x1f, 0x07, 0xc4, 0x99, 0x93, 0xf6, 0xb4, 0xb2, 0x49, 0xf6, 0xad,
    0x0d, 0x70, 0x2d, 0x7f,
};
static const unsigned char kat355_persstr[] = {
    0x9a, 0x31, 0x0d, 0x97, 0x0d, 0xaa, 0xff, 0x8a, 0xb0, 0xb2, 0x6a, 0x0e,
    0xdd, 0x2a, 0xa2, 0xe5, 0xc4, 0x0f, 0x83, 0x3c, 0xd1, 0x54, 0x21, 0x8d,
    0xd7, 0x12, 0xb1, 0x1c, 0x52, 0x93, 0x4a, 0xb2,
};
static const unsigned char kat355_addin0[] = {
    0xa3, 0x8c, 0x3e, 0x35, 0xc7, 0x3f, 0x78, 0xb3, 0x81, 0x6e, 0xe9, 0x0b,
    0x7d, 0x13, 0x16, 0xc4, 0x24, 0x71, 0x90, 0x47, 0xe1, 0x04, 0xe3, 0x09,
    0x73, 0xf6, 0x96, 0x10, 0xcb, 0x41, 0x40, 0x8a,
};
static const unsigned char kat355_addin1[] = {
    0x14, 0xae, 0xf4, 0x4a, 0x84, 0xb7, 0xc0, 0x21, 0xd4, 0xfc, 0xf3, 0x82,
    0x58, 0xcb, 0xfb, 0x95, 0xb1, 0x07, 0xbf, 0xf7, 0x51, 0xb5, 0x40, 0x5b,
    0xf2, 0x1e, 0x9d, 0x08, 0x8f, 0xd7, 0x2d, 0xab,
};
static const unsigned char kat355_retbits[] = {
    0x8a, 0x84, 0x0c, 0x68, 0x78, 0x67, 0x1d, 0x31, 0xac, 0x43, 0x84, 0xaa,
    0x7e, 0x11, 0x3b, 0x09, 0x74, 0x4f, 0x6d, 0x89, 0xa0, 0x9b, 0x0e, 0x5c,
    0xb7, 0xcd, 0x14, 0x80, 0xfd, 0x89, 0x4e, 0xa6, 0x9e, 0x56, 0x8d, 0x6e,
    0xe8, 0x31, 0xa9, 0xef, 0x4b, 0x90, 0xdb, 0x52, 0x77, 0xf6, 0xee, 0x48,
    0xac, 0x66, 0xbb, 0xc3, 0x78, 0x65, 0x68, 0x7c, 0xed, 0x77, 0xf9, 0x56,
    0xa2, 0xca, 0xcf, 0x90,
};
static const struct drbg_kat_no_reseed kat355_t = {
    9, kat355_entropyin, kat355_nonce, kat355_persstr,
    kat355_addin0, kat355_addin1, kat355_retbits
};
static const struct drbg_kat kat355 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat355_t
};

static const unsigned char kat356_entropyin[] = {
    0x61, 0xd3, 0x83, 0x95, 0x16, 0x61, 0x74, 0x27, 0x61, 0x38, 0xd6, 0x43,
    0x68, 0x3b, 0x3e, 0x63, 0xd5, 0x90, 0x9e, 0x5d, 0x15, 0xd7, 0x31, 0x4a,
};
static const unsigned char kat356_nonce[] = {
    0x21, 0x14, 0x1c, 0xe4, 0x19, 0xc9, 0x54, 0x1d, 0x49, 0xbc, 0xff, 0xcc,
    0xf0, 0x16, 0xd5, 0x93,
};
static const unsigned char kat356_persstr[] = {
    0x63, 0xb3, 0xac, 0x79, 0x89, 0x91, 0x37, 0x69, 0x23, 0xc1, 0x11, 0x71,
    0x08, 0xdd, 0x09, 0x62, 0x74, 0x57, 0xb0, 0x4f, 0x20, 0xeb, 0x06, 0x3c,
    0xac, 0xb4, 0xe0, 0xc5, 0x6a, 0xee, 0xde, 0x43,
};
static const unsigned char kat356_addin0[] = {
    0x15, 0x07, 0xf2, 0x57, 0x0a, 0xc3, 0x47, 0xe2, 0x59, 0x3a, 0x77, 0x87,
    0x3e, 0xc2, 0xab, 0x50, 0x86, 0xd9, 0xb3, 0x65, 0x9f, 0xba, 0x3c, 0xa2,
    0xba, 0xe6, 0x04, 0x14, 0x21, 0x9b, 0xd7, 0x34,
};
static const unsigned char kat356_addin1[] = {
    0xc6, 0x2c, 0x67, 0x91, 0xec, 0x08, 0xe6, 0x4b, 0xe2, 0xfa, 0x59, 0xd2,
    0xfa, 0x9e, 0x17, 0x49, 0xc1, 0x86, 0x40, 0x85, 0x54, 0xa0, 0x55, 0x14,
    0x4e, 0x3e, 0xe9, 0x2d, 0x7b, 0xf1, 0x30, 0xbf,
};
static const unsigned char kat356_retbits[] = {
    0xff, 0x3d, 0xf5, 0x33, 0xee, 0xab, 0x76, 0x97, 0xef, 0x66, 0x8d, 0x30,
    0x1e, 0x9d, 0x69, 0x20, 0xb1, 0x55, 0xd6, 0x78, 0x1e, 0x4a, 0xbd, 0xad,
    0xb0, 0xef, 0xce, 0x45, 0x75, 0xa6, 0x49, 0xd1, 0x32, 0x97, 0xe0, 0xb2,
    0x98, 0x69, 0x40, 0x60, 0x8c, 0x65, 0x57, 0xfe, 0x19, 0x6d, 0x65, 0xb9,
    0xaa, 0x4f, 0x64, 0x21, 0x96, 0xca, 0x83, 0x6f, 0xd7, 0x57, 0xe0, 0xc3,
    0xc3, 0xcc, 0x5d, 0x1b,
};
static const struct drbg_kat_no_reseed kat356_t = {
    10, kat356_entropyin, kat356_nonce, kat356_persstr,
    kat356_addin0, kat356_addin1, kat356_retbits
};
static const struct drbg_kat kat356 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat356_t
};

static const unsigned char kat357_entropyin[] = {
    0xa1, 0x32, 0xa5, 0x32, 0x44, 0xa7, 0x84, 0x07, 0x88, 0x00, 0x47, 0x4c,
    0xfe, 0xe3, 0xd7, 0x72, 0xcc, 0x00, 0xda, 0x9e, 0xfb, 0x5a, 0xab, 0xce,
};
static const unsigned char kat357_nonce[] = {
    0x6a, 0x29, 0x9b, 0x43, 0xe7, 0xd6, 0x71, 0xdf, 0xb2, 0xdb, 0x27, 0xb7,
    0x92, 0x02, 0xf3, 0x36,
};
static const unsigned char kat357_persstr[] = {
    0xf3, 0xf6, 0xe2, 0xff, 0xc0, 0x24, 0x06, 0x88, 0xfa, 0xa1, 0x2b, 0xfc,
    0x60, 0xcb, 0x28, 0x71, 0x9b, 0x8c, 0x4f, 0xfc, 0x42, 0x0f, 0xe9, 0x13,
    0x29, 0xcf, 0x0f, 0xa7, 0x92, 0xe3, 0x2e, 0xda,
};
static const unsigned char kat357_addin0[] = {
    0x13, 0x7e, 0x39, 0xe2, 0xd8, 0xb6, 0x5c, 0xac, 0x5a, 0x65, 0x4b, 0x00,
    0xa9, 0xfc, 0xe3, 0x82, 0x52, 0x16, 0x2e, 0xdb, 0x4e, 0xec, 0xc7, 0x92,
    0x07, 0x55, 0xbf, 0xfc, 0x96, 0xa5, 0xd6, 0x41,
};
static const unsigned char kat357_addin1[] = {
    0x98, 0xb6, 0xe4, 0x17, 0x61, 0x6f, 0x3c, 0xfe, 0x40, 0x90, 0x00, 0x87,
    0x28, 0x65, 0xf4, 0xd0, 0x8f, 0xb0, 0x32, 0x2b, 0xdd, 0xa4, 0x5e, 0xf1,
    0xcb, 0x70, 0x00, 0xfa, 0x45, 0x26, 0x6b, 0x20,
};
static const unsigned char kat357_retbits[] = {
    0x9a, 0x1b, 0xad, 0x8d, 0x87, 0x77, 0x7f, 0x45, 0xf9, 0xcb, 0x8b, 0xde,
    0x14, 0xc7, 0xea, 0xaa, 0xcd, 0x93, 0x74, 0xb9, 0x74, 0x26, 0x4e, 0x14,
    0xd1, 0xba, 0x31, 0xec, 0x59, 0x98, 0x3a, 0x35, 0x90, 0x74, 0x1d, 0xdf,
    0x6c, 0x85, 0xe9, 0x93, 0xff, 0xb7, 0xb8, 0x50, 0x35, 0x20, 0x2d, 0x24,
    0xf1, 0x06, 0x83, 0xa5, 0x3e, 0x87, 0xac, 0x92, 0x02, 0xf5, 0x5d, 0x8b,
    0x99, 0x18, 0x91, 0xc9,
};
static const struct drbg_kat_no_reseed kat357_t = {
    11, kat357_entropyin, kat357_nonce, kat357_persstr,
    kat357_addin0, kat357_addin1, kat357_retbits
};
static const struct drbg_kat kat357 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat357_t
};

static const unsigned char kat358_entropyin[] = {
    0x44, 0x40, 0x2c, 0xfd, 0xba, 0xb2, 0x13, 0x4d, 0xfe, 0x82, 0x56, 0x06,
    0xd2, 0xa8, 0xe1, 0xa4, 0x92, 0xc7, 0x60, 0x84, 0xf1, 0x24, 0x54, 0x31,
};
static const unsigned char kat358_nonce[] = {
    0x31, 0x29, 0x85, 0x42, 0x7c, 0xf3, 0xc3, 0xe4, 0xe1, 0x49, 0x8f, 0x59,
    0x66, 0x30, 0xd0, 0xe6,
};
static const unsigned char kat358_persstr[] = {
    0xf3, 0x33, 0xe2, 0x9a, 0x3a, 0x03, 0xe7, 0x01, 0x1c, 0x05, 0x85, 0xbd,
    0x95, 0xbf, 0xd8, 0x19, 0xb8, 0x40, 0x32, 0xa8, 0xe0, 0x54, 0x9c, 0xdb,
    0xd0, 0xe1, 0x1a, 0xf1, 0x6b, 0xcb, 0x69, 0x3e,
};
static const unsigned char kat358_addin0[] = {
    0x9b, 0x3c, 0x03, 0x3e, 0x37, 0x43, 0x73, 0x2c, 0x9e, 0xce, 0x7d, 0xf9,
    0x41, 0xd1, 0xae, 0x4b, 0xce, 0x7e, 0x66, 0x31, 0x0c, 0x31, 0x75, 0x82,
    0x4e, 0x6e, 0xc6, 0xe8, 0x2c, 0x02, 0xc6, 0xa1,
};
static const unsigned char kat358_addin1[] = {
    0x9c, 0xd3, 0x9c, 0xdd, 0x2d, 0x43, 0x74, 0x0f, 0x30, 0x34, 0x73, 0x63,
    0x59, 0xe8, 0x9b, 0x7a, 0x78, 0xd2, 0x5b, 0xd1, 0x8a, 0xce, 0x9a, 0x14,
    0xdf, 0x1c, 0x36, 0xce, 0x7d, 0xde, 0x50, 0x5c,
};
static const unsigned char kat358_retbits[] = {
    0xe5, 0x84, 0xa0, 0x46, 0x49, 0x4b, 0x50, 0x76, 0x75, 0x76, 0x43, 0xba,
    0x84, 0xc7, 0xa7, 0xe0, 0x8c, 0x00, 0x45, 0xfc, 0xa4, 0x32, 0x5c, 0x3f,
    0x9c, 0x58, 0xf6, 0x66, 0x24, 0x52, 0x6c, 0xd4, 0x3a, 0x12, 0x8d, 0x10,
    0x54, 0xf8, 0x1b, 0x3c, 0x4b, 0x6f, 0x57, 0x35, 0xf3, 0xfb, 0x6f, 0x33,
    0xd2, 0x82, 0x50, 0x62, 0xec, 0x27, 0x05, 0x3f, 0x4d, 0x0d, 0xe2, 0x41,
    0xcc, 0xf7, 0x4a, 0xc8,
};
static const struct drbg_kat_no_reseed kat358_t = {
    12, kat358_entropyin, kat358_nonce, kat358_persstr,
    kat358_addin0, kat358_addin1, kat358_retbits
};
static const struct drbg_kat kat358 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat358_t
};

static const unsigned char kat359_entropyin[] = {
    0x33, 0xba, 0xb5, 0xbb, 0x7c, 0x01, 0x37, 0xed, 0x68, 0xbb, 0xb9, 0x8b,
    0x2e, 0x16, 0x5d, 0xc9, 0x24, 0x77, 0xe2, 0x4c, 0xca, 0xd6, 0x0c, 0x3c,
};
static const unsigned char kat359_nonce[] = {
    0x72, 0x00, 0x6a, 0x08, 0x65, 0x3c, 0x6b, 0x83, 0x7f, 0xe2, 0x02, 0x43,
    0x8b, 0xaa, 0x43, 0x89,
};
static const unsigned char kat359_persstr[] = {
    0x09, 0x8f, 0x89, 0x66, 0x20, 0xff, 0x0e, 0x41, 0x43, 0x79, 0x7c, 0x1c,
    0x6b, 0xe8, 0x1c, 0x62, 0x5e, 0xd3, 0x7f, 0x79, 0xbc, 0x05, 0xa2, 0xd3,
    0x01, 0x84, 0x59, 0x09, 0x7f, 0xd1, 0x40, 0xb4,
};
static const unsigned char kat359_addin0[] = {
    0xc2, 0xa2, 0x89, 0xd2, 0x18, 0xdc, 0xd2, 0x80, 0xe6, 0xa8, 0x82, 0xe3,
    0x0e, 0x18, 0x2d, 0xa6, 0xa1, 0x3f, 0xd0, 0x96, 0x36, 0x85, 0xb3, 0xa9,
    0xaf, 0xca, 0xae, 0x8a, 0xe3, 0xac, 0xad, 0x69,
};
static const unsigned char kat359_addin1[] = {
    0x36, 0x1a, 0xcb, 0x27, 0xf5, 0x01, 0x40, 0x80, 0xa9, 0xe5, 0x21, 0xe0,
    0x42, 0x0d, 0xfd, 0x58, 0xe9, 0x20, 0x91, 0x11, 0xdf, 0x3b, 0x1b, 0xc7,
    0x7b, 0xef, 0x23, 0x55, 0x15, 0x81, 0x88, 0x72,
};
static const unsigned char kat359_retbits[] = {
    0xb0, 0x07, 0xae, 0xb8, 0x7b, 0x47, 0x85, 0x81, 0x5c, 0xf0, 0xb5, 0x46,
    0xff, 0x25, 0xdc, 0xf5, 0xbf, 0x96, 0xb3, 0xfa, 0xa8, 0x8d, 0xf8, 0xe4,
    0x5a, 0x12, 0xb7, 0x88, 0xc4, 0x74, 0x00, 0xce, 0xab, 0x79, 0xd9, 0x61,
    0xc9, 0xd6, 0x5b, 0xc4, 0x75, 0xb5, 0x6a, 0xbf, 0xf9, 0xba, 0x1d, 0x0e,
    0x3a, 0x5f, 0xf5, 0xfe, 0x56, 0xfb, 0xda, 0x06, 0xcd, 0xbd, 0x71, 0x02,
    0xa2, 0x72, 0x5a, 0xaa,
};
static const struct drbg_kat_no_reseed kat359_t = {
    13, kat359_entropyin, kat359_nonce, kat359_persstr,
    kat359_addin0, kat359_addin1, kat359_retbits
};
static const struct drbg_kat kat359 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat359_t
};

static const unsigned char kat360_entropyin[] = {
    0xc6, 0xe8, 0x0e, 0x41, 0xa5, 0x24, 0x87, 0xd6, 0x26, 0xa4, 0x4e, 0xeb,
    0xc4, 0x35, 0xa1, 0xa4, 0x15, 0xda, 0x62, 0x53, 0x68, 0xff, 0xfa, 0xab,
};
static const unsigned char kat360_nonce[] = {
    0x8e, 0xb6, 0x09, 0x35, 0xa5, 0x9e, 0x8d, 0x55, 0xfe, 0x82, 0x66, 0xb4,
    0x34, 0x19, 0xd1, 0x05,
};
static const unsigned char kat360_persstr[] = {
    0xcc, 0x80, 0x42, 0xc0, 0xbe, 0x1c, 0x21, 0x27, 0x73, 0xc7, 0xd6, 0xe5,
    0xd8, 0x61, 0x82, 0xde, 0xe8, 0x1c, 0xde, 0x58, 0x70, 0x3e, 0x9b, 0x1d,
    0x6e, 0xd0, 0x8b, 0x66, 0x4e, 0x04, 0x77, 0x9c,
};
static const unsigned char kat360_addin0[] = {
    0x21, 0x12, 0xe9, 0xf2, 0x16, 0xee, 0xfd, 0xae, 0xe0, 0xc3, 0x7f, 0x6b,
    0x5c, 0xcd, 0xd0, 0x0c, 0xdc, 0x80, 0x91, 0xa2, 0x8d, 0x09, 0x4e, 0x56,
    0x7b, 0x69, 0x71, 0x09, 0xa7, 0x11, 0x0b, 0x7e,
};
static const unsigned char kat360_addin1[] = {
    0x4c, 0x32, 0x95, 0x4a, 0x5c, 0xa0, 0x45, 0x03, 0x78, 0xf5, 0x90, 0x7b,
    0x6e, 0xa0, 0xbe, 0x77, 0x0e, 0x40, 0x79, 0x70, 0x80, 0xa0, 0x57, 0xb1,
    0xa1, 0x4b, 0x45, 0x36, 0xe3, 0x89, 0xde, 0x98,
};
static const unsigned char kat360_retbits[] = {
    0xa5, 0x1d, 0x5f, 0x66, 0x1b, 0xb1, 0xd5, 0x0f, 0x2a, 0xae, 0xb0, 0xb8,
    0xb7, 0xaf, 0x40, 0x61, 0x42, 0xa1, 0x13, 0x57, 0x78, 0x46, 0x2b, 0x80,
    0x20, 0x16, 0xc5, 0xe3, 0x2b, 0xa5, 0x56, 0x3a, 0xdd, 0x27, 0x44, 0xd6,
    0x2c, 0x82, 0xc1, 0xdf, 0x91, 0xe6, 0xa9, 0x1d, 0xba, 0xa9, 0x53, 0x28,
    0x76, 0x3d, 0x23, 0x20, 0xac, 0x6c, 0x20, 0xb4, 0x24, 0xaa, 0x72, 0x9d,
    0x2b, 0x1c, 0x1a, 0x44,
};
static const struct drbg_kat_no_reseed kat360_t = {
    14, kat360_entropyin, kat360_nonce, kat360_persstr,
    kat360_addin0, kat360_addin1, kat360_retbits
};
static const struct drbg_kat kat360 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat360_t
};

static const unsigned char kat361_entropyin[] = {
    0x4d, 0x5b, 0xe9, 0x19, 0x99, 0xc1, 0xdc, 0x6a, 0xe2, 0xe5, 0xf6, 0xde,
    0xb5, 0x63, 0xb1, 0x25, 0xbc, 0x84, 0x39, 0xe8, 0x5f, 0x25, 0x76, 0xfc,
};
static const unsigned char kat361_nonce[] = {
    0x11, 0xe1, 0xc2, 0xd4, 0xd4, 0xf7, 0x3b, 0x9c, 0x45, 0x7f, 0xca, 0xa0,
    0x6f, 0x4a, 0xf2, 0x2e,
};
static const unsigned char kat361_persstr[] = {0};
static const unsigned char kat361_addin0[] = {0};
static const unsigned char kat361_addin1[] = {0};
static const unsigned char kat361_retbits[] = {
    0x53, 0xc2, 0xc0, 0x23, 0x7e, 0xdf, 0x94, 0x25, 0x84, 0x2a, 0xb9, 0x1c,
    0x63, 0x72, 0x36, 0x16, 0xe9, 0x8b, 0xb6, 0xc1, 0x2d, 0x16, 0xbf, 0x80,
    0x87, 0x77, 0x2d, 0x0a, 0x08, 0x0d, 0x28, 0x9f, 0x8b, 0x4f, 0x35, 0xdf,
    0xda, 0xef, 0x5b, 0x11, 0xcd, 0x58, 0x88, 0x14, 0xc6, 0xbc, 0x01, 0xdc,
    0x7e, 0x23, 0xb9, 0xbd, 0xd3, 0x9c, 0x0a, 0xee, 0x74, 0x07, 0xf7, 0x10,
    0x54, 0xa7, 0xc9, 0xaa,
};
static const struct drbg_kat_no_reseed kat361_t = {
    0, kat361_entropyin, kat361_nonce, kat361_persstr,
    kat361_addin0, kat361_addin1, kat361_retbits
};
static const struct drbg_kat kat361 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat361_t
};

static const unsigned char kat362_entropyin[] = {
    0x24, 0xd8, 0xc6, 0x0f, 0x62, 0x90, 0x8b, 0x44, 0x74, 0xb6, 0xa0, 0x1e,
    0xc8, 0x8c, 0x99, 0x5b, 0x35, 0x7f, 0x82, 0xe2, 0x0d, 0x21, 0xc8, 0xf5,
};
static const unsigned char kat362_nonce[] = {
    0x45, 0xcb, 0x53, 0x4e, 0x0e, 0xa5, 0xc1, 0xb1, 0xa7, 0x5e, 0x6a, 0x66,
    0xd9, 0x90, 0xc7, 0x15,
};
static const unsigned char kat362_persstr[] = {0};
static const unsigned char kat362_addin0[] = {0};
static const unsigned char kat362_addin1[] = {0};
static const unsigned char kat362_retbits[] = {
    0x86, 0xa1, 0xde, 0xb9, 0xa3, 0x28, 0x63, 0xf7, 0x13, 0xc3, 0xe6, 0xae,
    0xa8, 0x15, 0x04, 0xc7, 0xec, 0x76, 0x6d, 0x0b, 0x7e, 0x5a, 0xa8, 0x00,
    0xef, 0x0b, 0x44, 0x9c, 0xc3, 0x34, 0x08, 0xc7, 0xf8, 0x7e, 0x71, 0x2c,
    0xfd, 0x58, 0x84, 0x2c, 0x90, 0x5f, 0x8a, 0x26, 0x2a, 0x4e, 0x2a, 0xf7,
    0x45, 0xfa, 0x58, 0x4b, 0x37, 0x0b, 0x17, 0xe2, 0xaf, 0x89, 0xfc, 0xb7,
    0x3d, 0x39, 0x9c, 0xf3,
};
static const struct drbg_kat_no_reseed kat362_t = {
    1, kat362_entropyin, kat362_nonce, kat362_persstr,
    kat362_addin0, kat362_addin1, kat362_retbits
};
static const struct drbg_kat kat362 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat362_t
};

static const unsigned char kat363_entropyin[] = {
    0x33, 0x54, 0xa9, 0x53, 0xf8, 0xe6, 0xcc, 0x3b, 0x64, 0xa8, 0x9f, 0xa7,
    0x4b, 0xcb, 0x45, 0x97, 0x1f, 0x14, 0xc4, 0xd1, 0xf7, 0x9c, 0xf9, 0x4a,
};
static const unsigned char kat363_nonce[] = {
    0xd2, 0xa8, 0x30, 0x7d, 0x96, 0x59, 0xfd, 0x46, 0xa0, 0x73, 0xeb, 0x33,
    0xf3, 0x48, 0x1a, 0xb7,
};
static const unsigned char kat363_persstr[] = {0};
static const unsigned char kat363_addin0[] = {0};
static const unsigned char kat363_addin1[] = {0};
static const unsigned char kat363_retbits[] = {
    0xf6, 0x42, 0x33, 0x32, 0xc2, 0xb0, 0xd7, 0xb6, 0xed, 0xbd, 0x09, 0xe5,
    0x90, 0x20, 0x7b, 0xfb, 0xec, 0xa6, 0x2b, 0x4e, 0x8d, 0x80, 0xe8, 0x19,
    0xd6, 0x77, 0x34, 0x11, 0xb3, 0x61, 0xac, 0x63, 0xc5, 0x8d, 0x58, 0x7d,
    0x6e, 0xbb, 0xae, 0x5e, 0x0a, 0x21, 0x1d, 0x1f, 0xf8, 0xde, 0x7f, 0x90,
    0x1d, 0xf1, 0x53, 0x37, 0x38, 0x25, 0x6e, 0x44, 0xed, 0xaa, 0x2b, 0xb7,
    0xc1, 0xa1, 0x4e, 0x0e,
};
static const struct drbg_kat_no_reseed kat363_t = {
    2, kat363_entropyin, kat363_nonce, kat363_persstr,
    kat363_addin0, kat363_addin1, kat363_retbits
};
static const struct drbg_kat kat363 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat363_t
};

static const unsigned char kat364_entropyin[] = {
    0x93, 0x18, 0x37, 0xf9, 0x6b, 0x3b, 0xe2, 0x58, 0x7c, 0xe6, 0x37, 0xe3,
    0x32, 0xa7, 0xc4, 0x44, 0x17, 0x37, 0xf9, 0xbf, 0x9b, 0x4a, 0x5a, 0x33,
};
static const unsigned char kat364_nonce[] = {
    0xdb, 0xab, 0x3e, 0x4e, 0xec, 0xb4, 0x53, 0xa1, 0x6c, 0xea, 0x08, 0xd6,
    0xfa, 0xfc, 0x2f, 0x36,
};
static const unsigned char kat364_persstr[] = {0};
static const unsigned char kat364_addin0[] = {0};
static const unsigned char kat364_addin1[] = {0};
static const unsigned char kat364_retbits[] = {
    0xa8, 0x46, 0x4a, 0x65, 0xfd, 0x94, 0x2c, 0x2b, 0xee, 0xa2, 0x02, 0x95,
    0xcd, 0x25, 0x56, 0x01, 0x21, 0x08, 0x0f, 0xb6, 0x9c, 0xdc, 0xb4, 0x55,
    0x9a, 0xde, 0xd0, 0xc6, 0x70, 0xc8, 0x28, 0x38, 0xf6, 0xba, 0x46, 0xb0,
    0x0f, 0x5d, 0x59, 0x35, 0xcc, 0x77, 0xa1, 0x4a, 0x8c, 0x4e, 0xda, 0x02,
    0xae, 0x19, 0x8e, 0xa8, 0x7c, 0x9c, 0x98, 0x20, 0x4d, 0x38, 0xa6, 0x6a,
    0x8c, 0x68, 0x6f, 0x67,
};
static const struct drbg_kat_no_reseed kat364_t = {
    3, kat364_entropyin, kat364_nonce, kat364_persstr,
    kat364_addin0, kat364_addin1, kat364_retbits
};
static const struct drbg_kat kat364 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat364_t
};

static const unsigned char kat365_entropyin[] = {
    0x55, 0x01, 0x82, 0x72, 0x5f, 0x54, 0x5e, 0x7c, 0xd4, 0xfd, 0x7b, 0x97,
    0x7f, 0x58, 0x74, 0x51, 0x11, 0x7c, 0xc7, 0x31, 0x92, 0x5b, 0xa2, 0x7c,
};
static const unsigned char kat365_nonce[] = {
    0xde, 0x0f, 0xc3, 0xbf, 0x40, 0xb6, 0x1e, 0xdc, 0xc3, 0x1e, 0x0c, 0xe6,
    0xf2, 0x01, 0x1c, 0xda,
};
static const unsigned char kat365_persstr[] = {0};
static const unsigned char kat365_addin0[] = {0};
static const unsigned char kat365_addin1[] = {0};
static const unsigned char kat365_retbits[] = {
    0xea, 0x0d, 0xfb, 0x57, 0xcf, 0x50, 0x19, 0xc7, 0x3f, 0x58, 0x61, 0x8f,
    0xc0, 0x00, 0x31, 0x40, 0xa3, 0x6e, 0xd6, 0x87, 0xac, 0x14, 0x51, 0xa5,
    0xbd, 0xe0, 0x90, 0x50, 0xaa, 0x1b, 0x2f, 0x6f, 0xd1, 0xcd, 0x33, 0x7d,
    0xb4, 0x38, 0xca, 0x79, 0x59, 0x84, 0xca, 0x6a, 0x2b, 0x93, 0x56, 0x17,
    0xd9, 0x84, 0x66, 0x99, 0xd9, 0x9b, 0x69, 0x78, 0xfe, 0x87, 0x8c, 0x12,
    0xc3, 0xc8, 0xf9, 0xc9,
};
static const struct drbg_kat_no_reseed kat365_t = {
    4, kat365_entropyin, kat365_nonce, kat365_persstr,
    kat365_addin0, kat365_addin1, kat365_retbits
};
static const struct drbg_kat kat365 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat365_t
};

static const unsigned char kat366_entropyin[] = {
    0x39, 0xc7, 0xcd, 0x1d, 0x98, 0x62, 0x13, 0x3a, 0x1a, 0xd4, 0xd4, 0x55,
    0x92, 0x11, 0x6c, 0x0e, 0x09, 0x89, 0x6f, 0xa6, 0x68, 0xad, 0xc7, 0x8a,
};
static const unsigned char kat366_nonce[] = {
    0xd0, 0x87, 0x82, 0x92, 0x52, 0xbf, 0x38, 0x39, 0x3c, 0xd0, 0x0f, 0x87,
    0xf7, 0x4d, 0xf4, 0xca,
};
static const unsigned char kat366_persstr[] = {0};
static const unsigned char kat366_addin0[] = {0};
static const unsigned char kat366_addin1[] = {0};
static const unsigned char kat366_retbits[] = {
    0xf4, 0x90, 0x21, 0xcd, 0x85, 0x70, 0x71, 0x85, 0x83, 0xed, 0x21, 0xec,
    0x55, 0x46, 0x8b, 0x4c, 0x45, 0xf8, 0x7a, 0x7b, 0x8f, 0xf6, 0xdd, 0x51,
    0x72, 0xd9, 0x37, 0xd6, 0xe9, 0xd9, 0xb4, 0x08, 0x6a, 0x75, 0x9a, 0xad,
    0x92, 0x7e, 0xf9, 0xe8, 0xb7, 0xc2, 0x65, 0x9b, 0x0f, 0x06, 0xc2, 0xa4,
    0xf8, 0xe3, 0x5e, 0xd8, 0xb6, 0x71, 0x47, 0x0d, 0x9d, 0xc6, 0x1b, 0x03,
    0x1d, 0x7d, 0x17, 0x82,
};
static const struct drbg_kat_no_reseed kat366_t = {
    5, kat366_entropyin, kat366_nonce, kat366_persstr,
    kat366_addin0, kat366_addin1, kat366_retbits
};
static const struct drbg_kat kat366 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat366_t
};

static const unsigned char kat367_entropyin[] = {
    0x88, 0xfa, 0x53, 0x8a, 0xeb, 0xf8, 0x34, 0x90, 0x74, 0xee, 0xb4, 0x5d,
    0x95, 0x67, 0xa9, 0x00, 0xc1, 0xc9, 0x76, 0x43, 0xb7, 0x9a, 0x2c, 0x16,
};
static const unsigned char kat367_nonce[] = {
    0xbb, 0x41, 0x66, 0x91, 0xb8, 0x63, 0x62, 0xfc, 0x96, 0x77, 0xe3, 0xfe,
    0x50, 0xd1, 0x57, 0xa9,
};
static const unsigned char kat367_persstr[] = {0};
static const unsigned char kat367_addin0[] = {0};
static const unsigned char kat367_addin1[] = {0};
static const unsigned char kat367_retbits[] = {
    0x9e, 0xf3, 0x1c, 0xf9, 0x31, 0xf2, 0x1d, 0x89, 0x86, 0x8e, 0xaa, 0x24,
    0x5f, 0xf0, 0x48, 0x14, 0x60, 0x59, 0x68, 0x4e, 0x2f, 0xa5, 0x61, 0x9a,
    0x44, 0x0f, 0x1d, 0x8a, 0x19, 0x55, 0x00, 0x35, 0x5f, 0xe3, 0x94, 0xaf,
    0xe3, 0xd3, 0xf9, 0xec, 0x45, 0xd2, 0x06, 0xb3, 0x7f, 0xd8, 0xfb, 0x47,
    0x5b, 0xdb, 0x1a, 0xc8, 0x80, 0x0c, 0xe5, 0x4f, 0x04, 0xc9, 0x94, 0x03,
    0xeb, 0x76, 0xed, 0xee,
};
static const struct drbg_kat_no_reseed kat367_t = {
    6, kat367_entropyin, kat367_nonce, kat367_persstr,
    kat367_addin0, kat367_addin1, kat367_retbits
};
static const struct drbg_kat kat367 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat367_t
};

static const unsigned char kat368_entropyin[] = {
    0x81, 0xa8, 0xb2, 0x19, 0x2d, 0x59, 0xa2, 0x5d, 0x54, 0x80, 0x46, 0x5e,
    0x50, 0x8c, 0x11, 0x93, 0x1a, 0x1a, 0xe5, 0xa4, 0x27, 0xf6, 0xd1, 0x17,
};
static const unsigned char kat368_nonce[] = {
    0x1b, 0x61, 0x0d, 0xf5, 0xaf, 0x73, 0xb7, 0x0e, 0x85, 0xd9, 0x7c, 0x15,
    0x9a, 0x13, 0xbc, 0x71,
};
static const unsigned char kat368_persstr[] = {0};
static const unsigned char kat368_addin0[] = {0};
static const unsigned char kat368_addin1[] = {0};
static const unsigned char kat368_retbits[] = {
    0xcb, 0x0b, 0xab, 0x82, 0xb9, 0x1a, 0x72, 0x7c, 0xa3, 0xe1, 0x04, 0xb9,
    0xea, 0xca, 0xe0, 0x86, 0x05, 0xe0, 0x6d, 0x18, 0xee, 0xa0, 0xee, 0x2c,
    0x71, 0xeb, 0xab, 0xde, 0x5c, 0x98, 0xcc, 0xca, 0x7e, 0xdb, 0x47, 0x93,
    0x83, 0x6a, 0x65, 0xf2, 0x83, 0x8b, 0x82, 0x83, 0xcf, 0x4a, 0xdf, 0x49,
    0xfa, 0x1f, 0x38, 0x36, 0x87, 0x76, 0xe5, 0xba, 0x4e, 0xf7, 0xc5, 0x14,
    0x37, 0x88, 0x8f, 0xdf,
};
static const struct drbg_kat_no_reseed kat368_t = {
    7, kat368_entropyin, kat368_nonce, kat368_persstr,
    kat368_addin0, kat368_addin1, kat368_retbits
};
static const struct drbg_kat kat368 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat368_t
};

static const unsigned char kat369_entropyin[] = {
    0xa4, 0x95, 0x85, 0xca, 0x40, 0xea, 0x98, 0x00, 0x20, 0x8a, 0xbe, 0xf7,
    0x03, 0xa4, 0xc2, 0x0b, 0xd3, 0xc4, 0xca, 0x59, 0x65, 0x25, 0x68, 0x2a,
};
static const unsigned char kat369_nonce[] = {
    0x11, 0xf7, 0xe4, 0x63, 0x0d, 0xc9, 0x6e, 0x2e, 0x4a, 0x35, 0x6f, 0x8e,
    0xff, 0xe8, 0xee, 0xf6,
};
static const unsigned char kat369_persstr[] = {0};
static const unsigned char kat369_addin0[] = {0};
static const unsigned char kat369_addin1[] = {0};
static const unsigned char kat369_retbits[] = {
    0x63, 0x3a, 0x67, 0x28, 0xbf, 0x62, 0x16, 0xea, 0x30, 0x3f, 0x3b, 0x27,
    0xe4, 0xfa, 0xa5, 0x2d, 0x2d, 0xa4, 0x76, 0x8e, 0x6e, 0x7c, 0xed, 0x7d,
    0xcd, 0x33, 0xd4, 0xdf, 0x25, 0xad, 0x03, 0xe9, 0x1f, 0x5a, 0xe9, 0xe9,
    0xe1, 0xf1, 0x9c, 0x4d, 0xad, 0x74, 0xe3, 0xa1, 0xed, 0xe1, 0x52, 0x84,
    0xaf, 0xcc, 0xf5, 0x14, 0x8f, 0x18, 0x8b, 0x95, 0xa0, 0x56, 0x76, 0x70,
    0xe2, 0x06, 0xec, 0xa8,
};
static const struct drbg_kat_no_reseed kat369_t = {
    8, kat369_entropyin, kat369_nonce, kat369_persstr,
    kat369_addin0, kat369_addin1, kat369_retbits
};
static const struct drbg_kat kat369 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat369_t
};

static const unsigned char kat370_entropyin[] = {
    0xbd, 0xe5, 0x3c, 0xa3, 0xe0, 0xb0, 0x11, 0x6b, 0x61, 0x19, 0x15, 0xa3,
    0xbe, 0xa8, 0x01, 0x41, 0x80, 0x14, 0x73, 0x40, 0x88, 0x63, 0x9c, 0xdd,
};
static const unsigned char kat370_nonce[] = {
    0x71, 0xb4, 0x99, 0x3c, 0xaf, 0xd8, 0x68, 0xac, 0x20, 0x9d, 0xb7, 0x22,
    0xcb, 0x75, 0x7f, 0xbc,
};
static const unsigned char kat370_persstr[] = {0};
static const unsigned char kat370_addin0[] = {0};
static const unsigned char kat370_addin1[] = {0};
static const unsigned char kat370_retbits[] = {
    0x6c, 0xf1, 0xdc, 0x52, 0x14, 0xaf, 0x97, 0xc1, 0xc6, 0xd4, 0x3c, 0x31,
    0xac, 0xe4, 0x0c, 0xed, 0xbf, 0x3c, 0x32, 0xbc, 0x7f, 0x8d, 0x69, 0x28,
    0xbc, 0xb6, 0x13, 0xb6, 0xa9, 0xb3, 0x70, 0x11, 0xd4, 0x0a, 0x16, 0xe8,
    0x73, 0x9b, 0x2f, 0xba, 0xfd, 0xfe, 0x77, 0x47, 0xc8, 0xd5, 0x16, 0x3d,
    0xc4, 0x22, 0x3c, 0xe3, 0x30, 0x35, 0x32, 0xc7, 0x19, 0xab, 0xb1, 0x69,
    0xe6, 0x9d, 0x3d, 0xb9,
};
static const struct drbg_kat_no_reseed kat370_t = {
    9, kat370_entropyin, kat370_nonce, kat370_persstr,
    kat370_addin0, kat370_addin1, kat370_retbits
};
static const struct drbg_kat kat370 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat370_t
};

static const unsigned char kat371_entropyin[] = {
    0x12, 0x40, 0xc0, 0x7f, 0x0c, 0x42, 0xda, 0x18, 0x3e, 0xbf, 0x8f, 0x44,
    0x13, 0xab, 0x31, 0x6a, 0x88, 0x24, 0x61, 0x66, 0xaa, 0x7b, 0xb7, 0x59,
};
static const unsigned char kat371_nonce[] = {
    0x6e, 0x52, 0x9d, 0x4e, 0x92, 0x47, 0x75, 0x79, 0xd0, 0x7f, 0xb9, 0xd2,
    0xde, 0x05, 0x4d, 0x8b,
};
static const unsigned char kat371_persstr[] = {0};
static const unsigned char kat371_addin0[] = {0};
static const unsigned char kat371_addin1[] = {0};
static const unsigned char kat371_retbits[] = {
    0xdc, 0xdf, 0xa1, 0xa5, 0x73, 0x3e, 0x5b, 0xec, 0x22, 0xa2, 0x4c, 0x01,
    0xb6, 0xd4, 0x6a, 0xe1, 0x02, 0x09, 0x35, 0x12, 0x40, 0xdb, 0xb4, 0x26,
    0x86, 0xbc, 0xa8, 0x11, 0x7d, 0xaa, 0xa2, 0x28, 0x73, 0x26, 0x94, 0x20,
    0x21, 0xf1, 0x98, 0xec, 0x6b, 0xd2, 0xfd, 0x63, 0x4f, 0xe8, 0xcc, 0xd2,
    0x4a, 0x56, 0x2c, 0x48, 0x29, 0x76, 0x30, 0xf9, 0xf7, 0x18, 0x62, 0x3b,
    0xca, 0x0a, 0x64, 0xd7,
};
static const struct drbg_kat_no_reseed kat371_t = {
    10, kat371_entropyin, kat371_nonce, kat371_persstr,
    kat371_addin0, kat371_addin1, kat371_retbits
};
static const struct drbg_kat kat371 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat371_t
};

static const unsigned char kat372_entropyin[] = {
    0x72, 0xed, 0x4b, 0x06, 0xe7, 0x5b, 0x59, 0xc6, 0xfb, 0xd1, 0x7b, 0xc7,
    0x26, 0xc2, 0xe9, 0x2a, 0x22, 0x01, 0x62, 0xbc, 0x6c, 0x1f, 0xfe, 0xb5,
};
static const unsigned char kat372_nonce[] = {
    0x46, 0x6e, 0xeb, 0x26, 0x5f, 0xa1, 0x47, 0xea, 0x1f, 0x77, 0x3f, 0x98,
    0x08, 0x7f, 0xbb, 0x83,
};
static const unsigned char kat372_persstr[] = {0};
static const unsigned char kat372_addin0[] = {0};
static const unsigned char kat372_addin1[] = {0};
static const unsigned char kat372_retbits[] = {
    0x46, 0xc3, 0x56, 0x17, 0x4b, 0x28, 0xae, 0x3a, 0x90, 0x30, 0x71, 0x64,
    0x3a, 0xbc, 0xed, 0x78, 0x4b, 0x31, 0x68, 0xbb, 0x20, 0xa7, 0x8a, 0x08,
    0x2a, 0xbc, 0x3c, 0x8d, 0x76, 0xab, 0x53, 0x67, 0x3c, 0xec, 0x21, 0x81,
    0xd4, 0x0a, 0x42, 0x94, 0xcf, 0xfc, 0xc8, 0xa2, 0xfa, 0x43, 0xb8, 0x9b,
    0xcb, 0x96, 0x2f, 0x5e, 0xf3, 0x30, 0x0e, 0x7d, 0xf5, 0x00, 0xfb, 0x0c,
    0x96, 0xbe, 0x85, 0x8a,
};
static const struct drbg_kat_no_reseed kat372_t = {
    11, kat372_entropyin, kat372_nonce, kat372_persstr,
    kat372_addin0, kat372_addin1, kat372_retbits
};
static const struct drbg_kat kat372 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat372_t
};

static const unsigned char kat373_entropyin[] = {
    0xc4, 0x47, 0xb5, 0xee, 0xc4, 0x1e, 0x14, 0xa9, 0xce, 0x9b, 0xbd, 0x85,
    0x93, 0xe8, 0xa7, 0xc0, 0xf5, 0x55, 0x9f, 0x4b, 0x0e, 0x11, 0x76, 0x88,
};
static const unsigned char kat373_nonce[] = {
    0x46, 0x55, 0xd3, 0xc7, 0x8c, 0xaa, 0x1c, 0x1c, 0xfe, 0x6d, 0x6e, 0x11,
    0x66, 0x46, 0x8b, 0x28,
};
static const unsigned char kat373_persstr[] = {0};
static const unsigned char kat373_addin0[] = {0};
static const unsigned char kat373_addin1[] = {0};
static const unsigned char kat373_retbits[] = {
    0x68, 0xd4, 0x1f, 0xe2, 0x8b, 0xdb, 0x7d, 0xe4, 0xa7, 0x61, 0xcb, 0x60,
    0x04, 0xec, 0xed, 0xfa, 0x31, 0x09, 0x46, 0x07, 0x50, 0x92, 0x11, 0x1e,
    0x80, 0x6b, 0xee, 0xd6, 0x34, 0xff, 0x54, 0x0c, 0x38, 0x98, 0x7f, 0x16,
    0xa8, 0x9b, 0xe5, 0x2d, 0x35, 0xe3, 0x07, 0x59, 0x03, 0x0c, 0xbc, 0xa2,
    0x40, 0xbb, 0xe9, 0xa5, 0xdb, 0xbf, 0xf1, 0x9d, 0x15, 0x0b, 0xe3, 0xa2,
    0x29, 0x6b, 0x8c, 0xeb,
};
static const struct drbg_kat_no_reseed kat373_t = {
    12, kat373_entropyin, kat373_nonce, kat373_persstr,
    kat373_addin0, kat373_addin1, kat373_retbits
};
static const struct drbg_kat kat373 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat373_t
};

static const unsigned char kat374_entropyin[] = {
    0x24, 0x04, 0xf9, 0x87, 0x09, 0x45, 0xbd, 0x40, 0x4f, 0xe1, 0x84, 0xb1,
    0x19, 0xa8, 0xa3, 0x03, 0x71, 0xae, 0xfa, 0x80, 0x15, 0xd4, 0x55, 0xcc,
};
static const unsigned char kat374_nonce[] = {
    0xe5, 0xfd, 0x16, 0x64, 0x17, 0x28, 0xb4, 0xd4, 0x6e, 0x3f, 0x0a, 0x43,
    0xae, 0x34, 0x84, 0x40,
};
static const unsigned char kat374_persstr[] = {0};
static const unsigned char kat374_addin0[] = {0};
static const unsigned char kat374_addin1[] = {0};
static const unsigned char kat374_retbits[] = {
    0xea, 0x4b, 0xc0, 0xde, 0xc0, 0xf1, 0xa6, 0x1c, 0xf2, 0xac, 0x89, 0x0d,
    0x0c, 0x7d, 0x8d, 0xb4, 0x29, 0x32, 0x91, 0x48, 0x6d, 0x65, 0xf3, 0xed,
    0x8b, 0x5f, 0x75, 0x8e, 0xee, 0xe9, 0xd5, 0xde, 0xed, 0x2a, 0x1d, 0x05,
    0x06, 0xfe, 0xbe, 0x20, 0xbf, 0xd1, 0x8b, 0x68, 0x0c, 0x5a, 0x8d, 0x77,
    0x7c, 0x6b, 0x3b, 0xbb, 0x8c, 0xa1, 0xbf, 0xfa, 0x0f, 0x10, 0x5e, 0xe7,
    0xc1, 0x50, 0x73, 0x59,
};
static const struct drbg_kat_no_reseed kat374_t = {
    13, kat374_entropyin, kat374_nonce, kat374_persstr,
    kat374_addin0, kat374_addin1, kat374_retbits
};
static const struct drbg_kat kat374 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat374_t
};

static const unsigned char kat375_entropyin[] = {
    0x45, 0x70, 0xff, 0x9a, 0xb9, 0x2e, 0x5b, 0x69, 0x4b, 0x8a, 0xce, 0xb3,
    0xd7, 0x65, 0xf4, 0x87, 0x2f, 0x2e, 0xa6, 0xad, 0x77, 0x07, 0xc5, 0x10,
};
static const unsigned char kat375_nonce[] = {
    0xa3, 0x46, 0x1b, 0xc1, 0x7d, 0x5d, 0xb6, 0x92, 0xf8, 0xcd, 0xd8, 0xcf,
    0x79, 0x18, 0x62, 0xb5,
};
static const unsigned char kat375_persstr[] = {0};
static const unsigned char kat375_addin0[] = {0};
static const unsigned char kat375_addin1[] = {0};
static const unsigned char kat375_retbits[] = {
    0xb7, 0x76, 0x1d, 0x93, 0x00, 0xa2, 0x21, 0x8b, 0x28, 0x3b, 0x55, 0xce,
    0x29, 0xb4, 0x3c, 0x4e, 0x7a, 0x63, 0xfc, 0xc9, 0xb1, 0xd0, 0xc1, 0x68,
    0x6e, 0xfd, 0x8d, 0x5f, 0x0e, 0x40, 0x31, 0xc2, 0xdc, 0x74, 0x5f, 0x58,
    0xa5, 0x0b, 0x35, 0xbe, 0x81, 0xfc, 0xd5, 0xe9, 0x7d, 0x70, 0xf8, 0x96,
    0xb1, 0x7d, 0x3a, 0x64, 0x2a, 0x30, 0xba, 0x56, 0x72, 0x5d, 0x0a, 0xde,
    0xf1, 0x42, 0xef, 0xd6,
};
static const struct drbg_kat_no_reseed kat375_t = {
    14, kat375_entropyin, kat375_nonce, kat375_persstr,
    kat375_addin0, kat375_addin1, kat375_retbits
};
static const struct drbg_kat kat375 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat375_t
};

static const unsigned char kat376_entropyin[] = {
    0x52, 0x69, 0x4c, 0x7c, 0xf7, 0x0f, 0xd3, 0xd2, 0x07, 0xd2, 0xe7, 0xef,
    0x19, 0x2a, 0x04, 0x9d, 0xf3, 0x20, 0xe7, 0xf9, 0x19, 0x1b, 0x93, 0xc4,
};
static const unsigned char kat376_nonce[] = {
    0xff, 0xc0, 0xc8, 0xe1, 0x68, 0x41, 0xec, 0x38, 0x4c, 0x08, 0xb3, 0xcc,
    0xd8, 0x16, 0x03, 0x31,
};
static const unsigned char kat376_persstr[] = {0};
static const unsigned char kat376_addin0[] = {
    0x28, 0x7b, 0xba, 0x8b, 0xc6, 0xdf, 0xbd, 0x83, 0x6a, 0x21, 0xcb, 0x3d,
    0xda, 0x67, 0x8d, 0x3d, 0x7b, 0xa4, 0x1c, 0x5d, 0xfe, 0x08, 0xbd, 0xb3,
    0x56, 0xd9, 0x30, 0x72, 0x52, 0x82, 0x85, 0xe4,
};
static const unsigned char kat376_addin1[] = {
    0x1e, 0xea, 0x34, 0x1f, 0xf8, 0x1f, 0x90, 0x58, 0x3a, 0xff, 0xaa, 0x27,
    0x65, 0x9e, 0x2b, 0x06, 0xe3, 0xe0, 0x62, 0x06, 0x8c, 0xa4, 0xf1, 0xb4,
    0x08, 0x59, 0xa9, 0x22, 0x99, 0xa6, 0xb4, 0x90,
};
static const unsigned char kat376_retbits[] = {
    0xb3, 0x17, 0x56, 0x92, 0xec, 0xa9, 0xd7, 0xfe, 0x5e, 0xba, 0xb6, 0xdc,
    0x5d, 0x12, 0xc3, 0xd1, 0x10, 0x5e, 0x46, 0x73, 0x7c, 0xe6, 0xfe, 0xc8,
    0x9b, 0x26, 0x63, 0xef, 0xfd, 0xb4, 0x1a, 0x8f, 0x85, 0xf9, 0x83, 0x15,
    0x1a, 0xe4, 0xdb, 0xf7, 0x00, 0x20, 0x85, 0x3f, 0x48, 0x4f, 0xd1, 0xe9,
    0x6d, 0x74, 0x3d, 0x5f, 0x6c, 0xc4, 0x15, 0x68, 0x58, 0x90, 0xa0, 0xe3,
    0xf2, 0x5e, 0x3b, 0x69,
};
static const struct drbg_kat_no_reseed kat376_t = {
    0, kat376_entropyin, kat376_nonce, kat376_persstr,
    kat376_addin0, kat376_addin1, kat376_retbits
};
static const struct drbg_kat kat376 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat376_t
};

static const unsigned char kat377_entropyin[] = {
    0x0f, 0xf3, 0xd4, 0x27, 0x5e, 0xde, 0x36, 0xd4, 0xe2, 0x9c, 0x42, 0xb1,
    0x7e, 0xd1, 0xdf, 0x14, 0xde, 0x2e, 0x77, 0x94, 0x9a, 0xf5, 0x5e, 0xb7,
};
static const unsigned char kat377_nonce[] = {
    0xac, 0x33, 0xac, 0x97, 0x0a, 0x24, 0x75, 0xb9, 0x1f, 0x0a, 0x3b, 0xde,
    0xf1, 0x78, 0xbf, 0xd4,
};
static const unsigned char kat377_persstr[] = {0};
static const unsigned char kat377_addin0[] = {
    0xd3, 0x9f, 0xde, 0x81, 0x4e, 0xa3, 0xa6, 0x7e, 0x3b, 0x33, 0xe3, 0x25,
    0xab, 0x5e, 0x05, 0x53, 0xaa, 0x9a, 0x0f, 0x30, 0x14, 0xa7, 0x4f, 0xed,
    0x1b, 0x03, 0x02, 0x19, 0x54, 0x2f, 0xb0, 0x45,
};
static const unsigned char kat377_addin1[] = {
    0x1a, 0x1f, 0xc9, 0x24, 0x2e, 0x3e, 0xc8, 0x71, 0x25, 0x67, 0xe7, 0x80,
    0xdf, 0x05, 0x83, 0x06, 0x00, 0x1e, 0x6e, 0x10, 0xd5, 0xb6, 0xe4, 0xc2,
    0x1c, 0x6c, 0x17, 0x93, 0x14, 0x90, 0x34, 0xd3,
};
static const unsigned char kat377_retbits[] = {
    0xd2, 0x12, 0x92, 0x6f, 0x27, 0x63, 0x52, 0x74, 0x66, 0xff, 0x75, 0x11,
    0xdc, 0x56, 0xcf, 0x11, 0x78, 0xeb, 0x41, 0x50, 0xc5, 0xf6, 0x47, 0xbf,
    0x2c, 0x9c, 0x15, 0x05, 0xc0, 0x18, 0xf2, 0x71, 0xaf, 0x3a, 0x1b, 0xae,
    0x45, 0x61, 0x9f, 0x13, 0x07, 0xc1, 0x81, 0xbe, 0xdf, 0xc3, 0x62, 0xaf,
    0x34, 0x61, 0xc6, 0x03, 0x67, 0xaf, 0x08, 0xa3, 0x22, 0xf3, 0x67, 0xb3,
    0x2b, 0x0e, 0x0c, 0x1d,
};
static const struct drbg_kat_no_reseed kat377_t = {
    1, kat377_entropyin, kat377_nonce, kat377_persstr,
    kat377_addin0, kat377_addin1, kat377_retbits
};
static const struct drbg_kat kat377 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat377_t
};

static const unsigned char kat378_entropyin[] = {
    0x71, 0x32, 0x63, 0xcb, 0x43, 0x59, 0x77, 0x66, 0xc7, 0xa9, 0x25, 0x38,
    0x1c, 0x3e, 0xce, 0x46, 0xc1, 0x26, 0xa4, 0x73, 0x3b, 0x0b, 0xb5, 0xd9,
};
static const unsigned char kat378_nonce[] = {
    0x58, 0xfa, 0x56, 0x55, 0x73, 0xa9, 0xe9, 0xc7, 0x2e, 0x22, 0xe5, 0xeb,
    0xb2, 0xbe, 0xb8, 0x18,
};
static const unsigned char kat378_persstr[] = {0};
static const unsigned char kat378_addin0[] = {
    0x66, 0x73, 0x26, 0x0a, 0x58, 0x78, 0xb3, 0xda, 0x0b, 0xc6, 0x97, 0xb6,
    0x72, 0xd9, 0x8c, 0x36, 0x3e, 0x80, 0xe2, 0x55, 0x58, 0x8e, 0xfe, 0x79,
    0x60, 0x03, 0x0c, 0xdf, 0xb5, 0x87, 0x4c, 0x81,
};
static const unsigned char kat378_addin1[] = {
    0xb4, 0x2d, 0x33, 0xc1, 0x03, 0xea, 0xcb, 0xe8, 0xa2, 0xdc, 0x67, 0xcb,
    0x5f, 0x48, 0x57, 0x6e, 0x9b, 0xc8, 0x96, 0xda, 0xf0, 0x91, 0xa8, 0x33,
    0x9b, 0x0f, 0x04, 0x4c, 0x20, 0xc7, 0x2c, 0x8a,
};
static const unsigned char kat378_retbits[] = {
    0x6d, 0xfb, 0x11, 0x3e, 0x16, 0xaa, 0xeb, 0x39, 0x19, 0x13, 0x58, 0x27,
    0x31, 0x7f, 0xbd, 0xc2, 0x83, 0xe0, 0x66, 0x80, 0xdd, 0x4c, 0x83, 0x81,
    0x09, 0xd3, 0x73, 0x4b, 0x35, 0xfd, 0x44, 0x2d, 0x7f, 0xb7, 0x3e, 0x19,
    0x83, 0xd7, 0x4b, 0x1e, 0x3b, 0x4d, 0xa8, 0xe5, 0x9e, 0xa0, 0x21, 0xe9,
    0x29, 0x61, 0xc7, 0x04, 0xb6, 0x62, 0x7f, 0x15, 0xe8, 0xbe, 0xb5, 0xa3,
    0x22, 0x4b, 0x36, 0x18,
};
static const struct drbg_kat_no_reseed kat378_t = {
    2, kat378_entropyin, kat378_nonce, kat378_persstr,
    kat378_addin0, kat378_addin1, kat378_retbits
};
static const struct drbg_kat kat378 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat378_t
};

static const unsigned char kat379_entropyin[] = {
    0x26, 0x08, 0x08, 0xc8, 0x8f, 0x5f, 0x6b, 0xe1, 0x11, 0xb6, 0x83, 0xf6,
    0xf2, 0x97, 0x4f, 0xb5, 0xbb, 0xb4, 0x9e, 0x3d, 0x90, 0x38, 0x3b, 0x7f,
};
static const unsigned char kat379_nonce[] = {
    0xc5, 0xa8, 0xb8, 0x70, 0x66, 0x11, 0x6d, 0x0b, 0xd8, 0xae, 0x39, 0x30,
    0x25, 0xdb, 0x35, 0xcc,
};
static const unsigned char kat379_persstr[] = {0};
static const unsigned char kat379_addin0[] = {
    0x06, 0xb6, 0x26, 0xab, 0xc9, 0x66, 0xd9, 0x63, 0x06, 0x7f, 0x0f, 0xd9,
    0x3a, 0x84, 0xed, 0x71, 0x01, 0x62, 0x39, 0xe7, 0x9f, 0x63, 0x6f, 0xe9,
    0x09, 0x67, 0x2a, 0xe8, 0x4f, 0xf3, 0x6f, 0xa0,
};
static const unsigned char kat379_addin1[] = {
    0x3e, 0x32, 0x72, 0x6b, 0xa5, 0x5d, 0xf9, 0x15, 0xe0, 0xfb, 0x03, 0x2d,
    0xf0, 0x29, 0xdc, 0x96, 0xd4, 0x29, 0xcb, 0x5f, 0x45, 0x8b, 0x08, 0xe5,
    0xbe, 0x0b, 0x1c, 0x28, 0x33, 0x92, 0xbb, 0xe1,
};
static const unsigned char kat379_retbits[] = {
    0xb8, 0xfe, 0xbb, 0xd2, 0x88, 0x7b, 0xa6, 0x9e, 0x3f, 0xcb, 0x07, 0x8a,
    0xae, 0xff, 0xbd, 0x9b, 0x79, 0x08, 0x61, 0x80, 0x24, 0xe6, 0x03, 0x52,
    0x7c, 0xbe, 0x74, 0xc0, 0xf9, 0x9d, 0x6f, 0x40, 0xa0, 0xe8, 0x65, 0x18,
    0xac, 0xd9, 0xf9, 0xf7, 0xc3, 0x4b, 0x3f, 0x11, 0x78, 0x3a, 0x72, 0x78,
    0xfe, 0xc3, 0xc3, 0xb6, 0xd4, 0x07, 0xe1, 0x5a, 0x89, 0x84, 0x58, 0x43,
    0x23, 0x11, 0x3d, 0xe1,
};
static const struct drbg_kat_no_reseed kat379_t = {
    3, kat379_entropyin, kat379_nonce, kat379_persstr,
    kat379_addin0, kat379_addin1, kat379_retbits
};
static const struct drbg_kat kat379 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat379_t
};

static const unsigned char kat380_entropyin[] = {
    0xd5, 0x81, 0xd9, 0xca, 0xfb, 0x39, 0xad, 0xbd, 0x2b, 0xca, 0x4a, 0xfa,
    0x48, 0x3a, 0xcc, 0x39, 0x9d, 0x4e, 0xb4, 0x7b, 0xc7, 0x60, 0xd8, 0x43,
};
static const unsigned char kat380_nonce[] = {
    0x36, 0x43, 0x2c, 0x59, 0x7e, 0x6c, 0xd8, 0xfb, 0x70, 0xda, 0x84, 0xb2,
    0x5d, 0x8d, 0xcc, 0xc9,
};
static const unsigned char kat380_persstr[] = {0};
static const unsigned char kat380_addin0[] = {
    0x9c, 0x27, 0xf4, 0x75, 0x0a, 0x9b, 0x55, 0x32, 0x0d, 0x45, 0x15, 0xf2,
    0xe3, 0x52, 0x81, 0x48, 0xfc, 0x02, 0xd4, 0x8f, 0x84, 0x82, 0x24, 0x12,
    0x3a, 0xd6, 0xe9, 0x67, 0xff, 0x6a, 0x43, 0x50,
};
static const unsigned char kat380_addin1[] = {
    0xc7, 0xbf, 0xef, 0x3c, 0x24, 0x55, 0x2c, 0xd4, 0x66, 0xf2, 0xa5, 0x92,
    0x3c, 0x13, 0x4e, 0x56, 0xe9, 0xd1, 0x72, 0x55, 0xb4, 0x9c, 0x8b, 0x0f,
    0x2c, 0x81, 0x5f, 0x01, 0xfd, 0x72, 0xc4, 0xe0,
};
static const unsigned char kat380_retbits[] = {
    0xfc, 0x99, 0x80, 0xb6, 0x50, 0x06, 0xc6, 0x49, 0x34, 0x9f, 0x77, 0xb4,
    0x69, 0xe4, 0x0f, 0xb0, 0xb8, 0x03, 0xb2, 0xa7, 0xe0, 0xc1, 0x4a, 0xdc,
    0xd4, 0x0a, 0x4a, 0xe6, 0x51, 0xbb, 0xd1, 0xe1, 0x8d, 0xf6, 0xb8, 0xa1,
    0xc1, 0xa0, 0x28, 0xce, 0xb4, 0x3c, 0x26, 0x27, 0x8b, 0x43, 0xb9, 0x0d,
    0xe6, 0x17, 0x58, 0x37, 0x4f, 0x2c, 0x60, 0xba, 0x66, 0x3e, 0xf0, 0x4e,
    0xdd, 0xc3, 0x17, 0x48,
};
static const struct drbg_kat_no_reseed kat380_t = {
    4, kat380_entropyin, kat380_nonce, kat380_persstr,
    kat380_addin0, kat380_addin1, kat380_retbits
};
static const struct drbg_kat kat380 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat380_t
};

static const unsigned char kat381_entropyin[] = {
    0x41, 0xed, 0x09, 0xa7, 0x4f, 0xe7, 0x75, 0xbb, 0x08, 0x24, 0xa7, 0x82,
    0x08, 0x83, 0x33, 0x7a, 0xe4, 0x93, 0xb9, 0xaa, 0xa7, 0x1b, 0x41, 0xaa,
};
static const unsigned char kat381_nonce[] = {
    0x9b, 0x3f, 0xa2, 0xde, 0x23, 0x8c, 0x61, 0x1b, 0xfc, 0x34, 0x8f, 0x07,
    0x94, 0xb9, 0xf8, 0x65,
};
static const unsigned char kat381_persstr[] = {0};
static const unsigned char kat381_addin0[] = {
    0x27, 0x6c, 0xd3, 0xd2, 0x78, 0x77, 0x45, 0xc9, 0xf8, 0x93, 0xe7, 0xd0,
    0x97, 0xc4, 0x55, 0x2f, 0x3a, 0x0f, 0x29, 0x97, 0x7c, 0x33, 0xe6, 0x58,
    0x7a, 0xd1, 0x32, 0x59, 0x40, 0x89, 0xb6, 0xc0,
};
static const unsigned char kat381_addin1[] = {
    0xf9, 0x13, 0x36, 0x7e, 0x39, 0xcd, 0x59, 0xb7, 0xf4, 0x3d, 0x8a, 0x79,
    0x40, 0x57, 0x8f, 0x8a, 0x65, 0x19, 0xdb, 0xe6, 0xe0, 0x7a, 0xc2, 0xcc,
    0xb1, 0x18, 0x45, 0x3e, 0xd0, 0x28, 0x43, 0x32,
};
static const unsigned char kat381_retbits[] = {
    0xeb, 0x90, 0x27, 0xbb, 0x00, 0xa4, 0x79, 0x61, 0x12, 0x28, 0x2d, 0x14,
    0x7b, 0x6d, 0xde, 0xaf, 0xb4, 0x67, 0x3b, 0x83, 0x81, 0x00, 0x68, 0xdf,
    0x53, 0xb0, 0xbc, 0x8d, 0xda, 0x0a, 0x0b, 0xb7, 0x65, 0x1d, 0x2a, 0x38,
    0xd6, 0xd1, 0x92, 0x93, 0xa0, 0xe0, 0x17, 0x0e, 0xfe, 0x4e, 0xf8, 0x82,
    0x5d, 0xb8, 0xc7, 0xda, 0x6b, 0x48, 0x55, 0x4c, 0xdf, 0x68, 0x10, 0x4d,
    0xfe, 0x31, 0xc8, 0x46,
};
static const struct drbg_kat_no_reseed kat381_t = {
    5, kat381_entropyin, kat381_nonce, kat381_persstr,
    kat381_addin0, kat381_addin1, kat381_retbits
};
static const struct drbg_kat kat381 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat381_t
};

static const unsigned char kat382_entropyin[] = {
    0xd8, 0xe9, 0x27, 0xd9, 0x38, 0xbf, 0xca, 0x29, 0x76, 0x1c, 0xd6, 0x3a,
    0x09, 0xc2, 0xe6, 0x09, 0xa9, 0xe5, 0x03, 0x55, 0x91, 0xc2, 0x1b, 0xaa,
};
static const unsigned char kat382_nonce[] = {
    0x19, 0x57, 0xdb, 0x12, 0x71, 0x81, 0xd0, 0x51, 0xfc, 0xee, 0x6f, 0xb8,
    0x7f, 0x7c, 0x86, 0x89,
};
static const unsigned char kat382_persstr[] = {0};
static const unsigned char kat382_addin0[] = {
    0xfa, 0x05, 0x83, 0x2a, 0x2d, 0x88, 0x14, 0x70, 0xc9, 0xa6, 0x66, 0x02,
    0xc9, 0xef, 0x0a, 0x9f, 0x3c, 0x9a, 0xeb, 0xff, 0x5c, 0x92, 0x33, 0xd0,
    0x7e, 0xa4, 0xee, 0x46, 0x59, 0xa5, 0xdf, 0xf4,
};
static const unsigned char kat382_addin1[] = {
    0xf6, 0xb2, 0x35, 0xfc, 0x52, 0x4a, 0x46, 0xd6, 0xfa, 0x71, 0xeb, 0xcc,
    0xa2, 0x7c, 0x3f, 0x0c, 0x0f, 0x36, 0x66, 0x21, 0x6f, 0xdf, 0x89, 0xd2,
    0x5c, 0x45, 0xb2, 0x7b, 0x99, 0xa7, 0xac, 0x63,
};
static const unsigned char kat382_retbits[] = {
    0xbb, 0x51, 0xe9, 0xbb, 0x19, 0x85, 0x39, 0x4b, 0x44, 0x76, 0x53, 0xd1,
    0x44, 0x61, 0x29, 0x6f, 0x80, 0x4b, 0x01, 0xfd, 0x79, 0x42, 0x9b, 0x83,
    0x56, 0x4e, 0x97, 0x90, 0x8a, 0x3d, 0xb0, 0xf5, 0x4c, 0x5a, 0xf1, 0xa5,
    0x43, 0xc7, 0x13, 0x8e, 0x47, 0x8c, 0xa4, 0xd7, 0x95, 0xb0, 0x8c, 0xd1,
    0x2c, 0x77, 0x26, 0x00, 0xab, 0x0f, 0x78, 0xc5, 0x0c, 0xf2, 0x7c, 0x05,
    0xf5, 0x1e, 0x0a, 0xfb,
};
static const struct drbg_kat_no_reseed kat382_t = {
    6, kat382_entropyin, kat382_nonce, kat382_persstr,
    kat382_addin0, kat382_addin1, kat382_retbits
};
static const struct drbg_kat kat382 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat382_t
};

static const unsigned char kat383_entropyin[] = {
    0x6a, 0x58, 0x81, 0xe0, 0x9d, 0x36, 0xd4, 0x66, 0xa1, 0x66, 0x7f, 0x62,
    0x59, 0xea, 0xe9, 0x27, 0xe1, 0x81, 0x57, 0xd3, 0x31, 0x0b, 0xa1, 0xb6,
};
static const unsigned char kat383_nonce[] = {
    0xfd, 0x9a, 0x1c, 0x82, 0xd4, 0xed, 0x4c, 0xd7, 0xcc, 0x4a, 0x52, 0xeb,
    0xd1, 0x61, 0x6f, 0x90,
};
static const unsigned char kat383_persstr[] = {0};
static const unsigned char kat383_addin0[] = {
    0x82, 0x97, 0xd7, 0x02, 0x6e, 0x87, 0x61, 0x46, 0x6e, 0xf8, 0x8e, 0x28,
    0x23, 0xb2, 0x9f, 0xf8, 0xad, 0x1b, 0xf7, 0x4c, 0xbe, 0x6a, 0x55, 0x34,
    0x32, 0xd3, 0x59, 0xfc, 0xe9, 0x20, 0xcc, 0xb9,
};
static const unsigned char kat383_addin1[] = {
    0x3d, 0x4b, 0x7b, 0x99, 0x38, 0xd7, 0x71, 0x7f, 0xea, 0xa3, 0x0c, 0x7d,
    0x49, 0x07, 0xf3, 0x8a, 0xc2, 0x82, 0x70, 0x2c, 0xb5, 0x29, 0x2d, 0x96,
    0xe7, 0x23, 0x32, 0x80, 0x0a, 0x40, 0x86, 0x79,
};
static const unsigned char kat383_retbits[] = {
    0x97, 0xe5, 0xb0, 0xda, 0xb9, 0x74, 0x2c, 0xaa, 0xf5, 0x17, 0x0e, 0x17,
    0x23, 0xd5, 0xc3, 0x9f, 0xc7, 0x11, 0x8c, 0x72, 0xe1, 0xcb, 0x48, 0xe5,
    0x97, 0x46, 0x59, 0xf4, 0x99, 0x54, 0xef, 0xb7, 0x6f, 0x6b, 0x96, 0x62,
    0xb8, 0xb4, 0xcc, 0xfb, 0xf6, 0xe9, 0x4a, 0x79, 0x2a, 0xd1, 0x45, 0xa6,
    0x05, 0xb0, 0xc6, 0xc3, 0x43, 0x02, 0x6c, 0xba, 0x79, 0xf5, 0xcd, 0x10,
    0xbd, 0xc8, 0x66, 0x48,
};
static const struct drbg_kat_no_reseed kat383_t = {
    7, kat383_entropyin, kat383_nonce, kat383_persstr,
    kat383_addin0, kat383_addin1, kat383_retbits
};
static const struct drbg_kat kat383 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat383_t
};

static const unsigned char kat384_entropyin[] = {
    0x2c, 0xc4, 0x79, 0x70, 0x0f, 0x90, 0x9e, 0x7b, 0xa2, 0x67, 0x7f, 0x3e,
    0x90, 0x12, 0x81, 0x9d, 0xbd, 0xa2, 0xbf, 0xa9, 0x6b, 0xbf, 0x0e, 0x71,
};
static const unsigned char kat384_nonce[] = {
    0x36, 0x2b, 0xb1, 0x0b, 0x3e, 0x7f, 0x8c, 0x3f, 0xb3, 0xe4, 0x29, 0x6b,
    0x23, 0x71, 0xd4, 0x50,
};
static const unsigned char kat384_persstr[] = {0};
static const unsigned char kat384_addin0[] = {
    0x9d, 0xd4, 0x6b, 0xa2, 0x14, 0x56, 0x5b, 0xe1, 0xd6, 0xec, 0x71, 0xd5,
    0xfa, 0xa9, 0x2b, 0x45, 0x8b, 0x70, 0xe8, 0x99, 0x8a, 0x2e, 0x60, 0xab,
    0xf4, 0x28, 0xfb, 0x17, 0x96, 0xc5, 0x96, 0x51,
};
static const unsigned char kat384_addin1[] = {
    0xf4, 0x0e, 0x36, 0xed, 0x10, 0x05, 0xe8, 0xc4, 0x46, 0x8d, 0x9f, 0x93,
    0x4f, 0xbc, 0xf4, 0x6b, 0x34, 0x2e, 0x57, 0x45, 0x30, 0x9f, 0xa4, 0x00,
    0x4c, 0x15, 0x0b, 0xd4, 0xfb, 0x2b, 0x1e, 0x71,
};
static const unsigned char kat384_retbits[] = {
    0xcb, 0x69, 0xc3, 0x69, 0x35, 0x62, 0x0c, 0x7a, 0x23, 0x40, 0xd6, 0x72,
    0xb5, 0x59, 0xbe, 0x58, 0xcc, 0xaa, 0x82, 0xed, 0x45, 0x1a, 0x3f, 0x8e,
    0xa4, 0xfc, 0x06, 0xa9, 0x9c, 0x58, 0x9b, 0x96, 0x17, 0x67, 0x1d, 0xbf,
    0x26, 0x5a, 0xb9, 0xbf, 0xb9, 0xc5, 0x57, 0xdc, 0x6f, 0x6a, 0x70, 0x1b,
    0xa0, 0xcb, 0xc5, 0x03, 0x0e, 0x08, 0x5a, 0xf1, 0x6b, 0xc9, 0x73, 0x86,
    0x78, 0xcc, 0x31, 0xb4,
};
static const struct drbg_kat_no_reseed kat384_t = {
    8, kat384_entropyin, kat384_nonce, kat384_persstr,
    kat384_addin0, kat384_addin1, kat384_retbits
};
static const struct drbg_kat kat384 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat384_t
};

static const unsigned char kat385_entropyin[] = {
    0xcd, 0x60, 0xf9, 0xaa, 0x04, 0xb8, 0x32, 0xb6, 0xa7, 0x6a, 0xc4, 0x3c,
    0x52, 0x76, 0x09, 0x25, 0xfa, 0x8e, 0x9d, 0x99, 0x59, 0x2b, 0xdd, 0xef,
};
static const unsigned char kat385_nonce[] = {
    0x12, 0x94, 0x99, 0x7f, 0x49, 0x24, 0x8a, 0xdc, 0x70, 0x54, 0x0c, 0x18,
    0xe8, 0xed, 0xe4, 0x70,
};
static const unsigned char kat385_persstr[] = {0};
static const unsigned char kat385_addin0[] = {
    0xaf, 0x6c, 0xe3, 0x7c, 0x50, 0xc9, 0x27, 0x59, 0x37, 0x3d, 0x52, 0x06,
    0x62, 0x8f, 0x74, 0x7c, 0x16, 0x91, 0x53, 0xea, 0xbf, 0x80, 0x4d, 0x25,
    0x40, 0xc1, 0xc1, 0x77, 0x64, 0xae, 0x53, 0x8e,
};
static const unsigned char kat385_addin1[] = {
    0xfa, 0x2e, 0x5a, 0x76, 0x2d, 0xd5, 0x60, 0x96, 0x23, 0x0b, 0xa8, 0x34,
    0x43, 0xfd, 0x16, 0xc8, 0xbf, 0x6d, 0xa0, 0xf5, 0xaa, 0x07, 0xbe, 0x0e,
    0x52, 0x2d, 0x85, 0xb7, 0xfe, 0xc8, 0x8e, 0x96,
};
static const unsigned char kat385_retbits[] = {
    0xcb, 0x38, 0xaa, 0x9a, 0x8b, 0x89, 0x3e, 0xef, 0xb8, 0x81, 0x9b, 0x41,
    0x73, 0x47, 0x1b, 0x98, 0x3e, 0x64, 0xdd, 0x7f, 0xf6, 0x20, 0x92, 0xd4,
    0x0c, 0x22, 0x02, 0xa1, 0x2e, 0x07, 0x01, 0x36, 0x72, 0x84, 0x08, 0x04,
    0xd8, 0xf4, 0x9a, 0x17, 0x4a, 0xb0, 0x73, 0xb2, 0x93, 0xc6, 0x63, 0xd0,
    0x18, 0x2f, 0xe0, 0x03, 0x5f, 0x80, 0xe1, 0xd8, 0x82, 0x19, 0x8b, 0x4e,
    0x49, 0xfe, 0x70, 0x10,
};
static const struct drbg_kat_no_reseed kat385_t = {
    9, kat385_entropyin, kat385_nonce, kat385_persstr,
    kat385_addin0, kat385_addin1, kat385_retbits
};
static const struct drbg_kat kat385 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat385_t
};

static const unsigned char kat386_entropyin[] = {
    0x69, 0x31, 0xcd, 0x18, 0xa6, 0xc7, 0x72, 0xce, 0x1a, 0xd6, 0xb1, 0xff,
    0x80, 0x46, 0x11, 0x10, 0xb3, 0x14, 0xc9, 0xc1, 0xb2, 0x7d, 0x6e, 0x5d,
};
static const unsigned char kat386_nonce[] = {
    0x4f, 0x3c, 0x93, 0xd4, 0x04, 0xb1, 0xa2, 0x0b, 0x83, 0x5e, 0x50, 0x12,
    0xe1, 0x2a, 0x65, 0x4f,
};
static const unsigned char kat386_persstr[] = {0};
static const unsigned char kat386_addin0[] = {
    0x3c, 0x48, 0x21, 0x1e, 0x1f, 0x59, 0x62, 0xd1, 0xad, 0x49, 0xbb, 0x76,
    0x36, 0xab, 0x18, 0x2e, 0x8b, 0x84, 0x95, 0xe2, 0xff, 0xd3, 0x17, 0xfd,
    0x08, 0x52, 0x69, 0x50, 0x95, 0x08, 0xc4, 0x61,
};
static const unsigned char kat386_addin1[] = {
    0xaa, 0x0a, 0x70, 0x03, 0x1a, 0xf5, 0xfe, 0xdc, 0x2c, 0x00, 0x60, 0x5e,
    0x5b, 0x6e, 0x4b, 0x92, 0x59, 0xfd, 0x14, 0x21, 0x44, 0x63, 0x52, 0x03,
    0x4a, 0x4a, 0x4c, 0x80, 0x16, 0xe3, 0xe0, 0x16,
};
static const unsigned char kat386_retbits[] = {
    0xa4, 0x49, 0xf3, 0x78, 0x1a, 0x48, 0x84, 0x34, 0x92, 0x10, 0xc9, 0xd3,
    0xf6, 0x94, 0xf1, 0xa6, 0x38, 0x71, 0x60, 0x5f, 0xb0, 0x7a, 0xd3, 0x02,
    0xa2, 0xc2, 0xcd, 0x16, 0xd7, 0xc6, 0x44, 0xea, 0x7e, 0xb5, 0xfc, 0xbc,
    0x27, 0xb1, 0x2a, 0xd6, 0xa4, 0xb7, 0x5c, 0x84, 0xe2, 0xba, 0xdd, 0x4f,
    0x0a, 0x06, 0x62, 0x21, 0x64, 0xbc, 0xe3, 0xed, 0xc4, 0xbe, 0x8a, 0x35,
    0xba, 0x6c, 0x19, 0x7b,
};
static const struct drbg_kat_no_reseed kat386_t = {
    10, kat386_entropyin, kat386_nonce, kat386_persstr,
    kat386_addin0, kat386_addin1, kat386_retbits
};
static const struct drbg_kat kat386 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat386_t
};

static const unsigned char kat387_entropyin[] = {
    0x59, 0x29, 0xc6, 0xc6, 0xc5, 0x63, 0xdc, 0x6b, 0x85, 0x4c, 0xd7, 0x1c,
    0x3a, 0x2e, 0x9c, 0x8f, 0x36, 0x88, 0x6b, 0xdf, 0x24, 0x46, 0xb1, 0x1f,
};
static const unsigned char kat387_nonce[] = {
    0x36, 0x6e, 0x9c, 0xc6, 0x90, 0x07, 0x44, 0xa7, 0xfd, 0x31, 0xed, 0x92,
    0xf5, 0x78, 0x5d, 0x2b,
};
static const unsigned char kat387_persstr[] = {0};
static const unsigned char kat387_addin0[] = {
    0xdc, 0xc1, 0x53, 0x2d, 0x3d, 0x93, 0x48, 0x80, 0x7c, 0x19, 0x17, 0xa4,
    0x51, 0x39, 0xd7, 0x34, 0x5d, 0x15, 0xa8, 0xd9, 0xc7, 0xa6, 0xa1, 0x8c,
    0x82, 0x69, 0x45, 0xe8, 0x2b, 0x0b, 0x8a, 0x5e,
};
static const unsigned char kat387_addin1[] = {
    0x40, 0x22, 0xbc, 0x66, 0xf1, 0x81, 0x60, 0xeb, 0x8a, 0x87, 0x23, 0x76,
    0xba, 0x99, 0x8b, 0xbc, 0x40, 0x00, 0x31, 0xb7, 0xb9, 0x05, 0x03, 0x1a,
    0x38, 0x47, 0xd4, 0xec, 0x0d, 0xf9, 0x57, 0xcf,
};
static const unsigned char kat387_retbits[] = {
    0x36, 0x2f, 0xf3, 0xdf, 0x4d, 0x56, 0x3f, 0xca, 0xb2, 0x73, 0x0b, 0xb1,
    0x93, 0x2f, 0x27, 0x91, 0x99, 0x1e, 0x50, 0xe2, 0x46, 0x82, 0x1a, 0x74,
    0x68, 0xa2, 0xeb, 0x1a, 0xcf, 0x55, 0xd3, 0x30, 0xcd, 0x13, 0xba, 0x99,
    0x9c, 0x58, 0x52, 0xe8, 0x1d, 0x58, 0xc3, 0xc8, 0xe6, 0x38, 0x48, 0xbf,
    0xa6, 0xb2, 0x43, 0x3d, 0xe4, 0x53, 0xa0, 0xda, 0x6c, 0x79, 0x97, 0xa6,
    0x00, 0x0c, 0x9f, 0xc3,
};
static const struct drbg_kat_no_reseed kat387_t = {
    11, kat387_entropyin, kat387_nonce, kat387_persstr,
    kat387_addin0, kat387_addin1, kat387_retbits
};
static const struct drbg_kat kat387 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat387_t
};

static const unsigned char kat388_entropyin[] = {
    0x61, 0xae, 0x1d, 0x52, 0x00, 0x54, 0x09, 0x68, 0x42, 0xc1, 0x7a, 0xb7,
    0x0c, 0x60, 0xa2, 0x78, 0x48, 0x0a, 0xae, 0xd2, 0x99, 0x57, 0x5e, 0x59,
};
static const unsigned char kat388_nonce[] = {
    0x9a, 0xbc, 0xbd, 0xbe, 0x26, 0x0a, 0x64, 0x19, 0xc7, 0x87, 0x70, 0x61,
    0x5d, 0xa5, 0x4c, 0x03,
};
static const unsigned char kat388_persstr[] = {0};
static const unsigned char kat388_addin0[] = {
    0x0e, 0x5b, 0x2a, 0x49, 0xf3, 0x44, 0x51, 0x99, 0xcb, 0xbe, 0xca, 0x36,
    0x3b, 0x02, 0x3f, 0x4a, 0x29, 0x54, 0xc3, 0xe5, 0x67, 0xcb, 0xfc, 0x33,
    0xf1, 0xb2, 0x9f, 0xbc, 0x74, 0x20, 0xb1, 0x89,
};
static const unsigned char kat388_addin1[] = {
    0xa2, 0xff, 0xe5, 0x63, 0xdd, 0x17, 0x50, 0xbb, 0x3e, 0x07, 0xf2, 0xc4,
    0x98, 0xef, 0x18, 0xa6, 0xc4, 0xf2, 0x24, 0x53, 0xd9, 0xab, 0xc9, 0x93,
    0x55, 0x50, 0x9c, 0x42, 0xed, 0x01, 0x19, 0x60,
};
static const unsigned char kat388_retbits[] = {
    0x7b, 0xec, 0x6d, 0xaa, 0x15, 0xef, 0x65, 0x81, 0x72, 0x49, 0x04, 0x4d,
    0x37, 0x5f, 0xbd, 0x98, 0x57, 0x3a, 0xcd, 0xac, 0xb2, 0x11, 0x46, 0x78,
    0xd4, 0x11, 0x58, 0x9e, 0x44, 0xb7, 0x1c, 0xcc, 0xdf, 0xa7, 0xbd, 0x50,
    0x69, 0x5a, 0xdd, 0x94, 0x7b, 0x0e, 0xbb, 0x58, 0xb7, 0x38, 0x38, 0x65,
    0x37, 0xb6, 0x34, 0x40, 0xee, 0x84, 0xb7, 0x7f, 0x37, 0xe0, 0xc4, 0x58,
    0x77, 0x43, 0x0c, 0x70,
};
static const struct drbg_kat_no_reseed kat388_t = {
    12, kat388_entropyin, kat388_nonce, kat388_persstr,
    kat388_addin0, kat388_addin1, kat388_retbits
};
static const struct drbg_kat kat388 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat388_t
};

static const unsigned char kat389_entropyin[] = {
    0xb6, 0x73, 0x5d, 0x76, 0x0d, 0x4d, 0xca, 0xa5, 0x7d, 0xae, 0xfc, 0x5b,
    0xdb, 0xf0, 0xb5, 0xa9, 0x4f, 0x2e, 0x6b, 0xc4, 0xa3, 0x60, 0x80, 0x4b,
};
static const unsigned char kat389_nonce[] = {
    0xd0, 0xa8, 0xbc, 0xef, 0xaa, 0x6d, 0x9f, 0x4b, 0x6a, 0xfd, 0x4c, 0x8a,
    0xd4, 0x27, 0x6e, 0xf0,
};
static const unsigned char kat389_persstr[] = {0};
static const unsigned char kat389_addin0[] = {
    0x75, 0x61, 0x9e, 0xe0, 0xb4, 0xf2, 0x60, 0x1d, 0x5f, 0x1c, 0xb0, 0x83,
    0x3f, 0x32, 0x5e, 0x3a, 0x7b, 0x59, 0x21, 0x48, 0x17, 0x52, 0xe1, 0x22,
    0xc8, 0x78, 0x6d, 0xdc, 0x33, 0xa3, 0x8a, 0x27,
};
static const unsigned char kat389_addin1[] = {
    0x13, 0x60, 0xb2, 0x00, 0x07, 0x80, 0x37, 0x6f, 0x0a, 0xed, 0x1a, 0xef,
    0xfa, 0x3c, 0x26, 0xe6, 0xc2, 0x9d, 0x30, 0xb0, 0x74, 0x0d, 0xe1, 0x4b,
    0xbf, 0xe9, 0x31, 0xe5, 0xf1, 0xaa, 0xa9, 0x68,
};
static const unsigned char kat389_retbits[] = {
    0xae, 0x3b, 0xb8, 0x2b, 0xfb, 0xd1, 0xf4, 0x8c, 0x33, 0xe1, 0xeb, 0x0b,
    0x96, 0xca, 0x85, 0x0c, 0x35, 0x05, 0xa7, 0x5b, 0xf3, 0xfb, 0x07, 0xbb,
    0xad, 0xe7, 0xa2, 0xc1, 0x7b, 0xd1, 0xb0, 0x4d, 0x8a, 0xf5, 0xed, 0x1a,
    0x5c, 0x71, 0x56, 0x6c, 0xa2, 0xc0, 0x51, 0x35, 0xc4, 0x12, 0x8a, 0x0e,
    0x6e, 0x8e, 0x9a, 0x27, 0xb3, 0x7a, 0x76, 0xc9, 0xf7, 0x08, 0xdb, 0x02,
    0x7f, 0x94, 0x80, 0x1c,
};
static const struct drbg_kat_no_reseed kat389_t = {
    13, kat389_entropyin, kat389_nonce, kat389_persstr,
    kat389_addin0, kat389_addin1, kat389_retbits
};
static const struct drbg_kat kat389 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat389_t
};

static const unsigned char kat390_entropyin[] = {
    0xbb, 0x90, 0xe5, 0x3c, 0x12, 0x40, 0xd6, 0x58, 0x40, 0x6d, 0x7c, 0x85,
    0xc0, 0xff, 0x23, 0xd7, 0x47, 0x4a, 0x67, 0x6a, 0x59, 0x20, 0xb0, 0xe1,
};
static const unsigned char kat390_nonce[] = {
    0x48, 0x02, 0xe0, 0x2a, 0xdc, 0xc5, 0xa1, 0x50, 0x4d, 0x6d, 0xde, 0x82,
    0xb6, 0xf2, 0x0a, 0x67,
};
static const unsigned char kat390_persstr[] = {0};
static const unsigned char kat390_addin0[] = {
    0xfb, 0x2b, 0x97, 0x6d, 0x69, 0x4d, 0xf3, 0xfc, 0x2b, 0x32, 0xa3, 0x3f,
    0x7c, 0x98, 0x96, 0x83, 0xf5, 0x0c, 0x4c, 0xfc, 0xe6, 0x16, 0x8a, 0xd1,
    0xab, 0x9d, 0x1a, 0x79, 0xd5, 0xf5, 0x9d, 0x3c,
};
static const unsigned char kat390_addin1[] = {
    0x6c, 0x78, 0x61, 0x0d, 0xaf, 0x6e, 0xa4, 0x26, 0x64, 0x1b, 0xe5, 0x62,
    0x80, 0xcd, 0x62, 0x3e, 0x8b, 0x11, 0xdd, 0x96, 0x7a, 0xc6, 0x15, 0x4f,
    0x04, 0xa6, 0xd4, 0xa6, 0xc7, 0x49, 0xc9, 0x57,
};
static const unsigned char kat390_retbits[] = {
    0x97, 0xe3, 0x56, 0x01, 0x57, 0xab, 0x43, 0x42, 0xe7, 0x62, 0x48, 0x07,
    0x27, 0xd5, 0x50, 0x5a, 0x72, 0x29, 0x3e, 0x13, 0x53, 0x1c, 0x62, 0xea,
    0x70, 0x72, 0xb2, 0xac, 0xf8, 0xf7, 0x2e, 0xa8, 0xce, 0xd0, 0x9c, 0xbf,
    0x9e, 0xcb, 0x9e, 0x9d, 0x6d, 0xa6, 0x16, 0xfb, 0x8c, 0xa0, 0x61, 0x38,
    0x09, 0x38, 0x8e, 0xe9, 0x15, 0xaa, 0xcb, 0x59, 0xcb, 0x36, 0x27, 0xe3,
    0xf0, 0xc3, 0x89, 0xdf,
};
static const struct drbg_kat_no_reseed kat390_t = {
    14, kat390_entropyin, kat390_nonce, kat390_persstr,
    kat390_addin0, kat390_addin1, kat390_retbits
};
static const struct drbg_kat kat390 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat390_t
};

static const unsigned char kat391_entropyin[] = {
    0xa7, 0x99, 0x43, 0x34, 0x23, 0x77, 0xca, 0x01, 0x85, 0x59, 0xd0, 0x88,
    0x6d, 0x43, 0xdf, 0xe0, 0x18, 0xd6, 0x30, 0x59, 0x0d, 0xb1, 0x02, 0x3b,
};
static const unsigned char kat391_nonce[] = {
    0x1b, 0x96, 0xf2, 0x2b, 0xd6, 0x61, 0x79, 0xf5, 0x93, 0x80, 0x9d, 0xb9,
    0x0f, 0xdf, 0xa6, 0x14,
};
static const unsigned char kat391_persstr[] = {
    0x5b, 0xed, 0xfc, 0x44, 0xb3, 0x72, 0x88, 0x44, 0x52, 0x36, 0x72, 0x29,
    0xf1, 0xf6, 0x7e, 0x93, 0xbc, 0x44, 0x7f, 0x8f, 0xbe, 0xe0, 0x44, 0xc3,
    0x1e, 0x10, 0x96, 0x7e, 0xf0, 0x12, 0x0c, 0x6d,
};
static const unsigned char kat391_addin0[] = {0};
static const unsigned char kat391_addin1[] = {0};
static const unsigned char kat391_retbits[] = {
    0x49, 0xeb, 0xb3, 0x6a, 0xfd, 0x56, 0x3e, 0x07, 0x72, 0xed, 0x7d, 0xc8,
    0xec, 0xbb, 0xee, 0xba, 0x47, 0xcc, 0xba, 0xbf, 0xd0, 0xbe, 0xeb, 0xb5,
    0xc9, 0x9c, 0x97, 0x71, 0xe2, 0xdf, 0x9a, 0x33, 0xe1, 0x9c, 0x4b, 0xb7,
    0x16, 0x44, 0x9e, 0xb5, 0xe9, 0xb6, 0x67, 0x39, 0xbd, 0xca, 0x2e, 0xe8,
    0xca, 0x21, 0x7e, 0xcc, 0x1c, 0x4f, 0xf6, 0xc0, 0x34, 0xcd, 0xfc, 0xad,
    0xb8, 0xf1, 0xc7, 0xda,
};
static const struct drbg_kat_no_reseed kat391_t = {
    0, kat391_entropyin, kat391_nonce, kat391_persstr,
    kat391_addin0, kat391_addin1, kat391_retbits
};
static const struct drbg_kat kat391 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat391_t
};

static const unsigned char kat392_entropyin[] = {
    0xc3, 0x7b, 0xcf, 0xc7, 0x99, 0xb1, 0xa7, 0x51, 0xca, 0x6f, 0xc1, 0xfd,
    0xec, 0x29, 0x2a, 0xe8, 0xcd, 0x3a, 0x59, 0xb6, 0x15, 0x82, 0x8c, 0x89,
};
static const unsigned char kat392_nonce[] = {
    0x26, 0xeb, 0x7b, 0xb6, 0x85, 0xe4, 0xe9, 0x12, 0xe4, 0x74, 0x11, 0x5b,
    0x19, 0xfa, 0xb4, 0xa2,
};
static const unsigned char kat392_persstr[] = {
    0x02, 0x7c, 0xa3, 0x5a, 0x2a, 0xd5, 0x2c, 0x9e, 0xb8, 0x73, 0x07, 0xac,
    0x2b, 0x4d, 0xd0, 0x45, 0x9a, 0xb5, 0xfe, 0xf8, 0x74, 0xcd, 0x25, 0x34,
    0x27, 0x52, 0x88, 0x8c, 0x1d, 0xba, 0x63, 0x21,
};
static const unsigned char kat392_addin0[] = {0};
static const unsigned char kat392_addin1[] = {0};
static const unsigned char kat392_retbits[] = {
    0x0d, 0xb0, 0xad, 0x15, 0xbe, 0xe8, 0x9b, 0x26, 0xff, 0x32, 0xad, 0x9b,
    0x3e, 0x5e, 0xea, 0x25, 0xb3, 0x02, 0x6a, 0x0e, 0x76, 0xe8, 0x5d, 0x57,
    0x6a, 0x17, 0x40, 0x0e, 0x00, 0xe3, 0x8b, 0x82, 0x88, 0x3d, 0xd0, 0x1d,
    0x43, 0x03, 0x8c, 0xe6, 0xb5, 0x64, 0x5d, 0xe5, 0x8f, 0x3d, 0x08, 0xa2,
    0x38, 0xb8, 0x58, 0x97, 0x8d, 0xe4, 0x09, 0x06, 0xa8, 0x89, 0x08, 0x92,
    0x5a, 0x91, 0x82, 0x0d,
};
static const struct drbg_kat_no_reseed kat392_t = {
    1, kat392_entropyin, kat392_nonce, kat392_persstr,
    kat392_addin0, kat392_addin1, kat392_retbits
};
static const struct drbg_kat kat392 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat392_t
};

static const unsigned char kat393_entropyin[] = {
    0x5f, 0xd9, 0xfd, 0xe9, 0x5f, 0x75, 0x96, 0x85, 0xda, 0x08, 0xaf, 0x67,
    0xd9, 0xd8, 0x4a, 0xa0, 0xd3, 0x32, 0x23, 0x5a, 0x0d, 0x32, 0x45, 0xe7,
};
static const unsigned char kat393_nonce[] = {
    0x1d, 0x30, 0xeb, 0x84, 0xed, 0xd4, 0x48, 0x7c, 0xa3, 0xbe, 0x0f, 0x08,
    0xf8, 0x1c, 0x8a, 0xfc,
};
static const unsigned char kat393_persstr[] = {
    0x2a, 0xe9, 0x1a, 0xd3, 0xbb, 0xae, 0x3f, 0x5f, 0x80, 0xe1, 0x80, 0xb7,
    0xe9, 0x4a, 0xe5, 0x01, 0x90, 0x1c, 0x01, 0x5f, 0xb1, 0x2f, 0x50, 0xd8,
    0xac, 0x8f, 0x2a, 0xc2, 0x15, 0x0a, 0xa2, 0xb6,
};
static const unsigned char kat393_addin0[] = {0};
static const unsigned char kat393_addin1[] = {0};
static const unsigned char kat393_retbits[] = {
    0x2a, 0xa5, 0xb7, 0xef, 0x58, 0x74, 0x59, 0x46, 0xec, 0xcf, 0xaf, 0x9c,
    0xe1, 0xe2, 0x8c, 0x03, 0x6a, 0xd4, 0xe9, 0x9e, 0xd5, 0x05, 0x1b, 0x51,
    0xc5, 0x74, 0x66, 0xea, 0xb0, 0xad, 0x08, 0x31, 0xcf, 0xa0, 0x14, 0x42,
    0x6c, 0x2c, 0x3d, 0x4b, 0xe7, 0x0f, 0x0e, 0xc7, 0x7e, 0x02, 0x43, 0x40,
    0x9d, 0xac, 0xb8, 0x7e, 0xd3, 0x40, 0xaf, 0x0b, 0x8e, 0x39, 0x45, 0x61,
    0x20, 0x64, 0x0e, 0xb9,
};
static const struct drbg_kat_no_reseed kat393_t = {
    2, kat393_entropyin, kat393_nonce, kat393_persstr,
    kat393_addin0, kat393_addin1, kat393_retbits
};
static const struct drbg_kat kat393 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat393_t
};

static const unsigned char kat394_entropyin[] = {
    0xb0, 0x6b, 0xb1, 0x4c, 0xce, 0x0d, 0xb3, 0xf4, 0x1a, 0xf3, 0xe7, 0x79,
    0x10, 0x36, 0x1e, 0x77, 0x85, 0x42, 0x7c, 0x83, 0x02, 0xdd, 0xbd, 0xf4,
};
static const unsigned char kat394_nonce[] = {
    0xf3, 0x9d, 0x39, 0xc3, 0x05, 0xd1, 0x72, 0x7f, 0x7d, 0xcb, 0x62, 0x94,
    0xca, 0xfc, 0x30, 0x45,
};
static const unsigned char kat394_persstr[] = {
    0x06, 0x76, 0xb6, 0x8f, 0x21, 0xa6, 0xc2, 0x89, 0x52, 0x11, 0xce, 0x12,
    0x7a, 0x57, 0x48, 0xa0, 0x05, 0x0e, 0xeb, 0xec, 0x08, 0x04, 0x4e, 0xe2,
    0xfe, 0x26, 0x93, 0xbe, 0x05, 0xe2, 0x23, 0xe8,
};
static const unsigned char kat394_addin0[] = {0};
static const unsigned char kat394_addin1[] = {0};
static const unsigned char kat394_retbits[] = {
    0xbd, 0xe9, 0x73, 0x5f, 0x36, 0xb8, 0xd8, 0xaf, 0xe8, 0xac, 0x76, 0xa9,
    0x71, 0xec, 0x94, 0x66, 0x06, 0x5d, 0x22, 0xe1, 0x88, 0x5e, 0x75, 0xdb,
    0x61, 0xca, 0x11, 0x66, 0x28, 0x10, 0x6b, 0xeb, 0x15, 0xdd, 0xe7, 0xe5,
    0xe1, 0xc4, 0x86, 0xe4, 0xcc, 0x66, 0x05, 0x6e, 0xe1, 0x57, 0x0c, 0xd5,
    0x34, 0x9b, 0x6c, 0x62, 0x80, 0x94, 0xc9, 0x50, 0x67, 0x52, 0xde, 0x9f,
    0x3d, 0xa9, 0x5f, 0x8a,
};
static const struct drbg_kat_no_reseed kat394_t = {
    3, kat394_entropyin, kat394_nonce, kat394_persstr,
    kat394_addin0, kat394_addin1, kat394_retbits
};
static const struct drbg_kat kat394 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat394_t
};

static const unsigned char kat395_entropyin[] = {
    0xc2, 0x10, 0x9a, 0x53, 0x44, 0x62, 0x8f, 0xfa, 0x5c, 0x43, 0x46, 0x0e,
    0x9e, 0x38, 0x5f, 0x4e, 0xa4, 0xc0, 0x7a, 0xa4, 0x01, 0xec, 0x98, 0x1e,
};
static const unsigned char kat395_nonce[] = {
    0xd2, 0x42, 0xbb, 0xb3, 0xba, 0x28, 0x2b, 0x9b, 0xb0, 0xd3, 0x64, 0x8a,
    0xee, 0x33, 0xf6, 0x99,
};
static const unsigned char kat395_persstr[] = {
    0x00, 0x33, 0x3c, 0xfe, 0x95, 0xaf, 0x5e, 0xcb, 0x99, 0xf9, 0xab, 0x69,
    0x86, 0x76, 0x2a, 0xd2, 0xd2, 0xf7, 0x70, 0x2a, 0xa5, 0xec, 0x5c, 0xef,
    0xc4, 0x31, 0x77, 0xee, 0xb3, 0xb3, 0x3d, 0xdb,
};
static const unsigned char kat395_addin0[] = {0};
static const unsigned char kat395_addin1[] = {0};
static const unsigned char kat395_retbits[] = {
    0xf8, 0xc4, 0xed, 0x2c, 0x6d, 0x7f, 0x32, 0x38, 0x28, 0xc5, 0xe9, 0xe5,
    0x0c, 0x86, 0x31, 0xe1, 0x35, 0xd1, 0xca, 0x04, 0x17, 0x3e, 0xaf, 0x15,
    0x03, 0x6a, 0xf8, 0x25, 0x0e, 0xf0, 0x9b, 0xdd, 0x59, 0x5d, 0x03, 0xa7,
    0x76, 0xae, 0x06, 0x6d, 0xa4, 0x62, 0xc6, 0x31, 0xfc, 0x6a, 0x66, 0xea,
    0xb3, 0x75, 0xfb, 0x0e, 0x2e, 0xde, 0x9a, 0x3b, 0x50, 0x83, 0x98, 0x0f,
    0x4f, 0xf8, 0x10, 0xc9,
};
static const struct drbg_kat_no_reseed kat395_t = {
    4, kat395_entropyin, kat395_nonce, kat395_persstr,
    kat395_addin0, kat395_addin1, kat395_retbits
};
static const struct drbg_kat kat395 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat395_t
};

static const unsigned char kat396_entropyin[] = {
    0xb7, 0xa9, 0x0d, 0x58, 0xf3, 0x65, 0x08, 0x14, 0xad, 0x99, 0xd3, 0x08,
    0xb2, 0x3c, 0x79, 0xd9, 0x36, 0x57, 0xc7, 0x3d, 0x2d, 0xed, 0x54, 0x12,
};
static const unsigned char kat396_nonce[] = {
    0x8a, 0xe9, 0x69, 0x38, 0xdf, 0xfc, 0xcc, 0xef, 0xc8, 0xd5, 0xf9, 0x34,
    0x75, 0x3f, 0xd6, 0x70,
};
static const unsigned char kat396_persstr[] = {
    0xe3, 0x19, 0xf4, 0x51, 0x11, 0xce, 0xa3, 0x52, 0x90, 0xb0, 0x77, 0x30,
    0x2b, 0xd3, 0xb5, 0xd4, 0x5b, 0x0f, 0x7c, 0x5e, 0x9e, 0x1d, 0x97, 0x29,
    0xdd, 0x2f, 0xcc, 0xc7, 0x58, 0x85, 0x5c, 0x3a,
};
static const unsigned char kat396_addin0[] = {0};
static const unsigned char kat396_addin1[] = {0};
static const unsigned char kat396_retbits[] = {
    0x7b, 0xb1, 0x31, 0x47, 0x9e, 0x60, 0x64, 0xc7, 0x76, 0x7e, 0x17, 0xaf,
    0x46, 0xd5, 0x28, 0xe4, 0xb6, 0x59, 0xc7, 0xfa, 0xbf, 0x90, 0x6c, 0x0a,
    0x76, 0x79, 0xdd, 0x8a, 0xfd, 0xd4, 0xaa, 0xf1, 0x04, 0xf8, 0xe7, 0xca,
    0xab, 0x91, 0x53, 0x65, 0x2a, 0x30, 0xe3, 0x8f, 0x80, 0xda, 0xe1, 0x0e,
    0x67, 0xca, 0xf3, 0x02, 0xbf, 0x11, 0x71, 0x04, 0x94, 0x96, 0x49, 0xc7,
    0xb2, 0x0f, 0x5d, 0x38,
};
static const struct drbg_kat_no_reseed kat396_t = {
    5, kat396_entropyin, kat396_nonce, kat396_persstr,
    kat396_addin0, kat396_addin1, kat396_retbits
};
static const struct drbg_kat kat396 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat396_t
};

static const unsigned char kat397_entropyin[] = {
    0x76, 0x18, 0xe5, 0x54, 0x6a, 0x59, 0x81, 0xcb, 0x4c, 0x72, 0xa5, 0x17,
    0xf2, 0x9a, 0x5f, 0xe8, 0x88, 0x6d, 0x4d, 0x3e, 0x9a, 0xd9, 0xfc, 0xb0,
};
static const unsigned char kat397_nonce[] = {
    0xd4, 0x75, 0x70, 0x4c, 0x10, 0xc4, 0xa1, 0x86, 0x6b, 0xca, 0xa4, 0x12,
    0xe6, 0xbf, 0xbe, 0x7d,
};
static const unsigned char kat397_persstr[] = {
    0xae, 0x8c, 0x3c, 0xde, 0x39, 0xed, 0x90, 0xda, 0x31, 0xb7, 0x32, 0x0c,
    0xf3, 0x59, 0x69, 0xfd, 0xe8, 0xf8, 0x28, 0x94, 0xe2, 0x22, 0xec, 0x2e,
    0x8a, 0xa1, 0x5d, 0x67, 0xb5, 0x5e, 0xfd, 0x89,
};
static const unsigned char kat397_addin0[] = {0};
static const unsigned char kat397_addin1[] = {0};
static const unsigned char kat397_retbits[] = {
    0x75, 0x30, 0x48, 0xb2, 0x87, 0x9a, 0xbd, 0x1a, 0x0c, 0xf2, 0x42, 0x6f,
    0x1e, 0x1b, 0x04, 0x1f, 0xcb, 0x6b, 0xa3, 0xf0, 0x71, 0x0e, 0xe8, 0xec,
    0x41, 0x0d, 0x79, 0x2f, 0xb3, 0x2b, 0xd6, 0x6b, 0x1b, 0xac, 0x95, 0xaf,
    0x81, 0xa3, 0x09, 0x01, 0xf7, 0xbd, 0xd4, 0x2e, 0x5f, 0x8f, 0xe6, 0xcd,
    0x28, 0x62, 0x3f, 0xde, 0xe9, 0xc5, 0x89, 0xbc, 0x47, 0xde, 0x1c, 0x6a,
    0x29, 0x0b, 0xec, 0x62,
};
static const struct drbg_kat_no_reseed kat397_t = {
    6, kat397_entropyin, kat397_nonce, kat397_persstr,
    kat397_addin0, kat397_addin1, kat397_retbits
};
static const struct drbg_kat kat397 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat397_t
};

static const unsigned char kat398_entropyin[] = {
    0x1d, 0x38, 0xd2, 0x43, 0x0e, 0x12, 0x93, 0x2d, 0xa3, 0xc3, 0x8f, 0x44,
    0x93, 0x0c, 0x1a, 0x2f, 0xf4, 0xa6, 0x2e, 0xaf, 0xaf, 0x8c, 0x85, 0x6f,
};
static const unsigned char kat398_nonce[] = {
    0x18, 0x85, 0xf3, 0xbd, 0x33, 0x9f, 0x2c, 0x96, 0x59, 0x5c, 0x3b, 0x6d,
    0x99, 0x8e, 0x50, 0x83,
};
static const unsigned char kat398_persstr[] = {
    0x43, 0x0c, 0x78, 0x71, 0x90, 0x1e, 0x16, 0x56, 0xe0, 0x8f, 0xb3, 0x5e,
    0xb4, 0xec, 0x43, 0x98, 0x49, 0x37, 0xa7, 0x5a, 0xf5, 0x96, 0xc9, 0x45,
    0x86, 0xa8, 0xf0, 0x2a, 0xf1, 0x2b, 0x7f, 0xc3,
};
static const unsigned char kat398_addin0[] = {0};
static const unsigned char kat398_addin1[] = {0};
static const unsigned char kat398_retbits[] = {
    0x9e, 0xc5, 0x99, 0x81, 0x42, 0x37, 0x9d, 0x14, 0x89, 0x64, 0x51, 0x02,
    0x4f, 0xec, 0x60, 0x85, 0x34, 0x18, 0xa3, 0xef, 0x59, 0xd3, 0xa6, 0xbd,
    0x3f, 0xf1, 0x59, 0x28, 0x26, 0xb1, 0x13, 0x72, 0xf3, 0x21, 0xa9, 0x8c,
    0x54, 0xd1, 0xbf, 0x82, 0x25, 0xbb, 0x64, 0xec, 0x2f, 0x2f, 0x4f, 0x46,
    0xed, 0x64, 0x2a, 0xea, 0xf2, 0x8e, 0xb2, 0x0b, 0xc9, 0x93, 0xe6, 0xd6,
    0x84, 0xfa, 0x33, 0xde,
};
static const struct drbg_kat_no_reseed kat398_t = {
    7, kat398_entropyin, kat398_nonce, kat398_persstr,
    kat398_addin0, kat398_addin1, kat398_retbits
};
static const struct drbg_kat kat398 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat398_t
};

static const unsigned char kat399_entropyin[] = {
    0xab, 0xb6, 0xb7, 0x68, 0x06, 0x81, 0x3c, 0x22, 0x73, 0xaa, 0x8e, 0xd4,
    0x76, 0xaf, 0x00, 0x74, 0xa7, 0xa4, 0x73, 0xea, 0x54, 0x8c, 0xc1, 0xef,
};
static const unsigned char kat399_nonce[] = {
    0xd9, 0x91, 0x77, 0x34, 0x3e, 0xa3, 0x83, 0x7f, 0x92, 0xd9, 0xc7, 0x36,
    0x7a, 0x70, 0x43, 0x3e,
};
static const unsigned char kat399_persstr[] = {
    0xef, 0x6b, 0x87, 0x66, 0x1f, 0x2a, 0x67, 0x8a, 0xd2, 0xd4, 0xdc, 0x85,
    0xba, 0x08, 0x45, 0xec, 0x8e, 0xea, 0xfa, 0x2b, 0x82, 0xb8, 0xae, 0xda,
    0x4f, 0xc6, 0x19, 0x38, 0xa8, 0xdd, 0x60, 0x77,
};
static const unsigned char kat399_addin0[] = {0};
static const unsigned char kat399_addin1[] = {0};
static const unsigned char kat399_retbits[] = {
    0x81, 0x1a, 0xc8, 0x8d, 0x4c, 0x96, 0x15, 0xb5, 0x35, 0xb0, 0xf7, 0x2a,
    0x9f, 0x49, 0x10, 0x7b, 0xac, 0xa2, 0x39, 0xa1, 0x3f, 0x26, 0xb9, 0x7b,
    0xa7, 0x33, 0xe4, 0xaf, 0x3a, 0x24, 0xdb, 0x8a, 0x65, 0x82, 0x75, 0xee,
    0x24, 0x16, 0xc0, 0x2f, 0x59, 0x68, 0xc2, 0x2d, 0x13, 0xaa, 0x62, 0x55,
    0xde, 0xb0, 0x98, 0x14, 0x51, 0x05, 0x22, 0x6f, 0x4a, 0xa8, 0x0a, 0xe3,
    0xdf, 0x8c, 0x1b, 0xfd,
};
static const struct drbg_kat_no_reseed kat399_t = {
    8, kat399_entropyin, kat399_nonce, kat399_persstr,
    kat399_addin0, kat399_addin1, kat399_retbits
};
static const struct drbg_kat kat399 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat399_t
};

static const unsigned char kat400_entropyin[] = {
    0xde, 0xfc, 0x7c, 0x59, 0xd7, 0x8e, 0x50, 0xc4, 0x12, 0x32, 0xf7, 0xbe,
    0xbf, 0xc5, 0xe9, 0x40, 0x8c, 0xea, 0x26, 0x6f, 0xc1, 0xe3, 0xc5, 0xc8,
};
static const unsigned char kat400_nonce[] = {
    0x9f, 0xfa, 0x0d, 0xf6, 0x97, 0x93, 0xbb, 0x6d, 0x9e, 0x7d, 0x78, 0xeb,
    0xaf, 0x46, 0xf6, 0xa7,
};
static const unsigned char kat400_persstr[] = {
    0x98, 0xb3, 0xfa, 0xb2, 0xff, 0xe5, 0x47, 0xec, 0x1a, 0x41, 0x8e, 0xd9,
    0x8a, 0xe7, 0xac, 0x66, 0x9c, 0x07, 0x01, 0x17, 0x8b, 0xe6, 0x55, 0xa6,
    0xfa, 0xf9, 0x96, 0x2d, 0x80, 0xae, 0xe9, 0x51,
};
static const unsigned char kat400_addin0[] = {0};
static const unsigned char kat400_addin1[] = {0};
static const unsigned char kat400_retbits[] = {
    0xa9, 0xe6, 0x82, 0x0b, 0xf8, 0x3a, 0x8b, 0x29, 0x41, 0x46, 0xfc, 0xe2,
    0xd1, 0x94, 0x01, 0x9a, 0x99, 0x59, 0xcc, 0x58, 0x47, 0x29, 0xd6, 0x7d,
    0xc7, 0x17, 0x42, 0x39, 0x10, 0x60, 0xec, 0x6b, 0x24, 0xd6, 0x85, 0xe5,
    0x45, 0x80, 0x7b, 0x1c, 0xa9, 0xd8, 0xa2, 0x8b, 0x68, 0xe7, 0x6a, 0xf2,
    0x56, 0xd4, 0x35, 0xf6, 0xf8, 0x31, 0x63, 0x6c, 0x33, 0x2a, 0x7f, 0x13,
    0x80, 0x7b, 0x1f, 0x35,
};
static const struct drbg_kat_no_reseed kat400_t = {
    9, kat400_entropyin, kat400_nonce, kat400_persstr,
    kat400_addin0, kat400_addin1, kat400_retbits
};
static const struct drbg_kat kat400 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat400_t
};

static const unsigned char kat401_entropyin[] = {
    0x6c, 0xfa, 0xaf, 0xb3, 0x1b, 0x1d, 0x6a, 0x6b, 0x9b, 0xed, 0x1d, 0xe9,
    0xa9, 0x93, 0xb7, 0x0f, 0xa1, 0x4d, 0xc6, 0x1f, 0xa1, 0x7a, 0xb1, 0xd0,
};
static const unsigned char kat401_nonce[] = {
    0xad, 0xd4, 0x61, 0x04, 0x8f, 0xf6, 0x97, 0x53, 0xf8, 0x87, 0xab, 0x87,
    0xb7, 0x4c, 0x25, 0xed,
};
static const unsigned char kat401_persstr[] = {
    0xe2, 0xde, 0xdf, 0xe6, 0xb7, 0x0f, 0xd4, 0x26, 0x34, 0x8b, 0x98, 0x38,
    0xb6, 0x3f, 0x01, 0xd5, 0x21, 0x18, 0x20, 0xbf, 0xba, 0x78, 0xa2, 0x19,
    0xa8, 0x8b, 0x9b, 0x67, 0xad, 0xbf, 0xa4, 0x52,
};
static const unsigned char kat401_addin0[] = {0};
static const unsigned char kat401_addin1[] = {0};
static const unsigned char kat401_retbits[] = {
    0xc6, 0x0f, 0xcf, 0x4d, 0x36, 0x2c, 0x92, 0x3b, 0xbb, 0x70, 0xda, 0x80,
    0x6c, 0x92, 0x65, 0x1b, 0x23, 0x9e, 0x32, 0x75, 0x2b, 0xf4, 0x37, 0x66,
    0x9a, 0x39, 0xfa, 0x5e, 0x24, 0xd8, 0xa3, 0x04, 0x66, 0x8b, 0xac, 0x92,
    0xba, 0x12, 0x8a, 0x8b, 0x7e, 0x02, 0xdf, 0x00, 0xba, 0x17, 0x8f, 0x76,
    0xde, 0x44, 0xb6, 0x56, 0xf3, 0x45, 0x1f, 0x21, 0x25, 0x0d, 0xe7, 0x78,
    0xe0, 0x40, 0x5f, 0x14,
};
static const struct drbg_kat_no_reseed kat401_t = {
    10, kat401_entropyin, kat401_nonce, kat401_persstr,
    kat401_addin0, kat401_addin1, kat401_retbits
};
static const struct drbg_kat kat401 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat401_t
};

static const unsigned char kat402_entropyin[] = {
    0x90, 0x09, 0x14, 0x8d, 0xfc, 0x32, 0xc4, 0xcd, 0x2e, 0x67, 0x09, 0xc5,
    0x4a, 0x93, 0xf5, 0x7a, 0xa7, 0x5a, 0x93, 0x0a, 0x39, 0x5b, 0xf2, 0x01,
};
static const unsigned char kat402_nonce[] = {
    0xb2, 0x8d, 0xfc, 0x3c, 0x6b, 0x33, 0x9e, 0xb8, 0xbb, 0xaf, 0x29, 0xe5,
    0xf8, 0x19, 0x8b, 0x4e,
};
static const unsigned char kat402_persstr[] = {
    0xeb, 0xb2, 0xe3, 0x30, 0xa0, 0x5d, 0x1e, 0xaa, 0xfc, 0x51, 0x16, 0xe9,
    0x37, 0x62, 0x20, 0x73, 0x2c, 0xcf, 0x09, 0x1e, 0x1b, 0x0e, 0x40, 0x17,
    0xfa, 0xa9, 0x43, 0x46, 0xa8, 0x39, 0x45, 0xc6,
};
static const unsigned char kat402_addin0[] = {0};
static const unsigned char kat402_addin1[] = {0};
static const unsigned char kat402_retbits[] = {
    0x43, 0xc3, 0x1a, 0xc4, 0x9d, 0xe9, 0xf8, 0xe7, 0x74, 0xdf, 0xc8, 0xe4,
    0x37, 0xe4, 0x0a, 0x87, 0xe4, 0x25, 0x29, 0xe6, 0x25, 0xbd, 0xbd, 0x10,
    0xbf, 0x77, 0xbc, 0x0a, 0x81, 0x74, 0xfb, 0x9d, 0xd0, 0x8f, 0x6e, 0xdd,
    0x30, 0x8c, 0x6e, 0xeb, 0xa3, 0x17, 0x5b, 0xd3, 0x49, 0x6d, 0xa1, 0x14,
    0x36, 0xde, 0x61, 0x2a, 0x6d, 0x47, 0xf5, 0xd9, 0xe2, 0x63, 0x39, 0xeb,
    0xec, 0xda, 0x64, 0x9d,
};
static const struct drbg_kat_no_reseed kat402_t = {
    11, kat402_entropyin, kat402_nonce, kat402_persstr,
    kat402_addin0, kat402_addin1, kat402_retbits
};
static const struct drbg_kat kat402 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat402_t
};

static const unsigned char kat403_entropyin[] = {
    0xf3, 0xf4, 0xe1, 0xab, 0xac, 0x55, 0x87, 0xf2, 0xa8, 0x40, 0x9b, 0x1a,
    0x22, 0x7f, 0x9c, 0x92, 0x41, 0x6f, 0x65, 0x67, 0x04, 0x95, 0x01, 0xdb,
};
static const unsigned char kat403_nonce[] = {
    0xc1, 0x97, 0x96, 0x37, 0xe6, 0xb4, 0xd0, 0x4b, 0xc9, 0x70, 0xd3, 0x7e,
    0x11, 0xcd, 0xeb, 0xb5,
};
static const unsigned char kat403_persstr[] = {
    0x3d, 0x49, 0x16, 0x31, 0xd7, 0x4d, 0x3d, 0x90, 0x8a, 0xc5, 0x8c, 0xc4,
    0x74, 0x81, 0xb5, 0xe0, 0x96, 0x4b, 0xb2, 0x19, 0x22, 0x06, 0x1c, 0xfe,
    0xbd, 0x83, 0x38, 0x20, 0xe9, 0x12, 0xa0, 0x0b,
};
static const unsigned char kat403_addin0[] = {0};
static const unsigned char kat403_addin1[] = {0};
static const unsigned char kat403_retbits[] = {
    0x90, 0xed, 0xdd, 0x43, 0x9e, 0xed, 0x53, 0x79, 0x9e, 0xa5, 0x74, 0xc9,
    0x23, 0x84, 0x73, 0xff, 0xf0, 0x30, 0x38, 0x1e, 0x1a, 0x48, 0x54, 0x5b,
    0xc0, 0x7b, 0xb2, 0xcc, 0x9b, 0xb2, 0xd0, 0xea, 0xf2, 0x3a, 0x23, 0xf0,
    0x09, 0x88, 0x1d, 0x1c, 0x24, 0x64, 0xe6, 0xb3, 0x9f, 0x02, 0x30, 0x5b,
    0x32, 0x14, 0xd8, 0x99, 0x32, 0xf8, 0xc2, 0x37, 0x96, 0x49, 0x82, 0xc9,
    0xa7, 0xad, 0xcb, 0xfb,
};
static const struct drbg_kat_no_reseed kat403_t = {
    12, kat403_entropyin, kat403_nonce, kat403_persstr,
    kat403_addin0, kat403_addin1, kat403_retbits
};
static const struct drbg_kat kat403 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat403_t
};

static const unsigned char kat404_entropyin[] = {
    0xdc, 0x24, 0x65, 0x25, 0x7d, 0x55, 0x89, 0x89, 0xa3, 0x40, 0x26, 0x3f,
    0xaa, 0x86, 0xef, 0x3f, 0x98, 0xe8, 0xa9, 0xfb, 0x53, 0x98, 0x04, 0xa9,
};
static const unsigned char kat404_nonce[] = {
    0x1c, 0xa1, 0x60, 0x8e, 0xf5, 0xf2, 0x6b, 0xdf, 0xbb, 0xb1, 0xb5, 0x06,
    0xfd, 0x77, 0x24, 0x26,
};
static const unsigned char kat404_persstr[] = {
    0xdb, 0x9b, 0xa1, 0x82, 0x86, 0x42, 0x94, 0x65, 0xf7, 0x3b, 0x35, 0x73,
    0x9e, 0x6c, 0x98, 0x7e, 0x07, 0xc0, 0x39, 0x7d, 0xd6, 0x03, 0xe0, 0x36,
    0x85, 0x3c, 0x54, 0x6e, 0x49, 0x93, 0xfe, 0x89,
};
static const unsigned char kat404_addin0[] = {0};
static const unsigned char kat404_addin1[] = {0};
static const unsigned char kat404_retbits[] = {
    0xfd, 0x35, 0xff, 0xc4, 0x30, 0x51, 0x43, 0x57, 0x12, 0x37, 0x4a, 0xd9,
    0x03, 0xa4, 0x65, 0x5b, 0x36, 0xd3, 0x75, 0x4d, 0x78, 0x55, 0x1c, 0x3f,
    0x48, 0xd7, 0x04, 0x94, 0x74, 0x4e, 0xe2, 0xf3, 0x36, 0xbd, 0x19, 0xb7,
    0x80, 0xc8, 0xcb, 0x6e, 0x5d, 0xe4, 0xa6, 0x67, 0x11, 0x7d, 0x1f, 0xb6,
    0xb8, 0x1f, 0x03, 0xd3, 0x62, 0x0b, 0xb5, 0xbe, 0x18, 0x02, 0x15, 0x28,
    0x42, 0xc6, 0x97, 0x3e,
};
static const struct drbg_kat_no_reseed kat404_t = {
    13, kat404_entropyin, kat404_nonce, kat404_persstr,
    kat404_addin0, kat404_addin1, kat404_retbits
};
static const struct drbg_kat kat404 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat404_t
};

static const unsigned char kat405_entropyin[] = {
    0x1b, 0xeb, 0xd5, 0xd8, 0x76, 0xe0, 0x94, 0x9d, 0x1d, 0x8d, 0xeb, 0x41,
    0x0d, 0x3a, 0x77, 0x9c, 0xed, 0x95, 0xe7, 0x0c, 0x0b, 0x4a, 0xd5, 0x31,
};
static const unsigned char kat405_nonce[] = {
    0x4c, 0x1d, 0xb0, 0x88, 0x71, 0x71, 0x96, 0x23, 0xea, 0xb2, 0xa2, 0x4f,
    0xcb, 0x8f, 0xca, 0x1c,
};
static const unsigned char kat405_persstr[] = {
    0xf7, 0x3f, 0x41, 0xc6, 0x14, 0x6f, 0x2c, 0x3d, 0x6a, 0xde, 0x95, 0x46,
    0x2a, 0x70, 0xf1, 0x23, 0xb9, 0x60, 0x4b, 0xea, 0x02, 0xfd, 0xca, 0x6b,
    0x11, 0x54, 0xc1, 0xf8, 0x6b, 0x89, 0x9b, 0x48,
};
static const unsigned char kat405_addin0[] = {0};
static const unsigned char kat405_addin1[] = {0};
static const unsigned char kat405_retbits[] = {
    0x4a, 0x0e, 0x35, 0xcf, 0x6a, 0x1e, 0x19, 0x9e, 0x56, 0xfc, 0x6b, 0x5f,
    0x37, 0x84, 0xb2, 0x8c, 0x1a, 0xc7, 0x0e, 0x91, 0xfd, 0x74, 0xd8, 0xa7,
    0xef, 0x45, 0xa1, 0x68, 0xb3, 0xaa, 0xd2, 0x8f, 0x32, 0x44, 0x11, 0xee,
    0x88, 0x6b, 0xb7, 0x39, 0xbb, 0x81, 0x4d, 0x5b, 0xa1, 0xdd, 0x79, 0x4a,
    0x0d, 0x37, 0x2e, 0xed, 0x5f, 0x4a, 0x97, 0x1a, 0x0c, 0x55, 0x42, 0xb9,
    0x47, 0xc4, 0xe9, 0x77,
};
static const struct drbg_kat_no_reseed kat405_t = {
    14, kat405_entropyin, kat405_nonce, kat405_persstr,
    kat405_addin0, kat405_addin1, kat405_retbits
};
static const struct drbg_kat kat405 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat405_t
};

static const unsigned char kat406_entropyin[] = {
    0x05, 0xe7, 0x34, 0xb2, 0xb1, 0x28, 0x7f, 0x82, 0x73, 0x94, 0x22, 0x8f,
    0x81, 0x40, 0xda, 0xc0, 0x50, 0xad, 0xb8, 0x21, 0xbd, 0x37, 0xc4, 0xe2,
};
static const unsigned char kat406_nonce[] = {
    0xc5, 0x41, 0xec, 0xbe, 0x12, 0xd3, 0x00, 0xc1, 0x4a, 0x7c, 0x9f, 0xe9,
    0x0e, 0x38, 0xb1, 0xab,
};
static const unsigned char kat406_persstr[] = {
    0x01, 0x8b, 0x8e, 0x3d, 0x28, 0x5b, 0x9f, 0x7b, 0x92, 0x8c, 0xc8, 0x19,
    0xc5, 0x76, 0xfa, 0x21, 0x69, 0x2c, 0x52, 0xff, 0x71, 0x66, 0xda, 0xe7,
    0xb8, 0xaf, 0x78, 0x54, 0xcd, 0x3f, 0x8e, 0xf4,
};
static const unsigned char kat406_addin0[] = {
    0x1f, 0xd8, 0xd3, 0x7b, 0x2e, 0xa6, 0xb2, 0x0d, 0x74, 0x74, 0xd9, 0x5c,
    0xac, 0x58, 0x81, 0xa6, 0x9a, 0x12, 0x66, 0xba, 0xdc, 0x49, 0xc0, 0xda,
    0x8b, 0xe2, 0x95, 0xf3, 0xdf, 0x6a, 0xfb, 0x5e,
};
static const unsigned char kat406_addin1[] = {
    0x97, 0x26, 0x73, 0x24, 0x32, 0xe1, 0x56, 0x7e, 0x5d, 0x9e, 0x53, 0x55,
    0x4b, 0xcb, 0x88, 0x6f, 0x12, 0x9c, 0x80, 0x07, 0xda, 0x9e, 0x68, 0xb3,
    0x25, 0xbf, 0x16, 0xab, 0x74, 0x0f, 0x54, 0x0e,
};
static const unsigned char kat406_retbits[] = {
    0x3c, 0xc1, 0xec, 0x09, 0x29, 0x23, 0xa2, 0xba, 0x7f, 0xca, 0x0d, 0xbf,
    0x54, 0x3c, 0xce, 0xb9, 0xb7, 0x61, 0xc6, 0x1c, 0xad, 0xd0, 0x23, 0x35,
    0xdf, 0x36, 0x1b, 0xc4, 0x25, 0x34, 0xc9, 0xb0, 0x13, 0x73, 0xc8, 0xe5,
    0x13, 0xb0, 0x69, 0xef, 0x77, 0xdd, 0x0f, 0xf8, 0xe4, 0x62, 0x3e, 0xcf,
    0xd1, 0x2f, 0x29, 0x0a, 0x28, 0x45, 0xb8, 0xe7, 0xa1, 0xf2, 0xeb, 0x97,
    0x54, 0x7a, 0x16, 0xcc,
};
static const struct drbg_kat_no_reseed kat406_t = {
    0, kat406_entropyin, kat406_nonce, kat406_persstr,
    kat406_addin0, kat406_addin1, kat406_retbits
};
static const struct drbg_kat kat406 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat406_t
};

static const unsigned char kat407_entropyin[] = {
    0x0a, 0x07, 0xb4, 0x89, 0x3a, 0xdc, 0x1d, 0x05, 0xf2, 0xe6, 0xf4, 0x61,
    0xa7, 0x17, 0xc3, 0xb4, 0xed, 0x12, 0xf9, 0xdc, 0x6a, 0xb1, 0xee, 0xc0,
};
static const unsigned char kat407_nonce[] = {
    0x90, 0x0a, 0xda, 0xad, 0xf9, 0x0e, 0xf5, 0x85, 0x67, 0x84, 0xea, 0x86,
    0x05, 0x7b, 0x57, 0x14,
};
static const unsigned char kat407_persstr[] = {
    0x12, 0x1e, 0x2e, 0xad, 0x8d, 0xa5, 0xb5, 0xcb, 0xc8, 0x40, 0x85, 0x6c,
    0x13, 0x6e, 0x94, 0x58, 0xe6, 0x5b, 0x53, 0x6a, 0x51, 0x31, 0x01, 0x5e,
    0x8e, 0xf4, 0xeb, 0x16, 0x7a, 0xa1, 0xf9, 0x64,
};
static const unsigned char kat407_addin0[] = {
    0x4b, 0x79, 0x78, 0x58, 0x37, 0x73, 0x29, 0xb1, 0x9a, 0x69, 0x4a, 0x32,
    0x0d, 0xec, 0xa0, 0x92, 0x69, 0xbc, 0xee, 0x49, 0x61, 0x1f, 0x7e, 0xd6,
    0xf4, 0x0e, 0xb9, 0x45, 0x99, 0xf4, 0x20, 0xb6,
};
static const unsigned char kat407_addin1[] = {
    0xfb, 0x4e, 0xb2, 0x1d, 0x67, 0xcb, 0xbe, 0x06, 0xb9, 0x96, 0xcf, 0x80,
    0xca, 0xb0, 0xcb, 0xbd, 0x61, 0xe9, 0xcf, 0x35, 0x80, 0x48, 0x77, 0x8e,
    0xbc, 0x50, 0x52, 0x0e, 0x5f, 0xb0, 0x1b, 0xb2,
};
static const unsigned char kat407_retbits[] = {
    0xfc, 0xf3, 0xaa, 0x35, 0xd1, 0x4d, 0x2b, 0xf7, 0xfc, 0xc6, 0xb4, 0x64,
    0x60, 0xb5, 0x0d, 0x6b, 0xc9, 0xee, 0xb7, 0xb1, 0x17, 0xae, 0xef, 0xf4,
    0x55, 0x02, 0x2d, 0xa6, 0xeb, 0x32, 0xaa, 0x3b, 0x3e, 0x9b, 0xf7, 0xdb,
    0x05, 0xd2, 0x44, 0xa8, 0xa6, 0xbd, 0xbb, 0xee, 0x4a, 0x73, 0xa9, 0x6e,
    0x9b, 0x3a, 0xc8, 0xf1, 0x9a, 0xa6, 0x0a, 0xb1, 0x19, 0xf0, 0x0b, 0xca,
    0xcf, 0x61, 0xdd, 0x13,
};
static const struct drbg_kat_no_reseed kat407_t = {
    1, kat407_entropyin, kat407_nonce, kat407_persstr,
    kat407_addin0, kat407_addin1, kat407_retbits
};
static const struct drbg_kat kat407 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat407_t
};

static const unsigned char kat408_entropyin[] = {
    0x81, 0x98, 0x28, 0xbb, 0xcb, 0xc1, 0xd9, 0x5b, 0x72, 0x9e, 0xa4, 0xbd,
    0xe2, 0x7c, 0x4a, 0x78, 0x7f, 0x49, 0xe1, 0x67, 0x0f, 0xf7, 0xa5, 0xa2,
};
static const unsigned char kat408_nonce[] = {
    0xf8, 0x8a, 0x9c, 0xe0, 0x8c, 0x10, 0x86, 0xa6, 0x5e, 0x49, 0x95, 0x80,
    0x18, 0x62, 0xac, 0xb5,
};
static const unsigned char kat408_persstr[] = {
    0x7c, 0x63, 0x19, 0xe4, 0x88, 0x82, 0x0f, 0xb9, 0xe0, 0x49, 0x7b, 0x0b,
    0x2d, 0xf4, 0xdd, 0x7d, 0x8e, 0x1b, 0xfb, 0x4c, 0x21, 0xea, 0xd3, 0xaf,
    0x74, 0x47, 0x1c, 0x4d, 0xba, 0x63, 0xf5, 0x97,
};
static const unsigned char kat408_addin0[] = {
    0x66, 0x4d, 0x25, 0x83, 0x2d, 0x57, 0x30, 0x2f, 0xe8, 0x75, 0xe6, 0xad,
    0xeb, 0xac, 0xfa, 0x82, 0x5f, 0xfe, 0xf3, 0x94, 0xf6, 0x08, 0x56, 0xbc,
    0x21, 0xf7, 0x62, 0x07, 0x4d, 0x34, 0x91, 0x7d,
};
static const unsigned char kat408_addin1[] = {
    0x2a, 0x6c, 0xfa, 0x71, 0x0e, 0xc9, 0x20, 0x2b, 0x30, 0xe2, 0x76, 0xe1,
    0x8d, 0xb6, 0x99, 0x9e, 0x24, 0x5b, 0x76, 0x89, 0xd3, 0xff, 0x9a, 0xfe,
    0x1a, 0x79, 0x62, 0x72, 0xce, 0x86, 0xfe, 0x4c,
};
static const unsigned char kat408_retbits[] = {
    0x58, 0x53, 0x4c, 0x9c, 0xaa, 0xae, 0xf5, 0x47, 0xcc, 0x5b, 0x6e, 0x60,
    0x2c, 0x09, 0xa5, 0x2b, 0x04, 0x8c, 0x96, 0xd1, 0xeb, 0x03, 0xc1, 0xa1,
    0x7d, 0x3d, 0x8d, 0xe2, 0xca, 0x86, 0xed, 0xf4, 0x0f, 0xac, 0xd9, 0x9a,
    0x06, 0xcf, 0x8d, 0x22, 0x54, 0xd3, 0xa0, 0x5c, 0xb8, 0x07, 0x1b, 0xf8,
    0x5d, 0x54, 0x14, 0x08, 0x28, 0x01, 0x94, 0x65, 0x0e, 0x5c, 0x77, 0x64,
    0x3b, 0xbc, 0xf8, 0xea,
};
static const struct drbg_kat_no_reseed kat408_t = {
    2, kat408_entropyin, kat408_nonce, kat408_persstr,
    kat408_addin0, kat408_addin1, kat408_retbits
};
static const struct drbg_kat kat408 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat408_t
};

static const unsigned char kat409_entropyin[] = {
    0x0a, 0x48, 0x1e, 0xd8, 0xc9, 0x9e, 0x58, 0xd3, 0x34, 0x61, 0xe0, 0x36,
    0x00, 0xa7, 0xd6, 0x5e, 0x36, 0xea, 0xfa, 0xa3, 0x45, 0xca, 0x07, 0xa8,
};
static const unsigned char kat409_nonce[] = {
    0x10, 0x8b, 0x19, 0x04, 0x64, 0x8f, 0xae, 0x51, 0x6f, 0x4a, 0x1b, 0x60,
    0xfe, 0xae, 0x9e, 0x8f,
};
static const unsigned char kat409_persstr[] = {
    0xbf, 0xf9, 0x9f, 0xdd, 0xca, 0x34, 0xd0, 0x8d, 0x77, 0x08, 0x44, 0x00,
    0x30, 0xf9, 0xf4, 0xe7, 0x55, 0x5e, 0x5a, 0xeb, 0x61, 0x60, 0x15, 0xcf,
    0xc8, 0xe6, 0xb8, 0x59, 0x9d, 0xac, 0x91, 0x64,
};
static const unsigned char kat409_addin0[] = {
    0x9f, 0x8c, 0x91, 0x95, 0xe5, 0x2a, 0xb0, 0x51, 0xeb, 0x94, 0x8f, 0x02,
    0xe5, 0x97, 0x2c, 0x5b, 0xbd, 0xb1, 0xa9, 0x38, 0x4a, 0x27, 0x82, 0xe0,
    0xc8, 0x3a, 0x45, 0xd2, 0x0c, 0x6a, 0x8c, 0xf7,
};
static const unsigned char kat409_addin1[] = {
    0x88, 0x61, 0x60, 0xff, 0x53, 0x4f, 0x3a, 0x4e, 0xd9, 0x7f, 0x43, 0x38,
    0x67, 0x93, 0xde, 0xc5, 0xaa, 0xb8, 0x6a, 0xe7, 0x7b, 0x30, 0xd1, 0x37,
    0x3b, 0x55, 0x6e, 0x8b, 0xf9, 0x27, 0x72, 0xff,
};
static const unsigned char kat409_retbits[] = {
    0x41, 0xaa, 0x8a, 0x43, 0x48, 0x0e, 0xe9, 0x7c, 0x72, 0xe3, 0xe0, 0x14,
    0x35, 0xd4, 0xd3, 0x41, 0xf2, 0x31, 0xcd, 0xc5, 0x19, 0xe8, 0x82, 0x5e,
    0x0f, 0x98, 0xfb, 0x63, 0xd7, 0xe9, 0xf7, 0x1b, 0xc4, 0x2c, 0x90, 0xfc,
    0x20, 0x3f, 0xcc, 0x87, 0x5a, 0xb9, 0xe1, 0x1d, 0xbb, 0x40, 0xa3, 0xaa,
    0x9e, 0xaa, 0x6e, 0x1b, 0x01, 0xcf, 0xf5, 0xed, 0x72, 0x65, 0x70, 0x8a,
    0x21, 0x08, 0x29, 0x42,
};
static const struct drbg_kat_no_reseed kat409_t = {
    3, kat409_entropyin, kat409_nonce, kat409_persstr,
    kat409_addin0, kat409_addin1, kat409_retbits
};
static const struct drbg_kat kat409 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat409_t
};

static const unsigned char kat410_entropyin[] = {
    0xe4, 0x1f, 0x2d, 0xe0, 0xaa, 0xc4, 0x0d, 0xd3, 0xae, 0x65, 0x17, 0xfe,
    0xf3, 0xc1, 0x9c, 0x20, 0x48, 0xb4, 0x31, 0xb7, 0x70, 0x4c, 0x60, 0x76,
};
static const unsigned char kat410_nonce[] = {
    0x14, 0xdc, 0x27, 0xf4, 0x6c, 0xc0, 0xaa, 0x6c, 0x35, 0xf0, 0xa6, 0x60,
    0x85, 0x0b, 0x1f, 0xef,
};
static const unsigned char kat410_persstr[] = {
    0xde, 0x24, 0xa9, 0xd5, 0x6b, 0x0c, 0x0b, 0xc4, 0xb3, 0xa9, 0x1c, 0x62,
    0x43, 0x24, 0xfc, 0x98, 0xb6, 0xf6, 0x60, 0xc3, 0x0a, 0xd9, 0xc2, 0x04,
    0xa7, 0xb0, 0x2f, 0xee, 0xf6, 0xbf, 0xf0, 0x3e,
};
static const unsigned char kat410_addin0[] = {
    0xa2, 0x44, 0xcb, 0x28, 0xa1, 0x30, 0xca, 0x53, 0x31, 0xfb, 0x24, 0x11,
    0x63, 0x4b, 0x80, 0x9d, 0x32, 0x40, 0x8e, 0xd7, 0x45, 0xb2, 0xe3, 0x8c,
    0x4b, 0x90, 0x1a, 0x95, 0x5e, 0x1e, 0x24, 0xc3,
};
static const unsigned char kat410_addin1[] = {
    0xeb, 0xf7, 0xf1, 0x87, 0xad, 0x03, 0xa3, 0x91, 0x33, 0x17, 0x5f, 0xf9,
    0x09, 0x6a, 0x04, 0xb7, 0x2e, 0xfb, 0x0f, 0xfb, 0x09, 0x9d, 0xbf, 0x16,
    0xc5, 0x3d, 0xfb, 0x6f, 0x61, 0x82, 0xe3, 0xbb,
};
static const unsigned char kat410_retbits[] = {
    0xa0, 0xe7, 0xad, 0xcc, 0x6a, 0x7e, 0x36, 0x65, 0x61, 0x21, 0x17, 0xcf,
    0xc1, 0x56, 0x20, 0xe5, 0x69, 0xad, 0xd3, 0x1c, 0x02, 0x9c, 0x1d, 0xc1,
    0xa7, 0x9f, 0x07, 0x21, 0x23, 0xfc, 0xc5, 0x42, 0xb3, 0x97, 0xa2, 0xe7,
    0x04, 0x4a, 0x98, 0x3d, 0xaf, 0xe3, 0x62, 0x80, 0x59, 0x85, 0xff, 0xc1,
    0x64, 0xff, 0x83, 0xc5, 0x68, 0x21, 0xa9, 0x2e, 0xcd, 0xa3, 0x31, 0x55,
    0x1a, 0xa7, 0xf7, 0xb9,
};
static const struct drbg_kat_no_reseed kat410_t = {
    4, kat410_entropyin, kat410_nonce, kat410_persstr,
    kat410_addin0, kat410_addin1, kat410_retbits
};
static const struct drbg_kat kat410 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat410_t
};

static const unsigned char kat411_entropyin[] = {
    0x4b, 0x55, 0xf7, 0xfb, 0xed, 0x3b, 0xab, 0x6e, 0x29, 0xeb, 0xbb, 0xc5,
    0x76, 0x85, 0x3e, 0xac, 0x83, 0xc9, 0x3e, 0x02, 0xac, 0xd6, 0x44, 0x2d,
};
static const unsigned char kat411_nonce[] = {
    0x5e, 0x4c, 0xa3, 0xe9, 0xf8, 0x0a, 0x16, 0xf5, 0x78, 0xab, 0x7d, 0x63,
    0x30, 0x1a, 0x52, 0x8a,
};
static const unsigned char kat411_persstr[] = {
    0xf4, 0xd4, 0x11, 0x07, 0x0a, 0x57, 0xe6, 0x0a, 0x17, 0xa6, 0x63, 0xbd,
    0x78, 0xec, 0x27, 0x25, 0xb7, 0x80, 0xcc, 0xa2, 0x81, 0x47, 0x61, 0x29,
    0x51, 0x98, 0x8c, 0x15, 0x2f, 0x2e, 0x27, 0x9f,
};
static const unsigned char kat411_addin0[] = {
    0x94, 0x99, 0xe3, 0x50, 0x4d, 0x86, 0x38, 0x04, 0x28, 0x0e, 0x32, 0x72,
    0x08, 0x93, 0xb8, 0xe5, 0xe7, 0x3e, 0x39, 0x62, 0x59, 0x1e, 0xc6, 0xd8,
    0x34, 0xb4, 0x97, 0x42, 0x38, 0x90, 0xbe, 0xc6,
};
static const unsigned char kat411_addin1[] = {
    0x24, 0x88, 0xbf, 0x77, 0x3c, 0x15, 0x3d, 0x85, 0x4c, 0x2b, 0x72, 0xa0,
    0x21, 0x91, 0x57, 0x92, 0x64, 0xf7, 0x84, 0xc2, 0x9a, 0x50, 0x9e, 0x0c,
    0x4e, 0xc1, 0x97, 0x46, 0xca, 0x98, 0x48, 0xa0,
};
static const unsigned char kat411_retbits[] = {
    0x7d, 0x1a, 0xa3, 0x30, 0xdf, 0xcf, 0xac, 0x15, 0x7a, 0xf9, 0x1e, 0x67,
    0x52, 0x12, 0x9c, 0x3e, 0xc8, 0x02, 0x9b, 0xf8, 0xad, 0x54, 0x37, 0xc4,
    0xc0, 0x14, 0x40, 0x62, 0x48, 0x10, 0x2c, 0x0c, 0xe0, 0x95, 0xb6, 0xb2,
    0x7f, 0x25, 0x2a, 0xe3, 0x12, 0xf4, 0x9f, 0x11, 0x98, 0x5e, 0x67, 0x3d,
    0x9c, 0xc7, 0xb2, 0xac, 0x18, 0x1a, 0x30, 0x44, 0xdc, 0x0f, 0xb5, 0xe6,
    0xad, 0x7a, 0x1d, 0x6a,
};
static const struct drbg_kat_no_reseed kat411_t = {
    5, kat411_entropyin, kat411_nonce, kat411_persstr,
    kat411_addin0, kat411_addin1, kat411_retbits
};
static const struct drbg_kat kat411 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat411_t
};

static const unsigned char kat412_entropyin[] = {
    0xd5, 0xac, 0x19, 0x5a, 0x47, 0x31, 0x4d, 0x9f, 0x10, 0x5b, 0x29, 0x18,
    0x7d, 0x28, 0x3c, 0x6d, 0x24, 0xc4, 0x4c, 0x70, 0x54, 0x00, 0x64, 0x1d,
};
static const unsigned char kat412_nonce[] = {
    0xea, 0x92, 0xb7, 0x3e, 0xfc, 0x54, 0x31, 0xad, 0xd6, 0xca, 0x2a, 0xe8,
    0x7d, 0x9b, 0xaa, 0x31,
};
static const unsigned char kat412_persstr[] = {
    0x6e, 0x2b, 0x85, 0x89, 0xc4, 0x90, 0xe0, 0x48, 0x41, 0x3f, 0x0d, 0x88,
    0xc5, 0x51, 0x04, 0x7e, 0x3d, 0xe7, 0xff, 0xc7, 0xcf, 0x43, 0xdc, 0x9d,
    0x35, 0xdb, 0x93, 0x27, 0xb3, 0x5a, 0xb9, 0x70,
};
static const unsigned char kat412_addin0[] = {
    0x51, 0xed, 0x74, 0xbd, 0xd7, 0x55, 0x85, 0xc2, 0x14, 0x5c, 0xc2, 0x38,
    0x48, 0x62, 0xf3, 0x4e, 0xd0, 0xb4, 0x0c, 0x30, 0x44, 0xac, 0x60, 0x48,
    0x7e, 0xe1, 0x0b, 0xef, 0xaf, 0xd5, 0xb1, 0x6b,
};
static const unsigned char kat412_addin1[] = {
    0x89, 0x4a, 0xa5, 0x0d, 0x05, 0xa4, 0x0e, 0x5b, 0xff, 0x48, 0x14, 0xe7,
    0x56, 0x86, 0xe3, 0x42, 0x00, 0x4a, 0x66, 0x0e, 0x2e, 0x4d, 0xc7, 0x36,
    0x3e, 0xd7, 0x28, 0x88, 0xb7, 0x92, 0xc0, 0xc5,
};
static const unsigned char kat412_retbits[] = {
    0xcc, 0x3c, 0xe3, 0x94, 0xcb, 0x02, 0x5b, 0xcf, 0xf4, 0x36, 0xe6, 0xa6,
    0xc2, 0xf8, 0x03, 0x76, 0x7e, 0xa8, 0xed, 0xff, 0x89, 0x05, 0xa2, 0x3b,
    0xe4, 0x23, 0x9e, 0x64, 0xc2, 0x82, 0x6e, 0x54, 0xbe, 0xe2, 0xea, 0xb1,
    0xb1, 0x4f, 0xe4, 0x4a, 0x68, 0x2d, 0x15, 0x13, 0xc4, 0x05, 0x3b, 0x2e,
    0x46, 0x99, 0xb1, 0x8a, 0xd1, 0xc3, 0xc2, 0x5e, 0xf1, 0xa5, 0xe4, 0x22,
    0x73, 0x0a, 0xca, 0xfd,
};
static const struct drbg_kat_no_reseed kat412_t = {
    6, kat412_entropyin, kat412_nonce, kat412_persstr,
    kat412_addin0, kat412_addin1, kat412_retbits
};
static const struct drbg_kat kat412 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat412_t
};

static const unsigned char kat413_entropyin[] = {
    0x8b, 0x5d, 0xd1, 0xe8, 0x9f, 0xa9, 0x7c, 0x9d, 0x08, 0x4f, 0x27, 0x7e,
    0x2e, 0xc5, 0x83, 0xbb, 0x30, 0x5c, 0x6b, 0x4d, 0x07, 0x37, 0xd7, 0xcf,
};
static const unsigned char kat413_nonce[] = {
    0x15, 0xab, 0x99, 0xd1, 0x5e, 0xa1, 0x12, 0x81, 0xd0, 0xfd, 0xcf, 0xd7,
    0x44, 0xf9, 0x64, 0xd0,
};
static const unsigned char kat413_persstr[] = {
    0xc2, 0xbf, 0x9e, 0x13, 0xb3, 0xb7, 0x10, 0x94, 0xea, 0x1a, 0x84, 0xa4,
    0x36, 0xa0, 0xce, 0x25, 0xe5, 0x32, 0xc2, 0xa5, 0xd9, 0xe6, 0x23, 0x12,
    0xf9, 0x67, 0xf4, 0xfd, 0x06, 0x20, 0x4d, 0x13,
};
static const unsigned char kat413_addin0[] = {
    0xfe, 0x60, 0xba, 0x40, 0x6b, 0xf6, 0x37, 0x84, 0xa0, 0xf8, 0x01, 0x9c,
    0x8f, 0x62, 0xe5, 0x25, 0xf2, 0xb8, 0x44, 0x47, 0xb9, 0x7d, 0x85, 0x87,
    0xda, 0xcd, 0x67, 0xb6, 0x90, 0xb3, 0xd8, 0xa8,
};
static const unsigned char kat413_addin1[] = {
    0x49, 0x8d, 0xec, 0x11, 0xe7, 0xcd, 0x55, 0x6a, 0x12, 0x07, 0xdc, 0x37,
    0x9c, 0x03, 0x13, 0x8e, 0xf6, 0x80, 0xba, 0x56, 0x99, 0xe4, 0x68, 0x66,
    0xa3, 0x97, 0xf1, 0xf4, 0xb9, 0x10, 0xb2, 0xe2,
};
static const unsigned char kat413_retbits[] = {
    0xca, 0xd4, 0x22, 0x4e, 0x71, 0x77, 0xaf, 0xe9, 0xf6, 0x84, 0x96, 0xb7,
    0xde, 0x52, 0x37, 0x6d, 0x8e, 0x71, 0x80, 0x7b, 0x2b, 0x3a, 0xf4, 0xe9,
    0x7e, 0x81, 0x85, 0xf5, 0xbe, 0x91, 0x2f, 0x85, 0xd3, 0x3d, 0xda, 0x07,
    0xee, 0x48, 0x99, 0x3f, 0x22, 0x3e, 0x4b, 0xd9, 0x4d, 0x1a, 0x45, 0xea,
    0xa2, 0x74, 0x44, 0x8e, 0xa7, 0x39, 0x59, 0x86, 0x01, 0x63, 0x16, 0x94,
    0x39, 0xa2, 0x5a, 0xa6,
};
static const struct drbg_kat_no_reseed kat413_t = {
    7, kat413_entropyin, kat413_nonce, kat413_persstr,
    kat413_addin0, kat413_addin1, kat413_retbits
};
static const struct drbg_kat kat413 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat413_t
};

static const unsigned char kat414_entropyin[] = {
    0x78, 0xd2, 0xe9, 0x21, 0x65, 0x6d, 0xb1, 0x69, 0x8d, 0xcf, 0xd2, 0xbb,
    0x94, 0xe9, 0x48, 0xac, 0xb7, 0x59, 0xff, 0xd2, 0xc4, 0xfb, 0xa7, 0x28,
};
static const unsigned char kat414_nonce[] = {
    0x31, 0xe2, 0x0d, 0xc0, 0xf8, 0x61, 0x42, 0x7e, 0x78, 0xe0, 0xc4, 0xa3,
    0x06, 0x59, 0x64, 0xcf,
};
static const unsigned char kat414_persstr[] = {
    0x70, 0xa5, 0xbe, 0x6e, 0xb4, 0xd2, 0xde, 0xbd, 0x0b, 0xce, 0x7f, 0x02,
    0x7c, 0x08, 0xa6, 0x1e, 0x66, 0x5a, 0x0c, 0xfd, 0xe3, 0xe1, 0xb9, 0xa8,
    0x6d, 0xb8, 0x5a, 0x92, 0x30, 0x47, 0xcf, 0xb5,
};
static const unsigned char kat414_addin0[] = {
    0x0a, 0x20, 0x43, 0x22, 0xf4, 0x4c, 0x4e, 0x83, 0x23, 0x37, 0xf7, 0x40,
    0xa7, 0xcc, 0xba, 0xbc, 0xef, 0xda, 0x2f, 0xfa, 0x21, 0xc5, 0x11, 0x11,
    0x2a, 0xd5, 0x23, 0x0c, 0x41, 0x10, 0x9f, 0x67,
};
static const unsigned char kat414_addin1[] = {
    0x33, 0x02, 0xb3, 0x31, 0x5f, 0x17, 0x64, 0x27, 0xe9, 0x12, 0x2a, 0x08,
    0xc9, 0x82, 0xd6, 0xa9, 0x33, 0x5c, 0xe9, 0x6f, 0x0d, 0x24, 0x26, 0xbc,
    0xe3, 0x82, 0x01, 0x7d, 0xf2, 0xfb, 0x9e, 0x9f,
};
static const unsigned char kat414_retbits[] = {
    0xef, 0x96, 0xc7, 0x56, 0x99, 0xd6, 0x8e, 0xc0, 0x5e, 0xde, 0x3a, 0xc6,
    0x20, 0x54, 0xd8, 0xae, 0xc3, 0xfa, 0xb4, 0x4c, 0x06, 0x4a, 0xd6, 0x33,
    0x83, 0x73, 0x7a, 0x83, 0xe9, 0xc6, 0xd8, 0xaf, 0x3d, 0x61, 0xf2, 0xb8,
    0xd0, 0x81, 0xf2, 0xf8, 0x2f, 0x23, 0xbf, 0x3a, 0x82, 0x97, 0x60, 0x90,
    0x61, 0x81, 0x65, 0x98, 0xc5, 0xbf, 0x55, 0x01, 0x3a, 0xb4, 0xd2, 0x26,
    0x36, 0x0a, 0x7c, 0x2b,
};
static const struct drbg_kat_no_reseed kat414_t = {
    8, kat414_entropyin, kat414_nonce, kat414_persstr,
    kat414_addin0, kat414_addin1, kat414_retbits
};
static const struct drbg_kat kat414 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat414_t
};

static const unsigned char kat415_entropyin[] = {
    0xa5, 0x60, 0x03, 0x7e, 0x96, 0xbb, 0x5b, 0x76, 0xbc, 0xc8, 0x1e, 0xe6,
    0xfd, 0x12, 0x7e, 0x25, 0xba, 0xc8, 0x9d, 0xdc, 0xa7, 0x1d, 0xd9, 0x41,
};
static const unsigned char kat415_nonce[] = {
    0x6a, 0x1a, 0xe2, 0x07, 0x02, 0xa1, 0xf2, 0x9d, 0xb6, 0x58, 0xe5, 0xf2,
    0xb2, 0x47, 0x6e, 0xfc,
};
static const unsigned char kat415_persstr[] = {
    0xdb, 0xc9, 0x3a, 0x3b, 0x72, 0xaa, 0x98, 0x90, 0xc0, 0x56, 0x76, 0x08,
    0x67, 0xa9, 0xd2, 0x56, 0xcd, 0x13, 0xee, 0x44, 0xdd, 0x76, 0x84, 0x47,
    0xd0, 0x68, 0x39, 0xd7, 0xc2, 0xfd, 0xf5, 0x94,
};
static const unsigned char kat415_addin0[] = {
    0x51, 0xd1, 0xf7, 0xaa, 0x0d, 0x55, 0x10, 0x79, 0xdb, 0x5c, 0x94, 0x2f,
    0xb9, 0x19, 0x9b, 0x11, 0xdd, 0xc1, 0x59, 0x73, 0x7f, 0xdc, 0xf8, 0x26,
    0x29, 0xe0, 0xe8, 0x2b, 0xc1, 0xed, 0x28, 0xe6,
};
static const unsigned char kat415_addin1[] = {
    0x73, 0x38, 0x86, 0xbf, 0xed, 0x59, 0xcf, 0xac, 0x56, 0x23, 0xe4, 0x1a,
    0x81, 0x89, 0x78, 0x0a, 0x3b, 0x24, 0x8e, 0xfd, 0x7b, 0xd4, 0xe1, 0x44,
    0x94, 0xec, 0x7d, 0x79, 0x3d, 0xb0, 0x62, 0x85,
};
static const unsigned char kat415_retbits[] = {
    0x0d, 0xb2, 0x95, 0x41, 0x85, 0x35, 0xa4, 0xbe, 0x1a, 0x05, 0x7e, 0xfb,
    0x85, 0xdc, 0x48, 0xd2, 0x9c, 0x73, 0xb7, 0x9f, 0x11, 0x3e, 0x9e, 0x4c,
    0x62, 0x4f, 0x42, 0x16, 0xa9, 0x2d, 0x8d, 0xf7, 0xb2, 0x2c, 0xf3, 0x8d,
    0xa8, 0x15, 0x63, 0x46, 0xfb, 0x5c, 0x01, 0x53, 0x71, 0x16, 0x56, 0xba,
    0x08, 0x20, 0x8a, 0x33, 0x39, 0x9c, 0x9a, 0x2d, 0x45, 0xe7, 0x6c, 0x09,
    0x50, 0x0d, 0x59, 0xe9,
};
static const struct drbg_kat_no_reseed kat415_t = {
    9, kat415_entropyin, kat415_nonce, kat415_persstr,
    kat415_addin0, kat415_addin1, kat415_retbits
};
static const struct drbg_kat kat415 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat415_t
};

static const unsigned char kat416_entropyin[] = {
    0x9d, 0x57, 0x3e, 0xc9, 0x20, 0x5c, 0x17, 0x65, 0xd0, 0x04, 0x83, 0xff,
    0xa3, 0x6d, 0xea, 0x87, 0x38, 0x92, 0x88, 0xe5, 0x54, 0xf8, 0x43, 0x5b,
};
static const unsigned char kat416_nonce[] = {
    0x73, 0xa1, 0x75, 0xb5, 0xab, 0xa2, 0xb3, 0x65, 0xd4, 0x54, 0xe0, 0xe8,
    0x91, 0xb5, 0x4c, 0x48,
};
static const unsigned char kat416_persstr[] = {
    0x35, 0xdc, 0xa1, 0x49, 0xa7, 0x9b, 0x6e, 0x45, 0xe3, 0xf1, 0xdd, 0x16,
    0xb5, 0xc5, 0x55, 0xe8, 0x9f, 0x1b, 0x1c, 0xe3, 0x5d, 0x50, 0xcf, 0x3f,
    0x9f, 0xc0, 0x06, 0x26, 0x5d, 0x46, 0x4a, 0xc2,
};
static const unsigned char kat416_addin0[] = {
    0xf6, 0xcd, 0x2e, 0x99, 0x70, 0x50, 0x15, 0x87, 0x11, 0x26, 0x7d, 0xbf,
    0x2f, 0xd3, 0xdd, 0x4d, 0xe3, 0x37, 0xc4, 0xce, 0x1a, 0x13, 0x01, 0x41,
    0xc9, 0x4c, 0x02, 0xa8, 0xa6, 0x1a, 0x29, 0xc5,
};
static const unsigned char kat416_addin1[] = {
    0x45, 0x41, 0xd1, 0x9c, 0x8e, 0x93, 0x03, 0x4e, 0x61, 0xcf, 0x42, 0x65,
    0xab, 0x80, 0xf4, 0x1f, 0x95, 0x45, 0x23, 0xef, 0x58, 0x9c, 0xd0, 0x15,
    0x45, 0x5d, 0x4e, 0x69, 0x91, 0x9b, 0xa0, 0xfd,
};
static const unsigned char kat416_retbits[] = {
    0xcc, 0x14, 0x02, 0x9c, 0xac, 0x4f, 0x41, 0x79, 0xd2, 0x05, 0x61, 0xa8,
    0x9d, 0xef, 0xf9, 0x94, 0xb1, 0x32, 0xf7, 0x63, 0x21, 0x2f, 0x28, 0xda,
    0x1c, 0x4b, 0xbc, 0xac, 0x38, 0x68, 0xbd, 0xe0, 0xa7, 0x66, 0xac, 0xd1,
    0x00, 0xb0, 0xea, 0xdd, 0x7f, 0x4c, 0x47, 0xe4, 0x2a, 0x4d, 0x68, 0x2b,
    0xf1, 0xce, 0xec, 0xf6, 0x21, 0xb4, 0xbd, 0x5a, 0xaa, 0xb6, 0x8e, 0x04,
    0x6c, 0xa5, 0xd7, 0x4b,
};
static const struct drbg_kat_no_reseed kat416_t = {
    10, kat416_entropyin, kat416_nonce, kat416_persstr,
    kat416_addin0, kat416_addin1, kat416_retbits
};
static const struct drbg_kat kat416 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat416_t
};

static const unsigned char kat417_entropyin[] = {
    0xb8, 0x1a, 0xc1, 0x61, 0xf0, 0x41, 0x97, 0x73, 0x3b, 0xd1, 0xbb, 0xaf,
    0x9e, 0xda, 0x79, 0xee, 0x51, 0xa8, 0x61, 0xf8, 0x0b, 0xd8, 0xc1, 0x2d,
};
static const unsigned char kat417_nonce[] = {
    0x5c, 0x9e, 0x5f, 0xcf, 0x06, 0x2d, 0xe1, 0x14, 0x52, 0x3f, 0x55, 0x4b,
    0x64, 0xc9, 0x35, 0xeb,
};
static const unsigned char kat417_persstr[] = {
    0x6f, 0xb5, 0xf7, 0x5a, 0x56, 0x31, 0x93, 0x08, 0x4e, 0x96, 0xed, 0x39,
    0x4e, 0x69, 0x61, 0x81, 0x88, 0x5b, 0xb2, 0xbc, 0x7b, 0x63, 0x56, 0x22,
    0x66, 0xdc, 0x4b, 0x97, 0x99, 0x9d, 0x25, 0xd8,
};
static const unsigned char kat417_addin0[] = {
    0xfd, 0x53, 0xb9, 0x23, 0x44, 0x4f, 0x97, 0x58, 0x6b, 0xec, 0x15, 0x50,
    0xa7, 0x4f, 0x36, 0x6c, 0xa9, 0x28, 0xbf, 0xb7, 0xab, 0xf6, 0xd1, 0xcf,
    0xe2, 0x44, 0xc2, 0x5b, 0x3a, 0xd9, 0x5f, 0xb7,
};
static const unsigned char kat417_addin1[] = {
    0xaf, 0x46, 0xc2, 0x6d, 0xb8, 0x46, 0xbf, 0x6e, 0x9e, 0x82, 0xb0, 0xe8,
    0x68, 0x2d, 0x0b, 0x4e, 0xe8, 0x72, 0x8c, 0x76, 0xb5, 0x26, 0x48, 0xe3,
    0xfc, 0xca, 0x0d, 0xa8, 0x82, 0x22, 0xe5, 0xe7,
};
static const unsigned char kat417_retbits[] = {
    0xc6, 0x62, 0x0e, 0x78, 0x0d, 0x72, 0x3d, 0xb4, 0x54, 0xea, 0xc8, 0x25,
    0xd1, 0x21, 0x81, 0x4b, 0xaa, 0x55, 0xf2, 0x1a, 0x9a, 0xe8, 0xae, 0x59,
    0x7c, 0xd0, 0x27, 0xc5, 0xcb, 0x13, 0xdc, 0x40, 0x72, 0xaa, 0xec, 0xbc,
    0xb4, 0x78, 0xdd, 0xca, 0x61, 0x22, 0x18, 0x6d, 0xd4, 0x7f, 0x90, 0x43,
    0xd2, 0x82, 0x8b, 0x5a, 0xf8, 0x5d, 0x01, 0x1b, 0xcf, 0xb9, 0x9d, 0x58,
    0x8b, 0x0c, 0x7b, 0xcd,
};
static const struct drbg_kat_no_reseed kat417_t = {
    11, kat417_entropyin, kat417_nonce, kat417_persstr,
    kat417_addin0, kat417_addin1, kat417_retbits
};
static const struct drbg_kat kat417 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat417_t
};

static const unsigned char kat418_entropyin[] = {
    0x39, 0xdf, 0x42, 0x18, 0xbf, 0xf3, 0x1d, 0xd0, 0x10, 0x85, 0x96, 0x9a,
    0xb9, 0xf4, 0x81, 0x88, 0xb0, 0x11, 0x2e, 0xef, 0x75, 0x9e, 0xbc, 0x4e,
};
static const unsigned char kat418_nonce[] = {
    0x3e, 0x34, 0x41, 0xc2, 0x96, 0x66, 0xa5, 0xbd, 0x31, 0x37, 0xa3, 0x0c,
    0x6d, 0x05, 0x05, 0x3e,
};
static const unsigned char kat418_persstr[] = {
    0xd3, 0x82, 0xb3, 0xd6, 0x57, 0x73, 0xa9, 0xd6, 0x38, 0xfa, 0x83, 0x77,
    0xd9, 0xc0, 0x95, 0xa6, 0x7a, 0xb3, 0x36, 0xc5, 0x5d, 0xfd, 0xb8, 0x5d,
    0x45, 0x70, 0x31, 0x05, 0x85, 0x71, 0xc5, 0x72,
};
static const unsigned char kat418_addin0[] = {
    0xbd, 0xcd, 0xa7, 0xac, 0x59, 0x0a, 0x56, 0xc1, 0x50, 0xca, 0x85, 0x80,
    0x55, 0x29, 0x9c, 0x63, 0x86, 0x12, 0xfe, 0x04, 0x57, 0xc1, 0x8f, 0xbd,
    0x5a, 0x92, 0x04, 0x5f, 0x32, 0x58, 0x71, 0x95,
};
static const unsigned char kat418_addin1[] = {
    0xf4, 0x70, 0x51, 0xf2, 0xc2, 0x9a, 0xbc, 0xf5, 0x7a, 0xf9, 0x23, 0xf1,
    0xe0, 0x8c, 0x22, 0x85, 0x88, 0x06, 0xda, 0x9c, 0xfe, 0x14, 0x27, 0x57,
    0x21, 0x50, 0x20, 0xba, 0x67, 0x63, 0x80, 0x95,
};
static const unsigned char kat418_retbits[] = {
    0x1a, 0xa4, 0x1b, 0x4b, 0x31, 0x42, 0xd6, 0x11, 0xec, 0x62, 0x27, 0x62,
    0x42, 0xe7, 0x2a, 0x84, 0x0f, 0x82, 0xf5, 0x5e, 0x22, 0xa7, 0xc9, 0xd3,
    0xa3, 0x40, 0xa9, 0x3b, 0x43, 0x64, 0x8d, 0xa7, 0x5d, 0xd7, 0x3c, 0xae,
    0x3d, 0xc6, 0xa6, 0x0c, 0x9d, 0x59, 0x6c, 0x72, 0x3f, 0xf8, 0xd2, 0xf8,
    0xac, 0x6e, 0x6e, 0x23, 0x18, 0x32, 0xfd, 0x29, 0x7b, 0x54, 0xf3, 0xd8,
    0xd5, 0x89, 0xfe, 0xe8,
};
static const struct drbg_kat_no_reseed kat418_t = {
    12, kat418_entropyin, kat418_nonce, kat418_persstr,
    kat418_addin0, kat418_addin1, kat418_retbits
};
static const struct drbg_kat kat418 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat418_t
};

static const unsigned char kat419_entropyin[] = {
    0x7d, 0x8c, 0x64, 0x8e, 0xf9, 0x6f, 0xbe, 0xc0, 0xf8, 0x17, 0x9a, 0xce,
    0xcc, 0x55, 0x56, 0x9d, 0x3b, 0xf4, 0x56, 0xce, 0x3b, 0xb2, 0xcf, 0x73,
};
static const unsigned char kat419_nonce[] = {
    0x78, 0x18, 0x93, 0xf4, 0xf3, 0xf3, 0x6b, 0x0d, 0x92, 0xd3, 0xb1, 0xac,
    0xe6, 0xd5, 0x05, 0xa3,
};
static const unsigned char kat419_persstr[] = {
    0x80, 0x0f, 0xef, 0x9c, 0xac, 0xf3, 0xf9, 0x1c, 0x04, 0x39, 0x6a, 0x52,
    0xc7, 0xd3, 0xee, 0x6c, 0x64, 0x78, 0x91, 0x16, 0x0e, 0x35, 0x01, 0x0a,
    0x4f, 0xd5, 0x02, 0xe5, 0x1d, 0x6d, 0xd6, 0x2c,
};
static const unsigned char kat419_addin0[] = {
    0x0f, 0x60, 0x97, 0x74, 0xdb, 0xb0, 0xbb, 0x50, 0x57, 0x12, 0x27, 0x38,
    0xd8, 0xf7, 0x66, 0x56, 0xb1, 0xb7, 0x05, 0xc4, 0xe4, 0x47, 0xb0, 0x0b,
    0x7d, 0xf8, 0x4c, 0xa2, 0x9f, 0xf8, 0x15, 0xe2,
};
static const unsigned char kat419_addin1[] = {
    0x9e, 0x7d, 0x9c, 0xed, 0x55, 0x0e, 0x42, 0x79, 0x90, 0xc3, 0xe2, 0x36,
    0xb7, 0x48, 0xb9, 0xd8, 0xb1, 0x90, 0xaa, 0xe4, 0x10, 0x70, 0x01, 0x5e,
    0xb2, 0x36, 0x97, 0x4b, 0x53, 0x4f, 0x0e, 0x25,
};
static const unsigned char kat419_retbits[] = {
    0x3b, 0xd0, 0x3c, 0x79, 0xf0, 0x36, 0x87, 0x99, 0x9b, 0x14, 0x7f, 0x39,
    0xc8, 0x79, 0xfc, 0x7c, 0x0d, 0x34, 0xd0, 0x04, 0x44, 0x5d, 0x75, 0xc7,
    0x15, 0x98, 0xff, 0xd5, 0x5e, 0x3f, 0xc5, 0x5c, 0x0e, 0x17, 0x33, 0x18,
    0x26, 0xac, 0x7d, 0xee, 0x4a, 0x8e, 0x6b, 0xa7, 0x98, 0x6f, 0x1d, 0x5c,
    0x84, 0x8c, 0x36, 0x36, 0x62, 0x8f, 0x4d, 0xbf, 0x9e, 0x67, 0xbe, 0xa5,
    0x72, 0x37, 0xd8, 0x86,
};
static const struct drbg_kat_no_reseed kat419_t = {
    13, kat419_entropyin, kat419_nonce, kat419_persstr,
    kat419_addin0, kat419_addin1, kat419_retbits
};
static const struct drbg_kat kat419 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat419_t
};

static const unsigned char kat420_entropyin[] = {
    0x57, 0x45, 0xb2, 0x30, 0x14, 0x84, 0x54, 0xda, 0x9c, 0x4d, 0xd1, 0x2c,
    0xe1, 0xde, 0xdb, 0x5b, 0x14, 0x8f, 0x5b, 0xbe, 0x44, 0x56, 0x80, 0xc7,
};
static const unsigned char kat420_nonce[] = {
    0x73, 0x8a, 0xb9, 0x6e, 0x37, 0xad, 0x97, 0x1b, 0x7f, 0x5a, 0x38, 0x70,
    0x38, 0x3f, 0xc1, 0x72,
};
static const unsigned char kat420_persstr[] = {
    0x4d, 0xd3, 0x59, 0x3e, 0x0f, 0xa4, 0x94, 0xbf, 0xda, 0x36, 0x5b, 0x5f,
    0xc4, 0xe5, 0x48, 0x28, 0x01, 0xe5, 0xc3, 0x99, 0xa6, 0x5f, 0xc6, 0x98,
    0x62, 0x49, 0xbf, 0xd9, 0x4b, 0xe7, 0xb5, 0xa6,
};
static const unsigned char kat420_addin0[] = {
    0x80, 0xc1, 0xe2, 0xbb, 0x60, 0xd2, 0x2c, 0x8c, 0xe4, 0x4a, 0x2a, 0xbb,
    0x7a, 0x19, 0xdd, 0x60, 0xed, 0x6a, 0x38, 0x09, 0x8f, 0x2b, 0x09, 0x84,
    0x59, 0x71, 0x6a, 0xee, 0xf2, 0x47, 0x4e, 0xad,
};
static const unsigned char kat420_addin1[] = {
    0xc9, 0x6c, 0xf9, 0x96, 0xb2, 0xe9, 0xce, 0xb9, 0xa5, 0x78, 0x23, 0x80,
    0x0f, 0x89, 0x0c, 0x8c, 0xcf, 0xea, 0xfa, 0xc7, 0x9c, 0x96, 0x95, 0xca,
    0x21, 0xf0, 0xbb, 0x6f, 0x6f, 0xcf, 0x97, 0xa7,
};
static const unsigned char kat420_retbits[] = {
    0x8a, 0x7a, 0x6d, 0x38, 0x3f, 0x34, 0x4a, 0xd7, 0x17, 0xda, 0x7c, 0x04,
    0x4d, 0x94, 0x77, 0xe8, 0x14, 0x33, 0xe0, 0x4d, 0x3b, 0x9a, 0xe4, 0x33,
    0x4e, 0xce, 0x15, 0xe1, 0x0d, 0x7d, 0xfa, 0xa1, 0xca, 0xc7, 0x9d, 0xed,
    0x51, 0x7a, 0xb3, 0xe2, 0x4c, 0xf1, 0xd3, 0x1d, 0x34, 0x05, 0x90, 0x74,
    0x5f, 0xea, 0x2e, 0xef, 0x91, 0x53, 0x20, 0x4c, 0x12, 0x09, 0x43, 0x75,
    0xbb, 0x3c, 0x4e, 0xfa,
};
static const struct drbg_kat_no_reseed kat420_t = {
    14, kat420_entropyin, kat420_nonce, kat420_persstr,
    kat420_addin0, kat420_addin1, kat420_retbits
};
static const struct drbg_kat kat420 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat420_t
};

static const unsigned char kat421_entropyin[] = {
    0xab, 0x85, 0x43, 0x81, 0x85, 0x45, 0x57, 0x3c, 0xa2, 0x17, 0xce, 0x4b,
    0x84, 0x4b, 0x9c, 0x39, 0x66, 0x70, 0x36, 0x20, 0x78, 0x4f, 0x1e, 0xec,
};
static const unsigned char kat421_nonce[] = {
    0x4d, 0x1a, 0xb7, 0x1f, 0x18, 0x24, 0x56, 0x0a, 0xf0, 0xde, 0xb8, 0x65,
    0xba, 0x4b, 0x66, 0x20,
};
static const unsigned char kat421_persstr[] = {0};
static const unsigned char kat421_addin0[] = {0};
static const unsigned char kat421_addin1[] = {0};
static const unsigned char kat421_retbits[] = {
    0xcd, 0xd9, 0xab, 0xe9, 0x52, 0x6b, 0xc9, 0x18, 0x0c, 0xf6, 0x4b, 0xa2,
    0x67, 0x9d, 0x4c, 0x10, 0x1a, 0x5a, 0x8b, 0x52, 0x44, 0xf9, 0x32, 0x2a,
    0xff, 0x8a, 0x92, 0xed, 0x1d, 0x48, 0xa7, 0x7c, 0xe2, 0x0e, 0x39, 0xd1,
    0x91, 0x5e, 0x9a, 0x52, 0x75, 0xe8, 0xa1, 0xfe, 0x7a, 0x5a, 0xa8, 0xa2,
    0x8b, 0x06, 0x42, 0xda, 0xae, 0x9a, 0x70, 0xdc, 0x9e, 0xe4, 0xea, 0x76,
    0xac, 0x03, 0x82, 0x74,
};
static const struct drbg_kat_no_reseed kat421_t = {
    0, kat421_entropyin, kat421_nonce, kat421_persstr,
    kat421_addin0, kat421_addin1, kat421_retbits
};
static const struct drbg_kat kat421 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat421_t
};

static const unsigned char kat422_entropyin[] = {
    0x20, 0xb0, 0x1e, 0x96, 0x04, 0xd2, 0x63, 0x26, 0xb8, 0x6c, 0x4b, 0xb2,
    0x2b, 0x6c, 0x8b, 0x97, 0x4e, 0x2a, 0x42, 0xf5, 0xcb, 0x92, 0x04, 0xef,
};
static const unsigned char kat422_nonce[] = {
    0x9f, 0x9d, 0x96, 0xd1, 0x25, 0x01, 0x07, 0x69, 0x45, 0x65, 0xf5, 0x0e,
    0xf0, 0x5e, 0xe2, 0xd9,
};
static const unsigned char kat422_persstr[] = {0};
static const unsigned char kat422_addin0[] = {0};
static const unsigned char kat422_addin1[] = {0};
static const unsigned char kat422_retbits[] = {
    0x8c, 0x50, 0xe7, 0x36, 0x58, 0x38, 0x94, 0x89, 0x10, 0x32, 0xe5, 0xc3,
    0xa4, 0xf5, 0x09, 0x85, 0x44, 0x63, 0x68, 0x7c, 0xd1, 0xa4, 0xd1, 0x0b,
    0x77, 0x76, 0x0b, 0xbb, 0xea, 0xc8, 0x3b, 0xc7, 0xd9, 0xb6, 0x00, 0xaa,
    0x2f, 0xd3, 0xb1, 0xc2, 0x42, 0x10, 0xba, 0x25, 0xe2, 0x16, 0xec, 0x40,
    0x19, 0xaa, 0x7f, 0x75, 0xb7, 0x4d, 0x50, 0x6b, 0xa0, 0x91, 0x3f, 0xaa,
    0xab, 0xd0, 0x11, 0xbb,
};
static const struct drbg_kat_no_reseed kat422_t = {
    1, kat422_entropyin, kat422_nonce, kat422_persstr,
    kat422_addin0, kat422_addin1, kat422_retbits
};
static const struct drbg_kat kat422 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat422_t
};

static const unsigned char kat423_entropyin[] = {
    0xd3, 0xed, 0x0d, 0x0b, 0xb5, 0x00, 0xf7, 0x35, 0xba, 0x08, 0x96, 0xc5,
    0x1f, 0x72, 0x34, 0xd5, 0x7f, 0x32, 0x98, 0xab, 0x72, 0x71, 0xfb, 0x8c,
};
static const unsigned char kat423_nonce[] = {
    0x33, 0x6d, 0xcd, 0x6b, 0xfb, 0x58, 0xea, 0x09, 0x3b, 0x92, 0x3e, 0xeb,
    0xc8, 0x44, 0xec, 0x64,
};
static const unsigned char kat423_persstr[] = {0};
static const unsigned char kat423_addin0[] = {0};
static const unsigned char kat423_addin1[] = {0};
static const unsigned char kat423_retbits[] = {
    0xe7, 0xd9, 0xc7, 0x98, 0x11, 0x5e, 0xc4, 0x2e, 0x0a, 0xf0, 0x3e, 0x91,
    0xd8, 0x9d, 0x66, 0x32, 0x31, 0xdf, 0xb8, 0xdb, 0x11, 0x87, 0xe4, 0x37,
    0x17, 0x89, 0x9e, 0x1f, 0x80, 0x9f, 0x86, 0xfc, 0x0d, 0x59, 0x85, 0x70,
    0x98, 0x21, 0xed, 0xb8, 0xf6, 0x05, 0x45, 0xbf, 0x92, 0x2a, 0x82, 0x8e,
    0x8a, 0xc9, 0xc5, 0xba, 0x36, 0x23, 0xc2, 0xcf, 0x81, 0x67, 0x1e, 0xc3,
    0x36, 0x77, 0xbf, 0xdd,
};
static const struct drbg_kat_no_reseed kat423_t = {
    2, kat423_entropyin, kat423_nonce, kat423_persstr,
    kat423_addin0, kat423_addin1, kat423_retbits
};
static const struct drbg_kat kat423 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat423_t
};

static const unsigned char kat424_entropyin[] = {
    0xfb, 0xf3, 0xc2, 0x67, 0x26, 0x4f, 0x54, 0xe4, 0x56, 0xcc, 0x56, 0xd1,
    0xe0, 0xae, 0x7f, 0xd2, 0xe5, 0x84, 0x74, 0x99, 0x71, 0x6e, 0x15, 0x80,
};
static const unsigned char kat424_nonce[] = {
    0xbf, 0x60, 0xb7, 0x1c, 0xb1, 0x3f, 0xfb, 0xf2, 0x8d, 0x20, 0xf9, 0x68,
    0x23, 0x0c, 0x17, 0x11,
};
static const unsigned char kat424_persstr[] = {0};
static const unsigned char kat424_addin0[] = {0};
static const unsigned char kat424_addin1[] = {0};
static const unsigned char kat424_retbits[] = {
    0xec, 0x04, 0xec, 0x06, 0xaa, 0xe2, 0x0e, 0x31, 0x10, 0x1e, 0x8f, 0x3e,
    0x80, 0x57, 0x81, 0x3b, 0x15, 0xe0, 0x48, 0xc6, 0xbf, 0x05, 0x33, 0x2c,
    0x10, 0x82, 0x02, 0x4b, 0xde, 0x43, 0xbc, 0x69, 0xa3, 0xf9, 0x44, 0x4f,
    0x4a, 0xe9, 0x8e, 0xfb, 0xab, 0xf3, 0xd9, 0x86, 0x32, 0x7a, 0xf0, 0xc9,
    0x3b, 0x17, 0xf5, 0xd4, 0xe1, 0x3a, 0xf7, 0xa7, 0xe2, 0x19, 0xbc, 0x93,
    0xb6, 0xd2, 0x59, 0xc4,
};
static const struct drbg_kat_no_reseed kat424_t = {
    3, kat424_entropyin, kat424_nonce, kat424_persstr,
    kat424_addin0, kat424_addin1, kat424_retbits
};
static const struct drbg_kat kat424 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat424_t
};

static const unsigned char kat425_entropyin[] = {
    0x3e, 0x19, 0x1f, 0xf8, 0x92, 0x64, 0x4d, 0x77, 0x03, 0x1b, 0x24, 0xbf,
    0x3d, 0xd6, 0xdd, 0x70, 0x4a, 0x97, 0x40, 0xd2, 0x55, 0x8c, 0xb9, 0xbe,
};
static const unsigned char kat425_nonce[] = {
    0x88, 0x89, 0x1f, 0xb2, 0x5e, 0x11, 0xb8, 0x6a, 0xba, 0x15, 0xc2, 0x46,
    0x08, 0xfc, 0x0e, 0xcf,
};
static const unsigned char kat425_persstr[] = {0};
static const unsigned char kat425_addin0[] = {0};
static const unsigned char kat425_addin1[] = {0};
static const unsigned char kat425_retbits[] = {
    0x96, 0x71, 0x23, 0xe1, 0x3a, 0x69, 0xb8, 0xda, 0xb5, 0x2f, 0x08, 0x7b,
    0xf4, 0xa2, 0x1e, 0x43, 0x81, 0x20, 0xd1, 0xcf, 0x0e, 0x8e, 0xfc, 0x7c,
    0x12, 0x20, 0x73, 0x25, 0x99, 0xff, 0x67, 0x85, 0xe0, 0x2e, 0x0d, 0xf6,
    0xdf, 0x95, 0xc5, 0x0f, 0xb7, 0xcf, 0x6c, 0xf5, 0x7f, 0x13, 0xad, 0xae,
    0x64, 0xc2, 0xf8, 0xac, 0x00, 0xf5, 0x38, 0xeb, 0x4a, 0x7d, 0x24, 0x02,
    0x5e, 0x05, 0x01, 0xcb,
};
static const struct drbg_kat_no_reseed kat425_t = {
    4, kat425_entropyin, kat425_nonce, kat425_persstr,
    kat425_addin0, kat425_addin1, kat425_retbits
};
static const struct drbg_kat kat425 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat425_t
};

static const unsigned char kat426_entropyin[] = {
    0xf3, 0xd7, 0xd0, 0x1a, 0x2c, 0xb2, 0x73, 0x1b, 0xb1, 0xd6, 0xe9, 0x9a,
    0x16, 0x74, 0x5f, 0x31, 0x9a, 0x61, 0x7e, 0xf9, 0x5d, 0xcb, 0x55, 0x0f,
};
static const unsigned char kat426_nonce[] = {
    0x8a, 0x4d, 0x89, 0x0e, 0x32, 0x30, 0xff, 0x9a, 0x1e, 0xea, 0x9a, 0x66,
    0x47, 0x9e, 0x92, 0x6f,
};
static const unsigned char kat426_persstr[] = {0};
static const unsigned char kat426_addin0[] = {0};
static const unsigned char kat426_addin1[] = {0};
static const unsigned char kat426_retbits[] = {
    0x3c, 0xee, 0xf6, 0x0f, 0x8b, 0x23, 0xe5, 0x0e, 0x7d, 0x49, 0xa3, 0x0a,
    0xc2, 0xc4, 0xfd, 0x4d, 0xcc, 0xe3, 0xa1, 0x6f, 0x94, 0x33, 0x0a, 0xe7,
    0xff, 0xf7, 0xf2, 0x7f, 0x3d, 0x56, 0x9f, 0x25, 0x30, 0x0d, 0x59, 0xfe,
    0x87, 0xff, 0x7d, 0x75, 0xce, 0x90, 0xcd, 0x99, 0xe9, 0x4b, 0xa4, 0xf4,
    0xca, 0x30, 0xd7, 0x0c, 0x75, 0x97, 0x93, 0xbb, 0xcf, 0x7b, 0xce, 0x44,
    0xc2, 0x1d, 0xd8, 0x42,
};
static const struct drbg_kat_no_reseed kat426_t = {
    5, kat426_entropyin, kat426_nonce, kat426_persstr,
    kat426_addin0, kat426_addin1, kat426_retbits
};
static const struct drbg_kat kat426 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat426_t
};

static const unsigned char kat427_entropyin[] = {
    0xd7, 0x69, 0x0e, 0x2a, 0xce, 0x3c, 0x17, 0x4e, 0x9d, 0x1c, 0x8c, 0xdc,
    0x1c, 0x9c, 0xda, 0x65, 0x83, 0x15, 0x26, 0x61, 0x11, 0x05, 0x19, 0x8b,
};
static const unsigned char kat427_nonce[] = {
    0xb0, 0xa1, 0x1c, 0xfb, 0x27, 0x44, 0xae, 0x7b, 0x0d, 0x67, 0x5e, 0xd0,
    0xbf, 0x26, 0x34, 0xa9,
};
static const unsigned char kat427_persstr[] = {0};
static const unsigned char kat427_addin0[] = {0};
static const unsigned char kat427_addin1[] = {0};
static const unsigned char kat427_retbits[] = {
    0xd3, 0xea, 0x3d, 0xc6, 0x29, 0x6e, 0x36, 0x40, 0xf2, 0x2e, 0x75, 0xee,
    0x27, 0xe9, 0xf7, 0xf3, 0xc7, 0xff, 0x06, 0xf1, 0x79, 0x4d, 0x91, 0x5e,
    0x4f, 0xe9, 0xa4, 0x71, 0x43, 0x13, 0x17, 0xd0, 0x9d, 0x80, 0x17, 0x1b,
    0x1e, 0x7f, 0xd8, 0xfc, 0x57, 0xd8, 0xb8, 0xd1, 0xd1, 0xfa, 0x61, 0x16,
    0x2b, 0xbf, 0xc5, 0x67, 0x43, 0x83, 0x57, 0x42, 0xb4, 0xd5, 0x26, 0x66,
    0x6c, 0xf5, 0xcc, 0x03,
};
static const struct drbg_kat_no_reseed kat427_t = {
    6, kat427_entropyin, kat427_nonce, kat427_persstr,
    kat427_addin0, kat427_addin1, kat427_retbits
};
static const struct drbg_kat kat427 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat427_t
};

static const unsigned char kat428_entropyin[] = {
    0x82, 0x51, 0xbb, 0x48, 0x80, 0x97, 0xe7, 0xd0, 0x7f, 0x41, 0xa7, 0x68,
    0xc1, 0xf3, 0x7c, 0x42, 0x11, 0x80, 0xc9, 0x8e, 0xf5, 0x9b, 0x73, 0x0b,
};
static const unsigned char kat428_nonce[] = {
    0xb7, 0x2b, 0xc5, 0x2b, 0x35, 0x8c, 0x79, 0x33, 0xa3, 0x16, 0xb1, 0x7c,
    0xbf, 0x07, 0xcc, 0x02,
};
static const unsigned char kat428_persstr[] = {0};
static const unsigned char kat428_addin0[] = {0};
static const unsigned char kat428_addin1[] = {0};
static const unsigned char kat428_retbits[] = {
    0x31, 0xf3, 0x57, 0xab, 0xf5, 0xa2, 0xa0, 0x21, 0xee, 0x7e, 0x0e, 0xfb,
    0xad, 0x1f, 0xcb, 0x58, 0x11, 0x8e, 0x1b, 0xeb, 0xe4, 0x2e, 0xf6, 0x5a,
    0xc8, 0xcc, 0x93, 0xd6, 0xd0, 0xe5, 0x4f, 0x5a, 0xf7, 0xc4, 0x02, 0x2f,
    0x34, 0x3a, 0xd6, 0x3e, 0xfc, 0x94, 0xd9, 0x50, 0x35, 0x4b, 0x83, 0x8a,
    0x37, 0x8c, 0xc7, 0x7f, 0x20, 0xf7, 0x59, 0x70, 0x5b, 0xce, 0x43, 0xd8,
    0x73, 0x4e, 0x71, 0x7b,
};
static const struct drbg_kat_no_reseed kat428_t = {
    7, kat428_entropyin, kat428_nonce, kat428_persstr,
    kat428_addin0, kat428_addin1, kat428_retbits
};
static const struct drbg_kat kat428 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat428_t
};

static const unsigned char kat429_entropyin[] = {
    0xab, 0xdb, 0x2e, 0xd2, 0xe0, 0x9a, 0xa0, 0x0d, 0x4c, 0x39, 0x4f, 0x80,
    0xfc, 0x0d, 0x5b, 0xf2, 0x13, 0xb4, 0x28, 0x3b, 0x76, 0x8b, 0x58, 0x12,
};
static const unsigned char kat429_nonce[] = {
    0x89, 0xe1, 0x9d, 0xb4, 0x2a, 0x96, 0x2a, 0x1d, 0x35, 0xd4, 0x9c, 0xb8,
    0xc6, 0x43, 0xb7, 0x13,
};
static const unsigned char kat429_persstr[] = {0};
static const unsigned char kat429_addin0[] = {0};
static const unsigned char kat429_addin1[] = {0};
static const unsigned char kat429_retbits[] = {
    0x06, 0x7a, 0xe3, 0xea, 0x23, 0x04, 0x91, 0x71, 0x47, 0xcf, 0xf9, 0x76,
    0x86, 0x4d, 0x9b, 0x89, 0x4f, 0xb3, 0x8b, 0x1c, 0x8b, 0x0e, 0x57, 0x12,
    0x56, 0x47, 0x4a, 0x8e, 0x3c, 0x60, 0x20, 0x99, 0x6d, 0x16, 0x9c, 0x04,
    0x15, 0x45, 0x46, 0x64, 0x97, 0x6a, 0xdf, 0x95, 0x05, 0xc5, 0x3b, 0x5e,
    0x94, 0x6b, 0x4d, 0x0e, 0x65, 0x06, 0x67, 0x50, 0xcf, 0x0d, 0xe1, 0x41,
    0xe8, 0x46, 0x0b, 0x0b,
};
static const struct drbg_kat_no_reseed kat429_t = {
    8, kat429_entropyin, kat429_nonce, kat429_persstr,
    kat429_addin0, kat429_addin1, kat429_retbits
};
static const struct drbg_kat kat429 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat429_t
};

static const unsigned char kat430_entropyin[] = {
    0x4a, 0x44, 0x10, 0x1f, 0x7a, 0x1f, 0x22, 0xb1, 0x7e, 0x7c, 0xcc, 0x91,
    0xe9, 0x78, 0x9f, 0xcc, 0x6f, 0x1b, 0x4d, 0xc3, 0x09, 0x72, 0x56, 0x7b,
};
static const unsigned char kat430_nonce[] = {
    0xcb, 0x9f, 0xc3, 0xd4, 0x52, 0xf6, 0xcb, 0xe5, 0xc9, 0x83, 0x1e, 0x65,
    0x37, 0x02, 0x7d, 0x94,
};
static const unsigned char kat430_persstr[] = {0};
static const unsigned char kat430_addin0[] = {0};
static const unsigned char kat430_addin1[] = {0};
static const unsigned char kat430_retbits[] = {
    0xd9, 0x3b, 0x85, 0x86, 0xd6, 0x10, 0x91, 0xe3, 0xbb, 0xef, 0x18, 0x7e,
    0x5f, 0xed, 0x6a, 0x2b, 0x17, 0x00, 0xe9, 0x34, 0x10, 0x86, 0x6d, 0x10,
    0xbc, 0x02, 0xd3, 0xa6, 0x22, 0xa0, 0xa8, 0xb3, 0x8d, 0x8b, 0x08, 0x33,
    0x61, 0xad, 0x53, 0x19, 0x7b, 0xc1, 0x81, 0x17, 0x68, 0x20, 0x6e, 0x54,
    0x11, 0x15, 0xbf, 0x96, 0x12, 0x19, 0x65, 0xc1, 0x6d, 0x32, 0xe1, 0xc1,
    0x78, 0x0e, 0x4f, 0x24,
};
static const struct drbg_kat_no_reseed kat430_t = {
    9, kat430_entropyin, kat430_nonce, kat430_persstr,
    kat430_addin0, kat430_addin1, kat430_retbits
};
static const struct drbg_kat kat430 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat430_t
};

static const unsigned char kat431_entropyin[] = {
    0xf5, 0xa4, 0x66, 0x2d, 0x9f, 0x51, 0x56, 0xd3, 0xbc, 0x28, 0x2a, 0x4b,
    0xf8, 0x2e, 0x5d, 0x97, 0x73, 0x1d, 0x36, 0xc0, 0x01, 0x79, 0xad, 0xab,
};
static const unsigned char kat431_nonce[] = {
    0x3e, 0x5e, 0x20, 0x38, 0x62, 0xbc, 0x32, 0x8e, 0x99, 0x87, 0xa7, 0x21,
    0x89, 0x7d, 0x47, 0xcd,
};
static const unsigned char kat431_persstr[] = {0};
static const unsigned char kat431_addin0[] = {0};
static const unsigned char kat431_addin1[] = {0};
static const unsigned char kat431_retbits[] = {
    0x5e, 0x08, 0xab, 0x01, 0xb0, 0xa4, 0xca, 0x99, 0x2d, 0xa9, 0x96, 0xf1,
    0x64, 0x34, 0x1b, 0xe4, 0xe2, 0x01, 0xef, 0x0d, 0x1c, 0xd0, 0x49, 0xef,
    0xf6, 0x60, 0x59, 0x5a, 0x70, 0xce, 0xf1, 0x72, 0x31, 0x79, 0xcc, 0x58,
    0x8c, 0xa5, 0x2d, 0xe7, 0xef, 0xde, 0x20, 0x6a, 0x15, 0xed, 0x7a, 0xb4,
    0x14, 0xea, 0x7c, 0xfc, 0xf6, 0x71, 0xa0, 0x5c, 0xe9, 0x53, 0x43, 0x76,
    0x39, 0xa5, 0x8c, 0x1d,
};
static const struct drbg_kat_no_reseed kat431_t = {
    10, kat431_entropyin, kat431_nonce, kat431_persstr,
    kat431_addin0, kat431_addin1, kat431_retbits
};
static const struct drbg_kat kat431 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat431_t
};

static const unsigned char kat432_entropyin[] = {
    0x10, 0xb6, 0xe2, 0x73, 0x3b, 0xa8, 0x56, 0x03, 0xdf, 0xd5, 0xd5, 0xaa,
    0x5a, 0x20, 0x67, 0x52, 0xa0, 0xf6, 0x07, 0xf9, 0xd4, 0xd3, 0xd7, 0x3c,
};
static const unsigned char kat432_nonce[] = {
    0xc0, 0xc2, 0x80, 0xae, 0x01, 0x4d, 0xf2, 0x00, 0xd0, 0x2c, 0xcd, 0x5b,
    0x79, 0xfd, 0x81, 0xb0,
};
static const unsigned char kat432_persstr[] = {0};
static const unsigned char kat432_addin0[] = {0};
static const unsigned char kat432_addin1[] = {0};
static const unsigned char kat432_retbits[] = {
    0x9a, 0xf4, 0x6d, 0xc0, 0x1a, 0xa8, 0x60, 0xd9, 0xf1, 0xea, 0x68, 0xd3,
    0xef, 0x57, 0x33, 0x17, 0x50, 0x3e, 0x54, 0x65, 0x63, 0x63, 0xb5, 0x70,
    0xae, 0x26, 0x3b, 0x37, 0x60, 0xdc, 0x17, 0x49, 0x43, 0xe1, 0x81, 0x5f,
    0x97, 0x2c, 0xab, 0xbb, 0x42, 0xe6, 0x00, 0x90, 0x19, 0x29, 0x55, 0x3f,
    0x76, 0x73, 0x9e, 0x2d, 0x29, 0xf7, 0x7c, 0x5d, 0xd1, 0x13, 0x13, 0x8c,
    0xdf, 0x97, 0x11, 0x3e,
};
static const struct drbg_kat_no_reseed kat432_t = {
    11, kat432_entropyin, kat432_nonce, kat432_persstr,
    kat432_addin0, kat432_addin1, kat432_retbits
};
static const struct drbg_kat kat432 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat432_t
};

static const unsigned char kat433_entropyin[] = {
    0x55, 0xfb, 0x21, 0x93, 0x9e, 0xbd, 0x06, 0xd2, 0x93, 0x2b, 0x6b, 0xce,
    0x9e, 0x02, 0x3e, 0x62, 0x35, 0x55, 0xf2, 0x7d, 0x73, 0xfa, 0xd8, 0x3a,
};
static const unsigned char kat433_nonce[] = {
    0x43, 0x65, 0x9f, 0x6b, 0xbb, 0x52, 0xf4, 0x16, 0xae, 0x94, 0x17, 0x90,
    0x8b, 0xeb, 0x7b, 0x2c,
};
static const unsigned char kat433_persstr[] = {0};
static const unsigned char kat433_addin0[] = {0};
static const unsigned char kat433_addin1[] = {0};
static const unsigned char kat433_retbits[] = {
    0x41, 0xe8, 0x0b, 0x17, 0xf8, 0x52, 0xdb, 0xf0, 0xe3, 0x1e, 0xbe, 0x51,
    0x27, 0xf8, 0xc5, 0x28, 0xd0, 0x67, 0x44, 0x9a, 0xe6, 0xaa, 0x03, 0xc6,
    0x82, 0x5b, 0xb4, 0xc5, 0xdc, 0xc6, 0xad, 0x5a, 0x72, 0x84, 0x12, 0x13,
    0x30, 0x77, 0xc6, 0x8c, 0xba, 0x1e, 0x40, 0x33, 0xd7, 0x19, 0x85, 0x6b,
    0xbb, 0x30, 0xd0, 0x4b, 0x82, 0xd8, 0x40, 0xfb, 0x5a, 0x91, 0x05, 0x7c,
    0x43, 0xda, 0x8d, 0xa7,
};
static const struct drbg_kat_no_reseed kat433_t = {
    12, kat433_entropyin, kat433_nonce, kat433_persstr,
    kat433_addin0, kat433_addin1, kat433_retbits
};
static const struct drbg_kat kat433 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat433_t
};

static const unsigned char kat434_entropyin[] = {
    0x93, 0xbf, 0x67, 0xed, 0xa3, 0x7b, 0xf7, 0x4b, 0xa8, 0x4b, 0x43, 0x57,
    0x49, 0xe5, 0xc3, 0x12, 0x4b, 0x98, 0x9b, 0x6a, 0xc6, 0xad, 0x72, 0x61,
};
static const unsigned char kat434_nonce[] = {
    0x29, 0x4c, 0xb2, 0x6c, 0x39, 0x58, 0x6e, 0x48, 0x95, 0xba, 0x8a, 0xe7,
    0x79, 0x10, 0x5d, 0xd2,
};
static const unsigned char kat434_persstr[] = {0};
static const unsigned char kat434_addin0[] = {0};
static const unsigned char kat434_addin1[] = {0};
static const unsigned char kat434_retbits[] = {
    0xc4, 0x60, 0x11, 0x78, 0x92, 0x4d, 0x24, 0x3f, 0xee, 0x92, 0x55, 0x7e,
    0xa3, 0x0b, 0x9d, 0x48, 0xb8, 0x16, 0x25, 0x96, 0x42, 0xda, 0x4c, 0xbe,
    0x4d, 0x86, 0x8a, 0x94, 0x8c, 0xa7, 0xbd, 0xbd, 0x41, 0xe6, 0xcd, 0x5e,
    0xb8, 0x00, 0xbf, 0x44, 0xed, 0x60, 0x6d, 0x2d, 0xdc, 0xcb, 0x34, 0xdf,
    0xcf, 0x3b, 0xac, 0xa7, 0x06, 0x06, 0x1f, 0x12, 0x62, 0x1c, 0x79, 0x9f,
    0x55, 0x1f, 0x44, 0x8e,
};
static const struct drbg_kat_no_reseed kat434_t = {
    13, kat434_entropyin, kat434_nonce, kat434_persstr,
    kat434_addin0, kat434_addin1, kat434_retbits
};
static const struct drbg_kat kat434 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat434_t
};

static const unsigned char kat435_entropyin[] = {
    0xc8, 0x03, 0x6e, 0x05, 0xb5, 0xc9, 0x7e, 0x1f, 0x17, 0x0d, 0xdd, 0x6c,
    0x95, 0x8c, 0xfe, 0x18, 0x8d, 0x44, 0x74, 0xb1, 0x1d, 0x67, 0x82, 0x62,
};
static const unsigned char kat435_nonce[] = {
    0xa1, 0x82, 0x9e, 0x13, 0x50, 0x31, 0xee, 0xf0, 0x0a, 0x27, 0xa6, 0xfe,
    0x02, 0x24, 0x8c, 0x1b,
};
static const unsigned char kat435_persstr[] = {0};
static const unsigned char kat435_addin0[] = {0};
static const unsigned char kat435_addin1[] = {0};
static const unsigned char kat435_retbits[] = {
    0x15, 0xbe, 0x79, 0xf7, 0x7f, 0x5c, 0x18, 0x43, 0x10, 0x41, 0x6e, 0x92,
    0xd7, 0xa4, 0x7f, 0xc3, 0x25, 0x67, 0xe0, 0xc4, 0x85, 0x85, 0x40, 0xbc,
    0x54, 0x0c, 0x81, 0x95, 0x04, 0xb3, 0x9b, 0xd8, 0x2b, 0xcb, 0x96, 0x1e,
    0xaf, 0x50, 0xe3, 0x8d, 0xb9, 0x0c, 0x59, 0x30, 0x90, 0x51, 0x71, 0x7c,
    0x76, 0x74, 0xfa, 0x3e, 0x75, 0xa4, 0xf5, 0xec, 0x33, 0x66, 0x9c, 0x73,
    0x6e, 0xc4, 0x3a, 0x19,
};
static const struct drbg_kat_no_reseed kat435_t = {
    14, kat435_entropyin, kat435_nonce, kat435_persstr,
    kat435_addin0, kat435_addin1, kat435_retbits
};
static const struct drbg_kat kat435 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 0, 64, &kat435_t
};

static const unsigned char kat436_entropyin[] = {
    0x17, 0x4d, 0xb4, 0xee, 0xa5, 0x56, 0x56, 0x9b, 0xc0, 0x45, 0xef, 0xc3,
    0x5f, 0x80, 0x15, 0x71, 0x75, 0x66, 0x50, 0x22, 0x88, 0xa1, 0xe7, 0xd0,
};
static const unsigned char kat436_nonce[] = {
    0x38, 0x87, 0x89, 0x0e, 0xdb, 0x10, 0x5c, 0x15, 0x41, 0xdb, 0x3a, 0xd5,
    0x95, 0x5e, 0x12, 0x62,
};
static const unsigned char kat436_persstr[] = {0};
static const unsigned char kat436_addin0[] = {
    0x54, 0x95, 0x18, 0x3b, 0xa0, 0x88, 0xd7, 0x3c, 0xff, 0x04, 0xc6, 0x20,
    0xa0, 0xdc, 0x11, 0x55, 0xd3, 0xf2, 0x82, 0x5b, 0xb1, 0xfc, 0xb9, 0x49,
    0x15, 0xf1, 0xc1, 0xa5, 0x08, 0x7a, 0xeb, 0x2c,
};
static const unsigned char kat436_addin1[] = {
    0xd3, 0x0b, 0xdb, 0xfb, 0x41, 0xa1, 0xb4, 0x76, 0xc0, 0x05, 0x6b, 0x18,
    0x0f, 0x24, 0x34, 0xd3, 0xd8, 0x35, 0x22, 0x78, 0xaf, 0x47, 0x7f, 0x50,
    0x07, 0x94, 0x75, 0xc1, 0xbd, 0xbe, 0x62, 0x10,
};
static const unsigned char kat436_retbits[] = {
    0x24, 0xd3, 0x2b, 0x58, 0xe5, 0x9d, 0x00, 0x00, 0xff, 0x57, 0x4f, 0xd4,
    0x7e, 0x67, 0x02, 0xff, 0xd1, 0xa9, 0xb8, 0xdb, 0xcd, 0xb4, 0xf1, 0x64,
    0xab, 0xf1, 0x73, 0x14, 0x5c, 0xb6, 0xd2, 0xd9, 0x23, 0x65, 0x6b, 0x55,
    0xc4, 0xe8, 0x85, 0xca, 0x34, 0xd1, 0x8c, 0x15, 0x62, 0xf1, 0x7b, 0xb5,
    0x4d, 0x10, 0xd4, 0x6b, 0x1a, 0x53, 0xae, 0x14, 0x6e, 0xae, 0x07, 0x7d,
    0xdf, 0x93, 0xed, 0x53,
};
static const struct drbg_kat_no_reseed kat436_t = {
    0, kat436_entropyin, kat436_nonce, kat436_persstr,
    kat436_addin0, kat436_addin1, kat436_retbits
};
static const struct drbg_kat kat436 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat436_t
};

static const unsigned char kat437_entropyin[] = {
    0x02, 0x17, 0x32, 0x7c, 0x7e, 0x3a, 0x61, 0xb0, 0x45, 0x9b, 0xb4, 0x92,
    0xb1, 0x0a, 0x73, 0xdd, 0x99, 0x86, 0xd2, 0x60, 0x5a, 0xdd, 0xb6, 0x64,
};
static const unsigned char kat437_nonce[] = {
    0x37, 0xeb, 0x5d, 0x66, 0x92, 0x88, 0x3c, 0xc6, 0xa6, 0xa4, 0x3c, 0x80,
    0xdf, 0x83, 0xf6, 0xe2,
};
static const unsigned char kat437_persstr[] = {0};
static const unsigned char kat437_addin0[] = {
    0xa9, 0xd2, 0x22, 0x84, 0xc5, 0x03, 0x25, 0xd7, 0x96, 0xc7, 0x94, 0xba,
    0xd2, 0x8d, 0xfe, 0xf9, 0x2d, 0x13, 0x17, 0xc9, 0x3e, 0x20, 0x0b, 0xa2,
    0x20, 0xe0, 0x0b, 0xf4, 0xf3, 0xac, 0xec, 0xc6,
};
static const unsigned char kat437_addin1[] = {
    0xbc, 0x0f, 0xb7, 0x0a, 0xf9, 0x2d, 0xff, 0xf5, 0xd5, 0x03, 0x4f, 0x90,
    0xbd, 0xbb, 0x05, 0x29, 0x56, 0x25, 0xd1, 0xda, 0x45, 0x70, 0x7f, 0x7f,
    0xe1, 0x71, 0x50, 0xbf, 0xb9, 0x88, 0xad, 0xa2,
};
static const unsigned char kat437_retbits[] = {
    0x58, 0x59, 0xc8, 0x53, 0x9a, 0x37, 0x96, 0x1f, 0x74, 0x61, 0xe8, 0xb3,
    0x55, 0x24, 0x85, 0xce, 0x37, 0xf0, 0xc1, 0x96, 0x46, 0xf7, 0x84, 0xe6,
    0x82, 0x3a, 0x8d, 0xd6, 0x06, 0x41, 0x93, 0x13, 0x31, 0x21, 0x3a, 0x22,
    0x11, 0xb3, 0x2c, 0xf5, 0x7a, 0x1a, 0x26, 0xb5, 0xa3, 0xb6, 0xe6, 0xee,
    0xbd, 0xc5, 0xab, 0xbc, 0xe0, 0xd4, 0x3d, 0xdb, 0x96, 0x7a, 0x6d, 0xe5,
    0x7c, 0x97, 0xa8, 0xf6,
};
static const struct drbg_kat_no_reseed kat437_t = {
    1, kat437_entropyin, kat437_nonce, kat437_persstr,
    kat437_addin0, kat437_addin1, kat437_retbits
};
static const struct drbg_kat kat437 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat437_t
};

static const unsigned char kat438_entropyin[] = {
    0x9e, 0x2f, 0x8e, 0xd0, 0x86, 0x1f, 0xcc, 0x64, 0x77, 0x9f, 0x01, 0xae,
    0x37, 0xb7, 0x61, 0x54, 0x40, 0x2e, 0x09, 0xc2, 0xac, 0x84, 0xec, 0x24,
};
static const unsigned char kat438_nonce[] = {
    0x6c, 0x1e, 0x77, 0x9a, 0x17, 0xff, 0x4d, 0xdc, 0x33, 0x25, 0x28, 0x41,
    0xda, 0x58, 0xc4, 0x81,
};
static const unsigned char kat438_persstr[] = {0};
static const unsigned char kat438_addin0[] = {
    0x96, 0x0d, 0x53, 0x1c, 0x3a, 0x8e, 0x7c, 0x8c, 0x26, 0x9c, 0xed, 0x9f,
    0x74, 0x64, 0xfe, 0xde, 0xc8, 0xd4, 0xf4, 0x9e, 0x73, 0x96, 0xc6, 0x0f,
    0xbd, 0x89, 0xb2, 0xea, 0x4a, 0xf6, 0xfe, 0x40,
};
static const unsigned char kat438_addin1[] = {
    0xe3, 0x29, 0x50, 0x33, 0x92, 0xa8, 0xe7, 0xeb, 0x56, 0x2e, 0xf1, 0x30,
    0xd0, 0xf9, 0xfd, 0x5e, 0x66, 0xd5, 0xf2, 0x3f, 0x24, 0x87, 0x99, 0x55,
    0xe6, 0x60, 0x52, 0xd2, 0x00, 0x9f, 0xac, 0xda,
};
static const unsigned char kat438_retbits[] = {
    0x4a, 0x02, 0x0e, 0x7a, 0xd3, 0x3b, 0xa3, 0x80, 0xa5, 0x62, 0xad, 0x13,
    0x2d, 0x5b, 0x73, 0xee, 0xad, 0x1e, 0x89, 0xd9, 0x20, 0x7a, 0x6e, 0x4a,
    0xdb, 0x12, 0xe0, 0x8d, 0x62, 0xf4, 0x88, 0x6a, 0x27, 0xc2, 0x18, 0xb0,
    0x36, 0x41, 0xf7, 0xb6, 0xcf, 0x3a, 0x90, 0xe4, 0x46, 0x0b, 0x36, 0x39,
    0xe0, 0xaa, 0x9e, 0x70, 0xab, 0xc9, 0xae, 0x6b, 0xdc, 0xca, 0x60, 0x82,
    0x7a, 0x3a, 0x07, 0x53,
};
static const struct drbg_kat_no_reseed kat438_t = {
    2, kat438_entropyin, kat438_nonce, kat438_persstr,
    kat438_addin0, kat438_addin1, kat438_retbits
};
static const struct drbg_kat kat438 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat438_t
};

static const unsigned char kat439_entropyin[] = {
    0x12, 0xf0, 0x6b, 0xc8, 0xea, 0x0c, 0x9e, 0x95, 0xb3, 0xf5, 0xb8, 0xd9,
    0xbe, 0x8d, 0x4b, 0x94, 0xfb, 0x80, 0x88, 0x33, 0x93, 0xd4, 0x44, 0x14,
};
static const unsigned char kat439_nonce[] = {
    0x49, 0xdf, 0xee, 0x73, 0x11, 0x72, 0x5f, 0x1f, 0x46, 0xc1, 0xd5, 0x00,
    0x10, 0x05, 0x41, 0xd1,
};
static const unsigned char kat439_persstr[] = {0};
static const unsigned char kat439_addin0[] = {
    0xf7, 0xec, 0xc3, 0x52, 0xda, 0xc2, 0x80, 0x62, 0xf1, 0xda, 0xb5, 0x17,
    0x85, 0xb6, 0xb0, 0xe3, 0x1b, 0x81, 0xdb, 0x6e, 0xc7, 0x6e, 0xe0, 0x53,
    0xc3, 0x2b, 0xbc, 0xa8, 0x7c, 0x7e, 0x3b, 0x9c,
};
static const unsigned char kat439_addin1[] = {
    0xef, 0x29, 0xf6, 0x89, 0xef, 0x40, 0xb3, 0xa4, 0xf1, 0x86, 0xa3, 0xfd,
    0x57, 0xbd, 0x48, 0x3f, 0xf2, 0x8b, 0x5e, 0xa6, 0x29, 0x3e, 0x17, 0x33,
    0xa3, 0x9c, 0xa3, 0xdd, 0x33, 0x12, 0xff, 0xf4,
};
static const unsigned char kat439_retbits[] = {
    0x17, 0x3a, 0x29, 0x5b, 0xdd, 0x79, 0xad, 0x14, 0x8b, 0x15, 0xc2, 0x6d,
    0x8d, 0x13, 0x13, 0x3a, 0x0c, 0x3d, 0x52, 0xc5, 0xee, 0xb6, 0x47, 0x4a,
    0x73, 0xab, 0xb7, 0x6c, 0xbb, 0xbd, 0x2f, 0x74, 0x0a, 0xb8, 0xb6, 0x57,
    0xcb, 0xcf, 0x7b, 0x5f, 0xcc, 0x86, 0x27, 0x14, 0x08, 0xb1, 0x9f, 0xd9,
    0x35, 0xe4, 0xb7, 0x4a, 0x0d, 0xf9, 0xdd, 0x7f, 0xf5, 0x09, 0x9a, 0xe0,
    0x84, 0x9f, 0x4e, 0xa0,
};
static const struct drbg_kat_no_reseed kat439_t = {
    3, kat439_entropyin, kat439_nonce, kat439_persstr,
    kat439_addin0, kat439_addin1, kat439_retbits
};
static const struct drbg_kat kat439 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat439_t
};

static const unsigned char kat440_entropyin[] = {
    0x3f, 0x4d, 0xab, 0x4b, 0xd4, 0x56, 0x79, 0x10, 0x1e, 0x4e, 0x9e, 0x1d,
    0x05, 0x38, 0xef, 0x4d, 0x76, 0x32, 0x30, 0xa0, 0xaf, 0x43, 0xfa, 0xe8,
};
static const unsigned char kat440_nonce[] = {
    0x8b, 0xb6, 0x8e, 0xd2, 0x94, 0x0f, 0x71, 0xb6, 0x0b, 0x7c, 0x7d, 0x8f,
    0x60, 0xc1, 0x5c, 0xa6,
};
static const unsigned char kat440_persstr[] = {0};
static const unsigned char kat440_addin0[] = {
    0xce, 0xdf, 0xb5, 0x65, 0xd5, 0x54, 0x24, 0x9a, 0xcd, 0xf9, 0xe6, 0x1c,
    0xe9, 0x96, 0xee, 0xfe, 0xf0, 0xcf, 0x7e, 0xb7, 0xca, 0x0e, 0xcc, 0xda,
    0x77, 0xb7, 0xe6, 0x71, 0x5f, 0x38, 0xb7, 0x7c,
};
static const unsigned char kat440_addin1[] = {
    0xba, 0xa6, 0x40, 0xa1, 0x35, 0x06, 0xf7, 0x53, 0x66, 0x8e, 0xca, 0x79,
    0x0a, 0x8d, 0x3f, 0xc4, 0xc5, 0x3f, 0x16, 0xea, 0xa9, 0x3d, 0x08, 0x8d,
    0x69, 0xf1, 0xe5, 0x88, 0x15, 0xd4, 0x2b, 0x3e,
};
static const unsigned char kat440_retbits[] = {
    0xce, 0x36, 0xad, 0x85, 0x3b, 0x12, 0xd5, 0xd5, 0x09, 0x3b, 0xa3, 0x5f,
    0xde, 0xcf, 0xeb, 0x25, 0xf1, 0xb1, 0x33, 0x0c, 0xeb, 0x37, 0xe5, 0x8e,
    0x2e, 0x58, 0xa9, 0x40, 0xf0, 0x5f, 0x90, 0x02, 0xe0, 0xfe, 0x6b, 0x8d,
    0x36, 0xb5, 0x14, 0x6c, 0x18, 0x8a, 0xb1, 0xe3, 0x3b, 0xa8, 0x4e, 0x95,
    0x4d, 0x17, 0xa5, 0x20, 0x33, 0xcc, 0x08, 0x1a, 0xa8, 0xb2, 0x50, 0x7b,
    0xe8, 0xd5, 0x16, 0x13,
};
static const struct drbg_kat_no_reseed kat440_t = {
    4, kat440_entropyin, kat440_nonce, kat440_persstr,
    kat440_addin0, kat440_addin1, kat440_retbits
};
static const struct drbg_kat kat440 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat440_t
};

static const unsigned char kat441_entropyin[] = {
    0x1a, 0xa5, 0xd2, 0x36, 0x13, 0xdd, 0x1f, 0x71, 0x71, 0x4a, 0x01, 0xe7,
    0x5e, 0xcc, 0x29, 0xa6, 0xd1, 0x48, 0x1c, 0x76, 0x15, 0x7a, 0x81, 0x77,
};
static const unsigned char kat441_nonce[] = {
    0x2c, 0xe6, 0xba, 0x4c, 0x82, 0xe7, 0xcc, 0x2e, 0x59, 0x8c, 0xda, 0x4b,
    0x3e, 0x6a, 0xc8, 0xa7,
};
static const unsigned char kat441_persstr[] = {0};
static const unsigned char kat441_addin0[] = {
    0x02, 0x6c, 0x22, 0x49, 0x12, 0xb4, 0x9e, 0xa7, 0xb2, 0xa1, 0xa7, 0xcc,
    0xd5, 0x04, 0x18, 0x09, 0x0c, 0x13, 0x83, 0xc6, 0x4f, 0x28, 0x52, 0x34,
    0x1c, 0x28, 0xc1, 0xf6, 0x56, 0x2f, 0x9b, 0xf7,
};
static const unsigned char kat441_addin1[] = {
    0xd9, 0x81, 0x0d, 0x1a, 0x33, 0x92, 0x1f, 0x06, 0x20, 0x61, 0xd7, 0xf6,
    0x67, 0x50, 0x6b, 0x02, 0x79, 0xde, 0x3e, 0x14, 0x3e, 0x55, 0xf5, 0x72,
    0x28, 0x46, 0x93, 0x60, 0x9c, 0xb7, 0x37, 0x85,
};
static const unsigned char kat441_retbits[] = {
    0x47, 0x75, 0xcb, 0xe9, 0xf4, 0xa7, 0x4b, 0x72, 0xfa, 0xcf, 0x4a, 0x2c,
    0xf8, 0xe9, 0x58, 0x1a, 0xa0, 0x25, 0x32, 0x62, 0x60, 0x37, 0xfa, 0x3b,
    0x63, 0xef, 0xef, 0xc3, 0xc7, 0xc1, 0xc2, 0xdb, 0x0e, 0xb5, 0x90, 0x3e,
    0xa1, 0xa9, 0xcc, 0x9e, 0x40, 0xb2, 0xb9, 0xe6, 0x28, 0xac, 0xfd, 0x9d,
    0x74, 0xe6, 0x65, 0x72, 0xa8, 0x65, 0x6c, 0x75, 0xdb, 0x6a, 0x5f, 0xbf,
    0x5a, 0xc0, 0xad, 0x1e,
};
static const struct drbg_kat_no_reseed kat441_t = {
    5, kat441_entropyin, kat441_nonce, kat441_persstr,
    kat441_addin0, kat441_addin1, kat441_retbits
};
static const struct drbg_kat kat441 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat441_t
};

static const unsigned char kat442_entropyin[] = {
    0x32, 0x6f, 0x6e, 0x91, 0x22, 0x70, 0x3e, 0x58, 0x31, 0x6e, 0x7e, 0x0e,
    0x2e, 0x58, 0x78, 0x61, 0x04, 0x23, 0xf0, 0xf5, 0x59, 0x73, 0x09, 0xa1,
};
static const unsigned char kat442_nonce[] = {
    0x42, 0x6e, 0x56, 0xf1, 0x3e, 0x11, 0x26, 0x9e, 0xa6, 0xbb, 0xaf, 0xa0,
    0x09, 0x86, 0x11, 0xf5,
};
static const unsigned char kat442_persstr[] = {0};
static const unsigned char kat442_addin0[] = {
    0x8a, 0xe8, 0x8c, 0xdd, 0x32, 0x0e, 0x96, 0xe9, 0x21, 0xd0, 0x6d, 0xe2,
    0x3c, 0x92, 0xa7, 0x1b, 0x5c, 0x37, 0x0e, 0xcb, 0x4d, 0xba, 0x98, 0x87,
    0xf6, 0x03, 0x44, 0xe2, 0x08, 0x56, 0xa9, 0x04,
};
static const unsigned char kat442_addin1[] = {
    0xc5, 0xbe, 0xd3, 0x4c, 0xd6, 0x08, 0x50, 0x64, 0x04, 0x40, 0xd0, 0xc7,
    0x0d, 0xb4, 0xde, 0x17, 0x85, 0x9a, 0x25, 0xc2, 0xbd, 0xce, 0xee, 0xfe,
    0x5a, 0xe0, 0xcf, 0x8f, 0xc7, 0x80, 0x36, 0x2d,
};
static const unsigned char kat442_retbits[] = {
    0xf4, 0x62, 0x28, 0x0c, 0x4e, 0xb3, 0x9c, 0x83, 0xa2, 0x08, 0x5f, 0x8d,
    0x62, 0xa4, 0xbc, 0xec, 0x7c, 0x27, 0x62, 0x12, 0x02, 0x0b, 0xed, 0x35,
    0xe8, 0x98, 0xa5, 0x54, 0xd6, 0xaf, 0xb9, 0x8e, 0x77, 0x33, 0xd5, 0x71,
    0x2e, 0xaf, 0xf8, 0x84, 0x05, 0x8c, 0xa8, 0x8a, 0xb3, 0xe4, 0x3f, 0x3e,
    0x7f, 0x58, 0xd9, 0x8e, 0xa2, 0xbb, 0x97, 0xef, 0x8a, 0x5d, 0xac, 0x63,
    0x63, 0x56, 0x66, 0x70,
};
static const struct drbg_kat_no_reseed kat442_t = {
    6, kat442_entropyin, kat442_nonce, kat442_persstr,
    kat442_addin0, kat442_addin1, kat442_retbits
};
static const struct drbg_kat kat442 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat442_t
};

static const unsigned char kat443_entropyin[] = {
    0x00, 0x71, 0x2f, 0x30, 0xc7, 0xf3, 0x2e, 0x21, 0xde, 0xbf, 0x40, 0x26,
    0x78, 0xa7, 0xe2, 0xf7, 0x20, 0x1e, 0x96, 0xe7, 0xc2, 0x55, 0x17, 0x96,
};
static const unsigned char kat443_nonce[] = {
    0x97, 0x23, 0x77, 0x2e, 0x04, 0x71, 0x0d, 0x54, 0xa3, 0x76, 0x51, 0x07,
    0xaa, 0xd9, 0x14, 0xf5,
};
static const unsigned char kat443_persstr[] = {0};
static const unsigned char kat443_addin0[] = {
    0x17, 0x77, 0xd9, 0x9d, 0xbf, 0x76, 0xe0, 0x26, 0x5f, 0xf6, 0x85, 0x3b,
    0x83, 0x51, 0x8f, 0x34, 0x1b, 0x32, 0x78, 0x81, 0x71, 0xdb, 0x4f, 0x17,
    0xca, 0x65, 0x04, 0xd7, 0x88, 0xc2, 0xf1, 0x37,
};
static const unsigned char kat443_addin1[] = {
    0x79, 0xf3, 0xa3, 0x69, 0x5e, 0x2c, 0xfa, 0xa7, 0xf0, 0x87, 0x87, 0x29,
    0xe9, 0x31, 0x0b, 0xb4, 0x79, 0xd3, 0x00, 0xcd, 0xc1, 0x5c, 0xa3, 0x7e,
    0x36, 0x8e, 0x3a, 0x45, 0xe7, 0xf1, 0xea, 0x26,
};
static const unsigned char kat443_retbits[] = {
    0xe7, 0xbd, 0x54, 0x5d, 0x18, 0x53, 0x05, 0xa0, 0xe4, 0x43, 0x51, 0x65,
    0x01, 0xe7, 0xfa, 0x9a, 0xbf, 0xa8, 0x42, 0x27, 0x4d, 0x34, 0x2a, 0xe3,
    0xae, 0xb2, 0x95, 0x83, 0xc1, 0x50, 0xea, 0x55, 0x0b, 0xc6, 0xfb, 0xa9,
    0x52, 0xa4, 0x9e, 0x30, 0xf8, 0x4c, 0x29, 0x34, 0x29, 0x9c, 0x1f, 0xd2,
    0xd5, 0x58, 0x19, 0x1e, 0xb7, 0xad, 0x80, 0xd8, 0x28, 0xe6, 0x3a, 0xa9,
    0xff, 0x22, 0x31, 0x07,
};
static const struct drbg_kat_no_reseed kat443_t = {
    7, kat443_entropyin, kat443_nonce, kat443_persstr,
    kat443_addin0, kat443_addin1, kat443_retbits
};
static const struct drbg_kat kat443 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat443_t
};

static const unsigned char kat444_entropyin[] = {
    0x0e, 0x1e, 0x1a, 0x43, 0x04, 0xbf, 0xe2, 0x37, 0xe9, 0xbd, 0xe5, 0x4c,
    0xcb, 0x87, 0x6b, 0xdb, 0x0e, 0x83, 0x3b, 0xfa, 0xa3, 0x60, 0x9e, 0xe1,
};
static const unsigned char kat444_nonce[] = {
    0x4a, 0xb3, 0xd5, 0x27, 0x40, 0xb7, 0xb9, 0xde, 0x6e, 0xd3, 0x9a, 0x01,
    0x32, 0x15, 0xd1, 0x4a,
};
static const unsigned char kat444_persstr[] = {0};
static const unsigned char kat444_addin0[] = {
    0x02, 0x99, 0x8f, 0x33, 0xf6, 0xbe, 0x3b, 0xf8, 0x95, 0x5b, 0x94, 0x45,
    0x01, 0x75, 0x69, 0x71, 0x67, 0x26, 0x86, 0x6b, 0xe7, 0xd6, 0xda, 0x47,
    0x99, 0xe4, 0xf0, 0x72, 0x0a, 0xf5, 0x35, 0x93,
};
static const unsigned char kat444_addin1[] = {
    0x1d, 0x96, 0xd9, 0xe0, 0xff, 0x54, 0x79, 0x2f, 0x4f, 0xcc, 0xe9, 0x20,
    0xbf, 0x79, 0x0c, 0xf7, 0x2d, 0x58, 0xfc, 0x86, 0x5f, 0xe8, 0xf4, 0x46,
    0xfb, 0x03, 0xcb, 0xc2, 0xeb, 0x21, 0xea, 0x2f,
};
static const unsigned char kat444_retbits[] = {
    0xad, 0xb0, 0xe9, 0x3e, 0x74, 0x79, 0x19, 0x97, 0xdd, 0xcc, 0xf9, 0x6f,
    0x56, 0xbb, 0x29, 0x78, 0x93, 0xee, 0x40, 0xf5, 0x28, 0x9b, 0xc9, 0x92,
    0xb6, 0x78, 0x70, 0xaf, 0x19, 0xc9, 0x85, 0xaa, 0x9d, 0x91, 0x1b, 0x14,
    0x17, 0x56, 0x01, 0x72, 0xca, 0x70, 0xa9, 0xf6, 0x07, 0x65, 0x56, 0xf9,
    0x6b, 0x49, 0x57, 0xcc, 0x57, 0x4c, 0x09, 0x89, 0xfa, 0xdf, 0x1c, 0xed,
    0xa9, 0xdf, 0x9b, 0xc2,
};
static const struct drbg_kat_no_reseed kat444_t = {
    8, kat444_entropyin, kat444_nonce, kat444_persstr,
    kat444_addin0, kat444_addin1, kat444_retbits
};
static const struct drbg_kat kat444 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat444_t
};

static const unsigned char kat445_entropyin[] = {
    0x16, 0x65, 0x4b, 0x44, 0xfd, 0x38, 0xc0, 0xb2, 0x44, 0x89, 0x67, 0x82,
    0x9b, 0xe7, 0xbf, 0xc4, 0x62, 0x78, 0x6f, 0x08, 0x8b, 0x1e, 0xb4, 0x57,
};
static const unsigned char kat445_nonce[] = {
    0xfe, 0xfa, 0x7f, 0xcc, 0x14, 0xfd, 0xe2, 0x5b, 0x6f, 0x4e, 0xe9, 0xa2,
    0x23, 0x14, 0x14, 0xf0,
};
static const unsigned char kat445_persstr[] = {0};
static const unsigned char kat445_addin0[] = {
    0x14, 0x60, 0xfe, 0x86, 0x87, 0x84, 0xae, 0x02, 0xb2, 0x39, 0x42, 0x7a,
    0x0a, 0x44, 0x87, 0x20, 0x94, 0x26, 0x71, 0x38, 0xa2, 0x6c, 0xc4, 0x02,
    0xe7, 0x2a, 0xab, 0x14, 0x76, 0xdf, 0x21, 0x6a,
};
static const unsigned char kat445_addin1[] = {
    0x2a, 0x2f, 0x4d, 0x85, 0xd1, 0xfc, 0x79, 0x18, 0xe8, 0x15, 0x40, 0x53,
    0xd8, 0x43, 0x02, 0xb5, 0x49, 0x6c, 0xba, 0x91, 0xc3, 0x6e, 0x47, 0x7f,
    0xf0, 0x2c, 0x75, 0x48, 0x9e, 0xe7, 0x70, 0xbf,
};
static const unsigned char kat445_retbits[] = {
    0x9e, 0xa5, 0xe4, 0x07, 0x68, 0xaa, 0x35, 0xec, 0x4a, 0x70, 0x9b, 0x8a,
    0x7d, 0x6c, 0x27, 0x61, 0xe7, 0xa7, 0x7d, 0xce, 0x06, 0xcb, 0x25, 0x37,
    0x63, 0xeb, 0xd7, 0x0a, 0x9f, 0x69, 0xed, 0x62, 0x5c, 0xcc, 0x75, 0x2c,
    0x3e, 0x7e, 0x80, 0x97, 0x69, 0xfe, 0x99, 0xdd, 0x3a, 0xb8, 0x05, 0x64,
    0x5e, 0x96, 0xc6, 0x02, 0xff, 0x9b, 0x4f, 0x78, 0x12, 0x2e, 0xf2, 0xe8,
    0x09, 0xc3, 0x9e, 0x60,
};
static const struct drbg_kat_no_reseed kat445_t = {
    9, kat445_entropyin, kat445_nonce, kat445_persstr,
    kat445_addin0, kat445_addin1, kat445_retbits
};
static const struct drbg_kat kat445 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat445_t
};

static const unsigned char kat446_entropyin[] = {
    0xee, 0xfa, 0x26, 0x21, 0x7e, 0xcf, 0xe8, 0xe0, 0x52, 0xd0, 0xd4, 0xc0,
    0x89, 0x39, 0x20, 0x83, 0x7b, 0xbd, 0x4b, 0x3f, 0x96, 0x6a, 0x34, 0x8b,
};
static const unsigned char kat446_nonce[] = {
    0xe9, 0x2a, 0x5c, 0xde, 0xa0, 0x72, 0xa2, 0xcb, 0xb9, 0x8a, 0x5d, 0x1a,
    0xb4, 0x1d, 0x7c, 0xcd,
};
static const unsigned char kat446_persstr[] = {0};
static const unsigned char kat446_addin0[] = {
    0x48, 0xf1, 0xcc, 0xfb, 0xb5, 0x9d, 0x7a, 0xe7, 0x28, 0xfb, 0x22, 0x72,
    0x98, 0x12, 0x36, 0x6c, 0xf6, 0x6a, 0x7d, 0xf6, 0x35, 0xd3, 0x3e, 0xd1,
    0x85, 0x7f, 0x9b, 0x88, 0x01, 0x3b, 0x1f, 0x21,
};
static const unsigned char kat446_addin1[] = {
    0xb2, 0x47, 0x06, 0xd1, 0xde, 0x83, 0xe2, 0x47, 0x0c, 0x37, 0xee, 0x21,
    0x4c, 0xbd, 0x72, 0x71, 0xc3, 0x2b, 0xb1, 0x65, 0x05, 0x96, 0xed, 0x12,
    0x2a, 0xf4, 0x23, 0x93, 0x50, 0xbc, 0xef, 0x9e,
};
static const unsigned char kat446_retbits[] = {
    0x65, 0x0a, 0x98, 0xb3, 0x65, 0x57, 0x6e, 0x2f, 0xe8, 0x5b, 0xf7, 0x36,
    0xcf, 0x21, 0x6b, 0x45, 0x19, 0x2b, 0xeb, 0x91, 0x25, 0xbb, 0x1a, 0xe4,
    0x89, 0x12, 0xec, 0x06, 0xf3, 0x9d, 0x94, 0xe2, 0xda, 0x02, 0xea, 0xb9,
    0x70, 0xb9, 0xd9, 0xcb, 0xc7, 0x7e, 0x30, 0x27, 0x8e, 0x46, 0xa7, 0x98,
    0x2d, 0x7a, 0x2f, 0x7b, 0x75, 0xca, 0x6c, 0x7d, 0x53, 0x83, 0x90, 0xf9,
    0x54, 0x07, 0x38, 0x2f,
};
static const struct drbg_kat_no_reseed kat446_t = {
    10, kat446_entropyin, kat446_nonce, kat446_persstr,
    kat446_addin0, kat446_addin1, kat446_retbits
};
static const struct drbg_kat kat446 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat446_t
};

static const unsigned char kat447_entropyin[] = {
    0xe7, 0x65, 0x32, 0x77, 0x4d, 0x4e, 0xf4, 0x5d, 0xeb, 0x39, 0xbf, 0xb5,
    0x2a, 0x40, 0x46, 0x1d, 0x31, 0x25, 0xe3, 0x62, 0xd2, 0x4a, 0x56, 0xb8,
};
static const unsigned char kat447_nonce[] = {
    0xaa, 0x47, 0x05, 0x55, 0xee, 0x44, 0x9f, 0x9f, 0x06, 0xba, 0x28, 0x12,
    0x97, 0x0c, 0xe6, 0x8e,
};
static const unsigned char kat447_persstr[] = {0};
static const unsigned char kat447_addin0[] = {
    0xb2, 0xec, 0xe7, 0x17, 0xa6, 0x11, 0x19, 0xe9, 0xfd, 0x32, 0x45, 0x80,
    0x81, 0xd0, 0x0f, 0xbd, 0xd7, 0xc1, 0xb5, 0x34, 0x22, 0xa5, 0x3a, 0xa3,
    0x2f, 0xf0, 0x92, 0x37, 0xe5, 0x02, 0x86, 0x87,
};
static const unsigned char kat447_addin1[] = {
    0x6b, 0x9e, 0x89, 0x44, 0x99, 0xb1, 0xe4, 0x78, 0xce, 0x1f, 0x43, 0x91,
    0x50, 0xb4, 0x80, 0xef, 0x8b, 0x80, 0x14, 0x99, 0x3e, 0xa9, 0x31, 0xd8,
    0xae, 0xbe, 0xfc, 0x8b, 0x76, 0xa9, 0x62, 0x72,
};
static const unsigned char kat447_retbits[] = {
    0xf9, 0x8f, 0xdc, 0x5d, 0x33, 0xea, 0xb1, 0xa4, 0x56, 0xfc, 0x68, 0x08,
    0x0d, 0x59, 0x3b, 0x83, 0xbf, 0x6c, 0xf0, 0xc7, 0xcd, 0xb9, 0x83, 0x40,
    0x82, 0xde, 0x36, 0x86, 0x91, 0x9d, 0x07, 0x52, 0x36, 0x5a, 0x5c, 0xa6,
    0x4f, 0x32, 0x24, 0x37, 0x67, 0x1f, 0x05, 0x43, 0xff, 0x74, 0xee, 0x64,
    0x90, 0xa0, 0xe8, 0x70, 0x63, 0x4f, 0xed, 0x4b, 0x5e, 0xc9, 0x84, 0x82,
    0xef, 0x09, 0xa2, 0xf2,
};
static const struct drbg_kat_no_reseed kat447_t = {
    11, kat447_entropyin, kat447_nonce, kat447_persstr,
    kat447_addin0, kat447_addin1, kat447_retbits
};
static const struct drbg_kat kat447 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat447_t
};

static const unsigned char kat448_entropyin[] = {
    0x4b, 0xcc, 0xcc, 0xe3, 0x5f, 0xea, 0xfd, 0xe5, 0xda, 0x22, 0xd9, 0x56,
    0x40, 0xd2, 0x8b, 0x23, 0x2c, 0xa7, 0xc6, 0xec, 0x67, 0xf4, 0x6e, 0x05,
};
static const unsigned char kat448_nonce[] = {
    0x0d, 0xf8, 0xf3, 0xcc, 0xa0, 0x12, 0x8a, 0xc3, 0x9c, 0x11, 0xc0, 0x38,
    0xf8, 0xdf, 0xe0, 0xae,
};
static const unsigned char kat448_persstr[] = {0};
static const unsigned char kat448_addin0[] = {
    0xb3, 0xe1, 0x9b, 0x4a, 0xfc, 0x44, 0x66, 0x29, 0xf4, 0x5c, 0xea, 0x68,
    0xe7, 0x6b, 0x86, 0x43, 0xc7, 0x2c, 0x92, 0x6e, 0xa0, 0xf0, 0xbd, 0x47,
    0x3a, 0xff, 0x44, 0xc8, 0x43, 0x7c, 0xa2, 0xa8,
};
static const unsigned char kat448_addin1[] = {
    0xaf, 0x06, 0xe0, 0xf1, 0xbd, 0xb1, 0x1c, 0x07, 0xd6, 0x96, 0xd5, 0x9c,
    0x05, 0x61, 0x89, 0x6b, 0x76, 0x25, 0x83, 0x25, 0x39, 0x70, 0xa9, 0x5c,
    0xcd, 0xbb, 0x16, 0x90, 0xb5, 0xd3, 0xa2, 0x11,
};
static const unsigned char kat448_retbits[] = {
    0x11, 0x26, 0x68, 0x69, 0xe6, 0xa0, 0x9f, 0xb1, 0xc3, 0x63, 0xca, 0x25,
    0xf8, 0xf3, 0x28, 0x45, 0xea, 0x68, 0x50, 0x80, 0x78, 0x72, 0xde, 0x59,
    0x4c, 0xef, 0xa5, 0x5f, 0xb5, 0xd8, 0x4b, 0xc6, 0x89, 0xcc, 0x51, 0x8b,
    0x4e, 0xb8, 0x3b, 0xb4, 0x83, 0xc1, 0x5c, 0x62, 0xac, 0xcf, 0x47, 0x4c,
    0xc6, 0xaf, 0xec, 0xa5, 0x6f, 0xfa, 0x84, 0xc3, 0xc0, 0x1b, 0xee, 0xf6,
    0xae, 0xa5, 0xe2, 0x3a,
};
static const struct drbg_kat_no_reseed kat448_t = {
    12, kat448_entropyin, kat448_nonce, kat448_persstr,
    kat448_addin0, kat448_addin1, kat448_retbits
};
static const struct drbg_kat kat448 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat448_t
};

static const unsigned char kat449_entropyin[] = {
    0x48, 0xa4, 0x79, 0x9f, 0xbb, 0x49, 0xbf, 0xc9, 0xdd, 0x09, 0x77, 0xae,
    0x3b, 0x1c, 0x89, 0x0d, 0x2f, 0xa6, 0x9f, 0x57, 0x75, 0x54, 0xc3, 0xd3,
};
static const unsigned char kat449_nonce[] = {
    0x6b, 0xfb, 0x2d, 0x6b, 0xb9, 0x2b, 0xb2, 0xed, 0x88, 0x17, 0x99, 0x2a,
    0xd9, 0x3c, 0xc6, 0xa9,
};
static const unsigned char kat449_persstr[] = {0};
static const unsigned char kat449_addin0[] = {
    0x2c, 0xbe, 0xe2, 0xfb, 0x8b, 0xa4, 0x99, 0x65, 0xc8, 0x5a, 0x9f, 0x66,
    0x03, 0x68, 0xcf, 0x1d, 0x01, 0x81, 0xb6, 0xaf, 0xe6, 0x3b, 0x34, 0x82,
    0x21, 0x48, 0x22, 0x80, 0xc9, 0x05, 0x81, 0x2d,
};
static const unsigned char kat449_addin1[] = {
    0xbb, 0xd3, 0xe2, 0x4c, 0x95, 0xd8, 0xfb, 0x5c, 0x60, 0xd7, 0x15, 0x44,
    0xb6, 0x69, 0x6b, 0x56, 0x2a, 0x2d, 0x05, 0xce, 0x2e, 0xef, 0x65, 0x8e,
    0x21, 0x3f, 0x67, 0xaa, 0x2c, 0x1f, 0xe5, 0xd2,
};
static const unsigned char kat449_retbits[] = {
    0x8d, 0x78, 0xf3, 0xb7, 0x22, 0xd4, 0xfd, 0xa0, 0xd8, 0x18, 0x37, 0xa6,
    0x1f, 0x33, 0x10, 0x3b, 0x36, 0xf7, 0x72, 0xa2, 0x0b, 0x03, 0xf9, 0xfa,
    0x80, 0xd2, 0xe7, 0xfe, 0x0c, 0x84, 0x5e, 0xdf, 0x32, 0x61, 0x65, 0xd7,
    0x16, 0x85, 0x52, 0x79, 0x31, 0xa8, 0x9d, 0x7b, 0x1d, 0xeb, 0x01, 0x24,
    0x0a, 0xeb, 0x1f, 0x8f, 0x5f, 0xef, 0x8d, 0xab, 0x57, 0xe1, 0x43, 0x84,
    0x58, 0x0b, 0x69, 0x3f,
};
static const struct drbg_kat_no_reseed kat449_t = {
    13, kat449_entropyin, kat449_nonce, kat449_persstr,
    kat449_addin0, kat449_addin1, kat449_retbits
};
static const struct drbg_kat kat449 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat449_t
};

static const unsigned char kat450_entropyin[] = {
    0xfc, 0xe8, 0x5e, 0x48, 0xde, 0xd7, 0x33, 0x7f, 0xf3, 0xd3, 0xea, 0xe3,
    0x91, 0xe2, 0xb2, 0xa3, 0x95, 0x42, 0xbe, 0x05, 0x05, 0xf2, 0x93, 0x0c,
};
static const unsigned char kat450_nonce[] = {
    0xe5, 0x90, 0x5c, 0xe6, 0x53, 0xb9, 0x27, 0x5f, 0x14, 0xb0, 0xe6, 0xa1,
    0x06, 0x23, 0xbd, 0x87,
};
static const unsigned char kat450_persstr[] = {0};
static const unsigned char kat450_addin0[] = {
    0xbe, 0xea, 0x35, 0x64, 0xab, 0x77, 0xad, 0xf6, 0x3a, 0x2f, 0xb5, 0x14,
    0x29, 0x7f, 0xe9, 0x3f, 0xc8, 0x33, 0x23, 0x1c, 0x22, 0xdb, 0x98, 0x57,
    0xa3, 0x9d, 0x6e, 0x29, 0x70, 0x85, 0x84, 0x65,
};
static const unsigned char kat450_addin1[] = {
    0x62, 0x4f, 0x46, 0xac, 0x56, 0xc6, 0x87, 0xfc, 0x33, 0x6d, 0xd4, 0xc7,
    0xd7, 0xdb, 0x44, 0xa5, 0x62, 0x91, 0xeb, 0x7f, 0x66, 0xd1, 0x88, 0x42,
    0x76, 0x06, 0x6b, 0x38, 0x55, 0x0f, 0xfc, 0x17,
};
static const unsigned char kat450_retbits[] = {
    0x69, 0x8b, 0x1e, 0xbd, 0x23, 0x91, 0xc2, 0xf1, 0x4c, 0xc8, 0xb2, 0x0d,
    0x96, 0x4f, 0x39, 0xd5, 0xea, 0xfb, 0x56, 0xd6, 0x13, 0xda, 0xa8, 0xd3,
    0x8a, 0x45, 0xcc, 0x6f, 0xd9, 0xcc, 0x98, 0xca, 0x4e, 0xac, 0xc4, 0x79,
    0x4b, 0xc9, 0xc3, 0x17, 0xaa, 0xab, 0x14, 0x10, 0xb8, 0xd5, 0x65, 0x48,
    0x43, 0x67, 0xe3, 0xd9, 0x30, 0x67, 0x1d, 0x00, 0x17, 0x13, 0xc3, 0x29,
    0xe6, 0x5d, 0x28, 0xd9,
};
static const struct drbg_kat_no_reseed kat450_t = {
    14, kat450_entropyin, kat450_nonce, kat450_persstr,
    kat450_addin0, kat450_addin1, kat450_retbits
};
static const struct drbg_kat kat450 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 0, 32, 64, &kat450_t
};

static const unsigned char kat451_entropyin[] = {
    0x2a, 0x9b, 0x56, 0xc3, 0x5d, 0x17, 0xa5, 0xeb, 0xfc, 0x5b, 0x62, 0xae,
    0x44, 0xe9, 0x29, 0xac, 0x3a, 0x07, 0x47, 0x90, 0x7c, 0x15, 0xef, 0xa6,
};
static const unsigned char kat451_nonce[] = {
    0x8d, 0xf8, 0xca, 0x01, 0x19, 0x67, 0x19, 0xe5, 0x26, 0xff, 0x2f, 0xfe,
    0xe2, 0x01, 0xef, 0x45,
};
static const unsigned char kat451_persstr[] = {
    0xa4, 0xf5, 0xfa, 0xbe, 0xd0, 0x64, 0x69, 0x39, 0x13, 0x88, 0x0e, 0x33,
    0xf5, 0xae, 0xc5, 0xed, 0x13, 0x2f, 0x42, 0x9f, 0xdf, 0xeb, 0x22, 0x6b,
    0x0e, 0x83, 0x4e, 0x72, 0xd3, 0xff, 0xb4, 0x49,
};
static const unsigned char kat451_addin0[] = {0};
static const unsigned char kat451_addin1[] = {0};
static const unsigned char kat451_retbits[] = {
    0x51, 0xe1, 0x9a, 0x13, 0xb1, 0x18, 0x15, 0xb1, 0xec, 0xb0, 0x65, 0xd5,
    0x4b, 0xbf, 0xa4, 0x5e, 0x31, 0xd9, 0x4a, 0xde, 0xca, 0x33, 0x85, 0x62,
    0x54, 0xf4, 0x34, 0x81, 0x94, 0x45, 0x13, 0xde, 0x8f, 0xa6, 0xcf, 0x23,
    0xc4, 0xfb, 0x24, 0xb3, 0x32, 0x34, 0x6d, 0x00, 0x46, 0x4b, 0x06, 0xe9,
    0xae, 0x80, 0xd9, 0x8d, 0xa9, 0xc6, 0xfd, 0x38, 0x39, 0xcf, 0x0c, 0xa7,
    0x53, 0x1c, 0xcb, 0x89,
};
static const struct drbg_kat_no_reseed kat451_t = {
    0, kat451_entropyin, kat451_nonce, kat451_persstr,
    kat451_addin0, kat451_addin1, kat451_retbits
};
static const struct drbg_kat kat451 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat451_t
};

static const unsigned char kat452_entropyin[] = {
    0x86, 0x08, 0xbd, 0xf7, 0xd3, 0x3d, 0x89, 0xc0, 0x93, 0x24, 0xb4, 0x98,
    0x95, 0x41, 0x10, 0xb8, 0xc0, 0xec, 0xcb, 0x52, 0x0c, 0xf8, 0x60, 0x68,
};
static const unsigned char kat452_nonce[] = {
    0x57, 0x2e, 0x58, 0x16, 0xca, 0x90, 0xe0, 0x29, 0x10, 0x2d, 0x5d, 0x68,
    0x21, 0x89, 0xd8, 0x56,
};
static const unsigned char kat452_persstr[] = {
    0xc4, 0xef, 0x4c, 0x15, 0x72, 0x13, 0x37, 0x20, 0x9a, 0xb0, 0xc1, 0x03,
    0xdb, 0xee, 0xdb, 0x46, 0x32, 0x93, 0x58, 0xaf, 0xc4, 0xaf, 0x0a, 0xb7,
    0x4a, 0x27, 0x82, 0x00, 0x88, 0xce, 0xbb, 0x5b,
};
static const unsigned char kat452_addin0[] = {0};
static const unsigned char kat452_addin1[] = {0};
static const unsigned char kat452_retbits[] = {
    0xc3, 0x93, 0xeb, 0xef, 0x36, 0xf8, 0x6f, 0x0f, 0xaa, 0x9e, 0x40, 0x92,
    0xea, 0x0a, 0xad, 0x0d, 0x8b, 0x81, 0x92, 0x0d, 0x76, 0x29, 0x66, 0xf3,
    0x1f, 0x1f, 0xfd, 0xc9, 0x0f, 0xd2, 0x13, 0x06, 0xf9, 0x04, 0x74, 0x22,
    0xa9, 0xde, 0x67, 0xeb, 0x2c, 0x51, 0xec, 0xfb, 0x27, 0xcb, 0xd0, 0x68,
    0x64, 0x85, 0x96, 0xc5, 0x86, 0xd9, 0xc9, 0x4f, 0x31, 0xe9, 0xe8, 0x20,
    0x47, 0xcd, 0xaa, 0xfd,
};
static const struct drbg_kat_no_reseed kat452_t = {
    1, kat452_entropyin, kat452_nonce, kat452_persstr,
    kat452_addin0, kat452_addin1, kat452_retbits
};
static const struct drbg_kat kat452 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat452_t
};

static const unsigned char kat453_entropyin[] = {
    0x41, 0xc6, 0x98, 0x33, 0x96, 0x8c, 0xed, 0x2e, 0x0d, 0x26, 0x85, 0xe0,
    0x68, 0xea, 0x7d, 0x59, 0xf6, 0xe6, 0x73, 0xaf, 0x2b, 0x7d, 0x35, 0x99,
};
static const unsigned char kat453_nonce[] = {
    0xc6, 0x30, 0xe4, 0x8e, 0xe0, 0xd5, 0x35, 0x3f, 0xc7, 0xed, 0x4d, 0x60,
    0x7d, 0xef, 0x81, 0x31,
};
static const unsigned char kat453_persstr[] = {
    0x49, 0xcc, 0x83, 0x10, 0xe0, 0xe8, 0x98, 0x46, 0xb6, 0x4f, 0x58, 0xfb,
    0xe8, 0xe2, 0xbb, 0xf6, 0x8c, 0xf9, 0x99, 0x12, 0x6f, 0x44, 0x34, 0xdf,
    0x3e, 0x53, 0x26, 0xf6, 0x19, 0x6e, 0xfd, 0x41,
};
static const unsigned char kat453_addin0[] = {0};
static const unsigned char kat453_addin1[] = {0};
static const unsigned char kat453_retbits[] = {
    0xf7, 0xf4, 0xbd, 0x0d, 0xb9, 0x84, 0xa2, 0x38, 0xdf, 0xa9, 0xdd, 0xe2,
    0xb5, 0x19, 0xdd, 0xfd, 0x0e, 0xb9, 0x9a, 0x54, 0xfb, 0x0a, 0xbb, 0x73,
    0x87, 0x3b, 0x3c, 0x1f, 0x50, 0x84, 0xdc, 0x64, 0x12, 0xbe, 0x35, 0x3f,
    0x58, 0x0b, 0xe0, 0x1c, 0xba, 0x58, 0xdc, 0x1d, 0x90, 0xa5, 0x8b, 0xc9,
    0x63, 0xbd, 0xb7, 0x7d, 0x21, 0x70, 0x15, 0x99, 0x87, 0x25, 0x9b, 0x60,
    0xc6, 0x6b, 0x86, 0x10,
};
static const struct drbg_kat_no_reseed kat453_t = {
    2, kat453_entropyin, kat453_nonce, kat453_persstr,
    kat453_addin0, kat453_addin1, kat453_retbits
};
static const struct drbg_kat kat453 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat453_t
};

static const unsigned char kat454_entropyin[] = {
    0xc1, 0xc7, 0x21, 0x67, 0x85, 0xa9, 0x1e, 0x28, 0x16, 0xc8, 0xd4, 0xd1,
    0x41, 0x77, 0x2d, 0x12, 0x2a, 0xc3, 0xa1, 0x08, 0xd6, 0xea, 0x03, 0x44,
};
static const unsigned char kat454_nonce[] = {
    0x8e, 0x22, 0x8b, 0x8f, 0xd6, 0xe4, 0x67, 0x16, 0x09, 0xa2, 0x46, 0x08,
    0x2c, 0xaa, 0x3f, 0x2b,
};
static const unsigned char kat454_persstr[] = {
    0xd4, 0x0f, 0x20, 0x72, 0xa2, 0x73, 0x54, 0xdc, 0xe5, 0xd4, 0xb1, 0x90,
    0xc0, 0x3c, 0x79, 0xc7, 0x04, 0x41, 0xf1, 0xb9, 0xac, 0x0e, 0x61, 0x34,
    0x5b, 0x76, 0x71, 0xdd, 0x57, 0x7f, 0xf4, 0xab,
};
static const unsigned char kat454_addin0[] = {0};
static const unsigned char kat454_addin1[] = {0};
static const unsigned char kat454_retbits[] = {
    0x55, 0xd4, 0x62, 0xeb, 0x21, 0x08, 0x3b, 0x55, 0xfa, 0x5b, 0xa9, 0xb6,
    0x06, 0xfb, 0xf6, 0x66, 0x91, 0x90, 0x23, 0x5c, 0xf1, 0x12, 0x12, 0x3a,
    0x40, 0xa4, 0xa2, 0xcb, 0xf2, 0x4a, 0xad, 0xb6, 0x18, 0x14, 0xfe, 0x50,
    0x35, 0x8e, 0xaa, 0x13, 0x77, 0xfc, 0x82, 0x93, 0x12, 0xd9, 0x83, 0x4d,
    0xc6, 0xa3, 0x73, 0x5d, 0x93, 0x6a, 0xb3, 0x05, 0x57, 0x53, 0x2b, 0x69,
    0xcd, 0x7b, 0x26, 0x8e,
};
static const struct drbg_kat_no_reseed kat454_t = {
    3, kat454_entropyin, kat454_nonce, kat454_persstr,
    kat454_addin0, kat454_addin1, kat454_retbits
};
static const struct drbg_kat kat454 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat454_t
};

static const unsigned char kat455_entropyin[] = {
    0xb3, 0x47, 0x43, 0x30, 0xa2, 0x63, 0x61, 0xab, 0xaf, 0x69, 0x51, 0x1f,
    0x92, 0x01, 0x80, 0x69, 0x2a, 0x10, 0xb8, 0x29, 0xe6, 0x0f, 0x9e, 0x27,
};
static const unsigned char kat455_nonce[] = {
    0xbe, 0x42, 0x0f, 0xf2, 0x6f, 0x33, 0xae, 0x5c, 0x11, 0x35, 0x05, 0x7e,
    0x37, 0x9c, 0xe8, 0xf7,
};
static const unsigned char kat455_persstr[] = {
    0x2b, 0x21, 0xb8, 0x48, 0xb8, 0xae, 0xa5, 0x1b, 0x43, 0x24, 0xf9, 0xbd,
    0xd6, 0x27, 0x52, 0xf5, 0x48, 0xfc, 0xdc, 0x5a, 0xb3, 0x20, 0xcb, 0x33,
    0xdb, 0x1f, 0x23, 0xf9, 0x38, 0x3b, 0x57, 0x65,
};
static const unsigned char kat455_addin0[] = {0};
static const unsigned char kat455_addin1[] = {0};
static const unsigned char kat455_retbits[] = {
    0x31, 0x40, 0x15, 0x58, 0x8b, 0xa1, 0x8c, 0x37, 0x5f, 0xeb, 0xd6, 0x4e,
    0xe9, 0x54, 0x23, 0xd4, 0x2a, 0x76, 0x79, 0xda, 0x10, 0xe7, 0xe1, 0x17,
    0xd7, 0xb9, 0x70, 0x89, 0xbd, 0xd2, 0x00, 0x27, 0x01, 0x0d, 0x7e, 0x60,
    0x2d, 0x6e, 0x79, 0xb8, 0xec, 0x41, 0xa7, 0xe2, 0xbf, 0x6a, 0xc4, 0x6a,
    0x41, 0xea, 0x66, 0x16, 0x01, 0xbb, 0xf6, 0x66, 0xaa, 0xf9, 0x9f, 0xb5,
    0x46, 0x7f, 0x22, 0x1e,
};
static const struct drbg_kat_no_reseed kat455_t = {
    4, kat455_entropyin, kat455_nonce, kat455_persstr,
    kat455_addin0, kat455_addin1, kat455_retbits
};
static const struct drbg_kat kat455 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat455_t
};

static const unsigned char kat456_entropyin[] = {
    0xa9, 0x5c, 0x12, 0xce, 0x08, 0xf5, 0x87, 0xfd, 0xec, 0xc6, 0x9e, 0x8f,
    0x41, 0xe7, 0xb7, 0x17, 0x90, 0x09, 0x4d, 0x46, 0xdd, 0x5b, 0xa7, 0xd9,
};
static const unsigned char kat456_nonce[] = {
    0xbe, 0x3e, 0xca, 0x66, 0x8b, 0x3e, 0xb2, 0x5d, 0xdc, 0x33, 0x36, 0xac,
    0x70, 0xe3, 0x4a, 0x56,
};
static const unsigned char kat456_persstr[] = {
    0x01, 0xbf, 0xca, 0x95, 0x6d, 0x03, 0x70, 0xb8, 0x50, 0x9e, 0xbd, 0x40,
    0x83, 0x48, 0x72, 0x28, 0xea, 0x95, 0x72, 0xeb, 0x52, 0xc6, 0x4e, 0x83,
    0x46, 0x2c, 0x09, 0x07, 0x72, 0x08, 0x87, 0x60,
};
static const unsigned char kat456_addin0[] = {0};
static const unsigned char kat456_addin1[] = {0};
static const unsigned char kat456_retbits[] = {
    0xa9, 0x94, 0xa0, 0xb9, 0x7c, 0x21, 0xcc, 0x35, 0x51, 0xcd, 0xc6, 0x46,
    0x02, 0xda, 0x0b, 0x1f, 0x67, 0x5d, 0x72, 0x29, 0xf4, 0x50, 0x40, 0x3e,
    0x30, 0x3d, 0x1a, 0x7c, 0xb9, 0x37, 0x6e, 0x9b, 0x5f, 0x88, 0x99, 0x2c,
    0x43, 0x86, 0xab, 0x75, 0x86, 0x4c, 0x24, 0xc5, 0xba, 0xe2, 0xfc, 0xe8,
    0xb0, 0x0e, 0x9b, 0xa6, 0xa6, 0x2f, 0x17, 0x62, 0x09, 0x89, 0x45, 0x65,
    0x04, 0x4c, 0x94, 0x21,
};
static const struct drbg_kat_no_reseed kat456_t = {
    5, kat456_entropyin, kat456_nonce, kat456_persstr,
    kat456_addin0, kat456_addin1, kat456_retbits
};
static const struct drbg_kat kat456 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat456_t
};

static const unsigned char kat457_entropyin[] = {
    0xfd, 0xbf, 0x44, 0x18, 0x64, 0x51, 0xe1, 0x9c, 0x0e, 0x0e, 0x71, 0x98,
    0x49, 0x55, 0xd3, 0x01, 0x43, 0x46, 0x14, 0x67, 0x9c, 0xa0, 0x53, 0x10,
};
static const unsigned char kat457_nonce[] = {
    0xc4, 0x46, 0xf2, 0xc5, 0x0f, 0x81, 0x48, 0xb4, 0x43, 0x5f, 0xa7, 0xf0,
    0x39, 0x01, 0xad, 0x16,
};
static const unsigned char kat457_persstr[] = {
    0x0c, 0x40, 0x15, 0x31, 0x7c, 0x92, 0x2f, 0xbc, 0xe0, 0x93, 0x4b, 0xc2,
    0x96, 0xec, 0x4d, 0x0c, 0x5e, 0x87, 0x94, 0x0c, 0xc3, 0x5b, 0x53, 0x5e,
    0xbb, 0x32, 0x53, 0xc5, 0xd1, 0x0c, 0xb5, 0x0d,
};
static const unsigned char kat457_addin0[] = {0};
static const unsigned char kat457_addin1[] = {0};
static const unsigned char kat457_retbits[] = {
    0x0d, 0x20, 0x84, 0x32, 0x19, 0x43, 0x3b, 0x90, 0x3e, 0x8a, 0x8e, 0x70,
    0xe4, 0x55, 0xcf, 0x57, 0x4b, 0xb3, 0x15, 0x30, 0x8c, 0xf9, 0xf2, 0xd8,
    0x37, 0xb7, 0xb2, 0x35, 0xa9, 0x5f, 0x2a, 0xc9, 0x89, 0x66, 0x1c, 0xb8,
    0x65, 0xe8, 0xf4, 0xb7, 0x74, 0x94, 0xa3, 0xdc, 0xa3, 0x14, 0x80, 0x15,
    0xab, 0xde, 0x0a, 0x73, 0xc6, 0x56, 0xa7, 0xa3, 0x6a, 0xae, 0x91, 0xf0,
    0x2d, 0xce, 0x30, 0xbd,
};
static const struct drbg_kat_no_reseed kat457_t = {
    6, kat457_entropyin, kat457_nonce, kat457_persstr,
    kat457_addin0, kat457_addin1, kat457_retbits
};
static const struct drbg_kat kat457 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat457_t
};

static const unsigned char kat458_entropyin[] = {
    0x3e, 0xf9, 0x4f, 0xda, 0x2d, 0x12, 0xa0, 0x76, 0xee, 0x5a, 0xbf, 0xd7,
    0xef, 0x39, 0xe6, 0xc7, 0xd7, 0x5a, 0x7f, 0xb9, 0x28, 0x1b, 0x4b, 0x55,
};
static const unsigned char kat458_nonce[] = {
    0xd8, 0x10, 0x72, 0xc4, 0x6e, 0xfb, 0xd9, 0x1e, 0x9c, 0x6b, 0xda, 0x13,
    0xcc, 0xc8, 0x7c, 0x7e,
};
static const unsigned char kat458_persstr[] = {
    0x57, 0x11, 0xa9, 0x70, 0x5d, 0x80, 0x0f, 0xa1, 0xa2, 0x1b, 0xd7, 0x75,
    0x2e, 0x3f, 0x1d, 0x3d, 0xa9, 0x0a, 0x1f, 0x33, 0x69, 0xfe, 0xa4, 0x03,
    0xa9, 0xcb, 0x65, 0x81, 0x24, 0x4c, 0x7d, 0xe9,
};
static const unsigned char kat458_addin0[] = {0};
static const unsigned char kat458_addin1[] = {0};
static const unsigned char kat458_retbits[] = {
    0x7d, 0x70, 0x72, 0x2a, 0x76, 0x82, 0x52, 0x64, 0xb3, 0x70, 0x19, 0xb4,
    0xda, 0x28, 0x6c, 0x73, 0x47, 0x89, 0x0f, 0x59, 0xf0, 0xa4, 0x98, 0x72,
    0x8e, 0x63, 0x59, 0xbd, 0x9d, 0x04, 0xed, 0x96, 0xa7, 0x58, 0x81, 0x66,
    0xc6, 0xef, 0x39, 0x1d, 0xa0, 0x79, 0x94, 0xd6, 0x3d, 0xb8, 0xa2, 0xfb,
    0x2c, 0xdc, 0xe3, 0x18, 0x7b, 0x56, 0xae, 0xe9, 0xa2, 0xd5, 0xcc, 0x0f,
    0x85, 0x2b, 0x2e, 0xd9,
};
static const struct drbg_kat_no_reseed kat458_t = {
    7, kat458_entropyin, kat458_nonce, kat458_persstr,
    kat458_addin0, kat458_addin1, kat458_retbits
};
static const struct drbg_kat kat458 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat458_t
};

static const unsigned char kat459_entropyin[] = {
    0xf3, 0x51, 0x9e, 0x7d, 0xa5, 0x79, 0x70, 0x17, 0xef, 0xa1, 0xa1, 0xa0,
    0x15, 0xf9, 0xa6, 0x14, 0x47, 0xfa, 0x20, 0xd0, 0xf2, 0xfa, 0x4d, 0x51,
};
static const unsigned char kat459_nonce[] = {
    0xfd, 0x2d, 0x1e, 0xae, 0xe6, 0xd8, 0x45, 0x5a, 0x03, 0xe8, 0x3a, 0x0f,
    0x80, 0xe1, 0x5b, 0x17,
};
static const unsigned char kat459_persstr[] = {
    0x1a, 0xb1, 0x08, 0xec, 0x09, 0x92, 0x70, 0x04, 0x93, 0x26, 0xd1, 0xde,
    0xde, 0xee, 0x52, 0x6b, 0x29, 0xd1, 0x77, 0x73, 0xf7, 0x66, 0x57, 0x18,
    0x66, 0xf4, 0xe9, 0x38, 0x3c, 0x25, 0xcc, 0xfa,
};
static const unsigned char kat459_addin0[] = {0};
static const unsigned char kat459_addin1[] = {0};
static const unsigned char kat459_retbits[] = {
    0xd2, 0xe7, 0x66, 0x43, 0xe8, 0xf9, 0xec, 0x5a, 0x09, 0xf8, 0x66, 0x55,
    0xfc, 0x65, 0x73, 0x8d, 0xd0, 0xd2, 0x0a, 0x52, 0x5f, 0x14, 0x97, 0xb3,
    0x27, 0x29, 0xaf, 0xce, 0xe8, 0x73, 0xc8, 0x09, 0x9f, 0x6b, 0x61, 0xc4,
    0xd9, 0x32, 0x4e, 0x02, 0xa2, 0xd8, 0x9e, 0x50, 0xf3, 0x75, 0x84, 0xc2,
    0xcd, 0x27, 0x76, 0x34, 0x22, 0x10, 0xda, 0x42, 0x05, 0x8f, 0x07, 0xdf,
    0x3f, 0x85, 0xf8, 0xac,
};
static const struct drbg_kat_no_reseed kat459_t = {
    8, kat459_entropyin, kat459_nonce, kat459_persstr,
    kat459_addin0, kat459_addin1, kat459_retbits
};
static const struct drbg_kat kat459 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat459_t
};

static const unsigned char kat460_entropyin[] = {
    0x41, 0x07, 0x2a, 0x1c, 0x20, 0x92, 0x0c, 0xe5, 0xc5, 0x2c, 0x9e, 0xb6,
    0x8b, 0x87, 0xf9, 0x7a, 0x65, 0x8a, 0x66, 0x97, 0xd4, 0xe9, 0x67, 0xdb,
};
static const unsigned char kat460_nonce[] = {
    0xcb, 0x03, 0xef, 0xae, 0xca, 0x21, 0x53, 0x4c, 0x59, 0xa9, 0xbb, 0x12,
    0xf1, 0x46, 0xee, 0x21,
};
static const unsigned char kat460_persstr[] = {
    0x9a, 0x09, 0xae, 0x50, 0xf7, 0x61, 0x49, 0xa9, 0xa1, 0xf5, 0x5a, 0x37,
    0xec, 0x30, 0xaa, 0x08, 0xcb, 0x57, 0x30, 0x57, 0xd4, 0xf5, 0xb5, 0xfa,
    0x77, 0x13, 0xef, 0x7a, 0x1c, 0x08, 0x9b, 0xaa,
};
static const unsigned char kat460_addin0[] = {0};
static const unsigned char kat460_addin1[] = {0};
static const unsigned char kat460_retbits[] = {
    0x52, 0x74, 0x42, 0x04, 0x1c, 0x23, 0x0a, 0xd4, 0x96, 0x82, 0xd7, 0x89,
    0x2c, 0x85, 0x16, 0xf6, 0x58, 0xed, 0x0b, 0x16, 0x38, 0xd1, 0xed, 0x01,
    0xb2, 0x79, 0x14, 0x7a, 0x6b, 0x4a, 0x81, 0x5e, 0x41, 0x66, 0x67, 0x90,
    0x99, 0xfc, 0x2e, 0xa1, 0xfc, 0x44, 0xb4, 0xe5, 0xe4, 0xb6, 0x10, 0x6f,
    0x66, 0x91, 0x50, 0x5d, 0xd0, 0x62, 0xea, 0x3b, 0x3d, 0xd6, 0x73, 0x32,
    0xec, 0x48, 0x2e, 0xf3,
};
static const struct drbg_kat_no_reseed kat460_t = {
    9, kat460_entropyin, kat460_nonce, kat460_persstr,
    kat460_addin0, kat460_addin1, kat460_retbits
};
static const struct drbg_kat kat460 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat460_t
};

static const unsigned char kat461_entropyin[] = {
    0xfb, 0x67, 0xde, 0x62, 0x27, 0x1a, 0xda, 0xff, 0x78, 0xdd, 0x3c, 0x06,
    0xfa, 0xb6, 0xb9, 0xbd, 0x01, 0xa8, 0x49, 0x6f, 0xc9, 0xbe, 0xb6, 0xa1,
};
static const unsigned char kat461_nonce[] = {
    0x1c, 0x5f, 0x50, 0xe7, 0x37, 0x2f, 0x8f, 0x8d, 0x40, 0x44, 0x67, 0x4c,
    0x1f, 0xf1, 0x98, 0x98,
};
static const unsigned char kat461_persstr[] = {
    0x8e, 0x4f, 0x33, 0xea, 0x52, 0xa9, 0x94, 0x43, 0x83, 0xeb, 0xe9, 0x0a,
    0x30, 0x42, 0xaa, 0xfe, 0x13, 0x03, 0xc3, 0x28, 0x29, 0x34, 0x04, 0x37,
    0x95, 0x7f, 0x83, 0x35, 0x6c, 0x83, 0x7e, 0x2a,
};
static const unsigned char kat461_addin0[] = {0};
static const unsigned char kat461_addin1[] = {0};
static const unsigned char kat461_retbits[] = {
    0xf3, 0x26, 0x9b, 0x27, 0xab, 0x4f, 0x7e, 0xb1, 0xff, 0x07, 0x0e, 0xb2,
    0xbb, 0x87, 0x94, 0xb4, 0xf2, 0x2b, 0x0e, 0xec, 0x45, 0xc2, 0xad, 0x1b,
    0xbf, 0x2c, 0x5c, 0x0b, 0x7c, 0xd3, 0x2d, 0x4d, 0xe0, 0x68, 0xf3, 0xe8,
    0x28, 0x2c, 0x3d, 0xcc, 0x35, 0xc9, 0xf4, 0x69, 0xef, 0x0a, 0x9d, 0x31,
    0x75, 0x6a, 0xed, 0xce, 0x00, 0xcf, 0xae, 0xb3, 0x09, 0x59, 0x4b, 0x81,
    0x64, 0x91, 0xcc, 0xad,
};
static const struct drbg_kat_no_reseed kat461_t = {
    10, kat461_entropyin, kat461_nonce, kat461_persstr,
    kat461_addin0, kat461_addin1, kat461_retbits
};
static const struct drbg_kat kat461 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat461_t
};

static const unsigned char kat462_entropyin[] = {
    0xa2, 0x2d, 0x4f, 0xac, 0x12, 0x66, 0xce, 0x8f, 0x7c, 0xba, 0xff, 0x6b,
    0x9d, 0x47, 0x42, 0x4a, 0x10, 0xf2, 0x69, 0xe1, 0x6b, 0x8c, 0x52, 0x78,
};
static const unsigned char kat462_nonce[] = {
    0x7a, 0x04, 0xd4, 0x04, 0x45, 0x33, 0xe9, 0x83, 0x92, 0x60, 0x23, 0xc5,
    0xa8, 0xc0, 0xb1, 0x0b,
};
static const unsigned char kat462_persstr[] = {
    0xa3, 0x75, 0xcb, 0x70, 0xba, 0x80, 0x06, 0xc7, 0x4c, 0xcd, 0x06, 0xa9,
    0xc8, 0xe4, 0x1e, 0xad, 0xc4, 0x45, 0x33, 0x1e, 0x14, 0xb8, 0xe1, 0x95,
    0xf4, 0xa3, 0xfa, 0xb0, 0xc1, 0xdf, 0x6c, 0x96,
};
static const unsigned char kat462_addin0[] = {0};
static const unsigned char kat462_addin1[] = {0};
static const unsigned char kat462_retbits[] = {
    0xb3, 0xf9, 0xa3, 0x93, 0x85, 0x61, 0x6f, 0x68, 0x51, 0xbe, 0xb7, 0xc3,
    0x77, 0x09, 0xac, 0x1f, 0x30, 0x66, 0x7d, 0xf1, 0x1c, 0x9d, 0x04, 0x7d,
    0xfc, 0xd5, 0xc7, 0x80, 0x3e, 0x93, 0x0b, 0xf3, 0x84, 0x07, 0x0e, 0x35,
    0x0f, 0x1d, 0x0b, 0x8f, 0xaa, 0x39, 0xf2, 0x9c, 0x98, 0xd9, 0xdf, 0x59,
    0xd3, 0x5f, 0xfb, 0x52, 0x0c, 0x8b, 0xc9, 0x0b, 0xe2, 0x2f, 0xc7, 0x7d,
    0x4d, 0x67, 0xf1, 0x34,
};
static const struct drbg_kat_no_reseed kat462_t = {
    11, kat462_entropyin, kat462_nonce, kat462_persstr,
    kat462_addin0, kat462_addin1, kat462_retbits
};
static const struct drbg_kat kat462 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat462_t
};

static const unsigned char kat463_entropyin[] = {
    0x48, 0xc4, 0xfc, 0xab, 0x28, 0x5f, 0x5b, 0xb3, 0xed, 0xaa, 0x34, 0x8c,
    0x45, 0x06, 0xd2, 0x36, 0xbb, 0x9f, 0xbb, 0x29, 0x7d, 0xd4, 0xf3, 0x28,
};
static const unsigned char kat463_nonce[] = {
    0x8a, 0xda, 0x0b, 0x51, 0x6d, 0xc4, 0x9a, 0xc8, 0xc3, 0x44, 0xf3, 0xb7,
    0x28, 0x27, 0x40, 0x18,
};
static const unsigned char kat463_persstr[] = {
    0x52, 0x8e, 0xed, 0x74, 0x30, 0x85, 0xd3, 0xd4, 0x16, 0xbd, 0x06, 0x98,
    0x4d, 0xe2, 0x44, 0x16, 0x1b, 0xde, 0xe0, 0xbf, 0xb5, 0x98, 0xa7, 0x47,
    0xf2, 0x9c, 0x37, 0xfa, 0x94, 0x12, 0x97, 0x6f,
};
static const unsigned char kat463_addin0[] = {0};
static const unsigned char kat463_addin1[] = {0};
static const unsigned char kat463_retbits[] = {
    0x2b, 0x47, 0x3e, 0x1c, 0x19, 0x69, 0x3b, 0x30, 0x8d, 0xef, 0x41, 0xf0,
    0xcd, 0x99, 0xcd, 0xbf, 0x33, 0x1c, 0x7d, 0x7d, 0x21, 0x5c, 0xd5, 0x02,
    0x8e, 0x36, 0x19, 0xad, 0x91, 0xc8, 0xe7, 0xd5, 0xe0, 0x24, 0x80, 0x48,
    0x29, 0xf4, 0xb2, 0x90, 0x5a, 0x33, 0x54, 0x49, 0x96, 0x19, 0x6d, 0x5c,
    0xaf, 0xab, 0xd2, 0x27, 0x0d, 0x3c, 0xdd, 0x66, 0x89, 0xdd, 0x12, 0x46,
    0x24, 0x68, 0xad, 0x50,
};
static const struct drbg_kat_no_reseed kat463_t = {
    12, kat463_entropyin, kat463_nonce, kat463_persstr,
    kat463_addin0, kat463_addin1, kat463_retbits
};
static const struct drbg_kat kat463 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat463_t
};

static const unsigned char kat464_entropyin[] = {
    0x60, 0x7f, 0xf9, 0x9f, 0xac, 0xb2, 0x69, 0xd5, 0xe8, 0x5b, 0xaf, 0x03,
    0xdd, 0x93, 0xde, 0x5e, 0xa5, 0x43, 0x16, 0x8e, 0x69, 0x39, 0xd9, 0x72,
};
static const unsigned char kat464_nonce[] = {
    0xf8, 0x49, 0xa0, 0xfa, 0x95, 0xd0, 0x4b, 0x74, 0x1e, 0x46, 0xd1, 0x8f,
    0x06, 0x12, 0xee, 0xb7,
};
static const unsigned char kat464_persstr[] = {
    0x95, 0xfe, 0x53, 0xeb, 0x4d, 0xb0, 0x9a, 0xa8, 0x5b, 0x94, 0x06, 0x53,
    0x0c, 0x9e, 0x99, 0x4c, 0x69, 0x5b, 0x86, 0xc9, 0xf0, 0x4e, 0xe5, 0x3f,
    0x7f, 0x49, 0x2c, 0x4d, 0x75, 0x82, 0x02, 0xad,
};
static const unsigned char kat464_addin0[] = {0};
static const unsigned char kat464_addin1[] = {0};
static const unsigned char kat464_retbits[] = {
    0xca, 0xe0, 0x4e, 0xd6, 0x42, 0xfa, 0x6d, 0xd8, 0x75, 0x6a, 0x97, 0xb6,
    0xa3, 0x24, 0x94, 0xd6, 0xb9, 0x5d, 0x5a, 0x52, 0x65, 0x41, 0x2d, 0x27,
    0xa0, 0xda, 0x9a, 0x70, 0xad, 0x91, 0x4e, 0xd3, 0x1a, 0x67, 0x4c, 0x6e,
    0xd9, 0x00, 0xbe, 0x9a, 0xfa, 0x49, 0x3f, 0x8e, 0x14, 0x1a, 0x08, 0xab,
    0xa8, 0xda, 0xb7, 0x5c, 0xba, 0x57, 0x3c, 0x46, 0x63, 0x7a, 0x9f, 0xaf,
    0x4c, 0x6a, 0x71, 0x61,
};
static const struct drbg_kat_no_reseed kat464_t = {
    13, kat464_entropyin, kat464_nonce, kat464_persstr,
    kat464_addin0, kat464_addin1, kat464_retbits
};
static const struct drbg_kat kat464 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat464_t
};

static const unsigned char kat465_entropyin[] = {
    0x99, 0xe2, 0xdf, 0x3b, 0xd5, 0x50, 0x08, 0xe9, 0xce, 0xa0, 0xd0, 0xea,
    0x58, 0xc3, 0x02, 0x2e, 0xff, 0x1d, 0x1e, 0x48, 0x9c, 0xd6, 0xf8, 0xee,
};
static const unsigned char kat465_nonce[] = {
    0xb3, 0x90, 0x7b, 0x88, 0x20, 0xc9, 0xf2, 0x29, 0x79, 0xb6, 0x63, 0x29,
    0x2f, 0x7a, 0xc7, 0x88,
};
static const unsigned char kat465_persstr[] = {
    0x28, 0xbe, 0xb6, 0x61, 0x18, 0xcb, 0xce, 0x5e, 0xed, 0x82, 0x5b, 0xdc,
    0xa9, 0xc8, 0xb6, 0x5b, 0x6c, 0x6a, 0x49, 0x8d, 0xb6, 0x22, 0x6e, 0x72,
    0xe1, 0xfc, 0x01, 0xe7, 0x1f, 0xd2, 0x2c, 0x82,
};
static const unsigned char kat465_addin0[] = {0};
static const unsigned char kat465_addin1[] = {0};
static const unsigned char kat465_retbits[] = {
    0x43, 0xeb, 0x72, 0x1f, 0x36, 0x69, 0xf9, 0x18, 0xfd, 0x84, 0x1d, 0x79,
    0xe2, 0xa2, 0x36, 0x48, 0xf8, 0x8c, 0x98, 0x69, 0x26, 0x0b, 0x62, 0x4b,
    0x8a, 0x33, 0x81, 0xbd, 0xc1, 0xd2, 0xb1, 0x29, 0x6b, 0x6b, 0xa1, 0xe0,
    0x75, 0xcd, 0x4d, 0x3e, 0xe0, 0x6a, 0x8b, 0x4f, 0x98, 0x6c, 0x00, 0xbb,
    0x2b, 0x9c, 0x5a, 0x1e, 0xb2, 0x3b, 0x82, 0xce, 0x08, 0x1f, 0x51, 0x91,
    0x3e, 0x46, 0xa4, 0xdc,
};
static const struct drbg_kat_no_reseed kat465_t = {
    14, kat465_entropyin, kat465_nonce, kat465_persstr,
    kat465_addin0, kat465_addin1, kat465_retbits
};
static const struct drbg_kat kat465 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 0, 64, &kat465_t
};

static const unsigned char kat466_entropyin[] = {
    0x43, 0xf2, 0x47, 0xa2, 0xee, 0x1c, 0xb5, 0x94, 0x3f, 0x0a, 0x4a, 0x69,
    0x42, 0x35, 0x5b, 0x16, 0x33, 0xab, 0x82, 0xd0, 0xcf, 0x60, 0xd4, 0x33,
};
static const unsigned char kat466_nonce[] = {
    0x81, 0x81, 0xab, 0x27, 0xf9, 0x3f, 0x66, 0x23, 0x25, 0xf8, 0xcd, 0xf7,
    0x7b, 0xd4, 0x63, 0x99,
};
static const unsigned char kat466_persstr[] = {
    0xd3, 0x93, 0x50, 0x8a, 0x6d, 0xa0, 0x49, 0xa6, 0x46, 0xaa, 0xc8, 0xc9,
    0xcf, 0x8f, 0xf5, 0x3a, 0x70, 0xf6, 0x64, 0x5a, 0x67, 0xec, 0x46, 0x6d,
    0x6e, 0xa6, 0x24, 0x1e, 0x0f, 0xac, 0xfa, 0xd9,
};
static const unsigned char kat466_addin0[] = {
    0xdc, 0x3e, 0x8d, 0x4c, 0x2b, 0xd4, 0xae, 0xe2, 0x78, 0xde, 0x3f, 0x99,
    0x8c, 0xa6, 0x03, 0xec, 0x50, 0x23, 0x32, 0xb6, 0xa1, 0xcb, 0x6c, 0x82,
    0x85, 0xb8, 0x4c, 0x84, 0xba, 0x54, 0xd7, 0x32,
};
static const unsigned char kat466_addin1[] = {
    0x1d, 0x0d, 0x64, 0xb5, 0xa5, 0x0d, 0xe0, 0x3e, 0xe2, 0x65, 0x4f, 0xa5,
    0x90, 0x00, 0xfb, 0xff, 0x87, 0xb4, 0xb8, 0xb3, 0x2e, 0x90, 0xa4, 0xde,
    0xe9, 0x0c, 0x9a, 0x76, 0x12, 0xea, 0xa5, 0x75,
};
static const unsigned char kat466_retbits[] = {
    0xd6, 0xd7, 0xc6, 0x4d, 0xd9, 0x22, 0xf9, 0xc7, 0xf3, 0xf5, 0x38, 0xc4,
    0xcb, 0xc7, 0x91, 0xda, 0x2d, 0xd1, 0x3f, 0x27, 0x81, 0x35, 0x22, 0x5c,
    0xd7, 0xce, 0x36, 0x73, 0x8c, 0xe2, 0x72, 0xf3, 0x51, 0x55, 0xd4, 0xc2,
    0x69, 0x6e, 0x7e, 0x10, 0xa8, 0xf4, 0x8c, 0x73, 0x60, 0xa1, 0xb3, 0x1a,
    0x30, 0x03, 0x7f, 0x66, 0x04, 0xff, 0xb7, 0xbc, 0x7b, 0x29, 0xef, 0x0e,
    0xf5, 0xac, 0xa8, 0x4a,
};
static const struct drbg_kat_no_reseed kat466_t = {
    0, kat466_entropyin, kat466_nonce, kat466_persstr,
    kat466_addin0, kat466_addin1, kat466_retbits
};
static const struct drbg_kat kat466 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat466_t
};

static const unsigned char kat467_entropyin[] = {
    0xb4, 0xe8, 0xfb, 0xf6, 0x33, 0x51, 0x38, 0xa2, 0x4e, 0xfb, 0x12, 0x0e,
    0xe9, 0xff, 0x7f, 0x99, 0xe1, 0x94, 0xa1, 0x7e, 0x64, 0x49, 0x62, 0xe0,
};
static const unsigned char kat467_nonce[] = {
    0x8e, 0x4d, 0xad, 0x78, 0x74, 0x19, 0x52, 0xb3, 0x4a, 0x27, 0xbb, 0x85,
    0xce, 0x2b, 0xeb, 0xd7,
};
static const unsigned char kat467_persstr[] = {
    0x44, 0x08, 0xdc, 0x6f, 0x19, 0xad, 0x7c, 0xab, 0x07, 0x60, 0x56, 0x0c,
    0x21, 0x75, 0x32, 0xab, 0x7d, 0xd5, 0x65, 0x81, 0x20, 0xa9, 0x4a, 0x60,
    0x3f, 0xe3, 0xb6, 0x2f, 0x8f, 0xc5, 0x58, 0x38,
};
static const unsigned char kat467_addin0[] = {
    0x2f, 0x18, 0xe4, 0x2c, 0x7a, 0x50, 0x80, 0x1b, 0xf5, 0x13, 0x5a, 0x0d,
    0x8f, 0xce, 0x0a, 0x9b, 0x80, 0x2b, 0xea, 0x0b, 0xef, 0x4c, 0x7f, 0x35,
    0xe4, 0xd8, 0x94, 0x82, 0x11, 0x54, 0x72, 0x21,
};
static const unsigned char kat467_addin1[] = {
    0xde, 0x9e, 0x72, 0xfe, 0x41, 0x30, 0xfb, 0x75, 0xc8, 0xca, 0xed, 0x41,
    0x5e, 0x14, 0xbd, 0x0b, 0x92, 0x2b, 0x9b, 0xef, 0x8b, 0xe1, 0x44, 0xf4,
    0x36, 0xde, 0xfa, 0xda, 0x10, 0x96, 0xf9, 0x81,
};
static const unsigned char kat467_retbits[] = {
    0x85, 0x7a, 0x93, 0x1d, 0x68, 0xe2, 0xf1, 0xf4, 0x1d, 0x9e, 0x92, 0xa7,
    0x4f, 0x57, 0x8b, 0xcc, 0x86, 0x08, 0x26, 0x04, 0x03, 0x2f, 0x3b, 0xca,
    0x15, 0x2e, 0x70, 0xcb, 0xb4, 0xc0, 0xfa, 0xda, 0xdc, 0x26, 0xbd, 0x26,
    0x67, 0x19, 0x4a, 0x6c, 0x72, 0x7e, 0x32, 0xec, 0xc1, 0xb2, 0xfc, 0x63,
    0x4b, 0x5b, 0xdd, 0x2d, 0x90, 0x06, 0xe6, 0xfa, 0xaf, 0xfc, 0x23, 0x34,
    0x53, 0xda, 0x47, 0x25,
};
static const struct drbg_kat_no_reseed kat467_t = {
    1, kat467_entropyin, kat467_nonce, kat467_persstr,
    kat467_addin0, kat467_addin1, kat467_retbits
};
static const struct drbg_kat kat467 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat467_t
};

static const unsigned char kat468_entropyin[] = {
    0xd3, 0xd9, 0x0c, 0x95, 0x3e, 0x60, 0xcc, 0xe2, 0x9d, 0x49, 0x13, 0xb6,
    0xba, 0x67, 0x66, 0xf9, 0xd8, 0x5f, 0x1f, 0xa0, 0x54, 0xf2, 0x35, 0x9c,
};
static const unsigned char kat468_nonce[] = {
    0x90, 0x23, 0xad, 0x28, 0xe8, 0xc5, 0xf2, 0xe4, 0xbe, 0x8f, 0xb2, 0xd8,
    0x6f, 0x62, 0x25, 0x05,
};
static const unsigned char kat468_persstr[] = {
    0xd4, 0x9d, 0xb6, 0x64, 0xdc, 0xa3, 0xc4, 0x0e, 0x69, 0x12, 0x7b, 0x04,
    0xf8, 0xec, 0x7f, 0xb0, 0x07, 0x73, 0xb8, 0xdd, 0xdd, 0xb9, 0x1a, 0xe2,
    0xe1, 0xe3, 0xc8, 0x4c, 0x68, 0xff, 0x98, 0x55,
};
static const unsigned char kat468_addin0[] = {
    0xee, 0x10, 0xa8, 0x06, 0x78, 0x41, 0xb5, 0xb4, 0xa3, 0xa6, 0xf3, 0x49,
    0x67, 0x04, 0xe9, 0x74, 0x78, 0xa4, 0x4c, 0xdc, 0xc8, 0x57, 0xe4, 0xf2,
    0x96, 0x80, 0xf6, 0x69, 0xec, 0x5a, 0x83, 0xd1,
};
static const unsigned char kat468_addin1[] = {
    0xef, 0xbb, 0x9a, 0x69, 0x6e, 0xfc, 0x90, 0xa7, 0x74, 0xee, 0x87, 0xb7,
    0xed, 0x43, 0x45, 0x03, 0x79, 0x2c, 0x1a, 0x29, 0xbe, 0xec, 0xe9, 0x34,
    0x2e, 0x0b, 0x57, 0x24, 0xc3, 0xf5, 0xbc, 0xc6,
};
static const unsigned char kat468_retbits[] = {
    0x28, 0x4f, 0xec, 0xb8, 0x33, 0xa6, 0xe4, 0x18, 0xe9, 0xe5, 0x16, 0xc1,
    0x71, 0xfc, 0xe8, 0x52, 0xe7, 0x02, 0xa4, 0x48, 0x86, 0x74, 0x77, 0x1c,
    0x91, 0x3e, 0x85, 0x16, 0x9f, 0x19, 0x5b, 0xb7, 0x99, 0xf0, 0x19, 0xac,
    0x1b, 0x55, 0xa7, 0xe3, 0x93, 0x7f, 0xa7, 0x70, 0x33, 0x07, 0x22, 0xa8,
    0x01, 0x51, 0x13, 0x84, 0xd8, 0x56, 0xfb, 0x19, 0x9b, 0x76, 0xe2, 0x64,
    0xca, 0xa0, 0x6b, 0x6d,
};
static const struct drbg_kat_no_reseed kat468_t = {
    2, kat468_entropyin, kat468_nonce, kat468_persstr,
    kat468_addin0, kat468_addin1, kat468_retbits
};
static const struct drbg_kat kat468 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat468_t
};

static const unsigned char kat469_entropyin[] = {
    0x90, 0x7b, 0xc8, 0xc3, 0xa8, 0xc6, 0x94, 0x64, 0x69, 0xc4, 0x88, 0xbf,
    0x18, 0xa8, 0xe6, 0x7f, 0x7c, 0xc0, 0x45, 0xff, 0x4e, 0xd6, 0x9d, 0x5f,
};
static const unsigned char kat469_nonce[] = {
    0xc2, 0xe0, 0xcb, 0xb2, 0x6a, 0x48, 0x23, 0x2d, 0xd3, 0x23, 0x10, 0xb1,
    0xce, 0xb3, 0x2e, 0x9b,
};
static const unsigned char kat469_persstr[] = {
    0xab, 0x67, 0xf1, 0xfd, 0x53, 0xe1, 0x2c, 0x36, 0x47, 0x22, 0xd6, 0xb5,
    0x47, 0x5d, 0x55, 0x55, 0xa1, 0x79, 0x51, 0xeb, 0x10, 0xd2, 0x00, 0x55,
    0xca, 0x3e, 0x40, 0x86, 0xbb, 0xde, 0xd2, 0xdb,
};
static const unsigned char kat469_addin0[] = {
    0xac, 0x77, 0x29, 0x46, 0x2a, 0x01, 0x24, 0x14, 0x36, 0xc3, 0xbe, 0x8e,
    0x85, 0x40, 0x18, 0x73, 0x3a, 0x84, 0x25, 0x4f, 0x1e, 0x0a, 0x23, 0x43,
    0xca, 0x69, 0x05, 0x8c, 0x54, 0x71, 0x6f, 0x1b,
};
static const unsigned char kat469_addin1[] = {
    0x85, 0xd4, 0x36, 0x47, 0xb3, 0x6e, 0x29, 0x4a, 0xbd, 0xdb, 0x01, 0x36,
    0xb6, 0x55, 0x0d, 0xf3, 0xcb, 0x31, 0x4c, 0x88, 0x92, 0xfe, 0x27, 0x9b,
    0x96, 0x5f, 0x4e, 0x89, 0x31, 0xed, 0xf2, 0xbf,
};
static const unsigned char kat469_retbits[] = {
    0xad, 0x8a, 0x6f, 0xd4, 0xc3, 0xc8, 0x5c, 0x4d, 0xf1, 0x87, 0xf0, 0x35,
    0x43, 0x32, 0x50, 0xb6, 0x50, 0xec, 0x96, 0xad, 0x96, 0x2c, 0x36, 0x5a,
    0x8b, 0x14, 0x42, 0x5c, 0x4d, 0x37, 0xb0, 0xc4, 0xfd, 0x58, 0xd0, 0x0f,
    0x10, 0x94, 0xea, 0x76, 0x27, 0xac, 0xaf, 0x67, 0xb2, 0x37, 0x34, 0x46,
    0x83, 0xb5, 0x13, 0xe8, 0xf5, 0x24, 0xee, 0xc4, 0x96, 0xea, 0x90, 0x7b,
    0x0c, 0xb8, 0x1c, 0x27,
};
static const struct drbg_kat_no_reseed kat469_t = {
    3, kat469_entropyin, kat469_nonce, kat469_persstr,
    kat469_addin0, kat469_addin1, kat469_retbits
};
static const struct drbg_kat kat469 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat469_t
};

static const unsigned char kat470_entropyin[] = {
    0x9e, 0xc0, 0x4e, 0x7c, 0x20, 0xf7, 0x76, 0x63, 0xd3, 0x30, 0xa4, 0x3e,
    0xf1, 0x93, 0xf3, 0xa3, 0xba, 0xee, 0xc8, 0x9e, 0xc0, 0x86, 0x0b, 0x7c,
};
static const unsigned char kat470_nonce[] = {
    0x32, 0x67, 0xaa, 0xcc, 0x67, 0x60, 0x2d, 0x44, 0xb4, 0x06, 0x63, 0xf3,
    0xe7, 0x9d, 0x99, 0xa1,
};
static const unsigned char kat470_persstr[] = {
    0xad, 0xc1, 0x05, 0xb4, 0x10, 0x39, 0xe0, 0x58, 0x63, 0x5b, 0x67, 0x6b,
    0xf7, 0xc0, 0x7d, 0x00, 0x45, 0xf7, 0x5e, 0xbf, 0x64, 0x23, 0x91, 0xf5,
    0x68, 0x49, 0x99, 0xa5, 0xc2, 0xf3, 0xa4, 0xfd,
};
static const unsigned char kat470_addin0[] = {
    0xa5, 0xb3, 0x74, 0xce, 0x0a, 0x74, 0x9e, 0xd1, 0x20, 0xb7, 0xb3, 0xad,
    0xcc, 0x5e, 0xe9, 0x3d, 0xcc, 0xb2, 0xb0, 0x54, 0x55, 0xc6, 0x26, 0xdf,
    0x03, 0xd1, 0x86, 0xc9, 0x65, 0xbb, 0x6b, 0xbc,
};
static const unsigned char kat470_addin1[] = {
    0x1e, 0xb5, 0x92, 0x13, 0x3f, 0x72, 0xec, 0x56, 0xc3, 0xda, 0xb6, 0xa5,
    0x37, 0x89, 0x72, 0x0b, 0xd2, 0xeb, 0xc1, 0xbd, 0x1f, 0x74, 0x6f, 0x7a,
    0x05, 0x49, 0x9d, 0xdb, 0x7d, 0x9c, 0x35, 0xa9,
};
static const unsigned char kat470_retbits[] = {
    0x69, 0x9f, 0x32, 0xe3, 0x06, 0x6a, 0xc7, 0xe1, 0xf9, 0x3c, 0x8e, 0x0b,
    0x41, 0xdd, 0xea, 0x89, 0x8d, 0xf4, 0x22, 0x89, 0xf8, 0x63, 0x2b, 0x97,
    0x6f, 0x5e, 0x64, 0xea, 0xa8, 0x23, 0xff, 0x28, 0xdf, 0xd9, 0xd4, 0xe1,
    0x24, 0x44, 0x9a, 0x7f, 0xfe, 0x1a, 0x60, 0x3e, 0x04, 0x07, 0xb1, 0x3f,
    0x7b, 0x44, 0xb4, 0xa5, 0x01, 0x02, 0x5e, 0x12, 0x24, 0xd8, 0x4b, 0xd2,
    0xe7, 0x80, 0xa6, 0x44,
};
static const struct drbg_kat_no_reseed kat470_t = {
    4, kat470_entropyin, kat470_nonce, kat470_persstr,
    kat470_addin0, kat470_addin1, kat470_retbits
};
static const struct drbg_kat kat470 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat470_t
};

static const unsigned char kat471_entropyin[] = {
    0x9d, 0xa2, 0x56, 0xb4, 0xe1, 0x5b, 0x21, 0xd4, 0x8a, 0x9f, 0x59, 0x25,
    0xc1, 0x1d, 0x27, 0x5c, 0x8b, 0x45, 0x8a, 0xef, 0x75, 0x0e, 0xa2, 0xb5,
};
static const unsigned char kat471_nonce[] = {
    0x95, 0xa9, 0xbf, 0x02, 0xb8, 0xb9, 0xa9, 0x8f, 0x71, 0x20, 0xa8, 0x66,
    0x76, 0x41, 0x03, 0x54,
};
static const unsigned char kat471_persstr[] = {
    0xf6, 0xe8, 0xc3, 0x6b, 0x41, 0xd3, 0xd0, 0xd7, 0x47, 0x13, 0x31, 0x98,
    0xa0, 0x30, 0x3c, 0x8a, 0x9a, 0x9d, 0xe2, 0xaa, 0x78, 0x46, 0x39, 0xaf,
    0x12, 0x2b, 0xac, 0x75, 0xd6, 0xeb, 0x85, 0x35,
};
static const unsigned char kat471_addin0[] = {
    0x62, 0xda, 0x4a, 0xa2, 0x14, 0x44, 0x26, 0xd1, 0xee, 0x35, 0x81, 0xa8,
    0xc0, 0xf9, 0x1b, 0x84, 0xe1, 0x7a, 0xce, 0x43, 0xf5, 0x66, 0xb2, 0x57,
    0xd8, 0x5c, 0xf9, 0xf9, 0xb1, 0xdb, 0x02, 0x1d,
};
static const unsigned char kat471_addin1[] = {
    0xd7, 0x1f, 0xfa, 0x97, 0xfd, 0x0a, 0x7a, 0x80, 0xb9, 0x19, 0xc2, 0x1d,
    0xd7, 0xaf, 0xfc, 0x57, 0x77, 0x00, 0xac, 0xa7, 0xdc, 0xa6, 0x31, 0x1d,
    0x3b, 0x90, 0x64, 0xf5, 0x02, 0x08, 0x43, 0x5b,
};
static const unsigned char kat471_retbits[] = {
    0x12, 0x09, 0x92, 0xa8, 0x2a, 0xd8, 0xfe, 0xd5, 0xad, 0x49, 0xdb, 0x7b,
    0x75, 0x6b, 0x94, 0x18, 0x5e, 0x15, 0xb1, 0xfb, 0x53, 0xaa, 0xa6, 0x4e,
    0x6c, 0x7b, 0x9a, 0xc1, 0x44, 0x87, 0x30, 0x7d, 0x0f, 0xbb, 0x5f, 0x51,
    0x95, 0x27, 0xe4, 0x68, 0xfd, 0x97, 0x82, 0x91, 0x38, 0x1f, 0x95, 0x92,
    0xcc, 0xf5, 0x3d, 0xf5, 0x87, 0x58, 0x6c, 0xa0, 0xfd, 0xce, 0x0c, 0x81,
    0x31, 0x0d, 0xbb, 0xb3,
};
static const struct drbg_kat_no_reseed kat471_t = {
    5, kat471_entropyin, kat471_nonce, kat471_persstr,
    kat471_addin0, kat471_addin1, kat471_retbits
};
static const struct drbg_kat kat471 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat471_t
};

static const unsigned char kat472_entropyin[] = {
    0xa8, 0x13, 0xae, 0x3d, 0x13, 0x8b, 0xb9, 0x76, 0x87, 0x59, 0xe2, 0x59,
    0x0a, 0xdd, 0xea, 0x63, 0x90, 0x26, 0x64, 0x05, 0xb8, 0xf5, 0x9b, 0x68,
};
static const unsigned char kat472_nonce[] = {
    0x52, 0x49, 0xee, 0x76, 0xa7, 0xe7, 0x92, 0xe6, 0xa1, 0xce, 0x5c, 0xad,
    0xe7, 0x89, 0xb8, 0x46,
};
static const unsigned char kat472_persstr[] = {
    0x7b, 0x4b, 0x71, 0x74, 0x7d, 0x17, 0x90, 0x2f, 0x3f, 0xbb, 0xeb, 0x3a,
    0x60, 0x00, 0x5c, 0x2a, 0x30, 0xe9, 0x02, 0xf6, 0x96, 0xa8, 0xd2, 0xd8,
    0xff, 0x68, 0x28, 0xd7, 0x55, 0x1d, 0x87, 0x49,
};
static const unsigned char kat472_addin0[] = {
    0x6d, 0xdb, 0x92, 0x55, 0x50, 0x8d, 0x29, 0xaf, 0x7a, 0x99, 0x8e, 0x14,
    0x6d, 0x82, 0x06, 0xbe, 0xaf, 0x82, 0xc1, 0xbb, 0x7d, 0x42, 0x3d, 0x92,
    0x60, 0x5e, 0x69, 0xa6, 0xa0, 0x31, 0x80, 0xe4,
};
static const unsigned char kat472_addin1[] = {
    0x84, 0x85, 0xf2, 0x57, 0xd4, 0x4a, 0x82, 0xbf, 0x92, 0x00, 0x13, 0xcc,
    0x8b, 0x83, 0x21, 0xa3, 0x57, 0xd1, 0x93, 0x3b, 0x64, 0x39, 0x5c, 0x12,
    0x55, 0x40, 0x90, 0x8d, 0xa9, 0xea, 0x9e, 0x99,
};
static const unsigned char kat472_retbits[] = {
    0x52, 0xfb, 0x04, 0x00, 0xf8, 0xdf, 0xcf, 0x69, 0x4f, 0x0d, 0xaa, 0xda,
    0x7b, 0xa7, 0x81, 0xe0, 0x9a, 0xf5, 0xe4, 0x8f, 0xcf, 0xcb, 0xe7, 0x18,
    0x05, 0x2b, 0x46, 0xa3, 0x89, 0xdf, 0x71, 0x3a, 0xb3, 0xaf, 0x71, 0x69,
    0x9c, 0x41, 0xd4, 0x1e, 0x65, 0x42, 0xb8, 0x7c, 0xd9, 0xbe, 0xc4, 0xce,
    0x0f, 0xd2, 0x92, 0xae, 0x52, 0xea, 0xd7, 0x8c, 0xf5, 0x2e, 0x0c, 0xac,
    0x5c, 0xcd, 0x4f, 0x01,
};
static const struct drbg_kat_no_reseed kat472_t = {
    6, kat472_entropyin, kat472_nonce, kat472_persstr,
    kat472_addin0, kat472_addin1, kat472_retbits
};
static const struct drbg_kat kat472 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat472_t
};

static const unsigned char kat473_entropyin[] = {
    0x67, 0xe4, 0x44, 0x71, 0xbc, 0x65, 0x6e, 0xec, 0xee, 0x78, 0x05, 0x38,
    0x0e, 0x98, 0x49, 0xf9, 0xe3, 0xf4, 0xf2, 0xf4, 0xac, 0xba, 0x52, 0x05,
};
static const unsigned char kat473_nonce[] = {
    0x60, 0x18, 0xf7, 0xe1, 0xf9, 0x43, 0xfd, 0xcf, 0x7f, 0xc9, 0x59, 0x5e,
    0xd0, 0xa0, 0x81, 0x07,
};
static const unsigned char kat473_persstr[] = {
    0xff, 0x0f, 0x82, 0xbd, 0x94, 0xfa, 0x8c, 0x65, 0x54, 0xe0, 0x5c, 0x69,
    0x57, 0x97, 0x3b, 0xb8, 0xc4, 0xf3, 0xbd, 0x3e, 0x88, 0x57, 0xab, 0xfc,
    0xbc, 0xde, 0x87, 0xc5, 0x84, 0xf7, 0x56, 0x11,
};
static const unsigned char kat473_addin0[] = {
    0xf0, 0x9d, 0xfa, 0xdc, 0xbd, 0xa7, 0x19, 0x08, 0xda, 0x9d, 0xd7, 0xbb,
    0x7a, 0x9e, 0x16, 0xa3, 0xc2, 0x5e, 0x30, 0x20, 0x50, 0x23, 0x5e, 0xaf,
    0xd9, 0x43, 0x2a, 0x8a, 0x78, 0x78, 0xc1, 0x0d,
};
static const unsigned char kat473_addin1[] = {
    0x1f, 0xc8, 0xb6, 0x0f, 0xc0, 0x30, 0xf7, 0xc4, 0x17, 0x7a, 0x51, 0x47,
    0x07, 0x15, 0x13, 0xfc, 0x2f, 0x17, 0x46, 0x59, 0xb0, 0x12, 0x88, 0x9a,
    0x0c, 0x58, 0x3b, 0xed, 0x7c, 0x38, 0x18, 0xb7,
};
static const unsigned char kat473_retbits[] = {
    0x35, 0xb2, 0xbc, 0x5f, 0x59, 0x24, 0x28, 0xd6, 0x1b, 0xb7, 0xe0, 0x62,
    0xbf, 0x96, 0x30, 0x31, 0x02, 0x11, 0x8a, 0xc5, 0xb1, 0x66, 0x50, 0x8a,
    0xd5, 0xcd, 0x22, 0x81, 0xf5, 0x9b, 0x38, 0x06, 0xbe, 0x11, 0x8e, 0x97,
    0x88, 0x7c, 0x57, 0x32, 0xfd, 0x96, 0xc7, 0xb1, 0x71, 0x43, 0x17, 0x59,
    0x7a, 0xaf, 0x20, 0xa1, 0xa6, 0xdd, 0xf7, 0x57, 0x8e, 0xfb, 0x74, 0x36,
    0x98, 0x0f, 0x99, 0x38,
};
static const struct drbg_kat_no_reseed kat473_t = {
    7, kat473_entropyin, kat473_nonce, kat473_persstr,
    kat473_addin0, kat473_addin1, kat473_retbits
};
static const struct drbg_kat kat473 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat473_t
};

static const unsigned char kat474_entropyin[] = {
    0x94, 0x32, 0x32, 0x27, 0x08, 0x8d, 0x2d, 0x5d, 0xdf, 0x36, 0xa1, 0xc4,
    0x24, 0x35, 0xe2, 0x05, 0xb9, 0xe7, 0xef, 0x19, 0xb6, 0xaf, 0xf0, 0x92,
};
static const unsigned char kat474_nonce[] = {
    0x9f, 0x96, 0x64, 0xc8, 0x7d, 0x7a, 0x43, 0x14, 0x2c, 0xe3, 0x79, 0x9d,
    0x17, 0x91, 0xc0, 0xc6,
};
static const unsigned char kat474_persstr[] = {
    0xa5, 0xf8, 0x3a, 0x66, 0x27, 0x7c, 0xb6, 0xc9, 0x4b, 0x20, 0x86, 0x0d,
    0xa0, 0x88, 0x74, 0xf3, 0x3c, 0x2a, 0x35, 0x34, 0xcf, 0xc7, 0xaa, 0x53,
    0xf4, 0x0b, 0xeb, 0x4f, 0xde, 0xff, 0x87, 0x0b,
};
static const unsigned char kat474_addin0[] = {
    0x41, 0x36, 0xba, 0xcf, 0x19, 0xb6, 0xed, 0xa7, 0x5f, 0x69, 0x92, 0x71,
    0xe5, 0xb3, 0x82, 0xf0, 0xf5, 0x8d, 0xd8, 0xb5, 0x9e, 0x0e, 0xaa, 0xc6,
    0xe4, 0x37, 0x99, 0x52, 0x5f, 0x7e, 0x3b, 0x76,
};
static const unsigned char kat474_addin1[] = {
    0x2c, 0xec, 0x57, 0xe8, 0xec, 0xb9, 0x34, 0x63, 0x80, 0xb3, 0x80, 0xbf,
    0x9e, 0xe7, 0x7d, 0x1b, 0x07, 0x7d, 0x8b, 0x79, 0x20, 0x4d, 0x73, 0xe0,
    0xe4, 0x46, 0xef, 0x38, 0xe1, 0xa9, 0xf2, 0x6e,
};
static const unsigned char kat474_retbits[] = {
    0x53, 0x11, 0x2d, 0x08, 0x4e, 0x01, 0x60, 0xa7, 0x62, 0xfd, 0x90, 0x8e,
    0x05, 0x40, 0xd5, 0x3b, 0x07, 0xcb, 0xe6, 0x48, 0x4b, 0xe9, 0xcd, 0xbd,
    0xa8, 0xf3, 0xa1, 0xc1, 0x4f, 0x79, 0xf2, 0xd2, 0xe0, 0x69, 0x39, 0x62,
    0xf1, 0x7e, 0x10, 0xed, 0xd0, 0x7c, 0x66, 0xd4, 0x2b, 0x40, 0xaf, 0xb4,
    0xf0, 0x2d, 0x69, 0x5e, 0x49, 0x84, 0x6f, 0x37, 0x6d, 0xcd, 0x2b, 0x5c,
    0xd2, 0x1c, 0x9c, 0xb6,
};
static const struct drbg_kat_no_reseed kat474_t = {
    8, kat474_entropyin, kat474_nonce, kat474_persstr,
    kat474_addin0, kat474_addin1, kat474_retbits
};
static const struct drbg_kat kat474 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat474_t
};

static const unsigned char kat475_entropyin[] = {
    0xe1, 0xce, 0x5b, 0x2c, 0x05, 0xf4, 0x5a, 0x89, 0x46, 0xf5, 0xb7, 0x94,
    0x92, 0x51, 0x7e, 0x33, 0x2b, 0xae, 0xda, 0x4d, 0x56, 0xc2, 0xab, 0xf2,
};
static const unsigned char kat475_nonce[] = {
    0xcd, 0x2d, 0x68, 0x47, 0x5b, 0xbf, 0x8a, 0xd8, 0x75, 0xdd, 0x30, 0x9a,
    0xd4, 0x28, 0xd6, 0x44,
};
static const unsigned char kat475_persstr[] = {
    0x74, 0xc6, 0xb1, 0x70, 0x79, 0x1c, 0x53, 0xaa, 0x1c, 0xd5, 0x2e, 0x38,
    0x2e, 0xee, 0x21, 0x84, 0x82, 0x27, 0x56, 0x7a, 0xf4, 0x0c, 0x55, 0x81,
    0xf9, 0x87, 0x48, 0xf4, 0xef, 0xc6, 0x05, 0xbd,
};
static const unsigned char kat475_addin0[] = {
    0x49, 0xd5, 0xa1, 0x77, 0x0f, 0x85, 0x13, 0x94, 0xb6, 0xe1, 0x49, 0x19,
    0x80, 0xbf, 0xff, 0x76, 0x21, 0xbc, 0x56, 0x10, 0xb5, 0x77, 0x7e, 0x71,
    0xe9, 0xcd, 0xa3, 0x0d, 0xac, 0x1e, 0xea, 0xeb,
};
static const unsigned char kat475_addin1[] = {
    0x4c, 0x5b, 0x15, 0x82, 0xc5, 0xf7, 0xdf, 0x4f, 0xdf, 0x91, 0x12, 0x53,
    0x30, 0xf8, 0xda, 0xdc, 0x36, 0x68, 0x77, 0xad, 0x8f, 0xea, 0xd6, 0x51,
    0xc7, 0xfb, 0xee, 0xd8, 0xee, 0x4e, 0x41, 0x78,
};
static const unsigned char kat475_retbits[] = {
    0x3c, 0x37, 0xe1, 0xec, 0x6c, 0x18, 0x0b, 0x32, 0xa9, 0x63, 0xbc, 0x04,
    0x40, 0x22, 0x1a, 0xc5, 0xc1, 0xc3, 0x3d, 0x89, 0x27, 0x2b, 0x9f, 0x69,
    0x7c, 0x75, 0x73, 0x7d, 0xd9, 0xcd, 0x2c, 0x74, 0x9c, 0x25, 0xf3, 0xc8,
    0x45, 0x53, 0xdf, 0xae, 0xf8, 0x6a, 0xf4, 0x61, 0x13, 0xe0, 0x6f, 0xe1,
    0xf6, 0xc8, 0x91, 0xc6, 0xa3, 0x90, 0x59, 0x0e, 0x16, 0x68, 0xce, 0xb7,
    0x5c, 0x95, 0xd3, 0xc9,
};
static const struct drbg_kat_no_reseed kat475_t = {
    9, kat475_entropyin, kat475_nonce, kat475_persstr,
    kat475_addin0, kat475_addin1, kat475_retbits
};
static const struct drbg_kat kat475 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat475_t
};

static const unsigned char kat476_entropyin[] = {
    0x7b, 0x9b, 0x3a, 0xaf, 0xb3, 0xda, 0x4f, 0xb6, 0x93, 0x5d, 0xe0, 0xa7,
    0x1a, 0xc9, 0xfb, 0x40, 0x95, 0xc6, 0x17, 0x35, 0x07, 0x2c, 0x28, 0xdb,
};
static const unsigned char kat476_nonce[] = {
    0x28, 0xbe, 0x54, 0xc1, 0x96, 0xbe, 0x07, 0x2f, 0xff, 0x92, 0xbc, 0x6c,
    0xaa, 0x8c, 0xcc, 0x02,
};
static const unsigned char kat476_persstr[] = {
    0x88, 0xea, 0x40, 0xac, 0xa2, 0xa8, 0xaa, 0xe4, 0x80, 0x85, 0x80, 0x74,
    0xbb, 0x14, 0x03, 0x69, 0x51, 0x66, 0x75, 0xbd, 0xff, 0xb7, 0xfa, 0xb4,
    0x31, 0x8c, 0xc5, 0x08, 0x35, 0x7b, 0x7b, 0xec,
};
static const unsigned char kat476_addin0[] = {
    0xc0, 0xe3, 0xe9, 0xd7, 0x00, 0x8d, 0x64, 0xfa, 0x30, 0x69, 0xaa, 0xb3,
    0x80, 0x5c, 0x0c, 0x81, 0x7c, 0x9a, 0x09, 0x71, 0x44, 0x6a, 0x8b, 0x53,
    0xd0, 0x0f, 0xca, 0xb0, 0x2a, 0x4c, 0xcb, 0x59,
};
static const unsigned char kat476_addin1[] = {
    0xdb, 0xac, 0x68, 0x1f, 0xbc, 0x3e, 0xb3, 0xc5, 0x73, 0x94, 0x52, 0xa5,
    0xe1, 0x50, 0xf5, 0x95, 0x65, 0x7c, 0x04, 0xf0, 0x7a, 0x52, 0x06, 0x20,
    0x52, 0xed, 0x1b, 0x19, 0xae, 0x53, 0xae, 0xd8,
};
static const unsigned char kat476_retbits[] = {
    0x4c, 0x4d, 0x6b, 0x05, 0x8f, 0xb1, 0x33, 0x44, 0xc4, 0x17, 0x5f, 0x02,
    0x39, 0x59, 0xa7, 0x70, 0x11, 0x52, 0xae, 0x35, 0x6e, 0x40, 0x09, 0x3f,
    0x70, 0x19, 0xd4, 0x7a, 0xe5, 0x1e, 0xe5, 0x19, 0x67, 0xe4, 0xfa, 0xbb,
    0x20, 0x9b, 0x67, 0xb2, 0x0a, 0x95, 0xa3, 0x37, 0xee, 0x8d, 0x4f, 0xe5,
    0xf6, 0x66, 0xcd, 0x08, 0xfd, 0xde, 0x3a, 0x74, 0xdc, 0x9f, 0xce, 0x0a,
    0x5a, 0xe0, 0xf9, 0x31,
};
static const struct drbg_kat_no_reseed kat476_t = {
    10, kat476_entropyin, kat476_nonce, kat476_persstr,
    kat476_addin0, kat476_addin1, kat476_retbits
};
static const struct drbg_kat kat476 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat476_t
};

static const unsigned char kat477_entropyin[] = {
    0x4c, 0x59, 0xc6, 0xc6, 0x7c, 0x48, 0x44, 0x17, 0x5e, 0x5f, 0x1a, 0x34,
    0xad, 0x9e, 0x08, 0x43, 0x4c, 0x20, 0x51, 0x51, 0xb7, 0x44, 0x7a, 0x01,
};
static const unsigned char kat477_nonce[] = {
    0x82, 0x8d, 0xea, 0x3b, 0xc0, 0x7a, 0xbd, 0x45, 0xdf, 0x79, 0x7c, 0xe7,
    0xdc, 0x8e, 0xdd, 0x77,
};
static const unsigned char kat477_persstr[] = {
    0xfb, 0x12, 0x99, 0x2f, 0x39, 0x4b, 0x21, 0x3b, 0x08, 0x7b, 0x32, 0x8f,
    0xb2, 0x74, 0xa2, 0x8f, 0x0a, 0x2d, 0x5d, 0xed, 0x77, 0x0f, 0xb8, 0xd6,
    0xfd, 0x7b, 0xf3, 0x82, 0x1d, 0x86, 0xee, 0x39,
};
static const unsigned char kat477_addin0[] = {
    0x5a, 0xe8, 0xe9, 0x30, 0x56, 0x82, 0x1c, 0x7f, 0xb0, 0x7b, 0x48, 0xc5,
    0xc6, 0xaa, 0xd9, 0xfc, 0xf3, 0x34, 0x34, 0xec, 0x1c, 0x93, 0x94, 0x29,
    0x30, 0xec, 0x06, 0xf0, 0x31, 0x1c, 0xd0, 0x9e,
};
static const unsigned char kat477_addin1[] = {
    0x09, 0x22, 0x28, 0x75, 0xe6, 0x9e, 0x81, 0x1b, 0x92, 0xba, 0x62, 0xc4,
    0x85, 0xa3, 0xd8, 0x10, 0xa8, 0xca, 0x89, 0xf0, 0xbe, 0x46, 0x8c, 0x7c,
    0x13, 0x61, 0x41, 0x2e, 0x19, 0x3b, 0x3c, 0x32,
};
static const unsigned char kat477_retbits[] = {
    0xf7, 0x1b, 0x7f, 0x6c, 0xcf, 0x4d, 0xc0, 0xbc, 0x0b, 0x89, 0x8b, 0xd3,
    0x8a, 0xb8, 0x10, 0x24, 0x76, 0xd0, 0xbb, 0x7e, 0x9a, 0x1a, 0xdd, 0x22,
    0x56, 0x82, 0x16, 0xb0, 0xd5, 0xc5, 0x2c, 0x4b, 0x17, 0x19, 0xa6, 0xf2,
    0x22, 0x3b, 0xc2, 0x4a, 0x52, 0x48, 0xc9, 0x15, 0xf6, 0x5d, 0x5e, 0x18,
    0x3f, 0xb9, 0xfc, 0xfa, 0xea, 0xd6, 0x5f, 0x1a, 0xe0, 0x4b, 0xee, 0xff,
    0x1b, 0xf4, 0x25, 0x96,
};
static const struct drbg_kat_no_reseed kat477_t = {
    11, kat477_entropyin, kat477_nonce, kat477_persstr,
    kat477_addin0, kat477_addin1, kat477_retbits
};
static const struct drbg_kat kat477 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat477_t
};

static const unsigned char kat478_entropyin[] = {
    0x81, 0xf4, 0xab, 0x51, 0x4e, 0x4e, 0xf1, 0xc8, 0x9f, 0x18, 0x61, 0x72,
    0x55, 0x39, 0x90, 0xaf, 0x49, 0xa3, 0xe0, 0x74, 0x72, 0x9b, 0xba, 0xec,
};
static const unsigned char kat478_nonce[] = {
    0x35, 0xd5, 0xbb, 0x6e, 0xff, 0x10, 0xd2, 0x70, 0x35, 0xe7, 0x88, 0x8f,
    0xd6, 0xc0, 0xc5, 0x70,
};
static const unsigned char kat478_persstr[] = {
    0xff, 0x5a, 0xa8, 0xcb, 0x94, 0x89, 0xf7, 0xfa, 0xd8, 0x82, 0x52, 0x2c,
    0x96, 0xa3, 0xc8, 0xca, 0xe8, 0x3a, 0x9b, 0x71, 0x13, 0xd0, 0xc1, 0xe4,
    0x9d, 0xa6, 0x81, 0x59, 0x8a, 0xe2, 0xfa, 0x56,
};
static const unsigned char kat478_addin0[] = {
    0x29, 0x41, 0x78, 0x55, 0x09, 0x14, 0x8b, 0x16, 0x81, 0x3f, 0x88, 0xf1,
    0xeb, 0x43, 0x0b, 0x28, 0x62, 0xe3, 0x03, 0xfd, 0x71, 0x6b, 0x71, 0x72,
    0xb1, 0x41, 0xb9, 0x99, 0x42, 0x3f, 0xa3, 0x0e,
};
static const unsigned char kat478_addin1[] = {
    0x3b, 0xeb, 0x13, 0xe4, 0x1b, 0x29, 0xc3, 0xf9, 0xe7, 0xc0, 0x4b, 0xd8,
    0xb9, 0x3a, 0xc2, 0xc8, 0xfb, 0xfe, 0xd0, 0x6e, 0xd3, 0x3b, 0x8e, 0xa9,
    0x31, 0x7f, 0xa4, 0xca, 0x3c, 0x25, 0x31, 0x01,
};
static const unsigned char kat478_retbits[] = {
    0x3c, 0x33, 0xd7, 0x78, 0xb1, 0xd6, 0x1c, 0x12, 0xce, 0x1f, 0x71, 0x18,
    0x0e, 0x8d, 0xda, 0xf4, 0xf3, 0x09, 0x8b, 0x09, 0x27, 0xad, 0x9e, 0x86,
    0x3d, 0x8f, 0x74, 0xfe, 0x87, 0x1e, 0xfc, 0xa1, 0xd6, 0x72, 0xb2, 0xd0,
    0x7c, 0xa5, 0x7a, 0x89, 0x11, 0x80, 0x62, 0xbb, 0x35, 0x49, 0x94, 0xc1,
    0xc5, 0xae, 0xa5, 0xab, 0x1b, 0xe2, 0xf9, 0x25, 0x71, 0xc0, 0x72, 0x01,
    0x97, 0x43, 0x5e, 0x73,
};
static const struct drbg_kat_no_reseed kat478_t = {
    12, kat478_entropyin, kat478_nonce, kat478_persstr,
    kat478_addin0, kat478_addin1, kat478_retbits
};
static const struct drbg_kat kat478 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat478_t
};

static const unsigned char kat479_entropyin[] = {
    0x6c, 0xce, 0xcf, 0xbe, 0x10, 0xea, 0xb1, 0xca, 0xa8, 0xfd, 0xa9, 0x38,
    0x40, 0x2d, 0xdc, 0x3a, 0x32, 0x93, 0xed, 0x7a, 0xed, 0x4d, 0x87, 0xa9,
};
static const unsigned char kat479_nonce[] = {
    0x0e, 0x4e, 0x20, 0xa3, 0x5f, 0x77, 0xbd, 0xdc, 0x04, 0xb8, 0xd1, 0x72,
    0xf7, 0x8b, 0xbd, 0xee,
};
static const unsigned char kat479_persstr[] = {
    0x2a, 0x1e, 0x1d, 0x1f, 0x13, 0x10, 0xae, 0x69, 0xaa, 0x80, 0x6b, 0x2b,
    0x14, 0xed, 0x2b, 0x1b, 0xa3, 0x08, 0x8c, 0xd3, 0x9a, 0x4c, 0x4d, 0x8f,
    0x27, 0xb4, 0x8b, 0x9e, 0x62, 0x7d, 0xe6, 0x8e,
};
static const unsigned char kat479_addin0[] = {
    0xd4, 0x09, 0x75, 0x2d, 0x88, 0x87, 0xb3, 0x47, 0xe7, 0x78, 0x44, 0x72,
    0xc0, 0x28, 0xab, 0xd9, 0xf0, 0x45, 0xf9, 0x64, 0x1d, 0xb4, 0x63, 0x8d,
    0x67, 0x04, 0xa8, 0x2d, 0x77, 0x67, 0x16, 0xe9,
};
static const unsigned char kat479_addin1[] = {
    0xed, 0xa7, 0x99, 0x69, 0x15, 0x48, 0x2c, 0x06, 0x09, 0xa4, 0x32, 0xe7,
    0x25, 0xfe, 0xf5, 0x8f, 0xef, 0xd9, 0x15, 0xf7, 0x8a, 0xfb, 0x80, 0xd0,
    0x66, 0x4e, 0x99, 0x68, 0xbe, 0xf6, 0x5a, 0x91,
};
static const unsigned char kat479_retbits[] = {
    0x57, 0x99, 0x53, 0x12, 0x4f, 0x77, 0x96, 0xda, 0x17, 0xa3, 0x83, 0xc3,
    0x59, 0xc5, 0xcb, 0xae, 0x15, 0x1e, 0x18, 0xb1, 0x5f, 0x61, 0xb1, 0xe4,
    0x05, 0xed, 0x2c, 0xcd, 0xab, 0xae, 0x43, 0xe4, 0x49, 0x0d, 0x26, 0x92,
    0x72, 0x32, 0x6d, 0x13, 0x88, 0xe4, 0x59, 0x01, 0x40, 0xf4, 0xe5, 0x8d,
    0xfc, 0x6e, 0xca, 0x21, 0xf6, 0x36, 0x03, 0xdf, 0x0a, 0x62, 0x6b, 0xb7,
    0x6a, 0x28, 0x51, 0xd4,
};
static const struct drbg_kat_no_reseed kat479_t = {
    13, kat479_entropyin, kat479_nonce, kat479_persstr,
    kat479_addin0, kat479_addin1, kat479_retbits
};
static const struct drbg_kat kat479 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat479_t
};

static const unsigned char kat480_entropyin[] = {
    0x3d, 0x17, 0x4d, 0x98, 0xec, 0xa8, 0x0a, 0xe3, 0x9c, 0x75, 0xd2, 0x7f,
    0x97, 0xb9, 0x9e, 0x9b, 0x0f, 0x74, 0x2b, 0x32, 0xe3, 0x7b, 0x94, 0xe8,
};
static const unsigned char kat480_nonce[] = {
    0xbc, 0xfc, 0x46, 0x13, 0x17, 0x5b, 0x6e, 0x35, 0x34, 0x08, 0xe2, 0x3b,
    0x85, 0x81, 0xe7, 0x05,
};
static const unsigned char kat480_persstr[] = {
    0x3d, 0x0e, 0x9f, 0xa5, 0x61, 0x6f, 0xa2, 0x5a, 0x0d, 0x58, 0xdd, 0xbd,
    0xc4, 0xa6, 0xa8, 0xe5, 0x5b, 0xef, 0x05, 0x1d, 0xb6, 0xfe, 0xdc, 0x8b,
    0xe8, 0xd4, 0xc7, 0x52, 0xf1, 0xd2, 0xc9, 0x4a,
};
static const unsigned char kat480_addin0[] = {
    0x87, 0x18, 0xb1, 0x5a, 0x55, 0x0f, 0x97, 0x89, 0x39, 0xf5, 0x82, 0x79,
    0xec, 0xda, 0xaf, 0xe7, 0x98, 0x0a, 0x1b, 0xbf, 0xe6, 0xb9, 0xde, 0x85,
    0xb9, 0xfe, 0x10, 0x7c, 0x72, 0xa8, 0xfe, 0x20,
};
static const unsigned char kat480_addin1[] = {
    0x07, 0x88, 0x99, 0xf5, 0xa1, 0xed, 0xd1, 0xba, 0x8e, 0x10, 0xeb, 0x8e,
    0xfa, 0x2c, 0x98, 0xba, 0x17, 0x4c, 0x1b, 0xd3, 0x5a, 0x74, 0x41, 0x4c,
    0xff, 0x86, 0x23, 0x63, 0x3a, 0xc1, 0xc2, 0x5b,
};
static const unsigned char kat480_retbits[] = {
    0x74, 0xe1, 0x55, 0xa5, 0x70, 0x7e, 0xb1, 0x9b, 0x79, 0xcf, 0x7e, 0xfc,
    0xe1, 0xc4, 0x23, 0x7b, 0xac, 0x69, 0xe3, 0xf9, 0x31, 0x14, 0x11, 0xc9,
    0x2d, 0xac, 0xfd, 0x8a, 0xc9, 0x2e, 0xa1, 0x1a, 0x87, 0x87, 0xa4, 0x7b,
    0xc4, 0x4f, 0x1c, 0x29, 0x44, 0x44, 0x5b, 0x2b, 0x05, 0x27, 0xe1, 0x6a,
    0xec, 0x6c, 0x01, 0x16, 0x54, 0x82, 0xd7, 0xe3, 0xb6, 0x44, 0x37, 0x09,
    0x3b, 0x87, 0xbf, 0xbc,
};
static const struct drbg_kat_no_reseed kat480_t = {
    14, kat480_entropyin, kat480_nonce, kat480_persstr,
    kat480_addin0, kat480_addin1, kat480_retbits
};
static const struct drbg_kat kat480 = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat480_t
};

static const unsigned char kat481_entropyin[] = {
    0x36, 0x40, 0x19, 0x40, 0xfa, 0x8b, 0x1f, 0xba, 0x91, 0xa1, 0x66, 0x1f,
    0x21, 0x1d, 0x78, 0xa0, 0xb9, 0x38, 0x9a, 0x74, 0xe5, 0xbc, 0xcf, 0xec,
    0xe8, 0xd7, 0x66, 0xaf, 0x1a, 0x6d, 0x3b, 0x14,
};
static const unsigned char kat481_nonce[] = {
    0x49, 0x6f, 0x25, 0xb0, 0xf1, 0x30, 0x1b, 0x4f, 0x50, 0x1b, 0xe3, 0x03,
    0x80, 0xa1, 0x37, 0xeb,
};
static const unsigned char kat481_persstr[] = {0};
static const unsigned char kat481_addin0[] = {0};
static const unsigned char kat481_addin1[] = {0};
static const unsigned char kat481_retbits[] = {
    0x58, 0x62, 0xeb, 0x38, 0xbd, 0x55, 0x8d, 0xd9, 0x78, 0xa6, 0x96, 0xe6,
    0xdf, 0x16, 0x47, 0x82, 0xdd, 0xd8, 0x87, 0xe7, 0xe9, 0xa6, 0xc9, 0xf3,
    0xf1, 0xfb, 0xaf, 0xb7, 0x89, 0x41, 0xb5, 0x35, 0xa6, 0x49, 0x12, 0xdf,
    0xd2, 0x24, 0xc6, 0xdc, 0x74, 0x54, 0xe5, 0x25, 0x0b, 0x3d, 0x97, 0x16,
    0x5e, 0x16, 0x26, 0x0c, 0x2f, 0xaf, 0x1c, 0xc7, 0x73, 0x5c, 0xb7, 0x5f,
    0xb4, 0xf0, 0x7e, 0x1d,
};
static const struct drbg_kat_no_reseed kat481_t = {
    0, kat481_entropyin, kat481_nonce, kat481_persstr,
    kat481_addin0, kat481_addin1, kat481_retbits
};
static const struct drbg_kat kat481 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat481_t
};

static const unsigned char kat482_entropyin[] = {
    0x13, 0x19, 0x90, 0x90, 0xa4, 0x7f, 0xbd, 0x19, 0x84, 0xeb, 0x5f, 0xa9,
    0x58, 0x93, 0x45, 0x15, 0x46, 0x99, 0xef, 0x73, 0xf0, 0x0c, 0xd6, 0x2b,
    0x07, 0xc3, 0x41, 0x67, 0xc0, 0x32, 0x7e, 0x53,
};
static const unsigned char kat482_nonce[] = {
    0x5f, 0x96, 0x8f, 0x93, 0xb6, 0x59, 0xd8, 0xa5, 0x75, 0x0a, 0x95, 0x34,
    0x5a, 0x8a, 0xe2, 0x0c,
};
static const unsigned char kat482_persstr[] = {0};
static const unsigned char kat482_addin0[] = {0};
static const unsigned char kat482_addin1[] = {0};
static const unsigned char kat482_retbits[] = {
    0xd1, 0x68, 0x78, 0xc5, 0xb0, 0x6d, 0x7b, 0x6c, 0xed, 0x8e, 0x8a, 0xeb,
    0x3a, 0x48, 0xd9, 0x5e, 0xc8, 0xdd, 0x65, 0x57, 0x33, 0xee, 0xc6, 0xef,
    0x47, 0x3a, 0x80, 0x78, 0xdf, 0xde, 0xa6, 0x00, 0xc0, 0xcc, 0x02, 0x16,
    0x8b, 0x4d, 0x6d, 0x74, 0x4e, 0xe8, 0x28, 0xba, 0x50, 0x31, 0x94, 0x1f,
    0x8e, 0x3d, 0x96, 0x58, 0x64, 0x07, 0xaf, 0x79, 0xeb, 0xa6, 0x0d, 0x14,
    0xaf, 0x47, 0xd5, 0x3a,
};
static const struct drbg_kat_no_reseed kat482_t = {
    1, kat482_entropyin, kat482_nonce, kat482_persstr,
    kat482_addin0, kat482_addin1, kat482_retbits
};
static const struct drbg_kat kat482 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat482_t
};

static const unsigned char kat483_entropyin[] = {
    0xd6, 0xcc, 0xf8, 0xc8, 0x14, 0x3a, 0xbf, 0xe5, 0xfd, 0x70, 0x62, 0x6a,
    0xfc, 0x17, 0xf8, 0xae, 0xf1, 0x72, 0x02, 0x7c, 0x68, 0xc3, 0x8f, 0x94,
    0xce, 0x59, 0xf7, 0xae, 0xd5, 0xe9, 0x66, 0x57,
};
static const unsigned char kat483_nonce[] = {
    0x2e, 0xbc, 0x66, 0xd2, 0xfd, 0x66, 0xb4, 0xbf, 0x1e, 0xd2, 0x4f, 0xaf,
    0x74, 0x4f, 0xfb, 0xc9,
};
static const unsigned char kat483_persstr[] = {0};
static const unsigned char kat483_addin0[] = {0};
static const unsigned char kat483_addin1[] = {0};
static const unsigned char kat483_retbits[] = {
    0x6d, 0x47, 0x4b, 0xa9, 0x71, 0xa8, 0x33, 0x9e, 0xca, 0x90, 0x4a, 0x4c,
    0x0d, 0xcf, 0x62, 0x65, 0x11, 0x6f, 0xbc, 0x66, 0xcb, 0xe5, 0xdd, 0xdf,
    0xdc, 0x42, 0x10, 0x45, 0x02, 0xeb, 0x21, 0x0e, 0x36, 0x60, 0xe1, 0xb1,
    0xb7, 0x10, 0xb9, 0x7d, 0x83, 0x0c, 0x27, 0x21, 0x2b, 0x33, 0x13, 0x1d,
    0x85, 0xd2, 0xf7, 0x3f, 0x39, 0x76, 0x07, 0x82, 0xf4, 0xb4, 0x7d, 0x44,
    0x7b, 0xa6, 0xa6, 0x8a,
};
static const struct drbg_kat_no_reseed kat483_t = {
    2, kat483_entropyin, kat483_nonce, kat483_persstr,
    kat483_addin0, kat483_addin1, kat483_retbits
};
static const struct drbg_kat kat483 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat483_t
};

static const unsigned char kat484_entropyin[] = {
    0x39, 0x5d, 0x06, 0xb7, 0x54, 0x90, 0x73, 0xc4, 0x82, 0x52, 0xfb, 0x01,
    0xf3, 0x95, 0x42, 0x64, 0x56, 0x00, 0x31, 0x72, 0x20, 0x09, 0x00, 0x29,
    0xb2, 0xba, 0xc5, 0x8a, 0x7a, 0x4c, 0x35, 0xdf,
};
static const unsigned char kat484_nonce[] = {
    0x57, 0x26, 0xb9, 0x91, 0x1d, 0xa8, 0xf1, 0x66, 0xa8, 0x4f, 0x82, 0xc0,
    0x6f, 0x53, 0xdc, 0x9e,
};
static const unsigned char kat484_persstr[] = {0};
static const unsigned char kat484_addin0[] = {0};
static const unsigned char kat484_addin1[] = {0};
static const unsigned char kat484_retbits[] = {
    0x0e, 0x18, 0x10, 0xb5, 0x04, 0x4f, 0x28, 0xef, 0x2c, 0xc7, 0x92, 0x8b,
    0xd6, 0x32, 0xd7, 0x03, 0x5b, 0xcd, 0xb9, 0x80, 0x1e, 0x9d, 0x84, 0xf5,
    0x69, 0xa5, 0xb6, 0xd0, 0x2d, 0x3c, 0xb5, 0xaa, 0xc0, 0xa1, 0x90, 0xbd,
    0x58, 0xd6, 0xa0, 0x8b, 0x67, 0x89, 0x52, 0x93, 0x20, 0xc7, 0x68, 0x17,
    0xf2, 0x7b, 0x7d, 0x33, 0x10, 0x85, 0x34, 0x67, 0x35, 0xad, 0x37, 0x1b,
    0x5c, 0x91, 0x89, 0xcd,
};
static const struct drbg_kat_no_reseed kat484_t = {
    3, kat484_entropyin, kat484_nonce, kat484_persstr,
    kat484_addin0, kat484_addin1, kat484_retbits
};
static const struct drbg_kat kat484 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat484_t
};

static const unsigned char kat485_entropyin[] = {
    0xe5, 0x02, 0x71, 0x8e, 0x54, 0xc8, 0xa7, 0x9f, 0x31, 0x52, 0x9a, 0xba,
    0x42, 0x40, 0x48, 0x08, 0xe6, 0x52, 0x47, 0x7f, 0x59, 0x5a, 0xb3, 0x5b,
    0xc5, 0x4e, 0xaa, 0xc7, 0xaf, 0xaa, 0x22, 0x8a,
};
static const unsigned char kat485_nonce[] = {
    0xae, 0xe3, 0x28, 0xae, 0x82, 0x27, 0x4d, 0x9d, 0xff, 0xdb, 0x27, 0x72,
    0x31, 0x54, 0x89, 0xb2,
};
static const unsigned char kat485_persstr[] = {0};
static const unsigned char kat485_addin0[] = {0};
static const unsigned char kat485_addin1[] = {0};
static const unsigned char kat485_retbits[] = {
    0x9d, 0x10, 0xba, 0xac, 0x91, 0x77, 0x0e, 0x97, 0xbe, 0x49, 0x0d, 0xb4,
    0xd8, 0x0d, 0x70, 0x07, 0xd6, 0xa2, 0x04, 0x07, 0x81, 0x3e, 0xee, 0x12,
    0x8a, 0xcb, 0x16, 0x1c, 0x6e, 0x36, 0xc2, 0x25, 0xeb, 0xc4, 0x2c, 0xa3,
    0x7b, 0x10, 0x7f, 0x04, 0x30, 0xb6, 0x98, 0x26, 0xad, 0xd2, 0xe5, 0x20,
    0xc2, 0xf1, 0x8f, 0xc0, 0x7e, 0x32, 0xec, 0x0a, 0x7b, 0x33, 0x46, 0x3b,
    0xcf, 0x48, 0xe5, 0x76,
};
static const struct drbg_kat_no_reseed kat485_t = {
    4, kat485_entropyin, kat485_nonce, kat485_persstr,
    kat485_addin0, kat485_addin1, kat485_retbits
};
static const struct drbg_kat kat485 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat485_t
};

static const unsigned char kat486_entropyin[] = {
    0x63, 0xe6, 0x0f, 0xf1, 0x73, 0x9d, 0x46, 0x05, 0xf5, 0xc5, 0x11, 0xfd,
    0x0e, 0x39, 0x51, 0xdd, 0x3d, 0xe6, 0x57, 0x50, 0x8a, 0x60, 0xd7, 0xc8,
    0x7c, 0xe9, 0x5f, 0x39, 0x48, 0x1a, 0x75, 0x53,
};
static const unsigned char kat486_nonce[] = {
    0xb1, 0xc1, 0x7b, 0xb3, 0x4b, 0xaf, 0x2c, 0x7f, 0x3b, 0x03, 0xb7, 0x6e,
    0x68, 0x97, 0x31, 0x6f,
};
static const unsigned char kat486_persstr[] = {0};
static const unsigned char kat486_addin0[] = {0};
static const unsigned char kat486_addin1[] = {0};
static const unsigned char kat486_retbits[] = {
    0x08, 0x19, 0x50, 0x61, 0xde, 0xd1, 0xab, 0x84, 0xbe, 0x7d, 0x5d, 0xcd,
    0x63, 0x0e, 0x7b, 0x90, 0x38, 0x54, 0xf1, 0x28, 0x43, 0x89, 0xe5, 0xe7,
    0x78, 0x49, 0xbb, 0xa1, 0x0c, 0x89, 0x08, 0x3b, 0xb6, 0xf3, 0x2f, 0x68,
    0x15, 0xcf, 0xac, 0x45, 0xb7, 0xca, 0xd0, 0xdf, 0xb5, 0x54, 0x98, 0xa4,
    0x7e, 0x87, 0x5d, 0x21, 0xed, 0x18, 0x42, 0xf5, 0x87, 0x79, 0x84, 0x5f,
    0x2f, 0x24, 0xc6, 0xe2,
};
static const struct drbg_kat_no_reseed kat486_t = {
    5, kat486_entropyin, kat486_nonce, kat486_persstr,
    kat486_addin0, kat486_addin1, kat486_retbits
};
static const struct drbg_kat kat486 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat486_t
};

static const unsigned char kat487_entropyin[] = {
    0x63, 0xec, 0x70, 0x79, 0x2a, 0x6c, 0x78, 0x54, 0x0e, 0x40, 0xdb, 0x15,
    0xc1, 0xb8, 0x73, 0xfc, 0xde, 0x9b, 0xea, 0x1d, 0x74, 0x1d, 0x88, 0xb8,
    0x1b, 0xd1, 0xfd, 0x66, 0x13, 0x9d, 0xf7, 0x0c,
};
static const unsigned char kat487_nonce[] = {
    0xfc, 0x5f, 0x90, 0x2b, 0xdd, 0x0d, 0x7f, 0xe2, 0x16, 0xb1, 0x42, 0x3d,
    0x4f, 0x41, 0xf1, 0x2f,
};
static const unsigned char kat487_persstr[] = {0};
static const unsigned char kat487_addin0[] = {0};
static const unsigned char kat487_addin1[] = {0};
static const unsigned char kat487_retbits[] = {
    0xe3, 0x66, 0xae, 0x29, 0x88, 0xf6, 0xf3, 0x78, 0x54, 0x85, 0x96, 0x02,
    0x31, 0x9e, 0x64, 0x38, 0x52, 0x44, 0x11, 0x50, 0x04, 0x38, 0x6a, 0x21,
    0xeb, 0x25, 0xe6, 0x94, 0x56, 0xc7, 0x78, 0x94, 0x7a, 0xe1, 0x1c, 0x95,
    0x32, 0xe5, 0x79, 0x4e, 0x62, 0x04, 0x60, 0x87, 0x7a, 0x5b, 0xcd, 0x77,
    0xf3, 0x70, 0xe8, 0xa9, 0x0d, 0x4b, 0xe3, 0x82, 0x06, 0xca, 0x39, 0x3e,
    0xdc, 0xea, 0xe3, 0xac,
};
static const struct drbg_kat_no_reseed kat487_t = {
    6, kat487_entropyin, kat487_nonce, kat487_persstr,
    kat487_addin0, kat487_addin1, kat487_retbits
};
static const struct drbg_kat kat487 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat487_t
};

static const unsigned char kat488_entropyin[] = {
    0xea, 0xdc, 0x5b, 0x6a, 0x6f, 0x8d, 0x10, 0xa0, 0xa7, 0x89, 0x17, 0x03,
    0x55, 0x0e, 0xf4, 0x0f, 0xc2, 0x1c, 0xa5, 0x00, 0xee, 0x60, 0x3a, 0x68,
    0x5f, 0xf3, 0xd1, 0xfb, 0x56, 0xea, 0xd7, 0x0e,
};
static const unsigned char kat488_nonce[] = {
    0x6c, 0x5a, 0x65, 0xd7, 0xe8, 0xfb, 0xc2, 0xa7, 0xcf, 0xd9, 0xfa, 0x7a,
    0x5e, 0xfb, 0xdc, 0xd7,
};
static const unsigned char kat488_persstr[] = {0};
static const unsigned char kat488_addin0[] = {0};
static const unsigned char kat488_addin1[] = {0};
static const unsigned char kat488_retbits[] = {
    0xd4, 0x28, 0xa2, 0x98, 0xcb, 0xfd, 0xf2, 0x0e, 0xf7, 0xbe, 0xe8, 0x91,
    0x3a, 0x26, 0xc5, 0x3e, 0xea, 0x49, 0x93, 0x3a, 0x2d, 0xde, 0x42, 0x1b,
    0xca, 0x4b, 0x1c, 0x6b, 0x86, 0x50, 0x6c, 0x6c, 0xa0, 0xef, 0x0b, 0xfa,
    0x13, 0xc0, 0x30, 0x50, 0x57, 0x48, 0xd5, 0x73, 0x7b, 0x23, 0x24, 0x80,
    0xed, 0xc0, 0xbb, 0xc9, 0xe7, 0xb7, 0xf3, 0xff, 0x8a, 0xef, 0xc2, 0x92,
    0x10, 0x6a, 0x62, 0x54,
};
static const struct drbg_kat_no_reseed kat488_t = {
    7, kat488_entropyin, kat488_nonce, kat488_persstr,
    kat488_addin0, kat488_addin1, kat488_retbits
};
static const struct drbg_kat kat488 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat488_t
};

static const unsigned char kat489_entropyin[] = {
    0x0f, 0x91, 0x06, 0x58, 0x8b, 0xc9, 0x27, 0xec, 0xa2, 0x85, 0xe0, 0x5c,
    0x8c, 0x17, 0x0a, 0x68, 0xe9, 0xa1, 0x03, 0x10, 0x2a, 0x10, 0x67, 0x81,
    0xd8, 0xfb, 0x0a, 0x76, 0x88, 0xe4, 0x91, 0xc2,
};
static const unsigned char kat489_nonce[] = {
    0x2e, 0x58, 0xa8, 0x59, 0xb7, 0xbd, 0xc8, 0x16, 0xfb, 0x40, 0xb9, 0xee,
    0x59, 0x16, 0xa9, 0x25,
};
static const unsigned char kat489_persstr[] = {0};
static const unsigned char kat489_addin0[] = {0};
static const unsigned char kat489_addin1[] = {0};
static const unsigned char kat489_retbits[] = {
    0xe3, 0xde, 0xcd, 0x08, 0x10, 0xe1, 0xdb, 0x9b, 0x77, 0x49, 0x0c, 0xbe,
    0x78, 0xca, 0xd2, 0x52, 0xc2, 0x61, 0xf0, 0xf3, 0x80, 0x51, 0xe5, 0xba,
    0x1e, 0x1f, 0xf9, 0x85, 0x2d, 0x0a, 0x68, 0xa1, 0xfb, 0xdc, 0x79, 0x8e,
    0xb1, 0x96, 0xf2, 0xba, 0x13, 0x2e, 0x6a, 0x45, 0x1d, 0xfe, 0x6a, 0x98,
    0x88, 0x83, 0x36, 0x66, 0xbb, 0xc3, 0x04, 0xc6, 0x17, 0xc2, 0xd6, 0x10,
    0xbd, 0x5e, 0x48, 0x9e,
};
static const struct drbg_kat_no_reseed kat489_t = {
    8, kat489_entropyin, kat489_nonce, kat489_persstr,
    kat489_addin0, kat489_addin1, kat489_retbits
};
static const struct drbg_kat kat489 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat489_t
};

static const unsigned char kat490_entropyin[] = {
    0x41, 0x62, 0xa4, 0x2c, 0xb0, 0xd8, 0x3d, 0x33, 0x84, 0xa1, 0xdc, 0xb9,
    0xda, 0xdc, 0xc6, 0xd0, 0x70, 0xeb, 0xd1, 0x68, 0x3b, 0x3c, 0x09, 0x00,
    0xf3, 0x1d, 0x7c, 0xbe, 0x05, 0x0e, 0x6a, 0xca,
};
static const unsigned char kat490_nonce[] = {
    0x15, 0xf1, 0x54, 0xac, 0x7c, 0x82, 0x52, 0x58, 0xde, 0x3d, 0x66, 0x89,
    0xfb, 0x9a, 0xb4, 0x6f,
};
static const unsigned char kat490_persstr[] = {0};
static const unsigned char kat490_addin0[] = {0};
static const unsigned char kat490_addin1[] = {0};
static const unsigned char kat490_retbits[] = {
    0x3e, 0xad, 0xb8, 0xac, 0xf7, 0xde, 0x3b, 0xd0, 0x98, 0x29, 0x70, 0x67,
    0x36, 0x61, 0x83, 0xa0, 0x79, 0x90, 0x55, 0x0a, 0xde, 0xad, 0x0b, 0xbb,
    0xe4, 0x8f, 0xe8, 0x7b, 0xfd, 0xe2, 0x1c, 0xa8, 0x78, 0xe3, 0xce, 0xc7,
    0x73, 0x79, 0xbb, 0x88, 0x4f, 0x28, 0x54, 0x6c, 0x57, 0xc4, 0x0e, 0x28,
    0xe7, 0x23, 0xf8, 0xcb, 0x8c, 0x9e, 0x04, 0xa8, 0x50, 0xa4, 0x6d, 0xc7,
    0x5f, 0xe9, 0x03, 0x68,
};
static const struct drbg_kat_no_reseed kat490_t = {
    9, kat490_entropyin, kat490_nonce, kat490_persstr,
    kat490_addin0, kat490_addin1, kat490_retbits
};
static const struct drbg_kat kat490 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat490_t
};

static const unsigned char kat491_entropyin[] = {
    0x36, 0x52, 0x20, 0x1d, 0x2c, 0xed, 0x05, 0x6c, 0x83, 0x66, 0x71, 0x57,
    0xd7, 0xb0, 0xed, 0x02, 0xdc, 0x4b, 0xa3, 0xf6, 0xd4, 0x37, 0x99, 0xfd,
    0xa4, 0xa6, 0xd5, 0x20, 0x4c, 0x4e, 0x3b, 0xb4,
};
static const unsigned char kat491_nonce[] = {
    0x37, 0xf1, 0x81, 0x83, 0x70, 0x52, 0xd7, 0x29, 0xa7, 0x0b, 0x46, 0x2f,
    0x5c, 0x1f, 0xec, 0xc2,
};
static const unsigned char kat491_persstr[] = {0};
static const unsigned char kat491_addin0[] = {0};
static const unsigned char kat491_addin1[] = {0};
static const unsigned char kat491_retbits[] = {
    0xb5, 0xbc, 0x59, 0x13, 0x8d, 0xf3, 0x51, 0x49, 0x00, 0x57, 0xcf, 0x9d,
    0xf9, 0x7e, 0x99, 0x17, 0x45, 0xe0, 0x3b, 0x30, 0xee, 0x96, 0x84, 0xe6,
    0x18, 0x12, 0xd8, 0x45, 0x3d, 0xac, 0x5b, 0xcf, 0x54, 0x99, 0x6d, 0x95,
    0xca, 0x1d, 0x71, 0xfb, 0xc5, 0x99, 0x2d, 0x18, 0xde, 0x9b, 0xf2, 0x24,
    0xf2, 0xba, 0x42, 0xd2, 0x4e, 0x3f, 0x8d, 0x13, 0xe3, 0x41, 0x18, 0x22,
    0x74, 0xcb, 0xcd, 0x32,
};
static const struct drbg_kat_no_reseed kat491_t = {
    10, kat491_entropyin, kat491_nonce, kat491_persstr,
    kat491_addin0, kat491_addin1, kat491_retbits
};
static const struct drbg_kat kat491 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat491_t
};

static const unsigned char kat492_entropyin[] = {
    0xc1, 0xd3, 0xcb, 0xa4, 0x8d, 0x32, 0x8a, 0xd2, 0xe4, 0x1c, 0x75, 0xd2,
    0x1c, 0xb5, 0x3b, 0x69, 0xff, 0x16, 0xca, 0xfb, 0x51, 0xc2, 0x41, 0xd3,
    0x6e, 0x1d, 0x75, 0x2f, 0xa3, 0xde, 0x2d, 0xd3,
};
static const unsigned char kat492_nonce[] = {
    0xb1, 0xb6, 0xe9, 0xd1, 0xa5, 0x0d, 0x4f, 0xa6, 0x54, 0xeb, 0xd4, 0x46,
    0x63, 0xa8, 0xec, 0x01,
};
static const unsigned char kat492_persstr[] = {0};
static const unsigned char kat492_addin0[] = {0};
static const unsigned char kat492_addin1[] = {0};
static const unsigned char kat492_retbits[] = {
    0x2b, 0x9b, 0xa1, 0xa9, 0xbc, 0x56, 0xc7, 0xf3, 0xe5, 0xb2, 0xd5, 0x87,
    0xaa, 0xf9, 0x8d, 0x1e, 0x7d, 0xa5, 0x66, 0xcf, 0xf5, 0xf2, 0xe7, 0xf9,
    0x17, 0x10, 0xd4, 0x3f, 0x8c, 0xf7, 0xe4, 0x85, 0x63, 0x2f, 0xf4, 0x51,
    0x11, 0x0a, 0xf8, 0x73, 0xe0, 0xb3, 0x3f, 0x0d, 0xff, 0x46, 0x8f, 0x09,
    0xc5, 0xe3, 0x23, 0x3d, 0x38, 0x07, 0xcd, 0x73, 0x99, 0x82, 0xb5, 0xe5,
    0x88, 0x54, 0x34, 0xc9,
};
static const struct drbg_kat_no_reseed kat492_t = {
    11, kat492_entropyin, kat492_nonce, kat492_persstr,
    kat492_addin0, kat492_addin1, kat492_retbits
};
static const struct drbg_kat kat492 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat492_t
};

static const unsigned char kat493_entropyin[] = {
    0xd6, 0x70, 0x2e, 0xd5, 0xd0, 0x35, 0x96, 0xd2, 0xd1, 0xda, 0xf9, 0xff,
    0xe9, 0xc0, 0xa1, 0x9a, 0x47, 0x94, 0x77, 0xf7, 0xe9, 0x14, 0x65, 0x4f,
    0x7b, 0x11, 0x4d, 0x18, 0xb6, 0x3f, 0x50, 0x7c,
};
static const unsigned char kat493_nonce[] = {
    0x53, 0xb2, 0x9d, 0x4b, 0x42, 0xd3, 0xc8, 0x79, 0x8d, 0xc1, 0xe6, 0x3d,
    0xf9, 0xf3, 0x0b, 0xbf,
};
static const unsigned char kat493_persstr[] = {0};
static const unsigned char kat493_addin0[] = {0};
static const unsigned char kat493_addin1[] = {0};
static const unsigned char kat493_retbits[] = {
    0xfb, 0x4e, 0x3f, 0x7b, 0x0c, 0x6e, 0xfa, 0xe1, 0x48, 0x13, 0x0f, 0x82,
    0xd5, 0x9c, 0xbc, 0x99, 0x38, 0x12, 0x41, 0x2e, 0x0f, 0x57, 0xb2, 0x06,
    0xff, 0x08, 0x64, 0xf4, 0xdb, 0xb8, 0xe4, 0x07, 0xeb, 0x7d, 0xe2, 0xc6,
    0x8a, 0xf8, 0xa5, 0x14, 0x16, 0x10, 0xed, 0x47, 0x65, 0x27, 0x49, 0x35,
    0xe6, 0xf3, 0x7e, 0xf4, 0xee, 0xd3, 0x7d, 0xe2, 0x73, 0x78, 0x85, 0x76,
    0xe7, 0xb8, 0x9d, 0xaa,
};
static const struct drbg_kat_no_reseed kat493_t = {
    12, kat493_entropyin, kat493_nonce, kat493_persstr,
    kat493_addin0, kat493_addin1, kat493_retbits
};
static const struct drbg_kat kat493 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat493_t
};

static const unsigned char kat494_entropyin[] = {
    0x7e, 0xe6, 0x1f, 0xaa, 0x3b, 0x3f, 0xd9, 0x83, 0xba, 0x3b, 0x35, 0x0b,
    0xee, 0x25, 0xbd, 0x07, 0x5e, 0x2e, 0x6f, 0x17, 0x2b, 0xe9, 0x64, 0xbb,
    0xe4, 0xff, 0x21, 0x0f, 0xc9, 0x2d, 0xe6, 0x51,
};
static const unsigned char kat494_nonce[] = {
    0xa0, 0xc4, 0xeb, 0xcc, 0x0a, 0xef, 0x84, 0x8e, 0xd7, 0xb0, 0x0d, 0x3b,
    0x61, 0x29, 0x72, 0xc9,
};
static const unsigned char kat494_persstr[] = {0};
static const unsigned char kat494_addin0[] = {0};
static const unsigned char kat494_addin1[] = {0};
static const unsigned char kat494_retbits[] = {
    0xa2, 0xa1, 0x90, 0xf0, 0xfd, 0xfa, 0xf6, 0x63, 0x14, 0xd4, 0xe0, 0x07,
    0x1d, 0x4e, 0xc3, 0xd2, 0x91, 0x4f, 0xa9, 0x3f, 0x89, 0x64, 0xef, 0x56,
    0x08, 0xec, 0x38, 0x6b, 0xba, 0x03, 0x46, 0x70, 0xec, 0x01, 0x77, 0x35,
    0x0a, 0x78, 0x02, 0x14, 0xba, 0xc5, 0x79, 0x29, 0x09, 0x18, 0xfb, 0xa9,
    0xb2, 0x1e, 0xf9, 0xf4, 0xf1, 0x21, 0x00, 0x34, 0xd0, 0x80, 0x9f, 0xed,
    0xc4, 0x4d, 0x33, 0xb9,
};
static const struct drbg_kat_no_reseed kat494_t = {
    13, kat494_entropyin, kat494_nonce, kat494_persstr,
    kat494_addin0, kat494_addin1, kat494_retbits
};
static const struct drbg_kat kat494 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat494_t
};

static const unsigned char kat495_entropyin[] = {
    0x86, 0xe3, 0xcc, 0xf0, 0x93, 0x82, 0xf4, 0x61, 0x92, 0xd8, 0x93, 0x7e,
    0xba, 0x1d, 0xfc, 0x04, 0xad, 0x8e, 0x12, 0xef, 0x19, 0x6d, 0x6d, 0x94,
    0x19, 0x70, 0xb1, 0x6e, 0x15, 0x43, 0x8b, 0xb2,
};
static const unsigned char kat495_nonce[] = {
    0xe2, 0xf9, 0xcc, 0x7e, 0xab, 0x26, 0x49, 0x1e, 0x3e, 0xab, 0x3c, 0xfd,
    0xfa, 0xad, 0xda, 0x90,
};
static const unsigned char kat495_persstr[] = {0};
static const unsigned char kat495_addin0[] = {0};
static const unsigned char kat495_addin1[] = {0};
static const unsigned char kat495_retbits[] = {
    0xd8, 0xbc, 0x7e, 0x82, 0x20, 0xcb, 0xe0, 0x79, 0xb4, 0x40, 0x3d, 0x6b,
    0x47, 0xe7, 0x97, 0x66, 0x05, 0x73, 0x5f, 0x95, 0x3a, 0xab, 0xf9, 0x3f,
    0xd0, 0x03, 0x93, 0x96, 0xdd, 0xb4, 0x08, 0x8e, 0x43, 0x60, 0x52, 0x72,
    0x2c, 0x43, 0x25, 0xf9, 0x0e, 0xb8, 0x68, 0x51, 0x2a, 0xe7, 0xce, 0x39,
    0x27, 0x37, 0x09, 0x40, 0xb2, 0x00, 0xf3, 0xcc, 0x81, 0xda, 0x77, 0x47,
    0x79, 0xe9, 0x0d, 0x7c,
};
static const struct drbg_kat_no_reseed kat495_t = {
    14, kat495_entropyin, kat495_nonce, kat495_persstr,
    kat495_addin0, kat495_addin1, kat495_retbits
};
static const struct drbg_kat kat495 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat495_t
};

static const unsigned char kat496_entropyin[] = {
    0x81, 0x48, 0xd6, 0x5d, 0x86, 0x51, 0x3c, 0xe7, 0xd3, 0x89, 0x23, 0xec,
    0x2f, 0x26, 0xb9, 0xe7, 0xc6, 0x77, 0xdc, 0xc8, 0x99, 0x7e, 0x32, 0x5b,
    0x73, 0x72, 0x61, 0x9e, 0x75, 0x3e, 0xd9, 0x44,
};
static const unsigned char kat496_nonce[] = {
    0x41, 0xc7, 0x1a, 0x24, 0xd1, 0x7d, 0x97, 0x41, 0x90, 0x98, 0x2b, 0xb7,
    0x51, 0x5c, 0xe7, 0xf5,
};
static const unsigned char kat496_persstr[] = {0};
static const unsigned char kat496_addin0[] = {
    0x55, 0xb4, 0x46, 0x04, 0x6c, 0x2d, 0x14, 0xbd, 0xd0, 0xcd, 0xba, 0x4b,
    0x71, 0x87, 0x3f, 0xd4, 0x76, 0x26, 0x50, 0x69, 0x5a, 0x11, 0x50, 0x79,
    0x49, 0x46, 0x2d, 0xa8, 0xd9, 0x64, 0xab, 0x6a,
};
static const unsigned char kat496_addin1[] = {
    0x91, 0x46, 0x8f, 0x1a, 0x09, 0x7d, 0x99, 0xee, 0x33, 0x94, 0x62, 0xca,
    0x91, 0x6c, 0xb4, 0xa1, 0x0f, 0x63, 0xd5, 0x38, 0x50, 0xa4, 0xf1, 0x7f,
    0x59, 0x8e, 0xac, 0x49, 0x02, 0x99, 0xb0, 0x2e,
};
static const unsigned char kat496_retbits[] = {
    0x54, 0x60, 0x3d, 0x1a, 0x50, 0x61, 0x32, 0xbb, 0xfa, 0x05, 0xb1, 0x53,
    0xa0, 0x4f, 0x22, 0xa1, 0xd5, 0x16, 0xcc, 0x46, 0x32, 0x3c, 0xef, 0x15,
    0x11, 0x1a, 0xf2, 0x21, 0xf0, 0x30, 0xf3, 0x8d, 0x68, 0x41, 0xd4, 0x67,
    0x05, 0x18, 0xb4, 0x91, 0x4a, 0x46, 0x31, 0xaf, 0x68, 0x2e, 0x74, 0x21,
    0xdf, 0xfa, 0xac, 0x98, 0x6a, 0x38, 0xe9, 0x4d, 0x92, 0xbf, 0xa7, 0x58,
    0xe2, 0xeb, 0x10, 0x1f,
};
static const struct drbg_kat_no_reseed kat496_t = {
    0, kat496_entropyin, kat496_nonce, kat496_persstr,
    kat496_addin0, kat496_addin1, kat496_retbits
};
static const struct drbg_kat kat496 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat496_t
};

static const unsigned char kat497_entropyin[] = {
    0xeb, 0x4a, 0x0a, 0xdd, 0x69, 0x70, 0x97, 0xf1, 0xce, 0x3a, 0x71, 0x9d,
    0x0d, 0x4a, 0xe6, 0x9b, 0x17, 0x21, 0xdc, 0xe3, 0xec, 0x0e, 0x6c, 0x0e,
    0x90, 0x5d, 0x78, 0xee, 0x21, 0x28, 0x63, 0xb1,
};
static const unsigned char kat497_nonce[] = {
    0x5f, 0x36, 0x8e, 0x85, 0xc1, 0xf1, 0x7b, 0x64, 0x63, 0xa2, 0x78, 0x37,
    0x7f, 0x69, 0x1f, 0x37,
};
static const unsigned char kat497_persstr[] = {0};
static const unsigned char kat497_addin0[] = {
    0xf9, 0x78, 0x01, 0xbc, 0xe9, 0x81, 0xb3, 0x50, 0x81, 0xc2, 0x58, 0x01,
    0x40, 0x0e, 0xc2, 0x07, 0x43, 0x3d, 0xa4, 0xf1, 0x7f, 0x32, 0x65, 0xa1,
    0x6e, 0x9e, 0x4e, 0x68, 0x37, 0x22, 0x70, 0x8b,
};
static const unsigned char kat497_addin1[] = {
    0xae, 0x54, 0xb4, 0x9a, 0x41, 0x12, 0xb3, 0xd9, 0x78, 0xe9, 0x66, 0xe2,
    0xdd, 0xa0, 0x62, 0xe3, 0x65, 0x2b, 0x58, 0xa1, 0x4b, 0xef, 0x4f, 0xfe,
    0x03, 0x85, 0x20, 0xc9, 0xa6, 0x75, 0xd3, 0x53,
};
static const unsigned char kat497_retbits[] = {
    0x6a, 0xee, 0x0b, 0x3a, 0x81, 0x5c, 0x82, 0xf9, 0xbb, 0x01, 0x19, 0xf8,
    0x6a, 0xf9, 0x07, 0x93, 0xfc, 0x1f, 0x99, 0x96, 0xdd, 0x5b, 0x72, 0xbb,
    0xc3, 0x26, 0xac, 0x4e, 0x6a, 0x5e, 0x87, 0x48, 0x50, 0xb2, 0xfe, 0xc1,
    0xd7, 0x20, 0x2c, 0x35, 0x58, 0x0b, 0xd6, 0x72, 0x70, 0x29, 0x60, 0x9f,
    0x24, 0x71, 0xe6, 0xc9, 0xb6, 0x16, 0x29, 0xd1, 0x74, 0xb8, 0x94, 0xcd,
    0x17, 0x8a, 0xdf, 0xd4,
};
static const struct drbg_kat_no_reseed kat497_t = {
    1, kat497_entropyin, kat497_nonce, kat497_persstr,
    kat497_addin0, kat497_addin1, kat497_retbits
};
static const struct drbg_kat kat497 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat497_t
};

static const unsigned char kat498_entropyin[] = {
    0x7f, 0xd6, 0xe2, 0x62, 0xd8, 0x21, 0xd5, 0xe5, 0xb6, 0x60, 0x48, 0x57,
    0x55, 0xea, 0x79, 0x61, 0x57, 0x96, 0x31, 0xa4, 0xb9, 0x64, 0xcf, 0xb4,
    0xc2, 0xc3, 0x5a, 0xfd, 0xe6, 0x9f, 0xfe, 0xa1,
};
static const unsigned char kat498_nonce[] = {
    0xae, 0x8c, 0x54, 0xaf, 0xfd, 0xb7, 0x6c, 0x5f, 0xd1, 0x96, 0xfb, 0xd5,
    0xa2, 0xc4, 0x77, 0xec,
};
static const unsigned char kat498_persstr[] = {0};
static const unsigned char kat498_addin0[] = {
    0xab, 0x81, 0x03, 0x5c, 0xb3, 0xc0, 0x17, 0xcb, 0xe5, 0x1a, 0x2b, 0xc6,
    0x47, 0x51, 0xce, 0x61, 0xf8, 0xae, 0x02, 0xe8, 0x0a, 0xfe, 0xf8, 0x37,
    0x8f, 0x42, 0xac, 0x67, 0x06, 0x0f, 0xfa, 0xf9,
};
static const unsigned char kat498_addin1[] = {
    0x4f, 0x7d, 0xb0, 0x2d, 0x34, 0x6b, 0xd4, 0x16, 0x68, 0xfc, 0xf6, 0x1b,
    0xac, 0x93, 0x93, 0x60, 0x03, 0xd2, 0x2f, 0xeb, 0xd3, 0xb9, 0xf8, 0xc0,
    0x23, 0x4d, 0x15, 0x1b, 0x49, 0x2b, 0x16, 0xe7,
};
static const unsigned char kat498_retbits[] = {
    0x93, 0x0c, 0x26, 0x43, 0x1a, 0x0f, 0xab, 0xb4, 0x5a, 0xbe, 0x41, 0x8d,
    0xb9, 0xaf, 0x10, 0xfe, 0x27, 0x55, 0x80, 0xf4, 0x99, 0xcd, 0xd7, 0x17,
    0xf7, 0xfc, 0xc9, 0x4b, 0x59, 0xf9, 0x52, 0xa0, 0x4e, 0xef, 0x8f, 0x1d,
    0x5a, 0xa0, 0xa4, 0x82, 0x0d, 0xdb, 0xe4, 0x13, 0xb5, 0xc3, 0xd7, 0xa0,
    0x89, 0x27, 0x34, 0x6a, 0x90, 0x63, 0x5e, 0xa2, 0xc5, 0x1b, 0x0a, 0xb0,
    0xe9, 0xce, 0xdb, 0x1d,
};
static const struct drbg_kat_no_reseed kat498_t = {
    2, kat498_entropyin, kat498_nonce, kat498_persstr,
    kat498_addin0, kat498_addin1, kat498_retbits
};
static const struct drbg_kat kat498 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat498_t
};

static const unsigned char kat499_entropyin[] = {
    0x70, 0x50, 0x79, 0x0a, 0x8b, 0x22, 0xff, 0xea, 0x19, 0xa5, 0x05, 0xe4,
    0xfb, 0xb3, 0xbc, 0x2a, 0x3b, 0x41, 0xfd, 0x94, 0x7c, 0xe3, 0xdd, 0x50,
    0xb4, 0xf7, 0x38, 0xd8, 0xc2, 0x2f, 0xbe, 0xb0,
};
static const unsigned char kat499_nonce[] = {
    0x36, 0x56, 0x28, 0xa7, 0xb6, 0xe7, 0xa7, 0x16, 0x60, 0xfd, 0x36, 0x38,
    0x35, 0x1e, 0x6c, 0x12,
};
static const unsigned char kat499_persstr[] = {0};
static const unsigned char kat499_addin0[] = {
    0x1d, 0xb1, 0xc1, 0x98, 0x48, 0x58, 0x0e, 0xe1, 0xf6, 0x9a, 0x63, 0x81,
    0x4b, 0x41, 0xad, 0x65, 0xea, 0x1a, 0x54, 0x26, 0x1e, 0x90, 0x7d, 0x3e,
    0xdb, 0x80, 0xb5, 0xe9, 0xc5, 0x58, 0x19, 0x9d,
};
static const unsigned char kat499_addin1[] = {
    0x06, 0x18, 0x12, 0x90, 0x6e, 0xd3, 0x31, 0x4a, 0x9d, 0xf0, 0xef, 0x61,
    0xb3, 0xb5, 0x2b, 0x68, 0x5c, 0xcc, 0x45, 0x60, 0x1d, 0x69, 0xd9, 0x84,
    0x4c, 0xdb, 0xdb, 0x45, 0x62, 0x7d, 0x42, 0x94,
};
static const unsigned char kat499_retbits[] = {
    0x0e, 0x6f, 0xc5, 0xde, 0x18, 0x29, 0x7e, 0x15, 0xb4, 0x32, 0xa1, 0x24,
    0xa9, 0xd8, 0x87, 0x7f, 0x9a, 0xdb, 0xd2, 0x33, 0x72, 0x40, 0x6f, 0x26,
    0x3e, 0x2d, 0xd7, 0x7c, 0x69, 0xbd, 0xe7, 0x55, 0x96, 0xeb, 0x9f, 0x8a,
    0x7a, 0x25, 0xd0, 0xb4, 0x5a, 0x40, 0x3a, 0x4c, 0x57, 0x58, 0x77, 0x1f,
    0x22, 0x49, 0x09, 0xd8, 0x84, 0x08, 0xf1, 0x74, 0x48, 0x7f, 0xdb, 0x30,
    0x55, 0x4c, 0x1f, 0x3b,
};
static const struct drbg_kat_no_reseed kat499_t = {
    3, kat499_entropyin, kat499_nonce, kat499_persstr,
    kat499_addin0, kat499_addin1, kat499_retbits
};
static const struct drbg_kat kat499 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat499_t
};

static const unsigned char kat500_entropyin[] = {
    0x29, 0x68, 0xf8, 0x57, 0xd5, 0x83, 0xab, 0x28, 0x74, 0x10, 0xf4, 0x55,
    0xbf, 0x44, 0x62, 0xe8, 0xa7, 0x17, 0xfb, 0x93, 0xbc, 0x20, 0x0d, 0xb4,
    0x3d, 0x12, 0xb1, 0xd1, 0x33, 0xbe, 0x00, 0x3b,
};
static const unsigned char kat500_nonce[] = {
    0xdc, 0xdd, 0x14, 0x6a, 0x6f, 0x2f, 0x9f, 0x0f, 0x86, 0x80, 0x4f, 0x30,
    0x92, 0x16, 0xa7, 0xbe,
};
static const unsigned char kat500_persstr[] = {0};
static const unsigned char kat500_addin0[] = {
    0x65, 0x5e, 0xac, 0x56, 0xcb, 0xb1, 0x24, 0xe1, 0x75, 0x82, 0xa1, 0xfa,
    0x7f, 0xa6, 0x19, 0x9e, 0xbe, 0x32, 0x81, 0x10, 0x1e, 0xc7, 0x8b, 0xac,
    0x74, 0x9a, 0x4d, 0x1d, 0x28, 0x7e, 0x1b, 0x82,
};
static const unsigned char kat500_addin1[] = {
    0x26, 0xba, 0x4b, 0x54, 0x01, 0xb9, 0xd1, 0x24, 0x62, 0x5a, 0x60, 0xd5,
    0x3f, 0x2d, 0x7b, 0x3f, 0x45, 0xdb, 0x5b, 0xc4, 0x36, 0x11, 0xd4, 0xcc,
    0x59, 0xc8, 0x3a, 0x18, 0x18, 0x16, 0x9c, 0xac,
};
static const unsigned char kat500_retbits[] = {
    0xa7, 0xbf, 0x0e, 0x65, 0x50, 0x20, 0xa2, 0xc9, 0xa5, 0x06, 0x43, 0xcc,
    0xde, 0x05, 0xfd, 0x95, 0xa6, 0x44, 0x66, 0xda, 0x7a, 0x75, 0x6b, 0xf1,
    0xb9, 0xd1, 0x95, 0xfb, 0xac, 0x0e, 0xe2, 0x05, 0x9b, 0xa9, 0xd5, 0xf5,
    0xe3, 0x43, 0xfd, 0x8d, 0x67, 0x55, 0xa2, 0xe5, 0x4c, 0x38, 0x80, 0x3c,
    0xcf, 0xc4, 0xe9, 0x19, 0x12, 0x4c, 0x1e, 0xcd, 0x48, 0x53, 0x71, 0x54,
    0x86, 0xbe, 0x4f, 0x1a,
};
static const struct drbg_kat_no_reseed kat500_t = {
    4, kat500_entropyin, kat500_nonce, kat500_persstr,
    kat500_addin0, kat500_addin1, kat500_retbits
};
static const struct drbg_kat kat500 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat500_t
};

static const unsigned char kat501_entropyin[] = {
    0x8f, 0xd3, 0xb3, 0x30, 0xfc, 0x13, 0x8f, 0x42, 0x1a, 0x66, 0x92, 0x5f,
    0x72, 0x64, 0x89, 0x85, 0xe5, 0xa2, 0x55, 0xe5, 0x99, 0x7f, 0xf2, 0x47,
    0xe5, 0x33, 0x62, 0xdf, 0xf7, 0xa9, 0xa0, 0xf9,
};
static const unsigned char kat501_nonce[] = {
    0x3c, 0x67, 0x0e, 0x24, 0x67, 0xe3, 0xfb, 0x08, 0x7c, 0x7e, 0xa6, 0xcc,
    0x0a, 0xf5, 0x56, 0xff,
};
static const unsigned char kat501_persstr[] = {0};
static const unsigned char kat501_addin0[] = {
    0xa8, 0x4e, 0xfb, 0x4a, 0x46, 0x4c, 0xc8, 0xaa, 0x75, 0x49, 0x7f, 0x86,
    0xc3, 0x51, 0x5c, 0x64, 0x76, 0xce, 0xdc, 0x7b, 0x94, 0x0f, 0x9d, 0x46,
    0xce, 0x40, 0x1b, 0xd1, 0xd5, 0x23, 0x49, 0xfd,
};
static const unsigned char kat501_addin1[] = {
    0x78, 0xdc, 0x05, 0x30, 0x7d, 0x06, 0x50, 0xa1, 0xa4, 0xc8, 0x64, 0x91,
    0xd5, 0xb3, 0x12, 0x54, 0xf5, 0x48, 0x50, 0xd5, 0x7e, 0xe4, 0xec, 0xfb,
    0x69, 0xc9, 0xa4, 0xd2, 0x68, 0x57, 0x0a, 0xc4,
};
static const unsigned char kat501_retbits[] = {
    0x3c, 0xb7, 0x70, 0x2d, 0x62, 0xe7, 0x6b, 0xee, 0xc9, 0x2f, 0x66, 0xc7,
    0x2c, 0x63, 0x13, 0x14, 0x3c, 0x56, 0x2b, 0x52, 0xee, 0x94, 0x84, 0xc8,
    0x20, 0x0d, 0x33, 0x15, 0x09, 0xcb, 0xe2, 0xd0, 0xf4, 0xcb, 0x4d, 0xba,
    0x7d, 0xfc, 0xd4, 0x18, 0x39, 0xf2, 0xe7, 0x67, 0x49, 0xab, 0x80, 0x15,
    0x61, 0x4a, 0x14, 0xaf, 0xfd, 0xdc, 0xff, 0x0f, 0xdd, 0x4e, 0xb1, 0x82,
    0xf2, 0xbc, 0x48, 0x4d,
};
static const struct drbg_kat_no_reseed kat501_t = {
    5, kat501_entropyin, kat501_nonce, kat501_persstr,
    kat501_addin0, kat501_addin1, kat501_retbits
};
static const struct drbg_kat kat501 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat501_t
};

static const unsigned char kat502_entropyin[] = {
    0x18, 0x28, 0xce, 0x53, 0x41, 0xed, 0x77, 0x1c, 0x2c, 0x8e, 0x68, 0xef,
    0x5d, 0x57, 0x28, 0xa5, 0x25, 0xa3, 0xf7, 0x65, 0x53, 0x63, 0xc5, 0xec,
    0x01, 0xc3, 0xfc, 0xb3, 0x0c, 0x0d, 0x10, 0x25,
};
static const unsigned char kat502_nonce[] = {
    0xfd, 0x4c, 0x5d, 0x43, 0x54, 0x5e, 0x88, 0x56, 0xb6, 0x1c, 0xa4, 0x23,
    0xe9, 0x58, 0x0c, 0x7d,
};
static const unsigned char kat502_persstr[] = {0};
static const unsigned char kat502_addin0[] = {
    0x38, 0xc2, 0x81, 0x99, 0x54, 0x46, 0xe6, 0xc2, 0x73, 0x63, 0x00, 0x80,
    0x08, 0xf9, 0xb2, 0x8f, 0xcb, 0x5c, 0x00, 0x38, 0x3a, 0x20, 0x95, 0x87,
    0x85, 0x5c, 0x1d, 0x81, 0x9d, 0x48, 0xbd, 0xa0,
};
static const unsigned char kat502_addin1[] = {
    0x9b, 0x1f, 0x6b, 0x84, 0xc4, 0x63, 0x2a, 0x5b, 0x21, 0x5f, 0xa3, 0xf8,
    0x41, 0x10, 0xc4, 0x5b, 0x6b, 0xd5, 0x3b, 0xcc, 0xca, 0x79, 0x2b, 0x4c,
    0x4d, 0x20, 0x73, 0xcc, 0xc9, 0x9e, 0xe3, 0x63,
};
static const unsigned char kat502_retbits[] = {
    0xa3, 0xd7, 0x96, 0x54, 0xcd, 0x4f, 0x56, 0x70, 0xa6, 0x37, 0xad, 0x40,
    0xf4, 0x7f, 0x7f, 0xd5, 0xce, 0x21, 0xd4, 0xd1, 0x8c, 0x6e, 0xab, 0x6d,
    0xd0, 0x2c, 0x79, 0x14, 0x28, 0xb6, 0xa9, 0x65, 0xd3, 0xf7, 0xeb, 0x0e,
    0x62, 0x03, 0xcc, 0xb8, 0xdd, 0xc0, 0x6f, 0x3e, 0x29, 0x8a, 0xb3, 0x13,
    0x16, 0x12, 0x94, 0x52, 0x2f, 0x14, 0x6d, 0x30, 0x84, 0x4c, 0xa7, 0x74,
    0x2e, 0x02, 0x20, 0x7e,
};
static const struct drbg_kat_no_reseed kat502_t = {
    6, kat502_entropyin, kat502_nonce, kat502_persstr,
    kat502_addin0, kat502_addin1, kat502_retbits
};
static const struct drbg_kat kat502 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat502_t
};

static const unsigned char kat503_entropyin[] = {
    0x7c, 0x15, 0xca, 0xae, 0x3d, 0x1d, 0x58, 0xac, 0xde, 0x9f, 0xf7, 0x4d,
    0x35, 0x16, 0xc8, 0xc2, 0xe9, 0x60, 0xec, 0xa6, 0xda, 0xa1, 0x59, 0x6f,
    0x7b, 0x01, 0xd0, 0x8e, 0x91, 0x76, 0xb1, 0xa3,
};
static const unsigned char kat503_nonce[] = {
    0xf0, 0x86, 0x3c, 0x9d, 0x0f, 0x60, 0xaf, 0xf6, 0xbd, 0x1a, 0xbe, 0x2e,
    0xa2, 0x4c, 0xa9, 0x5c,
};
static const unsigned char kat503_persstr[] = {0};
static const unsigned char kat503_addin0[] = {
    0x6e, 0x55, 0xd1, 0x3b, 0xfc, 0x16, 0x62, 0xb7, 0xfe, 0xfb, 0x1e, 0xc2,
    0x22, 0xe3, 0x86, 0x6b, 0x4d, 0xf1, 0xb0, 0xba, 0xed, 0xc3, 0x55, 0xf4,
    0xa5, 0x23, 0xdb, 0x43, 0x45, 0x60, 0x23, 0x3f,
};
static const unsigned char kat503_addin1[] = {
    0x31, 0xa7, 0x1c, 0x65, 0xdb, 0xab, 0xfd, 0x7c, 0x06, 0x2c, 0xb2, 0xe1,
    0x91, 0x17, 0x77, 0x38, 0x7b, 0xb9, 0x97, 0x09, 0x0b, 0x43, 0xe5, 0x3d,
    0xc9, 0x54, 0x34, 0x65, 0xa0, 0xee, 0xa6, 0xb0,
};
static const unsigned char kat503_retbits[] = {
    0xc3, 0x65, 0xb1, 0xda, 0xaa, 0x9c, 0xfd, 0x4c, 0xa1, 0x68, 0x10, 0x1d,
    0x99, 0x6f, 0x7b, 0x89, 0x82, 0xec, 0xd0, 0x63, 0x85, 0x81, 0x11, 0x1c,
    0x9b, 0xfd, 0xa5, 0x28, 0x9a, 0x30, 0xc4, 0x19, 0xea, 0x77, 0xc3, 0x13,
    0x26, 0x47, 0x8b, 0x07, 0x2f, 0xaa, 0x0f, 0x00, 0x31, 0x23, 0x85, 0xdc,
    0x0a, 0x3b, 0xf2, 0x6b, 0x6c, 0xd4, 0xff, 0xf0, 0xce, 0x2e, 0xfe, 0x4b,
    0x0b, 0x55, 0x7a, 0x5e,
};
static const struct drbg_kat_no_reseed kat503_t = {
    7, kat503_entropyin, kat503_nonce, kat503_persstr,
    kat503_addin0, kat503_addin1, kat503_retbits
};
static const struct drbg_kat kat503 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat503_t
};

static const unsigned char kat504_entropyin[] = {
    0x73, 0xc1, 0x6b, 0xa1, 0x9c, 0xfc, 0x3f, 0x8d, 0x3d, 0x41, 0x71, 0xa2,
    0xa5, 0xff, 0x0e, 0xbd, 0x8e, 0x1b, 0x1e, 0x5e, 0x15, 0x2c, 0x1b, 0x45,
    0x9a, 0x58, 0x88, 0x76, 0xd2, 0xdb, 0x5f, 0xb6,
};
static const unsigned char kat504_nonce[] = {
    0xf2, 0x98, 0x62, 0xb5, 0x7a, 0x08, 0x96, 0xb5, 0x64, 0x44, 0x93, 0xed,
    0x97, 0x0e, 0x14, 0x2c,
};
static const unsigned char kat504_persstr[] = {0};
static const unsigned char kat504_addin0[] = {
    0x7e, 0x48, 0x1b, 0xda, 0x6e, 0x72, 0x20, 0x7b, 0x6d, 0xf7, 0x4f, 0x30,
    0xf5, 0x9c, 0xfe, 0xa6, 0x99, 0xba, 0xaf, 0xdf, 0xb8, 0xc5, 0x04, 0xf8,
    0x74, 0xc9, 0x8d, 0xd3, 0xf2, 0x15, 0x78, 0x7c,
};
static const unsigned char kat504_addin1[] = {
    0x00, 0x3e, 0x69, 0x41, 0x85, 0xf0, 0x2d, 0x97, 0x38, 0xa7, 0x67, 0xb4,
    0x4a, 0x55, 0xf7, 0x95, 0x54, 0xc1, 0x46, 0x0a, 0x10, 0xc0, 0x67, 0x6a,
    0x7d, 0x22, 0x7e, 0xe4, 0x84, 0xc5, 0x8d, 0x88,
};
static const unsigned char kat504_retbits[] = {
    0xbc, 0xf8, 0x50, 0xfc, 0xd2, 0xff, 0x56, 0xdf, 0x24, 0xe4, 0x91, 0xea,
    0x33, 0x69, 0xb6, 0x5d, 0x69, 0x17, 0xc4, 0xaa, 0x34, 0x59, 0x28, 0x29,
    0xa7, 0x77, 0xf9, 0x5d, 0xd3, 0xd8, 0x6a, 0x3b, 0x8e, 0x88, 0x4d, 0x9d,
    0xa6, 0xe5, 0x32, 0x02, 0x90, 0xd1, 0xfd, 0x89, 0x95, 0xf1, 0xb5, 0x6e,
    0xf4, 0x7a, 0xec, 0x49, 0x2d, 0xc7, 0x20, 0xa2, 0x11, 0x89, 0x8d, 0xbc,
    0x5f, 0x50, 0x75, 0x4a,
};
static const struct drbg_kat_no_reseed kat504_t = {
    8, kat504_entropyin, kat504_nonce, kat504_persstr,
    kat504_addin0, kat504_addin1, kat504_retbits
};
static const struct drbg_kat kat504 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat504_t
};

static const unsigned char kat505_entropyin[] = {
    0xba, 0x82, 0x14, 0x84, 0x74, 0x0f, 0xce, 0xf6, 0x01, 0x65, 0x28, 0xfc,
    0x68, 0xb8, 0x80, 0x96, 0xd1, 0x03, 0xb2, 0x4d, 0x1f, 0x08, 0xb9, 0xf9,
    0xd9, 0xfd, 0x12, 0xdd, 0xce, 0xfc, 0x1a, 0x87,
};
static const unsigned char kat505_nonce[] = {
    0xef, 0x53, 0xbe, 0x20, 0xeb, 0x5a, 0xaa, 0x86, 0x54, 0x49, 0xd7, 0x4c,
    0x48, 0x28, 0x18, 0x15,
};
static const unsigned char kat505_persstr[] = {0};
static const unsigned char kat505_addin0[] = {
    0xee, 0x7a, 0x7d, 0x73, 0x8c, 0x39, 0x09, 0x1c, 0xeb, 0x31, 0x91, 0x1b,
    0x61, 0x49, 0xdb, 0x26, 0xc0, 0x57, 0x13, 0x2a, 0xa7, 0xe4, 0x06, 0x25,
    0xab, 0x4c, 0x2e, 0x24, 0xa1, 0xae, 0x1e, 0xc1,
};
static const unsigned char kat505_addin1[] = {
    0x71, 0xd4, 0xe3, 0x7d, 0x8e, 0x0b, 0xbc, 0xae, 0x40, 0x81, 0x3c, 0x33,
    0x09, 0x55, 0x81, 0xbf, 0xfd, 0xb1, 0x5d, 0x42, 0xc6, 0xba, 0xfd, 0x75,
    0x81, 0xbb, 0xca, 0x26, 0xbe, 0xa7, 0xba, 0xdf,
};
static const unsigned char kat505_retbits[] = {
    0xfd, 0xb8, 0x32, 0xec, 0xb3, 0x09, 0xab, 0x8f, 0xd4, 0x6d, 0x1a, 0x5b,
    0xc6, 0x4c, 0xb3, 0x7a, 0x19, 0x9a, 0x90, 0x68, 0x47, 0x85, 0xd3, 0x2d,
    0x13, 0x39, 0xd7, 0x5c, 0xed, 0xa5, 0x33, 0x8a, 0xe6, 0x96, 0x1e, 0xe4,
    0x1c, 0xd1, 0x63, 0x63, 0x32, 0x46, 0xd6, 0x7f, 0xa6, 0x03, 0xc4, 0xff,
    0xda, 0xd5, 0x6a, 0xbc, 0x3e, 0xec, 0xfd, 0x9e, 0x93, 0x97, 0xb6, 0x29,
    0xec, 0xb9, 0x65, 0xbd,
};
static const struct drbg_kat_no_reseed kat505_t = {
    9, kat505_entropyin, kat505_nonce, kat505_persstr,
    kat505_addin0, kat505_addin1, kat505_retbits
};
static const struct drbg_kat kat505 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat505_t
};

static const unsigned char kat506_entropyin[] = {
    0xaf, 0xd3, 0x91, 0xb5, 0x0a, 0x0f, 0x2e, 0x81, 0xfe, 0xf3, 0xe0, 0x44,
    0xdb, 0x92, 0x68, 0xd7, 0x85, 0x9c, 0x2b, 0x61, 0xd5, 0x3c, 0x7d, 0x19,
    0x02, 0xa9, 0xaf, 0x69, 0xf0, 0x41, 0x42, 0x81,
};
static const unsigned char kat506_nonce[] = {
    0xf1, 0xbb, 0x5a, 0x11, 0x69, 0x44, 0x5a, 0x2d, 0xd8, 0xc5, 0xa8, 0x45,
    0x27, 0xc1, 0x0f, 0x7f,
};
static const unsigned char kat506_persstr[] = {0};
static const unsigned char kat506_addin0[] = {
    0xb5, 0x94, 0xb0, 0x46, 0xa5, 0x79, 0x60, 0x94, 0x1e, 0x83, 0x4a, 0x00,
    0xa2, 0xcd, 0xc7, 0xd6, 0xf2, 0x43, 0x0b, 0x93, 0xf5, 0xe2, 0x69, 0x40,
    0xb2, 0xd6, 0x94, 0x1f, 0x42, 0x4d, 0x4b, 0xac,
};
static const unsigned char kat506_addin1[] = {
    0xe7, 0xd7, 0x67, 0xd3, 0xa8, 0x1e, 0x39, 0xb6, 0xb0, 0xe9, 0x8f, 0xf3,
    0xed, 0x0c, 0xe6, 0xff, 0xbb, 0x43, 0xdd, 0x5b, 0x98, 0x6b, 0x9c, 0xb8,
    0xc2, 0x74, 0x2b, 0xdf, 0xa0, 0x2f, 0xae, 0xd1,
};
static const unsigned char kat506_retbits[] = {
    0x28, 0x36, 0x80, 0x0d, 0x4e, 0x0a, 0x01, 0x80, 0xb2, 0x79, 0xf0, 0xee,
    0x1f, 0xf6, 0x14, 0x8b, 0x0b, 0x36, 0x77, 0x09, 0xf9, 0xa9, 0xa3, 0xd1,
    0x03, 0x03, 0xd4, 0x8e, 0xc8, 0x03, 0x6a, 0x53, 0x17, 0x92, 0xd3, 0x21,
    0x0d, 0x9d, 0x67, 0x17, 0x46, 0xe2, 0x03, 0xfc, 0x07, 0xcd, 0xdb, 0x29,
    0xf7, 0x87, 0x77, 0x77, 0x75, 0xed, 0xe3, 0xc9, 0x8c, 0x50, 0x88, 0x5d,
    0xf7, 0xdd, 0xfe, 0x6b,
};
static const struct drbg_kat_no_reseed kat506_t = {
    10, kat506_entropyin, kat506_nonce, kat506_persstr,
    kat506_addin0, kat506_addin1, kat506_retbits
};
static const struct drbg_kat kat506 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat506_t
};

static const unsigned char kat507_entropyin[] = {
    0x05, 0x92, 0x51, 0x95, 0xf9, 0x73, 0x5e, 0x38, 0x45, 0x7a, 0xce, 0xf4,
    0xb7, 0x26, 0xcb, 0xc5, 0x8f, 0x83, 0xca, 0x37, 0x40, 0xc5, 0x58, 0x31,
    0x76, 0xb1, 0x13, 0x5c, 0x15, 0x84, 0xda, 0xed,
};
static const unsigned char kat507_nonce[] = {
    0xd5, 0xcb, 0xe9, 0x50, 0x34, 0xea, 0xad, 0x6e, 0x10, 0xe5, 0x5a, 0x05,
    0x15, 0x5a, 0xe6, 0xb9,
};
static const unsigned char kat507_persstr[] = {0};
static const unsigned char kat507_addin0[] = {
    0xae, 0x5e, 0x51, 0x1f, 0x35, 0x7e, 0x36, 0x2e, 0x39, 0x20, 0xca, 0x3e,
    0x7c, 0xda, 0xd2, 0xae, 0x44, 0x91, 0x2d, 0x64, 0xbc, 0x09, 0x62, 0x9d,
    0x2e, 0xeb, 0x9c, 0xc5, 0x40, 0xb3, 0x04, 0xf6,
};
static const unsigned char kat507_addin1[] = {
    0x3f, 0xe1, 0xaa, 0xf5, 0xc7, 0x91, 0xb8, 0x38, 0x65, 0xe5, 0x6b, 0xd1,
    0x9e, 0xd1, 0x9a, 0x01, 0x9d, 0x25, 0x90, 0xc0, 0xcd, 0x59, 0x6e, 0xb7,
    0x2f, 0x46, 0x44, 0xc8, 0xc4, 0x14, 0x90, 0xf4,
};
static const unsigned char kat507_retbits[] = {
    0x31, 0x23, 0xc4, 0xe8, 0x1a, 0x8d, 0x06, 0x71, 0xfe, 0xe1, 0xd5, 0x28,
    0x32, 0x8e, 0xad, 0x1c, 0xa4, 0xde, 0x8d, 0x07, 0xb3, 0xe5, 0x70, 0xcd,
    0x0c, 0x5a, 0xa1, 0xc6, 0x66, 0x15, 0x8b, 0x86, 0xc3, 0xb5, 0x48, 0xfa,
    0xe6, 0x00, 0xd0, 0x02, 0x50, 0xf3, 0xa8, 0x56, 0x55, 0xca, 0xe3, 0xf4,
    0x09, 0x99, 0xe4, 0x39, 0x32, 0xea, 0x75, 0x64, 0x7a, 0x4d, 0xf1, 0xc7,
    0x5a, 0x19, 0xfa, 0x80,
};
static const struct drbg_kat_no_reseed kat507_t = {
    11, kat507_entropyin, kat507_nonce, kat507_persstr,
    kat507_addin0, kat507_addin1, kat507_retbits
};
static const struct drbg_kat kat507 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat507_t
};

static const unsigned char kat508_entropyin[] = {
    0xae, 0x13, 0xa2, 0xd2, 0x37, 0x34, 0xb2, 0xd6, 0x57, 0xf6, 0x7d, 0x6b,
    0xab, 0x36, 0xe1, 0x12, 0x1d, 0x5d, 0x2c, 0x1e, 0xdb, 0x4e, 0x36, 0xf4,
    0xef, 0x2f, 0xfb, 0xc1, 0x0b, 0x62, 0x8c, 0xfb,
};
static const unsigned char kat508_nonce[] = {
    0x6a, 0x1d, 0xb3, 0x53, 0xe3, 0xc5, 0xa6, 0xe6, 0x65, 0x7c, 0x86, 0xca,
    0xa7, 0x26, 0x33, 0xd8,
};
static const unsigned char kat508_persstr[] = {0};
static const unsigned char kat508_addin0[] = {
    0xcb, 0xfc, 0xce, 0x20, 0x9f, 0xde, 0x42, 0x08, 0xef, 0xab, 0xb2, 0x41,
    0x79, 0x52, 0xb1, 0xc9, 0xfc, 0x22, 0x84, 0x3c, 0xe2, 0x17, 0x46, 0x08,
    0xbe, 0x04, 0xa4, 0x5f, 0x1d, 0x98, 0x27, 0xc2,
};
static const unsigned char kat508_addin1[] = {
    0xa8, 0xd0, 0xca, 0x78, 0x88, 0x00, 0x85, 0x2c, 0x5d, 0xf5, 0xea, 0xbb,
    0xea, 0xf1, 0xf5, 0xfb, 0xd9, 0x4b, 0xeb, 0x74, 0x61, 0x03, 0xb0, 0x43,
    0x68, 0x76, 0xbc, 0x80, 0x92, 0xda, 0x8b, 0xad,
};
static const unsigned char kat508_retbits[] = {
    0xd3, 0x86, 0x88, 0x90, 0xe7, 0x38, 0xbb, 0xf5, 0x60, 0x2c, 0xa4, 0x6e,
    0x00, 0x9d, 0xab, 0xaf, 0xb1, 0x6f, 0x56, 0x86, 0xfd, 0xdb, 0x78, 0x7c,
    0x8c, 0xca, 0x32, 0x89, 0x23, 0x60, 0xf6, 0x10, 0x99, 0xf0, 0xd7, 0x7c,
    0xce, 0x39, 0xdc, 0x8f, 0x6e, 0x53, 0x05, 0x40, 0xfc, 0x5f, 0xab, 0x3e,
    0xb6, 0x57, 0x11, 0x48, 0x47, 0xd3, 0xa3, 0x95, 0xfa, 0xf4, 0x0f, 0x8a,
    0xf6, 0x4f, 0x4d, 0x2d,
};
static const struct drbg_kat_no_reseed kat508_t = {
    12, kat508_entropyin, kat508_nonce, kat508_persstr,
    kat508_addin0, kat508_addin1, kat508_retbits
};
static const struct drbg_kat kat508 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat508_t
};

static const unsigned char kat509_entropyin[] = {
    0xfa, 0x9f, 0x4f, 0xd3, 0x46, 0x55, 0x5d, 0xb3, 0x5d, 0x3e, 0xe8, 0xf1,
    0x57, 0x95, 0x7c, 0x81, 0x91, 0xe3, 0xe8, 0x61, 0x84, 0x4b, 0x4f, 0x50,
    0x00, 0xda, 0xb4, 0x82, 0x61, 0xf8, 0x5f, 0x74,
};
static const unsigned char kat509_nonce[] = {
    0x5e, 0x18, 0x8b, 0x4e, 0xe0, 0x78, 0xd5, 0x7f, 0x91, 0xd7, 0xde, 0x21,
    0x9f, 0xc4, 0x25, 0xed,
};
static const unsigned char kat509_persstr[] = {0};
static const unsigned char kat509_addin0[] = {
    0x5e, 0xb2, 0x48, 0xcd, 0x13, 0x8b, 0x8f, 0xef, 0xa2, 0xb7, 0xe8, 0xfc,
    0x75, 0x2c, 0x0f, 0xbc, 0xb0, 0x82, 0xbe, 0x51, 0xd0, 0xf4, 0xe2, 0xa9,
    0x88, 0xbf, 0x5f, 0x87, 0xfc, 0x73, 0x74, 0x47,
};
static const unsigned char kat509_addin1[] = {
    0x7c, 0xf9, 0x60, 0x6d, 0x30, 0x2f, 0x53, 0x56, 0x7e, 0x45, 0xb9, 0x8c,
    0x38, 0xd4, 0xda, 0xae, 0x23, 0x2c, 0x92, 0x5c, 0xb9, 0x2e, 0x54, 0xc7,
    0x4b, 0xcf, 0x14, 0x10, 0x4e, 0x32, 0xa1, 0x13,
};
static const unsigned char kat509_retbits[] = {
    0x89, 0x58, 0xd2, 0x45, 0xd7, 0x02, 0x35, 0x55, 0x5a, 0x1e, 0xcc, 0xf4,
    0xfa, 0xf2, 0x00, 0x9d, 0x32, 0x76, 0x2e, 0x33, 0x24, 0x94, 0xef, 0xe5,
    0xf4, 0x8f, 0xa5, 0xd7, 0xff, 0x30, 0xc7, 0xcb, 0xf0, 0x7c, 0x9b, 0xcc,
    0x3e, 0x70, 0xd6, 0x2b, 0x94, 0xa4, 0xed, 0x21, 0xdb, 0x5e, 0x39, 0xcd,
    0xd0, 0x1a, 0x1e, 0x96, 0x68, 0x8b, 0x3b, 0xb9, 0xb2, 0xc4, 0xac, 0x0a,
    0x7a, 0xfe, 0xec, 0x24,
};
static const struct drbg_kat_no_reseed kat509_t = {
    13, kat509_entropyin, kat509_nonce, kat509_persstr,
    kat509_addin0, kat509_addin1, kat509_retbits
};
static const struct drbg_kat kat509 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat509_t
};

static const unsigned char kat510_entropyin[] = {
    0xc7, 0x64, 0xbc, 0xcc, 0x19, 0x05, 0x3d, 0x7f, 0x12, 0xdf, 0x71, 0x01,
    0x1b, 0x35, 0x87, 0x8d, 0xd0, 0xbb, 0x58, 0x7d, 0xeb, 0x88, 0x7f, 0x8a,
    0x05, 0xf0, 0x40, 0xec, 0x45, 0xe3, 0x24, 0xbc,
};
static const unsigned char kat510_nonce[] = {
    0xa4, 0xb1, 0xa9, 0xff, 0x04, 0x91, 0x72, 0x47, 0x6a, 0xb9, 0x9f, 0x7a,
    0xe0, 0x64, 0x5e, 0x6c,
};
static const unsigned char kat510_persstr[] = {0};
static const unsigned char kat510_addin0[] = {
    0xb6, 0xf2, 0xaa, 0x8a, 0x99, 0x3a, 0xa0, 0x42, 0x52, 0xd6, 0xc0, 0x14,
    0xea, 0xc8, 0xad, 0x0b, 0x90, 0xcf, 0x0e, 0xd4, 0x38, 0x88, 0x81, 0x73,
    0xe1, 0x5e, 0xab, 0x28, 0x0f, 0x73, 0xab, 0xe0,
};
static const unsigned char kat510_addin1[] = {
    0xc6, 0x38, 0x4c, 0x9a, 0x2c, 0x9d, 0x8f, 0xc4, 0xc7, 0x42, 0x2f, 0xe4,
    0x0e, 0x43, 0x0e, 0x4f, 0x36, 0xe6, 0xae, 0x9b, 0xd6, 0xb0, 0x39, 0xd5,
    0x5a, 0xbf, 0xdd, 0x8f, 0x05, 0xd6, 0xa2, 0x0e,
};
static const unsigned char kat510_retbits[] = {
    0x63, 0x00, 0xf2, 0x89, 0x41, 0xdc, 0x17, 0x3a, 0x9f, 0x6f, 0x5e, 0x4c,
    0x11, 0xf1, 0xd1, 0xd7, 0xd4, 0xfd, 0x7f, 0xbb, 0xaf, 0x6f, 0xc3, 0xa9,
    0x7a, 0xcd, 0xfc, 0x00, 0x8a, 0x00, 0x10, 0x9a, 0x67, 0x0e, 0x48, 0x7e,
    0x5d, 0xce, 0x42, 0x3a, 0x3e, 0xee, 0x48, 0x82, 0x34, 0x7e, 0xf2, 0x64,
    0x49, 0x88, 0x8c, 0x8d, 0xd7, 0x3b, 0x9a, 0xaa, 0x73, 0xd1, 0x94, 0xc1,
    0x71, 0x46, 0x9a, 0xf6,
};
static const struct drbg_kat_no_reseed kat510_t = {
    14, kat510_entropyin, kat510_nonce, kat510_persstr,
    kat510_addin0, kat510_addin1, kat510_retbits
};
static const struct drbg_kat kat510 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat510_t
};

static const unsigned char kat511_entropyin[] = {
    0x54, 0x16, 0xe7, 0x7b, 0x5e, 0x1d, 0x87, 0x2d, 0x4f, 0xf9, 0x19, 0x73,
    0xb1, 0xbe, 0x66, 0xbc, 0x07, 0xf4, 0xa9, 0x9e, 0x30, 0xdb, 0x7d, 0x00,
    0x06, 0xda, 0x00, 0x6f, 0xcf, 0xb0, 0x82, 0xdb,
};
static const unsigned char kat511_nonce[] = {
    0x7a, 0x81, 0x1c, 0xe6, 0x2b, 0x9f, 0xd3, 0x4a, 0xf1, 0x86, 0xb2, 0xb3,
    0xe5, 0x0e, 0xaf, 0x5d,
};
static const unsigned char kat511_persstr[] = {
    0x71, 0xee, 0x0c, 0x76, 0x99, 0xac, 0x0e, 0x80, 0x56, 0x32, 0xf2, 0x05,
    0x8d, 0xe3, 0x8b, 0xf8, 0x72, 0xb8, 0x34, 0x0f, 0x89, 0x99, 0x8f, 0x7a,
    0x8a, 0x2a, 0xd4, 0xac, 0x04, 0x5a, 0xe6, 0xef,
};
static const unsigned char kat511_addin0[] = {0};
static const unsigned char kat511_addin1[] = {0};
static const unsigned char kat511_retbits[] = {
    0x68, 0xf5, 0x85, 0x9c, 0xf7, 0x6f, 0x94, 0xc4, 0x45, 0xd9, 0xfc, 0xd3,
    0x4f, 0xc1, 0x7a, 0xc2, 0x24, 0xc3, 0xd7, 0xd7, 0xc2, 0xfc, 0x38, 0xfa,
    0xaf, 0x3c, 0x24, 0xbe, 0x6c, 0xd3, 0xcd, 0x93, 0xb7, 0xf9, 0xd8, 0xa6,
    0x14, 0x6f, 0x5a, 0xc8, 0x3a, 0xc1, 0xd7, 0xb1, 0xb2, 0xb7, 0xe7, 0xec,
    0xbc, 0x1a, 0x2e, 0x38, 0x76, 0x0e, 0xf8, 0x6a, 0x57, 0x7d, 0x40, 0x2d,
    0x85, 0x99, 0x0d, 0x9b,
};
static const struct drbg_kat_no_reseed kat511_t = {
    0, kat511_entropyin, kat511_nonce, kat511_persstr,
    kat511_addin0, kat511_addin1, kat511_retbits
};
static const struct drbg_kat kat511 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat511_t
};

static const unsigned char kat512_entropyin[] = {
    0x70, 0x8e, 0xca, 0x2e, 0x3a, 0x92, 0x65, 0xa7, 0x90, 0x60, 0x7e, 0xdb,
    0xe0, 0x5f, 0xe3, 0x42, 0x66, 0x3f, 0x84, 0xc6, 0x61, 0x7e, 0xda, 0x14,
    0xf2, 0x52, 0x76, 0xa9, 0x43, 0x90, 0x1f, 0xda,
};
static const unsigned char kat512_nonce[] = {
    0x75, 0xaf, 0xb4, 0x9a, 0x18, 0x4b, 0x23, 0x50, 0x6b, 0xe1, 0x49, 0x26,
    0xcd, 0x4a, 0x03, 0xf0,
};
static const unsigned char kat512_persstr[] = {
    0xcb, 0xb4, 0x8e, 0xf8, 0x41, 0x46, 0xc1, 0x0e, 0x02, 0x24, 0x0d, 0x87,
    0x40, 0xd3, 0x48, 0x7b, 0x6a, 0x42, 0x08, 0x40, 0x53, 0x83, 0xc0, 0x1a,
    0x66, 0x4e, 0xc7, 0xd3, 0xad, 0xa0, 0x7e, 0x2d,
};
static const unsigned char kat512_addin0[] = {0};
static const unsigned char kat512_addin1[] = {0};
static const unsigned char kat512_retbits[] = {
    0x26, 0xb0, 0xaa, 0x6e, 0x82, 0x2c, 0x4c, 0xc9, 0x12, 0xcf, 0x1d, 0xba,
    0xe6, 0x69, 0xc7, 0xda, 0xd0, 0xbd, 0xcf, 0xf6, 0x5f, 0x22, 0x81, 0x3a,
    0xfd, 0x06, 0x22, 0x5b, 0x7f, 0xf7, 0x99, 0xf7, 0x80, 0x3b, 0x3a, 0xd4,
    0x8b, 0xc8, 0x8d, 0x2b, 0xe0, 0xf5, 0xa3, 0x57, 0xf6, 0x20, 0xcc, 0x61,
    0x7f, 0x44, 0x6f, 0xc6, 0xd2, 0x12, 0x59, 0x2a, 0xda, 0x69, 0xb7, 0xdc,
    0x8f, 0xf4, 0xa2, 0x22,
};
static const struct drbg_kat_no_reseed kat512_t = {
    1, kat512_entropyin, kat512_nonce, kat512_persstr,
    kat512_addin0, kat512_addin1, kat512_retbits
};
static const struct drbg_kat kat512 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat512_t
};

static const unsigned char kat513_entropyin[] = {
    0x44, 0xcc, 0x6b, 0x44, 0x33, 0xce, 0xc6, 0x15, 0xc3, 0xc2, 0x14, 0xe1,
    0x66, 0xc7, 0xdc, 0xff, 0x25, 0x8f, 0x8c, 0xfe, 0x57, 0x48, 0xe6, 0x42,
    0x32, 0x1c, 0xda, 0x2f, 0x7d, 0xb4, 0x26, 0xe3,
};
static const unsigned char kat513_nonce[] = {
    0x6a, 0x25, 0x26, 0x95, 0x4b, 0x5d, 0xf9, 0x89, 0xd6, 0x1e, 0x1f, 0xaf,
    0x93, 0xdd, 0xa2, 0xae,
};
static const unsigned char kat513_persstr[] = {
    0x88, 0x22, 0x63, 0x13, 0xc7, 0xf1, 0xec, 0x03, 0xcd, 0xe3, 0x77, 0x97,
    0x0c, 0x8e, 0xa7, 0xd7, 0x41, 0xa9, 0xf2, 0x1a, 0x8f, 0x54, 0xb6, 0xb9,
    0x70, 0x43, 0xbc, 0x3e, 0x8d, 0xa4, 0x0b, 0x1e,
};
static const unsigned char kat513_addin0[] = {0};
static const unsigned char kat513_addin1[] = {0};
static const unsigned char kat513_retbits[] = {
    0xc1, 0x95, 0x6c, 0x41, 0x95, 0xad, 0xfc, 0x3e, 0xe7, 0x15, 0x82, 0xab,
    0x2c, 0x63, 0xed, 0xc0, 0xa7, 0x8a, 0xf4, 0x9e, 0xcc, 0x23, 0xa3, 0xdd,
    0xbc, 0xf2, 0xdf, 0xaf, 0x80, 0xc7, 0x61, 0xfd, 0x63, 0x43, 0xaf, 0x6d,
    0x14, 0x31, 0x0e, 0x71, 0x9d, 0x8c, 0xd3, 0xc6, 0xbb, 0xb4, 0x91, 0xc8,
    0x69, 0x0a, 0x7d, 0xd8, 0xa1, 0x68, 0xcd, 0x8a, 0x48, 0x02, 0x17, 0xe5,
    0xdd, 0x20, 0x88, 0xaf,
};
static const struct drbg_kat_no_reseed kat513_t = {
    2, kat513_entropyin, kat513_nonce, kat513_persstr,
    kat513_addin0, kat513_addin1, kat513_retbits
};
static const struct drbg_kat kat513 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat513_t
};

static const unsigned char kat514_entropyin[] = {
    0x54, 0xcc, 0xb1, 0xe5, 0xf0, 0x44, 0x44, 0x7d, 0xce, 0x52, 0xa4, 0x70,
    0xf4, 0x7f, 0xe2, 0x68, 0x27, 0x17, 0xdd, 0x29, 0x6d, 0x64, 0x49, 0x1e,
    0xe2, 0xac, 0xc9, 0x9e, 0x9a, 0xd6, 0x56, 0x6f,
};
static const unsigned char kat514_nonce[] = {
    0xff, 0x4c, 0xd3, 0x18, 0x56, 0x11, 0xcb, 0xe0, 0x67, 0x84, 0xe3, 0x25,
    0x80, 0xb2, 0xf2, 0x3c,
};
static const unsigned char kat514_persstr[] = {
    0x13, 0x2e, 0xf5, 0x42, 0xf9, 0x07, 0xb8, 0x4c, 0x44, 0x3d, 0x19, 0x73,
    0xb3, 0x90, 0x9b, 0x6d, 0x9a, 0x0d, 0x91, 0x24, 0xd3, 0x8b, 0xd1, 0xe7,
    0xc8, 0x83, 0x3f, 0x48, 0xae, 0xcf, 0xc0, 0x8d,
};
static const unsigned char kat514_addin0[] = {0};
static const unsigned char kat514_addin1[] = {0};
static const unsigned char kat514_retbits[] = {
    0x0f, 0x2f, 0x56, 0xea, 0x8b, 0x91, 0x1c, 0xbe, 0x59, 0xa7, 0xb8, 0xfa,
    0xb1, 0xc7, 0x10, 0xa7, 0xeb, 0xb6, 0xec, 0x9a, 0x09, 0x55, 0x5c, 0xa4,
    0x9c, 0xcd, 0xdd, 0x9a, 0xfd, 0x38, 0xed, 0x61, 0xb8, 0x55, 0xcf, 0x3f,
    0x33, 0xf2, 0xc5, 0xbc, 0x61, 0x6d, 0xf6, 0xcb, 0x17, 0x26, 0x96, 0x84,
    0x83, 0xc6, 0x9c, 0x18, 0x49, 0xe0, 0xf1, 0xb4, 0x6b, 0xa0, 0x29, 0xaa,
    0x6f, 0x5d, 0xeb, 0xdb,
};
static const struct drbg_kat_no_reseed kat514_t = {
    3, kat514_entropyin, kat514_nonce, kat514_persstr,
    kat514_addin0, kat514_addin1, kat514_retbits
};
static const struct drbg_kat kat514 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat514_t
};

static const unsigned char kat515_entropyin[] = {
    0x3d, 0x3f, 0xdd, 0x9d, 0x90, 0xac, 0xbc, 0xee, 0x07, 0x00, 0x2f, 0x17,
    0x37, 0x00, 0x45, 0xfe, 0xb5, 0xea, 0xa3, 0x34, 0xfd, 0x74, 0x59, 0x4e,
    0x11, 0x21, 0x14, 0xd3, 0x92, 0x8d, 0xd5, 0xd9,
};
static const unsigned char kat515_nonce[] = {
    0xf8, 0x50, 0x95, 0x29, 0x4e, 0xbc, 0x5f, 0xdd, 0xf4, 0x49, 0x41, 0xbe,
    0x5f, 0xfa, 0xf1, 0x0c,
};
static const unsigned char kat515_persstr[] = {
    0x59, 0xe2, 0xff, 0xa1, 0x64, 0x73, 0x3f, 0xf1, 0x1b, 0x5a, 0x95, 0xeb,
    0x99, 0xa7, 0x83, 0x66, 0x90, 0x6d, 0xe4, 0xfa, 0xc6, 0x4e, 0x51, 0x24,
    0x00, 0x08, 0x11, 0x16, 0xac, 0xce, 0x53, 0x90,
};
static const unsigned char kat515_addin0[] = {0};
static const unsigned char kat515_addin1[] = {0};
static const unsigned char kat515_retbits[] = {
    0x12, 0x44, 0xe5, 0x32, 0x79, 0x9f, 0x1e, 0xa4, 0xed, 0x32, 0x18, 0x94,
    0xda, 0xc5, 0x1b, 0x3c, 0x78, 0xd2, 0xfa, 0x5f, 0x0e, 0x1c, 0x92, 0x2f,
    0xfd, 0x2f, 0xf6, 0x08, 0x27, 0x54, 0x00, 0x83, 0x4d, 0x03, 0x45, 0x49,
    0x42, 0xd3, 0x1a, 0x20, 0x14, 0xcc, 0xfe, 0x07, 0xc2, 0x35, 0x41, 0x12,
    0x36, 0x3c, 0x60, 0xf4, 0x8d, 0xd1, 0x2b, 0x29, 0xa3, 0x73, 0x41, 0x28,
    0xa5, 0x9b, 0xca, 0x21,
};
static const struct drbg_kat_no_reseed kat515_t = {
    4, kat515_entropyin, kat515_nonce, kat515_persstr,
    kat515_addin0, kat515_addin1, kat515_retbits
};
static const struct drbg_kat kat515 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat515_t
};

static const unsigned char kat516_entropyin[] = {
    0x25, 0x36, 0xd7, 0xea, 0x90, 0x6d, 0x1e, 0xee, 0xa8, 0x3e, 0x1c, 0x9d,
    0xa0, 0x8c, 0xf7, 0x6d, 0x09, 0x5b, 0x7d, 0x4f, 0x55, 0xb4, 0x33, 0xbb,
    0xd5, 0x22, 0x5a, 0xa8, 0x70, 0xcc, 0x3f, 0x50,
};
static const unsigned char kat516_nonce[] = {
    0x17, 0x70, 0x4a, 0x7c, 0x50, 0xf9, 0x37, 0xac, 0xf6, 0x26, 0xe8, 0x54,
    0xe0, 0xb9, 0x5f, 0x80,
};
static const unsigned char kat516_persstr[] = {
    0xb6, 0x1f, 0xcc, 0xd4, 0x0d, 0x9c, 0xeb, 0xc4, 0xb9, 0x27, 0xba, 0x77,
    0x3b, 0x93, 0x29, 0x57, 0xc2, 0x99, 0xf7, 0x79, 0xc4, 0x26, 0x6a, 0x83,
    0xf1, 0x69, 0xb6, 0xde, 0x50, 0x7d, 0x31, 0x27,
};
static const unsigned char kat516_addin0[] = {0};
static const unsigned char kat516_addin1[] = {0};
static const unsigned char kat516_retbits[] = {
    0x8c, 0x3b, 0x0b, 0x71, 0xd6, 0x43, 0x2e, 0x10, 0x22, 0x97, 0x6c, 0x9c,
    0x8e, 0xbb, 0x14, 0x06, 0xb1, 0xda, 0x99, 0x5e, 0x29, 0x37, 0x22, 0x1d,
    0x18, 0xd7, 0x51, 0x81, 0x68, 0x25, 0xfc, 0xa0, 0x64, 0x53, 0x4e, 0x21,
    0x69, 0xcc, 0x63, 0xb5, 0x07, 0x05, 0x29, 0xff, 0x02, 0xb5, 0xee, 0x5b,
    0x70, 0x81, 0xa0, 0x8e, 0xbd, 0xd8, 0x78, 0x62, 0x59, 0x5e, 0xa3, 0x7a,
    0x95, 0xc1, 0xe4, 0xa7,
};
static const struct drbg_kat_no_reseed kat516_t = {
    5, kat516_entropyin, kat516_nonce, kat516_persstr,
    kat516_addin0, kat516_addin1, kat516_retbits
};
static const struct drbg_kat kat516 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat516_t
};

static const unsigned char kat517_entropyin[] = {
    0xfe, 0xe9, 0x82, 0xd8, 0x7d, 0xf4, 0xc3, 0x20, 0x57, 0xcb, 0x49, 0x9d,
    0xf6, 0xea, 0x21, 0xdd, 0x69, 0x16, 0xdd, 0x38, 0x85, 0x0a, 0x87, 0x2a,
    0xbe, 0x34, 0x36, 0x0f, 0x29, 0x14, 0xb5, 0x6c,
};
static const unsigned char kat517_nonce[] = {
    0x5a, 0x1f, 0xf0, 0x69, 0x2d, 0x9a, 0xdc, 0x5e, 0xdb, 0x1a, 0xf7, 0x74,
    0x61, 0xfd, 0x05, 0x31,
};
static const unsigned char kat517_persstr[] = {
    0x1c, 0x0d, 0x18, 0x5f, 0xa4, 0xbe, 0xde, 0xdb, 0x78, 0x1b, 0x8b, 0x73,
    0x61, 0x6d, 0xef, 0x74, 0x40, 0x82, 0x21, 0x72, 0xd0, 0x1a, 0x61, 0x9a,
    0x20, 0x66, 0xb7, 0x9e, 0xfa, 0xce, 0x68, 0x5f,
};
static const unsigned char kat517_addin0[] = {0};
static const unsigned char kat517_addin1[] = {0};
static const unsigned char kat517_retbits[] = {
    0xd7, 0xc2, 0x28, 0xe7, 0x00, 0x2b, 0xc2, 0xa1, 0xa2, 0x3a, 0x16, 0xe4,
    0x89, 0xba, 0x8f, 0x0a, 0x61, 0x62, 0x7a, 0xb4, 0xb2, 0x44, 0x4f, 0x00,
    0xf2, 0x3e, 0xd3, 0x71, 0x7b, 0x87, 0x94, 0x4f, 0xee, 0xbb, 0x16, 0x9e,
    0xe4, 0xe9, 0x79, 0x08, 0x9c, 0x35, 0xe8, 0x3f, 0x24, 0x30, 0x79, 0xe4,
    0x77, 0x19, 0x18, 0x86, 0x2e, 0x15, 0x03, 0x6e, 0xc6, 0xa9, 0xc9, 0xee,
    0xe4, 0xbb, 0x00, 0xb5,
};
static const struct drbg_kat_no_reseed kat517_t = {
    6, kat517_entropyin, kat517_nonce, kat517_persstr,
    kat517_addin0, kat517_addin1, kat517_retbits
};
static const struct drbg_kat kat517 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat517_t
};

static const unsigned char kat518_entropyin[] = {
    0x25, 0x32, 0x7b, 0x05, 0xf8, 0x6e, 0x50, 0x79, 0xab, 0x55, 0x2e, 0x98,
    0x7e, 0xc7, 0xe2, 0x81, 0x62, 0x59, 0x51, 0x09, 0x33, 0xbc, 0x0c, 0x7a,
    0x05, 0xc3, 0x5e, 0x16, 0x3b, 0x47, 0xb4, 0xce,
};
static const unsigned char kat518_nonce[] = {
    0x7e, 0xf8, 0xd6, 0x30, 0x8e, 0xe6, 0xb5, 0x06, 0x1c, 0xbc, 0xb8, 0x50,
    0x48, 0x5a, 0x15, 0xa7,
};
static const unsigned char kat518_persstr[] = {
    0x9a, 0x39, 0xed, 0x86, 0xb2, 0xe8, 0x29, 0x0f, 0x74, 0xca, 0x88, 0x6b,
    0x32, 0x0d, 0x8e, 0x23, 0xd8, 0xa1, 0xd4, 0x95, 0x3c, 0x27, 0x62, 0xf8,
    0x1f, 0x07, 0x1d, 0x84, 0x2c, 0xe9, 0x51, 0xe3,
};
static const unsigned char kat518_addin0[] = {0};
static const unsigned char kat518_addin1[] = {0};
static const unsigned char kat518_retbits[] = {
    0x21, 0x19, 0x71, 0x63, 0x2f, 0x8e, 0x6c, 0x60, 0x82, 0xf1, 0x10, 0x76,
    0xbb, 0x70, 0x7e, 0xc9, 0xc6, 0x5e, 0x34, 0x34, 0xc2, 0x72, 0x7e, 0x59,
    0xa5, 0xd1, 0x3e, 0x54, 0x35, 0x62, 0xa1, 0xda, 0x4d, 0x7b, 0xc2, 0xe4,
    0x83, 0x03, 0x5d, 0x46, 0x68, 0x53, 0x6a, 0xd1, 0xd6, 0x9f, 0x61, 0x19,
    0x04, 0x59, 0x99, 0x52, 0x4d, 0x92, 0xc4, 0x8c, 0x4a, 0xed, 0xe6, 0x22,
    0xee, 0x53, 0x63, 0x8d,
};
static const struct drbg_kat_no_reseed kat518_t = {
    7, kat518_entropyin, kat518_nonce, kat518_persstr,
    kat518_addin0, kat518_addin1, kat518_retbits
};
static const struct drbg_kat kat518 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat518_t
};

static const unsigned char kat519_entropyin[] = {
    0x7b, 0x16, 0x3d, 0xc1, 0x1e, 0x15, 0x6b, 0x1b, 0xb1, 0x70, 0xc2, 0x2d,
    0xcd, 0xd1, 0x21, 0xea, 0xea, 0x30, 0x1d, 0x61, 0x72, 0xe3, 0xc0, 0xb4,
    0xbb, 0xc4, 0xf3, 0x77, 0xbe, 0x04, 0x09, 0x40,
};
static const unsigned char kat519_nonce[] = {
    0x32, 0x33, 0xf1, 0x56, 0x37, 0x28, 0x69, 0xd9, 0x37, 0xf3, 0x69, 0x4c,
    0x50, 0xea, 0x98, 0x32,
};
static const unsigned char kat519_persstr[] = {
    0x7d, 0x61, 0xda, 0xf2, 0x37, 0xbd, 0xea, 0x1b, 0xa3, 0xb0, 0x30, 0x1f,
    0x02, 0xcb, 0xe6, 0x1c, 0xea, 0xa1, 0x88, 0xee, 0xf8, 0x5b, 0x95, 0x84,
    0x24, 0x32, 0x47, 0x0c, 0x65, 0xa1, 0x33, 0x67,
};
static const unsigned char kat519_addin0[] = {0};
static const unsigned char kat519_addin1[] = {0};
static const unsigned char kat519_retbits[] = {
    0x8b, 0x9e, 0x1b, 0x2a, 0xa9, 0x50, 0xd0, 0x16, 0xae, 0xe5, 0x5c, 0x3a,
    0xbf, 0x0c, 0x7f, 0x1d, 0x3c, 0x1c, 0x0d, 0x8a, 0x62, 0xce, 0x5d, 0x59,
    0xe8, 0x4b, 0x72, 0xa1, 0x32, 0x2f, 0x17, 0x46, 0xef, 0xaf, 0x0e, 0x46,
    0xe6, 0x7b, 0x17, 0x66, 0x43, 0xc7, 0xb5, 0x7e, 0x3d, 0xbf, 0xfb, 0xdf,
    0x32, 0xce, 0x7a, 0xff, 0xfd, 0x6f, 0xf7, 0xf2, 0x30, 0x81, 0xae, 0x4c,
    0x4f, 0x97, 0x3e, 0xb0,
};
static const struct drbg_kat_no_reseed kat519_t = {
    8, kat519_entropyin, kat519_nonce, kat519_persstr,
    kat519_addin0, kat519_addin1, kat519_retbits
};
static const struct drbg_kat kat519 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat519_t
};

static const unsigned char kat520_entropyin[] = {
    0x96, 0xcb, 0x5f, 0x53, 0x1e, 0x02, 0xac, 0x25, 0x44, 0x45, 0x67, 0x3d,
    0xe5, 0xe1, 0xdc, 0xad, 0x51, 0x76, 0x10, 0x26, 0xe7, 0x78, 0xa0, 0x33,
    0x21, 0xb4, 0x9c, 0x59, 0x77, 0x45, 0x73, 0x1b,
};
static const unsigned char kat520_nonce[] = {
    0x13, 0x91, 0x98, 0xde, 0x0c, 0xa9, 0x82, 0xc4, 0x9b, 0x8a, 0x88, 0xe5,
    0xab, 0x88, 0x6c, 0xb8,
};
static const unsigned char kat520_persstr[] = {
    0x00, 0x6f, 0x17, 0x2e, 0x51, 0x86, 0xfb, 0x47, 0x9f, 0x3c, 0xd8, 0xe3,
    0x42, 0x5a, 0x75, 0x2b, 0x8f, 0x8d, 0x56, 0xa3, 0xf1, 0x76, 0xc6, 0xfa,
    0xba, 0x6c, 0xa4, 0xa4, 0xdd, 0xea, 0xe6, 0x37,
};
static const unsigned char kat520_addin0[] = {0};
static const unsigned char kat520_addin1[] = {0};
static const unsigned char kat520_retbits[] = {
    0x38, 0x8e, 0x5c, 0x98, 0xc0, 0x54, 0x39, 0x30, 0x84, 0xa9, 0x44, 0x15,
    0xe1, 0xa9, 0xba, 0x32, 0x04, 0x4c, 0x87, 0x45, 0x62, 0xc1, 0x85, 0x39,
    0x9b, 0x96, 0x99, 0x94, 0xb6, 0x40, 0x6f, 0x9e, 0xf8, 0x63, 0x19, 0x10,
    0x2f, 0xd8, 0x9d, 0xf6, 0x70, 0x90, 0x35, 0x67, 0xf8, 0x6d, 0xe8, 0x67,
    0x6b, 0x0f, 0x72, 0x29, 0x8a, 0xa0, 0x01, 0x91, 0x15, 0x12, 0x62, 0xe7,
    0xe0, 0x89, 0x87, 0x91,
};
static const struct drbg_kat_no_reseed kat520_t = {
    9, kat520_entropyin, kat520_nonce, kat520_persstr,
    kat520_addin0, kat520_addin1, kat520_retbits
};
static const struct drbg_kat kat520 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat520_t
};

static const unsigned char kat521_entropyin[] = {
    0x58, 0x5a, 0xf2, 0x7f, 0x7f, 0x52, 0x4a, 0xb4, 0xc5, 0xa7, 0xfc, 0xa4,
    0x3b, 0x70, 0xb4, 0x64, 0xcd, 0x71, 0xd7, 0xc5, 0xbf, 0xb4, 0x45, 0x5e,
    0x6d, 0xbd, 0x10, 0x2d, 0x89, 0x33, 0x5e, 0x89,
};
static const unsigned char kat521_nonce[] = {
    0x0a, 0xc9, 0xcb, 0x4d, 0x14, 0xbf, 0x52, 0x26, 0x16, 0xc6, 0x13, 0x06,
    0x86, 0x98, 0xb1, 0x6b,
};
static const unsigned char kat521_persstr[] = {
    0x0c, 0xa0, 0xed, 0x30, 0x27, 0xd3, 0xb4, 0x7f, 0x78, 0x3f, 0xee, 0xf1,
    0x8a, 0x40, 0x34, 0x0c, 0xc5, 0x85, 0x1b, 0x34, 0x2f, 0x39, 0xf2, 0xb7,
    0xd9, 0x08, 0x91, 0x0b, 0x96, 0x79, 0x2e, 0x3a,
};
static const unsigned char kat521_addin0[] = {0};
static const unsigned char kat521_addin1[] = {0};
static const unsigned char kat521_retbits[] = {
    0xf4, 0xc2, 0xb3, 0xb8, 0x64, 0x71, 0xb2, 0xfb, 0x44, 0x69, 0x92, 0x79,
    0x19, 0x42, 0x15, 0x6d, 0x85, 0xfa, 0xc3, 0xc9, 0x37, 0xa8, 0xe5, 0x0e,
    0x79, 0xac, 0xa7, 0x92, 0xc6, 0x69, 0x50, 0x92, 0xd6, 0x7f, 0x80, 0x7e,
    0xb7, 0xac, 0x0d, 0xae, 0xe3, 0x1d, 0x68, 0xf6, 0x6a, 0x83, 0x48, 0xc3,
    0x3c, 0x31, 0x5e, 0x4b, 0xe1, 0xef, 0x58, 0x6d, 0xbd, 0x40, 0x8e, 0x63,
    0xdb, 0x1d, 0x8a, 0xb0,
};
static const struct drbg_kat_no_reseed kat521_t = {
    10, kat521_entropyin, kat521_nonce, kat521_persstr,
    kat521_addin0, kat521_addin1, kat521_retbits
};
static const struct drbg_kat kat521 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat521_t
};

static const unsigned char kat522_entropyin[] = {
    0x4e, 0x7b, 0x5b, 0x7b, 0x30, 0xb8, 0xb9, 0x0b, 0x1b, 0xdc, 0x04, 0x3c,
    0x78, 0x80, 0x54, 0x58, 0x4f, 0xd3, 0x94, 0x5f, 0x45, 0xca, 0x54, 0x70,
    0x9f, 0x89, 0xc1, 0x9b, 0x75, 0x56, 0x6d, 0xc3,
};
static const unsigned char kat522_nonce[] = {
    0x85, 0xa4, 0x00, 0x11, 0x6e, 0xb9, 0x03, 0x0c, 0x92, 0x49, 0xce, 0xae,
    0xe8, 0xde, 0xc9, 0xc7,
};
static const unsigned char kat522_persstr[] = {
    0x09, 0x1f, 0x59, 0x0f, 0x58, 0x42, 0x30, 0x69, 0x61, 0x63, 0xb6, 0xb5,
    0x3a, 0x11, 0xf5, 0x2c, 0x2e, 0x92, 0x59, 0xaa, 0x09, 0x82, 0xd9, 0x89,
    0x7e, 0x7d, 0xc5, 0x94, 0x19, 0x9d, 0x48, 0xcb,
};
static const unsigned char kat522_addin0[] = {0};
static const unsigned char kat522_addin1[] = {0};
static const unsigned char kat522_retbits[] = {
    0xc8, 0x9b, 0xc0, 0x8d, 0x62, 0x33, 0x5d, 0x3c, 0x28, 0x46, 0x4e, 0xe2,
    0x76, 0x22, 0x85, 0xce, 0xb0, 0xee, 0xfc, 0x95, 0x77, 0x19, 0x5a, 0x8f,
    0x89, 0xd6, 0x3e, 0x05, 0x52, 0x02, 0x01, 0x82, 0x3a, 0xf8, 0x87, 0x30,
    0x5c, 0xff, 0x9b, 0x2d, 0x7f, 0xd1, 0x7f, 0x60, 0xb5, 0x11, 0x7f, 0xdc,
    0xd0, 0xed, 0x3e, 0xf1, 0x8b, 0xac, 0x7e, 0x5f, 0x15, 0x08, 0x08, 0x02,
    0xa8, 0xab, 0x13, 0x9e,
};
static const struct drbg_kat_no_reseed kat522_t = {
    11, kat522_entropyin, kat522_nonce, kat522_persstr,
    kat522_addin0, kat522_addin1, kat522_retbits
};
static const struct drbg_kat kat522 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat522_t
};

static const unsigned char kat523_entropyin[] = {
    0xac, 0x2d, 0xd6, 0x82, 0xa9, 0x6e, 0xe0, 0xe8, 0x18, 0x82, 0x9d, 0xb5,
    0x62, 0xfb, 0xc3, 0x0a, 0x7c, 0x00, 0xe0, 0x19, 0x31, 0x47, 0xc5, 0x33,
    0xe2, 0xa5, 0x1b, 0x37, 0x4a, 0x46, 0x79, 0x58,
};
static const unsigned char kat523_nonce[] = {
    0xb8, 0xe6, 0x39, 0x64, 0x49, 0xd8, 0x84, 0xe9, 0xc6, 0xee, 0x7c, 0xed,
    0x8c, 0x7a, 0xd9, 0x2e,
};
static const unsigned char kat523_persstr[] = {
    0x10, 0x87, 0xed, 0x01, 0xb4, 0xa8, 0x53, 0xef, 0xd8, 0xb9, 0xd3, 0xad,
    0xc5, 0x89, 0x39, 0x36, 0x49, 0x9d, 0xd2, 0x97, 0x45, 0xcd, 0xc6, 0xd3,
    0xd8, 0x93, 0xad, 0xc8, 0x1d, 0x5a, 0x1b, 0xcd,
};
static const unsigned char kat523_addin0[] = {0};
static const unsigned char kat523_addin1[] = {0};
static const unsigned char kat523_retbits[] = {
    0xe1, 0x15, 0x4b, 0x37, 0x6b, 0x8c, 0x7c, 0x26, 0x15, 0x1b, 0x7f, 0x68,
    0xa5, 0xdb, 0x27, 0x4d, 0x73, 0x10, 0x18, 0x03, 0xc1, 0xe7, 0xc2, 0xf6,
    0xd1, 0x0f, 0x37, 0x4e, 0x4b, 0x88, 0x3b, 0x10, 0x30, 0x5a, 0x5d, 0x58,
    0x76, 0x76, 0x02, 0x0d, 0x52, 0xc6, 0xc4, 0xa7, 0xe5, 0x53, 0xbb, 0x96,
    0xa9, 0x53, 0xb6, 0xe9, 0xf2, 0xe4, 0xa7, 0xdd, 0x60, 0xed, 0x4f, 0x94,
    0xd1, 0x23, 0xe4, 0xa3,
};
static const struct drbg_kat_no_reseed kat523_t = {
    12, kat523_entropyin, kat523_nonce, kat523_persstr,
    kat523_addin0, kat523_addin1, kat523_retbits
};
static const struct drbg_kat kat523 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat523_t
};

static const unsigned char kat524_entropyin[] = {
    0x40, 0x44, 0x29, 0x9a, 0x1a, 0x97, 0x12, 0x6a, 0x52, 0x45, 0x97, 0xc8,
    0xdf, 0x58, 0x0c, 0x9f, 0x54, 0xad, 0xb4, 0x14, 0x3a, 0x82, 0x82, 0x33,
    0x64, 0x96, 0xd9, 0xed, 0x41, 0x88, 0x13, 0x57,
};
static const unsigned char kat524_nonce[] = {
    0x78, 0x7d, 0xa4, 0xf0, 0x15, 0x98, 0xb6, 0xf0, 0x9e, 0x3a, 0x7b, 0x68,
    0x65, 0xf9, 0x09, 0x87,
};
static const unsigned char kat524_persstr[] = {
    0xc7, 0x78, 0x25, 0x4d, 0x48, 0x62, 0x8e, 0x48, 0xf4, 0xdc, 0xac, 0x3c,
    0x96, 0xdf, 0xb2, 0xcf, 0x84, 0x10, 0xf9, 0xcd, 0xb6, 0xc0, 0x0d, 0x62,
    0x5b, 0x41, 0x1a, 0x01, 0x47, 0xbf, 0xe1, 0x6d,
};
static const unsigned char kat524_addin0[] = {0};
static const unsigned char kat524_addin1[] = {0};
static const unsigned char kat524_retbits[] = {
    0x1b, 0xfa, 0xb9, 0x3a, 0x54, 0xb7, 0xa0, 0xbb, 0xc6, 0xa7, 0x7d, 0x63,
    0xd8, 0x13, 0x2a, 0x5d, 0x6c, 0x19, 0x1c, 0x84, 0xe0, 0x9a, 0x85, 0x91,
    0xd2, 0xb7, 0xd8, 0x8f, 0x33, 0x9c, 0xad, 0x29, 0xd4, 0x3c, 0xc8, 0xd0,
    0xbf, 0x01, 0xd6, 0xea, 0x9f, 0xfe, 0x27, 0xfa, 0x1f, 0xf7, 0x69, 0x9c,
    0x87, 0x1a, 0x84, 0x11, 0x33, 0x2a, 0xb0, 0xbc, 0xeb, 0xce, 0x01, 0x7a,
    0xde, 0xd3, 0xef, 0xb6,
};
static const struct drbg_kat_no_reseed kat524_t = {
    13, kat524_entropyin, kat524_nonce, kat524_persstr,
    kat524_addin0, kat524_addin1, kat524_retbits
};
static const struct drbg_kat kat524 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat524_t
};

static const unsigned char kat525_entropyin[] = {
    0x56, 0xe5, 0xdd, 0x70, 0x78, 0x2f, 0xca, 0x50, 0x49, 0x3b, 0x5d, 0x22,
    0x2b, 0x83, 0xb6, 0xdd, 0x7b, 0xdf, 0x10, 0x22, 0xd3, 0x04, 0xf4, 0xf2,
    0x92, 0xdb, 0x38, 0x5f, 0x3a, 0xcd, 0x8a, 0xb0,
};
static const unsigned char kat525_nonce[] = {
    0xf0, 0xec, 0x46, 0xde, 0x4a, 0x9a, 0xab, 0x3f, 0x9e, 0x7f, 0xd4, 0x7f,
    0x60, 0xd2, 0x04, 0x50,
};
static const unsigned char kat525_persstr[] = {
    0xcc, 0x54, 0x2e, 0x50, 0xfc, 0x0a, 0xbb, 0x15, 0xb7, 0xdd, 0xe7, 0xb0,
    0xdf, 0xe8, 0x41, 0xc7, 0x9f, 0x74, 0x82, 0x44, 0xa0, 0xd1, 0xd0, 0xa0,
    0x2e, 0x9f, 0x90, 0x8a, 0xb9, 0x9c, 0x61, 0xc2,
};
static const unsigned char kat525_addin0[] = {0};
static const unsigned char kat525_addin1[] = {0};
static const unsigned char kat525_retbits[] = {
    0x74, 0x27, 0xc7, 0x01, 0x8c, 0x51, 0xf3, 0xb0, 0x60, 0xcb, 0xd8, 0xfe,
    0x25, 0x3a, 0x8b, 0x62, 0xb6, 0x72, 0xa3, 0x1f, 0xb8, 0x48, 0xf5, 0x58,
    0x31, 0xbe, 0x1b, 0x04, 0x5f, 0x60, 0x85, 0xc9, 0x13, 0x64, 0x85, 0x3c,
    0x7e, 0x35, 0x3b, 0xe0, 0xa5, 0x2d, 0xca, 0xdb, 0x30, 0x92, 0x8e, 0x94,
    0xe1, 0x77, 0x2b, 0xed, 0xbf, 0x27, 0x10, 0x1a, 0x2a, 0x8f, 0xf8, 0x71,
    0x3d, 0x9c, 0x9f, 0x4c,
};
static const struct drbg_kat_no_reseed kat525_t = {
    14, kat525_entropyin, kat525_nonce, kat525_persstr,
    kat525_addin0, kat525_addin1, kat525_retbits
};
static const struct drbg_kat kat525 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat525_t
};

static const unsigned char kat526_entropyin[] = {
    0x87, 0xb5, 0x6e, 0x96, 0x4e, 0xba, 0x22, 0x71, 0x54, 0x72, 0x4b, 0xb9,
    0x48, 0x4b, 0x81, 0x2d, 0x3e, 0x2c, 0x0c, 0x43, 0xb3, 0xd1, 0x7f, 0x60,
    0x98, 0xd9, 0x52, 0x6e, 0x16, 0xe6, 0xd0, 0xef,
};
static const unsigned char kat526_nonce[] = {
    0x9b, 0xea, 0x6a, 0x7f, 0xf2, 0x35, 0x8d, 0xf1, 0x42, 0xe6, 0xc2, 0x3e,
    0x21, 0x57, 0xfb, 0x83,
};
static const unsigned char kat526_persstr[] = {
    0x98, 0x60, 0xb4, 0x32, 0xed, 0xd5, 0x8d, 0x1c, 0xcb, 0xfe, 0xec, 0xbc,
    0xe9, 0x9f, 0xfa, 0xee, 0x7d, 0x93, 0x5a, 0x61, 0x48, 0x60, 0xd4, 0xe9,
    0x65, 0xbd, 0x67, 0x04, 0x14, 0x03, 0x09, 0x6b,
};
static const unsigned char kat526_addin0[] = {
    0x99, 0xa5, 0xcc, 0x87, 0x92, 0x4e, 0x8e, 0xa6, 0x5a, 0x59, 0x6f, 0x81,
    0xfd, 0x17, 0xd6, 0x3f, 0x5b, 0x45, 0x42, 0xfe, 0x6e, 0x8e, 0x15, 0x11,
    0xb5, 0xd3, 0x5c, 0x83, 0x5d, 0xfa, 0xdb, 0x0b,
};
static const unsigned char kat526_addin1[] = {
    0x9a, 0x8d, 0xec, 0x54, 0x73, 0x4a, 0x34, 0x58, 0x2a, 0x23, 0x32, 0xf3,
    0x45, 0x2e, 0x82, 0x31, 0x35, 0x24, 0xc3, 0xe0, 0xdf, 0xb4, 0x85, 0xfa,
    0xea, 0xc6, 0xca, 0x5f, 0xc0, 0xff, 0x50, 0x4d,
};
static const unsigned char kat526_retbits[] = {
    0xdb, 0xc6, 0xa2, 0x33, 0x0b, 0x19, 0xb5, 0xcd, 0xdd, 0x8c, 0xd6, 0x39,
    0x2e, 0xc1, 0xfb, 0x50, 0x86, 0x78, 0xc8, 0x05, 0xe8, 0x7d, 0x1a, 0xca,
    0x07, 0xac, 0x26, 0x50, 0x07, 0x63, 0x25, 0x03, 0x04, 0x4a, 0x00, 0x61,
    0x0c, 0x79, 0xd9, 0x83, 0x75, 0xaf, 0xa7, 0xab, 0x4c, 0xca, 0x1a, 0x90,
    0x98, 0x9c, 0xbf, 0xe7, 0xc6, 0x74, 0xaf, 0x5d, 0x82, 0x3c, 0xed, 0x11,
    0xc4, 0x7e, 0x9a, 0xf6,
};
static const struct drbg_kat_no_reseed kat526_t = {
    0, kat526_entropyin, kat526_nonce, kat526_persstr,
    kat526_addin0, kat526_addin1, kat526_retbits
};
static const struct drbg_kat kat526 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat526_t
};

static const unsigned char kat527_entropyin[] = {
    0xb3, 0x60, 0x32, 0xf5, 0xd7, 0x77, 0x25, 0x08, 0x26, 0xd8, 0x31, 0x56,
    0x6e, 0xc5, 0x85, 0x45, 0x2d, 0x70, 0xb9, 0x20, 0x65, 0x43, 0x55, 0xac,
    0xf8, 0xf6, 0x91, 0x94, 0x16, 0x43, 0xee, 0x95,
};
static const unsigned char kat527_nonce[] = {
    0xda, 0xcf, 0x74, 0x7e, 0x85, 0xfa, 0xa6, 0xa3, 0xeb, 0x01, 0x6d, 0xf9,
    0x29, 0xc9, 0x0e, 0x8b,
};
static const unsigned char kat527_persstr[] = {
    0xf0, 0x32, 0x65, 0xb2, 0xf2, 0x17, 0x4c, 0xea, 0x93, 0x8f, 0xf2, 0x3c,
    0x7e, 0x60, 0xa7, 0x5d, 0xcb, 0xa1, 0xe4, 0xe4, 0x12, 0xbb, 0xad, 0x4b,
    0x5d, 0x3b, 0x3e, 0x23, 0x68, 0x5e, 0x80, 0xd8,
};
static const unsigned char kat527_addin0[] = {
    0xd4, 0x77, 0x23, 0x80, 0xde, 0x77, 0x4b, 0xbb, 0xb6, 0x10, 0x0d, 0x93,
    0x39, 0x59, 0x0e, 0xff, 0x03, 0x3f, 0xf5, 0x48, 0xb8, 0x26, 0x68, 0x55,
    0x53, 0xa2, 0xe8, 0x57, 0x80, 0x0a, 0x07, 0xe2,
};
static const unsigned char kat527_addin1[] = {
    0x05, 0x01, 0x1d, 0x3d, 0xd4, 0xdd, 0xcf, 0x19, 0x07, 0x6f, 0xae, 0x65,
    0x69, 0x73, 0xaa, 0xc9, 0xa1, 0x16, 0x41, 0xb2, 0x10, 0x96, 0x3c, 0xec,
    0x81, 0xd1, 0xea, 0x58, 0xdb, 0x7b, 0xb7, 0xe0,
};
static const unsigned char kat527_retbits[] = {
    0x3d, 0x35, 0x31, 0x05, 0x79, 0x77, 0x40, 0x10, 0x72, 0xce, 0x44, 0xe2,
    0xe6, 0x63, 0x17, 0xa8, 0x08, 0xd4, 0x7c, 0x44, 0xaa, 0xd4, 0xf9, 0x8c,
    0x08, 0xd8, 0x8e, 0xac, 0x7b, 0x59, 0x8c, 0x40, 0x71, 0x4a, 0xd1, 0x24,
    0x17, 0xb6, 0x16, 0x99, 0xd1, 0x12, 0x6e, 0xa4, 0xc6, 0x42, 0xb0, 0x9f,
    0xe9, 0xf5, 0xde, 0xd3, 0x6f, 0x2e, 0x37, 0xed, 0x2c, 0xce, 0x97, 0x2e,
    0x0d, 0xfc, 0xc7, 0xce,
};
static const struct drbg_kat_no_reseed kat527_t = {
    1, kat527_entropyin, kat527_nonce, kat527_persstr,
    kat527_addin0, kat527_addin1, kat527_retbits
};
static const struct drbg_kat kat527 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat527_t
};

static const unsigned char kat528_entropyin[] = {
    0x32, 0x42, 0xa1, 0xb9, 0x7c, 0x11, 0xe5, 0xea, 0x8a, 0x1b, 0x96, 0xcc,
    0xdc, 0xc2, 0x56, 0x28, 0xe7, 0x9e, 0xc5, 0xd1, 0x4b, 0x04, 0x15, 0x58,
    0xd3, 0x12, 0xea, 0x72, 0xa3, 0xdd, 0x0d, 0xd0,
};
static const unsigned char kat528_nonce[] = {
    0xd9, 0xf1, 0xa8, 0xdd, 0x83, 0xb0, 0xf1, 0x13, 0x17, 0xa9, 0x2d, 0x20,
    0x51, 0xe0, 0x7e, 0x97,
};
static const unsigned char kat528_persstr[] = {
    0x93, 0x0b, 0x0c, 0x3b, 0x96, 0x26, 0x3c, 0x3e, 0xc7, 0xed, 0xf2, 0x58,
    0x90, 0xcc, 0x5d, 0x7e, 0xa4, 0x1d, 0x65, 0x64, 0xa8, 0x1f, 0x17, 0x89,
    0x9f, 0xc7, 0xb4, 0x9c, 0x5c, 0x40, 0xb0, 0x91,
};
static const unsigned char kat528_addin0[] = {
    0xa9, 0x05, 0xda, 0xb4, 0xad, 0xe7, 0x5e, 0xe5, 0xe6, 0x8a, 0x07, 0x09,
    0xd1, 0xd0, 0x69, 0x9a, 0x87, 0xf5, 0xd4, 0xf8, 0xe4, 0x9c, 0x8c, 0x95,
    0xd9, 0x59, 0x0d, 0x7c, 0x12, 0x38, 0xd0, 0xf9,
};
static const unsigned char kat528_addin1[] = {
    0x4d, 0x8e, 0x2d, 0x92, 0x29, 0x97, 0x65, 0xb9, 0xce, 0xf0, 0x39, 0xd9,
    0x47, 0xd4, 0x42, 0x7d, 0x4b, 0x9c, 0x14, 0xf4, 0x2c, 0x81, 0xe1, 0x2b,
    0xc9, 0x1b, 0x4e, 0x29, 0x72, 0x33, 0xfc, 0xe1,
};
static const unsigned char kat528_retbits[] = {
    0xfa, 0x3a, 0xdb, 0x6a, 0x35, 0xe0, 0x50, 0x85, 0x34, 0x80, 0x3e, 0xda,
    0x5f, 0x39, 0xd9, 0xfd, 0x62, 0x22, 0x74, 0x74, 0xd9, 0xe1, 0x43, 0x5e,
    0xd0, 0xe5, 0xe6, 0xda, 0x01, 0xe2, 0xe4, 0xd8, 0x3a, 0x3d, 0xa9, 0x39,
    0x5a, 0x3c, 0x96, 0xef, 0xec, 0xdf, 0xe9, 0x2b, 0x89, 0xe1, 0x5e, 0x8c,
    0xaf, 0x25, 0xe8, 0x17, 0x24, 0xb3, 0x97, 0x35, 0x51, 0x97, 0x2a, 0x8f,
    0xf5, 0xb5, 0xcd, 0x0f,
};
static const struct drbg_kat_no_reseed kat528_t = {
    2, kat528_entropyin, kat528_nonce, kat528_persstr,
    kat528_addin0, kat528_addin1, kat528_retbits
};
static const struct drbg_kat kat528 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat528_t
};

static const unsigned char kat529_entropyin[] = {
    0xbb, 0x2c, 0x5c, 0x0d, 0xe9, 0xdd, 0x27, 0x69, 0x42, 0xca, 0x0a, 0x54,
    0xfa, 0xf9, 0xa4, 0xf7, 0x25, 0xe2, 0x7b, 0x19, 0x6a, 0x6f, 0x16, 0xe9,
    0xdd, 0x34, 0x59, 0x53, 0x79, 0xe5, 0xb8, 0x69,
};
static const unsigned char kat529_nonce[] = {
    0x76, 0x30, 0xfb, 0xb7, 0x1d, 0x49, 0xba, 0x0b, 0xa3, 0x6d, 0xd8, 0x8d,
    0x2d, 0x49, 0x1d, 0x61,
};
static const unsigned char kat529_persstr[] = {
    0xa0, 0x26, 0x4b, 0x50, 0x05, 0xdb, 0xfc, 0x57, 0x19, 0x4b, 0xac, 0x89,
    0x08, 0x9c, 0x16, 0xa7, 0x85, 0x22, 0x2a, 0xf0, 0x08, 0x31, 0x0b, 0x20,
    0xa6, 0x08, 0x5e, 0xe7, 0x41, 0x89, 0x2c, 0xed,
};
static const unsigned char kat529_addin0[] = {
    0x63, 0x4f, 0xd7, 0xca, 0xd0, 0x00, 0xb7, 0x86, 0xfe, 0x36, 0x2c, 0x91,
    0x48, 0x57, 0xb5, 0xd9, 0xed, 0x26, 0x69, 0xcd, 0x5c, 0x77, 0x7c, 0x52,
    0x07, 0x5c, 0xff, 0x38, 0x40, 0xbb, 0x58, 0xab,
};
static const unsigned char kat529_addin1[] = {
    0xfd, 0xfe, 0x16, 0x1f, 0x0c, 0xa0, 0xe6, 0xec, 0xd6, 0x35, 0x6a, 0xdb,
    0x64, 0x6e, 0xf6, 0x29, 0x35, 0xd8, 0xcf, 0xc7, 0x52, 0x22, 0x44, 0xa6,
    0x04, 0x1c, 0xb3, 0xd7, 0xcc, 0x6e, 0x68, 0x39,
};
static const unsigned char kat529_retbits[] = {
    0xe3, 0xc7, 0x8a, 0x24, 0xd5, 0xa3, 0x4b, 0x2a, 0x70, 0x26, 0x4d, 0x13,
    0x70, 0x7c, 0xa6, 0x35, 0xc9, 0x54, 0x26, 0xf4, 0x22, 0xdb, 0x78, 0xc1,
    0x8c, 0x91, 0xe3, 0xc4, 0x0f, 0xb7, 0x39, 0x2c, 0x02, 0xb0, 0xa9, 0xed,
    0x2e, 0x1d, 0x5a, 0x9c, 0x73, 0xef, 0x45, 0x6c, 0x37, 0xaf, 0xe6, 0xae,
    0x0c, 0xc1, 0x86, 0x72, 0x63, 0xb4, 0x63, 0x0a, 0xbd, 0x73, 0x84, 0xab,
    0x68, 0x29, 0x03, 0x42,
};
static const struct drbg_kat_no_reseed kat529_t = {
    3, kat529_entropyin, kat529_nonce, kat529_persstr,
    kat529_addin0, kat529_addin1, kat529_retbits
};
static const struct drbg_kat kat529 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat529_t
};

static const unsigned char kat530_entropyin[] = {
    0x73, 0x74, 0xb7, 0x23, 0xd4, 0x1a, 0xc8, 0x7c, 0xd1, 0xee, 0x53, 0x88,
    0x0c, 0x0d, 0x76, 0xbd, 0x95, 0x8b, 0x50, 0xa8, 0x71, 0x4f, 0xfd, 0x07,
    0x0f, 0x45, 0x3b, 0xd8, 0x49, 0x86, 0x22, 0xc8,
};
static const unsigned char kat530_nonce[] = {
    0x86, 0xe1, 0x23, 0xcd, 0x4f, 0x07, 0x4c, 0x7f, 0x07, 0x31, 0x63, 0xfb,
    0x18, 0x9a, 0xf9, 0xfd,
};
static const unsigned char kat530_persstr[] = {
    0x16, 0xb6, 0x26, 0x75, 0xe9, 0xf9, 0x24, 0xc6, 0xfc, 0xaa, 0x13, 0x3c,
    0x0c, 0x75, 0xc4, 0x1b, 0x5e, 0xde, 0x75, 0xbd, 0xfd, 0x4a, 0x70, 0x11,
    0x0e, 0xa7, 0x4a, 0xfb, 0x73, 0xd2, 0x56, 0x97,
};
static const unsigned char kat530_addin0[] = {
    0xfc, 0x06, 0x04, 0x40, 0x87, 0x2b, 0xa7, 0xee, 0x4b, 0x26, 0xb0, 0xca,
    0xb6, 0x70, 0x7c, 0xd3, 0x88, 0xc6, 0x9d, 0x85, 0xac, 0xb5, 0x7d, 0x65,
    0x79, 0xc8, 0xd8, 0xbc, 0x4a, 0xf1, 0xad, 0xfe,
};
static const unsigned char kat530_addin1[] = {
    0xcd, 0xd1, 0x50, 0xf8, 0xd7, 0x0d, 0x5a, 0xad, 0x87, 0x84, 0xea, 0x74,
    0xc5, 0x09, 0x86, 0x05, 0xf8, 0x3c, 0xb0, 0x7f, 0x69, 0xdb, 0xad, 0xe1,
    0x93, 0x89, 0x3b, 0xc0, 0x18, 0x4d, 0xab, 0x39,
};
static const unsigned char kat530_retbits[] = {
    0xcb, 0x7e, 0x38, 0xfc, 0xfa, 0xd9, 0x1f, 0x1e, 0xb8, 0x2d, 0xac, 0xd2,
    0xce, 0x99, 0x11, 0xb7, 0x9b, 0xfd, 0xb0, 0x7e, 0xdb, 0xab, 0x2e, 0xdf,
    0x24, 0x19, 0x0a, 0x2d, 0x9d, 0xa1, 0x71, 0xc1, 0x8c, 0x62, 0xbd, 0x10,
    0xe3, 0xb7, 0xd7, 0xff, 0x01, 0x9b, 0x66, 0x6e, 0xbf, 0xf4, 0x26, 0x77,
    0xd7, 0x9b, 0x40, 0x1a, 0xe6, 0x4a, 0x41, 0x37, 0x28, 0x76, 0x36, 0xd4,
    0x9c, 0xad, 0xdc, 0xdc,
};
static const struct drbg_kat_no_reseed kat530_t = {
    4, kat530_entropyin, kat530_nonce, kat530_persstr,
    kat530_addin0, kat530_addin1, kat530_retbits
};
static const struct drbg_kat kat530 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat530_t
};

static const unsigned char kat531_entropyin[] = {
    0x0d, 0xb7, 0x5d, 0x22, 0xf0, 0xd1, 0x69, 0x02, 0xc7, 0xa1, 0x81, 0x3b,
    0x0a, 0xda, 0xb2, 0xeb, 0xc6, 0x75, 0xf4, 0x30, 0x57, 0x38, 0x46, 0x9b,
    0x14, 0x2b, 0x7f, 0x9a, 0x39, 0xd5, 0xea, 0xf7,
};
static const unsigned char kat531_nonce[] = {
    0x1a, 0xce, 0x39, 0x3a, 0x30, 0xcb, 0x36, 0xda, 0xbf, 0xcd, 0x85, 0x48,
    0x34, 0x91, 0xb9, 0x48,
};
static const unsigned char kat531_persstr[] = {
    0x1f, 0x9f, 0x7a, 0xbf, 0xca, 0x5d, 0x53, 0x65, 0xc6, 0x44, 0xf7, 0xd9,
    0xd3, 0x3d, 0x12, 0x88, 0xcb, 0x62, 0xb2, 0x7e, 0x87, 0x3a, 0x4d, 0x52,
    0xaa, 0xd0, 0xe7, 0xcc, 0x1c, 0x89, 0x53, 0x88,
};
static const unsigned char kat531_addin0[] = {
    0xb0, 0x21, 0x01, 0x79, 0xbb, 0x14, 0x59, 0x8a, 0xd3, 0x0c, 0x4f, 0xe1,
    0xf2, 0xbf, 0x43, 0xc7, 0x81, 0xea, 0x9d, 0x8b, 0x51, 0xa2, 0x6b, 0x88,
    0x55, 0xa8, 0x12, 0xb5, 0x5a, 0x1c, 0xf8, 0x94,
};
static const unsigned char kat531_addin1[] = {
    0x73, 0xa2, 0x8b, 0x72, 0xcd, 0xa9, 0xf8, 0x31, 0xe2, 0x1b, 0x77, 0x6b,
    0xc9, 0x2e, 0x34, 0x04, 0x13, 0x2a, 0xbe, 0x5d, 0x7c, 0x2f, 0xe1, 0xc2,
    0x6a, 0xd4, 0xca, 0x6c, 0xa6, 0xe4, 0xea, 0x47,
};
static const unsigned char kat531_retbits[] = {
    0xf7, 0xb6, 0x39, 0x82, 0x60, 0xd3, 0x0e, 0x5b, 0x24, 0x0d, 0xfd, 0x5b,
    0xa0, 0x13, 0x6b, 0xa2, 0xda, 0x98, 0xb8, 0xbf, 0x25, 0x3f, 0x7a, 0x7c,
    0x1b, 0xad, 0x91, 0x32, 0x62, 0x54, 0x04, 0xe8, 0x99, 0x88, 0xd2, 0x71,
    0xba, 0xf0, 0xa2, 0x6e, 0x19, 0x78, 0x61, 0x06, 0xba, 0xc9, 0x20, 0xb9,
    0x43, 0x3c, 0xc8, 0xb4, 0xf5, 0xb9, 0x85, 0x3e, 0x24, 0x7c, 0x05, 0x06,
    0xd1, 0xb1, 0x14, 0x88,
};
static const struct drbg_kat_no_reseed kat531_t = {
    5, kat531_entropyin, kat531_nonce, kat531_persstr,
    kat531_addin0, kat531_addin1, kat531_retbits
};
static const struct drbg_kat kat531 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat531_t
};

static const unsigned char kat532_entropyin[] = {
    0x6b, 0xd0, 0x81, 0xc5, 0x86, 0x37, 0xe5, 0x55, 0x10, 0x5e, 0x14, 0x1d,
    0xca, 0xca, 0x4b, 0x95, 0xb5, 0xbe, 0xcf, 0x39, 0x96, 0x73, 0xdb, 0x6c,
    0xc0, 0xe9, 0x6c, 0x66, 0xe1, 0x60, 0x92, 0xcd,
};
static const unsigned char kat532_nonce[] = {
    0xce, 0xa7, 0x1d, 0xe8, 0x81, 0x9b, 0x59, 0x99, 0x37, 0xdb, 0xa4, 0x84,
    0x77, 0xae, 0x9f, 0xba,
};
static const unsigned char kat532_persstr[] = {
    0xeb, 0x04, 0x5c, 0x29, 0x85, 0xab, 0x91, 0x69, 0x07, 0x58, 0xc2, 0xb5,
    0x6c, 0x7d, 0xba, 0xd2, 0x56, 0xdb, 0x16, 0xef, 0xc4, 0x14, 0x34, 0x4a,
    0x4b, 0xba, 0xe2, 0x54, 0xbe, 0x41, 0x70, 0xfa,
};
static const unsigned char kat532_addin0[] = {
    0xf1, 0x32, 0xbb, 0x2e, 0x73, 0x1c, 0xd1, 0xcc, 0xfa, 0x37, 0x3b, 0x4d,
    0x1e, 0x8a, 0x52, 0x7e, 0xd0, 0x8b, 0x42, 0x94, 0x47, 0xf3, 0x2b, 0x23,
    0xb9, 0x6d, 0x31, 0x2e, 0x5b, 0xab, 0xdd, 0x23,
};
static const unsigned char kat532_addin1[] = {
    0x0a, 0x55, 0x4d, 0xed, 0x57, 0xc1, 0xef, 0xec, 0xc4, 0xd4, 0xa0, 0x1a,
    0x82, 0xff, 0x17, 0x99, 0xc6, 0xe9, 0x36, 0xf9, 0xf9, 0x34, 0x02, 0xbc,
    0xe4, 0x62, 0x10, 0x2c, 0x2d, 0x25, 0x91, 0x0a,
};
static const unsigned char kat532_retbits[] = {
    0x7d, 0xe6, 0x0b, 0x0a, 0x40, 0xfb, 0x83, 0x59, 0xda, 0x92, 0x92, 0xc6,
    0x8f, 0x26, 0x16, 0x31, 0xd1, 0x84, 0xb0, 0x3e, 0x90, 0x47, 0x33, 0xb5,
    0xc9, 0x63, 0x96, 0x6a, 0x74, 0xd4, 0x87, 0x5a, 0x8f, 0x36, 0x50, 0x87,
    0xde, 0xe9, 0xc2, 0xcb, 0x8d, 0x05, 0x77, 0x8b, 0x0e, 0x0b, 0x46, 0xe5,
    0x26, 0xbc, 0xa1, 0x3c, 0x4c, 0x50, 0xbe, 0x49, 0x06, 0x25, 0xef, 0x05,
    0x0a, 0x79, 0x45, 0x28,
};
static const struct drbg_kat_no_reseed kat532_t = {
    6, kat532_entropyin, kat532_nonce, kat532_persstr,
    kat532_addin0, kat532_addin1, kat532_retbits
};
static const struct drbg_kat kat532 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat532_t
};

static const unsigned char kat533_entropyin[] = {
    0xb7, 0x1b, 0xd1, 0x4e, 0x15, 0x87, 0xf6, 0x33, 0x4b, 0x7f, 0x94, 0x64,
    0x26, 0x16, 0xa3, 0x6a, 0x55, 0xff, 0xe4, 0xa7, 0x5e, 0xea, 0xde, 0x86,
    0x4e, 0xa7, 0x05, 0x1e, 0x64, 0x6b, 0xad, 0x52,
};
static const unsigned char kat533_nonce[] = {
    0xa4, 0x3d, 0xc9, 0x7f, 0x41, 0x1b, 0xe9, 0x7c, 0xd1, 0x8d, 0xbe, 0xcf,
    0xce, 0x56, 0x2f, 0x23,
};
static const unsigned char kat533_persstr[] = {
    0xeb, 0xbd, 0xb8, 0x1e, 0xc0, 0x3a, 0x13, 0xa1, 0x5a, 0x03, 0xdb, 0x56,
    0x98, 0xf3, 0x4a, 0xa8, 0xc0, 0x29, 0x38, 0xbf, 0x87, 0x85, 0xf1, 0x3f,
    0xde, 0x75, 0xcb, 0xcd, 0x82, 0x3e, 0x61, 0xf7,
};
static const unsigned char kat533_addin0[] = {
    0x58, 0x17, 0xe3, 0x93, 0x93, 0xe8, 0x8f, 0x09, 0x39, 0xd5, 0xa8, 0xd7,
    0x03, 0x83, 0x08, 0x3f, 0xc6, 0x6d, 0x9c, 0x53, 0x3b, 0xae, 0xe6, 0x28,
    0x28, 0x2d, 0xf7, 0x80, 0x46, 0xfc, 0xd8, 0xfa,
};
static const unsigned char kat533_addin1[] = {
    0xf9, 0x67, 0xa6, 0xd7, 0x9c, 0x65, 0x7d, 0x0b, 0xff, 0xd3, 0x34, 0x6f,
    0xb2, 0x80, 0x46, 0x91, 0x1b, 0x1f, 0x8c, 0x30, 0x31, 0xcc, 0x6e, 0x22,
    0x04, 0xc2, 0xc9, 0xe8, 0x88, 0x43, 0x2f, 0x77,
};
static const unsigned char kat533_retbits[] = {
    0xe1, 0xba, 0x47, 0xd4, 0x5d, 0x73, 0xba, 0x54, 0xda, 0x50, 0x30, 0x15,
    0x98, 0x7d, 0x8e, 0x12, 0x14, 0xc3, 0x46, 0x87, 0xa6, 0xff, 0x78, 0x6a,
    0x88, 0x0c, 0x98, 0xe5, 0xe2, 0xe5, 0x59, 0x0e, 0xae, 0x69, 0x09, 0x5a,
    0x93, 0x21, 0x6d, 0x12, 0x73, 0x50, 0x4d, 0x94, 0xde, 0xa6, 0x7c, 0x51,
    0xf4, 0xa8, 0xa5, 0xd9, 0x7d, 0x02, 0x42, 0x89, 0x0c, 0x39, 0x4b, 0xf7,
    0x3e, 0xf5, 0x2d, 0x63,
};
static const struct drbg_kat_no_reseed kat533_t = {
    7, kat533_entropyin, kat533_nonce, kat533_persstr,
    kat533_addin0, kat533_addin1, kat533_retbits
};
static const struct drbg_kat kat533 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat533_t
};

static const unsigned char kat534_entropyin[] = {
    0x0d, 0x97, 0x0e, 0xa0, 0xa1, 0xe8, 0x9d, 0x36, 0xfb, 0x66, 0xa7, 0x3a,
    0xe7, 0xa7, 0xbb, 0x47, 0x60, 0xca, 0x9b, 0x0e, 0x22, 0x96, 0xe2, 0xd2,
    0x41, 0x62, 0x9b, 0x99, 0x24, 0x9a, 0xc9, 0xf3,
};
static const unsigned char kat534_nonce[] = {
    0xc0, 0x30, 0x31, 0xdc, 0x1c, 0xd9, 0xcf, 0x62, 0xc4, 0xd0, 0x3e, 0x2e,
    0x37, 0x20, 0x39, 0x1b,
};
static const unsigned char kat534_persstr[] = {
    0x91, 0x68, 0x2b, 0xe8, 0x19, 0x34, 0xaf, 0xd8, 0x1d, 0xcb, 0x63, 0x8b,
    0xa8, 0x88, 0x3a, 0x88, 0x91, 0x3e, 0x32, 0x65, 0xda, 0x1b, 0x92, 0xc8,
    0xd9, 0xb3, 0x4a, 0x2b, 0x90, 0x66, 0x24, 0xcd,
};
static const unsigned char kat534_addin0[] = {
    0xc3, 0xb3, 0x3d, 0xc2, 0xf9, 0x8d, 0xc8, 0xde, 0xdc, 0x22, 0x54, 0xb3,
    0x82, 0x65, 0x4c, 0x75, 0x38, 0x1a, 0x4d, 0xf0, 0x71, 0x33, 0x4c, 0xe2,
    0x28, 0x60, 0xe9, 0x2c, 0x61, 0x64, 0xf8, 0xcc,
};
static const unsigned char kat534_addin1[] = {
    0x53, 0xb2, 0xae, 0x27, 0xf9, 0x64, 0xfc, 0x86, 0x2b, 0x39, 0xf4, 0xa1,
    0x6d, 0x73, 0xa9, 0x98, 0x0e, 0x81, 0x94, 0x6c, 0xce, 0x5c, 0x2a, 0xe4,
    0xa5, 0xef, 0x6a, 0x25, 0xb9, 0x2a, 0x08, 0xaa,
};
static const unsigned char kat534_retbits[] = {
    0xe3, 0x47, 0x1a, 0xbe, 0xe1, 0x16, 0x7a, 0x57, 0x66, 0x71, 0x32, 0xdd,
    0x31, 0x59, 0x0e, 0x95, 0x12, 0x5f, 0x61, 0x95, 0x7e, 0xcd, 0x62, 0x1d,
    0xdb, 0xa4, 0x9e, 0xbf, 0xa5, 0x3a, 0x8f, 0x5a, 0xfe, 0xaa, 0x30, 0xac,
    0xa1, 0xad, 0xf1, 0x1c, 0xa2, 0x0d, 0xe2, 0xcb, 0x08, 0xf0, 0xdd, 0x43,
    0x29, 0xd9, 0x50, 0xd0, 0x53, 0x92, 0x09, 0x26, 0x93, 0x1f, 0x67, 0x28,
    0x27, 0x93, 0x1f, 0x8c,
};
static const struct drbg_kat_no_reseed kat534_t = {
    8, kat534_entropyin, kat534_nonce, kat534_persstr,
    kat534_addin0, kat534_addin1, kat534_retbits
};
static const struct drbg_kat kat534 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat534_t
};

static const unsigned char kat535_entropyin[] = {
    0xbe, 0xd9, 0x60, 0xee, 0x26, 0xeb, 0x33, 0xfd, 0xf4, 0xa6, 0x2c, 0xdf,
    0xfd, 0xe8, 0x4e, 0x6d, 0x18, 0x32, 0xc6, 0x91, 0x48, 0xa2, 0xae, 0x5c,
    0xbc, 0x9b, 0xa5, 0x03, 0x63, 0x13, 0xc2, 0xd2,
};
static const unsigned char kat535_nonce[] = {
    0x53, 0xe0, 0x92, 0xe7, 0x08, 0xe8, 0xb8, 0xa4, 0xa0, 0x36, 0xed, 0x6d,
    0x0c, 0xd1, 0x28, 0x70,
};
static const unsigned char kat535_persstr[] = {
    0x08, 0x03, 0xf2, 0x8c, 0x79, 0xd5, 0x3f, 0x05, 0x79, 0x53, 0xdb, 0x4a,
    0x1a, 0xb5, 0xa3, 0xca, 0xb5, 0x81, 0xf2, 0xc1, 0x6d, 0x1e, 0x08, 0x3f,
    0x29, 0x48, 0xac, 0xd0, 0xfb, 0xf9, 0xbe, 0xec,
};
static const unsigned char kat535_addin0[] = {
    0x42, 0x9a, 0x33, 0x10, 0xc1, 0xae, 0x67, 0x31, 0xd4, 0x68, 0xef, 0x6f,
    0x2b, 0x1c, 0x99, 0x49, 0xbb, 0xfc, 0x8d, 0x36, 0xeb, 0x66, 0xfb, 0xee,
    0x0c, 0x86, 0x2e, 0x42, 0x58, 0x6a, 0xac, 0x3f,
};
static const unsigned char kat535_addin1[] = {
    0xd7, 0xa5, 0xd5, 0x80, 0x04, 0x8d, 0xba, 0x0c, 0xe6, 0x76, 0xb5, 0x04,
    0x7a, 0x98, 0x9e, 0xcc, 0xf3, 0xe4, 0x05, 0x2a, 0x60, 0x28, 0xef, 0xff,
    0x29, 0xe7, 0xcb, 0x0a, 0x13, 0xab, 0x64, 0xce,
};
static const unsigned char kat535_retbits[] = {
    0x27, 0x03, 0x84, 0x5c, 0xa0, 0x40, 0xff, 0x03, 0x49, 0xb6, 0x6c, 0x1b,
    0xf8, 0x22, 0x6f, 0xf8, 0x9f, 0x27, 0x87, 0x57, 0xba, 0xad, 0xb7, 0x4d,
    0x27, 0x74, 0xfd, 0xb5, 0x4d, 0xf0, 0x17, 0x74, 0xf0, 0x4b, 0xbe, 0x9a,
    0xcd, 0x6b, 0xc8, 0xe8, 0xa1, 0xf2, 0xd9, 0x7a, 0xeb, 0x89, 0x13, 0x03,
    0xa5, 0x33, 0x7b, 0xcd, 0x69, 0xce, 0x4a, 0xfd, 0x4c, 0x6b, 0x62, 0x42,
    0x64, 0x49, 0xdd, 0x63,
};
static const struct drbg_kat_no_reseed kat535_t = {
    9, kat535_entropyin, kat535_nonce, kat535_persstr,
    kat535_addin0, kat535_addin1, kat535_retbits
};
static const struct drbg_kat kat535 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat535_t
};

static const unsigned char kat536_entropyin[] = {
    0x8e, 0x7b, 0x62, 0x75, 0xc4, 0x8f, 0xa6, 0x0a, 0xae, 0x68, 0xbc, 0xd6,
    0xae, 0xde, 0xd1, 0x7f, 0xce, 0xcc, 0xe7, 0x12, 0xba, 0x47, 0xa5, 0xda,
    0x79, 0xdc, 0x88, 0x83, 0xa2, 0xb2, 0xb4, 0x43,
};
static const unsigned char kat536_nonce[] = {
    0xc1, 0xaa, 0x1c, 0x23, 0x7e, 0x69, 0x85, 0x3d, 0xcf, 0x44, 0xc7, 0x55,
    0x8d, 0xca, 0x07, 0xd8,
};
static const unsigned char kat536_persstr[] = {
    0xd5, 0x73, 0x7e, 0x7b, 0x1f, 0x4e, 0x53, 0xdd, 0x92, 0x41, 0x51, 0xfd,
    0x71, 0xc5, 0x1a, 0x77, 0xe6, 0x55, 0xf5, 0x17, 0x3c, 0xcd, 0xf0, 0xe9,
    0x73, 0x51, 0x24, 0x1b, 0x57, 0x0e, 0xa4, 0xf0,
};
static const unsigned char kat536_addin0[] = {
    0xbb, 0x70, 0xb8, 0xdb, 0x19, 0xd3, 0xc1, 0xab, 0xfd, 0x6d, 0x25, 0x13,
    0x65, 0x3f, 0x24, 0x83, 0xbe, 0x43, 0x30, 0x06, 0x7f, 0xf6, 0x6b, 0x8f,
    0xb6, 0x2f, 0xc8, 0xe6, 0x72, 0xdf, 0x96, 0x28,
};
static const unsigned char kat536_addin1[] = {
    0x00, 0x05, 0xe3, 0x1b, 0x93, 0x34, 0x24, 0x2b, 0xeb, 0x2f, 0x3c, 0xa9,
    0x88, 0x60, 0x3f, 0x5d, 0xcb, 0xa6, 0x85, 0x57, 0x6e, 0xb7, 0x60, 0x78,
    0x1d, 0x9f, 0x47, 0x25, 0x2b, 0xa5, 0xe2, 0x9f,
};
static const unsigned char kat536_retbits[] = {
    0xe0, 0x30, 0x08, 0x5f, 0x66, 0x79, 0x37, 0xf8, 0x66, 0xe1, 0x3a, 0xd0,
    0x85, 0x4f, 0x24, 0x5b, 0x8f, 0x39, 0x2e, 0x07, 0x01, 0x5e, 0x03, 0x71,
    0xd5, 0xec, 0x72, 0xb9, 0x27, 0x1f, 0xf9, 0xf0, 0x94, 0x6a, 0x6d, 0x44,
    0x69, 0xc7, 0x70, 0x1c, 0x48, 0x5f, 0xa4, 0x9a, 0xd7, 0x9e, 0xf5, 0xdb,
    0xd0, 0xb1, 0x01, 0x0d, 0x58, 0xd6, 0xb1, 0x7d, 0xd4, 0x05, 0x4c, 0x13,
    0x08, 0x15, 0x00, 0x65,
};
static const struct drbg_kat_no_reseed kat536_t = {
    10, kat536_entropyin, kat536_nonce, kat536_persstr,
    kat536_addin0, kat536_addin1, kat536_retbits
};
static const struct drbg_kat kat536 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat536_t
};

static const unsigned char kat537_entropyin[] = {
    0x80, 0xdb, 0x28, 0xc5, 0x45, 0x7f, 0x9b, 0x55, 0x72, 0x39, 0xdb, 0x59,
    0x51, 0xce, 0x1a, 0x29, 0xa5, 0xb8, 0xe4, 0x85, 0x6c, 0x7f, 0x89, 0xd0,
    0x28, 0x58, 0x94, 0x6a, 0x27, 0x8c, 0xb7, 0x05,
};
static const unsigned char kat537_nonce[] = {
    0x94, 0x14, 0xcf, 0xc5, 0x45, 0x2a, 0xfb, 0xe4, 0xd9, 0x3a, 0xfa, 0xf1,
    0x62, 0x60, 0x4b, 0x0e,
};
static const unsigned char kat537_persstr[] = {
    0x78, 0x81, 0x6b, 0xa6, 0xd7, 0xb6, 0x3d, 0xc2, 0x01, 0xb5, 0x9c, 0x74,
    0xb1, 0x1d, 0x29, 0x18, 0xb0, 0x58, 0xe7, 0x10, 0x58, 0x28, 0xcf, 0xff,
    0x5d, 0xcc, 0x17, 0x81, 0xcf, 0x1d, 0xfb, 0xe5,
};
static const unsigned char kat537_addin0[] = {
    0x63, 0x56, 0xdb, 0x39, 0xf5, 0x9e, 0x61, 0xf8, 0x8d, 0xa6, 0x15, 0xa7,
    0x9e, 0x62, 0x17, 0x11, 0x3f, 0x79, 0xa1, 0x7c, 0x17, 0xa9, 0x12, 0x34,
    0x34, 0x12, 0xef, 0xd0, 0x56, 0x8e, 0xc2, 0x33,
};
static const unsigned char kat537_addin1[] = {
    0x44, 0x8c, 0x62, 0xe9, 0x72, 0xab, 0xf9, 0xc4, 0x0b, 0xf9, 0x18, 0x5d,
    0x42, 0xb3, 0x45, 0xaa, 0x6f, 0xba, 0x9b, 0x98, 0xee, 0xe2, 0x35, 0x3a,
    0x5f, 0xd0, 0x15, 0x97, 0x38, 0x7b, 0x60, 0x96,
};
static const unsigned char kat537_retbits[] = {
    0xe6, 0x60, 0xe4, 0xe2, 0xb2, 0x42, 0x58, 0x71, 0x31, 0x8c, 0x45, 0x3a,
    0x6b, 0x31, 0x1d, 0x0b, 0xbb, 0x01, 0x85, 0x2b, 0xde, 0x28, 0x0b, 0x0e,
    0x38, 0xd6, 0xc3, 0xd0, 0x7f, 0x4b, 0x8a, 0x9b, 0x3a, 0x4f, 0xd1, 0xa2,
    0x6d, 0x26, 0x3b, 0xb7, 0x57, 0x44, 0x7a, 0xca, 0x15, 0x30, 0xa0, 0x41,
    0xc1, 0x27, 0xa7, 0x7e, 0xe2, 0xe5, 0xed, 0x83, 0x25, 0x9f, 0x64, 0x62,
    0xe1, 0xdb, 0xef, 0x6d,
};
static const struct drbg_kat_no_reseed kat537_t = {
    11, kat537_entropyin, kat537_nonce, kat537_persstr,
    kat537_addin0, kat537_addin1, kat537_retbits
};
static const struct drbg_kat kat537 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat537_t
};

static const unsigned char kat538_entropyin[] = {
    0xac, 0xed, 0xba, 0x21, 0x5c, 0x39, 0x0b, 0xab, 0xcf, 0x0f, 0x1d, 0x46,
    0x19, 0x3c, 0x18, 0x9e, 0x15, 0xc5, 0x80, 0xc1, 0xbf, 0x49, 0x09, 0x65,
    0xbd, 0xee, 0x96, 0x3d, 0x38, 0xfe, 0x97, 0x13,
};
static const unsigned char kat538_nonce[] = {
    0xbb, 0xa7, 0x79, 0x9f, 0x72, 0x41, 0xe1, 0x26, 0x78, 0x6d, 0x53, 0x34,
    0x16, 0x86, 0x3a, 0x33,
};
static const unsigned char kat538_persstr[] = {
    0x7b, 0x1e, 0x05, 0x7e, 0xe5, 0x2c, 0xa0, 0xd4, 0x15, 0x3c, 0x1c, 0xe6,
    0x48, 0x28, 0xc8, 0x70, 0x52, 0x30, 0xd1, 0x9f, 0x90, 0x0d, 0x7a, 0x77,
    0x1a, 0xa7, 0x71, 0x82, 0xfc, 0xc7, 0x88, 0x07,
};
static const unsigned char kat538_addin0[] = {
    0x6c, 0xd1, 0x68, 0x49, 0x88, 0xd7, 0x0d, 0x56, 0x9b, 0x0a, 0xf5, 0x58,
    0x5f, 0xb9, 0xb8, 0xba, 0x69, 0x8b, 0x59, 0x9d, 0xc4, 0xa0, 0xbd, 0x7b,
    0x21, 0x9c, 0x2e, 0x92, 0x01, 0x4a, 0x1e, 0xf4,
};
static const unsigned char kat538_addin1[] = {
    0x85, 0x7a, 0x8b, 0x64, 0xa8, 0x20, 0xa2, 0x1b, 0xc4, 0x9a, 0x16, 0x3f,
    0x72, 0x17, 0x5f, 0xda, 0x94, 0xa2, 0x87, 0x82, 0x87, 0x86, 0x6c, 0xde,
    0x86, 0x36, 0xd4, 0x6f, 0xba, 0xd9, 0xde, 0x33,
};
static const unsigned char kat538_retbits[] = {
    0xa3, 0x21, 0xe4, 0x2e, 0xab, 0x68, 0x41, 0x50, 0xfe, 0x55, 0xdb, 0x9e,
    0x1f, 0x4e, 0xfc, 0x9a, 0xb6, 0xbc, 0x94, 0xad, 0xfd, 0x78, 0xb8, 0x7a,
    0x8d, 0x58, 0xeb, 0xd1, 0x1d, 0x20, 0xab, 0x3c, 0xae, 0x9b, 0x0e, 0x1b,
    0xdb, 0x31, 0xf1, 0xec, 0x1c, 0x1d, 0x85, 0xc1, 0xf0, 0x71, 0xac, 0xe6,
    0x39, 0xc0, 0x53, 0xa0, 0x68, 0x2a, 0x07, 0x71, 0x47, 0x31, 0x45, 0x06,
    0xc0, 0x7b, 0xda, 0xad,
};
static const struct drbg_kat_no_reseed kat538_t = {
    12, kat538_entropyin, kat538_nonce, kat538_persstr,
    kat538_addin0, kat538_addin1, kat538_retbits
};
static const struct drbg_kat kat538 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat538_t
};

static const unsigned char kat539_entropyin[] = {
    0x5d, 0xce, 0xd2, 0xaf, 0xdf, 0x72, 0xd2, 0x2d, 0x2b, 0x42, 0x80, 0xe1,
    0x81, 0x18, 0x6e, 0xb3, 0xfc, 0xd5, 0x5c, 0xbe, 0xc0, 0xd4, 0xa0, 0x26,
    0x93, 0x62, 0x62, 0xbd, 0x0f, 0x7d, 0xf6, 0x36,
};
static const unsigned char kat539_nonce[] = {
    0x07, 0x6c, 0xde, 0x76, 0x2a, 0xc3, 0x9b, 0x18, 0xe4, 0x48, 0x82, 0x6e,
    0xdc, 0xea, 0x42, 0x9c,
};
static const unsigned char kat539_persstr[] = {
    0xef, 0x7d, 0xb5, 0xdd, 0x64, 0x02, 0x8a, 0x67, 0x45, 0xbf, 0xe0, 0xe5,
    0x7c, 0x83, 0x05, 0xc4, 0x5e, 0xc2, 0x44, 0x38, 0xd9, 0xa6, 0x53, 0x71,
    0xb7, 0xd9, 0x8c, 0x93, 0xab, 0x73, 0x20, 0xcc,
};
static const unsigned char kat539_addin0[] = {
    0x64, 0x3b, 0xe9, 0x51, 0x2f, 0xb8, 0x23, 0x5a, 0x12, 0xdb, 0x98, 0x11,
    0x11, 0x07, 0x9b, 0x5c, 0x45, 0x36, 0x99, 0xd4, 0x8f, 0x0b, 0x1b, 0xc0,
    0x0d, 0x64, 0x3c, 0xe0, 0xe0, 0x17, 0xdd, 0x04,
};
static const unsigned char kat539_addin1[] = {
    0xb8, 0x53, 0xe7, 0x9d, 0xdb, 0x46, 0xb2, 0x59, 0x15, 0x89, 0xfe, 0xe4,
    0x3d, 0x3d, 0xd7, 0xad, 0x9e, 0xa1, 0x69, 0x33, 0x2f, 0x83, 0xd4, 0x46,
    0x44, 0xc3, 0x0b, 0x0c, 0xf5, 0xc9, 0x79, 0x2d,
};
static const unsigned char kat539_retbits[] = {
    0x48, 0x78, 0x78, 0xdc, 0x78, 0xa9, 0xe8, 0x6f, 0x5f, 0xaa, 0x7f, 0x77,
    0x99, 0x76, 0xf3, 0xf0, 0xc2, 0x07, 0xf8, 0xe9, 0xcf, 0xd3, 0x1b, 0xea,
    0x54, 0xbf, 0x12, 0x1c, 0x41, 0x22, 0x7c, 0x86, 0xd7, 0x25, 0x64, 0xc0,
    0xa1, 0xb5, 0x85, 0xe6, 0xf2, 0x84, 0xaa, 0x73, 0x2b, 0x5d, 0xbe, 0xac,
    0x78, 0xab, 0x5c, 0xe1, 0x72, 0xb2, 0x86, 0xf6, 0xdd, 0x2d, 0x7b, 0x97,
    0x63, 0xe7, 0x67, 0x25,
};
static const struct drbg_kat_no_reseed kat539_t = {
    13, kat539_entropyin, kat539_nonce, kat539_persstr,
    kat539_addin0, kat539_addin1, kat539_retbits
};
static const struct drbg_kat kat539 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat539_t
};

static const unsigned char kat540_entropyin[] = {
    0x4d, 0x6d, 0x0d, 0x6c, 0x03, 0x55, 0x28, 0x80, 0xc0, 0xd0, 0xc5, 0x67,
    0xf1, 0x43, 0xe2, 0x97, 0xf7, 0x7d, 0x9a, 0x5e, 0xcb, 0x54, 0x7c, 0x28,
    0xab, 0xcf, 0xf1, 0x71, 0x5b, 0x06, 0x10, 0x84,
};
static const unsigned char kat540_nonce[] = {
    0x1b, 0x47, 0x76, 0x61, 0x13, 0x28, 0x34, 0xab, 0xcd, 0x71, 0x3b, 0xe2,
    0x27, 0x97, 0x88, 0x20,
};
static const unsigned char kat540_persstr[] = {
    0x1c, 0x58, 0x44, 0x91, 0xac, 0xb3, 0xf1, 0xa6, 0x2e, 0xcf, 0x84, 0x12,
    0x0e, 0x4c, 0x6a, 0xdb, 0x67, 0x6c, 0x23, 0x3a, 0x16, 0x84, 0xf2, 0xd1,
    0xb8, 0xcd, 0x54, 0x2a, 0x62, 0x9f, 0x6c, 0x84,
};
static const unsigned char kat540_addin0[] = {
    0xf2, 0x4b, 0x84, 0x67, 0xae, 0x57, 0x66, 0x9e, 0x4a, 0xd6, 0x75, 0x88,
    0xa4, 0x1f, 0x95, 0x12, 0xfd, 0x48, 0x97, 0x66, 0x42, 0xdb, 0x17, 0x37,
    0x12, 0x20, 0x64, 0xa1, 0xc1, 0x23, 0x44, 0x48,
};
static const unsigned char kat540_addin1[] = {
    0x7a, 0x6f, 0x1a, 0x19, 0x11, 0x17, 0xc2, 0xdc, 0xd5, 0xc9, 0xc1, 0x75,
    0xc8, 0x27, 0xd8, 0xda, 0x04, 0xe3, 0x80, 0x2a, 0xe8, 0x35, 0xb2, 0x51,
    0x82, 0xfa, 0x83, 0xbf, 0x91, 0xdc, 0x68, 0x71,
};
static const unsigned char kat540_retbits[] = {
    0x1d, 0xce, 0xe0, 0xaf, 0xbf, 0xfb, 0x8a, 0x0c, 0xf4, 0x00, 0x2c, 0xfc,
    0x26, 0x2c, 0x4f, 0xe0, 0x58, 0xd1, 0x3e, 0x35, 0xc7, 0x4c, 0x70, 0xb0,
    0x10, 0x47, 0x6c, 0x71, 0x53, 0x31, 0x3d, 0x71, 0x45, 0xdf, 0x88, 0x9c,
    0x2f, 0x99, 0x1f, 0x8f, 0x97, 0x8f, 0x0d, 0x73, 0xc9, 0x5e, 0x86, 0x7c,
    0x4d, 0x2d, 0xa4, 0xf3, 0xf4, 0xd9, 0x32, 0xd0, 0xb9, 0xa2, 0x3c, 0x78,
    0xe0, 0xf1, 0x12, 0xa5,
};
static const struct drbg_kat_no_reseed kat540_t = {
    14, kat540_entropyin, kat540_nonce, kat540_persstr,
    kat540_addin0, kat540_addin1, kat540_retbits
};
static const struct drbg_kat kat540 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat540_t
};

static const unsigned char kat541_entropyin[] = {
    0x8b, 0x0b, 0xcb, 0x3f, 0x93, 0x21, 0x70, 0x41, 0x67, 0x39, 0xea, 0x42,
    0xe7, 0xdc, 0xdc, 0x6f, 0xa9, 0x60, 0x64, 0x5b, 0xc0, 0x18, 0x82, 0x01,
    0x34, 0xf7, 0x14, 0xb3, 0xc6, 0x91, 0x2b, 0x56,
};
static const unsigned char kat541_nonce[] = {
    0xba, 0xc0, 0xfd, 0xc0, 0xc4, 0x17, 0xaa, 0x26, 0x9b, 0xbd, 0xea, 0x77,
    0xe9, 0x28, 0xf9, 0xf8,
};
static const unsigned char kat541_persstr[] = {0};
static const unsigned char kat541_addin0[] = {0};
static const unsigned char kat541_addin1[] = {0};
static const unsigned char kat541_retbits[] = {
    0xd9, 0xc4, 0xfd, 0x81, 0xf6, 0x62, 0x1a, 0x8c, 0xf0, 0x6d, 0x61, 0x2e,
    0x9a, 0x84, 0xb8, 0x0f, 0xa1, 0x3d, 0x09, 0x8d, 0xce, 0xaf, 0x2c, 0x08,
    0x3d, 0xc8, 0x1c, 0xd8, 0x0c, 0xae, 0xdd, 0x10, 0x5c, 0x7f, 0x27, 0x89,
    0x96, 0x3a, 0x16, 0x7d, 0x72, 0xf7, 0x6e, 0x81, 0x17, 0x80, 0x01, 0xfd,
    0x93, 0xde, 0x46, 0x23, 0xc2, 0x60, 0xfe, 0x9e, 0xeb, 0xce, 0xd8, 0x9f,
    0x7b, 0x4b, 0x04, 0x7a,
};
static const struct drbg_kat_no_reseed kat541_t = {
    0, kat541_entropyin, kat541_nonce, kat541_persstr,
    kat541_addin0, kat541_addin1, kat541_retbits
};
static const struct drbg_kat kat541 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat541_t
};

static const unsigned char kat542_entropyin[] = {
    0x67, 0xb6, 0xe8, 0x4d, 0x5a, 0x56, 0x0a, 0xf4, 0xd9, 0x27, 0x45, 0x85,
    0x3d, 0xa8, 0x3c, 0x4e, 0x8d, 0xcf, 0xf4, 0x69, 0x86, 0x9e, 0xca, 0x69,
    0x98, 0x10, 0x55, 0xba, 0x4c, 0x6f, 0x84, 0xc3,
};
static const unsigned char kat542_nonce[] = {
    0xaa, 0xbc, 0x8d, 0x3a, 0xb5, 0x93, 0xdb, 0xea, 0x35, 0xfa, 0xb1, 0xff,
    0x6c, 0xdc, 0x26, 0xfb,
};
static const unsigned char kat542_persstr[] = {0};
static const unsigned char kat542_addin0[] = {0};
static const unsigned char kat542_addin1[] = {0};
static const unsigned char kat542_retbits[] = {
    0xe7, 0x4a, 0xd6, 0x22, 0xa7, 0x12, 0x98, 0x98, 0x3a, 0xa2, 0x10, 0x66,
    0xd7, 0x88, 0xfd, 0xcd, 0x6a, 0xfd, 0xc9, 0xaa, 0xf7, 0xfc, 0x8a, 0x55,
    0x53, 0x4e, 0xc0, 0x91, 0x7d, 0x68, 0x40, 0xd1, 0x5c, 0x1b, 0xa2, 0xf0,
    0xa7, 0x03, 0xf0, 0x4b, 0x14, 0x8b, 0xd7, 0xbc, 0x49, 0x83, 0xb2, 0x79,
    0xa4, 0x14, 0xe3, 0x93, 0x7c, 0x17, 0xa8, 0x18, 0x1e, 0x64, 0x4e, 0xa0,
    0x66, 0x2d, 0xbe, 0xbc,
};
static const struct drbg_kat_no_reseed kat542_t = {
    1, kat542_entropyin, kat542_nonce, kat542_persstr,
    kat542_addin0, kat542_addin1, kat542_retbits
};
static const struct drbg_kat kat542 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat542_t
};

static const unsigned char kat543_entropyin[] = {
    0xbe, 0x57, 0xcf, 0x16, 0xb2, 0x64, 0x81, 0xaa, 0xb3, 0x16, 0x4b, 0x80,
    0x60, 0xc2, 0x9f, 0x17, 0x98, 0x27, 0x11, 0xb4, 0x51, 0x18, 0x8d, 0xea,
    0xcd, 0xc9, 0x80, 0x5e, 0xf7, 0xe0, 0x16, 0xbe,
};
static const unsigned char kat543_nonce[] = {
    0x85, 0x48, 0x4d, 0xaa, 0x20, 0xb8, 0x60, 0x25, 0x07, 0xb3, 0xd7, 0x68,
    0x50, 0x93, 0x9e, 0x59,
};
static const unsigned char kat543_persstr[] = {0};
static const unsigned char kat543_addin0[] = {0};
static const unsigned char kat543_addin1[] = {0};
static const unsigned char kat543_retbits[] = {
    0x4b, 0x95, 0x46, 0x92, 0x16, 0x03, 0x7d, 0xe3, 0xaf, 0xb2, 0x79, 0x0d,
    0xd1, 0x52, 0x34, 0x73, 0xcb, 0x8d, 0xbd, 0xf7, 0x23, 0x0b, 0x0f, 0x85,
    0x43, 0xf7, 0x38, 0xc6, 0xba, 0xa0, 0xa1, 0xa4, 0x6d, 0x13, 0x36, 0x6f,
    0xe3, 0x16, 0x4f, 0x24, 0x56, 0x76, 0xdf, 0xe1, 0xaf, 0x02, 0x14, 0xc5,
    0x58, 0x1e, 0x82, 0x79, 0x0f, 0xde, 0x30, 0xb0, 0x20, 0x3e, 0x45, 0x54,
    0x80, 0x4b, 0x98, 0x26,
};
static const struct drbg_kat_no_reseed kat543_t = {
    2, kat543_entropyin, kat543_nonce, kat543_persstr,
    kat543_addin0, kat543_addin1, kat543_retbits
};
static const struct drbg_kat kat543 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat543_t
};

static const unsigned char kat544_entropyin[] = {
    0x3c, 0xbb, 0xdc, 0x1b, 0xbb, 0x6f, 0x00, 0x58, 0x97, 0xd6, 0x53, 0x84,
    0xed, 0x79, 0x79, 0xdf, 0x6d, 0x71, 0x08, 0x55, 0x9c, 0x3e, 0x76, 0x19,
    0xd6, 0xdf, 0xc8, 0xda, 0xd8, 0xe6, 0x54, 0x9f,
};
static const unsigned char kat544_nonce[] = {
    0x7c, 0x9f, 0x78, 0xb7, 0xd1, 0x5e, 0xa7, 0x32, 0x44, 0x12, 0x3f, 0xfd,
    0xb4, 0x48, 0x9f, 0x0d,
};
static const unsigned char kat544_persstr[] = {0};
static const unsigned char kat544_addin0[] = {0};
static const unsigned char kat544_addin1[] = {0};
static const unsigned char kat544_retbits[] = {
    0xc2, 0x1a, 0xf2, 0x65, 0x18, 0xc9, 0x7d, 0x67, 0x63, 0xb7, 0x53, 0x93,
    0x9a, 0xb0, 0xdf, 0x09, 0xcd, 0x2d, 0x59, 0xfa, 0x10, 0x90, 0x93, 0x3e,
    0x26, 0x41, 0xc9, 0x43, 0x9f, 0x79, 0xfb, 0x3b, 0x90, 0x22, 0xfa, 0x6e,
    0x07, 0xc9, 0x95, 0x0c, 0xe9, 0xea, 0xdc, 0x33, 0x27, 0xdc, 0x49, 0xf9,
    0x1d, 0xc8, 0x00, 0x6c, 0x7d, 0xa5, 0x2b, 0x02, 0x1e, 0x5a, 0xe8, 0x35,
    0x82, 0xf3, 0x04, 0x75,
};
static const struct drbg_kat_no_reseed kat544_t = {
    3, kat544_entropyin, kat544_nonce, kat544_persstr,
    kat544_addin0, kat544_addin1, kat544_retbits
};
static const struct drbg_kat kat544 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat544_t
};

static const unsigned char kat545_entropyin[] = {
    0x06, 0x31, 0x1c, 0xc8, 0x1e, 0x75, 0xb1, 0x22, 0x69, 0xbc, 0x67, 0xed,
    0x0e, 0x13, 0x13, 0x48, 0x0f, 0x32, 0x4b, 0x75, 0x2a, 0x1f, 0xd7, 0x83,
    0xce, 0x09, 0x77, 0x0d, 0x1d, 0x40, 0x00, 0xfb,
};
static const unsigned char kat545_nonce[] = {
    0xae, 0x2c, 0xc5, 0x17, 0xb2, 0x15, 0x85, 0x5b, 0x1e, 0x91, 0xd1, 0xf7,
    0x94, 0x2d, 0x93, 0x1e,
};
static const unsigned char kat545_persstr[] = {0};
static const unsigned char kat545_addin0[] = {0};
static const unsigned char kat545_addin1[] = {0};
static const unsigned char kat545_retbits[] = {
    0x5e, 0xd9, 0x14, 0x43, 0x8a, 0x62, 0xcc, 0x4d, 0x46, 0x33, 0x67, 0xb1,
    0x7c, 0x89, 0x60, 0x42, 0x90, 0xe9, 0x72, 0x35, 0x22, 0xef, 0x80, 0x1e,
    0xad, 0x51, 0x5c, 0xa3, 0x52, 0xe0, 0x99, 0xd6, 0xfa, 0x13, 0x62, 0xde,
    0xd3, 0x2a, 0x3f, 0xa3, 0x6b, 0x24, 0x53, 0x42, 0x2f, 0x11, 0x4f, 0x8c,
    0xf2, 0x46, 0xc1, 0x8c, 0x8c, 0xdd, 0x78, 0x6a, 0xa2, 0x43, 0x80, 0x1c,
    0xae, 0xf2, 0xb2, 0xed,
};
static const struct drbg_kat_no_reseed kat545_t = {
    4, kat545_entropyin, kat545_nonce, kat545_persstr,
    kat545_addin0, kat545_addin1, kat545_retbits
};
static const struct drbg_kat kat545 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat545_t
};

static const unsigned char kat546_entropyin[] = {
    0x59, 0x27, 0xad, 0x80, 0xb1, 0x2b, 0x86, 0x26, 0x9e, 0xb3, 0x1e, 0x85,
    0xc3, 0xb1, 0x22, 0xa7, 0xb8, 0x2d, 0x8e, 0xd5, 0x78, 0x22, 0x76, 0x34,
    0x4b, 0xe3, 0xf5, 0x2e, 0xdf, 0x62, 0x80, 0xdb,
};
static const unsigned char kat546_nonce[] = {
    0xe6, 0x74, 0x0d, 0x37, 0x13, 0xda, 0x61, 0x7b, 0xaa, 0x5a, 0x5c, 0xd5,
    0xb4, 0xd6, 0xbe, 0xd8,
};
static const unsigned char kat546_persstr[] = {0};
static const unsigned char kat546_addin0[] = {0};
static const unsigned char kat546_addin1[] = {0};
static const unsigned char kat546_retbits[] = {
    0x52, 0xf3, 0x10, 0xec, 0x9a, 0xdc, 0xf4, 0x91, 0xde, 0x58, 0xdc, 0x52,
    0x59, 0x5d, 0xdb, 0xb1, 0xfe, 0xe9, 0x66, 0xe6, 0x9e, 0x60, 0xba, 0x0e,
    0x65, 0xbe, 0xd7, 0xfb, 0xf8, 0xdc, 0x3d, 0x58, 0x14, 0x85, 0x38, 0xa1,
    0xb4, 0x56, 0xae, 0xce, 0x1b, 0x41, 0x47, 0x9d, 0xea, 0x06, 0xe1, 0x01,
    0x5b, 0x7b, 0x10, 0xde, 0xb6, 0x6c, 0x97, 0x74, 0x02, 0x9f, 0x0b, 0x52,
    0xce, 0xb1, 0xf5, 0x8a,
};
static const struct drbg_kat_no_reseed kat546_t = {
    5, kat546_entropyin, kat546_nonce, kat546_persstr,
    kat546_addin0, kat546_addin1, kat546_retbits
};
static const struct drbg_kat kat546 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat546_t
};

static const unsigned char kat547_entropyin[] = {
    0x6e, 0xc1, 0x35, 0x79, 0x2f, 0x27, 0x02, 0x67, 0x72, 0x73, 0x9e, 0x89,
    0x3a, 0xc5, 0x08, 0x52, 0x20, 0x35, 0x1f, 0x59, 0x05, 0x75, 0xfe, 0xb9,
    0xa3, 0xd3, 0x79, 0x2a, 0xc9, 0x13, 0xd8, 0xa4,
};
static const unsigned char kat547_nonce[] = {
    0xa0, 0x96, 0x9c, 0xdb, 0x40, 0x59, 0x14, 0x36, 0xcb, 0x63, 0x7f, 0xe0,
    0x93, 0x15, 0x6f, 0x16,
};
static const unsigned char kat547_persstr[] = {0};
static const unsigned char kat547_addin0[] = {0};
static const unsigned char kat547_addin1[] = {0};
static const unsigned char kat547_retbits[] = {
    0x08, 0xd2, 0x73, 0x57, 0x18, 0xf9, 0xc8, 0x82, 0x32, 0x2f, 0xde, 0xf7,
    0xb1, 0x21, 0x33, 0x9d, 0x5a, 0x7f, 0x59, 0xb0, 0x74, 0xd3, 0xc6, 0xf8,
    0xb8, 0xfc, 0x51, 0x7a, 0x07, 0x2d, 0x35, 0x6b, 0xb7, 0x02, 0xd9, 0x0d,
    0xbc, 0xec, 0x40, 0xe0, 0x1c, 0x4d, 0xdb, 0x62, 0x81, 0x38, 0x7e, 0x73,
    0x1b, 0x2b, 0xd0, 0xe7, 0x2a, 0x61, 0x64, 0xef, 0x4a, 0x72, 0x1d, 0xad,
    0x01, 0x02, 0x3b, 0x2a,
};
static const struct drbg_kat_no_reseed kat547_t = {
    6, kat547_entropyin, kat547_nonce, kat547_persstr,
    kat547_addin0, kat547_addin1, kat547_retbits
};
static const struct drbg_kat kat547 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat547_t
};

static const unsigned char kat548_entropyin[] = {
    0xe9, 0xc4, 0x8b, 0xe6, 0xd6, 0x5c, 0x8d, 0xd3, 0xbb, 0x37, 0x22, 0x31,
    0xdf, 0xa8, 0xa7, 0x07, 0x8d, 0xf5, 0x01, 0xa9, 0x23, 0x8d, 0x63, 0x11,
    0x61, 0xce, 0xe3, 0xf0, 0xaf, 0xa9, 0x78, 0x89,
};
static const unsigned char kat548_nonce[] = {
    0x1d, 0x32, 0xb5, 0xa3, 0x7b, 0xe3, 0xad, 0x60, 0x96, 0xe8, 0x0c, 0x7f,
    0xab, 0xf8, 0xa7, 0x9b,
};
static const unsigned char kat548_persstr[] = {0};
static const unsigned char kat548_addin0[] = {0};
static const unsigned char kat548_addin1[] = {0};
static const unsigned char kat548_retbits[] = {
    0x02, 0x4f, 0xa9, 0x5b, 0x3c, 0xef, 0x85, 0xdd, 0x9a, 0x22, 0x7c, 0x93,
    0xa8, 0x22, 0x3b, 0x1d, 0x02, 0xd5, 0x7f, 0xd5, 0x06, 0xed, 0x42, 0xa1,
    0x03, 0xf7, 0xee, 0x0d, 0x92, 0x05, 0xca, 0xd9, 0xfc, 0x51, 0xbc, 0xbd,
    0xb7, 0xc4, 0x01, 0xd1, 0x5f, 0xcb, 0x06, 0xe4, 0xa5, 0xed, 0xa1, 0x75,
    0x36, 0xfe, 0x9e, 0x93, 0x81, 0x2e, 0xb9, 0x9a, 0x87, 0x9b, 0x91, 0x06,
    0xca, 0x0b, 0x6d, 0x8d,
};
static const struct drbg_kat_no_reseed kat548_t = {
    7, kat548_entropyin, kat548_nonce, kat548_persstr,
    kat548_addin0, kat548_addin1, kat548_retbits
};
static const struct drbg_kat kat548 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat548_t
};

static const unsigned char kat549_entropyin[] = {
    0x6f, 0x37, 0xbb, 0x62, 0xe2, 0xb3, 0x01, 0xde, 0x8f, 0xf6, 0xc5, 0xa8,
    0x0a, 0xd8, 0x63, 0x4a, 0xc7, 0x18, 0x73, 0x1b, 0xa7, 0x89, 0xb2, 0x47,
    0xc3, 0x50, 0x72, 0x16, 0xf6, 0x4b, 0xae, 0x42,
};
static const unsigned char kat549_nonce[] = {
    0x41, 0xe5, 0x14, 0x98, 0x0d, 0x2c, 0x79, 0x57, 0x25, 0x84, 0xd0, 0xe6,
    0x0f, 0x20, 0x2c, 0x67,
};
static const unsigned char kat549_persstr[] = {0};
static const unsigned char kat549_addin0[] = {0};
static const unsigned char kat549_addin1[] = {0};
static const unsigned char kat549_retbits[] = {
    0xc1, 0xad, 0x07, 0x4f, 0xb5, 0x74, 0x8f, 0x09, 0xa3, 0x4b, 0x8f, 0x8b,
    0x8b, 0x15, 0x78, 0x9d, 0x26, 0xa6, 0xaf, 0xf4, 0x71, 0x40, 0x95, 0xc5,
    0x4f, 0x62, 0x61, 0x1d, 0x88, 0xee, 0x2a, 0x45, 0xa4, 0xac, 0x11, 0x10,
    0xa3, 0xe4, 0xec, 0x9d, 0x59, 0xba, 0x85, 0xec, 0x1d, 0x80, 0xda, 0xac,
    0xdd, 0xae, 0x3e, 0xa5, 0x8f, 0x95, 0x39, 0x21, 0x9a, 0xfa, 0xf5, 0x7f,
    0xe9, 0x53, 0x53, 0x6e,
};
static const struct drbg_kat_no_reseed kat549_t = {
    8, kat549_entropyin, kat549_nonce, kat549_persstr,
    kat549_addin0, kat549_addin1, kat549_retbits
};
static const struct drbg_kat kat549 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat549_t
};

static const unsigned char kat550_entropyin[] = {
    0x54, 0x90, 0x2f, 0x1c, 0xe7, 0x88, 0xfd, 0xdb, 0x8d, 0x81, 0x5a, 0x80,
    0xe5, 0xb0, 0x2d, 0xa3, 0x6a, 0xe0, 0x9f, 0xe2, 0xce, 0xcb, 0x6c, 0xad,
    0x76, 0x6b, 0x75, 0x72, 0x38, 0xa0, 0x95, 0x46,
};
static const unsigned char kat550_nonce[] = {
    0x52, 0xfc, 0x08, 0x26, 0x5f, 0x84, 0xbb, 0x3b, 0x2f, 0x5d, 0xab, 0x01,
    0xeb, 0x8a, 0xde, 0x3c,
};
static const unsigned char kat550_persstr[] = {0};
static const unsigned char kat550_addin0[] = {0};
static const unsigned char kat550_addin1[] = {0};
static const unsigned char kat550_retbits[] = {
    0x0d, 0x11, 0x4d, 0x27, 0xd1, 0xc4, 0xae, 0x86, 0x18, 0x6c, 0x31, 0x3e,
    0x4f, 0x1b, 0xfb, 0x00, 0x2d, 0xe3, 0xf2, 0xa8, 0x27, 0xb7, 0x15, 0x39,
    0x41, 0xf9, 0x0f, 0x99, 0x4b, 0x56, 0xc8, 0xc2, 0x66, 0x81, 0x6c, 0xc6,
    0x9e, 0x10, 0x72, 0xa2, 0xb3, 0xbe, 0x19, 0xc2, 0x9f, 0x80, 0xab, 0x3d,
    0x03, 0x9e, 0xd5, 0x72, 0x2f, 0xed, 0xcc, 0x01, 0x6f, 0xa8, 0x2b, 0x39,
    0x61, 0x78, 0x28, 0x58,
};
static const struct drbg_kat_no_reseed kat550_t = {
    9, kat550_entropyin, kat550_nonce, kat550_persstr,
    kat550_addin0, kat550_addin1, kat550_retbits
};
static const struct drbg_kat kat550 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat550_t
};

static const unsigned char kat551_entropyin[] = {
    0xfe, 0x55, 0xe2, 0xf9, 0x6b, 0x16, 0xd2, 0xdd, 0x91, 0x5c, 0xa1, 0x1e,
    0x25, 0x04, 0xd7, 0x20, 0xf1, 0xaa, 0x19, 0x18, 0x63, 0x1c, 0x47, 0x2c,
    0xb0, 0xb4, 0x49, 0x9b, 0x7d, 0x88, 0x6b, 0x56,
};
static const unsigned char kat551_nonce[] = {
    0x85, 0x30, 0xfc, 0x3e, 0xba, 0x5a, 0x01, 0xb7, 0x63, 0x08, 0x9e, 0x80,
    0x43, 0xdf, 0xdf, 0x79,
};
static const unsigned char kat551_persstr[] = {0};
static const unsigned char kat551_addin0[] = {0};
static const unsigned char kat551_addin1[] = {0};
static const unsigned char kat551_retbits[] = {
    0xbe, 0x3e, 0x72, 0xb3, 0xaa, 0x0b, 0x07, 0xd0, 0x75, 0x66, 0x45, 0xa5,
    0xf5, 0xa7, 0x4a, 0xa6, 0xa0, 0x15, 0xb7, 0xce, 0xa6, 0x85, 0x81, 0xfc,
    0x8e, 0xd7, 0x62, 0x66, 0x0d, 0x9e, 0x54, 0xa7, 0x81, 0x0e, 0x71, 0x2d,
    0x81, 0x98, 0x8e, 0xa7, 0x06, 0xad, 0x2b, 0x93, 0x8e, 0x33, 0x65, 0x1a,
    0x3e, 0x8a, 0x67, 0x55, 0xd5, 0x92, 0xf5, 0x21, 0x7f, 0xf2, 0xc9, 0xea,
    0x8f, 0x41, 0xa5, 0xc4,
};
static const struct drbg_kat_no_reseed kat551_t = {
    10, kat551_entropyin, kat551_nonce, kat551_persstr,
    kat551_addin0, kat551_addin1, kat551_retbits
};
static const struct drbg_kat kat551 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat551_t
};

static const unsigned char kat552_entropyin[] = {
    0xf5, 0xfc, 0x14, 0x97, 0x1b, 0x96, 0x6a, 0x4f, 0x95, 0xbb, 0x13, 0xd5,
    0x5a, 0x68, 0x2b, 0xfd, 0xc7, 0xda, 0x7f, 0x26, 0xc9, 0x43, 0xdb, 0x40,
    0x1a, 0x17, 0x9f, 0x23, 0xf4, 0x7e, 0xc3, 0xbe,
};
static const unsigned char kat552_nonce[] = {
    0xf0, 0x51, 0xee, 0xba, 0xbc, 0xe3, 0x59, 0x90, 0x21, 0xc3, 0x64, 0x81,
    0x1b, 0x11, 0xf2, 0x8b,
};
static const unsigned char kat552_persstr[] = {0};
static const unsigned char kat552_addin0[] = {0};
static const unsigned char kat552_addin1[] = {0};
static const unsigned char kat552_retbits[] = {
    0x9e, 0x18, 0x4c, 0x66, 0x83, 0x4d, 0x73, 0xb4, 0x61, 0x0e, 0x53, 0x62,
    0xcd, 0xf9, 0x08, 0xa5, 0xf8, 0x49, 0x39, 0xec, 0xf0, 0x1a, 0x58, 0xeb,
    0xe6, 0xc2, 0xdc, 0x50, 0xca, 0x0c, 0x24, 0xbe, 0xd9, 0x94, 0xa8, 0x2e,
    0xec, 0x99, 0x8e, 0x18, 0xaf, 0xb9, 0x57, 0x27, 0xc4, 0x27, 0x86, 0xf5,
    0xe9, 0x4d, 0x60, 0xc6, 0x06, 0x57, 0x6a, 0xf8, 0x3f, 0x3b, 0x33, 0x52,
    0x79, 0xa1, 0x01, 0x4d,
};
static const struct drbg_kat_no_reseed kat552_t = {
    11, kat552_entropyin, kat552_nonce, kat552_persstr,
    kat552_addin0, kat552_addin1, kat552_retbits
};
static const struct drbg_kat kat552 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat552_t
};

static const unsigned char kat553_entropyin[] = {
    0x26, 0xe6, 0xf4, 0x4b, 0x11, 0x2f, 0x52, 0x40, 0xf0, 0xe2, 0x24, 0x9c,
    0x2a, 0xa4, 0xd5, 0xdf, 0x3d, 0xf0, 0x94, 0x31, 0x29, 0x6f, 0xee, 0xad,
    0x01, 0xcf, 0x65, 0x9c, 0x62, 0xde, 0x50, 0x07,
};
static const unsigned char kat553_nonce[] = {
    0x8c, 0xaa, 0x8a, 0x24, 0xc2, 0x8a, 0x17, 0xcf, 0xb6, 0x86, 0x13, 0xe2,
    0xfb, 0x3b, 0xf3, 0x7e,
};
static const unsigned char kat553_persstr[] = {0};
static const unsigned char kat553_addin0[] = {0};
static const unsigned char kat553_addin1[] = {0};
static const unsigned char kat553_retbits[] = {
    0x91, 0xa3, 0xad, 0x60, 0xda, 0x8a, 0x72, 0xd7, 0x04, 0x05, 0xa1, 0xba,
    0x81, 0x9e, 0xcb, 0x14, 0x7f, 0x33, 0x88, 0x68, 0xa4, 0x16, 0x68, 0xdb,
    0x59, 0xff, 0xfb, 0x03, 0x11, 0xa8, 0x34, 0x4a, 0xe0, 0xea, 0x58, 0xd9,
    0x7b, 0x6d, 0x3c, 0x31, 0x64, 0x90, 0xb5, 0xc6, 0x23, 0x65, 0x3e, 0x81,
    0x6c, 0xc1, 0xb9, 0x1d, 0x5f, 0xcd, 0x9b, 0x4d, 0x70, 0xe7, 0x71, 0x7d,
    0x6e, 0xe2, 0xb7, 0x02,
};
static const struct drbg_kat_no_reseed kat553_t = {
    12, kat553_entropyin, kat553_nonce, kat553_persstr,
    kat553_addin0, kat553_addin1, kat553_retbits
};
static const struct drbg_kat kat553 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat553_t
};

static const unsigned char kat554_entropyin[] = {
    0xf2, 0x91, 0xd7, 0xe5, 0xc2, 0x96, 0x13, 0x13, 0x18, 0x4f, 0x80, 0x23,
    0x27, 0x7b, 0x8f, 0xac, 0x09, 0xa5, 0x43, 0xa2, 0x68, 0xee, 0x50, 0x6a,
    0x53, 0x41, 0x70, 0x67, 0xa8, 0xcc, 0x12, 0xb6,
};
static const unsigned char kat554_nonce[] = {
    0x55, 0x24, 0x3c, 0xc5, 0x46, 0x0e, 0x24, 0xbf, 0x0c, 0x2f, 0x92, 0xc0,
    0xcf, 0xb2, 0x3e, 0xb6,
};
static const unsigned char kat554_persstr[] = {0};
static const unsigned char kat554_addin0[] = {0};
static const unsigned char kat554_addin1[] = {0};
static const unsigned char kat554_retbits[] = {
    0x80, 0x9d, 0x31, 0xa2, 0xa2, 0x6c, 0x94, 0x0b, 0x3f, 0x04, 0x5f, 0x16,
    0xb8, 0x59, 0xf2, 0x70, 0x8f, 0xd9, 0x19, 0x6a, 0x39, 0x83, 0x51, 0x41,
    0x04, 0xd8, 0x0f, 0x1f, 0x68, 0x16, 0x6a, 0xf0, 0xe4, 0x8b, 0x98, 0x27,
    0x1b, 0x76, 0xa2, 0x67, 0xea, 0x1e, 0x05, 0x25, 0x0a, 0x3f, 0x7f, 0x25,
    0x81, 0xc1, 0xd6, 0x35, 0x67, 0x08, 0xc2, 0x93, 0xf2, 0x08, 0x94, 0x4b,
    0x3d, 0xa0, 0xa4, 0x25,
};
static const struct drbg_kat_no_reseed kat554_t = {
    13, kat554_entropyin, kat554_nonce, kat554_persstr,
    kat554_addin0, kat554_addin1, kat554_retbits
};
static const struct drbg_kat kat554 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat554_t
};

static const unsigned char kat555_entropyin[] = {
    0x8b, 0x08, 0x78, 0xd7, 0x33, 0xc8, 0xa4, 0x8a, 0x79, 0xfa, 0x67, 0x4d,
    0xfa, 0x88, 0x9b, 0x34, 0xfd, 0x05, 0x22, 0x01, 0x6d, 0x2f, 0x68, 0xfd,
    0xdd, 0x3d, 0xd0, 0x5f, 0x1a, 0xf7, 0x85, 0x06,
};
static const unsigned char kat555_nonce[] = {
    0xf7, 0x72, 0x49, 0x63, 0x34, 0xfd, 0x93, 0xb4, 0x66, 0x47, 0x29, 0x7a,
    0xc4, 0x44, 0x62, 0x9c,
};
static const unsigned char kat555_persstr[] = {0};
static const unsigned char kat555_addin0[] = {0};
static const unsigned char kat555_addin1[] = {0};
static const unsigned char kat555_retbits[] = {
    0x91, 0x91, 0xc5, 0x97, 0x0a, 0x37, 0xfd, 0xd5, 0x71, 0x5a, 0x7c, 0x6a,
    0x09, 0xa7, 0xdd, 0xdb, 0xfb, 0xee, 0x8d, 0x37, 0xee, 0x45, 0xa0, 0x35,
    0xb3, 0x38, 0x23, 0xc6, 0xe8, 0x6f, 0xe0, 0xe2, 0xe2, 0xe5, 0x7f, 0xf7,
    0xd0, 0x8e, 0x0a, 0x4f, 0x39, 0xd8, 0x1c, 0x18, 0xb9, 0xb5, 0xfd, 0x48,
    0x7e, 0x84, 0xca, 0x95, 0xdf, 0x37, 0x2d, 0x36, 0xdc, 0xbf, 0xac, 0x68,
    0x2e, 0x04, 0x4b, 0x34,
};
static const struct drbg_kat_no_reseed kat555_t = {
    14, kat555_entropyin, kat555_nonce, kat555_persstr,
    kat555_addin0, kat555_addin1, kat555_retbits
};
static const struct drbg_kat kat555 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat555_t
};

static const unsigned char kat556_entropyin[] = {
    0xd6, 0x74, 0x39, 0xab, 0xf1, 0xe1, 0x62, 0xe5, 0xb2, 0x59, 0x41, 0x60,
    0x5a, 0x8a, 0xeb, 0xa7, 0xd6, 0x86, 0xde, 0xc1, 0x33, 0x25, 0x7f, 0x6c,
    0x22, 0x0e, 0x1c, 0x59, 0x5e, 0x95, 0x4a, 0x07,
};
static const unsigned char kat556_nonce[] = {
    0x69, 0xff, 0x33, 0x10, 0x14, 0x1d, 0xbf, 0x3e, 0xce, 0x40, 0x9a, 0xde,
    0x58, 0x74, 0x51, 0x13,
};
static const unsigned char kat556_persstr[] = {0};
static const unsigned char kat556_addin0[] = {
    0x03, 0xe7, 0x95, 0xbe, 0x83, 0x79, 0xc4, 0x81, 0xcb, 0x32, 0x53, 0x40,
    0x11, 0xca, 0x6b, 0xf5, 0x24, 0xdc, 0x75, 0x49, 0x78, 0xee, 0x5e, 0xbe,
    0xe4, 0x75, 0x12, 0x9a, 0xd3, 0x9e, 0xca, 0x98,
};
static const unsigned char kat556_addin1[] = {
    0x56, 0x85, 0xc7, 0x33, 0x0f, 0x33, 0x00, 0x45, 0x15, 0xf8, 0xc0, 0xab,
    0x27, 0xf2, 0xa1, 0xcb, 0xe0, 0xc8, 0xa4, 0xa6, 0x80, 0x6d, 0x6c, 0x84,
    0x86, 0xe0, 0x21, 0x7b, 0x43, 0xe8, 0x59, 0xf2,
};
static const unsigned char kat556_retbits[] = {
    0xa6, 0xd2, 0x2a, 0x43, 0x70, 0x25, 0x1c, 0x51, 0x97, 0x8f, 0xed, 0xc7,
    0xe7, 0x75, 0x3c, 0x78, 0x17, 0x9e, 0xd1, 0x94, 0x3d, 0x2f, 0xf1, 0xb5,
    0xa3, 0x74, 0x86, 0x01, 0x06, 0x04, 0x1a, 0x30, 0x4b, 0x12, 0x4d, 0x47,
    0xcf, 0xa3, 0x04, 0xc9, 0x09, 0xf7, 0xd4, 0x17, 0x84, 0x38, 0x46, 0xd5,
    0x2d, 0xcc, 0x7e, 0xbc, 0xf5, 0xc9, 0x3a, 0xfe, 0xf8, 0x85, 0xc8, 0x93,
    0xb4, 0x0c, 0x81, 0xed,
};
static const struct drbg_kat_no_reseed kat556_t = {
    0, kat556_entropyin, kat556_nonce, kat556_persstr,
    kat556_addin0, kat556_addin1, kat556_retbits
};
static const struct drbg_kat kat556 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat556_t
};

static const unsigned char kat557_entropyin[] = {
    0x8f, 0x7c, 0x8c, 0xd0, 0xbc, 0xdf, 0xcc, 0xe6, 0x61, 0x4c, 0xc6, 0x51,
    0x1d, 0x51, 0x95, 0xad, 0xe6, 0xda, 0xd5, 0xf6, 0x1f, 0xef, 0x59, 0x88,
    0x6f, 0x24, 0x02, 0x12, 0x2e, 0x43, 0x0a, 0x8d,
};
static const unsigned char kat557_nonce[] = {
    0x17, 0xd1, 0x41, 0x2b, 0x83, 0x44, 0x59, 0x9a, 0x39, 0xb9, 0x60, 0x76,
    0x1c, 0x6a, 0xc3, 0x9f,
};
static const unsigned char kat557_persstr[] = {0};
static const unsigned char kat557_addin0[] = {
    0xe5, 0x39, 0x59, 0x3c, 0xfc, 0xc7, 0x9e, 0xbd, 0x0a, 0x5e, 0x7b, 0xe3,
    0x24, 0x3e, 0x51, 0xa7, 0x7b, 0xf3, 0x81, 0x76, 0x90, 0xb2, 0xff, 0xc8,
    0x0c, 0xe5, 0xdc, 0x35, 0xf2, 0xb2, 0xd4, 0xb8,
};
static const unsigned char kat557_addin1[] = {
    0xe6, 0xa2, 0x4e, 0x9f, 0x76, 0x24, 0xaf, 0xb3, 0xa5, 0x5d, 0x99, 0x74,
    0xf8, 0xcb, 0x1a, 0xdd, 0xc4, 0x43, 0x2f, 0xdf, 0xea, 0xc7, 0xc3, 0x5a,
    0x61, 0x61, 0x11, 0x58, 0x1c, 0xd1, 0x9b, 0x2f,
};
static const unsigned char kat557_retbits[] = {
    0x5f, 0xc2, 0x07, 0x36, 0xda, 0x9c, 0xf5, 0xa8, 0x10, 0x36, 0x4b, 0x6a,
    0xca, 0x24, 0xed, 0xf7, 0x58, 0xbd, 0x20, 0xeb, 0xd3, 0x31, 0x73, 0xdb,
    0x87, 0x4b, 0x64, 0x1b, 0x84, 0x70, 0xab, 0x9a, 0x8a, 0x63, 0x3d, 0x12,
    0x38, 0xba, 0x99, 0x01, 0x03, 0x95, 0x6c, 0x0f, 0x5e, 0x2b, 0x28, 0x4f,
    0x3b, 0x47, 0x3c, 0x28, 0xd0, 0x05, 0x5d, 0x7e, 0x9b, 0xec, 0x0b, 0x83,
    0x90, 0x88, 0x91, 0x7a,
};
static const struct drbg_kat_no_reseed kat557_t = {
    1, kat557_entropyin, kat557_nonce, kat557_persstr,
    kat557_addin0, kat557_addin1, kat557_retbits
};
static const struct drbg_kat kat557 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat557_t
};

static const unsigned char kat558_entropyin[] = {
    0xd3, 0x00, 0x5b, 0x86, 0x84, 0x6d, 0x8b, 0x45, 0x9e, 0xd9, 0x10, 0x52,
    0x18, 0xa1, 0x78, 0x7c, 0x1d, 0x80, 0x01, 0x20, 0x53, 0xca, 0xa3, 0x8a,
    0xde, 0x19, 0x6b, 0xff, 0x82, 0x73, 0xc8, 0x93,
};
static const unsigned char kat558_nonce[] = {
    0x3a, 0x68, 0x9f, 0x93, 0x0c, 0xd4, 0xc5, 0x38, 0x62, 0xc6, 0x81, 0x91,
    0xb5, 0xad, 0xbf, 0x14,
};
static const unsigned char kat558_persstr[] = {0};
static const unsigned char kat558_addin0[] = {
    0x70, 0xc4, 0xbb, 0xeb, 0x4f, 0x7f, 0x52, 0x7d, 0xf7, 0xe0, 0xad, 0x69,
    0x85, 0x1b, 0x66, 0x40, 0x8b, 0x21, 0x54, 0xe7, 0xa2, 0x6f, 0xa5, 0x42,
    0xd9, 0x2f, 0x0e, 0x5b, 0x1a, 0x96, 0x95, 0x75,
};
static const unsigned char kat558_addin1[] = {
    0x86, 0x05, 0x72, 0xf8, 0x81, 0xd5, 0xa9, 0x7c, 0xd7, 0xc9, 0xc8, 0xee,
    0x39, 0xe4, 0xfe, 0xfb, 0x67, 0xb9, 0x14, 0x7d, 0x37, 0xfe, 0xa5, 0xa6,
    0x4f, 0x58, 0xcd, 0x2e, 0x7a, 0xb6, 0x83, 0x84,
};
static const unsigned char kat558_retbits[] = {
    0x4d, 0x8b, 0x57, 0x8e, 0x27, 0xad, 0x27, 0xde, 0x26, 0x4c, 0x63, 0xc3,
    0xd2, 0xf8, 0x71, 0x3d, 0x58, 0x02, 0x2c, 0x54, 0x14, 0x99, 0x33, 0x51,
    0x10, 0xa2, 0x80, 0x99, 0x0b, 0xb0, 0xb3, 0xd3, 0x8a, 0x13, 0x94, 0x3c,
    0xfa, 0xbb, 0x7a, 0x6b, 0xfd, 0xc2, 0xdb, 0x2c, 0x05, 0x09, 0x09, 0x89,
    0x89, 0xe1, 0x31, 0xc8, 0x9a, 0x76, 0x22, 0x23, 0x5b, 0x76, 0x9e, 0xcc,
    0x2c, 0x50, 0x9d, 0xd4,
};
static const struct drbg_kat_no_reseed kat558_t = {
    2, kat558_entropyin, kat558_nonce, kat558_persstr,
    kat558_addin0, kat558_addin1, kat558_retbits
};
static const struct drbg_kat kat558 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat558_t
};

static const unsigned char kat559_entropyin[] = {
    0x0b, 0x99, 0x25, 0xbf, 0xf6, 0x8f, 0x29, 0x84, 0x3c, 0x8a, 0x05, 0xc6,
    0x95, 0xde, 0x37, 0x8b, 0x37, 0x0e, 0x8d, 0xc9, 0xa6, 0xdd, 0xe3, 0x3f,
    0xe8, 0x61, 0x52, 0xfa, 0x58, 0x7a, 0xf1, 0xf1,
};
static const unsigned char kat559_nonce[] = {
    0xb4, 0x38, 0x5b, 0x66, 0x09, 0x6b, 0x2d, 0x2d, 0x0d, 0xac, 0x68, 0x9d,
    0x04, 0x3c, 0x60, 0x91,
};
static const unsigned char kat559_persstr[] = {0};
static const unsigned char kat559_addin0[] = {
    0xb8, 0x0b, 0xd9, 0x34, 0xcc, 0x3e, 0x73, 0xea, 0x85, 0xac, 0x4a, 0x7f,
    0xb5, 0x19, 0x0d, 0x1f, 0xa2, 0x98, 0x8d, 0x29, 0xaf, 0xb3, 0xa3, 0x08,
    0xc9, 0x22, 0x05, 0xc0, 0xb7, 0x45, 0xb0, 0x70,
};
static const unsigned char kat559_addin1[] = {
    0x4d, 0xaf, 0xcb, 0x75, 0x28, 0x23, 0x20, 0x34, 0xd2, 0x61, 0x70, 0x63,
    0x61, 0xcc, 0x87, 0x67, 0x18, 0x75, 0xee, 0x67, 0xac, 0xe9, 0x6c, 0xad,
    0x09, 0x9b, 0x12, 0x66, 0xa7, 0x5c, 0xb7, 0x28,
};
static const unsigned char kat559_retbits[] = {
    0x5b, 0x56, 0x1f, 0x9f, 0xa2, 0x60, 0x2b, 0x55, 0x3b, 0xc3, 0xd4, 0xc9,
    0x00, 0x1f, 0x8b, 0x4d, 0x98, 0x59, 0x1c, 0x8d, 0x81, 0x33, 0x66, 0xa7,
    0x92, 0x06, 0xeb, 0x3d, 0x7e, 0x92, 0xb4, 0x1c, 0x34, 0xae, 0xf3, 0x23,
    0x80, 0xd7, 0x9d, 0x4f, 0x7d, 0xf6, 0x73, 0x56, 0x96, 0xf1, 0x47, 0xce,
    0x56, 0x90, 0xc3, 0xc5, 0x67, 0x6d, 0xfd, 0xd2, 0x0a, 0xee, 0x3b, 0x4f,
    0xc9, 0x55, 0x9e, 0x17,
};
static const struct drbg_kat_no_reseed kat559_t = {
    3, kat559_entropyin, kat559_nonce, kat559_persstr,
    kat559_addin0, kat559_addin1, kat559_retbits
};
static const struct drbg_kat kat559 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat559_t
};

static const unsigned char kat560_entropyin[] = {
    0xb7, 0xc3, 0x22, 0xf8, 0x12, 0xcd, 0xd3, 0xc1, 0x86, 0xdc, 0x9d, 0xd5,
    0x89, 0xf8, 0x7d, 0x46, 0x9f, 0x64, 0xe5, 0xed, 0x80, 0xa0, 0x4e, 0x41,
    0x44, 0x18, 0xe2, 0x64, 0x00, 0x86, 0x48, 0xc1,
};
static const unsigned char kat560_nonce[] = {
    0x8b, 0x66, 0x9f, 0xc0, 0xe3, 0xac, 0xb7, 0x94, 0x90, 0x41, 0x00, 0x5a,
    0x27, 0xfc, 0x14, 0x45,
};
static const unsigned char kat560_persstr[] = {0};
static const unsigned char kat560_addin0[] = {
    0xde, 0x10, 0x1f, 0xca, 0xf1, 0xc3, 0xbf, 0x36, 0x8e, 0x70, 0x40, 0xd3,
    0x9d, 0x57, 0x66, 0x5c, 0x83, 0x5c, 0x40, 0x92, 0x7c, 0xa8, 0x63, 0xd7,
    0x0f, 0x74, 0x34, 0x1c, 0x1a, 0x11, 0x3b, 0x68,
};
static const unsigned char kat560_addin1[] = {
    0x7f, 0xed, 0xdb, 0x0e, 0x87, 0x86, 0x4d, 0x35, 0x87, 0x94, 0x66, 0xda,
    0xc2, 0x32, 0xeb, 0x86, 0xfa, 0xc0, 0xc4, 0xfd, 0x74, 0x58, 0x51, 0x26,
    0x1a, 0x63, 0x55, 0xd3, 0x6e, 0x01, 0x80, 0x05,
};
static const unsigned char kat560_retbits[] = {
    0xf7, 0x91, 0x66, 0xbc, 0x53, 0xf7, 0x3b, 0xdf, 0x87, 0x3d, 0x76, 0xba,
    0x81, 0xe1, 0xeb, 0x62, 0x97, 0x03, 0x27, 0xc8, 0xe6, 0xfa, 0xc1, 0x0d,
    0xc2, 0x36, 0x67, 0xb9, 0xda, 0x74, 0xec, 0x58, 0x30, 0xf5, 0x50, 0x25,
    0xd9, 0x52, 0xea, 0xac, 0x65, 0xdd, 0x3b, 0x52, 0xc1, 0x51, 0x7c, 0x79,
    0xb0, 0x55, 0x9a, 0x39, 0xc5, 0x4d, 0x30, 0xc9, 0x7c, 0xce, 0x83, 0x2c,
    0x1b, 0xdd, 0xe4, 0xf2,
};
static const struct drbg_kat_no_reseed kat560_t = {
    4, kat560_entropyin, kat560_nonce, kat560_persstr,
    kat560_addin0, kat560_addin1, kat560_retbits
};
static const struct drbg_kat kat560 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat560_t
};

static const unsigned char kat561_entropyin[] = {
    0xda, 0xba, 0x89, 0xb6, 0x2a, 0x44, 0x75, 0xe8, 0xc2, 0x21, 0x40, 0x52,
    0xd2, 0x04, 0xc1, 0xff, 0xd2, 0x48, 0xe0, 0xc3, 0x00, 0x8a, 0x0b, 0xd1,
    0x23, 0x83, 0x0d, 0xb5, 0x02, 0x3e, 0x53, 0xc2,
};
static const unsigned char kat561_nonce[] = {
    0x77, 0xfc, 0x74, 0x10, 0x14, 0xe5, 0x14, 0x24, 0x30, 0x58, 0xd3, 0xa2,
    0xcf, 0x03, 0x63, 0x74,
};
static const unsigned char kat561_persstr[] = {0};
static const unsigned char kat561_addin0[] = {
    0xaa, 0x93, 0x2a, 0x71, 0xbb, 0xbd, 0x86, 0xb3, 0xf3, 0xa7, 0x02, 0xfa,
    0x38, 0x4d, 0x49, 0x06, 0x1f, 0x31, 0xc2, 0x5e, 0x80, 0x9d, 0xd9, 0xa0,
    0x30, 0x34, 0xac, 0x3c, 0xdf, 0x6b, 0x2e, 0xf2,
};
static const unsigned char kat561_addin1[] = {
    0x92, 0x62, 0xca, 0xaf, 0xb3, 0x38, 0x3f, 0xaf, 0x78, 0xd9, 0xb0, 0x44,
    0x0f, 0x0c, 0x5b, 0xe2, 0x8e, 0x26, 0x0f, 0x3d, 0xb7, 0x77, 0x28, 0xd7,
    0xfc, 0xa6, 0x79, 0x98, 0x4c, 0xc3, 0x8d, 0xdf,
};
static const unsigned char kat561_retbits[] = {
    0x78, 0x4c, 0x8d, 0x58, 0x81, 0x34, 0x9a, 0x13, 0xbc, 0x1a, 0x38, 0xed,
    0xac, 0x6e, 0x8f, 0xe7, 0xdc, 0x3a, 0x3b, 0x66, 0x2b, 0x40, 0xb7, 0x40,
    0xeb, 0xe8, 0xc8, 0x9d, 0x14, 0x4c, 0xca, 0xb3, 0x19, 0xb4, 0xcd, 0x90,
    0x82, 0x0d, 0xd3, 0xf5, 0xec, 0xc3, 0x10, 0x0c, 0x9f, 0x72, 0x87, 0x8d,
    0x5e, 0xeb, 0x21, 0xf4, 0x62, 0x34, 0xf4, 0x09, 0x4b, 0x71, 0xb8, 0xb3,
    0x42, 0x85, 0xc5, 0x50,
};
static const struct drbg_kat_no_reseed kat561_t = {
    5, kat561_entropyin, kat561_nonce, kat561_persstr,
    kat561_addin0, kat561_addin1, kat561_retbits
};
static const struct drbg_kat kat561 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat561_t
};

static const unsigned char kat562_entropyin[] = {
    0x8f, 0x7c, 0x64, 0x0c, 0xa1, 0x5c, 0x75, 0x84, 0x04, 0x46, 0xd4, 0x42,
    0xf4, 0x13, 0xbb, 0xfe, 0xaf, 0xb4, 0xc2, 0xb8, 0x87, 0x5f, 0x7f, 0x1b,
    0xfd, 0x16, 0x12, 0x2b, 0x81, 0xe1, 0xac, 0x1f,
};
static const unsigned char kat562_nonce[] = {
    0x69, 0x60, 0x13, 0xa4, 0xc7, 0xe7, 0xc7, 0x24, 0xa4, 0x07, 0x23, 0xb5,
    0x55, 0x7b, 0x00, 0x56,
};
static const unsigned char kat562_persstr[] = {0};
static const unsigned char kat562_addin0[] = {
    0xb7, 0xd2, 0x57, 0x70, 0xe0, 0xb2, 0xbd, 0x1b, 0x6a, 0x82, 0x20, 0xf8,
    0x6b, 0x3d, 0x75, 0x9d, 0x20, 0xd8, 0x50, 0x89, 0x80, 0xcf, 0x59, 0x1f,
    0x44, 0x3d, 0xde, 0x3a, 0xcd, 0x3e, 0xf6, 0x48,
};
static const unsigned char kat562_addin1[] = {
    0xbc, 0x08, 0xf8, 0x46, 0x21, 0xaf, 0xb6, 0xa7, 0xd3, 0x31, 0xa0, 0x1b,
    0xfc, 0x57, 0x16, 0xa3, 0x1b, 0xcc, 0xcf, 0x05, 0xa7, 0x8e, 0x69, 0x6d,
    0xc7, 0x68, 0x9b, 0x98, 0x53, 0x89, 0xe1, 0x2b,
};
static const unsigned char kat562_retbits[] = {
    0xd4, 0x86, 0xcb, 0xac, 0x79, 0x4e, 0xc9, 0xc6, 0xb2, 0xac, 0xc6, 0xaa,
    0xdb, 0x93, 0xe3, 0x2b, 0x30, 0xb2, 0x7e, 0xa3, 0x72, 0x1f, 0xb6, 0x14,
    0xe8, 0x23, 0x47, 0xef, 0xf4, 0x61, 0x64, 0x9c, 0xb8, 0xd0, 0xf6, 0xb4,
    0x3a, 0x39, 0xb2, 0xe5, 0xf6, 0xf4, 0x03, 0xff, 0x4a, 0xfe, 0xfd, 0x90,
    0xad, 0xba, 0x5e, 0x96, 0x16, 0x4a, 0xc7, 0x3b, 0x27, 0xd8, 0xae, 0x20,
    0x98, 0xe6, 0x32, 0xcb,
};
static const struct drbg_kat_no_reseed kat562_t = {
    6, kat562_entropyin, kat562_nonce, kat562_persstr,
    kat562_addin0, kat562_addin1, kat562_retbits
};
static const struct drbg_kat kat562 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat562_t
};

static const unsigned char kat563_entropyin[] = {
    0xb8, 0xbe, 0xd1, 0x95, 0x21, 0xf6, 0xd2, 0x47, 0x1b, 0x76, 0x14, 0x37,
    0xf1, 0x0c, 0x9d, 0xb9, 0xaf, 0x2a, 0xc3, 0x03, 0x21, 0x27, 0xdc, 0x5e,
    0xb3, 0x8e, 0x15, 0x71, 0x82, 0xe2, 0xe2, 0x5e,
};
static const unsigned char kat563_nonce[] = {
    0xe0, 0x0b, 0x1b, 0x87, 0x12, 0xb2, 0x19, 0xb7, 0x84, 0x44, 0xa3, 0xfc,
    0xab, 0xdd, 0x0f, 0x68,
};
static const unsigned char kat563_persstr[] = {0};
static const unsigned char kat563_addin0[] = {
    0xed, 0x74, 0x4e, 0x78, 0x61, 0x75, 0x9e, 0x33, 0x7e, 0x0d, 0x8d, 0xb5,
    0xed, 0xa6, 0xf6, 0x6c, 0xb1, 0x42, 0x40, 0xd1, 0xff, 0x7a, 0x86, 0xf9,
    0x0c, 0x96, 0xdf, 0xa2, 0xf4, 0xce, 0x85, 0xf9,
};
static const unsigned char kat563_addin1[] = {
    0x90, 0xf3, 0xb8, 0x92, 0x0d, 0xbd, 0xbd, 0x8f, 0xf6, 0xc6, 0x60, 0x1b,
    0x57, 0xa5, 0xfc, 0x90, 0xaa, 0x13, 0xbb, 0xa9, 0x88, 0x43, 0xdc, 0x51,
    0x43, 0xc0, 0x18, 0x2d, 0xdc, 0x23, 0x1d, 0xfb,
};
static const unsigned char kat563_retbits[] = {
    0x33, 0x10, 0xbe, 0x30, 0x08, 0x80, 0x29, 0xe1, 0x35, 0x24, 0xea, 0x9e,
    0x8d, 0x6b, 0xd9, 0xef, 0xaa, 0x72, 0xd1, 0xc5, 0x8d, 0x4f, 0x9d, 0x7e,
    0x98, 0x53, 0x50, 0x8b, 0xa1, 0x63, 0xb9, 0x06, 0xc4, 0x58, 0x02, 0x0b,
    0xe3, 0xcb, 0x7c, 0x5f, 0x3a, 0x2f, 0xa8, 0x89, 0x53, 0xf1, 0x27, 0xe7,
    0xe1, 0x17, 0xc1, 0xf3, 0x89, 0x47, 0xa6, 0x68, 0x0b, 0x53, 0xd2, 0x8d,
    0x56, 0x5a, 0x3d, 0xdf,
};
static const struct drbg_kat_no_reseed kat563_t = {
    7, kat563_entropyin, kat563_nonce, kat563_persstr,
    kat563_addin0, kat563_addin1, kat563_retbits
};
static const struct drbg_kat kat563 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat563_t
};

static const unsigned char kat564_entropyin[] = {
    0x01, 0x42, 0x76, 0xad, 0x54, 0x57, 0x97, 0x88, 0xdf, 0xda, 0xf1, 0x2f,
    0x2d, 0xf1, 0x88, 0xff, 0xef, 0xab, 0xd3, 0x51, 0x64, 0xf3, 0x1c, 0xc8,
    0x73, 0xc6, 0xd3, 0x91, 0x49, 0x6e, 0x09, 0xee,
};
static const unsigned char kat564_nonce[] = {
    0x46, 0x62, 0xec, 0x7d, 0x07, 0xbf, 0x4f, 0xf1, 0x0b, 0xcd, 0x0c, 0x83,
    0x41, 0x06, 0xe1, 0x79,
};
static const unsigned char kat564_persstr[] = {0};
static const unsigned char kat564_addin0[] = {
    0xe6, 0x6f, 0x96, 0x0f, 0x9a, 0x89, 0x3d, 0x39, 0xd1, 0x1c, 0x04, 0x26,
    0xbc, 0x9e, 0x7e, 0x1f, 0x6f, 0x07, 0x50, 0x55, 0x7a, 0x69, 0x69, 0xe4,
    0x45, 0xce, 0xbb, 0x47, 0xd2, 0xf5, 0xd4, 0x27,
};
static const unsigned char kat564_addin1[] = {
    0x6b, 0x39, 0xf4, 0x5b, 0xbb, 0x78, 0x5c, 0xa9, 0xec, 0x23, 0x30, 0x3f,
    0xbc, 0xe7, 0x09, 0x3a, 0x26, 0xfb, 0x87, 0x9b, 0x13, 0xfa, 0xdd, 0x92,
    0x7d, 0xc9, 0x50, 0x19, 0xb8, 0xa2, 0x5c, 0xb5,
};
static const unsigned char kat564_retbits[] = {
    0xdd, 0xa5, 0x11, 0x8a, 0xe2, 0x34, 0xe6, 0xae, 0xb5, 0xcf, 0x63, 0x4e,
    0x57, 0xa3, 0xa3, 0xce, 0x76, 0x81, 0xbd, 0x80, 0x36, 0x90, 0xf8, 0xd6,
    0x88, 0x86, 0xba, 0x52, 0x26, 0xfb, 0x40, 0x18, 0x26, 0x44, 0xa6, 0x43,
    0xa5, 0x54, 0xcc, 0x58, 0x6d, 0x9c, 0xcd, 0x14, 0x8a, 0x75, 0x24, 0x01,
    0x42, 0xcd, 0x8f, 0x62, 0xeb, 0xe8, 0x74, 0x7b, 0x4b, 0xe6, 0xe7, 0x53,
    0x23, 0x58, 0x20, 0x06,
};
static const struct drbg_kat_no_reseed kat564_t = {
    8, kat564_entropyin, kat564_nonce, kat564_persstr,
    kat564_addin0, kat564_addin1, kat564_retbits
};
static const struct drbg_kat kat564 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat564_t
};

static const unsigned char kat565_entropyin[] = {
    0xcf, 0xfc, 0xfb, 0xed, 0xb3, 0x5f, 0xf5, 0xbf, 0x3c, 0x79, 0x6d, 0x1a,
    0xb9, 0x4a, 0x8a, 0xe1, 0xb9, 0x35, 0xad, 0xdf, 0x6f, 0xdd, 0x8b, 0xe0,
    0x00, 0xae, 0x30, 0x99, 0xda, 0xea, 0x7b, 0xd4,
};
static const unsigned char kat565_nonce[] = {
    0x1f, 0x7e, 0xc8, 0x65, 0x8f, 0x17, 0xe6, 0x4d, 0x1e, 0xd1, 0xb4, 0x98,
    0x15, 0xef, 0x9b, 0xef,
};
static const unsigned char kat565_persstr[] = {0};
static const unsigned char kat565_addin0[] = {
    0x8d, 0x44, 0xea, 0xe3, 0x90, 0x8e, 0x4a, 0xd5, 0x3f, 0xc0, 0xfa, 0xb4,
    0x17, 0x99, 0xbb, 0x03, 0x56, 0xb3, 0xdd, 0xd4, 0x73, 0xfd, 0xe4, 0x29,
    0xaf, 0x21, 0xd7, 0xe0, 0x4b, 0x42, 0x6b, 0x80,
};
static const unsigned char kat565_addin1[] = {
    0x9e, 0x3f, 0x82, 0x8e, 0x8d, 0xdf, 0x07, 0xce, 0xb5, 0x8f, 0xe8, 0xf1,
    0x69, 0x7d, 0xae, 0xc6, 0x66, 0xb9, 0xe2, 0x50, 0xfa, 0x5a, 0x83, 0x6f,
    0xfd, 0x18, 0xee, 0x17, 0x36, 0x50, 0xe7, 0x2c,
};
static const unsigned char kat565_retbits[] = {
    0x10, 0x01, 0x62, 0xaa, 0x41, 0x80, 0xc5, 0xe5, 0xa1, 0x8c, 0x51, 0xbf,
    0x7d, 0x2b, 0x78, 0xca, 0x94, 0x4c, 0x70, 0xe6, 0x2b, 0xde, 0xa5, 0x54,
    0x83, 0x7a, 0x86, 0x81, 0x35, 0xcc, 0xbe, 0x1b, 0x79, 0xbe, 0xbd, 0xfa,
    0xe4, 0x09, 0x22, 0x1f, 0x43, 0x8d, 0xf7, 0x28, 0x33, 0xb8, 0xdd, 0x9d,
    0x17, 0xdc, 0xbe, 0x06, 0xef, 0x8d, 0x19, 0x01, 0x01, 0x27, 0x31, 0xc7,
    0xbe, 0x12, 0xfc, 0x53,
};
static const struct drbg_kat_no_reseed kat565_t = {
    9, kat565_entropyin, kat565_nonce, kat565_persstr,
    kat565_addin0, kat565_addin1, kat565_retbits
};
static const struct drbg_kat kat565 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat565_t
};

static const unsigned char kat566_entropyin[] = {
    0xdb, 0x31, 0xef, 0x9d, 0xd1, 0x26, 0x6c, 0x60, 0xea, 0xaf, 0xbe, 0xca,
    0x54, 0x2f, 0x52, 0xb4, 0xfd, 0x98, 0x9b, 0x4b, 0x43, 0x69, 0x35, 0xea,
    0x65, 0xeb, 0xc6, 0x25, 0x54, 0xbe, 0x96, 0x2c,
};
static const unsigned char kat566_nonce[] = {
    0xd0, 0x3a, 0x8b, 0xa0, 0x34, 0x85, 0x2b, 0x4f, 0x96, 0x25, 0x0b, 0xdd,
    0x89, 0xf3, 0x68, 0x02,
};
static const unsigned char kat566_persstr[] = {0};
static const unsigned char kat566_addin0[] = {
    0x62, 0xeb, 0xe7, 0x71, 0x5a, 0x7d, 0xdc, 0x82, 0x0e, 0x44, 0xab, 0x31,
    0x37, 0x67, 0x77, 0x6f, 0x87, 0x85, 0xb1, 0x81, 0x42, 0x6f, 0xbf, 0x49,
    0x3e, 0x7c, 0xbc, 0xce, 0x13, 0x60, 0xfd, 0x23,
};
static const unsigned char kat566_addin1[] = {
    0x05, 0x28, 0xb6, 0x90, 0x41, 0xc4, 0xe5, 0xea, 0xda, 0xb6, 0x18, 0xa9,
    0x29, 0xf1, 0x76, 0xf7, 0x7c, 0x34, 0x5b, 0x6d, 0x89, 0xae, 0xe9, 0xaa,
    0x95, 0x5c, 0x5a, 0x8c, 0xa8, 0xfc, 0x8b, 0x92,
};
static const unsigned char kat566_retbits[] = {
    0x01, 0x7c, 0xf4, 0xff, 0x68, 0x6d, 0xf4, 0xc0, 0x6f, 0xbb, 0xc0, 0xbe,
    0x97, 0x89, 0xb8, 0x28, 0xf1, 0x5b, 0xe5, 0x4a, 0xaf, 0xba, 0x72, 0x87,
    0xb4, 0x39, 0x1e, 0x9e, 0xc7, 0xb7, 0xd0, 0x02, 0x5d, 0xb0, 0x12, 0xdf,
    0x4b, 0xb3, 0x5c, 0x74, 0x06, 0x16, 0x77, 0x20, 0x7a, 0x7d, 0x7a, 0x06,
    0x1e, 0xf1, 0x23, 0x89, 0x3e, 0xa4, 0xd7, 0xc8, 0x22, 0x0e, 0xca, 0xca,
    0xd5, 0x68, 0x48, 0xb1,
};
static const struct drbg_kat_no_reseed kat566_t = {
    10, kat566_entropyin, kat566_nonce, kat566_persstr,
    kat566_addin0, kat566_addin1, kat566_retbits
};
static const struct drbg_kat kat566 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat566_t
};

static const unsigned char kat567_entropyin[] = {
    0x06, 0xb2, 0x6d, 0x4f, 0xfa, 0xb4, 0xa8, 0x61, 0x14, 0x14, 0xc8, 0xfd,
    0xe8, 0x7e, 0x51, 0x69, 0x2f, 0xfe, 0x50, 0x78, 0x83, 0x4b, 0x51, 0x98,
    0x19, 0x3d, 0xf7, 0xf9, 0x8f, 0x1c, 0x52, 0x61,
};
static const unsigned char kat567_nonce[] = {
    0xde, 0x49, 0xf9, 0xfa, 0xba, 0xea, 0x3f, 0x8d, 0xd1, 0xec, 0x58, 0x1c,
    0x71, 0xd5, 0xc1, 0xda,
};
static const unsigned char kat567_persstr[] = {0};
static const unsigned char kat567_addin0[] = {
    0x6b, 0xba, 0x85, 0xa3, 0xf3, 0x93, 0x86, 0x5a, 0x6a, 0x7b, 0x73, 0x39,
    0x16, 0xc2, 0x8a, 0xe3, 0x9b, 0xa4, 0x03, 0xa0, 0x34, 0x8a, 0x2d, 0x3b,
    0x7a, 0x33, 0xfd, 0xf8, 0x19, 0x57, 0x1b, 0xf9,
};
static const unsigned char kat567_addin1[] = {
    0xc3, 0x2f, 0xd2, 0x84, 0xc4, 0x3a, 0xa0, 0x72, 0xeb, 0x72, 0xa7, 0xe4,
    0x7f, 0x40, 0x8a, 0x87, 0x7a, 0xe4, 0x0e, 0x12, 0x09, 0xd1, 0x12, 0xc9,
    0x9d, 0x45, 0x7e, 0x7f, 0x96, 0x53, 0xcb, 0x86,
};
static const unsigned char kat567_retbits[] = {
    0x1f, 0x0f, 0x54, 0x04, 0xf5, 0x1d, 0x48, 0xc3, 0xf3, 0x62, 0x3b, 0x93,
    0x84, 0x32, 0xf7, 0xdd, 0x42, 0x49, 0x64, 0x9f, 0x61, 0xb6, 0xf1, 0x03,
    0xc2, 0xfd, 0x8b, 0x30, 0xdb, 0xf5, 0xb4, 0xea, 0x95, 0x7b, 0x5d, 0xe0,
    0x96, 0xac, 0x47, 0xcb, 0x10, 0x7e, 0x52, 0x2d, 0xf0, 0x13, 0x80, 0x76,
    0x34, 0x2d, 0x6c, 0x2e, 0xa0, 0x8e, 0xcc, 0x2b, 0x48, 0x1c, 0x3e, 0x73,
    0xef, 0x48, 0xa0, 0x46,
};
static const struct drbg_kat_no_reseed kat567_t = {
    11, kat567_entropyin, kat567_nonce, kat567_persstr,
    kat567_addin0, kat567_addin1, kat567_retbits
};
static const struct drbg_kat kat567 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat567_t
};

static const unsigned char kat568_entropyin[] = {
    0x21, 0xbe, 0x14, 0x42, 0xe6, 0x89, 0x53, 0xaf, 0x26, 0x56, 0xec, 0x3a,
    0xdb, 0xa3, 0xa6, 0x5a, 0x7b, 0x45, 0x9b, 0x7a, 0xb4, 0xe7, 0x9c, 0x77,
    0x5a, 0xd2, 0x54, 0x4c, 0xe9, 0xa4, 0xf8, 0xb5,
};
static const unsigned char kat568_nonce[] = {
    0xb6, 0x22, 0x1e, 0x28, 0x2e, 0x81, 0xf3, 0x10, 0xa6, 0xa2, 0xef, 0xba,
    0x73, 0xb0, 0x49, 0x0b,
};
static const unsigned char kat568_persstr[] = {0};
static const unsigned char kat568_addin0[] = {
    0xf2, 0x76, 0xb8, 0xae, 0x6e, 0xa4, 0xd9, 0x9b, 0x27, 0xee, 0x89, 0x19,
    0x35, 0x30, 0x0f, 0x30, 0x9a, 0x7e, 0x6f, 0x15, 0x4f, 0x5a, 0xc6, 0xee,
    0x27, 0x38, 0x83, 0xca, 0x53, 0x57, 0xba, 0x6f,
};
static const unsigned char kat568_addin1[] = {
    0x93, 0x26, 0x8a, 0x42, 0x42, 0x6c, 0xc3, 0x10, 0xee, 0x18, 0x76, 0x7d,
    0x96, 0x63, 0x26, 0x9f, 0x5e, 0x1c, 0x46, 0xb9, 0x6c, 0x6b, 0x17, 0xb2,
    0xf8, 0xbd, 0x6e, 0xce, 0xc3, 0x43, 0x5e, 0x42,
};
static const unsigned char kat568_retbits[] = {
    0x45, 0xe8, 0x5e, 0xe0, 0x30, 0x0a, 0x94, 0x62, 0x01, 0x6c, 0x4b, 0x7d,
    0x4b, 0x16, 0x19, 0xc7, 0x6d, 0x4a, 0xa9, 0x42, 0x2b, 0x5c, 0x45, 0xcf,
    0xac, 0xad, 0xae, 0x24, 0xb9, 0x34, 0xba, 0x19, 0x0f, 0x76, 0xa8, 0xa4,
    0x55, 0x1a, 0x23, 0xe8, 0xe1, 0x44, 0x73, 0xde, 0xd8, 0xf7, 0x9b, 0xdf,
    0x91, 0xb4, 0xfe, 0x0c, 0xe1, 0x63, 0x92, 0x8a, 0xf2, 0xc1, 0xea, 0xdf,
    0xc9, 0x6d, 0x30, 0x1c,
};
static const struct drbg_kat_no_reseed kat568_t = {
    12, kat568_entropyin, kat568_nonce, kat568_persstr,
    kat568_addin0, kat568_addin1, kat568_retbits
};
static const struct drbg_kat kat568 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat568_t
};

static const unsigned char kat569_entropyin[] = {
    0xdf, 0xf8, 0xc1, 0x4a, 0x61, 0x19, 0x28, 0x12, 0x76, 0xe7, 0x1c, 0x91,
    0xd6, 0x6d, 0x97, 0xe2, 0x9a, 0x14, 0x1e, 0x06, 0x53, 0x7e, 0xaf, 0x6c,
    0x2f, 0xeb, 0xd8, 0x23, 0x49, 0x3b, 0x7e, 0x3f,
};
static const unsigned char kat569_nonce[] = {
    0x1e, 0x98, 0x17, 0x4e, 0xdb, 0x22, 0x8d, 0x55, 0xcd, 0x4f, 0x58, 0x04,
    0x27, 0xac, 0xb3, 0x7a,
};
static const unsigned char kat569_persstr[] = {0};
static const unsigned char kat569_addin0[] = {
    0xf1, 0x7f, 0x9d, 0xdc, 0x60, 0xc6, 0x3f, 0xa8, 0x04, 0x36, 0x04, 0x1e,
    0x5a, 0x21, 0x19, 0x7e, 0x5e, 0x37, 0x8e, 0xcf, 0x70, 0xc7, 0xee, 0x39,
    0x2b, 0xd6, 0xf8, 0x92, 0x9d, 0xf1, 0xee, 0xef,
};
static const unsigned char kat569_addin1[] = {
    0x13, 0x90, 0xbb, 0x27, 0x69, 0x29, 0x27, 0x2a, 0x99, 0x42, 0xf2, 0x94,
    0xd5, 0xcb, 0x64, 0xd6, 0x32, 0x2c, 0x1c, 0xdc, 0x9a, 0x6f, 0x87, 0xf1,
    0xbd, 0x4e, 0x9a, 0xd7, 0xba, 0x6d, 0x82, 0x9c,
};
static const unsigned char kat569_retbits[] = {
    0xc1, 0xad, 0xc5, 0x14, 0x5d, 0xb9, 0xb5, 0xeb, 0x35, 0x92, 0x0a, 0x94,
    0xb7, 0x63, 0x2e, 0x50, 0x30, 0xe7, 0xfa, 0x38, 0x10, 0x30, 0x01, 0x17,
    0xa9, 0x36, 0xb8, 0x87, 0xb8, 0xbb, 0xfb, 0x0c, 0x51, 0x16, 0x9e, 0x70,
    0xf2, 0x6c, 0x0b, 0xf8, 0x79, 0x83, 0x85, 0xbc, 0xcd, 0xd7, 0xbe, 0x4f,
    0x7c, 0x53, 0xb3, 0xbb, 0xf9, 0x7a, 0x65, 0xb1, 0x6c, 0xcf, 0xfc, 0xb1,
    0xd5, 0x2e, 0xbd, 0x14,
};
static const struct drbg_kat_no_reseed kat569_t = {
    13, kat569_entropyin, kat569_nonce, kat569_persstr,
    kat569_addin0, kat569_addin1, kat569_retbits
};
static const struct drbg_kat kat569 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat569_t
};

static const unsigned char kat570_entropyin[] = {
    0xe8, 0xd1, 0xca, 0x5d, 0xba, 0xec, 0x67, 0x47, 0x22, 0x36, 0x9d, 0x32,
    0x21, 0xa3, 0xdc, 0xf2, 0x68, 0x1f, 0xb3, 0x70, 0x06, 0x1e, 0x48, 0x88,
    0x27, 0xa0, 0x07, 0x90, 0xab, 0xbb, 0x40, 0x80,
};
static const unsigned char kat570_nonce[] = {
    0xe5, 0xe6, 0x14, 0x4b, 0xda, 0x93, 0x75, 0x29, 0x89, 0xc5, 0x4a, 0x70,
    0x20, 0x81, 0xaf, 0x08,
};
static const unsigned char kat570_persstr[] = {0};
static const unsigned char kat570_addin0[] = {
    0x9e, 0xba, 0x53, 0xd0, 0xab, 0xbc, 0xdf, 0xb9, 0x84, 0x39, 0xce, 0xc3,
    0x2a, 0x14, 0xdc, 0x79, 0x24, 0x4a, 0xc3, 0x61, 0x6a, 0x71, 0xda, 0xfe,
    0xb7, 0x17, 0x5b, 0x71, 0xf9, 0x82, 0xe2, 0x14,
};
static const unsigned char kat570_addin1[] = {
    0x76, 0xe4, 0xae, 0xfb, 0xb6, 0x88, 0x96, 0x41, 0x1b, 0xd7, 0x93, 0x66,
    0x7e, 0x0e, 0x3b, 0xb3, 0xde, 0xd8, 0x37, 0xd3, 0xa2, 0xbb, 0x37, 0x0d,
    0x94, 0x4c, 0x20, 0xe5, 0xbf, 0x03, 0xe6, 0xd1,
};
static const unsigned char kat570_retbits[] = {
    0x6f, 0x6b, 0xed, 0x60, 0x5b, 0xc0, 0x41, 0x9e, 0x36, 0x7a, 0x42, 0x04,
    0x85, 0x85, 0xcb, 0x10, 0x54, 0x0d, 0x3a, 0x03, 0x5f, 0xc9, 0x90, 0x5f,
    0xa0, 0x04, 0x72, 0x9f, 0xef, 0xea, 0x46, 0x71, 0xef, 0x17, 0xaa, 0xc2,
    0xcc, 0x2c, 0x96, 0x4f, 0x35, 0xe7, 0x40, 0x98, 0x8e, 0x63, 0x6d, 0x8d,
    0x25, 0x11, 0x7c, 0x95, 0x3d, 0x0a, 0xf6, 0xa1, 0x99, 0x06, 0x7a, 0xd3,
    0x9a, 0xa6, 0x16, 0x63,
};
static const struct drbg_kat_no_reseed kat570_t = {
    14, kat570_entropyin, kat570_nonce, kat570_persstr,
    kat570_addin0, kat570_addin1, kat570_retbits
};
static const struct drbg_kat kat570 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat570_t
};

static const unsigned char kat571_entropyin[] = {
    0x87, 0x82, 0xd5, 0x16, 0xab, 0x2e, 0x07, 0x20, 0x81, 0x6d, 0x31, 0xe8,
    0x41, 0xc4, 0x97, 0x65, 0x83, 0xf5, 0xf2, 0x35, 0x6d, 0x4a, 0x6b, 0x75,
    0xba, 0xa0, 0xc8, 0x54, 0xd8, 0x1e, 0x87, 0xdf,
};
static const unsigned char kat571_nonce[] = {
    0xd3, 0xa0, 0xdf, 0x6e, 0x41, 0x0c, 0xba, 0x3a, 0xf8, 0x2b, 0x2e, 0x91,
    0x4e, 0x52, 0xb1, 0x9a,
};
static const unsigned char kat571_persstr[] = {
    0x94, 0x60, 0xe6, 0x67, 0x3c, 0x94, 0xac, 0x44, 0xf8, 0x12, 0x67, 0x3c,
    0x25, 0xb8, 0x90, 0x54, 0x56, 0xc3, 0x2f, 0xa7, 0xa8, 0x8d, 0x01, 0x9c,
    0x9b, 0x9a, 0xf0, 0xe9, 0xe6, 0xdf, 0xde, 0x32,
};
static const unsigned char kat571_addin0[] = {0};
static const unsigned char kat571_addin1[] = {0};
static const unsigned char kat571_retbits[] = {
    0x73, 0xbe, 0x5a, 0xca, 0x78, 0x6c, 0x4d, 0x20, 0x01, 0xf0, 0x26, 0xa4,
    0x8f, 0xc3, 0x2e, 0x0d, 0x5b, 0x9c, 0x43, 0xf5, 0x58, 0x15, 0x89, 0x80,
    0x9f, 0x10, 0x3c, 0xf9, 0x1f, 0xdc, 0x33, 0xaa, 0x00, 0x07, 0x03, 0xc5,
    0xb9, 0xa7, 0x39, 0x1c, 0x4c, 0x75, 0x12, 0x6b, 0xa0, 0x0f, 0x9f, 0x9c,
    0xf3, 0x68, 0xb0, 0xf9, 0x2a, 0x72, 0x90, 0x5e, 0xc1, 0x1f, 0x67, 0x02,
    0x44, 0xd0, 0x2e, 0x33,
};
static const struct drbg_kat_no_reseed kat571_t = {
    0, kat571_entropyin, kat571_nonce, kat571_persstr,
    kat571_addin0, kat571_addin1, kat571_retbits
};
static const struct drbg_kat kat571 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat571_t
};

static const unsigned char kat572_entropyin[] = {
    0xa3, 0xa0, 0x58, 0xec, 0x8f, 0x49, 0x63, 0xe3, 0xe4, 0xa5, 0xe7, 0xae,
    0xad, 0xea, 0xd4, 0x8e, 0x48, 0xa1, 0x30, 0xf0, 0x4a, 0xe6, 0x78, 0x5c,
    0x18, 0x4d, 0x76, 0xff, 0x8c, 0x78, 0x13, 0x4e,
};
static const unsigned char kat572_nonce[] = {
    0xca, 0x4f, 0xf0, 0xc8, 0xc0, 0x5d, 0xb6, 0xd7, 0x66, 0xf3, 0x56, 0x21,
    0x6c, 0x3b, 0x5f, 0xb4,
};
static const unsigned char kat572_persstr[] = {
    0xcf, 0x95, 0x33, 0x8c, 0xe6, 0x92, 0x72, 0x32, 0x4c, 0x75, 0x17, 0x59,
    0x56, 0x6e, 0x99, 0xeb, 0x9a, 0x2a, 0x61, 0x8c, 0xed, 0xee, 0xa9, 0x77,
    0xc3, 0x60, 0xa3, 0x5b, 0xe7, 0xdb, 0x80, 0x7c,
};
static const unsigned char kat572_addin0[] = {0};
static const unsigned char kat572_addin1[] = {0};
static const unsigned char kat572_retbits[] = {
    0xf5, 0x93, 0xfe, 0xcd, 0xec, 0xfd, 0x70, 0xd9, 0xf7, 0xcc, 0x09, 0x3b,
    0x4c, 0xf0, 0x50, 0x2f, 0x17, 0x8c, 0x99, 0x97, 0xce, 0x7f, 0x3b, 0x95,
    0xcb, 0xaf, 0xba, 0xf6, 0xe5, 0x75, 0x63, 0x7d, 0x34, 0x4e, 0x2c, 0x9b,
    0x7e, 0xbc, 0xb9, 0xed, 0x60, 0x48, 0x65, 0x06, 0x39, 0xea, 0x48, 0xd3,
    0x21, 0xc6, 0x26, 0x08, 0x6b, 0x28, 0x00, 0x2d, 0x86, 0x3c, 0xaf, 0xed,
    0xe0, 0x91, 0xe7, 0xe5,
};
static const struct drbg_kat_no_reseed kat572_t = {
    1, kat572_entropyin, kat572_nonce, kat572_persstr,
    kat572_addin0, kat572_addin1, kat572_retbits
};
static const struct drbg_kat kat572 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat572_t
};

static const unsigned char kat573_entropyin[] = {
    0x91, 0xf7, 0x4d, 0x6c, 0x79, 0x8f, 0x6e, 0x18, 0x42, 0xe3, 0x6a, 0xa6,
    0x10, 0x19, 0x68, 0x2e, 0x24, 0x6a, 0x2e, 0xec, 0x04, 0xaa, 0xc8, 0xf7,
    0xc5, 0xe8, 0x49, 0xdb, 0xd6, 0xfa, 0x67, 0x7d,
};
static const unsigned char kat573_nonce[] = {
    0x80, 0x07, 0x23, 0x00, 0x8b, 0x74, 0x43, 0x51, 0x97, 0x9a, 0xe8, 0x5d,
    0x92, 0xfd, 0x21, 0x7f,
};
static const unsigned char kat573_persstr[] = {
    0xc9, 0xb3, 0x8f, 0x9b, 0x98, 0xb7, 0xa0, 0x04, 0x3b, 0x13, 0xd1, 0x92,
    0x6c, 0x27, 0x26, 0x55, 0x21, 0xf0, 0x13, 0x16, 0xe8, 0xfe, 0x79, 0xd2,
    0xef, 0xa8, 0xb8, 0x17, 0xb2, 0x3a, 0xaf, 0xc5,
};
static const unsigned char kat573_addin0[] = {0};
static const unsigned char kat573_addin1[] = {0};
static const unsigned char kat573_retbits[] = {
    0x5a, 0x63, 0x77, 0x02, 0x30, 0xa4, 0xa0, 0x48, 0x64, 0x5c, 0xe1, 0xa7,
    0x5e, 0x50, 0xea, 0x79, 0x2c, 0x21, 0x96, 0x34, 0x56, 0x5f, 0x24, 0xec,
    0x52, 0xfa, 0xe6, 0x04, 0x65, 0x06, 0xc5, 0xb0, 0x52, 0x9a, 0x79, 0x8c,
    0x6b, 0xb7, 0x16, 0x19, 0xa2, 0x4b, 0xbd, 0x71, 0xf9, 0x03, 0x35, 0xe9,
    0x3c, 0x41, 0xde, 0x3f, 0xd0, 0xfd, 0x1f, 0x3e, 0xe3, 0x20, 0x4b, 0x9c,
    0x60, 0x64, 0xb7, 0x35,
};
static const struct drbg_kat_no_reseed kat573_t = {
    2, kat573_entropyin, kat573_nonce, kat573_persstr,
    kat573_addin0, kat573_addin1, kat573_retbits
};
static const struct drbg_kat kat573 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat573_t
};

static const unsigned char kat574_entropyin[] = {
    0xaa, 0x40, 0xfd, 0x98, 0xee, 0xa7, 0x52, 0xb7, 0x31, 0x54, 0x5a, 0x6b,
    0x93, 0x86, 0xb2, 0xff, 0x35, 0x6e, 0xf7, 0xd9, 0xce, 0x88, 0xda, 0xa2,
    0x21, 0x9a, 0x5c, 0x5f, 0xe5, 0x71, 0x09, 0xc8,
};
static const unsigned char kat574_nonce[] = {
    0x11, 0x0a, 0xcd, 0xc8, 0x6c, 0x06, 0xed, 0xcf, 0x8d, 0x61, 0x2a, 0x4f,
    0x2d, 0xf6, 0xea, 0x72,
};
static const unsigned char kat574_persstr[] = {
    0xf1, 0x7a, 0x8e, 0x0d, 0x46, 0x0e, 0x75, 0x87, 0x47, 0xc4, 0x61, 0x78,
    0x2a, 0xee, 0x6d, 0xad, 0x4a, 0x2e, 0xa8, 0xcc, 0x26, 0xc3, 0x3b, 0x34,
    0xe7, 0x97, 0xc9, 0xae, 0x8f, 0x8f, 0xc6, 0x32,
};
static const unsigned char kat574_addin0[] = {0};
static const unsigned char kat574_addin1[] = {0};
static const unsigned char kat574_retbits[] = {
    0x3d, 0xa6, 0x3f, 0x9f, 0xee, 0x35, 0x54, 0x41, 0xfa, 0xc4, 0x08, 0x4b,
    0xb9, 0x76, 0x68, 0x51, 0xcb, 0x6c, 0x60, 0xb7, 0xa9, 0x48, 0x42, 0xd2,
    0xc7, 0x62, 0x3b, 0x80, 0x7a, 0x46, 0x20, 0xcc, 0xae, 0x17, 0x48, 0x10,
    0xc2, 0x1d, 0x92, 0xd5, 0xf2, 0x67, 0x6f, 0x9e, 0x84, 0xa5, 0xc9, 0x8b,
    0x9a, 0x8a, 0x23, 0xad, 0xf7, 0x2d, 0xdf, 0xe3, 0x9f, 0xb7, 0x88, 0xf2,
    0x89, 0x21, 0x71, 0x87,
};
static const struct drbg_kat_no_reseed kat574_t = {
    3, kat574_entropyin, kat574_nonce, kat574_persstr,
    kat574_addin0, kat574_addin1, kat574_retbits
};
static const struct drbg_kat kat574 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat574_t
};

static const unsigned char kat575_entropyin[] = {
    0xbd, 0x5d, 0xaa, 0x18, 0xdf, 0xf5, 0x7e, 0x90, 0x76, 0x2d, 0xff, 0xd3,
    0x5a, 0x05, 0xa4, 0xa7, 0x39, 0xce, 0x7e, 0xbf, 0x08, 0x7f, 0x42, 0x93,
    0xf2, 0xc7, 0xa0, 0x31, 0xd1, 0x7d, 0xf9, 0xeb,
};
static const unsigned char kat575_nonce[] = {
    0xaf, 0x62, 0x92, 0x3e, 0xf2, 0x14, 0x46, 0x24, 0x18, 0x43, 0x9e, 0xc8,
    0xdc, 0x55, 0x3c, 0x9b,
};
static const unsigned char kat575_persstr[] = {
    0xfc, 0x9e, 0x13, 0x8b, 0xe9, 0x17, 0x0c, 0x83, 0x12, 0x28, 0x81, 0x91,
    0x03, 0x9b, 0x03, 0x3b, 0xf4, 0x1e, 0xf1, 0xd4, 0x7f, 0x4e, 0x64, 0x23,
    0x57, 0x86, 0x6b, 0x87, 0x5c, 0x7f, 0x18, 0x3e,
};
static const unsigned char kat575_addin0[] = {0};
static const unsigned char kat575_addin1[] = {0};
static const unsigned char kat575_retbits[] = {
    0xd4, 0x0e, 0xcb, 0x4e, 0x47, 0xe5, 0x54, 0x60, 0xc4, 0x00, 0x47, 0xd6,
    0x0f, 0x85, 0x28, 0x78, 0xb9, 0x15, 0x26, 0x8a, 0x4f, 0x13, 0x79, 0x6c,
    0xf5, 0xd9, 0xaa, 0x0d, 0x67, 0xf6, 0xda, 0x88, 0x09, 0x84, 0x74, 0x68,
    0xd7, 0xe0, 0x4c, 0x03, 0x9a, 0x9f, 0x9e, 0x3d, 0x9e, 0x5b, 0x4d, 0x53,
    0xce, 0x8f, 0x66, 0xfe, 0x7d, 0x88, 0xa4, 0x98, 0x3c, 0x51, 0x11, 0xce,
    0xf6, 0x03, 0x7b, 0x33,
};
static const struct drbg_kat_no_reseed kat575_t = {
    4, kat575_entropyin, kat575_nonce, kat575_persstr,
    kat575_addin0, kat575_addin1, kat575_retbits
};
static const struct drbg_kat kat575 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat575_t
};

static const unsigned char kat576_entropyin[] = {
    0x5a, 0x57, 0x10, 0x9b, 0xf7, 0xde, 0xe0, 0x09, 0x92, 0x26, 0x23, 0xcb,
    0x8d, 0xd7, 0xc6, 0x02, 0x9d, 0x14, 0xe6, 0xf3, 0x99, 0x7e, 0x9a, 0x54,
    0x05, 0x73, 0x9b, 0xd8, 0x50, 0x27, 0xd3, 0x1e,
};
static const unsigned char kat576_nonce[] = {
    0xa4, 0xc9, 0x51, 0x1f, 0x90, 0x66, 0x5b, 0x44, 0x21, 0x8d, 0x45, 0xc6,
    0x27, 0xd8, 0xbe, 0x7d,
};
static const unsigned char kat576_persstr[] = {
    0x2a, 0x6b, 0xe5, 0x24, 0xc3, 0xf7, 0xa5, 0x80, 0x15, 0x0f, 0x69, 0x9b,
    0xcc, 0x6b, 0xa1, 0x0b, 0x1e, 0x26, 0xcb, 0x85, 0x71, 0x26, 0x21, 0xce,
    0x00, 0xda, 0x3f, 0x36, 0x3c, 0x8b, 0x1c, 0x46,
};
static const unsigned char kat576_addin0[] = {0};
static const unsigned char kat576_addin1[] = {0};
static const unsigned char kat576_retbits[] = {
    0xaa, 0xf4, 0xc6, 0xe3, 0xad, 0xa5, 0x1a, 0x1a, 0xe6, 0x2c, 0x24, 0x38,
    0x1b, 0x21, 0xad, 0x4a, 0xed, 0x83, 0xd8, 0xae, 0x39, 0x45, 0xeb, 0x71,
    0x93, 0x8c, 0x1c, 0xe2, 0xd5, 0x86, 0x27, 0x11, 0x5e, 0x5e, 0xfc, 0x3c,
    0x58, 0xe8, 0x05, 0x6e, 0x7f, 0x71, 0x90, 0xae, 0x55, 0x0a, 0x8f, 0x1b,
    0x46, 0xd1, 0xb5, 0x8a, 0xe3, 0xea, 0x9a, 0xf0, 0xb7, 0xb1, 0x08, 0x1a,
    0xa7, 0xfd, 0x36, 0x30,
};
static const struct drbg_kat_no_reseed kat576_t = {
    5, kat576_entropyin, kat576_nonce, kat576_persstr,
    kat576_addin0, kat576_addin1, kat576_retbits
};
static const struct drbg_kat kat576 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat576_t
};

static const unsigned char kat577_entropyin[] = {
    0xf4, 0x3d, 0x16, 0x31, 0x39, 0x7b, 0xcf, 0xd6, 0xc6, 0xb3, 0x85, 0x10,
    0x6f, 0xef, 0x8e, 0x72, 0xf9, 0x72, 0x84, 0x50, 0x28, 0x62, 0xd6, 0x46,
    0x4e, 0x61, 0x0c, 0xed, 0xe7, 0xb9, 0xd8, 0xae,
};
static const unsigned char kat577_nonce[] = {
    0x74, 0x1c, 0xcd, 0x9a, 0xa4, 0x87, 0xfd, 0x77, 0x53, 0x78, 0xea, 0xf3,
    0x1e, 0xb8, 0x77, 0xd8,
};
static const unsigned char kat577_persstr[] = {
    0x96, 0x07, 0xf8, 0x4f, 0x0b, 0x22, 0x37, 0x62, 0x6d, 0xa9, 0x25, 0x0b,
    0xcb, 0xa6, 0x46, 0xa0, 0x6d, 0xd8, 0x00, 0x49, 0xf9, 0xa2, 0x8c, 0x4f,
    0x0b, 0x2b, 0x63, 0x1f, 0xfe, 0x73, 0x0d, 0x5c,
};
static const unsigned char kat577_addin0[] = {0};
static const unsigned char kat577_addin1[] = {0};
static const unsigned char kat577_retbits[] = {
    0x50, 0x97, 0xa9, 0xf2, 0x7e, 0xb3, 0xae, 0x42, 0x32, 0x81, 0xb9, 0x3b,
    0x77, 0x20, 0x82, 0x58, 0x15, 0x03, 0x42, 0xea, 0x42, 0x45, 0xdb, 0x7f,
    0x1c, 0x4e, 0x73, 0x47, 0xb6, 0xa3, 0x80, 0xc8, 0x1e, 0x27, 0xb0, 0xeb,
    0x00, 0x61, 0x0e, 0x84, 0x2e, 0x2a, 0xb5, 0x7d, 0xdc, 0x03, 0x45, 0x67,
    0x67, 0x19, 0xb5, 0x2a, 0xf6, 0x63, 0x0d, 0x52, 0x90, 0x97, 0x5e, 0x65,
    0x75, 0xa0, 0x17, 0xb3,
};
static const struct drbg_kat_no_reseed kat577_t = {
    6, kat577_entropyin, kat577_nonce, kat577_persstr,
    kat577_addin0, kat577_addin1, kat577_retbits
};
static const struct drbg_kat kat577 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat577_t
};

static const unsigned char kat578_entropyin[] = {
    0xb9, 0xf8, 0xf6, 0xf6, 0x73, 0x50, 0xb2, 0xb6, 0x74, 0xa9, 0x8e, 0xc3,
    0xcc, 0xf9, 0xf9, 0x4d, 0x5e, 0xb9, 0xed, 0x74, 0xc6, 0x74, 0xec, 0x30,
    0x5d, 0xd9, 0x8f, 0xc5, 0xd8, 0xc6, 0x4d, 0x93,
};
static const unsigned char kat578_nonce[] = {
    0xed, 0x55, 0xa0, 0xe6, 0x82, 0xba, 0xc6, 0xad, 0x5d, 0x6f, 0x86, 0xeb,
    0x31, 0xf3, 0xb1, 0x5e,
};
static const unsigned char kat578_persstr[] = {
    0x9c, 0x29, 0x9b, 0x3d, 0xc0, 0x78, 0x2f, 0x07, 0x46, 0xee, 0x9a, 0xa1,
    0x9c, 0x24, 0xfc, 0xee, 0xb7, 0xb3, 0x50, 0xdd, 0x3d, 0xe9, 0x72, 0x7a,
    0x19, 0x70, 0x8f, 0x41, 0xd9, 0x9c, 0x86, 0xaa,
};
static const unsigned char kat578_addin0[] = {0};
static const unsigned char kat578_addin1[] = {0};
static const unsigned char kat578_retbits[] = {
    0x2c, 0xe4, 0xaf, 0x0c, 0xe1, 0xd6, 0x5a, 0x83, 0x78, 0xd3, 0xb8, 0x96,
    0xad, 0x2e, 0x16, 0xb9, 0xe8, 0xbd, 0x2a, 0x92, 0xd5, 0x95, 0xc1, 0x07,
    0x15, 0x9c, 0x38, 0x6a, 0x59, 0x93, 0x70, 0x54, 0x63, 0x6b, 0x64, 0x25,
    0xec, 0x73, 0x1c, 0x1d, 0x83, 0x8c, 0x53, 0x01, 0x53, 0x08, 0x64, 0x68,
    0xe7, 0x52, 0xea, 0x34, 0xd2, 0xad, 0x64, 0xc0, 0x68, 0xf0, 0x01, 0x53,
    0x41, 0xae, 0xa6, 0xa1,
};
static const struct drbg_kat_no_reseed kat578_t = {
    7, kat578_entropyin, kat578_nonce, kat578_persstr,
    kat578_addin0, kat578_addin1, kat578_retbits
};
static const struct drbg_kat kat578 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat578_t
};

static const unsigned char kat579_entropyin[] = {
    0xc6, 0x99, 0x20, 0x58, 0x59, 0x90, 0x4c, 0x44, 0xa7, 0x82, 0x9c, 0x81,
    0xa5, 0x81, 0xd6, 0x36, 0xa9, 0x7c, 0x3f, 0x57, 0x21, 0x12, 0x57, 0x69,
    0xd8, 0x07, 0x0b, 0x63, 0x8f, 0xab, 0x07, 0xb1,
};
static const unsigned char kat579_nonce[] = {
    0x4d, 0xdb, 0xba, 0x6a, 0x5a, 0x13, 0x7e, 0x24, 0xce, 0x4b, 0x28, 0x3a,
    0x8f, 0xf5, 0x27, 0xe3,
};
static const unsigned char kat579_persstr[] = {
    0x22, 0xdc, 0x48, 0x6c, 0x4f, 0xe1, 0x9a, 0xf1, 0xf8, 0x20, 0x89, 0xba,
    0xb3, 0xd0, 0x1e, 0xf1, 0x60, 0xb6, 0xf5, 0xe5, 0x94, 0x83, 0x69, 0xc0,
    0xac, 0xac, 0xd6, 0xa5, 0xf4, 0x11, 0xf7, 0x37,
};
static const unsigned char kat579_addin0[] = {0};
static const unsigned char kat579_addin1[] = {0};
static const unsigned char kat579_retbits[] = {
    0xed, 0x8e, 0xab, 0x6c, 0x64, 0x0e, 0x6e, 0x06, 0x17, 0x25, 0xd7, 0x18,
    0xf7, 0x67, 0xca, 0x51, 0x08, 0x49, 0x54, 0x21, 0x37, 0xda, 0x84, 0x70,
    0x0b, 0xdc, 0xac, 0xad, 0xf1, 0xca, 0xe5, 0xdd, 0x03, 0xf6, 0xed, 0xba,
    0xc0, 0x05, 0x3f, 0x1b, 0x7f, 0xe2, 0xcf, 0xd5, 0x36, 0x3a, 0x2f, 0xb8,
    0x12, 0x9e, 0x14, 0x9a, 0xc7, 0x47, 0x64, 0x47, 0x0b, 0x86, 0x23, 0x47,
    0x4e, 0x8b, 0x20, 0x4d,
};
static const struct drbg_kat_no_reseed kat579_t = {
    8, kat579_entropyin, kat579_nonce, kat579_persstr,
    kat579_addin0, kat579_addin1, kat579_retbits
};
static const struct drbg_kat kat579 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat579_t
};

static const unsigned char kat580_entropyin[] = {
    0x16, 0x4a, 0xf0, 0xb9, 0x1b, 0xa3, 0x67, 0xe7, 0xd9, 0x49, 0xe0, 0x68,
    0x7a, 0x7c, 0xf1, 0x82, 0x59, 0x19, 0x74, 0x5a, 0x2f, 0xf4, 0x1a, 0x28,
    0xc0, 0x6e, 0x9d, 0x64, 0x7c, 0xea, 0x54, 0x5a,
};
static const unsigned char kat580_nonce[] = {
    0xa6, 0x10, 0xa3, 0xcf, 0xc5, 0x1c, 0xd6, 0xad, 0x86, 0x9f, 0xa8, 0x50,
    0x86, 0x12, 0x4a, 0xa7,
};
static const unsigned char kat580_persstr[] = {
    0x5d, 0x03, 0x58, 0x6d, 0xcd, 0xa1, 0xf0, 0x22, 0xde, 0xa6, 0x43, 0xa6,
    0x1f, 0xbc, 0x9e, 0x25, 0xc7, 0xf0, 0x7f, 0xc1, 0x5b, 0xa5, 0x54, 0xd5,
    0xad, 0xb8, 0xd4, 0x78, 0x5c, 0xa7, 0x6b, 0xc7,
};
static const unsigned char kat580_addin0[] = {0};
static const unsigned char kat580_addin1[] = {0};
static const unsigned char kat580_retbits[] = {
    0xa7, 0x1a, 0xa1, 0x59, 0xb4, 0x81, 0x35, 0x3e, 0x1e, 0xa8, 0xa2, 0xfb,
    0x9f, 0x0f, 0xa6, 0xfb, 0xca, 0xac, 0x1c, 0x24, 0x10, 0x47, 0x17, 0xf9,
    0x89, 0x45, 0x35, 0xf0, 0x87, 0xe2, 0x90, 0x79, 0xb1, 0xc3, 0x16, 0x8f,
    0xb7, 0x45, 0xe0, 0x50, 0x31, 0x01, 0xed, 0xa8, 0x48, 0x42, 0x98, 0x0f,
    0x91, 0x91, 0x72, 0x7e, 0xa4, 0x7f, 0xfb, 0x57, 0x0c, 0xfb, 0xaa, 0xed,
    0xff, 0x05, 0x5d, 0x9c,
};
static const struct drbg_kat_no_reseed kat580_t = {
    9, kat580_entropyin, kat580_nonce, kat580_persstr,
    kat580_addin0, kat580_addin1, kat580_retbits
};
static const struct drbg_kat kat580 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat580_t
};

static const unsigned char kat581_entropyin[] = {
    0x6a, 0x59, 0x15, 0x57, 0xdf, 0x6a, 0xfe, 0x71, 0xca, 0xd5, 0xac, 0xff,
    0xbb, 0xf7, 0x58, 0xf6, 0x82, 0x9e, 0xa8, 0x87, 0x55, 0x9b, 0xc1, 0xc3,
    0xab, 0x6f, 0x1d, 0xdb, 0xef, 0x92, 0x8b, 0x0f,
};
static const unsigned char kat581_nonce[] = {
    0xc8, 0x72, 0xc4, 0xf0, 0xd9, 0xaf, 0xef, 0x9b, 0xe4, 0x08, 0xe0, 0xac,
    0x48, 0x87, 0x8b, 0xae,
};
static const unsigned char kat581_persstr[] = {
    0xdf, 0xa1, 0xc2, 0xdb, 0x43, 0xf6, 0x12, 0x9b, 0x31, 0xfb, 0x4b, 0x32,
    0x66, 0xb1, 0x7d, 0x97, 0x2b, 0xb4, 0x67, 0x95, 0x72, 0x61, 0xdf, 0x3a,
    0x67, 0x67, 0x87, 0x94, 0x49, 0x76, 0x01, 0xb6,
};
static const unsigned char kat581_addin0[] = {0};
static const unsigned char kat581_addin1[] = {0};
static const unsigned char kat581_retbits[] = {
    0x44, 0xf4, 0xad, 0x62, 0x71, 0x0d, 0x87, 0xa1, 0x09, 0x91, 0x09, 0x08,
    0xa4, 0xdd, 0x1a, 0x57, 0x91, 0x51, 0x65, 0x4b, 0x7c, 0x44, 0x06, 0x86,
    0xf9, 0x03, 0xb2, 0x13, 0x28, 0x5e, 0xa8, 0x86, 0xb7, 0x26, 0x05, 0xd8,
    0xe6, 0xaf, 0x0c, 0x2c, 0xb9, 0x28, 0x6f, 0x46, 0xb8, 0x7d, 0x6c, 0xdf,
    0xe1, 0x70, 0x24, 0x81, 0x24, 0x8a, 0x81, 0x6e, 0x88, 0x7d, 0x76, 0x68,
    0x58, 0xb2, 0x21, 0xf4,
};
static const struct drbg_kat_no_reseed kat581_t = {
    10, kat581_entropyin, kat581_nonce, kat581_persstr,
    kat581_addin0, kat581_addin1, kat581_retbits
};
static const struct drbg_kat kat581 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat581_t
};

static const unsigned char kat582_entropyin[] = {
    0xe2, 0xbc, 0x6e, 0x86, 0x5a, 0xbe, 0xb1, 0xbb, 0xf1, 0x3e, 0xbe, 0xd7,
    0xe1, 0x23, 0xff, 0xa3, 0x26, 0x87, 0x62, 0x5c, 0x67, 0xef, 0x56, 0x1d,
    0xe9, 0xc2, 0xfc, 0x6f, 0x14, 0xef, 0xdd, 0x77,
};
static const unsigned char kat582_nonce[] = {
    0x8e, 0x67, 0x5b, 0xf0, 0xac, 0x4b, 0x6b, 0x1f, 0xd4, 0x3a, 0x5b, 0xa3,
    0xd5, 0x5e, 0xb5, 0xfe,
};
static const unsigned char kat582_persstr[] = {
    0x2a, 0xd7, 0x9b, 0x01, 0xb4, 0x77, 0xc2, 0x64, 0x6b, 0xe5, 0x76, 0x12,
    0x72, 0x33, 0x0d, 0x98, 0x40, 0x5e, 0xdd, 0x77, 0x58, 0x91, 0xf6, 0x19,
    0xc9, 0x66, 0x67, 0x91, 0xdd, 0x2a, 0x9b, 0x68,
};
static const unsigned char kat582_addin0[] = {0};
static const unsigned char kat582_addin1[] = {0};
static const unsigned char kat582_retbits[] = {
    0xf4, 0x81, 0xf0, 0xf8, 0x1f, 0x46, 0xfe, 0xef, 0x26, 0x34, 0x10, 0xac,
    0x39, 0xef, 0x51, 0xf1, 0xfa, 0xaa, 0xcf, 0x22, 0x54, 0x1d, 0xe8, 0xe4,
    0xf9, 0x14, 0x52, 0xb6, 0x90, 0xbb, 0x5c, 0x6a, 0x21, 0x1d, 0xc8, 0x75,
    0x1d, 0xcf, 0x8b, 0xdb, 0xf4, 0xdc, 0xd3, 0x55, 0x48, 0x94, 0xbb, 0x66,
    0x44, 0xfb, 0x79, 0x95, 0xb4, 0x0c, 0x04, 0xea, 0x4c, 0x3a, 0x97, 0x9e,
    0x97, 0xb5, 0xbc, 0xfb,
};
static const struct drbg_kat_no_reseed kat582_t = {
    11, kat582_entropyin, kat582_nonce, kat582_persstr,
    kat582_addin0, kat582_addin1, kat582_retbits
};
static const struct drbg_kat kat582 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat582_t
};

static const unsigned char kat583_entropyin[] = {
    0xff, 0xed, 0x52, 0x4b, 0x1d, 0x1f, 0x1c, 0x5c, 0xee, 0x98, 0x76, 0x28,
    0x58, 0x38, 0x7d, 0x1a, 0x40, 0x32, 0x45, 0xeb, 0xfa, 0xfe, 0xc0, 0xc0,
    0xa4, 0x0a, 0xd8, 0x15, 0xdb, 0x9c, 0x83, 0xf2,
};
static const unsigned char kat583_nonce[] = {
    0x95, 0xf9, 0xd8, 0x65, 0x52, 0x11, 0xa0, 0xf8, 0x0e, 0xf8, 0xe3, 0x76,
    0xab, 0xd8, 0xf1, 0x71,
};
static const unsigned char kat583_persstr[] = {
    0x38, 0xb2, 0xef, 0x1f, 0xcf, 0x66, 0xab, 0x0a, 0xd6, 0xf1, 0x4e, 0xf6,
    0x8b, 0x7d, 0xb3, 0x6a, 0x73, 0x11, 0xff, 0x99, 0x64, 0xff, 0x4b, 0xd5,
    0x01, 0xc4, 0x47, 0xfd, 0x04, 0x16, 0x73, 0x40,
};
static const unsigned char kat583_addin0[] = {0};
static const unsigned char kat583_addin1[] = {0};
static const unsigned char kat583_retbits[] = {
    0x2e, 0xd8, 0x10, 0x83, 0x0b, 0x5c, 0x7b, 0xb2, 0x87, 0xee, 0xa8, 0x5c,
    0xd3, 0x2e, 0xde, 0x0f, 0xe0, 0x70, 0x04, 0x9a, 0x89, 0x8e, 0x0b, 0xbd,
    0x7a, 0x69, 0x49, 0xef, 0xe3, 0xbd, 0x3c, 0x19, 0x07, 0x5a, 0xcf, 0x87,
    0xff, 0x27, 0x71, 0x44, 0xab, 0xc2, 0xe2, 0x3c, 0xeb, 0x41, 0x6b, 0x70,
    0x5b, 0x7a, 0x2b, 0xe2, 0xfb, 0x0a, 0x0d, 0xc9, 0x16, 0x47, 0x86, 0xf3,
    0x52, 0x7c, 0xfe, 0x14,
};
static const struct drbg_kat_no_reseed kat583_t = {
    12, kat583_entropyin, kat583_nonce, kat583_persstr,
    kat583_addin0, kat583_addin1, kat583_retbits
};
static const struct drbg_kat kat583 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat583_t
};

static const unsigned char kat584_entropyin[] = {
    0x2a, 0x92, 0x90, 0x3d, 0x4d, 0x3a, 0xba, 0x09, 0x16, 0xcb, 0xc0, 0x72,
    0xf9, 0x8a, 0x17, 0x05, 0x44, 0x6a, 0x12, 0x6d, 0x3e, 0x68, 0x4d, 0x0d,
    0x27, 0x1b, 0x78, 0xa4, 0x6a, 0xe8, 0x5c, 0x6e,
};
static const unsigned char kat584_nonce[] = {
    0x2c, 0x94, 0x36, 0x1b, 0xdb, 0x04, 0x08, 0x75, 0xa6, 0x4e, 0x1f, 0x21,
    0xb9, 0xca, 0x85, 0x50,
};
static const unsigned char kat584_persstr[] = {
    0x35, 0x3f, 0x18, 0xf9, 0xc3, 0x31, 0xa1, 0xf7, 0x7a, 0x1b, 0x0a, 0xa5,
    0xf9, 0x70, 0x76, 0x2e, 0xba, 0xcb, 0xac, 0xb4, 0x27, 0x3a, 0x94, 0x11,
    0x14, 0xa3, 0xc7, 0x83, 0xb6, 0xe4, 0x08, 0x8e,
};
static const unsigned char kat584_addin0[] = {0};
static const unsigned char kat584_addin1[] = {0};
static const unsigned char kat584_retbits[] = {
    0xf9, 0xe3, 0xf3, 0x31, 0x77, 0x25, 0xf9, 0x6b, 0x13, 0x7b, 0xaa, 0x2e,
    0xe2, 0x16, 0xd7, 0x0c, 0xc4, 0xb4, 0x5c, 0x38, 0x05, 0x20, 0xf8, 0x1d,
    0x6f, 0xde, 0x13, 0x68, 0xef, 0x70, 0x10, 0x87, 0xa2, 0x90, 0x98, 0x93,
    0x3a, 0x38, 0xee, 0x75, 0x7b, 0x9f, 0x70, 0xb1, 0x82, 0xe7, 0x06, 0x3e,
    0x36, 0x30, 0xa4, 0x3b, 0xdc, 0x1f, 0x21, 0xc3, 0x0c, 0x4a, 0x47, 0xbf,
    0x7f, 0xbb, 0x98, 0xe5,
};
static const struct drbg_kat_no_reseed kat584_t = {
    13, kat584_entropyin, kat584_nonce, kat584_persstr,
    kat584_addin0, kat584_addin1, kat584_retbits
};
static const struct drbg_kat kat584 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat584_t
};

static const unsigned char kat585_entropyin[] = {
    0xc2, 0x75, 0x29, 0xd8, 0x65, 0x0a, 0x49, 0x8f, 0xe1, 0xaa, 0xcd, 0xf1,
    0x2d, 0x8a, 0x56, 0xc7, 0x3e, 0xd1, 0xc4, 0xb5, 0x66, 0xfc, 0xfc, 0x57,
    0xb4, 0x3e, 0xeb, 0x2f, 0x8c, 0x17, 0x90, 0xad,
};
static const unsigned char kat585_nonce[] = {
    0x09, 0x45, 0x15, 0x3f, 0xe4, 0x78, 0x9c, 0xd0, 0xa7, 0x45, 0x10, 0xdf,
    0xcd, 0xa5, 0xde, 0x71,
};
static const unsigned char kat585_persstr[] = {
    0xcd, 0x98, 0xbf, 0x62, 0x9d, 0xa5, 0xde, 0xc8, 0x6d, 0x82, 0x3b, 0xbd,
    0x72, 0x0e, 0x27, 0x3f, 0xa0, 0x9b, 0x75, 0xa3, 0x04, 0xa2, 0xc3, 0x13,
    0x9c, 0x38, 0x38, 0x6b, 0xb4, 0x46, 0x85, 0x53,
};
static const unsigned char kat585_addin0[] = {0};
static const unsigned char kat585_addin1[] = {0};
static const unsigned char kat585_retbits[] = {
    0xba, 0x34, 0xd4, 0x6c, 0x29, 0xaf, 0xc9, 0x1e, 0xc3, 0x4c, 0xad, 0x84,
    0x85, 0xb5, 0xdd, 0x6e, 0x2f, 0xb5, 0x80, 0xa2, 0x17, 0x85, 0x9f, 0x70,
    0x49, 0xa6, 0xb2, 0x3e, 0xb9, 0x3e, 0x66, 0xe4, 0x29, 0xdd, 0xff, 0x22,
    0xcc, 0x64, 0x71, 0x59, 0xef, 0x11, 0x95, 0xbb, 0x67, 0xc4, 0x0c, 0xb3,
    0x38, 0x83, 0xeb, 0xd8, 0xa6, 0xb9, 0xf7, 0x1a, 0xb0, 0xac, 0xb8, 0x9d,
    0xe7, 0x74, 0xa5, 0x93,
};
static const struct drbg_kat_no_reseed kat585_t = {
    14, kat585_entropyin, kat585_nonce, kat585_persstr,
    kat585_addin0, kat585_addin1, kat585_retbits
};
static const struct drbg_kat kat585 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat585_t
};

static const unsigned char kat586_entropyin[] = {
    0xed, 0x12, 0xdf, 0x77, 0x81, 0x55, 0x85, 0xfc, 0x9a, 0xe7, 0x39, 0x66,
    0x20, 0xee, 0xe4, 0xae, 0x68, 0xcc, 0x82, 0xa8, 0x2e, 0xc3, 0x0a, 0x79,
    0x29, 0x01, 0xe2, 0x85, 0x8a, 0x59, 0x70, 0x5d,
};
static const unsigned char kat586_nonce[] = {
    0x23, 0x2a, 0x3d, 0xb9, 0x70, 0xb5, 0xcf, 0x1f, 0x31, 0xa5, 0xe0, 0x9f,
    0x02, 0xc0, 0xa9, 0x7e,
};
static const unsigned char kat586_persstr[] = {
    0x2f, 0x92, 0x94, 0xdb, 0x48, 0x53, 0x05, 0xd4, 0x88, 0x63, 0xb6, 0xf5,
    0x37, 0xc3, 0xfa, 0xed, 0x90, 0x3b, 0x9f, 0xeb, 0x94, 0xbb, 0x84, 0x8d,
    0x00, 0xdc, 0x58, 0xe7, 0x7d, 0x8f, 0x47, 0xc0,
};
static const unsigned char kat586_addin0[] = {
    0xc9, 0x96, 0x9a, 0x56, 0x33, 0x74, 0x48, 0x0b, 0xc0, 0x8f, 0x61, 0xd4,
    0xb4, 0x6e, 0x58, 0x7a, 0xfc, 0x55, 0x12, 0x6d, 0x38, 0x09, 0xe6, 0x03,
    0xe2, 0x0e, 0x44, 0xa0, 0x76, 0x36, 0xc6, 0x78,
};
static const unsigned char kat586_addin1[] = {
    0x03, 0xcf, 0xba, 0xa7, 0x39, 0xb3, 0x3c, 0x1b, 0xc6, 0x0a, 0xbb, 0x1c,
    0x73, 0x0e, 0x15, 0x5f, 0xae, 0x07, 0x83, 0x70, 0x54, 0xb0, 0x8e, 0xe8,
    0x48, 0xc4, 0x58, 0xc8, 0x85, 0x69, 0xff, 0xc1,
};
static const unsigned char kat586_retbits[] = {
    0x78, 0xbd, 0x67, 0xeb, 0x4e, 0x66, 0x0a, 0x4f, 0xe3, 0x47, 0x4e, 0xc1,
    0xe9, 0x5b, 0x1f, 0xbd, 0xc1, 0xe4, 0xdc, 0x68, 0x67, 0x18, 0x4e, 0xe4,
    0xea, 0x9e, 0x15, 0x68, 0x14, 0xc5, 0x84, 0x9c, 0x3c, 0x12, 0xd7, 0xba,
    0x06, 0xcc, 0xed, 0x8c, 0x87, 0x27, 0x12, 0xc2, 0xb9, 0x6e, 0x74, 0x68,
    0x53, 0x6e, 0x11, 0xa2, 0x0e, 0x93, 0xe5, 0x3b, 0x8c, 0x77, 0x8e, 0x9c,
    0x06, 0x34, 0xc6, 0xcb,
};
static const struct drbg_kat_no_reseed kat586_t = {
    0, kat586_entropyin, kat586_nonce, kat586_persstr,
    kat586_addin0, kat586_addin1, kat586_retbits
};
static const struct drbg_kat kat586 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat586_t
};

static const unsigned char kat587_entropyin[] = {
    0x2c, 0x8c, 0x91, 0x62, 0xa1, 0xdd, 0x63, 0xc1, 0xf2, 0x89, 0x47, 0x14,
    0xd8, 0x91, 0x58, 0x03, 0x0a, 0x5f, 0x67, 0x7a, 0xa0, 0x14, 0xd7, 0x8b,
    0xcd, 0x55, 0x8d, 0x8f, 0xfb, 0xa2, 0xab, 0x25,
};
static const unsigned char kat587_nonce[] = {
    0x42, 0x06, 0xb6, 0xc3, 0xc1, 0xf5, 0x43, 0xb1, 0x60, 0x8f, 0xb9, 0xcd,
    0xb6, 0x2f, 0xc2, 0xc7,
};
static const unsigned char kat587_persstr[] = {
    0xaa, 0xdd, 0x7d, 0x9f, 0x9c, 0xee, 0x1f, 0x93, 0xf4, 0x3a, 0xff, 0x31,
    0x32, 0x83, 0x77, 0x58, 0xe8, 0x89, 0x55, 0x35, 0x0f, 0x6d, 0xee, 0xb7,
    0x7b, 0xb4, 0xf8, 0x5c, 0xc0, 0x41, 0x04, 0x54,
};
static const unsigned char kat587_addin0[] = {
    0xe7, 0x49, 0xfb, 0x5d, 0x67, 0xae, 0x61, 0x77, 0x04, 0xff, 0xfe, 0xbb,
    0xde, 0xb9, 0x98, 0xb2, 0x69, 0x2d, 0xb7, 0x2a, 0xf8, 0xac, 0x21, 0x7f,
    0x7b, 0xc5, 0x41, 0x6f, 0x93, 0xa7, 0x7a, 0x8f,
};
static const unsigned char kat587_addin1[] = {
    0x46, 0xa5, 0x13, 0x49, 0xdb, 0x45, 0x45, 0x6d, 0xb9, 0x4e, 0xd1, 0x25,
    0x46, 0xea, 0x6a, 0x62, 0x14, 0x89, 0xac, 0xfb, 0x40, 0xb0, 0xfa, 0x31,
    0x6a, 0x3c, 0x8f, 0x5f, 0x48, 0x0a, 0x00, 0x88,
};
static const unsigned char kat587_retbits[] = {
    0xee, 0x1f, 0x95, 0xda, 0x9b, 0x2d, 0x79, 0xf0, 0xcb, 0xf8, 0x33, 0x5e,
    0xfc, 0xc6, 0x91, 0x2f, 0x16, 0x39, 0x46, 0xe4, 0x45, 0x6d, 0x32, 0x84,
    0xb9, 0x18, 0x57, 0x9b, 0x50, 0xd6, 0x88, 0x1d, 0xb4, 0xcc, 0x08, 0x6d,
    0x3d, 0x21, 0x2a, 0xf2, 0xf3, 0x42, 0xb4, 0xbf, 0x46, 0x57, 0x37, 0x0b,
    0x02, 0x5c, 0xd4, 0xad, 0x2c, 0x1e, 0xef, 0xf3, 0xcf, 0x60, 0x70, 0xdb,
    0xdd, 0x50, 0x78, 0x61,
};
static const struct drbg_kat_no_reseed kat587_t = {
    1, kat587_entropyin, kat587_nonce, kat587_persstr,
    kat587_addin0, kat587_addin1, kat587_retbits
};
static const struct drbg_kat kat587 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat587_t
};

static const unsigned char kat588_entropyin[] = {
    0x91, 0x97, 0x3e, 0xa6, 0x17, 0xf9, 0x5b, 0x5c, 0xd9, 0x4c, 0xbb, 0xc2,
    0xbf, 0x02, 0x03, 0x81, 0x51, 0x85, 0x73, 0x63, 0xd8, 0x50, 0x25, 0x6b,
    0xdf, 0x9f, 0x78, 0xbd, 0x22, 0x34, 0x0b, 0x49,
};
static const unsigned char kat588_nonce[] = {
    0x9a, 0xb4, 0x15, 0x60, 0xf8, 0x54, 0x52, 0x07, 0x00, 0x52, 0x68, 0x1a,
    0x25, 0xd9, 0xf0, 0xf8,
};
static const unsigned char kat588_persstr[] = {
    0x69, 0x16, 0x3f, 0xff, 0x9b, 0xab, 0x52, 0x97, 0xe0, 0x9f, 0xc9, 0x35,
    0x1f, 0x96, 0xa5, 0xbe, 0xe9, 0x97, 0x30, 0x86, 0x1f, 0xde, 0xf1, 0x0a,
    0xdf, 0xd4, 0xac, 0x38, 0xa0, 0x80, 0x0b, 0x3e,
};
static const unsigned char kat588_addin0[] = {
    0xc4, 0x2c, 0x50, 0x3d, 0x41, 0x7d, 0x87, 0x67, 0x9f, 0x53, 0x0f, 0x4b,
    0xa0, 0x5c, 0xbc, 0xe3, 0x40, 0x48, 0x74, 0xd4, 0x61, 0x16, 0x3f, 0x5c,
    0x57, 0x0a, 0x3c, 0xfa, 0xd7, 0x5a, 0xbe, 0xed,
};
static const unsigned char kat588_addin1[] = {
    0xba, 0xc9, 0xd9, 0xa1, 0x6d, 0x9f, 0x1c, 0xe7, 0xa4, 0xeb, 0xb7, 0xc3,
    0x78, 0x12, 0x43, 0x39, 0x4e, 0x22, 0x80, 0xa5, 0x2d, 0xe6, 0x41, 0x1d,
    0xb5, 0x2e, 0xd8, 0xc3, 0x09, 0x01, 0x3e, 0xcf,
};
static const unsigned char kat588_retbits[] = {
    0x23, 0x64, 0x0c, 0xf3, 0x57, 0x0c, 0x88, 0xf3, 0x5c, 0x44, 0x0f, 0x5a,
    0x1d, 0x61, 0x95, 0xde, 0x06, 0xd0, 0x86, 0xb9, 0x11, 0x8d, 0xd6, 0xda,
    0xf8, 0x97, 0x48, 0x65, 0xd5, 0x2e, 0x11, 0x76, 0x6e, 0xaa, 0x8b, 0x14,
    0x62, 0xc7, 0x7b, 0x54, 0x2d, 0x6a, 0x53, 0x91, 0xdf, 0x63, 0x68, 0xca,
    0x68, 0x2f, 0x63, 0x98, 0xaf, 0x51, 0x56, 0xee, 0x7c, 0xd8, 0xce, 0x24,
    0x50, 0x25, 0x8f, 0xcb,
};
static const struct drbg_kat_no_reseed kat588_t = {
    2, kat588_entropyin, kat588_nonce, kat588_persstr,
    kat588_addin0, kat588_addin1, kat588_retbits
};
static const struct drbg_kat kat588 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat588_t
};

static const unsigned char kat589_entropyin[] = {
    0xef, 0xfa, 0xd8, 0x21, 0x04, 0x04, 0x17, 0x86, 0x15, 0x9b, 0xe5, 0xa1,
    0xeb, 0xc0, 0xbe, 0x77, 0x68, 0x8e, 0x6f, 0x1c, 0x07, 0x5d, 0xc2, 0x0a,
    0x83, 0x2e, 0x3f, 0xfa, 0xfb, 0x11, 0xc5, 0x4c,
};
static const unsigned char kat589_nonce[] = {
    0xe1, 0x3b, 0xad, 0xfd, 0xe3, 0x6a, 0x8c, 0x10, 0x7e, 0x4f, 0x3c, 0xea,
    0x69, 0x3d, 0xa3, 0x6d,
};
static const unsigned char kat589_persstr[] = {
    0x01, 0xea, 0x78, 0xe5, 0xbe, 0x0b, 0xf9, 0x48, 0x96, 0x2d, 0x7c, 0x33,
    0x4a, 0xe2, 0x37, 0xc5, 0x60, 0xbb, 0x49, 0xe6, 0x63, 0x59, 0x58, 0x03,
    0x59, 0x36, 0xc4, 0x8d, 0x90, 0x82, 0xde, 0x39,
};
static const unsigned char kat589_addin0[] = {
    0x72, 0x63, 0x49, 0xa1, 0xe5, 0x2b, 0xe8, 0xb8, 0x4b, 0x15, 0x75, 0x82,
    0x7e, 0xae, 0x96, 0xef, 0xc1, 0xc1, 0x15, 0x51, 0xb3, 0x5c, 0x68, 0xa0,
    0x4e, 0x89, 0xbd, 0x12, 0x55, 0x89, 0x82, 0x7a,
};
static const unsigned char kat589_addin1[] = {
    0xe9, 0x43, 0x56, 0x9b, 0xac, 0x0d, 0x0e, 0x6b, 0x41, 0x00, 0xba, 0x89,
    0xc6, 0xf3, 0x2d, 0xd7, 0xed, 0x64, 0x57, 0x33, 0x57, 0xc5, 0x77, 0x83,
    0xf3, 0xd0, 0xe0, 0xf6, 0x9e, 0xaf, 0xdd, 0x1f,
};
static const unsigned char kat589_retbits[] = {
    0xba, 0x58, 0x11, 0x9f, 0x27, 0x47, 0xdd, 0xe7, 0xfc, 0x00, 0x0e, 0x22,
    0x09, 0x0a, 0xe3, 0x25, 0x6a, 0xec, 0xd2, 0x64, 0xcd, 0xcf, 0x77, 0xd3,
    0xc5, 0x1f, 0x32, 0xfa, 0x96, 0xb2, 0x44, 0xfe, 0xed, 0x32, 0x05, 0xfa,
    0xd9, 0x0d, 0x98, 0xa4, 0x27, 0x27, 0xb5, 0x51, 0xcd, 0xdc, 0xb3, 0xfa,
    0x28, 0x87, 0x0c, 0x08, 0xf0, 0x2d, 0x86, 0x5b, 0x4c, 0x39, 0xcd, 0x1e,
    0x36, 0x81, 0x8a, 0x23,
};
static const struct drbg_kat_no_reseed kat589_t = {
    3, kat589_entropyin, kat589_nonce, kat589_persstr,
    kat589_addin0, kat589_addin1, kat589_retbits
};
static const struct drbg_kat kat589 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat589_t
};

static const unsigned char kat590_entropyin[] = {
    0x85, 0x24, 0x10, 0x08, 0x36, 0x34, 0x04, 0x7a, 0x3f, 0x9a, 0xa8, 0xc7,
    0x72, 0x51, 0x7c, 0x6c, 0x91, 0x2f, 0xa9, 0x5b, 0xab, 0x1d, 0xa0, 0x9a,
    0x3a, 0x70, 0x53, 0xc4, 0xbb, 0xaf, 0x75, 0x01,
};
static const unsigned char kat590_nonce[] = {
    0x49, 0xcc, 0xe5, 0xd8, 0x7b, 0xc9, 0x05, 0xe9, 0x8b, 0xc5, 0xbd, 0x3f,
    0x2f, 0x6b, 0xaa, 0x0e,
};
static const unsigned char kat590_persstr[] = {
    0x79, 0xe7, 0x15, 0x35, 0xed, 0x28, 0xe0, 0x80, 0xa2, 0xa0, 0x24, 0xbe,
    0xf6, 0x02, 0x04, 0x37, 0x74, 0x19, 0xc1, 0x3b, 0x0f, 0x57, 0x37, 0x39,
    0xb2, 0xe3, 0x8a, 0xd2, 0xd9, 0x4e, 0x98, 0x01,
};
static const unsigned char kat590_addin0[] = {
    0x7c, 0x8d, 0xd0, 0xaa, 0xc5, 0x36, 0x11, 0x4f, 0x88, 0xe6, 0xdb, 0xa6,
    0x8c, 0x73, 0xda, 0x37, 0xb7, 0x70, 0x06, 0xc5, 0xa4, 0x4e, 0x78, 0x6e,
    0x5f, 0x62, 0xae, 0xa3, 0x65, 0x58, 0xf9, 0x45,
};
static const unsigned char kat590_addin1[] = {
    0x05, 0xac, 0xd9, 0x92, 0x1c, 0x8d, 0xec, 0xd6, 0x0b, 0xc9, 0xcd, 0x12,
    0x49, 0x25, 0xc8, 0xb5, 0x1b, 0xac, 0x8a, 0x63, 0x29, 0xe4, 0x63, 0x2d,
    0xbf, 0x07, 0x65, 0x32, 0xeb, 0xff, 0x92, 0x96,
};
static const unsigned char kat590_retbits[] = {
    0x90, 0xb4, 0x1b, 0x64, 0x4a, 0xd0, 0x53, 0xd3, 0x6c, 0xff, 0x4c, 0xeb,
    0x3d, 0x67, 0xc0, 0xf0, 0x64, 0xec, 0x43, 0x87, 0x42, 0xd7, 0x14, 0xcd,
    0xdd, 0x62, 0x4f, 0xed, 0x5d, 0xef, 0xce, 0x1c, 0xea, 0xca, 0xb3, 0x25,
    0xc1, 0x40, 0x01, 0x73, 0x16, 0xf2, 0x9a, 0xef, 0xf2, 0x32, 0x24, 0x4c,
    0x5d, 0x0a, 0x02, 0x04, 0xee, 0xd8, 0xcd, 0xd7, 0xda, 0x34, 0x6d, 0xb2,
    0x9d, 0xe7, 0xaa, 0x8e,
};
static const struct drbg_kat_no_reseed kat590_t = {
    4, kat590_entropyin, kat590_nonce, kat590_persstr,
    kat590_addin0, kat590_addin1, kat590_retbits
};
static const struct drbg_kat kat590 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat590_t
};

static const unsigned char kat591_entropyin[] = {
    0xbb, 0x56, 0x82, 0x66, 0x31, 0x8d, 0x93, 0x65, 0x09, 0x26, 0x57, 0xec,
    0xfa, 0xf0, 0x5a, 0xc7, 0x0c, 0x8d, 0xfd, 0x79, 0x7a, 0x47, 0xc7, 0xe8,
    0x5b, 0x51, 0xcb, 0x61, 0x26, 0xa0, 0xe6, 0x93,
};
static const unsigned char kat591_nonce[] = {
    0xdc, 0xd3, 0x5d, 0x62, 0x80, 0xf4, 0x0b, 0xeb, 0xc8, 0x21, 0x25, 0xea,
    0x28, 0xbe, 0x7a, 0x80,
};
static const unsigned char kat591_persstr[] = {
    0x7b, 0x25, 0x4b, 0xba, 0x67, 0x74, 0xfc, 0x1f, 0x60, 0x1c, 0x0f, 0x2f,
    0x9f, 0x90, 0x02, 0xa2, 0x90, 0x26, 0x9c, 0xb6, 0x56, 0x2d, 0x15, 0x0d,
    0x4b, 0x3a, 0x32, 0x11, 0x5c, 0xe4, 0x31, 0x3d,
};
static const unsigned char kat591_addin0[] = {
    0x97, 0x14, 0xe5, 0x46, 0x0f, 0xed, 0x9b, 0x43, 0xb3, 0x03, 0x83, 0x3a,
    0x60, 0x01, 0xfe, 0x0b, 0x07, 0x1a, 0x43, 0x94, 0xd1, 0x03, 0x49, 0xb5,
    0x6a, 0xd1, 0xaa, 0x5c, 0x51, 0x14, 0xf9, 0x26,
};
static const unsigned char kat591_addin1[] = {
    0xb1, 0x3a, 0x4e, 0x44, 0x21, 0xac, 0x55, 0x3f, 0x5c, 0x16, 0x6f, 0xd9,
    0x48, 0xb7, 0x58, 0x79, 0x4e, 0x57, 0xcc, 0x0c, 0xc0, 0xe7, 0x3e, 0x76,
    0x0d, 0xff, 0x26, 0xbe, 0xc8, 0xcf, 0x81, 0xe8,
};
static const unsigned char kat591_retbits[] = {
    0x5d, 0xb4, 0x68, 0x50, 0x19, 0x62, 0x02, 0x48, 0xac, 0x08, 0xd0, 0x23,
    0xd2, 0xd6, 0x05, 0x06, 0xe2, 0x1d, 0x94, 0xd1, 0x56, 0xb9, 0xa1, 0xc3,
    0x90, 0x90, 0x3b, 0xf2, 0x51, 0xd1, 0xb1, 0xff, 0x3b, 0x46, 0xe0, 0x13,
    0xa2, 0xc6, 0xd9, 0x2a, 0x9f, 0x70, 0xa2, 0xb1, 0x8b, 0xbd, 0x75, 0x84,
    0xa2, 0xf0, 0xa1, 0x9d, 0x77, 0x0c, 0xa0, 0x16, 0xee, 0x5a, 0x5f, 0x83,
    0xfa, 0xf1, 0x78, 0xd7,
};
static const struct drbg_kat_no_reseed kat591_t = {
    5, kat591_entropyin, kat591_nonce, kat591_persstr,
    kat591_addin0, kat591_addin1, kat591_retbits
};
static const struct drbg_kat kat591 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat591_t
};

static const unsigned char kat592_entropyin[] = {
    0x34, 0x00, 0x27, 0x77, 0x37, 0x42, 0xa6, 0xe9, 0x15, 0x53, 0xa7, 0xba,
    0x65, 0x44, 0x38, 0x8e, 0x74, 0xe3, 0xdd, 0xfc, 0x11, 0x51, 0xfd, 0xb9,
    0xb7, 0x63, 0xa9, 0xd1, 0x6c, 0xbe, 0x59, 0x7a,
};
static const unsigned char kat592_nonce[] = {
    0xe6, 0x04, 0x47, 0x01, 0x69, 0x51, 0x28, 0x08, 0xcd, 0x4a, 0x13, 0x1b,
    0xb1, 0x82, 0x78, 0x0b,
};
static const unsigned char kat592_persstr[] = {
    0x21, 0x0f, 0x36, 0xc9, 0x6b, 0x6b, 0x19, 0xf4, 0xb5, 0x59, 0x16, 0x1c,
    0x60, 0xc4, 0x8d, 0xf2, 0x96, 0xbf, 0xfc, 0xcc, 0xce, 0x09, 0x08, 0x59,
    0x9a, 0xba, 0xa4, 0x00, 0x17, 0x7a, 0x96, 0xe9,
};
static const unsigned char kat592_addin0[] = {
    0xd5, 0x5d, 0x98, 0x16, 0xd5, 0xc4, 0xe2, 0xe2, 0xe1, 0x81, 0x12, 0x6b,
    0xc0, 0x80, 0x40, 0xa9, 0xf6, 0x53, 0xed, 0x95, 0xea, 0x12, 0x4c, 0x64,
    0xe4, 0x05, 0x9e, 0x2f, 0x48, 0xff, 0xef, 0x10,
};
static const unsigned char kat592_addin1[] = {
    0x6c, 0x30, 0xa6, 0x4b, 0x1c, 0x73, 0xfb, 0xd0, 0x8d, 0xa5, 0xce, 0xbd,
    0xa1, 0x86, 0x42, 0xf7, 0x8a, 0x02, 0xbd, 0x32, 0x47, 0xab, 0x1d, 0x78,
    0xc1, 0xc0, 0x3f, 0x8a, 0x6c, 0xff, 0xdd, 0x89,
};
static const unsigned char kat592_retbits[] = {
    0x4f, 0xff, 0x81, 0xdb, 0xf1, 0xf4, 0x2e, 0xf5, 0x7b, 0xdb, 0xf6, 0xda,
    0x19, 0x43, 0x11, 0xdb, 0xc4, 0xb4, 0x64, 0xd3, 0xa1, 0x1a, 0x33, 0xc6,
    0x64, 0xc0, 0xd5, 0x98, 0x34, 0xec, 0x60, 0xa5, 0x83, 0xb3, 0x90, 0x8a,
    0xde, 0x9c, 0xc6, 0xfb, 0xb8, 0x9d, 0xeb, 0xb1, 0x98, 0xd1, 0x14, 0x3a,
    0xba, 0x2b, 0xc5, 0x75, 0x39, 0x83, 0x08, 0x6b, 0x42, 0x20, 0x98, 0xa2,
    0x38, 0x77, 0xcb, 0x2a,
};
static const struct drbg_kat_no_reseed kat592_t = {
    6, kat592_entropyin, kat592_nonce, kat592_persstr,
    kat592_addin0, kat592_addin1, kat592_retbits
};
static const struct drbg_kat kat592 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat592_t
};

static const unsigned char kat593_entropyin[] = {
    0xb3, 0x6a, 0x98, 0xe6, 0xc5, 0xb2, 0x36, 0x98, 0x13, 0x5e, 0xb3, 0x5f,
    0x28, 0xdc, 0x83, 0x0a, 0xc6, 0xbc, 0x17, 0x17, 0xe5, 0xc3, 0xee, 0x3e,
    0xa1, 0x38, 0x01, 0xf8, 0xa8, 0xc9, 0x36, 0x5d,
};
static const unsigned char kat593_nonce[] = {
    0x7f, 0x13, 0x19, 0x97, 0xa9, 0x29, 0x22, 0x41, 0x3d, 0x83, 0x7c, 0xa5,
    0x0e, 0x17, 0xd9, 0x77,
};
static const unsigned char kat593_persstr[] = {
    0x1e, 0xd6, 0xb0, 0x7c, 0xb5, 0x25, 0x13, 0x4e, 0xce, 0xe3, 0x22, 0x00,
    0x71, 0x36, 0x01, 0xc1, 0x8a, 0xbd, 0x9c, 0x79, 0x20, 0xe7, 0xad, 0x1d,
    0xc0, 0x76, 0x4c, 0xd9, 0x2e, 0x3d, 0x56, 0xff,
};
static const unsigned char kat593_addin0[] = {
    0xa4, 0xef, 0x95, 0x75, 0xb9, 0x66, 0x82, 0xd7, 0x55, 0x2f, 0x9e, 0xb7,
    0xba, 0x26, 0x9c, 0x43, 0xa5, 0x08, 0x35, 0xf2, 0x3f, 0xb2, 0xcf, 0x1f,
    0x81, 0x83, 0x8e, 0x9c, 0x01, 0xb1, 0xea, 0x40,
};
static const unsigned char kat593_addin1[] = {
    0x18, 0x9b, 0xcd, 0x45, 0xfa, 0x13, 0x93, 0x98, 0x38, 0x66, 0x3a, 0xde,
    0x2e, 0xaf, 0x19, 0x79, 0xb3, 0x3c, 0x17, 0x2f, 0x5e, 0xd7, 0x41, 0x12,
    0x3f, 0x06, 0xa2, 0xe3, 0xca, 0x25, 0x42, 0x7f,
};
static const unsigned char kat593_retbits[] = {
    0xc8, 0xb6, 0xcc, 0x90, 0x1c, 0x42, 0x36, 0xbd, 0x39, 0x14, 0xbc, 0x77,
    0x98, 0x7d, 0x9d, 0x63, 0x99, 0x73, 0xd6, 0x9c, 0x96, 0xe0, 0x40, 0x19,
    0x77, 0x37, 0xb9, 0xa7, 0x2b, 0xbf, 0x3b, 0x5e, 0x89, 0x98, 0xc5, 0x49,
    0x67, 0xbd, 0xae, 0x26, 0x14, 0x67, 0x72, 0x71, 0x6e, 0xd2, 0xae, 0x0c,
    0x6b, 0x20, 0x84, 0x71, 0xbd, 0xf2, 0x1a, 0x85, 0xb2, 0x77, 0x4b, 0xf7,
    0xe5, 0x16, 0xb3, 0x4f,
};
static const struct drbg_kat_no_reseed kat593_t = {
    7, kat593_entropyin, kat593_nonce, kat593_persstr,
    kat593_addin0, kat593_addin1, kat593_retbits
};
static const struct drbg_kat kat593 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat593_t
};

static const unsigned char kat594_entropyin[] = {
    0xc0, 0x0c, 0xdf, 0x7e, 0x0e, 0x78, 0x6a, 0x07, 0x93, 0x78, 0x3b, 0x6c,
    0xed, 0xc3, 0x7d, 0x6d, 0x05, 0x04, 0xd8, 0xc5, 0xeb, 0xff, 0x06, 0x30,
    0x14, 0xcf, 0x81, 0x09, 0x2c, 0xf6, 0x1d, 0x11,
};
static const unsigned char kat594_nonce[] = {
    0x94, 0x21, 0x43, 0x51, 0x9f, 0x87, 0xb3, 0x83, 0x5a, 0x0e, 0xdc, 0xa2,
    0xd4, 0xcd, 0x22, 0x85,
};
static const unsigned char kat594_persstr[] = {
    0x3a, 0x95, 0x3e, 0x04, 0x1c, 0x7e, 0x5d, 0x30, 0x63, 0x2d, 0x34, 0xa7,
    0xe8, 0x14, 0xf4, 0xfa, 0xcc, 0x85, 0x8d, 0xe6, 0xb8, 0xfb, 0x3c, 0x43,
    0x29, 0x73, 0xa4, 0xdb, 0x4a, 0x5a, 0x78, 0xc2,
};
static const unsigned char kat594_addin0[] = {
    0x1f, 0xab, 0x18, 0x06, 0x26, 0x0c, 0x24, 0x25, 0x95, 0x36, 0xe1, 0x45,
    0x74, 0xbb, 0xef, 0x9b, 0xe5, 0x0e, 0xda, 0x48, 0x93, 0x63, 0x32, 0xd6,
    0x0b, 0xc3, 0xdb, 0xcf, 0x6b, 0x10, 0xf5, 0xf7,
};
static const unsigned char kat594_addin1[] = {
    0x18, 0xcc, 0xe3, 0x08, 0x95, 0x6e, 0x63, 0x50, 0x5d, 0x43, 0xdd, 0xc2,
    0x87, 0x09, 0xea, 0xcc, 0x78, 0x3a, 0x14, 0x93, 0x15, 0x93, 0xd0, 0x79,
    0xd6, 0xbd, 0x74, 0xa6, 0x95, 0xc3, 0x07, 0x5a,
};
static const unsigned char kat594_retbits[] = {
    0x9f, 0x4d, 0x77, 0xd3, 0x94, 0x81, 0x51, 0x48, 0x16, 0x2d, 0x6c, 0xca,
    0xa8, 0xb7, 0x3c, 0xbd, 0x61, 0xb5, 0x4c, 0x5d, 0xb1, 0x04, 0xe7, 0xc0,
    0x47, 0xbf, 0x0a, 0x22, 0x5a, 0xe0, 0x55, 0x94, 0x33, 0x2b, 0xb5, 0x6c,
    0x60, 0xd5, 0x1d, 0x29, 0x86, 0xe5, 0x75, 0x13, 0xf4, 0xc2, 0x66, 0xdf,
    0x39, 0xd9, 0xbe, 0xee, 0x9b, 0x69, 0x64, 0x98, 0x91, 0xb6, 0xcc, 0xae,
    0xf4, 0xb9, 0x86, 0xbd,
};
static const struct drbg_kat_no_reseed kat594_t = {
    8, kat594_entropyin, kat594_nonce, kat594_persstr,
    kat594_addin0, kat594_addin1, kat594_retbits
};
static const struct drbg_kat kat594 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat594_t
};

static const unsigned char kat595_entropyin[] = {
    0xfb, 0x5f, 0x69, 0x58, 0x20, 0x3f, 0xa7, 0xbd, 0x06, 0x66, 0x55, 0xef,
    0x7b, 0x5f, 0xde, 0x99, 0xa5, 0x71, 0x94, 0x7e, 0x1f, 0x74, 0xb8, 0xb8,
    0xf5, 0x3a, 0x85, 0x4d, 0xbb, 0xbc, 0x35, 0x2e,
};
static const unsigned char kat595_nonce[] = {
    0x02, 0x90, 0xfb, 0x96, 0xdd, 0xae, 0xd0, 0xd8, 0x17, 0x9a, 0x07, 0x95,
    0xee, 0x1c, 0xd5, 0x29,
};
static const unsigned char kat595_persstr[] = {
    0xc5, 0x9d, 0x3d, 0x2f, 0xe3, 0x0b, 0x85, 0xed, 0x64, 0xdb, 0x2c, 0x9a,
    0xe8, 0x63, 0xb9, 0xda, 0xad, 0x7b, 0xf9, 0x6c, 0x33, 0x1d, 0x66, 0xdf,
    0x94, 0x5f, 0x5f, 0x3a, 0x1f, 0x8d, 0x25, 0x15,
};
static const unsigned char kat595_addin0[] = {
    0xc9, 0x9d, 0x7f, 0xe2, 0xde, 0xb6, 0xaf, 0x99, 0xa1, 0x78, 0x2e, 0x8f,
    0xc1, 0x2a, 0x51, 0x96, 0x36, 0x47, 0x1f, 0xc3, 0xbf, 0xad, 0x0a, 0xc2,
    0xa7, 0x89, 0xa2, 0xf2, 0x0f, 0x4d, 0xf6, 0x3f,
};
static const unsigned char kat595_addin1[] = {
    0xe4, 0x79, 0xac, 0x18, 0xb2, 0xfe, 0xd6, 0xd9, 0xf2, 0xee, 0x44, 0x1a,
    0x20, 0xbd, 0x50, 0xaa, 0xdd, 0x2a, 0x90, 0xba, 0x96, 0x25, 0xac, 0xe0,
    0x1a, 0x44, 0x49, 0xa6, 0xce, 0x6e, 0xfa, 0x92,
};
static const unsigned char kat595_retbits[] = {
    0x0a, 0xb8, 0x70, 0xc2, 0x51, 0x6e, 0xac, 0x1e, 0xf1, 0xe6, 0xdd, 0x3f,
    0x2d, 0xe9, 0xdd, 0x05, 0x64, 0xa5, 0x63, 0x63, 0xf0, 0xb8, 0xc4, 0x2b,
    0x4b, 0xae, 0x4b, 0x22, 0xe1, 0x4c, 0x99, 0x12, 0xb4, 0xc1, 0xcb, 0xa8,
    0xf3, 0x15, 0x8e, 0xb5, 0x88, 0x1b, 0x86, 0x52, 0x4c, 0x94, 0xcb, 0xb2,
    0xc8, 0x16, 0x9f, 0xc8, 0xcc, 0x2c, 0x13, 0x10, 0x3a, 0x64, 0x78, 0x33,
    0x7f, 0x2b, 0xb0, 0x0a,
};
static const struct drbg_kat_no_reseed kat595_t = {
    9, kat595_entropyin, kat595_nonce, kat595_persstr,
    kat595_addin0, kat595_addin1, kat595_retbits
};
static const struct drbg_kat kat595 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat595_t
};

static const unsigned char kat596_entropyin[] = {
    0xa4, 0xc9, 0x75, 0x00, 0x48, 0xb3, 0x35, 0x54, 0xe2, 0x47, 0x59, 0xa3,
    0x76, 0x27, 0xc4, 0x0f, 0xa7, 0xd2, 0xbc, 0xc0, 0x45, 0xb0, 0x6b, 0xd2,
    0x2a, 0x8d, 0x90, 0xae, 0x1a, 0x8c, 0xa2, 0x79,
};
static const unsigned char kat596_nonce[] = {
    0xba, 0xa1, 0x37, 0xff, 0xca, 0xd7, 0x2d, 0x25, 0x82, 0xc4, 0xc3, 0x48,
    0xb1, 0x2b, 0x81, 0x58,
};
static const unsigned char kat596_persstr[] = {
    0xa4, 0xf1, 0x15, 0x88, 0x5e, 0x0f, 0x45, 0xc4, 0x58, 0x0d, 0x04, 0xca,
    0xfc, 0xfb, 0xa8, 0xec, 0x09, 0xc5, 0xe0, 0x81, 0xb6, 0x5d, 0x64, 0x85,
    0x67, 0xc5, 0x5a, 0x9d, 0x65, 0x1e, 0x52, 0x1c,
};
static const unsigned char kat596_addin0[] = {
    0xeb, 0x8e, 0x2a, 0xa1, 0x82, 0xfb, 0xf7, 0x85, 0xf2, 0x5d, 0xa7, 0xf5,
    0x70, 0x54, 0xb7, 0x01, 0x71, 0xe5, 0xa2, 0x6a, 0xfc, 0x5a, 0x61, 0xaa,
    0xa1, 0xf9, 0x2f, 0xb4, 0xcb, 0x3a, 0xc5, 0x0c,
};
static const unsigned char kat596_addin1[] = {
    0xa4, 0x4a, 0xa0, 0xb5, 0x25, 0x5b, 0xa3, 0x39, 0xbc, 0x3c, 0xd1, 0x71,
    0x5e, 0xc6, 0xa8, 0xd5, 0x56, 0x99, 0xe1, 0xb4, 0x99, 0x6b, 0x9d, 0x73,
    0x7d, 0xa8, 0x3b, 0x8a, 0x35, 0x70, 0x45, 0xb2,
};
static const unsigned char kat596_retbits[] = {
    0x53, 0xf7, 0x4f, 0xe7, 0xab, 0x0a, 0xd3, 0x1d, 0x5d, 0xf0, 0x32, 0x7e,
    0x6d, 0xbc, 0xf5, 0xde, 0xeb, 0x3f, 0x0b, 0xd9, 0xd1, 0xff, 0x50, 0xb7,
    0xa9, 0xd5, 0x22, 0xfd, 0x9c, 0x9c, 0x3f, 0x63, 0x3e, 0x53, 0xce, 0x1d,
    0x31, 0x66, 0xaa, 0x7f, 0x2a, 0x25, 0xe3, 0x2d, 0x04, 0x88, 0xff, 0x07,
    0x89, 0xa1, 0xb9, 0x81, 0x88, 0x6d, 0x4a, 0xbd, 0xd2, 0x01, 0x4d, 0xe3,
    0x25, 0x9d, 0x40, 0x64,
};
static const struct drbg_kat_no_reseed kat596_t = {
    10, kat596_entropyin, kat596_nonce, kat596_persstr,
    kat596_addin0, kat596_addin1, kat596_retbits
};
static const struct drbg_kat kat596 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat596_t
};

static const unsigned char kat597_entropyin[] = {
    0x30, 0xb0, 0x28, 0x27, 0xca, 0xcd, 0xcb, 0xd1, 0xd9, 0x97, 0x1a, 0xf6,
    0x88, 0xa4, 0x5a, 0x6b, 0x31, 0x1e, 0x49, 0x56, 0x14, 0x29, 0xfa, 0xec,
    0x6a, 0xef, 0x87, 0x67, 0x1b, 0xac, 0xc6, 0xf9,
};
static const unsigned char kat597_nonce[] = {
    0x1b, 0x17, 0x46, 0xf7, 0x69, 0x39, 0xeb, 0x07, 0x18, 0x32, 0x86, 0x96,
    0x5e, 0xe6, 0x48, 0xab,
};
static const unsigned char kat597_persstr[] = {
    0x6b, 0x4f, 0x4b, 0x56, 0x56, 0x93, 0x48, 0x26, 0x93, 0xf6, 0x96, 0x51,
    0x21, 0x04, 0xec, 0x1a, 0xe1, 0x2a, 0xc6, 0x5a, 0x7c, 0xe0, 0xba, 0xbb,
    0xf1, 0x93, 0xf1, 0xfd, 0xc4, 0xc2, 0x31, 0xed,
};
static const unsigned char kat597_addin0[] = {
    0x1e, 0x3a, 0xdf, 0x2b, 0x30, 0xf2, 0x88, 0xb2, 0xa1, 0x8d, 0x28, 0x7c,
    0x1b, 0x05, 0x3a, 0xfb, 0x7e, 0x51, 0x3f, 0x6d, 0xe9, 0x03, 0x86, 0xab,
    0xb9, 0xda, 0x60, 0x41, 0x94, 0x15, 0xcd, 0x72,
};
static const unsigned char kat597_addin1[] = {
    0xe1, 0x92, 0x54, 0xcf, 0x8b, 0xb5, 0x02, 0x3f, 0xdf, 0x51, 0x26, 0x9b,
    0x83, 0xa9, 0x61, 0xf5, 0xa3, 0x47, 0xf5, 0x42, 0x98, 0xa3, 0x19, 0x66,
    0x98, 0xd4, 0xc9, 0xfa, 0xe3, 0xca, 0x2c, 0xcb,
};
static const unsigned char kat597_retbits[] = {
    0x47, 0x3b, 0xe3, 0x98, 0xde, 0x0e, 0x2e, 0x5c, 0xdc, 0x50, 0x52, 0x9e,
    0x6c, 0xeb, 0x02, 0x22, 0xb4, 0x03, 0x8e, 0x8e, 0x07, 0x76, 0xe9, 0x6c,
    0x27, 0x74, 0x21, 0xed, 0xc1, 0x0f, 0x2a, 0x84, 0x1d, 0xec, 0x9f, 0x07,
    0xc8, 0x2f, 0x00, 0xe2, 0xf5, 0x30, 0x62, 0xab, 0x68, 0xe3, 0xf7, 0xd4,
    0xeb, 0xdc, 0x18, 0x99, 0x9e, 0x36, 0x2e, 0xe9, 0x04, 0xa0, 0xba, 0xf4,
    0x42, 0x29, 0xb5, 0x9c,
};
static const struct drbg_kat_no_reseed kat597_t = {
    11, kat597_entropyin, kat597_nonce, kat597_persstr,
    kat597_addin0, kat597_addin1, kat597_retbits
};
static const struct drbg_kat kat597 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat597_t
};

static const unsigned char kat598_entropyin[] = {
    0x55, 0x4c, 0x46, 0x7d, 0xfd, 0x02, 0xc9, 0x63, 0x97, 0x25, 0x18, 0x37,
    0x85, 0xae, 0xcd, 0x6c, 0x0f, 0x5a, 0x32, 0x80, 0xc6, 0x1c, 0x82, 0x2b,
    0xab, 0x02, 0xf1, 0xd7, 0x41, 0x3f, 0x43, 0x61,
};
static const unsigned char kat598_nonce[] = {
    0x6f, 0xf6, 0xa4, 0xac, 0xc5, 0xa5, 0x8c, 0x68, 0x1d, 0xe9, 0xa5, 0x44,
    0x1a, 0x9f, 0x75, 0x6f,
};
static const unsigned char kat598_persstr[] = {
    0x6a, 0xce, 0xc4, 0xa4, 0xc0, 0xbe, 0x17, 0x21, 0x8e, 0x7c, 0x3f, 0x42,
    0x46, 0x64, 0x09, 0x20, 0xaa, 0xc4, 0x8f, 0xb3, 0x80, 0x09, 0x14, 0xdb,
    0x2f, 0x86, 0xdc, 0x53, 0x90, 0x3b, 0x09, 0xaf,
};
static const unsigned char kat598_addin0[] = {
    0xcd, 0x24, 0x87, 0x5c, 0xdf, 0x70, 0xeb, 0x4f, 0xb7, 0x1c, 0x01, 0x23,
    0xe0, 0xeb, 0x73, 0x8a, 0x71, 0xdd, 0x76, 0xf3, 0x6f, 0xf5, 0x4b, 0x9b,
    0x7e, 0xc7, 0x06, 0xce, 0xf0, 0xe0, 0x35, 0xf5,
};
static const unsigned char kat598_addin1[] = {
    0x99, 0xd6, 0x71, 0xe8, 0xf8, 0xe7, 0x2c, 0xf8, 0xe9, 0xa0, 0x2e, 0x76,
    0x30, 0x64, 0x8c, 0xed, 0x53, 0xac, 0x39, 0x3f, 0x6f, 0xc0, 0x42, 0x5e,
    0x5f, 0x52, 0x5e, 0xb4, 0x13, 0xd3, 0x75, 0xdd,
};
static const unsigned char kat598_retbits[] = {
    0x35, 0xbe, 0xae, 0x85, 0x58, 0xe9, 0xb5, 0xec, 0xb3, 0x0b, 0x1b, 0xf5,
    0xab, 0xc5, 0x23, 0x10, 0x21, 0x91, 0x96, 0xb5, 0x81, 0x23, 0x25, 0x70,
    0x43, 0xd2, 0x79, 0x3c, 0x05, 0xc9, 0x65, 0xe6, 0x25, 0x8a, 0xf7, 0x83,
    0xec, 0xee, 0x80, 0xb3, 0x39, 0x29, 0xa3, 0xfd, 0xd7, 0x7a, 0x8b, 0x1a,
    0xdf, 0xfb, 0x72, 0x3b, 0xcf, 0xe7, 0xe6, 0x5f, 0x7c, 0xac, 0x68, 0x28,
    0x62, 0x22, 0x0f, 0x4e,
};
static const struct drbg_kat_no_reseed kat598_t = {
    12, kat598_entropyin, kat598_nonce, kat598_persstr,
    kat598_addin0, kat598_addin1, kat598_retbits
};
static const struct drbg_kat kat598 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat598_t
};

static const unsigned char kat599_entropyin[] = {
    0x93, 0xee, 0x61, 0xdd, 0x85, 0x0c, 0x0c, 0x97, 0x27, 0x36, 0xaa, 0x98,
    0xdc, 0x18, 0xbb, 0x3a, 0xdd, 0x52, 0x9e, 0xb9, 0x25, 0x83, 0x21, 0x1a,
    0x29, 0x8f, 0xb0, 0x8d, 0xf1, 0x7a, 0x7a, 0xf7,
};
static const unsigned char kat599_nonce[] = {
    0x0a, 0x4d, 0xf3, 0x17, 0x78, 0x47, 0x51, 0x08, 0x2b, 0x98, 0x2c, 0x85,
    0x71, 0x7c, 0xdc, 0x2c,
};
static const unsigned char kat599_persstr[] = {
    0xf4, 0xf1, 0x0e, 0x09, 0x5e, 0x9f, 0x5f, 0xaa, 0xcc, 0xa2, 0x4f, 0x53,
    0x95, 0xa7, 0x3c, 0xb1, 0x53, 0x85, 0x65, 0xdf, 0x67, 0x9c, 0x87, 0x54,
    0xb7, 0xad, 0x94, 0xe8, 0x36, 0xf6, 0x91, 0x68,
};
static const unsigned char kat599_addin0[] = {
    0x87, 0x6e, 0x00, 0x2f, 0x90, 0xf5, 0x06, 0x3d, 0xdd, 0x5c, 0x57, 0x8d,
    0x9d, 0x9f, 0x4a, 0x2a, 0x9b, 0xde, 0x7e, 0xf2, 0x8a, 0x41, 0xd9, 0xc2,
    0x77, 0xa4, 0x89, 0xe8, 0x5a, 0xfb, 0x63, 0x5b,
};
static const unsigned char kat599_addin1[] = {
    0xc2, 0x56, 0x2e, 0xc2, 0x35, 0x3e, 0x63, 0xe1, 0x3e, 0x6c, 0x6f, 0x6f,
    0x06, 0xe1, 0xba, 0x22, 0x0b, 0x08, 0x48, 0xbe, 0x35, 0x3e, 0x5b, 0xd4,
    0x46, 0x89, 0x21, 0xb9, 0xbf, 0x26, 0x49, 0xfa,
};
static const unsigned char kat599_retbits[] = {
    0xd4, 0xcf, 0x47, 0x8c, 0x91, 0xba, 0x19, 0x73, 0xa7, 0xcf, 0xae, 0x15,
    0x9b, 0x8e, 0x4c, 0x05, 0x25, 0xb2, 0x96, 0xb5, 0x44, 0xfd, 0x85, 0x44,
    0xf4, 0xff, 0x10, 0xe7, 0xa2, 0xc5, 0xae, 0xbb, 0xe8, 0xb5, 0xa1, 0x0f,
    0xef, 0x16, 0xf4, 0xa9, 0x1c, 0x19, 0x93, 0x09, 0x0b, 0xe7, 0x62, 0x8c,
    0xe2, 0x1f, 0x53, 0xe2, 0xb4, 0xe7, 0xd6, 0xc2, 0xf1, 0x4b, 0xe3, 0x44,
    0x18, 0x8f, 0xa4, 0xe6,
};
static const struct drbg_kat_no_reseed kat599_t = {
    13, kat599_entropyin, kat599_nonce, kat599_persstr,
    kat599_addin0, kat599_addin1, kat599_retbits
};
static const struct drbg_kat kat599 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat599_t
};

static const unsigned char kat600_entropyin[] = {
    0xac, 0x38, 0xc3, 0x8c, 0x49, 0x8b, 0xf9, 0xb7, 0xcc, 0x72, 0x4f, 0x79,
    0x27, 0x06, 0xa0, 0x7b, 0x87, 0xe4, 0xbb, 0xa1, 0x92, 0x9a, 0x41, 0x8b,
    0x03, 0xb7, 0x42, 0xe4, 0xb0, 0xca, 0xe3, 0x9a,
};
static const unsigned char kat600_nonce[] = {
    0x80, 0x85, 0xcb, 0x0e, 0xf7, 0xd5, 0x46, 0xee, 0x8d, 0x4f, 0x9b, 0xb1,
    0x83, 0xde, 0x56, 0xe4,
};
static const unsigned char kat600_persstr[] = {
    0x73, 0xf2, 0xd1, 0x80, 0xa4, 0xd2, 0xc7, 0x36, 0x4a, 0x64, 0xbe, 0x7a,
    0xa7, 0x57, 0xbe, 0x6d, 0xf2, 0x7d, 0x93, 0x7a, 0xe8, 0xda, 0x40, 0x6c,
    0x61, 0xd0, 0x35, 0x3d, 0xda, 0x31, 0x9f, 0xd5,
};
static const unsigned char kat600_addin0[] = {
    0xd3, 0xef, 0x42, 0xed, 0xc3, 0x18, 0xfe, 0x36, 0x24, 0xe7, 0x2f, 0x27,
    0x18, 0xd2, 0x74, 0xd7, 0xdb, 0xe2, 0xd8, 0x9f, 0xd8, 0xe3, 0xc4, 0xcf,
    0xc8, 0xa7, 0xa7, 0xa1, 0xb5, 0x02, 0x21, 0x6f,
};
static const unsigned char kat600_addin1[] = {
    0x1f, 0xbb, 0xd7, 0x3b, 0x62, 0xd4, 0x31, 0x0a, 0x63, 0x6f, 0xd5, 0x6d,
    0x28, 0x37, 0x44, 0xf4, 0x85, 0x91, 0xf9, 0x6a, 0x0b, 0x65, 0xf2, 0x4d,
    0x87, 0x11, 0xec, 0xf9, 0x2b, 0x3c, 0xe3, 0xdc,
};
static const unsigned char kat600_retbits[] = {
    0x75, 0xb4, 0x29, 0xd2, 0x65, 0x7e, 0x16, 0xc7, 0x32, 0x5d, 0x05, 0x70,
    0x00, 0x5c, 0x42, 0xb2, 0x21, 0x66, 0x4e, 0x8e, 0x1b, 0x0c, 0x1e, 0x69,
    0x54, 0xaa, 0xc3, 0x06, 0x2b, 0xbc, 0x84, 0xfd, 0xc6, 0x9e, 0xb2, 0xa5,
    0x4a, 0xff, 0xcc, 0x49, 0xb4, 0x01, 0xa2, 0x22, 0x1f, 0x68, 0xa9, 0x17,
    0x46, 0x31, 0x24, 0x45, 0xb0, 0x3d, 0xf1, 0xb8, 0xf9, 0xe9, 0xf2, 0xe3,
    0x4a, 0x7a, 0x1f, 0x70,
};
static const struct drbg_kat_no_reseed kat600_t = {
    14, kat600_entropyin, kat600_nonce, kat600_persstr,
    kat600_addin0, kat600_addin1, kat600_retbits
};
static const struct drbg_kat kat600 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat600_t
};

static const unsigned char kat601_entropyin[] = {
    0x58, 0xa5, 0xf7, 0x9d, 0xa4, 0x4b, 0x9f, 0x23, 0xa9, 0x8a, 0x39, 0x35,
    0x29, 0x72, 0xad, 0x16, 0x03, 0x1f, 0xe1, 0x36, 0x37, 0xbd, 0x18, 0xd6,
    0xcb, 0x6c, 0x9f, 0x52, 0x69, 0xd8, 0xe2, 0x40,
};
static const unsigned char kat601_nonce[] = {
    0xaa, 0xa4, 0x66, 0x10, 0x68, 0x11, 0x67, 0xff, 0x8d, 0x4d, 0x2c, 0x51,
    0xe7, 0x79, 0x11, 0xd4,
};
static const unsigned char kat601_persstr[] = {0};
static const unsigned char kat601_addin0[] = {0};
static const unsigned char kat601_addin1[] = {0};
static const unsigned char kat601_retbits[] = {
    0xc1, 0x71, 0x4f, 0x89, 0x45, 0x9c, 0xe7, 0x46, 0xb1, 0x51, 0x50, 0x9e,
    0x50, 0x66, 0xd4, 0x81, 0x1a, 0x06, 0xad, 0x06, 0xc1, 0xe9, 0xb1, 0x3b,
    0x50, 0xc0, 0xfc, 0x7c, 0xdd, 0x77, 0xce, 0xed, 0xc2, 0x33, 0x90, 0x8e,
    0xbe, 0x1e, 0xa8, 0x14, 0x0e, 0xc2, 0xdc, 0x26, 0x2a, 0x43, 0x20, 0x1b,
    0xe6, 0x67, 0x00, 0x8e, 0x08, 0x1e, 0x54, 0x76, 0xb1, 0x9b, 0x27, 0x21,
    0x41, 0x11, 0xd3, 0x25,
};
static const struct drbg_kat_no_reseed kat601_t = {
    0, kat601_entropyin, kat601_nonce, kat601_persstr,
    kat601_addin0, kat601_addin1, kat601_retbits
};
static const struct drbg_kat kat601 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat601_t
};

static const unsigned char kat602_entropyin[] = {
    0xa9, 0x43, 0xe8, 0x09, 0x63, 0x04, 0x13, 0xde, 0x62, 0x07, 0x74, 0x6d,
    0x0d, 0x03, 0x41, 0x91, 0x3f, 0x46, 0x6a, 0xf0, 0xae, 0x89, 0x3c, 0xfb,
    0x34, 0x06, 0x57, 0x0b, 0x2f, 0xb7, 0x91, 0xcf,
};
static const unsigned char kat602_nonce[] = {
    0x90, 0x7b, 0x9c, 0xf7, 0xf9, 0xed, 0xf0, 0x4f, 0xcf, 0x35, 0x10, 0x31,
    0x5d, 0xd0, 0xc3, 0x81,
};
static const unsigned char kat602_persstr[] = {0};
static const unsigned char kat602_addin0[] = {0};
static const unsigned char kat602_addin1[] = {0};
static const unsigned char kat602_retbits[] = {
    0xc3, 0xb8, 0x48, 0x88, 0xc5, 0x24, 0x4c, 0xdc, 0xbb, 0x94, 0x62, 0x99,
    0xcc, 0x38, 0x48, 0xc3, 0x79, 0xa9, 0xb7, 0x80, 0xe2, 0x1f, 0x02, 0x9f,
    0x0b, 0xb2, 0xfe, 0x81, 0x5a, 0x2d, 0x03, 0x9d, 0xd7, 0xaa, 0x8a, 0x2e,
    0x80, 0x8c, 0x2a, 0xc4, 0x7b, 0x8a, 0x9c, 0xb6, 0x86, 0x0b, 0x97, 0x04,
    0x40, 0x04, 0x9a, 0x65, 0xd8, 0x15, 0xe3, 0x36, 0x9e, 0xd8, 0x33, 0xc7,
    0x61, 0x24, 0xaa, 0xc1,
};
static const struct drbg_kat_no_reseed kat602_t = {
    1, kat602_entropyin, kat602_nonce, kat602_persstr,
    kat602_addin0, kat602_addin1, kat602_retbits
};
static const struct drbg_kat kat602 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat602_t
};

static const unsigned char kat603_entropyin[] = {
    0x4d, 0x43, 0xce, 0xb1, 0xce, 0x9c, 0xac, 0xf5, 0x64, 0x03, 0xa0, 0xc9,
    0x90, 0x5d, 0xaa, 0x67, 0xa2, 0xac, 0xdd, 0xd0, 0xe4, 0xbe, 0x6a, 0x33,
    0x4b, 0x8c, 0x44, 0x34, 0xf4, 0xc6, 0x04, 0x55,
};
static const unsigned char kat603_nonce[] = {
    0x97, 0x72, 0xaa, 0xea, 0x3c, 0xd3, 0x0c, 0xa7, 0x76, 0xd6, 0x74, 0xbc,
    0xfb, 0x88, 0x4e, 0x18,
};
static const unsigned char kat603_persstr[] = {0};
static const unsigned char kat603_addin0[] = {0};
static const unsigned char kat603_addin1[] = {0};
static const unsigned char kat603_retbits[] = {
    0x85, 0xc8, 0x7c, 0x47, 0x15, 0xc1, 0x6c, 0x4b, 0x75, 0x28, 0xd0, 0x40,
    0x23, 0x05, 0x63, 0x98, 0xff, 0x28, 0x28, 0xe0, 0xb6, 0x49, 0xfb, 0xd1,
    0x0a, 0x29, 0x7b, 0x74, 0xfc, 0x3d, 0xc0, 0xdf, 0x49, 0x66, 0xbc, 0xfc,
    0xd4, 0xf8, 0x2f, 0xdb, 0x22, 0x8f, 0xaf, 0x10, 0x2d, 0x52, 0xcc, 0xa0,
    0xd3, 0xae, 0x8a, 0xf7, 0xf0, 0xc5, 0xb3, 0x0f, 0xff, 0x62, 0xd0, 0xc5,
    0x45, 0xd3, 0xde, 0x79,
};
static const struct drbg_kat_no_reseed kat603_t = {
    2, kat603_entropyin, kat603_nonce, kat603_persstr,
    kat603_addin0, kat603_addin1, kat603_retbits
};
static const struct drbg_kat kat603 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat603_t
};

static const unsigned char kat604_entropyin[] = {
    0xd4, 0x10, 0xcf, 0x13, 0xca, 0xe3, 0x65, 0xfa, 0xf3, 0x17, 0x2f, 0xb0,
    0xc2, 0x36, 0x84, 0x01, 0xf4, 0x43, 0xe7, 0x89, 0xa6, 0x2b, 0x3c, 0xe6,
    0xbc, 0x40, 0x02, 0x32, 0x49, 0xfe, 0x7d, 0xee,
};
static const unsigned char kat604_nonce[] = {
    0x22, 0xc3, 0x12, 0xb5, 0x2a, 0x06, 0x92, 0xeb, 0x38, 0x76, 0x33, 0x32,
    0xb6, 0xcd, 0x4a, 0xe9,
};
static const unsigned char kat604_persstr[] = {0};
static const unsigned char kat604_addin0[] = {0};
static const unsigned char kat604_addin1[] = {0};
static const unsigned char kat604_retbits[] = {
    0xa0, 0x05, 0xf4, 0x0f, 0xde, 0xfa, 0x1e, 0x1d, 0xce, 0x22, 0xf7, 0x35,
    0xbf, 0x3e, 0x87, 0xea, 0xfc, 0x8e, 0xc3, 0x58, 0x4a, 0x6b, 0x2b, 0x80,
    0x45, 0xbe, 0x53, 0xbd, 0xe0, 0xb1, 0xcd, 0xc4, 0x6b, 0xe3, 0x80, 0xbe,
    0x86, 0x05, 0x38, 0xca, 0x0e, 0x97, 0x6e, 0xed, 0xda, 0xe4, 0xad, 0xd2,
    0x20, 0x42, 0x62, 0x35, 0x0d, 0x5f, 0x6e, 0x19, 0xe3, 0x4d, 0xb0, 0xfc,
    0x47, 0xdc, 0xd0, 0xec,
};
static const struct drbg_kat_no_reseed kat604_t = {
    3, kat604_entropyin, kat604_nonce, kat604_persstr,
    kat604_addin0, kat604_addin1, kat604_retbits
};
static const struct drbg_kat kat604 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat604_t
};

static const unsigned char kat605_entropyin[] = {
    0xf8, 0x09, 0x72, 0xa5, 0xcd, 0x4e, 0x2e, 0x14, 0xb1, 0xf5, 0x21, 0x4d,
    0xd9, 0x3c, 0x54, 0x9d, 0xc5, 0x1e, 0xdb, 0x97, 0xc1, 0x44, 0x7d, 0x52,
    0xf3, 0xe9, 0x1b, 0x30, 0xc1, 0x5b, 0x74, 0x8c,
};
static const unsigned char kat605_nonce[] = {
    0xe1, 0xdf, 0xe4, 0xaa, 0x77, 0x7a, 0x0b, 0xeb, 0xd1, 0xfe, 0x93, 0x66,
    0x35, 0xa5, 0x19, 0x3d,
};
static const unsigned char kat605_persstr[] = {0};
static const unsigned char kat605_addin0[] = {0};
static const unsigned char kat605_addin1[] = {0};
static const unsigned char kat605_retbits[] = {
    0xf3, 0x89, 0xde, 0xfd, 0x88, 0xce, 0x73, 0xa6, 0xa5, 0xd8, 0x1a, 0x32,
    0xfe, 0xdf, 0x26, 0xe0, 0x05, 0xa5, 0xd4, 0x2f, 0x78, 0x68, 0xfb, 0xa4,
    0x0d, 0xdf, 0x20, 0xdf, 0x63, 0x25, 0xfe, 0x34, 0x73, 0x8d, 0xa3, 0xce,
    0xbb, 0x62, 0xb6, 0x02, 0x21, 0x72, 0x47, 0xfe, 0xf7, 0x78, 0x37, 0xfc,
    0x73, 0xdb, 0xef, 0x33, 0xb8, 0x13, 0xb2, 0x6e, 0xb0, 0x6b, 0xe2, 0xad,
    0x05, 0x06, 0x98, 0x82,
};
static const struct drbg_kat_no_reseed kat605_t = {
    4, kat605_entropyin, kat605_nonce, kat605_persstr,
    kat605_addin0, kat605_addin1, kat605_retbits
};
static const struct drbg_kat kat605 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat605_t
};

static const unsigned char kat606_entropyin[] = {
    0x92, 0xae, 0x36, 0x3f, 0xcb, 0xe3, 0x5f, 0xd5, 0xf6, 0x06, 0xd2, 0x1d,
    0x20, 0x94, 0xf9, 0x5b, 0x90, 0xe6, 0xb1, 0x1c, 0x21, 0x49, 0x4f, 0xe0,
    0xea, 0x3f, 0x7e, 0x8d, 0x0c, 0xcb, 0x2b, 0xc3,
};
static const unsigned char kat606_nonce[] = {
    0x88, 0xb3, 0x1b, 0x93, 0x1f, 0x49, 0xf8, 0x6c, 0x43, 0x05, 0x71, 0xaf,
    0x1a, 0x35, 0x2a, 0x3c,
};
static const unsigned char kat606_persstr[] = {0};
static const unsigned char kat606_addin0[] = {0};
static const unsigned char kat606_addin1[] = {0};
static const unsigned char kat606_retbits[] = {
    0x4f, 0x4a, 0x1e, 0x0e, 0xa8, 0x21, 0xbb, 0x8a, 0xdd, 0x74, 0xaf, 0xe6,
    0xe1, 0x26, 0x59, 0x81, 0x33, 0x38, 0xed, 0xe3, 0xd1, 0xd3, 0x53, 0x58,
    0x08, 0x34, 0x2a, 0x0c, 0xb3, 0xd0, 0x66, 0xec, 0x13, 0xa8, 0xd5, 0x29,
    0x97, 0x33, 0x66, 0x07, 0x29, 0x7e, 0x56, 0xff, 0x53, 0xca, 0x5a, 0x50,
    0x14, 0x75, 0x90, 0x23, 0x2d, 0x26, 0xf0, 0x41, 0xf7, 0x6f, 0xf9, 0xb5,
    0x82, 0x3f, 0x36, 0xf4,
};
static const struct drbg_kat_no_reseed kat606_t = {
    5, kat606_entropyin, kat606_nonce, kat606_persstr,
    kat606_addin0, kat606_addin1, kat606_retbits
};
static const struct drbg_kat kat606 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat606_t
};

static const unsigned char kat607_entropyin[] = {
    0x01, 0xab, 0x1c, 0x40, 0x96, 0xc4, 0xed, 0x6d, 0x24, 0x10, 0x74, 0xc6,
    0xe8, 0xb3, 0xe4, 0xab, 0xe7, 0x93, 0x52, 0x35, 0xd6, 0xf4, 0x8e, 0x91,
    0x3b, 0x78, 0x0d, 0x21, 0x9a, 0x71, 0xfe, 0x2d,
};
static const unsigned char kat607_nonce[] = {
    0x47, 0x2e, 0x39, 0xd4, 0x80, 0x0e, 0xf0, 0xee, 0xce, 0x7b, 0xed, 0x95,
    0x44, 0xd5, 0xaf, 0x5a,
};
static const unsigned char kat607_persstr[] = {0};
static const unsigned char kat607_addin0[] = {0};
static const unsigned char kat607_addin1[] = {0};
static const unsigned char kat607_retbits[] = {
    0xbf, 0x7a, 0xcb, 0x20, 0x90, 0x7e, 0x0b, 0xa1, 0x41, 0xe4, 0x65, 0x19,
    0xaa, 0x31, 0xd4, 0x6a, 0xd4, 0x5d, 0xa4, 0x6e, 0x01, 0x4b, 0x24, 0x20,
    0xd4, 0x8d, 0x30, 0x9c, 0xb0, 0xd4, 0x70, 0x3e, 0x82, 0xc6, 0xa7, 0x8e,
    0x83, 0x5e, 0xfa, 0xbb, 0x17, 0x23, 0xb7, 0x5b, 0x8d, 0xac, 0xe5, 0x96,
    0x80, 0x42, 0x41, 0x60, 0x75, 0xcb, 0x71, 0xb9, 0xc1, 0x49, 0x80, 0x6c,
    0xc4, 0x21, 0x47, 0x58,
};
static const struct drbg_kat_no_reseed kat607_t = {
    6, kat607_entropyin, kat607_nonce, kat607_persstr,
    kat607_addin0, kat607_addin1, kat607_retbits
};
static const struct drbg_kat kat607 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat607_t
};

static const unsigned char kat608_entropyin[] = {
    0xf6, 0x30, 0x21, 0xbc, 0x49, 0x22, 0x30, 0x5c, 0x82, 0x67, 0xb2, 0xce,
    0xf7, 0x67, 0xe3, 0xbb, 0xda, 0xb7, 0x2e, 0x11, 0x9b, 0xc8, 0x31, 0x7a,
    0xd8, 0xfe, 0x2c, 0x27, 0x69, 0x4f, 0x9e, 0x00,
};
static const unsigned char kat608_nonce[] = {
    0xe6, 0xcc, 0xef, 0x20, 0xf8, 0x75, 0x19, 0xd5, 0xd6, 0x77, 0x16, 0xe1,
    0x2e, 0x1b, 0xd7, 0x60,
};
static const unsigned char kat608_persstr[] = {0};
static const unsigned char kat608_addin0[] = {0};
static const unsigned char kat608_addin1[] = {0};
static const unsigned char kat608_retbits[] = {
    0xd7, 0xee, 0xc5, 0x1c, 0x18, 0x6f, 0x59, 0x90, 0xbc, 0x3f, 0x55, 0xdb,
    0x34, 0x2a, 0x41, 0xa2, 0xea, 0x15, 0x50, 0x34, 0xaf, 0x65, 0x1c, 0x3c,
    0x29, 0x13, 0x10, 0x24, 0x1d, 0x46, 0x55, 0x73, 0x3c, 0xc9, 0xd7, 0x81,
    0xd6, 0x88, 0x61, 0x38, 0xb3, 0xf9, 0xac, 0x59, 0x5b, 0x95, 0x6f, 0x94,
    0xda, 0xff, 0x64, 0xdf, 0xde, 0x94, 0xd3, 0xc8, 0xf2, 0xb8, 0x11, 0x82,
    0xb8, 0x7e, 0xb8, 0x32,
};
static const struct drbg_kat_no_reseed kat608_t = {
    7, kat608_entropyin, kat608_nonce, kat608_persstr,
    kat608_addin0, kat608_addin1, kat608_retbits
};
static const struct drbg_kat kat608 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat608_t
};

static const unsigned char kat609_entropyin[] = {
    0xa8, 0xfd, 0xae, 0x7e, 0x11, 0x16, 0x12, 0x2d, 0x2b, 0x42, 0xd1, 0x54,
    0xa6, 0x32, 0x9a, 0xee, 0x99, 0xae, 0x6f, 0x3b, 0xd6, 0x51, 0x80, 0xa0,
    0x38, 0x28, 0xa3, 0x54, 0x4b, 0x80, 0xf9, 0x9b,
};
static const unsigned char kat609_nonce[] = {
    0x10, 0xb1, 0x4f, 0x56, 0x8e, 0x42, 0xff, 0x31, 0xcf, 0x73, 0xb1, 0xf3,
    0x94, 0x3a, 0xec, 0xb8,
};
static const unsigned char kat609_persstr[] = {0};
static const unsigned char kat609_addin0[] = {0};
static const unsigned char kat609_addin1[] = {0};
static const unsigned char kat609_retbits[] = {
    0x4a, 0xbd, 0xe8, 0x19, 0x9c, 0x07, 0x76, 0xac, 0xe8, 0xaf, 0xba, 0x19,
    0x6d, 0x2b, 0xef, 0xc6, 0xd6, 0x88, 0x8f, 0x3e, 0x95, 0x60, 0x10, 0x88,
    0x95, 0x68, 0x90, 0x17, 0xac, 0x65, 0x4a, 0x55, 0x57, 0x22, 0xc0, 0xf1,
    0x4c, 0x53, 0xb1, 0x9f, 0x3c, 0xf6, 0x71, 0xa8, 0x8c, 0x10, 0xa9, 0x15,
    0xea, 0x72, 0x0e, 0x7d, 0xc9, 0xb9, 0xe5, 0xe0, 0x31, 0x91, 0x5e, 0x33,
    0x0e, 0xbb, 0x0e, 0x4c,
};
static const struct drbg_kat_no_reseed kat609_t = {
    8, kat609_entropyin, kat609_nonce, kat609_persstr,
    kat609_addin0, kat609_addin1, kat609_retbits
};
static const struct drbg_kat kat609 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat609_t
};

static const unsigned char kat610_entropyin[] = {
    0x9a, 0xef, 0xe4, 0x3a, 0xc5, 0x49, 0x0d, 0xef, 0xc5, 0xba, 0xad, 0xb1,
    0x2e, 0x66, 0x83, 0x8d, 0xe9, 0xdd, 0xaf, 0xc9, 0x44, 0xcd, 0x68, 0x6e,
    0xcb, 0x05, 0x96, 0x35, 0x40, 0x34, 0x52, 0xaf,
};
static const unsigned char kat610_nonce[] = {
    0xe9, 0x25, 0x6f, 0x49, 0x02, 0xc7, 0xd7, 0xf6, 0xbb, 0xb1, 0xdc, 0xc1,
    0x56, 0xdd, 0xf9, 0xbd,
};
static const unsigned char kat610_persstr[] = {0};
static const unsigned char kat610_addin0[] = {0};
static const unsigned char kat610_addin1[] = {0};
static const unsigned char kat610_retbits[] = {
    0xd9, 0x6d, 0x90, 0xac, 0x5a, 0xca, 0xb5, 0x4b, 0xe1, 0x8d, 0xa1, 0x9c,
    0x76, 0x15, 0xb4, 0xce, 0x47, 0x1a, 0x22, 0x41, 0x6f, 0xfe, 0x20, 0x43,
    0xe7, 0x82, 0xbe, 0xb8, 0x85, 0x26, 0x9b, 0xbb, 0xac, 0xe4, 0xc4, 0x16,
    0x05, 0x0c, 0xb4, 0xfb, 0x90, 0x15, 0x8f, 0xda, 0x6d, 0xa5, 0x8b, 0xf8,
    0x66, 0x0e, 0xf9, 0x16, 0x0d, 0x1c, 0x15, 0xfd, 0xd4, 0xc0, 0x67, 0xc5,
    0xc9, 0x8c, 0xcd, 0xb7,
};
static const struct drbg_kat_no_reseed kat610_t = {
    9, kat610_entropyin, kat610_nonce, kat610_persstr,
    kat610_addin0, kat610_addin1, kat610_retbits
};
static const struct drbg_kat kat610 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat610_t
};

static const unsigned char kat611_entropyin[] = {
    0x36, 0x18, 0xb5, 0x63, 0x86, 0xb7, 0xae, 0x61, 0xdf, 0xc1, 0x84, 0x99,
    0x9c, 0xc4, 0x58, 0x7c, 0xf0, 0x7e, 0xda, 0x8b, 0xe8, 0x61, 0xf1, 0xc6,
    0x4a, 0x28, 0xec, 0xb2, 0x87, 0x58, 0x90, 0xd3,
};
static const unsigned char kat611_nonce[] = {
    0xde, 0xb0, 0x3a, 0xcd, 0x0e, 0xa6, 0xf0, 0x59, 0x15, 0x80, 0xb8, 0xaf,
    0xbc, 0x2f, 0xf8, 0x80,
};
static const unsigned char kat611_persstr[] = {0};
static const unsigned char kat611_addin0[] = {0};
static const unsigned char kat611_addin1[] = {0};
static const unsigned char kat611_retbits[] = {
    0xf6, 0x05, 0x30, 0x5d, 0x87, 0xf2, 0x09, 0xac, 0x29, 0x3a, 0xb9, 0xc8,
    0xc1, 0xd3, 0x12, 0x84, 0x18, 0xfc, 0xc9, 0x17, 0x5a, 0xec, 0x25, 0x65,
    0x26, 0x5b, 0x1d, 0xac, 0x75, 0xe7, 0xaa, 0x80, 0x57, 0x95, 0x3e, 0x0f,
    0x68, 0x54, 0x80, 0x0f, 0x37, 0xce, 0x40, 0x5e, 0xeb, 0x24, 0x92, 0x7a,
    0xb7, 0x8e, 0x94, 0xfb, 0x69, 0x4b, 0x66, 0x8c, 0xaa, 0xeb, 0xa5, 0x27,
    0x1f, 0xc4, 0x86, 0xf4,
};
static const struct drbg_kat_no_reseed kat611_t = {
    10, kat611_entropyin, kat611_nonce, kat611_persstr,
    kat611_addin0, kat611_addin1, kat611_retbits
};
static const struct drbg_kat kat611 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat611_t
};

static const unsigned char kat612_entropyin[] = {
    0xc0, 0x9c, 0xbd, 0x47, 0x8e, 0x38, 0xc7, 0x52, 0x0c, 0xd9, 0x24, 0x32,
    0xb5, 0xfc, 0xe8, 0xc1, 0x57, 0x58, 0xef, 0x56, 0xc4, 0xe4, 0x57, 0xfd,
    0x4f, 0x2c, 0x97, 0xd5, 0x2e, 0x8f, 0x45, 0x66,
};
static const unsigned char kat612_nonce[] = {
    0x05, 0x3e, 0xa2, 0x92, 0x36, 0x74, 0x76, 0x41, 0x5d, 0xf7, 0xfe, 0xdb,
    0xf2, 0x5d, 0x69, 0x54,
};
static const unsigned char kat612_persstr[] = {0};
static const unsigned char kat612_addin0[] = {0};
static const unsigned char kat612_addin1[] = {0};
static const unsigned char kat612_retbits[] = {
    0x00, 0xa8, 0x1e, 0x0b, 0x7d, 0xb1, 0x56, 0x98, 0xb8, 0x9a, 0x75, 0xf5,
    0x61, 0x75, 0xf4, 0x49, 0xdb, 0xc4, 0x21, 0x1b, 0xa3, 0x54, 0xb4, 0x6f,
    0x24, 0x74, 0xac, 0xb5, 0x5d, 0xc1, 0x70, 0xbf, 0x6c, 0x0e, 0xe1, 0x81,
    0xdc, 0xea, 0xc3, 0xda, 0xf5, 0xfc, 0x81, 0xa5, 0x7c, 0x63, 0xdd, 0xc2,
    0x81, 0xcd, 0x24, 0xcd, 0x6e, 0x7f, 0x25, 0xa1, 0xb4, 0x0a, 0xdb, 0x4d,
    0x27, 0xc7, 0x13, 0x19,
};
static const struct drbg_kat_no_reseed kat612_t = {
    11, kat612_entropyin, kat612_nonce, kat612_persstr,
    kat612_addin0, kat612_addin1, kat612_retbits
};
static const struct drbg_kat kat612 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat612_t
};

static const unsigned char kat613_entropyin[] = {
    0x59, 0xef, 0x31, 0xed, 0x68, 0x54, 0x01, 0x6a, 0xb4, 0xab, 0x7c, 0x4a,
    0xf7, 0x2f, 0x5b, 0x16, 0xad, 0x2d, 0x7b, 0x57, 0xa1, 0x8f, 0x79, 0xf8,
    0xf2, 0x9d, 0xe2, 0x6e, 0xe4, 0x20, 0x09, 0x61,
};
static const unsigned char kat613_nonce[] = {
    0x8e, 0x8d, 0x6f, 0x83, 0x78, 0x63, 0x43, 0x14, 0xf7, 0x2a, 0xe0, 0x57,
    0x99, 0x1a, 0x33, 0x3e,
};
static const unsigned char kat613_persstr[] = {0};
static const unsigned char kat613_addin0[] = {0};
static const unsigned char kat613_addin1[] = {0};
static const unsigned char kat613_retbits[] = {
    0x1a, 0xcf, 0x34, 0x3f, 0xca, 0x1f, 0x42, 0xd4, 0x7a, 0x11, 0x31, 0xe5,
    0xd1, 0x89, 0x06, 0xb5, 0xb7, 0x3f, 0xff, 0x61, 0x46, 0x12, 0x37, 0x88,
    0x2a, 0x58, 0xbc, 0x5a, 0xdd, 0x75, 0x27, 0xa5, 0xdd, 0x03, 0xd2, 0xb1,
    0x96, 0x7b, 0x70, 0x22, 0x6d, 0x11, 0x19, 0xae, 0x14, 0x35, 0x91, 0x15,
    0x15, 0xe3, 0xea, 0xa6, 0x7a, 0xcf, 0x14, 0x42, 0x14, 0x71, 0x2f, 0xf3,
    0x34, 0x64, 0x90, 0xf2,
};
static const struct drbg_kat_no_reseed kat613_t = {
    12, kat613_entropyin, kat613_nonce, kat613_persstr,
    kat613_addin0, kat613_addin1, kat613_retbits
};
static const struct drbg_kat kat613 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat613_t
};

static const unsigned char kat614_entropyin[] = {
    0x04, 0x13, 0x8e, 0x8f, 0xe8, 0xfe, 0xf7, 0xfe, 0xea, 0xb5, 0x4e, 0xe3,
    0xe5, 0x80, 0x2e, 0x02, 0x7b, 0x1f, 0x39, 0x44, 0x87, 0xca, 0xc3, 0x6d,
    0x1b, 0xb0, 0xf0, 0x52, 0xa7, 0xa1, 0x07, 0x5c,
};
static const unsigned char kat614_nonce[] = {
    0xd6, 0xb6, 0x48, 0x4d, 0xbb, 0x46, 0x38, 0x9e, 0xdf, 0xc1, 0xac, 0x96,
    0x9e, 0x46, 0x22, 0x14,
};
static const unsigned char kat614_persstr[] = {0};
static const unsigned char kat614_addin0[] = {0};
static const unsigned char kat614_addin1[] = {0};
static const unsigned char kat614_retbits[] = {
    0x09, 0xf2, 0xc6, 0x86, 0xd9, 0xb0, 0x1d, 0x49, 0xf1, 0x23, 0x48, 0xf1,
    0x2f, 0xf8, 0x32, 0xdb, 0x4e, 0xa9, 0x10, 0x5e, 0xa6, 0xa1, 0x48, 0xfc,
    0xaa, 0xea, 0x32, 0x49, 0x0d, 0x2d, 0x84, 0x3a, 0x28, 0x2c, 0xdc, 0x30,
    0xc0, 0x11, 0x19, 0x1f, 0x8b, 0xa7, 0x7c, 0x22, 0x4c, 0x29, 0x70, 0x12,
    0xe2, 0x61, 0xc8, 0xa7, 0x39, 0xd0, 0x77, 0x72, 0x98, 0xeb, 0x59, 0x1c,
    0x99, 0x02, 0x8c, 0xa5,
};
static const struct drbg_kat_no_reseed kat614_t = {
    13, kat614_entropyin, kat614_nonce, kat614_persstr,
    kat614_addin0, kat614_addin1, kat614_retbits
};
static const struct drbg_kat kat614 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat614_t
};

static const unsigned char kat615_entropyin[] = {
    0x11, 0xb1, 0xa3, 0x07, 0xc2, 0x66, 0xd4, 0x07, 0xc3, 0xab, 0x12, 0xbe,
    0x77, 0x4e, 0xe3, 0x1c, 0xdc, 0x91, 0x8a, 0xe1, 0x4c, 0xd9, 0x8f, 0xcf,
    0x02, 0xc3, 0xac, 0xc8, 0x82, 0xdd, 0xb8, 0x2c,
};
static const unsigned char kat615_nonce[] = {
    0xe4, 0xef, 0xda, 0xf6, 0xf0, 0x1e, 0xb5, 0x47, 0x5d, 0x82, 0xc9, 0x4e,
    0x12, 0x9a, 0x35, 0xe1,
};
static const unsigned char kat615_persstr[] = {0};
static const unsigned char kat615_addin0[] = {0};
static const unsigned char kat615_addin1[] = {0};
static const unsigned char kat615_retbits[] = {
    0x6a, 0x52, 0x63, 0x64, 0x4a, 0x10, 0x1e, 0x9f, 0x03, 0xa4, 0x02, 0x60,
    0xa6, 0x8f, 0xa7, 0x27, 0x1a, 0x39, 0x9b, 0xa9, 0x54, 0xfe, 0xe2, 0x30,
    0x61, 0x3a, 0x18, 0xbc, 0x18, 0xcd, 0x5c, 0x54, 0x03, 0xd6, 0x79, 0xc7,
    0x73, 0x5c, 0xb1, 0x67, 0x1b, 0x17, 0x4e, 0x60, 0xed, 0x82, 0x8e, 0x70,
    0x73, 0x77, 0xf5, 0x51, 0x78, 0xbb, 0x53, 0x8b, 0xf4, 0x26, 0x74, 0x53,
    0xe5, 0xc8, 0x5b, 0x97,
};
static const struct drbg_kat_no_reseed kat615_t = {
    14, kat615_entropyin, kat615_nonce, kat615_persstr,
    kat615_addin0, kat615_addin1, kat615_retbits
};
static const struct drbg_kat kat615 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat615_t
};

static const unsigned char kat616_entropyin[] = {
    0x00, 0x1e, 0xc3, 0xb1, 0x92, 0xdd, 0xc7, 0x65, 0x55, 0x3e, 0x15, 0x74,
    0x2d, 0xff, 0xeb, 0x21, 0xcc, 0x7d, 0x97, 0xa4, 0xbc, 0xf8, 0x66, 0xe3,
    0x66, 0x4d, 0x8a, 0x5e, 0xcb, 0x4c, 0x24, 0x63,
};
static const unsigned char kat616_nonce[] = {
    0x6c, 0xa8, 0x48, 0x65, 0x1d, 0x42, 0x0f, 0xb0, 0x2f, 0x9b, 0x66, 0xf0,
    0x6b, 0x37, 0x7e, 0x59,
};
static const unsigned char kat616_persstr[] = {0};
static const unsigned char kat616_addin0[] = {
    0x99, 0xf1, 0x39, 0xab, 0x5e, 0xe4, 0xf7, 0xee, 0xd6, 0x14, 0x8e, 0x82,
    0xd7, 0x9a, 0xd5, 0xf2, 0xb9, 0xfa, 0x63, 0x8d, 0x57, 0x4e, 0x5d, 0xb7,
    0x9b, 0x65, 0x0c, 0x0e, 0x68, 0x2c, 0xa4, 0x66,
};
static const unsigned char kat616_addin1[] = {
    0x6e, 0x7b, 0xf0, 0xae, 0x28, 0xa7, 0x97, 0xcc, 0xbb, 0x47, 0x10, 0x1f,
    0x26, 0xbf, 0xe5, 0xa0, 0xb1, 0xe4, 0x50, 0xc5, 0x7a, 0xed, 0xf7, 0x31,
    0x27, 0x24, 0x11, 0xfa, 0x7b, 0x6c, 0x4e, 0xd4,
};
static const unsigned char kat616_retbits[] = {
    0x86, 0x5b, 0x6d, 0xd4, 0x36, 0x3c, 0x59, 0x40, 0xd6, 0x22, 0x8c, 0xc9,
    0x0b, 0xa8, 0xf1, 0xa2, 0x1e, 0xfb, 0xaa, 0x99, 0xb0, 0xc7, 0xb3, 0x73,
    0x61, 0xf7, 0xfe, 0xd7, 0xe9, 0x69, 0xa9, 0x7b, 0x68, 0xd5, 0x50, 0xdd,
    0x6a, 0xd4, 0xbb, 0xfa, 0xf6, 0x62, 0x67, 0x79, 0xbf, 0xb4, 0x3c, 0x66,
    0x84, 0x5c, 0x29, 0x23, 0xdf, 0x9f, 0x55, 0x30, 0x7c, 0x8b, 0xc9, 0xf0,
    0xa3, 0x87, 0x2f, 0xa7,
};
static const struct drbg_kat_no_reseed kat616_t = {
    0, kat616_entropyin, kat616_nonce, kat616_persstr,
    kat616_addin0, kat616_addin1, kat616_retbits
};
static const struct drbg_kat kat616 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat616_t
};

static const unsigned char kat617_entropyin[] = {
    0x53, 0xaa, 0x66, 0x68, 0xd0, 0x6d, 0x2b, 0xdb, 0x4a, 0xca, 0x98, 0x9d,
    0x29, 0x4d, 0x68, 0xb0, 0x00, 0x36, 0xe1, 0xb4, 0x66, 0xd1, 0x55, 0x31,
    0x86, 0xb9, 0xed, 0xa9, 0xde, 0x69, 0x3a, 0x68,
};
static const unsigned char kat617_nonce[] = {
    0xb7, 0x7d, 0x9b, 0x74, 0x57, 0x47, 0x31, 0xdc, 0xd9, 0x6a, 0xed, 0x38,
    0x35, 0x05, 0x27, 0x6d,
};
static const unsigned char kat617_persstr[] = {0};
static const unsigned char kat617_addin0[] = {
    0x0e, 0x00, 0x58, 0x9f, 0x59, 0x26, 0xad, 0x32, 0xa0, 0xac, 0xb3, 0x37,
    0xef, 0xb6, 0x1d, 0x0f, 0x8b, 0x6c, 0x4f, 0x25, 0x26, 0xea, 0x6d, 0x1a,
    0xaa, 0x20, 0x23, 0xd3, 0x93, 0xb0, 0xf9, 0x22,
};
static const unsigned char kat617_addin1[] = {
    0x70, 0x40, 0x4e, 0x72, 0x9a, 0x59, 0x6e, 0x11, 0xc5, 0xd1, 0x4a, 0xb9,
    0xe4, 0x35, 0xd5, 0x0e, 0x47, 0xaf, 0xb7, 0x35, 0xd5, 0x58, 0x29, 0x3a,
    0x8d, 0x11, 0x97, 0xcb, 0xf8, 0x54, 0x36, 0xfe,
};
static const unsigned char kat617_retbits[] = {
    0xb8, 0x37, 0x78, 0xfb, 0x3f, 0xe1, 0x6b, 0xfa, 0x43, 0x23, 0x0a, 0xc1,
    0x01, 0xc9, 0xb3, 0x81, 0x68, 0x27, 0xf5, 0x50, 0x0c, 0x65, 0x06, 0x02,
    0x98, 0xd5, 0x8b, 0xd4, 0xfa, 0xcb, 0x17, 0xa0, 0x62, 0xee, 0x03, 0x98,
    0x1e, 0x6d, 0x19, 0xeb, 0x2c, 0x98, 0x51, 0xfb, 0x00, 0xae, 0x2b, 0x4b,
    0xc5, 0x17, 0xee, 0x33, 0x8e, 0xf5, 0x98, 0x06, 0xe3, 0xc8, 0xb0, 0xb9,
    0x9f, 0xb6, 0x7a, 0x31,
};
static const struct drbg_kat_no_reseed kat617_t = {
    1, kat617_entropyin, kat617_nonce, kat617_persstr,
    kat617_addin0, kat617_addin1, kat617_retbits
};
static const struct drbg_kat kat617 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat617_t
};

static const unsigned char kat618_entropyin[] = {
    0x3d, 0x38, 0x73, 0x6b, 0x9f, 0x03, 0xdd, 0x30, 0x6b, 0x10, 0xd5, 0xbb,
    0x91, 0xc1, 0x6b, 0x16, 0x1e, 0xb4, 0xb6, 0xf0, 0x54, 0xd2, 0xbc, 0x4d,
    0x56, 0x1a, 0x93, 0x98, 0x50, 0xc0, 0x40, 0xea,
};
static const unsigned char kat618_nonce[] = {
    0xa0, 0xd0, 0xc1, 0x88, 0xcb, 0x2f, 0x32, 0xa3, 0x12, 0x74, 0x89, 0xb7,
    0x4b, 0x83, 0xae, 0x78,
};
static const unsigned char kat618_persstr[] = {0};
static const unsigned char kat618_addin0[] = {
    0x37, 0xcf, 0xe2, 0xab, 0xe1, 0x90, 0x6a, 0x58, 0x9c, 0xdc, 0x67, 0x1e,
    0x01, 0xb5, 0x0e, 0xca, 0x78, 0xb4, 0x27, 0x02, 0xe8, 0x4e, 0x08, 0x8d,
    0xbf, 0xd2, 0x1c, 0x22, 0xe0, 0x43, 0x2e, 0x46,
};
static const unsigned char kat618_addin1[] = {
    0x2f, 0x1f, 0x52, 0x3b, 0x30, 0x5b, 0xbe, 0x79, 0x9a, 0x92, 0x0a, 0x65,
    0x7a, 0xb3, 0x6f, 0xf4, 0x20, 0x71, 0x88, 0x99, 0x8f, 0x6c, 0x7c, 0xd3,
    0x9b, 0xe4, 0xfc, 0x5d, 0x06, 0x93, 0xa0, 0x70,
};
static const unsigned char kat618_retbits[] = {
    0xdc, 0x1a, 0x5b, 0x28, 0xe1, 0x9e, 0xf8, 0x9c, 0x86, 0x9e, 0xa6, 0x95,
    0xf7, 0xd9, 0xa5, 0x79, 0xf6, 0x17, 0xd0, 0x92, 0x3b, 0xd0, 0x64, 0x11,
    0x35, 0xa0, 0x6a, 0x86, 0xe0, 0x11, 0x75, 0x1c, 0xbc, 0x2d, 0x9f, 0x2f,
    0xa2, 0x61, 0x2c, 0x3e, 0x6a, 0x2a, 0xff, 0x53, 0x8c, 0x64, 0x53, 0x80,
    0xa6, 0x18, 0xf1, 0xf1, 0x8f, 0xbd, 0x98, 0x7a, 0xe9, 0xf4, 0xff, 0x1d,
    0x70, 0x47, 0x49, 0xf7,
};
static const struct drbg_kat_no_reseed kat618_t = {
    2, kat618_entropyin, kat618_nonce, kat618_persstr,
    kat618_addin0, kat618_addin1, kat618_retbits
};
static const struct drbg_kat kat618 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat618_t
};

static const unsigned char kat619_entropyin[] = {
    0x89, 0xf2, 0x09, 0xa7, 0xd8, 0xce, 0x9f, 0x93, 0x68, 0xf9, 0x47, 0x1a,
    0x33, 0xbf, 0x7b, 0x46, 0x9e, 0xf8, 0x0f, 0x99, 0x4c, 0xa5, 0xbf, 0xfa,
    0xaa, 0x18, 0xf1, 0x63, 0xa4, 0x50, 0x3d, 0x97,
};
static const unsigned char kat619_nonce[] = {
    0x37, 0x65, 0x56, 0x04, 0xc0, 0x45, 0xca, 0x1f, 0x52, 0x69, 0x41, 0xa0,
    0x50, 0xa2, 0xa7, 0x8e,
};
static const unsigned char kat619_persstr[] = {0};
static const unsigned char kat619_addin0[] = {
    0x1d, 0x24, 0xe7, 0xe3, 0xbc, 0xb7, 0x70, 0x9a, 0xb1, 0x80, 0x73, 0x4b,
    0xd1, 0x2f, 0x45, 0x54, 0xa2, 0x30, 0x3a, 0x38, 0xa8, 0x35, 0x51, 0xd9,
    0x45, 0x4f, 0xe0, 0x45, 0xab, 0xea, 0x54, 0x25,
};
static const unsigned char kat619_addin1[] = {
    0x15, 0x64, 0x6b, 0x33, 0x1b, 0x80, 0x8d, 0xc9, 0x71, 0xdd, 0x2b, 0xc9,
    0xca, 0xf8, 0x03, 0x0b, 0x57, 0x08, 0x5f, 0x8d, 0xd6, 0x2c, 0x4b, 0xfb,
    0x28, 0x11, 0x0c, 0x75, 0x97, 0x7d, 0xd1, 0xd0,
};
static const unsigned char kat619_retbits[] = {
    0x59, 0xac, 0x0b, 0x9b, 0xbd, 0x27, 0x99, 0xb0, 0x80, 0x85, 0x27, 0xa8,
    0x50, 0xb9, 0x98, 0xee, 0x3d, 0x6b, 0x7a, 0xec, 0xdb, 0xa3, 0x40, 0xf1,
    0x7b, 0x68, 0x22, 0x09, 0x22, 0x2b, 0x67, 0x83, 0xcf, 0x9a, 0x48, 0xec,
    0x45, 0xca, 0xf4, 0x04, 0x46, 0xae, 0x37, 0xe9, 0x53, 0x98, 0x99, 0x41,
    0x0c, 0xb6, 0x3f, 0x79, 0xb3, 0x14, 0xd3, 0xb2, 0xb8, 0x33, 0xb5, 0x4c,
    0x26, 0xa2, 0xca, 0xbc,
};
static const struct drbg_kat_no_reseed kat619_t = {
    3, kat619_entropyin, kat619_nonce, kat619_persstr,
    kat619_addin0, kat619_addin1, kat619_retbits
};
static const struct drbg_kat kat619 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat619_t
};

static const unsigned char kat620_entropyin[] = {
    0x0b, 0xee, 0x49, 0x37, 0xa0, 0x71, 0xfd, 0x93, 0x9f, 0x00, 0x07, 0xc3,
    0xce, 0x3a, 0xbe, 0xf4, 0x3f, 0xfa, 0x3b, 0xd0, 0xe0, 0x5e, 0xfc, 0x01,
    0x90, 0xed, 0x5f, 0x2a, 0x2d, 0x64, 0x47, 0x78,
};
static const unsigned char kat620_nonce[] = {
    0xc5, 0x4f, 0x6c, 0xf1, 0x0f, 0x64, 0x54, 0xc6, 0x9b, 0x66, 0x7b, 0x64,
    0x16, 0xad, 0xcd, 0x6f,
};
static const unsigned char kat620_persstr[] = {0};
static const unsigned char kat620_addin0[] = {
    0x4d, 0x84, 0x2e, 0x42, 0x6b, 0x64, 0xe9, 0x42, 0x2e, 0x7f, 0x77, 0xf6,
    0x00, 0x9c, 0x11, 0x57, 0x57, 0x08, 0x32, 0x23, 0x5b, 0xe6, 0x5d, 0x33,
    0x69, 0x53, 0x0f, 0x29, 0x9f, 0xbf, 0xc8, 0x2e,
};
static const unsigned char kat620_addin1[] = {
    0x72, 0x7e, 0x31, 0xe8, 0x39, 0xa7, 0x87, 0x5b, 0x65, 0x4b, 0xcf, 0x1a,
    0x54, 0x28, 0xd8, 0xaf, 0x90, 0x0e, 0x5a, 0xa4, 0x06, 0x36, 0xaa, 0x80,
    0xfd, 0x85, 0x1c, 0x33, 0x78, 0x7c, 0xab, 0xf3,
};
static const unsigned char kat620_retbits[] = {
    0x5a, 0x63, 0xf6, 0x62, 0x53, 0x0b, 0xb2, 0xec, 0x41, 0x62, 0xf7, 0x52,
    0x36, 0x15, 0x30, 0xcc, 0x8d, 0x5c, 0x89, 0x05, 0x87, 0xd2, 0x26, 0xb2,
    0x0f, 0x14, 0x2b, 0xae, 0x2a, 0x2d, 0xbd, 0x69, 0x47, 0x80, 0xd2, 0xa5,
    0x55, 0x53, 0xb0, 0x35, 0xcd, 0x35, 0x52, 0xba, 0xb0, 0xfa, 0xf9, 0x4e,
    0x63, 0x89, 0x53, 0x30, 0x0e, 0xa9, 0x50, 0x87, 0xa7, 0x32, 0xc9, 0x48,
    0x9e, 0xa4, 0x1e, 0xac,
};
static const struct drbg_kat_no_reseed kat620_t = {
    4, kat620_entropyin, kat620_nonce, kat620_persstr,
    kat620_addin0, kat620_addin1, kat620_retbits
};
static const struct drbg_kat kat620 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat620_t
};

static const unsigned char kat621_entropyin[] = {
    0x26, 0x9d, 0xe0, 0xa0, 0x01, 0x9b, 0x68, 0x51, 0x02, 0x95, 0x3c, 0x21,
    0xd4, 0xd4, 0x82, 0x5e, 0xe0, 0x0f, 0xdb, 0x77, 0x4c, 0x7c, 0xa7, 0x21,
    0xf1, 0xf3, 0xcc, 0x1a, 0x22, 0x20, 0xcc, 0x98,
};
static const unsigned char kat621_nonce[] = {
    0xe1, 0x08, 0x06, 0xc0, 0xd8, 0xb9, 0x17, 0x5f, 0x22, 0xfc, 0x96, 0xc7,
    0xd7, 0x7d, 0xfd, 0xfc,
};
static const unsigned char kat621_persstr[] = {0};
static const unsigned char kat621_addin0[] = {
    0x18, 0x5e, 0xa8, 0x97, 0xe2, 0x52, 0x36, 0x05, 0x86, 0xb3, 0x9b, 0x5a,
    0x2e, 0x3e, 0x12, 0xbb, 0x30, 0x88, 0x5b, 0x99, 0xba, 0x06, 0x8d, 0x10,
    0xac, 0x7d, 0xfd, 0xfc, 0x26, 0x3e, 0xbc, 0xd5,
};
static const unsigned char kat621_addin1[] = {
    0x25, 0x9b, 0xac, 0x2e, 0x62, 0xda, 0x63, 0x89, 0xbf, 0x6a, 0x5a, 0xb3,
    0x44, 0x17, 0x7c, 0xa1, 0x6b, 0xa5, 0xaf, 0xe7, 0x75, 0x5b, 0x5a, 0x5b,
    0xb4, 0xb9, 0x4b, 0x3e, 0xcc, 0x6a, 0x8f, 0x7b,
};
static const unsigned char kat621_retbits[] = {
    0x18, 0x1c, 0x99, 0x5f, 0x85, 0x92, 0x38, 0x4d, 0xfb, 0x56, 0x5a, 0xeb,
    0x11, 0x49, 0x44, 0x2b, 0x87, 0x1b, 0xe2, 0x9a, 0xaa, 0x32, 0x40, 0x47,
    0xb1, 0x25, 0x64, 0x62, 0x2b, 0xc1, 0xae, 0x7d, 0x38, 0x2c, 0x06, 0xaa,
    0xb3, 0xb2, 0xaa, 0x6d, 0x3c, 0xe1, 0x51, 0xaa, 0x17, 0x11, 0xf1, 0x6d,
    0x58, 0x7a, 0x03, 0x66, 0x23, 0x63, 0x0f, 0xef, 0xbb, 0x69, 0xcc, 0x15,
    0x38, 0xb4, 0x45, 0x1a,
};
static const struct drbg_kat_no_reseed kat621_t = {
    5, kat621_entropyin, kat621_nonce, kat621_persstr,
    kat621_addin0, kat621_addin1, kat621_retbits
};
static const struct drbg_kat kat621 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat621_t
};

static const unsigned char kat622_entropyin[] = {
    0x92, 0xdf, 0x43, 0x3d, 0xef, 0x81, 0xe1, 0xd7, 0x29, 0x88, 0xd2, 0x95,
    0xa5, 0x6a, 0x67, 0x9d, 0x0e, 0x61, 0x5b, 0x55, 0x08, 0xc9, 0x6c, 0x75,
    0xad, 0x5d, 0x22, 0xd8, 0xca, 0xa5, 0x25, 0xbb,
};
static const unsigned char kat622_nonce[] = {
    0x71, 0xbd, 0xbd, 0x0b, 0x0b, 0xc5, 0xb1, 0xf0, 0x3e, 0xbe, 0xd3, 0x00,
    0x66, 0x76, 0x85, 0x86,
};
static const unsigned char kat622_persstr[] = {0};
static const unsigned char kat622_addin0[] = {
    0x06, 0x08, 0x30, 0xec, 0xbb, 0x10, 0xc3, 0xfc, 0x18, 0xe7, 0x8d, 0x0e,
    0x8f, 0x12, 0xa6, 0x51, 0x02, 0xd1, 0xff, 0x8b, 0x37, 0xf8, 0x99, 0xe5,
    0x44, 0x5b, 0x5a, 0x03, 0x63, 0xa1, 0x18, 0x4f,
};
static const unsigned char kat622_addin1[] = {
    0x62, 0x8c, 0x96, 0xa7, 0xf5, 0x62, 0x03, 0x4d, 0x05, 0x6a, 0x4f, 0xec,
    0x0b, 0xfa, 0x2e, 0x9f, 0x0c, 0xcb, 0x6b, 0xe9, 0xcd, 0x33, 0xc6, 0x22,
    0x41, 0xa5, 0x07, 0xfd, 0xda, 0x00, 0x7f, 0x81,
};
static const unsigned char kat622_retbits[] = {
    0x09, 0xc4, 0x50, 0x31, 0x89, 0x57, 0xe2, 0x9f, 0x09, 0xe1, 0x5b, 0xd3,
    0xb0, 0x1c, 0x1d, 0x50, 0x2b, 0xbe, 0x90, 0x96, 0xf4, 0x0a, 0xa9, 0x51,
    0x73, 0x3a, 0xb8, 0xf7, 0xc0, 0x36, 0x30, 0x53, 0xca, 0x45, 0xbf, 0x81,
    0x9a, 0x02, 0xf3, 0xe1, 0x9e, 0x8f, 0xd8, 0x0c, 0xc0, 0x47, 0x09, 0xce,
    0x07, 0xa6, 0xb5, 0x80, 0x9b, 0xa8, 0xf7, 0x48, 0x6e, 0x14, 0xab, 0x95,
    0xce, 0xd2, 0xe0, 0x87,
};
static const struct drbg_kat_no_reseed kat622_t = {
    6, kat622_entropyin, kat622_nonce, kat622_persstr,
    kat622_addin0, kat622_addin1, kat622_retbits
};
static const struct drbg_kat kat622 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat622_t
};

static const unsigned char kat623_entropyin[] = {
    0x39, 0x34, 0x15, 0xa3, 0x28, 0xe2, 0x10, 0x93, 0xbb, 0x34, 0xeb, 0xb0,
    0x81, 0x97, 0xd2, 0x13, 0x94, 0xe2, 0x6e, 0x43, 0xd8, 0x2c, 0x1d, 0x2a,
    0xbf, 0xbd, 0xd4, 0x6a, 0x16, 0x5c, 0x27, 0xa8,
};
static const unsigned char kat623_nonce[] = {
    0xae, 0xe4, 0x9e, 0x30, 0x28, 0xd4, 0x07, 0x58, 0xb3, 0xf4, 0x2f, 0xd6,
    0xb8, 0xe6, 0x8b, 0xef,
};
static const unsigned char kat623_persstr[] = {0};
static const unsigned char kat623_addin0[] = {
    0xd5, 0x3d, 0xc5, 0xa6, 0x12, 0xcd, 0x15, 0x4a, 0xde, 0x83, 0xaf, 0xc0,
    0x00, 0x93, 0x3a, 0xd8, 0x8a, 0xed, 0xb8, 0x24, 0xa0, 0x2e, 0x2f, 0xd9,
    0x7a, 0x86, 0x97, 0xbe, 0x56, 0xda, 0x48, 0xb1,
};
static const unsigned char kat623_addin1[] = {
    0xbd, 0xa3, 0x92, 0xc5, 0x79, 0xac, 0x3d, 0x33, 0xd6, 0xc7, 0xb9, 0x45,
    0x5d, 0x6b, 0x8e, 0x71, 0x27, 0x84, 0x9b, 0x5a, 0x18, 0x6b, 0xf0, 0x06,
    0x0a, 0x3a, 0xf1, 0xe8, 0x30, 0x1e, 0xaf, 0x67,
};
static const unsigned char kat623_retbits[] = {
    0x5c, 0x33, 0x1c, 0xcc, 0xbc, 0x93, 0x52, 0x2f, 0xc8, 0x3f, 0x4c, 0xe8,
    0xc2, 0x2f, 0xff, 0x78, 0x1e, 0xe6, 0xe7, 0xd3, 0x82, 0x14, 0x1c, 0xce,
    0x7f, 0x0e, 0xfe, 0xe5, 0x68, 0x35, 0xf7, 0x80, 0x26, 0x5a, 0x4b, 0xe5,
    0x3c, 0x54, 0x12, 0xf9, 0x80, 0x83, 0x2a, 0xed, 0x12, 0xce, 0x88, 0x7c,
    0xf1, 0x5d, 0x28, 0x1a, 0xaa, 0x64, 0x93, 0x3c, 0x51, 0x27, 0xe1, 0x44,
    0x20, 0x97, 0x90, 0x36,
};
static const struct drbg_kat_no_reseed kat623_t = {
    7, kat623_entropyin, kat623_nonce, kat623_persstr,
    kat623_addin0, kat623_addin1, kat623_retbits
};
static const struct drbg_kat kat623 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat623_t
};

static const unsigned char kat624_entropyin[] = {
    0xb7, 0xc3, 0xac, 0xfb, 0xfd, 0xff, 0x34, 0x42, 0xef, 0x1c, 0x79, 0xf4,
    0xac, 0x90, 0xc9, 0x92, 0x3c, 0x3f, 0x90, 0xd9, 0xbf, 0x1b, 0x02, 0xa3,
    0xca, 0x73, 0xb9, 0xb5, 0x60, 0xa6, 0x9e, 0x8c,
};
static const unsigned char kat624_nonce[] = {
    0x9e, 0xaf, 0xc9, 0x73, 0x76, 0x01, 0x47, 0xdb, 0x55, 0x05, 0x74, 0x5a,
    0x70, 0x4b, 0xa0, 0x56,
};
static const unsigned char kat624_persstr[] = {0};
static const unsigned char kat624_addin0[] = {
    0xb3, 0xaf, 0xa7, 0x63, 0xbd, 0xb6, 0xcf, 0x2f, 0xf7, 0x90, 0xa5, 0x27,
    0x1e, 0xe6, 0x9d, 0xd0, 0xd7, 0xb3, 0x55, 0x4b, 0x63, 0x77, 0x52, 0x6c,
    0x46, 0x8c, 0x50, 0xc1, 0x55, 0x05, 0xce, 0x7c,
};
static const unsigned char kat624_addin1[] = {
    0x53, 0x81, 0x09, 0xfe, 0xb9, 0x58, 0xe5, 0x2d, 0x31, 0xe8, 0xfb, 0xb2,
    0xc2, 0x6a, 0x10, 0x37, 0xfb, 0x11, 0xff, 0xb9, 0xc2, 0xb2, 0x55, 0x9d,
    0x18, 0xf8, 0x41, 0xa8, 0x37, 0xf8, 0xa6, 0x9d,
};
static const unsigned char kat624_retbits[] = {
    0xe3, 0xd3, 0x02, 0x19, 0xe1, 0xc4, 0xc9, 0x69, 0x04, 0x97, 0xce, 0xde,
    0xcf, 0xf7, 0xab, 0xb4, 0x25, 0x14, 0xe9, 0xfa, 0x05, 0x91, 0xb3, 0x66,
    0x9c, 0xf6, 0x70, 0x04, 0xc0, 0xe2, 0x01, 0x69, 0x12, 0x9c, 0x09, 0xc7,
    0xe6, 0x04, 0xf9, 0xac, 0x67, 0xcc, 0x22, 0x04, 0xe3, 0x29, 0x31, 0xca,
    0x19, 0x14, 0xae, 0x18, 0xb6, 0x52, 0xbf, 0xde, 0x98, 0xf5, 0x3e, 0xb4,
    0xc1, 0xd4, 0x06, 0xa3,
};
static const struct drbg_kat_no_reseed kat624_t = {
    8, kat624_entropyin, kat624_nonce, kat624_persstr,
    kat624_addin0, kat624_addin1, kat624_retbits
};
static const struct drbg_kat kat624 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat624_t
};

static const unsigned char kat625_entropyin[] = {
    0xd3, 0xea, 0xbe, 0x6b, 0xe4, 0xd1, 0x74, 0x69, 0xdd, 0x74, 0x87, 0x18,
    0x17, 0xe1, 0x74, 0x86, 0xe8, 0x98, 0xc8, 0xe6, 0x5f, 0x36, 0x9a, 0xd6,
    0xe3, 0x8b, 0x41, 0xaa, 0x22, 0xa5, 0xbd, 0x27,
};
static const unsigned char kat625_nonce[] = {
    0x06, 0x7d, 0x66, 0x10, 0x71, 0x7a, 0x8f, 0xfb, 0xf8, 0xdb, 0xf9, 0x40,
    0x5e, 0x7f, 0x15, 0xcf,
};
static const unsigned char kat625_persstr[] = {0};
static const unsigned char kat625_addin0[] = {
    0x91, 0x1a, 0xb9, 0xc5, 0x75, 0x3c, 0x5d, 0x70, 0x93, 0xaf, 0x43, 0x16,
    0xd7, 0xd5, 0xc0, 0x3a, 0x2b, 0x97, 0x1d, 0xb8, 0xcf, 0x7b, 0x0c, 0x58,
    0xcb, 0x7f, 0xd8, 0xd5, 0xb9, 0xd8, 0xe0, 0x1b,
};
static const unsigned char kat625_addin1[] = {
    0x49, 0x53, 0x5f, 0xc0, 0xba, 0x86, 0x3d, 0xaf, 0x3b, 0x29, 0xdd, 0x8e,
    0x8d, 0x56, 0xf2, 0x06, 0x50, 0x8a, 0xe7, 0xb0, 0xc3, 0x43, 0x7a, 0x41,
    0x84, 0x7a, 0xea, 0x77, 0x18, 0x4b, 0x3e, 0x13,
};
static const unsigned char kat625_retbits[] = {
    0xec, 0x4b, 0xe9, 0xf3, 0xab, 0x19, 0x67, 0xe5, 0xbf, 0x96, 0x3f, 0xe3,
    0xca, 0xc8, 0x73, 0xc6, 0xcd, 0x87, 0xd9, 0x1a, 0x8a, 0x9d, 0x20, 0xe6,
    0xfd, 0x5f, 0x64, 0x18, 0xa3, 0x97, 0x2b, 0xfc, 0x1b, 0x9f, 0xcd, 0xa3,
    0xab, 0x35, 0x9e, 0xb7, 0x7b, 0x85, 0x85, 0x2d, 0x78, 0xa2, 0xa4, 0x8f,
    0xca, 0x43, 0x6e, 0x3a, 0xdf, 0x1b, 0x45, 0x0f, 0x27, 0x04, 0x5e, 0xba,
    0x6b, 0x2f, 0x1d, 0x6d,
};
static const struct drbg_kat_no_reseed kat625_t = {
    9, kat625_entropyin, kat625_nonce, kat625_persstr,
    kat625_addin0, kat625_addin1, kat625_retbits
};
static const struct drbg_kat kat625 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat625_t
};

static const unsigned char kat626_entropyin[] = {
    0xf4, 0x6b, 0x8c, 0x14, 0x80, 0x4b, 0x70, 0x3b, 0x93, 0x87, 0x47, 0x65,
    0x10, 0xd8, 0xa6, 0x50, 0xc0, 0x32, 0xdf, 0xe9, 0x83, 0x7b, 0xc4, 0xac,
    0xc3, 0xac, 0x40, 0xfc, 0xab, 0xcd, 0x79, 0xc8,
};
static const unsigned char kat626_nonce[] = {
    0x6a, 0x06, 0x15, 0x2e, 0x43, 0x83, 0xb2, 0x12, 0x8d, 0x42, 0x92, 0xcd,
    0x24, 0xd6, 0xf2, 0xd6,
};
static const unsigned char kat626_persstr[] = {0};
static const unsigned char kat626_addin0[] = {
    0x7e, 0xce, 0x8c, 0x73, 0xb7, 0x0b, 0x80, 0xcc, 0x28, 0xf4, 0xd7, 0xb5,
    0xc1, 0x7e, 0xdd, 0xf8, 0x15, 0x0c, 0x62, 0x33, 0xe6, 0xa3, 0x27, 0x47,
    0x29, 0x66, 0x57, 0x86, 0xf3, 0x5f, 0x4b, 0xab,
};
static const unsigned char kat626_addin1[] = {
    0x9e, 0xcf, 0x34, 0x1b, 0x59, 0xc1, 0xe0, 0x23, 0x95, 0xda, 0xba, 0x3f,
    0xaa, 0xe6, 0xf4, 0x36, 0x60, 0xb9, 0x87, 0x91, 0xd5, 0x55, 0xab, 0xa1,
    0xd8, 0x10, 0x96, 0x15, 0x43, 0x2a, 0x7b, 0x81,
};
static const unsigned char kat626_retbits[] = {
    0xd2, 0xf8, 0x97, 0x6d, 0xdb, 0xe8, 0xc6, 0x1f, 0xb7, 0x20, 0x59, 0xac,
    0x6e, 0xfa, 0x8f, 0x2b, 0xc9, 0x68, 0x8b, 0x68, 0xe4, 0xbe, 0x46, 0x40,
    0xa2, 0x02, 0xb5, 0xdf, 0xa1, 0x1d, 0x75, 0x52, 0xd4, 0xf8, 0x9c, 0xe7,
    0xbc, 0x81, 0x34, 0xc9, 0x8d, 0xf5, 0x3c, 0xd0, 0x56, 0xf6, 0x33, 0x4f,
    0xcb, 0xac, 0xe2, 0xdc, 0xc3, 0xd6, 0x4b, 0xf3, 0x55, 0x3e, 0xd9, 0x05,
    0xff, 0xc4, 0x28, 0xbc,
};
static const struct drbg_kat_no_reseed kat626_t = {
    10, kat626_entropyin, kat626_nonce, kat626_persstr,
    kat626_addin0, kat626_addin1, kat626_retbits
};
static const struct drbg_kat kat626 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat626_t
};

static const unsigned char kat627_entropyin[] = {
    0x02, 0x7d, 0x70, 0x1b, 0x53, 0x1e, 0x61, 0x8c, 0xe1, 0xb3, 0x64, 0x91,
    0xb2, 0x5d, 0x96, 0x7d, 0x6d, 0x3d, 0xcd, 0xa0, 0x14, 0x5e, 0x9e, 0x56,
    0x49, 0xb3, 0x69, 0x8a, 0x37, 0x4a, 0x7b, 0x2e,
};
static const unsigned char kat627_nonce[] = {
    0x44, 0xa5, 0x09, 0xef, 0x42, 0x08, 0x66, 0x09, 0x8c, 0x3e, 0x0f, 0x4a,
    0xbf, 0xe7, 0x3f, 0x21,
};
static const unsigned char kat627_persstr[] = {0};
static const unsigned char kat627_addin0[] = {
    0x50, 0x52, 0xae, 0xed, 0x2c, 0x37, 0xd7, 0x6a, 0xbb, 0xe9, 0x33, 0x83,
    0x16, 0x7d, 0x25, 0x6c, 0xd1, 0xb4, 0x93, 0xa1, 0xb6, 0xb4, 0x24, 0xda,
    0x50, 0x40, 0x71, 0x97, 0xe9, 0x12, 0x19, 0xf5,
};
static const unsigned char kat627_addin1[] = {
    0xa7, 0xcb, 0x70, 0xc5, 0x29, 0xfd, 0xf6, 0x1a, 0xa3, 0x04, 0xcb, 0xba,
    0x1f, 0x39, 0x7a, 0xf4, 0x10, 0x7a, 0x6e, 0x41, 0x39, 0xa3, 0x60, 0x14,
    0xe5, 0x4c, 0x7a, 0x41, 0x1d, 0x87, 0x2d, 0x8f,
};
static const unsigned char kat627_retbits[] = {
    0xbd, 0x11, 0x62, 0x0a, 0xde, 0x95, 0x56, 0x9f, 0x4e, 0x1c, 0xef, 0x6d,
    0xa8, 0xd8, 0x64, 0x5d, 0xd8, 0x27, 0x48, 0x1e, 0xa4, 0x2e, 0xf9, 0x82,
    0xe2, 0x11, 0x76, 0x1b, 0xe4, 0xfa, 0x9a, 0xae, 0xe9, 0x2f, 0x30, 0xde,
    0xd2, 0x68, 0xaa, 0x7a, 0xc9, 0x27, 0x07, 0xd0, 0xd6, 0x48, 0xa2, 0xe2,
    0xae, 0xc5, 0x30, 0xa4, 0x84, 0x94, 0xd2, 0x69, 0x9b, 0x3a, 0x3f, 0xd7,
    0x39, 0xda, 0x20, 0x9a,
};
static const struct drbg_kat_no_reseed kat627_t = {
    11, kat627_entropyin, kat627_nonce, kat627_persstr,
    kat627_addin0, kat627_addin1, kat627_retbits
};
static const struct drbg_kat kat627 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat627_t
};

static const unsigned char kat628_entropyin[] = {
    0x73, 0x42, 0xd6, 0x94, 0xe0, 0x8b, 0xd0, 0xbb, 0x7b, 0xa9, 0x17, 0x11,
    0x87, 0x98, 0x33, 0x9e, 0x36, 0xb9, 0xb3, 0x0e, 0x1f, 0x1d, 0x03, 0x44,
    0xe6, 0xf8, 0x53, 0x2c, 0xd4, 0x2d, 0x58, 0xa8,
};
static const unsigned char kat628_nonce[] = {
    0xec, 0xea, 0xbd, 0x4c, 0x87, 0x43, 0x56, 0xb3, 0xb3, 0x0a, 0xfb, 0xad,
    0xdc, 0xe2, 0x32, 0xba,
};
static const unsigned char kat628_persstr[] = {0};
static const unsigned char kat628_addin0[] = {
    0x15, 0x83, 0x1e, 0x1f, 0xc3, 0x0e, 0xf5, 0x2c, 0x79, 0x05, 0x02, 0x67,
    0x24, 0x0b, 0x74, 0x64, 0xa7, 0x7f, 0xfb, 0x90, 0xaa, 0x56, 0x81, 0x38,
    0x37, 0x9a, 0x4e, 0x77, 0x70, 0x81, 0xe2, 0xbb,
};
static const unsigned char kat628_addin1[] = {
    0x47, 0x7c, 0xd2, 0xcb, 0x4c, 0x80, 0xf3, 0xd1, 0x68, 0x16, 0xce, 0x2a,
    0xf1, 0xc3, 0x92, 0x0f, 0x20, 0x17, 0xf2, 0xd4, 0xeb, 0x48, 0x2a, 0xd2,
    0xa5, 0x86, 0x8a, 0xc4, 0x0a, 0x0e, 0x31, 0xc2,
};
static const unsigned char kat628_retbits[] = {
    0x56, 0x66, 0x30, 0xa2, 0x2c, 0x26, 0xf9, 0x26, 0xf1, 0x3f, 0x30, 0xba,
    0xc8, 0xc2, 0xf9, 0x53, 0xde, 0xeb, 0x2a, 0xd9, 0x0f, 0xac, 0xd3, 0xcd,
    0xc8, 0x8b, 0x3f, 0x44, 0xfc, 0xee, 0x53, 0xfb, 0x1e, 0xcf, 0x13, 0xf3,
    0x78, 0x04, 0xad, 0x88, 0x9a, 0x70, 0x52, 0xb5, 0xfd, 0x7e, 0xe9, 0x28,
    0xa2, 0x59, 0x1f, 0x6f, 0xdb, 0xb9, 0x86, 0xfa, 0x38, 0xf3, 0x7c, 0xc8,
    0xe0, 0x2d, 0x4e, 0x9a,
};
static const struct drbg_kat_no_reseed kat628_t = {
    12, kat628_entropyin, kat628_nonce, kat628_persstr,
    kat628_addin0, kat628_addin1, kat628_retbits
};
static const struct drbg_kat kat628 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat628_t
};

static const unsigned char kat629_entropyin[] = {
    0xe9, 0x08, 0xe2, 0xa4, 0xa6, 0x44, 0x4b, 0x85, 0x26, 0xfc, 0x41, 0x83,
    0x5f, 0x52, 0xf9, 0x44, 0x6b, 0x72, 0xf5, 0x33, 0x1b, 0x32, 0x86, 0xa4,
    0x06, 0xc8, 0xd5, 0x40, 0x83, 0x26, 0xa8, 0x60,
};
static const unsigned char kat629_nonce[] = {
    0x71, 0xcf, 0x47, 0x0f, 0x85, 0xa8, 0x0e, 0x7b, 0x33, 0xc4, 0x13, 0x3d,
    0xb7, 0x7e, 0xe4, 0x8b,
};
static const unsigned char kat629_persstr[] = {0};
static const unsigned char kat629_addin0[] = {
    0x92, 0x9a, 0x61, 0xe3, 0x54, 0x4c, 0x94, 0x21, 0x8b, 0x13, 0x98, 0xc8,
    0xd6, 0x90, 0x4b, 0xde, 0x66, 0x2d, 0x12, 0x4a, 0x01, 0x3a, 0x54, 0xe4,
    0x82, 0xc2, 0x77, 0xd8, 0x5e, 0x0d, 0xc2, 0xae,
};
static const unsigned char kat629_addin1[] = {
    0x57, 0x14, 0xc6, 0x72, 0xc3, 0x91, 0xa3, 0x6a, 0x5b, 0x2f, 0x09, 0x9a,
    0xae, 0x8a, 0xaa, 0x16, 0x39, 0x28, 0x1d, 0xc0, 0xf5, 0x5b, 0x04, 0xc1,
    0x01, 0x86, 0x28, 0x23, 0x41, 0xbb, 0x44, 0xfc,
};
static const unsigned char kat629_retbits[] = {
    0xd4, 0x6a, 0xf2, 0x2d, 0xfd, 0x1b, 0xc5, 0x55, 0xe4, 0x2a, 0xc0, 0xec,
    0xc7, 0xec, 0x28, 0xc8, 0x52, 0xc8, 0xe3, 0x35, 0xac, 0xc7, 0x9c, 0x65,
    0xe1, 0x42, 0xfa, 0xf4, 0x43, 0x41, 0x53, 0x89, 0x64, 0x83, 0xbd, 0xd1,
    0xc6, 0x4d, 0x5d, 0xe2, 0xc8, 0x34, 0x45, 0x79, 0x76, 0x48, 0x37, 0x1f,
    0x59, 0xa3, 0x16, 0xc4, 0xe0, 0xd8, 0x37, 0xd6, 0xaa, 0x17, 0x54, 0x6f,
    0x8c, 0xda, 0x82, 0x87,
};
static const struct drbg_kat_no_reseed kat629_t = {
    13, kat629_entropyin, kat629_nonce, kat629_persstr,
    kat629_addin0, kat629_addin1, kat629_retbits
};
static const struct drbg_kat kat629 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat629_t
};

static const unsigned char kat630_entropyin[] = {
    0x6c, 0xf5, 0x51, 0x90, 0x3c, 0x86, 0xeb, 0x4a, 0xe6, 0xff, 0xde, 0xf3,
    0x5f, 0x57, 0xd6, 0xd8, 0xa7, 0x46, 0x17, 0xbd, 0x3d, 0x97, 0x90, 0xe6,
    0x4a, 0x2e, 0x76, 0xb9, 0xb0, 0x14, 0x05, 0x55,
};
static const unsigned char kat630_nonce[] = {
    0x34, 0x2e, 0x55, 0x01, 0x86, 0x20, 0x60, 0xe4, 0x97, 0x4b, 0xc8, 0x8e,
    0x7c, 0x63, 0x2c, 0x69,
};
static const unsigned char kat630_persstr[] = {0};
static const unsigned char kat630_addin0[] = {
    0x91, 0xad, 0x32, 0x67, 0x62, 0x6a, 0xe5, 0xc3, 0x60, 0xfd, 0xd7, 0xba,
    0xa5, 0xff, 0x88, 0x17, 0x8a, 0xe8, 0x4f, 0xf7, 0x8b, 0x69, 0x4e, 0x2f,
    0xbf, 0x90, 0x77, 0x9b, 0xf4, 0xeb, 0x89, 0x8b,
};
static const unsigned char kat630_addin1[] = {
    0xcc, 0xd9, 0x7e, 0xc3, 0x1b, 0x03, 0x0e, 0x73, 0x2e, 0xdb, 0xfc, 0xdf,
    0xb9, 0x84, 0x8c, 0x2b, 0x36, 0x73, 0xbd, 0x3b, 0x10, 0x71, 0x50, 0x26,
    0x58, 0xda, 0xa4, 0x60, 0xb2, 0x62, 0xfb, 0x1c,
};
static const unsigned char kat630_retbits[] = {
    0x37, 0x31, 0xcf, 0xe5, 0xeb, 0xbc, 0x5b, 0x05, 0xc1, 0x12, 0xe8, 0x8f,
    0x1b, 0x4c, 0xb3, 0xde, 0xee, 0xe4, 0xbc, 0xc9, 0xec, 0xd1, 0x63, 0x5e,
    0x9c, 0x85, 0x46, 0x92, 0x57, 0x15, 0x06, 0x61, 0xa5, 0xdb, 0xac, 0x2e,
    0x63, 0x4b, 0x8f, 0xb0, 0xde, 0x08, 0xb5, 0xcd, 0x5e, 0xee, 0x12, 0x65,
    0xf8, 0x2e, 0x92, 0x78, 0xdc, 0xcd, 0x77, 0xb1, 0xfe, 0xd2, 0x32, 0x3c,
    0x55, 0x44, 0x45, 0xc3,
};
static const struct drbg_kat_no_reseed kat630_t = {
    14, kat630_entropyin, kat630_nonce, kat630_persstr,
    kat630_addin0, kat630_addin1, kat630_retbits
};
static const struct drbg_kat kat630 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat630_t
};

static const unsigned char kat631_entropyin[] = {
    0x47, 0xd5, 0x70, 0xe3, 0xa0, 0xa2, 0x0c, 0x0a, 0x20, 0x10, 0x67, 0x3a,
    0x61, 0x31, 0xdc, 0xc3, 0x20, 0x26, 0x79, 0xeb, 0x06, 0xf3, 0xc1, 0xb8,
    0x2a, 0x71, 0x0e, 0xdf, 0x92, 0xd3, 0x77, 0x4b,
};
static const unsigned char kat631_nonce[] = {
    0x88, 0x1f, 0xa7, 0xfd, 0xfb, 0x70, 0xd1, 0x06, 0xb9, 0xb9, 0xeb, 0x02,
    0x54, 0xb0, 0xeb, 0x6b,
};
static const unsigned char kat631_persstr[] = {
    0x0f, 0x66, 0x78, 0x7e, 0xf9, 0xb9, 0x03, 0x64, 0x51, 0x7e, 0x31, 0x51,
    0xb1, 0x58, 0xbe, 0xcd, 0x9d, 0xf4, 0x06, 0x0c, 0xd9, 0x2e, 0xc8, 0x8d,
    0xa3, 0xa6, 0xdd, 0x7b, 0x3b, 0x18, 0xe5, 0x4f,
};
static const unsigned char kat631_addin0[] = {0};
static const unsigned char kat631_addin1[] = {0};
static const unsigned char kat631_retbits[] = {
    0xe6, 0x4e, 0x8d, 0xca, 0xc9, 0x5e, 0x0e, 0x46, 0xf5, 0xe6, 0xc5, 0x57,
    0x1d, 0x07, 0x7b, 0x57, 0x4b, 0x1e, 0xab, 0xe4, 0x88, 0x0b, 0xbc, 0x0b,
    0xab, 0x8e, 0x08, 0xe2, 0x14, 0x80, 0x51, 0x44, 0x11, 0x65, 0xc3, 0x05,
    0xfc, 0x09, 0xd6, 0x07, 0x65, 0x19, 0x03, 0x46, 0xaf, 0x27, 0xa0, 0xdf,
    0x81, 0x56, 0x53, 0xe8, 0x1f, 0x78, 0x2a, 0xb7, 0xfe, 0xe5, 0x5d, 0xad,
    0x23, 0xec, 0x51, 0xd1,
};
static const struct drbg_kat_no_reseed kat631_t = {
    0, kat631_entropyin, kat631_nonce, kat631_persstr,
    kat631_addin0, kat631_addin1, kat631_retbits
};
static const struct drbg_kat kat631 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat631_t
};

static const unsigned char kat632_entropyin[] = {
    0x21, 0xcf, 0x7b, 0x1f, 0x01, 0x49, 0x95, 0xff, 0xe7, 0xfe, 0x84, 0x54,
    0x3f, 0x3e, 0x9a, 0x75, 0xcb, 0x3f, 0x99, 0x85, 0x1c, 0xf2, 0x1c, 0x4a,
    0xbb, 0xdc, 0x38, 0x73, 0x30, 0xd5, 0xc7, 0xe9,
};
static const unsigned char kat632_nonce[] = {
    0x49, 0xa6, 0xee, 0xa4, 0x26, 0x3e, 0xe1, 0xf5, 0xd4, 0x61, 0x90, 0x7d,
    0xc5, 0x8b, 0x44, 0xfb,
};
static const unsigned char kat632_persstr[] = {
    0x14, 0xd5, 0x39, 0x75, 0xf8, 0x52, 0xbc, 0xc9, 0xa1, 0xc5, 0xec, 0x9f,
    0x48, 0x25, 0xa0, 0x47, 0x21, 0xec, 0xfd, 0x87, 0xf2, 0xad, 0xef, 0x09,
    0x9a, 0x5b, 0x88, 0xe2, 0x7d, 0x77, 0x7b, 0x03,
};
static const unsigned char kat632_addin0[] = {0};
static const unsigned char kat632_addin1[] = {0};
static const unsigned char kat632_retbits[] = {
    0xa2, 0x6c, 0x99, 0x05, 0xc9, 0xae, 0x13, 0x8d, 0x94, 0x8b, 0xe7, 0x3c,
    0x42, 0x71, 0xe7, 0xe0, 0xda, 0xa2, 0x31, 0x61, 0xbc, 0x65, 0x95, 0x15,
    0x48, 0x81, 0xae, 0x60, 0x53, 0x59, 0x9a, 0x21, 0xaa, 0x97, 0xe5, 0x7f,
    0x3c, 0xe3, 0x4d, 0x30, 0xf6, 0x96, 0x47, 0xe9, 0x70, 0xe7, 0x82, 0x70,
    0x39, 0x93, 0x26, 0x15, 0xd9, 0x70, 0xb4, 0x75, 0x75, 0x96, 0x4c, 0xeb,
    0x8f, 0x7a, 0x43, 0x7d,
};
static const struct drbg_kat_no_reseed kat632_t = {
    1, kat632_entropyin, kat632_nonce, kat632_persstr,
    kat632_addin0, kat632_addin1, kat632_retbits
};
static const struct drbg_kat kat632 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat632_t
};

static const unsigned char kat633_entropyin[] = {
    0x17, 0x9e, 0x22, 0xe2, 0xf4, 0x95, 0xba, 0x9a, 0xe3, 0x52, 0xb9, 0x3c,
    0x83, 0x6b, 0x69, 0x33, 0xe2, 0x8a, 0x2a, 0x18, 0x4f, 0x89, 0x82, 0xc0,
    0x4e, 0x25, 0xe7, 0xee, 0xe6, 0x6f, 0x9f, 0x7c,
};
static const unsigned char kat633_nonce[] = {
    0xc1, 0x84, 0xe8, 0x42, 0xd2, 0x55, 0x5e, 0x56, 0x88, 0x8b, 0x7b, 0x75,
    0x18, 0x9e, 0x77, 0x75,
};
static const unsigned char kat633_persstr[] = {
    0x0f, 0xc7, 0x4e, 0x50, 0xa0, 0xfd, 0xa7, 0x9b, 0xb3, 0x1d, 0x5e, 0xbb,
    0x30, 0x8a, 0xa9, 0x7c, 0xcd, 0x6e, 0x6f, 0x17, 0xdc, 0xec, 0x14, 0x97,
    0x6f, 0x4e, 0x6b, 0x15, 0xca, 0x1b, 0xe3, 0x41,
};
static const unsigned char kat633_addin0[] = {0};
static const unsigned char kat633_addin1[] = {0};
static const unsigned char kat633_retbits[] = {
    0x47, 0x49, 0x76, 0x47, 0x11, 0x17, 0x44, 0xc7, 0x3d, 0xd2, 0xd0, 0x5c,
    0xf5, 0x7d, 0x65, 0xce, 0xae, 0x22, 0xe2, 0x2c, 0xac, 0x44, 0xb0, 0x98,
    0x18, 0x9c, 0x5a, 0x5c, 0x2f, 0x78, 0x1b, 0x74, 0x66, 0x9e, 0x6d, 0x66,
    0x9e, 0x38, 0xea, 0x8e, 0x5b, 0x46, 0x60, 0xe0, 0x4c, 0x04, 0x01, 0xc4,
    0xa4, 0xe6, 0x4c, 0x33, 0x1d, 0x79, 0x6d, 0x19, 0xb7, 0x35, 0x0a, 0x6a,
    0x3e, 0x46, 0x19, 0xfc,
};
static const struct drbg_kat_no_reseed kat633_t = {
    2, kat633_entropyin, kat633_nonce, kat633_persstr,
    kat633_addin0, kat633_addin1, kat633_retbits
};
static const struct drbg_kat kat633 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat633_t
};

static const unsigned char kat634_entropyin[] = {
    0x60, 0xbf, 0x6d, 0x95, 0x73, 0xea, 0x93, 0x98, 0x07, 0x4c, 0x3d, 0x6e,
    0x04, 0xe0, 0xe8, 0x22, 0xf0, 0xee, 0x95, 0xb6, 0x7d, 0xd2, 0x55, 0x59,
    0x88, 0x12, 0xe5, 0x33, 0x6a, 0xcc, 0x23, 0x36,
};
static const unsigned char kat634_nonce[] = {
    0xad, 0x27, 0xd7, 0xb2, 0x74, 0xf3, 0xa2, 0x18, 0x9d, 0x27, 0xbc, 0x54,
    0x7d, 0x6a, 0xc4, 0x10,
};
static const unsigned char kat634_persstr[] = {
    0x58, 0x03, 0x5b, 0xae, 0x9b, 0xa6, 0x7b, 0x89, 0x0b, 0x89, 0x2e, 0x3a,
    0x97, 0x4e, 0x33, 0x1d, 0x99, 0xd1, 0x5c, 0x60, 0x75, 0x93, 0xed, 0x21,
    0xa5, 0x1d, 0xec, 0x71, 0x03, 0x4d, 0x14, 0x2c,
};
static const unsigned char kat634_addin0[] = {0};
static const unsigned char kat634_addin1[] = {0};
static const unsigned char kat634_retbits[] = {
    0x51, 0x8d, 0xfe, 0xa2, 0x50, 0xe1, 0x79, 0xe2, 0x73, 0x82, 0x83, 0xde,
    0x6b, 0xc2, 0x9a, 0x06, 0x9c, 0xd9, 0xf6, 0x79, 0x3c, 0xe1, 0x1e, 0xc8,
    0x01, 0xab, 0x2d, 0x32, 0x39, 0x14, 0xe1, 0x34, 0x55, 0x87, 0x6c, 0x4d,
    0xb8, 0x32, 0x8a, 0x5b, 0x9a, 0x78, 0xfd, 0x06, 0x1c, 0xad, 0x66, 0x23,
    0x2d, 0x10, 0x57, 0xe5, 0xd6, 0xa3, 0x85, 0xa2, 0x97, 0xad, 0xca, 0x9c,
    0xdc, 0xa5, 0x7c, 0xaa,
};
static const struct drbg_kat_no_reseed kat634_t = {
    3, kat634_entropyin, kat634_nonce, kat634_persstr,
    kat634_addin0, kat634_addin1, kat634_retbits
};
static const struct drbg_kat kat634 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat634_t
};

static const unsigned char kat635_entropyin[] = {
    0xd3, 0x9f, 0xb3, 0x56, 0xf6, 0x6c, 0xea, 0xb8, 0x1c, 0x4c, 0xda, 0xd8,
    0x9e, 0xe2, 0xe6, 0x65, 0x7f, 0x8b, 0x36, 0xe8, 0x45, 0xd6, 0xb8, 0xb7,
    0x53, 0x0a, 0x61, 0x88, 0xc4, 0xc2, 0x6a, 0x8c,
};
static const unsigned char kat635_nonce[] = {
    0xb1, 0x09, 0x22, 0xb4, 0xa3, 0x08, 0x6e, 0x4a, 0x48, 0x4b, 0x50, 0x39,
    0x99, 0x6a, 0x54, 0xb4,
};
static const unsigned char kat635_persstr[] = {
    0x04, 0xdf, 0x5d, 0x29, 0xdc, 0x5c, 0x07, 0x93, 0xfe, 0x5b, 0x4a, 0xb3,
    0xda, 0x3e, 0xf7, 0xd2, 0x64, 0xc4, 0xcd, 0x67, 0x5f, 0xb0, 0x6b, 0xd2,
    0x1a, 0xc2, 0x1d, 0x7c, 0x77, 0xa2, 0xce, 0x73,
};
static const unsigned char kat635_addin0[] = {0};
static const unsigned char kat635_addin1[] = {0};
static const unsigned char kat635_retbits[] = {
    0x0f, 0x2c, 0x2d, 0xbe, 0x5d, 0xa6, 0x67, 0x42, 0xe9, 0xaa, 0xaa, 0x48,
    0x61, 0x2a, 0x0e, 0x07, 0xdd, 0x69, 0x14, 0x94, 0x2c, 0x52, 0xe8, 0x7d,
    0x32, 0x66, 0x15, 0x00, 0x13, 0xaa, 0xc3, 0x13, 0xde, 0x17, 0x17, 0x08,
    0x8e, 0x01, 0xb9, 0x3d, 0xd0, 0xd8, 0xc6, 0xab, 0xd0, 0xc5, 0xd6, 0x3d,
    0x56, 0x49, 0x51, 0x40, 0x45, 0x8c, 0x4a, 0x98, 0x0a, 0xb4, 0xff, 0x7d,
    0x98, 0x9e, 0x00, 0xe4,
};
static const struct drbg_kat_no_reseed kat635_t = {
    4, kat635_entropyin, kat635_nonce, kat635_persstr,
    kat635_addin0, kat635_addin1, kat635_retbits
};
static const struct drbg_kat kat635 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat635_t
};

static const unsigned char kat636_entropyin[] = {
    0x9a, 0x1a, 0x99, 0xf8, 0xf2, 0xbb, 0x88, 0xa7, 0x45, 0xbb, 0x24, 0xcb,
    0xcd, 0x4a, 0xc3, 0xaa, 0xbc, 0xf9, 0xad, 0x96, 0x37, 0x05, 0xdd, 0xee,
    0x88, 0x7c, 0xa1, 0x79, 0x2a, 0x1f, 0xa9, 0xf0,
};
static const unsigned char kat636_nonce[] = {
    0xa9, 0x10, 0x06, 0x54, 0xab, 0xc3, 0xac, 0xd6, 0xfb, 0x0c, 0x34, 0x44,
    0x04, 0xed, 0xb9, 0x74,
};
static const unsigned char kat636_persstr[] = {
    0x2f, 0xaa, 0xd0, 0xea, 0x6e, 0xb2, 0xb5, 0xbe, 0xfe, 0x02, 0x41, 0x19,
    0x20, 0x3b, 0xf7, 0x9c, 0x13, 0xed, 0x1f, 0xed, 0xac, 0xf5, 0x35, 0x88,
    0x10, 0xf4, 0xf8, 0x70, 0x17, 0x5e, 0x08, 0xc1,
};
static const unsigned char kat636_addin0[] = {0};
static const unsigned char kat636_addin1[] = {0};
static const unsigned char kat636_retbits[] = {
    0x27, 0x98, 0xf2, 0xe4, 0xb6, 0x6b, 0xca, 0x62, 0x12, 0xe1, 0x05, 0x5d,
    0x10, 0x0f, 0xfa, 0x4b, 0xac, 0x73, 0x3b, 0x05, 0xdc, 0xa1, 0x9a, 0x36,
    0x02, 0xfc, 0xd5, 0xe6, 0x84, 0x87, 0x5a, 0x7c, 0x58, 0xc1, 0xe7, 0xa8,
    0x94, 0x1a, 0x77, 0x32, 0x12, 0x4d, 0xaf, 0x13, 0xf9, 0x7c, 0x57, 0xcd,
    0x7d, 0x14, 0xe4, 0x03, 0x7a, 0xbf, 0x2f, 0x89, 0x10, 0x28, 0x1d, 0xbf,
    0xdf, 0x91, 0x13, 0xbb,
};
static const struct drbg_kat_no_reseed kat636_t = {
    5, kat636_entropyin, kat636_nonce, kat636_persstr,
    kat636_addin0, kat636_addin1, kat636_retbits
};
static const struct drbg_kat kat636 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat636_t
};

static const unsigned char kat637_entropyin[] = {
    0x9f, 0x63, 0xa3, 0x28, 0x2c, 0xd1, 0x8a, 0x09, 0xa9, 0x1a, 0x50, 0x08,
    0x6c, 0xe0, 0xe8, 0xd9, 0x1b, 0xe0, 0x42, 0x61, 0xda, 0x62, 0x38, 0x75,
    0xb1, 0x5f, 0x9b, 0xac, 0x5a, 0xfe, 0x7c, 0x6d,
};
static const unsigned char kat637_nonce[] = {
    0xef, 0x81, 0xc6, 0xf3, 0x67, 0xff, 0xbd, 0x5d, 0x97, 0xcd, 0xdf, 0x81,
    0x9a, 0x99, 0xf9, 0x96,
};
static const unsigned char kat637_persstr[] = {
    0xf1, 0x8f, 0xc5, 0x00, 0xcc, 0x26, 0x0b, 0xeb, 0x53, 0xaa, 0x6a, 0x1d,
    0x1e, 0xbd, 0x5b, 0xaf, 0x6e, 0xc3, 0xad, 0x79, 0x57, 0x8a, 0x71, 0xcd,
    0xb2, 0x76, 0xc6, 0x9d, 0x67, 0x50, 0x11, 0xcc,
};
static const unsigned char kat637_addin0[] = {0};
static const unsigned char kat637_addin1[] = {0};
static const unsigned char kat637_retbits[] = {
    0x47, 0x8f, 0xb3, 0x11, 0x2c, 0x99, 0x36, 0x3b, 0x7c, 0x6e, 0x5d, 0x78,
    0x1c, 0xcf, 0xc2, 0x05, 0x66, 0x94, 0xdd, 0xdb, 0x70, 0xd5, 0xe4, 0xe4,
    0x57, 0xf2, 0x2c, 0xfb, 0x15, 0x4d, 0x1d, 0x01, 0x73, 0xc0, 0x9d, 0xaa,
    0x88, 0x06, 0xba, 0x9c, 0x7c, 0xef, 0xcd, 0xd7, 0xa7, 0x15, 0xb3, 0xb5,
    0x39, 0x1b, 0xe1, 0x0c, 0x37, 0x28, 0xf3, 0xaf, 0x25, 0xab, 0x2a, 0x4b,
    0x8a, 0x8a, 0x5c, 0xaa,
};
static const struct drbg_kat_no_reseed kat637_t = {
    6, kat637_entropyin, kat637_nonce, kat637_persstr,
    kat637_addin0, kat637_addin1, kat637_retbits
};
static const struct drbg_kat kat637 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat637_t
};

static const unsigned char kat638_entropyin[] = {
    0x10, 0x15, 0xc9, 0xc0, 0xd0, 0x6e, 0x6a, 0xc7, 0xac, 0xbb, 0xab, 0x0d,
    0x2d, 0xd1, 0xad, 0xea, 0x75, 0x60, 0x57, 0xcc, 0x39, 0x38, 0xef, 0x87,
    0x85, 0xe3, 0xd5, 0x1e, 0x5a, 0xfc, 0x01, 0x8e,
};
static const unsigned char kat638_nonce[] = {
    0x95, 0x1d, 0x78, 0x61, 0xd8, 0x40, 0x9b, 0x0a, 0x62, 0xb1, 0x8c, 0x43,
    0xbb, 0xb6, 0x55, 0x23,
};
static const unsigned char kat638_persstr[] = {
    0xe7, 0x6a, 0x06, 0x02, 0x4d, 0x00, 0xbe, 0x09, 0xc6, 0xa2, 0x2b, 0x78,
    0xb7, 0x4c, 0xad, 0xba, 0xf3, 0x68, 0x92, 0x0b, 0x1f, 0x74, 0xa4, 0x41,
    0xd5, 0x10, 0xae, 0xe0, 0x2c, 0xae, 0x72, 0x1d,
};
static const unsigned char kat638_addin0[] = {0};
static const unsigned char kat638_addin1[] = {0};
static const unsigned char kat638_retbits[] = {
    0xe1, 0xad, 0x22, 0x2c, 0x7f, 0x45, 0xd8, 0x2d, 0x15, 0x93, 0x09, 0xfe,
    0xa1, 0x5e, 0x81, 0xbf, 0x9f, 0x14, 0x8c, 0xac, 0xd5, 0x7b, 0x52, 0xf5,
    0xe8, 0xfd, 0x3f, 0xaf, 0x73, 0xd9, 0xb8, 0x32, 0xcc, 0x80, 0x3d, 0x4f,
    0x36, 0xe3, 0xd9, 0xaf, 0xb0, 0x3c, 0xba, 0xcb, 0xf7, 0x2d, 0x6b, 0x29,
    0x7a, 0x01, 0xd0, 0xf0, 0x0b, 0x82, 0x5f, 0x64, 0xe6, 0x76, 0xa7, 0x01,
    0x81, 0x59, 0xda, 0x2b,
};
static const struct drbg_kat_no_reseed kat638_t = {
    7, kat638_entropyin, kat638_nonce, kat638_persstr,
    kat638_addin0, kat638_addin1, kat638_retbits
};
static const struct drbg_kat kat638 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat638_t
};

static const unsigned char kat639_entropyin[] = {
    0xca, 0xd9, 0x36, 0xac, 0xd1, 0xef, 0x96, 0x85, 0xf6, 0x6b, 0xef, 0x62,
    0xa8, 0xbf, 0x4f, 0x91, 0x9e, 0x18, 0x02, 0x4f, 0xda, 0x5c, 0x63, 0xc5,
    0x7c, 0xed, 0xb7, 0x02, 0xdf, 0x3e, 0x94, 0xa9,
};
static const unsigned char kat639_nonce[] = {
    0x85, 0xd3, 0x23, 0x8a, 0xfa, 0x86, 0xc6, 0x54, 0x70, 0x2d, 0xfa, 0x79,
    0x3e, 0x11, 0xd3, 0xf6,
};
static const unsigned char kat639_persstr[] = {
    0xd6, 0x32, 0x4b, 0x2a, 0x3e, 0x38, 0x94, 0x6c, 0x7d, 0xf7, 0xc6, 0xe8,
    0xd3, 0x01, 0xf6, 0xb5, 0xe6, 0x3e, 0x55, 0x53, 0x5c, 0x8a, 0xe0, 0xbe,
    0xd6, 0x77, 0xce, 0x0e, 0xa8, 0x6e, 0xa4, 0x99,
};
static const unsigned char kat639_addin0[] = {0};
static const unsigned char kat639_addin1[] = {0};
static const unsigned char kat639_retbits[] = {
    0x30, 0x03, 0x46, 0x83, 0x65, 0xb5, 0xec, 0xcd, 0xd6, 0x42, 0xd3, 0xd4,
    0x30, 0xd4, 0xc4, 0xd4, 0xc0, 0x89, 0xc5, 0xdf, 0xb7, 0xdd, 0x4a, 0xda,
    0xfb, 0xdb, 0x46, 0x31, 0x61, 0x08, 0x05, 0xe1, 0x58, 0x7d, 0x35, 0x5b,
    0x56, 0xd6, 0x59, 0xc5, 0x9f, 0xa0, 0x12, 0xe4, 0x90, 0xbe, 0xae, 0x5f,
    0xc7, 0x67, 0xbd, 0xbb, 0x34, 0xb3, 0x1a, 0x5b, 0x30, 0x4c, 0x23, 0xdc,
    0x27, 0xb4, 0xe9, 0xe0,
};
static const struct drbg_kat_no_reseed kat639_t = {
    8, kat639_entropyin, kat639_nonce, kat639_persstr,
    kat639_addin0, kat639_addin1, kat639_retbits
};
static const struct drbg_kat kat639 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat639_t
};

static const unsigned char kat640_entropyin[] = {
    0x64, 0x8e, 0x4e, 0x48, 0x4b, 0x26, 0xfc, 0xb3, 0x30, 0xbf, 0x46, 0xa9,
    0xee, 0xe4, 0x18, 0xcb, 0x46, 0xbd, 0xb5, 0x66, 0x4d, 0xc0, 0xb5, 0xa8,
    0x75, 0xee, 0x8e, 0x58, 0x45, 0x47, 0x6a, 0x56,
};
static const unsigned char kat640_nonce[] = {
    0xba, 0x61, 0x0a, 0x74, 0xb6, 0x64, 0x72, 0xc6, 0x9f, 0x81, 0xad, 0x0f,
    0xbf, 0xe8, 0xb3, 0x9d,
};
static const unsigned char kat640_persstr[] = {
    0x32, 0x2f, 0x58, 0x43, 0x29, 0xdf, 0xc1, 0xf6, 0x43, 0xf3, 0x59, 0xec,
    0xfb, 0x98, 0x33, 0xc4, 0xaa, 0x28, 0x66, 0x67, 0xf2, 0xff, 0xe6, 0xbe,
    0xd9, 0xc4, 0xc6, 0xe9, 0x12, 0x38, 0x03, 0x89,
};
static const unsigned char kat640_addin0[] = {0};
static const unsigned char kat640_addin1[] = {0};
static const unsigned char kat640_retbits[] = {
    0x7f, 0x84, 0x98, 0x28, 0x2c, 0x32, 0xe6, 0xb5, 0x35, 0x12, 0x15, 0xf9,
    0x0e, 0x58, 0xdd, 0x3d, 0x5f, 0xf5, 0xcb, 0x7b, 0x61, 0xc3, 0xfd, 0x9e,
    0x2b, 0x93, 0xdc, 0xbd, 0xc5, 0xef, 0x85, 0xba, 0x9e, 0xda, 0x9f, 0xc5,
    0x34, 0x56, 0x5f, 0x77, 0xb7, 0x8b, 0x42, 0x0e, 0x24, 0xcf, 0xc9, 0x5d,
    0x7d, 0x55, 0x5b, 0x74, 0x3a, 0x80, 0xb2, 0x19, 0x45, 0x08, 0xc3, 0x1e,
    0x4c, 0x30, 0xba, 0xe3,
};
static const struct drbg_kat_no_reseed kat640_t = {
    9, kat640_entropyin, kat640_nonce, kat640_persstr,
    kat640_addin0, kat640_addin1, kat640_retbits
};
static const struct drbg_kat kat640 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat640_t
};

static const unsigned char kat641_entropyin[] = {
    0xc5, 0x2a, 0x30, 0x1c, 0x95, 0x25, 0x50, 0xa3, 0x1a, 0xb3, 0xf3, 0xd3,
    0xc7, 0x89, 0x19, 0xcc, 0x55, 0x09, 0xa5, 0xb9, 0x84, 0xc0, 0xa4, 0x7a,
    0x9c, 0xae, 0x9f, 0xe8, 0xdc, 0xf6, 0xce, 0x97,
};
static const unsigned char kat641_nonce[] = {
    0x87, 0xe2, 0x8a, 0xef, 0xf6, 0x6c, 0x45, 0xe4, 0x09, 0xf0, 0x2f, 0x99,
    0xd4, 0x59, 0xf6, 0x97,
};
static const unsigned char kat641_persstr[] = {
    0x95, 0x6e, 0x32, 0xd3, 0xbd, 0x82, 0xc6, 0x37, 0x9b, 0x2b, 0x7a, 0x7b,
    0x67, 0xfd, 0xe6, 0x58, 0x3f, 0x98, 0xd5, 0xbc, 0x2e, 0x22, 0x08, 0x2e,
    0x79, 0x3b, 0x95, 0xe0, 0xdc, 0xbb, 0x4f, 0x94,
};
static const unsigned char kat641_addin0[] = {0};
static const unsigned char kat641_addin1[] = {0};
static const unsigned char kat641_retbits[] = {
    0x6d, 0x0f, 0x4a, 0x8c, 0xb3, 0x06, 0x1b, 0x6a, 0xf2, 0xf1, 0x5d, 0x4b,
    0xf0, 0xaf, 0xba, 0xa4, 0x62, 0x4c, 0x06, 0x3f, 0xd7, 0xde, 0xd3, 0x90,
    0x3d, 0xc9, 0x0f, 0xb3, 0x59, 0xf3, 0x50, 0xb9, 0xec, 0x31, 0xa1, 0x91,
    0x40, 0xbc, 0xc4, 0xd4, 0xea, 0xb4, 0x49, 0xa6, 0xce, 0x4e, 0x0a, 0xbe,
    0x6b, 0x1e, 0x81, 0x91, 0x64, 0xd3, 0x98, 0x84, 0x42, 0x0a, 0xa2, 0x0c,
    0x85, 0x40, 0x8c, 0x9f,
};
static const struct drbg_kat_no_reseed kat641_t = {
    10, kat641_entropyin, kat641_nonce, kat641_persstr,
    kat641_addin0, kat641_addin1, kat641_retbits
};
static const struct drbg_kat kat641 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat641_t
};

static const unsigned char kat642_entropyin[] = {
    0xb5, 0x2b, 0x8b, 0x33, 0x34, 0xbf, 0xf2, 0x30, 0xd0, 0x33, 0xc8, 0x9c,
    0xe5, 0xf7, 0x4d, 0x21, 0xc1, 0xab, 0x07, 0xf1, 0x7a, 0xce, 0xc3, 0x09,
    0x4f, 0x54, 0x37, 0xe5, 0x09, 0x13, 0x9d, 0x85,
};
static const unsigned char kat642_nonce[] = {
    0x61, 0x7f, 0x1f, 0x58, 0x40, 0xa4, 0x85, 0xd0, 0x42, 0xa5, 0xa7, 0xbc,
    0x55, 0xca, 0x37, 0x46,
};
static const unsigned char kat642_persstr[] = {
    0xa6, 0xf4, 0x0a, 0x52, 0xf7, 0x92, 0x30, 0x37, 0xa9, 0x39, 0x1d, 0x45,
    0x09, 0x52, 0xaf, 0x3b, 0x4c, 0x82, 0x23, 0xe9, 0xc1, 0x9e, 0x24, 0xd2,
    0xba, 0x11, 0xe2, 0x0d, 0x15, 0x96, 0x4d, 0x34,
};
static const unsigned char kat642_addin0[] = {0};
static const unsigned char kat642_addin1[] = {0};
static const unsigned char kat642_retbits[] = {
    0xce, 0xd9, 0xf3, 0xe3, 0xfd, 0xe7, 0x50, 0x9f, 0x5c, 0x1c, 0x46, 0xd5,
    0xa3, 0xbf, 0xff, 0xe6, 0x7c, 0xac, 0xff, 0xb1, 0x03, 0xfe, 0x4f, 0xac,
    0xfb, 0x45, 0xb6, 0x6e, 0x53, 0xd5, 0x94, 0xcf, 0x4e, 0x19, 0x28, 0x44,
    0xa1, 0xba, 0xc8, 0x18, 0x7c, 0x65, 0xcb, 0xc9, 0xcd, 0x84, 0x3f, 0x23,
    0x83, 0x51, 0x0d, 0x8f, 0x65, 0x76, 0xc3, 0x9d, 0x7f, 0x37, 0xd0, 0x00,
    0xb8, 0xce, 0x75, 0x21,
};
static const struct drbg_kat_no_reseed kat642_t = {
    11, kat642_entropyin, kat642_nonce, kat642_persstr,
    kat642_addin0, kat642_addin1, kat642_retbits
};
static const struct drbg_kat kat642 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat642_t
};

static const unsigned char kat643_entropyin[] = {
    0xa5, 0x79, 0x37, 0x42, 0x88, 0xc8, 0xa2, 0xde, 0x14, 0x4f, 0x46, 0x0c,
    0xb0, 0xdc, 0x2f, 0xff, 0xbe, 0xc7, 0x0e, 0xb7, 0x59, 0xc5, 0xbf, 0x17,
    0x12, 0xa4, 0x5b, 0x5a, 0xbf, 0x63, 0x04, 0x5c,
};
static const unsigned char kat643_nonce[] = {
    0xbd, 0xb7, 0x03, 0xe6, 0x46, 0x66, 0x6d, 0x1b, 0x21, 0x63, 0xc2, 0x5f,
    0xd5, 0x63, 0x02, 0x9d,
};
static const unsigned char kat643_persstr[] = {
    0x51, 0x28, 0x52, 0x7d, 0x90, 0x77, 0xbd, 0x74, 0xb4, 0x4a, 0xb7, 0xde,
    0xad, 0xeb, 0x3f, 0x1f, 0xc2, 0x09, 0x9a, 0xb7, 0x11, 0xed, 0x44, 0xa6,
    0x12, 0xcb, 0xd6, 0x7c, 0x8e, 0x94, 0x62, 0x73,
};
static const unsigned char kat643_addin0[] = {0};
static const unsigned char kat643_addin1[] = {0};
static const unsigned char kat643_retbits[] = {
    0x8f, 0xba, 0x16, 0x08, 0x67, 0xf2, 0x57, 0xb9, 0x0d, 0x76, 0xec, 0x68,
    0x0f, 0x3e, 0x16, 0x90, 0x4a, 0x8d, 0x14, 0x43, 0xb0, 0x28, 0x33, 0xea,
    0xa7, 0x8a, 0x33, 0x00, 0x71, 0x37, 0xc4, 0x22, 0x66, 0x91, 0x33, 0xd7,
    0xb3, 0xde, 0x20, 0x35, 0x83, 0x0a, 0xf7, 0x94, 0x00, 0x73, 0x0d, 0xed,
    0x71, 0x49, 0x54, 0x6d, 0x56, 0xa9, 0x0f, 0x20, 0x4e, 0x86, 0x47, 0x7c,
    0x42, 0xfe, 0xc1, 0xa9,
};
static const struct drbg_kat_no_reseed kat643_t = {
    12, kat643_entropyin, kat643_nonce, kat643_persstr,
    kat643_addin0, kat643_addin1, kat643_retbits
};
static const struct drbg_kat kat643 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat643_t
};

static const unsigned char kat644_entropyin[] = {
    0x1f, 0x44, 0x48, 0x91, 0xec, 0x1d, 0x5d, 0x06, 0x0b, 0xd1, 0xb0, 0x37,
    0x36, 0x3b, 0xd7, 0x50, 0xdf, 0x90, 0x17, 0xb2, 0x74, 0x79, 0xe2, 0x11,
    0x85, 0xb9, 0x49, 0x7f, 0x6b, 0xa0, 0x31, 0x1f,
};
static const unsigned char kat644_nonce[] = {
    0x73, 0xb5, 0x49, 0x79, 0x76, 0xd6, 0x71, 0xa2, 0x60, 0xb3, 0x4f, 0xa3,
    0x14, 0xcc, 0xdd, 0x74,
};
static const unsigned char kat644_persstr[] = {
    0xd7, 0x0d, 0x63, 0xe5, 0x8d, 0x23, 0x12, 0x3e, 0xba, 0x0e, 0x5a, 0x6b,
    0xde, 0x07, 0x79, 0xa8, 0x86, 0x66, 0x83, 0x36, 0x9e, 0xe1, 0xc1, 0xd1,
    0xdd, 0xe8, 0x1a, 0x14, 0x67, 0x17, 0xd8, 0xbc,
};
static const unsigned char kat644_addin0[] = {0};
static const unsigned char kat644_addin1[] = {0};
static const unsigned char kat644_retbits[] = {
    0x43, 0x56, 0xaf, 0xd0, 0x40, 0x81, 0xe8, 0xc0, 0xf4, 0xc4, 0x46, 0xa6,
    0x22, 0xcd, 0x3f, 0x04, 0x9e, 0x7e, 0x6e, 0x8f, 0x90, 0x6c, 0x07, 0x60,
    0x0d, 0xa0, 0x67, 0x1e, 0xea, 0xe8, 0xf0, 0xa7, 0x32, 0xf4, 0x5f, 0x7b,
    0x99, 0xee, 0x98, 0xc1, 0xb0, 0x61, 0x33, 0x5d, 0x0d, 0x58, 0x96, 0xda,
    0x1e, 0x1c, 0x08, 0x1e, 0x2a, 0x6f, 0x25, 0xc5, 0x65, 0xfc, 0x47, 0xdf,
    0xff, 0xeb, 0xbe, 0x98,
};
static const struct drbg_kat_no_reseed kat644_t = {
    13, kat644_entropyin, kat644_nonce, kat644_persstr,
    kat644_addin0, kat644_addin1, kat644_retbits
};
static const struct drbg_kat kat644 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat644_t
};

static const unsigned char kat645_entropyin[] = {
    0x75, 0xd5, 0xb7, 0x82, 0xc5, 0x34, 0x5c, 0xee, 0x69, 0xe0, 0x45, 0x84,
    0x47, 0x83, 0xba, 0x53, 0xa3, 0x15, 0x88, 0x22, 0x82, 0x74, 0x0a, 0x45,
    0xb4, 0xa0, 0x4f, 0xd9, 0xe7, 0x7e, 0xc1, 0xb6,
};
static const unsigned char kat645_nonce[] = {
    0x17, 0x5c, 0x36, 0xb3, 0x96, 0x47, 0x26, 0xca, 0xab, 0x9b, 0x5a, 0x12,
    0x7e, 0xe9, 0xcd, 0x91,
};
static const unsigned char kat645_persstr[] = {
    0xc7, 0xb4, 0x5b, 0x62, 0xad, 0xab, 0xf3, 0x45, 0xdc, 0xc0, 0x18, 0xe2,
    0xea, 0xfe, 0x3e, 0xf2, 0x12, 0xa2, 0x53, 0xed, 0xc9, 0x46, 0x0f, 0x12,
    0xa7, 0xc0, 0x26, 0x0f, 0xe9, 0x59, 0xdf, 0x6c,
};
static const unsigned char kat645_addin0[] = {0};
static const unsigned char kat645_addin1[] = {0};
static const unsigned char kat645_retbits[] = {
    0x5c, 0x74, 0xc0, 0x46, 0xd2, 0x00, 0xce, 0xf4, 0x36, 0x13, 0xee, 0xb2,
    0xb0, 0x83, 0x58, 0x38, 0x60, 0x9c, 0x4a, 0xe2, 0x69, 0x78, 0x89, 0xd5,
    0x1c, 0x17, 0x9a, 0x27, 0x37, 0x1b, 0x74, 0x21, 0xd9, 0xd2, 0x8d, 0x03,
    0x8b, 0xd2, 0x2a, 0xaa, 0x7b, 0xe1, 0xe3, 0x1c, 0xa0, 0x9f, 0x3c, 0xf3,
    0xd3, 0xea, 0x1d, 0x3f, 0x18, 0x34, 0x0d, 0x9c, 0xc7, 0x3d, 0xd8, 0x41,
    0x51, 0x22, 0xca, 0xaa,
};
static const struct drbg_kat_no_reseed kat645_t = {
    14, kat645_entropyin, kat645_nonce, kat645_persstr,
    kat645_addin0, kat645_addin1, kat645_retbits
};
static const struct drbg_kat kat645 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 0, 64, &kat645_t
};

static const unsigned char kat646_entropyin[] = {
    0x50, 0x6b, 0xfe, 0x78, 0x5b, 0xd1, 0x7b, 0x7a, 0x2d, 0xec, 0x8a, 0xbb,
    0xe2, 0x02, 0xa2, 0x41, 0x40, 0x62, 0xb4, 0xc2, 0xff, 0x22, 0xaa, 0xc3,
    0x89, 0x01, 0x33, 0x80, 0x1c, 0x54, 0x96, 0x1f,
};
static const unsigned char kat646_nonce[] = {
    0x64, 0x88, 0x5c, 0x54, 0xfd, 0x46, 0x16, 0xe6, 0x0d, 0xab, 0x9c, 0x4a,
    0x42, 0x4c, 0xb2, 0x00,
};
static const unsigned char kat646_persstr[] = {
    0xdb, 0x1a, 0xab, 0xae, 0x13, 0x8e, 0x6b, 0xb9, 0xca, 0x30, 0xe7, 0xb1,
    0x07, 0x11, 0x00, 0x46, 0xad, 0x18, 0x8b, 0xef, 0x4a, 0x71, 0xc9, 0x0d,
    0x23, 0x29, 0xee, 0x42, 0x0e, 0xfb, 0x4b, 0x9d,
};
static const unsigned char kat646_addin0[] = {
    0x0e, 0x22, 0x4a, 0x4d, 0x7b, 0x8c, 0xa1, 0xff, 0xf0, 0x46, 0x56, 0xf9,
    0xf4, 0xb5, 0xb9, 0x57, 0x7f, 0xce, 0xfc, 0xa0, 0xc2, 0x83, 0x28, 0x76,
    0x77, 0xbb, 0x84, 0xb1, 0xc3, 0x08, 0x34, 0x96,
};
static const unsigned char kat646_addin1[] = {
    0x58, 0xac, 0xad, 0xc5, 0x4f, 0x21, 0x95, 0xef, 0x4d, 0x13, 0x53, 0x75,
    0x99, 0x47, 0xe6, 0xe5, 0x2d, 0xba, 0x26, 0x38, 0x04, 0x07, 0x76, 0xab,
    0x0b, 0xe3, 0xb6, 0x3a, 0x4b, 0x2d, 0x66, 0x3b,
};
static const unsigned char kat646_retbits[] = {
    0x9f, 0x54, 0x75, 0xa3, 0x95, 0x98, 0x8b, 0x36, 0xcc, 0x3c, 0x41, 0x58,
    0x72, 0x31, 0xf1, 0x8f, 0x23, 0x2f, 0xb3, 0x03, 0xcf, 0x82, 0xf2, 0x4c,
    0xbf, 0xe7, 0x95, 0x69, 0x68, 0x1f, 0x7f, 0x8d, 0xab, 0x8c, 0x7a, 0x58,
    0x86, 0x10, 0x6d, 0x53, 0x0f, 0xe7, 0x88, 0x88, 0x6f, 0x8e, 0x5d, 0x13,
    0x15, 0x71, 0x54, 0x84, 0xd1, 0x88, 0x2b, 0x1d, 0x0c, 0x24, 0x12, 0xe8,
    0x79, 0x6f, 0x27, 0x0f,
};
static const struct drbg_kat_no_reseed kat646_t = {
    0, kat646_entropyin, kat646_nonce, kat646_persstr,
    kat646_addin0, kat646_addin1, kat646_retbits
};
static const struct drbg_kat kat646 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat646_t
};

static const unsigned char kat647_entropyin[] = {
    0x52, 0xe7, 0xd8, 0xab, 0xd3, 0x0c, 0x4b, 0x14, 0x78, 0x67, 0x56, 0xe8,
    0x2d, 0xd7, 0xf8, 0x99, 0x07, 0x6e, 0x1c, 0xea, 0x07, 0xdc, 0x72, 0x2f,
    0x8e, 0x12, 0x16, 0x11, 0x41, 0xf6, 0xd9, 0xa5,
};
static const unsigned char kat647_nonce[] = {
    0xca, 0xb6, 0x8c, 0xe9, 0xde, 0xb7, 0xe5, 0x45, 0xe3, 0x3e, 0x5a, 0x27,
    0xc4, 0x87, 0x85, 0x97,
};
static const unsigned char kat647_persstr[] = {
    0x9a, 0xc3, 0xbf, 0x47, 0xf6, 0x30, 0x6a, 0x36, 0xee, 0x84, 0xed, 0x4e,
    0xe6, 0xae, 0xa8, 0xe1, 0xd7, 0xe8, 0xb1, 0x6b, 0x5c, 0x40, 0x7b, 0xd1,
    0x58, 0x3e, 0x7c, 0xb5, 0x2d, 0xa9, 0x12, 0x75,
};
static const unsigned char kat647_addin0[] = {
    0xd7, 0xbd, 0xd5, 0xcb, 0xbe, 0xfd, 0x1b, 0x4d, 0x0c, 0xdb, 0x32, 0x93,
    0x7f, 0xeb, 0x8d, 0x01, 0x9d, 0x50, 0x3c, 0xae, 0x80, 0xa5, 0x24, 0x24,
    0x95, 0x66, 0x55, 0x65, 0xf3, 0x2f, 0xc4, 0x87,
};
static const unsigned char kat647_addin1[] = {
    0x63, 0x61, 0xac, 0x7a, 0x3c, 0x20, 0x90, 0xbe, 0x66, 0xa4, 0x6f, 0xf8,
    0x29, 0xdf, 0x38, 0xff, 0x06, 0x3b, 0x2f, 0x9c, 0x53, 0x1c, 0x7e, 0x42,
    0x80, 0x30, 0x7e, 0xc4, 0x5c, 0x4f, 0xa0, 0xa6,
};
static const unsigned char kat647_retbits[] = {
    0x98, 0x34, 0xb9, 0xe1, 0x61, 0x8d, 0x5f, 0x01, 0xee, 0x90, 0x83, 0xee,
    0x89, 0xcc, 0xb3, 0x3c, 0x18, 0x59, 0x6e, 0x67, 0x5e, 0x5f, 0x37, 0xc3,
    0xf4, 0xf5, 0x9a, 0x94, 0x6c, 0xa0, 0x93, 0xe1, 0xd8, 0xfb, 0x06, 0x8c,
    0xd8, 0xd6, 0xbb, 0x0f, 0xac, 0xeb, 0xb7, 0xed, 0x8d, 0x97, 0x42, 0x9d,
    0x22, 0x22, 0x3d, 0x2e, 0x2d, 0xd8, 0x7d, 0x04, 0x83, 0x93, 0xd3, 0x54,
    0x99, 0x31, 0x33, 0x9b,
};
static const struct drbg_kat_no_reseed kat647_t = {
    1, kat647_entropyin, kat647_nonce, kat647_persstr,
    kat647_addin0, kat647_addin1, kat647_retbits
};
static const struct drbg_kat kat647 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat647_t
};

static const unsigned char kat648_entropyin[] = {
    0xdb, 0xe7, 0xa4, 0x62, 0x2d, 0x8c, 0xbf, 0xcc, 0x19, 0x1d, 0xc7, 0x40,
    0x56, 0x6f, 0xa0, 0x58, 0x8d, 0x77, 0x9a, 0x0d, 0x22, 0x7b, 0x03, 0x7f,
    0x73, 0x18, 0xa4, 0x28, 0x2a, 0x08, 0x0b, 0x0f,
};
static const unsigned char kat648_nonce[] = {
    0xc4, 0xe3, 0x46, 0x9a, 0xc3, 0xa8, 0xd2, 0x31, 0x89, 0xc9, 0xc9, 0xe4,
    0x41, 0x2a, 0xbc, 0xd8,
};
static const unsigned char kat648_persstr[] = {
    0x08, 0xdb, 0x81, 0xc1, 0x29, 0x71, 0x68, 0x1e, 0x78, 0x0b, 0xd9, 0xff,
    0x53, 0x76, 0x84, 0xde, 0x80, 0xc4, 0xbb, 0x21, 0x4d, 0x6e, 0xb1, 0x3a,
    0x92, 0x09, 0x04, 0x4a, 0xe4, 0x62, 0xa7, 0x40,
};
static const unsigned char kat648_addin0[] = {
    0xf9, 0x53, 0x0b, 0x74, 0xa8, 0xcb, 0x02, 0x45, 0x56, 0xce, 0x54, 0xae,
    0xd8, 0x0c, 0x32, 0xef, 0x22, 0x01, 0xfb, 0x19, 0xf4, 0xaa, 0x56, 0x01,
    0x25, 0x85, 0x96, 0xc5, 0x97, 0x5e, 0x18, 0x4f,
};
static const unsigned char kat648_addin1[] = {
    0xd1, 0x9a, 0x2e, 0xa8, 0x53, 0xde, 0xe8, 0x3d, 0xbf, 0xad, 0x41, 0x6f,
    0xec, 0x5f, 0xfe, 0xbc, 0xb1, 0xc6, 0x93, 0x6b, 0x35, 0x9a, 0xce, 0x38,
    0xc5, 0xf0, 0x57, 0x0e, 0x3a, 0xae, 0xe7, 0xd1,
};
static const unsigned char kat648_retbits[] = {
    0xfe, 0xcf, 0xbc, 0x6e, 0xb3, 0xd0, 0x42, 0x23, 0x67, 0x58, 0x30, 0x44,
    0xbe, 0x8a, 0xfb, 0x65, 0x71, 0x77, 0x23, 0xf5, 0xfc, 0xc5, 0x3b, 0xde,
    0x92, 0x94, 0x86, 0x25, 0x56, 0x97, 0x0f, 0xe9, 0xde, 0x96, 0x4a, 0x27,
    0xd3, 0x1a, 0xcd, 0x1d, 0x41, 0xca, 0x77, 0xa1, 0xe3, 0xb0, 0xe7, 0x3f,
    0xfc, 0x0b, 0x1a, 0xa9, 0xc6, 0xd3, 0xcb, 0xe8, 0x42, 0x6c, 0xe9, 0x11,
    0x19, 0x3a, 0x16, 0x7b,
};
static const struct drbg_kat_no_reseed kat648_t = {
    2, kat648_entropyin, kat648_nonce, kat648_persstr,
    kat648_addin0, kat648_addin1, kat648_retbits
};
static const struct drbg_kat kat648 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat648_t
};

static const unsigned char kat649_entropyin[] = {
    0xb1, 0x97, 0x8e, 0xa5, 0xdc, 0x18, 0xa9, 0x93, 0xab, 0x37, 0xb8, 0x81,
    0xe3, 0x3d, 0x85, 0xab, 0x54, 0x8e, 0x98, 0xaa, 0x2a, 0x79, 0x7e, 0xcf,
    0x8a, 0x19, 0x86, 0x21, 0xc0, 0xa1, 0xe1, 0x17,
};
static const unsigned char kat649_nonce[] = {
    0xdb, 0x53, 0xb0, 0xde, 0x8f, 0x51, 0x78, 0x71, 0x7e, 0x0a, 0x69, 0xd8,
    0xbd, 0xee, 0x9f, 0x37,
};
static const unsigned char kat649_persstr[] = {
    0x4f, 0x6d, 0xda, 0x5e, 0x9c, 0xc1, 0xd2, 0x53, 0x8e, 0xea, 0xce, 0x39,
    0xd2, 0x53, 0xe1, 0x62, 0x14, 0x57, 0xd6, 0x94, 0xa8, 0xe2, 0x4e, 0x5c,
    0x76, 0x02, 0xfc, 0xc8, 0x19, 0xfb, 0x83, 0x8b,
};
static const unsigned char kat649_addin0[] = {
    0xe2, 0xad, 0xa7, 0x1c, 0x24, 0xf2, 0xde, 0xce, 0x43, 0x15, 0xc7, 0xda,
    0x2a, 0x94, 0xaf, 0x4e, 0x47, 0xa7, 0xca, 0x52, 0x9d, 0x9c, 0x36, 0x8a,
    0x32, 0xae, 0x45, 0x0d, 0x1a, 0x64, 0x5e, 0xe0,
};
static const unsigned char kat649_addin1[] = {
    0xa9, 0x1a, 0x32, 0xbd, 0x1d, 0x9d, 0x83, 0xd1, 0x12, 0xf7, 0xac, 0xcf,
    0x95, 0x61, 0x15, 0xbd, 0x04, 0x8a, 0xb5, 0xf4, 0x24, 0x45, 0x31, 0xe5,
    0x93, 0x2e, 0x38, 0x2d, 0xd1, 0x43, 0x59, 0x73,
};
static const unsigned char kat649_retbits[] = {
    0x39, 0x7d, 0xd1, 0x76, 0x9d, 0xe4, 0xdd, 0xa7, 0xaf, 0xf6, 0x45, 0xe3,
    0x2c, 0xfa, 0x1e, 0xd2, 0x2f, 0x3a, 0x43, 0x97, 0xb6, 0x28, 0x57, 0x89,
    0x6b, 0xaa, 0x0f, 0xba, 0x92, 0x62, 0xca, 0x8e, 0x46, 0xcf, 0x46, 0xf2,
    0x0b, 0x12, 0x77, 0x24, 0xd4, 0xc6, 0x86, 0x3c, 0x21, 0x11, 0xf1, 0xde,
    0xea, 0x9e, 0xfa, 0xdd, 0xb9, 0x18, 0x55, 0xbc, 0xd0, 0x18, 0x48, 0x1a,
    0xcf, 0x17, 0xf2, 0x5f,
};
static const struct drbg_kat_no_reseed kat649_t = {
    3, kat649_entropyin, kat649_nonce, kat649_persstr,
    kat649_addin0, kat649_addin1, kat649_retbits
};
static const struct drbg_kat kat649 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat649_t
};

static const unsigned char kat650_entropyin[] = {
    0x9c, 0xe4, 0x9c, 0x6e, 0x9c, 0x5f, 0x2d, 0x43, 0xa1, 0xa2, 0xa1, 0xc2,
    0xa9, 0x0e, 0x8d, 0x63, 0xf1, 0xf4, 0xbd, 0x83, 0xfc, 0xde, 0xb7, 0x34,
    0xf1, 0x89, 0x42, 0x62, 0xd9, 0x1b, 0x5e, 0x91,
};
static const unsigned char kat650_nonce[] = {
    0xf5, 0xa0, 0x88, 0x5f, 0x0c, 0x21, 0xa3, 0xf1, 0x6c, 0xfc, 0x3b, 0xb8,
    0x28, 0x24, 0x63, 0x33,
};
static const unsigned char kat650_persstr[] = {
    0x0f, 0xa1, 0x4c, 0x2a, 0xb3, 0xbb, 0x6f, 0xc3, 0xf2, 0x22, 0xf1, 0x85,
    0x38, 0xee, 0x6c, 0xb9, 0x8b, 0x2e, 0x1a, 0x4c, 0xa7, 0xb5, 0x3e, 0x4c,
    0x91, 0x9b, 0xef, 0x88, 0x81, 0x38, 0x02, 0x62,
};
static const unsigned char kat650_addin0[] = {
    0x90, 0x95, 0x89, 0xe0, 0x79, 0x5e, 0x57, 0x1c, 0x53, 0x08, 0x79, 0x50,
    0x3b, 0x56, 0x60, 0xf4, 0x20, 0x00, 0x3b, 0xcd, 0x19, 0xad, 0x05, 0x1a,
    0x50, 0x20, 0xa5, 0xd4, 0x46, 0x1b, 0x93, 0x27,
};
static const unsigned char kat650_addin1[] = {
    0x96, 0xd4, 0x62, 0x87, 0x64, 0xc4, 0xa2, 0xe5, 0x2c, 0xed, 0x37, 0x57,
    0x4e, 0xa8, 0x61, 0x8b, 0x3e, 0xbf, 0xe7, 0x03, 0x5d, 0x41, 0xed, 0xcc,
    0x9b, 0x82, 0x0a, 0x22, 0xaa, 0x16, 0x70, 0x8b,
};
static const unsigned char kat650_retbits[] = {
    0x1a, 0xaa, 0x09, 0xfc, 0x05, 0xa3, 0x22, 0xd3, 0x61, 0x00, 0xa6, 0x80,
    0x5c, 0xc7, 0x9b, 0xc3, 0xb5, 0x34, 0x38, 0x95, 0x03, 0x97, 0x69, 0x77,
    0x75, 0xc8, 0xf8, 0xf5, 0xb6, 0x87, 0x9d, 0x4f, 0x73, 0x86, 0xd8, 0x84,
    0xa0, 0x9b, 0xbd, 0x57, 0xa6, 0x8e, 0xfd, 0x79, 0xb7, 0xc0, 0xaf, 0x33,
    0xb3, 0xd4, 0x5d, 0x9e, 0xb5, 0xc1, 0x90, 0x9d, 0x4a, 0xb3, 0xe8, 0x31,
    0xe2, 0xd2, 0x6a, 0x67,
};
static const struct drbg_kat_no_reseed kat650_t = {
    4, kat650_entropyin, kat650_nonce, kat650_persstr,
    kat650_addin0, kat650_addin1, kat650_retbits
};
static const struct drbg_kat kat650 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat650_t
};

static const unsigned char kat651_entropyin[] = {
    0x46, 0xb2, 0xd8, 0x67, 0x16, 0xa4, 0x29, 0x35, 0x8c, 0x6c, 0x39, 0x51,
    0x79, 0x7e, 0x64, 0x17, 0x10, 0x4a, 0x48, 0xd8, 0xf7, 0x50, 0x72, 0xa1,
    0x80, 0xbf, 0x7f, 0xdf, 0x6d, 0xd8, 0x88, 0xe2,
};
static const unsigned char kat651_nonce[] = {
    0x94, 0x4b, 0x76, 0x18, 0x0c, 0x1a, 0x36, 0x05, 0x55, 0x09, 0x8f, 0x3d,
    0xe9, 0x62, 0xdb, 0x43,
};
static const unsigned char kat651_persstr[] = {
    0x7b, 0xb9, 0xcb, 0x64, 0x23, 0xd4, 0x68, 0x5d, 0x86, 0x6e, 0x47, 0xf9,
    0x84, 0x73, 0xa7, 0xdc, 0x72, 0x0b, 0xe0, 0xe4, 0xa4, 0x2e, 0xb3, 0x8b,
    0x0e, 0x38, 0x3b, 0x3a, 0xf6, 0xa0, 0x12, 0x56,
};
static const unsigned char kat651_addin0[] = {
    0xaa, 0x56, 0xfe, 0xc9, 0x90, 0x6c, 0x89, 0xfa, 0xad, 0xb7, 0x8d, 0xf0,
    0xfd, 0x67, 0x07, 0xe7, 0x5e, 0xae, 0x96, 0xc7, 0x96, 0xd3, 0x01, 0x35,
    0x7c, 0x95, 0x54, 0xe2, 0x99, 0x76, 0x8e, 0x76,
};
static const unsigned char kat651_addin1[] = {
    0xc3, 0x70, 0x47, 0x12, 0x29, 0x73, 0xa6, 0xbe, 0xe2, 0xa3, 0xa9, 0x48,
    0x1a, 0x24, 0xd5, 0x48, 0x2c, 0x61, 0x13, 0xaf, 0xf3, 0x86, 0x5d, 0x6f,
    0xc1, 0x08, 0x98, 0x96, 0x6b, 0x7c, 0x8e, 0x6f,
};
static const unsigned char kat651_retbits[] = {
    0xaf, 0xc8, 0x13, 0xfb, 0x36, 0x47, 0x04, 0x0d, 0x07, 0x96, 0x3b, 0x87,
    0xbb, 0x78, 0x80, 0x4a, 0x07, 0x29, 0xa3, 0x11, 0xb6, 0xee, 0x4e, 0xd2,
    0x66, 0x56, 0x67, 0x9e, 0x56, 0xe3, 0x71, 0x43, 0x5f, 0x3f, 0xa4, 0x50,
    0x45, 0x44, 0x82, 0x70, 0x3c, 0xbf, 0x91, 0x8c, 0xfe, 0x09, 0x45, 0xc7,
    0x0f, 0x37, 0x57, 0xe2, 0xc6, 0xab, 0x3a, 0x67, 0x4a, 0xc7, 0x3b, 0x21,
    0xde, 0x42, 0xc9, 0x74,
};
static const struct drbg_kat_no_reseed kat651_t = {
    5, kat651_entropyin, kat651_nonce, kat651_persstr,
    kat651_addin0, kat651_addin1, kat651_retbits
};
static const struct drbg_kat kat651 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat651_t
};

static const unsigned char kat652_entropyin[] = {
    0x4e, 0xcf, 0x9a, 0xa3, 0xd6, 0xe3, 0x38, 0x08, 0xc5, 0x2a, 0x5d, 0x70,
    0x08, 0xa7, 0x67, 0xb8, 0x32, 0xeb, 0x92, 0x5a, 0xc1, 0xd6, 0x7d, 0xdc,
    0x76, 0x11, 0x5e, 0x3c, 0x16, 0x38, 0xab, 0x8e,
};
static const unsigned char kat652_nonce[] = {
    0xd5, 0x82, 0xd4, 0x64, 0x65, 0xf0, 0x17, 0x9d, 0x1d, 0x3a, 0x95, 0x90,
    0xf8, 0x86, 0x18, 0xc0,
};
static const unsigned char kat652_persstr[] = {
    0xfb, 0x33, 0x96, 0xb7, 0xfc, 0xdf, 0xa4, 0x49, 0x21, 0x0b, 0x4c, 0xe4,
    0x4d, 0x39, 0x1f, 0xd8, 0x7e, 0xc5, 0x66, 0x3e, 0x97, 0xca, 0x95, 0x3d,
    0x2f, 0xdc, 0xed, 0x45, 0x33, 0x08, 0xf8, 0xa6,
};
static const unsigned char kat652_addin0[] = {
    0x42, 0x3b, 0x59, 0xa0, 0x38, 0x7f, 0xd1, 0xdc, 0xe9, 0x3c, 0x4b, 0xc4,
    0xb2, 0xc4, 0xab, 0xcb, 0xe0, 0x56, 0xe1, 0x0f, 0x98, 0xc1, 0x1c, 0xc1,
    0x0c, 0x3b, 0xe1, 0x39, 0x72, 0x9e, 0x04, 0xc2,
};
static const unsigned char kat652_addin1[] = {
    0x5f, 0x6f, 0xe2, 0xa4, 0xbe, 0x57, 0x01, 0xd0, 0xf8, 0x73, 0x5f, 0x1e,
    0x63, 0xaa, 0x1f, 0x4e, 0xb8, 0xc7, 0x21, 0x3a, 0x78, 0x7b, 0x8b, 0x14,
    0x4b, 0x83, 0xce, 0xca, 0x54, 0x54, 0x7a, 0x94,
};
static const unsigned char kat652_retbits[] = {
    0xeb, 0x5c, 0xbd, 0xab, 0x07, 0xb7, 0x1c, 0xd3, 0x56, 0x04, 0xe8, 0x65,
    0xd4, 0xf8, 0xb2, 0x6e, 0xb8, 0x40, 0xa6, 0x5f, 0x83, 0xa1, 0xdf, 0xe7,
    0xe3, 0xf6, 0xb2, 0xfc, 0xd2, 0x41, 0x2c, 0xbc, 0x21, 0xf9, 0xad, 0x09,
    0x39, 0x93, 0xcf, 0xab, 0x3b, 0x41, 0x48, 0x97, 0x28, 0x33, 0x8b, 0x58,
    0xbe, 0xf0, 0xb7, 0x3d, 0x08, 0x63, 0x8c, 0xc1, 0x25, 0x8c, 0x39, 0x09,
    0x89, 0xeb, 0x7c, 0x28,
};
static const struct drbg_kat_no_reseed kat652_t = {
    6, kat652_entropyin, kat652_nonce, kat652_persstr,
    kat652_addin0, kat652_addin1, kat652_retbits
};
static const struct drbg_kat kat652 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat652_t
};

static const unsigned char kat653_entropyin[] = {
    0x94, 0xaa, 0x46, 0x3c, 0x75, 0x46, 0xa1, 0x59, 0xea, 0x14, 0xaf, 0x23,
    0x55, 0xaa, 0x78, 0x5a, 0x03, 0xc9, 0x12, 0x17, 0x97, 0x71, 0x9d, 0xc8,
    0xe8, 0xd5, 0xee, 0xeb, 0x2c, 0xc5, 0x1c, 0x59,
};
static const unsigned char kat653_nonce[] = {
    0x26, 0x9e, 0x5d, 0x4a, 0xd3, 0xcf, 0x2d, 0x14, 0xa1, 0x7b, 0xae, 0x19,
    0xc7, 0xd9, 0xc0, 0x85,
};
static const unsigned char kat653_persstr[] = {
    0x35, 0x11, 0xa4, 0x24, 0x9d, 0x85, 0x50, 0x72, 0xf6, 0x5b, 0x31, 0xb2,
    0x15, 0x81, 0x99, 0x2c, 0x75, 0x4c, 0xc3, 0x9d, 0xc5, 0x72, 0x59, 0x24,
    0xe5, 0x7e, 0x5a, 0x39, 0xe3, 0xce, 0x9e, 0x30,
};
static const unsigned char kat653_addin0[] = {
    0xad, 0x0c, 0x18, 0x97, 0x1e, 0x6c, 0xf6, 0xc7, 0x74, 0x66, 0x71, 0xac,
    0x37, 0x6c, 0x30, 0x86, 0x1f, 0xc7, 0xca, 0x66, 0x13, 0x46, 0x06, 0x7c,
    0x44, 0xf6, 0x0f, 0x48, 0x39, 0x8c, 0xc6, 0x03,
};
static const unsigned char kat653_addin1[] = {
    0x1f, 0x4a, 0x35, 0xab, 0xe8, 0x55, 0xd3, 0xfb, 0x5d, 0x69, 0x60, 0x7b,
    0xb8, 0xa5, 0x88, 0x8a, 0x10, 0x54, 0xd9, 0x95, 0x71, 0xf7, 0xf6, 0x02,
    0x0e, 0xc6, 0x52, 0x48, 0xd4, 0x72, 0x7d, 0x4f,
};
static const unsigned char kat653_retbits[] = {
    0x7d, 0xd6, 0x6c, 0xa4, 0x01, 0x70, 0x9b, 0x52, 0xc7, 0x7f, 0x67, 0xc2,
    0x20, 0x26, 0xfd, 0x92, 0x1c, 0xb6, 0x03, 0xfe, 0xc7, 0x28, 0x9d, 0xb4,
    0x44, 0x00, 0xa7, 0xcd, 0x83, 0x22, 0xf2, 0xb2, 0xf8, 0xe2, 0xda, 0x65,
    0x2b, 0xe5, 0x6a, 0xe3, 0x7b, 0x2a, 0x59, 0x84, 0xeb, 0xbd, 0x88, 0x9c,
    0x63, 0xfc, 0x39, 0x9c, 0x24, 0x29, 0x6a, 0x6c, 0xb6, 0x5f, 0xdb, 0x2c,
    0x2e, 0x86, 0x9d, 0x0b,
};
static const struct drbg_kat_no_reseed kat653_t = {
    7, kat653_entropyin, kat653_nonce, kat653_persstr,
    kat653_addin0, kat653_addin1, kat653_retbits
};
static const struct drbg_kat kat653 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat653_t
};

static const unsigned char kat654_entropyin[] = {
    0x92, 0x1e, 0x44, 0xa4, 0xa0, 0xd1, 0x8d, 0xe3, 0x27, 0x16, 0xd4, 0x15,
    0xc0, 0xeb, 0xb7, 0x44, 0x7f, 0x88, 0xc9, 0x4f, 0xe1, 0x28, 0x01, 0x3b,
    0xb6, 0xcc, 0xb9, 0x1b, 0xa3, 0x70, 0xa9, 0x42,
};
static const unsigned char kat654_nonce[] = {
    0x02, 0xb1, 0xac, 0x59, 0xd8, 0x95, 0xeb, 0x97, 0x19, 0xa7, 0xff, 0x3b,
    0xbe, 0x26, 0xcf, 0x41,
};
static const unsigned char kat654_persstr[] = {
    0x21, 0xa1, 0xa3, 0x4a, 0xb5, 0xaa, 0xb7, 0x6d, 0xf9, 0x22, 0x80, 0x68,
    0x39, 0xbd, 0x4a, 0xce, 0x2f, 0x17, 0xde, 0xf5, 0x9b, 0xc7, 0x35, 0xbd,
    0x9f, 0x9e, 0xb0, 0x18, 0x00, 0x4b, 0x02, 0x02,
};
static const unsigned char kat654_addin0[] = {
    0x8f, 0x8d, 0x89, 0x50, 0x5d, 0xb0, 0xf5, 0x9e, 0x62, 0xea, 0xc8, 0x3a,
    0x07, 0x94, 0xd5, 0x7e, 0x7a, 0xa0, 0xb6, 0x89, 0x82, 0x5b, 0x1f, 0x32,
    0xe4, 0xbe, 0x82, 0xa2, 0x80, 0x5e, 0xf7, 0x94,
};
static const unsigned char kat654_addin1[] = {
    0x0b, 0xd6, 0xd9, 0xde, 0x89, 0x60, 0x6b, 0x1b, 0x6f, 0xb2, 0xd8, 0x55,
    0x60, 0x42, 0xf3, 0xbe, 0x2e, 0x18, 0xa2, 0x24, 0x65, 0xbe, 0xf4, 0x4c,
    0x4b, 0x32, 0x29, 0x2d, 0x91, 0x03, 0x92, 0x08,
};
static const unsigned char kat654_retbits[] = {
    0x86, 0xdf, 0xd5, 0x8f, 0xc6, 0xb9, 0x3d, 0x89, 0x57, 0x10, 0xbb, 0x6b,
    0x06, 0x76, 0x74, 0x97, 0xad, 0x1c, 0xa0, 0xe8, 0xc0, 0xf5, 0xca, 0xdd,
    0x8a, 0x01, 0x2f, 0xce, 0xd9, 0x2c, 0x4f, 0xa6, 0x42, 0x3b, 0x77, 0xb9,
    0x6a, 0x1d, 0xc2, 0x89, 0x71, 0xf7, 0x71, 0x5b, 0x78, 0x23, 0xdd, 0x5d,
    0x91, 0x0e, 0xbe, 0x2e, 0xaf, 0xee, 0x37, 0x79, 0xa6, 0x1e, 0x1f, 0x70,
    0x7f, 0x53, 0x25, 0xd7,
};
static const struct drbg_kat_no_reseed kat654_t = {
    8, kat654_entropyin, kat654_nonce, kat654_persstr,
    kat654_addin0, kat654_addin1, kat654_retbits
};
static const struct drbg_kat kat654 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat654_t
};

static const unsigned char kat655_entropyin[] = {
    0x0c, 0x7c, 0x98, 0x7a, 0x32, 0xbb, 0x36, 0x9f, 0x5c, 0x13, 0x3c, 0x02,
    0x94, 0x36, 0x0a, 0xb3, 0xf0, 0x6f, 0xff, 0x4a, 0xf3, 0x0a, 0xcf, 0x4a,
    0x36, 0x55, 0x4a, 0xbc, 0x81, 0xea, 0x17, 0xd7,
};
static const unsigned char kat655_nonce[] = {
    0x49, 0x63, 0x0b, 0x51, 0x6b, 0x1e, 0x44, 0xca, 0xd4, 0x62, 0x1b, 0xb1,
    0xb2, 0x94, 0x34, 0xff,
};
static const unsigned char kat655_persstr[] = {
    0xfc, 0x70, 0x06, 0x12, 0xef, 0x74, 0x77, 0x9c, 0x25, 0xef, 0x84, 0x64,
    0x32, 0x10, 0x6e, 0xe4, 0x32, 0x1d, 0x35, 0x7c, 0x9c, 0x74, 0xed, 0x0b,
    0xcf, 0xe6, 0xd7, 0x31, 0x2e, 0xc0, 0xc6, 0xba,
};
static const unsigned char kat655_addin0[] = {
    0x55, 0xd3, 0xde, 0xaf, 0x9f, 0x26, 0x70, 0xe7, 0x2b, 0xdc, 0x3d, 0x13,
    0xd6, 0x2e, 0xaf, 0xcc, 0x8f, 0x2c, 0x2e, 0xd0, 0xcb, 0xf1, 0x79, 0x82,
    0xf5, 0x71, 0xd0, 0x86, 0x73, 0x8f, 0x40, 0xa2,
};
static const unsigned char kat655_addin1[] = {
    0x03, 0xe4, 0x80, 0xed, 0x33, 0xc8, 0x1f, 0x60, 0xee, 0x1d, 0x55, 0x8d,
    0xf4, 0x9e, 0x6e, 0x0c, 0x2e, 0x4f, 0x8c, 0xd4, 0x00, 0x7c, 0x0c, 0xa9,
    0xd0, 0x1a, 0x65, 0xe3, 0x2d, 0x1f, 0xd1, 0x3c,
};
static const unsigned char kat655_retbits[] = {
    0x1c, 0x02, 0xa1, 0x3d, 0xb6, 0xf4, 0xbe, 0xfd, 0x3a, 0xa7, 0x4b, 0x04,
    0xe9, 0xea, 0xb0, 0x10, 0x18, 0x4a, 0x32, 0x2e, 0xfa, 0xeb, 0x86, 0x0f,
    0x60, 0x7e, 0x98, 0xd5, 0xb1, 0x5d, 0x43, 0x37, 0x98, 0x3d, 0xee, 0xf8,
    0x77, 0x99, 0xc0, 0x86, 0x55, 0x17, 0xa1, 0x2c, 0x4b, 0x49, 0xbe, 0xd1,
    0x01, 0x6c, 0xb7, 0x3e, 0xba, 0x6e, 0xa0, 0x2f, 0xca, 0x60, 0xf7, 0xef,
    0xe5, 0x50, 0xdf, 0x3a,
};
static const struct drbg_kat_no_reseed kat655_t = {
    9, kat655_entropyin, kat655_nonce, kat655_persstr,
    kat655_addin0, kat655_addin1, kat655_retbits
};
static const struct drbg_kat kat655 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat655_t
};

static const unsigned char kat656_entropyin[] = {
    0xaf, 0xdb, 0xc8, 0xac, 0xbe, 0xeb, 0xbb, 0x3a, 0x58, 0xb1, 0xbb, 0xbe,
    0xb1, 0x04, 0x83, 0x53, 0x47, 0x73, 0xa2, 0xe1, 0x63, 0x1f, 0x98, 0xc3,
    0x50, 0x3f, 0x6a, 0x4b, 0x33, 0x1b, 0x14, 0xb3,
};
static const unsigned char kat656_nonce[] = {
    0xdc, 0x8d, 0xff, 0x64, 0xe4, 0x4d, 0x50, 0xd4, 0x5d, 0xfd, 0x66, 0xfd,
    0x7c, 0x54, 0x7a, 0xd0,
};
static const unsigned char kat656_persstr[] = {
    0x2e, 0xc2, 0xbd, 0x1a, 0xb6, 0x14, 0x79, 0xcf, 0x78, 0x0b, 0x97, 0x7c,
    0xc4, 0x80, 0xa6, 0x9b, 0x53, 0x06, 0x5f, 0x38, 0x12, 0x70, 0xea, 0x95,
    0xa0, 0x41, 0xeb, 0x41, 0x6c, 0x81, 0x92, 0xf8,
};
static const unsigned char kat656_addin0[] = {
    0x69, 0x6f, 0xa5, 0x74, 0x8d, 0xf7, 0x5c, 0xf6, 0x46, 0xd3, 0x4f, 0xb4,
    0x78, 0xce, 0x2c, 0x9b, 0xfe, 0xd2, 0xe7, 0x97, 0x81, 0xf5, 0xed, 0x1e,
    0xdc, 0x2e, 0xe7, 0xd9, 0xb5, 0x8b, 0xee, 0xaa,
};
static const unsigned char kat656_addin1[] = {
    0xe7, 0xfe, 0xf0, 0xf7, 0x55, 0x3d, 0x29, 0x19, 0xb8, 0x03, 0xca, 0x8b,
    0x83, 0xa6, 0xd4, 0x63, 0xce, 0x74, 0xff, 0xa6, 0x04, 0x0b, 0x02, 0x26,
    0x85, 0xea, 0xee, 0x81, 0x65, 0xcb, 0x49, 0x38,
};
static const unsigned char kat656_retbits[] = {
    0xd5, 0x57, 0xec, 0x72, 0x87, 0x87, 0xd1, 0xf2, 0xc5, 0x53, 0xc2, 0xba,
    0x91, 0x34, 0x51, 0x31, 0x6e, 0x7c, 0x5a, 0xa4, 0x99, 0xd9, 0xe7, 0x47,
    0x40, 0xba, 0xb1, 0x59, 0xdc, 0x80, 0xbb, 0xc8, 0x87, 0x08, 0x41, 0xed,
    0x83, 0x73, 0xc1, 0x0a, 0xfc, 0xc3, 0x8f, 0x91, 0xb0, 0x66, 0x7d, 0x6a,
    0x44, 0x77, 0xde, 0xe8, 0x8e, 0x07, 0xd9, 0x0a, 0x10, 0x64, 0x9d, 0x57,
    0x21, 0x01, 0x1b, 0x97,
};
static const struct drbg_kat_no_reseed kat656_t = {
    10, kat656_entropyin, kat656_nonce, kat656_persstr,
    kat656_addin0, kat656_addin1, kat656_retbits
};
static const struct drbg_kat kat656 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat656_t
};

static const unsigned char kat657_entropyin[] = {
    0x80, 0xca, 0xfa, 0x90, 0xe1, 0x3c, 0x24, 0xfd, 0x0d, 0x53, 0x61, 0xf5,
    0x0f, 0x38, 0x58, 0xcb, 0x46, 0xbe, 0x74, 0x6c, 0x0b, 0xf9, 0xf9, 0xc8,
    0x6b, 0xfa, 0x57, 0x4f, 0x6f, 0x63, 0xa6, 0xa3,
};
static const unsigned char kat657_nonce[] = {
    0xa9, 0xdd, 0xc1, 0x8c, 0x92, 0x9e, 0x55, 0x05, 0xda, 0x59, 0xed, 0xdc,
    0xf5, 0x1c, 0x6c, 0x48,
};
static const unsigned char kat657_persstr[] = {
    0x2b, 0xb7, 0x5c, 0xe2, 0x9b, 0xab, 0x75, 0xb2, 0xe9, 0x19, 0x2b, 0x13,
    0x72, 0x98, 0x77, 0x73, 0x10, 0x13, 0xc5, 0xde, 0x00, 0x6c, 0x3f, 0x58,
    0x25, 0xaf, 0x8e, 0xe8, 0xd7, 0xbe, 0x70, 0x73,
};
static const unsigned char kat657_addin0[] = {
    0x8e, 0x97, 0xf8, 0x40, 0x57, 0x4b, 0x9f, 0xe2, 0xed, 0x8a, 0x86, 0xd2,
    0xbd, 0xf3, 0x72, 0xb0, 0xf3, 0x1f, 0x89, 0x29, 0xa8, 0xd0, 0xd2, 0xb1,
    0x55, 0x5f, 0x03, 0xd6, 0xcc, 0x96, 0x03, 0x23,
};
static const unsigned char kat657_addin1[] = {
    0x65, 0xd9, 0x9e, 0x66, 0x4a, 0x5f, 0x3a, 0xc7, 0xb0, 0xd0, 0x79, 0xba,
    0x3c, 0xbd, 0x2d, 0x1b, 0x73, 0x3b, 0x9a, 0x46, 0xb4, 0x5b, 0xf8, 0x05,
    0xbb, 0xb1, 0x44, 0x26, 0x92, 0xd1, 0xa4, 0xd3,
};
static const unsigned char kat657_retbits[] = {
    0x9b, 0x9b, 0xde, 0xea, 0x2c, 0x91, 0x2f, 0xc1, 0x6d, 0x1b, 0x98, 0xcb,
    0x68, 0x02, 0x01, 0x34, 0x16, 0x76, 0x08, 0x09, 0x07, 0x3c, 0x53, 0x12,
    0x48, 0xe2, 0x6c, 0x64, 0x27, 0x2d, 0x44, 0x2c, 0x89, 0xdb, 0x23, 0x42,
    0xda, 0x1b, 0x44, 0xca, 0x4a, 0x1e, 0x5d, 0x98, 0x34, 0x99, 0x99, 0x9c,
    0xd8, 0x85, 0x1e, 0x20, 0xfc, 0x05, 0x59, 0xf8, 0x37, 0xf6, 0x67, 0x96,
    0xdd, 0x56, 0xb3, 0xd9,
};
static const struct drbg_kat_no_reseed kat657_t = {
    11, kat657_entropyin, kat657_nonce, kat657_persstr,
    kat657_addin0, kat657_addin1, kat657_retbits
};
static const struct drbg_kat kat657 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat657_t
};

static const unsigned char kat658_entropyin[] = {
    0x28, 0x16, 0xc4, 0xa6, 0xe8, 0x59, 0xc7, 0xda, 0x65, 0x71, 0xf5, 0x92,
    0x1b, 0x8e, 0x62, 0x1e, 0x9b, 0xdb, 0x9d, 0x97, 0xeb, 0x3f, 0x9a, 0x44,
    0x1a, 0x3e, 0x85, 0x37, 0x53, 0x6b, 0xfe, 0x21,
};
static const unsigned char kat658_nonce[] = {
    0x9b, 0x90, 0x61, 0x36, 0xc6, 0x85, 0xd9, 0xae, 0x68, 0xf4, 0x98, 0x63,
    0xdc, 0xcf, 0xc3, 0x1b,
};
static const unsigned char kat658_persstr[] = {
    0x72, 0x22, 0x82, 0xe8, 0xc3, 0xd4, 0xae, 0x4c, 0xd3, 0x2c, 0xcb, 0x06,
    0xad, 0x25, 0x0b, 0x33, 0xcf, 0x6e, 0xb0, 0xcf, 0x96, 0x4e, 0xbd, 0x8a,
    0x1f, 0xb8, 0xb5, 0x8b, 0xed, 0x00, 0xea, 0xb1,
};
static const unsigned char kat658_addin0[] = {
    0x5c, 0xa9, 0xd7, 0xbd, 0x6e, 0x95, 0x34, 0xdd, 0xfa, 0x5a, 0x6c, 0x4d,
    0xc0, 0x0b, 0x40, 0x4f, 0x25, 0x9f, 0x04, 0x2d, 0xed, 0x1f, 0x19, 0x9c,
    0xe6, 0x6a, 0xa5, 0x4d, 0xaa, 0x3e, 0x50, 0x19,
};
static const unsigned char kat658_addin1[] = {
    0x1d, 0xd4, 0x1f, 0x32, 0x17, 0x02, 0x4b, 0xd5, 0xf0, 0xcb, 0xdd, 0x44,
    0xa5, 0x91, 0x5e, 0xda, 0x49, 0xf3, 0xc3, 0x03, 0xc9, 0xa7, 0x0f, 0xd7,
    0x4e, 0x67, 0x50, 0x28, 0xe5, 0x67, 0xbb, 0x37,
};
static const unsigned char kat658_retbits[] = {
    0x4e, 0x3b, 0x60, 0xae, 0xf2, 0xe3, 0xb7, 0x5e, 0xd0, 0x70, 0xd4, 0x36,
    0x3e, 0x68, 0x44, 0xc6, 0x00, 0xa5, 0x58, 0x7c, 0x5a, 0x8c, 0x8c, 0x4e,
    0x11, 0xa6, 0x25, 0x87, 0x61, 0xcc, 0x07, 0x2f, 0xa4, 0x16, 0x50, 0x7d,
    0x73, 0xa4, 0xbe, 0xa1, 0x9c, 0x8a, 0x31, 0x38, 0xa8, 0xdb, 0x32, 0x97,
    0x16, 0xd6, 0xa8, 0x30, 0xb2, 0x87, 0x0b, 0x5b, 0x69, 0x95, 0x8a, 0x4d,
    0xae, 0xd4, 0xb7, 0x14,
};
static const struct drbg_kat_no_reseed kat658_t = {
    12, kat658_entropyin, kat658_nonce, kat658_persstr,
    kat658_addin0, kat658_addin1, kat658_retbits
};
static const struct drbg_kat kat658 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat658_t
};

static const unsigned char kat659_entropyin[] = {
    0x5f, 0x44, 0x0d, 0xb9, 0x7d, 0x11, 0x47, 0x58, 0x56, 0xe1, 0x20, 0x4d,
    0xb7, 0x00, 0xf2, 0x0e, 0xf5, 0x2c, 0xc9, 0x50, 0x5d, 0xe7, 0xfc, 0x23,
    0x53, 0xd6, 0x42, 0x9c, 0x8c, 0xde, 0xcc, 0xca,
};
static const unsigned char kat659_nonce[] = {
    0xb5, 0xf1, 0xe8, 0x27, 0x26, 0x3a, 0xac, 0x0d, 0xfd, 0xf6, 0x41, 0xe8,
    0x99, 0x25, 0x3a, 0x83,
};
static const unsigned char kat659_persstr[] = {
    0x2a, 0x5b, 0x6b, 0x82, 0x92, 0xe5, 0xe8, 0x7c, 0x53, 0xcd, 0x2b, 0x27,
    0x31, 0x2f, 0x77, 0x00, 0xcc, 0x40, 0xac, 0x0f, 0x4a, 0xd4, 0xbc, 0x7b,
    0x9f, 0xbc, 0xe4, 0xdd, 0x52, 0xa0, 0x94, 0xf9,
};
static const unsigned char kat659_addin0[] = {
    0x3a, 0xd3, 0x5c, 0x3b, 0x31, 0x85, 0x34, 0xed, 0xe1, 0xba, 0x6f, 0x10,
    0xfc, 0x75, 0x9f, 0x24, 0x91, 0x4a, 0x6c, 0x6e, 0x59, 0x34, 0xba, 0x83,
    0xa7, 0x76, 0xeb, 0xdb, 0x57, 0x71, 0x7a, 0xa2,
};
static const unsigned char kat659_addin1[] = {
    0x81, 0x20, 0x2a, 0x62, 0x8b, 0xc3, 0x1e, 0x86, 0x82, 0x1d, 0xc0, 0xd8,
    0xa3, 0xed, 0xb9, 0xd8, 0x64, 0xe5, 0x1b, 0x27, 0x4c, 0x23, 0x04, 0xc6,
    0x33, 0x44, 0xae, 0x50, 0x18, 0xcd, 0x03, 0x87,
};
static const unsigned char kat659_retbits[] = {
    0xf9, 0x1d, 0x2f, 0xd3, 0xa2, 0x0e, 0x54, 0x5e, 0xaf, 0xbe, 0x12, 0x4b,
    0x20, 0x33, 0x2e, 0x4a, 0xc2, 0x4a, 0xbe, 0x5f, 0x2c, 0x5b, 0xc0, 0xc4,
    0x5d, 0x21, 0x84, 0x88, 0xd2, 0xfc, 0x94, 0xde, 0xd5, 0x65, 0xaa, 0xf5,
    0x57, 0x5d, 0x8c, 0x93, 0x51, 0x98, 0x5f, 0xaf, 0xb2, 0x2e, 0x7f, 0xc5,
    0xa2, 0xc8, 0xe9, 0x53, 0x2e, 0x5b, 0xaf, 0x4b, 0x03, 0xb9, 0xd1, 0xac,
    0x8a, 0x96, 0x05, 0x05,
};
static const struct drbg_kat_no_reseed kat659_t = {
    13, kat659_entropyin, kat659_nonce, kat659_persstr,
    kat659_addin0, kat659_addin1, kat659_retbits
};
static const struct drbg_kat kat659 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat659_t
};

static const unsigned char kat660_entropyin[] = {
    0xec, 0x7e, 0x9d, 0x66, 0x9b, 0x31, 0x77, 0x03, 0x16, 0xb4, 0x20, 0x51,
    0x0e, 0x41, 0x55, 0x3f, 0x68, 0x3e, 0x01, 0x24, 0x03, 0xe6, 0x57, 0xf6,
    0x39, 0x92, 0x0a, 0x60, 0x35, 0xee, 0xd3, 0x4b,
};
static const unsigned char kat660_nonce[] = {
    0x30, 0x79, 0xe0, 0xa6, 0x21, 0x50, 0xe6, 0x0e, 0x58, 0xad, 0xe5, 0xad,
    0x62, 0x22, 0x5f, 0xd9,
};
static const unsigned char kat660_persstr[] = {
    0xaf, 0x32, 0xfc, 0x72, 0xf2, 0x34, 0x31, 0x3d, 0xbb, 0x9f, 0xd6, 0x6b,
    0x4d, 0x91, 0xbb, 0xa7, 0x43, 0x20, 0x6d, 0x6c, 0xdf, 0x28, 0x9d, 0xcd,
    0x4f, 0xab, 0x1e, 0xf7, 0xab, 0x46, 0xb1, 0xfa,
};
static const unsigned char kat660_addin0[] = {
    0x03, 0x18, 0x60, 0xc8, 0xc0, 0x18, 0x32, 0xad, 0xfb, 0xfc, 0xb9, 0x32,
    0x2f, 0x28, 0xe8, 0xbf, 0xf4, 0x0f, 0xac, 0x8b, 0x3d, 0xc2, 0x8b, 0x36,
    0x60, 0x88, 0xc2, 0x5f, 0x2c, 0x4b, 0xd7, 0xdd,
};
static const unsigned char kat660_addin1[] = {
    0x31, 0xed, 0xd2, 0xa5, 0xcb, 0x0a, 0x51, 0x08, 0x89, 0x51, 0x76, 0x80,
    0x33, 0xc9, 0x67, 0x22, 0x40, 0x30, 0x93, 0x42, 0xdc, 0x90, 0xee, 0x00,
    0x93, 0x53, 0xf0, 0xc6, 0x65, 0xdd, 0xc7, 0xde,
};
static const unsigned char kat660_retbits[] = {
    0x74, 0x90, 0xd6, 0xb1, 0x08, 0xe0, 0x16, 0x83, 0x8c, 0x6c, 0xf3, 0x3d,
    0xba, 0x4b, 0x78, 0xee, 0x3b, 0x6b, 0x36, 0x82, 0x9d, 0xe5, 0x10, 0x93,
    0xc9, 0xe6, 0x29, 0x09, 0x4a, 0x2f, 0x46, 0x2f, 0xdb, 0x55, 0x52, 0xf6,
    0xa0, 0x59, 0x33, 0x3f, 0xcc, 0x2c, 0xc3, 0xfb, 0x7f, 0xf1, 0x37, 0xc0,
    0xd8, 0xec, 0x1f, 0xcd, 0xa1, 0x7b, 0x58, 0x94, 0xe1, 0xa0, 0xfe, 0x24,
    0xf1, 0xe6, 0x7a, 0x8e,
};
static const struct drbg_kat_no_reseed kat660_t = {
    14, kat660_entropyin, kat660_nonce, kat660_persstr,
    kat660_addin0, kat660_addin1, kat660_retbits
};
static const struct drbg_kat kat660 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat660_t
};

static const unsigned char kat661_entropyin[] = {
    0x4e, 0xe6, 0x8b, 0x33, 0x52, 0xb8, 0x74, 0xe1, 0xcc, 0x29, 0x37, 0x50,
    0x28, 0x85, 0x1d, 0xee, 0x9d, 0x5d, 0xfd, 0x88, 0xa4, 0x06, 0x64, 0xc7,
    0x9e, 0x2b, 0x72, 0x4f, 0xb1, 0x1b, 0x28, 0x08,
};
static const unsigned char kat661_nonce[] = {
    0x1c, 0x6a, 0x80, 0xd8, 0x20, 0x12, 0xc3, 0x9c, 0x9f, 0x14, 0xa8, 0x08,
    0x64, 0x3f, 0x08, 0xe7,
};
static const unsigned char kat661_persstr[] = {0};
static const unsigned char kat661_addin0[] = {0};
static const unsigned char kat661_addin1[] = {0};
static const unsigned char kat661_retbits[] = {
    0x7c, 0x58, 0xd2, 0xa5, 0x52, 0x2a, 0x88, 0x34, 0x1f, 0xb5, 0x5f, 0xac,
    0xef, 0xdb, 0x6e, 0x24, 0x84, 0x0c, 0xae, 0x28, 0x39, 0x48, 0xd5, 0x31,
    0x48, 0xa3, 0x84, 0xe1, 0x3b, 0x54, 0x07, 0xd7, 0x71, 0x2c, 0x33, 0x43,
    0x4b, 0xd3, 0xd1, 0x94, 0x48, 0xb4, 0x32, 0x70, 0xc5, 0x48, 0x60, 0xbf,
    0x34, 0x95, 0x57, 0x90, 0x57, 0xc7, 0x0b, 0xff, 0x30, 0x84, 0xdd, 0xdf,
    0xf0, 0x8a, 0x09, 0x1d,
};
static const struct drbg_kat_no_reseed kat661_t = {
    0, kat661_entropyin, kat661_nonce, kat661_persstr,
    kat661_addin0, kat661_addin1, kat661_retbits
};
static const struct drbg_kat kat661 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat661_t
};

static const unsigned char kat662_entropyin[] = {
    0x94, 0x42, 0xe3, 0xf7, 0x67, 0x75, 0x09, 0x3a, 0xc2, 0x63, 0x5d, 0x9b,
    0x21, 0x79, 0x74, 0xe8, 0xc7, 0xcc, 0x9c, 0xce, 0x8b, 0xba, 0x2f, 0x04,
    0xde, 0x57, 0x43, 0x2f, 0xe6, 0xcf, 0x0f, 0x4a,
};
static const unsigned char kat662_nonce[] = {
    0xb9, 0x4a, 0x55, 0x8d, 0xe7, 0xf8, 0x87, 0xf7, 0xf5, 0x0d, 0x3f, 0x0c,
    0xd4, 0xf7, 0x6f, 0x43,
};
static const unsigned char kat662_persstr[] = {0};
static const unsigned char kat662_addin0[] = {0};
static const unsigned char kat662_addin1[] = {0};
static const unsigned char kat662_retbits[] = {
    0x31, 0xca, 0xae, 0xe5, 0xd5, 0x0c, 0x63, 0x42, 0xfd, 0x6b, 0x3b, 0x18,
    0xd0, 0xf8, 0x8e, 0x72, 0xb8, 0x57, 0xed, 0x3f, 0xe5, 0xcb, 0xaa, 0xf7,
    0x6b, 0xe1, 0xa6, 0xac, 0xf0, 0x85, 0x51, 0xcf, 0x3e, 0xb1, 0x5f, 0x4b,
    0x57, 0x3c, 0xa9, 0x89, 0x50, 0xc7, 0x7d, 0x30, 0xea, 0x1d, 0xc3, 0xb9,
    0xfa, 0x73, 0x33, 0x5c, 0xba, 0xa8, 0xe3, 0xa5, 0x16, 0x21, 0x11, 0x26,
    0x9a, 0xf7, 0x33, 0x3a,
};
static const struct drbg_kat_no_reseed kat662_t = {
    1, kat662_entropyin, kat662_nonce, kat662_persstr,
    kat662_addin0, kat662_addin1, kat662_retbits
};
static const struct drbg_kat kat662 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat662_t
};

static const unsigned char kat663_entropyin[] = {
    0x27, 0xf1, 0xcf, 0xb9, 0x37, 0x18, 0x5e, 0xff, 0xf2, 0x48, 0xe1, 0xb1,
    0x18, 0x8c, 0xf1, 0xfd, 0x9f, 0xb4, 0x89, 0xa7, 0xc8, 0x79, 0x5e, 0xf2,
    0xc7, 0xe0, 0xf8, 0xa7, 0xd7, 0xf7, 0x11, 0xe0,
};
static const unsigned char kat663_nonce[] = {
    0xe7, 0xac, 0x79, 0x5a, 0xdc, 0xda, 0xae, 0x1a, 0x93, 0x11, 0x68, 0x66,
    0xc0, 0x09, 0xc5, 0xe5,
};
static const unsigned char kat663_persstr[] = {0};
static const unsigned char kat663_addin0[] = {0};
static const unsigned char kat663_addin1[] = {0};
static const unsigned char kat663_retbits[] = {
    0x14, 0xf2, 0x2a, 0xb6, 0x9b, 0x2d, 0x5a, 0xc9, 0x16, 0x91, 0x70, 0x89,
    0x82, 0x7b, 0xd6, 0x57, 0xf8, 0xd6, 0xd1, 0xd9, 0x80, 0xb7, 0x92, 0x11,
    0xbc, 0x35, 0x0b, 0x0b, 0x32, 0x27, 0x96, 0x8d, 0x87, 0x66, 0x74, 0x29,
    0x7a, 0x89, 0x87, 0xdd, 0xb0, 0xa9, 0x44, 0xad, 0x1e, 0x22, 0xdf, 0x4c,
    0xf1, 0xb6, 0x12, 0xaf, 0x3f, 0x11, 0x01, 0x3a, 0x59, 0x7e, 0x2d, 0x20,
    0x1d, 0xed, 0x33, 0xc9,
};
static const struct drbg_kat_no_reseed kat663_t = {
    2, kat663_entropyin, kat663_nonce, kat663_persstr,
    kat663_addin0, kat663_addin1, kat663_retbits
};
static const struct drbg_kat kat663 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat663_t
};

static const unsigned char kat664_entropyin[] = {
    0x65, 0x89, 0xa9, 0x0d, 0xda, 0xc0, 0x83, 0x8c, 0x73, 0xb7, 0xa4, 0x52,
    0x9f, 0x2c, 0x64, 0x7d, 0x70, 0x7d, 0x3f, 0x5f, 0x17, 0xcb, 0x76, 0xa8,
    0xdf, 0x26, 0x5f, 0x26, 0x4e, 0x33, 0xc8, 0xb9,
};
static const unsigned char kat664_nonce[] = {
    0x00, 0x6a, 0x8e, 0x6c, 0x2f, 0xac, 0xb2, 0x35, 0x5f, 0xd6, 0xa4, 0x63,
    0x8d, 0xdb, 0x7c, 0x91,
};
static const unsigned char kat664_persstr[] = {0};
static const unsigned char kat664_addin0[] = {0};
static const unsigned char kat664_addin1[] = {0};
static const unsigned char kat664_retbits[] = {
    0xfb, 0xed, 0x16, 0x31, 0x23, 0xf1, 0xd0, 0x40, 0x44, 0x6c, 0xaf, 0xe4,
    0x4b, 0x96, 0x60, 0xa7, 0x21, 0x1d, 0x0f, 0xf0, 0xee, 0xae, 0xba, 0x86,
    0xa6, 0x12, 0xd8, 0x1d, 0x88, 0xee, 0x8c, 0x6a, 0xda, 0x33, 0xd2, 0x61,
    0x15, 0x27, 0x24, 0x21, 0xe9, 0xb8, 0x4a, 0x34, 0xd6, 0xbd, 0x6d, 0x7b,
    0xbb, 0xe6, 0x04, 0x3e, 0x38, 0x2f, 0x34, 0x8f, 0x0d, 0x7d, 0xaa, 0x94,
    0xdc, 0x72, 0xa1, 0x52,
};
static const struct drbg_kat_no_reseed kat664_t = {
    3, kat664_entropyin, kat664_nonce, kat664_persstr,
    kat664_addin0, kat664_addin1, kat664_retbits
};
static const struct drbg_kat kat664 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat664_t
};

static const unsigned char kat665_entropyin[] = {
    0xe8, 0x76, 0xc1, 0x0c, 0x9e, 0x42, 0xc7, 0x53, 0x46, 0xd5, 0x93, 0xd6,
    0xeb, 0x04, 0x7a, 0x1f, 0xb3, 0x36, 0x73, 0x98, 0xd6, 0x23, 0x16, 0xd1,
    0x16, 0xa9, 0x29, 0xeb, 0x9e, 0xce, 0xcb, 0x18,
};
static const unsigned char kat665_nonce[] = {
    0x81, 0x5a, 0xb7, 0x63, 0x32, 0xdb, 0x44, 0xe7, 0x13, 0xa8, 0xe9, 0x67,
    0xb2, 0x0b, 0x5c, 0x1f,
};
static const unsigned char kat665_persstr[] = {0};
static const unsigned char kat665_addin0[] = {0};
static const unsigned char kat665_addin1[] = {0};
static const unsigned char kat665_retbits[] = {
    0x70, 0x22, 0xdb, 0x94, 0x7f, 0xd4, 0x18, 0xf6, 0xc9, 0x2b, 0xf9, 0xa1,
    0x2b, 0x6d, 0x1b, 0x2b, 0xd2, 0x17, 0x75, 0x8f, 0xa2, 0xe3, 0x67, 0x76,
    0xf3, 0x5c, 0x9d, 0x33, 0xa4, 0x89, 0xf6, 0x91, 0x3a, 0x1d, 0x07, 0xb4,
    0xb4, 0x61, 0xa1, 0x39, 0x11, 0x46, 0x9c, 0xcf, 0x4f, 0x3b, 0x52, 0x11,
    0x11, 0x7b, 0xdc, 0xac, 0x05, 0x2a, 0xa8, 0xee, 0x0b, 0xe7, 0xe2, 0x7c,
    0x4c, 0xa2, 0xa3, 0x45,
};
static const struct drbg_kat_no_reseed kat665_t = {
    4, kat665_entropyin, kat665_nonce, kat665_persstr,
    kat665_addin0, kat665_addin1, kat665_retbits
};
static const struct drbg_kat kat665 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat665_t
};

static const unsigned char kat666_entropyin[] = {
    0x9b, 0x3d, 0xee, 0xc2, 0x3e, 0xde, 0x86, 0x62, 0x5c, 0x5e, 0xad, 0x5f,
    0x2c, 0x77, 0xe6, 0x05, 0x46, 0x6b, 0x60, 0x22, 0x40, 0xf5, 0x56, 0xa1,
    0x80, 0xec, 0xd2, 0xbe, 0xee, 0x20, 0x1a, 0x35,
};
static const unsigned char kat666_nonce[] = {
    0xdb, 0x3d, 0x77, 0x43, 0xde, 0xfc, 0x4d, 0xca, 0x54, 0x5f, 0xbf, 0xc1,
    0xd3, 0x7a, 0x09, 0xc5,
};
static const unsigned char kat666_persstr[] = {0};
static const unsigned char kat666_addin0[] = {0};
static const unsigned char kat666_addin1[] = {0};
static const unsigned char kat666_retbits[] = {
    0x9d, 0xcd, 0xf1, 0x42, 0x78, 0xbc, 0xf9, 0x95, 0x9e, 0x2f, 0xc0, 0x7f,
    0x7b, 0x7e, 0xe4, 0x2b, 0x9d, 0x51, 0x11, 0x4e, 0x96, 0x65, 0x64, 0x34,
    0x06, 0x03, 0x62, 0x41, 0x22, 0x1c, 0x82, 0x6a, 0x59, 0x5a, 0xa1, 0xa3,
    0x39, 0x40, 0x62, 0x00, 0xe2, 0x33, 0x22, 0x20, 0x3e, 0x34, 0x67, 0xde,
    0x3f, 0x14, 0xd1, 0xe2, 0x62, 0xed, 0x8f, 0x44, 0x24, 0xf3, 0xe1, 0xe2,
    0x04, 0x3b, 0x22, 0x8b,
};
static const struct drbg_kat_no_reseed kat666_t = {
    5, kat666_entropyin, kat666_nonce, kat666_persstr,
    kat666_addin0, kat666_addin1, kat666_retbits
};
static const struct drbg_kat kat666 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat666_t
};

static const unsigned char kat667_entropyin[] = {
    0x28, 0x9f, 0x5e, 0x0e, 0x23, 0x7e, 0x45, 0xb8, 0xf7, 0xb2, 0x26, 0x63,
    0xfb, 0xb5, 0x8e, 0xbb, 0x91, 0x01, 0xbf, 0xa9, 0x71, 0xce, 0xdc, 0x1f,
    0x69, 0x77, 0xcf, 0xef, 0x02, 0x49, 0x09, 0x53,
};
static const unsigned char kat667_nonce[] = {
    0xec, 0xef, 0xca, 0xa5, 0x9d, 0x31, 0xc9, 0x69, 0x14, 0x19, 0x60, 0xf8,
    0xf5, 0xca, 0xa8, 0x57,
};
static const unsigned char kat667_persstr[] = {0};
static const unsigned char kat667_addin0[] = {0};
static const unsigned char kat667_addin1[] = {0};
static const unsigned char kat667_retbits[] = {
    0x68, 0x38, 0x06, 0x23, 0xab, 0xef, 0xe6, 0xff, 0x32, 0x10, 0xe6, 0x62,
    0xcb, 0x2c, 0xd0, 0x4e, 0xf3, 0x10, 0x92, 0xac, 0xb1, 0xde, 0xe3, 0x4f,
    0xac, 0x3a, 0x5f, 0x70, 0xcc, 0x9d, 0xa2, 0xc4, 0xf9, 0x3f, 0x87, 0x5c,
    0xbd, 0xc9, 0xef, 0x29, 0x73, 0x9c, 0x3b, 0xa6, 0xc9, 0x00, 0xc3, 0x37,
    0x50, 0x89, 0x5f, 0x10, 0x7e, 0x3b, 0xc7, 0xf2, 0x06, 0x11, 0xc4, 0x04,
    0xfb, 0x1d, 0x3d, 0x12,
};
static const struct drbg_kat_no_reseed kat667_t = {
    6, kat667_entropyin, kat667_nonce, kat667_persstr,
    kat667_addin0, kat667_addin1, kat667_retbits
};
static const struct drbg_kat kat667 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat667_t
};

static const unsigned char kat668_entropyin[] = {
    0x00, 0x8b, 0xf6, 0x98, 0x42, 0xd1, 0x05, 0x68, 0x81, 0xff, 0x2e, 0x35,
    0x71, 0x82, 0x7a, 0x69, 0x90, 0x05, 0x0f, 0x56, 0x88, 0x37, 0x7b, 0x10,
    0xb0, 0x60, 0x44, 0x12, 0xce, 0x86, 0xf5, 0x76,
};
static const unsigned char kat668_nonce[] = {
    0x7a, 0xc5, 0x16, 0x3c, 0x1a, 0x96, 0xb5, 0xf4, 0xa5, 0xb3, 0xd2, 0xf2,
    0xfe, 0xa9, 0x5d, 0x21,
};
static const unsigned char kat668_persstr[] = {0};
static const unsigned char kat668_addin0[] = {0};
static const unsigned char kat668_addin1[] = {0};
static const unsigned char kat668_retbits[] = {
    0x18, 0xa6, 0x26, 0x1e, 0xde, 0xfd, 0x65, 0xf6, 0x34, 0xfb, 0x8c, 0x80,
    0x6c, 0x61, 0x5c, 0xd2, 0xaa, 0x82, 0xca, 0x11, 0xc6, 0x08, 0xbb, 0x6e,
    0x9e, 0x83, 0x1e, 0x1e, 0xe3, 0x36, 0xac, 0xa2, 0xa2, 0x47, 0x43, 0x70,
    0xa4, 0x61, 0xf9, 0xfd, 0x1a, 0x49, 0x6a, 0xcb, 0xe9, 0x08, 0xca, 0x58,
    0xeb, 0x7b, 0xee, 0xfe, 0x2f, 0xd5, 0xc4, 0x80, 0x2c, 0xd5, 0x27, 0xae,
    0x0a, 0x1c, 0xc6, 0xa6,
};
static const struct drbg_kat_no_reseed kat668_t = {
    7, kat668_entropyin, kat668_nonce, kat668_persstr,
    kat668_addin0, kat668_addin1, kat668_retbits
};
static const struct drbg_kat kat668 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat668_t
};

static const unsigned char kat669_entropyin[] = {
    0x41, 0x9d, 0xe0, 0x8d, 0x88, 0x2e, 0x1d, 0xea, 0x84, 0x14, 0xfd, 0x01,
    0xf1, 0x85, 0x05, 0x86, 0xe5, 0x35, 0x90, 0x4e, 0xf0, 0xde, 0x3b, 0xdd,
    0x8b, 0x66, 0x7d, 0x16, 0xfe, 0xde, 0xd6, 0x3e,
};
static const unsigned char kat669_nonce[] = {
    0x52, 0x57, 0xd3, 0xe3, 0x2c, 0xf4, 0xcf, 0x5a, 0x1f, 0x61, 0x57, 0xd6,
    0x03, 0x04, 0x51, 0x0b,
};
static const unsigned char kat669_persstr[] = {0};
static const unsigned char kat669_addin0[] = {0};
static const unsigned char kat669_addin1[] = {0};
static const unsigned char kat669_retbits[] = {
    0x0b, 0x7d, 0x93, 0x54, 0x14, 0xd0, 0x72, 0xec, 0x7b, 0x24, 0x34, 0xa5,
    0x0b, 0x9b, 0xc9, 0x0a, 0x29, 0x87, 0xc0, 0x5f, 0xe5, 0x59, 0x62, 0x82,
    0xff, 0x17, 0x00, 0xf4, 0xcc, 0x44, 0x22, 0x47, 0x79, 0xc4, 0xef, 0x2f,
    0x1f, 0xe6, 0x3a, 0x4f, 0x37, 0xea, 0x88, 0x91, 0x85, 0x0b, 0xa5, 0xea,
    0xa1, 0xf9, 0x08, 0x2b, 0xa5, 0xb4, 0x99, 0xba, 0x31, 0x20, 0x36, 0x85,
    0x69, 0xd0, 0x9d, 0xb2,
};
static const struct drbg_kat_no_reseed kat669_t = {
    8, kat669_entropyin, kat669_nonce, kat669_persstr,
    kat669_addin0, kat669_addin1, kat669_retbits
};
static const struct drbg_kat kat669 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat669_t
};

static const unsigned char kat670_entropyin[] = {
    0x80, 0x39, 0x45, 0x69, 0xdc, 0x1b, 0x8d, 0xaa, 0x7d, 0x7d, 0x42, 0xc3,
    0x2f, 0x26, 0x21, 0xfa, 0xc4, 0xd5, 0xb6, 0xe7, 0x93, 0xe6, 0x82, 0x45,
    0x63, 0x14, 0x97, 0x8a, 0x2b, 0xfb, 0x49, 0xa4,
};
static const unsigned char kat670_nonce[] = {
    0x9b, 0x85, 0x63, 0x42, 0x62, 0xbd, 0xae, 0xd1, 0xc1, 0xbb, 0xfc, 0x66,
    0x9d, 0x6c, 0xd3, 0xe2,
};
static const unsigned char kat670_persstr[] = {0};
static const unsigned char kat670_addin0[] = {0};
static const unsigned char kat670_addin1[] = {0};
static const unsigned char kat670_retbits[] = {
    0x08, 0x21, 0xe1, 0x8a, 0x67, 0xfd, 0x7e, 0x78, 0xc3, 0x2f, 0x65, 0x4e,
    0x77, 0x74, 0xc1, 0xdc, 0x12, 0x44, 0x86, 0x5e, 0x9a, 0xfc, 0x12, 0xa4,
    0x30, 0xc9, 0x8c, 0x61, 0x13, 0x0f, 0x95, 0x33, 0xe9, 0x9f, 0x64, 0x1f,
    0x65, 0x45, 0x8e, 0x15, 0x5b, 0xd6, 0x7f, 0xef, 0x2f, 0x7b, 0x35, 0xeb,
    0x7e, 0xc7, 0xcb, 0x38, 0xb1, 0xe8, 0x7e, 0xb4, 0xdf, 0xa8, 0xa1, 0x73,
    0xb2, 0x34, 0x78, 0xfb,
};
static const struct drbg_kat_no_reseed kat670_t = {
    9, kat670_entropyin, kat670_nonce, kat670_persstr,
    kat670_addin0, kat670_addin1, kat670_retbits
};
static const struct drbg_kat kat670 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat670_t
};

static const unsigned char kat671_entropyin[] = {
    0xdc, 0xc9, 0xad, 0xf0, 0x02, 0x91, 0x78, 0xd4, 0xd7, 0x1d, 0xfe, 0x39,
    0x79, 0x0a, 0xa8, 0x50, 0x5e, 0xa7, 0x16, 0x81, 0xf5, 0x65, 0x29, 0x47,
    0x84, 0xfc, 0xc3, 0x63, 0x12, 0xd4, 0xd7, 0xf4,
};
static const unsigned char kat671_nonce[] = {
    0xe4, 0xa3, 0x44, 0xda, 0xab, 0x48, 0xeb, 0x31, 0xd7, 0xe8, 0xdc, 0x38,
    0x93, 0x6f, 0x18, 0x84,
};
static const unsigned char kat671_persstr[] = {0};
static const unsigned char kat671_addin0[] = {0};
static const unsigned char kat671_addin1[] = {0};
static const unsigned char kat671_retbits[] = {
    0x5e, 0xac, 0xae, 0xd6, 0x00, 0x52, 0xac, 0x87, 0xa1, 0x5e, 0x42, 0x99,
    0x7c, 0xfc, 0x77, 0xb6, 0x9e, 0x89, 0x8c, 0xbc, 0x61, 0xe3, 0x6e, 0x00,
    0x0a, 0xf2, 0x0d, 0xc5, 0x3d, 0xd5, 0xc8, 0xa1, 0xfc, 0xda, 0x92, 0x4e,
    0x10, 0x30, 0xb5, 0x35, 0xc9, 0xe4, 0xdc, 0x87, 0xc3, 0xec, 0x8d, 0x3c,
    0xe4, 0xc0, 0x61, 0xdd, 0x46, 0xde, 0x12, 0xf9, 0x53, 0x10, 0x67, 0xca,
    0x87, 0xb8, 0xbd, 0x6d,
};
static const struct drbg_kat_no_reseed kat671_t = {
    10, kat671_entropyin, kat671_nonce, kat671_persstr,
    kat671_addin0, kat671_addin1, kat671_retbits
};
static const struct drbg_kat kat671 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat671_t
};

static const unsigned char kat672_entropyin[] = {
    0x1f, 0xee, 0x8b, 0x75, 0x7e, 0xab, 0xea, 0xb6, 0x60, 0xdd, 0x95, 0xdc,
    0x0d, 0x0d, 0x69, 0xd1, 0x82, 0x22, 0x8f, 0x24, 0x1d, 0x48, 0xc6, 0x40,
    0x67, 0xf8, 0x0d, 0xd9, 0x61, 0x58, 0xcf, 0xae,
};
static const unsigned char kat672_nonce[] = {
    0x6e, 0xb7, 0x6c, 0x5c, 0x85, 0xc3, 0xa3, 0x5d, 0x6f, 0x2a, 0xf4, 0x61,
    0x9c, 0x54, 0xae, 0x21,
};
static const unsigned char kat672_persstr[] = {0};
static const unsigned char kat672_addin0[] = {0};
static const unsigned char kat672_addin1[] = {0};
static const unsigned char kat672_retbits[] = {
    0x8a, 0x6f, 0x21, 0x15, 0x6f, 0xd6, 0x1e, 0x1a, 0x83, 0x6f, 0x6b, 0x1f,
    0x54, 0x5a, 0x06, 0xc8, 0x10, 0x08, 0xe8, 0x7c, 0x85, 0xfe, 0x52, 0x6c,
    0x3b, 0x82, 0xdb, 0xbb, 0xc1, 0x19, 0x43, 0xdb, 0xaf, 0x25, 0xfe, 0x48,
    0xee, 0x31, 0xfb, 0x70, 0x18, 0x51, 0x0b, 0x2e, 0xec, 0xdd, 0x86, 0xb2,
    0x1d, 0x17, 0x79, 0xd5, 0xf7, 0x0e, 0xa2, 0x8b, 0x44, 0xb1, 0xb2, 0x94,
    0xa5, 0x66, 0x34, 0xb6,
};
static const struct drbg_kat_no_reseed kat672_t = {
    11, kat672_entropyin, kat672_nonce, kat672_persstr,
    kat672_addin0, kat672_addin1, kat672_retbits
};
static const struct drbg_kat kat672 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat672_t
};

static const unsigned char kat673_entropyin[] = {
    0xb3, 0xed, 0x24, 0x78, 0x72, 0x21, 0x0b, 0x15, 0xc9, 0x76, 0xf4, 0x23,
    0x09, 0x6e, 0x3d, 0x7f, 0x6e, 0x62, 0x60, 0x45, 0xb2, 0x88, 0x38, 0x03,
    0x26, 0x2a, 0x96, 0xac, 0xa7, 0x15, 0xe2, 0xf7,
};
static const unsigned char kat673_nonce[] = {
    0xa1, 0xf2, 0xb4, 0x5c, 0xa6, 0x3d, 0x4c, 0x2a, 0x98, 0xa3, 0x80, 0xcb,
    0x24, 0xe4, 0x8b, 0xcd,
};
static const unsigned char kat673_persstr[] = {0};
static const unsigned char kat673_addin0[] = {0};
static const unsigned char kat673_addin1[] = {0};
static const unsigned char kat673_retbits[] = {
    0xbe, 0x7b, 0x55, 0x72, 0x51, 0x0a, 0x1c, 0x1d, 0x24, 0x5c, 0xc2, 0x72,
    0x5f, 0xf9, 0x27, 0xb3, 0x71, 0xee, 0xee, 0xa6, 0xea, 0xcc, 0x62, 0xba,
    0x6a, 0xee, 0xb9, 0x54, 0x3a, 0xf5, 0x34, 0x9f, 0xb9, 0xda, 0x16, 0x6e,
    0xec, 0xa6, 0xe9, 0x51, 0x03, 0x16, 0xe8, 0xd8, 0x9d, 0x5d, 0x06, 0x02,
    0x34, 0x01, 0x25, 0x87, 0xab, 0x0b, 0xaf, 0x57, 0xa7, 0x08, 0xc1, 0xb6,
    0x70, 0x3d, 0xfc, 0x9d,
};
static const struct drbg_kat_no_reseed kat673_t = {
    12, kat673_entropyin, kat673_nonce, kat673_persstr,
    kat673_addin0, kat673_addin1, kat673_retbits
};
static const struct drbg_kat kat673 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat673_t
};

static const unsigned char kat674_entropyin[] = {
    0x70, 0xcc, 0x7b, 0x7b, 0xc8, 0x03, 0xab, 0x30, 0x36, 0x89, 0x72, 0xcc,
    0x02, 0x1c, 0x1c, 0x51, 0xa3, 0xf1, 0x51, 0x33, 0x5a, 0xdd, 0x72, 0x6f,
    0x5b, 0xd0, 0x84, 0x13, 0x86, 0xc9, 0x66, 0x05,
};
static const unsigned char kat674_nonce[] = {
    0x26, 0x67, 0x87, 0x77, 0x91, 0x9a, 0xc8, 0xf1, 0xb3, 0x8f, 0x93, 0x6b,
    0xc0, 0xab, 0x9e, 0x11,
};
static const unsigned char kat674_persstr[] = {0};
static const unsigned char kat674_addin0[] = {0};
static const unsigned char kat674_addin1[] = {0};
static const unsigned char kat674_retbits[] = {
    0x55, 0xa4, 0x81, 0xfe, 0x7b, 0x3c, 0x1c, 0xd2, 0x9b, 0x94, 0x2a, 0x71,
    0x64, 0xac, 0x7c, 0x74, 0x13, 0xdb, 0x97, 0x11, 0xfb, 0xab, 0x14, 0xcf,
    0x00, 0x58, 0x87, 0xcc, 0x53, 0xfc, 0x73, 0x9c, 0x68, 0xe3, 0x7c, 0x0b,
    0x13, 0xcb, 0x73, 0xe5, 0x3e, 0x9d, 0x27, 0x2a, 0xf8, 0x03, 0x17, 0x04,
    0x43, 0xee, 0x09, 0x11, 0x08, 0x63, 0xbf, 0x7c, 0x10, 0xc4, 0x10, 0x2f,
    0x2a, 0xce, 0xd8, 0xea,
};
static const struct drbg_kat_no_reseed kat674_t = {
    13, kat674_entropyin, kat674_nonce, kat674_persstr,
    kat674_addin0, kat674_addin1, kat674_retbits
};
static const struct drbg_kat kat674 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat674_t
};

static const unsigned char kat675_entropyin[] = {
    0x43, 0x74, 0x96, 0xcc, 0x89, 0x6a, 0x20, 0xdd, 0x6d, 0xbf, 0x33, 0xcd,
    0x47, 0x53, 0x2f, 0x99, 0x24, 0xe7, 0x5f, 0x26, 0xe4, 0x3a, 0x37, 0xdb,
    0xe9, 0xf4, 0x40, 0xf5, 0xf6, 0x11, 0x36, 0xe2,
};
static const unsigned char kat675_nonce[] = {
    0x65, 0x40, 0xd6, 0xc5, 0x86, 0xeb, 0x24, 0xd1, 0x64, 0xe3, 0xc5, 0xdb,
    0x93, 0x83, 0x82, 0xa0,
};
static const unsigned char kat675_persstr[] = {0};
static const unsigned char kat675_addin0[] = {0};
static const unsigned char kat675_addin1[] = {0};
static const unsigned char kat675_retbits[] = {
    0x4b, 0x64, 0x8f, 0x49, 0xa1, 0x48, 0x88, 0x05, 0x71, 0xbb, 0xf7, 0x17,
    0xab, 0x21, 0x3d, 0x2d, 0x92, 0xfa, 0x1f, 0x2a, 0x69, 0x83, 0x58, 0x4b,
    0xc5, 0xf3, 0xdb, 0x21, 0x2e, 0x67, 0x47, 0xc5, 0xee, 0x8a, 0x66, 0x05,
    0xe1, 0xbd, 0x14, 0xab, 0x9e, 0xcd, 0xf5, 0x2c, 0x2a, 0xc6, 0x51, 0x88,
    0x21, 0xe0, 0xd3, 0x72, 0xae, 0xa5, 0x7e, 0x79, 0xa2, 0x61, 0x7a, 0x5d,
    0x43, 0x5c, 0x20, 0xb7,
};
static const struct drbg_kat_no_reseed kat675_t = {
    14, kat675_entropyin, kat675_nonce, kat675_persstr,
    kat675_addin0, kat675_addin1, kat675_retbits
};
static const struct drbg_kat kat675 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 0, 64, &kat675_t
};

static const unsigned char kat676_entropyin[] = {
    0x48, 0x1e, 0x50, 0x5b, 0xf7, 0xa3, 0x6f, 0x9d, 0x96, 0x69, 0x0d, 0x49,
    0x15, 0x4d, 0x98, 0xd6, 0xa2, 0x47, 0xc1, 0x4a, 0x70, 0x3d, 0xbf, 0xed,
    0x7c, 0xf1, 0xb7, 0xa7, 0x1b, 0xee, 0x73, 0x7f,
};
static const unsigned char kat676_nonce[] = {
    0x70, 0xbd, 0xed, 0xbc, 0x68, 0x25, 0xc4, 0xfe, 0x0a, 0x9f, 0x7e, 0x45,
    0x29, 0x0d, 0xdd, 0x51,
};
static const unsigned char kat676_persstr[] = {0};
static const unsigned char kat676_addin0[] = {
    0x5b, 0x07, 0x61, 0x0c, 0x2c, 0x94, 0x6e, 0xda, 0x29, 0x75, 0xa2, 0x6d,
    0xda, 0xdf, 0x7d, 0x73, 0xe3, 0xd2, 0x87, 0xe9, 0x23, 0xd9, 0xb1, 0xa2,
    0xd2, 0x07, 0x07, 0x76, 0xa4, 0x46, 0xd8, 0xe6,
};
static const unsigned char kat676_addin1[] = {
    0x27, 0x92, 0xa9, 0x88, 0xeb, 0xb2, 0xe7, 0x68, 0xee, 0xe0, 0xd5, 0xc2,
    0x63, 0xbc, 0xd7, 0x6a, 0x67, 0x5d, 0x6f, 0x33, 0x9e, 0x5f, 0x1a, 0xb2,
    0xca, 0x59, 0x5e, 0x6b, 0x3b, 0x4d, 0x02, 0x4a,
};
static const unsigned char kat676_retbits[] = {
    0x30, 0x34, 0x48, 0xa3, 0x55, 0xfc, 0x0a, 0x69, 0xa1, 0x30, 0xb6, 0xab,
    0x19, 0x49, 0x97, 0xb2, 0x20, 0x97, 0x0b, 0xf6, 0x80, 0x91, 0x49, 0x13,
    0xda, 0x90, 0x4e, 0x92, 0x10, 0x9d, 0xee, 0x3d, 0x9f, 0x23, 0x87, 0x11,
    0x30, 0xc4, 0x07, 0x04, 0x5c, 0xf4, 0x63, 0xce, 0x78, 0x3a, 0x5d, 0xfa,
    0xfd, 0x60, 0x3a, 0x83, 0x84, 0x79, 0x05, 0x73, 0xaf, 0x38, 0x5d, 0x47,
    0x9a, 0xcd, 0x72, 0x06,
};
static const struct drbg_kat_no_reseed kat676_t = {
    0, kat676_entropyin, kat676_nonce, kat676_persstr,
    kat676_addin0, kat676_addin1, kat676_retbits
};
static const struct drbg_kat kat676 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat676_t
};

static const unsigned char kat677_entropyin[] = {
    0xe4, 0xb6, 0x1f, 0x03, 0x62, 0xcc, 0xf3, 0xc1, 0x50, 0x1c, 0x60, 0x93,
    0x70, 0xfd, 0x56, 0x05, 0x88, 0xb9, 0xc1, 0xa5, 0x25, 0x95, 0xe0, 0x66,
    0xf7, 0x90, 0xf4, 0x35, 0x5b, 0xa1, 0x41, 0x6d,
};
static const unsigned char kat677_nonce[] = {
    0xae, 0x47, 0x32, 0x1a, 0xb5, 0xff, 0xec, 0x92, 0x7e, 0x54, 0x61, 0x69,
    0x61, 0x23, 0xbe, 0x8f,
};
static const unsigned char kat677_persstr[] = {0};
static const unsigned char kat677_addin0[] = {
    0xa5, 0x9f, 0x6d, 0x3e, 0xe5, 0xc8, 0x71, 0x14, 0x7e, 0xbc, 0x2d, 0x5f,
    0x6e, 0x6c, 0x70, 0xfd, 0x9b, 0x98, 0x5d, 0xa7, 0xf7, 0xdd, 0x04, 0x9c,
    0xe1, 0x94, 0x46, 0x2d, 0x9c, 0x83, 0xdf, 0xc6,
};
static const unsigned char kat677_addin1[] = {
    0x9f, 0xe2, 0xc7, 0xdb, 0x11, 0x36, 0x79, 0x81, 0x47, 0x41, 0x86, 0xd9,
    0x22, 0xd9, 0x3e, 0xdf, 0x6a, 0xc7, 0xaa, 0x72, 0xa3, 0xe1, 0x59, 0xf5,
    0xc4, 0x0c, 0xcf, 0x90, 0x1d, 0x52, 0x3e, 0x28,
};
static const unsigned char kat677_retbits[] = {
    0x70, 0xa7, 0x8b, 0x73, 0x03, 0xf9, 0x02, 0xa7, 0x62, 0x21, 0xa4, 0x01,
    0xeb, 0xe1, 0x34, 0xa6, 0x31, 0x7c, 0xbe, 0x61, 0x77, 0xd0, 0xb8, 0x27,
    0x99, 0x36, 0x0c, 0x49, 0x13, 0xaf, 0xa2, 0xa8, 0xc2, 0xb3, 0x6c, 0x0e,
    0x8a, 0x13, 0x58, 0x71, 0xc3, 0xc4, 0x00, 0x09, 0x60, 0xfa, 0xed, 0x37,
    0x28, 0xc1, 0xfb, 0xd0, 0x1e, 0xe0, 0xef, 0xc5, 0xc6, 0x29, 0xa0, 0x96,
    0x77, 0xc7, 0xa8, 0x50,
};
static const struct drbg_kat_no_reseed kat677_t = {
    1, kat677_entropyin, kat677_nonce, kat677_persstr,
    kat677_addin0, kat677_addin1, kat677_retbits
};
static const struct drbg_kat kat677 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat677_t
};

static const unsigned char kat678_entropyin[] = {
    0xd7, 0x53, 0x45, 0x7f, 0x99, 0xc1, 0xc6, 0xf1, 0xda, 0x21, 0x1a, 0xa2,
    0xa5, 0xef, 0xd0, 0xe8, 0x80, 0xae, 0x06, 0x89, 0xe8, 0xae, 0x29, 0x6c,
    0xec, 0x3b, 0x1b, 0x6c, 0x8f, 0x81, 0x61, 0x89,
};
static const unsigned char kat678_nonce[] = {
    0xf6, 0x5f, 0x7b, 0x22, 0x2b, 0x51, 0x8e, 0x00, 0x72, 0x10, 0x8f, 0xbe,
    0x99, 0xb6, 0x20, 0xe7,
};
static const unsigned char kat678_persstr[] = {0};
static const unsigned char kat678_addin0[] = {
    0x6d, 0x58, 0x7c, 0xaf, 0xec, 0x41, 0x39, 0x76, 0x6e, 0xdf, 0xf5, 0xac,
    0xe6, 0xb6, 0x3d, 0x96, 0x92, 0x86, 0x2c, 0x99, 0xbf, 0xcd, 0xe6, 0x04,
    0xfe, 0xde, 0xd6, 0xd9, 0xf6, 0xb3, 0xb4, 0x70,
};
static const unsigned char kat678_addin1[] = {
    0xdb, 0x18, 0x4a, 0xe8, 0xb3, 0xf7, 0x3a, 0xa2, 0x2b, 0x63, 0xdc, 0xc2,
    0xd2, 0x1f, 0x14, 0xb9, 0xfd, 0x17, 0xae, 0xf3, 0xa9, 0x6b, 0xa5, 0xf9,
    0x82, 0xca, 0xaa, 0x0c, 0x16, 0x69, 0x03, 0x71,
};
static const unsigned char kat678_retbits[] = {
    0xeb, 0x5e, 0x5c, 0xa3, 0xc8, 0x27, 0x57, 0x6e, 0x89, 0x7e, 0x26, 0x84,
    0xe2, 0x03, 0x75, 0x3b, 0xb1, 0xa3, 0xba, 0xcb, 0x7a, 0x1d, 0x55, 0x17,
    0xc6, 0x18, 0x5a, 0xdb, 0x9d, 0xdb, 0xed, 0xa8, 0x68, 0x4f, 0x6c, 0xad,
    0x3d, 0xde, 0x64, 0x17, 0x10, 0x67, 0x04, 0xfa, 0x29, 0xf8, 0xa1, 0x02,
    0x72, 0x5b, 0x7d, 0x80, 0xee, 0x2c, 0xac, 0xa1, 0x4c, 0x0c, 0x4c, 0x87,
    0x3c, 0x7d, 0x64, 0x6e,
};
static const struct drbg_kat_no_reseed kat678_t = {
    2, kat678_entropyin, kat678_nonce, kat678_persstr,
    kat678_addin0, kat678_addin1, kat678_retbits
};
static const struct drbg_kat kat678 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat678_t
};

static const unsigned char kat679_entropyin[] = {
    0x0e, 0x99, 0xc1, 0x46, 0x6a, 0xfb, 0x22, 0xf0, 0xe0, 0x1f, 0x7e, 0x69,
    0x19, 0xa8, 0x75, 0x95, 0x9c, 0x3d, 0x37, 0xc7, 0x1b, 0xca, 0x6d, 0x5d,
    0x59, 0xe0, 0xec, 0xee, 0xe4, 0x91, 0x86, 0xcd,
};
static const unsigned char kat679_nonce[] = {
    0x95, 0xa8, 0xdd, 0x75, 0xb0, 0x76, 0x45, 0x0e, 0xa6, 0xf8, 0xa7, 0x02,
    0x07, 0xfb, 0x15, 0x04,
};
static const unsigned char kat679_persstr[] = {0};
static const unsigned char kat679_addin0[] = {
    0x99, 0x92, 0xb0, 0x88, 0xe9, 0xc7, 0x6f, 0x49, 0xdd, 0xc5, 0x45, 0x40,
    0x9d, 0x84, 0x91, 0xa9, 0x04, 0x57, 0x57, 0x0b, 0x3d, 0x29, 0xae, 0x69,
    0xe3, 0xcc, 0x18, 0x97, 0xe9, 0x8d, 0x66, 0xfc,
};
static const unsigned char kat679_addin1[] = {
    0xc5, 0xee, 0xfd, 0xfb, 0x46, 0x56, 0xd3, 0xf9, 0x88, 0xca, 0x19, 0xef,
    0xff, 0xee, 0xbe, 0xc7, 0x81, 0xa1, 0xe1, 0xb1, 0xa2, 0x99, 0x7c, 0x3f,
    0xb7, 0x6e, 0x1d, 0x0d, 0xac, 0xda, 0xac, 0x8e,
};
static const unsigned char kat679_retbits[] = {
    0x4b, 0x0d, 0x76, 0x47, 0xbe, 0xb9, 0x59, 0xf7, 0x12, 0xc1, 0x01, 0xc4,
    0x07, 0xa2, 0x5f, 0xf0, 0x45, 0x7f, 0xa8, 0x28, 0x7f, 0x25, 0xc5, 0xf2,
    0xdb, 0x97, 0x11, 0x5d, 0x4c, 0xb8, 0xcc, 0x35, 0x39, 0xc5, 0x0e, 0x6d,
    0x51, 0xc5, 0xf3, 0xa7, 0x25, 0xb4, 0x3e, 0xfb, 0x0f, 0xea, 0x33, 0x95,
    0x29, 0x5c, 0xbb, 0xac, 0x1e, 0x91, 0x9d, 0x41, 0xff, 0xc9, 0xf4, 0xaf,
    0xf8, 0x2e, 0x8f, 0x7e,
};
static const struct drbg_kat_no_reseed kat679_t = {
    3, kat679_entropyin, kat679_nonce, kat679_persstr,
    kat679_addin0, kat679_addin1, kat679_retbits
};
static const struct drbg_kat kat679 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat679_t
};

static const unsigned char kat680_entropyin[] = {
    0x2b, 0xc4, 0x49, 0xcb, 0xc2, 0x23, 0x75, 0x6c, 0x12, 0x03, 0x54, 0x64,
    0x9a, 0xdc, 0x79, 0xd0, 0xb5, 0x3a, 0xef, 0x53, 0x43, 0xa6, 0x19, 0x00,
    0xb2, 0x54, 0x23, 0x5c, 0xb6, 0x15, 0x24, 0xba,
};
static const unsigned char kat680_nonce[] = {
    0x6a, 0xa9, 0xdd, 0x15, 0x7d, 0xbb, 0x1b, 0xbf, 0x41, 0x85, 0xe1, 0xcc,
    0x2c, 0xa3, 0xb5, 0x53,
};
static const unsigned char kat680_persstr[] = {0};
static const unsigned char kat680_addin0[] = {
    0xc0, 0x7d, 0x22, 0xdb, 0x1a, 0xdb, 0x41, 0xd8, 0xae, 0xc8, 0x46, 0xff,
    0xaf, 0x44, 0xcc, 0x83, 0x3e, 0xa1, 0xe2, 0x6d, 0x5d, 0x07, 0x18, 0xa5,
    0x16, 0x9f, 0xa0, 0x69, 0xd6, 0x3e, 0xdc, 0x5c,
};
static const unsigned char kat680_addin1[] = {
    0xd4, 0x1f, 0x91, 0xb2, 0x02, 0x89, 0xdf, 0xa4, 0x9c, 0x2a, 0x33, 0x52,
    0x20, 0x04, 0x27, 0xd4, 0x57, 0x2e, 0x98, 0x13, 0x38, 0x11, 0x27, 0x32,
    0x4f, 0x49, 0x80, 0x7b, 0x47, 0x54, 0x17, 0x48,
};
static const unsigned char kat680_retbits[] = {
    0x82, 0x0d, 0xb9, 0xb7, 0xef, 0x67, 0x8c, 0xbf, 0xa8, 0x44, 0x9c, 0xe3,
    0xdf, 0xf4, 0xb7, 0x68, 0x36, 0x20, 0x9a, 0xb8, 0x8e, 0xd8, 0xdc, 0xa3,
    0x39, 0x08, 0x4e, 0x46, 0xd1, 0xbf, 0xbd, 0x0a, 0xa5, 0x53, 0xd4, 0x10,
    0x09, 0xd9, 0x46, 0x59, 0xe6, 0x10, 0xac, 0xa9, 0xb8, 0xe1, 0xeb, 0xe2,
    0x35, 0xb6, 0x43, 0x7f, 0x36, 0x02, 0x71, 0xed, 0xc5, 0xc0, 0x5b, 0xae,
    0x2a, 0x63, 0xfd, 0xc3,
};
static const struct drbg_kat_no_reseed kat680_t = {
    4, kat680_entropyin, kat680_nonce, kat680_persstr,
    kat680_addin0, kat680_addin1, kat680_retbits
};
static const struct drbg_kat kat680 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat680_t
};

static const unsigned char kat681_entropyin[] = {
    0x3f, 0xbe, 0xc8, 0xe6, 0x8c, 0xaa, 0x29, 0x1d, 0x44, 0xb6, 0x5e, 0xf6,
    0x22, 0x81, 0x8f, 0x80, 0xad, 0x29, 0x5b, 0xeb, 0x06, 0x7d, 0x78, 0x5f,
    0xae, 0x69, 0x30, 0xec, 0x87, 0x17, 0x51, 0x7a,
};
static const unsigned char kat681_nonce[] = {
    0x8a, 0x3b, 0xb4, 0xd1, 0x5f, 0xf0, 0xb8, 0x43, 0x7b, 0xe6, 0xab, 0xc0,
    0x03, 0xaf, 0x0f, 0xa6,
};
static const unsigned char kat681_persstr[] = {0};
static const unsigned char kat681_addin0[] = {
    0x6c, 0x8c, 0x9c, 0x35, 0x85, 0x3e, 0x23, 0x82, 0x12, 0xa8, 0x33, 0x59,
    0xb5, 0xe0, 0x18, 0x52, 0xb6, 0xec, 0xc3, 0x4d, 0xfe, 0x7a, 0x25, 0x8f,
    0x51, 0x5c, 0x5d, 0x70, 0x58, 0xbb, 0x02, 0x86,
};
static const unsigned char kat681_addin1[] = {
    0x20, 0x7e, 0xc2, 0x1f, 0x30, 0x51, 0xe0, 0x39, 0x59, 0xfd, 0x2d, 0x44,
    0x74, 0x76, 0xb8, 0x93, 0xdb, 0xa1, 0xfe, 0xf8, 0xfe, 0x42, 0x26, 0x32,
    0xfe, 0xef, 0x23, 0xe3, 0xef, 0x8a, 0x02, 0x2a,
};
static const unsigned char kat681_retbits[] = {
    0xb9, 0x43, 0x5c, 0x54, 0x9a, 0x05, 0x05, 0x15, 0xfd, 0xdc, 0xa4, 0x54,
    0xf6, 0x02, 0xfd, 0xcd, 0x17, 0xfc, 0xb8, 0x12, 0xbe, 0x0e, 0x1c, 0x64,
    0x0f, 0xf8, 0x59, 0xe1, 0x17, 0x2c, 0x2a, 0xb0, 0x44, 0x91, 0xdb, 0x15,
    0x28, 0x19, 0xfa, 0x9a, 0x33, 0xfc, 0xb1, 0xe6, 0x6d, 0xc8, 0x21, 0x1c,
    0x8f, 0x36, 0x76, 0xa2, 0x81, 0x03, 0xb2, 0x03, 0xbd, 0x2d, 0x0a, 0x62,
    0x98, 0xbc, 0xc1, 0x67,
};
static const struct drbg_kat_no_reseed kat681_t = {
    5, kat681_entropyin, kat681_nonce, kat681_persstr,
    kat681_addin0, kat681_addin1, kat681_retbits
};
static const struct drbg_kat kat681 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat681_t
};

static const unsigned char kat682_entropyin[] = {
    0x55, 0xfa, 0xfc, 0x70, 0xde, 0xeb, 0xba, 0x97, 0x30, 0x6d, 0xa5, 0xb7,
    0x44, 0xff, 0x4e, 0x48, 0x0e, 0x3c, 0xc2, 0xbc, 0xc8, 0xfa, 0x20, 0xff,
    0xd1, 0x7c, 0xa8, 0x51, 0x36, 0x8b, 0xf5, 0x64,
};
static const unsigned char kat682_nonce[] = {
    0x40, 0x8f, 0x2d, 0xf2, 0x7e, 0x5b, 0xe6, 0x8d, 0x25, 0x48, 0x16, 0x68,
    0x7a, 0x63, 0xa4, 0x19,
};
static const unsigned char kat682_persstr[] = {0};
static const unsigned char kat682_addin0[] = {
    0x64, 0x30, 0x4f, 0x3d, 0xf4, 0x48, 0x6c, 0xf6, 0xf0, 0x12, 0x25, 0x2f,
    0x9c, 0xa9, 0xc1, 0x71, 0x04, 0xab, 0xe4, 0x86, 0x3d, 0x1e, 0x3d, 0xb5,
    0xab, 0x81, 0x74, 0x76, 0xd4, 0x30, 0x05, 0xa6,
};
static const unsigned char kat682_addin1[] = {
    0xfd, 0xe8, 0x74, 0x9c, 0x2b, 0xef, 0xe8, 0xb0, 0x6c, 0x79, 0xcb, 0xc1,
    0x60, 0xcc, 0xfe, 0x95, 0x79, 0xa7, 0xe7, 0x62, 0x55, 0x0c, 0xb8, 0x22,
    0x17, 0x75, 0xa5, 0x4e, 0x98, 0x2f, 0xb7, 0x4c,
};
static const unsigned char kat682_retbits[] = {
    0x70, 0x9a, 0xa6, 0x47, 0xef, 0xe0, 0xe6, 0x8d, 0xfa, 0xdf, 0x24, 0x37,
    0xea, 0x75, 0x56, 0x38, 0x1e, 0xa4, 0x30, 0x90, 0x94, 0xc1, 0x04, 0xcd,
    0xb8, 0xe4, 0xbc, 0xc3, 0x1c, 0x74, 0x71, 0x6f, 0xe9, 0xee, 0xa2, 0x32,
    0x15, 0x81, 0x70, 0xfa, 0x97, 0x8a, 0x7c, 0x05, 0x81, 0x3f, 0x25, 0x92,
    0xb3, 0x52, 0x2e, 0x5c, 0x44, 0xf2, 0x52, 0x6e, 0x76, 0x4e, 0x66, 0xe3,
    0x12, 0xec, 0xd4, 0x9b,
};
static const struct drbg_kat_no_reseed kat682_t = {
    6, kat682_entropyin, kat682_nonce, kat682_persstr,
    kat682_addin0, kat682_addin1, kat682_retbits
};
static const struct drbg_kat kat682 = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 0, 32, 64, &kat682_t
};

static const unsigned char kat683_entropyin[] = {
    0x57, 0x3a, 0x24, 0x3d, 0x3b, 0x38, 0xfb, 0x8d, 0x6f, 0xcc, 0xd