# Copyright 2015 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Model for a filesystem on a partition or a block device."""

from __future__ import (
    absolute_import,
    print_function,
    unicode_literals,
    )

str = None

__metaclass__ = type
__all__ = [
    'Filesystem',
    ]


from django.db.models import (
    BooleanField,
    CharField,
    ForeignKey,
)
from maasserver import DefaultMeta
from maasserver.enum import (
    FILESYSTEM_TYPE,
    FILESYSTEM_TYPE_CHOICES,
)
from maasserver.models.blockdevice import BlockDevice
from maasserver.models.cacheset import CacheSet
from maasserver.models.cleansave import CleanSave
from maasserver.models.filesystemgroup import FilesystemGroup
from maasserver.models.partition import Partition
from maasserver.models.timestampedmodel import TimestampedModel


class Filesystem(CleanSave, TimestampedModel):
    """A filesystem on partition or a block device.

    :ivar uuid: UUID of the filesystem.
    :ivar fstype: Type of filesystem. This can even be filesystems that
        cannot be mounted directly, e.g. LVM.
    :ivar partition: `Partition` this filesystem is on. If empty the filesystem
        must be directly on a `BlockDevice`.
    :ivar block_device: `BlockDevice` this filesystem is on. If empty the
        filesystem must be on a `Partition`.
    :ivar filesystem_group: `FilesystemGroup` this filesystem belongs to.
    :ivar create_params: Parameters that can be passed during the `mkfs`
        command when the filesystem is created.
    :ivar mount_point: Path to where this filesystem is mounted on the deployed
        operating system.
    :ivar mount_params: Parameters that are used to mount this filesystem
        on the deployed operating system.
    """

    class Meta(DefaultMeta):
        """Needed for South to recognize this model."""
        unique_together = (
            ("partition", "acquired"),
            ("block_device", "acquired"),
            )

    uuid = CharField(
        max_length=36, unique=False, null=False, blank=False, editable=False)

    fstype = CharField(
        max_length=20, choices=FILESYSTEM_TYPE_CHOICES,
        default=FILESYSTEM_TYPE.EXT4)

    partition = ForeignKey(
        Partition, unique=False, null=True, blank=True)

    block_device = ForeignKey(
        BlockDevice, unique=False, null=True, blank=True)

    label = CharField(
        max_length=255, null=True, blank=True)

    filesystem_group = ForeignKey(
        FilesystemGroup, null=True, blank=True, related_name='filesystems')

    create_params = CharField(
        max_length=255, null=True, blank=True)

    mount_point = CharField(
        max_length=255, null=True, blank=True)

    mount_params = CharField(
        max_length=255, null=True, blank=True)

    cache_set = ForeignKey(
        CacheSet, null=True, blank=True, related_name='filesystems')

    # When a node is allocated all Filesystem objects assigned to that node
    # with mountable filesystems will be duplicated with this field set to
    # True. This allows a standard user to change this object as they want
    # and format other free devices. Once the node is released these objects
    # will be delete.
    acquired = BooleanField(default=False)
