/*
 * Collection of filters to modify streams of tagged items
 *
 * Copyright (C) 2003,2004,2005,2006  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <tests/test-utils.h>
#include <tagcoll/stream/filters.h>

#include <tagcoll/stream/sink.h>
#include <tagcoll/coll/simple.h>

namespace tut {
using namespace std;
using namespace tagcoll;
using namespace tagcoll::stream;
using namespace tagcoll::tests;

struct tagcoll_stream_filters_shar {
};
TESTGRP(tagcoll_stream_filters);

template<> template<>
void to::test<1>()
{
	std::string input_coll(
			"a: \n"
			"b: c::D, e::F, f::g\n"
			"c: c::D, e, f::g\n"
			);
	std::string output_coll(
			"b: c::D, e::F, f::g\n"
			"c: c::D, e, f::g\n"
			);
	coll::Simple<string, string> result;
	parseCollection(input_coll, untaggedRemover(inserter(result)));

	coll::Simple<string, string> reference;
	parseCollection(output_coll, inserter(reference));

	ensure_coll_equals(reference, result);
}

template<> template<>
void to::test<2>()
{
	std::string input_coll(
			"a: \n"
			"b: c::D, e::F, f::g\n"
			"c: c::D, e, f::g\n"
			);
	std::string output_coll(
			"a: \n"
			);
	coll::Simple<string, string> result;
	parseCollection(input_coll, untaggedRemover(inserter(result), true));

	coll::Simple<string, string> reference;
	parseCollection(output_coll, inserter(reference));

	ensure_coll_equals(reference, result);
}

template<> template<>
void to::test<3>()
{
	std::string input_coll(
			"a: b, c, d::e\n"
			"b: c::D, e::F, f::g\n"
			"c: c::D, e, f::g\n"
			);
	std::string output_coll(
			"a: d::e\n"
			"b: c::D, e::F, f::g\n"
			"c: c::D, f::g\n"
			);
	coll::Simple<string, string> result;
	parseCollection(input_coll, unfacetedRemover(inserter(result))); 

	coll::Simple<string, string> reference;
	parseCollection(output_coll, inserter(reference));

	ensure_coll_equals(reference, result);
}

template<> template<>
void to::test<4>()
{
	std::string input_coll(
			"a: b, c, d---e\n"
			"b: c---D, e---F, f---g\n"
			"c: c---D, e, f---g\n"
			);
	std::string output_coll(
			"a: d---e\n"
			"b: c---D, e---F, f---g\n"
			"c: c---D, f---g\n"
			);
	coll::Simple<string, string> result;
	parseCollection(input_coll, unfacetedRemover(inserter(result), "---")); 

	coll::Simple<string, string> reference;
	parseCollection(output_coll, inserter(reference));

	ensure_coll_equals(reference, result);
}

template<> template<>
void to::test<5>()
{
	std::string input_coll(
			"a: b, c, d::e\n"
			"b: \n"
			"c: c::D, e, f::g\n"
			);
	std::string output_coll(
			"a: \n"
			"b:\n"
			"c: \n"
			);
	coll::Simple<string, string> result;
	parseCollection(input_coll, itemsOnly(inserter(result))); 

	coll::Simple<string, string> reference;
	parseCollection(output_coll, inserter(reference));

	ensure_coll_equals(reference, result);
}

template<> template<>
void to::test<6>()
{
	std::string input_coll(
			"a: 1, 2\n"
			"b: \n"
			"c: 2, 3\n"
			);
	std::string output_coll(
			"1: a\n"
			"2: a, c\n"
			"3: c\n"
			"z: b\n"
			);
	coll::Simple<string, string> result;
	parseCollection(input_coll, reverser(string("z"), inserter(result))); 

	coll::Simple<string, string> reference;
	parseCollection(output_coll, inserter(reference));

	ensure_coll_equals(reference, result);
}

template<> template<>
void to::test<7>()
{
	std::string input_coll(
			"a: 1, 2\n"
			"b: \n"
			"c: 2, 3\n"
			);
	int count1Items = 0, count1Tags = 0;
	int count2Items = 0, count2Tags = 0;

	parseCollection(input_coll, teeFilter(
				stream::countingSink(count1Items, count1Tags),
				stream::countingSink(count2Items, count2Tags)));

	ensure_equals(count1Items, 3);
	ensure_equals(count1Tags, 4);
	ensure_equals(count2Items, 3);
	ensure_equals(count2Tags, 4);
}

template<> template<>
void to::test<8>()
{
	std::string input_coll(
			"a: a1, a2, a3\n"
			"b: \n"
			"c: b1, b2, b3, b4, b5\n"
			);
	coll::Simple<string, size_t> result;
	parseCollection(input_coll, tagCounter(inserter(result))); 

	std::set<size_t> tagset;

	tagset = result.getTagsOfItem("a");
	ensure_equals(tagset.size(), 1u);
	ensure_equals(*tagset.begin(), 3u);
	tagset = result.getTagsOfItem("b");
	ensure_equals(tagset.size(), 1u);
	ensure_equals(*tagset.begin(), 0u);
	tagset = result.getTagsOfItem("c");
	ensure_equals(tagset.size(), 1u);
	ensure_equals(*tagset.begin(), 5u);
}

}

#include <tagcoll/tests/test-utils.tcc>
#include <tagcoll/TextFormat.tcc>
#include <tagcoll/stream/filters.tcc>
#include <tagcoll/coll/simple.tcc>

// vim:set ts=4 sw=4:
