# Copyright (C) 2009 Canonical Ltd
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA

"Menu definitions for IDEs."""

import os, sys

try:
    # python 2.5
    from xml.etree.ElementTree import ElementTree, Element
except ImportError:
    # python 2.4
    from elementtree.ElementTree import ElementTree, Element

try:
    from bzrlib import trace
except ImportError:
    trace = None


# Root folder name
MENU_ROOT_TITLE = "Bazaar Menus"


class MenuEntry(object):
    """Any item that can appear in a menu."""


class MenuAction(MenuEntry):
    """An action."""

    def __init__(self, text, command, icon=None, text_short=None,
        text_tip=None, conditions=None):
        """Create an action.

        :param text: text to display
        :param command: comand to run
        :param icon: suggested icon (using the free-desktop/Tango names)
        :param text_short: text to display on a toolbar
        :param text_tip: explanation of action
        :param conditions: list of bzr conditions that the menu is enabled for
        """
        self.text = text
        self.command = command
        self.icon = icon
        self.text_short = text_short
        self.text_tip = text_tip
        self.conditions = conditions

    def __repr__(self):
        return "MenuAction(%s, %s)" % (self.text, self.command)


class Menu(MenuEntry):

    def __init__(self, text):
        self.text = text
        self._children = []

    def __repr__(self):
        return "Menu(%s)" % (self.text,)

    def append(self, entry):
        self._children.append(entry)

    def __len__(self):
        return len(self._children)

    def __iter__(self):
        return iter(self._children)


class MenuSeparator(MenuEntry):
    """A separator within a menu."""

    def __repr__(self):
        return "MenuSeparator()"


class MenuStore(object):
    """A persistent store of menus."""

    def __init__(self, path=None):
        self._path = path
        self.mapper = _ETreeMenuMapper()
        self.load()

    def load(self):
        """Load the collection."""
        if self._path is None or not os.path.exists(self._path):
            self._root = Menu(MENU_ROOT_TITLE)
        else:
            try:
                etree = ElementTree(file=self._path)
            except Exception:
                if trace:
                    trace.mutter("failed to parse menus file %s" % (self._path,))
                    trace.report_exception(sys.exc_info(), sys.stderr)
                else:
                    print "failed to parse menus file %s" % (self._path,)
                self._root = Menu(MENU_ROOT_TITLE)
            else:
                self._root = self.mapper.etree_to_folder(etree)

    def save(self, path=None):
        """Save the collection.
        
        :param path: if non-None, the path to save to
          (instead of the default path set in the constructor)
        """
        if path is None:
            path = self._path

        # Backup old file if it exists
        if os.path.exists(path):
            backup_path = "%s.bak" % (path,)
            if os.path.exists(backup_path):
                os.unlink(backup_path)
            os.rename(path, backup_path)

        # Convert to xml and dump to a file
        etree = self.mapper.folder_to_etree(self._root)
        etree.write(path)
    
    def root(self):
        """Return the root folder of this collection."""
        return self._root


def _etree_indent(elem, level=0):
    # From http://effbot.org/zone/element-lib.htm
    i = "\n" + level*"  "
    if len(elem):
        if not elem.text or not elem.text.strip():
            elem.text = i + "  "
        if not elem.tail or not elem.tail.strip():
            elem.tail = i
        for elem in elem:
            _etree_indent(elem, level+1)
        if not elem.tail or not elem.tail.strip():
            elem.tail = i
    else:
        if level and (not elem.tail or not elem.tail.strip()):
            elem.tail = i


class _ETreeMenuMapper(object):

    def folder_to_etree(self, menu):
        """Convert a Menu to an ElementTree.

        :param menu: the Menu to convert
        :return: the ElementTree generated
        """
        root = self._entry_to_element(menu)
        # Make the output editable (by xml standards at least)
        _etree_indent(root)
        return ElementTree(root)

    def _entry_to_element(self, entry):
        if isinstance(entry, MenuAction):
            element = Element('action', text=entry.text, command=entry.command)
            if entry.icon:
                element.attrib['icon'] = entry.icon
            if entry.text_short:
                element.attrib['text_short'] = entry.text_short
            if entry.text_tip:
                element.attrib['text_tip'] = entry.text_tip
            if entry.conditions:
                element.attrib['conditions'] = " ".join(entry.conditions)
            return element
        elif isinstance(entry, Menu):
            folder_element = Element('menu', text=entry.text)
            for child in entry:
                child_element = self._entry_to_element(child)
                folder_element.append(child_element)
            return folder_element
        elif isinstance(entry, MenuSeparator):
            return Element('separator')
        else:
            raise AssertionError("unexpected entry %r" % (entry,))

    def etree_to_folder(self, etree):
        """Convert an ElementTree to a Menu.

        :param etree: the ElementTree to convert
        :return: the Menu generated
        """
        root = etree.getroot()
        return self._element_to_entry(root)

    def _element_to_entry(self, element):
        tag = element.tag
        if tag == 'action':
            text = element.get('text')
            command = element.get('command')
            icon = element.get('icon')
            text_short = element.get('text_short')
            text_tip = element.get('text_tip')
            conditions = element.get('conditions')
            if conditions:
                conditions = conditions.split()
            return MenuAction(text, command, icon, text_short, text_tip, conditions)
        elif tag == 'menu':
            text = element.get('text')
            folder = Menu(text)
            for child in element:
                child_entry = self._element_to_entry(child)
                folder.append(child_entry)
            return folder
        elif tag == 'separator':
            return MenuSeparator()
        else:
            raise AssertionError("unexpected element tag %s" % (tag,))


# testing
if __name__ == '__main__':
    store = MenuStore("menus-in.xml")
    store.save("menus-out.xml")
