/******************************************
 *
 * $GAMGI/src/mesa/gamgi_mesa_axes.c
 *
 * Copyright (C) 2004 Carlos Pereira
 *
 * Distributed under the terms of the GNU
 * General Public License: $GAMGI/LICENSE
 *
 */

#include "gamgi_engine.h"
#include "gamgi_mesa.h"
#include "gamgi_math.h"
#include "gamgi_chem.h"
#include "gamgi_phys.h"

#include "gamgi_math_vector.h"
#include "gamgi_math_cell.h"

static void static_sphere (double width)
{
glPushMatrix ();
glScalef (width, width, width);
glCallList (GAMGI_MESA_SPHERE);
glPopMatrix ();
}

static void static_axis (double *v, double width)
{
double small[3], big[3];
double length, angle;
double height;

length = gamgi_math_vector_length (v);
angle = acos (v[2] / length) * GAMGI_MATH_RAD_DEG;
height = length - 2 * width;
gamgi_math_vector_scale (v, small, width / length);
gamgi_math_vector_scale (v, big, height / length);

/*************************************
 * draw axis: sphere-cylinder-sphere *
 *************************************/

glPushMatrix ();
glTranslatef (small[0], small[1], small[2]);
static_sphere (width);

glPushMatrix ();
if (length - fabs (v[2]) > GAMGI_MATH_TOLERANCE_LENGTH)
  glRotatef (angle, -v[1], v[0], 0.0);
else
  glRotatef (angle, 1.0, 0.0, 0.0);
glScalef (width, width, height);
glCallList (GAMGI_MESA_CYLINDER_1);
glPopMatrix ();

glTranslatef (big[0], big[1], big[2]);
static_sphere (width);
glPopMatrix ();
}

void gamgi_mesa_axes (double *origin, double *a, double *b, double *c)
{
glTranslatef (origin[0], origin[1], origin[2]);

/*********************************
 * draw axes along a,b,c vectors *
 *********************************/

static_axis (a, GAMGI_MESA_AXES_WIDTH);
static_axis (b, GAMGI_MESA_AXES_WIDTH);
static_axis (c, GAMGI_MESA_AXES_WIDTH);
}

void gamgi_mesa_axes_color (double *origin, double *a, double *b, double *c)
{
glTranslatef (origin[0], origin[1], origin[2]);

/*********************************
 * draw axes along a,b,c vectors *
 *********************************/

glColor3f (GAMGI_MESA_AXIS_1_R, GAMGI_MESA_AXIS_1_G, GAMGI_MESA_AXIS_1_B);
static_axis (a, GAMGI_MESA_AXES_WIDTH);

glColor3f (GAMGI_MESA_AXIS_2_R, GAMGI_MESA_AXIS_2_G, GAMGI_MESA_AXIS_2_B);
static_axis (b, GAMGI_MESA_AXES_WIDTH);

glColor3f (GAMGI_MESA_AXIS_3_R, GAMGI_MESA_AXIS_3_G, GAMGI_MESA_AXIS_3_B);
static_axis (c, GAMGI_MESA_AXES_WIDTH);
}

void gamgi_mesa_axes_orbital (gamgi_orbital *orbital, gamgi_bool color)
{
double offset[3], v1[3], v2[3], v3[3];
double radius, length;

/************************************************************
 * when a frame is present axes are positioned at the lower *
 * frame corner, otherwise they are placed at the origin    *
 ************************************************************/

radius = orbital->radius; 
if (orbital->frame == TRUE)
  gamgi_math_vector_absolute (offset, -radius, -radius, -radius); 
else
  gamgi_math_vector_zero (offset);

/*****************************************************
 * axes length can be 1.0, 0.5292 (the Bohr radius), *
 * the radius (frame no) or 2 * radius (frame yes)   *
 *****************************************************/

length = 1.0;
if (orbital->axes == GAMGI_CHEM_BOHR) length = GAMGI_CHEM_A0;
else if (orbital->axes == GAMGI_CHEM_RADIUS)
  { length = (orbital->frame == TRUE) ? 2 * radius : radius; }

gamgi_math_vector_absolute (v1, length, 0.0, 0.0);
gamgi_math_vector_absolute (v2, 0.0, length, 0.0);
gamgi_math_vector_absolute (v3, 0.0, 0.0, length);

/************************************************************
 * TRUE is for mesa_draw, FALSE is for mesa_pick, mesa_grab *
 ************************************************************/

if (color == TRUE)
  gamgi_mesa_axes_color (offset, v1, v2, v3);
else
  gamgi_mesa_axes (offset, v1, v2, v3);
}

void gamgi_mesa_axes_cell (gamgi_cell *cell, gamgi_bool color)
{
double *v1, *v2, *v3;
double offset[3];

gamgi_math_cell_offset (cell, offset);

/**************************************************
 * build axes for conventional or primitive cells *
 **************************************************/

if (cell->axes_vectors == GAMGI_PHYS_CONVENTIONAL)
  { v1 = cell->a1; v2 = cell->a2; v3 = cell->a3; }
else
  { v1 = cell->p1; v2 = cell->p2; v3 = cell->p3; }

/************************************************************
 * TRUE is for mesa_draw, FALSE is for mesa_pick, mesa_grab *
 ************************************************************/

if (color == TRUE)
  gamgi_mesa_axes_color (offset, v1, v2, v3);
else
  gamgi_mesa_axes (offset, v1, v2, v3);
}

void gamgi_mesa_axes_layer (gamgi_layer *layer, gamgi_bool color)
{
double offset[3], v1[3], v2[3], v3[3];

gamgi_math_vector_zero (offset);

gamgi_math_vector_absolute (v1, 1.0, 0.0, 0.0);
gamgi_math_vector_absolute (v2, 0.0, 1.0, 0.0);
gamgi_math_vector_absolute (v3, 0.0, 0.0, 1.0);

/*************************************************
 * TRUE is for mesa_draw, FALSE is for mesa_pick *
 * (mesa_grab does not render layer axes)        *
 *************************************************/

if (color == TRUE)
  gamgi_mesa_axes_color (offset, v1, v2, v3);
else
  gamgi_mesa_axes (offset, v1, v2, v3);
}
