/*
 *  $Id: resample.c 22467 2019-09-06 14:05:01Z yeti-dn $
 *  Copyright (C) 2019 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libprocess/gwyprocesstypes.h>
#include <libprocess/arithmetic.h>
#include <libprocess/filters.h>
#include <libgwydgets/gwydgetutils.h>
#include <libgwydgets/gwycombobox.h>
#include <libgwydgets/gwystock.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwyapp.h>
#include "preview.h"

#define RESAMPLE_RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)
/* Physical pixel size change with inverse factors compared to resolutions! */
#define MAX_UPSAMPLE 16.0
#define MAX_DOWNSAMPLE 16.0

typedef struct {
    gdouble dx;
    gdouble dy;
    GwyInterpolationType interp;
    gboolean match_size;
    GwyAppDataId image;
    /* Interface only. */
    gint xypow10;
    gboolean square;
    gdouble orig_dx;
    gdouble orig_dy;
    gint orig_xres;
    gint orig_yres;
    gint xres;
    gint yres;
} ResampleArgs;

typedef struct {
    ResampleArgs *args;
    GtkObject *dx;
    GtkObject *dy;
    GtkWidget *interp;
    GtkWidget *image;
    GtkWidget *match_size;
    GtkWidget *square;
    GtkWidget *newsize;
    gboolean in_update;
    GwyDataField *dfield;
} ResampleControls;

static gboolean module_register            (void);
static void     resample                   (GwyContainer *data,
                                            GwyRunType run);
static gboolean resample_dialog            (ResampleArgs *args,
                                            GwyDataField *dfield);
static void     image_changed              (GwyDataChooser *chooser,
                                            ResampleControls *controls);
static void     match_size_changed         (GtkToggleButton *toggle,
                                            ResampleControls *controls);
static void     dx_changed                 (GtkAdjustment *adj,
                                            ResampleControls *controls);
static void     dy_changed                 (GtkAdjustment *adj,
                                            ResampleControls *controls);
static void     square_changed             (GtkToggleButton *toggle,
                                            ResampleControls *controls);
static void     resample_dialog_update     (ResampleControls *controls,
                                            ResampleArgs *args);
static void     update_new_resolutions     (ResampleControls *controls);
static void     update_sensitivity         (ResampleControls *controls);
static gboolean mould_filter               (GwyContainer *data,
                                            gint id,
                                            gpointer user_data);
static void     recalculate_new_resolutions(ResampleArgs *args);
static void     resample_sanitize_args     (ResampleArgs *args);
static void     resample_load_args         (GwyContainer *container,
                                            ResampleArgs *args);
static void     resample_save_args         (GwyContainer *container,
                                            ResampleArgs *args);

static GwyAppDataId image_id = GWY_APP_DATA_ID_NONE;

static const ResampleArgs resample_defaults = {
    1.0,
    1.0,
    GWY_INTERPOLATION_LINEAR,
    FALSE,
    GWY_APP_DATA_ID_NONE,
    /* Interface */
    0, FALSE, 0.0, 0.0, 0, 0, 0, 0,
};

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Resamples data to specified pixel size."),
    "Yeti <yeti@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti)",
    "2019",
};

GWY_MODULE_QUERY2(module_info, resample)

static gboolean
module_register(void)
{
    gwy_process_func_register("resample",
                              (GwyProcessFunc)&resample,
                              N_("/_Basic Operations/_Resample..."),
                              NULL,
                              RESAMPLE_RUN_MODES,
                              GWY_MENU_FLAG_DATA,
                              N_("Resample to pixel size"));

    return TRUE;
}

static void
resample(GwyContainer *data, GwyRunType run)
{
    GwyDataField *dfields[3], *otherfield;
    GwyContainer *otherdata;
    GQuark quark;
    gint oldid, newid;
    ResampleArgs args;
    gdouble xoff, yoff;
    gboolean ok;

    g_return_if_fail(run & RESAMPLE_RUN_MODES);
    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, dfields + 0,
                                     GWY_APP_MASK_FIELD, dfields + 1,
                                     GWY_APP_SHOW_FIELD, dfields + 2,
                                     GWY_APP_DATA_FIELD_ID, &oldid,
                                     0);
    g_return_if_fail(dfields[0]);

    resample_load_args(gwy_app_settings_get(), &args);

    if (args.match_size
        && (args.image.datano <= 0
            || args.image.id < 0
            || !mould_filter(gwy_app_data_browser_get(args.image.datano),
                             args.image.id,
                             dfields[0])))
        args.match_size = FALSE;

    args.orig_xres = gwy_data_field_get_xres(dfields[0]);
    args.orig_yres = gwy_data_field_get_yres(dfields[0]);
    args.orig_dx = gwy_data_field_get_dx(dfields[0]);
    args.orig_dy = gwy_data_field_get_dy(dfields[0]);
    xoff = gwy_data_field_get_xoffset(dfields[0]);
    yoff = gwy_data_field_get_yoffset(dfields[0]);
    if (args.match_size) {
        quark = gwy_app_get_data_key_for_id(args.image.id);
        otherdata = gwy_app_data_browser_get(args.image.datano);
        otherfield = gwy_container_get_object(otherdata, quark);
        args.dx = gwy_data_field_get_dx(otherfield);
        args.dy = gwy_data_field_get_dy(otherfield);
    }
    else {
        args.dx = CLAMP(args.dx,
                        args.orig_dx/MAX_UPSAMPLE,
                        args.orig_dx*MAX_DOWNSAMPLE);
        args.dy = CLAMP(args.dy,
                        args.orig_dy/MAX_UPSAMPLE,
                        args.orig_dy*MAX_DOWNSAMPLE);
    }
    args.square = (args.dx == args.dy);
    args.xypow10 = 3*GWY_ROUND(0.5*log10(args.orig_dx*args.orig_dy)/3.0);

    if (run == GWY_RUN_INTERACTIVE) {
        ok = resample_dialog(&args, dfields[0]);
        resample_save_args(gwy_app_settings_get(), &args);
        if (!ok)
            return;
    }
    else
        recalculate_new_resolutions(&args);

    xoff *= (args.xres*args.dx)/(args.orig_xres*args.orig_dx);
    yoff *= (args.yres*args.dy)/(args.orig_yres*args.orig_dy);
    dfields[0] = gwy_data_field_new_resampled(dfields[0],
                                              args.xres, args.yres,
                                              args.interp);
    gwy_data_field_set_xreal(dfields[0], args.xres*args.dx);
    gwy_data_field_set_yreal(dfields[0], args.yres*args.dy);
    gwy_data_field_set_xoffset(dfields[0], xoff);
    gwy_data_field_set_yoffset(dfields[0], yoff);
    if (dfields[1]) {
        dfields[1] = gwy_data_field_new_resampled(dfields[1],
                                                  args.xres, args.yres,
                                                  GWY_INTERPOLATION_LINEAR);
        gwy_data_field_set_xreal(dfields[1], args.xres*args.dx);
        gwy_data_field_set_yreal(dfields[1], args.yres*args.dy);
        gwy_data_field_set_xoffset(dfields[1], xoff);
        gwy_data_field_set_yoffset(dfields[1], yoff);
        gwy_data_field_threshold(dfields[1], 0.5, 0.0, 1.0);
    }
    if (dfields[2]) {
        dfields[2] = gwy_data_field_new_resampled(dfields[2],
                                                  args.xres, args.yres,
                                                  args.interp);
        gwy_data_field_set_xreal(dfields[2], args.xres*args.dx);
        gwy_data_field_set_yreal(dfields[2], args.yres*args.dy);
        gwy_data_field_set_xoffset(dfields[2], xoff);
        gwy_data_field_set_yoffset(dfields[2], yoff);
    }

    newid = gwy_app_data_browser_add_data_field(dfields[0], data, TRUE);
    g_object_unref(dfields[0]);
    gwy_app_sync_data_items(data, data, oldid, newid, FALSE,
                            GWY_DATA_ITEM_GRADIENT,
                            GWY_DATA_ITEM_RANGE,
                            GWY_DATA_ITEM_MASK_COLOR,
                            0);
    if (dfields[1]) {
        quark = gwy_app_get_mask_key_for_id(newid);
        gwy_container_set_object(data, quark, dfields[1]);
        g_object_unref(dfields[1]);
    }
    if (dfields[2]) {
        quark = gwy_app_get_show_key_for_id(newid);
        gwy_container_set_object(data, quark, dfields[2]);
        g_object_unref(dfields[2]);
    }

    gwy_app_set_data_field_title(data, newid, _("Resampled Data"));
    gwy_app_channel_log_add_proc(data, oldid, newid);
}

static gboolean
resample_dialog(ResampleArgs *args, GwyDataField *dfield)
{
    GtkWidget *dialog, *table;
    GwyDataChooser *chooser;
    ResampleControls controls;
    gint response, row;
    GwySIUnit *xyunit;
    GwySIValueFormat *vf;
    gdouble mag;

    gwy_clear(&controls, 1);
    controls.args = args;

    dialog = gtk_dialog_new_with_buttons(gwy_sgettext("Resample"), NULL, 0,
                                         _("_Reset"), RESPONSE_RESET,
                                         GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                         GTK_STOCK_OK, GTK_RESPONSE_OK,
                                         NULL);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);
    gwy_help_add_to_proc_dialog(GTK_DIALOG(dialog), GWY_HELP_DEFAULT);

    table = gtk_table_new(7, 3, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), table,
                       FALSE, FALSE, 4);
    row = 0;

    controls.image = gwy_data_chooser_new_channels();
    chooser = GWY_DATA_CHOOSER(controls.image);
    gwy_data_chooser_set_filter(chooser, mould_filter, dfield, NULL);
    gwy_data_chooser_set_active_id(chooser, &args->image);
    gwy_table_attach_adjbar(table, row++, _("_Match pixel size:"), NULL,
                            GTK_OBJECT(controls.image),
                            GWY_HSCALE_CHECK | GWY_HSCALE_WIDGET_NO_EXPAND);
    controls.match_size
        = gwy_table_hscale_get_check(GTK_OBJECT(controls.image));

    if (gwy_data_chooser_get_active_id(chooser, &args->image)) {
        g_signal_connect(controls.image, "changed",
                         G_CALLBACK(image_changed), &controls);
        g_signal_connect(controls.match_size, "toggled",
                         G_CALLBACK(match_size_changed), &controls);
    }
    else
        args->match_size = FALSE;

    xyunit = gwy_data_field_get_si_unit_xy(dfield);
    vf = gwy_si_unit_get_format_for_power10(xyunit,
                                            GWY_SI_UNIT_FORMAT_VFMARKUP,
                                            args->xypow10, NULL);
    mag = vf->magnitude;

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    controls.dx = gtk_adjustment_new(args->dx/mag,
                                     args->orig_dx/MAX_UPSAMPLE/mag,
                                     args->orig_dx*MAX_DOWNSAMPLE/mag,
                                     args->orig_dx/MAX_UPSAMPLE/64/mag,
                                     args->orig_dx/mag,
                                     0);
    gwy_table_attach_adjbar(table, row++, _("_X pixel size:"), vf->units,
                            controls.dx, GWY_HSCALE_LOG);

    controls.dy = gtk_adjustment_new(args->dy/mag,
                                     args->orig_dy/MAX_UPSAMPLE/mag,
                                     args->orig_dy*MAX_DOWNSAMPLE/mag,
                                     args->orig_dy/MAX_UPSAMPLE/64/mag,
                                     args->orig_dy/mag,
                                     0);
    gwy_table_attach_adjbar(table, row++, _("_Y pixel size:"), vf->units,
                            controls.dy, GWY_HSCALE_LOG);

    gwy_si_unit_value_format_free(vf);
    g_signal_connect(controls.dx, "value-changed",
                     G_CALLBACK(dx_changed), &controls);
    g_signal_connect(controls.dy, "value-changed",
                     G_CALLBACK(dy_changed), &controls);

    controls.square
        = gtk_check_button_new_with_mnemonic(_("_Square samples"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.square),
                                 args->square);
    g_signal_connect(controls.square, "toggled",
                     G_CALLBACK(square_changed), &controls);
    gtk_table_attach(GTK_TABLE(table), controls.square,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    controls.interp
        = gwy_enum_combo_box_new(gwy_interpolation_type_get_enum(), -1,
                                 G_CALLBACK(gwy_enum_combo_box_update_int),
                                 &args->interp, args->interp, TRUE);
    gwy_table_attach_adjbar(table, row++, _("_Interpolation type:"), NULL,
                            GTK_OBJECT(controls.interp),
                            GWY_HSCALE_WIDGET_NO_EXPAND);

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    controls.newsize = gtk_label_new("a × b");
    gwy_table_attach_adjbar(table, row++, _("New pixel dimensions:"), _("px"),
                            GTK_OBJECT(controls.newsize),
                            GWY_HSCALE_WIDGET_NO_EXPAND);

    controls.in_update = FALSE;
    resample_dialog_update(&controls, args);
    update_sensitivity(&controls);
    gtk_widget_show_all(dialog);

    do {
        response = gtk_dialog_run(GTK_DIALOG(dialog));
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            gtk_widget_destroy(dialog);
            case GTK_RESPONSE_NONE:
            return FALSE;
            break;

            case GTK_RESPONSE_OK:
            break;

            case RESPONSE_RESET:
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.match_size),
                                         FALSE);
            args->dx = args->orig_dx;
            args->dy = args->orig_dy;
            args->interp = resample_defaults.interp;
            resample_dialog_update(&controls, args);
            break;

            default:
            g_assert_not_reached();
            break;
        }
    } while (response != GTK_RESPONSE_OK);

    gtk_widget_destroy(dialog);

    return TRUE;
}

static void
image_changed(GwyDataChooser *chooser, ResampleControls *controls)
{
    ResampleArgs *args = controls->args;
    GwyContainer *otherdata;
    GwyDataField *otherfield;
    GQuark quark;

    gwy_data_chooser_get_active_id(chooser, &args->image);
    if (controls->in_update)
        return;

    quark = gwy_app_get_data_key_for_id(args->image.id);
    otherdata = gwy_app_data_browser_get(args->image.datano);
    otherfield = gwy_container_get_object(otherdata, quark);
    args->dx = gwy_data_field_get_dx(otherfield);
    args->dy = gwy_data_field_get_dy(otherfield);
    resample_dialog_update(controls, args);
}

static void
match_size_changed(GtkToggleButton *toggle, ResampleControls *controls)
{
    ResampleArgs *args = controls->args;

    args->match_size = gtk_toggle_button_get_active(toggle);
    if (controls->in_update)
        return;

    if (args->match_size)
        image_changed(GWY_DATA_CHOOSER(controls->image), controls);
    update_sensitivity(controls);
}

static void
dx_changed(GtkAdjustment *adj, ResampleControls *controls)
{
    ResampleArgs *args = controls->args;
    gdouble v;

    v = gtk_adjustment_get_value(adj);
    args->dx = v*pow10(args->xypow10);
    if (controls->in_update)
        return;

    if (args->square && !args->match_size) {
        controls->in_update = TRUE;
        gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->dy), v);
        controls->in_update = FALSE;
    }
    update_new_resolutions(controls);
}

static void
dy_changed(GtkAdjustment *adj, ResampleControls *controls)
{
    ResampleArgs *args = controls->args;
    gdouble v;

    v = gtk_adjustment_get_value(adj);
    args->dy = v*pow10(args->xypow10);
    if (controls->in_update)
        return;

    if (args->square && !args->match_size) {
        controls->in_update = TRUE;
        gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->dx), v);
        controls->in_update = FALSE;
    }
    update_new_resolutions(controls);
}

static void
square_changed(GtkToggleButton *toggle, ResampleControls *controls)
{
    ResampleArgs *args = controls->args;

    args->square = gtk_toggle_button_get_active(toggle);
    if (controls->in_update)
        return;

    if (fabs(log(args->dx/args->dy)) < 1e-6)
        return;

    args->dx = args->dy = sqrt(args->dx*args->dy);
    resample_dialog_update(controls, args);
}

static void
resample_dialog_update(ResampleControls *controls, ResampleArgs *args)
{
    controls->in_update = TRUE;
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->dx),
                             args->dx/pow10(args->xypow10));
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->dy),
                             args->dy/pow10(args->xypow10));
    gwy_enum_combo_box_set_active(GTK_COMBO_BOX(controls->interp),
                                  args->interp);
    controls->in_update = FALSE;

    update_new_resolutions(controls);
}

static void
update_new_resolutions(ResampleControls *controls)
{
    ResampleArgs *args = controls->args;
    gchar *s;

    recalculate_new_resolutions(args);
    s = g_strdup_printf("%d × %d", args->xres, args->yres);
    gtk_label_set_text(GTK_LABEL(controls->newsize), s);
    g_free(s);
}

static void
update_sensitivity(ResampleControls *controls)
{
    ResampleArgs *args = controls->args;
    gboolean any_image = (args->image.datano > 0 && args->image.id >= 0);
    gboolean is_match = args->match_size;

    gwy_table_hscale_set_sensitive(GTK_OBJECT(controls->image), any_image);
    gwy_table_hscale_set_sensitive(GTK_OBJECT(controls->dx), !is_match);
    gwy_table_hscale_set_sensitive(GTK_OBJECT(controls->dy), !is_match);
    gtk_widget_set_sensitive(controls->square, !is_match);
}

static gboolean
mould_filter(GwyContainer *data, gint id, gpointer user_data)
{
    GwyDataField *otherfield, *dfield = (GwyDataField*)user_data;
    GQuark quark;
    gdouble d, dother;

    quark = gwy_app_get_data_key_for_id(id);
    otherfield = gwy_container_get_object(data, quark);
    if (otherfield == dfield)
        return FALSE;
    if (gwy_data_field_check_compatibility(dfield, otherfield,
                                           GWY_DATA_COMPATIBILITY_LATERAL))
        return FALSE;

    d = gwy_data_field_get_dx(dfield);
    dother = gwy_data_field_get_dx(otherfield);
    if (dother > d*MAX_DOWNSAMPLE || dother < d/MAX_UPSAMPLE)
        return FALSE;

    d = gwy_data_field_get_dy(dfield);
    dother = gwy_data_field_get_dy(otherfield);
    if (dother > d*MAX_DOWNSAMPLE || dother < d/MAX_UPSAMPLE)
        return FALSE;

    return TRUE;
}

static void
recalculate_new_resolutions(ResampleArgs *args)
{
    args->xres = GWY_ROUND(args->orig_xres*args->orig_dx/args->dx);
    args->xres = MAX(args->xres, 1);
    args->yres = GWY_ROUND(args->orig_yres*args->orig_dy/args->dy);
    args->yres = MAX(args->yres, 1);
}

static const gchar dx_key[]         = "/module/resample/dx";
static const gchar dy_key[]         = "/module/resample/dy";
static const gchar interp_key[]     = "/module/resample/interp";
static const gchar match_size_key[] = "/module/resample/match_size";

static void
resample_sanitize_args(ResampleArgs *args)
{
    /* We must sanitize dx, dy according to current pixel size. */
    args->interp = gwy_enum_sanitize_value(args->interp,
                                           GWY_TYPE_INTERPOLATION_TYPE);
    args->match_size = !!args->match_size;
    gwy_app_data_id_verify_channel(&args->image);
}

static void
resample_load_args(GwyContainer *container, ResampleArgs *args)
{
    *args = resample_defaults;

    gwy_container_gis_double_by_name(container, dx_key, &args->dx);
    gwy_container_gis_double_by_name(container, dy_key, &args->dy);
    gwy_container_gis_enum_by_name(container, interp_key, &args->interp);
    gwy_container_gis_boolean_by_name(container, match_size_key,
                                      &args->match_size);
    args->image = image_id;
    resample_sanitize_args(args);
}

static void
resample_save_args(GwyContainer *container, ResampleArgs *args)
{
    gwy_container_set_double_by_name(container, dx_key, args->dx);
    gwy_container_set_double_by_name(container, dy_key, args->dy);
    gwy_container_set_enum_by_name(container, interp_key, args->interp);
    gwy_container_set_boolean_by_name(container, match_size_key,
                                      args->match_size);
    image_id = args->image;
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
