/*
 * Copyright (C) 2000-2007 the xine project
 *
 * This file is part of xine, a free video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * 
 * Xine plugin for Mozilla/Firefox 
 *      written by Claudio Ciccani <klan@users.sf.net>
 *      
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <fcntl.h>
#include <unistd.h>
#include <sys/mman.h>

#include <xine.h>
#include <xine/xineutils.h>
#include <xine/xmlparser.h>

#include "playlist.h"


#ifdef ENABLE_PLAYLIST

static inline char *trim (char *s) {
  char *e;

  for (; isspace(*s); s++);

  e = s + strlen(s) - 1;
  for (; e > s && isspace(*e); *e-- = '\0');

  return s;
}

static char *get_line (FILE *fp, char *buf, int size) {
  if (fgets (buf, size, fp) == NULL)
    return NULL;

  return trim (buf);
}

static int parse_time (const char *str) {
  int t = 0;
  int i;
  
  if (!str)
    return 0;

  if (!strncmp (str, "npt=", 4))
    str += 4;
  else if (!strncmp (str, "smpte=", 6))
    str += 6;
  
  for (i = 0; i < 3; i++) {
    t *= 60;
    t += atoi (str);
    str = strchr (str, ':');
    if (!str)
      break;
    str++;
  }

  return t*1000;
}


static int m3u_playlist_parse (FILE *fp, playlist_entry_t **list) {
  char  buf[4096];
  char *line;
  int   num = 0;

  while ((line = get_line (fp, buf, sizeof(buf)))) {
    if (*line == '\0' || *line == '#')
      continue;
    if (playlist_insert (list, line, 0))
      num++;
  }

  return num;
}

static int ram_playlist_parse (FILE *fp, playlist_entry_t **list) {
  char  buf[4096];
  char *line, *tmp;
  int   num = 0;

  while ((line = get_line (fp, buf, sizeof(buf)))) {
    if (*line == '\0' || *line == '#')
      continue;
    if (!strncmp (line, "--stop--", 8))
      break;
    /* Skip "Ref" prefix */
    if (!strncmp (line, "Ref", 3)) {
      tmp = strchr (line+3, '=');
      if (tmp)
        line = tmp + 1;
    }
    if (*line) {
      /* Strip mrl parameters (metadata) */
      if (!strncmp (line, "rtsp://", 7) || !strncmp (line, "pnm://", 6)) {
        tmp = strrchr (line, '?');
        if (tmp)
          *tmp = '\0';
      }
      if (playlist_insert (list, line, 0))
        num++;
    }
  }

  return num;
}

static int pls_playlist_parse (FILE *fp, playlist_entry_t **list) {
  char  buf[4096];
  char *line;
  int   num = 0;

  while ((line = get_line (fp, buf, sizeof(buf)))) {
    if (!strncmp (line, "File", 4)) {
      line = strchr (line+4, '=');
      if (line && *(line+1)) {
        if (playlist_insert (list, line+1, 0))
          num++;
      }
    }
  }

  return num;
}

static int asx_playlist_parse (FILE *fp, playlist_entry_t **list) {
  void       *ptr;
  size_t      size;
  xml_node_t *root, *node, *tmp;
  int         is_asx = 0;
  int         num    = 0;
  
  fseek (fp, 0, SEEK_END);
  size = ftell (fp);
  fseek (fp, 0, SEEK_SET);
  
  ptr = mmap (NULL, size, PROT_READ, MAP_SHARED, fileno(fp), 0);
  if (ptr == MAP_FAILED) {
    perror ("mmap() failed");
    return 0;
  }
    
  xml_parser_init (ptr, size, XML_PARSER_CASE_INSENSITIVE);
  
  if (xml_parser_build_tree (&root) >= 0) {
    if (!strcasecmp (root->name, "asx")) {
      is_asx = 1;
      
      for (node = root->child; node; node = node->next) {
        if (!strcasecmp (node->name, "entry")) {
          const char *src   = NULL;
          const char *start = NULL;
          
          for (tmp = node->child; tmp; tmp = tmp->next) {
            if (!strcasecmp (tmp->name, "ref")) {
              src = xml_parser_get_property (tmp, "href");
            }
            else if (!strcasecmp (tmp->name, "starttime")) {
              start = xml_parser_get_property (tmp, "value");
            }
          }
          
          if (src) {
            if (playlist_insert (list, src, parse_time(start)))
              num++;
          }
        }
      }
    }

    xml_parser_free_tree (root);
  }
  
  munmap (ptr, size);
  
  if (!is_asx) {
    char  buf[4096];
    char *src;
    /* No tags found? Might be a references list. */
    while ((src = get_line (fp, buf, sizeof(buf)))) {
      if (!strncmp (src, "Ref", 3)) {
        src = strchr (src+3, '=');
        if (src && *(src+1)) {
          if (playlist_insert (list, src+1, 0))
            num++;
        }
      }
    }
  }
  
  return num;      
}

static int smil_playlist_parse (FILE *fp, playlist_entry_t **list) {
  void       *ptr;
  size_t      size;
  xml_node_t *root, *node, *tmp;
  int         is_smi = 0;
  int         num    = 0;
  
  fseek (fp, 0, SEEK_END);
  size = ftell (fp);
  fseek (fp, 0, SEEK_SET);
  
  ptr = mmap (NULL, size, PROT_READ, MAP_SHARED, fileno(fp), 0);
  if (ptr == MAP_FAILED) {
    perror ("mmap() failed");
    return 0;
  }
    
  xml_parser_init (ptr, size, XML_PARSER_CASE_SENSITIVE);
  
  if (xml_parser_build_tree (&root) >= 0) {
    for (node = root; node; node = node->next) {
      if (!strcmp (node->name, "smil"))
        break;
    }
    
    if (node) {
      is_smi = 1;
       
      for (node = node->child; node; node = node->next) {
        if (!strcmp (node->name, "body")) {
          for (tmp = node->child; tmp; tmp = tmp->next) {
            if (!strcmp (tmp->name, "audio") || !strcmp (tmp->name, "video")) {
              const char *src, *start;
              
              src   = xml_parser_get_property (tmp, "src");
              start = xml_parser_get_property (tmp, "clipBegin") ? :
                      xml_parser_get_property (tmp, "clip-begin");
                     
              if (src) {
                if (playlist_insert (list, src, parse_time(start)))
                  num++;
              }
            }
          }
        }
      }
    }
    
    xml_parser_free_tree (root);
  }
  
  munmap (ptr, size);
  
  if (!is_smi) {
    /* No tags found? Might be a RAM playlist. */
    num = ram_playlist_parse (fp, list);
  }
  
  return num;
}

static int xspf_playlist_parse (FILE *fp, playlist_entry_t **list) {
  void       *ptr;
  size_t      size;
  xml_node_t *root, *node, *tmp;
  int         num = 0;
  
  fseek (fp, 0, SEEK_END);
  size = ftell (fp);
  fseek (fp, 0, SEEK_SET);
  
  ptr = mmap (NULL, size, PROT_READ, MAP_SHARED, fileno(fp), 0);
  if (ptr == MAP_FAILED) {
    perror ("mmap() failed");
    return 0;
  }
    
  xml_parser_init (ptr, size, XML_PARSER_CASE_SENSITIVE);
  
  if (xml_parser_build_tree (&root) >= 0) {
    for (node = root; node; node = node->next) {
      if (!strcmp (node->name, "playlist"))
        break;
    }
    if (node) {
      for (node = node->child; node; node = node->next) {
        if (!strcmp (node->name, "trackList"))
          break;
      }
    }
    if (node) { 
      for (node = node->child; node; node = node->next) {
        if (!strcmp (node->name, "track")) {
          char *src = NULL;
          
          for (tmp = node->child; tmp; tmp = tmp->next) {
            if (!strcmp (tmp->name, "location")) {
              src = tmp->data;
              break;
            }
          }
          
          if (src && playlist_insert (list, trim(src), 0))
            num++;
        }
      }
    }
    
    xml_parser_free_tree (root);
  }
  
  munmap (ptr, size);
  
  return num;
}

static int qtl_playlist_parse (FILE *fp, playlist_entry_t **list) {
  void       *ptr;
  size_t      size;
  xml_node_t *root, *node;
  int         num = 0;
  
  fseek (fp, 0, SEEK_END);
  size = ftell (fp);
  fseek (fp, 0, SEEK_SET);
  
  ptr = mmap (NULL, size, PROT_READ, MAP_SHARED, fileno(fp), 0);
  if (ptr == MAP_FAILED) {
    perror ("mmap() failed");
    return 0;
  }
    
  xml_parser_init (ptr, size, XML_PARSER_CASE_SENSITIVE);
  
  if (xml_parser_build_tree (&root) >= 0) {
    for (node = root; node; node = node->next) {
      if (!strcmp (node->name, "embed")) {
        const char *src;
        
        src = xml_parser_get_property (node, "src");
        if (src) {
          if (playlist_insert (list, src, 0))
            num++;
        }
      }
    }
    
    xml_parser_free_tree (root);
  }
  
  munmap (ptr, size);
  
  return num;
}

#endif /* ENABLE_PLAYLIST */

/******************************************************************************/

int playlist_type (const char *mimetype, const char *filename) {
#ifdef ENABLE_PLAYLIST
  char *tmp;

  if (mimetype) {
    tmp = strchr (mimetype, '/');
    if (tmp)
      tmp++;
    else
      tmp = (char *) mimetype;
      
    if (!strncmp (tmp, "x-", 2))
      tmp += 2;
    
    if (!strcmp (tmp, "mpegurl"))
      return XINE_PLT_M3U;
    if (!strcmp (tmp, "scpls"))
      return XINE_PLT_PLS;
    if (!strcmp (tmp, "ms-wvx") ||
        !strcmp (tmp, "ms-wax"))
      return XINE_PLT_ASX;
    if (!strcmp (tmp, "smil"))
      return XINE_PLT_SMI;
    if (!strcmp (tmp, "xspf+xml"))
      return XINE_PLT_XSPF;
  }
  
  if (filename) {  
    tmp = strrchr (filename, '.');
    if (tmp) {
      if (!strcasecmp (tmp, ".m3u"))
        return XINE_PLT_M3U;
      if (!strcasecmp (tmp, ".ram") || 
          !strcasecmp (tmp, ".rpm"))
        return XINE_PLT_RAM;
      if (!strcasecmp (tmp, ".pls"))
        return XINE_PLT_PLS;
      if (!strcasecmp (tmp, ".asx") ||
          !strcasecmp (tmp, ".wax") || 
          !strcasecmp (tmp, ".wvx"))
        return XINE_PLT_ASX;
      if (!strcasecmp (tmp, ".smi") || 
          !strcasecmp (tmp, ".smil"))
        return XINE_PLT_SMI;
      if (!strcasecmp (tmp, ".xspf"))
        return XINE_PLT_XSPF;
      if (!strcasecmp (tmp, ".qtl"))
        return XINE_PLT_QTL;
    }
  }
#endif /* ENABLE_PLAYLIST */

  return XINE_PLT_NONE;
}

int playlist_load (int type, const char *filename, playlist_entry_t **list) {
#ifdef ENABLE_PLAYLIST
  FILE *fp;
  int   num = 0;

  fp = fopen (filename, "r");
  if (!fp)
    return 0;

  switch (type) {
    case XINE_PLT_M3U:
      num = m3u_playlist_parse (fp, list);
      break;
    case XINE_PLT_RAM:
      num = ram_playlist_parse (fp, list);
      break;
    case XINE_PLT_PLS:
      num = pls_playlist_parse (fp, list);
      break;
    case XINE_PLT_ASX:
      num = asx_playlist_parse (fp, list);
      break;
    case XINE_PLT_SMI:
      num = smil_playlist_parse (fp, list);
      break;
    case XINE_PLT_XSPF:
      num = xspf_playlist_parse (fp, list);
      break;
    case XINE_PLT_QTL:
      num = qtl_playlist_parse (fp, list);
      break;
    default:
      break;
  }

  fclose (fp);
  
  return num;
#else
  return 0;
#endif /* ENABLE_PLAYLIST */
}
