##
##  Copyright (c) 2002 Ivo van der Wijk, Amaze Internet Services (ivo@amaze.nl)
## 
##  All rights reserved. 
## 
##  Redistribution and use in source and binary forms, with or without 
##  modification, are permitted provided that the following conditions
##  are met:
## 
## 1. Redistributions of source code must retain the above copyright
##    notice, this list of conditions and the following disclaimer.
## 2. Redistributions in binary form must reproduce the above copyright
##    notice, this list of conditions and the following disclaimer in the
##    documentation and/or other materials provided with the distribution.
## 3. The name of the author may not be used to endorse or promote products
##    derived from this software without specific prior written permission. 
##   
##   
## THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
## IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
## OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
## IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
## INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
## NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
## DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
## THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
## (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
## THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
##

import App, Globals, OFS
import string

from Persistence import Persistent

from Globals import DTMLFile
from OFS.Folder import Folder
from Acquisition import aq_base, aq_parent, Acquired, Implicit
from AccessControl import getSecurityManager, SpecialUsers

from zLOG import LOG, WARNING

def manage_addQuotaFolder(self, id, title, quota_bytes=0, quota_objects=0, 
                          quota_maxsize=0, REQUEST=None):
    """ Add a new QuotaFolder """

    o = QuotaFolder(id, title, quota_bytes, quota_maxsize, quota_objects, REQUEST)
    self._setObject(id, o)
    o = self._getOb(id)

    if REQUEST:
        return self.manage_main(self, REQUEST, update_menu=1)
    
def _replaceFolder(parent, id, quota_bytes=0, quota_objects=0, quota_maxsize=0):
    # Replaces an OFS.Folder with a QuotaFolder. Borrowed from BTreeFolder
    ob = QuotaFolder(id, "", quota_bytes, quota_maxsize, quota_objects)
    f = parent._getOb(id)
    # Copy the contents of the folder to the new QuotaFolder

    ids = f.objectIds()
    for key, value in f.__dict__.items():
        if key not in ids:
            # Not an ObjectManager item.
            ob.__dict__[key] = value
    for key in ids:
        subob = f._getOb(key)
        subob = getattr(subob, 'aq_base', subob)
        ob._setOb(key, subob)
    parent._setOb(id, ob)


manage_addQuotaFolderForm=DTMLFile('manage_addQuotaFolderForm', globals())

QuotaExceededException='Quota Exceeded'

class QuotaFolder(Folder, Persistent, Implicit):
    """ A QuotaFolder """

    meta_type='QuotaFolder'

    manage_options=(
        Folder.manage_options+
        (
        {'label': 'Quota', 'action': 'manage_editQuotaForm'},
        )
        )

    __ac_permissions__= \
        Folder.__ac_permissions__ +  \
        (
        ('View current quota', ('currentCount','currentSize')),
        )

    manage_editQuotaForm=DTMLFile('manage_editQuotaForm', globals())    

    def __init__(self, id, title="", quota=0, maxFileSize=0, maxNumberOfFiles=0,
                 manager_role=1, REQUEST=None):
        self.id=id
        self.title=title
        self._quota_bytes=int(quota)
        self._quota_objects=int(maxNumberOfFiles)
        self._quota_maxsize=int(maxFileSize)

        ##
        ## current accounting
        self._quota_filecount = 0
        self._quota_size = 0

        self._manager_role = 0
        if manager_role:
            self._manager_role = 1
        self._debug = 0
        self._allow_nested = 0

    def _check_quota(self, filechange, sizechange=0, objsize=0):
        """ Check if the change in usage violates the quota settings """
        #
        # print "I'm a quota folder: ", filechange, sizechange, objsize
        # print "Current quota: ", self._quota_filecount, self._quota_size 

        #
        # The emergency user can create *some* objects (i.e. userfolders).
        # make sure this never gives a QuotaExceeded exception

        emergency = 0
        user=getSecurityManager().getUser()
        if (SpecialUsers.emergency_user and
            aq_base(user) is SpecialUsers.emergency_user):
            LOG('QuotaFolder', INFO, 
                "Emergency user detected - not enforcing quota")
            emergency = 1

        if not emergency and \
           filechange > 0 and \
           self._quota_filecount + filechange > self._quota_objects \
                                              > 0:
            raise QuotaExceededException, \
                  "Too many files, this operation would require a quotum " + \
                  "of %d files, your current quotum is %d files" % \
                  (self._quota_filecount + filechange, self._quota_objects)
        if not emergency and \
           sizechange > 0 and \
           self._quota_size + sizechange > self._quota_bytes > 0:
            raise QuotaExceededException, \
                 "Too much space, this operation would require a " + \
                 "quotum of %d bytes, your current quotum is %d bytes" % \
                 (self._quota_size + sizechange, self._quota_bytes)
        if not emergency and objsize > self._quota_maxsize > 0:
            raise QuotaExceededException, \
                  "The object is too large. It requires %d bytes, " + \
                  "your maximum object size limit is %d bytes" % \
                  (objsize, self._quota_maxsize)

        self._quota_filecount = self._quota_filecount + filechange
        self._quota_size = self._quota_size + sizechange

        ##
        ## This should not happen... 
        path = string.join(self.getPhysicalPath(), "/")
        if self._quota_filecount < 0:
            LOG('QuotaFolder', WARNING, "Negative quota filecount in %s" % path)
            if not self._debug:
                self._quota_filecount = 0
        if self._quota_size < 0:
            LOG('QuotaFolder', WARNING, "Negative quota size in %s" % path)
            if not self._debug:
                self._quota_size = 0

        # print "New quota: ", self._quota_filecount, self._quota_size 

    def current_count(self):
        return self._quota_filecount

    def current_size(self):
        return self._quota_size

    def quota_bytes(self):
        return self._quota_bytes

    def quota_objects(self):
        return self._quota_objects

    def quota_maxsize(self):
        return self._quota_maxsize

    def authorized_to_edit(self):
        """ 
            return true/false if user is authorized to change in current context
        """

        security=getSecurityManager()
        security.addContext(self)
        user = security.getUser()
        if not self._manager_role:
            ## XXX use permission?
            return user.allowed(self, ['Manager'])
            
        ## use aq_base? 
        return user.allowed(aq_parent(self), ['Manager'])
        
    def manager_role(self):
        return self._manager_role

    def manage_editQuota(self, quota=-1, maxFileSize=-1, maxNumberOfFiles=-1,
                         REQUEST=None):
        """ Update the current Quota """
        ##
        ## Perhaps always sync after edit?

        if REQUEST.has_key('Sync'):
            self._qa_sync()
            ## XXX report difference to parent?
            if REQUEST:
                ## A redirect selects the proper tab
                message = "Quota synchronized"
                REQUEST.RESPONSE.redirect(
                      'manage_editQuotaForm?manage_tabs_message=%s' % message)
                return self.manage_editQuotaForm(self, REQUEST,
                                                 manage_tabs_message=message)
            return ''

        if not self.authorized_to_edit():
           raise "Unauthorized", \
                 "You are not authorized to edit this quota folder"

        if quota != -1:
            self._quota_bytes=int(quota)
        if maxFileSize != -1:
            self._quota_maxsize=int(maxFileSize)
        if maxNumberOfFiles != -1:
            self._quota_objects=int(maxNumberOfFiles)
        self._manager_role = REQUEST.get("manager_role", 0)
        if REQUEST:
            message = "Changes saved"
            ## A redirect selects the proper tab
            REQUEST.RESPONSE.redirect(
                     'manage_editQuotaForm?manage_tabs_message=%s' % message)
            return self.manage_editQuotaForm(self, REQUEST,
                                                 manage_tabs_message=message)
        return ''

    def manage_afterAdd(self, item, container):
        # print "afterAdd %s %s"%(str(item), str(container))
        if not self._allow_nested:
            for parent in self.aq_chain[1:]:
                if hasattr(parent, "meta_type") and \
                   parent.meta_type == "QuotaFolder":
                    raise QuotaExceededException, \
                          "Cannot add a QuotaFolder inside another QuotaFolder"

        Folder.manage_afterAdd(self, item, container)

Globals.default__class_init__(QuotaFolder)
